/*
 * Copyright 2021 DataCanvas
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.dingodb.exec.transaction.impl;

import com.google.common.collect.Iterators;
import io.dingodb.common.CommonId;
import io.dingodb.common.codec.PrimitiveCodec;
import io.dingodb.common.log.LogUtils;
import io.dingodb.common.store.KeyValue;
import io.dingodb.exec.Services;
import io.dingodb.exec.transaction.base.CacheToObject;
import io.dingodb.exec.transaction.base.TxnLocalData;
import io.dingodb.exec.transaction.util.TransactionCacheToMutation;
import io.dingodb.exec.utils.ByteUtils;
import io.dingodb.store.api.StoreInstance;
import io.dingodb.store.api.transaction.data.Op;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

import java.util.Iterator;
import java.util.List;

import static io.dingodb.common.util.NoBreakFunctions.wrap;

@Slf4j
public class TransactionCache {
    private final StoreInstance cache = Services.LOCAL_STORE.getInstance(null, null);
    private final CommonId txnId;

    @Setter
    private CommonId jobId;

    private final boolean pessimisticRollback;

    private final boolean optimisticRollback;

    private final boolean pessimisticResidualLock;

    private boolean pessimisticTransaction;

    private final boolean cleanCache;

    private final boolean cleanExtraDataCache;

    public TransactionCache(CommonId txnId) {
        this.txnId = txnId;
        this.pessimisticRollback = false;
        this.optimisticRollback = false;
        this.cleanCache = false;
        this.pessimisticResidualLock = false;
        this.cleanExtraDataCache = false;
    }

    public TransactionCache(CommonId txnId, long jobSeqId) {
        this.txnId = txnId;
        this.jobId = new CommonId(CommonId.CommonType.JOB, txnId.seq, jobSeqId);
        this.pessimisticRollback = true;
        this.optimisticRollback = false;
        this.cleanCache = false;
        this.pessimisticResidualLock = false;
        this.cleanExtraDataCache = false;
    }

    public TransactionCache(CommonId txnId, long jobSeqId, boolean optimisticRollback) {
        this.txnId = txnId;
        this.jobId = new CommonId(CommonId.CommonType.JOB, txnId.seq, jobSeqId);
        this.pessimisticRollback = false;
        this.optimisticRollback = optimisticRollback;
        this.cleanCache = false;
        this.pessimisticResidualLock = false;
        this.cleanExtraDataCache = false;
    }

    public TransactionCache(CommonId txnId, boolean cleanCache, boolean pessimisticTransaction) {
        this.txnId = txnId;
        this.pessimisticRollback = false;
        this.optimisticRollback = false;
        this.cleanCache = cleanCache;
        this.pessimisticTransaction = pessimisticTransaction;
        this.pessimisticResidualLock = false;
        this.cleanExtraDataCache = false;
    }

    public TransactionCache(CommonId txnId, boolean pessimisticResidualLock) {
        this.txnId = txnId;
        this.pessimisticRollback = false;
        this.optimisticRollback = false;
        this.cleanCache = false;
        this.pessimisticTransaction = true;
        this.pessimisticResidualLock = pessimisticResidualLock;
        this.cleanExtraDataCache = false;
    }

    public TransactionCache(CommonId txnId, boolean cleanCache, boolean pessimisticTransaction, boolean cleanExtraDataCache) {
        this.txnId = txnId;
        this.pessimisticRollback = false;
        this.optimisticRollback = false;
        this.cleanCache = cleanCache;
        this.pessimisticTransaction = pessimisticTransaction;
        this.pessimisticResidualLock = false;
        this.cleanExtraDataCache = cleanExtraDataCache;
    }


    public CacheToObject getPrimaryKey() {
        // call StoreService
        CacheToObject primaryKey = null;
        Iterator<KeyValue> iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_DATA, txnId));
        while (iterator.hasNext()) {
            KeyValue keyValue = iterator.next();
            Object[] tuple = ByteUtils.decode(keyValue);
            TxnLocalData txnLocalData = (TxnLocalData) tuple[0];
            CommonId.CommonType type = txnLocalData.getDataType();
            CommonId txnId = txnLocalData.getTxnId();
            CommonId tableId = txnLocalData.getTableId();
            CommonId newPartId = txnLocalData.getPartId();
            int op = txnLocalData.getOp().getCode();
            byte[] key = txnLocalData.getKey();
            byte[] value = txnLocalData.getValue();
            byte[] txnIdByte = txnId.encode();
            byte[] tableIdByte = tableId.encode();
            byte[] partIdByte = newPartId.encode();
            int len = txnIdByte.length + tableIdByte.length + partIdByte.length;
            byte[] checkBytes = ByteUtils.encode(
                CommonId.CommonType.TXN_CACHE_CHECK_DATA,
                key,
                Op.CheckNotExists.getCode(),
                len,
                txnIdByte, tableIdByte, partIdByte);
            keyValue = cache.get(checkBytes);
            if (keyValue != null && keyValue.getValue() != null) {
                switch (Op.forNumber(op)) {
                    case PUT:
                        op = Op.PUTIFABSENT.getCode();
                        break;
                    case DELETE:
                        op = Op.CheckNotExists.getCode();
                        break;
                    default:
                        break;
                }
            }
            primaryKey = new CacheToObject(TransactionCacheToMutation.cacheToMutation(
                op,
                key,
                value,
                0L,
                tableId,
                newPartId, txnId), tableId, newPartId
            );
            LogUtils.debug(log, "txnId:{} primary key is {}" , txnId, primaryKey);
            if (op != Op.CheckNotExists.getCode()) {
                break;
            }
        }
        if (primaryKey == null) {
            throw new RuntimeException(txnId + ",PrimaryKey is null");
        }
        return primaryKey;
    }

    public KeyValue get(byte[] key) {
        return cache.get(key);
    }

    public List<KeyValue> getKeys(List<byte[]> keys) {
        return cache.get(keys);
    }

    public void deleteKey(byte[] key) {
        cache.delete(key);
    }

    public byte[] getScanPrefix(CommonId.CommonType type, long startTs) {
        byte[] startTsByte = PrimitiveCodec.encodeLong(startTs);
        byte[] result = new byte[startTsByte.length + CommonId.TYPE_LEN + CommonId.TYPE_LEN];
        result[0] = (byte) type.getCode();
        result[1] = (byte) CommonId.CommonType.JOB.getCode();
        System.arraycopy(startTsByte, 0, result, CommonId.TYPE_LEN + CommonId.TYPE_LEN, startTsByte.length);
        return result;
    }

    public byte[] getScanPrefix(CommonId.CommonType type, CommonId commonId) {
        byte[] txnByte = commonId.encode();
        byte[] result = new byte[txnByte.length + CommonId.TYPE_LEN];
        result[0] = (byte) type.getCode();
        System.arraycopy(txnByte, 0, result, CommonId.TYPE_LEN, txnByte.length);
        return result;
    }

    private static byte[] getScanCacheDataPrefix(CommonId txnId, CommonId tableId, CommonId partId) {
        return ByteUtils.encodeCacheData(CommonId.LEN * 3, txnId.encode(), tableId.encode(), partId.encode());
    }

    public boolean checkContinue() {
        Iterator<KeyValue> iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_DATA, txnId));
        return iterator.hasNext();
    }

    public boolean checkCleanContinue(boolean isPessimistic) {
        Iterator<KeyValue> iterator;
        if (isPessimistic) {
            iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_LOCK, txnId));
        } else {
            iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_DATA, txnId));
        }
        return iterator.hasNext();
    }

    public boolean checkCleanExtraDataContinue() {
        Iterator<KeyValue> iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_EXTRA_DATA, txnId.seq));
        return iterator.hasNext();
    }

    public boolean checkPessimisticLockContinue() {
        Iterator<KeyValue> iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_EXTRA_DATA, jobId));
        return iterator.hasNext();
    }

    public boolean checkOptimisticLockContinue() {
        Iterator<KeyValue> iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_EXTRA_DATA, jobId));
        return iterator.hasNext();
    }

    public boolean checkResidualPessimisticLockContinue() {
        Iterator<KeyValue> iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_RESIDUAL_LOCK, txnId));
        return iterator.hasNext();
    }

    public Iterator<Object[]> iterator() {
        Iterator<KeyValue> iterator;
        if (pessimisticRollback) {
            iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_EXTRA_DATA, jobId));
            return Iterators.transform(iterator, wrap(ByteUtils::decode)::apply);
        } else if (optimisticRollback) {
            iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_EXTRA_DATA, jobId));
            return Iterators.transform(iterator, wrap(ByteUtils::decodeTxnCleanUp)::apply);
        } else if (pessimisticResidualLock) {
            iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_RESIDUAL_LOCK, txnId));
            return Iterators.transform(iterator, wrap(ByteUtils::decode)::apply);
        } else if (cleanCache) {
            if (pessimisticTransaction) {
                iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_LOCK, txnId));
            } else {
                iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_DATA, txnId));
            }
            return Iterators.transform(iterator, wrap(ByteUtils::decodeTxnCleanUp)::apply);
        } else if (cleanExtraDataCache) {
            iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_EXTRA_DATA, txnId.seq));
            return Iterators.transform(iterator, wrap(ByteUtils::decodeTxnCleanUp)::apply);
        } else {
            iterator = cache.scan(getScanPrefix(CommonId.CommonType.TXN_CACHE_DATA, txnId));
            return Iterators.transform(iterator, wrap(ByteUtils::decode)::apply);
        }
    }

    public static Iterator<Object[]> getCacheData(CommonId txnId, CommonId tableId, CommonId partId) {
        StoreInstance storeInstance = Services.LOCAL_STORE.getInstance(tableId, partId);
        Iterator<KeyValue> iterator = storeInstance.scan(getScanCacheDataPrefix(txnId, tableId, partId));
        return Iterators.transform(iterator, wrap(ByteUtils::decode)::apply);
    }

    public void checkCache() {
        Iterator<KeyValue> iterator = cache.scan((byte[]) null);
        Iterator<Object[]> transform = Iterators.transform(iterator, wrap(ByteUtils::decode)::apply);
        while (transform.hasNext()) {
            Object[] next = transform.next();
            TxnLocalData txnLocalData = (TxnLocalData) next[0];
            LogUtils.info(log, "txnId:{} tableId:{} partId:{} Op:{} Key:{} ", txnLocalData.getTxnId(), txnLocalData.getTableId(),
                txnLocalData.getPartId(), txnLocalData.getOp(), txnLocalData.getKey());
        }
    }

    public void cleanCache() {
        Iterator<KeyValue> iterator = cache.scan((byte[]) null);
        Iterator<Object[]> transform = Iterators.transform(iterator, wrap(ByteUtils::decode)::apply);
        while (transform.hasNext()) {
            Object[] next = transform.next();
            TxnLocalData txnLocalData = (TxnLocalData) next[0];
            LogUtils.info(log, "txnId:{} tableId:{} partId:{} Op:{} Key:{} ", txnLocalData.getTxnId(), txnLocalData.getTableId(),
                txnLocalData.getPartId(), txnLocalData.getOp(), txnLocalData.getKey());
        }
        while (iterator.hasNext()) {
            cache.delete(iterator.next().getKey());
            LogUtils.info(log, "key is {}", iterator.next().getKey());
        }
    }
}
