/*
 * Copyright (c) 2024. Devtron Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package health

import (
	"fmt"
	"github.com/devtron-labs/common-lib/utils/k8s/commonBean"

	"k8s.io/apimachinery/pkg/apis/meta/v1/unstructured"
	"k8s.io/apimachinery/pkg/runtime"
	apiregistrationv1 "k8s.io/kube-aggregator/pkg/apis/apiregistration/v1"
	apiregistrationv1beta1 "k8s.io/kube-aggregator/pkg/apis/apiregistration/v1beta1"
)

func getAPIServiceHealth(obj *unstructured.Unstructured) (*HealthStatus, error) {
	gvk := obj.GroupVersionKind()
	switch gvk {
	case apiregistrationv1.SchemeGroupVersion.WithKind(commonBean.APIServiceKind):
		var apiService apiregistrationv1.APIService
		err := runtime.DefaultUnstructuredConverter.FromUnstructured(obj.Object, &apiService)
		if err != nil {
			return nil, fmt.Errorf("failed to convert unstructured APIService to typed: %v", err)
		}
		return getApiregistrationv1APIServiceHealth(&apiService)
	case apiregistrationv1beta1.SchemeGroupVersion.WithKind(commonBean.APIServiceKind):
		var apiService apiregistrationv1beta1.APIService
		err := runtime.DefaultUnstructuredConverter.FromUnstructured(obj.Object, &apiService)
		if err != nil {
			return nil, fmt.Errorf("failed to convert unstructured APIService to typed: %v", err)
		}
		return getApiregistrationv1beta1APIServiceHealth(&apiService)
	default:
		return nil, fmt.Errorf("unsupported APIService GVK: %s", gvk)
	}
}

func getApiregistrationv1APIServiceHealth(apiservice *apiregistrationv1.APIService) (*HealthStatus, error) {
	for _, c := range apiservice.Status.Conditions {
		switch c.Type {
		case apiregistrationv1.Available:
			if c.Status == apiregistrationv1.ConditionTrue {
				return &HealthStatus{
					Status:  HealthStatusHealthy,
					Message: fmt.Sprintf("%s: %s", c.Reason, c.Message),
				}, nil
			} else {
				return &HealthStatus{
					Status:  HealthStatusProgressing,
					Message: fmt.Sprintf("%s: %s", c.Reason, c.Message),
				}, nil
			}
		}
	}
	return &HealthStatus{
		Status:  HealthStatusProgressing,
		Message: "Waiting to be processed",
	}, nil
}

func getApiregistrationv1beta1APIServiceHealth(apiservice *apiregistrationv1beta1.APIService) (*HealthStatus, error) {
	for _, c := range apiservice.Status.Conditions {
		switch c.Type {
		case apiregistrationv1beta1.Available:
			if c.Status == apiregistrationv1beta1.ConditionTrue {
				return &HealthStatus{
					Status:  HealthStatusHealthy,
					Message: fmt.Sprintf("%s: %s", c.Reason, c.Message),
				}, nil
			} else {
				return &HealthStatus{
					Status:  HealthStatusProgressing,
					Message: fmt.Sprintf("%s: %s", c.Reason, c.Message),
				}, nil
			}
		}
	}
	return &HealthStatus{
		Status:  HealthStatusProgressing,
		Message: "Waiting to be processed",
	}, nil
}
