#!/bin/bash
set -e

# Optional: Import test library
source dev-container-features-test-lib

FAILED=()

echoStderr()
{
    echo "$@" 1>&2
}

check-version-ge() {
    LABEL=$1
    CURRENT_VERSION=$2
    REQUIRED_VERSION=$3
    shift
    echo -e "\n🧪 Testing $LABEL: '$CURRENT_VERSION' is >= '$REQUIRED_VERSION'"
    local GREATER_VERSION=$((echo ${CURRENT_VERSION}; echo ${REQUIRED_VERSION}) | sort -V | tail -1)
    if [ "${CURRENT_VERSION}" == "${GREATER_VERSION}" ]; then
        echo "✅  Passed!"
        return 0
    else
        echoStderr "❌ $LABEL check failed."
        FAILED+=("$LABEL")
        return 1
    fi
}

checkPythonPackageVersion()
{
    PACKAGE=$1
    REQUIRED_VERSION=$2

    current_version=$(python -c "import importlib.metadata; print(importlib.metadata.version('${PACKAGE}'))")
    check-version-ge "${PACKAGE}-requirement" "${current_version}" "${REQUIRED_VERSION}"
}

checkPythonPackageVersion "setuptools" "65.5.1"

# Check that tools can execute - make sure something didn't get messed up in this scenario
check "autopep8" autopep8 --version
check "black" black --version
check "yapf" yapf --version
check "bandit" bandit --version
check "flake8" flake8 --version
check "mypy" mypy --version
check "pycodestyle" pycodestyle --version
check "pydocstyle" pydocstyle --version
check "pylint" pylint --version
check "pytest" pytest --version
check "setuptools" pip list | grep setuptools

# Check paths in settings
check "which autopep8" bash -c "which autopep8 | grep /usr/local/py-utils/bin/autopep8"
check "which black" bash -c "which black | grep /usr/local/py-utils/bin/black"
check "which yapf" bash -c "which yapf | grep /usr/local/py-utils/bin/yapf"
check "which bandit" bash -c "which bandit | grep /usr/local/py-utils/bin/bandit"
check "which flake8" bash -c "which flake8 | grep /usr/local/py-utils/bin/flake8"
check "which mypy" bash -c "which mypy | grep /usr/local/py-utils/bin/mypy"
check "which pycodestyle" bash -c "which pycodestyle | grep /usr/local/py-utils/bin/pycodestyle"
check "which pydocstyle" bash -c "which pydocstyle | grep /usr/local/py-utils/bin/pydocstyle"
check "which pylint" bash -c "which pylint | grep /usr/local/py-utils/bin/pylint"
check "which pytest" bash -c "which pytest | grep /usr/local/py-utils/bin/pytest"

HL="\033[1;33m"
N="\033[0;37m"

checkVulnerableFile_OR_DIR()
{
    for arg in "$@"; do
        if [[ -e $arg ]]; then
            echo -e "\n✅${HL} Vulnerable:- ${N}${arg} - ${HL}EXISTS ${N}in v3.10 as skipVulnerabilityPatching=true"
        else
            echo -e "\n❌${HL} Vulnerable:- ${arg} - ${N}don't ${HL}EXISTS ${N}in v3.10 as skipVulnerabilityPatching=false"
        fi
    done
}

# print setuptools
check "Show All Files/Folders which include setuptools" bash -c 'find / -name "*setuptools*"'

# only for 3.10
checkVulnerableFile_OR_DIR "/usr/local/py-utils/shared/lib/python3.10/site-packages/setuptools-65.5.0.dist-info" "/usr/local/lib/python3.10/ensurepip/_bundled/setuptools-65.5.0-py3-none-any.whl"

# Function to check if a package is installed
checkPackageInstalled() {
    if python -c "import $1" &>/dev/null; then
        echo -e "\n✅ Passed! \n$1 is installed"
    else
        echo -e "$1 is NOT installed\n"
        echoStderr "❌ check failed."
    fi
}

# Function to install a package using pip
installPackage() {
    python3 -m pip install "$1"
}

checkPipWorkingCorrectly() {
    echo -e "\n🧪 Testing whether pip install works fine \n"
    # List of packages to install via pip
    packages=("numpy" "requests" "matplotlib")
    # Install packages and check if installation was successful
    for package in "${packages[@]}"; do
        echo -e "\n🧪 Testing pip install $package\n"
        installPackage "$package"
        checkPackageInstalled "$package"
    done
}

checkPipWorkingCorrectly

# Report result
reportResults
