{
 * This Source Code Form is subject to the terms of the Mozilla Public License,
 * v. 2.0. If a copy of the MPL was not distributed with this file, You can
 * obtain one at http://mozilla.org/MPL/2.0/
 *
 * Copyright (C) 2006-2012, Peter Johnson (www.delphidabbler.com).
 *
 * Implements a class that enables a source code document to the highlighted and
 * provides access to highlighted code.
}


unit UDocument;


interface


uses
  // Delphi
  Classes,
  // Project
  UPasHi, UInputData, UOutputData, UOptions;


type
  ///  <summary>Enumeration of types of output document: complete document or
  ///  document fragment.</summary>
  TDocOutputType = (
    doComplete,
    doFragment
  );

type
  ///  <summary>Enumeration of supported input sources: binary data of files.
  ///  </summary>
  TDocInputSource = (
    isData,
    isFiles
  );

type
  ///  <summary>Class that enables a source code document to the highlighted and
  ///  provides access to highlighted code.</summary>
  TDocument = class(TObject)
  strict private
    var
      ///  <summary>Value of OutputType property.</summary>
      fOutputType: TDocOutputType;
      ///  <summary>Records source of input to be highlighted.</summary>
      fInputSource: TDocInputSource;
      ///  <summary>Value of InputData property.</summary>
      fInputData: IInputData;
      ///  <summary>Value of InputFiles property.</summary>
      fInputFiles: TArray<string>;
      ///  <summary>Reference to object that interacts with PasHi.</summary>
      fPasHi: TPasHi;
      ///  <summary>Stream that contains highlighted source code.</summary>
      fHilitedStream: TMemoryStream;
    ///  <summary>Write accessor for InputData property.</summary>
    ///  <remarks>Clears InputFiles property and records input source type.
    ///  </remarks>
    procedure SetInputData(const Value: IInputData);
    ///  <summary>Write accessor for InputFiles property.</summary>
    ///  <remarks>Clears InputData property and records input source type.
    ///  </remarks>
    procedure SetInputFiles(const Value: TArray<string>);
    ///  <summary>Read accessor for DisplayHTML property.</summary>
    function GetDisplayHTML: string;
    ///  <summary>Read accessor for HilitedCode property.</summary>
    function GetHilitedCode: string;
    ///  <summary>Creates and returns a complete XHTML document containing a
    ///  highlighted HTML fragment.</summary>
    ///  <remarks>For use only when document type is HTML fragment.</remarks>
    function SampleHTMLFragmentDoc: string;
    ///  <summary>Loads HTML template from resources that is used for generating
    ///  complete HTML documents from HTML fragments.</summary>
    function LoadHTMLTemplate: string;
  public

    ///  <summary>Constructs initialised object instance.</summary>
    constructor Create;

    ///  <summary>Destroys object instance.</summary>
    destructor Destroy; override;

    ///  <summary>Uses PasHi to highlight Pascal code read from input source
    ///  specified by InputData or InputFiles property using given Options.
    ///  </summary>
    ///  <remarks>Highlighted code is stored in fHilitedStream for later use by
    ///  HilitedCode and DisplayHTML properties and Save method.</remarks>
    procedure Highlight(const Options: TOptions);

    ///  <summary>Saves highlighted source code to output sink specified by
    ///  OutputData property.</summary>
    procedure Save(const OutputData: IOutputData);

    ///  <summary>Specifies type of hilighted document required: either complete
    ///  HTML document of HTML frgament.</summary>
    property OutputType: TDocOutputType read fOutputType write fOutputType;

    ///  <summary>Contains raw source code to be highlighted, as a binary data.
    ///  </summary>
    ///  <remarks>This property and InputFiles are mutually exclusive. Setting
    ///  this property clears InputFiles.</remarks>
    property InputData: IInputData read fInputData write SetInputData;

    ///  <summary>Array of names of files containing raw source code to be
    ///  highlighted.</summary>
    ///  <remarks>This property and InputData are mutually exclusive. Setting
    ///  this property clears InputData.</remarks>
    property InputFiles: TArray<string> read fInputFiles write SetInputFiles;

    ///  <summary>A string containing highlighted source exactly as output from
    ///  PasHi during the last call to the Highlight method.</summary>
    ///  <remarks>Returns empty string if Hilight has not been called.</remarks>
    property HilitedCode: string read GetHilitedCode;

    ///  <summary>A string containing a complete HTML document containing
    ///  highlighted source generated by last call to Highlight method. The
    ///  content depends on the value of OutputType: if doComplete DisplayHTML
    ///  is the same as HilitedCode; if OutputType = doFragment DisplayHTML
    ///  contains the code fragment from HilitedCode embedded in a complete HTML
    ///  document.</summary>
    ///  <remarks>The HTML document will have no displayable content if the
    ///  Highlight method has not been called.</remarks>
    property DisplayHTML: string read GetDisplayHTML;
  end;


implementation


uses
  // Delphi
  SysUtils, Windows,
  // Project
  UUtils;


{ TDocument }

constructor TDocument.Create;
begin
  inherited Create;
  fPasHi := TPasHi.Create;
  fHilitedStream := TMemoryStream.Create;
end;

destructor TDocument.Destroy;
begin
  FreeAndNil(fHilitedStream);
  FreeAndNil(fPasHi);
  inherited;
end;

function TDocument.GetDisplayHTML: string;
begin
  if fOutputType = doFragment then
    Result := SampleHTMLFragmentDoc
  else
    Result := GetHilitedCode;
end;

function TDocument.GetHilitedCode: string;
begin
  Result := StringFromStream(fHilitedStream);
end;

procedure TDocument.Highlight(const Options: TOptions);
var
  SourceStm: TStream;
begin
  fHilitedStream.Size := 0;
  case fInputSource of
    isData:
    begin
      if not Assigned(fInputData) then
        Exit;
      SourceStm := TMemoryStream.Create;
      try
        fInputData.ReadData(SourceStm);
        SourceStm.Position := 0;
        fPasHi.Hilite(SourceStm, fHilitedStream, Options);
      finally
        SourceStm.Free;
      end;
    end;
    isFiles:
    begin
      if Length(fInputFiles) = 0 then
        Exit;
      fPasHi.Hilite(fInputFiles, fHilitedStream, Options);
    end;
  end;
end;

function TDocument.LoadHTMLTemplate: string;
var
  ResStm: TResourceStream;    // stream onto template resource
begin
  ResStm := TResourceStream.Create(
    HInstance, 'FRAGMENTTPLT', RT_RCDATA    // ** do not localise
  );
  try
    Result := StringFromStream(ResStm);
  finally
    ResStm.Free;
  end;
end;

function TDocument.SampleHTMLFragmentDoc: string;
begin
  Assert(fOutputType = doFragment,
    'TDocument.SampleHTMLFragmentDoc: Fragment not true');
  Result := StringReplace(
    LoadHTMLTemplate,
    '<%Fragment-Here%>',    // ** do not localise
    GetHilitedCode,
    []
  );
end;

procedure TDocument.Save(const OutputData: IOutputData);
begin
  fHilitedStream.Position := 0;
  OutputData.WriteData(fHilitedStream);
end;

procedure TDocument.SetInputData(const Value: IInputData);
begin
  fInputData := Value;
  fInputSource := isData;
  SetLength(fInputFiles, 0);
end;

procedure TDocument.SetInputFiles(const Value: TArray<string>);
begin
  fInputFiles := Copy(Value);
  fInputSource := isFiles;
  fInputData := nil;
end;

end.

