package com.deer.wms.framework.shiro.web.session;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import com.deer.wms.system.domain.SysUser;
import com.deer.wms.system.service.ISysUserService;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.shiro.session.ExpiredSessionException;
import org.apache.shiro.session.InvalidSessionException;
import org.apache.shiro.session.Session;
import org.apache.shiro.session.mgt.DefaultSessionKey;
import org.apache.shiro.session.mgt.SessionKey;
import org.apache.shiro.web.session.mgt.DefaultWebSessionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.deer.wms.common.constant.ShiroConstants;
import com.deer.wms.common.utils.spring.SpringUtils;
import com.deer.wms.framework.shiro.session.OnlineSession;
import com.deer.wms.system.domain.SysUserOnline;
import com.deer.wms.system.service.ISysUserOnlineService;
import org.springframework.beans.factory.annotation.Value;

/**
 * 主要是在此如果会话的属性修改了 就标识下其修改了 然后方便 OnlineSessionDao同步
 * 
 * @author deer
 */
public class OnlineWebSessionManager extends DefaultWebSessionManager
{
    private static final Logger log = LoggerFactory.getLogger(OnlineWebSessionManager.class);

    @Value("${user.accountId}")
    private long accountId;
    
    @Override
    public void setAttribute(SessionKey sessionKey, Object attributeKey, Object value) throws InvalidSessionException
    {
        super.setAttribute(sessionKey, attributeKey, value);
        if (value != null && needMarkAttributeChanged(attributeKey))
        {
            OnlineSession s = (OnlineSession) doGetSession(sessionKey);
            s.markAttributeChanged();
        }
    }

    private boolean needMarkAttributeChanged(Object attributeKey)
    {
        if (attributeKey == null)
        {
            return false;
        }
        String attributeKeyStr = attributeKey.toString();
        // 优化 flash属性没必要持久化
        if (attributeKeyStr.startsWith("org.springframework"))
        {
            return false;
        }
        if (attributeKeyStr.startsWith("javax.servlet"))
        {
            return false;
        }
        if (attributeKeyStr.equals(ShiroConstants.CURRENT_USERNAME))
        {
            return false;
        }
        return true;
    }

    @Override
    public Object removeAttribute(SessionKey sessionKey, Object attributeKey) throws InvalidSessionException
    {
        Object removed = super.removeAttribute(sessionKey, attributeKey);
        if (removed != null)
        {
            OnlineSession s = (OnlineSession) doGetSession(sessionKey);
            s.markAttributeChanged();
        }

        return removed;
    }

    /**
     * 验证session是否有效 用于删除过期session
     */
    @Override
    public void validateSessions()
    {
        if (log.isInfoEnabled())
        {
            log.info("invalidation sessions...");
        }

        int invalidCount = 0;

        int timeout = (int) this.getGlobalSessionTimeout();
        Date expiredDate = DateUtils.addMilliseconds(new Date(), 0 - timeout);
        ISysUserOnlineService userOnlineService = SpringUtils.getBean(ISysUserOnlineService.class);
        List<SysUserOnline> userOnlineList = userOnlineService.selectOnlineByExpired(expiredDate);
        // 批量过期删除
        List<String> needOfflineIdList = new ArrayList<String>();

        //创建用户
        ISysUserService sysUserService = SpringUtils.getBean(ISysUserService.class);
        //根据报警账户Id查询账户
        List<SysUserOnline> sysUserOnlines = userOnlineService.selectByAccountId(accountId);
        //当前时间
        Date date = new Date();
        //查询到的报警登录session
        if(sysUserOnlines.size()>0){
            for(int i=0;i<sysUserOnlines.size()-1;i++){
                SysUserOnline sysUserOnline = sysUserOnlines.get(i);
                if(sysUserOnlines.get(i).getIpaddr().equals(sysUserOnlines.get(i+1).getIpaddr())){
                    needOfflineIdList.add(sysUserOnline.getSessionId());
                    invalidCount++;
                }
                else if(!sysUserOnlines.get(i).getIpaddr().equals(sysUserOnlines.get(i+1).getIpaddr())){
                    sysUserOnline.setLastAccessTime(date);
                    userOnlineService.updateBySessionId(sysUserOnline);
                }
                if(i == (sysUserOnlines.size()-2)){
                    sysUserOnline.setLastAccessTime(date);
                    userOnlineService.updateBySessionId(sysUserOnline);
                }
            }
        }
        for (SysUserOnline userOnline : userOnlineList)
        {
            try
            {
                //判断是否是报警账户
                if(!sysUserService.selectUserByLoginName(userOnline.getLoginName()).getUserId().equals(accountId)){
                    SessionKey key = new DefaultSessionKey(userOnline.getSessionId());
                    Session session = retrieveSession(key);
                    if (session != null)
                    {
                        throw new InvalidSessionException();
                    }
                }
            }
            catch (InvalidSessionException e)
            {
                if (log.isDebugEnabled())
                {
                    boolean expired = (e instanceof ExpiredSessionException);
                    String msg = "Invalidated session with id [" + userOnline.getSessionId() + "]"
                            + (expired ? " (expired)" : " (stopped)");
                    log.debug(msg);
                }
                invalidCount++;
                needOfflineIdList.add(userOnline.getSessionId());
            }

        }
        if (needOfflineIdList.size() > 0)
        {
            try
            {
                userOnlineService.batchDeleteOnline(needOfflineIdList);
            }
            catch (Exception e)
            {
                log.error("batch delete db session error.", e);
            }
        }

        if (log.isInfoEnabled())
        {
            String msg = "Finished invalidation session.";
            if (invalidCount > 0)
            {
                msg += " [" + invalidCount + "] sessions were stopped.";
            }
            else
            {
                msg += " No sessions were stopped.";
            }
            log.info(msg);
        }

    }

    @Override
    protected Collection<Session> getActiveSessions()
    {
        throw new UnsupportedOperationException("getActiveSessions method not supported");
    }
}
