/*
 * Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 * with the License. A copy of the License is located at
 *
 * http://aws.amazon.com/apache2.0/
 *
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES
 * OR CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */
package ai.djl.integration.tests.modality.cv;

import ai.djl.integration.util.TestUtils;
import ai.djl.modality.cv.Image;
import ai.djl.modality.cv.ImageFactory;
import ai.djl.ndarray.NDArray;
import ai.djl.ndarray.NDManager;
import ai.djl.ndarray.types.DataType;
import ai.djl.ndarray.types.Shape;
import ai.djl.testing.Assertions;

import org.testng.Assert;
import org.testng.annotations.Test;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Base64;

public class BufferedImageFactoryTest {

    @Test
    public void testLoadImage() throws IOException {
        try (NDManager manager = NDManager.newBaseManager(TestUtils.getEngine())) {
            ImageFactory factory = ImageFactory.getInstance();
            Image img = factory.fromUrl("https://resources.djl.ai/images/dog_bike_car.jpg");
            NDArray array = img.toNDArray(manager);
            Assert.assertEquals(new Shape(img.getHeight(), img.getWidth(), 3), array.getShape());

            try (ByteArrayOutputStream bos = new ByteArrayOutputStream()) {
                img.save(bos, "png");
                String data = Base64.getEncoder().encodeToString(bos.toByteArray());
                Image img2 = factory.fromUrl("data:image/png;base64," + data);
                Assert.assertEquals(img2.getWidth(), 768);
            }
        }
    }

    @Test
    public void testFormNDArray() {
        try (NDManager manager = NDManager.newBaseManager(TestUtils.getEngine())) {
            NDArray array = manager.arange(0.0f, 12.0f).reshape(3, 2, 2);
            ImageFactory factory = ImageFactory.getInstance();
            Image image = factory.fromNDArray(array);
            NDArray converted =
                    image.toNDArray(manager).transpose(2, 0, 1).toType(DataType.FLOAT32, true);
            Assertions.assertAlmostEquals(array, converted);

            array = array.transpose(1, 2, 0);
            image = factory.fromNDArray(array);
            converted = image.toNDArray(manager).toType(DataType.FLOAT32, false);
            Assertions.assertAlmostEquals(array, converted);
        }
    }
}
