import 'package:langchain/langchain.dart';
import 'package:test/test.dart';

void main() {
  group('MemoryVectorStore tests', () {
    test('Test MemoryVectorStore.fromDocuments', () async {
      const embeddings = _FakeEmbeddings();
      final store = await MemoryVectorStore.fromDocuments(
        documents: [
          const Document(id: '1', pageContent: 'hello'),
          const Document(id: '2', pageContent: 'hi'),
          const Document(id: '3', pageContent: 'bye'),
          const Document(id: '4', pageContent: "what's this"),
        ],
        embeddings: embeddings,
      );

      final results = await store.similaritySearch(
        query: 'chao',
        config: const VectorStoreSimilaritySearch(k: 1),
      );

      expect(results.length, 1);
      expect(results.first.id, '3');
      expect(results.first.pageContent, 'bye');
    });

    test('Test MemoryVectorStore.fromText', () async {
      const embeddings = _FakeEmbeddings();
      final store = await MemoryVectorStore.fromText(
        ids: const ['1', '2', '3', '4'],
        texts: const ['hello', 'hi', 'bye', "what's this"],
        embeddings: embeddings,
      );

      final results = await store.similaritySearch(
        query: 'chao',
        config: const VectorStoreSimilaritySearch(k: 1),
      );

      expect(results.length, 1);
      expect(results.first.id, '3');
      expect(results.first.pageContent, 'bye');
    });

    test('Test MemoryVectorStore with initialMemoryVectors', () async {
      const embeddings = _FakeEmbeddings();
      final store = MemoryVectorStore(
        embeddings: embeddings,
        initialMemoryVectors: [
          MemoryVector(
            document: const Document(id: '1', pageContent: 'hello'),
            embedding: _helloVector,
          ),
          MemoryVector(
            document: const Document(id: '2', pageContent: 'hi'),
            embedding: _hiVector,
          ),
          MemoryVector(
            document: const Document(id: '3', pageContent: 'bye'),
            embedding: _byeVector,
          ),
          MemoryVector(
            document: const Document(id: '4', pageContent: "what's this"),
            embedding: _whatsThisVector,
          ),
        ],
      );

      final results = await store.similaritySearch(
        query: 'chao',
        config: const VectorStoreSimilaritySearch(k: 1),
      );

      expect(results.length, 1);
      expect(results.first.id, '3');
      expect(results.first.pageContent, 'bye');
    });

    test('Test delete entry', () async {
      const embeddings = _FakeEmbeddings();
      final store = await MemoryVectorStore.fromDocuments(
        documents: [
          const Document(id: '1', pageContent: 'hello'),
          const Document(id: '2', pageContent: 'hi'),
          const Document(id: '3', pageContent: 'bye'),
          const Document(id: '4', pageContent: "what's this"),
        ],
        embeddings: embeddings,
      );
      await store.delete(ids: ['3']);

      final results = await store.similaritySearch(
        query: 'chao',
        config: const VectorStoreSimilaritySearch(k: 1),
      );

      expect(results.length, 1);
      expect(results.first.id, '2');
      expect(results.first.pageContent, 'hi');
    });

    test('Test scoreThreshold filter', () async {
      const embeddings = _FakeEmbeddings();
      final store = await MemoryVectorStore.fromText(
        ids: const ['1', '2', '3', '4'],
        texts: const ['hello', 'hi', 'bye', "what's this"],
        embeddings: embeddings,
      );

      final res = await store.similaritySearchWithScores(
        query: 'chao',
        config: const VectorStoreSimilaritySearch(scoreThreshold: 0.85),
      );

      for (final (_, score) in res) {
        expect(score, greaterThan(0.85));
      }
    });

    test('Test filtering', () async {
      const embeddings = _FakeEmbeddings();
      final store = await MemoryVectorStore.fromDocuments(
        documents: [
          const Document(
            id: '1',
            pageContent: 'hello',
            metadata: {'type': 'a'},
          ),
          const Document(
            id: '2',
            pageContent: 'hi',
            metadata: {'type': 'a'},
          ),
          const Document(
            id: '3',
            pageContent: 'bye',
            metadata: {'type': 'b'},
          ),
          const Document(
            id: '4',
            pageContent: "what's this",
            metadata: {'type': 'b'},
          ),
        ],
        embeddings: embeddings,
      );

      final results = await store.similaritySearch(
        query: 'chao',
        config: const VectorStoreSimilaritySearch(filter: {'type': 'b'}),
      );

      for (final doc in results) {
        expect(doc.metadata['type'], 'b');
      }
    });

    test('Test toMap and fromMap', () async {
      const embeddings = _FakeEmbeddings();
      final store = MemoryVectorStore(
        embeddings: embeddings,
        initialMemoryVectors: const [
          MemoryVector(
            document: Document(id: '1', pageContent: 'hello'),
            embedding: [1, 2, 3],
          ),
          MemoryVector(
            document: Document(id: '2', pageContent: 'hi'),
            embedding: [4, 5, 6],
          ),
        ],
      );

      final map = store.memoryVectors.map((final v) => v.toMap());
      final expectedMap = [
        {
          'document': {
            'id': '1',
            'pageContent': 'hello',
            'metadata': <String, dynamic>{},
          },
          'embedding': [1.0, 2.0, 3.0],
        },
        {
          'document': {
            'id': '2',
            'pageContent': 'hi',
            'metadata': <String, dynamic>{},
          },
          'embedding': [4.0, 5.0, 6.0],
        },
      ];
      expect(map, expectedMap);

      final newMap =
          expectedMap.map(MemoryVector.fromMap).toList(growable: false);
      expect(newMap, store.memoryVectors);
    });
  });
}

class _FakeEmbeddings implements Embeddings {
  const _FakeEmbeddings();

  @override
  Future<List<double>> embedQuery(
    final String query,
  ) async {
    return embedText(query);
  }

  @override
  Future<List<List<double>>> embedDocuments(
    final List<Document> documents,
  ) async {
    return [
      for (final document in documents) embedText(document.pageContent),
    ];
  }

  List<double> embedText(final String text) {
    return switch (text) {
      'hello' => _helloVector,
      'hi' => _hiVector,
      'bye' => _byeVector,
      "what's this" => _whatsThisVector,
      'chao' => _chaoVector,
      _ => throw Exception('Unknown text: $text'),
    };
  }
}

final _helloVector = [
  -0.02502486,
  -0.019432934,
  -0.027708454,
  -0.031039815,
  -0.024667928,
  0.027417623,
  -0.0124992095,
  -0.008513476,
  -0.017463202,
  -0.008453987,
  0.03252042,
  0.004293092,
  -0.024496071,
  -0.00063867593,
  0.01414506,
  -0.0015045651,
  0.039421093,
  0.002020959,
  0.026875615,
  -0.012585138,
  -0.020992856,
  0.008896846,
  0.008427547,
  -0.0030620089,
  -0.00541346,
  -0.009491732,
  0.011091313,
  -0.0017053391,
  0.0034767764,
  -0.023200544,
  0.0067255134,
  -0.00793181,
  -0.023887966,
  -0.008943115,
  0.0068477956,
  -0.013682372,
  0.009485122,
  -0.01413184,
  0.021706719,
  -0.010569137,
  0.003407373,
  -0.01458131,
  0.0052250796,
  -0.014925022,
  -0.031832997,
  0.016220551,
  -0.005416765,
  -0.0066726347,
  -0.009663588,
  0.034714885,
  0.026333608,
  0.009220729,
  -0.01960479,
  -0.008004518,
  -0.0056448043,
  -0.0057736966,
  -0.013603053,
  0.006160372,
  0.00090224337,
  -0.038601473,
  -0.00040691835,
  0.018983465,
  -0.01540093,
  0.01874551,
  -0.0004096036,
  0.01082692,
  0.019036343,
  0.0040683574,
  -0.004699597,
  0.02209009,
  0.025143836,
  0.017331004,
  0.006940333,
  -0.018031647,
  0.019089222,
  -0.010793871,
  -0.0073038745,
  -0.005377106,
  -0.01455487,
  0.0013773257,
  0.01579752,
  -0.024826564,
  -0.008434158,
  0.013398148,
  0.018877707,
  0.015176196,
  -0.021733157,
  0.023848308,
  -0.009789175,
  -0.028871788,
  -0.001676421,
  0.017991988,
  0.008824138,
  0.024866223,
  -0.021442326,
  0.0028620614,
  -0.014911802,
  0.03159504,
  -0.0018210113,
  -0.00832179,
  -0.001288919,
  0.013669152,
  -0.00067503005,
  -0.004921027,
  -0.03254686,
  -0.009452073,
  -0.011924154,
  -0.007330314,
  0.02372933,
  0.009914762,
  -0.005281263,
  0.026611222,
  -0.0024258117,
  -0.044365253,
  -0.014964681,
  -0.0065569626,
  -0.0051788106,
  -0.003744475,
  -0.012234816,
  -0.020688804,
  0.016749337,
  0.0022407363,
  0.026822736,
  -0.0035164356,
  0.021971112,
  0.004497997,
  -0.017357443,
  -0.016947633,
  0.0044318987,
  -0.0063950215,
  0.03371019,
  0.003704816,
  -0.0038039638,
  -0.0043988493,
  -0.020477287,
  0.019353615,
  -0.021402666,
  0.0249852,
  -0.039685488,
  -0.02671698,
  -0.014343356,
  0.032652617,
  -0.0003639544,
  -0.00354618,
  -0.02333274,
  0.008526695,
  0.013232903,
  -0.0010823616,
  0.0066957693,
  -0.014449113,
  0.00090306957,
  0.0012756994,
  0.012267865,
  -0.004455033,
  0.0020507032,
  0.0136427125,
  -0.010086617,
  0.008440767,
  -0.006755258,
  -0.013120535,
  -0.011943983,
  0.0031148877,
  0.012690895,
  -0.008691941,
  -0.0104105,
  0.023041908,
  0.017661495,
  0.016617142,
  -0.0014607749,
  -0.003065314,
  0.011408586,
  0.011124363,
  -0.035217233,
  0.021349788,
  0.0025943627,
  0.009888323,
  0.018137405,
  0.015929718,
  -0.02420524,
  -0.04153624,
  -0.016180892,
  0.008513476,
  0.019287517,
  0.038442835,
  -0.0125454785,
  0.009280217,
  0.017212028,
  0.009643759,
  0.0026174972,
  -0.02083422,
  0.008731601,
  0.022394143,
  0.033472236,
  -0.01333205,
  -0.7051908,
  -0.0057869162,
  0.026518684,
  0.00916785,
  0.013576614,
  0.0483576,
  0.025090957,
  0.013087486,
  -0.022420581,
  0.037358824,
  0.012552089,
  0.01747642,
  0.0070989695,
  -0.009392585,
  0.008923286,
  0.00091463677,
  -0.003420593,
  -0.0031710714,
  0.005952162,
  0.002078795,
  -0.013265952,
  0.006900674,
  -0.01997494,
  0.015242294,
  0.004534351,
  0.0030752288,
  0.016960854,
  -0.0074823406,
  -0.012704115,
  0.021204371,
  -0.013404758,
  0.005925723,
  0.007204727,
  -0.0041080164,
  0.038892306,
  -0.0031561991,
  0.007693855,
  0.022711415,
  0.009326486,
  0.039711926,
  -0.010655064,
  -0.016722899,
  0.0034899963,
  -0.0024208543,
  0.012214987,
  0.006163677,
  -0.0012319091,
  -0.0009567745,
  -0.00031541337,
  -0.0076806354,
  0.01500434,
  0.000087632034,
  -0.022169407,
  0.0048119645,
  0.005244909,
  -0.009908152,
  0.016180892,
  -0.018031647,
  0.0037808293,
  0.023570694,
  0.009485122,
  -0.0025745332,
  -0.016035475,
  -0.006913894,
  -0.014634188,
  0.0020143492,
  -0.013563395,
  -0.015136536,
  -0.005747257,
  0.0031495893,
  -0.0038072686,
  0.014277257,
  -0.012168718,
  0.0032156878,
  0.012889191,
  0.026783077,
  0.018930586,
  -0.015295173,
  -0.006834576,
  0.009385975,
  0.004412069,
  -0.007198117,
  -0.015744641,
  -0.013550174,
  0.03532299,
  -0.0149779,
  -0.02420524,
  0.003794049,
  0.003043832,
  0.010542697,
  0.027682016,
  0.028660271,
  0.0073964125,
  -0.024786904,
  -0.003784134,
  0.0009724729,
  -0.02749694,
  0.009564441,
  -0.0011377189,
  -0.027946409,
  -0.0027017726,
  -0.00048127907,
  -0.012836312,
  -0.011739078,
  0.034291856,
  0.009941202,
  -0.024443192,
  0.03910382,
  0.026055995,
  -0.018811608,
  -0.003873367,
  0.0064016315,
  -0.008639063,
  0.012459551,
  0.00916124,
  -0.026822736,
  -0.004336056,
  0.015625665,
  0.02207687,
  -0.026730198,
  -0.0063851066,
  0.0005552267,
  0.018930586,
  -0.003873367,
  -0.0014723422,
  0.025910579,
  0.000886545,
  -0.0075219995,
  -0.01788623,
  0.009544611,
  0.01667002,
  -0.0020953198,
  0.008024347,
  -0.014396234,
  0.015493468,
  -0.024271337,
  -0.002667071,
  -0.022605658,
  -0.0022027297,
  -0.005337447,
  0.0045971447,
  0.003628803,
  -0.007409632,
  -0.005403545,
  0.004527741,
  -0.011858055,
  -0.0062000314,
  -0.0062463,
  -0.0014624274,
  0.0031363696,
  0.010489818,
  0.0031760288,
  -0.003965905,
  0.017621838,
  0.022235505,
  -0.016828656,
  0.007753344,
  -0.0127503835,
  -0.023663232,
  -0.008797699,
  -0.007720295,
  0.014422674,
  -0.023398839,
  -0.0021531559,
  -0.007978079,
  -0.027206106,
  0.0036750718,
  0.010080008,
  0.0057670865,
  -0.04576654,
  0.025249593,
  -0.020226113,
  -0.021746378,
  0.0141715,
  0.018851267,
  0.0049408562,
  -0.011613491,
  0.0046467185,
  -0.0031727238,
  -0.01331883,
  0.008169764,
  -0.0045607905,
  -0.008619233,
  0.005991821,
  0.026769858,
  -0.0012087747,
  0.0059224176,
  0.02253956,
  -0.021561302,
  0.018613312,
  0.012135669,
  0.013999644,
  -0.014660628,
  -0.000008397855,
  0.013841008,
  0.0021829002,
  0.0007828531,
  0.009445463,
  0.0060942736,
  0.017317785,
  0.036169052,
  -0.01414506,
  0.018401798,
  -0.022235505,
  -0.016167672,
  -0.026743418,
  -0.003628803,
  -0.015691763,
  0.02037153,
  0.025632964,
  -0.0061504575,
  -0.014224378,
  0.0030206975,
  0.0030867958,
  0.025038078,
  0.03577246,
  -0.010628625,
  0.007832662,
  -0.0142375985,
  -0.0072443862,
  -0.0015450504,
  -0.010344402,
  0.008308571,
  -0.0039824294,
  0.007621147,
  0.02287005,
  0.014911802,
  0.032414664,
  0.00541346,
  -0.0016970767,
  -0.0142375985,
  0.011051655,
  0.009802395,
  0.010483208,
  -0.00024539037,
  -0.01163993,
  0.013027998,
  -0.021521643,
  0.031515725,
  -0.02002782,
  -0.0011104534,
  0.03204451,
  0.03410678,
  -0.03408034,
  0.001022873,
  0.016617142,
  0.0081565445,
  0.0046962923,
  0.023636792,
  0.026386486,
  0.0011162369,
  0.003870062,
  0.0020804475,
  -0.008592794,
  0.0018358835,
  -0.020490509,
  -0.011884495,
  0.00998747,
  0.015876839,
  0.025289252,
  0.0017681326,
  0.011078094,
  0.0062991786,
  -0.00093611877,
  0.0100072995,
  0.0075550484,
  -0.00791859,
  -0.0077665634,
  -0.0018689326,
  0.006537133,
  0.008097055,
  -0.0029463368,
  0.032203145,
  -0.030960497,
  0.023557475,
  0.00748895,
  -0.00395599,
  0.0050267843,
  -0.016313089,
  -0.018150624,
  -0.023134444,
  -0.026968153,
  0.004418679,
  0.01083353,
  0.01002052,
  -0.008116885,
  -0.053592592,
  0.00092207285,
  0.01038406,
  0.016736118,
  0.0072708256,
  0.013153584,
  0.028237242,
  -0.021640621,
  -0.004626889,
  0.006414851,
  0.03495284,
  -0.0016557652,
  0.0032355173,
  0.011071485,
  -0.004375715,
  0.00045070855,
  0.015638884,
  -0.019009903,
  -0.004504607,
  0.0050267843,
  -0.0112169,
  -0.011282999,
  0.00353296,
  0.0030041728,
  -0.0051127123,
  0.0149779,
  -0.01126978,
  -0.002100277,
  0.0027414316,
  0.0022803952,
  -0.020609485,
  -0.008546525,
  0.020596266,
  -0.023412058,
  -0.018044867,
  -0.009339706,
  -0.0010980599,
  0.010608795,
  0.055681303,
  0.024760466,
  0.0021548083,
  -0.013417978,
  0.0010278303,
  0.018917365,
  -0.020464068,
  -0.012975118,
  -0.01500434,
  -0.018428238,
  0.002268828,
  -0.009359535,
  -0.020900318,
  0.030828299,
  0.017331004,
  0.0024786906,
  0.011124363,
  -0.0029281597,
  -0.0033330126,
  0.0030901008,
  -0.0042666527,
  -0.012942069,
  0.005116017,
  0.017687935,
  0.022195848,
  -0.007046091,
  0.01751608,
  0.022737853,
  -0.00047838726,
  -0.014951461,
  -0.0036056684,
  -0.0000073263377,
  0.01207618,
  0.008645672,
  0.012446331,
  0.024429973,
  -0.013801348,
  0.014224378,
  0.0084077185,
  -0.002850494,
  0.012201767,
  0.012439721,
  0.009974251,
  -0.021481983,
  0.0011368927,
  -0.010146107,
  -0.013827788,
  0.024244897,
  -0.006788307,
  -0.015043999,
  0.03455625,
  0.002721602,
  -0.017198807,
  -0.017026952,
  -0.0036321077,
  0.006289264,
  -0.01413184,
  -0.0069271135,
  -0.006913894,
  -0.0022837003,
  0.0004709512,
  -0.02000138,
  -0.00058207917,
  0.0057869162,
  -0.029030424,
  -0.03965905,
  -0.026161753,
  0.00007110743,
  -0.012532259,
  0.008804308,
  -0.013292391,
  -0.006090969,
  -0.008969555,
  -0.011567222,
  0.00094768597,
  0.013603053,
  0.0006960989,
  -0.011633321,
  0.020252554,
  0.005991821,
  -0.0016706374,
  -0.032229587,
  -0.0017367358,
  -0.023676451,
  -0.00499704,
  -0.003068619,
  -0.0068081366,
  -0.004412069,
  -0.01084675,
  0.01663036,
  0.025606526,
  0.016498163,
  -0.0026406315,
  -0.0010955812,
  0.009809004,
  -0.0024737332,
  0.013312221,
  0.0060942736,
  -0.012016691,
  -0.011560612,
  0.011950593,
  -0.005122627,
  -0.006580097,
  -0.009240558,
  0.017991988,
  0.0021101919,
  0.012598357,
  0.025884138,
  -0.004015479,
  -0.010284913,
  0.03503216,
  -0.012770213,
  0.016921194,
  -0.0021151493,
  -0.0044913874,
  0.0068081366,
  0.006295874,
  0.029241938,
  -0.026293948,
  -0.014753166,
  0.0031148877,
  -0.05821948,
  0.017317785,
  0.007165068,
  0.006005041,
  0.0011195418,
  0.019300736,
  -0.012988338,
  0.0023134444,
  0.0016466767,
  -0.0071452386,
  0.015691763,
  -0.008189593,
  -0.017331004,
  -0.03167436,
  -0.0059025884,
  -0.012195157,
  0.0027563039,
  -0.009194289,
  -0.036512762,
  -0.03003512,
  -0.014938242,
  0.013080876,
  -0.027391182,
  -0.018124186,
  -0.047141388,
  -0.0051821154,
  0.005800136,
  -0.00045649215,
  0.013655932,
  -0.0065701823,
  0.010192376,
  -0.027391182,
  -0.027946409,
  -0.008506866,
  -0.024033383,
  -0.0022803952,
  -0.004712817,
  0.038416397,
  0.028898226,
  0.015731422,
  0.0121224485,
  0.008843968,
  -0.0056117554,
  0.003701511,
  0.018243162,
  -0.00055687915,
  0.0041807247,
  -0.010377451,
  0.01455487,
  0.02581804,
  0.03172724,
  0.004828489,
  -0.012148889,
  0.008182984,
  0.013933546,
  -0.006900674,
  -0.00018517884,
  -0.0057241227,
  -0.019393275,
  -0.013140365,
  0.0018011817,
  -0.011877885,
  0.004739256,
  0.01291563,
  -0.009214119,
  0.026029555,
  0.027708454,
  0.031013375,
  0.0014467291,
  0.0299558,
  -0.011203681,
  0.030748982,
  0.010172546,
  0.018005207,
  0.019234639,
  -0.0142375985,
  -0.016841875,
  -0.018586874,
  0.009762736,
  -0.0011724206,
  0.02039797,
  -0.015242294,
  -0.007627757,
  -0.017317785,
  0.02958565,
  -0.012723944,
  -0.008520085,
  -0.006593317,
  -0.0103510115,
  -0.012842922,
  -0.03873367,
  -0.014620969,
  -0.008718381,
  0.00459384,
  0.00026625267,
  -0.0166568,
  0.03320784,
  -0.021918233,
  -0.01830926,
  0.0023960674,
  -0.0062793493,
  0.021058954,
  -0.0011897715,
  0.014819264,
  0.011124363,
  -0.022936149,
  -0.028686712,
  -0.00042282327,
  -0.011606881,
  -0.014515212,
  0.020754902,
  0.0011302829,
  0.0011220205,
  -0.018018428,
  0.00036808554,
  -0.010298133,
  -0.01744998,
  -0.009399194,
  0.012585138,
  0.017648276,
  -0.008936506,
  -0.019697327,
  -0.0042038593,
  -0.021270469,
  0.026479024,
  0.010265084,
  -0.010648455,
  -0.0086588925,
  0.0046665478,
  -0.014118621,
  0.019472592,
  -0.01252565,
  0.009775955,
  0.02333274,
  -0.017225247,
  -0.017145928,
  0.0058992836,
  -0.01873229,
  0.014462333,
  -0.0073964125,
  0.01500434,
  -0.012016691,
  0.0018243162,
  0.014541651,
  -0.007667416,
  -0.03127777,
  -0.0121224485,
  -0.0020920148,
  0.026492244,
  -0.024932321,
  0.015678544,
  -0.0038965014,
  0.00014221486,
  0.01458131,
  0.00082251214,
  -0.012552089,
  -0.028528076,
  -0.020107137,
  -0.0012286042,
  0.026029555,
  0.010701333,
  -0.0025513987,
  -0.0033644093,
  -0.0028934581,
  -0.026082434,
  -0.033948146,
  0.00874482,
  0.0026373267,
  -0.0007097317,
  -0.012307525,
  -0.013827788,
  -0.0032074256,
  0.009015824,
  0.014250818,
  -0.026412927,
  0.02497198,
  0.0154141495,
  -0.032441102,
  -0.0047822203,
  -0.0049078073,
  -0.009088532,
  -0.029876484,
  0.0129222395,
  -0.0052845683,
  -0.019406494,
  0.01001391,
  -0.017555738,
  -0.02002782,
  0.014885362,
  0.0125454785,
  0.011481294,
  0.013867447,
  0.0077665634,
  0.017317785,
  0.018031647,
  0.006090969,
  -0.008103666,
  -0.019750206,
  0.012452941,
  -0.01877195,
  0.012426502,
  -0.0072443862,
  -0.011745688,
  0.021151492,
  -0.016009036,
  0.019221418,
  0.006880845,
  -0.023147665,
  -0.0021151493,
  -0.0005064791,
  -0.00066718087,
  -0.01791267,
  -0.0065007787,
  -0.0026224544,
  -0.011567222,
  -0.02334596,
  0.007528609,
  0.011408586,
  -0.011943983,
  0.01582396,
  -0.0044847773,
  -0.002293615,
  0.0029083302,
  -0.0036486324,
  -0.0006188464,
  -0.008189593,
  -0.011983642,
  -0.0058992836,
  -0.009438854,
  0.00040051507,
  0.03212383,
  0.010800481,
  -0.024297778,
  -0.018917365,
  0.008890237,
  -0.01919498,
  -0.008520085,
  -0.0059786015,
  0.0012897453,
  0.019380055,
  0.017767254,
  0.0058497097,
  0.017172368,
  -0.008771259,
  0.020530168,
  -0.020212894,
  -0.006999822,
  0.014634188,
  0.00914802,
  0.013431198,
  -0.007250996,
  -0.018586874,
  -0.033128526,
  0.008044177,
  0.0013451027,
  -0.010959117,
  0.019485813,
  -0.012043131,
  -0.018481117,
  -0.013801348,
  0.008249082,
  -0.028422318,
  -0.007819442,
  0.031066254,
  -0.024667928,
  0.0003176855,
  0.01041711,
  -0.020781342,
  -0.0018276211,
  -0.017106269,
  0.0016004079,
  -0.0016499816,
  0.016498163,
  -0.0054861684,
  -0.0048582335,
  0.022314824,
  -0.031489283,
  0.020860659,
  0.008830748,
  -0.017291345,
  0.031039815,
  -0.0032008158,
  0.006788307,
  -0.016868316,
  0.004954076,
  -0.01413184,
  0.0004742561,
  0.014211159,
  -0.017674716,
  -0.025183495,
  -0.030907618,
  -0.02161418,
  -0.021138273,
  0.012036521,
  -0.030484589,
  -0.0019614703,
  -0.0021234115,
  0.018018428,
  0.0010063483,
  -0.008929895,
  -0.013213073,
  -0.01038406,
  -0.017754033,
  0.0103047425,
  -0.009656978,
  0.014766386,
  -0.0011757255,
  0.000121972225,
  0.0113226585,
  -0.018229943,
  -0.018190283,
  0.0075153895,
  -0.0010294828,
  0.015731422,
  0.02626751,
  0.20929402,
  -0.0133915385,
  -0.0062231654,
  0.047749493,
  0.016960854,
  0.012776824,
  0.0138938865,
  0.0133915385,
  -0.0048119645,
  0.03497928,
  -0.015731422,
  0.011891104,
  -0.016207332,
  0.006632976,
  0.014885362,
  0.004111321,
  -0.02501164,
  -0.01834892,
  -0.023226982,
  -0.0015987554,
  -0.0071187993,
  -0.0103510115,
  -0.021495204,
  -0.015229074,
  0.036169052,
  0.00056803325,
  -0.009974251,
  0.00013457224,
  0.008863797,
  0.0029893008,
  -0.020146796,
  -0.037728973,
  0.007872321,
  0.010093228,
  -0.005819965,
  -0.0026042776,
  0.007759954,
  0.002749694,
  0.03080186,
  0.0050631384,
  0.0020688802,
  -0.0061868113,
  0.0034701666,
  -0.0052713486,
  -0.014806044,
  -0.0032834387,
  -0.005459729,
  -0.016934413,
  -0.001425247,
  0.021376226,
  -0.0333136,
  0.0066395854,
  0.02456217,
  0.034688447,
  0.005201945,
  -0.00015584767,
  0.005585316,
  0.005694378,
  -0.010337791,
  0.007568268,
  -0.01788623,
  0.033657312,
  -0.0052746534,
  0.0144887725,
  -0.029056862,
  0.008143324,
  -0.003946075,
  0.00010617058,
  0.008910066,
  -0.008916676,
  -0.018547215,
  -0.014938242,
  0.006418156,
  -0.0037378652,
  -0.01540093,
  -0.033948146,
  0.016207332,
  0.016339527,
  0.02368967,
  0.015850399,
  0.00458723,
  0.0006630497,
  -0.0071584582,
  -0.022565998,
  -0.01123012,
  -0.021006076,
  0.002749694,
  0.019380055,
  0.0017367358,
  -0.0046665478,
  -0.00177309,
  -0.015665324,
  -0.0065503526,
  0.011329268,
  0.022526339,
  0.012988338,
  0.003542875,
  0.018428238,
  -0.01918176,
  0.024932321,
  -0.025368571,
  -0.00750217,
  -0.0067651724,
  0.013867447,
  -0.008262302,
  0.01959157,
  0.0034371174,
  0.0002048018,
  0.001364106,
  -0.017238466,
  -0.019803084,
  -0.016326308,
  -0.00208871,
  -0.006090969,
  0.019380055,
  0.008797699,
  0.012148889,
  -0.009075312,
  0.0012542173,
  0.009742906,
  0.012862751,
  -0.005700988,
  0.0024836478,
  -0.0010460074,
  -0.008361449,
  -0.015942937,
  -0.018824829,
  -0.0041939444,
  0.007865711,
  -0.030299513,
  -0.005585316,
  -0.011335878,
  0.026148532,
  -0.0056381947,
  -0.013213073,
  -0.005165591,
  0.0011261518,
  0.00049945613,
  -0.005330837,
  0.016788997,
  0.00052506925,
  -0.020860659,
  0.0028058777,
  0.017555738,
  -0.0076409765,
  -0.025632964,
  0.025104178,
  -0.020953197,
  -0.011487904,
  0.0016169325,
  -0.020331873,
  0.0030289597,
  -0.007627757,
  0.001105496,
  0.030590346,
  -0.0041873343,
  -0.031489283,
  -0.03326072,
  -0.0010336139,
  -0.007323704,
  -0.005502693,
  -0.0020110442,
  0.018851267,
  0.0030917532,
  -0.021640621,
  -0.025223155,
  -0.17259617,
  0.048859946,
  0.0058166604,
  -0.02667732,
  0.020490509,
  0.017846571,
  0.023385618,
  -0.0012244731,
  0.00033586257,
  0.013312221,
  0.015691763,
  0.000437902,
  -0.034794204,
  -0.0038204882,
  -0.012591748,
  -0.007720295,
  0.017132709,
  0.006900674,
  0.004322836,
  0.034397613,
  0.03757034,
  -0.024773685,
  0.0030372222,
  -0.0022275166,
  0.008249082,
  -0.004630194,
  0.007409632,
  0.01877195,
  0.011864665,
  0.0023811953,
  -0.02129691,
  -0.014409454,
  0.018904146,
  0.011302829,
  0.011646541,
  0.003456947,
  0.0031760288,
  -0.024496071,
  -0.012981729,
  0.022830391,
  0.026069215,
  0.030828299,
  0.0079384195,
  -0.017317785,
  -0.014290477,
  0.014660628,
  0.016987292,
  0.010522868,
  0.015295173,
  -0.017621838,
  0.009914762,
  -0.009141411,
  0.0050697485,
  0.00031004287,
  0.021984331,
  0.0037147307,
  0.003101668,
  0.015929718,
  0.016881535,
  0.00015698373,
  -0.012036521,
  -0.004835099,
  0.018877707,
  -0.0059389425,
  0.01080709,
  -0.015242294,
  -0.011620101,
  0.018507555,
  -0.0140657425,
  0.028422318,
  0.007865711,
  -0.03838996,
  0.00353957,
  -0.040505107,
  0.021058954,
  0.018692631,
  -0.023226982,
  0.007198117,
  -0.016471725,
  -0.006537133,
  -0.021389447,
  0.01165315,
  -0.00959088,
  0.00706592,
  0.0064908643,
  0.0017813522,
  0.0107740415,
  -0.002746389,
  -0.020173235,
  -0.037120868,
  0.021891795,
  -0.013173413,
  -0.0043856297,
  -0.020662364,
  0.01750286,
  0.006176897,
  0.010093228,
  0.014277257,
  -0.01579752,
  -0.0041807247,
  0.008453987,
  -0.0027116875,
  -0.018467898,
  0.016445285,
  0.024879443,
  -0.019432934,
  0.018904146,
  0.021984331,
  0.008923286,
  -0.020331873,
  -0.005258129,
  0.0077070748,
  0.012419892,
  0.033181403,
  0.0019449458,
  0.012618187,
  -0.0011889453,
  -0.007634367,
  0.0037709144,
  -0.016762558,
  0.041932832,
  -0.013265952,
  -0.007422852,
  -0.010575746,
  -0.00458723,
  0.0058232704,
  -0.12056351,
  -0.03659208,
  0.0016210636,
  0.015057218,
  -0.022010772,
  0.0333136,
  0.010549307,
  0.007455901,
  -0.013629492,
  0.040161397,
  -0.011877885,
  -0.025659405,
  -0.029479893,
  0.0027513464,
  0.020239335,
  -0.013576614,
  -0.015202635,
  -0.008361449,
  -0.005241604,
  0.0048780628,
  0.0014888668,
  -0.0022754378,
  0.0075219995,
  -0.007330314,
  -0.004633499,
  -0.0014244209,
  0.0023167494,
  0.022552779,
  0.017066611,
  -0.013550174,
  0.009247168,
  -0.033102084,
  0.013285781,
  -0.014343356,
  -0.0055059977,
  0.0025613136,
  -0.049415175,
  0.021931453,
  0.018031647,
  -0.026505463,
  0.019776646,
  0.012737164,
  0.0037213406,
  -0.04248806,
  0.011084704,
  -0.021442326,
  -0.010694724,
  0.034794204,
  0.0092075085,
  -0.0015227422,
  -0.030537467,
  -0.015890058,
  -0.034344736,
  -0.0083284,
  0.027417623,
  -0.001956513,
  0.01418472,
  0.019525472,
  -0.012849531,
  0.021838916,
  0.010873189,
  0.0053143124,
  -0.0048483186,
  0.028633833,
  -0.007297265,
  -0.0216274,
  -0.013001557,
  0.0050631384,
  0.00047590857,
  -0.010218815,
  -0.008566354,
  0.00003459839,
  -0.014858923,
  0.012565308,
  -0.02626751,
  0.013100705,
  -0.025117397,
  -0.019208198,
  0.0040286984,
  -0.0258577,
  -0.02829012,
  -0.016299868,
  0.016987292,
  -0.002553051,
  0.013497296,
  -0.000829122,
  -0.0028604087,
  -0.011983642,
  -0.0034272028,
  -0.032837693,
  -0.0032784813,
  -0.003919636,
  0.009405804,
  -0.004461643,
  0.012274476,
  0.020226113,
  -0.005915808,
  -0.008923286,
  0.016114794,
  0.009650368,
  -0.018626533,
  -0.008645672,
  -0.062291145,
  0.037702534,
  -0.0040981015,
  0.00562167,
  0.010291523,
  0.00007792383,
  0.020781342,
  -0.010456769,
  -0.009438854,
  -0.016022256,
  -0.02125725,
  -0.0006593317,
  -0.006834576,
  0.0056481096,
  -0.032996327,
  -0.006699074,
  0.016207332,
  0.0021845526,
  0.020873878,
  0.011117753,
  0.007462511,
  -0.020305432,
  0.013576614,
  -0.007746734,
  -0.0032685664,
  0.016445285,
  -0.017463202,
  0.014832484,
  0.01664358,
  -0.01124334,
  0.0040088687,
  -0.029374136,
  0.017727595,
  0.015678544,
  0.0048086597,
  -0.002336579,
  0.0070593106,
  0.01538771,
  -0.0047756103,
  0.01664358,
  -0.027311863,
  -0.025897358,
  0.008910066,
  -0.0026356743,
  0.0083284,
  0.010225425,
  -0.00061760703,
  -0.005241604,
  0.01252565,
  0.006841186,
  0.004759086,
  0.008843968,
  -0.022989027,
  -0.008896846,
  -0.017793693,
  0.00012661978,
  -0.0028736284,
  0.009465293,
  -0.0018408408,
  -0.011064874,
  0.018031647,
  0.007674026,
  0.013199853,
  -0.0005915808,
  0.005859624,
  0.0042699575,
  -0.019274298,
  -0.010397281,
  0.009055482,
  -0.030193755,
  -0.022843612,
  -0.007574878,
  -0.0108137,
  -0.0008617581,
  0.024628269,
  0.009670198,
  -0.002817445,
  0.023002248,
  -0.018031647,
  0.028448757,
  0.013973204,
  0.0077335145,
  -0.03984412,
  -0.0021201067,
  0.021058954,
  0.028580954,
  -0.010893019,
  0.001489693,
  -0.006613146,
  0.01875873,
  -0.022737853,
  -0.016207332,
  0.0077269045,
  0.011421806,
  -0.010793871,
  -0.0096305385,
  -0.0021085395,
  -0.002100277,
  0.026386486,
  0.022711415,
  0.014620969,
  0.0049672956,
  -0.008169764,
  -0.018600093,
  0.0029810385,
  -0.0025166972,
  -0.021984331,
  -0.04462965,
  -0.010238644,
  0.013490686,
  0.01955191,
  0.007383193,
  0.012380233,
  0.01874551,
  -0.0140657425,
  0.005261434,
  -0.011567222,
  -0.0092075085,
  -0.03296989,
  0.032679055,
  0.004970601,
  -0.0030306121,
  0.02165384,
  -0.015876839,
  0.026307167,
  -0.006666025,
  -0.006322313,
  -0.0055522667,
  0.0108533595,
  -0.002412592,
  0.009703247,
  -0.010760821,
  -0.010655064,
  -0.019499032,
  -0.021164712,
  0.009954421,
  -0.00028029858,
  0.027391182,
  -0.02076812,
  0.06710311,
  -0.011170632,
  -0.0065239132,
  0.0108996285,
  0.011468074,
  0.02045085,
  0.007713685,
  0.0050201747,
  -0.01459453,
  -0.006715599,
  -0.004911112,
  -0.014197939,
  -0.013523735,
  0.00019974115,
  -0.018838048,
  0.007799613,
  -0.011144193,
  0.015850399,
  -0.02129691,
  0.0043162266,
  0.012664456,
  -0.012776824,
  0.013265952,
  -0.0071452386,
  -0.007363363,
  -0.010628625,
  0.00040175443,
  0.0016970767,
  -0.031462844,
  -0.015096878,
  0.00417742,
  0.00012971813,
  -0.01788623,
  -0.011382147,
  0.020477287,
  -0.019313956,
  -0.004382325,
  -0.023134444,
  -0.007614537,
  0.005492778,
  0.011798567,
  0.005162286,
  -0.021006076,
  -0.023425277,
  0.029003983,
  0.0023811953,
  -0.016299868,
  -0.0050565284,
  -0.006120713,
];
final _hiVector = [
  -0.035099167,
  -0.020636523,
  -0.015421565,
  -0.03990691,
  -0.027375247,
  0.021122552,
  -0.022002658,
  -0.019467426,
  -0.009484131,
  -0.013129348,
  0.029608354,
  -0.00469609,
  -0.015145711,
  -0.014134245,
  0.009057214,
  0.015171982,
  0.038356874,
  -0.0058257785,
  0.023920503,
  -0.01276811,
  -0.014961808,
  -0.003100076,
  -0.006856948,
  -0.008439826,
  -0.02280395,
  -0.00014428982,
  0.013543129,
  -0.016984738,
  0.0045417426,
  -0.0223836,
  0.014620274,
  -0.00089570525,
  -0.044951104,
  -0.009654898,
  -0.009799393,
  -0.015736828,
  0.009858505,
  -0.02107001,
  0.015158847,
  -0.005609036,
  0.0082296515,
  -0.0050343396,
  0.0069423313,
  -0.013286979,
  -0.01842969,
  -0.0084858015,
  -0.0023890946,
  0.0063150916,
  -0.010114655,
  0.020781018,
  0.021766212,
  0.0051624146,
  -0.028084587,
  -0.010915946,
  -0.016367352,
  -0.006640205,
  -0.014318148,
  0.014436372,
  -0.011172096,
  -0.018613592,
  -0.010193471,
  0.003914503,
  -0.0026879366,
  0.012163858,
  0.0061508925,
  0.001247912,
  0.022278512,
  -0.00041029212,
  0.0007828185,
  0.0068897875,
  0.030054975,
  0.030054975,
  -0.0047190776,
  -0.013608809,
  0.02431458,
  -0.01157931,
  -0.00028057495,
  0.004364408,
  -0.006702601,
  -0.0081705395,
  0.02640319,
  -0.026744723,
  -0.014830449,
  0.015605467,
  0.002666591,
  0.014423235,
  0.0005213317,
  0.019979728,
  -0.012374032,
  -0.031946547,
  0.0062724,
  0.025746394,
  0.01405543,
  0.010351102,
  -0.011310023,
  0.02093865,
  -0.0029654328,
  0.013162187,
  -0.0090178065,
  0.005136143,
  0.0015262292,
  0.007953797,
  -0.0072050495,
  -0.006134473,
  -0.036806837,
  -0.015618604,
  -0.014699089,
  -0.0077370545,
  0.017102962,
  -0.009562947,
  -0.006955467,
  0.03675429,
  -0.0025483677,
  -0.032314353,
  -0.010778018,
  -0.007060555,
  0.009779689,
  0.000030171555,
  -0.019454291,
  -0.022488687,
  0.006810972,
  -0.004676386,
  0.008741952,
  0.0019490415,
  0.017562719,
  0.008111428,
  -0.02955581,
  -0.0070999623,
  0.0082887625,
  0.0016469155,
  0.04316462,
  0.012282081,
  0.0061279046,
  -0.008065452,
  -0.03257707,
  0.030002432,
  -0.024262035,
  0.027716782,
  -0.021687396,
  -0.032918606,
  -0.01064666,
  0.037174642,
  -0.0021953399,
  -0.021372134,
  -0.013818983,
  -0.010850267,
  0.017746622,
  -0.002655097,
  -0.0038356874,
  -0.0022774395,
  0.0027519744,
  -0.0007926704,
  0.02286963,
  -0.0016034027,
  0.028820198,
  0.022580639,
  -0.008065452,
  -0.010672932,
  -0.009497267,
  -0.0062888195,
  0.00076804054,
  -0.0042954446,
  0.02315862,
  -0.006397191,
  -0.0050277715,
  0.02036067,
  0.020820426,
  0.029529538,
  0.011881435,
  -0.00075695716,
  0.01064666,
  0.02202893,
  -0.03058041,
  0.0049391044,
  -0.0033447326,
  0.0149224,
  0.010429917,
  0.015828779,
  -0.031631283,
  -0.042954445,
  -0.016459303,
  0.014515187,
  0.035703417,
  0.03888231,
  -0.0069094915,
  0.010640091,
  0.03139484,
  0.005132859,
  0.018561048,
  -0.025680715,
  0.016327944,
  0.026587093,
  0.012492255,
  -0.008991534,
  -0.6990409,
  -0.026639637,
  -0.007809302,
  -0.0050047836,
  0.015027488,
  0.049995296,
  0.02968717,
  0.022685727,
  -0.023644648,
  0.018101292,
  -0.010863402,
  0.0062132883,
  0.014068565,
  -0.013247571,
  0.000043230502,
  0.0036485007,
  0.0019868072,
  -0.0036911923,
  -0.011283752,
  0.008439826,
  -0.0052313786,
  -0.0051000193,
  -0.025588764,
  0.026363783,
  0.009904481,
  -0.0056878515,
  0.0005443195,
  -0.01064666,
  -0.024459075,
  0.013897799,
  -0.009687738,
  0.008006341,
  0.003881663,
  -0.015500381,
  0.056011543,
  0.0019096337,
  -0.009740282,
  0.022265377,
  0.00065638527,
  0.05648444,
  0.0014211419,
  -0.016170312,
  0.00007896943,
  0.0017848426,
  0.005326614,
  0.010009567,
  -0.009287092,
  -0.0016657982,
  0.011467654,
  -0.0033233867,
  0.0076910784,
  -0.009287092,
  -0.004531891,
  0.015395293,
  0.004167369,
  -0.0027339123,
  0.015447836,
  -0.022633182,
  -0.0027536163,
  0.02273827,
  0.0011592446,
  0.0028537777,
  -0.015080031,
  -0.01951997,
  -0.014016022,
  0.016262263,
  -0.020886106,
  -0.014830449,
  0.0039703306,
  -0.008183676,
  -0.010232878,
  0.018482232,
  -0.02997616,
  -0.011388839,
  0.000016471207,
  0.024406532,
  0.021306455,
  -0.00092772406,
  -0.0053397496,
  0.0079143895,
  -0.00342519,
  -0.007316705,
  0.0021427963,
  -0.009589219,
  0.02524723,
  -0.022015795,
  -0.033023693,
  0.016551254,
  -0.0036058088,
  -0.0063118073,
  0.01193398,
  0.028636295,
  0.013661352,
  -0.021910707,
  0.0030721622,
  0.0072772973,
  -0.014646546,
  0.006702601,
  -0.0006354499,
  -0.018403418,
  -0.0045515946,
  0.0034777336,
  -0.0038553912,
  -0.008906151,
  0.018705543,
  -0.0044005318,
  -0.013661352,
  0.028715111,
  0.026928626,
  -0.017877981,
  0.002625541,
  -0.005287206,
  -0.0035335612,
  0.010948786,
  0.009457859,
  -0.03210418,
  -0.013385498,
  0.032524526,
  0.023776008,
  -0.016616933,
  0.003256065,
  -0.004206777,
  0.020032272,
  -0.012295217,
  -0.002027857,
  0.021569174,
  -0.009201709,
  -0.0036189447,
  -0.025404861,
  0.006200152,
  0.008301899,
  -0.003257707,
  0.021779347,
  -0.020400077,
  0.013365794,
  -0.007999772,
  0.00342519,
  -0.015552924,
  -0.0035302774,
  -0.011585877,
  -0.0078552775,
  -0.0009154091,
  0.0047913254,
  -0.005566344,
  0.0073889527,
  -0.02811086,
  -0.030081246,
  -0.012557935,
  -0.0071196663,
  0.0036025248,
  0.0026271832,
  -0.0036320807,
  -0.00961549,
  0.020268718,
  0.031946547,
  -0.017286865,
  -0.001231492,
  -0.021713668,
  -0.0091557335,
  -0.0000045924394,
  0.001249554,
  0.025102735,
  -0.021923844,
  0.0018948559,
  0.003599241,
  -0.030028703,
  -0.004377544,
  0.021135688,
  0.0013735242,
  -0.039512835,
  0.017063554,
  -0.017523311,
  -0.0052182423,
  0.002953939,
  0.021135688,
  0.00060343114,
  -0.006249412,
  0.006177164,
  0.0027289866,
  -0.03270843,
  0.004978512,
  0.0033792143,
  -0.009195141,
  0.009759985,
  0.01994032,
  0.009589219,
  0.016879652,
  0.027716782,
  -0.014554595,
  0.0061804485,
  0.0081508355,
  0.014173653,
  -0.016367352,
  -0.0039276388,
  0.0016231065,
  0.011559606,
  0.0028291477,
  0.008380714,
  0.005178835,
  0.0223836,
  0.034678817,
  0.0083281705,
  0.039223842,
  -0.005641876,
  -0.0026830107,
  -0.031972818,
  0.0036123767,
  -0.009306796,
  0.02116196,
  0.020189902,
  0.00029863682,
  -0.018941991,
  0.0051098713,
  0.0031903854,
  0.018994534,
  0.018114427,
  -0.008111428,
  0.013050532,
  -0.019795824,
  0.006587662,
  0.009063782,
  -0.014370692,
  0.012131018,
  -0.0023595388,
  0.01157931,
  0.017339408,
  0.01842969,
  0.031289753,
  0.013464313,
  -0.0033660783,
  -0.007290433,
  -0.003272485,
  0.003114854,
  0.0011961893,
  -0.010935649,
  0.0032232252,
  0.0065613897,
  -0.022593774,
  0.027191345,
  -0.008157403,
  0.0036255126,
  0.028032044,
  0.03286606,
  -0.024459075,
  0.005727259,
  0.024853153,
  0.014265604,
  0.00591773,
  0.020255582,
  0.024695521,
  0.005421849,
  0.011290319,
  -0.005438269,
  0.011342864,
  0.012643319,
  -0.012052203,
  -0.01694533,
  -0.0034580298,
  0.009346204,
  0.018547913,
  0.00006952799,
  0.01987464,
  0.019625058,
  -0.023119211,
  0.015789371,
  -0.0031526198,
  -0.0028275058,
  -0.0014268889,
  0.00080005935,
  0.0012405231,
  -0.0039768983,
  -0.0027749622,
  0.0256019,
  -0.020255582,
  0.028610025,
  0.015145711,
  -0.005198539,
  -0.00082058425,
  -0.005457973,
  -0.0043414203,
  -0.02218656,
  -0.035388157,
  0.016932195,
  0.01842969,
  -0.004702658,
  -0.02511587,
  -0.03386439,
  0.007822438,
  0.0038159834,
  0.0072050495,
  -0.0064891423,
  0.01714237,
  0.031316023,
  -0.023079803,
  -0.0031542617,
  0.0039867503,
  0.042218834,
  -0.008728816,
  0.002825864,
  0.010397078,
  -0.0021148825,
  -0.00027626473,
  0.001712595,
  -0.005011352,
  -0.006134473,
  0.021595445,
  -0.008354442,
  -0.022764541,
  -0.0072970013,
  0.023723463,
  -0.0012610479,
  0.026232423,
  -0.013884663,
  0.010377374,
  0.0014490557,
  0.009957024,
  -0.0062691155,
  -0.00090473617,
  0.023618376,
  -0.011736941,
  -0.020754747,
  -0.0011748434,
  -0.014436372,
  0.0002672338,
  0.053489447,
  0.021188231,
  0.0015689209,
  -0.0024580583,
  0.0086828405,
  0.020150494,
  0.005389009,
  -0.020846698,
  -0.014436372,
  -0.013950342,
  0.009201709,
  -0.013608809,
  -0.005763383,
  -0.00095892185,
  0.009516971,
  -0.0008784644,
  0.022436144,
  0.00067691016,
  0.0074217925,
  -0.00742836,
  -0.0039900346,
  0.011237776,
  0.010725475,
  0.023552697,
  0.011566173,
  0.01585505,
  0.024301443,
  0.024406532,
  0.007008011,
  -0.027296433,
  -0.0050573274,
  0.002651813,
  0.014186789,
  0.009825665,
  -0.0048668566,
  0.028688839,
  -0.010876538,
  0.0097862575,
  0.017904254,
  -0.006535118,
  0.0038291195,
  0.015369021,
  0.007336409,
  -0.014252468,
  0.021542901,
  -0.020978058,
  -0.016511846,
  0.03055414,
  -0.0065843775,
  -0.010764883,
  0.028662568,
  0.0005328256,
  -0.010344533,
  -0.011099849,
  -0.006019533,
  0.01061382,
  -0.018508505,
  -0.0052018226,
  -0.005300342,
  0.014515187,
  -0.015171982,
  -0.016551254,
  0.012045635,
  -0.008157403,
  -0.020045407,
  -0.031263478,
  -0.02981853,
  -0.012400304,
  -0.016012682,
  0.003451462,
  -0.014830449,
  -0.00074176874,
  -0.018245786,
  -0.010889674,
  0.0061607445,
  0.011401975,
  0.0046041384,
  -0.008439826,
  0.02389423,
  0.008630296,
  -0.013129348,
  -0.032025363,
  -0.0018915718,
  -0.019086486,
  0.013845255,
  0.012071907,
  -0.011067009,
  -0.0082099475,
  -0.0034744497,
  0.0015130932,
  0.026232423,
  0.010232878,
  0.0031345577,
  -0.0074349283,
  -0.0011378987,
  -0.011421679,
  0.016038952,
  0.0055959,
  -0.012860062,
  -0.013135916,
  0.0068241083,
  -0.00044333717,
  0.0043282844,
  -0.0089981025,
  0.015763098,
  0.0005008068,
  0.008203379,
  0.016643206,
  -0.018272059,
  -0.013779575,
  0.031552467,
  -0.010915946,
  0.009306796,
  -0.0029490131,
  -0.0003255244,
  0.01714237,
  0.0003099255,
  0.019927183,
  0.0009236191,
  -0.03058041,
  0.004659966,
  -0.039854366,
  0.03399575,
  -0.009227981,
  0.003999886,
  0.0018915718,
  0.013063668,
  -0.016827108,
  -0.00742836,
  -0.0009679528,
  -0.008091724,
  0.023237435,
  0.011152392,
  -0.012229538,
  -0.022882765,
  -0.012787814,
  -0.022501823,
  0.0067847003,
  -0.00004271738,
  -0.032025363,
  -0.018272059,
  -0.0067321565,
  0.0018899299,
  -0.0067715645,
  -0.023066668,
  -0.04508246,
  -0.014094837,
  0.015369021,
  -0.013142483,
  0.006968603,
  -0.0050507598,
  0.0019326216,
  -0.025102735,
  -0.035072893,
  -0.012531663,
  -0.031815186,
  -0.015802506,
  -0.0004962913,
  0.035125438,
  0.026849812,
  0.02167426,
  0.0056648636,
  0.007145938,
  0.0050770314,
  0.013963479,
  0.013221299,
  -0.0068044043,
  0.00974685,
  -0.014974943,
  0.024787473,
  0.026284967,
  0.017917389,
  0.004820881,
  0.0026994306,
  0.018600456,
  0.012689294,
  -0.0050244876,
  -0.015999544,
  -0.008420122,
  -0.029240549,
  -0.008413554,
  -0.012623615,
  -0.01605209,
  0.011021033,
  0.006567958,
  -0.01266959,
  0.032524526,
  0.017930524,
  0.039644193,
  0.0055302205,
  0.02209461,
  0.00590131,
  0.02617988,
  0.006334795,
  0.010200039,
  0.015973274,
  -0.005737111,
  -0.0055499244,
  -0.003756872,
  0.016879652,
  -0.0001526024,
  0.006292104,
  -0.013405202,
  0.0072772973,
  -0.015960138,
  0.016643206,
  -0.004187073,
  -0.018114427,
  -0.013017693,
  -0.0072444575,
  -0.016643206,
  -0.014344419,
  -0.0010533362,
  -0.02473493,
  0.012078474,
  -0.004945672,
  -0.01849537,
  0.02209461,
  -0.016695749,
  -0.03142111,
  0.0041148257,
  -0.014515187,
  0.030606683,
  -0.010055543,
  0.016144041,
  0.012708998,
  -0.026482007,
  -0.026219288,
  -0.005336466,
  0.0027421224,
  -0.00038012056,
  0.017523311,
  0.0018242503,
  -0.004344704,
  -0.02167426,
  0.008617161,
  0.0005118902,
  0.007152506,
  -0.01672202,
  0.008242787,
  0.012085042,
  -0.0087813595,
  -0.02360524,
  -0.003265917,
  -0.024340851,
  0.010797722,
  0.008709112,
  0.0050441916,
  -0.009372476,
  -0.012610479,
  -0.004988364,
  0.022252241,
  -0.0298448,
  0.012485688,
  0.022462416,
  -0.016511846,
  -0.011408542,
  0.011250911,
  -0.023106076,
  0.022475552,
  0.009050646,
  0.011743508,
  -0.019178437,
  0.023697192,
  0.011152392,
  -0.002157574,
  -0.022081474,
  -0.0010730401,
  0.0003431758,
  0.029345635,
  -0.0075334474,
  0.00006573089,
  -0.0018554481,
  0.012229538,
  0.006659909,
  -0.0016304955,
  -0.010725475,
  -0.02273827,
  -0.025326045,
  -0.0017322989,
  0.039223842,
  0.019808961,
  -0.017891116,
  -0.019533107,
  -0.016669476,
  -0.024879424,
  -0.026455734,
  0.0040064543,
  0.001709311,
  0.0026386771,
  -0.02231792,
  -0.025864618,
  0.0035007214,
  0.017405089,
  0.017851708,
  -0.009589219,
  -0.011296887,
  0.01727373,
  -0.02074161,
  0.010863402,
  -0.0066040815,
  -0.0018324602,
  -0.04363751,
  0.018810632,
  -0.0052576503,
  -0.018508505,
  0.0081705395,
  -0.008512073,
  -0.03483645,
  -0.0076451027,
  0.0100818155,
  0.014961808,
  0.023434473,
  -0.008045748,
  0.021950115,
  0.008610592,
  -0.0035040055,
  -0.003428474,
  -0.022212833,
  0.009149165,
  -0.024406532,
  0.006715737,
  -0.008656569,
  0.00038935675,
  0.018823767,
  -0.014974943,
  0.024078133,
  -0.0030541003,
  -0.020400077,
  -0.020255582,
  -0.006988307,
  0.021556037,
  -0.011723804,
  0.017444495,
  -0.0034843015,
  -0.006663193,
  -0.022278512,
  0.010935649,
  0.0049128323,
  -0.008636865,
  0.016393622,
  0.003740452,
  -0.011927411,
  0.005927582,
  -0.0050671794,
  -0.01778603,
  0.0013374004,
  -0.029240549,
  -0.005632024,
  0.0026222572,
  -0.005789655,
  0.01885004,
  0.013910934,
  -0.033338953,
  -0.021871299,
  0.0069357636,
  -0.022396736,
  -0.00685038,
  0.0019079917,
  0.015513516,
  0.029765984,
  0.0063315113,
  0.006232992,
  0.019966591,
  0.0070671225,
  0.016866516,
  -0.0037667237,
  -0.017129233,
  0.007178778,
  -0.0049325363,
  0.01556606,
  0.0030672364,
  -0.035177983,
  -0.020006,
  0.019638194,
  -0.0034843015,
  -0.012938877,
  0.023421338,
  -0.006968603,
  -0.002356255,
  -0.015605467,
  0.01054814,
  0.0007011295,
  0.0004942388,
  0.008446394,
  -0.00987164,
  -0.0072444575,
  0.02218656,
  -0.024051862,
  -0.00021715311,
  -0.008249355,
  0.000620672,
  -0.006282252,
  0.0033086087,
  -0.027480336,
  -0.0067912685,
  0.015999544,
  -0.025299773,
  0.013832119,
  0.0055959,
  -0.02666591,
  0.038383145,
  -0.017194914,
  -0.0031017181,
  -0.026744723,
  0.0072313217,
  -0.009477563,
  -0.015789371,
  0.021477222,
  -0.008991534,
  -0.009195141,
  -0.00993732,
  -0.005441553,
  -0.03315505,
  -0.0007163179,
  -0.033260137,
  -0.00075121014,
  -0.0194937,
  0.032655887,
  -0.0024564161,
  0.0026764427,
  0.0025303056,
  -0.008301899,
  -0.017194914,
  -0.0084660975,
  -0.00742836,
  0.020268718,
  0.013241003,
  -0.01058098,
  -0.0010853551,
  -0.0036977602,
  -0.026219288,
  -0.006837244,
  0.018272059,
  -0.00071057095,
  0.02022931,
  0.19672348,
  -0.0039506266,
  0.0016715452,
  0.04332225,
  0.012196697,
  0.0021033885,
  0.01605209,
  0.02231792,
  -0.016774565,
  0.02216029,
  -0.026232423,
  0.010003,
  -0.020111086,
  0.003720748,
  0.01463341,
  0.0014318147,
  -0.034205925,
  -0.0064694383,
  -0.026613366,
  -0.0072444575,
  -0.011599014,
  -0.006988307,
  -0.014396964,
  -0.0032002374,
  0.030606683,
  0.007323273,
  0.005737111,
  0.0064267465,
  0.015894458,
  0.005632024,
  -0.011907708,
  -0.03139484,
  0.016511846,
  0.0087813595,
  -0.000110731664,
  0.000021923124,
  -0.008242787,
  0.0039900346,
  0.015907593,
  -0.004325,
  -0.0041509494,
  0.010429917,
  0.014869857,
  -0.014515187,
  -0.017050419,
  0.0050737476,
  0.015369021,
  -0.029765984,
  0.0019884491,
  0.012203266,
  -0.018692408,
  0.0025762815,
  0.02753288,
  0.027165074,
  -0.008630296,
  -0.00278974,
  0.008183676,
  -0.008630296,
  -0.010344533,
  0.00017825847,
  -0.0073955203,
  0.023434473,
  -0.0093133645,
  0.0088601755,
  -0.012925741,
  0.013004556,
  -0.011165529,
  -0.0072641615,
  -0.002346403,
  -0.022882765,
  -0.007375817,
  -0.044268034,
  0.011651557,
  -0.013595672,
  -0.027454063,
  -0.0028915433,
  0.03515171,
  0.021332728,
  0.01643303,
  0.016472438,
  0.017733486,
  0.0050343396,
  -0.006111485,
  -0.027243888,
  -0.011750077,
  -0.027348977,
  0.015382157,
  0.0023858107,
  -0.011395407,
  0.0061935843,
  -0.0063380795,
  -0.026324375,
  -0.008406986,
  -0.00688322,
  0.0023726746,
  0.022790814,
  -0.00012704893,
  0.02158231,
  0.0011773064,
  0.015080031,
  -0.014882992,
  -0.016235992,
  -0.0056977035,
  0.018863175,
  -0.004797893,
  0.02473493,
  0.0020771166,
  0.011021033,
  -0.009214845,
  -0.02145095,
  -0.0006465333,
  -0.03215672,
  0.022239106,
  -0.0064563025,
  0.023421338,
  0.013845255,
  0.0003680109,
  -0.006492426,
  0.014094837,
  -0.0000065519225,
  0.0086434325,
  -0.015671147,
  0.008374146,
  0.014081702,
  -0.010489029,
  -0.021148823,
  -0.032471985,
  -0.0010081815,
  0.0033578684,
  -0.024301443,
  0.007809302,
  -0.009300228,
  0.02158231,
  -0.009851936,
  -0.016406758,
  0.0011411826,
  0.0070539867,
  -0.0005283101,
  -0.012347761,
  0.014738497,
  -0.003421906,
  -0.019112756,
  0.013556265,
  0.008282195,
  0.002630467,
  -0.029293092,
  0.032603342,
  -0.027348977,
  0.004091838,
  -0.009044078,
  -0.025417997,
  -0.007651671,
  -0.0052313786,
  0.0083478745,
  0.03299742,
  -0.01308994,
  -0.049758848,
  -0.022633182,
  -0.0019933751,
  0.0021083145,
  -0.02122764,
  0.007040851,
  0.024656113,
  -0.013293547,
  0.0016863232,
  -0.023106076,
  -0.17087199,
  0.018048748,
  0.00784871,
  -0.015789371,
  0.016393622,
  -0.004650114,
  0.033522855,
  -0.016800836,
  0.00026661804,
  0.015894458,
  0.019414883,
  0.0021198085,
  -0.034022022,
  -0.017168641,
  -0.0008883163,
  -0.010383941,
  -0.010318262,
  0.0081311315,
  0.01534275,
  0.04093151,
  0.03270843,
  -0.02595657,
  0.019283524,
  -0.01951997,
  0.025851483,
  0.0031838175,
  -0.0044202358,
  0.02813713,
  0.01994032,
  -0.00562874,
  0.005944002,
  -0.018771224,
  0.023263706,
  0.01276811,
  -0.0007959544,
  0.011750077,
  -0.009530107,
  -0.031263478,
  -0.0063512153,
  0.022002658,
  0.026284967,
  0.016262263,
  0.0045811506,
  -0.005441553,
  -0.011789484,
  0.028610025,
  0.029503267,
  0.01183546,
  0.008334738,
  -0.024669249,
  0.0072050495,
  -0.020203039,
  0.020426348,
  0.0014539816,
  0.014909265,
  0.01205877,
  -0.003287263,
  0.015408429,
  0.0077173505,
  -0.013260706,
  -0.007500608,
  -0.020899242,
  0.021253912,
  0.00016686718,
  0.015001216,
  -0.040379804,
  -0.015684282,
  0.017733486,
  -0.029949887,
  0.015316478,
  0.007113098,
  -0.035466973,
  -0.0005964527,
  -0.022396736,
  0.020465756,
  0.0067847003,
  -0.01643303,
  0.00246791,
  -0.011973387,
  0.0045745824,
  -0.017391952,
  0.023789143,
  -0.010600684,
  0.002648529,
  0.007842141,
  -0.006091781,
  -0.0032478552,
  -0.009339636,
  -0.0044235196,
  -0.01283379,
  0.02007168,
  -0.018705543,
  -0.019375475,
  -0.018350873,
  0.022856493,
  0.02286963,
  0.0015984768,
  -0.0027306285,
  0.006955467,
  0.006682897,
  -0.0059505696,
  0.003740452,
  -0.02286963,
  0.007888117,
  0.016669476,
  -0.02273827,
  0.012183562,
  0.023224298,
  0.022817085,
  -0.020203039,
  -0.012873197,
  0.00836101,
  -0.007303569,
  0.016735157,
  0.0069620353,
  0.019730145,
  -0.0081377,
  -0.013595672,
  0.023933638,
  -0.018482232,
  0.03659666,
  -0.013818983,
  -0.01405543,
  -0.008196811,
  -0.009957024,
  -0.014002886,
  -0.11296888,
  -0.027138801,
  0.01022631,
  0.01122464,
  -0.008761656,
  0.021661125,
  0.006955467,
  -0.0024038726,
  -0.03244571,
  0.016984738,
  -0.012242673,
  -0.028032044,
  -0.028767655,
  -0.007375817,
  0.016708884,
  -0.0066566253,
  -0.018127564,
  -0.0056681475,
  -0.010502164,
  0.028373579,
  -0.01376644,
  -0.00075983064,
  -0.0050376235,
  -0.030895673,
  -0.013569401,
  -0.0070736906,
  -0.011638422,
  0.035834778,
  0.027348977,
  0.0030803722,
  -0.009076918,
  -0.022659454,
  0.016472438,
  -0.021017466,
  0.007671375,
  0.021267047,
  -0.03386439,
  0.0039637624,
  0.0035204254,
  -0.030895673,
  0.0075794235,
  0.025063327,
  0.00070974993,
  -0.039092485,
  -0.0019276956,
  -0.0090178065,
  -0.007809302,
  0.022462416,
  0.012997989,
  -0.011526766,
  -0.03701701,
  -0.015316478,
  -0.033916935,
  -0.009602354,
  0.0228959,
  0.008794496,
  0.010331398,
  0.026140472,
  -0.009819097,
  0.019625058,
  -0.0024301445,
  -0.0042691724,
  -0.01607836,
  0.030212605,
  -0.01109328,
  -0.015868185,
  -0.003083656,
  0.0046041384,
  -0.012071907,
  -0.0076779425,
  -0.011264048,
  0.0042330488,
  -0.019598786,
  -0.0016945332,
  -0.026390055,
  0.02244928,
  -0.028583752,
  -0.016314806,
  0.010370805,
  -0.02229165,
  -0.022593774,
  -0.013648217,
  0.012505392,
  -0.015145711,
  0.023447609,
  -0.0028669136,
  0.002014721,
  0.002625541,
  0.010594116,
  -0.013011124,
  -0.013595672,
  0.011303456,
  0.014948672,
  -0.014252468,
  0.02007168,
  0.011172096,
  -0.013661352,
  0.0044891993,
  0.016984738,
  0.00961549,
  -0.004502335,
  -0.030107519,
  -0.056011543,
  0.020794155,
  -0.01302426,
  -0.002510602,
  0.02145095,
  0.000038484126,
  0.012919173,
  -0.008597457,
  -0.008617161,
  -0.013477449,
  -0.004062282,
  -0.008439826,
  -0.0050573274,
  0.0045581628,
  -0.031815186,
  -0.020951785,
  0.015881322,
  0.022790814,
  0.00900467,
  0.020689066,
  0.01398975,
  -0.005300342,
  0.013674488,
  -0.0048570046,
  -0.010318262,
  0.007008011,
  -0.013543129,
  -0.0014112899,
  0.0007659881,
  -0.015671147,
  0.0071393703,
  -0.044399396,
  0.002331625,
  0.012091611,
  0.0101015195,
  -0.009149165,
  0.0067715645,
  0.036386486,
  0.00033455534,
  0.017733486,
  -0.01816697,
  -0.012387169,
  -0.011388839,
  -0.019204708,
  -0.0092739565,
  0.011605581,
  -0.022252241,
  0.014541458,
  0.027059985,
  -0.00064694387,
  0.018350873,
  0.015986409,
  -0.012702431,
  -0.00535617,
  -0.016879652,
  -0.01498808,
  -0.013976614,
  0.008847039,
  -0.008085156,
  -0.012892901,
  0.021148823,
  0.0074349283,
  0.01829833,
  -0.0032051634,
  0.018731816,
  -0.015658012,
  -0.015868185,
  0.019756418,
  0.0027109245,
  -0.028032044,
  -0.010587548,
  0.0078749815,
  -0.013201595,
  -0.0015262292,
  0.007585991,
  0.013648217,
  -0.008814199,
  0.0052773543,
  -0.019533107,
  0.02910919,
  0.012912605,
  -0.012938877,
  -0.039670464,
  0.007566287,
  0.026350647,
  0.019598786,
  -0.007303569,
  0.0091360295,
  0.010666364,
  0.017378816,
  -0.0046107066,
  -0.015421565,
  0.0017503607,
  0.0031854594,
  0.0059932615,
  0.016091496,
  0.001255301,
  -0.0013029187,
  0.016025817,
  0.022633182,
  0.008952127,
  -0.0052018226,
  0.006374203,
  -0.012091611,
  0.0013981541,
  0.009582651,
  -0.015027488,
  -0.033785574,
  -0.006830676,
  0.01456773,
  0.012676159,
  0.0059932615,
  0.023697192,
  0.02151663,
  -0.01592073,
  0.022002658,
  -0.02280395,
  -0.011625285,
  -0.031788915,
  0.039223842,
  -0.008833903,
  0.022265377,
  0.028032044,
  -0.011809188,
  0.01807502,
  0.0016116126,
  -0.0062132883,
  -0.017838573,
  0.020570844,
  0.005733827,
  0.0075400155,
  -0.014239333,
  -0.013234435,
  -0.01376644,
  -0.029056646,
  0.0027536163,
  -0.0089783985,
  0.033470314,
  -0.025759531,
  0.05627426,
  0.0012273871,
  -0.01054814,
  0.026705317,
  0.018547913,
  0.013569401,
  0.010830563,
  -0.007947229,
  -0.041246776,
  -0.00068881456,
  -0.00010241909,
  -0.01109328,
  0.002343119,
  -0.013188459,
  -0.016695749,
  0.0044465074,
  -0.011901139,
  0.013503721,
  -0.018403418,
  -0.0015910878,
  0.007671375,
  0.0015221242,
  0.005280638,
  -0.002957223,
  -0.02682354,
  -0.023881095,
  0.019414883,
  -0.000051209547,
  -0.021753076,
  -0.016997876,
  -0.0013571043,
  0.015552924,
  -0.0223836,
  -0.011053873,
  0.01885004,
  -0.0306855,
  -0.0046369783,
  -0.0086828405,
  -0.007027715,
  0.01176978,
  0.015053759,
  0.013181891,
  -0.033181325,
  -0.015290205,
  0.023749735,
  0.014291876,
  -0.011592446,
  -0.000933471,
  0.004673102,
];
final _byeVector = [
  -0.0084679825,
  -0.018047791,
  0.0074078683,
  -0.03914664,
  -0.02015509,
  0.01239816,
  -0.014182254,
  -0.026761409,
  -0.017000604,
  -0.003285383,
  0.041241013,
  0.0033225517,
  0.0005603574,
  -0.0024498971,
  0.0044473065,
  -0.005316729,
  0.03451834,
  0.00036663614,
  0.03413049,
  -0.022624379,
  -0.013820264,
  0.010045225,
  -0.012566227,
  -0.019599177,
  -0.015462147,
  -0.0031900376,
  0.0016984445,
  -0.025041956,
  0.002327079,
  -0.025507372,
  0.01631541,
  0.008325771,
  -0.030562304,
  0.0021137635,
  -0.0049838275,
  -0.031338,
  -0.00298157,
  -0.005840322,
  0.02218482,
  -0.008461518,
  0.012805399,
  -0.0066451035,
  -0.0053555137,
  -0.001552194,
  -0.033742648,
  0.0031851893,
  0.014802808,
  0.00008080136,
  -0.015294081,
  0.030820869,
  0.02545566,
  0.007582399,
  -0.024486043,
  -0.025248807,
  -0.0029169288,
  0.021990897,
  -0.005552669,
  0.020168018,
  -0.013044571,
  -0.022352887,
  0.008552016,
  -0.011506113,
  -0.004473163,
  0.029734898,
  0.001489169,
  -0.0044376105,
  0.008829972,
  0.0027310858,
  -0.020193875,
  0.005972836,
  0.03226883,
  0.019754315,
  0.0063994676,
  -0.020206803,
  0.01993531,
  -0.01609563,
  -0.02399477,
  0.0065287496,
  -0.03508718,
  0.0011594994,
  0.005468636,
  -0.0375694,
  -0.032294687,
  0.017892651,
  0.022740733,
  -0.0055203484,
  -0.006690352,
  0.006147367,
  -0.015552645,
  -0.01965089,
  0.00500322,
  0.030795014,
  -0.015138942,
  0.016845467,
  0.003451834,
  0.01084031,
  -0.010976057,
  0.027433677,
  -0.0042598476,
  -0.017711658,
  -0.0017081407,
  0.0043244883,
  -0.0067743855,
  0.0030817636,
  -0.0012346447,
  -0.0036813098,
  0.0051777507,
  -0.019754315,
  0.03930178,
  -0.020749789,
  -0.019547464,
  0.047860257,
  -0.003936642,
  -0.03648343,
  -0.011519041,
  -0.011480257,
  -0.014350321,
  -0.008073672,
  -0.011150587,
  -0.020840285,
  0.02045244,
  0.015100157,
  0.022391671,
  -0.031441424,
  0.007970246,
  0.0070394143,
  -0.0012669653,
  -0.02261145,
  0.02131863,
  0.002567867,
  0.0343632,
  0.0033839608,
  0.022016753,
  -0.012249486,
  -0.010452463,
  0.010704564,
  -0.024046483,
  0.0074078683,
  -0.010349038,
  -0.022598524,
  0.0077181454,
  0.034544196,
  -0.0023545516,
  -0.01192628,
  0.006506125,
  0.00714284,
  0.020129234,
  0.007097591,
  -0.0013849352,
  0.007705217,
  0.006496429,
  -0.0027246217,
  0.023257863,
  0.00014816949,
  0.01565607,
  0.004696175,
  -0.010142186,
  -0.0046121413,
  -0.021693548,
  0.010400751,
  0.009101464,
  -0.0015562341,
  0.03379436,
  -0.021525482,
  0.0024062644,
  0.029890038,
  0.016418835,
  0.02399477,
  0.011887495,
  -0.010711027,
  -0.0010649619,
  0.020491224,
  -0.03878465,
  0.0047931364,
  -0.0056787194,
  0.028959205,
  0.0008597264,
  0.014350321,
  -0.022314101,
  -0.023038082,
  -0.019107904,
  0.026334777,
  0.020840285,
  0.028674785,
  -0.0015117934,
  0.020620506,
  0.019586248,
  0.010232683,
  0.0024773695,
  -0.020620506,
  0.006509357,
  0.021279845,
  0.021564266,
  -0.019159617,
  -0.66895765,
  -0.012230094,
  0.0019182242,
  -0.001672588,
  -0.00018907517,
  0.00431156,
  0.016470548,
  0.0088041155,
  -0.023555212,
  0.0130833555,
  -0.0003403757,
  0.007382012,
  0.0032966954,
  -0.015759496,
  -0.0021105313,
  0.010206827,
  0.0051034135,
  -0.012462801,
  -0.0004932923,
  0.015862921,
  -0.010601138,
  0.025688367,
  0.0047349595,
  0.020775644,
  0.0008043775,
  -0.012630868,
  0.0022543578,
  0.007873284,
  -0.015849994,
  -0.00021331558,
  -0.020413654,
  0.008306379,
  0.0049030264,
  -0.013910761,
  0.050859604,
  0.014363249,
  -0.0015319937,
  0.013678053,
  0.0032433665,
  0.052850552,
  0.006625711,
  -0.050083913,
  -0.009760804,
  0.006638639,
  -0.0014019036,
  0.0031173164,
  0.007330299,
  0.0055106524,
  0.004537804,
  -0.012882968,
  0.012689045,
  -0.00074579654,
  0.004957971,
  0.007937925,
  0.002991266,
  0.001612795,
  0.030717444,
  -0.01261794,
  0.01478988,
  0.018577848,
  0.013548771,
  0.0070588063,
  -0.009734947,
  -0.0008500303,
  -0.006373611,
  0.009179034,
  -0.0065966225,
  -0.0059146592,
  0.015565573,
  -0.010924343,
  -0.010484784,
  0.025636654,
  -0.022094322,
  0.0012815095,
  0.015759496,
  0.028623072,
  0.019418182,
  -0.0048674736,
  -0.013871977,
  0.009851301,
  0.020349013,
  -0.00024604012,
  -0.014880378,
  -0.011538434,
  0.036095582,
  -0.018358069,
  -0.02820937,
  0.0014972491,
  -0.006483501,
  -0.018306356,
  0.003396889,
  0.024395546,
  -0.006916596,
  -0.01892691,
  0.005025844,
  -0.0050452366,
  -0.010336109,
  0.011131194,
  0.004360041,
  -0.012630868,
  -0.016832538,
  -0.0087653315,
  0.017414307,
  -0.010161579,
  -0.007453117,
  -0.0032045818,
  -0.020478295,
  0.018215857,
  0.027537102,
  -0.0123076625,
  0.017931437,
  0.00057368964,
  0.0063348264,
  0.008817044,
  0.025843505,
  -0.029372908,
  -0.0019473126,
  0.0018551991,
  0.026166711,
  -0.031493135,
  0.006441484,
  -0.009217818,
  0.013949546,
  -0.009237211,
  0.012411088,
  0.029372908,
  0.0039948192,
  -0.024951458,
  -0.039560344,
  0.00262766,
  -0.0032886153,
  0.0006258065,
  0.027252682,
  0.0028215833,
  -0.00031936736,
  -0.0057272003,
  0.00391725,
  -0.01478988,
  0.00523916,
  0.016199056,
  -0.005126038,
  -0.00045450762,
  0.002931473,
  0.007692289,
  0.0064673405,
  -0.010633458,
  -0.015281152,
  -0.013432417,
  -0.01936647,
  0.021512553,
  0.00953456,
  -0.0092630675,
  0.010284397,
  0.029476335,
  0.019043263,
  -0.020969568,
  -0.022107251,
  -0.014880378,
  -0.03263082,
  -0.006554606,
  0.020336086,
  0.036948845,
  -0.022870015,
  0.016354194,
  -0.008920469,
  -0.03131214,
  0.01084031,
  0.028623072,
  -0.003636061,
  -0.03524232,
  0.0041531897,
  -0.030252028,
  0.01287004,
  0.01181639,
  0.018383924,
  -0.01062053,
  0.0036748457,
  0.00068761955,
  -0.008112456,
  -0.0044957874,
  0.0002446261,
  -0.0008459902,
  -0.00881058,
  -0.006625711,
  0.023400072,
  0.001670972,
  0.0045119477,
  0.021887472,
  -0.02102128,
  0.019263044,
  0.0014423042,
  0.028648928,
  -0.01776337,
  -0.0019796332,
  0.0055365087,
  0.00020664946,
  0.008642513,
  0.011118267,
  0.017634088,
  0.010484784,
  0.024524827,
  -0.013180317,
  0.027976662,
  0.004906258,
  0.004647694,
  -0.029450478,
  0.009831909,
  -0.030381309,
  0.003393657,
  0.006305738,
  -0.013070427,
  -0.016069774,
  -0.017556518,
  0.011952137,
  0.015061373,
  0.02646406,
  -0.009657378,
  0.0026939171,
  -0.033820216,
  -0.013445346,
  0.0024127285,
  -0.011861639,
  0.0017695497,
  -0.0001308982,
  0.001552194,
  0.012999322,
  0.014751095,
  0.02015509,
  0.010943736,
  -0.017608231,
  -0.020555865,
  0.0046153734,
  -0.00027310857,
  0.0010205212,
  -0.009166106,
  -0.018138288,
  0.025145382,
  -0.014156397,
  0.037905533,
  -0.037802108,
  -0.00022341574,
  0.024977315,
  0.03387193,
  -0.02443433,
  -0.011221692,
  0.0047446555,
  0.009890086,
  -0.007511294,
  0.013794407,
  0.020142162,
  -0.020129234,
  0.013652197,
  -0.0213962,
  0.00689074,
  0.005294105,
  -0.01688425,
  -0.017104032,
  0.026011573,
  0.031570707,
  0.028235225,
  0.007931461,
  0.0038655368,
  -0.0006855995,
  0.013484131,
  0.0013146381,
  -0.0065352134,
  0.0026292761,
  0.007931461,
  -0.009922407,
  -0.0038461445,
  -0.031338,
  -0.022249462,
  0.013962475,
  -0.012889432,
  0.030950151,
  0.020982496,
  0.01762116,
  0.00049975637,
  -0.02756296,
  -0.016625687,
  -0.034621764,
  -0.021137634,
  0.02204261,
  0.015643142,
  -0.0037782714,
  -0.027252682,
  -0.023451785,
  0.01058821,
  -0.006638639,
  0.011577218,
  -0.007828035,
  0.03407878,
  -0.007369084,
  -0.0058694105,
  -0.01790558,
  0.002739166,
  0.022546811,
  0.005911427,
  0.00060075807,
  0.02508074,
  -0.014156397,
  -0.009127321,
  0.0054427795,
  -0.027743954,
  0.010853238,
  0.0032401343,
  -0.0048771696,
  -0.026864834,
  -0.023193222,
  0.004182278,
  -0.022740733,
  -0.005436315,
  -0.019314757,
  -0.015914636,
  0.0071880887,
  0.010303789,
  -0.018681273,
  -0.013031643,
  0.029269483,
  -0.016302481,
  -0.018823484,
  -0.002210725,
  0.0033613364,
  -0.011525505,
  0.070432924,
  0.039172497,
  0.008099528,
  0.021370342,
  0.0078345,
  0.025041956,
  -0.013445346,
  -0.023917202,
  0.008713618,
  0.002031346,
  -0.0026712928,
  -0.010174506,
  -0.012223629,
  0.0038946252,
  0.021939185,
  0.0015287617,
  0.000108071814,
  -0.0056819515,
  0.008875221,
  -0.00061328226,
  -0.001976401,
  -0.011784069,
  -0.0015764345,
  0.039715484,
  0.011984457,
  0.020025808,
  -0.0018147985,
  0.036276575,
  0.007162232,
  -0.006690352,
  0.0058144657,
  0.000418147,
  0.014233966,
  0.002323847,
  -0.017013533,
  0.02261145,
  0.008073672,
  -0.011738821,
  0.0013017098,
  -0.022314101,
  0.013018714,
  0.019948239,
  -0.002275366,
  -0.02536516,
  0.021525482,
  -0.02704583,
  -0.008739474,
  0.0148933055,
  -0.0062863454,
  -0.014402034,
  0.033897784,
  0.009605665,
  0.00026482643,
  -0.010568817,
  0.0020992192,
  0.0071105193,
  -0.018577848,
  -0.012436945,
  0.00964445,
  -0.0021961809,
  -0.017491877,
  -0.02566251,
  -0.000070196176,
  -0.0068390267,
  -0.0020555865,
  -0.028881637,
  -0.021628907,
  -0.011260477,
  -0.02589522,
  0.0070070936,
  0.0051034135,
  -0.012314127,
  -0.012721365,
  0.0008815428,
  -0.00063509867,
  0.020517081,
  0.0013307984,
  -0.011247548,
  0.011603075,
  0.012837719,
  -0.009159641,
  -0.015578501,
  -0.010930807,
  -0.009961192,
  -0.0010261772,
  0.009075608,
  -0.003985123,
  -0.00536521,
  -0.022197748,
  0.019301828,
  0.030252028,
  0.0057498245,
  0.011221692,
  -0.011784069,
  0.009870694,
  -0.012062026,
  0.0087653315,
  0.007905604,
  -0.018047791,
  0.008377485,
  0.014389105,
  -0.010917879,
  -0.0077246097,
  -0.012333519,
  0.017142816,
  0.015358722,
  0.02204261,
  0.041241013,
  -0.004948275,
  -0.010601138,
  0.026632126,
  -0.0379831,
  0.0022802143,
  0.0042727757,
  -0.00069287163,
  0.016535189,
  0.008920469,
  0.04312853,
  -0.009140249,
  -0.013484131,
  0.0049288827,
  -0.05822869,
  0.02378792,
  0.00045127558,
  0.0079896385,
  0.006021317,
  0.004954739,
  -0.022818303,
  -0.0146347415,
  -0.0034712262,
  -0.009987048,
  0.013975402,
  -0.0034679943,
  -0.040465318,
  -0.04589517,
  -0.020038737,
  -0.020258516,
  0.023232006,
  -0.018797627,
  -0.016005132,
  -0.023645708,
  0.0077698585,
  0.001608755,
  -0.01464767,
  -0.019107904,
  -0.054195084,
  -0.0069553806,
  0.014402034,
  -0.004534572,
  0.0072074807,
  -0.0151130855,
  0.0037136304,
  -0.010898487,
  -0.008299915,
  0.015035517,
  -0.022857087,
  -0.011796998,
  0.0007021638,
  0.018965695,
  0.034544196,
  0.03154485,
  -0.0011659636,
  0.015849994,
  -0.000030628762,
  0.0035584916,
  0.008371021,
  -0.008222346,
  0.009637985,
  0.00451518,
  0.0049159545,
  0.015992204,
  0.014091756,
  0.0014406882,
  -0.002325463,
  0.004010979,
  0.02204261,
  -0.008442125,
  -0.007963781,
  -0.017440164,
  -0.017491877,
  -0.0065255174,
  -0.019482823,
  -0.0049385787,
  0.0027197737,
  0.00067711534,
  0.00016927884,
  0.027795667,
  0.012333519,
  -0.0021590122,
  -0.016573975,
  0.030846726,
  -0.011557826,
  0.015513861,
  -0.0067162085,
  -0.010575281,
  0.0004718799,
  -0.0057465923,
  0.015281152,
  -0.0068842755,
  0.006877811,
  -0.0011481872,
  0.01378148,
  -0.01891398,
  -0.014130541,
  0.0009292156,
  0.012624403,
  -0.010749812,
  -0.027950805,
  0.018009007,
  -0.0085908,
  -0.0017921741,
  -0.034337346,
  -0.025326377,
  -0.014970875,
  -0.0047963685,
  0.002322231,
  -0.011628931,
  0.009372957,
  -0.0085908,
  -0.0023610156,
  -0.017996078,
  -0.00044279144,
  0.020116305,
  -0.016496405,
  0.032941096,
  0.014583029,
  -0.006137671,
  -0.011460864,
  0.008060743,
  -0.009870694,
  -0.0030656033,
  0.011667715,
  0.005294105,
  0.004692943,
  0.006496429,
  -0.013639268,
  0.014389105,
  -0.016496405,
  -0.017892651,
  0.004414986,
  0.019508678,
  -0.011906887,
  -0.012359375,
  -0.011363903,
  -0.015707783,
  0.01603099,
  -0.011447936,
  0.008868757,
  -0.018500278,
  -0.011803462,
  -0.010381358,
  0.042663116,
  -0.034906186,
  0.02204261,
  0.011331582,
  -0.020775644,
  -0.016354194,
  0.0024094963,
  -0.01762116,
  0.004298632,
  -0.014660598,
  0.033251375,
  -0.021499624,
  0.0069359886,
  0.01428568,
  -0.0065966225,
  -0.022831231,
  -0.009036823,
  0.0027763345,
  0.025753008,
  -0.017802155,
  -0.004395594,
  0.002550091,
  -0.0071234475,
  0.0050937175,
  -0.008377485,
  -0.009612129,
  -0.03110529,
  -0.014802808,
  0.0011304109,
  0.0045539644,
  0.012786007,
  -0.015384578,
  -0.026257208,
  -0.01217838,
  -0.026321849,
  -0.0128247915,
  0.013639268,
  0.00067024725,
  -0.010646387,
  -0.007912069,
  -0.010323181,
  -0.00030219706,
  0.03167413,
  0.0012596932,
  -0.021990897,
  0.00511311,
  0.012430481,
  -0.012992858,
  0.02094371,
  -0.010329645,
  0.000017927801,
  -0.022301175,
  -0.013458273,
  -0.014414961,
  -0.021939185,
  0.015630214,
  0.00017968201,
  -0.019948239,
  0.0029928822,
  -0.005420155,
  0.015617286,
  0.020801501,
  -0.005662559,
  0.010659315,
  0.019999951,
  0.0006625711,
  -0.0054169227,
  -0.030614018,
  0.007828035,
  -0.0041241013,
  0.015772425,
  -0.0010067848,
  -0.013419489,
  0.007265658,
  0.0032546786,
  0.009385886,
  -0.011635395,
  -0.024046483,
  0.00005070285,
  -0.0035358672,
  0.012514514,
  -0.009127321,
  -0.030872582,
  -0.009444063,
  -0.026451131,
  -0.0059793005,
  0.03873294,
  0.018901054,
  -0.016276626,
  0.0018875196,
  0.018435637,
  -0.0088429,
  0.021939185,
  0.000892855,
  -0.011609538,
  0.01250805,
  0.0025145381,
  -0.009269532,
  0.0028797602,
  0.008836436,
  0.020219732,
  0.00703295,
  -0.024033556,
  -0.0059889965,
  0.0087847235,
  -0.012424016,
  0.003390425,
  -0.0059534437,
  0.024317976,
  0.00786682,
  0.0068002422,
  0.0066515678,
  0.03467348,
  -0.0021008353,
  0.012611476,
  -0.019883597,
  -0.008157705,
  0.0038558408,
  0.024951458,
  0.029760756,
  0.003458298,
  -0.027252682,
  -0.015811209,
  0.015345793,
  -0.0055946857,
  -0.015229439,
  0.008868757,
  -0.00036017204,
  -0.009987048,
  -0.016328339,
  0.006147367,
  -0.01688425,
  0.007737538,
  0.028157657,
  -0.02523588,
  0.0176858,
  0.011544897,
  -0.0123076625,
  0.0020507383,
  0.0018923677,
  0.017789226,
  -0.0055203484,
  0.0012281806,
  -0.0056787194,
  0.0068842755,
  0.007950854,
  -0.035500884,
  -0.003881697,
  0.01515187,
  -0.019586248,
  0.046024453,
  -0.016703255,
  -0.015164798,
  -0.012605011,
  -0.0072979783,
  -0.011098874,
  -0.0050096842,
  0.020077521,
  -0.01087263,
  -0.0059857643,
  -0.02261145,
  -0.012074955,
  -0.047756832,
  0.0020151858,
  -0.03363922,
  -0.017375523,
  -0.020387799,
  0.004754352,
  0.017104032,
  -0.015578501,
  -0.011350974,
  0.0030833797,
  -0.006221704,
  -0.011990922,
  -0.009082072,
  0.019314757,
  -0.012249486,
  -0.015811209,
  0.014324464,
  -0.0057659848,
  -0.016729113,
  0.0110342335,
  0.010762741,
  0.0021687085,
  0.010284397,
  0.21843515,
  -0.012598547,
  0.002550091,
  0.041034162,
  0.009198426,
  0.019172546,
  -0.0024062644,
  0.010445999,
  0.011325118,
  0.024524827,
  -0.018164145,
  0.01740138,
  -0.015604358,
  -0.003691006,
  0.019663818,
  -0.007582399,
  -0.06412396,
  -0.023826703,
  -0.029553903,
  -0.014014187,
  -0.0063962354,
  -0.011881031,
  -0.0075500784,
  -0.004457003,
  0.03834509,
  0.007511294,
  -0.0047608158,
  0.0009195194,
  0.0028700642,
  0.0022365814,
  -0.024111124,
  -0.04661915,
  0.002202645,
  0.021874543,
  -0.015707783,
  0.0013202941,
  0.0148933055,
  -0.0054039946,
  0.042688973,
  0.024783392,
  -0.0006411588,
  0.00052520883,
  -0.012734294,
  -0.0123270545,
  -0.020801501,
  0.010439535,
  0.008138313,
  -0.031027721,
  0.015604358,
  0.0058015375,
  -0.04212013,
  0.014686454,
  0.021615978,
  0.052178282,
  0.0030963079,
  0.016405907,
  0.0070394143,
  0.0117065,
  -0.023322504,
  -0.0071880887,
  -0.0062895776,
  0.024524827,
  -0.017750442,
  0.01675497,
  -0.02435676,
  0.007886212,
  -0.014699383,
  0.005135734,
  0.021564266,
  -0.017789226,
  -0.0010698099,
  -0.019870669,
  0.0053555137,
  0.0070523424,
  -0.010581746,
  -0.022908801,
  0.019883597,
  0.010892022,
  0.011758213,
  0.005782145,
  -0.0053813704,
  0.0040174434,
  -0.011363903,
  -0.012546835,
  -0.010575281,
  -0.04183571,
  0.031415567,
  0.017013533,
  0.01250805,
  -0.007899141,
  0.012055562,
  -0.040930733,
  -0.010685171,
  -0.0027924948,
  0.009954727,
  0.011945672,
  0.017310882,
  0.025675438,
  -0.030096889,
  0.02784738,
  -0.00989655,
  -0.027976662,
  0.00549126,
  0.015164798,
  -0.02212018,
  -0.009127321,
  0.0044957874,
  0.03154485,
  0.0021800206,
  -0.019637961,
  0.0046864785,
  -0.022301175,
  0.00786682,
  0.0018051023,
  0.043800797,
  0.0071880887,
  0.008306379,
  0.0020523544,
  -0.00024583814,
  -0.018603705,
  0.024020627,
  0.0026712928,
  0.012482193,
  0.006244329,
  -0.0074272607,
  0.0011215229,
  -0.021577194,
  -0.008293451,
  0.021512553,
  -0.037310835,
  0.01370391,
  -0.0024256567,
  0.02298637,
  -0.013962475,
  -0.009204891,
  -0.008532623,
  0.0047414238,
  -0.010646387,
  -0.014040044,
  0.025197094,
  -0.013206174,
  -0.00022482977,
  0.009075608,
  0.00076114875,
  -0.0060342452,
  -0.031777557,
  0.0146347415,
  -0.007640576,
  -0.024990244,
  -0.010284397,
  -0.023232006,
  -0.009689699,
  -0.01965089,
  0.011842247,
  0.0271234,
  -0.001018097,
  -0.022947585,
  -0.030174458,
  -0.00346153,
  -0.004350345,
  -0.026295993,
  -0.01696182,
  0.021292774,
  -0.002685837,
  -0.04095659,
  -0.016199056,
  -0.16630858,
  0.046826,
  0.022107251,
  -0.013380704,
  0.0271234,
  0.006916596,
  0.033587508,
  0.012223629,
  0.002503226,
  -0.009553952,
  0.027873235,
  0.008461518,
  -0.024305047,
  -0.0005595494,
  0.000106758795,
  0.010601138,
  -0.0037039341,
  0.008552016,
  0.021874543,
  0.013497058,
  0.02218482,
  -0.032217115,
  0.018177073,
  -0.005956676,
  0.016573975,
  0.00084356614,
  0.017672872,
  0.0098189805,
  -0.011596611,
  -0.0074725095,
  -0.031208716,
  -0.005313497,
  0.03653514,
  0.02009045,
  0.0077181454,
  0.0039948192,
  0.008358092,
  -0.014272751,
  0.00848091,
  0.011719429,
  0.025171239,
  0.022469241,
  -0.00689074,
  0.011195836,
  -0.015255296,
  0.007873284,
  0.006066566,
  0.019922383,
  0.007466045,
  -0.009075608,
  0.009237211,
  -0.020749789,
  -0.0021751726,
  -0.0067420653,
  -0.0007490286,
  0.00514543,
  -0.015565573,
  0.0049224184,
  -0.0021477,
  -0.008661905,
  -0.012676117,
  -0.017297953,
  0.0021250756,
  0.010943736,
  -0.0032611426,
  -0.012184844,
  -0.006441484,
  0.014466675,
  -0.026528701,
  0.00986423,
  0.014479603,
  -0.025584942,
  0.011796998,
  -0.015617286,
  0.024124052,
  0.020504152,
  -0.019547464,
  0.02212018,
  0.0012023242,
  0.0032643748,
  -0.025701296,
  0.047653407,
  -0.03283767,
  0.0071816244,
  0.022714878,
  0.008610193,
  -0.0027052294,
  -0.00942467,
  -0.021861615,
  -0.03211369,
  0.020633435,
  -0.024951458,
  -0.014324464,
  -0.014751095,
  0.015190654,
  0.017375523,
  0.004056228,
  0.014182254,
  -0.0051551266,
  -0.002621196,
  0.008358092,
  -0.00811892,
  -0.029786611,
  -0.00030866117,
  0.03873294,
  -0.024563612,
  0.0054169227,
  0.020103376,
  0.026063286,
  -0.018797627,
  -0.022714878,
  0.00084841426,
  0.021085922,
  0.01944404,
  -0.008306379,
  0.015061373,
  -0.0072204093,
  -0.013264351,
  0.018435637,
  -0.017569447,
  0.040982448,
  -0.0047317273,
  -0.016263697,
  -0.0036651494,
  -0.0065869265,
  -0.012488658,
  -0.073432274,
  -0.033535793,
  0.0075371503,
  0.016367123,
  -0.024744608,
  -0.0037168623,
  -0.0039043215,
  0.017453093,
  -0.009411742,
  0.011208764,
  -0.005074325,
  -0.021939185,
  -0.026037429,
  -0.015061373,
  0.020904927,
  -0.010956664,
  -0.015488003,
  -0.03573359,
  -0.008758867,
  0.0112734055,
  0.010601138,
  -0.012579155,
  -0.009321244,
  -0.00047026388,
  -0.010413678,
  -0.008209418,
  -0.015785353,
  0.03283767,
  0.0151130855,
  -0.0067291367,
  0.0027116935,
  -0.021047138,
  0.022081394,
  -0.0197931,
  -0.0029444015,
  -0.013742695,
  -0.03943106,
  0.009625058,
  0.020219732,
  -0.01870713,
  0.008720082,
  0.010032296,
  -0.001247573,
  -0.040646315,
  0.007621184,
  -0.006189384,
  -0.021163492,
  0.012120203,
  0.00689074,
  -0.026140854,
  -0.028235225,
  -0.021952113,
  -0.032346398,
  -0.02081443,
  0.0405946,
  0.0017420772,
  0.0022963744,
  0.016173199,
  -0.022469241,
  0.017440164,
  0.009541024,
  0.0014043276,
  -0.01943111,
  0.030769156,
  0.009592737,
  -0.004870706,
  -0.010168043,
  0.008752403,
  0.002676141,
  -0.0014835129,
  0.00689074,
  0.031622417,
  0.011318654,
  0.010400751,
  -0.014001259,
  0.016263697,
  -0.0065416778,
  -0.024964387,
  0.01265026,
  -0.024602396,
  -0.009586273,
  -0.017789226,
  0.010458928,
  -0.008170633,
  0.014996732,
  0.0047349595,
  0.026942404,
  0.003522939,
  0.013471202,
  -0.023244934,
  0.009838373,
  0.013206174,
  0.015035517,
  -0.023697421,
  0.0038849292,
  0.004828689,
  -0.017802155,
  -0.0061182785,
  0.013031643,
  0.008429198,
  -0.008409805,
  -0.017750442,
  -0.045662463,
  0.022895873,
  -0.0148933055,
  0.007886212,
  0.023955986,
  -0.013820264,
  0.0076535046,
  -0.0056076143,
  -0.01675497,
  -0.00071024394,
  -0.00089770305,
  -0.008959254,
  -0.009793124,
  -0.008552016,
  -0.037181552,
  -0.021215204,
  0.035785303,
  0.017659944,
  0.028907493,
  0.018939838,
  0.012721365,
  0.0015392659,
  0.012954073,
  0.002858752,
  -0.020271445,
  -0.0024563612,
  -0.009915942,
  0.018112432,
  0.000893663,
  -0.0474207,
  0.014389105,
  -0.04387837,
  0.0040788525,
  0.022469241,
  0.012029706,
  -0.0064770365,
  -0.0069812373,
  0.0146347415,
  0.015966348,
  0.0052747126,
  -0.0213962,
  -0.020038737,
  0.0031771094,
  -0.015358722,
  -0.005361978,
  0.0015618902,
  -0.014751095,
  0.016108558,
  0.0012225246,
  0.010387822,
  -0.0009623442,
  0.023568138,
  0.005232696,
  -0.0066709598,
  -0.008907542,
  -0.0032676067,
  0.00645118,
  0.0016281473,
  -0.010801525,
  -0.016354194,
  0.02196504,
  0.030950151,
  0.01573364,
  0.011628931,
  0.0019505448,
  0.0055688294,
  -0.025132453,
  0.002150932,
  0.0062152403,
  -0.0385778,
  -0.012773078,
  0.0057465923,
  0.010736885,
  0.00008569994,
  0.01515187,
  0.00608919,
  -0.014492531,
  0.007149304,
  -0.039353494,
  0.0166774,
  0.02268902,
  -0.020555865,
  -0.027821522,
  0.007369084,
  0.032966953,
  0.028105944,
  0.003749183,
  0.0075500784,
  0.0017113728,
  0.0141176125,
  -0.031622417,
  -0.002317383,
  0.030691586,
  0.0098189805,
  -0.005290873,
  0.017918508,
  -0.0020151858,
  -0.000026942202,
  0.010614066,
  0.024537755,
  -0.005982532,
  -0.0056528626,
  -0.018396853,
  -0.020633435,
  0.0019618568,
  0.026373563,
  -0.008655441,
  -0.031027721,
  0.0059922286,
  0.01109241,
  0.011700036,
  0.016173199,
  0.022701949,
  0.029347053,
  -0.0040432997,
  0.015475076,
  -0.011616003,
  -0.017828012,
  -0.020672219,
  0.054919064,
  0.0060536377,
  0.006961845,
  0.023038082,
  0.00068438746,
  0.027019974,
  -0.0010641539,
  -0.01862956,
  -0.025688367,
  0.0102973245,
  -0.015669,
  0.021124706,
  -0.02610207,
  -0.0171816,
  -0.031932697,
  -0.010904951,
  -0.006570766,
  0.0062314006,
  0.037672825,
  -0.032889385,
  0.050782036,
  0.013497058,
  -0.014583029,
  0.0016297633,
  -0.0020588185,
  0.02893335,
  -0.0018341908,
  0.0010673859,
  -0.024188694,
  -0.015488003,
  0.0187847,
  -0.012798935,
  -0.0029411693,
  -0.021590123,
  -0.011997385,
  -0.021809902,
  -0.019107904,
  0.010820918,
  -0.0031092362,
  0.023464713,
  0.021447912,
  0.015669,
  0.024589468,
  0.008926934,
  -0.022443384,
  0.010665779,
  0.015992204,
  -0.0046800147,
  -0.0044279145,
  -0.0562636,
  0.007750466,
  0.007575935,
  -0.010704564,
  -0.0013744311,
  0.028519647,
  -0.017297953,
  -0.016367123,
  -0.012520978,
  -0.010245612,
  0.01652226,
  0.015979277,
  -0.0067162085,
  -0.025507372,
  -0.03488033,
  0.031208716,
  0.0074337246,
  -0.026218424,
  0.007129912,
  -0.025584942,
];
final _whatsThisVector = [
  -0.01297786,
  -0.0028681739,
  0.010790355,
  0.008147955,
  -0.020644162,
  -0.00613438,
  -0.02576841,
  -0.014275646,
  -0.015881155,
  -0.022022223,
  0.02437697,
  0.0021824879,
  -0.028096396,
  -0.008562711,
  0.009077812,
  0.01051608,
  0.027159851,
  -0.005559073,
  0.0036324628,
  -0.03293968,
  0.001190752,
  0.00011487329,
  0.0020319715,
  -0.0049503176,
  0.009766843,
  0.0067230663,
  0.007378649,
  -0.02559448,
  -0.0100009795,
  -0.011151594,
  0.021594089,
  -0.018316176,
  0.0003518321,
  -0.02049699,
  -0.01733949,
  -0.011512833,
  -0.0030972932,
  -0.03031735,
  0.008923951,
  0.0058667953,
  0.0127905505,
  0.010649872,
  -0.012061382,
  -0.010288633,
  -0.012262071,
  -0.0077532674,
  0.011084697,
  -0.03176231,
  -0.009546085,
  0.009619671,
  0.0088169165,
  0.0051275925,
  -0.030370867,
  -0.016041705,
  -0.010676631,
  -0.004752974,
  -0.015399502,
  0.01052277,
  0.0014483024,
  -0.00047621722,
  -0.0026607956,
  0.0113857305,
  -0.021861672,
  0.0004212369,
  0.020992022,
  0.0016046722,
  -0.004696112,
  0.00074756483,
  -0.00929188,
  -0.0032578441,
  0.02295877,
  0.038960338,
  0.00285145,
  -0.0001723622,
  0.020028718,
  -0.007699751,
  -0.032511543,
  -0.017620455,
  0.011559661,
  -0.0012367432,
  0.03138769,
  -0.017834522,
  -0.012603241,
  0.020403337,
  0.00011957693,
  -0.009385535,
  -0.008823606,
  0.029568113,
  -0.018958379,
  -0.016938115,
  0.001041908,
  -0.0065457914,
  0.013011307,
  0.015881155,
  -0.012422621,
  -0.0010302012,
  0.013807372,
  0.035695802,
  -0.0014683713,
  -0.011245249,
  -0.024992414,
  0.012817308,
  0.0014416127,
  -0.011305455,
  -0.015827637,
  -0.011833935,
  -0.0071244435,
  -0.0022042291,
  0.017138802,
  0.0108371815,
  -0.01210152,
  0.027373917,
  0.012757102,
  -0.058172923,
  0.021861672,
  -0.013874268,
  0.027066195,
  -0.00007494463,
  -0.0021941948,
  -0.024136143,
  0.019841408,
  0.035695802,
  0.025741652,
  -0.013051446,
  0.02191519,
  0.010021049,
  -0.014677023,
  -0.0028414153,
  -0.00648893,
  -0.01909217,
  0.039415233,
  -0.0046860776,
  0.011619867,
  -0.018704172,
  -0.0020771264,
  0.007151202,
  -0.01156635,
  0.01228214,
  -0.014275646,
  -0.0043683206,
  0.0151185375,
  0.031039828,
  -0.03401002,
  -0.018329553,
  -0.008729951,
  0.02017589,
  0.017673971,
  0.0058634505,
  0.00079564645,
  -0.009398914,
  0.014717161,
  -0.0012869153,
  0.00622469,
  0.009505948,
  -0.002963501,
  0.026062753,
  -0.005733003,
  0.00042416362,
  -0.009907325,
  0.0112118,
  -0.026062753,
  0.01593467,
  0.0073920283,
  -0.008107818,
  0.011158284,
  0.03326078,
  0.0025688135,
  0.0040572537,
  0.002083816,
  -0.013626752,
  -0.0092785,
  0.0056828307,
  -0.034839533,
  0.020737818,
  -0.0036424971,
  0.007231477,
  -0.00822823,
  0.00063049654,
  -0.02540717,
  -0.008174714,
  -0.013178548,
  0.01875769,
  0.014703781,
  0.017781006,
  -0.0057731406,
  -0.010663251,
  0.011532902,
  0.008254989,
  0.012656758,
  0.007077616,
  0.018490106,
  0.017446525,
  -0.0069170655,
  0.0026758474,
  -0.67345726,
  -0.015279088,
  -0.0025270034,
  0.0014140181,
  0.00822823,
  0.0031156898,
  0.0073385113,
  -0.0012200192,
  -0.016790941,
  0.0059169675,
  0.030130042,
  0.0030370867,
  0.0011021146,
  -0.0032929645,
  -0.007646234,
  -0.019145688,
  0.0010920801,
  -0.016844459,
  -0.0039870124,
  0.0056293136,
  -0.027909087,
  0.006221345,
  -0.0053884876,
  0.0041642874,
  0.034491673,
  0.014998124,
  0.012783861,
  -0.0037294622,
  -0.022383463,
  0.004485389,
  -0.057423685,
  0.010007669,
  0.017419767,
  -0.0058935536,
  0.049369384,
  -0.005579142,
  -0.014302404,
  0.02788233,
  -0.00040534907,
  0.001988489,
  -0.013392616,
  -0.010984354,
  0.039789848,
  -0.0033799296,
  -0.0016749132,
  -0.0032177065,
  0.014289024,
  0.01875769,
  0.035321184,
  -0.020108992,
  0.004873387,
  -0.004013771,
  0.0023748146,
  0.029648388,
  0.017085286,
  0.0024667967,
  0.030130042,
  0.0068836175,
  0.013004618,
  0.020818092,
  -0.026209924,
  0.025367033,
  -0.012436001,
  -0.01033546,
  -0.0056192796,
  0.014248887,
  -0.029487837,
  0.01629591,
  0.02365449,
  -0.022370083,
  0.022022223,
  0.029514596,
  0.0004707819,
  0.0036358077,
  0.011947658,
  0.0364718,
  0.002831381,
  -0.0022627634,
  -0.0030989656,
  0.007311753,
  0.0047730426,
  0.0026373821,
  -0.015011503,
  -0.00893064,
  0.03676614,
  -0.021808155,
  -0.030023007,
  0.010388977,
  0.007325132,
  0.016041705,
  -0.0057029,
  -0.0057029,
  -0.020590644,
  -0.0034886359,
  0.0048265597,
  0.033421334,
  0.0016866201,
  0.0039535644,
  0.01281062,
  0.02051037,
  -0.010964285,
  0.008000784,
  0.019306239,
  -0.0013028032,
  0.015988188,
  0.007612786,
  -0.025982477,
  0.020309681,
  0.030156799,
  -0.052553643,
  0.010435805,
  -0.00024291677,
  -0.01754018,
  -0.007311753,
  -0.028123155,
  -0.031227138,
  0.020256164,
  0.014489713,
  0.019078791,
  -0.02507269,
  0.017285973,
  -0.00640531,
  -0.0036424971,
  -0.007345201,
  -0.012081451,
  0.020403337,
  -0.0037829792,
  -0.015867775,
  -0.046907604,
  -0.012415932,
  0.010041117,
  0.0021674363,
  0.040378537,
  -0.018878102,
  0.002087161,
  0.010944216,
  0.016402945,
  0.01594805,
  0.0064788954,
  0.0033331024,
  -0.029942732,
  0.00420108,
  0.01628253,
  -0.015265709,
  -0.021540571,
  -0.018904861,
  0.009679878,
  -0.01245607,
  -0.011445937,
  -0.0026474164,
  -0.012221932,
  -0.008703194,
  -0.02349394,
  0.021607468,
  0.00026674854,
  -0.016349427,
  -0.0014976383,
  -0.030023007,
  -0.0057463823,
  -0.029675147,
  -0.018597139,
  0.017647212,
  0.010308702,
  -0.01069001,
  -0.016857838,
  -0.031628516,
  -0.01576074,
  0.013245445,
  -0.0148108145,
  -0.03577608,
  0.0114726955,
  -0.0007283322,
  -0.027320402,
  0.022182774,
  0.0055624177,
  0.0141953705,
  -0.0017643869,
  0.0016606977,
  -0.02368125,
  -0.019761132,
  -0.01664377,
  0.003662566,
  -0.0036692556,
  -0.005910278,
  0.0025520893,
  0.002384849,
  0.018142246,
  -0.004712836,
  -0.020443473,
  -0.01733949,
  0.0032812578,
  -0.0106967,
  -0.0034919807,
  0.009592913,
  -0.037301313,
  0.0055390038,
  0.0062113106,
  0.019172447,
  0.01735287,
  0.0228116,
  0.035829596,
  0.009385535,
  0.009626361,
  -0.0017050165,
  0.0028865703,
  -0.044205,
  0.011947658,
  -0.044365548,
  0.02472483,
  0.0020905058,
  0.01822252,
  -0.029568113,
  -0.001438268,
  -0.0072649256,
  0.009740084,
  0.027748536,
  -0.01648322,
  -0.012001175,
  -0.025206482,
  -0.002418297,
  0.0002621494,
  0.00858278,
  0.0025270034,
  -0.010743527,
  -0.03120038,
  0.022329945,
  0.029675147,
  0.022691185,
  0.0016899648,
  -0.045141544,
  0.0024952276,
  0.006990651,
  0.025474066,
  0.026076132,
  -0.005100834,
  0.00067481527,
  0.008776779,
  0.0017066889,
  0.02295877,
  -0.007994094,
  -0.0036826348,
  0.0068969964,
  0.023119321,
  -0.026276821,
  0.028979426,
  0.004237873,
  0.033100232,
  0.018289417,
  0.012837377,
  0.023600973,
  -0.012061382,
  0.0012133295,
  -0.03154824,
  0.017031768,
  0.015386122,
  -0.029327286,
  -0.017754247,
  0.00041266583,
  0.016964871,
  0.019761132,
  0.019921683,
  -0.0079606455,
  0.021875052,
  -0.014797436,
  -0.004257942,
  -0.0106164245,
  -0.007291684,
  0.0069705825,
  -0.005789865,
  -0.003930151,
  -0.010917457,
  -0.027721778,
  -0.0005874321,
  -0.002351401,
  -0.004167632,
  -0.00587014,
  0.015653707,
  0.015680466,
  -0.0026223303,
  0.005856761,
  -0.007311753,
  -0.029782182,
  0.0035321184,
  0.011057939,
  -0.02382842,
  -0.031815823,
  -0.0116265565,
  0.00034848732,
  -0.010308702,
  0.016764184,
  -0.010797044,
  0.0050707306,
  -0.0151185375,
  -0.009191535,
  0.003339792,
  -0.0069705825,
  0.025152965,
  0.0020052132,
  0.00089640886,
  -0.008027542,
  0.005328281,
  0.0036157386,
  0.023266492,
  -0.00035287737,
  0.02524662,
  0.0072114086,
  0.012041313,
  -0.021152573,
  0.0043917345,
  0.0106164245,
  -0.020831471,
  0.0078937495,
  0.0023614352,
  -0.010094634,
  -0.008040921,
  0.0077733365,
  0.0008324394,
  0.0044084582,
  0.029434321,
  0.0126366895,
  -0.004940283,
  -0.02666482,
  -0.021540571,
  -0.016670529,
  0.08899868,
  0.025875444,
  0.0032544993,
  0.018436588,
  -0.009606292,
  -0.028176673,
  -0.034919806,
  -0.0259691,
  0.011017801,
  0.011238559,
  0.0016941458,
  -0.018650657,
  -0.00007578083,
  0.001331234,
  0.0062180003,
  0.014302404,
  -0.014436197,
  0.0021774708,
  0.0068267556,
  -0.021982085,
  0.0036057043,
  -0.012469448,
  0.03173555,
  0.015546673,
  0.0014483024,
  -0.0011012785,
  0.007853612,
  0.0067397906,
  0.0041776667,
  -0.018797828,
  -0.013559856,
  -0.015800878,
  -0.016550116,
  -0.024631174,
  -0.00057823386,
  0.009780222,
  -0.0093654655,
  0.016322669,
  0.012576482,
  -0.0058801747,
  0.010609735,
  0.023253113,
  0.026758473,
  0.0020737818,
  0.00023100089,
  0.0015862758,
  0.000036557718,
  0.022998909,
  0.0074789934,
  -0.003277913,
  0.022490498,
  0.0054821423,
  -0.045061268,
  -0.008870434,
  0.014449576,
  -0.018904861,
  -0.00059872086,
  -0.00011142396,
  0.00042938988,
  -0.038157582,
  0.010295322,
  -0.005799899,
  -0.011813866,
  -0.0063718613,
  -0.023895316,
  -0.024644554,
  0.001029365,
  -0.016028326,
  -0.0095193265,
  0.00569621,
  -0.0038565649,
  -0.004191046,
  -0.02823019,
  -0.000525135,
  0.01877107,
  0.010723459,
  0.000826586,
  0.015198813,
  -0.028685084,
  0.036338005,
  0.010810424,
  -0.016683908,
  -0.014342542,
  -0.01577412,
  0.005227937,
  0.01105125,
  0.009559465,
  0.010890699,
  0.004960352,
  0.00787368,
  0.008649676,
  -0.005973829,
  0.010094634,
  -0.019480169,
  0.024831863,
  -0.00201692,
  0.006672894,
  0.0038264617,
  0.004699457,
  -0.019185826,
  -0.0068100314,
  -0.016376186,
  -0.0031641894,
  -0.0054252804,
  0.012589862,
  0.021982085,
  0.027480952,
  0.021085678,
  -0.014610127,
  -0.0055289697,
  -0.004401769,
  -0.027066195,
  0.0030120006,
  0.002777864,
  0.027066195,
  0.008248299,
  -0.0032461374,
  0.026771853,
  -0.038478684,
  0.008114507,
  0.003170879,
  -0.023373526,
  -0.003418395,
  0.011245249,
  -0.011713522,
  -0.008622918,
  0.008469057,
  -0.041582666,
  -0.022677805,
  -0.011680074,
  -0.0033531713,
  0.012148347,
  -0.021112435,
  0.00946581,
  -0.021634225,
  -0.001541957,
  -0.002070437,
  0.025821926,
  -0.0007676337,
  -0.011499454,
  -0.006642791,
  -0.004803146,
  -0.0058835195,
  0.011900831,
  -0.012375794,
  -0.03628449,
  -0.0029283804,
  0.01086394,
  -0.0009181501,
  0.013459512,
  -0.013446133,
  -0.0025387101,
  -0.012509586,
  -0.0367929,
  -0.0047596633,
  -0.018958379,
  -0.006381896,
  0.0039000474,
  0.029300528,
  0.00911795,
  0.027253505,
  0.01961396,
  0.015225572,
  0.02103216,
  -0.041127775,
  -0.011987796,
  0.00587014,
  0.019332998,
  -0.008395471,
  0.013238755,
  0.0054453495,
  0.014556609,
  0.0024450554,
  0.0077198194,
  -0.015586811,
  0.007111064,
  0.018490106,
  -0.008656366,
  -0.0060708285,
  -0.036177456,
  0.016028326,
  0.0031658618,
  0.004050564,
  -0.010395667,
  -0.011205111,
  -0.01297117,
  -0.0030471212,
  0.008917261,
  0.005207868,
  -0.014757298,
  0.030424384,
  -0.039763093,
  0.024631174,
  0.014396058,
  0.001806197,
  -0.0069170655,
  -0.007151202,
  -0.0035454978,
  -0.011559661,
  0.009131329,
  -0.007204719,
  0.023520699,
  -0.013165168,
  0.007064237,
  0.0238418,
  0.003455188,
  -0.023226354,
  0.0044753547,
  0.0131451,
  -0.010850561,
  0.0025972442,
  -0.0066160327,
  -0.02666482,
  -0.031441208,
  0.017647212,
  0.0071913395,
  -0.033367816,
  0.016603632,
  0.0052480055,
  -0.0032511544,
  0.0024316763,
  0.020764574,
  0.00552228,
  0.00613438,
  0.021313123,
  0.014623506,
  0.0021055574,
  -0.02769502,
  0.0073719593,
  0.0064722057,
  0.003612394,
  0.02630358,
  0.013024687,
  0.0059704846,
  -0.0012384156,
  0.023641111,
  0.011673384,
  -0.026985921,
  -0.03783648,
  -0.006820066,
  0.0318961,
  -0.0020152475,
  -0.010027738,
  -0.023386905,
  -0.0110780075,
  0.017205698,
  -0.007044168,
  0.00082993077,
  -0.008087749,
  -0.020604024,
  -0.02523324,
  0.008107818,
  -0.015667086,
  0.016724046,
  0.001025184,
  -0.0016623702,
  -0.014315783,
  -0.030023007,
  -0.002996949,
  0.0012049675,
  0.014128474,
  0.03243127,
  -0.015640328,
  0.014543231,
  0.011907521,
  -0.004522182,
  -0.0238418,
  0.010930836,
  0.006080863,
  0.025755031,
  -0.017125422,
  -0.012375794,
  0.002951794,
  0.007913819,
  0.008489125,
  0.003423412,
  -0.0022811599,
  -0.009077812,
  -0.012703585,
  -0.005408556,
  0.014449576,
  0.010756906,
  -0.03890682,
  0.001401475,
  -0.011305455,
  -0.0053583845,
  -0.014784057,
  -0.0032461374,
  0.00964643,
  -0.012308897,
  -0.012462759,
  -0.0018781103,
  0.01209483,
  0.018075349,
  -0.007646234,
  -0.0116265565,
  0.004344907,
  0.035562012,
  -0.014275646,
  -0.0031976376,
  -0.004030495,
  0.022731323,
  -0.018664036,
  0.014141853,
  0.009559465,
  -0.035454977,
  0.013553167,
  -0.011445937,
  -0.020389957,
  -0.023373526,
  0.023801662,
  0.016938115,
  0.008141265,
  -0.01210152,
  0.020189269,
  0.020951884,
  -0.004676043,
  0.013914406,
  -0.02611627,
  0.018316176,
  0.0011330541,
  -0.004401769,
  0.0119209,
  -0.007432166,
  0.007024099,
  0.021446917,
  0.0025370377,
  0.018864723,
  -0.018985137,
  -0.015038262,
  -0.005903588,
  0.0060340357,
  -0.01664377,
  -0.009318639,
  -0.018102108,
  -0.011091387,
  -0.016429702,
  -0.0050974893,
  -0.013593305,
  -0.008027542,
  -0.018115487,
  0.021808155,
  -0.020015338,
  0.018650657,
  -0.014436197,
  0.0007325132,
  -0.010910767,
  -0.014677023,
  -0.02683875,
  -0.031949617,
  -0.019212583,
  0.020282922,
  0.012850757,
  -0.02332001,
  -0.017821142,
  0.006234724,
  -0.0036424971,
  -0.00022431127,
  0.008341954,
  -0.00928519,
  0.014436197,
  0.008890503,
  -0.014717161,
  0.017807763,
  0.014382679,
  -0.008040921,
  -0.007940577,
  -0.0029049667,
  -0.00605076,
  -0.012750412,
  0.009492569,
  0.018637277,
  -0.02138002,
  -0.016188877,
  0.008783469,
  -0.016911356,
  0.030718727,
  0.025728272,
  0.0015369398,
  -0.010736837,
  0.002789571,
  0.0062982757,
  -0.008676435,
  0.027454194,
  0.010101324,
  -0.04222487,
  -0.008281748,
  -0.007726509,
  0.024082625,
  -0.011512833,
  0.014302404,
  0.0009173139,
  -0.014556609,
  0.01576074,
  0.0055390038,
  -0.013566546,
  0.0062782066,
  0.017834522,
  0.02579517,
  0.0036759453,
  -0.001806197,
  0.041636184,
  0.008435609,
  -0.029354045,
  -0.00578652,
  -0.0057229684,
  -0.023708008,
  0.0127236545,
  0.005963795,
  -0.041555908,
  -0.025380412,
  -0.012436001,
  0.011639936,
  -0.01368027,
  -0.0080476105,
  0.0022627634,
  -0.0066528255,
  -0.019132309,
  0.008422229,
  0.00052388076,
  -0.009840429,
  -0.009492569,
  -0.008642986,
  -0.02067092,
  0.001235907,
  -0.008321885,
  0.018811207,
  0.0019767822,
  -0.024082625,
  0.029166736,
  -0.006261483,
  0.0014583368,
  -0.0101347715,
  0.010201668,
  -0.021875052,
  -0.0073719593,
  0.21299745,
  0.007980715,
  0.010496011,
  0.016001567,
  -0.0021557296,
  0.012931032,
  0.017580317,
  -0.009325328,
  -0.008609539,
  0.017660592,
  0.015412881,
  0.02333339,
  -0.020938504,
  0.012837377,
  -0.016670529,
  -0.027989363,
  -0.030558176,
  -0.012262071,
  -0.021821534,
  -0.025567722,
  -0.006749825,
  -0.012549724,
  -0.005375108,
  -0.0038599097,
  0.029835697,
  -0.0037762895,
  0.011706832,
  -0.007639544,
  0.021781398,
  0.011305455,
  -0.0037629104,
  -0.014034819,
  -0.0009457448,
  0.0073050633,
  -0.007519131,
  0.0031658618,
  0.004017116,
  -0.02101878,
  0.034090295,
  0.037595656,
  -0.0036324628,
  0.0012242001,
  -0.010027738,
  0.02016251,
  -0.00037879962,
  -0.0032762405,
  -0.0056059,
  -0.027226746,
  -0.003277913,
  0.01017491,
  -0.021607468,
  0.0055089006,
  -0.0005723805,
  0.0416897,
  -0.0020486957,
  0.000101755366,
  -0.008188093,
  -0.008295127,
  -0.011231869,
  -0.002426659,
  -0.011365661,
  0.029220253,
  -0.009853808,
  0.045328856,
  -0.021072298,
  0.029755423,
  -0.046693537,
  0.011158284,
  -0.016269151,
  0.0070909956,
  -0.0000098449755,
  -0.011887452,
  -0.003963599,
  -0.010542839,
  -0.02207574,
  -0.024925519,
  0.019841408,
  0.028069638,
  0.035294425,
  0.0054286253,
  -0.019720994,
  -0.015078399,
  -0.021701122,
  -0.015252329,
  -0.017981693,
  -0.02456428,
  0.015573432,
  -0.005227937,
  -0.015841017,
  0.008957399,
  -0.019734373,
  -0.0062815514,
  -0.008977468,
  -0.01998858,
  0.008843675,
  0.03141445,
  -0.005318247,
  0.019038653,
  -0.005906933,
  0.020229407,
  -0.026865507,
  -0.007980715,
  0.010736837,
  0.0094725,
  -0.014797436,
  0.00042458173,
  -0.02017589,
  0.022209533,
  -0.0012526311,
  -0.027641503,
  -0.026397234,
  -0.027802054,
  -0.004104081,
  -0.010248495,
  -0.0012049675,
  0.02121947,
  -0.008562711,
  -0.010148151,
  0.0090577435,
  -0.017954936,
  0.011733591,
  -0.012168416,
  -0.019520307,
  -0.0032578441,
  -0.008188093,
  -0.011231869,
  -0.0037261173,
  0.011954348,
  0.0041274945,
  -0.03315375,
  0.02034982,
  -0.021540571,
  -0.007398718,
  0.015439639,
  -0.014556609,
  -0.009786911,
  -0.004224494,
  0.014315783,
  -0.028096396,
  0.01683108,
  -0.010669941,
  0.00005236717,
  -0.0053550396,
  -0.02192857,
  -0.007967335,
  -0.03384947,
  0.025928961,
  -0.0006660351,
  -0.012964481,
  -0.013345788,
  -0.03138769,
  -0.0221025,
  0.010589666,
  -0.035669044,
  0.029862456,
  -0.010656562,
  -0.0046492848,
  -0.02471145,
  -0.00012145838,
  0.019600581,
  -0.019894924,
  0.01033546,
  0.046827327,
  -0.0009833738,
  -0.008923951,
  0.014917849,
  -0.17318083,
  0.016603632,
  0.011312145,
  -0.029862456,
  0.035107117,
  0.037916757,
  0.011365661,
  0.007198029,
  0.0039033922,
  0.012716965,
  0.013620063,
  -0.01333241,
  -0.01998858,
  0.0017292664,
  0.026557785,
  -0.016148739,
  -0.02052375,
  -0.002603934,
  0.0077064405,
  0.009405604,
  0.006679584,
  0.0035521872,
  0.020992022,
  -0.0018630587,
  0.0077934056,
  -0.0063517927,
  0.0070174094,
  0.021834914,
  -0.0019951786,
  -0.011552971,
  0.0046927673,
  -0.015586811,
  -0.0073585804,
  0.033742435,
  0.012857446,
  -0.013238755,
  0.0052647297,
  0.0070575476,
  -0.0072649256,
  0.0047262153,
  0.03644504,
  -0.0017593696,
  0.020978643,
  -0.0086630555,
  0.014917849,
  0.037006967,
  -0.0074923723,
  0.0012635017,
  0.02262429,
  -0.02084485,
  0.01577412,
  -0.012034623,
  -0.0046225265,
  0.011874072,
  0.019761132,
  0.009940773,
  -0.010475942,
  0.031227138,
  -0.0014892763,
  -0.0075258208,
  -0.0023296596,
  -0.020256164,
  0.0050439723,
  -0.015225572,
  0.009124639,
  -0.015172054,
  -0.020430094,
  0.0054955212,
  -0.026704956,
  0.020590644,
  0.013011307,
  -0.009526016,
  0.009238363,
  -0.0013638459,
  -0.0012910963,
  0.0032662062,
  -0.041315082,
  0.036043663,
  0.014596747,
  0.008375403,
  0.0080543,
  0.026357096,
  -0.00210054,
  -0.0003024962,
  0.03384947,
  0.0027845537,
  0.007084306,
  0.005023903,
  0.020644162,
  0.0075325104,
  0.017045148,
  -0.015345984,
  0.009526016,
  -0.019868167,
  0.007064237,
  0.0060106223,
  0.010328771,
  0.013967923,
  -0.016710667,
  -0.019774511,
  0.0009883911,
  0.005318247,
  -0.025647996,
  0.014650264,
  0.031146863,
  0.004575699,
  0.01227545,
  0.013626752,
  0.04597106,
  -0.009385535,
  -0.01297786,
  -0.0015160347,
  0.017406387,
  0.03748862,
  0.0059303464,
  0.010455874,
  0.007499062,
  0.0043081143,
  0.03609718,
  -0.01680432,
  0.02595572,
  0.016229015,
  -0.032324236,
  -0.00085292634,
  -0.0099876,
  -0.015024883,
  -0.09713326,
  -0.020256164,
  0.010455874,
  0.030558176,
  -0.008743331,
  0.02508607,
  0.0009833738,
  0.0007329313,
  -0.004492079,
  0.01735287,
  -0.00025148783,
  -0.012429311,
  -0.021540571,
  -0.0062982757,
  0.00078603014,
  -0.0016548444,
  -0.017259216,
  0.005291488,
  -0.0016933096,
  0.028765358,
  -0.009158088,
  -0.015894534,
  0.0009833738,
  -0.016509978,
  0.00029078935,
  -0.011767039,
  -0.025326895,
  0.019306239,
  0.028283706,
  0.01787466,
  -0.002261091,
  -0.023761524,
  0.0071779606,
  -0.022156015,
  -0.008214851,
  -0.0027410712,
  -0.023413664,
  0.01350634,
  0.03417057,
  -0.024631174,
  0.008917261,
  -0.011158284,
  0.017901419,
  -0.01594805,
  0.009599602,
  -0.023239734,
  -0.004492079,
  0.016750805,
  0.0063852407,
  -0.007137823,
  -0.024430485,
  -0.022771461,
  -0.024470624,
  0.009151398,
  0.020122372,
  -0.021527192,
  -0.01280393,
  0.027828813,
  -0.023988971,
  -0.008448988,
  -0.012148347,
  0.003306344,
  -0.020095613,
  0.0364718,
  0.02330663,
  0.007432166,
  -0.008101128,
  -0.01350634,
  0.04522182,
  -0.021165952,
  -0.0064220335,
  0.015386122,
  -0.031307414,
  0.011813866,
  -0.02156733,
  0.017393008,
  -0.015305847,
  0.008475746,
  0.0063183447,
  -0.0119209,
  -0.009773532,
  -0.01577412,
  0.009204915,
  -0.00045573025,
  0.014476334,
  0.0012877515,
  0.028658325,
  -0.014556609,
  -0.008482436,
  -0.047416016,
  -0.0032963094,
  0.024617795,
  0.014382679,
  -0.024992414,
  -0.01017491,
  0.010897389,
  0.014409438,
  -0.0023279872,
  -0.00030855864,
  0.0038799786,
  0.007733199,
  -0.006064139,
  -0.049877793,
  0.0015938015,
  -0.0026691577,
  -0.02259753,
  0.0035454978,
  0.0010853906,
  0.013051446,
  -0.022704564,
  0.018610518,
  0.02753447,
  -0.030825762,
  0.012522966,
  0.010884009,
  -0.007111064,
  -0.02560786,
  -0.004167632,
  0.023279872,
  -0.00929188,
  0.019386513,
  0.011091387,
  0.02032306,
  -0.0061042765,
  0.037729446,
  0.018302796,
  -0.008254989,
  0.012121588,
  -0.03152148,
  0.009512637,
  -0.043669827,
  -0.007238167,
  0.015158675,
  -0.038987096,
  -0.01909217,
  0.021099057,
  -0.005411901,
  -0.0043382174,
  0.018610518,
  0.01593467,
  0.011680074,
  -0.025005793,
  -0.014155232,
  -0.027828813,
  0.014891091,
  -0.01121849,
  -0.034652222,
  0.008823606,
  -0.032190442,
  0.014396058,
  0.037408344,
  -0.0011924244,
  0.014155232,
  0.032003134,
  -0.015586811,
  -0.013432753,
  0.0011129852,
  -0.016402945,
  0.010295322,
  -0.0077198194,
  0.0078937495,
  -0.035321184,
  0.011285386,
  0.03470574,
  0.021647604,
  -0.01034884,
  0.0139545435,
  0.019707616,
  0.0035990146,
  0.00066478085,
  0.032056652,
  -0.026932403,
  -0.005324936,
  0.0029083116,
  0.012650068,
  0.0036257731,
  0.009793601,
  0.0010979336,
  0.0075592687,
  -0.0071779606,
  -0.02420304,
  0.035347942,
  0.021607468,
  -0.0182359,
  -0.012288829,
  0.00013504666,
  0.023012288,
  -0.017647212,
  -0.00059788465,
  0.010797044,
  -0.005662762,
  0.01929286,
  -0.03384947,
  0.015131917,
  -0.0013429408,
  -0.015827637,
  0.021139193,
  -0.0016013274,
  0.010950905,
  -0.014128474,
  0.005241316,
  0.026397234,
  0.018704172,
  -0.015439639,
  -0.035160635,
  -0.021339882,
  -0.0014140181,
  -0.0031775688,
  -0.010235116,
  -0.022196153,
  0.009579534,
  -0.0013053118,
  0.0013446133,
  -0.0122420015,
  -0.0013772252,
  0.010542839,
  -0.007947267,
  0.0068836175,
  -0.013352478,
  -0.022744702,
  -0.026597923,
  0.0037261173,
  0.0033916365,
  0.012589862,
  0.023039045,
  -0.01244269,
  0.022329945,
  0.018784449,
  0.0058032437,
  -0.010255185,
  0.008676435,
  0.009358776,
  -0.02119271,
  0.017326111,
  -0.00447201,
  -0.004702802,
  -0.0045623197,
  -0.0023597628,
  -0.014115095,
  0.008375403,
  0.0026842093,
  0.049851038,
  -0.008984158,
  0.0009950807,
  0.0025437274,
  -0.021995464,
  0.013399306,
  0.013593305,
  -0.0005794882,
  -0.003906737,
  -0.027427435,
  0.027092954,
  -0.010268564,
  0.022329945,
  -0.003306344,
  -0.011559661,
  0.024617795,
  -0.024176281,
  0.015573432,
  -0.0100009795,
  -0.007499062,
  0.014850953,
  -0.00402715,
  0.016764184,
  -0.0066160327,
  -0.016683908,
  -0.0066528255,
  0.0245509,
  -0.017767627,
  -0.021139193,
  -0.01893162,
  0.008796848,
  0.018557,
  -0.021674363,
  -0.025888823,
  0.013245445,
  -0.025045931,
  0.00005748891,
  -0.006137725,
  0.0175268,
  0.028016122,
  0.01733949,
  0.03928144,
  -0.029166736,
  -0.022851735,
  0.031976376,
  -0.028123155,
  -0.013459512,
  -0.014369301,
  -0.01910555,
];
final _chaoVector = [
  0.014674897,
  -0.012181209,
  0.013539028,
  -0.023239605,
  -0.018304454,
  -0.00029049508,
  -0.015523534,
  -0.04358079,
  -0.017886663,
  -0.008740964,
  -0.0019469698,
  0.009680993,
  0.0063745715,
  0.003120374,
  0.005698926,
  -0.0020106176,
  0.021150652,
  -0.021150652,
  0.044651378,
  -0.018239174,
  -0.008969443,
  0.012572887,
  0.010359903,
  -0.014100435,
  -0.008427621,
  -0.007572456,
  0.012142041,
  -0.02035424,
  0.005992685,
  -0.03407605,
  0.041857403,
  0.004749105,
  -0.029062564,
  -0.020589245,
  -0.01650273,
  -0.039428994,
  -0.0128927585,
  0.005911085,
  0.0099421125,
  0.0044847215,
  -0.001860474,
  -0.00749412,
  0.003410869,
  0.003450037,
  -0.0137740355,
  0.0025197999,
  -0.0032803095,
  -0.01547131,
  -0.029924257,
  0.033814933,
  0.015027408,
  0.0021183293,
  -0.03389327,
  -0.012050649,
  0.0060938685,
  0.015249359,
  -0.019309763,
  0.016189389,
  -0.014962128,
  -0.03433717,
  -0.003789492,
  -0.010131423,
  -0.011652443,
  0.016750794,
  0.008375397,
  0.0031987098,
  0.014857681,
  0.0054182224,
  -0.0016238348,
  0.0025948717,
  0.012357464,
  0.0076703755,
  0.007259113,
  -0.015627982,
  0.012481496,
  0.014949072,
  -0.021085372,
  0.00499064,
  -0.0010379486,
  -0.013721812,
  0.00920445,
  -0.039402883,
  -0.0005516142,
  0.019492546,
  0.015171023,
  0.01376098,
  0.0055096145,
  0.0054998226,
  -0.022103738,
  -0.020497855,
  0.002456152,
  0.013539028,
  0.027809191,
  0.029193122,
  -0.00007185877,
  0.009321954,
  0.008923748,
  0.030968733,
  0.0070241056,
  -0.022404024,
  0.0025654957,
  -0.007703015,
  -0.022900151,
  -0.014492113,
  -0.013995987,
  -0.02134649,
  0.026255531,
  -0.011456603,
  0.03391938,
  0.002208089,
  -0.017338313,
  0.046923112,
  0.010581854,
  -0.024101298,
  -0.02419269,
  -0.005630382,
  0.017586377,
  -0.0048927204,
  0.0066977064,
  -0.033997715,
  0.0016825866,
  0.0072525847,
  0.023944627,
  -0.014701009,
  0.02105926,
  0.0032052377,
  0.009426402,
  -0.01579771,
  0.0030485662,
  0.021007037,
  0.012324825,
  0.03587777,
  0.03297935,
  -0.01033379,
  -0.02603358,
  0.009583074,
  -0.022913206,
  0.0070241056,
  -0.0009139171,
  -0.009517794,
  0.020132288,
  0.036739465,
  -0.0052843993,
  -0.01669857,
  0.008871524,
  0.027417513,
  0.012559832,
  0.007259113,
  -0.008662628,
  0.01566715,
  0.029114787,
  -0.017129418,
  -0.0033260053,
  0.00008578173,
  0.016006604,
  0.004664241,
  0.0031921817,
  0.023200437,
  -0.0067499303,
  -0.01253372,
  0.014962128,
  -0.00066258985,
  0.0022962166,
  -0.019792832,
  0.018565573,
  0.029637026,
  0.014048211,
  0.010353375,
  -0.00035128687,
  -0.022717368,
  0.004703409,
  0.016946634,
  -0.027443623,
  -0.015915213,
  0.0016254667,
  0.015562702,
  -0.0029065828,
  0.015575758,
  -0.029532578,
  -0.019923393,
  -0.012847063,
  0.000943293,
  0.029741473,
  0.0053986385,
  -0.019531714,
  0.027861414,
  0.03159542,
  0.011835226,
  0.00862346,
  0.010686302,
  0.019270593,
  0.006195052,
  0.027600296,
  -0.010869085,
  -0.68684787,
  0.0022113528,
  -0.0024218801,
  -0.01056227,
  0.002521432,
  0.02997648,
  0.014217938,
  -0.0113391,
  -0.00034904288,
  -0.016124109,
  0.006841322,
  -0.0025883438,
  0.012834007,
  -0.0022831606,
  0.0098572485,
  -0.02752196,
  0.019858113,
  -0.015184079,
  0.012422744,
  0.029271457,
  -0.012259545,
  -0.0060873404,
  -0.0064431154,
  0.0054704463,
  0.005548782,
  0.0048829284,
  -0.005297455,
  -0.023357108,
  -0.019910336,
  -0.012076762,
  -0.046087533,
  0.013160406,
  -0.0027368553,
  -0.02461048,
  0.045173615,
  -0.0021199612,
  0.022208184,
  0.01805639,
  -0.0054671825,
  0.043633014,
  -0.007990247,
  -0.017756103,
  0.0131799895,
  -0.007957607,
  0.005290927,
  -0.0056956615,
  0.02438853,
  -0.0042105466,
  -0.0051440476,
  -0.009641825,
  -0.0085190125,
  -0.010758109,
  0.012709975,
  -0.016803019,
  -0.020223679,
  0.011613275,
  0.034102164,
  -0.02887978,
  0.0022423607,
  0.021777337,
  -0.007879271,
  0.007213417,
  -0.03773172,
  -0.010327263,
  -0.019453378,
  0.0019339138,
  -0.021020092,
  0.012285656,
  0.010255455,
  -0.017612487,
  -0.0043182583,
  0.009217506,
  -0.0363739,
  0.0054084305,
  0.006919658,
  0.02105926,
  0.03556443,
  0.009106531,
  -0.009439458,
  0.010901725,
  -0.010170591,
  0.008257894,
  0.008101222,
  0.010059616,
  0.019531714,
  -0.0108233895,
  -0.026947498,
  -0.023187382,
  -0.012422744,
  -0.0024365683,
  -0.0014981712,
  0.034728847,
  -0.00794455,
  0.010431711,
  0.010261983,
  0.011554523,
  -0.022377912,
  -0.011117148,
  0.0122987125,
  -0.016111052,
  0.0023158006,
  -0.014583506,
  -0.0065736747,
  -0.0028249829,
  -0.015014352,
  0.034572177,
  -0.025641901,
  0.015627982,
  0.039063428,
  -0.01056227,
  0.0113325715,
  0.0012231801,
  0.0067172903,
  -0.00820567,
  0.016111052,
  -0.032065433,
  -0.013721812,
  0.006195052,
  0.02105926,
  -0.01682913,
  0.02445381,
  -0.004732785,
  0.022978486,
  0.005313775,
  0.015301583,
  0.013969875,
  0.017403591,
  -0.015849933,
  -0.025406895,
  -0.009478626,
  0.0087736035,
  0.01395682,
  0.010738526,
  -0.00746148,
  0.016450508,
  -0.023618229,
  0.007761767,
  -0.017194698,
  -0.0027221672,
  0.0039265794,
  0.009047779,
  0.0029588065,
  -0.003727476,
  -0.011234652,
  0.0070959134,
  -0.019701442,
  0.0067368746,
  0.006828266,
  -0.025080496,
  0.022665143,
  0.022273464,
  -0.026020525,
  0.0001858434,
  0.023004599,
  -0.0023109047,
  -0.026999721,
  -0.008401509,
  -0.012540248,
  -0.020954812,
  -0.010575326,
  0.020693693,
  0.024271026,
  -0.004954736,
  0.005819693,
  0.0032819414,
  -0.037000585,
  -0.004683825,
  0.036974475,
  -0.00930237,
  -0.026947498,
  0.017782215,
  -0.031856537,
  0.004758897,
  0.009478626,
  0.0032444056,
  0.013408469,
  -0.0060449084,
  -0.0013904596,
  0.00042268666,
  -0.011267292,
  -0.00094818894,
  -0.018552516,
  -0.015353806,
  -0.002208089,
  0.029245347,
  -0.011221596,
  0.011600219,
  0.028018085,
  -0.007448424,
  0.021007037,
  0.014270162,
  0.009667937,
  -0.040160127,
  -0.010092256,
  0.014100435,
  -0.000120971614,
  0.01331055,
  0.0144007215,
  0.019636162,
  0.012481496,
  0.008767076,
  0.013284437,
  0.026307756,
  -0.005898029,
  0.013813204,
  -0.034102164,
  -0.0010020448,
  -0.005052656,
  -0.0020563134,
  0.013153878,
  -0.008669157,
  -0.010993117,
  -0.0057185097,
  -0.0017625544,
  0.012651224,
  0.025315503,
  -0.0065214513,
  0.004093043,
  0.0045695854,
  -0.0023288566,
  -0.0032672535,
  -0.01172425,
  -0.0073766164,
  0.008133862,
  -0.02984592,
  0.006273388,
  0.020432575,
  0.037287816,
  0.003720948,
  0.0052223834,
  -0.010353375,
  0.0029947103,
  0.0039167874,
  0.007983718,
  0.0013251797,
  -0.0035381645,
  0.03326658,
  -0.0077682952,
  0.029454242,
  -0.0064072115,
  -0.0010501887,
  0.026895273,
  0.019401154,
  -0.009863776,
  0.0025279599,
  0.023892403,
  0.0028967906,
  -0.009054307,
  0.012481496,
  0.026242476,
  -0.020993982,
  0.0054508625,
  -0.011064924,
  -0.012187737,
  0.02609886,
  -0.019257538,
  -0.005532462,
  0.019832,
  0.008003302,
  0.008310118,
  0.0069849375,
  0.006211372,
  0.009896416,
  0.0025867117,
  0.010137952,
  -0.0010526367,
  0.0061297724,
  -0.013969875,
  0.0005222383,
  -0.011933146,
  -0.00625054,
  -0.020262847,
  -0.015980493,
  -0.00097593287,
  0.025250223,
  -0.007454952,
  -0.005715246,
  -0.005920877,
  0.0023109047,
  0.004037555,
  -0.001888218,
  -0.019492546,
  0.021685947,
  0.0034010771,
  0.0042758263,
  -0.009622241,
  -0.018970307,
  0.014217938,
  -0.020523965,
  -0.014962128,
  -0.008133862,
  0.014466002,
  0.018852804,
  -0.0035544846,
  -0.0313343,
  0.01560187,
  0.03908954,
  -0.0038645635,
  0.00926973,
  0.011561051,
  -0.008701796,
  0.0006523899,
  -0.01363042,
  -0.03459829,
  -0.01711636,
  0.0035349007,
  -0.02070675,
  -0.011495771,
  0.008616933,
  0.0023908722,
  0.00038800677,
  -0.02064147,
  -0.029558688,
  -0.017051082,
  -0.002787447,
  -0.0020350975,
  -0.024976049,
  -0.0023827124,
  0.032326553,
  -0.0054247505,
  -0.00035842686,
  -0.007761767,
  -0.019596994,
  -0.008838884,
  0.013656532,
  0.026190251,
  -0.005114672,
  0.0053529427,
  0.0106210215,
  0.012664279,
  -0.0036034444,
  -0.026542762,
  0.00010373367,
  0.005627118,
  -0.016842186,
  -0.020824254,
  0.0025426478,
  0.011959258,
  0.027443623,
  0.023500724,
  0.013656532,
  -0.01831751,
  0.0015030672,
  -0.0038221318,
  -0.0045728493,
  -0.0070959134,
  0.0031889179,
  0.036478348,
  -0.0035577486,
  0.02684305,
  -0.0015822189,
  0.013747924,
  0.0013814836,
  -0.021633722,
  -0.010555742,
  0.012285656,
  0.010040032,
  0.0051179356,
  -0.024440754,
  0.0005450863,
  -0.010699358,
  0.0067238184,
  -0.018461125,
  -0.0036524043,
  0.014727121,
  0.0155365905,
  -0.0057739974,
  -0.017821383,
  0.012599,
  -0.018565573,
  -0.026686378,
  0.019871168,
  -0.020197567,
  -0.0158891,
  0.039011203,
  0.015993549,
  -0.023566004,
  -0.019466434,
  0.0066781226,
  0.009184866,
  0.0047099367,
  -0.012873175,
  0.016124109,
  0.0057772617,
  -0.01505352,
  -0.025654959,
  -0.0023696565,
  -0.010418654,
  -0.010934365,
  -0.009576545,
  -0.029375905,
  -0.007931494,
  -0.028226981,
  0.004119155,
  -0.015236303,
  -0.00374706,
  -0.017808327,
  -0.010020448,
  0.028279206,
  0.0064496435,
  0.005927405,
  -0.011149788,
  0.016306892,
  0.012161625,
  0.012997206,
  -0.037783943,
  -0.009210979,
  -0.021294268,
  0.006860906,
  0.00943293,
  0.024976049,
  -0.013839316,
  -0.006893546,
  0.021620667,
  0.02997648,
  -0.010359903,
  0.0049025123,
  -0.0045630573,
  0.0009041251,
  0.003427189,
  0.0003776028,
  0.0055814222,
  -0.0003439429,
  0.012938455,
  -0.004625073,
  -0.021516219,
  -0.026790826,
  0.00045124657,
  0.016920522,
  -0.00041228268,
  0.016581066,
  0.026451372,
  -0.0038417156,
  0.001564267,
  0.012422744,
  -0.027809191,
  0.023944627,
  -0.000006802103,
  -0.00061118207,
  0.012102873,
  0.011900506,
  0.021568444,
  -0.011482716,
  -0.0006552459,
  -0.0070632733,
  -0.0128078945,
  0.0014239154,
  0.011143261,
  0.0101118395,
  0.011228124,
  0.008845411,
  -0.002475736,
  -0.008864996,
  -0.008616933,
  0.0029506464,
  0.01967533,
  -0.008662628,
  -0.019701442,
  -0.028357541,
  -0.0029359586,
  0.0011195485,
  0.021907898,
  -0.027809191,
  -0.0047001448,
  -0.041883513,
  -0.004634865,
  -0.020249791,
  -0.017455816,
  -0.03587777,
  -0.036974475,
  0.000617302,
  0.0035708046,
  0.01062755,
  0.02199929,
  0.009060835,
  0.010346847,
  -0.026738603,
  0.007259113,
  -0.0040114434,
  -0.022234296,
  -0.013460693,
  -0.020954812,
  0.030759837,
  0.009348066,
  0.03284879,
  -0.016124109,
  0.01505352,
  0.011685083,
  0.009694049,
  0.0003461869,
  -0.02393157,
  -0.0237749,
  -0.020262847,
  0.011097564,
  0.010438238,
  0.023657396,
  0.0033782292,
  -0.017808327,
  -0.016672458,
  0.0017135945,
  -0.0030045023,
  -0.0002735631,
  0.0049449443,
  -0.018944195,
  -0.017273033,
  -0.015027408,
  -0.0038972036,
  0.012938455,
  0.021646779,
  0.0011334204,
  0.03300546,
  0.019923393,
  0.02051091,
  -0.0029555424,
  0.03013315,
  0.00083435734,
  0.0026666794,
  -0.013800148,
  -0.00015391751,
  -0.00746148,
  0.009491681,
  0.01192009,
  -0.0022619448,
  0.019166147,
  -0.003397813,
  0.03869786,
  0.0049025123,
  -0.012592471,
  0.0037699079,
  0.0073831445,
  0.005532462,
  -0.020262847,
  -0.0012860119,
  -0.0223518,
  -0.02064147,
  -0.03734004,
  -0.0011880922,
  -0.06366085,
  0.010647134,
  0.0035251088,
  -0.015314639,
  0.025707182,
  -0.006221164,
  -0.00868874,
  0.01702497,
  0.016907467,
  0.035459984,
  -0.022195129,
  0.003436981,
  0.018173894,
  -0.0114174355,
  -0.0013406837,
  0.023370165,
  0.0076964875,
  -0.0176386,
  0.0101118395,
  0.02820087,
  -0.0027303272,
  -0.002836407,
  0.016816074,
  0.0190617,
  -0.012292185,
  -0.023409333,
  0.0012876439,
  0.012122457,
  0.0032036058,
  -0.0034696208,
  0.0012623479,
  -0.017521096,
  0.012057178,
  -0.012279129,
  0.0048568165,
  -0.019805888,
  -0.0073048086,
  -0.015680205,
  0.033710483,
  -0.01240316,
  0.019388098,
  0.020654526,
  -0.008904164,
  -0.011991898,
  0.0057772617,
  -0.013930707,
  -0.0042921463,
  0.00566955,
  0.05057878,
  -0.009726689,
  0.022155961,
  0.0070698014,
  0.018147781,
  -0.013108182,
  -0.0032395096,
  -0.011195485,
  0.018160839,
  -0.009876832,
  0.0013210998,
  -0.03026371,
  -0.024036018,
  0.005672814,
  0.0021052733,
  -0.0018131462,
  -0.022599863,
  -0.023814067,
  -0.0038156037,
  0.017599432,
  0.0031562778,
  -0.023122102,
  -0.032457113,
  -0.03159542,
  -0.032901015,
  -0.010092256,
  0.0051114075,
  0.0013072278,
  0.029062564,
  -0.012938455,
  -0.019270593,
  -0.00620158,
  0.01540603,
  0.007761767,
  -0.009224035,
  0.0073178643,
  0.006795626,
  -0.016776906,
  0.017756103,
  -0.024009908,
  -0.005640174,
  -0.027417513,
  0.004690353,
  -0.0045891693,
  -0.014857681,
  -0.0014279954,
  -0.006580203,
  -0.014348498,
  -0.020119231,
  0.01853946,
  0.014048211,
  0.014217938,
  -0.0055357264,
  0.009772385,
  0.022142906,
  0.0010991485,
  -0.006514923,
  -0.022926262,
  0.011423963,
  -0.013930707,
  0.012990679,
  0.0060416446,
  0.009374178,
  0.0063680434,
  0.014622673,
  0.021215932,
  0.009047779,
  -0.027626408,
  0.012736087,
  -0.0019453377,
  0.010079199,
  -0.022586808,
  -0.021111485,
  -0.0328749,
  -0.010255455,
  -0.015876045,
  0.0013096757,
  0.0035610127,
  -0.01175689,
  0.0011032284,
  0.029375905,
  -0.0017233865,
  -0.00797719,
  0.004895984,
  -0.0057837893,
  0.0062962356,
  -0.000310487,
  -0.022939319,
  -0.011319515,
  0.00095879694,
  0.017912775,
  0.020628413,
  -0.01812167,
  -0.01327791,
  0.0046479213,
  -0.005917613,
  0.0137740355,
  -0.020171456,
  0.0030958941,
  0.00009649169,
  0.0005614062,
  -0.006204844,
  0.03553832,
  0.0014157555,
  0.019910336,
  -0.026216364,
  0.019009475,
  0.012664279,
  0.009132642,
  0.013930707,
  -0.008114278,
  -0.025459118,
  -0.011482716,
  0.012918871,
  0.012677335,
  -0.009583074,
  0.014844624,
  -0.010588382,
  -0.005385583,
  -0.005274607,
  0.007187305,
  -0.000634846,
  -0.0029131107,
  0.029924257,
  -0.008904164,
  -0.0023696565,
  -0.009341538,
  -0.027417513,
  0.0101967035,
  -0.01498824,
  -0.018291397,
  0.0011268924,
  0.011143261,
  -0.015915213,
  -0.016868297,
  0.018291397,
  -0.020158399,
  0.01579771,
  -0.012794839,
  0.004687089,
  0.026947498,
  0.005313775,
  -0.031073181,
  -0.019662272,
  -0.011391323,
  -0.0012076761,
  0.011084508,
  0.012168153,
  -0.0044716657,
  0.0019518657,
  -0.002849463,
  0.025798574,
  -0.015549646,
  0.0033406932,
  -0.014557393,
  -0.0051016156,
  -0.008003302,
  0.025889965,
  0.016737739,
  0.0057576774,
  0.009563489,
  -0.0054965583,
  0.000012864709,
  -0.022534585,
  -0.020053951,
  -0.012174681,
  -0.02467576,
  0.0032688854,
  -0.0000045517354,
  -0.018800579,
  -0.010784222,
  -0.0037046282,
  -0.00095308496,
  0.0002170553,
  0.0013602676,
  0.20221068,
  -0.010692829,
  -0.0123835765,
  0.037392262,
  0.007135081,
  0.016933577,
  0.01230524,
  0.009152227,
  -0.0011415803,
  0.015510478,
  0.0018425221,
  0.013362773,
  -0.0047621606,
  0.0037046282,
  -0.0004912304,
  -0.0070502176,
  -0.032039322,
  -0.010229344,
  0.0026487275,
  -0.043058548,
  -0.0009930688,
  -0.023761844,
  0.0013961715,
  0.0033553813,
  0.022952374,
  0.015027408,
  -0.011861338,
  -0.0010648766,
  0.03324047,
  0.01085603,
  -0.011202012,
  -0.0014998032,
  -0.0027629673,
  0.018774468,
  -0.021490106,
  0.00020297935,
  0.017481929,
  -0.013317077,
  0.01246844,
  0.020053951,
  -0.015419086,
  -0.005943725,
  0.005633646,
  -0.020523965,
  -0.010320735,
  0.026268588,
  -0.0041158907,
  -0.022665143,
  0.0151971355,
  0.032770455,
  -0.02303071,
  -0.003773172,
  0.009106531,
  0.056192845,
  -0.0081795575,
  -0.0019012738,
  0.0027433832,
  0.0035218447,
  -0.04269298,
  -0.0035544846,
  0.0008143654,
  0.03365826,
  -0.004592433,
  0.006341932,
  -0.015118799,
  0.0000065343543,
  -0.009002083,
  -0.0012272601,
  0.0011831962,
  -0.016058829,
  -0.016776906,
  -0.010686302,
  -0.0006944138,
  -0.017364424,
  -0.03188265,
  -0.014257106,
  0.021085372,
  0.013591252,
  0.0028478308,
  0.01560187,
  -0.013982931,
  -0.0051636314,
  0.014714065,
  -0.005630382,
  -0.009224035,
  -0.024714928,
  0.0076638474,
  0.008362341,
  -0.019179203,
  -0.00018502741,
  0.005548782,
  -0.02412741,
  -0.014204882,
  0.0062864437,
  0.014753233,
  0.025106607,
  0.022991542,
  0.011783002,
  -0.0055389903,
  0.012690391,
  -0.031438746,
  -0.03136041,
  0.01844807,
  0.010053088,
  -0.009628769,
  -0.006821738,
  -0.0061558844,
  0.01799111,
  0.009870305,
  -0.009687521,
  -0.032744344,
  -0.020693693,
  0.01961005,
  -0.0016034349,
  0.029114787,
  -0.004674033,
  -0.0076050954,
  -0.0063810996,
  0.01812167,
  -0.00072623766,
  0.013108182,
  -0.013153878,
  -0.009831136,
  0.010131423,
  -0.0056662858,
  -0.012651224,
  -0.021411771,
  0.02038035,
  0.004660977,
  -0.011391323,
  0.012266072,
  -0.026203308,
  0.0005483502,
  0.01844807,
  -0.020171456,
  -0.00025928317,
  0.007193833,
  -0.0006805418,
  -0.01792583,
  -0.00018390542,
  -0.020093119,
  -0.014296274,
  0.0029637024,
  -0.007108969,
  0.012664279,
  -0.0029212707,
  0.013323605,
  -0.004625073,
  -0.03159542,
  -0.0009506369,
  -0.017233865,
  0.0007184857,
  -0.018800579,
  0.001213388,
  0.017403591,
  -0.018356677,
  -0.030394271,
  -0.033867154,
  -0.0024430961,
  0.0010632445,
  -0.027756967,
  -0.013297494,
  0.018839747,
  -0.0013504757,
  -0.028932003,
  -0.007676903,
  -0.16857854,
  0.022808759,
  0.006596523,
  -0.0136957,
  0.0033080534,
  -0.0017413384,
  0.040316798,
  -0.010967005,
  -0.016163277,
  0.00618526,
  0.030342046,
  0.0024496242,
  -0.033632148,
  -0.0011440284,
  0.009067363,
  0.0013610836,
  -0.005209327,
  0.002834775,
  0.004037555,
  0.021281213,
  0.0039592194,
  -0.019166147,
  0.036086667,
  -0.0075985678,
  0.008473317,
  0.00044675855,
  0.028044198,
  0.019518657,
  0.0026405675,
  -0.013251797,
  -0.036008332,
  -0.021790395,
  0.006527979,
  0.0061003966,
  0.031438746,
  -0.0005675262,
  0.017899718,
  -0.03720948,
  -0.0026813673,
  0.028540324,
  0.016111052,
  0.040812925,
  -0.009426402,
  0.0024137204,
  -0.011521883,
  0.021085372,
  0.0020269374,
  -0.002181977,
  0.016659403,
  -0.004093043,
  0.0058001094,
  -0.011019229,
  0.020693693,
  -0.005917613,
  0.031151516,
  0.01647662,
  -0.011097564,
  0.021437883,
  0.0035871244,
  0.012696919,
  -0.0029620705,
  -0.019518657,
  -0.023801012,
  -0.01233788,
  0.003472885,
  -0.017442761,
  -0.01695969,
  -0.0035675406,
  -0.008388453,
  0.011508827,
  0.013800148,
  -0.014883792,
  0.016711626,
  -0.003737268,
  0.022874039,
  0.0153668625,
  -0.027548071,
  0.014387666,
  0.0060710204,
  -0.009752801,
  -0.014505169,
  0.014022099,
  -0.017481929,
  -0.018513348,
  0.0049188323,
  0.02861866,
  0.016319947,
  -0.00079478143,
  0.01014448,
  -0.015105744,
  0.010477406,
  -0.014531282,
  -0.010033504,
  -0.035094414,
  0.012612056,
  0.010967005,
  -0.003066518,
  0.008845411,
  -0.0023631284,
  -0.02261292,
  0.00019594138,
  -0.00023011127,
  -0.0020710013,
  -0.013095126,
  0.008727908,
  -0.004608753,
  0.0069849375,
  0.020863421,
  0.039193988,
  -0.014635729,
  -0.0071154973,
  0.034702737,
  0.034546066,
  0.034728847,
  -0.0055291983,
  0.026111916,
  -0.024532145,
  -0.027313065,
  -0.0008771972,
  -0.0029539105,
  0.058542915,
  -0.019192258,
  0.007892326,
  0.014518226,
  0.004347634,
  -0.013160406,
  -0.083505906,
  -0.011978841,
  -0.000059618807,
  0.014635729,
  -0.012664279,
  0.007859686,
  0.010033504,
  0.022495415,
  -0.024362419,
  0.014544337,
  -0.005914349,
  -0.029323682,
  -0.027078057,
  -0.02035424,
  0.024218803,
  -0.009524321,
  -0.009733217,
  -0.009583074,
  0.001558555,
  0.002771127,
  0.005068976,
  -0.004964528,
  0.0055291983,
  -0.018957252,
  -0.028226981,
  -0.0013219158,
  -0.014426834,
  0.025446063,
  0.010947421,
  0.005865389,
  -0.0041452665,
  -0.020471742,
  0.01853946,
  -0.0027368553,
  0.0042301305,
  0.002539384,
  -0.01628078,
  -0.007840103,
  0.012918871,
  -0.0127556715,
  0.0144007215,
  -0.018108614,
  -0.005046128,
  -0.037914503,
  0.025315503,
  0.006550827,
  -0.0030028704,
  -0.0014206514,
  0.007141609,
  -0.05311164,
  -0.03475496,
  -0.0033782292,
  -0.020876477,
  -0.0020856892,
  0.025981357,
  0.0049090404,
  0.008251365,
  0.010483935,
  -0.020850366,
  0.009687521,
  0.019570882,
  0.002134649,
  -0.03159542,
  0.021228988,
  0.00494168,
  -0.00684785,
  -0.017938886,
  -0.0041974904,
  0.021790395,
  -0.011058397,
  -0.0075332876,
  0.030629277,
  -0.009700577,
  0.015458254,
  -0.035773326,
  -0.011358684,
  -0.015706317,
  -0.016816074,
  0.021777337,
  -0.015641037,
  -0.021281213,
  -0.027025834,
  0.013656532,
  -0.012847063,
  0.035799436,
  0.013721812,
  0.020341182,
  -0.010346847,
  0.011489243,
  -0.023344053,
  0.0024382002,
  0.028696995,
  -0.008251365,
  -0.018395845,
  0.0054704463,
  0.018891972,
  0.0033390613,
  -0.0036230283,
  0.020080063,
  0.0039690114,
  -0.002828247,
  -0.008982499,
  -0.059639618,
  0.025198,
  -0.009511266,
  -0.0067303465,
  0.013173462,
  -0.017299145,
  0.017064137,
  -0.013813204,
  -0.005943725,
  -0.033632148,
  -0.006873962,
  0.011032284,
  -0.0014883792,
  0.011626331,
  -0.035668876,
  -0.021150652,
  0.04222297,
  0.03136041,
  0.013512917,
  0.008571236,
  0.008408037,
  0.0022766327,
  0.029402018,
  -0.006926186,
  -0.004716465,
  0.0041420027,
  -0.016254667,
  0.009413346,
  -0.008035942,
  -0.029245347,
  0.01014448,
  -0.040186238,
  0.0036262923,
  0.02261292,
  0.017220808,
  -0.007219945,
  -0.011352155,
  0.029349795,
  0.025041327,
  0.018278342,
  -0.037000585,
  -0.025459118,
  0.0030159263,
  -0.024401587,
  -0.023357108,
  0.00943293,
  -0.031987097,
  -0.0061101886,
  0.019322818,
  0.0074288403,
  0.009818081,
  0.008727908,
  -0.00006777878,
  0.005330095,
  0.0015283631,
  -0.025916077,
  0.013003734,
  0.0060220608,
  -0.01547131,
  -0.022730423,
  0.029036451,
  0.024714928,
  0.009106531,
  -0.0063223476,
  -0.004951472,
  0.0011040445,
  0.0018245701,
  -0.016946634,
  0.001860474,
  -0.06261638,
  -0.038488965,
  0.014949072,
  0.0019371778,
  0.026738603,
  -0.00061077403,
  0.010053088,
  0.00006946178,
  -0.005202799,
  -0.032639895,
  0.03240489,
  0.0035512205,
  -0.014152658,
  -0.020902589,
  0.019805888,
  0.016554955,
  0.005339887,
  -0.00923709,
  -0.004008179,
  -0.0054378067,
  0.009511266,
  -0.045304175,
  -0.0038482437,
  0.007905383,
  -0.008780132,
  -0.011893977,
  0.0034924687,
  0.0061819963,
  -0.0019420738,
  0.017899718,
  0.015836878,
  -0.014661841,
  0.0016091468,
  -0.014361554,
  0.0064006834,
  -0.014322386,
  0.019388098,
  -0.001853946,
  -0.008812772,
  0.013539028,
  0.014674897,
  0.022795703,
  0.010790749,
  0.0019371778,
  0.021072317,
  -0.008923748,
  0.006002477,
  -0.0057837893,
  -0.010914781,
  -0.019414209,
  0.03653057,
  0.015876045,
  0.040917374,
  -0.009791968,
  0.004964528,
  0.0037699079,
  -0.0049710562,
  -0.0021020093,
  -0.038384516,
  0.011698139,
  -0.013042903,
  0.00694577,
  -0.014661841,
  -0.016672458,
  -0.027156392,
  -0.016084941,
  -0.0031840217,
  -0.0018588421,
  0.01641134,
  -0.028827555,
  0.088258274,
  0.012696919,
  0.0033651732,
  0.0106210215,
  0.031543195,
  0.018591685,
  0.015719373,
  0.008440677,
  -0.013995987,
  -0.013591252,
  0.011880922,
  -0.015210191,
  -0.0011285244,
  -0.014479058,
  -0.009935584,
  0.013460693,
  -0.01702497,
  0.0005169343,
  -0.009250146,
  0.0037046282,
  0.029323682,
  0.0059731007,
  0.015118799,
  0.007422312,
  -0.010477406,
  0.0026356715,
  -0.0017641863,
  -0.0011505563,
  -0.035747215,
  -0.020798141,
  0.009785441,
  -0.0023370164,
  -0.025641901,
  -0.00429541,
  0.005209327,
  -0.0030273504,
  -0.011051869,
  -0.0016581067,
  -0.023122102,
  0.0038776195,
  -0.0035708046,
  0.014139603,
  -0.0075920396,
  -0.02642526,
  0.04522584,
  -0.0033553813,
  -0.0019796097,
  -0.0054214867,
  -0.00085516524,
];
