/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.elasticsearch.cluster.routing.allocation;

import static org.assertj.core.api.Assertions.assertThat;
import static org.elasticsearch.cluster.metadata.IndexMetadata.SETTING_INDEX_UUID;
import static org.elasticsearch.cluster.routing.ShardRoutingState.INITIALIZING;
import static org.elasticsearch.cluster.routing.ShardRoutingState.STARTED;
import static org.elasticsearch.cluster.routing.ShardRoutingState.UNASSIGNED;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.Version;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ESAllocationTestCase;
import org.elasticsearch.cluster.metadata.AutoExpandReplicas;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.routing.RoutingNodes;
import org.elasticsearch.cluster.routing.RoutingTable;
import org.elasticsearch.cluster.routing.allocation.decider.ClusterRebalanceAllocationDecider;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.settings.Settings;

public class IndexBalanceTests extends ESAllocationTestCase {

    private final Logger logger = LogManager.getLogger(IndexBalanceTests.class);

    private Settings.Builder indexSettings() {
        return settings(Version.CURRENT)
            // Crate will lookup by index UUID by default, thus a unique UUID must be set explicitly
            // (see https://github.com/crate/crate/commit/517691c7fb4a58fe01428ee2b30f9c0577c27b91)
            .put(SETTING_INDEX_UUID, UUIDs.randomBase64UUID())
            // Crate defaults to '0-1' for auto expand replicas, we must explicitly disable this here
            .put(AutoExpandReplicas.SETTING.getKey(), "false");
    }

    public void testBalanceAllNodesStarted() {
        AllocationService strategy = createAllocationService(Settings.builder()
            .put("cluster.routing.allocation.node_concurrent_recoveries", 10)
            .put("cluster.routing.allocation.node_initial_primaries_recoveries", 10)
            .put(ClusterRebalanceAllocationDecider.CLUSTER_ROUTING_ALLOCATION_ALLOW_REBALANCE_SETTING.getKey(), "always")
            .put("cluster.routing.allocation.cluster_concurrent_rebalance", -1).build());

        logger.info("Building initial routing table");

        Metadata metadata = Metadata.builder()
            .put(IndexMetadata.builder("test")
                .settings(indexSettings())
                .numberOfShards(3)
                .numberOfReplicas(1))
            .put(IndexMetadata.builder("test1")
                .settings(indexSettings())
                .numberOfShards(3)
                .numberOfReplicas(1))
            .build();

        RoutingTable initialRoutingTable = RoutingTable.builder()
            .addAsNew(metadata.index("test")).addAsNew(metadata.index("test1")).build();

        ClusterState clusterState = ClusterState.builder(org.elasticsearch.cluster.ClusterName.CLUSTER_NAME_SETTING
            .getDefault(Settings.EMPTY)).metadata(metadata).routingTable(initialRoutingTable).build();

        assertThat(clusterState.routingTable().index("test").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards().get(0).state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test").shard(i).shards().get(1).state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test").shard(i).shards().get(0).currentNodeId()).isNull();
            assertThat(clusterState.routingTable().index("test").shard(i).shards().get(1).currentNodeId()).isNull();
        }

        assertThat(clusterState.routingTable().index("test1").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test1").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test1").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).shards().get(0).state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test1").shard(i).shards().get(1).state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test1").shard(i).shards().get(0).currentNodeId()).isNull();
            assertThat(clusterState.routingTable().index("test1").shard(i).shards().get(1).currentNodeId()).isNull();
        }

        logger.info("Adding three node and performing rerouting");
        clusterState = ClusterState.builder(clusterState)
            .nodes(DiscoveryNodes.builder().add(newNode("node1")).add(newNode("node2"))
                .add(newNode("node3"))).build();

        ClusterState newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;

        assertThat(clusterState.routingTable().index("test").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).primaryShard().state()).isEqualTo(INITIALIZING);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards()).hasSize(1);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards().getFirst().state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards().getFirst().currentNodeId()).isNull();
        }

        logger.info("Another round of rebalancing");
        clusterState = ClusterState.builder(clusterState).nodes(DiscoveryNodes.builder(clusterState.nodes())).build();
        newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isEqualTo(clusterState);

        newState = startInitializingShardsAndReroute(strategy, clusterState);
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;
        assertThat(clusterState.routingTable().index("test").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).primaryShard().state()).isEqualTo(STARTED);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards()).hasSize(1);
            // backup shards are initializing as well, we make sure that they
            // recover from primary *started* shards in the
            // IndicesClusterStateService
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards().getFirst().state()).isEqualTo(INITIALIZING);
        }

        logger.info("Reroute, nothing should change");
        newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isEqualTo(clusterState);

        logger.info("Start the more shards");
        newState = startInitializingShardsAndReroute(strategy, clusterState);
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;
        RoutingNodes routingNodes = clusterState.getRoutingNodes();

        assertThat(clusterState.routingTable().index("test").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).primaryShard().state()).isEqualTo(STARTED);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards()).hasSize(1);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards().getFirst().state()).isEqualTo(STARTED);
        }
        assertThat(clusterState.routingTable().index("test1").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test1").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test1").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).primaryShard().state()).isEqualTo(STARTED);
            assertThat(clusterState.routingTable().index("test1").shard(i).replicaShards()).hasSize(1);
            assertThat(clusterState.routingTable().index("test1").shard(i).replicaShards().getFirst().state()).isEqualTo(STARTED);
        }

        assertThat(routingNodes.node("node1").numberOfShardsWithState(STARTED)).isEqualTo(4);
        assertThat(routingNodes.node("node2").numberOfShardsWithState(STARTED)).isEqualTo(4);
        assertThat(routingNodes.node("node3").numberOfShardsWithState(STARTED)).isEqualTo(4);

        assertThat(routingNodes.node("node1").shardsWithState("test", STARTED)).hasSize(2);
        assertThat(routingNodes.node("node2").shardsWithState("test", STARTED)).hasSize(2);
        assertThat(routingNodes.node("node3").shardsWithState("test", STARTED)).hasSize(2);

        assertThat(routingNodes.node("node1").shardsWithState("test1", STARTED)).hasSize(2);
        assertThat(routingNodes.node("node2").shardsWithState("test1", STARTED)).hasSize(2);
        assertThat(routingNodes.node("node3").shardsWithState("test1", STARTED)).hasSize(2);
    }

    public void testBalanceIncrementallyStartNodes() {
        AllocationService strategy = createAllocationService(Settings.builder()
            .put("cluster.routing.allocation.node_concurrent_recoveries", 10)
            .put("cluster.routing.allocation.node_initial_primaries_recoveries", 10)
            .put(ClusterRebalanceAllocationDecider.CLUSTER_ROUTING_ALLOCATION_ALLOW_REBALANCE_SETTING.getKey(), "always")
            .put("cluster.routing.allocation.cluster_concurrent_rebalance", -1).build());

        logger.info("Building initial routing table");

        Metadata metadata = Metadata.builder()
            .put(IndexMetadata.builder("test")
                .settings(indexSettings())
                .numberOfShards(3)
                .numberOfReplicas(1))
            .put(IndexMetadata.builder("test1")
                .settings(indexSettings())
                .numberOfShards(3)
                .numberOfReplicas(1))
            .build();

        RoutingTable initialRoutingTable = RoutingTable.builder()
            .addAsNew(metadata.index("test")).addAsNew(metadata.index("test1")).build();

        ClusterState clusterState = ClusterState.builder(org.elasticsearch.cluster.ClusterName.CLUSTER_NAME_SETTING
            .getDefault(Settings.EMPTY)).metadata(metadata).routingTable(initialRoutingTable).build();

        assertThat(clusterState.routingTable().index("test").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards().get(0).state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test").shard(i).shards().get(1).state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test").shard(i).shards().get(0).currentNodeId()).isNull();
            assertThat(clusterState.routingTable().index("test").shard(i).shards().get(1).currentNodeId()).isNull();
        }

        assertThat(clusterState.routingTable().index("test1").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test1").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test1").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).shards().get(0).state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test1").shard(i).shards().get(1).state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test1").shard(i).shards().get(0).currentNodeId()).isNull();
            assertThat(clusterState.routingTable().index("test1").shard(i).shards().get(1).currentNodeId()).isNull();
        }

        logger.info("Adding one node and performing rerouting");
        clusterState = ClusterState.builder(clusterState).nodes(DiscoveryNodes.builder().add(newNode("node1"))).build();

        ClusterState newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;
        assertThat(clusterState.routingTable().index("test").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).primaryShard().state()).isEqualTo(INITIALIZING);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards()).hasSize(1);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards().getFirst().state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards().getFirst().currentNodeId()).isNull();
        }

        logger.info("Add another node and perform rerouting, nothing will happen since primary not started");
        clusterState = ClusterState.builder(clusterState)
            .nodes(DiscoveryNodes.builder(clusterState.nodes()).add(newNode("node2"))).build();
        newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isEqualTo(clusterState);


        logger.info("Start the primary shard");
        newState = startInitializingShardsAndReroute(strategy, clusterState);
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;
        assertThat(clusterState.routingTable().index("test").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).primaryShard().state()).isEqualTo(STARTED);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards()).hasSize(1);
            // backup shards are initializing as well, we make sure that they
            // recover from primary *started* shards in the
            // IndicesClusterStateService
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards().getFirst().state()).isEqualTo(INITIALIZING);
        }

        logger.info("Reroute, nothing should change");
        newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isEqualTo(clusterState);

        logger.info("Start the backup shard");
        newState = startInitializingShardsAndReroute(strategy, clusterState);
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;

        assertThat(clusterState.routingTable().index("test").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).primaryShard().state()).isEqualTo(STARTED);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards()).hasSize(1);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards().getFirst().state()).isEqualTo(STARTED);
        }
        assertThat(clusterState.routingTable().index("test1").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test1").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test1").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).primaryShard().state()).isEqualTo(STARTED);
            assertThat(clusterState.routingTable().index("test1").shard(i).replicaShards()).hasSize(1);
            assertThat(clusterState.routingTable().index("test1").shard(i).replicaShards().getFirst().state()).isEqualTo(STARTED);
        }

        logger.info("Add another node and perform rerouting, nothing will happen since primary not started");
        clusterState = ClusterState.builder(clusterState)
            .nodes(DiscoveryNodes.builder(clusterState.nodes()).add(newNode("node3"))).build();
        newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;

        logger.info("Reroute, nothing should change");
        newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isEqualTo(clusterState);

        logger.info("Start the backup shard");
        newState = startInitializingShardsAndReroute(strategy, clusterState);
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;

        assertThat(clusterState.routingTable().index("test").shards()).hasSize(3);

        newState = startInitializingShardsAndReroute(strategy, clusterState);
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;
        RoutingNodes routingNodes = clusterState.getRoutingNodes();

        assertThat(clusterState.routingTable().index("test1").shards()).hasSize(3);

        assertThat(routingNodes.node("node1").numberOfShardsWithState(STARTED)).isEqualTo(4);
        assertThat(routingNodes.node("node2").numberOfShardsWithState(STARTED)).isEqualTo(4);
        assertThat(routingNodes.node("node3").numberOfShardsWithState(STARTED)).isEqualTo(4);

        assertThat(routingNodes.node("node1").shardsWithState("test", STARTED)).hasSize(2);
        assertThat(routingNodes.node("node2").shardsWithState("test", STARTED)).hasSize(2);
        assertThat(routingNodes.node("node3").shardsWithState("test", STARTED)).hasSize(2);

        assertThat(routingNodes.node("node1").shardsWithState("test1", STARTED)).hasSize(2);
        assertThat(routingNodes.node("node2").shardsWithState("test1", STARTED)).hasSize(2);
        assertThat(routingNodes.node("node3").shardsWithState("test1", STARTED)).hasSize(2);
    }

    public void testBalanceAllNodesStartedAddIndex() {
        AllocationService strategy = createAllocationService(Settings.builder()
            .put("cluster.routing.allocation.node_concurrent_recoveries", 10)
            .put("cluster.routing.allocation.node_initial_primaries_recoveries", 10)
            .put(ClusterRebalanceAllocationDecider.CLUSTER_ROUTING_ALLOCATION_ALLOW_REBALANCE_SETTING.getKey(), "always")
            .put("cluster.routing.allocation.cluster_concurrent_rebalance", -1).build());

        logger.info("Building initial routing table");

        Metadata metadata = Metadata.builder()
            .put(IndexMetadata.builder("test")
                .settings(indexSettings())
                .numberOfShards(3)
                .numberOfReplicas(1))
            .build();

        RoutingTable initialRoutingTable = RoutingTable.builder().addAsNew(metadata.index("test")).build();

        ClusterState clusterState = ClusterState.builder(org.elasticsearch.cluster.ClusterName.CLUSTER_NAME_SETTING
            .getDefault(Settings.EMPTY)).metadata(metadata).routingTable(initialRoutingTable).build();

        assertThat(clusterState.routingTable().index("test").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards().get(0).state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test").shard(i).shards().get(1).state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test").shard(i).shards().get(0).currentNodeId()).isNull();
            assertThat(clusterState.routingTable().index("test").shard(i).shards().get(1).currentNodeId()).isNull();
        }

        logger.info("Adding three node and performing rerouting");
        clusterState = ClusterState.builder(clusterState)
            .nodes(DiscoveryNodes.builder().add(newNode("node1")).add(newNode("node2")).add(newNode("node3"))).build();

        ClusterState newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;
        assertThat(clusterState.routingTable().index("test").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).primaryShard().state()).isEqualTo(INITIALIZING);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards()).hasSize(1);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards().getFirst().state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards().getFirst().currentNodeId()).isNull();
        }

        logger.info("Another round of rebalancing");
        clusterState = ClusterState.builder(clusterState).nodes(DiscoveryNodes.builder(clusterState.nodes())).build();
        newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isEqualTo(clusterState);


        newState = startInitializingShardsAndReroute(strategy, clusterState);
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;
        assertThat(clusterState.routingTable().index("test").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).primaryShard().state()).isEqualTo(STARTED);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards()).hasSize(1);
            // backup shards are initializing as well, we make sure that they
            // recover from primary *started* shards in the
            // IndicesClusterStateService
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards().getFirst().state()).isEqualTo(INITIALIZING);
        }

        logger.info("Reroute, nothing should change");
        newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isEqualTo(clusterState);

        logger.info("Start the more shards");
        newState = startInitializingShardsAndReroute(strategy, clusterState);
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;
        RoutingNodes routingNodes = clusterState.getRoutingNodes();
        assertThat(clusterState.routingTable().index("test").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test").shard(i).primaryShard().state()).isEqualTo(STARTED);
            assertThat(clusterState.routingTable().index("test").shard(i).replicaShards()).hasSize(1);
        }
        assertThat(routingNodes.node("node1").numberOfShardsWithState(STARTED)).isEqualTo(2);
        assertThat(routingNodes.node("node2").numberOfShardsWithState(STARTED)).isEqualTo(2);
        assertThat(routingNodes.node("node3").numberOfShardsWithState(STARTED)).isEqualTo(2);

        assertThat(routingNodes.node("node1").shardsWithState("test", STARTED)).hasSize(2);
        assertThat(routingNodes.node("node2").shardsWithState("test", STARTED)).hasSize(2);
        assertThat(routingNodes.node("node3").shardsWithState("test", STARTED)).hasSize(2);

        logger.info("Add new index 3 shards 1 replica");

        Metadata updatedMetadata = Metadata.builder(clusterState.metadata())
            .put(IndexMetadata.builder("test1").settings(settings(Version.CURRENT)
                .put(IndexMetadata.SETTING_NUMBER_OF_SHARDS, 3)
                .put(IndexMetadata.SETTING_NUMBER_OF_REPLICAS, 1)
            ))
            .build();
        RoutingTable updatedRoutingTable = RoutingTable.builder(clusterState.routingTable())
            .addAsNew(updatedMetadata.index("test1"))
            .build();
        clusterState = ClusterState.builder(clusterState).metadata(updatedMetadata).routingTable(updatedRoutingTable).build();


        assertThat(clusterState.routingTable().index("test1").shards()).hasSize(3);

        newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;
        assertThat(clusterState.routingTable().index("test1").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test1").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test1").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).primaryShard().state()).isEqualTo(INITIALIZING);
            assertThat(clusterState.routingTable().index("test1").shard(i).replicaShards()).hasSize(1);
            assertThat(clusterState.routingTable().index("test1").shard(i).replicaShards().getFirst().state()).isEqualTo(UNASSIGNED);
            assertThat(clusterState.routingTable().index("test1").shard(i).replicaShards().getFirst().currentNodeId()).isNull();
        }

        logger.info("Another round of rebalancing");
        clusterState = ClusterState.builder(clusterState).nodes(DiscoveryNodes.builder(clusterState.nodes())).build();
        newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isEqualTo(clusterState);


        newState = startInitializingShardsAndReroute(strategy, clusterState);
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;
        assertThat(clusterState.routingTable().index("test1").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test1").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test1").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).primaryShard().state()).isEqualTo(STARTED);
            assertThat(clusterState.routingTable().index("test1").shard(i).replicaShards()).hasSize(1);
            // backup shards are initializing as well, we make sure that they
            // recover from primary *started* shards in the
            // IndicesClusterStateService
            assertThat(clusterState.routingTable().index("test1").shard(i).replicaShards().getFirst().state()).isEqualTo(INITIALIZING);
        }

        logger.info("Reroute, nothing should change");
        newState = strategy.reroute(clusterState, "reroute");
        assertThat(newState).isEqualTo(clusterState);

        logger.info("Start the more shards");
        newState = startInitializingShardsAndReroute(strategy, clusterState);
        assertThat(newState).isNotEqualTo(clusterState);
        clusterState = newState;
        routingNodes = clusterState.getRoutingNodes();
        assertThat(clusterState.routingTable().index("test1").shards()).hasSize(3);
        for (int i = 0; i < clusterState.routingTable().index("test1").shards().size(); i++) {
            assertThat(clusterState.routingTable().index("test1").shard(i)).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).shards()).hasSize(2);
            assertThat(clusterState.routingTable().index("test1").shard(i).primaryShard().state()).isEqualTo(STARTED);
            assertThat(clusterState.routingTable().index("test1").shard(i).replicaShards()).hasSize(1);
        }
        assertThat(routingNodes.node("node1").numberOfShardsWithState(STARTED)).isEqualTo(4);
        assertThat(routingNodes.node("node2").numberOfShardsWithState(STARTED)).isEqualTo(4);
        assertThat(routingNodes.node("node3").numberOfShardsWithState(STARTED)).isEqualTo(4);

        assertThat(routingNodes.node("node1").shardsWithState("test1", STARTED)).hasSize(2);
        assertThat(routingNodes.node("node2").shardsWithState("test1", STARTED)).hasSize(2);
        assertThat(routingNodes.node("node3").shardsWithState("test1", STARTED)).hasSize(2);
    }
}
