/*
 * Licensed to Crate.io GmbH ("Crate") under one or more contributor
 * license agreements.  See the NOTICE file distributed with this work for
 * additional information regarding copyright ownership.  Crate licenses
 * this file to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.  You may
 * obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 * However, if you have executed another commercial license agreement
 * with Crate these terms will supersede the license and you may use the
 * software solely pursuant to the terms of the relevant commercial agreement.
 */

package io.crate.lucene;

import java.lang.reflect.Method;

import org.apache.lucene.search.Query;
import org.elasticsearch.Version;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.rules.TestName;

import io.crate.expression.symbol.Symbol;
import io.crate.test.integration.CrateDummyClusterServiceUnitTest;
import io.crate.testing.IndexVersionCreated;
import io.crate.testing.QueryTester;

public abstract class LuceneQueryBuilderTest extends CrateDummyClusterServiceUnitTest {

    @Rule
    public TestName testName = new TestName();

    protected QueryTester queryTester;

    @Before
    public void prepare() throws Exception {
        QueryTester.Builder builder = new QueryTester.Builder(
            THREAD_POOL,
            clusterService,
            indexVersion(),
            createStmt()
        );
        queryTester = builder.build();
    }

    @After
    public void tearDownQueryTester() throws Exception {
        queryTester.close();
    }

    protected Query convert(String expression) {
        return queryTester.toQuery(expression);
    }

    protected Query convert(String expression, Object ... params) {
        return queryTester.toQuery(expression, params);
    }

    protected Query convert(Symbol expression) {
        return queryTester.toQuery(expression);
    }

    protected String createStmt() {
        return "create table users (" +
               " name string," +
               " tags string index using fulltext not null," +
               " content string index using fulltext," +
               " text_no_index text index off storage with (columnstore = false)," +
               " x integer not null," +
               " x_no_docvalues int storage with (columnstore = false)," +
               " y integer," +
               " f float," +
               " d double," +
               " obj object as (" +
               "     x integer," +
               "     y integer," +
               "     o_array array(object as (x int))" +
               " )," +
               " obj_no_sub_columns object," +
               " obj_ignored object (ignored), " +
               " d_array array(double)," +
               " y_array array(long)," +
               " x_array_no_docvalues array(int) storage with (columnstore = false)," +
               " o_array array(object as (xs array(integer), obj object as (x int), o_array_2 array(object as (x int))))," +
               " ts_array array(timestamp with time zone)," +
               " string_array array(string)," +
               " shape geo_shape," +
               " bkd_shape geo_shape index using bkdtree," +
               " point geo_point," +
               " ts timestamp with time zone," +
               " addr ip," +
               " vchar_name varchar(40)," +
               " byte_col byte, " +
               " bool_col boolean, " +
               " d_array_index_off_no_docvalues array(double) index off storage with (columnstore = false) " +
               ")";
    }

    private Version indexVersion() {
        try {
            Class<?> clazz = this.getClass();
            Method method = clazz.getMethod(testName.getMethodName());
            IndexVersionCreated annotation = method.getAnnotation(IndexVersionCreated.class);
            if (annotation == null) {
                annotation = clazz.getAnnotation(IndexVersionCreated.class);
                if (annotation == null) {
                    return Version.CURRENT;
                }
            }
            int value = annotation.value();
            if (value == -1) {
                return Version.CURRENT;
            }
            return Version.fromId(value);
        } catch (NoSuchMethodException ignored) {
            return Version.CURRENT;
        }
    }
}
