/*
 * Licensed to Crate.io GmbH ("Crate") under one or more contributor
 * license agreements.  See the NOTICE file distributed with this work for
 * additional information regarding copyright ownership.  Crate licenses
 * this file to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.  You may
 * obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 * However, if you have executed another commercial license agreement
 * with Crate these terms will supersede the license and you may use the
 * software solely pursuant to the terms of the relevant commercial agreement.
 */

package io.crate.integrationtests;

import static io.crate.protocols.postgres.PGErrorStatus.INTERNAL_ERROR;
import static io.crate.testing.Asserts.assertThat;
import static io.netty.handler.codec.http.HttpResponseStatus.BAD_REQUEST;

import java.util.ArrayList;
import java.util.List;

import org.elasticsearch.test.IntegTestCase;
import org.junit.Test;

import io.crate.testing.Asserts;

public class SysClusterTest extends IntegTestCase {

    @Test
    public void testSysCluster() throws Exception {
        execute("select id from sys.cluster");
        assertThat(response).hasRowCount(1L);
        assertThat(((String) response.rows()[0][0]).length()).isEqualTo(22); // looks like a uuid generated by UUIDs.randomBase64UUID
    }

    public void testSysClusterMasterNode() throws Exception {
        execute("select id from sys.nodes");
        List<String> nodes = new ArrayList<>();
        for (Object[] nodeId : response.rows()) {
            nodes.add((String) nodeId[0]);
        }

        execute("select master_node from sys.cluster");
        assertThat(response.rowCount()).isEqualTo(1L);
        String node = (String) response.rows()[0][0];
        assertThat(nodes.contains(node)).isTrue();
    }

    @Test
    public void testExplainSysCluster() throws Exception {
        execute("explain (costs false) select * from sys.cluster limit 2");
        assertThat(response.rowCount()).isEqualTo(1L);
        assertThat(response).hasRows(
            "Limit[2::bigint;0]\n" +
            "  └ Collect[sys.cluster | [id, license, master_node, name, settings] | true]");
    }

    @Test
    public void testScalarEvaluatesInErrorOnSysCluster() throws Exception {
        Asserts.assertSQLError(() -> execute("select 1/0 from sys.cluster"))
            .hasPGError(INTERNAL_ERROR)
            .hasHTTPError(BAD_REQUEST, 4000)
            .hasMessageContaining("/ by zero");
    }
}
