/*
 * Licensed to Crate.io GmbH ("Crate") under one or more contributor
 * license agreements.  See the NOTICE file distributed with this work for
 * additional information regarding copyright ownership.  Crate licenses
 * this file to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.  You may
 * obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 * However, if you have executed another commercial license agreement
 * with Crate these terms will supersede the license and you may use the
 * software solely pursuant to the terms of the relevant commercial agreement.
 */

package io.crate.common.collections;

import java.util.Arrays;
import java.util.Iterator;
import java.util.NoSuchElementException;

public final class RingBuffer<T> implements Iterable<T> {

    private final T[] buf;
    private int pos = 0;

    @SuppressWarnings("unchecked")
    public RingBuffer(int size) {
        this.buf = (T[]) new Object[size];
    }

    public void add(T x) {
        buf[pos % buf.length] = x;
        pos++;
    }

    public void reset() {
        pos = 0;
        Arrays.fill(buf, null);
    }

    @Override
    public Iterator<T> iterator() {
        return new Iterator<>() {

            int idx = -1;

            @Override
            public boolean hasNext() {
                return idx + 1 < buf.length && buf[idx + 1] != null;
            }

            @Override
            public T next() {
                if (!hasNext()) {
                    throw new NoSuchElementException("RingBuffer iterator has no more elements");
                }
                idx++;
                return buf[idx];
            }
        };
    }
}
