#!/usr/bin/env python3

# Licensed to Crate.io GmbH ("Crate") under one or more contributor
# license agreements.  See the NOTICE file distributed with this work for
# additional information regarding copyright ownership.  Crate licenses
# this file to you under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.  You may
# obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
# License for the specific language governing permissions and limitations
# under the License.
#
# However, if you have executed another commercial license agreement
# with Crate these terms will supersede the license and you may use the
# software solely pursuant to the terms of the relevant commercial agreement.

"""Script to generate a keystore with node and client certificates.

Requires keystore and openssl to be available in $PATH
"""

import os
import argparse
from os.path import join, splitext, basename
from subprocess import run


def int_or(val, default):
    if val:
        return int(val)
    return default


def create_key_and_csr(key, csr):
    cn = splitext(basename(csr))[0]
    run([
        'openssl', 'req', '-newkey', 'rsa:2048', '-nodes',
        '-subj', f'/C=AT/ST=Dummy State/L=Dummy Country/O=Dummy Company/CN={cn}',
        '-keyout', key,
        '-out', csr
    ])


def create_crt(csr, crt, root_ca_crt, root_ca_key, out_dir):
    cn = splitext(basename(csr))[0]
    ssl_ext_template = f"""authorityKeyIdentifier=keyid,issuer
basicConstraints=CA:FALSE
keyUsage = digitalSignature, nonRepudiation, keyEncipherment, dataEncipherment
subjectAltName = @alt_names

[alt_names]
DNS.1 = {cn}
"""
    with open(join(out_dir, 'ssl.ext'),'w') as f:
        f.write(ssl_ext_template)

    run(['openssl', 'x509', '-req',
         '-in', csr,
         '-CA', root_ca_crt,
         '-CAkey', root_ca_key,
         '-CAcreateserial',
         '-out', crt,
         '-sha256',
         '-days', '365',
         '-extfile', join(out_dir, 'ssl.ext')
    ])


def generate_for(root_ca_key, root_ca_crt, out_dir, entity, num_default):
    num = int_or(input(f'How many {entity} certs do you want to generate? [{num_default}]: '), num_default)
    certs_and_keys = []
    for i in range(num):
        name = entity + str(i + 1)
        supplied_name = input(f'Name (CN) of {entity} {i + 1} [{name}]: ')
        name = supplied_name or name

        key = join(out_dir, name + '.key')
        csr = join(out_dir, name + '.csr')
        crt = join(out_dir, name + '.crt')
        certs_and_keys.append((crt, key))
        print(f'Creating {entity} key, csr and cert for {name}')
        create_key_and_csr(key, csr)
        create_crt(csr, crt, root_ca_crt, root_ca_key, out_dir)
    print('')
    print('')
    return certs_and_keys


def import_into_keystores(certs_and_keys, entity, keystore_pw, ca_crt, keystore, keystorep12):
    print(f'Importing {entity} certificates into keystore, Use "{keystore_pw}" as pw.')
    for (cert, key) in certs_and_keys:
        run([
            'openssl', 'pkcs12', '-export',
            '-in', cert,
            '-inkey', key,
            '-out', keystorep12,
            '-name', splitext(cert)[0],
            '-CAfile', ca_crt,
            '-caname', 'myCA',
            '-chain',
        ])
        run([
            'keytool', '-importkeystore',
            '-deststorepass', keystore_pw,
            '-destkeypass', keystore_pw,
            '-destkeystore', keystore,
            '-srckeystore', keystorep12,
            '-srcstoretype', 'PKCS12',
            '-srcstorepass', keystore_pw,
            '-alias', splitext(cert)[0]
        ])


def create_certs(out_dir, keystore_pw):
    ca_key = join(out_dir, 'rootCA.key')
    ca_crt = join(out_dir, 'rootCA.crt')
    print(f'Generating rootCA key: {ca_key}')
    print(f'Generating rootCA certificate: {ca_crt}')
    run([
        'openssl', 'req', '-x509', '-sha256', '-nodes',
        '-days', '365',
        '-subj', f'/C=AT/ST=Dummy State/L=Dummy Country/O=Dummy Company/CN=myCA',
        '-newkey', 'rsa:2048',
        '-keyout', ca_key,
        '-out', ca_crt
    ])

    certs_and_keys = generate_for(ca_key, ca_crt, out_dir, 'node', 1)
    keystore = join(out_dir, 'keystore.jks')
    keystore_p12 = join(out_dir, 'keystore.p12')
    import_into_keystores(certs_and_keys, 'node', keystore_pw, ca_crt, keystore, keystore_p12)
    # the CA certificate should also be in the keystore for the
    # node to be able to verify the client certificate
    run(['keytool', '-importcert',
         '-storepass', keystore_pw,
         '-keystore', keystore,
         '-file', ca_crt,
         '-alias', 'therootca'
    ])

    certs_and_keys = generate_for(ca_key, ca_crt, out_dir, 'client', 1)
    keystore_client = join(out_dir, 'keystore_client.jks')
    keystore_client_p12 = join(out_dir, 'keystore_client.p12')
    import_into_keystores(certs_and_keys, 'client', keystore_pw, ca_crt, keystore_client, keystore_client_p12)


def main():
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument('--out-dir', type=str, required=True)
    parser.add_argument('--keystore-pw', type=str, default='changeit')
    args = parser.parse_args()
    os.makedirs(args.out_dir, exist_ok=True)
    create_certs(args.out_dir, args.keystore_pw)


if __name__ == "__main__":
    main()
