﻿namespace Reportr.Data.Querying
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Linq;

    /// <summary>
    /// Represents the data for a single query grouping
    /// </summary>
    public sealed class QueryGrouping : IEnumerable<QueryRow>
    {
        /// <summary>
        /// Constructs the query grouping with the details
        /// </summary>
        /// <param name="columns">The columns</param>
        /// <param name="rows">The rows</param>
        public QueryGrouping
            (
                QueryColumnInfo[] columns,
                params QueryRow[] rows
            )
        {
            SetData(columns, rows);
        }

        /// <summary>
        /// Constructs the query grouping with the details
        /// </summary>
        /// <param name="groupingValues">The grouping values</param>
        /// <param name="columns">The columns</param>
        /// <param name="rows">The rows</param>
        public QueryGrouping
            (
                Dictionary<string, object> groupingValues,
                QueryColumnInfo[] columns,
                params QueryRow[] rows
            )
        {
            Validate.IsNotNull(groupingValues);

            this.GroupingValues = groupingValues;

            SetData(columns, rows);
        }

        /// <summary>
        /// Gets the grouping values by column
        /// </summary>
        public Dictionary<string, object> GroupingValues { get; private set; }

        /// <summary>
        /// Gets an array of the rows in the result
        /// </summary>
        public QueryRow[] Rows { get; private set; }

        /// <summary>
        /// Sets the query grouping data
        /// </summary>
        /// <param name="columns">The queries columns</param>
        /// <param name="rows">The rows generated by the query</param>
        private void SetData
            (
                IEnumerable<QueryColumnInfo> columns,
                params QueryRow[] rows
            )
        {
            Validate.IsNotNull(columns);
            Validate.IsNotNull(rows);

            if (false == columns.Any())
            {
                throw new ArgumentException
                (
                    "At least one column is required to populate the query."
                );
            }

            foreach (var set in columns.ToList())
            {
                var name = set.Column.Name;

                var matchCount = columns.Count
                (
                    s => s.Column.Name.Equals(name, StringComparison.OrdinalIgnoreCase)
                );

                if (matchCount > 1)
                {
                    throw new ArgumentException
                    (
                        $"The column name '{name}' can only be used once."
                    );
                }
            }

            ValidateRows
            (
                columns.ToArray(),
                rows
            );

            this.Rows = rows;
        }

        /// <summary>
        /// Validates a collection of rows against columns
        /// </summary>
        /// <param name="columns">The columns</param>
        /// <param name="rows">The rows to validate</param>
        private void ValidateRows
            (
                QueryColumnInfo[] columns,
                QueryRow[] rows
            )
        {
            var columnCount = columns.Length;

            foreach (var row in rows)
            {
                var cellCount = row.Cells.Length;

                if (cellCount != columnCount)
                {
                    throw new InvalidOperationException
                    (
                        $"{columnCount} cells were expected, but {cellCount} were found."
                    );
                }

                // Ensure the column names match those in the cells
                foreach (var info in columns)
                {
                    var name = info.Column.Name;

                    var matchFound = row.Cells.Any
                    (
                        c => c.Column.Name.Equals(name, StringComparison.OrdinalIgnoreCase)
                    );

                    if (false == matchFound)
                    {
                        throw new InvalidOperationException
                        (
                            $"No data was found for the column '{name}'."
                        );
                    }
                }
            }
        }

        /// <summary>
        /// Gets the row at the index specified
        /// </summary>
        /// <param name="index">The row index (zero based)</param>
        /// <returns>The matching query row</returns>
        public QueryRow this[int index]
        {
            get
            {
                return this.Rows[index];
            }
        }

        /// <summary>
        /// Gets an enumerator for the collection of rows
        /// </summary>
        /// <returns>The enumerator</returns>
        public IEnumerator<QueryRow> GetEnumerator()
        {
            return this.Rows.ToList().GetEnumerator();
        }

        /// <summary>
        /// Gets a generic enumerator for the collection of rows
        /// </summary>
        /// <returns>The generic enumerator</returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }
    }
}
