﻿namespace Reportr.Data.Querying
{
    using Nito.AsyncEx.Synchronous;
    using Reportr.Globalization;
    using Reportr.Filtering;
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Reflection;
    using System.Threading.Tasks;

    /// <summary>
    /// Represents a base implementation for a report query
    /// </summary>
    public abstract class QueryBase : IQuery
    {
        private readonly Dictionary<string, QuerySortingRule> _sortingRules;
        private readonly List<string> _groupingColumns;

        /// <summary>
        /// Constructs the query with a data source
        /// </summary>
        /// <param name="dataSource">The data source</param>
        /// <param name="maximumRows">The maximum rows (optional)</param>
        public QueryBase
            (
                IDataSource dataSource,
                int? maximumRows = null
            )
        {
            Validate.IsNotNull(dataSource);

            _sortingRules = new Dictionary<string, QuerySortingRule>
            (
                StringComparer.OrdinalIgnoreCase
            );

            _groupingColumns = new List<string>();

            this.QueryId = Guid.NewGuid();
            this.DataSource = dataSource;
            this.MaximumRows = maximumRows;
        }

        /// <summary>
        /// Gets the unique ID of the query
        /// </summary>
        public Guid QueryId { get; private set; }

        /// <summary>
        /// Gets the name of the query
        /// </summary>
        public virtual string Name
        {
            get
            {
                return this.GetType().Name.Replace
                (
                    "Query",
                    String.Empty
                );
            }
        }

        /// <summary>
        /// Gets the data source being used by the query
        /// </summary>
        public IDataSource DataSource { get; private set; }

        /// <summary>
        /// Gets an array of the columns generated by the query
        /// </summary>
        public abstract QueryColumnInfo[] Columns { get; }

        /// <summary>
        /// Determines if the query has a column with the name specified
        /// </summary>
        /// <param name="name">The column name</param>
        /// <returns>True, if a matching column is found; otherwise false</returns>
        public bool HasColumn
            (
                string name
            )
        {
            Validate.IsNotEmpty(name);

            return this.Columns.Any
            (
                info => info.Column.Name.Equals
                (
                    name,
                    StringComparison.OrdinalIgnoreCase
                )
            );
        }

        /// <summary>
        /// Gets a column from the query matching the name specified
        /// </summary>
        /// <param name="name">The column name</param>
        /// <returns>The matching column</returns>
        public QueryColumnInfo GetColumn
            (
                string name
            )
        {
            Validate.IsNotEmpty(name);

            var column = this.Columns.FirstOrDefault
            (
                info => info.Column.Name.Equals
                (
                    name,
                    StringComparison.OrdinalIgnoreCase
                )
            );

            if (column == null)
            {
                throw new KeyNotFoundException
                (
                    $"No column was found matching the name '{name}'."
                );
            }

            return column;
        }

        /// <summary>
        /// Gets an array of parameters accepted by the query
        /// </summary>
        public abstract Filtering.ParameterInfo[] Parameters { get; }

        /// <summary>
        /// Gets a flag indicating if at least one parameter value is required to run the query
        /// </summary>
        public bool OnlyRunWithParameterValues { get; protected set; }

        /// <summary>
        /// Gets the maximum number of rows the query will return
        /// </summary>
        /// <remarks>
        /// This is optional and, if null, all rows are returned.
        /// </remarks>
        public int? MaximumRows { get; protected set; }

        /// <summary>
        /// Gets an array of sorting rules for the query
        /// </summary>
        public QuerySortingRule[] SortingRules
        {
            get
            {
                var rules = _sortingRules.Select
                (
                    pair => pair.Value
                );

                return rules.ToArray();
            }
        }

        /// <summary>
        /// Specifies a sorting rule against a column in the query
        /// </summary>
        /// <param name="columnName">The column name</param>
        /// <param name="direction">The sort direction</param>
        public void SortColumn
            (
                string columnName,
                SortDirection direction
            )
        {
            Validate.IsNotEmpty(columnName);

            var columnFound = HasColumn(columnName);

            if (false == columnFound)
            {
                throw new InvalidOperationException
                (
                    $"The column '{columnName}' does not exist."
                );
            }

            var rule = new QuerySortingRule
            (
                columnName,
                direction
            );

            _sortingRules[columnName] = rule;
        }

        /// <summary>
        /// Gets an array of grouping columns for the query
        /// </summary>
        public string[] GroupingColumns
        {
            get
            {
                return _groupingColumns.ToArray();
            }
        }

        /// <summary>
        /// Adds a grouping column to the query
        /// </summary>
        /// <param name="columnName">The column name</param>
        public void AddGrouping
            (
                string columnName
            )
        {
            Validate.IsNotEmpty(columnName);

            var columnFound = HasColumn(columnName);

            if (false == columnFound)
            {
                throw new InvalidOperationException
                (
                    $"The column '{columnName}' does not exist."
                );
            }

            _groupingColumns.Add(columnName);
        }

        /// <summary>
        /// Executes the query using the parameter values supplied
        /// </summary>
        /// <param name="parameterValues">The parameter values</param>
        /// <returns>The query results</returns>
        public virtual QueryResults Execute
            (
                params ParameterValue[] parameterValues
            )
        {
            var task = ExecuteAsync(parameterValues);

            return task.WaitAndUnwrapException();
        }

        /// <summary>
        /// Asynchronously executes the query using the parameter values supplied
        /// </summary>
        /// <param name="parameterValues">The parameter values</param>
        /// <returns>The query results</returns>
        public virtual async Task<QueryResults> ExecuteAsync
            (
                params ParameterValue[] parameterValues
            )
        {
            var watch = Stopwatch.StartNew();

            if (parameterValues == null)
            {
                parameterValues = new ParameterValue[] { };
            }

            if (this.OnlyRunWithParameterValues)
            {
                var valueFound = parameterValues.Any
                (
                    pv => pv.Value != null && false == pv.ValueAutoSetByConstraint
                );

                if (false == valueFound)
                {
                    return new QueryResults
                    (
                        this,
                        0
                    );
                }
            }

            var parameterErrors = ValidateParameterValues
            (
                parameterValues
            );

            if (parameterErrors.Any())
            {
                var results = new QueryResults
                (
                    this,
                    0,
                    false
                );

                return results.WithErrors
                (
                    parameterErrors
                );
            }
            else
            {
                var fetchTask = FetchDataAsync
                (
                    parameterValues
                );

                var rows = await fetchTask.ConfigureAwait
                (
                    false
                );

                EnsureRowCountValid(rows);

                rows = SortRows(rows);

                var groupings = GroupRows(rows);

                watch.Stop();

                var executionTime = watch.ElapsedMilliseconds;

                var results = new QueryResults
                (
                    this,
                    executionTime
                );

                return results.WithData
                (
                    groupings
                );
            }
        }

        /// <summary>
        /// Asynchronously fetches the query data using the parameter values
        /// </summary>
        /// <param name="parameterValues">The parameter values</param>
        /// <returns>The query data in the form of an array of rows</returns>
        protected abstract Task<IEnumerable<QueryRow>> FetchDataAsync
        (
            params ParameterValue[] parameterValues
        );

        /// <summary>
        /// Ensures the number of rows returned by a query are valid
        /// </summary>
        /// <typeparam name="T">The returned data type</typeparam>
        /// <param name="data">The data generated by the query</param>
        protected virtual void EnsureRowCountValid<T>
            (
                IEnumerable<T> data
            )
        {
            if (this.MaximumRows.HasValue)
            {
                var name = this.Name;
                var rowCount = data.Count();
                var maxRows = this.MaximumRows.Value;

                if (rowCount > maxRows)
                {
                    throw new InvalidOperationException
                    (
                        $"The query '{name}' returned {rowCount} rows, " +
                        $"but the maximum number of rows allowed is {maxRows}."
                    );
                }
            }
        }

        /// <summary>
        /// Converts a collection of data into a list of query rows
        /// </summary>
        /// <typeparam name="T">The data type</typeparam>
        /// <param name="data">The data to convert</param>
        /// <returns>A list of query rows</returns>
        protected virtual List<QueryRow> ConvertToRows<T>
            (
                IEnumerable<T> data
            )
        {
            var localeConfiguration = this.DataSource.LocaleConfiguration;
            var rows = new List<QueryRow>();
            var entityType = typeof(T);

            foreach (var item in data)
            {
                var cells = new List<QueryCell>();

                foreach (var info in this.Columns)
                {
                    var property = entityType.GetProperty
                    (
                        info.Column.Name
                    );

                    var propertyValue = property?.GetValue
                    (
                        item
                    );

                    var transformer = CulturalTransformerFactory.GetInstance
                    (
                        propertyValue
                    );

                    propertyValue = transformer.Transform
                    (
                        propertyValue,
                        localeConfiguration
                    );

                    cells.Add
                    (
                        new QueryCell
                        (
                            info.Column,
                            propertyValue
                        )
                    );
                }

                rows.Add
                (
                    new QueryRow
                    (
                        cells.ToArray()
                    )
                );
            }

            return rows;
        }

        /// <summary>
        /// Sorts a collection of rows by the queries sorting rules
        /// </summary>
        /// <param name="rows">The rows to sort</param>
        /// <returns>A collection of sorted rows</returns>
        protected virtual IEnumerable<QueryRow> SortRows
            (
                IEnumerable<QueryRow> rows
            )
        {
            Validate.IsNotNull(rows);

            if (false == this.SortingRules.Any())
            {
                return rows;
            }
            else
            {
                var ruleNumber = 1;
                var sortedRows = (IOrderedEnumerable<QueryRow>)rows;

                foreach (var rule in this.SortingRules)
                {
                    object keySelector(QueryRow row) => row.First
                    (
                        cell => cell.Column.Name.Equals
                        (
                            rule.ColumnName,
                            StringComparison.OrdinalIgnoreCase
                        )
                    )
                    .Value;

                    if (rule.Direction == SortDirection.Ascending)
                    {
                        if (ruleNumber == 1)
                        {
                            sortedRows = sortedRows.OrderBy
                            (
                                keySelector
                            );
                        }
                        else
                        {
                            sortedRows = sortedRows.ThenBy
                            (
                                keySelector
                            );
                        }
                    }
                    else
                    {
                        if (ruleNumber == 1)
                        {
                            sortedRows = sortedRows.OrderByDescending
                            (
                                keySelector
                            );
                        }
                        else
                        {
                            sortedRows = sortedRows.ThenByDescending
                            (
                                keySelector
                            );
                        }
                    }

                    ruleNumber++;
                }

                return rows;
            }
        }

        /// <summary>
        /// Groups a collection of rows by the queries grouping rules
        /// </summary>
        /// <param name="rows">The rows to group</param>
        /// <returns>An array of query groupings</returns>
        protected virtual QueryGrouping[] GroupRows
            (
                IEnumerable<QueryRow> rows
            )
        {
            Validate.IsNotNull(rows);

            if (false == _groupingColumns.Any())
            {
                return new QueryGrouping[]
                {
                    new QueryGrouping
                    (
                        this.Columns,
                        rows.ToArray()
                    )
                };
            }
            else
            {
                var groupings = new List<QueryGrouping>();
                var groupedRows = new Dictionary<string, List<QueryRow>>();

                foreach (var row in rows)
                {
                    var groupingValue = String.Empty;

                    foreach (var columnName in _groupingColumns)
                    {
                        groupingValue += row[columnName].Value;
                    }

                    if (groupedRows.ContainsKey(groupingValue))
                    {
                        groupedRows[groupingValue].Add
                        (
                            row
                        );
                    }
                    else
                    {
                        groupedRows.Add
                        (
                            groupingValue,
                            new List<QueryRow> { row }
                        );
                    }
                }

                foreach (var item in groupedRows)
                {
                    var firstRow = item.Value.First();
                    var groupingValues = new Dictionary<string, object>();
                    var allColumns = this.Columns;

                    foreach (var columnName in _groupingColumns)
                    {
                        groupingValues.Add
                        (
                            columnName,
                            firstRow[columnName].Value
                        );
                    }

                    var grouping = new QueryGrouping
                    (
                        groupingValues,
                        allColumns,
                        item.Value.ToArray()
                    );

                    groupings.Add(grouping);
                }

                return groupings.ToArray();
            }
        }

        /// <summary>
        /// Validates parameter values against the query
        /// </summary>
        /// <param name="parameterValues">The parameter values</param>
        /// <returns>A dictionary of errors generated</returns>
        private IDictionary<string, string> ValidateParameterValues
            (
                params ParameterValue[] parameterValues
            )
        {
            var errors = new Dictionary<string, string>();
            var parameters = this.Parameters;

            foreach (var value in parameterValues)
            {
                var parameterName = value.Parameter.Name;

                var matchingParameter = parameters.FirstOrDefault
                (
                    p => p.Name.Equals
                    (
                        parameterName,
                        StringComparison.OrdinalIgnoreCase
                    )
                );

                if (matchingParameter == null)
                {
                    var message = 
                        $"'{parameterName}' did not match any " +
                        $"parameters in the query '{this.Name}'.";

                    errors.Add
                    (
                        parameterName,
                        message
                    );
                }
                else
                {
                    // Ensure all parameter values match the expected type
                    if (value.Value != null)
                    {
                        var actualType = value.Value.GetType();

                        if (matchingParameter.ExpectedType != actualType)
                        {
                            var canConvert = actualType.CanConvert
                            (
                                matchingParameter.ExpectedType,
                                value.Value
                            );

                            if (false == canConvert)
                            {
                                var message =
                                    $"Type {actualType.Name} is not valid " +
                                    $"for the parameter '{parameterName}'.";

                                errors.Add
                                (
                                    parameterName,
                                    message
                                );
                            }
                        }
                    }
                }
            }

            // Ensure all required parameters have been supplied
            foreach (var parameter in parameters)
            {
                if (parameter.ValueRequired)
                {
                    var valueFound = parameterValues.Any
                    (
                        value => value.Name.Equals
                        (
                            parameter.Name,
                            StringComparison.OrdinalIgnoreCase
                        )
                        && value.Value != null
                    );

                    if (false == valueFound)
                    {
                        var message = 
                            $"A value is required for the " +
                            $"parameter '{parameter.Name}'.";

                        errors.Add
                        (
                            parameter.Name,
                            message
                        );
                    }
                }
            }

            return errors;
        }

        /// <summary>
        /// Gets a parameter value from the parameters supplied
        /// </summary>
        /// <typeparam name="TValue">The parameter value type to return</typeparam>
        /// <param name="parameterName">The parameter name</param>
        /// <param name="parameterValues">The parameter values</param>
        /// <returns>The parameter value as the type specified</returns>
        protected virtual TValue GetParameterValue<TValue>
            (
                string parameterName,
                params ParameterValue[] parameterValues
            )
        {
            Validate.IsNotEmpty(parameterName);

            var matchingItem = parameterValues.FirstOrDefault
            (
                pv => pv.Name.Equals
                (
                    parameterName,
                    StringComparison.OrdinalIgnoreCase
                )
            );

            if (matchingItem == null || matchingItem.Value == null)
            {
                return default;
            }
            else
            {
                var currentType = matchingItem.Value.GetType();

                if (currentType == typeof(TValue))
                {
                    return (TValue)matchingItem.Value;
                }
                else
                {
                    var converter = new ObjectConverter<TValue>();

                    return converter.Convert
                    (
                        matchingItem.Value
                    );
                }
            }
        }

        /// <summary>
        /// Resolves the data table schema for a specific output type
        /// </summary>
        /// <typeparam name="TOutput">The query output type</typeparam>
        /// <returns>The table schema</returns>
        protected virtual DataTableSchema ResolveTableSchema<TOutput>()
        {
            var outputType = typeof(TOutput);
            var dataSource = this.DataSource;

            var tableSchema = dataSource.Schema.FirstOrDefault
            (
                dts => dts.Name == outputType.Name
            );

            if (tableSchema == null)
            {
                var properties = outputType.GetProperties
                (
                    BindingFlags.Public | BindingFlags.Instance
                );

                var columnSchemas = new List<DataColumnSchema>();

                foreach (var property in properties)
                {
                    columnSchemas.Add
                    (
                        new DataColumnSchema
                        (
                            property.Name,
                            property.PropertyType
                        )
                    );
                }

                tableSchema = new DataTableSchema
                (
                    outputType.Name,
                    columnSchemas.ToArray()
                );
            }

            return tableSchema;
        }

        /// <summary>
        /// Resolves the queries columns from the output type
        /// </summary>
        /// <typeparam name="T">The output type</typeparam>
        /// <returns>An array of columns</returns>
        protected virtual QueryColumnInfo[] ResolveColumns<T>()
        {
            var entityType = typeof(T);
            var tableSchema = ResolveTableSchema<T>();

            var properties = entityType.GetProperties
            (
                BindingFlags.Public | BindingFlags.Instance
            );

            var columnInfos = new List<QueryColumnInfo>();

            foreach (var property in properties)
            {
                columnInfos.Add
                (
                    new QueryColumnInfo
                    (
                        tableSchema,
                        new DataColumnSchema
                        (
                            property.Name,
                            property.PropertyType
                        )
                    )
                );
            }

            return columnInfos.ToArray();
        }

        /// <summary>
        /// Provides a custom description of the query
        /// </summary>
        /// <returns>The query name</returns>
        public override string ToString()
        {
            return this.Name;
        }
    }
}
