﻿namespace Reportr.Data.Querying
{
    using Reportr.Data;
    using Reportr.Filtering;
    using System;
    using System.Threading.Tasks;

    /// <summary>
    /// Defines a contract for a single report query
    /// </summary>
    public interface IQuery : ISortable
    {
        /// <summary>
        /// Gets the unique ID of the query
        /// </summary>
        Guid QueryId { get; }

        /// <summary>
        /// Gets the name of the query
        /// </summary>
        string Name { get; }

        /// <summary>
        /// Gets the data source being used by the query
        /// </summary>
        IDataSource DataSource { get; }

        /// <summary>
        /// Gets an array of the columns generated by the query
        /// </summary>
        QueryColumnInfo[] Columns { get; }

        /// <summary>
        /// Determines if the query has a column with the name specified
        /// </summary>
        /// <param name="name">The column name</param>
        /// <returns>True, if a matching column is found; otherwise false</returns>
        bool HasColumn
        (
            string name
        );

        /// <summary>
        /// Gets a column from the query matching the name specified
        /// </summary>
        /// <param name="name">The column name</param>
        /// <returns>The matching column</returns>
        QueryColumnInfo GetColumn
        (
            string name
        );

        /// <summary>
        /// Gets an array of parameters accepted by the component
        /// </summary>
        ParameterInfo[] Parameters { get; }

        /// <summary>
        /// Gets a flag indicating if at least one parameter value is required to run the query
        /// </summary>
        bool OnlyRunWithParameterValues { get; }

        /// <summary>
        /// Gets the maximum number of rows the query will return
        /// </summary>
        /// <remarks>
        /// This is optional and, if null, all rows are returned.
        /// </remarks>
        int? MaximumRows { get; }

        /// <summary>
        /// Gets an array of grouping columns for the query
        /// </summary>
        string[] GroupingColumns { get; }

        /// <summary>
        /// Adds a grouping column to the query
        /// </summary>
        /// <param name="columnName">The column name</param>
        void AddGrouping
        (
            string columnName
        );

        /// <summary>
        /// Executes the query using the parameter values supplied
        /// </summary>
        /// <param name="parameterValues">The parameter values</param>
        /// <returns>The query results</returns>
        QueryResults Execute
        (
            params ParameterValue[] parameterValues
        );

        /// <summary>
        /// Asynchronously executes the query using the parameter values supplied
        /// </summary>
        /// <param name="parameterValues">The parameter values</param>
        /// <returns>The query results</returns>
        Task<QueryResults> ExecuteAsync
        (
            params ParameterValue[] parameterValues
        );
    }
}
