﻿namespace Reportr.Components.Metrics
{
    using Reportr.Globalization;
    using System;
    using System.Runtime.Serialization;

    /// <summary>
    /// Represents a single report statistic
    /// </summary>
    [DataContract]
    public class Statistic : ReportComponentBase
    {
        /// <summary>
        /// Constructs the statistic with the details
        /// </summary>
        /// <param name="definition">The statistic definition</param>
        /// <param name="value">The value calculated</param>
        public Statistic
            (
                StatisticDefinition definition,
                double value
            )
            : base(definition)
        {
            this.Value = value;
            this.HasRange = false;
        }

        /// <summary>
        /// Gets the value generated by the statistic
        /// </summary>
        [DataMember]
        public double Value { get; private set; }

        /// <summary>
        /// Gets the lower range value
        /// </summary>
        [DataMember]
        public double? LowerRange { get; private set; }

        /// <summary>
        /// Gets the upper range value
        /// </summary>
        [DataMember]
        public double? UpperRange { get; private set; }

        /// <summary>
        /// Gets a flag indicating if the value has a range that it fits into
        /// </summary>
        [DataMember]
        public bool HasRange { get; private set; }

        /// <summary>
        /// Adds the range values to the statistic
        /// </summary>
        /// <param name="lower">The lower range</param>
        /// <param name="upper">The upper range</param>
        /// <returns>The updated statistic</returns>
        public Statistic WithRange
            (
                double? lower,
                double? upper
            )
        {
            if (lower == null && upper == null)
            {
                throw new InvalidOperationException
                (
                    "A lower or upper range value must be specified."
                );
            }

            this.LowerRange = lower;
            this.UpperRange = upper;
            this.HasRange = true;

            return this;
        }

        /// <summary>
        /// Gets the statistic action
        /// </summary>
        [DataMember]
        public ReportAction Action { get; private set; }

        /// <summary>
        /// Adds the action to the statistic
        /// </summary>
        /// <param name="action">The statistic action</param>
        /// <returns>The updated statistic</returns>
        public Statistic WithAction
            (
                ReportAction action
            )
        {
            Validate.IsNotNull(action);

            this.Action = action;

            return this;
        }
    }
}
