﻿namespace Reportr.Data.Sql
{
    using Reportr.Data.Querying;
    using Reportr.Filtering;
    using System;
    using System.Collections.Generic;
    using System.Data;
    using System.Data.SqlClient;
    using System.Threading.Tasks;

    /// <summary>
    /// Represents a base class for stored procedure queries
    /// </summary>
    public abstract class StoredProcedureQuery : QueryBase
    {
        private List<QueryColumnInfo> _columns;

        /// <summary>
        /// Constructs the query with an SQL data source
        /// </summary>
        /// <param name="dataSource">The SQL data source</param>
        public StoredProcedureQuery
            (
                SqlDataSource dataSource
            )

            : base(dataSource)
        {
            _columns = null;
        }

        /// <summary>
        /// Resolves the queries columns from the output type
        /// </summary>
        private void ResolveColumns()
        {
            _columns = new List<QueryColumnInfo>();

            var connectionString = GetConnectionString();
            var procedureName = this.StoredProcedureName;

            using (var connection = new SqlConnection(connectionString))
            {
                connection.Open();

                using (var command = new SqlCommand(procedureName, connection))
                {
                    command.CommandType = CommandType.StoredProcedure;
                    
                    var reader = command.ExecuteReader();
                    var schema = reader.GetSchemaTable();
                    
                    foreach (DataColumn column in schema.Columns)
                    {
                        var columnName = column.ColumnName;
                        var valueType = column.DataType;

                        var tableSchema = this.DataSource.GetSchemaTable
                        (
                            column.Table.TableName
                        );

                        var columnSchema = new DataColumnSchema
                        (
                            columnName,
                            valueType
                        );

                        _columns.Add
                        (
                            new QueryColumnInfo
                            (
                                tableSchema,
                                columnSchema
                            )
                        );
                    }
                }
            }
        }

        /// <summary>
        /// Gets an array of the columns generated by the query
        /// </summary>
        public override QueryColumnInfo[] Columns
        {
            get
            {
                if (_columns == null)
                {
                    ResolveColumns();
                }

                return _columns.ToArray();
            }
        }

        /// <summary>
        /// Gets the database connection string from the data source
        /// </summary>
        /// <returns>The connection string</returns>
        protected string GetConnectionString()
        {
            return ((SqlDataSource)this.DataSource).ConnectionString;
        }

        /// <summary>
        /// Gets the name of the stored procedure
        /// </summary>
        protected abstract string StoredProcedureName { get; }

        /// <summary>
        /// Asynchronously fetches the query data using the parameter values
        /// </summary>
        /// <param name="parameterValues">The parameter values</param>
        /// <returns>The query data in the form of an array of rows</returns>
        protected override async Task<IEnumerable<QueryRow>> FetchDataAsync
            (
                params ParameterValue[] parameterValues
            )
        {
            Validate.IsNotNull(parameterValues);

            var connectionString = GetConnectionString();
            var procedureName = this.StoredProcedureName;

            using (var connection = new SqlConnection(connectionString))
            {
                using (var command = new SqlCommand(procedureName, connection))
                {
                    command.CommandType = CommandType.StoredProcedure;
                    connection.Open();

                    foreach (var parameter in parameterValues)
                    {
                        command.Parameters.Add
                        (
                            new SqlParameter
                            (
                                parameter.Name,
                                parameter.Value
                            )
                        );
                    }

                    var readTask = command.ExecuteReaderAsync();

                    using (var reader = await readTask.ConfigureAwait(false))
                    {
                        return reader.ToQueryRows
                        (
                            this.DataSource.LocaleConfiguration
                        );
                    }
                }
            }
        }
    }
}
