﻿namespace Reportr.Data.Dapper
{
    using global::Dapper;
    using Nito.AsyncEx.Synchronous;
    using Reportr.Data.Querying;
    using Reportr.Filtering;
    using System;
    using System.Collections.Generic;
    using System.Data;
    using System.Threading.Tasks;

    /// <summary>
    /// Represents a base class for generic Dapper report queries
    /// </summary>
    /// <typeparam name="T">The query output type</typeparam>
    public abstract class DapperQuery<T> : QueryBase
    {
        private QueryColumnInfo[] _columns = null;
        private readonly int _commandTimeout;

        /// <summary>
        /// Constructs the query with a Dapper data source
        /// </summary>
        /// <param name="dataSource">The data source</param>
        /// <param name="commandTimeout">Command execution timeout (in seconds)</param>
        public DapperQuery
            (
                DapperDataSource dataSource,
                int commandTimeout = 60
            )

            : base(dataSource)
        {
            _columns = ResolveColumns<T>();
            _commandTimeout = commandTimeout;
        }
        
        /// <summary>
        /// Gets an array of the columns generated by the query
        /// </summary>
        public override QueryColumnInfo[] Columns
        {
            get
            {
                if (_columns == null || _columns.Length == 0)
                {
                    _columns = ResolveColumns<T>();
                }

                return _columns;
            }
        }

        /// <summary>
        /// Gets the database connection from the data source
        /// </summary>
        /// <returns>The database connection</returns>
        protected IDbConnection GetConnection()
        {
            return ((DapperDataSource)this.DataSource).Connection;
        }

        /// <summary>
        /// Compiles the SQL statement to be executed by the query
        /// </summary>
        /// <param name="parameterValues">The parameter values</param>
        /// <returns>The SQL query generated</returns>
        protected abstract SqlStatement CompileSql
        (
            params ParameterValue[] parameterValues
        );
        
        /// <summary>
        /// Asynchronously fetches the query data using the parameter values
        /// </summary>
        /// <param name="parameterValues">The parameter values</param>
        /// <returns>The query data in the form of an array of rows</returns>
        protected override async Task<IEnumerable<QueryRow>> FetchDataAsync
            (
                params ParameterValue[] parameterValues
            )
        {
            Validate.IsNotNull(parameterValues);

            var statement = CompileSql
            (
                parameterValues
            );

            var queryResults = await ExecuteQueryAsync<T>
            (
                statement
            )
            .ConfigureAwait
            (
                false
            );
            
            var rows = ConvertToRows<T>
            (
                queryResults
            );

            return rows;
        }

        /// <summary>
        /// Executes an SQL statement and returns the results
        /// </summary>
        /// <typeparam name="TOutput">The query output type</typeparam>
        /// <param name="statement">The SQL statement</param>
        /// <returns>The results</returns>
        protected IEnumerable<TOutput> ExecuteQuery<TOutput>
            (
                SqlStatement statement
            )
        {
            Validate.IsNotNull(statement);

            var task = ExecuteQueryAsync<TOutput>
            (
                statement
            );

            return task.WaitAndUnwrapException();
        }

        /// <summary>
        /// Executes an SQL statement and returns the results
        /// </summary>
        /// <typeparam name="TOutput">The query output type</typeparam>
        /// <param name="statement">The SQL statement</param>
        /// <returns>The results</returns>
        protected async Task<IEnumerable<TOutput>> ExecuteQueryAsync<TOutput>
            (
                SqlStatement statement
            )
        {
            Validate.IsNotNull(statement);

            var connection = GetConnection();
            var sql = statement.Sql;
            var parameters = new DynamicParameters();

            foreach (var item in statement.Parameters)
            {
                parameters.Add
                (
                    item.Key,
                    item.Value
                );
            }

            var queryResults = await connection.QueryAsync<TOutput>
            (
                sql,
                parameters,
                null,
                _commandTimeout
            )
            .ConfigureAwait
            (
                false
            );

            return queryResults;
        }
    }
}
