require_relative '../../../spec_helper.rb'

describe Salus::Scanners::ReportNodeModules do
  describe '#run' do
    it 'should throw an error if package.json and friends are absent' do
      repo = Salus::Repo.new('spec/fixtures/blank_repository')
      scanner = Salus::Scanners::ReportNodeModules.new(repository: repo, config: {})

      expect { scanner.run }.to raise_error(
        Salus::Scanners::Base::InvalidScannerInvocationError,
        'Cannot report on Node modules without package.json, '\
          'package-lock.json or yarn.lock files.'
      )
    end

    it 'should report all the deps in package.json file in the absence of a lockfile' do
      repo = Salus::Repo.new('spec/fixtures/report_node_modules/package_json_only')
      scanner = Salus::Scanners::ReportNodeModules.new(repository: repo, config: {})
      scanner.run

      info = scanner.report.to_h.fetch(:info)

      expect(info[:package_json_node_version]).to eq('>= 6.9')
      expect(info[:package_json_npm_version]).to eq('>= 4.0.0')

      expect(info[:dependencies]).to match_array(
        [
          {
            dependency_file: 'package.json',
            type: 'node_module',
            name: '@nerv-hq/control-system',
            version: '1.2.3',
            source: 'https://npm.hq.nerv.net',
            scope: 'required'
          },
          {
            dependency_file: 'package.json',
            type: 'node_module',
            name: '@magi-core/prediction',
            version: '0.0.1',
            source: 'https://npm.magi.nerv.net',
            scope: 'required'
          },
          {
            dependency_file: 'package.json',
            type: 'node_module',
            name: 'classnames',
            version: '^2.2.5',
            source: '<package manager default>',
            scope: 'required'
          },
          {
            dependency_file: 'package.json',
            type: 'node_module',
            name: 'mobx',
            version: '^3.2.1',
            source: '<package manager default>',
            scope: 'required'
          }
        ]
      )
    end

    it 'should report all deps from package-lock.json,if present' do
      repo = Salus::Repo.new('spec/fixtures/report_node_modules/package_lock_json')
      scanner = Salus::Scanners::ReportNodeModules.new(repository: repo, config: {})
      scanner.run

      info = scanner.report.to_h.fetch(:info)

      expect(info[:package_lock_version]).to eq('1')

      expect(info[:dependencies]).to match_array(
        [
          {
            dependency_file: 'package-lock.json',
            type: 'node_module',
            name: '@nerv-hq/control-system',
            version: '1.2.3',
            source: 'https://npm.hq.nerv.net/control-system/v1.2.3/'\
              'tarball#sha1-ux/eKKZxz2ojD4icjhvReee++74=',
            scope: 'required'
          },
          {
            dependency_file: 'package-lock.json',
            type: 'node_module',
            name: '@magi-core/prediction',
            version: '0.0.1',
            source: 'https://npm.magi.nerv.net/prediction/v0.0.1/'\
              'tarball#sha1-ux/EyKzTRzvejDFis90vRLj8++74=',
            scope: 'required'
          },
          {
            dependency_file: 'package-lock.json',
            type: 'node_module',
            name: 'classnames',
            version: '2.2.5',
            source: 'https://registry.npmjs.org/classnames/-/classnames-2.2.5.tgz'\
              '#sha1-+zgB1FNGdknvNgPH1hoCvRKb3m0=',
            scope: 'required'
          },
          {
            dependency_file: 'package-lock.json',
            type: 'node_module',
            name: 'mobx',
            version: '3.2.1',
            source: 'https://registry.npmjs.org/mobx/-/mobx-3.2.1.tgz'\
              '#sha1-aureASDMP3i6pXGVAxYwoK6PE88=',
            scope: 'required'
          }
        ]
      )
    end

    it 'should report all deps from yarn.lock,if present' do
      repo = Salus::Repo.new('spec/fixtures/report_node_modules/yarn_lock')
      scanner = Salus::Scanners::ReportNodeModules.new(repository: repo, config: {})

      scanner.run

      info = scanner.report.to_h.fetch(:info)

      expect(info[:dependencies]).to match_array(
        [
          {
            dependency_file: 'yarn.lock',
            type: 'node_module',
            name: '@nerv-hq/control-system',
            version: '1.2.3',
            source: 'https://npm.hq.nerv.net/control-system/v1.2.3/'\
              'tarball#fb3801d453467639ef560fc7a61a02bd129bde6d',
            scope: 'required'
          },
          {
            dependency_file: 'yarn.lock',
            type: 'node_module',
            name: '@magi-core/prediction',
            version: '0.0.1',
            source: 'https://npm.magi.nerv.net/prediction/v0.0.1/'\
              'tarball#fb3801d453467639ef5602c7a51a02fd129bbbbd',
            scope: 'required'
          },
          {
            dependency_file: 'yarn.lock',
            type: 'node_module',
            name: 'classnames',
            version: '2.2.5',
            source: 'https://registry.yarnpkg.com/classnames/-/classnames-2.2.5.tgz'\
              '#fb3801d453467649ef3603c7d61a02bd129bde6d',
            scope: 'required'
          },
          {
            dependency_file: 'yarn.lock',
            type: 'node_module',
            name: 'mobx',
            version: '3.2.1',
            source: 'https://registry.yarnpkg.com/mobx/-/mobx-3.2.1.tgz'\
              '#6aeade0120cc3f78baa57195031630a0ae8f13cf',
            scope: 'required'
          },
          {
            dependency_file: 'yarn.lock',
            type: 'node_module',
            name: 'eslint-plugin-internal',
            version: '0.0.0',
            source: 'file:./lib/lint-rules',
            scope: 'required'
          }
        ]
      )
    end

    it 'should report all deps from latest yarn.lock,if present' do
      repo = Salus::Repo.new('spec/fixtures/report_node_modules/yarn_lock_v3')
      scanner = Salus::Scanners::ReportNodeModules.new(repository: repo, config: {})

      scanner.run
      info = scanner.report.to_h.fetch(:info)
      # rubocop:disable Style/HashSyntax
      expect(info[:dependencies]).to match_array(
        [
          {
            :type => "node_module",
            :name => "@next/env",
            :version => "12.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@next/swc-android-arm64",
            :version => "12.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@next/swc-darwin-arm64",
            :version => "12.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@next/swc-darwin-x64",
            :version => "12.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@next/swc-linux-arm-gnueabihf",
            :version => "12.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@next/swc-linux-arm64-gnu",
            :version => "12.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@next/swc-linux-arm64-musl",
            :version => "12.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@next/swc-linux-x64-gnu",
            :version => "12.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@next/swc-linux-x64-musl",
            :version => "12.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@next/swc-win32-arm64-msvc",
            :version => "12.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@next/swc-win32-ia32-msvc",
            :version => "12.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@next/swc-win32-x64-msvc",
            :version => "12.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@types/node",
            :version => "17.0.15",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@types/prop-types",
            :version => "15.7.4",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@types/react-dom",
            :version => "17.0.11",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@types/react",
            :version => "17.0.39",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "@types/scheduler",
            :version => "0.16.2",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "caniuse-lite",
            :version => "1.0.30001307",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "csstype",
            :version => "3.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "js-tokens",
            :version => "4.0.0",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "loose-envify",
            :version => "1.4.0",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "nanoid",
            :version => "3.2.0",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "next",
            :version => "12.0.10",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "object-assign",
            :version => "4.1.1",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "picocolors",
            :version => "1.0.0",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "postcss",
            :version => "8.4.5",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "react-dom",
            :version => "17.0.2",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "react",
            :version => "17.0.2",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "scheduler",
            :version => "0.20.2",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "source-map-js",
            :version => "1.0.2",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "styled-jsx",
            :version => "5.0.0",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "typescript",
            :version => "4.5.5",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "typescript@patch:typescript@npm%3A4.5.5#~builtin<compat/"\
                "typescript>::version=4.5.5&hash=493e53",
            :version => "4.5.5",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "use-subscription",
            :version => "1.5.1",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          },
          {
            :type => "node_module",
            :name => "yarn3next@workspace:.",
            :version => "0.0.0-use.local",
            :source => "https://registry.yarnpkg.com",
            :scope => "required",
            :dependency_file => "yarn.lock"
          }
        ]
      )
      # rubocop:enable Style/HashSyntax
    end

    it 'should report all the deps in bower.json,if present' do
      repo = Salus::Repo.new('spec/fixtures/report_node_modules/bower_json')
      scanner = Salus::Scanners::ReportNodeModules.new(repository: repo, config: {})
      scanner.run

      info = scanner.report.to_h.fetch(:info)

      expect(info[:dependencies]).to match_array(
        [
          {
            dependency_file: 'bower.json',
            type: 'node_module',
            name: 'lodash',
            version: '3.0.0',
            source: 'https://registry.bower.io',
            scope: 'required'
          },
          {
            dependency_file: 'bower.json',
            type: 'node_module',
            name: 'angular',
            version: '1.3.0',
            source: 'https://registry.bower.io',
            scope: 'required'
          },
          {
            dependency_file: 'bower.json',
            type: 'node_module',
            name: 'axios',
            version: '0.17.0',
            source: 'https://registry.bower.io',
            scope: 'required'
          },
          {
            dependency_file: 'bower.json',
            type: 'node_module',
            name: 'angular-ui-router',
            version: '0.2.13',
            source: 'https://registry.bower.io',
            scope: 'required'
          }
        ]
      )
    end

    it 'should report all deps from package-lock.json including deps,if present' do
      path = 'spec/fixtures/report_node_modules/validate_include_dev_deps_flag'
      config_file = YAML.load_file("#{path}/salus.yaml")
      scanner_config = config_file['scanner_configs']["ReportNodeModules"]

      repo = Salus::Repo.new(path)
      scanner = Salus::Scanners::ReportNodeModules.new(repository: repo, config: scanner_config)
      scanner.run

      info = scanner.report.to_h.fetch(:info)

      expect(info[:package_lock_version]).to eq('1')

      expect(info[:dependencies]).to match_array(
        [
          {
            dependency_file: 'package-lock.json',
            type: 'node_module',
            name: '@storybook/components',
            version: '3.4.11',
            source: 'https://registry.npmjs.org/@storybook/components/-/components-3.4.11.tgz'\
              '#sha512-M3WhGPR4LNB2NabKyLtxDMevB1LAHOrmrII2U19XYIph93k3SReIwLKWEds0/'\
              'jWwajgQtI3hBftDCu/QA5bTOA==',
            scope: 'optional'
          },
          {
            dependency_file: 'package-lock.json',
            type: 'node_module',
            name: 'parseurl',
            version: '1.3.2',
            source: 'https://registry.npmjs.org/parseurl/-/parseurl-1.3.2.tgz'\
              '#sha1-/CidTtiZMRlGDBViUyYs3I3mW/M=',
            scope: 'required'
          },
          {
            dependency_file: 'package-lock.json',
            type: 'node_module',
            name: 'prop-types',
            version: '15.7.1',
            source: 'https://registry.npmjs.org/prop-types/-/prop-types-15.7.1.tgz'\
              '#sha512-f8Lku2z9kERjOCcnDOPm68EBJAO2K00Q5mSgPAUE/'\
              'gJuBgsYLbVy6owSrtcHj90zt8PvW+z0qaIIgsIhHOa1Qw==',
            scope: 'optional'
          }
        ]
      )
    end
  end

  it 'should report all deps from package-lock.json excluding deps,if present' do
    path = 'spec/fixtures/report_node_modules/validate_exclude_dev_deps_flag'
    config_file = YAML.load_file("#{path}/salus.yaml")
    scanner_config = config_file['scanner_configs']["ReportNodeModules"]

    repo = Salus::Repo.new(path)
    scanner = Salus::Scanners::ReportNodeModules.new(repository: repo, config: scanner_config)
    scanner.run

    info = scanner.report.to_h.fetch(:info)

    expect(info[:package_lock_version]).to eq('1')

    expect(info[:dependencies]).to match_array(
      [
        {
          dependency_file: 'package-lock.json',
          type: 'node_module',
          name: 'parseurl',
          version: '1.3.2',
          source: 'https://registry.npmjs.org/parseurl/-/parseurl-1.3.2.tgz'\
            '#sha1-/CidTtiZMRlGDBViUyYs3I3mW/M=',
          scope: 'required'
        }
      ]
    )
  end

  describe '#should_run?' do
    context 'no relevant node dependency files present' do
      it 'should return false' do
        repo = Salus::Repo.new('spec/fixtures/blank_repository')
        expect(repo.package_json_present?).to eq(false)
        expect(repo.package_lock_json_present?).to eq(false)
        expect(repo.yarn_lock_present?).to eq(false)
        scanner = Salus::Scanners::ReportNodeModules.new(repository: repo, config: {})
        expect(scanner.should_run?).to eq(false)
      end
    end

    context 'package.json is present' do
      it 'should return true' do
        repo = Salus::Repo.new('spec/fixtures/report_node_modules/package_json_only')
        expect(repo.package_json_present?).to eq(true)
        expect(repo.package_lock_json_present?).to eq(false)
        expect(repo.yarn_lock_present?).to eq(false)
        scanner = Salus::Scanners::ReportNodeModules.new(repository: repo, config: {})
        expect(scanner.should_run?).to eq(true)
      end
    end

    context 'package.json and package-lock.json are present' do
      it 'should return true' do
        repo = Salus::Repo.new('spec/fixtures/report_node_modules/package_lock_json')
        expect(repo.package_json_present?).to eq(true)
        expect(repo.package_lock_json_present?).to eq(true)
        expect(repo.yarn_lock_present?).to eq(false)
        scanner = Salus::Scanners::ReportNodeModules.new(repository: repo, config: {})
        expect(scanner.should_run?).to eq(true)
      end
    end

    context 'package.json and yarn.lock are present' do
      it 'should return true' do
        repo = Salus::Repo.new('spec/fixtures/report_node_modules/yarn_lock')
        expect(repo.package_json_present?).to eq(true)
        expect(repo.package_lock_json_present?).to eq(false)
        expect(repo.yarn_lock_present?).to eq(true)
        scanner = Salus::Scanners::ReportNodeModules.new(repository: repo, config: {})
        expect(scanner.should_run?).to eq(true)
      end
    end
  end

  describe '#version_valid?' do
    context 'scanner version is valid' do
      it 'should return true' do
        repo = Salus::Repo.new("dir")
        scanner = Salus::Scanners::ReportNodeModules.new(repository: repo, config: {})
        expect(scanner.version).to eq('')
      end
    end
  end

  describe '#supported_languages' do
    context 'should return supported languages' do
      it 'should return expected langs' do
        langs = Salus::Scanners::ReportNodeModules.supported_languages
        expect(langs).to eq(['javascript'])
      end
    end
  end
end
