require 'salus/scanners/report_base'
require 'cyclonedx/base'

# Report node module dependencies.

module Salus::Scanners
  class ReportNodeModules < ReportBase
    class ParseError < StandardError; end

    SCOPE_REQUIRED = Cyclonedx::Base::SCOPE_REQUIRED
    SCOPE_OPTIONAL = Cyclonedx::Base::SCOPE_OPTIONAL
    YARN_VERSION_COMMAND = 'yarn --version'.freeze
    YARN_LIST_PACKAGES_COMMAND = 'yarn info --all --json --recursive'.freeze
    YARN_REGISTRY_SOURCE_COMMAND = 'yarn config get npmRegistryServer'.freeze
    DEFAULT_YARN_REGISTRY = 'https://registry.yarnpkg.com'.freeze
    BREAKING_VERSION = "2.0.0".freeze

    def run
      # A lockfile is the most definitive source of truth for what will run
      # in production. It also lists the dependencies of dependencies.
      # We preference parsing dependencies in the following order:
      #  - yarn.lock
      #  - package-lock.json
      # We will also always parse package.json if it's present since there is some unique data.

      if @repository.yarn_lock_present?
        record_dependencies_from_yarn_lock

        if @repository.package_json_present?
          record_dependencies_from_package_json(record_modules: false)
        end
      elsif @repository.package_lock_json_present?
        record_dependencies_from_package_lock_json

        if @repository.package_json_present?
          record_dependencies_from_package_json(record_modules: false)
        end
      elsif @repository.bower_json_present?
        record_dependencies_from_bower_json

        if @repository.package_json_present?
          record_dependencies_from_package_json(record_modules: false)
        end
      elsif @repository.package_json_present?
        record_dependencies_from_package_json(record_modules: true)
      else
        raise InvalidScannerInvocationError,
              'Cannot report on Node modules without package.json, '\
                'package-lock.json or yarn.lock files.'
      end
    end

    def should_run?
      @repository.yarn_lock_present? ||
        @repository.package_lock_json_present? ||
        @repository.bower_json_present? ||
        @repository.package_json_present?
    end

    def self.supported_languages
      ['javascript']
    end

    private

    def yarn_version
      shell_return = run_shell(YARN_VERSION_COMMAND)
      # stdout looks like "1.22.0\n"
      shell_return.stdout&.strip
    end

    def record_dependencies_from_yarn_lock
      if Gem::Version.new(yarn_version) >= Gem::Version.new(BREAKING_VERSION)
        record_dependencies_from_latest_yarn_lock
      else
        record_dependencies_from_legacy_yarn_lock
      end
    end

    def record_dependencies_from_latest_yarn_lock
      packages_shell_return = run_shell(
        YARN_LIST_PACKAGES_COMMAND, chdir: @repository.path_to_repo
      )
      unless packages_shell_return.success?
        err_msg = "ReportNodeModules: Failed to fetch yarn.lock file."
        report_stderr(err_msg)
        report_error(err_msg)
        return
      end

      source_shell_return = run_shell(
        YARN_REGISTRY_SOURCE_COMMAND, chdir: @repository.path_to_repo
      )
      source = DEFAULT_YARN_REGISTRY unless source_shell_return.success?
      # source shell output - https://registry.yarnpkg.com
      source = source_shell_return.stdout.strip

      dependencies = []
      yarn_packages = packages_shell_return.stdout
      # yarn info shell output -
      # {"value"=>"@adobe/css-tools@npm:4.0.1", "children"=>{"Version"=>"4.0.1"}}
      yarn_packages.each_line do |line|
        line = JSON.parse(line)
        name = line.dig("value")
        version = line.dig("children", "Version")

        unless name.nil? && version.nil?
          dependencies.append(
            {
              "name" => name.split("@npm:").first,
              "version" => version
            }
          )
        end
      end

      dependencies.each do |dependency|
        record_node_module(
          name: dependency["name"],
          version: dependency["version"],
          source: source,
          dependency_file: 'yarn.lock'
        )
      end
    end

    def record_dependencies_from_legacy_yarn_lock
      shell_return = run_shell("bin/parse_yarn_lock #{@repository.path_to_repo}/yarn.lock",
                               chdir: nil)

      unless shell_return.success?
        raise ParseError, "Failed to parse yarn.lock file: #{shell_return.stderr}"
      end

      yarn_lock = JSON.parse(shell_return.stdout)

      if yarn_lock['type'] != 'success'
        raise ParseError, '@yarnpkg/lockfile failed to parse the yarn.lock file.'
      end

      # At this poinst we only have block of dependencies left. Record all.
      yarn_lock['object'].each do |dependency, info|
        # The dependency will be the <package name>@<version or location>.
        parts = dependency.scan(/(.*)@(.*)/)[0]
        name = parts[0]
        version_or_location = parts[1]

        # If there is a value for "resolved" then this is the source.
        # Otherwise, it was part of the dependency and referenced directly from local file.
        source = info['resolved']
        source ||= version_or_location
        version = info['version']

        record_node_module(
          name: name,
          version: version,
          source: source,
          dependency_file: 'yarn.lock'
        )
      end
    end

    def record_dependency_from_package_lock_json(dependency, data, dev_flag)
      record_node_module(
        name: dependency,
        version: data['version'],
        source: "#{data['resolved']}#{"##{data['integrity']}" if data['integrity']}",
        dependency_file: 'package-lock.json',
        scope: dev_flag ? SCOPE_OPTIONAL : SCOPE_REQUIRED
      )
    end

    def record_dependencies_from_package_lock_json
      include_dev_dependencies = @config.fetch('include_dev_deps', true)
      package_lock = JSON.parse(@repository.package_lock_json)

      # Record the lock file version.
      report_info(
        :package_lock_version,
        package_lock.fetch('lockfileVersion', '<unknown>').to_s
      )

      # Record each dependency.
      package_lock['dependencies']&.each do |dependency, data|
        is_dev = data['dev'] || false

        if !is_dev || is_dev && include_dev_dependencies
          record_dependency_from_package_lock_json(dependency, data, is_dev)

          # Handling recursive dependencies.
          if data.key?("dependencies")
            nested_dependencies = data.dig("dependencies")
            nested_dependencies.each do |nested_dependency, nested_data|
              nested_is_dev = nested_data['dev'] || false
              if !nested_is_dev || nested_is_dev && include_dev_dependencies
                record_dependency_from_package_lock_json(nested_dependency,
                                                         nested_data, nested_is_dev)
              end
            end
          end

        end
      end
    end

    def record_dependencies_from_bower_json
      begin
        packages = JSON.parse(@repository.bower_json)
      rescue JSON::ParserError => e
        error_msg = "Failed to parse bower.json when recording bower dependencies."
        bugsnag_notify(e.message)
        report_error(error_msg)
        return
      end
      # Record if custom registry is defined in .bowerrc
      # Supports single and multiple registry -
      # {"registry": "https://registry.bower.io"}
      # {"registry": {"search": ["http://localhost:9000","https://registry.bower.io"] }}
      sources = 'https://registry.bower.io'
      if @repository.bowerrc_present?
        begin
          file = JSON.parse(@repository.bowerrc)
        rescue JSON::ParserError => e
          error_msg = "Failed to parse .bowerrc when recording bower dependencies."
          bugsnag_notify(e.message)
          report_error(error_msg)
        end
        if file.is_a?(Hash)
          if file.key?("registry") && file["registry"].is_a?(Hash) &&
              file.dig("registry", "search").is_a?(Array)
            sources = file.dig("registry", "search").join(',')
          elsif file.key?("registry") && file["registry"].is_a?(String)
            sources = file.dig("registry")
          end
        end
      end

      # Record dependencies, version from bower.json
      packages['dependencies']&.each do |name, version|
        record_node_module(
          name: name,
          version: version,
          source: sources,
          dependency_file: 'bower.json'
        )
      end
    end

    def record_dependencies_from_package_json(record_modules: true)
      packages = JSON.parse(@repository.package_json)

      # Record the npm and node versions.
      if packages['engines']
        node_version = packages['engines']['node']
        npm_version = packages['engines']['npm']
        report_info(:package_json_node_version, node_version) unless node_version.nil?
        report_info(:package_json_npm_version, npm_version) unless npm_version.nil?
      end

      return unless record_modules

      # Record each dependency. Start by checking for non npmjs.com sources
      sources = {}
      if @repository.npmrc_present?
        @repository.npmrc.split("\n").each do |line|
          # if the line is blank
          next if line.empty?

          # npmrc is ini formatted
          key = line.split('=')[0]
          value = line.split('=')[1]
          next unless key.start_with?('@')

          # This line has the format "@source_name:registry=https://source.com"
          sources[key.split(':')[0]] = value
        end
      end

      packages['dependencies']&.each do |name, version|
        # Resolve the source if possible from .npmrc
        source = if name.start_with?('@')
                   sections = name.split(%r{\/})
                   sources[sections[0]]
                 else
                   '<package manager default>'
                 end
        record_node_module(
          name: name,
          version: version,
          source: source,
          dependency_file: 'package.json'
        )
      end
    end

    def record_node_module(dependency_file:, name:, version:, source:, scope: SCOPE_REQUIRED)
      report_dependency(
        dependency_file,
        type: 'node_module',
        name: name,
        version: version,
        source: source,
        scope: scope
      )
    end
  end
end
