require 'salus/repo_searcher'

module Salus
  class Repo
    attr_reader :path_to_repo

    IMPORTANT_FILES = [
      # Ruby
      { handle: :gemfile, filename: 'Gemfile' },
      { handle: :gemfile_lock, filename: 'Gemfile.lock' },
      { handle: :ruby_version, filename: '.ruby-version' },
      # JS
      { handle: :package_json, filename: 'package.json' },
      { handle: :package_lock_json, filename: 'package-lock.json' },
      { handle: :yarn_lock, filename: 'yarn.lock' },
      { handle: :npmrc, filename: '.npmrc' },
      { handle: :bower_json, filename: 'bower.json' },
      { handle: :bowerrc, filename: '.bowerrc' },
      # Go
      # https://go.googlesource.com/proposal/+/master/design/24301-versioned-go.md#compatibility
      # dep_lock: Dependency management lock file, generated by dep ensure and
      #           dep init
      # go_mod: Go package versioning introduced in Go 1.11^
      # go_sum: Contains the expected cryptographic checksums of the content of
      #         specific module versions, go.mod & go.sum are usually pairs
      { handle: :dep_lock, filename: 'Gopkg.lock' },
      { handle: :go_mod, filename: 'go.mod' },
      { handle: :go_sum, filename: 'go.sum' },
      # Protobufs
      { handle: :proto_files, filename: '*.proto', wildcard: true },
      # Python
      { handle: :requirements_txt, filename: 'requirements.txt' },
      { handle: :setup_cfg, filename: 'setup.cfg' },
      { handle: :py_files, filename: '*.py', wildcard: true },
      # Rust
      { handle: :cargo, filename: 'Cargo.toml' },
      { handle: :cargo_lock, filename: 'Cargo.lock' },
      # Mobile Scanners
      { handle: :android_app, filename: '*.apk', wildcard: true },
      { handle: :ios_app, filename: '*.ipa', wildcard: true },
      # Java
      { handle: :pom_xml, filename: 'pom.xml' },
      { handle: :build_gradle, filename: 'build.gradle' },
      # Swift
      { handle: :package_resolved, filename: 'Package.resolved' },
      # Apple Ecosystem (macOS, iOS, etc)
      { handle: :podfile_lock, filename: 'Podfile.lock' },
      # Solidity
      { handle: :sol_file, filename: '*.sol', wildcard: true },
      { handle: :truffle_js, filename: 'truffle.js' },
      { handle: :truffle_ts, filename: 'truffle.ts' },
      { handle: :truffle_config_js, filename: 'truffle-config.js' },
      { handle: :truffle_config_ts, filename: 'truffle-config.ts' },
      { handle: :hardhat_config_js, filename: 'hardhat.config.js' },
      { handle: :hardhat_config_ts, filename: 'hardhat.config.ts' }
    ].freeze

    # Define file checkers.
    IMPORTANT_FILES.each do |file|
      define_method :"#{file[:handle]}_present?" do
        if file[:wildcard]
          files = RepoSearcher.new(@path_to_repo, {})
            .run_rg("rg", "--files", "-g", file[:filename])
          return false unless files.any?

          # Prepend path_to_repo to the relative filepath
          appended_files = files.map { |filepath| File.join(path_to_repo, filepath) }
          return appended_files
        end

        File.exist?("#{@path_to_repo}/#{file[:filename]}")
      end
    end

    # Define cached file getters.
    IMPORTANT_FILES.each do |file|
      define_method file[:handle] do
        cache_handle = "@#{file[:handle]}_contents"

        if File.exist?("#{@path_to_repo}/#{file[:filename]}")
          if instance_variable_get(cache_handle).nil?
            instance_variable_set(cache_handle, File.read("#{@path_to_repo}/#{file[:filename]}"))
          end
        end

        instance_variable_get(cache_handle)
      end
    end

    # Define file path getters.
    # Returns nil for wildcard objects with a wildcard
    # property, as there can be more than one location
    # for files fitting the pattern
    IMPORTANT_FILES.each do |file|
      define_method :"#{file[:handle]}_path" do
        return File.expand_path("#{@path_to_repo}/#{file[:filename]}") if !file[:wildcard]

        return nil
      end
    end

    def initialize(path_to_repo = nil)
      @path_to_repo = path_to_repo
    end
  end
end
