/*
 * MIT License
 *
 * Copyright (c) 2020 Dheeraj Kotwani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package com.milad.githoob.utils

import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.flow
import retrofit2.Response

abstract class SafeApiRequest {

    suspend fun <T : Any> apiRequest(
        call: suspend () -> Response<T>
    ) = flow {
        emit(Result.loading(null))
        try {
            val response = call.invoke()
            if (response.isSuccessful)
                emit(
                    Result.success(
                        data = response.body()
                    )
                )
            else
                emit(
                    Result.error(
                        msg = "Connection failed: ${response.errorBody()?.string()}",
                        data = null
                    )
                )
        } catch (ex: Exception) {
            emit(
                Result.error(
                    msg = "Connection error: ${ex.message}",
                    data = null
                )
            )
        }
    }

    suspend fun <T : Any> apiResponseCode(call: suspend () -> Response<T>): Int {
        val response = call.invoke()
        return response.code()
    }
}

class ApiException(message: String) : Exception(message)