"""
A Flask application for ubuntu.com
"""

import os
import json
from functools import wraps
import math
import flask
import jinja2
import requests
import talisker.requests
from jinja2 import ChoiceLoader, FileSystemLoader
from canonicalwebteam.blog import BlogAPI, BlogViews, build_blueprint
from canonicalwebteam.discourse import (
    DiscourseAPI,
    DocParser,
    Docs,
    EngagePages,
    TutorialParser,
    Tutorials,
)
from canonicalwebteam.flask_base.app import FlaskBase
from canonicalwebteam.search import build_search_view
from canonicalwebteam.templatefinder import TemplateFinder

from webapp.certified.views import certified_routes
from webapp.handlers import init_handlers
from webapp.login import login_handler, logout
from webapp.security.views import (
    cve,
    cve_index,
    cves_sitemap,
    notice,
    notices,
    notices_feed,
    notices_sitemap,
    single_cves_sitemap,
    single_notices_sitemap,
)
from webapp.shop.advantage.views import (
    accept_renewal,
    activate_magic_attach,
    advantage_account_users_view,
    advantage_shop_view,
    advantage_view,
    blender_shop_view,
    cancel_advantage_subscriptions,
    cancel_trial,
    delete_account_user_role,
    get_account_offers,
    get_account_users,
    get_activate_view,
    get_advantage_offers,
    get_annotated_subscriptions,
    get_contract_token,
    get_renewal,
    get_user_subscriptions,
    magic_attach_view,
    post_account_user_role,
    post_advantage_purchase,
    post_auto_renewal_settings,
    post_offer,
    pro_activate_activation_key,
    pro_page_view,
    put_account_user_role,
    put_contract_entitlements,
    get_channel_offers,
    get_distributor_view,
    get_distributor_thank_you_view,
)
from webapp.shop.cred.views import (
    activate_activation_key,
    confidentiality_agreement_webhook,
    cred_assessments,
    cred_beta_activation,
    cred_cancel_exam,
    cred_dashboard,
    cred_dashboard_upcoming_exams,
    cred_dashboard_exam_results,
    cred_dashboard_system_statuses,
    cred_exam,
    cred_home,
    cred_redeem_code,
    cred_schedule,
    cred_self_study,
    cred_shop,
    cred_manage_shop,
    cred_shop_thank_you,
    cred_shop_webhook_responses,
    cred_shop_keys,
    cred_sign_up,
    cred_thank_you,
    cred_submit_form,
    cred_syllabus_data,
    cred_your_exams,
    get_activation_key_info,
    get_activation_keys,
    get_cue_products,
    get_issued_badges,
    get_issued_badges_bulk,
    get_test_taker_stats,
    issue_credly_badge,
    get_cred_user_permissions,
    get_my_issued_badges,
    get_webhook_response,
    issue_badges,
    rotate_activation_key,
    cancel_scheduled_exam,
)
from webapp.shop.views import (
    account_view,
    checkout,
    download_invoice,
    ensure_purchase_account,
    get_customer_info,
    get_last_purchase_ids,
    get_purchase,
    get_purchase_account_status,
    get_shop_status_page,
    invoices_view,
    maintenance_check,
    payment_methods_view,
    post_anonymised_customer_info,
    post_customer_info,
    post_payment_methods,
    post_purchase_calculate,
    post_retry_purchase,
    support,
)
from webapp.views import (
    BlogCustomGroup,
    BlogCustomTopic,
    BlogRedirects,
    BlogSitemapIndex,
    BlogSitemapPage,
    account_query,
    appliance_install,
    appliance_portfolio,
    build_case_study_index,
    build_engage_index,
    build_engage_page,
    build_engage_pages_sitemap,
    build_tutorials_index,
    build_tutorials_query,
    download_server_steps,
    download_thank_you,
    engage_thank_you,
    french_why_openstack,
    german_why_openstack,
    get_user_country_by_tz,
    json_asset_query,
    marketo_submit,
    mirrors_query,
    openstack_engage,
    openstack_install,
    releasenotes_redirect,
    show_template,
    sitemap_index,
    spanish_why_openstack,
    subscription_centre,
    thank_you,
    unlisted_engage_page,
    navigation_nojs,
)

DISCOURSE_API_KEY = os.getenv("DISCOURSE_API_KEY")
DISCOURSE_API_USERNAME = os.getenv("DISCOURSE_API_USERNAME")

CHARMHUB_DISCOURSE_API_KEY = os.getenv("CHARMHUB_DISCOURSE_API_KEY")
CHARMHUB_DISCOURSE_API_USERNAME = os.getenv("CHARMHUB_DISCOURSE_API_USERNAME")

# Set up application
# ===

app = FlaskBase(
    __name__,
    "ubuntu.com",
    template_folder="../templates",
    template_404="404.html",
    template_500="500.html",
    static_folder="../static",
)

# ChoiceLoader attempts loading templates from each path in successive order
loader = ChoiceLoader(
    [
        FileSystemLoader("templates"),
        FileSystemLoader("node_modules/vanilla-framework/templates"),
    ]
)

app.jinja_loader = loader

sentry = app.extensions["sentry"]
session = talisker.requests.get_session()
charmhub_session = requests.Session()
talisker.requests.configure(charmhub_session)

discourse_api = DiscourseAPI(
    base_url="https://discourse.ubuntu.com/",
    session=session,
    api_key=DISCOURSE_API_KEY,
    api_username=DISCOURSE_API_USERNAME,
    get_topics_query_id=2,
)

charmhub_discourse_api = DiscourseAPI(
    base_url="https://discourse.charmhub.io/",
    session=charmhub_session,
    api_key=CHARMHUB_DISCOURSE_API_KEY,
    api_username=CHARMHUB_DISCOURSE_API_USERNAME,
    get_topics_query_id=2,
)

# Web tribe websites custom search ID
search_engine_id = "adb2397a224a1fe55"

init_handlers(app, sentry)

# Routes
# ===

# Simple routes
app.add_url_rule("/asset/<file_name>", view_func=json_asset_query)
app.add_url_rule("/sitemap.xml", view_func=sitemap_index)
app.add_url_rule("/account.json", view_func=account_query)
app.add_url_rule("/mirrors.json", view_func=mirrors_query)
app.add_url_rule("/marketo/submit", view_func=marketo_submit, methods=["POST"])
app.add_url_rule("/thank-you", view_func=thank_you)
app.add_url_rule("/pro/activate", view_func=get_activate_view)
app.add_url_rule(
    "/pro/activate",
    view_func=pro_activate_activation_key,
    methods=["POST"],
)
app.add_url_rule("/navigation", view_func=navigation_nojs)
app.add_url_rule("/pro/dashboard", view_func=advantage_view)
app.add_url_rule("/pro/user-subscriptions", view_func=get_user_subscriptions)
app.add_url_rule(
    "/pro/subscriptions.json", view_func=get_annotated_subscriptions
)
app.add_url_rule(
    "/pro/contracts/<contract_id>/token", view_func=get_contract_token
)
app.add_url_rule("/pro/users", view_func=advantage_account_users_view)
app.add_url_rule(
    "/pro/distributor/users", view_func=advantage_account_users_view
)
app.add_url_rule("/pro/account-users", view_func=get_account_users)
app.add_url_rule(
    "/pro/accounts/<account_id>/user",
    view_func=post_account_user_role,
    methods=["POST"],
)
app.add_url_rule(
    "/pro/accounts/<account_id>/user",
    view_func=put_account_user_role,
    methods=["PUT"],
)
app.add_url_rule(
    "/pro/accounts/<account_id>/user",
    view_func=delete_account_user_role,
    methods=["DELETE"],
)
app.add_url_rule("/pro/subscribe", view_func=advantage_shop_view)
app.add_url_rule("/pro/subscribe/blender", view_func=blender_shop_view)
app.add_url_rule(
    "/pro/subscribe",
    view_func=cancel_advantage_subscriptions,
    methods=["DELETE"],
)
app.add_url_rule("/pro/offer", view_func=post_offer, methods=["POST"])
app.add_url_rule(
    "/pro/set-auto-renewal",
    view_func=post_auto_renewal_settings,
    methods=["POST"],
)
app.add_url_rule(
    "/pro/renewals/<renewal_id>", view_func=get_renewal, methods=["GET"]
)
app.add_url_rule(
    "/pro/trial/<account_id>",
    view_func=cancel_trial,
    methods=["DELETE"],
)

app.add_url_rule(
    "/pro/renewals/<renewal_id>/process-payment",
    view_func=accept_renewal,
    methods=["POST"],
)

app.add_url_rule(
    "/pro/contracts/<contract_id>/entitlements",
    view_func=put_contract_entitlements,
    methods=["PUT"],
)

app.add_url_rule(
    "/pro/offers",
    view_func=get_advantage_offers,
    methods=["GET"],
)

app.add_url_rule(
    "/pro/offers.json",
    view_func=get_account_offers,
    methods=["GET"],
)

app.add_url_rule("/pro/distributor", view_func=get_distributor_view)
app.add_url_rule("/pro/distributor/shop", view_func=get_distributor_view)
app.add_url_rule(
    "/pro/distributor/thank-you", view_func=get_distributor_thank_you_view
)
app.add_url_rule(
    "/pro/channel-offers.json",
    view_func=get_channel_offers,
    methods=["GET"],
)
app.add_url_rule(
    "/pro/attach", view_func=activate_magic_attach, methods=["POST"]
)
app.add_url_rule("/pro/attach", view_func=magic_attach_view, methods=["GET"])
# shop
app.add_url_rule(
    "/account",
    view_func=account_view,
)
app.add_url_rule(
    "/account/<marketplace>/purchase-account-status",
    view_func=get_purchase_account_status,
    methods=["GET"],
)
app.add_url_rule(
    "/account/invoices",
    view_func=invoices_view,
)
app.add_url_rule("/pro/distributor/invoice", view_func=invoices_view)
app.add_url_rule(
    "/account/invoices/download/<purchase_id>",
    view_func=download_invoice,
)
app.add_url_rule(
    "/account/payment-methods",
    view_func=payment_methods_view,
)
app.add_url_rule(
    "/account/payment-methods",
    view_func=post_payment_methods,
    methods=["POST"],
)
app.add_url_rule(
    "/account/purchase-account",
    view_func=ensure_purchase_account,
    methods=["POST"],
)
app.add_url_rule(
    "/account/customer-info/<account_id>",
    view_func=get_customer_info,
    methods=["GET"],
)
app.add_url_rule(
    "/account/customer-info",
    view_func=post_customer_info,
    methods=["POST"],
)
app.add_url_rule(
    "/account/customer-info-anon",
    view_func=post_anonymised_customer_info,
    methods=["POST"],
)
app.add_url_rule(
    "/account/purchases/<purchase_id>",
    view_func=get_purchase,
    methods=["GET"],
)
app.add_url_rule(
    "/account/purchases/<purchase_id>/retry",
    view_func=post_retry_purchase,
    methods=["POST"],
)
app.add_url_rule("/support", view_func=support)
app.add_url_rule(
    "/account/last-purchase-ids/<account_id>",
    view_func=get_last_purchase_ids,
)
app.add_url_rule(
    "/pro",
    view_func=pro_page_view,
    methods=["GET"],
)
app.add_url_rule(
    "/pro/purchase",
    view_func=post_advantage_purchase,
    methods=["POST"],
    defaults={"preview": False},
)
app.add_url_rule(
    "/pro/purchase/preview",
    view_func=post_advantage_purchase,
    methods=["POST"],
    defaults={"preview": True},
)
app.add_url_rule(
    "/account/checkout",
    view_func=checkout,
    methods=["GET"],
)
app.add_url_rule(
    "/account/<marketplace>/purchase/calculate",
    view_func=post_purchase_calculate,
    methods=["POST"],
)
app.add_url_rule(
    "/pro/status",
    view_func=get_shop_status_page,
    methods=["GET"],
)
app.add_url_rule(
    "/pro/maintenance-check",
    view_func=maintenance_check,
    methods=["GET"],
)

# end of shop

app.add_url_rule(
    (
        "/download"
        "/<regex('server|desktop|cloud|raspberry-pi'):category>"
        "/thank-you"
    ),
    view_func=download_thank_you,
)

app.add_url_rule(
    "/download/server",
    methods=["GET", "POST"],
    view_func=download_server_steps,
)

app.add_url_rule("/getubuntu/releasenotes", view_func=releasenotes_redirect)
app.add_url_rule(
    "/search",
    "search",
    build_search_view(
        app,
        session=session,
        template_path="search.html",
        search_engine_id=search_engine_id,
    ),
)

app.add_url_rule(
    (
        "/appliance/<regex('[a-z-]+'):appliance>/"
        "<regex('(raspberry-pi2?|intel-nuc|vm)'):device>"
    ),
    view_func=appliance_install,
)
app.add_url_rule(
    "/appliance/portfolio",
    view_func=appliance_portfolio,
)

# blog section

blog_views = BlogViews(
    api=BlogAPI(session=session, thumbnail_width=555, thumbnail_height=311),
    excluded_tags=[3184, 3265, 3408, 3960, 4491],
    per_page=11,
    blog_title="Ubuntu blog",
)
app.add_url_rule(
    "/blog/topics/<regex('maas|design|juju|robotics|snapcraft'):slug>",
    view_func=BlogCustomTopic.as_view("blog_topic", blog_views=blog_views),
)
app.add_url_rule(
    "/blog/<regex('cloud-and-server|desktop|internet-of-things|people-and-culture'):slug>",  # noqa: E501
    view_func=BlogCustomGroup.as_view("blog_group", blog_views=blog_views),
)
app.add_url_rule(
    "/blog/sitemap.xml",
    view_func=BlogSitemapIndex.as_view("sitemap", blog_views=blog_views),
)
app.add_url_rule(
    "/blog/sitemap/<regex('.+'):slug>.xml",
    view_func=BlogSitemapPage.as_view("sitemap_page", blog_views=blog_views),
)
app.add_url_rule(
    "/blog/<slug>",
    view_func=BlogRedirects.as_view("blog_redirects", blog_views=blog_views),
)
app.register_blueprint(build_blueprint(blog_views), url_prefix="/blog")

# usn section
app.add_url_rule("/security/notices", view_func=notices)

app.add_url_rule(
    "/security/notices/<notice_id>",
    view_func=notice,
)

app.add_url_rule(
    r"/security/notices/<regex('(lsn-|LSN-|usn-|USN-)\d{1,10}-\d{1,2}'):notice_id>",  # noqa: E501
    view_func=notice,
)

app.add_url_rule("/security/notices/<feed_type>.xml", view_func=notices_feed)

app.add_url_rule(
    "/security/notices/sitemap-<regex('[0-9]*'):offset>.xml",
    view_func=single_notices_sitemap,
)

app.add_url_rule("/security/notices/sitemap.xml", view_func=notices_sitemap)

app.add_url_rule(
    "/security/cves/sitemap-<regex('[0-9]*'):offset>.xml",
    view_func=single_cves_sitemap,
)

app.add_url_rule("/security/cves/sitemap.xml", view_func=cves_sitemap)

# cve section
app.add_url_rule("/security/cves", view_func=cve_index)

app.add_url_rule(
    r"/security/<regex('(cve-|CVE-)\d{4}-\d{4,7}'):cve_id>", view_func=cve
)

# Login
app.add_url_rule("/login", methods=["GET", "POST"], view_func=login_handler)
app.add_url_rule("/logout", view_func=logout)

# Engage pages and takeovers from Discourse
# This section needs to provide takeover data for /
engage_pages_discourse_api = DiscourseAPI(
    base_url="https://discourse.ubuntu.com/",
    session=session,
    get_topics_query_id=14,
    api_key=DISCOURSE_API_KEY,
    api_username=DISCOURSE_API_USERNAME,
)
takeovers_path = "/takeovers"
discourse_takeovers = EngagePages(
    api=engage_pages_discourse_api,
    page_type="takeovers",
    category_id=106,
    exclude_topics=[28426, 17250],
)

engage_path = "/engage"
engage_pages = EngagePages(
    api=engage_pages_discourse_api,
    category_id=51,
    page_type="engage-pages",
    exclude_topics=[17229, 18033, 17250],
)

app.add_url_rule(
    "/engage/sitemap.xml",
    view_func=build_engage_pages_sitemap(engage_pages),
)

app.add_url_rule(
    "/openstack/resources", view_func=openstack_engage(engage_pages)
)
# Custom engage page in German
app.add_url_rule(
    "/engage/de/warum-openstack",
    view_func=german_why_openstack,
)
app.add_url_rule(
    "/engage/fr/pourquoi-openstack",
    view_func=french_why_openstack,
)
app.add_url_rule(
    "/engage/es/por-que-openstack",
    view_func=spanish_why_openstack,
)
app.add_url_rule(engage_path, view_func=build_engage_index(engage_pages))
app.add_url_rule(
    "/engage/<page>",
    defaults={"language": None},
    view_func=build_engage_page(engage_pages),
)
app.add_url_rule("/case-study", view_func=build_case_study_index)
app.add_url_rule(
    "/engage/<language>/<page>",
    endpoint="language-engage-page",
    view_func=build_engage_page(engage_pages),
)
app.add_url_rule(
    "/engage/<page>/thank-you",
    defaults={"language": None},
    view_func=engage_thank_you(engage_pages),
)
app.add_url_rule(
    "/engage/<language>/<page>/thank-you",
    endpoint="alternative_thank-you",
    view_func=engage_thank_you(engage_pages),
)
app.add_url_rule(
    "/engage/unlisted/<slug>",
    view_func=unlisted_engage_page,
)


def takeovers_json():
    active_takeovers = discourse_takeovers.parse_active_takeovers()
    response = flask.jsonify(active_takeovers)
    response.cache_control.max_age = "300"

    return response


def takeovers_index():
    page = flask.request.args.get("page", default=1, type=int)
    limit = 50
    offset = (page - 1) * limit
    (
        all_takeovers,
        count,
        active_count,
        total_current,
    ) = discourse_takeovers.get_index(limit=limit, offset=offset)
    total_pages = math.ceil(count / limit)

    return flask.render_template(
        "takeovers/index.html",
        takeovers=all_takeovers,
        active_count=active_count,
        total_count=total_current,
        total_pages=total_pages,
        current_page=page,
    )


app.add_url_rule("/takeovers.json", view_func=takeovers_json)
app.add_url_rule("/takeovers", view_func=takeovers_index)

core_services_guide_url = "/core/services/guide"
core_services_guide = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=27473,
        url_prefix=core_services_guide_url,
    ),
    document_template="core/services/guide/document.html",
    url_prefix=core_services_guide_url,
    blueprint_name="core-services-guide",
)

app.add_url_rule(
    "/core/services/guide/search",
    "core-services-guide-search",
    build_search_view(
        app,
        session=session,
        site="ubuntu.com/core/services/guide",
        template_path="core/services/guide/search-results.html",
        search_engine_id=search_engine_id,
    ),
)

core_services_guide.init_app(app)

app.add_url_rule("/user-country-tz.json", view_func=get_user_country_by_tz)

# All other routes
template_finder_view = TemplateFinder.as_view("template_finder")
template_finder_view._exclude_xframe_options_header = True
app.add_url_rule("/", view_func=template_finder_view)
app.add_url_rule("/<path:subpath>", view_func=template_finder_view)

# Server docs
url_prefix = "/server/docs"
server_docs = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=11322,
        url_prefix=url_prefix,
    ),
    document_template="/server/docs/document.html",
    url_prefix=url_prefix,
    blueprint_name="server-docs",
)

# Server docs search
app.add_url_rule(
    "/server/docs/search",
    "server-docs-search",
    build_search_view(
        app,
        session=session,
        site="ubuntu.com/server/docs",
        template_path="/server/docs/search-results.html",
        search_engine_id=search_engine_id,
    ),
)

server_docs.init_app(app)

# Community docs
url_prefix = "/community"
community_docs = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=33115,
        url_prefix=url_prefix,
    ),
    document_template="/community/docs/document.html",
    url_prefix=url_prefix,
    blueprint_name="community-docs",
)

# Community docs search
app.add_url_rule(
    "/community/search",
    "community-search",
    build_search_view(
        app,
        session=session,
        site="ubuntu.com/community",
        template_path="/community/docs/search-results.html",
        search_engine_id=search_engine_id,
    ),
)

community_docs.init_app(app)

# Allow templates to be queried from discourse.ubuntu.com
app.add_url_rule(
    "/templates/<filename>",
    "templates",
    view_func=show_template,
)

tutorials_path = "/tutorials"
tutorials_docs = Tutorials(
    parser=TutorialParser(
        api=discourse_api,
        index_topic_id=13611,
        url_prefix=tutorials_path,
    ),
    document_template="/tutorials/tutorial.html",
    url_prefix=tutorials_path,
    blueprint_name="tutorials",
)
app.add_url_rule(
    tutorials_path,
    view_func=build_tutorials_index(session, tutorials_docs),
)
tutorials_docs.init_app(app)

app.add_url_rule(
    "/tutorials.json", view_func=build_tutorials_query(tutorials_docs)
)

# Ceph docs
ceph_docs = Docs(
    parser=DocParser(
        api=discourse_api, index_topic_id=17250, url_prefix="/ceph/docs"
    ),
    document_template="/ceph/docs/document.html",
    url_prefix="/ceph/docs",
    blueprint_name="ceph",
)
ceph_docs.init_app(app)

# Ceph docs search
app.add_url_rule(
    "/ceph/docs/search",
    "ceph-docs-search",
    build_search_view(
        app,
        session=session,
        site="ubuntu.com/ceph/docs",
        template_path="ceph/docs/search-results.html",
        search_engine_id=search_engine_id,
    ),
)

# Core docs
core_docs = Docs(
    parser=DocParser(
        api=discourse_api, index_topic_id=19764, url_prefix="/core/docs"
    ),
    document_template="/core/docs/document.html",
    url_prefix="/core/docs",
    blueprint_name="core",
)
# Core docs search
app.add_url_rule(
    "/core/docs/search",
    "core-docs-search",
    build_search_view(
        app,
        session=session,
        site="ubuntu.com/core/docs",
        template_path="/core/docs/search-results.html",
        search_engine_id=search_engine_id,
    ),
)
core_docs.init_app(app)

# Core docs - Modem Manager
core_modem_manager_docs = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=19901,
        url_prefix="/core/docs/modem-manager",
    ),
    document_template="/core/docs/document.html",
    url_prefix="/core/docs/modem-manager",
    blueprint_name="modem-manager",
)
core_modem_manager_docs.init_app(app)

# Core docs - Bluetooth (bluez) docs
core_bluetooth_docs = Docs(
    parser=DocParser(
        api=discourse_api, index_topic_id=19971, url_prefix="/core/docs/bluez"
    ),
    document_template="/core/docs/document.html",
    url_prefix="/core/docs/bluez",
    blueprint_name="bluez",
)
core_bluetooth_docs.init_app(app)

# Core docs - NetworkManager
core_network_manager_docs = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=19917,
        url_prefix="/core/docs/networkmanager",
    ),
    document_template="/core/docs/document.html",
    url_prefix="/core/docs/networkmanager",
    blueprint_name="networkmanager",
)
core_network_manager_docs.init_app(app)

# Core docs - wp-supplicant
core_wpa_supplicant_docs = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=19943,
        url_prefix="/core/docs/wpa-supplicant",
    ),
    document_template="/core/docs/document.html",
    url_prefix="/core/docs/wpa-supplicant",
    blueprint_name="wpa-supplicant",
)
core_wpa_supplicant_docs.init_app(app)

# Core docs - easy-openvpn
core_easy_openvpn_docs = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=19950,
        url_prefix="/core/docs/easy-openvpn",
    ),
    document_template="/core/docs/document.html",
    url_prefix="/core/docs/easy-openvpn",
    blueprint_name="easy-openvpn",
)
core_easy_openvpn_docs.init_app(app)

# Core docs - wifi-ap
core_wifi_ap_docs = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=19959,
        url_prefix="/core/docs/wifi-ap",
    ),
    document_template="/core/docs/document.html",
    url_prefix="/core/docs/wifi-ap",
    blueprint_name="wifi-ap",
)
core_wifi_ap_docs.init_app(app)

# Core docs - alsa-utils
core_als_autils_docs = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=19995,
        url_prefix="/core/docs/alsa-utils",
    ),
    document_template="/core/docs/document.html",
    url_prefix="/core/docs/alsa-utils",
    blueprint_name="alsa-utils",
)
core_als_autils_docs.init_app(app)

# Credentials
app.add_url_rule("/credentials", view_func=cred_home)
app.add_url_rule("/credentials/self-study", view_func=cred_self_study)
app.add_url_rule("/credentials/exam-content", view_func=cred_syllabus_data)
app.add_url_rule(
    "/credentials/sign-up", view_func=cred_sign_up, methods=["GET", "POST"]
)
app.add_url_rule(
    "/credentials/thank-you", view_func=cred_thank_you, methods=["GET"]
)
app.add_url_rule(
    "/credentials/schedule",
    view_func=cred_schedule,
    methods=["GET", "POST"],
)
app.add_url_rule("/credentials/your-exams", view_func=cred_your_exams)
app.add_url_rule("/credentials/cancel-exam", view_func=cred_cancel_exam)
app.add_url_rule("/credentials/assessments", view_func=cred_assessments)
app.add_url_rule("/credentials/exam", view_func=cred_exam)
app.add_url_rule(
    "/credentials/<string:type>/products",
    view_func=get_cue_products,
    methods=["GET"],
)
app.add_url_rule(
    "/credentials/exit-survey",
    view_func=cred_submit_form,
    methods=["GET", "POST"],
)
app.add_url_rule("/credentials/shop/", view_func=cred_shop)
app.add_url_rule(
    "/credentials/shop/manage/", view_func=cred_manage_shop, methods=["GET"]
)
app.add_url_rule("/credentials/shop/<p>", view_func=cred_shop)
app.add_url_rule("/credentials/shop/keys", view_func=cred_shop_keys)
app.add_url_rule(
    "/credentials/shop/order-thank-you", view_func=cred_shop_thank_you
)
app.add_url_rule(
    "/credentials/shop/webhook_responses",
    view_func=cred_shop_webhook_responses,
)
app.add_url_rule(
    "/credentials/redeem", view_func=cred_redeem_code, methods=["GET", "POST"]
)
app.add_url_rule(
    "/credentials/redeem/<code>",
    view_func=cred_redeem_code,
    methods=["GET", "POST"],
)
app.add_url_rule(
    "/credentials/keys/list",
    view_func=get_activation_keys,
    methods=["GET"],
)
app.add_url_rule(
    "/credentials/keys/rotate/<activation_key>",
    view_func=rotate_activation_key,
    methods=["GET"],
)
app.add_url_rule(
    "/credentials/keys/activate",
    view_func=activate_activation_key,
    methods=["POST"],
)
app.add_url_rule(
    "/credentials/keys/<key_id>",
    view_func=get_activation_key_info,
    methods=["GET"],
)
app.add_url_rule(
    "/credentials/beta/activation",
    view_func=cred_beta_activation,
    methods=["GET", "POST"],
)
app.add_url_rule(
    "/credentials/get_webhook_response",
    view_func=get_webhook_response,
    methods=["GET"],
)
app.add_url_rule(
    "/credentials/assessment_passed",
    view_func=issue_badges,
    methods=["POST"],
)
app.add_url_rule(
    "/credentials/dashboard",
    view_func=cred_dashboard,
    methods=["GET"],
)
app.add_url_rule(
    "/credentials/dashboard/<path:path>",
    view_func=cred_dashboard,
    methods=["GET"],
    defaults={"path": ""},
)
app.add_url_rule(
    "/credentials/api/upcoming-exams",
    view_func=cred_dashboard_upcoming_exams,
    methods=["GET"],
)
app.add_url_rule(
    "/credentials/api/exam-results",
    view_func=cred_dashboard_exam_results,
    methods=["GET"],
)
app.add_url_rule(
    "/credentials/api/system-statuses",
    view_func=cred_dashboard_system_statuses,
    methods=["GET"],
)
app.add_url_rule(
    "/credentials/api/issued-badges",
    view_func=get_issued_badges,
    methods=["GET"],
)
app.add_url_rule(
    "/credentials/api/issued-badges-bulk",
    view_func=get_issued_badges_bulk,
    methods=["GET"],
)
app.add_url_rule(
    "/credentials/api/test-taker-stats",
    view_func=get_test_taker_stats,
    methods=["GET"],
)
app.add_url_rule(
    "/credentials/api/issue-credly-badge",
    view_func=issue_credly_badge,
    methods=["POST"],
)
app.add_url_rule(
    "/credentials/api/user-permissions",
    view_func=get_cred_user_permissions,
    methods=["GET"],
)
app.add_url_rule(
    "/credentials/api/cancel-scheduled-exam/<reservation_id>",
    view_func=cancel_scheduled_exam,
    methods=["DELETE"],
)

app.add_url_rule(
    "/credentials/your-badges",
    view_func=get_my_issued_badges,
    methods=["GET"],
)

app.add_url_rule(
    "/credentials/confidentiality-agreement",
    view_func=confidentiality_agreement_webhook,
    methods=["POST"],
)

# Charmed OpenStack docs
openstack_docs = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=20991,
        url_prefix="/openstack/docs",
    ),
    document_template="openstack/docs/document.html",
    url_prefix="/openstack/docs",
    blueprint_name="openstack-docs",
)

# Charmed OpenStack docs search
app.add_url_rule(
    "/openstack/docs/search",
    "openstack-docs-search",
    build_search_view(
        app,
        session=session,
        site="ubuntu.com/openstack/docs",
        template_path="openstack/docs/search-results.html",
        search_engine_id=search_engine_id,
    ),
)

openstack_docs.init_app(app)

# Security Livepatch docs
security_livepatch_docs = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=22723,
        url_prefix="/security/livepatch/docs",
    ),
    document_template="/security/livepatch/docs/document.html",
    url_prefix="/security/livepatch/docs",
    blueprint_name="security-livepatch-docs",
)

# Security Livepatch search
app.add_url_rule(
    "/security/livepatch/docs/search",
    "security-livepatch-docs-search",
    build_search_view(
        app,
        session=session,
        site="ubuntu.com/security/livepatch/docs",
        template_path="/security/livepatch/docs/search-results.html",
        search_engine_id=search_engine_id,
    ),
)

security_livepatch_docs.init_app(app)

# Security Certifications docs
security_certs_docs = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=22810,
        url_prefix="/security/certifications/docs",
    ),
    document_template="/security/certifications/docs/document.html",
    url_prefix="/security/certifications/docs",
    blueprint_name="security-certs-docs",
)

# Security Certifications search
app.add_url_rule(
    "/security/certifications/docs/search",
    "security-certs-docs-search",
    build_search_view(
        app,
        session=session,
        site="ubuntu.com/security/certifications/docs",
        template_path="/security/certifications/docs/search-results.html",
        search_engine_id=search_engine_id,
    ),
)

security_certs_docs.init_app(app)

# Landscape docs
landscape_docs = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=23070,
        url_prefix="/landscape/docs",
    ),
    document_template="/landscape/docs/document.html",
    url_prefix="/landscape/docs",
    blueprint_name="landscape-docs",
)

# Landscape search
app.add_url_rule(
    "/landscape/docs/search",
    "landscape-docs-search",
    build_search_view(
        app,
        session=session,
        site="ubuntu.com/landscape/docs",
        template_path="/landscape/docs/search-results.html",
        search_engine_id=search_engine_id,
    ),
)

landscape_docs.init_app(app)

# Robotics docs
robotics_docs = Docs(
    parser=DocParser(
        api=discourse_api,
        index_topic_id=34683,
        url_prefix="/robotics/docs",
    ),
    document_template="/robotics/docs/document.html",
    url_prefix="/robotics/docs",
    blueprint_name="robotics-docs",
)

# Robotics search
app.add_url_rule(
    "/robotics/docs/search",
    "robotics-docs-search",
    build_search_view(
        app,
        session=session,
        site="ubuntu.com/robotics/docs",
        template_path="/robotics/docs/search-results.html",
        search_engine_id=search_engine_id,
    ),
)

robotics_docs.init_app(app)

certified_routes(app)

# Override openstack/install
app.add_url_rule(
    "/openstack/install",
    view_func=openstack_install,
)

# Subscription centre
app.add_url_rule(
    "/subscription-centre",
    view_func=subscription_centre,
    methods=["GET", "POST"],
)


# HPE blog section
def render_blogs():
    blogs = BlogViews(
        api=BlogAPI(
            session=session, thumbnail_width=555, thumbnail_height=311
        ),
        tag_ids=[4307],
        per_page=3,
        blog_title="HPE blogs",
    )
    hpe_articles = blogs.get_tag("hpe")
    return flask.render_template(
        "/hpe/index.html", blogs=hpe_articles["articles"]
    )


app.add_url_rule("/hpe", view_func=render_blogs)


# Public-cloud blog section
# tag_ids:
# public-cloud - 1350, aws - 1205, azure - 1748, google-cloud - 4191,
# ubuntu-on-aws - 4478, ubuntu-on-gcp - 4387, ubuntu-on-azure - 4540
def render_public_cloud_blogs():
    blogs = BlogViews(
        api=BlogAPI(
            session=session, thumbnail_width=1000, thumbnail_height=700
        ),
        tag_ids=[1205, 1350, 1748, 4191, 4478, 4540, 4387],
        per_page=3,
        blog_title="Public-cloud blogs",
    )
    public_cloud_articles = blogs.get_index()["articles"]
    sorted_articles = sorted(public_cloud_articles, key=lambda x: x["date"])
    return flask.render_template(
        "/cloud/public-cloud.html", blogs=sorted_articles
    )


app.add_url_rule("/cloud/public-cloud", view_func=render_public_cloud_blogs)


# Supermicro blog section
def render_supermicro_blogs():
    blogs = BlogViews(
        api=BlogAPI(
            session=session, thumbnail_width=555, thumbnail_height=311
        ),
        tag_ids=[2247],
        per_page=3,
        blog_title="Supermicro blogs",
    )
    supermicro_articles = blogs.get_tag("supermicro")
    return flask.render_template(
        "/supermicro/index.html", blogs=supermicro_articles["articles"]
    )


app.add_url_rule("/supermicro", view_func=render_supermicro_blogs)


def render_form(form, template_path, child=False):
    @wraps(render_form)
    def wrapper_func():
        try:
            if child:
                return flask.render_template(
                    template_path + ".html",
                    fieldsets=form["fieldsets"],
                    formData=form["formData"],
                    isModal=form.get("isModal"),
                    modalId=form.get("modalId"),
                    path=template_path,
                )
            else:
                return flask.render_template(
                    template_path + ".html",
                    fieldsets=form["fieldsets"],
                    formData=form["formData"],
                    isModal=form.get("isModal"),
                    modalId=form.get("modalId"),
                )
        except jinja2.exceptions.TemplateNotFound:
            flask.abort(
                404, description=f"Template {form['templatePath']} not found."
            )

    return wrapper_func


def set_form_rules():
    file_path = os.path.join(app.static_folder, "files", "forms-data.json")
    with open(file_path) as forms_json:
        data = json.load(forms_json)
        for path, form in data["forms"].items():
            try:
                if "childrenPaths" in form:
                    for child_path in form["childrenPaths"]:
                        app.add_url_rule(
                            child_path,
                            view_func=render_form(
                                form, child_path, child=True
                            ),
                            endpoint=child_path,
                        )
                app.add_url_rule(
                    path,
                    view_func=render_form(
                        form, form["templatePath"].split(".")[0]
                    ),
                    endpoint=path,
                )
            except AssertionError:
                app.logger.error(
                    f"Error setting form rules for {path} \n", AssertionError
                )


# Disabling for now, the forms in form-data.json are for testing purposes
set_form_rules()
