import {
  isValidCveId,
  isDomNode,
  disableField,
  enableField,
  disableOptionIfSelected,
} from "./cve-search.js";

describe("isValidCveId", () => {
  it("returns falsy if given no value", () => {
    expect(isValidCveId()).toBeFalsy();
  });

  it("returns truthy if given a valid CVE ID", () => {
    const validCveIds = [
      "cve-1234-1324",
      "CVE-1234-1234",
      "cve-1234-12345",
      "CVE-1234-12345",
      "cve-1234-123456",
      "CVE-1234-123456",
      "cve-1234-1234567",
      "CVE-1234-1234567",
      "1234-1234",
      "1234-12345",
      "1234-123456",
      "1234-1234567",
    ];

    validCveIds.forEach((cveId) => {
      expect(isValidCveId(cveId)).toBeTruthy();
    });
  });

  it("returns falsy if given an invalid CVE ID", () => {
    const invalidCveIds = [
      "cve-123-1234",
      "CVE-123-1234",
      "cve-123-12345",
      "CVE-123-12345",
      "cve-123-123456",
      "CVE-123-1234567",
      "cve-1234-123",
      "CVE-1234-123",
      "cve-1234-12345678",
      "CVE-1234-12345678",
      "123-1234",
      "123-12345",
      "123-123456",
      "123-1234567",
      "1234-123",
      "1234-12345678",
    ];

    invalidCveIds.forEach((cveId) => {
      expect(isValidCveId(cveId)).toBeFalsy();
    });
  });
});

describe("isDomNode", () => {
  it("returns falsy if no argument is given", () => {
    expect(isDomNode()).toBeFalsy();
  });

  it("returns truthy if argument is a DOM node", () => {
    const input = document.createElement("input");
    expect(isDomNode(input)).toBeTruthy();
  });

  it("returns falsy if argument is not a DOM node", () => {
    const input = '<input type="text">';
    expect(isDomNode(input)).toBeFalsy();
  });
});

describe("disableField", () => {
  it("returns falsy if no argument given", () => {
    expect(disableField()).toBeFalsy();
  });

  it("sets the `disabled` attribute to `true` on the field", () => {
    const input = document.createElement("input");
    disableField(input);
    expect(input.getAttribute("disabled")).toBeTruthy();

    input.setAttribute("disabled", false);
    expect(input.getAttribute("disabled")).toBe("false");
  });
});

describe("enableField", () => {
  it("returns falsy if no argument given", () => {
    expect(enableField()).toBeFalsy();
  });

  it("removes the `disabled` attribte from the field", () => {
    const input = document.createElement("input");
    input.setAttribute("disabled", true);
    enableField(input);
    expect(input.getAttribute("disabled")).toBeFalsy();
  });
});

describe("disableOptionIfSelected", () => {
  it("has no effect if option has no value", () => {
    const option = { value: "", selected: false, disabled: false };
    disableOptionIfSelected(option, []);
    expect(option.disabled).toBeFalsy();
  });

  it("has no effect if option is not selected and not in list", () => {
    const selectedVersions = ["20.04", "14.04", "current"];
    const option = { value: "18.04", selected: false, disabled: false };
    disableOptionIfSelected(option, selectedVersions);
    expect(option.disabled).toBeFalsy();
  });

  it("disables option if not selected and is in list", () => {
    const selectedVersions = ["20.04", "14.04", "current"];
    const option = { value: "14.04", selected: false, disabled: false };
    disableOptionIfSelected(option, selectedVersions);
    expect(option.disabled).toBeTruthy();
  });
});
