/**
 * isValidCveId
 *
 * Checks if the supplied value is a valid CVE ID in the format
 *
 * A CVE ID consists consists of `cve-` (case-insensitive)
 * followed by 4 integers, follwed by `-`, followed by 4 to 7 integers
 *
 * @param {string} value
 * @returns {boolean}
 */
function isValidCveId(value) {
  if (!value) {
    return;
  }

  const cveIdPattern = /^(cve-)?\d{4}-\d{4,7}$/gi;

  return value.match(cveIdPattern);
}

/**
 * isDomNode
 *
 * Checks that the supplied value is a DOM node
 *
 * @param {HTMLElement} node
 * @returns {boolean}
 */
function isDomNode(node) {
  try {
    return node instanceof HTMLElement;
  } catch (e) {
    return;
  }
}

/**
 * disableField
 *
 * Sets the `disabled` attribute on the supplied DOM node

 * @param {HTMLElement} field
 * @returns {undefined}
 */
function disableField(field) {
  if (!isDomNode(field)) {
    return;
  }

  field.setAttribute("disabled", true);
}

/**
 * enableField
 *
 * Removes the `disabled` attribute from the supplied DOM node
 *
 * @param {*} field
 * @returns {undefined}
 */
function enableField(field) {
  if (!isDomNode(field)) {
    return;
  }

  field.removeAttribute("disabled");
}

/**
 * attachEvents
 *
 * Adds event listeners to search row buttons and inputs
 *
 * @returns {undefined}
 */
function attachEvents() {
  const addRowButtons = document.querySelectorAll(".js-add-row");
  const removeRowButtons = document.querySelectorAll(".js-remove-row");
  const ubuntuVersionInputs = document.querySelectorAll(
    ".js-ubuntu-version-input",
  );
  const statusInputs = document.querySelectorAll(".js-status-input");

  addRowButtons.forEach((addRowButton) => {
    addRowButton.addEventListener("click", addRow);
  });

  removeRowButtons.forEach((removeRowButton) => {
    removeRowButton.addEventListener("click", removeRow);
  });

  ubuntuVersionInputs.forEach((input) => {
    input.addEventListener("change", () => {
      handleButtons();
      disableSelectedVersions();
    });
  });

  statusInputs.forEach((input) => {
    input.addEventListener("change", () => {
      handleButtons();
    });
  });
}

/**
 * addRow
 *
 * Adds Ubuntu version/status row to CVE search form
 *
 * @param {Event} e
 * @returns {undefined}
 */
function addRow(e) {
  e.preventDefault();

  const rowTemplate = document.getElementById("version-field-row-template");
  const newRowContainer = document.getElementById("new-row-container");
  const newRow = rowTemplate.content.cloneNode(true);

  newRowContainer.appendChild(newRow);

  attachEvents();

  handleButtons();

  const ubuntuVersionInputs = document.querySelectorAll(
    ".js-ubuntu-version-input",
  );
  const statusInputs = document.querySelectorAll(".js-status-input");

  ubuntuVersionInputs.forEach((input) => {
    input.addEventListener("change", () => {
      handleButtons();
      disableSelectedVersions();
    });
  });

  statusInputs.forEach((input) => {
    input.addEventListener("change", () => {
      handleButtons();
    });
  });

  disableSelectedVersions();
}

/**
 * removeRow
 *
 * Removes Ubuntu version/status row from CVE search form
 *
 * @param {Event} e
 * @returns {undefined}
 */
function removeRow(e) {
  e.preventDefault();

  const row = e.currentTarget.closest(".js-version-field-row");

  row.parentNode.removeChild(row);

  handleButtons();
}

/**
 * handleButtons
 *
 * Handles the enabled/disabled state of the add/remove row buttons
 * from the CVE search form
 */
function handleButtons() {
  const STATE = {
    allowNewVersionStatusRow: false,
    versionsAndStatuses: [],
  };

  const rows = document.querySelectorAll(".js-version-field-row");

  STATE.versionsAndStatuses = [...rows].map((row) => {
    const ubuntuVersionInput = row.querySelector(".js-ubuntu-version-input");
    const statusInput = row.querySelector(".js-status-input");

    return {
      version: ubuntuVersionInput.value,
      status: statusInput.value,
    };
  });

  const values = STATE.versionsAndStatuses;

  for (let i = 0, ii = values.length; i < ii; i++) {
    if (values[i].version === "" && values[i].status === "") {
      STATE.allowNewVersionStatusRow = false;
      break;
    } else {
      STATE.allowNewVersionStatusRow = true;
    }
  }

  const addButtons = document.querySelectorAll(".js-add-row");

  addButtons.forEach((addButton) => {
    if (STATE.allowNewVersionStatusRow) {
      enableField(addButton);
    } else {
      disableField(addButton);
    }
  });
}

/**
 * disableOptionIfSelected
 *
 * @param {Object} option - A select option
 * @param {Array} selectedVersions - All versions currently selected
 */
function disableOptionIfSelected(option, selectedVersions) {
  if (option.value === "") {
    return;
  }

  if (selectedVersions.includes(option.value) && !option.selected) {
    option.disabled = true;
  }
}

/**
 * disableSelectedVersions
 *
 * Disables options in the Ubuntu version select box if
 * that version is already selected in a previous row
 */
function disableSelectedVersions() {
  const inputs = document.querySelectorAll(".js-ubuntu-version-input");
  const selectedVersions = [...inputs].map((input) => input.value);

  inputs.forEach((input) => {
    [...input.options].forEach((option) => {
      disableOptionIfSelected(option, selectedVersions);
    });
  });
}

export {
  isValidCveId,
  isDomNode,
  disableField,
  enableField,
  attachEvents,
  handleButtons,
  disableOptionIfSelected,
  disableSelectedVersions,
};
