﻿declare namespace ASP {
    interface Server {
        /**
         * Sets or returns the maximum number of seconds a script can run before it is terminated.
         * Default is 90 seconds.
         */
        ScriptTimeout: number;

        /**
         * Creates an instance of an object.
         * Note: Objects created with this method have page scope. They are destroyed when the
         * server are finished processing the current ASP page. To create an object with session
         * or application scope, you can either use the <object> tag in the Global.asa file,
         * or store the object in a session or application variable.
         * @param progID The type of object to create.
         */
        CreateObject(progID: string): any;

        /**
         * Executes an ASP file from inside another ASP file. After executing the called .asp file,
         * the control is returned to the original .asp file.
         * @param path The location of the ASP file to execute.
         */
        Execute(path: string): void;

        /**
         * Returns an ASPError object that describes the error condition that occurred.
         *
         * By default, a Web site uses the file \iishelp\common\500-100.asp for processing ASP errors.
         * You can either use this file, or create your own. If you want to change the ASP file for
         * processing the 500;100 custom errors you can use the IIS snap-in.
         *
         * Note: A 500;100 custom error will be generated if IIS encounters an error while processing
         * either an ASP file or the application's Global.asa file.
         *
         * Note: This method is available only before the ASP file has sent any content to the browser.
         */
        GetLastError(): ASPError;

        /**
         * Applies HTML encoding to a specified string.
         * @param str The string to encode.
         */
        HTMLEncode(str: string): string;

        /**
         * Maps a specified path to a physical path.
         * @param path A relative or virtual path to map to a physical path. If this parameter starts
         *     with / or \, it returns a path as if this parameter is a full virtual path. If this
         *     parameter doesn't start with / or \, it returns a path relative to the directory of
         *     the .asp file being processed.
         * @returns The physical path.
         */
        MapPath(path: string): string;

        /**
         * Sends (transfers) all the information created in one ASP file to a second ASP file.
         * When the second ASP page completes its tasks, it will NOT return to the first ASP page
         * (like the Execute method).
         *
         * Note: The Transfer method is an efficient alternate for the Response.Redirect. A redirect
         * forces the Web server to handle an extra request while the Server.Transfer method transfers
         * execution to a different ASP page on the server, and avoids the extra round trip.
         *
         * @param path The location of the ASP file to which control should be transferred.
         */
        Transfer(path: string): void;

        /**
         * Applies URL encoding rules to a specified string.
         * @param str The string to encode.
         */
        URLEncode(str: string): string;
    }

    /**
     * You can use the ASPError object to obtain information about an error condition that has occurred
     * in script in an ASP page. The ASPError object is returned by the Server.GetLastError method.
     * The ASPError object exposes read-only properties.
     */
    interface ASPError {
        /** Returns an error code generated by IIS. */
        ASPCode(): string;

        /** Returns a more detailed description of the error, if it is an ASP-related error. */
        ASPDescription(): string;

        /** Indicates if the source of the error was internal to ASP, the scripting language, or an object. */
        Category(): string;

        /** Indicates the column position within the .asp file that generated the error. */
        Column(): number;

        /** Returns a short description of the error. */
        Description(): string;

        /** Indicates the name of the .asp file that was being processed when the error occurred. */
        File(): string;

        /** Indicates the line within the .asp file that generated the error. */
        Line(): number;

        /** Returns the standard COM error code. */
        Number(): number;

        /** Returns the actual source code, when available, of the line that caused the error. */
        Source(): string;
    }

    /**
     * The IVariantDictionary interface exposes methods that you can use to access the
     * members of the Application.Contents Collection, Application.StaticObjects Collection,
     * Session.Contents Collection, and Session.StaticObjects Colleciton collections.
     */
    interface IVariantDictionary {
        /**
         * Retrieves an item from the specified collection.
         * @param key Identifier that indicates which item to retrieve from the collection.
         */
        (key: string): any;

        /**
         * Retrieves an item from the specified collection.
         * @param key Identifier that indicates which item to retrieve from the collection.
         */
        Item(key: string): any;

        /** Returns a unique identifier for an item in either the Contents or StaticObjects collection. */
        Key(index: number): string;

        /** Returns a unique identifier for an item in either the Contents or StaticObjects collection. */
        Key(name: string): string;

        /** Retrieves the number of items in the collection. */
        Count(): number;

        /**
         * Deletes an item from the Contents collection.
         * @param name The name of the item to remove.
         */
        Remove(name: string): void;

        /**
         * Deletes an item from the Contents collection.
         * @param index The index of the item to remove.
         */
        Remove(index: number): void;

        /** Deletes all items from the Contents collection. */
        RemoveAll(): void;
    }

    interface Application {
        Contents: IVariantDictionary;
        StaticObjects: IVariantDictionary;
        Lock(): void;
        Unlock(): void;

        /**
         * Retrieves an item from the Contents collection.
         * @param key Identifier that indicates which item to retrieve from the collection.
         */
        (key: string): any;
    }

    interface Session {
        /**
         * The CodePage property specifies the character set that will be used when displaying
         * dynamic content.
         *
         * Examples:
         *     - 1252 - ANSI Latin 1; Western European (Windows)
         *     - 65001 - utf-8
         */
        CodePage: number;

        /**
         * Sets or returns an integer that specifies a location or region. Contents like date,
         * time, and currency will be displayed according to that location or region.
         *
         * Examples:
         *     - 1033 - English - United States
         *     - 1046 - Portuguese - Brazil
         */
        LCID: number;

        /** Returns a unique id for each user. The unique id is generated by the server. */
        SessionID: string;

        /**
         * Sets or returns the timeout period (in minutes) for the Session object in this application.
         */
        Timeout: number;

        /** Contains all the items appended to the session through a script command */
        Contents: IVariantDictionary;

        /** Contains all the objects appended to the session with the HTML <object> tag */
        StaticObjects: IVariantDictionary;

        /** Destroys a user session. */
        Abandon(): void;
        
        /**
         * Retrieves an item from the Contents collection.
         * @param key Identifier that indicates which item to retrieve from the collection.
         */
        (key: string): any;
    }

    interface IRequestDictionary {
        (): IRequestDictionary;
        (name: string): IStringList;
        Item(): string;
        Item(name: string): IStringList;
        Key(index: number): string;
        Key(name: string): string;
        Count(): number;
    }

    /**
     * The IStringList interface to retrieve individual items from the string lists contained
     * in the Request.QueryString Collection, Request.Form Collection, or Request.ServerVariables
     * Collection collections.
     */
    interface IStringList {
        (): string;

        /** @param index Starts at 1. */
        (index: number): string;
        Item(): string;

        /** @param index Starts at 1. */
        Item(index: number): string;

        Count(): number;
    }

    /**
     * The Request object is used to get information from a visitor.
     */
    interface Request {
        /**
         * Contains all the field values stored in the client certificate
         */
        ClientCertificate: IRequestDictionary;

        /**
         * Contains all the cookie values sent in a HTTP request
         */
        Cookies: IRequestDictionary;

        /**
         * Contains all the form (input) values from a form that uses the post method
         */
        Form: IRequestDictionary;

        /**
         * Contains all the variable values in a HTTP query string
         */
        QueryString: IRequestDictionary;

        /**
         * Contains all the server variable values
         */
        ServerVariables: IRequestDictionary;

        /**
         * Returns the total number of bytes the client sent in the body of the request
         */
        TotalBytes: number;

        /**
         * Retrieves the data sent to the server from the client as part of a POST request
         * and stores it in a safe array (an array that stores information about the number
         * of dimensions and the bounds of its dimensions).
         *
         * Note: A call to Request.Form after a call to BinaryRead, and vice-versa, will cause an error.
         *
         * @param count Specifies how many bytes to read from the client
         */
        BinaryRead(count: number): any[];
    }

    interface ICollection {
        [index: number]: string;
        [name: string]: string;
    }

    /** The ASP Response object is used to send output to the user from the server. */
    interface Response {
        /**
         * Adds a new HTTP header and a value to the HTTP response.
         * Once a header has been added, it cannot be removed.
         * @param name The name of the new header variable (cannot contain underscores).
         * @param value The initial value of the new header variable.
         */
        AddHeader(name: string, value: string): void;

        /**
         * Adds a string to the end of the server log entry. You can call this method
         * multiple times in a script. Each time it is called it will append the specified
         * string to the log entry.
         * @param str The text to append to the log file (cannot contain any comma characters).
         */
        AppendToLog(str: string): void;

        /**
         * Writes data directly to the output without any character conversion.
         * This method is used to write image data (BLOB) from a database to a browser.
         * @param data The binary information to be sent.
         */
        BinaryWrite(data: any): void;

        /**
         * Clears any buffered HTML output.
         * Note: This method does not clear the response headers, only the response body.
         * Note: If response.Buffer is false, this method will cause a run-time error.
         */
        Clear(): void;

        /**
         * Stops processing a script, and returns the current result.
         * Note: This method will flush the buffer if Response.Buffer has been set to true.
         * If you do not want to return any output to the user, you should call Response.Clear first.
         */
        End(): void;

        /**
         * Sends buffered HTML output immediately.
         * Note: If response.Buffer is false, this method will cause a run-time error.
         */
        Flush(): void;

        /**
         * Redirects the user to a different URL.
         * @param url The URL that the user (browser) is redirected to.
         */
        Redirect(url: string): void;

        /**
         * Writes a specified string to the output.
         * @param variant The data to write.
         */
        Write(variant: any): void;

        /**
         * Specifies whether to buffer the page output or not. When the output is buffered,
         * the server will hold back the response to the browser until all of the server scripts
         * have been processed, or until the script calls the Flush or End method.
         *
         * False indicates no buffering. The server will send the output as it is processed.
         * False is default for IIS version 4.0 (and earlier). Default for IIS version 5.0 (and later) is true.
         *
         * True indicates buffering. The server will not send output until all of the scripts on
         * the page have been processed, or until the Flush or End method has been called.
         *
         * Note: If this property is set, it should be before the <html> tag in the .asp file.
         */
        Buffer: boolean;

        /**
         * Sets whether a proxy server can cache the output generated by ASP or not.
         *
         * It can be set to "Public" or "Private".
         *
         * Private is default and indicates that only private caches may cache this page.
         * Proxy servers will not cache pages with this setting.
         *
         * Public indicates public caches. Proxy servers will cache pages with this setting.
         */
        CacheControl: string;

        /**
         * Appends the name of a character-set to the content-type header in the Response object.
         * Default character set is ISO-LATIN-1.
         *
         * Note: This property will accept any string, regardless of whether it is a valid
         * character set or not, for the name.
         */
        Charset: string;

        /** Sets the HTTP content type for the Response object. */
        ContentType: string;

        /**
         * Sets how long (in minutes) a page will be cached on a browser before it expires.
         * If a user returns to the same page before it expires, the cached version is displayed.
         */
        Expires: number;

        /**
         * Sets a date and time when a page cached on a browser will expire.
         * If a user returns to the same page before this date/time, the cached version is displayed.
         */
        ExpiresAbsolute: Date;

        /** Indicates if the client has disconnected from the server. */
        IsClientConnected: boolean;

        /** Appends a value to the PICS label response header */
        Pics(text: string): void;

        /**
         * Specifies the value of the status line returned by the server.
         * Use this property to modify the status line returned by the server.
         * The value is a three-digit number and a description of that code,
         * like "404 Not Found".
         */
        Status: string;

        /**
         * Sets a cookie value. If the cookie does not exist, it will be created, and take the
         * value that is specified.
         *
         * Note: The Response.Cookies command must appear before the <html> tag.
         */
        Cookies: ICollection;
    }
}

declare var Application: ASP.Application;
declare var Server: ASP.Server;
declare var Session: ASP.Session;

/** The Request object is used to get information from a visitor. */
declare var Request: ASP.Request;

/** The ASP Response object is used to send output to the user from the server. */
declare var Response: ASP.Response;