////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//   Wireless Communication Library 7                                         //
//                                                                            //
//   Copyright (C) 2006-2024 Mike Petrichenko                                 //
//                           Soft Service Company                             //
//                           All Rights Reserved                              //
//                                                                            //
//   http://www.btframework.com                                               //
//                                                                            //
//   support@btframework.com                                                  //
//   shop@btframework.com                                                     //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#pragma once

#include <windows.h>

#include "..\Common\wclWinAPI.h"

#include "wclWiFiErrors.h"

namespace wclWiFi
{
	namespace WlanApi
	{
		/* Native WiFi declarations. */

		typedef enum _DOT11_BSS_TYPE
		{
			dot11_BSS_type_infrastructure = 1,
			dot11_BSS_type_independent = 2,
			dot11_BSS_type_any = 3
		} DOT11_BSS_TYPE, *PDOT11_BSS_TYPE;

		#define DOT11_SSID_MAX_LENGTH	32

		typedef struct _DOT11_SSID
		{
			ULONG	uSSIDLength;
			UCHAR	ucSSID[DOT11_SSID_MAX_LENGTH];
		} DOT11_SSID, *PDOT11_SSID;

		typedef DWORD DOT11_AUTH_ALGORITHM, *PDOT11_AUTH_ALGORITHM;
		  
		#define DOT11_AUTH_ALGO_80211_OPEN			1
		#define DOT11_AUTH_ALGO_80211_SHARED_KEY	2
		#define DOT11_AUTH_ALGO_WPA					3
		#define DOT11_AUTH_ALGO_WPA_PSK				4
		#define DOT11_AUTH_ALGO_WPA_NONE			5
		#define DOT11_AUTH_ALGO_RSNA				6
		#define DOT11_AUTH_ALGO_RSNA_PSK			7
		#define DOT11_AUTH_ALGO_WPA3				8
		#define DOT11_AUTH_ALGO_WPA3_ENT_192		DOT11_AUTH_ALGO_WPA3
		#define DOT11_AUTH_ALGO_WPA3_SAE			9
		#define DOT11_AUTH_ALGO_OWE					10
		#define DOT11_AUTH_ALGO_WPA3_ENT			11
		#define DOT11_AUTH_ALGO_IHV_START			0x80000000
		#define DOT11_AUTH_ALGO_IHV_END				0xffffffff

		#define DOT11_AUTH_ALGORITHM_OPEN_SYSTEM	DOT11_AUTH_ALGO_80211_OPEN
		#define DOT11_AUTH_ALGORITHM_SHARED_KEY		DOT11_AUTH_ALGO_80211_SHARED_KEY
		#define DOT11_AUTH_ALGORITHM_WPA			DOT11_AUTH_ALGO_WPA
		#define DOT11_AUTH_ALGORITHM_WPA_PSK		DOT11_AUTH_ALGO_WPA_PSK
		#define DOT11_AUTH_ALGORITHM_WPA_NONE		DOT11_AUTH_ALGO_WPA_NONE
		#define DOT11_AUTH_ALGORITHM_RSNA			DOT11_AUTH_ALGO_RSNA
		#define DOT11_AUTH_ALGORITHM_RSNA_PSK		DOT11_AUTH_ALGO_RSNA_PSK
		#define DOT11_AUTH_ALGORITHM_WPA3			DOT11_AUTH_ALGO_WPA3
		#define DOT11_AUTH_ALGORITHM_WPA3_ENT_192	DOT11_AUTH_ALGO_WPA3_ENT_192
		#define DOT11_AUTH_ALGORITHM_WPA3_SAE		DOT11_AUTH_ALGO_WPA3_SAE
		#define DOT11_AUTH_ALGORITHM_OWE			DOT11_AUTH_ALGO_OWE
		#define DOT11_AUTH_ALGORITHM_WPA3_ENT		DOT11_AUTH_ALGO_WPA3_ENT

		typedef DWORD DOT11_CIPHER_ALGORITHM, *PDOT11_CIPHER_ALGORITHM;

		#define DOT11_CIPHER_ALGO_NONE				0x00
		#define DOT11_CIPHER_ALGO_WEP40				0x01
		#define DOT11_CIPHER_ALGO_TKIP				0x02
		#define DOT11_CIPHER_ALGO_CCMP				0x04
		#define DOT11_CIPHER_ALGO_WEP104			0x05
		#define DOT11_CIPHER_ALGO_BIP				0x06
		#define DOT11_CIPHER_ALGO_GCMP				0x08
		#define DOT11_CIPHER_ALGO_GCMP_256			0x09
		#define DOT11_CIPHER_ALGO_CCMP_256			0x0A
		#define DOT11_CIPHER_ALGO_BIP_GMAC_128		0x0B
		#define DOT11_CIPHER_ALGO_BIP_GMAC_256		0x0C
		#define DOT11_CIPHER_ALGO_BIP_CMAC_256		0x0D
		#define DOT11_CIPHER_ALGO_WPA_USE_GROUP		0x100
		#define DOT11_CIPHER_ALGO_RSN_USE_GROUP		0x100
		#define DOT11_CIPHER_ALGO_WEP				0x101
		#define DOT11_CIPHER_ALGO_IHV_START			0x80000000
		#define DOT11_CIPHER_ALGO_IHV_END			0xffffffff

		typedef struct _DOT11_AUTH_CIPHER_PAIR
		{
			DOT11_AUTH_ALGORITHM	AuthAlgoId;
			DOT11_CIPHER_ALGORITHM	CipherAlgoId;
		} DOT11_AUTH_CIPHER_PAIR, *PDOT11_AUTH_CIPHER_PAIR;

		#define DOT11_OI_MAX_LENGTH		5
		#define DOT11_OI_MIN_LENGTH		3

		typedef struct _DOT11_OI
		{
			USHORT	OILength;
			UCHAR	OI[DOT11_OI_MAX_LENGTH];
		} DOT11_OI, *PDOT11_OI;

		typedef struct DOT11_ACCESSNETWORKOPTIONS
		{
			UINT8	AccessNetworkType;
			UINT8	Internet;
			UINT8	ASRA;
			UINT8	ESR;
			UINT8	UESA;
		} DOT11_ACCESSNETWORKOPTIONS, *PDOT11_ACCESSNETWORKOPTIONS;

		typedef struct _DOT11_VENUEINFO
		{
			UINT8	VenueGroup;
			UINT8	VenueType;
		} DOT11_VENUEINFO, *PDOT11_VENUEINFO;


		#define WLAN_API_VERSION_1_0	0x00000001
		#define WLAN_API_VERSION_2_0	0x00000002

		#define WLAN_API_VERSION	WLAN_API_VERSION_2_0

		#define WLAN_MAX_NAME_LENGTH	256

		#define WLAN_PROFILE_GROUP_POLICY					0x00000001
		#define WLAN_PROFILE_USER							0x00000002
		#define WLAN_PROFILE_GET_PLAINTEXT_KEY				0x00000004
		#define WLAN_PROFILE_CONNECTION_MODE_SET_BY_CLIENT	0x00010000
		#define WLAN_PROFILE_CONNECTION_MODE_AUTO			0x00020000

		#define WLAN_SET_EAPHOST_DATA_ALL_USERS		0x00000001

		typedef struct _WLAN_PROFILE_INFO
		{
			WCHAR	strProfileName[WLAN_MAX_NAME_LENGTH];
			DWORD	dwFlags;
		} WLAN_PROFILE_INFO, *PWLAN_PROFILE_INFO;

		typedef struct _DOT11_NETWORK
		{
			DOT11_SSID		dot11Ssid;
			DOT11_BSS_TYPE	dot11BssType;
		} DOT11_NETWORK, *PDOT11_NETWORK;

		#define DOT11_PSD_IE_MAX_DATA_SIZE		240
		#define DOT11_PSD_IE_MAX_ENTRY_NUMBER	5

		typedef struct _WLAN_RAW_DATA
		{
			DWORD	dwDataSize;
			BYTE	DataBlob[DOT11_PSD_IE_MAX_DATA_SIZE];
		} WLAN_RAW_DATA, *PWLAN_RAW_DATA;

		typedef struct _DataListRec
		{
			DWORD	dwDataOffset;
			DWORD	dwDataSize;
		} DataListRec;

		typedef struct _WLAN_RAW_DATA_LIST
		{
			DWORD		dwTotalSize;
			DWORD		dwNumberOfItems;
			DataListRec	DataList[1];
		} WLAN_RAW_DATA_LIST, *PWLAN_RAW_DATA_LIST;

		// Use this structure for simple add PSD IE.
		// It supports only one record (IE) but for now it is enough.
		typedef struct _WCL_WLAN_RAW_DATA_LIST
		{
			DWORD	dwTotalSize;
			DWORD	dwNumberOfItems;
			DWORD	dwDataOffset;
			DWORD	dwDataSize;
			// The maximum IE length is DOT11_PSD_IE_MAX_DATA_SIZE. So simple
			// Allocate memory for maximum possible length. However, the dwTotalSize and
			// dwDataSize should be set with current real data length.
			BYTE	Data[DOT11_PSD_IE_MAX_DATA_SIZE];
		} WCL_WLAN_RAW_DATA_LIST, *PWCL_WLAN_RAW_DATA_LIST;

		typedef enum _WLAN_CONNECTION_MODE
		{
			wlan_connection_mode_profile = 0,
			wlan_connection_mode_temporary_profile,
			wlan_connection_mode_discovery_secure,
			wlan_connection_mode_discovery_unsecure,
			wlan_connection_mode_auto,
			wlan_connection_mode_invalid
		} WLAN_CONNECTION_MODE, *PWLAN_CONNECTION_MODE;

		typedef DWORD WLAN_REASON_CODE, *PWLAN_REASON_CODE;

		#define L2_REASON_CODE_GROUP_SIZE	0x10000
		#define L2_REASON_CODE_GEN_BASE		0x10000

		#define L2_REASON_CODE_DOT11_AC_BASE			(L2_REASON_CODE_GEN_BASE + L2_REASON_CODE_GROUP_SIZE)
		#define L2_REASON_CODE_DOT11_MSM_BASE			(L2_REASON_CODE_DOT11_AC_BASE + L2_REASON_CODE_GROUP_SIZE)
		#define L2_REASON_CODE_DOT11_SECURITY_BASE		(L2_REASON_CODE_DOT11_MSM_BASE + L2_REASON_CODE_GROUP_SIZE)
		#define L2_REASON_CODE_ONEX_BASE				(L2_REASON_CODE_DOT11_SECURITY_BASE + L2_REASON_CODE_GROUP_SIZE)
		#define L2_REASON_CODE_DOT3_AC_BASE				(L2_REASON_CODE_ONEX_BASE + L2_REASON_CODE_GROUP_SIZE)
		#define L2_REASON_CODE_DOT3_MSM_BASE			(L2_REASON_CODE_DOT3_AC_BASE + L2_REASON_CODE_GROUP_SIZE)
		#define L2_REASON_CODE_PROFILE_BASE				(L2_REASON_CODE_DOT3_MSM_BASE + L2_REASON_CODE_GROUP_SIZE)
		#define L2_REASON_CODE_IHV_BASE					(L2_REASON_CODE_PROFILE_BASE + L2_REASON_CODE_GROUP_SIZE)
		#define L2_REASON_CODE_WIMAX_BASE				(L2_REASON_CODE_IHV_BASE + L2_REASON_CODE_GROUP_SIZE)
		#define L2_REASON_CODE_RESERVED_BASE			(L2_REASON_CODE_WIMAX_BASE + L2_REASON_CODE_GROUP_SIZE)

		#define L2_REASON_CODE_SUCCESS	0

		#define L2_REASON_CODE_UNKNOWN	(L2_REASON_CODE_GEN_BASE + 1)

		#define L2_REASON_CODE_PROFILE_MISSING	0x00000001

		#define WLAN_REASON_CODE_SUCCESS	L2_REASON_CODE_SUCCESS
		#define WLAN_REASON_CODE_UNKNOWN	L2_REASON_CODE_UNKNOWN

		#define WLAN_REASON_CODE_RANGE_SIZE	L2_REASON_CODE_GROUP_SIZE
		#define WLAN_REASON_CODE_BASE		L2_REASON_CODE_DOT11_AC_BASE

		#define WLAN_REASON_CODE_AC_BASE			L2_REASON_CODE_DOT11_AC_BASE
		#define WLAN_REASON_CODE_AC_CONNECT_BASE	(WLAN_REASON_CODE_AC_BASE + WLAN_REASON_CODE_RANGE_SIZE / 2)
		#define WLAN_REASON_CODE_AC_END				(WLAN_REASON_CODE_AC_BASE + WLAN_REASON_CODE_RANGE_SIZE - 1)

		#define WLAN_REASON_CODE_PROFILE_BASE			L2_REASON_CODE_PROFILE_BASE
		#define WLAN_REASON_CODE_PROFILE_CONNECT_BASE	(WLAN_REASON_CODE_PROFILE_BASE + WLAN_REASON_CODE_RANGE_SIZE / 2)
		#define WLAN_REASON_CODE_PROFILE_END			(WLAN_REASON_CODE_PROFILE_BASE + WLAN_REASON_CODE_RANGE_SIZE - 1)

		#define WLAN_REASON_CODE_MSM_BASE			L2_REASON_CODE_DOT11_MSM_BASE
		#define WLAN_REASON_CODE_MSM_CONNECT_BASE	(WLAN_REASON_CODE_MSM_BASE + WLAN_REASON_CODE_RANGE_SIZE / 2)
		#define WLAN_REASON_CODE_MSM_END			(WLAN_REASON_CODE_MSM_BASE + WLAN_REASON_CODE_RANGE_SIZE - 1)

		#define WLAN_REASON_CODE_RESERVED_BASE	(L2_REASON_CODE_RESERVED_BASE)
		#define WLAN_REASON_CODE_RESERVED_END	(WLAN_REASON_CODE_RESERVED_BASE + WLAN_REASON_CODE_RANGE_SIZE - 1)

		#define WLAN_REASON_CODE_MSMSEC_BASE			L2_REASON_CODE_DOT11_SECURITY_BASE
		#define WLAN_REASON_CODE_MSMSEC_CONNECT_BASE	(WLAN_REASON_CODE_MSMSEC_BASE + WLAN_REASON_CODE_RANGE_SIZE / 2)
		#define WLAN_REASON_CODE_MSMSEC_END				(WLAN_REASON_CODE_MSMSEC_BASE + WLAN_REASON_CODE_RANGE_SIZE - 1)

		#define WLAN_REASON_CODE_NETWORK_NOT_COMPATIBLE		(WLAN_REASON_CODE_AC_BASE + 1)
		#define WLAN_REASON_CODE_PROFILE_NOT_COMPATIBLE		(WLAN_REASON_CODE_AC_BASE + 2)

		#define WLAN_REASON_CODE_NO_AUTO_CONNECTION					(WLAN_REASON_CODE_AC_CONNECT_BASE + 1)
		#define WLAN_REASON_CODE_NOT_VISIBLE						(WLAN_REASON_CODE_AC_CONNECT_BASE + 2)
		#define WLAN_REASON_CODE_GP_DENIED							(WLAN_REASON_CODE_AC_CONNECT_BASE + 3)
		#define WLAN_REASON_CODE_USER_DENIED						(WLAN_REASON_CODE_AC_CONNECT_BASE + 4)
		#define WLAN_REASON_CODE_BSS_TYPE_NOT_ALLOWED				(WLAN_REASON_CODE_AC_CONNECT_BASE + 5)
		#define WLAN_REASON_CODE_IN_FAILED_LIST						(WLAN_REASON_CODE_AC_CONNECT_BASE + 6)
		#define WLAN_REASON_CODE_IN_BLOCKED_LIST					(WLAN_REASON_CODE_AC_CONNECT_BASE + 7)
		#define WLAN_REASON_CODE_SSID_LIST_TOO_LONG					(WLAN_REASON_CODE_AC_CONNECT_BASE + 8)
		#define WLAN_REASON_CODE_CONNECT_CALL_FAIL					(WLAN_REASON_CODE_AC_CONNECT_BASE + 9)
		#define WLAN_REASON_CODE_SCAN_CALL_FAIL						(WLAN_REASON_CODE_AC_CONNECT_BASE + 10)
		#define WLAN_REASON_CODE_NETWORK_NOT_AVAILABLE				(WLAN_REASON_CODE_AC_CONNECT_BASE + 11)
		#define WLAN_REASON_CODE_PROFILE_CHANGED_OR_DELETED			(WLAN_REASON_CODE_AC_CONNECT_BASE + 12)
		#define WLAN_REASON_CODE_KEY_MISMATCH						(WLAN_REASON_CODE_AC_CONNECT_BASE + 13)
		#define WLAN_REASON_CODE_USER_NOT_RESPOND					(WLAN_REASON_CODE_AC_CONNECT_BASE + 14)
		#define WLAN_REASON_CODE_AP_PROFILE_NOT_ALLOWED_FOR_CLIENT	(WLAN_REASON_CODE_AC_CONNECT_BASE + 15)
		#define WLAN_REASON_CODE_AP_PROFILE_NOT_ALLOWED				(WLAN_REASON_CODE_AC_CONNECT_BASE + 16)
		#define WLAN_REASON_CODE_HOTSPOT2_PROFILE_DENIED			(WLAN_REASON_CODE_AC_CONNECT_BASE + 17)

		#define WLAN_REASON_CODE_INVALID_PROFILE_SCHEMA						(WLAN_REASON_CODE_PROFILE_BASE + 1)
		#define WLAN_REASON_CODE_PROFILE_MISSING							(WLAN_REASON_CODE_PROFILE_BASE + 2)
		#define WLAN_REASON_CODE_INVALID_PROFILE_NAME						(WLAN_REASON_CODE_PROFILE_BASE + 3)
		#define WLAN_REASON_CODE_INVALID_PROFILE_TYPE						(WLAN_REASON_CODE_PROFILE_BASE + 4)
		#define WLAN_REASON_CODE_INVALID_PHY_TYPE							(WLAN_REASON_CODE_PROFILE_BASE + 5)
		#define WLAN_REASON_CODE_MSM_SECURITY_MISSING						(WLAN_REASON_CODE_PROFILE_BASE + 6)
		#define WLAN_REASON_CODE_IHV_SECURITY_NOT_SUPPORTED					(WLAN_REASON_CODE_PROFILE_BASE + 7)
		#define WLAN_REASON_CODE_IHV_OUI_MISMATCH							(WLAN_REASON_CODE_PROFILE_BASE + 8)
		#define WLAN_REASON_CODE_IHV_OUI_MISSING							(WLAN_REASON_CODE_PROFILE_BASE + 9)
		#define WLAN_REASON_CODE_IHV_SETTINGS_MISSING						(WLAN_REASON_CODE_PROFILE_BASE + 10)
		#define WLAN_REASON_CODE_CONFLICT_SECURITY							(WLAN_REASON_CODE_PROFILE_BASE + 11)
		#define WLAN_REASON_CODE_SECURITY_MISSING							(WLAN_REASON_CODE_PROFILE_BASE + 12)
		#define WLAN_REASON_CODE_INVALID_BSS_TYPE							(WLAN_REASON_CODE_PROFILE_BASE + 13)
		#define WLAN_REASON_CODE_INVALID_ADHOC_CONNECTION_MODE				(WLAN_REASON_CODE_PROFILE_BASE + 14)
		#define WLAN_REASON_CODE_NON_BROADCAST_SET_FOR_ADHOC				(WLAN_REASON_CODE_PROFILE_BASE + 15)
		#define WLAN_REASON_CODE_AUTO_SWITCH_SET_FOR_ADHOC					(WLAN_REASON_CODE_PROFILE_BASE + 16)
		#define WLAN_REASON_CODE_AUTO_SWITCH_SET_FOR_MANUAL_CONNECTION		(WLAN_REASON_CODE_PROFILE_BASE + 17)
		#define WLAN_REASON_CODE_IHV_SECURITY_ONEX_MISSING					(WLAN_REASON_CODE_PROFILE_BASE + 18)
		#define WLAN_REASON_CODE_PROFILE_SSID_INVALID						(WLAN_REASON_CODE_PROFILE_BASE + 19)
		#define WLAN_REASON_CODE_TOO_MANY_SSID								(WLAN_REASON_CODE_PROFILE_BASE + 20)
		#define WLAN_REASON_CODE_IHV_CONNECTIVITY_NOT_SUPPORTED				(WLAN_REASON_CODE_PROFILE_BASE + 21)
		#define WLAN_REASON_CODE_BAD_MAX_NUMBER_OF_CLIENTS_FOR_AP			(WLAN_REASON_CODE_PROFILE_BASE + 22)
		#define WLAN_REASON_CODE_INVALID_CHANNEL							(WLAN_REASON_CODE_PROFILE_BASE + 23)
		#define WLAN_REASON_CODE_OPERATION_MODE_NOT_SUPPORTED				(WLAN_REASON_CODE_PROFILE_BASE + 24)
		#define WLAN_REASON_CODE_AUTO_AP_PROFILE_NOT_ALLOWED				(WLAN_REASON_CODE_PROFILE_BASE + 25)
		#define WLAN_REASON_CODE_AUTO_CONNECTION_NOT_ALLOWED				(WLAN_REASON_CODE_PROFILE_BASE + 26)
		#define WLAN_REASON_CODE_HOTSPOT2_PROFILE_NOT_ALLOWED				(WLAN_REASON_CODE_PROFILE_BASE + 27)

		#define WLAN_REASON_CODE_UNSUPPORTED_SECURITY_SET_BY_OS		(WLAN_REASON_CODE_MSM_BASE + 1)
		#define WLAN_REASON_CODE_UNSUPPORTED_SECURITY_SET			(WLAN_REASON_CODE_MSM_BASE + 2)
		#define WLAN_REASON_CODE_BSS_TYPE_UNMATCH					(WLAN_REASON_CODE_MSM_BASE + 3)
		#define WLAN_REASON_CODE_PHY_TYPE_UNMATCH					(WLAN_REASON_CODE_MSM_BASE + 4)
		#define WLAN_REASON_CODE_DATARATE_UNMATCH					(WLAN_REASON_CODE_MSM_BASE + 5)

		#define WLAN_REASON_CODE_USER_CANCELLED					(WLAN_REASON_CODE_MSM_CONNECT_BASE + 1)
		#define WLAN_REASON_CODE_ASSOCIATION_FAILURE			(WLAN_REASON_CODE_MSM_CONNECT_BASE + 2)
		#define WLAN_REASON_CODE_ASSOCIATION_TIMEOUT			(WLAN_REASON_CODE_MSM_CONNECT_BASE + 3)
		#define WLAN_REASON_CODE_PRE_SECURITY_FAILURE			(WLAN_REASON_CODE_MSM_CONNECT_BASE + 4)
		#define WLAN_REASON_CODE_START_SECURITY_FAILURE			(WLAN_REASON_CODE_MSM_CONNECT_BASE + 5)
		#define WLAN_REASON_CODE_SECURITY_FAILURE				(WLAN_REASON_CODE_MSM_CONNECT_BASE + 6)
		#define WLAN_REASON_CODE_SECURITY_TIMEOUT				(WLAN_REASON_CODE_MSM_CONNECT_BASE + 7)
		#define WLAN_REASON_CODE_ROAMING_FAILURE				(WLAN_REASON_CODE_MSM_CONNECT_BASE + 8)
		#define WLAN_REASON_CODE_ROAMING_SECURITY_FAILURE		(WLAN_REASON_CODE_MSM_CONNECT_BASE + 9)
		#define WLAN_REASON_CODE_ADHOC_SECURITY_FAILURE			(WLAN_REASON_CODE_MSM_CONNECT_BASE + 10)
		#define WLAN_REASON_CODE_DRIVER_DISCONNECTED			(WLAN_REASON_CODE_MSM_CONNECT_BASE + 11)
		#define WLAN_REASON_CODE_DRIVER_OPERATION_FAILURE		(WLAN_REASON_CODE_MSM_CONNECT_BASE + 12)
		#define WLAN_REASON_CODE_IHV_NOT_AVAILABLE				(WLAN_REASON_CODE_MSM_CONNECT_BASE + 13)
		#define WLAN_REASON_CODE_IHV_NOT_RESPONDING				(WLAN_REASON_CODE_MSM_CONNECT_BASE + 14)
		#define WLAN_REASON_CODE_DISCONNECT_TIMEOUT				(WLAN_REASON_CODE_MSM_CONNECT_BASE + 15)
		#define WLAN_REASON_CODE_INTERNAL_FAILURE				(WLAN_REASON_CODE_MSM_CONNECT_BASE + 16)
		#define WLAN_REASON_CODE_UI_REQUEST_TIMEOUT				(WLAN_REASON_CODE_MSM_CONNECT_BASE + 17)
		#define WLAN_REASON_CODE_TOO_MANY_SECURITY_ATTEMPTS		(WLAN_REASON_CODE_MSM_CONNECT_BASE + 18)
		#define WLAN_REASON_CODE_AP_STARTING_FAILURE			(WLAN_REASON_CODE_MSM_CONNECT_BASE + 19)
		#define WLAN_REASON_CODE_NO_VISIBLE_AP					(WLAN_REASON_CODE_MSM_CONNECT_BASE + 20)

		#define WLAN_REASON_CODE_MSMSEC_MIN		WLAN_REASON_CODE_MSMSEC_BASE

		#define WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_KEY_INDEX					(WLAN_REASON_CODE_MSMSEC_BASE + 1)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_PSK_PRESENT							(WLAN_REASON_CODE_MSMSEC_BASE + 2)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_KEY_LENGTH							(WLAN_REASON_CODE_MSMSEC_BASE + 3)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_PSK_LENGTH							(WLAN_REASON_CODE_MSMSEC_BASE + 4)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_NO_AUTH_CIPHER_SPECIFIED			(WLAN_REASON_CODE_MSMSEC_BASE + 5)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_TOO_MANY_AUTH_CIPHER_SPECIFIED		(WLAN_REASON_CODE_MSMSEC_BASE + 6)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_DUPLICATE_AUTH_CIPHER				(WLAN_REASON_CODE_MSMSEC_BASE + 7)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_RAWDATA_INVALID						(WLAN_REASON_CODE_MSMSEC_BASE + 8)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_AUTH_CIPHER					(WLAN_REASON_CODE_MSMSEC_BASE + 9)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_ONEX_DISABLED						(WLAN_REASON_CODE_MSMSEC_BASE + 10)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_ONEX_ENABLED						(WLAN_REASON_CODE_MSMSEC_BASE + 11)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_PMKCACHE_MODE				(WLAN_REASON_CODE_MSMSEC_BASE + 12)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_PMKCACHE_SIZE				(WLAN_REASON_CODE_MSMSEC_BASE + 13)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_PMKCACHE_TTL				(WLAN_REASON_CODE_MSMSEC_BASE + 14)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_PREAUTH_MODE				(WLAN_REASON_CODE_MSMSEC_BASE + 15)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_PREAUTH_THROTTLE			(WLAN_REASON_CODE_MSMSEC_BASE + 16)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_PREAUTH_ONLY_ENABLED				(WLAN_REASON_CODE_MSMSEC_BASE + 17)
		#define WLAN_REASON_CODE_MSMSEC_CAPABILITY_NETWORK							(WLAN_REASON_CODE_MSMSEC_BASE + 18)
		#define WLAN_REASON_CODE_MSMSEC_CAPABILITY_NIC								(WLAN_REASON_CODE_MSMSEC_BASE + 19)
		#define WLAN_REASON_CODE_MSMSEC_CAPABILITY_PROFILE							(WLAN_REASON_CODE_MSMSEC_BASE + 20)
		#define WLAN_REASON_CODE_MSMSEC_CAPABILITY_DISCOVERY						(WLAN_REASON_CODE_MSMSEC_BASE + 21)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_PASSPHRASE_CHAR						(WLAN_REASON_CODE_MSMSEC_BASE + 22)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_KEYMATERIAL_CHAR					(WLAN_REASON_CODE_MSMSEC_BASE + 23)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_WRONG_KEYTYPE						(WLAN_REASON_CODE_MSMSEC_BASE + 24)
		#define WLAN_REASON_CODE_MSMSEC_MIXED_CELL									(WLAN_REASON_CODE_MSMSEC_BASE + 25)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_AUTH_TIMERS_INVALID					(WLAN_REASON_CODE_MSMSEC_BASE + 26)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_GKEY_INTV					(WLAN_REASON_CODE_MSMSEC_BASE + 27)
		#define WLAN_REASON_CODE_MSMSEC_TRANSITION_NETWORK							(WLAN_REASON_CODE_MSMSEC_BASE + 28)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_KEY_UNMAPPED_CHAR					(WLAN_REASON_CODE_MSMSEC_BASE + 29)
		#define WLAN_REASON_CODE_MSMSEC_CAPABILITY_PROFILE_AUTH						(WLAN_REASON_CODE_MSMSEC_BASE + 30)
		#define WLAN_REASON_CODE_MSMSEC_CAPABILITY_PROFILE_CIPHER					(WLAN_REASON_CODE_MSMSEC_BASE + 31)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_SAFE_MODE							(WLAN_REASON_CODE_MSMSEC_BASE + 32)
		#define WLAN_REASON_CODE_MSMSEC_CAPABILITY_PROFILE_SAFE_MODE_NIC			(WLAN_REASON_CODE_MSMSEC_BASE + 33)
		#define WLAN_REASON_CODE_MSMSEC_CAPABILITY_PROFILE_SAFE_MODE_NW				(WLAN_REASON_CODE_MSMSEC_BASE + 34)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_UNSUPPORTED_AUTH					(WLAN_REASON_CODE_MSMSEC_BASE + 35)
		#define WLAN_REASON_CODE_MSMSEC_PROFILE_UNSUPPORTED_CIPHER					(WLAN_REASON_CODE_MSMSEC_BASE + 36)
		#define WLAN_REASON_CODE_MSMSEC_CAPABILITY_MFP_NW_NIC						(WLAN_REASON_CODE_MSMSEC_BASE + 37)

		#define WLAN_REASON_CODE_MSMSEC_UI_REQUEST_FAILURE			(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 1)
		#define WLAN_REASON_CODE_MSMSEC_AUTH_START_TIMEOUT			(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 2)
		#define WLAN_REASON_CODE_MSMSEC_AUTH_SUCCESS_TIMEOUT		(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 3)
		#define WLAN_REASON_CODE_MSMSEC_KEY_START_TIMEOUT			(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 4)
		#define WLAN_REASON_CODE_MSMSEC_KEY_SUCCESS_TIMEOUT			(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 5)
		#define WLAN_REASON_CODE_MSMSEC_M3_MISSING_KEY_DATA			(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 6)
		#define WLAN_REASON_CODE_MSMSEC_M3_MISSING_IE				(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 7)
		#define WLAN_REASON_CODE_MSMSEC_M3_MISSING_GRP_KEY			(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 8)
		#define WLAN_REASON_CODE_MSMSEC_PR_IE_MATCHING				(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 9)
		#define WLAN_REASON_CODE_MSMSEC_SEC_IE_MATCHING				(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 10)
		#define WLAN_REASON_CODE_MSMSEC_NO_PAIRWISE_KEY				(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 11)
		#define WLAN_REASON_CODE_MSMSEC_G1_MISSING_KEY_DATA			(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 12)
		#define WLAN_REASON_CODE_MSMSEC_G1_MISSING_GRP_KEY			(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 13)
		#define WLAN_REASON_CODE_MSMSEC_PEER_INDICATED_INSECURE		(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 14)
		#define WLAN_REASON_CODE_MSMSEC_NO_AUTHENTICATOR			(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 15)
		#define WLAN_REASON_CODE_MSMSEC_NIC_FAILURE					(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 16)
		#define WLAN_REASON_CODE_MSMSEC_CANCELLED					(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 17)
		#define WLAN_REASON_CODE_MSMSEC_KEY_FORMAT					(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 18)
		#define WLAN_REASON_CODE_MSMSEC_DOWNGRADE_DETECTED			(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 19)
		#define WLAN_REASON_CODE_MSMSEC_PSK_MISMATCH_SUSPECTED		(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 20)
		#define WLAN_REASON_CODE_MSMSEC_FORCED_FAILURE				(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 21)
		  
		#define WLAN_REASON_CODE_MSMSEC_M3_TOO_MANY_RSNIE			(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 22)
		#define WLAN_REASON_CODE_MSMSEC_M2_MISSING_KEY_DATA			(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 23)
		#define WLAN_REASON_CODE_MSMSEC_M2_MISSING_IE				(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 24)
		#define WLAN_REASON_CODE_MSMSEC_AUTH_WCN_COMPLETED			(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 25)
		#define WLAN_REASON_CODE_MSMSEC_M3_MISSING_MGMT_GRP_KEY		(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 26)
		#define WLAN_REASON_CODE_MSMSEC_G1_MISSING_MGMT_GRP_KEY		(WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 27)

		#define WLAN_REASON_CODE_MSMSEC_MAX		WLAN_REASON_CODE_MSMSEC_END

		typedef WLAN_REASON_CODE ONEX_REASON_CODE, *PONEX_REASON_CODE;

		#define ONEX_REASON_CODE_SUCCESS	0

		#define ONEX_REASON_START	L2_REASON_CODE_ONEX_BASE

		#define ONEX_UNABLE_TO_IDENTIFY_USER						(ONEX_REASON_START + 1)
		#define ONEX_IDENTITY_NOT_FOUND								(ONEX_REASON_START + 2)
		#define ONEX_UI_DISABLED									(ONEX_REASON_START + 3)
		#define ONEX_UI_FAILURE										(ONEX_REASON_START + 4)
		#define ONEX_EAP_FAILURE_RECEIVED							(ONEX_REASON_START + 5)
		#define ONEX_AUTHENTICATOR_NO_LONGER_PRESENT				(ONEX_REASON_START + 6)
		#define ONEX_NO_RESPONSE_TO_IDENTITY						(ONEX_REASON_START + 7)
		#define ONEX_PROFILE_VERSION_NOT_SUPPORTED					(ONEX_REASON_START + 8)
		#define ONEX_PROFILE_INVALID_LENGTH							(ONEX_REASON_START + 9)
		#define ONEX_PROFILE_DISALLOWED_EAP_TYPE					(ONEX_REASON_START + 10)
		#define ONEX_PROFILE_INVALID_EAP_TYPE_OR_FLAG				(ONEX_REASON_START + 11)
		#define ONEX_PROFILE_INVALID_ONEX_FLAGS						(ONEX_REASON_START + 12)
		#define ONEX_PROFILE_INVALID_TIMER_VALUE					(ONEX_REASON_START + 13)
		#define ONEX_PROFILE_INVALID_SUPPLICANT_MODE				(ONEX_REASON_START + 14)
		#define ONEX_PROFILE_INVALID_AUTH_MODE						(ONEX_REASON_START + 15)
		#define ONEX_PROFILE_INVALID_EAP_CONNECTION_PROPERTIES		(ONEX_REASON_START + 16)
		#define ONEX_UI_CANCELLED									(ONEX_REASON_START + 17)
		#define ONEX_PROFILE_INVALID_EXPLICIT_CREDENTIALS			(ONEX_REASON_START + 18)
		#define ONEX_PROFILE_EXPIRED_EXPLICIT_CREDENTIALS			(ONEX_REASON_START + 19)
		#define ONEX_UI_NOT_PERMITTED								(ONEX_REASON_START + 20)

		typedef ULONG WLAN_SIGNAL_QUALITY, *PWLAN_SIGNAL_QUALITY;

		#define WLAN_AVAILABLE_NETWORK_CONNECTED				0x00000001
		#define WLAN_AVAILABLE_NETWORK_HAS_PROFILE				0x00000002
		#define WLAN_AVAILABLE_NETWORK_CONSOLE_USER_PROFILE		0x00000004
		#define WLAN_AVAILABLE_NETWORK_INTERWORKING_SUPPORTED	0x00000008
		#define WLAN_AVAILABLE_NETWORK_HOTSPOT2_ENABLED			0x00000010
		#define WLAN_AVAILABLE_NETWORK_ANQP_SUPPORTED			0x00000020
		#define WLAN_AVAILABLE_NETWORK_HOTSPOT2_DOMAIN			0x00000040
		#define WLAN_AVAILABLE_NETWORK_HOTSPOT2_ROAMING			0x00000080
		#define WLAN_AVAILABLE_NETWORK_AUTO_CONNECT_FAILED		0x00000100

		#define WLAN_AVAILABLE_NETWORK_INCLUDE_ALL_ADHOC_PROFILES			0x00000001
		#define WLAN_AVAILABLE_NETWORK_INCLUDE_ALL_MANUAL_HIDDEN_PROFILES	0x00000002

		#define DOT11_RATE_SET_MAX_LENGTH	126

		typedef struct _WLAN_RATE_SET
		{
			ULONG	uRateSetLength;
			USHORT	usRateSet[DOT11_RATE_SET_MAX_LENGTH];
		} WLAN_RATE_SET, *PWLAN_RATE_SET;

		#define WLAN_MAX_PHY_TYPE_NUMBER	8

		typedef DWORD DOT11_PHY_TYPE, *PDOT11_PHY_TYPE;

		#define dot11_phy_type_unknown		0
		#define dot11_phy_type_any			dot11_phy_type_unknown
		#define dot11_phy_type_fhss			1
		#define dot11_phy_type_dsss			2
		#define dot11_phy_type_irbaseband	3
		#define dot11_phy_type_ofdm			4
		#define dot11_phy_type_hrdsss		5
		#define dot11_phy_type_erp			6
		#define dot11_phy_type_ht			7
		#define dot11_phy_type_vht			8
		#define dot11_phy_type_IHV_start	0x80000000
		#define dot11_phy_type_IHV_end		0xffffffff

		typedef struct _WLAN_AVAILABLE_NETWORK
		{
			WCHAR					strProfileName[WLAN_MAX_NAME_LENGTH];
			DOT11_SSID				dot11Ssid;
			DOT11_BSS_TYPE			dot11BssType;
			ULONG					uNumberOfBssids;
			BOOL					bNetworkConnectable;
			WLAN_REASON_CODE		wlanNotConnectableReason;
			ULONG					uNumberOfPhyTypes;
			DOT11_PHY_TYPE			dot11PhyTypes[WLAN_MAX_PHY_TYPE_NUMBER];
			BOOL					bMorePhyTypes;
			WLAN_SIGNAL_QUALITY		wlanSignalQuality;
			BOOL					bSecurityEnabled;
			DOT11_AUTH_ALGORITHM	dot11DefaultAuthAlgorithm;
			DOT11_CIPHER_ALGORITHM	dot11DefaultCipherAlgorithm;
			DWORD					dwFlags;
			DWORD					dwReserved;
		} WLAN_AVAILABLE_NETWORK, *PWLAN_AVAILABLE_NETWORK;

		#define DOT11_HESSID_LENGTH		6

		typedef UCHAR DOT11_HESSID[DOT11_HESSID_LENGTH], *PDOT11_HESSID;

		typedef struct _WLAN_AVAILABLE_NETWORK_V2
		{
			WCHAR						strProfileName[WLAN_MAX_NAME_LENGTH];
			DOT11_SSID					dot11Ssid;
			DOT11_BSS_TYPE				dot11BssType;
			ULONG						uNumberOfBssids;
			BOOL						bNetworkConnectable;
			WLAN_REASON_CODE			wlanNotConnectableReason;
			ULONG						uNumberOfPhyTypes;
			DOT11_PHY_TYPE				dot11PhyTypes[WLAN_MAX_PHY_TYPE_NUMBER];
			BOOL						bMorePhyTypes;
			WLAN_SIGNAL_QUALITY			wlanSignalQuality;
			BOOL						bSecurityEnabled;
			DOT11_AUTH_ALGORITHM		dot11DefaultAuthAlgorithm;
			DOT11_CIPHER_ALGORITHM		dot11DefaultCipherAlgorithm;
			DWORD						dwFlags;
			DOT11_ACCESSNETWORKOPTIONS	AccessNetworkOptions;
			DOT11_HESSID				dot11HESSID;
			DOT11_VENUEINFO				VenueInfo;
			DWORD						dwReserved;
		} WLAN_AVAILABLE_NETWORK_V2, *PWLAN_AVAILABLE_NETWORK_V2;

		typedef UCHAR DOT11_MAC_ADDRESS[6], *PDOT11_MAC_ADDRESS;

		typedef struct _WLAN_BSS_ENTRY
		{
			DOT11_SSID			dot11Ssid;
			ULONG				uPhyId;
			DOT11_MAC_ADDRESS	dot11Bssid;
			DOT11_BSS_TYPE		dot11BssType;
			DOT11_PHY_TYPE		dot11BssPhyType;
			LONG				lRssi;
			ULONG				uLinkQuality;
			BOOLEAN				bInRegDomain;
			USHORT				usBeaconPeriod;
			ULONGLONG			ullTimestamp;
			ULONGLONG			ullHostTimestamp;
			USHORT				usCapabilityInformation;
			ULONG				ulChCenterFrequency;
			WLAN_RATE_SET		wlanRateSet;
			ULONG				ulIeOffset;
			ULONG				ulIeSize;
		} WLAN_BSS_ENTRY, *PWLAN_BSS_ENTRY;

		typedef struct _WLAN_BSS_LIST
		{
			DWORD			dwTotalSize;
			DWORD			dwNumberOfItems;
			WLAN_BSS_ENTRY	wlanBssEntries[1];
		} WLAN_BSS_LIST, *PWLAN_BSS_LIST;

		typedef enum _WLAN_INTERFACE_STATE
		{
			wlan_interface_state_not_ready,
			wlan_interface_state_connected,
			wlan_interface_state_ad_hoc_network_formed,
			wlan_interface_state_disconnecting,
			wlan_interface_state_disconnected,
			wlan_interface_state_associating,
			wlan_interface_state_discovering,
			wlan_interface_state_authenticating
		} WLAN_INTERFACE_STATE, *PWLAN_INTERFACE_STATE;

		typedef enum _WLAN_ADHOC_NETWORK_STATE
		{
			wlan_adhoc_network_state_formed = 0,
			wlan_adhoc_network_state_connected
		} WLAN_ADHOC_NETWORK_STATE, *PWLAN_ADHOC_NETWORK_STATE;

		typedef struct _WLAN_INTERFACE_INFO
		{
			GUID					InterfaceGuid;
			WCHAR					strInterfaceDescription[WLAN_MAX_NAME_LENGTH];
			WLAN_INTERFACE_STATE	isState;
		} WLAN_INTERFACE_INFO, *PWLAN_INTERFACE_INFO;

		typedef struct _WLAN_ASSOCIATION_ATTRIBUTES
		{
			DOT11_SSID				dot11Ssid;
			DOT11_BSS_TYPE			dot11BssType;
			DOT11_MAC_ADDRESS		dot11Bssid;
			DOT11_PHY_TYPE			dot11PhyType;
			ULONG					uDot11PhyIndex;
			WLAN_SIGNAL_QUALITY		wlanSignalQuality;
			ULONG					ulRxRate;
			ULONG					ulTxRate;
		} WLAN_ASSOCIATION_ATTRIBUTES, *PWLAN_ASSOCIATION_ATTRIBUTES;

		typedef struct _WLAN_SECURITY_ATTRIBUTES
		{
			BOOL					bSecurityEnabled;
			BOOL					bOneXEnabled;
			DOT11_AUTH_ALGORITHM	dot11AuthAlgorithm;
			DOT11_CIPHER_ALGORITHM	dot11CipherAlgorithm;
		} WLAN_SECURITY_ATTRIBUTES, *PWLAN_SECURITY_ATTRIBUTES;

		typedef struct _WLAN_CONNECTION_ATTRIBUTES
		{
			WLAN_INTERFACE_STATE			isState;
			WLAN_CONNECTION_MODE			wlanConnectionMode;
			WCHAR							strProfileName[WLAN_MAX_NAME_LENGTH];
			WLAN_ASSOCIATION_ATTRIBUTES		wlanAssociationAttributes;
			WLAN_SECURITY_ATTRIBUTES		wlanSecurityAttributes;
		} WLAN_CONNECTION_ATTRIBUTES, *PWLAN_CONNECTION_ATTRIBUTES;

		typedef enum _DOT11_RADIO_STATE
		{
			dot11_radio_state_unknown = 0,
			dot11_radio_state_on,
			dot11_radio_state_off
		} DOT11_RADIO_STATE, *PDOT11_RADIO_STATE;

		#define WLAN_MAX_PHY_INDEX	64

		typedef struct _WLAN_PHY_RADIO_STATE
		{
			DWORD				dwPhyIndex;
			DOT11_RADIO_STATE	dot11SoftwareRadioState;
			DOT11_RADIO_STATE	dot11HardwareRadioState;
		} WLAN_PHY_RADIO_STATE, *PWLAN_PHY_RADIO_STATE;

		typedef struct _WLAN_RADIO_STATE
		{
			DWORD					dwNumberOfPhys;
			WLAN_PHY_RADIO_STATE	PhyRadioState[WLAN_MAX_PHY_INDEX];
		} WLAN_RADIO_STATE, *PWLAN_RADIO_STATE;

		typedef enum _WLAN_OPERATIONAL_STATE
		{
			wlan_operational_state_unknown = 0,
			wlan_operational_state_off,
			wlan_operational_state_on,
			wlan_operational_state_going_off,
			wlan_operational_state_going_on
		} WLAN_OPERATIONAL_STATE, *PWLAN_OPERATIONAL_STATE;

		typedef enum _WLAN_INTERFACE_TYPE
		{
			wlan_interface_type_emulated_802_11 = 0,
			wlan_interface_type_native_802_11,
			wlan_interface_type_invalid
		} WLAN_INTERFACE_TYPE, *PWLAN_INTERFACE_TYPE;

		typedef struct _WLAN_INTERFACE_CAPABILITY
		{
			WLAN_INTERFACE_TYPE		interfaceType;
			BOOL					bDot11DSupported;
			DWORD					dwMaxDesiredSsidListSize;
			DWORD					dwMaxDesiredBssidListSize;
			DWORD					dwNumberOfSupportedPhys;
			DOT11_PHY_TYPE			dot11PhyTypes[WLAN_MAX_PHY_INDEX];
		} WLAN_INTERFACE_CAPABILITY, *PWLAN_INTERFACE_CAPABILITY;

		typedef struct _WLAN_AUTH_CIPHER_PAIR_LIST
		{
			DWORD					dwNumberOfItems;
			DOT11_AUTH_CIPHER_PAIR	pAuthCipherPairList[1];
		} WLAN_AUTH_CIPHER_PAIR_LIST, *PWLAN_AUTH_CIPHER_PAIR_LIST;

		typedef UCHAR DOT11_COUNTRY_OR_REGION_STRING[3], *PDOT11_COUNTRY_OR_REGION_STRING;

		typedef struct _WLAN_COUNTRY_OR_REGION_STRING_LIST
		{
			DWORD							dwNumberOfItems;
			DOT11_COUNTRY_OR_REGION_STRING	pCountryOrRegionStringList[1];
		} WLAN_COUNTRY_OR_REGION_STRING_LIST, *PWLAN_COUNTRY_OR_REGION_STRING_LIST;

		typedef struct _WLAN_PROFILE_INFO_LIST
		{
			DWORD				dwNumberOfItems;
			DWORD				dwIndex;
			WLAN_PROFILE_INFO	ProfileInfo[1];
		} WLAN_PROFILE_INFO_LIST, *PWLAN_PROFILE_INFO_LIST;

		typedef struct _WLAN_AVAILABLE_NETWORK_LIST
		{
			DWORD					dwNumberOfItems;
			DWORD					dwIndex;
			WLAN_AVAILABLE_NETWORK	Network[1];
		} WLAN_AVAILABLE_NETWORK_LIST, *PWLAN_AVAILABLE_NETWORK_LIST;

		typedef struct _WLAN_AVAILABLE_NETWORK_LIST_V2
		{
			DWORD						dwNumberOfItems;
			DWORD						dwIndex;
			WLAN_AVAILABLE_NETWORK_V2	Network[1];
		} WLAN_AVAILABLE_NETWORK_LIST_V2, *PWLAN_AVAILABLE_NETWORK_LIST_V2;

		typedef struct _WLAN_INTERFACE_INFO_LIST
		{
			DWORD					dwNumberOfItems;
			DWORD					dwIndex;
			WLAN_INTERFACE_INFO		InterfaceInfo[1];
		} WLAN_INTERFACE_INFO_LIST, *PWLAN_INTERFACE_INFO_LIST;

		typedef struct _DOT11_NETWORK_LIST
		{
			DWORD			dwNumberOfItems;
			DWORD			dwIndex;
			DOT11_NETWORK	Network[1];
		} DOT11_NETWORK_LIST, *PDOT11_NETWORK_LIST;

		typedef enum WLAN_POWER_SETTING
		{
			wlan_power_setting_no_saving = 0,
			wlan_power_setting_low_saving,
			wlan_power_setting_medium_saving,
			wlan_power_setting_maximum_saving,
			wlan_power_setting_invalid
		} WLAN_POWER_SETTING, *PWLAN_POWER_SETTING;


		#define WLAN_CONNECTION_HIDDEN_NETWORK										0x00000001
		#define WLAN_CONNECTION_ADHOC_JOIN_ONLY										0x00000002
		#define WLAN_CONNECTION_IGNORE_PRIVACY_BIT									0x00000004
		#define WLAN_CONNECTION_EAPOL_PASSTHROUGH									0x00000008
		#define WLAN_CONNECTION_PERSIST_DISCOVERY_PROFILE							0x00000010
		#define WLAN_CONNECTION_PERSIST_DISCOVERY_PROFILE_CONNECTION_MODE_AUTO		0x00000020
		#define WLAN_CONNECTION_PERSIST_DISCOVERY_PROFILE_OVERWRITE_EXISTING		0x00000040

		#define NDIS_OBJECT_TYPE_DEFAULT		0x80
		#define NDIS_OBJECT_REVISION_1			1
		#define DOT11_BSSID_LIST_REVISION_1		1

		typedef struct _NDIS_OBJECT_HEADER
		{
			UCHAR	Type;
			UCHAR	Revision;
			USHORT	Size;
		} NDIS_OBJECT_HEADER, *PNDIS_OBJECT_HEADER;

		typedef struct _DOT11_BSSID_LIST
		{
			NDIS_OBJECT_HEADER	Header;
			ULONG				uNumOfEntries;
			ULONG				uTotalNumOfEntries;
			DOT11_MAC_ADDRESS	BSSIDs[1];
		} DOT11_BSSID_LIST, *PDOT11_BSSID_LIST;

		typedef struct _WLAN_CONNECTION_PARAMETERS
		{
			WLAN_CONNECTION_MODE	wlanConnectionMode;
			LPCWSTR					strProfile;
			PDOT11_SSID				pDot11Ssid;
			PDOT11_BSSID_LIST		pDesiredBssidList;
			DOT11_BSS_TYPE			dot11BssType;
			DWORD					dwFlags;
		} WLAN_CONNECTION_PARAMETERS, *PWLAN_CONNECTION_PARAMETERS;

		typedef struct _WLAN_CONNECTION_PARAMETERS_V2
		{
			WLAN_CONNECTION_MODE			wlanConnectionMode;
			LPCWSTR							strProfile;
			PDOT11_SSID						pDot11Ssid;
			PDOT11_HESSID					pDot11Hessid;
			PDOT11_BSSID_LIST				pDesiredBssidList;
			DOT11_BSS_TYPE					dot11BssType;
			DWORD							dwFlags;
			PDOT11_ACCESSNETWORKOPTIONS		pDot11AccessNetworkOptions;
		} WLAN_CONNECTION_PARAMETERS_V2, *PWLAN_CONNECTION_PARAMETERS_V2;

		typedef struct _WLAN_MSM_NOTIFICATION_DATA
		{
			WLAN_CONNECTION_MODE	wlanConnectionMode;
			WCHAR					strProfileName[WLAN_MAX_NAME_LENGTH];
			DOT11_SSID				dot11Ssid;
			DOT11_BSS_TYPE			dot11BssType;
			DOT11_MAC_ADDRESS		dot11MacAddr;
			BOOL					bSecurityEnabled;
			BOOL					bFirstPeer;
			BOOL					bLastPeer;
			WLAN_REASON_CODE		wlanReasonCode;
		} WLAN_MSM_NOTIFICATION_DATA, *PWLAN_MSM_NOTIFICATION_DATA;

		#define WLAN_CONNECTION_NOTIFICATION_ADHOC_NETWORK_FORMED	0x00000001
		#define WLAN_CONNECTION_NOTIFICATION_CONSOLE_USER_PROFILE	0x00000004

		typedef struct _WLAN_CONNECTION_NOTIFICATION_DATA
		{
			WLAN_CONNECTION_MODE	wlanConnectionMode;
			WCHAR					strProfileName[WLAN_MAX_NAME_LENGTH];
			DOT11_SSID				dot11Ssid;
			DOT11_BSS_TYPE			dot11BssType;
			BOOL					bSecurityEnabled;
			WLAN_REASON_CODE		wlanReasonCode;
			DWORD					dwFlags;
			WCHAR					strProfileXml[1];
		} WLAN_CONNECTION_NOTIFICATION_DATA, *PWLAN_CONNECTION_NOTIFICATION_DATA;

		#define L2_NOTIFICATION_SOURCE_NONE				0
		#define L2_NOTIFICATION_SOURCE_ALL				0x0000FFFF
		#define L2_NOTIFICATION_SOURCE_WLAN_ACM			0x00000008
		#define L2_NOTIFICATION_SOURCE_WLAN_MSM			0x00000010
		#define L2_NOTIFICATION_SOURCE_WLAN_SECURITY	0x00000020
		#define L2_NOTIFICATION_SOURCE_WLAN_IHV			0x00000040
		#define L2_NOTIFICATION_SOURCE_WLAN_HNWK		0x00000080
		#define L2_NOTIFICATION_SOURCE_ONEX				0x00000004
		  
		#define WLAN_NOTIFICATION_SOURCE_NONE	L2_NOTIFICATION_SOURCE_NONE
		#define WLAN_NOTIFICATION_SOURCE_ALL	L2_NOTIFICATION_SOURCE_ALL

		#define WLAN_NOTIFICATION_SOURCE_ACM		L2_NOTIFICATION_SOURCE_WLAN_ACM
		#define WLAN_NOTIFICATION_SOURCE_MSM		L2_NOTIFICATION_SOURCE_WLAN_MSM
		#define WLAN_NOTIFICATION_SOURCE_SECURITY	L2_NOTIFICATION_SOURCE_WLAN_SECURITY
		#define WLAN_NOTIFICATION_SOURCE_IHV		L2_NOTIFICATION_SOURCE_WLAN_IHV
		#define WLAN_NOTIFICATION_SOURCE_HNWK		L2_NOTIFICATION_SOURCE_WLAN_HNWK
		#define WLAN_NOTIFICATION_SOURCE_ONEX		L2_NOTIFICATION_SOURCE_ONEX

		#define L2_NOTIFICATION_CODE_PUBLIC_BEGIN	0x00000000

		typedef enum _WLAN_NOTIFICATION_ACM
		{
			wlan_notification_acm_start = L2_NOTIFICATION_CODE_PUBLIC_BEGIN,
			wlan_notification_acm_autoconf_enabled,
			wlan_notification_acm_autoconf_disabled,
			wlan_notification_acm_background_scan_enabled,
			wlan_notification_acm_background_scan_disabled,
			wlan_notification_acm_bss_type_change,
			wlan_notification_acm_power_setting_change,
			wlan_notification_acm_scan_complete,
			wlan_notification_acm_scan_fail,
			wlan_notification_acm_connection_start,
			wlan_notification_acm_connection_complete,
			wlan_notification_acm_connection_attempt_fail,
			wlan_notification_acm_filter_list_change,
			wlan_notification_acm_interface_arrival,
			wlan_notification_acm_interface_removal,
			wlan_notification_acm_profile_change,
			wlan_notification_acm_profile_name_change,
			wlan_notification_acm_profiles_exhausted,
			wlan_notification_acm_network_not_available,
			wlan_notification_acm_network_available,
			wlan_notification_acm_disconnecting,
			wlan_notification_acm_disconnected,
			wlan_notification_acm_adhoc_network_state_change,
			wlan_notification_acm_profile_unblocked,
			wlan_notification_acm_screen_power_change,
			wlan_notification_acm_profile_blocked,
			wlan_notification_acm_scan_list_refresh,
			wlan_notification_acm_operational_state_change,
			wlan_notification_acm_end
		} WLAN_NOTIFICATION_ACM, *PWLAN_NOTIFICATION_ACM;

		typedef enum _WLAN_NOTIFICATION_MSM
		{
			wlan_notification_msm_start = L2_NOTIFICATION_CODE_PUBLIC_BEGIN,
			wlan_notification_msm_associating,
			wlan_notification_msm_associated,
			wlan_notification_msm_authenticating,
			wlan_notification_msm_connected,
			wlan_notification_msm_roaming_start,
			wlan_notification_msm_roaming_end,
			wlan_notification_msm_radio_state_change,
			wlan_notification_msm_signal_quality_change,
			wlan_notification_msm_disassociating,
			wlan_notification_msm_disconnected,
			wlan_notification_msm_peer_join,
			wlan_notification_msm_peer_leave,
			wlan_notification_msm_adapter_removal,
			wlan_notification_msm_adapter_operation_mode_change,
			wlan_notification_msm_link_degraded,
			wlan_notification_msm_link_improved,
			wlan_notification_msm_end
		} WLAN_NOTIFICATION_MSM, *PWLAN_NOTIFICATION_MSM;


		typedef enum _WLAN_NOTIFICATION_SECURITY
		{
			wlan_notification_security_start = L2_NOTIFICATION_CODE_PUBLIC_BEGIN,
			wlan_notification_security_end
		} WLAN_NOTIFICATION_SECURITY, *PWLAN_NOTIFICATION_SECURITY;

		typedef enum _ONEX_NOTIFICATION_TYPE
		{
			OneXPublicNotificationBase = 0,
			OneXNotificationTypeResultUpdate,
			OneXNotificationTypeAuthRestarted,
			OneXNotificationTypeEventInvalid,
			OneXNumNotifications = OneXNotificationTypeEventInvalid
		} ONEX_NOTIFICATION_TYPE, *PONEX_NOTIFICATION_TYPE;

		typedef enum ONEX_AUTH_RESTART_REASON
		{
			OneXRestartReasonPeerInitiated,
			OneXRestartReasonMsmInitiated,
			OneXRestartReasonOneXHeldStateTimeout,
			OneXRestartReasonOneXAuthTimeout,
			OneXRestartReasonOneXConfigurationChanged,
			OneXRestartReasonOneXUserChanged,
			OneXRestartReasonQuarantineStateChanged,
			OneXRestartReasonAltCredsTrial,
			OneXRestartReasonInvalid
		} ONEX_AUTH_RESTART_REASON, *PONEX_AUTH_RESTART_REASON;

		typedef struct _L2_NOTIFICATION_DATA
		{
			DWORD	NotificationSource;
			DWORD	NotificationCode;
			GUID	InterfaceGuid;
			DWORD	dwDataSize;
			PVOID	pData;
		} L2_NOTIFICATION_DATA, *PL2_NOTIFICATION_DATA;

		typedef L2_NOTIFICATION_DATA WLAN_NOTIFICATION_DATA, *PWLAN_NOTIFICATION_DATA;

		typedef void (__stdcall *WLAN_NOTIFICATION_CALLBACK)(PWLAN_NOTIFICATION_DATA data, PVOID context);

		typedef enum _WLAN_OPCODE_VALUE_TYPE
		{
			wlan_opcode_value_type_query_only = 0,
			wlan_opcode_value_type_set_by_group_policy,
			wlan_opcode_value_type_set_by_user,
			wlan_opcode_value_type_invalid
		} WLAN_OPCODE_VALUE_TYPE, *PWLAN_OPCODE_VALUE_TYPE;

		typedef DWORD WLAN_INTF_OPCODE, *PWLAN_INTF_OPCODE;

		typedef enum _ONEX_AUTH_STATUS
		{
			OneXAuthNotStarted,
			OneXAuthInProgress,
			OneXAuthNoAuthenticatorFound,
			OneXAuthSuccess,
			OneXAuthFailure,
			OneXAuthInvalid
		} ONEX_AUTH_STATUS;

		typedef struct _ONEX_STATUS
		{
			ONEX_AUTH_STATUS	authStatus;
			DWORD				dwReason;
			DWORD				dwError;
		} ONEX_STATUS;

		typedef enum _ONEX_EAP_METHOD_BACKEND_SUPPORT
		{
			OneXEapMethodBackendSupportUnknown,
			OneXEapMethodBackendSupported,
			OneXEapMethodBackendUnsupported
		} ONEX_EAP_METHOD_BACKEND_SUPPORT;

		typedef struct _ONEX_VARIABLE_BLOB
		{
			DWORD	dwSize;
			DWORD	dwOffset;
		} ONEX_VARIABLE_BLOB;

		typedef struct _ONEX_RESULT_UPDATE_DATA
		{
			ONEX_STATUS							oneXStatus;
			ONEX_EAP_METHOD_BACKEND_SUPPORT		BackendSupport;
			BOOL								fBackendEngaged;
			DWORD								fOneXAuthParams : 1;
			DWORD								fEapError : 1;
			ONEX_VARIABLE_BLOB					authParams;
			ONEX_VARIABLE_BLOB					eapError;
		} ONEX_RESULT_UPDATE_DATA, *PONEX_RESULT_UPDATE_DATA;

		typedef enum _ONEX_SUPPLICANT_MODE
		{
			OneXSupplicantModeInhibitTransmission,
			OneXSupplicantModeLearn,
			OneXSupplicantModeCompliant
		} ONEX_SUPPLICANT_MODE;

		typedef enum _ONEX_AUTH_MODE
		{
			OneXAuthModeMachineOrUser,
			OneXAuthModeMachineOnly,
			OneXAuthModeUserOnly,
			OneXAuthModeGuest,
			OneXAuthModeUnspecified
		} ONEX_AUTH_MODE;

		typedef struct _ONEX_CONNECTION_PROFILE
		{
			DWORD					dwVersion;
			DWORD					dwTotalLen;
			DWORD					fOneXSupplicantFlags : 1;
			DWORD					fsupplicantMode : 1;
			DWORD					fauthMode : 1;
			DWORD					fHeldPeriod : 1;
			DWORD					fAuthPeriod : 1;
			DWORD					fStartPeriod : 1;
			DWORD					fMaxStart : 1;
			DWORD					fMaxAuthFailures : 1;
			DWORD					fNetworkAuthTimeout : 1;
			DWORD					fAllowLogonDialogs : 1;
			DWORD					fNetworkAuthWithUITimeout : 1;
			DWORD					fUserBasedVLan : 1;
			DWORD					dwOneXSupplicantFlags;
			ONEX_SUPPLICANT_MODE	supplicantMode;
			ONEX_AUTH_MODE			authMode;
			DWORD					dwHeldPeriod;
			DWORD					dwAuthPeriod;
			DWORD					dwStartPeriod;
			DWORD					dwMaxStart;
			DWORD					dwMaxAuthFailures;
			DWORD					dwNetworkAuthTimeout;
			DWORD					dwNetworkAuthWithUITimeout;
			BOOL					bAllowLogonDialogs;
			BOOL					bUserBasedVLan;
		} ONEX_CONNECTION_PROFILE, *PONEX_CONNECTION_PROFILE;

		typedef enum _ONEX_AUTH_IDENTITY
		{
			OneXAuthIdentityNone,
			OneXAuthIdentityMachine,
			OneXAuthIdentityUser,
			OneXAuthIdentityExplicitUser,
			OneXAuthIdentityGuest,
			OneXAuthIdentityInvalid
		} ONEX_AUTH_IDENTITY;

		typedef enum _ISOLATION_STATE
		{
			ISOLATION_STATE_UNKNOWN = 0,
			ISOLATION_STATE_NOT_RESTRICTED = 1,
			ISOLATION_STATE_IN_PROBATION = 2,
			ISOLATION_STATE_RESTRICTED_ACCESS = 3
		} ISOLATION_STATE;

		typedef struct _ONEX_AUTH_PARAMS
		{
			BOOL				fUpdatePending;
			ONEX_VARIABLE_BLOB	oneXConnProfile;
			ONEX_AUTH_IDENTITY	authIdentity;
			ISOLATION_STATE		dwQuarantineState;
			DWORD				fSessionId : 1;
			DWORD				fhUserToken : 1;
			DWORD				fOnexUserProfile : 1;
			DWORD				fIdentity : 1;
			DWORD				fUserName : 1;
			DWORD				fDomain : 1;
			DWORD				dwSessionId;
			HANDLE				hUserToken;
			ONEX_VARIABLE_BLOB	OneXUserProfile;
			ONEX_VARIABLE_BLOB	Identity;
			ONEX_VARIABLE_BLOB	UserName;
			ONEX_VARIABLE_BLOB	Domain;
		} ONEX_AUTH_PARAMS, *PONEX_AUTH_PARAMS;

		typedef struct _EAP_TYPE
		{
			BYTE	type;
			DWORD	dwVendorId;
			DWORD	dwVendorType;
		} EAP_TYPE;

		typedef struct _EAP_METHOD_TYPE
		{
			EAP_TYPE	eapType;
			DWORD		dwAuthorId;
		} EAP_METHOD_TYPE;

		typedef struct _ONEX_EAP_ERROR
		{
			DWORD				dwWinError;
			EAP_METHOD_TYPE		type;
			DWORD				dwReasonCode;
			GUID				rootCauseGuid;
			GUID				repairGuid;
			GUID				helpLinkGuid;
			DWORD				fRootCauseString : 1;
			DWORD				fRepairString : 1;
			ONEX_VARIABLE_BLOB	RootCauseString;
			ONEX_VARIABLE_BLOB	RepairString;
		} ONEX_EAP_ERROR, *PONEX_EAP_ERROR;

		#define wlan_intf_opcode_autoconf_start									0x000000000
		#define wlan_intf_opcode_autoconf_enabled								0x000000001
		#define wlan_intf_opcode_background_scan_enabled						0x000000002
		#define wlan_intf_opcode_media_streaming_mode							0x000000003
		#define wlan_intf_opcode_radio_state									0x000000004
		#define wlan_intf_opcode_bss_type										0x000000005
		#define wlan_intf_opcode_interface_state								0x000000006
		#define wlan_intf_opcode_current_connection								0x000000007
		#define wlan_intf_opcode_channel_number									0x000000008
		#define wlan_intf_opcode_supported_infrastructure_auth_cipher_pairs		0x000000009
		#define wlan_intf_opcode_supported_adhoc_auth_cipher_pairs				0x00000000A
		#define wlan_intf_opcode_supported_country_or_region_string_list		0x00000000B
		#define wlan_intf_opcode_current_operation_mode							0x00000000C
		#define wlan_intf_opcode_supported_safe_mode							0x00000000D
		#define wlan_intf_opcode_certified_safe_mode							0x00000000E
		#define wlan_intf_opcode_hosted_network_capable							0x00000000F
		#define wlan_intf_opcode_management_frame_protection_capable			0x000000010
		#define wlan_intf_opcode_secondary_sta_interfaces						0x000000011
		#define wlan_intf_opcode_secondary_sta_synchronized_connections			0x000000012
		#define wlan_intf_opcode_autoconf_end									0x0fffffff
		#define wlan_intf_opcode_msm_start										0x10000100
		#define wlan_intf_opcode_statistics										0x10000101
		#define wlan_intf_opcode_rssi											0x10000102
		#define wlan_intf_opcode_msm_end										0x1fffffff
		#define wlan_intf_opcode_security_start									0x20010000
		#define wlan_intf_opcode_security_end									0x2fffffff
		#define wlan_intf_opcode_ihv_start										0x30000000
		#define wlan_intf_opcode_ihv_end										0x3fffffff

		typedef enum _WLAN_AUTOCONF_OPCODE
		{
			wlan_autoconf_opcode_start = 0,
			wlan_autoconf_opcode_show_denied_networks,
			wlan_autoconf_opcode_power_setting,
			wlan_autoconf_opcode_only_use_gp_profiles_for_allowed_networks,
			wlan_autoconf_opcode_allow_explicit_creds,
			wlan_autoconf_opcode_block_period,
			wlan_autoconf_opcode_allow_virtual_station_extensibility,
			wlan_autoconf_opcode_end
		} WLAN_AUTOCONF_OPCODE, *PWLAN_AUTOCONF_OPCODE;

		typedef enum WLAN_IHV_CONTROL_TYPE
		{
			wlan_ihv_control_type_service,
			wlan_ihv_control_type_driver
		} WLAN_IHV_CONTROL_TYPE, *PWLAN_IHV_CONTROL_TYPE;

		typedef enum WLAN_FILTER_LIST_TYPE
		{
			wlan_filter_list_type_gp_permit,
			wlan_filter_list_type_gp_deny,
			wlan_filter_list_type_user_permit,
			wlan_filter_list_type_user_deny
		} WLAN_FILTER_LIST_TYPE, *PWLAN_FILTER_LIST_TYPE;


		typedef struct WLAN_PHY_FRAME_STATISTICS
		{
			ULONGLONG	ullTransmittedFrameCount;
			ULONGLONG	ullMulticastTransmittedFrameCount;
			ULONGLONG	ullFailedCount;
			ULONGLONG	ullRetryCount;
			ULONGLONG	ullMultipleRetryCount;
			ULONGLONG	ullMaxTXLifetimeExceededCount;
			ULONGLONG	ullTransmittedFragmentCount;
			ULONGLONG	ullRTSSuccessCount;
			ULONGLONG	ullRTSFailureCount;
			ULONGLONG	ullACKFailureCount;
			ULONGLONG	ullReceivedFrameCount;
			ULONGLONG	ullMulticastReceivedFrameCount;
			ULONGLONG	ullPromiscuousReceivedFrameCount;
			ULONGLONG	ullMaxRXLifetimeExceededCount;
			ULONGLONG	ullFrameDuplicateCount;
			ULONGLONG	ullReceivedFragmentCount;
			ULONGLONG	ullPromiscuousReceivedFragmentCount;
			ULONGLONG	ullFCSErrorCount;
		} WLAN_PHY_FRAME_STATISTICS, *PWLAN_PHY_FRAME_STATISTICS;

		typedef struct _WLAN_MAC_FRAME_STATISTICS
		{
			ULONGLONG	ullTransmittedFrameCount;
			ULONGLONG	ullReceivedFrameCount;
			ULONGLONG	ullWEPExcludedCount;
			ULONGLONG	ullTKIPLocalMICFailures;
			ULONGLONG	ullTKIPReplays;
			ULONGLONG	ullTKIPICVErrorCount;
			ULONGLONG	ullCCMPReplays;
			ULONGLONG	ullCCMPDecryptErrors;
			ULONGLONG	ullWEPUndecryptableCount;
			ULONGLONG	ullWEPICVErrorCount;
			ULONGLONG	ullDecryptSuccessCount;
			ULONGLONG	ullDecryptFailureCount;
		} WLAN_MAC_FRAME_STATISTICS, *PWLAN_MAC_FRAME_STATISTICS;


		typedef struct _WLAN_STATISTICS
		{
			ULONGLONG					ullFourWayHandshakeFailures;
			ULONGLONG					ullTKIPCounterMeasuresInvoked;
			ULONGLONG					ullReserved;
			WLAN_MAC_FRAME_STATISTICS	MacUcastCounters;
			WLAN_MAC_FRAME_STATISTICS	MacMcastCounters;
			DWORD						dwNumberOfPhys;
			WLAN_PHY_FRAME_STATISTICS	PhyCounters[1];
		} WLAN_STATISTICS, *PWLAN_STATISTICS;

		#define WLAN_READ_ACCESS	(STANDARD_RIGHTS_READ | FILE_READ_DATA)
		#define WLAN_EXECUTE_ACCESS	(WLAN_READ_ACCESS | STANDARD_RIGHTS_EXECUTE | FILE_EXECUTE)
		#define WLAN_WRITE_ACCESS	(WLAN_READ_ACCESS | WLAN_EXECUTE_ACCESS | STANDARD_RIGHTS_WRITE | FILE_WRITE_DATA | DELETE | WRITE_DAC)

		typedef enum _WLAN_SECURABLE_OBJECT
		{
			wlan_secure_permit_list = 0,
			wlan_secure_deny_list,
			wlan_secure_ac_enabled,
			wlan_secure_bc_scan_enabled,
			wlan_secure_bss_type,
			wlan_secure_show_denied,
			wlan_secure_interface_properties,
			wlan_secure_ihv_control,
			wlan_secure_all_user_profiles_order,
			wlan_secure_add_new_all_user_profiles,
			wlan_secure_add_new_per_user_profiles,
			wlan_secure_media_streaming_mode_enabled,
			wlan_secure_current_operation_mode,
			wlan_secure_get_plaintext_key,
			wlan_secure_hosted_network_elevated_access,
			wlan_secure_virtual_station_extensibility,
			wlan_secure_wfd_elevated_access,
			WLAN_SECURABLE_OBJECT_COUNT
		} WLAN_SECURABLE_OBJECT, PWLAN_SECURABLE_OBJECT;

		#define WFD_API_VERSION_1_0		0x00000001

		#define WFD_API_VERSION			WFD_API_VERSION_1_0

		typedef enum _WFD_ROLE_TYPE
		{
			WFD_ROLE_TYPE_NONE = 0x00,
			WFD_ROLE_TYPE_DEVICE = 0x01,
			WFD_ROLE_TYPE_GROUP_OWNER = 0x02,
			WFD_ROLE_TYPE_CLIENT = 0x04,
			WFD_ROLE_TYPE_MAX = 0x05
		} WFD_ROLE_TYPE, PWFD_ROLE_TYPE;

		typedef struct _WFD_GROUP_ID
		{
			DOT11_MAC_ADDRESS	DeviceAddress;
			DOT11_SSID			GroupSSID;
		} WFD_GROUP_ID, *PWFD_GROUP_ID;

		#define DOT11_OPERATION_MODE_UNKNOWN				0x00000000
		#define DOT11_OPERATION_MODE_STATION				0x00000001
		#define DOT11_OPERATION_MODE_AP						0x00000002
		#define DOT11_OPERATION_MODE_EXTENSIBLE_STATION		0x00000004
		#define DOT11_OPERATION_MODE_EXTENSIBLE_AP			0x00000008
		#define DOT11_OPERATION_MODE_WFD_DEVICE				0x00000010
		#define DOT11_OPERATION_MODE_WFD_GROUP_OWNER		0x00000020
		#define DOT11_OPERATION_MODE_WFD_CLIENT				0x00000040
		#define DOT11_OPERATION_MODE_MANUFACTURING			0x40000000
		#define DOT11_OPERATION_MODE_NETWORK_MONITOR		0x80000000

		#ifndef ERROR_NDIS_DOT11_POWER_STATE_INVALID
			#define ERROR_NDIS_DOT11_POWER_STATE_INVALID			0x80342002
		#endif
		#ifndef EAP_I_USER_ACCOUNT_OTHER_ERROR
			#define EAP_I_USER_ACCOUNT_OTHER_ERROR					0x40420110
		#endif
		#ifndef EAP_E_CERT_STORE_INACCESSIBLE
			#define EAP_E_CERT_STORE_INACCESSIBLE					0x80420010
		#endif
		#ifndef EAP_E_EAPHOST_METHOD_NOT_INSTALLED
			#define EAP_E_EAPHOST_METHOD_NOT_INSTALLED				0x80420011
		#endif
		#ifndef EAP_E_EAPHOST_EAPQEC_INACCESSIBLE
			#define EAP_E_EAPHOST_EAPQEC_INACCESSIBLE				0x80420013
		#endif
		#ifndef EAP_E_EAPHOST_IDENTITY_UNKNOWN
			#define EAP_E_EAPHOST_IDENTITY_UNKNOWN					0x80420014
		#endif
		#ifndef EAP_E_AUTHENTICATION_FAILED
			#define EAP_E_AUTHENTICATION_FAILED						0x80420015
		#endif
		#ifndef EAP_I_EAPHOST_EAP_NEGOTIATION_FAILED
			#define EAP_I_EAPHOST_EAP_NEGOTIATION_FAILED			0x80420016
		#endif
		#ifndef EAP_E_EAPHOST_METHOD_INVALID_PACKET
			#define EAP_E_EAPHOST_METHOD_INVALID_PACKET				0x80420017
		#endif
		#ifndef EAP_E_EAPHOST_REMOTE_INVALID_PACKET
			#define EAP_E_EAPHOST_REMOTE_INVALID_PACKET				0x80420018
		#endif
		#ifndef EAP_E_EAPHOST_XML_MALFORMED
			#define EAP_E_EAPHOST_XML_MALFORMED						0x80420019
		#endif
		#ifndef EAP_E_METHOD_CONFIG_DOES_NOT_SUPPORT_SSO
			#define EAP_E_METHOD_CONFIG_DOES_NOT_SUPPORT_SSO		0x8042001A
		#endif
		#ifndef EAP_E_EAPHOST_METHOD_OPERATION_NOT_SUPPORTED
			#define EAP_E_EAPHOST_METHOD_OPERATION_NOT_SUPPORTED	0x80420020
		#endif
		#ifndef EAP_E_USER_CERT_NOT_FOUND
			#define EAP_E_USER_CERT_NOT_FOUND						0x80420100
		#endif
		#ifndef EAP_E_USER_CERT_INVALID
			#define EAP_E_USER_CERT_INVALID							0x80420101
		#endif
		#ifndef EAP_E_USER_CERT_EXPIRED
			#define EAP_E_USER_CERT_EXPIRED							0x80420102
		#endif
		#ifndef EAP_E_USER_CERT_REVOKED
			#define EAP_E_USER_CERT_REVOKED							0x80420103
		#endif
		#ifndef EAP_E_USER_CERT_OTHER_ERROR
			#define EAP_E_USER_CERT_OTHER_ERROR						0x80420104
		#endif
		#ifndef EAP_E_USER_CERT_REJECTED
			#define EAP_E_USER_CERT_REJECTED						0x80420105
		#endif
		#ifndef EAP_E_USER_CREDENTIALS_REJECTED
			#define EAP_E_USER_CREDENTIALS_REJECTED					0x80420111
		#endif
		#ifndef EAP_E_USER_NAME_PASSWORD_REJECTED
			#define EAP_E_USER_NAME_PASSWORD_REJECTED				0x80420112
		#endif
		#ifndef EAP_E_NO_SMART_CARD_READER
			#define EAP_E_NO_SMART_CARD_READER						0x80420113
		#endif
		#ifndef EAP_E_SERVER_CERT_INVALID
			#define EAP_E_SERVER_CERT_INVALID						0x80420201
		#endif
		#ifndef EAP_E_SERVER_CERT_EXPIRED
			#define EAP_E_SERVER_CERT_EXPIRED						0x80420202
		#endif
		#ifndef EAP_E_SERVER_CERT_REVOKED
			#define EAP_E_SERVER_CERT_REVOKED						0x80420203
		#endif
		#ifndef EAP_E_SERVER_CERT_OTHER_ERROR
			#define EAP_E_SERVER_CERT_OTHER_ERROR					0x80420204
		#endif
		#ifndef EAP_E_USER_ROOT_CERT_NOT_FOUND
			#define EAP_E_USER_ROOT_CERT_NOT_FOUND					0x80420300
		#endif
		#ifndef EAP_E_USER_ROOT_CERT_INVALID
			#define EAP_E_USER_ROOT_CERT_INVALID					0x80420301
		#endif
		#ifndef EAP_E_USER_ROOT_CERT_EXPIRED
			#define EAP_E_USER_ROOT_CERT_EXPIRED					0x80420302
		#endif
		#ifndef EAP_E_SERVER_ROOT_CERT_NOT_FOUND
			#define EAP_E_SERVER_ROOT_CERT_NOT_FOUND				0x80420400
		#endif

		DWORD WlanOpenHandle(DWORD dwClientVersion, PVOID pReserved, PDWORD pdwNegotiatedVersion,
			PHANDLE phClientHandle);
		DWORD WlanCloseHandle(HANDLE hClientHandle, PVOID pReserved);
		DWORD WlanEnumInterfaces(HANDLE hClientHandle, PVOID pReserved, PWLAN_INTERFACE_INFO_LIST* ppInterfaceList);
		DWORD WlanSetAutoConfigParameter(HANDLE hClientHandle, WLAN_AUTOCONF_OPCODE OpCode, DWORD dwDataSize,
			PVOID pData, PVOID pReserved);
		DWORD WlanQueryAutoConfigParameter(HANDLE hClientHandle, WLAN_AUTOCONF_OPCODE OpCode, PVOID pReserved,
			PDWORD pdwDataSize, PVOID* ppData, PWLAN_OPCODE_VALUE_TYPE pWlanOpcodeValueType);
		DWORD WlanGetInterfaceCapability(HANDLE hClientHandle, GUID* pInterfaceGuid, PVOID pReserved,
			PWLAN_INTERFACE_CAPABILITY* ppCapability);
		DWORD WlanSetInterface(HANDLE hClientHandle, GUID* pInterfaceGuid, WLAN_INTF_OPCODE OpCode,
			DWORD dwDataSize, PVOID pData, PVOID pReserved);
		DWORD WlanQueryInterface(HANDLE hClientHandle, GUID* pInterfaceGuid, WLAN_INTF_OPCODE OpCode,
			PVOID pReserved, PDWORD pdwDataSize, PVOID* ppData, PWLAN_OPCODE_VALUE_TYPE pWlanOpcodeValueType);
		DWORD WlanIhvControl(HANDLE hClientHandle, GUID* pInterfaceGuid, WLAN_IHV_CONTROL_TYPE Type,
			DWORD dwInBufferSize, PVOID pInBuffer, DWORD dwOutBufferSize, PVOID pOutBuffer,
			PDWORD pdwBytesReturned);
		DWORD WlanScan(HANDLE hClientHandle, GUID* pInterfaceGuid, PDOT11_SSID pDot11Ssid, PWLAN_RAW_DATA pIeData,
			PVOID pReserved);
		DWORD WlanGetAvailableNetworkList(HANDLE hClientHandle, GUID* pInterfaceGuid, DWORD dwFlags, PVOID pReserved,
			PWLAN_AVAILABLE_NETWORK_LIST* ppAvailableNetworkList);
		DWORD WlanGetAvailableNetworkList2(HANDLE hClientHandle, GUID* pInterfaceGuid, DWORD dwFlags, PVOID pReserved,
			PWLAN_AVAILABLE_NETWORK_LIST_V2* ppAvailableNetworkList);
		DWORD WlanGetNetworkBssList(HANDLE hClientHandle, GUID* pInterfaceGuid, PDOT11_SSID pDot11Ssid,
			DOT11_BSS_TYPE dot11BssType, BOOL bSecurityEnabled, PVOID pReserved, PWLAN_BSS_LIST* ppWlanBssList);
		DWORD WlanConnect(HANDLE hClientHandle, GUID* pInterfaceGuid, PWLAN_CONNECTION_PARAMETERS pConnectionParameters,
			PVOID pReserved);
		DWORD WlanConnect2(HANDLE hClientHandle, GUID* pInterfaceGuid, PWLAN_CONNECTION_PARAMETERS_V2 pConnectionParameters,
			PVOID pReserved);
		DWORD WlanDisconnect(HANDLE hClientHandle, GUID* pInterfaceGuid, PVOID pReserved);
		DWORD WlanRegisterNotification(HANDLE hClientHandle, DWORD dwNotifSource, BOOL bIgnoreDuplicate,
			WLAN_NOTIFICATION_CALLBACK funcCallback, PVOID pCallbackContext, PVOID pReserved,
			PDWORD pdwPrevNotifSource);
		DWORD WlanGetProfile(HANDLE hClientHandle, GUID* pInterfaceGuid, LPCWSTR strProfileName, PVOID pReserved,
			LPWSTR* pstrProfileXml, PDWORD pdwFlags, PDWORD pdwGrantedAccess);
		DWORD WlanSetProfileEapUserData(HANDLE hClientHandle, GUID* pInterfaceGuid, LPCWSTR strProfileName,
			EAP_METHOD_TYPE eapType, DWORD dwFlags, DWORD dwEapUserDataSize, LPBYTE pbEapUserData,
			PVOID pReserved);
		DWORD WlanSetProfileEapXmlUserData(HANDLE hClientHandle, GUID* pInterfaceGuid, LPCWSTR strProfileName,
			DWORD dwFlags, LPCWSTR strEapXmlUserData, PVOID pReserved);
		DWORD WlanSetProfile(HANDLE hClientHandle, GUID* pInterfaceGuid, DWORD dwFlags, LPCWSTR strProfileXml,
			LPCWSTR strAllUserProfileSecurity, BOOL bOverwrite, PVOID pReserved, PDWORD pdwReasonCode);
		DWORD WlanDeleteProfile(HANDLE hClientHandle, GUID* pInterfaceGuid, LPCWSTR strProfileName,
			PVOID pReserved);
		DWORD WlanRenameProfile(HANDLE hClientHandle, GUID* pInterfaceGuid, LPCWSTR strOldProfileName,
			LPCWSTR strNewProfileName, PVOID pReserved);
		DWORD WlanGetProfileList(HANDLE hClientHandle, GUID* pInterfaceGuid, PVOID pReserved,
			PWLAN_PROFILE_INFO_LIST* ppProfileList);
		DWORD WlanSetProfileList(HANDLE hClientHandle, GUID* pInterfaceGuid, DWORD dwItems, LPCWSTR* strProfileNames,
			PVOID pReserved);
		DWORD WlanSetProfilePosition(HANDLE hClientHandle, GUID* pInterfaceGuid, LPCWSTR strProfileName,
			DWORD dwPosition, PVOID pReserved);
		DWORD WlanSetProfileCustomUserData(HANDLE hClientHandle, GUID* pInterfaceGuid, LPCWSTR strProfileName,
			DWORD dwDataSize, PBYTE pData, PVOID pReserved);
		DWORD WlanGetProfileCustomUserData(HANDLE hClientHandle, GUID* pInterfaceGuid, LPCWSTR strProfileName,
			PVOID pReserved, PDWORD pdwDataSize, PBYTE* ppData);
		DWORD WlanSetFilterList(HANDLE hClientHandle, WLAN_FILTER_LIST_TYPE wlanFilterListType,
			PDOT11_NETWORK_LIST pNetworkList, PVOID pReserved);
		DWORD WlanGetFilterList(HANDLE hClientHandle, WLAN_FILTER_LIST_TYPE wlanFilterListType, PVOID pReserved,
			PDOT11_NETWORK_LIST* ppNetworkList);
		DWORD WlanSetPsdIEDataList(HANDLE hClientHandle, LPCWSTR strFormat, PWCL_WLAN_RAW_DATA_LIST pPsdIEDataList,
			PVOID pReserved);
		DWORD WlanSaveTemporaryProfile(HANDLE hClientHandle, GUID* pInterfaceGuid, LPCWSTR strProfileName,
			LPCWSTR strAllUserProfileSecurity, DWORD dwFlags, BOOL bOverWrite, PVOID pReserved);
		DWORD WlanExtractPsdIEDataList(HANDLE hClientHandle, DWORD dwIeDataSize, PBYTE pRawIeData,
			LPCWSTR strFormat, PVOID pReserved, PWLAN_RAW_DATA_LIST* ppPsdIEDataList);
		DWORD WlanReasonCodeToString(DWORD dwReasonCode, DWORD dwBufferSize, PWCHAR pStringBuffer, PVOID pReserved);
		PVOID WlanAllocateMemory(DWORD dwMemorySize);
		void WlanFreeMemory(PVOID pMemory);
		DWORD WlanSetSecuritySettings(HANDLE hClientHandle, WLAN_SECURABLE_OBJECT SecurableObject,
			LPCWSTR strModifiedSDDL);
		DWORD WlanGetSecuritySettings(HANDLE hClientHandle, WLAN_SECURABLE_OBJECT SecurableObject,
			PWLAN_OPCODE_VALUE_TYPE pValueType, LPWSTR* pstrCurrentSDDL, PDWORD pdwGrantedAccess);

		/* Native WiFi UI declarations. */

		#define WLAN_UI_API_VERSION				1
		#define WLAN_UI_API_INITIAL_VERSION		1

		typedef enum _WL_DISPLAY_PAGES
		{
			WLConnectionPage,
			WLSecurityPage,
			WLAdvPage
		} WL_DISPLAY_PAGES, *PWL_DISPLAY_PAGES;

		DWORD WlanUIEditProfile(DWORD dwClientVersion, LPCWSTR wstrProfileName, GUID* pInterfaceGuid,
			HWND hWnd, WL_DISPLAY_PAGES wlStartPage, PVOID pReserved, PWLAN_REASON_CODE pWlanReasonCode);

		/* WiFi Hosted Network declarations. */

		typedef enum _WLAN_HOSTED_NETWORK_STATE
		{
			wlan_hosted_network_unavailable,
			wlan_hosted_network_idle,
			wlan_hosted_network_active
		} WLAN_HOSTED_NETWORK_STATE, *PWLAN_HOSTED_NETWORK_STATE;

		typedef enum _WLAN_HOSTED_NETWORK_REASON
		{
			wlan_hosted_network_reason_success = 0,
			wlan_hosted_network_reason_unspecified,
			wlan_hosted_network_reason_bad_parameters,
			wlan_hosted_network_reason_service_shutting_down,
			wlan_hosted_network_reason_insufficient_resources,
			wlan_hosted_network_reason_elevation_required,
			wlan_hosted_network_reason_read_only,
			wlan_hosted_network_reason_persistence_failed,
			wlan_hosted_network_reason_crypt_error,
			wlan_hosted_network_reason_impersonation,
			wlan_hosted_network_reason_stop_before_start,
			wlan_hosted_network_reason_interface_available,
			wlan_hosted_network_reason_interface_unavailable,
			wlan_hosted_network_reason_miniport_stopped,
			wlan_hosted_network_reason_miniport_started,
			wlan_hosted_network_reason_incompatible_connection_started,
			wlan_hosted_network_reason_incompatible_connection_stopped,
			wlan_hosted_network_reason_user_action,
			wlan_hosted_network_reason_client_abort,
			wlan_hosted_network_reason_ap_start_failed,
			wlan_hosted_network_reason_peer_arrived,
			wlan_hosted_network_reason_peer_departed,
			wlan_hosted_network_reason_peer_timeout,
			wlan_hosted_network_reason_gp_denied,
			wlan_hosted_network_reason_service_unavailable,
			wlan_hosted_network_reason_device_change,
			wlan_hosted_network_reason_properties_change,
			wlan_hosted_network_reason_virtual_station_blocking_use,
			wlan_hosted_network_reason_service_available_on_virtual_station
		} WLAN_HOSTED_NETWORK_REASON, *PWLAN_HOSTED_NETWORK_REASON;

		typedef enum _WLAN_HOSTED_NETWORK_PEER_AUTH_STATE
		{
			wlan_hosted_network_peer_state_invalid,
			wlan_hosted_network_peer_state_authenticated
		} WLAN_HOSTED_NETWORK_PEER_AUTH_STATE, *PWLAN_HOSTED_NETWORK_PEER_AUTH_STATE;

		DWORD WlanHostedNetworkStartUsing(HANDLE hClientHandle, PWLAN_HOSTED_NETWORK_REASON pFailReason,
			PVOID pvReserved);
		DWORD WlanHostedNetworkStopUsing(HANDLE hClientHandle, PWLAN_HOSTED_NETWORK_REASON pFailReason,
			PVOID pvReserved);
		DWORD WlanHostedNetworkForceStart(HANDLE hClientHandle, PWLAN_HOSTED_NETWORK_REASON pFailReason,
			PVOID pvReserved);
		DWORD WlanHostedNetworkForceStop(HANDLE hClientHandle, PWLAN_HOSTED_NETWORK_REASON pFailReason, PVOID pvReserved);

		typedef struct _WLAN_HOSTED_NETWORK_PEER_STATE
		{
			DOT11_MAC_ADDRESS						PeerMacAddress;
			WLAN_HOSTED_NETWORK_PEER_AUTH_STATE		PeerAuthState;
		} WLAN_HOSTED_NETWORK_PEER_STATE, *PWLAN_HOSTED_NETWORK_PEER_STATE;

		typedef struct _WLAN_HOSTED_NETWORK_RADIO_STATE
		{
			DOT11_RADIO_STATE	dot11SoftwareRadioState;
			DOT11_RADIO_STATE	dot11HardwareRadioState;
		} WLAN_HOSTED_NETWORK_RADIO_STATE, *PWLAN_HOSTED_NETWORK_RADIO_STATE;

		#define L2_NOTIFICATION_CODE_GROUP_SIZE		0x00001000
		#define L2_NOTIFICATION_CODE_V2_BEGIN		(L2_NOTIFICATION_CODE_PUBLIC_BEGIN + L2_NOTIFICATION_CODE_GROUP_SIZE)

		typedef DWORD WLAN_HOSTED_NETWORK_NOTIFICATION_CODE, *PWLAN_HOSTED_NETWORK_NOTIFICATION_CODE;

		#define wlan_hosted_network_state_change		L2_NOTIFICATION_CODE_V2_BEGIN
		#define wlan_hosted_network_peer_state_change	(L2_NOTIFICATION_CODE_V2_BEGIN + 1)
		#define wlan_hosted_network_radio_state_change	(L2_NOTIFICATION_CODE_V2_BEGIN + 2)

		typedef struct _WLAN_HOSTED_NETWORK_STATE_CHANGE_REC
		{
			WLAN_HOSTED_NETWORK_STATE	OldState;
			WLAN_HOSTED_NETWORK_STATE	NewState;
			WLAN_HOSTED_NETWORK_REASON	StateChangeReason;
		} WLAN_HOSTED_NETWORK_STATE_CHANGE_REC, *PWLAN_HOSTED_NETWORK_STATE_CHANGE_REC;

		typedef struct _WLAN_HOSTED_NETWORK_DATA_PEER_STATE_CHANGE_REC
		{
			WLAN_HOSTED_NETWORK_PEER_STATE	OldState;
			WLAN_HOSTED_NETWORK_PEER_STATE	NewState;
			WLAN_HOSTED_NETWORK_REASON	PeerStateChangeReason;
		} WLAN_HOSTED_NETWORK_DATA_PEER_STATE_CHANGE_REC, *PWLAN_HOSTED_NETWORK_DATA_PEER_STATE_CHANGE_REC;

		typedef enum _WLAN_HOSTED_NETWORK_OPCODE
		{
			wlan_hosted_network_opcode_connection_settings,
			wlan_hosted_network_opcode_security_settings,
			wlan_hosted_network_opcode_station_profile,
			wlan_hosted_network_opcode_enable
		} WLAN_HOSTED_NETWORK_OPCODE, *PWLAN_HOSTED_NETWORK_OPCODE;

		typedef struct _WLAN_HOSTED_NETWORK_CONNECTION_SETTINGS
		{
			DOT11_SSID	hostedNetworkSSID;
			DWORD		dwMaxNumberOfPeers;
		} WLAN_HOSTED_NETWORK_CONNECTION_SETTINGS, *PWLAN_HOSTED_NETWORK_CONNECTION_SETTINGS;

		typedef struct _WLAN_HOSTED_NETWORK_SECURITY_SETTINGS
		{
			DOT11_AUTH_ALGORITHM	dot11AuthAlgo;
			DOT11_CIPHER_ALGORITHM	dot11CipherAlgo;
		} WLAN_HOSTED_NETWORK_SECURITY_SETTINGS, *PWLAN_HOSTED_NETWORK_SECURITY_SETTINGS;

		DWORD WlanHostedNetworkQueryProperty(HANDLE hClientHandle, WLAN_HOSTED_NETWORK_OPCODE OpCode,
			PDWORD pdwDataSize, PVOID* ppvData, PWLAN_OPCODE_VALUE_TYPE pWlanOpcodeValueType,
			PVOID pvReserved);
		DWORD WlanHostedNetworkSetProperty(HANDLE hClientHandle, WLAN_HOSTED_NETWORK_OPCODE OpCode,
			DWORD dwDataSize, PVOID pvData, PWLAN_HOSTED_NETWORK_REASON pFailReason, PVOID pvReserved);
		DWORD WlanHostedNetworkInitSettings(HANDLE hClientHandle, PWLAN_HOSTED_NETWORK_REASON pFailReason,
			PVOID pvReserved);
		DWORD WlanHostedNetworkRefreshSecuritySettings(HANDLE hClientHandle, PWLAN_HOSTED_NETWORK_REASON pFailReason,
			PVOID pvReserved);

		typedef struct _WLAN_HOSTED_NETWORK_STATUS
		{
			WLAN_HOSTED_NETWORK_STATE		HostedNetworkState;
			GUID							IPDeviceID;
			DOT11_MAC_ADDRESS				wlanHostedNetworkBSSID;
			DOT11_PHY_TYPE					dot11PhyType;
			ULONG							ulChannelFrequency;
			DWORD							dwNumberOfPeers;
			WLAN_HOSTED_NETWORK_PEER_STATE	PeerList[1];
		} WLAN_HOSTED_NETWORK_STATUS, *PWLAN_HOSTED_NETWORK_STATUS;

		DWORD WlanHostedNetworkQueryStatus(HANDLE hClientHandle, PWLAN_HOSTED_NETWORK_STATUS* ppWlanHostedNetworkStatus,
			PVOID pvReserved);
		DWORD WlanHostedNetworkSetSecondaryKey(HANDLE hClientHandle, DWORD dwKeyLength, PUCHAR pucKeyData,
			BOOL bIsPassPhrase, BOOL bPersistent, PWLAN_HOSTED_NETWORK_REASON pFailReason,
			PVOID pvReserved);
		DWORD WlanHostedNetworkQuerySecondaryKey(HANDLE hClientHandle, PDWORD pdwKeyLength, PUCHAR* ppucKeyData,
			PBOOL pbIsPassPhrase, PBOOL pbPersistent, PWLAN_HOSTED_NETWORK_REASON pFailReason, PVOID pvReserved);
		DWORD WlanRegisterVirtualStationNotification(HANDLE hClientHandle, BOOL bRegister, PVOID pReserved);

		/* Direct WiFi declarations. */

		DWORD WFDOpenHandle(DWORD dwClientVersion, PDWORD pdwNegotiatedVersion, PHANDLE phClientHandle);
		DWORD WFDCloseHandle(HANDLE hClientHandle);

		typedef void (__stdcall *WFD_OPEN_SESSION_COMPLETE_CALLBACK)(HANDLE hSessionHandle, PVOID pvContext,
			GUID guidSessionInterface, DWORD dwError, DWORD dwReasonCode);

		DWORD WFDStartOpenSession(HANDLE hClientHandle, PDOT11_MAC_ADDRESS pDeviceAddress, PVOID pvContext,
			WFD_OPEN_SESSION_COMPLETE_CALLBACK pfnCallback, PHANDLE phSessionHandle);
		DWORD WFDCancelOpenSession(HANDLE hSessionHandle);
		DWORD WFDOpenLegacySession(HANDLE hClientHandle, PDOT11_MAC_ADDRESS pLegacyMacAddress, PHANDLE phSessionHandle,
			GUID* pGuidSessionInterface);
		DWORD WFDCloseSession(HANDLE hSessionHandle);
		DWORD WFDUpdateDeviceVisibility(PDOT11_MAC_ADDRESS pDeviceAddress);

		/* WiFi Direct WinRT API. */

		// WiFi Direct WinRT types.

		typedef enum
		{
			WiFiDirectConnectionStatus_Disconnected = 0,
			WiFiDirectConnectionStatus_Connected = 1
		} WiFiDirectConnectionStatus;

		typedef enum
		{
			WiFiDirectError_Success = 0,
			WiFiDirectError_RadioNotAvailable = 1,
			WiFiDirectError_ResourceInUse = 2
		} WiFiDirectError;

		typedef enum
		{
			WiFiDirectDeviceSelectorType_DeviceInterface = 0,
			WiFiDirectDeviceSelectorType_AssociationEndpoint = 1
		} WiFiDirectDeviceSelectorType;

		typedef enum
		{
			WiFiDirectAdvertisementListenStateDiscoverability_None = 0,
			WiFiDirectAdvertisementListenStateDiscoverability_Normal = 1,
			WiFiDirectAdvertisementListenStateDiscoverability_Intensive = 2
		} WiFiDirectAdvertisementListenStateDiscoverability;

		typedef enum
		{
			WiFiDirectAdvertisementPublisherStatus_Created = 0,
			WiFiDirectAdvertisementPublisherStatus_Started = 1,
			WiFiDirectAdvertisementPublisherStatus_Stopped = 2,
			WiFiDirectAdvertisementPublisherStatus_Aborted = 3
		} WiFiDirectAdvertisementPublisherStatus;

		typedef enum
		{
			WiFiDirectConfigurationMethod_ProvidePin = 0,
			WiFiDirectConfigurationMethod_DisplayPin = 1,
			WiFiDirectConfigurationMethod_PushButton = 2
		} WiFiDirectConfigurationMethod;

		typedef enum
		{
			WiFiDirectPairingProcedure_GroupOwnerNegotiation = 0,
			WiFiDirectPairingProcedure_Invitation = 1
		} WiFiDirectPairingProcedure;

		typedef enum
		{
			WiFiDirectServiceAdvertisementStatus_Created = 0,
			WiFiDirectServiceAdvertisementStatus_Started = 1,
			WiFiDirectServiceAdvertisementStatus_Stopped = 2,
			WiFiDirectServiceAdvertisementStatus_Aborted = 3
		} WiFiDirectServiceAdvertisementStatus;

		typedef enum
		{
			WiFiDirectServiceConfigurationMethod_Default = 0,
			WiFiDirectServiceConfigurationMethod_PinDisplay = 1,
			WiFiDirectServiceConfigurationMethod_PinEntry = 2
		} WiFiDirectServiceConfigurationMethod;

		typedef enum
		{
			WiFiDirectServiceError_Success = 0,
			WiFiDirectServiceError_RadioNotAvailable = 1,
			WiFiDirectServiceError_ResourceInUse = 2,
			WiFiDirectServiceError_UnsupportedHardware = 3,
			WiFiDirectServiceError_NoHardware = 4
		} WiFiDirectServiceError;

		typedef enum
		{
			WiFiDirectServiceIPProtocol_Tcp = 6,
			WiFiDirectServiceIPProtocol_Udp = 17
		} WiFiDirectServiceIPProtocol;

		typedef enum
		{
			WiFiDirectServiceSessionErrorStatus_Ok = 0,
			WiFiDirectServiceSessionErrorStatus_Disassociated = 1,
			WiFiDirectServiceSessionErrorStatus_LocalClose = 2,
			WiFiDirectServiceSessionErrorStatus_RemoteClose = 3,
			WiFiDirectServiceSessionErrorStatus_SystemFailure = 4,
			WiFiDirectServiceSessionErrorStatus_NoResponseFromRemote = 5
		} WiFiDirectServiceSessionErrorStatus;

		typedef enum
		{
			WiFiDirectServiceSessionStatus_Closed = 0,
			WiFiDirectServiceSessionStatus_Initiated = 1,
			WiFiDirectServiceSessionStatus_Requested = 2,
			WiFiDirectServiceSessionStatus_Open = 3
		} WiFiDirectServiceSessionStatus;

		typedef enum
		{
			WiFiDirectServiceStatus_Available = 0,
			WiFiDirectServiceStatus_Busy = 1,
			WiFiDirectServiceStatus_Custom = 2
		} WiFiDirectServiceStatus;

		typedef enum
		{
			TetheringOperationalState_Unknown = 0,
			TetheringOperationalState_On = 1,
			TetheringOperationalState_Off = 2,
			TetheringOperationalState_InTransition = 3
		} TetheringOperationalState;

		typedef enum
		{
			TetheringOperationStatus_Success = 0,
			TetheringOperationStatus_Unknown = 1,
			TetheringOperationStatus_MobileBroadbandDeviceOff = 2,
			TetheringOperationStatus_WiFiDeviceOff = 3,
			TetheringOperationStatus_EntitlementCheckTimeout = 4,
			TetheringOperationStatus_EntitlementCheckFailure = 5,
			TetheringOperationStatus_OperationInProgress = 6,
			TetheringOperationStatus_BluetoothDeviceOff = 7,
			TetheringOperationStatus_NetworkLimitedConnectivity = 8
		} TetheringOperationStatus;

		typedef enum
		{
			TetheringCapability_Enabled = 0,
			TetheringCapability_DisabledByGroupPolicy = 1,
			TetheringCapability_DisabledByHardwareLimitation = 2,
			TetheringCapability_DisabledByOperator = 3,
			TetheringCapability_DisabledBySku = 4,
			TetheringCapability_DisabledByRequiredAppNotInstalled = 5,
			TetheringCapability_DisabledDueToUnknownCause = 6,
			TetheringCapability_DisabledBySystemCapability = 7
		} TetheringCapability;

		typedef enum
		{
			TetheringWiFiBand_Auto = 0,
			TetheringWiFiBand_TwoPointFourGigahertz = 1,
			TetheringWiFiBand_FiveGigahertz = 2
		} TetheringWiFiBand;

		// WiFi Direct WinRT interface names.

		#define WiFiDirectAdvertisementPublisherName	_T("Windows.Devices.WiFiDirect.WiFiDirectAdvertisementPublisher")
		#define WiFiDirectConnectionListenerName		_T("Windows.Devices.WiFiDirect.WiFiDirectConnectionListener")
		#define WiFiDirectDeviceName					_T("Windows.Devices.WiFiDirect.WiFiDirectDevice")
		#define WiFiDirectInformationElementName		_T("Windows.Devices.WiFiDirect.WiFiDirectInformationElement")
		#define WiFiDirectConnectionParametersName		_T("Windows.Devices.WiFiDirect.WiFiDirectConnectionParameters")
		#define NetworkOperatorTetheringManagerName		_T("Windows.Networking.NetworkOperators.NetworkOperatorTetheringManager")
		#define NetworkInformationName					_T("Windows.Networking.Connectivity.NetworkInformation")

		// WiFi Direct WinRT interface definitions.

		typedef interface IWiFiDirectInformationElementIterator IWiFiDirectInformationElementIterator;
		typedef interface IWiFiDirectInformationElementIterable IWiFiDirectInformationElementIterable;
		typedef interface IWiFiDirectInformationElementVectorView IWiFiDirectInformationElementVectorView;
		typedef interface IWiFiDirectInformationElementVector IWiFiDirectInformationElementVector;
		typedef interface IWiFiDirectDeviceAsyncOperationCompletedHandler IWiFiDirectDeviceAsyncOperationCompletedHandler;
		typedef interface IWiFiDirectDeviceAsyncOperation IWiFiDirectDeviceAsyncOperation;
		typedef interface IWiFiDirectAdvertisementPublisherStatusChangedEventHandler IWiFiDirectAdvertisementPublisherStatusChangedEventHandler;
		typedef interface IWiFiDirectConnectionRequestedEventHandler IWiFiDirectConnectionRequestedEventHandler;
		typedef interface IWiFiDirectDeviceEventHandler IWiFiDirectDeviceEventHandler;
		typedef interface IWiFiDirectDeviceStatics IWiFiDirectDeviceStatics;
		typedef interface IWiFiDirectDeviceStatics2 IWiFiDirectDeviceStatics2;
		typedef interface IWiFiDirectInformationElementStatics IWiFiDirectInformationElementStatics;
		typedef interface IWiFiDirectInformationElement IWiFiDirectInformationElement;
		typedef interface IWiFiDirectLegacySettings IWiFiDirectLegacySettings;
		typedef interface IWiFiDirectAdvertisement IWiFiDirectAdvertisement;
		typedef interface IWiFiDirectAdvertisementPublisherStatusChangedEventArgs IWiFiDirectAdvertisementPublisherStatusChangedEventArgs;
		typedef interface IWiFiDirectAdvertisementPublisher IWiFiDirectAdvertisementPublisher;
		typedef interface IWiFiDirectConnectionParameters IWiFiDirectConnectionParameters;
		typedef interface IWiFiDirectConnectionRequestedEventArgs IWiFiDirectConnectionRequestedEventArgs;
		typedef interface IWiFiDirectConnectionListener IWiFiDirectConnectionListener;
		typedef interface IWiFiDirectDevice IWiFiDirectDevice;
		typedef interface IWiFiDirectConnectionRequest IWiFiDirectConnectionRequest;
		typedef interface IWiFiDirectDeviceEventHandler IWiFiDirectDeviceEventHandler;
		typedef interface IWiFiDirectDeviceStatics IWiFiDirectDeviceStatics;
		typedef interface IWiFiDirectDeviceStatics2 IWiFiDirectDeviceStatics2;
		typedef interface IWiFiDirectInformationElementStatics IWiFiDirectInformationElementStatics;
		typedef interface IWiFiDirectInformationElement IWiFiDirectInformationElement;
		typedef interface IWiFiDirectLegacySettings IWiFiDirectLegacySettings;
		typedef interface IWiFiDirectAdvertisement IWiFiDirectAdvertisement;
		typedef interface IWiFiDirectAdvertisementPublisherStatusChangedEventArgs IWiFiDirectAdvertisementPublisherStatusChangedEventArgs;
		typedef interface IWiFiDirectAdvertisementPublisher IWiFiDirectAdvertisementPublisher;
		typedef interface IWiFiDirectConnectionParameters IWiFiDirectConnectionParameters;
		typedef interface IWiFiDirectConnectionRequestedEventArgs IWiFiDirectConnectionRequestedEventArgs;
		typedef interface IWiFiDirectConnectionListener IWiFiDirectConnectionListener;
		typedef interface IWiFiDirectDevice IWiFiDirectDevice;
		typedef interface IWiFiDirectConnectionRequest IWiFiDirectConnectionRequest;
		typedef interface IWiFiDirectConfigurationMethodIterator IWiFiDirectConfigurationMethodIterator;
		typedef interface IWiFiDirectConfigurationMethodIterable IWiFiDirectConfigurationMethodIterable;
		typedef interface IWiFiDirectConfigurationMethodVectorView IWiFiDirectConfigurationMethodVectorView;
		typedef interface IWiFiDirectConfigurationMethodVector IWiFiDirectConfigurationMethodVector;
		typedef interface IWiFiDirectAdvertisement2 IWiFiDirectAdvertisement2;
		typedef interface IWiFiDirectConnectionParameters2 IWiFiDirectConnectionParameters2;
		typedef interface IWiFiDirectConnectionParametersStatics IWiFiDirectConnectionParametersStatics;
		typedef interface IWiFiDirectServiceConfigurationMethodIterator IWiFiDirectServiceConfigurationMethodIterator;
		typedef interface IWiFiDirectServiceConfigurationMethodIterable IWiFiDirectServiceConfigurationMethodIterable;
		typedef interface IWiFiDirectServiceConfigurationMethodVectorView IWiFiDirectServiceConfigurationMethodVectorView;
		typedef interface IWiFiDirectServiceConfigurationMethodVector IWiFiDirectServiceConfigurationMethodVector;
		typedef interface IWiFiDirectServiceAsyncOperationCompletedHandler IWiFiDirectServiceAsyncOperationCompletedHandler;
		typedef interface IWiFiDirectServiceAsyncOperation IWiFiDirectServiceAsyncOperation;
		typedef interface IWiFiDirectServiceProvisioningInfoAsyncOperationCompletedHandler IWiFiDirectServiceProvisioningInfoAsyncOperationCompletedHandler;
		typedef interface IWiFiDirectServiceProvisioningInfoAsyncOperation IWiFiDirectServiceProvisioningInfoAsyncOperation;
		typedef interface IWiFiDirectServiceSessionAsyncOperationCompletedHandler IWiFiDirectServiceSessionAsyncOperationCompletedHandler;
		typedef interface IWiFiDirectServiceSessionAsyncOperation IWiFiDirectServiceSessionAsyncOperation;
		typedef interface IWiFiDirectServiceSessionDeferredEventHandler IWiFiDirectServiceSessionDeferredEventHandler;
		typedef interface IWiFiDirectServiceAdvertiserEventHandler IWiFiDirectServiceAdvertiserEventHandler;
		typedef interface IWiFiDirectServiceAutoAcceptSessionConnectedEventHandler IWiFiDirectServiceAutoAcceptSessionConnectedEventHandler;
		typedef interface IWiFiDirectServiceSessionRequestedEventHandler IWiFiDirectServiceSessionRequestedEventHandler;
		typedef interface IWiFiDirectServiceSessionEventHandler IWiFiDirectServiceSessionEventHandler;
		typedef interface IWiFiDirectServiceRemotePortAddedEventHandler IWiFiDirectServiceRemotePortAddedEventHandler;
		typedef interface IWiFiDirectService IWiFiDirectService;
		typedef interface IWiFiDirectServiceAdvertiser IWiFiDirectServiceAdvertiser;
		typedef interface IWiFiDirectServiceAdvertiserFactory IWiFiDirectServiceAdvertiserFactory;
		typedef interface IWiFiDirectServiceAutoAcceptSessionConnectedEventArgs IWiFiDirectServiceAutoAcceptSessionConnectedEventArgs;
		typedef interface IWiFiDirectServiceProvisioningInfo IWiFiDirectServiceProvisioningInfo;
		typedef interface IWiFiDirectServiceRemotePortAddedEventArgs IWiFiDirectServiceRemotePortAddedEventArgs;
		typedef interface IWiFiDirectServiceSession IWiFiDirectServiceSession;
		typedef interface IWiFiDirectServiceSessionDeferredEventArgs IWiFiDirectServiceSessionDeferredEventArgs;
		typedef interface IWiFiDirectServiceSessionRequest IWiFiDirectServiceSessionRequest;
		typedef interface IWiFiDirectServiceSessionRequestedEventArgs IWiFiDirectServiceSessionRequestedEventArgs;
		typedef interface IWiFiDirectServiceStatics IWiFiDirectServiceStatics;
		typedef interface INetworkOperatorTetheringManager INetworkOperatorTetheringManager;
		typedef interface INetworkOperatorTetheringAccessPointConfiguration INetworkOperatorTetheringAccessPointConfiguration;
		typedef interface INetworkOperatorTetheringOperationResultAsyncOperation INetworkOperatorTetheringOperationResultAsyncOperation;
		typedef interface INetworkOperatorTetheringOperationResultAsyncOperationCompletedHandler INetworkOperatorTetheringOperationResultAsyncOperationCompletedHandler;
		typedef interface INetworkOperatorTetheringOperationResult INetworkOperatorTetheringOperationResult;
		typedef interface INetworkOperatorTetheringManagerStatics INetworkOperatorTetheringManagerStatics;
		typedef interface INetworkOperatorTetheringManagerStatics2 INetworkOperatorTetheringManagerStatics2;
		typedef interface INetworkOperatorTetheringManagerStatics3 INetworkOperatorTetheringManagerStatics3;
		typedef interface INetworkOperatorTetheringManagerStatics4 INetworkOperatorTetheringManagerStatics4;
		typedef interface INetworkInformationStatics INetworkInformationStatics;
		typedef interface INetworkOperatorTetheringClientManager INetworkOperatorTetheringClientManager;
		typedef interface INetworkOperatorTetheringClientVectorView INetworkOperatorTetheringClientVectorView;
		typedef interface INetworkOperatorTetheringClient INetworkOperatorTetheringClient;
		typedef interface INetworkOperatorTetheringAccessPointConfiguration2 INetworkOperatorTetheringAccessPointConfiguration2;

		// WiFi Direct WinRT interface declarations.

		MIDL_INTERFACE("cf806026-c915-553e-af3c-8da43871b693")
		IWiFiDirectInformationElementIterator : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_Current(IWiFiDirectInformationElement** current) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_HasCurrent(BOOLEAN* hasCurrent) = 0;
			virtual HRESULT STDMETHODCALLTYPE MoveNext(BOOLEAN* hasCurrent) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetMany(DWORD capacity, IWiFiDirectInformationElement** items,
				DWORD* actual) = 0;
		};

		MIDL_INTERFACE("19c1ca4e-9561-5253-96d9-dbaf28d47d89")
		IWiFiDirectInformationElementIterable : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE First(IWiFiDirectInformationElementIterator** first) = 0;
		};

		MIDL_INTERFACE("6dcffadb-04c5-535e-add4-1389b3be6eca")
		IWiFiDirectInformationElementVectorView : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetAt(DWORD index, IWiFiDirectInformationElement** item) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Size(DWORD* size) = 0;
			virtual HRESULT STDMETHODCALLTYPE IndexOf(IWiFiDirectInformationElement* item, DWORD* index, BOOLEAN* found) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetMany(DWORD startIndex, DWORD capacity, IWiFiDirectInformationElement** items,
				DWORD* actual) = 0;
		};

		MIDL_INTERFACE("b8c55492-e4de-5ba7-8476-d3bab557cdd6")
		IWiFiDirectInformationElementVector : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetAt(DWORD index, IWiFiDirectInformationElement** item) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Size(DWORD* size) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetView(IWiFiDirectInformationElementVectorView** view) = 0;
			virtual HRESULT STDMETHODCALLTYPE IndexOf(IWiFiDirectInformationElement item,
				DWORD* index, BOOLEAN* found) = 0;
			virtual HRESULT STDMETHODCALLTYPE SetAt(DWORD index, IWiFiDirectInformationElement* item) = 0;
			virtual HRESULT STDMETHODCALLTYPE InsertAt(DWORD index, IWiFiDirectInformationElement* item) = 0;
			virtual HRESULT STDMETHODCALLTYPE RemoveAt(DWORD index) = 0;
			virtual HRESULT STDMETHODCALLTYPE Append(IWiFiDirectInformationElement* item) = 0;
			virtual HRESULT STDMETHODCALLTYPE RemoveAtEnd() = 0;
			virtual HRESULT STDMETHODCALLTYPE Clear() = 0;
			virtual HRESULT STDMETHODCALLTYPE GetMany(DWORD startIndex, DWORD capacity,
				IWiFiDirectInformationElement** items, DWORD* actual) = 0;
			virtual HRESULT STDMETHODCALLTYPE ReplaceAll(DWORD count, IWiFiDirectInformationElement** value) = 0;
		};

		MIDL_INTERFACE("d34abe17-fb19-57be-bc41-0eb83dea151c")
		IWiFiDirectDeviceAsyncOperationCompletedHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(IWiFiDirectDeviceAsyncOperation* asyncInfo,
				wclCommon::WinApi::AsyncStatus status) = 0;
		};

		MIDL_INTERFACE("dad01b61-a82d-566c-ba82-224c11500669")
		IWiFiDirectDeviceAsyncOperation : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE put_Completed(
				IWiFiDirectDeviceAsyncOperationCompletedHandler* handler) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Completed(
				IWiFiDirectDeviceAsyncOperationCompletedHandler** handler) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetResults(IWiFiDirectDevice** results) = 0;
		};

		MIDL_INTERFACE("de73cba7-370d-550c-b23a-53dd0b4e480d")
		IWiFiDirectAdvertisementPublisherStatusChangedEventHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(IWiFiDirectAdvertisementPublisher* sender,
				IWiFiDirectAdvertisementPublisherStatusChangedEventArgs* e) = 0;
		};

		MIDL_INTERFACE("d04b0403-1fe2-532f-8e47-4823a14e624f")
		IWiFiDirectConnectionRequestedEventHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(IWiFiDirectConnectionListener* sender,
				IWiFiDirectConnectionRequestedEventArgs* e) = 0;
		};

		MIDL_INTERFACE("9208929a-2a3c-50ad-aa08-a0a986edbabe")
		IWiFiDirectDeviceEventHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(IWiFiDirectDevice* sender,
				wclCommon::WinApi::IInspectable* e) = 0;
		};

		MIDL_INTERFACE("E86CB57C-3AAC-4851-A792-482AAF931B04")
		IWiFiDirectDeviceStatics : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetDeviceSelector(wclCommon::WinApi::HSTRING* deviceSelector) = 0;
			virtual HRESULT STDMETHODCALLTYPE FromIdAsync(wclCommon::WinApi::HSTRING deviceId,
				IWiFiDirectDeviceAsyncOperation** asyncOp) = 0;
		};

		MIDL_INTERFACE("1A953E49-B103-437E-9226-AB67971342F9")
		IWiFiDirectDeviceStatics2 : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetDeviceSelector(WiFiDirectDeviceSelectorType type,
				wclCommon::WinApi::HSTRING* result) = 0;
			virtual HRESULT STDMETHODCALLTYPE FromIdAsync(wclCommon::WinApi::HSTRING deviceId,
				IWiFiDirectConnectionParameters* connectionParameters, IWiFiDirectDeviceAsyncOperation** result) = 0;
		};

		MIDL_INTERFACE("DBD02F16-11A5-4E60-8CAA-34772148378A")
		IWiFiDirectInformationElementStatics : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE CreateFromBuffer(wclCommon::WinApi::IBuffer* buffer,
				IWiFiDirectInformationElementVector** result) = 0;
			virtual HRESULT STDMETHODCALLTYPE CreateFromDeviceInformation(
				wclCommon::WinApi::IDeviceInformation* deviceInformation,
				IWiFiDirectInformationElementVector** result) = 0;
		};

		MIDL_INTERFACE("AFFB72D6-76BB-497E-AC8B-DC72838BC309")
		IWiFiDirectInformationElement : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_Oui(wclCommon::WinApi::IBuffer** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_Oui(wclCommon::WinApi::IBuffer* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_OuiType(BYTE* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_OuiType(BYTE value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Value(wclCommon::WinApi::IBuffer** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_Value(wclCommon::WinApi::IBuffer* value) = 0;
		};

		MIDL_INTERFACE("A64FDBBA-F2FD-4567-A91B-F5C2F5321057")
		IWiFiDirectLegacySettings : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_IsEnabled(BOOLEAN* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_IsEnabled(BOOLEAN value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Ssid(wclCommon::WinApi::HSTRING* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_Ssid(wclCommon::WinApi::HSTRING value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Passphrase(wclCommon::WinApi::IPasswordCredential** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_Passphrase(wclCommon::WinApi::IPasswordCredential* value) = 0;
		};

		MIDL_INTERFACE("AB511A2D-2A06-49A1-A584-61435C7905A6")
		IWiFiDirectAdvertisement : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_InformationElements(IWiFiDirectInformationElementVector** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_InformationElements(IWiFiDirectInformationElementVector* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_ListenStateDiscoverability(
				WiFiDirectAdvertisementListenStateDiscoverability* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_ListenStateDiscoverability(
				WiFiDirectAdvertisementListenStateDiscoverability value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_IsAutonomousGroupOwnerEnabled(BOOLEAN* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_IsAutonomousGroupOwnerEnabled(BOOLEAN value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_LegacySettings(IWiFiDirectLegacySettings** value) = 0;
		};

		MIDL_INTERFACE("AAFDE53C-5481-46E6-90DD-32116518F192")
		IWiFiDirectAdvertisementPublisherStatusChangedEventArgs : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_Status(WiFiDirectAdvertisementPublisherStatus* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Error(WiFiDirectError* value) = 0;
		};

		MIDL_INTERFACE("B35A2D1A-9B1F-45D9-925A-694D66DF68EF")
		IWiFiDirectAdvertisementPublisher : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_Advertisement(IWiFiDirectAdvertisement** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Status(WiFiDirectAdvertisementPublisherStatus* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE add_StatusChanged(
				IWiFiDirectAdvertisementPublisherStatusChangedEventHandler* handler,
				wclCommon::WinApi::EventRegistrationToken* token) = 0;
			virtual HRESULT STDMETHODCALLTYPE remove_StatusChanged(
				wclCommon::WinApi::EventRegistrationToken token) = 0;
			virtual HRESULT STDMETHODCALLTYPE Start() = 0;
			virtual HRESULT STDMETHODCALLTYPE Stop() = 0;
		};

		MIDL_INTERFACE("B2E55405-5702-4B16-A02C-BBCD21EF6098")
		IWiFiDirectConnectionParameters : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_GroupOwnerIntent(short* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_GroupOwnerIntent(short value) = 0;
		};

		MIDL_INTERFACE("F99D20BE-D38D-484F-8215-E7B65ABF244C")
		IWiFiDirectConnectionRequestedEventArgs : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetConnectionRequest(
				IWiFiDirectConnectionRequest** result) = 0;
		};

		MIDL_INTERFACE("699C1B0D-8D13-4EE9-B9EC-9C72F8251F7D")
		IWiFiDirectConnectionListener : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE add_ConnectionRequested(
				IWiFiDirectConnectionRequestedEventHandler* handler,
				wclCommon::WinApi::EventRegistrationToken* token) = 0;
			virtual HRESULT STDMETHODCALLTYPE remove_ConnectionRequested(
				wclCommon::WinApi::EventRegistrationToken token) = 0;
		};

		MIDL_INTERFACE("72DEAAA8-72EB-4DAE-8A28-8513355D2777")
		IWiFiDirectDevice : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_ConnectionStatus(WiFiDirectConnectionStatus* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_DeviceId(wclCommon::WinApi::HSTRING* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE add_ConnectionStatusChanged(IWiFiDirectDeviceEventHandler* handler,
				wclCommon::WinApi::EventRegistrationToken* token) = 0;
			virtual HRESULT STDMETHODCALLTYPE remove_ConnectionStatusChanged(
				wclCommon::WinApi::EventRegistrationToken token) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetConnectionEndpointPairs(
				wclCommon::WinApi::IEndpointPairVectorView** value) = 0;
		};

		MIDL_INTERFACE("8EB99605-914F-49C3-A614-D18DC5B19B43")
		IWiFiDirectConnectionRequest : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_DeviceInformation(
				wclCommon::WinApi::IDeviceInformation** value) = 0;
		};

		MIDL_INTERFACE("201940f9-a368-57f4-9ef2-3f64e243e0a4")
		IWiFiDirectConfigurationMethodIterator : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_Current(WiFiDirectConfigurationMethod* current) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_HasCurrent(BOOLEAN* hasCurrent) = 0;
			virtual HRESULT STDMETHODCALLTYPE MoveNext(BOOLEAN* hasCurrent) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetMany(DWORD capacity, WiFiDirectConfigurationMethod** items,
				DWORD* actual) = 0;
		};

		MIDL_INTERFACE("794f12da-2dc6-5277-82dc-b0781610537b")
		IWiFiDirectConfigurationMethodIterable : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE First(IWiFiDirectConfigurationMethodIterator** first) = 0;
		};

		MIDL_INTERFACE("61a32670-04d3-551d-ad66-bd04e9ef5c78")
		IWiFiDirectConfigurationMethodVectorView : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetAt(DWORD index, WiFiDirectConfigurationMethod* item) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Size(DWORD* size) = 0;
			virtual HRESULT STDMETHODCALLTYPE IndexOf(WiFiDirectConfigurationMethod item,
				DWORD* index, BOOLEAN* found) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetMany(DWORD startIndex, DWORD capacity,
				WiFiDirectConfigurationMethod* items, DWORD* actual) = 0;
		};

		MIDL_INTERFACE("9b498bc0-b474-5587-b65c-e600965f8fd0")
		IWiFiDirectConfigurationMethodVector : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetAt(DWORD index, WiFiDirectConfigurationMethod* item) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Size(DWORD* size) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetView(IWiFiDirectConfigurationMethodVectorView** view) = 0;
			virtual HRESULT STDMETHODCALLTYPE IndexOf(WiFiDirectConfigurationMethod item, DWORD* index,
				BOOLEAN* found) = 0;
			virtual HRESULT STDMETHODCALLTYPE SetAt(DWORD index, WiFiDirectConfigurationMethod item) = 0;
			virtual HRESULT STDMETHODCALLTYPE InsertAt(DWORD index, WiFiDirectConfigurationMethod item) = 0;
			virtual HRESULT STDMETHODCALLTYPE RemoveAt(DWORD index) = 0;
			virtual HRESULT STDMETHODCALLTYPE Append(WiFiDirectConfigurationMethod item) = 0;
			virtual HRESULT STDMETHODCALLTYPE RemoveAtEnd() = 0;
			virtual HRESULT STDMETHODCALLTYPE Clear() = 0;
			virtual HRESULT STDMETHODCALLTYPE GetMany(DWORD startIndex, DWORD capacity,
				WiFiDirectConfigurationMethod* items, DWORD* actual) = 0;
			virtual HRESULT STDMETHODCALLTYPE ReplaceAll(DWORD count,
				WiFiDirectConfigurationMethod* value) = 0;
		};

		MIDL_INTERFACE("B759AA46-D816-491B-917A-B40D7DC403A2")
		IWiFiDirectAdvertisement2 : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_SupportedConfigurationMethods(
				IWiFiDirectConfigurationMethodVector** value) = 0;
		};

		MIDL_INTERFACE("AB3B0FBE-AA82-44B4-88C8-E3056B89801D")
		IWiFiDirectConnectionParameters2 : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_PreferenceOrderedConfigurationMethods(
				IWiFiDirectConfigurationMethodVector** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_PreferredPairingProcedure(
				WiFiDirectPairingProcedure* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_PreferredPairingProcedure(
				WiFiDirectPairingProcedure value) = 0;
		};

		MIDL_INTERFACE("598AF493-7642-456F-B9D8-E8A9EB1F401A")
		IWiFiDirectConnectionParametersStatics : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetDevicePairingKinds(WiFiDirectConfigurationMethod configurationMethod,
				wclCommon::WinApi::DevicePairingKinds* result) = 0;
		};

		MIDL_INTERFACE("19889f5e-49ae-5e31-b059-083f9f1532c3")
		IWiFiDirectServiceConfigurationMethodIterator : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_Current(WiFiDirectServiceConfigurationMethod* current) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_HasCurrent(BOOLEAN* hasCurrent) = 0;
			virtual HRESULT STDMETHODCALLTYPE MoveNext(BOOLEAN* hasCurrent) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetMany(DWORD capacity, WiFiDirectServiceConfigurationMethod* items,
				DWORD* actual) = 0;
		};

		MIDL_INTERFACE("d9773b1a-a148-58bf-9c4b-afeac9be3ab4")
		IWiFiDirectServiceConfigurationMethodIterable : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE First(
				IWiFiDirectServiceConfigurationMethodIterator** first) = 0;
		};

		MIDL_INTERFACE("dc710fe1-7f04-515b-8ac1-1c5d3c0d2b28")
		IWiFiDirectServiceConfigurationMethodVectorView : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetAt(DWORD index,
				WiFiDirectServiceConfigurationMethod* item) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Size(DWORD* size) = 0;
			virtual HRESULT STDMETHODCALLTYPE IndexOf(WiFiDirectServiceConfigurationMethod item,
				DWORD* index, BOOLEAN* found) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetMany(DWORD startIndex, DWORD capacity,
				WiFiDirectServiceConfigurationMethod* items, DWORD* actual) = 0;
		};

		MIDL_INTERFACE("f6a6f91c-0579-565d-be07-4538a55690be")
		IWiFiDirectServiceConfigurationMethodVector : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetAt(DWORD index,
				WiFiDirectServiceConfigurationMethod* item) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Size(DWORD* size) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetView(IWiFiDirectServiceConfigurationMethodVectorView** view) = 0;
			virtual HRESULT STDMETHODCALLTYPE IndexOf(WiFiDirectServiceConfigurationMethod item,
				DWORD* index, BOOLEAN* found) = 0;
			virtual HRESULT STDMETHODCALLTYPE SetAt(DWORD index,
				WiFiDirectServiceConfigurationMethod item) = 0;
			virtual HRESULT STDMETHODCALLTYPE InsertAt(DWORD index,
				WiFiDirectServiceConfigurationMethod item) = 0;
			virtual HRESULT STDMETHODCALLTYPE RemoveAt(DWORD index) = 0;
			virtual HRESULT STDMETHODCALLTYPE Append(WiFiDirectServiceConfigurationMethod item) = 0;
			virtual HRESULT STDMETHODCALLTYPE RemoveAtEnd() = 0;
			virtual HRESULT STDMETHODCALLTYPE Clear() = 0;
			virtual HRESULT STDMETHODCALLTYPE GetMany(DWORD startIndex, DWORD capacity,
				WiFiDirectServiceConfigurationMethod* items, DWORD* actual) = 0;
			virtual HRESULT STDMETHODCALLTYPE ReplaceAll(DWORD count,
				WiFiDirectServiceConfigurationMethod* value) = 0;
		};

		MIDL_INTERFACE("f505a3c8-4837-5e0e-8a4d-1e2af5477e5c")
		IWiFiDirectServiceAsyncOperationCompletedHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(IWiFiDirectServiceAsyncOperation* asyncInfo,
				wclCommon::WinApi::AsyncStatus status) = 0;
		};

		MIDL_INTERFACE("c4fa2ae8-4ff7-5aa0-af97-ed85ea66f9ae")
		IWiFiDirectServiceAsyncOperation : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE put_Completed(
				IWiFiDirectServiceAsyncOperationCompletedHandler* handler) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Completed(
				IWiFiDirectServiceAsyncOperationCompletedHandler** handler) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetResults(IWiFiDirectService** results) = 0;
		};

		MIDL_INTERFACE("94cb9568-040a-5186-a3c9-52680ee17984")
		IWiFiDirectServiceProvisioningInfoAsyncOperationCompletedHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(IWiFiDirectServiceProvisioningInfoAsyncOperation* asyncInfo,
				wclCommon::WinApi::AsyncStatus status) = 0;
		};

		MIDL_INTERFACE("d7fa4dc4-4730-506e-bff0-801eb4a831a8")
		IWiFiDirectServiceProvisioningInfoAsyncOperation : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE put_Completed(
				IWiFiDirectServiceProvisioningInfoAsyncOperationCompletedHandler* handler) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Completed(
				IWiFiDirectServiceProvisioningInfoAsyncOperationCompletedHandler** handler) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetResults(
				IWiFiDirectServiceProvisioningInfo** results) = 0;
		};

		MIDL_INTERFACE("b29de711-60b8-59da-8f4d-fc79d8ccd422")
		IWiFiDirectServiceSessionAsyncOperationCompletedHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(IWiFiDirectServiceSessionAsyncOperation* asyncInfo,
				wclCommon::WinApi::AsyncStatus status) = 0;
		};

		MIDL_INTERFACE("c2da4e97-728b-5401-a9d9-3a0185450af2")
		IWiFiDirectServiceSessionAsyncOperation : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE put_Completed(
				IWiFiDirectServiceSessionAsyncOperationCompletedHandler* handler) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Completed(
				IWiFiDirectServiceSessionAsyncOperationCompletedHandler** handler) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetResults(
				IWiFiDirectServiceSession** results) = 0;
		};

		MIDL_INTERFACE("fc3dfc2c-9cfa-5822-ba3f-ff3afb65777e")
		IWiFiDirectServiceSessionDeferredEventHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(IWiFiDirectService* sender,
				IWiFiDirectServiceSessionDeferredEventArgs* e) = 0;
		};

		MIDL_INTERFACE("67fc3121-c1a0-5c23-af58-ecb7f2a7d773")
		IWiFiDirectServiceAdvertiserEventHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(IWiFiDirectServiceAdvertiser* sender,
				wclCommon::WinApi::IInspectable* e) = 0;
		};

		MIDL_INTERFACE("3be2d508-a856-5c09-9998-522597b44b07")
		IWiFiDirectServiceAutoAcceptSessionConnectedEventHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(IWiFiDirectServiceAdvertiser* sender,
				IWiFiDirectServiceAutoAcceptSessionConnectedEventArgs* e) = 0;
		};

		MIDL_INTERFACE("cb98fd74-871d-5730-91fe-81ef947fe78f")
		IWiFiDirectServiceSessionRequestedEventHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(IWiFiDirectServiceAdvertiser* sender,
				IWiFiDirectServiceSessionRequestedEventArgs* e) = 0;
		};

		MIDL_INTERFACE("10c33301-e31c-5cce-b2a0-c1dc2d8d0e13")
		IWiFiDirectServiceSessionEventHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(IWiFiDirectServiceSession* sender,
				wclCommon::WinApi::IInspectable* e) = 0;
		};

		MIDL_INTERFACE("8326a337-3c19-57a7-80ec-cca2ea62ef12")
		IWiFiDirectServiceRemotePortAddedEventHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(IWiFiDirectServiceSession* sender,
				IWiFiDirectServiceRemotePortAddedEventArgs* e) = 0;
		};

		MIDL_INTERFACE("50AABBB8-5F71-45EC-84F1-A1E4FC7879A3")
		IWiFiDirectService : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_RemoteServiceInfo(wclCommon::WinApi::IBuffer** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_SupportedConfigurationMethods(
				IWiFiDirectServiceConfigurationMethodVectorView** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_PreferGroupOwnerMode(BOOLEAN* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_PreferGroupOwnerMode(BOOLEAN value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_SessionInfo(wclCommon::WinApi::IBuffer** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_SessionInfo(wclCommon::WinApi::IBuffer* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_ServiceError(WiFiDirectServiceError* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE add_SessionDeferred(
				IWiFiDirectServiceSessionDeferredEventHandler* handler,
				wclCommon::WinApi::EventRegistrationToken* token) = 0;
			virtual HRESULT STDMETHODCALLTYPE remove_SessionDeferred(
				wclCommon::WinApi::EventRegistrationToken token) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetProvisioningInfoAsync(
				WiFiDirectServiceConfigurationMethod selectedConfigurationMethod,
				IWiFiDirectServiceProvisioningInfoAsyncOperation** result) = 0;
			virtual HRESULT STDMETHODCALLTYPE ConnectAsync(
				IWiFiDirectServiceSessionAsyncOperation** result) = 0;
			virtual HRESULT STDMETHODCALLTYPE ConnectAsyncWithPin(wclCommon::WinApi::HSTRING pin,
				IWiFiDirectServiceSessionAsyncOperation** result) = 0;
		};

		MIDL_INTERFACE("A4AA1EE1-9D8F-4F4F-93EE-7DDEA2E37F46")
		IWiFiDirectServiceAdvertiser : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_ServiceName(wclCommon::WinApi::HSTRING* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_ServiceNamePrefixes(
				wclCommon::WinApi::IHSTRINGVector** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_ServiceInfo(wclCommon::WinApi::IBuffer** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_ServiceInfo(wclCommon::WinApi::IBuffer* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_AutoAcceptSession(BOOLEAN* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_AutoAcceptSession(BOOLEAN value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_PreferGroupOwnerMode(BOOLEAN* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_PreferGroupOwnerMode(BOOLEAN value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_PreferredConfigurationMethods(
				IWiFiDirectServiceConfigurationMethodVector** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_ServiceStatus(WiFiDirectServiceStatus* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_ServiceStatus(WiFiDirectServiceStatus value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_CustomServiceStatusCode(UINT32* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_CustomServiceStatusCode(UINT32 value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_DeferredSessionInfo(wclCommon::WinApi::IBuffer** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_DeferredSessionInfo(wclCommon::WinApi::IBuffer* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_AdvertisementStatus(
				WiFiDirectServiceAdvertisementStatus* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_ServiceError(WiFiDirectServiceError* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE add_SessionRequested(
				IWiFiDirectServiceSessionRequestedEventHandler* handler,
				wclCommon::WinApi::EventRegistrationToken* token) = 0;
			virtual HRESULT STDMETHODCALLTYPE remove_SessionRequested(
				wclCommon::WinApi::EventRegistrationToken token) = 0;
			virtual HRESULT STDMETHODCALLTYPE add_AutoAcceptSessionConnected(
				IWiFiDirectServiceAutoAcceptSessionConnectedEventHandler* handler,
				wclCommon::WinApi::EventRegistrationToken* token) = 0;
			virtual HRESULT STDMETHODCALLTYPE remove_AutoAcceptSessionConnected(
				wclCommon::WinApi::EventRegistrationToken token) = 0;
			virtual HRESULT STDMETHODCALLTYPE add_AdvertisementStatusChanged(
				IWiFiDirectServiceAdvertiserEventHandler* handler,
				wclCommon::WinApi::EventRegistrationToken* token) = 0;
			virtual HRESULT STDMETHODCALLTYPE remove_AdvertisementStatusChanged(
				wclCommon::WinApi::EventRegistrationToken token) = 0;
			virtual HRESULT STDMETHODCALLTYPE ConnectAsync(wclCommon::WinApi::IDeviceInformation* deviceInfo,
				IWiFiDirectServiceSessionAsyncOperation** result) = 0;
			virtual HRESULT STDMETHODCALLTYPE ConnectAsyncWithPin(wclCommon::WinApi::IDeviceInformation* deviceInfo,
				wclCommon::WinApi::HSTRING pin, IWiFiDirectServiceSessionAsyncOperation** result) = 0;
			virtual HRESULT STDMETHODCALLTYPE Start() = 0;
			virtual HRESULT STDMETHODCALLTYPE Stop() = 0;
		};

		MIDL_INTERFACE("3106AC0D-B446-4F13-9F9A-8AE925FEBA2B")
		IWiFiDirectServiceAdvertiserFactory : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE CreateWiFiDirectServiceAdvertiser(
				wclCommon::WinApi::HSTRING serviceName, IWiFiDirectServiceAdvertiser** result) = 0;
		};

		MIDL_INTERFACE("DCD9E01E-83DF-43E5-8F43-CBE8479E84EB")
		IWiFiDirectServiceAutoAcceptSessionConnectedEventArgs : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_Session(
				IWiFiDirectServiceSession** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_SessionInfo(wclCommon::WinApi::IBuffer** value) = 0;
		};

		MIDL_INTERFACE("8BDB7CFE-97D9-45A2-8E99-DB50910FB6A6")
		IWiFiDirectServiceProvisioningInfo : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_SelectedConfigurationMethod(
				WiFiDirectServiceConfigurationMethod* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_IsGroupFormationNeeded(BOOLEAN* value) = 0;
		};

		MIDL_INTERFACE("D4CEBAC1-3FD3-4F0E-B7BD-782906F44411")
		IWiFiDirectServiceRemotePortAddedEventArgs : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_EndpointPairs(wclCommon::WinApi::IEndpointPairVectorView** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Protocol(WiFiDirectServiceIPProtocol* value) = 0;
		};

		MIDL_INTERFACE("81142163-E426-47CB-8640-E1B3588BF26F")
		IWiFiDirectServiceSession : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_ServiceName(wclCommon::WinApi::HSTRING* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Status(WiFiDirectServiceSessionStatus* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_ErrorStatus(WiFiDirectServiceSessionErrorStatus* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_SessionId(UINT32* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_AdvertisementId(UINT32* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_ServiceAddress(wclCommon::WinApi::HSTRING* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_SessionAddress(wclCommon::WinApi::HSTRING* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetConnectionEndpointPairs(
				wclCommon::WinApi::IEndpointPairVectorView** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE add_SessionStatusChanged(
				IWiFiDirectServiceSessionEventHandler* handler,
				wclCommon::WinApi::EventRegistrationToken* token) = 0;
			virtual HRESULT STDMETHODCALLTYPE remove_SessionStatusChanged(
				wclCommon::WinApi::EventRegistrationToken token) = 0;
			virtual HRESULT STDMETHODCALLTYPE AddStreamSocketListenerAsync(
				wclCommon::WinApi::IStreamSocketListener* value, wclCommon::WinApi::IAsyncAction** result) = 0;
			virtual HRESULT STDMETHODCALLTYPE AddDatagramSocketAsync(
				wclCommon::WinApi::IInspectable* value, //value: IDatagramSocket;
				wclCommon::WinApi::IAsyncAction** result) = 0;
			virtual HRESULT STDMETHODCALLTYPE add_RemotePortAdded(
				IWiFiDirectServiceRemotePortAddedEventHandler* handler,
				wclCommon::WinApi::EventRegistrationToken* token) = 0;
			virtual HRESULT STDMETHODCALLTYPE remove_RemotePortAdded(
				wclCommon::WinApi::EventRegistrationToken token) = 0;
		};

		MIDL_INTERFACE("8DFC197F-1201-4F1F-B6F4-5DF1B7B9FB2E")
		IWiFiDirectServiceSessionDeferredEventArgs : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_DeferredSessionInfo(wclCommon::WinApi::IBuffer** value) = 0;
		};

		MIDL_INTERFACE("A0E27C8B-50CB-4A58-9BCF-E472B99FBA04")
		IWiFiDirectServiceSessionRequest : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_DeviceInformation(wclCommon::WinApi::IDeviceInformation** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_ProvisioningInfo(
				IWiFiDirectServiceProvisioningInfo** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_SessionInfo(wclCommon::WinApi::IBuffer** value) = 0;
		};

		MIDL_INTERFACE("74BDCC11-53D6-4999-B4F8-6C8ECC1771E7")
		IWiFiDirectServiceSessionRequestedEventArgs : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetSessionRequest(
				IWiFiDirectServiceSessionRequest** value) = 0;
		};

		MIDL_INTERFACE("7DB40045-FD74-4688-B725-5DCE86ACF233")
		IWiFiDirectServiceStatics : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetSelector(wclCommon::WinApi::HSTRING serviceName,
				wclCommon::WinApi::HSTRING* serviceSelector) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetSelectorWithFilter(wclCommon::WinApi::HSTRING serviceName,
				wclCommon::WinApi::IBuffer* serviceInfoFilter, wclCommon::WinApi::HSTRING* serviceSelector) = 0;
			virtual HRESULT STDMETHODCALLTYPE FromIdAsync(wclCommon::WinApi::HSTRING deviceId,
				IWiFiDirectServiceAsyncOperation** asyncOp) = 0;
		};

		MIDL_INTERFACE("D45A8DA0-0E86-4D98-8BA4-DD70D4B764D3")
		INetworkOperatorTetheringManager : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_MaxClientCount(UINT32* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_ClientCount(UINT32* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_TetheringOperationalState(
				TetheringOperationalState* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetCurrentAccessPointConfiguration(
				INetworkOperatorTetheringAccessPointConfiguration** configuration) = 0;
			virtual HRESULT STDMETHODCALLTYPE ConfigureAccessPointAsync(
				INetworkOperatorTetheringAccessPointConfiguration* configuration,
				wclCommon::WinApi::IAsyncAction** asyncInfo) = 0;
			virtual HRESULT STDMETHODCALLTYPE StartTetheringAsync(
				INetworkOperatorTetheringOperationResultAsyncOperation** asyncInfo) = 0;
			virtual HRESULT STDMETHODCALLTYPE StopTetheringAsync(
				INetworkOperatorTetheringOperationResultAsyncOperation** asyncInfo) = 0;
		};

		MIDL_INTERFACE("0BCC0284-412E-403D-ACC6-B757E34774A4")
		INetworkOperatorTetheringAccessPointConfiguration : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_Ssid(wclCommon::WinApi::HSTRING* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_Ssid(wclCommon::WinApi::HSTRING value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Passphrase(wclCommon::WinApi::HSTRING* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_Passphrase(wclCommon::WinApi::HSTRING value) = 0;
		};

		MIDL_INTERFACE("601b30ed-9b7f-54b6-b61b-24a09bc56304")
		INetworkOperatorTetheringOperationResultAsyncOperation : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE put_Completed(
				INetworkOperatorTetheringOperationResultAsyncOperationCompletedHandler* handler) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Completed(
				INetworkOperatorTetheringOperationResultAsyncOperationCompletedHandler** handler) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetResults(
				INetworkOperatorTetheringOperationResult** results) = 0;
		};

		MIDL_INTERFACE("a936b927-7537-59c6-89de-33f36a9725ab")
		INetworkOperatorTetheringOperationResultAsyncOperationCompletedHandler : public IUnknown
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE Invoke(
				INetworkOperatorTetheringOperationResultAsyncOperation* asyncInfo,
				wclCommon::WinApi::AsyncStatus status) = 0;
		};

		MIDL_INTERFACE("EBD203A1-01BA-476D-B4B3-BF3D12C8F80C")
		INetworkOperatorTetheringOperationResult : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_Status(TetheringOperationStatus* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_AdditionalErrorMessage(
				wclCommon::WinApi::HSTRING** value) = 0;
		};

		MIDL_INTERFACE("3EBCBACC-F8C3-405C-9964-70A1EEABE194")
		INetworkOperatorTetheringManagerStatics : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetTetheringCapability(
				wclCommon::WinApi::HSTRING networkAccountId,
				TetheringCapability* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE CreateFromNetworkAccountId(
				wclCommon::WinApi::HSTRING networkAccountId,
				INetworkOperatorTetheringManager** ppManager) = 0;
		};

		MIDL_INTERFACE("5B235412-35F0-49E7-9B08-16D278FBAA42")
		INetworkOperatorTetheringManagerStatics2 : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetTetheringCapabilityFromConnectionProfile(
				wclCommon::WinApi::IConnectionProfile* profile, TetheringCapability* result) = 0;
			virtual HRESULT STDMETHODCALLTYPE CreateFromConnectionProfile(
				wclCommon::WinApi::IConnectionProfile* profile, INetworkOperatorTetheringManager** ppManager) = 0;
		};

		MIDL_INTERFACE("8FDAADB6-4AF9-4F21-9B58-D53E9F24231E")
		INetworkOperatorTetheringManagerStatics3 : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE CreateFromConnectionProfileWithTargetAdapter(
				wclCommon::WinApi::IConnectionProfile* profile,
				wclCommon::WinApi::INetworkAdapter* adapter,
				INetworkOperatorTetheringManager** ppManager) = 0;
		};

		MIDL_INTERFACE("b3b9f9d0-ebff-46a4-a847-d663d8b0977e")
		INetworkOperatorTetheringManagerStatics4 : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE IsNoConnectionsTimeoutEnabled(
				BOOLEAN* result) = 0;
			virtual HRESULT STDMETHODCALLTYPE EnableNoConnectionsTimeout() = 0;
			virtual HRESULT STDMETHODCALLTYPE EnableNoConnectionsTimeoutAsync(
				wclCommon::WinApi::IAsyncAction** operation) = 0;
			virtual HRESULT STDMETHODCALLTYPE DisableNoConnectionsTimeout() = 0;
			virtual HRESULT STDMETHODCALLTYPE DisableNoConnectionsTimeoutAsync(
				wclCommon::WinApi::IAsyncAction** operation) = 0;
		};

		MIDL_INTERFACE("5074F851-950D-4165-9C15-365619481EEA")
		INetworkInformationStatics : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetConnectionProfiles(void** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetInternetConnectionProfile(
				wclCommon::WinApi::IConnectionProfile** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetLanIdentifiers(void** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetHostNames(void** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetProxyConfigurationAsync(void* uri,
				void** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetSortedEndpointPairs(void* destinationList,
				UINT32 sortOptions, void** value) = 0;
			virtual HRESULT STDMETHODCALLTYPE add_NetworkStatusChanged(void* networkStatusHandler,
				wclCommon::WinApi::EventRegistrationToken* eventCookie) = 0;
			virtual HRESULT STDMETHODCALLTYPE remove_NetworkStatusChanged(
				wclCommon::WinApi::EventRegistrationToken eventCookie) = 0;
		};

		MIDL_INTERFACE("e21d6797-4ef6-553f-b7f9-5ed74bcebd7e")
		INetworkOperatorTetheringClientVectorView : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetAt(DWORD index,
				INetworkOperatorTetheringClient** item) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Size(DWORD* size) = 0;
			virtual HRESULT STDMETHODCALLTYPE IndexOf(INetworkOperatorTetheringClient* item,
				DWORD* index, BOOLEAN* found) = 0;
			virtual HRESULT STDMETHODCALLTYPE GetMany(DWORD startIndex, DWORD capacity,
				INetworkOperatorTetheringClient** titems, DWORD* actual) = 0;
		};

		MIDL_INTERFACE("709D254C-595F-4847-BB30-646935542918")
		INetworkOperatorTetheringClient : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE get_MacAddress(wclCommon::WinApi::HSTRING* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_HostNames(wclCommon::WinApi::IHostNameVectorView** value) = 0;
		};

		MIDL_INTERFACE("91B14016-8DCA-4225-BBED-EEF8B8D718D7")
		INetworkOperatorTetheringClientManager : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE GetTetheringClients(
				INetworkOperatorTetheringClientVectorView** value) = 0;
		};

		MIDL_INTERFACE("b1809142-7238-59a0-928b-74ab46fd64b6")
		INetworkOperatorTetheringAccessPointConfiguration2 : public wclCommon::WinApi::IInspectable
		{
		public:
			virtual HRESULT STDMETHODCALLTYPE IsBandSupported(TetheringWiFiBand band,
				BOOLEAN* result) = 0;
			virtual HRESULT STDMETHODCALLTYPE IsBandSupportedAsync(TetheringWiFiBand band,
				wclCommon::WinApi::IBooleanAsyncOperation** operation) = 0;
			virtual HRESULT STDMETHODCALLTYPE get_Band(TetheringWiFiBand* value) = 0;
			virtual HRESULT STDMETHODCALLTYPE put_Band(TetheringWiFiBand value) = 0;
		};

		/* WiFi subsystem control functions declarations. */

		/// <summary> Loads the WLAN subsystem. </summary>
		/// <returns> If the function completed with success the return value is
		///   <see cref="WCL_E_SUCCESS" />. Otherwise the function returns one of the
		///   WCL error codes. </returns>
		/// <remarks> An application must not call this function. Instead it must use
		///   high-level WiFi functions and classes. </remarks>
		int wclLoadWlan();

		/// <summary> Unloads the WLAN subsystem. </summary>
		/// <returns> If the function completed with success the return value is
		///   <see cref="WCL_E_SUCCESS" />. Otherwise the function returns one of the
		///   WCL error codes. </returns>
		/// <remarks> An application must not call this function. Instead it must use
		///   high-level WiFi functions and classes. </remarks>
		int wclUnloadWlan();
	}
}