/**
 * @module select
 */

modules.define(
    'select',
    ['i-bem-dom', 'popup', 'menu', 'menu__item', 'button', 'jquery', 'dom', 'keyboard__codes', 'strings__escape'],
    function(provide, bemDom, Popup, Menu, MenuItem, Button, $, dom, keyCodes, escape) {

/**
 * @exports
 * @class select
 * @bem
 *
 * @bemmod opened Represents opened state
 */
provide(bemDom.declBlock(this.name, /** @lends select.prototype */{
    beforeSetMod : {
        'opened' : {
            'true' : function() {
                return !this.hasMod('disabled');
            }
        },

        'focused' : {
            '' : function() {
                return !this._isPointerPressInProgress;
            }
        }
    },

    onSetMod : {
        'js' : {
            'inited' : function() {
                this._button = this.findChildBlock(Button);
                this._events(Button).on('click', this._onButtonClick, this);

                this._popup = this.findChildBlock(Popup)
                    .setAnchor(this._button);
                this._events(Popup).on({ modName : 'visible', modVal : '' }, this._onPopupHide, this);

                this._menu = this._popup.findChildBlock(Menu);
                this._events(this._menu)
                    .on('change', this._onMenuChange, this)
                    .on('item-click', this._onMenuItemClick, this)
                    .on('item-hover', this._onMenuItemHover, this);

                this._isPointerPressInProgress = false;
                this._buttonWidth = null;

                this.hasMod('focused') && this._focus();
            }
        },

        'focused' : {
            'true' : function() {
                this._focus();
            },

            '' : function() {
                this._blur();
            }
        },

        'opened' : {
            '*' : function(_, modVal) {
                this._menu.setMod('focused', modVal);
            },

            'true' : function() {
                this._buttonWidth === null && this._updateMenuWidth();

                this._updateMenuHeight();
                this._popup.setMod('visible');
                this._domEvents(bemDom.doc).on('pointerpress', this._onDocPointerPress);
                this.setMod('focused')
                    ._hoverCheckedOrFirstItem();
            },

            '' : function() {
                this._domEvents(bemDom.doc).un('pointerpress', this._onDocPointerPress);
                this._popup.delMod('visible');
            }
        },

        'disabled' : {
            '*' : function(modName, modVal) {
                this._button.setMod(modName, modVal);
                this._menu.setMod(modName, modVal);
            },

            'true' : function() {
                this._elems('control').forEach(function(control) {
                    control.domElem.attr('disabled', true);
                });
                this._popup.delMod('visible');
            },

            '' : function() {
                this._elems('control').forEach(function(control) {
                    control.domElem.removeAttr('disabled');
                });
            }
        }
    },

    /**
     * Get value
     * @returns {*}
     */
    getVal : function() {
        return this._menu.getVal();
    },

    /**
     * Set value
     * @param {*} val
     * @returns {select} this
     */
    setVal : function(val) {
        this._menu.setVal(val);
        return this;
    },

    /**
     * Get name
     * @returns {String}
     */
    getName : function() {
        return this.params.name;
    },

    _getDefaultParams : function() {
        return {
            optionsMaxHeight : Number.POSITIVE_INFINITY
        };
    },

    _focus : function() {
        this._domEvents('button')
            .on('keydown', this._onKeyDown)
            .on('keypress', this._onKeyPress);

        this._button.setMod('focused');
    },

    _blur : function() {
        this._domEvents('button')
            .un('keydown', this._onKeyDown)
            .un('keypress', this._onKeyPress);

        this.delMod('opened');
        this._button.delMod('focused');
    },

    _updateMenuWidth : function() {
        this._menu.domElem.css('min-width', this._buttonWidth = this._button.domElem.outerWidth());

        this._popup.redraw();
    },

    _updateMenuHeight : function() {
        var drawingParams = this._popup.calcPossibleDrawingParams(),
            menuDomElem = this._menu.domElem,
            menuWidth = menuDomElem.outerWidth(),
            bestHeight = 0;

        drawingParams.forEach(function(params) {
            params.width >= menuWidth && params.height > bestHeight && (bestHeight = params.height);
        });

        bestHeight && menuDomElem.css('max-height', Math.min(this.params.optionsMaxHeight, bestHeight));
    },

    _getCheckedItems : function() {
        return this._menu.getItems().filter(function(item) {
            return item.hasMod('checked');
        });
    },

    _hoverCheckedOrFirstItem : function() { // NOTE: may be it should be moved to menu
        (this._getCheckedItems().get(0) || this._menu.getItems().get(0))
            .setMod('hovered');
    },

    _onKeyDown : function(e) {
        if(this.hasMod('opened')) {
            if(e.keyCode === keyCodes.ESC) {
                // NOTE: stop propagation to prevent from being listened by global handlers
                e.stopPropagation();
                this.delMod('opened');
            }
        } else if((e.keyCode === keyCodes.UP || e.keyCode === keyCodes.DOWN) && !e.shiftKey) {
            e.preventDefault();
            this.setMod('opened');
        }
    },

    _onKeyPress : function(e) {
        // press a key: closed select - set value, opened select - set hover on menu-item.
        if(!this.hasMod('opened')) {
            var item = this._menu.searchItemByKeyboardEvent(e);
            item && this._setSingleVal(item.getVal());
        }
    },

    _setSingleVal : function(value) {
        this.setVal(value);
    },

    _onMenuChange : function() {
        this._updateControl();
        this._updateButton();

        this.hasMod('opened')?
            this._updateMenuWidth() :
            this._buttonWidth = null;

        this._emit('change');
    },

    _onMenuItemClick : function() {},

    _onMenuItemHover : function(e, data) {
        var item = data.item;
        item.hasMod('hovered')?
            this._button.domElem.attr('aria-activedescendant', item.domElem.attr('id')) :
            this._button.domElem.removeAttr('aria-activedescendant');
    },

    _updateControl : function() {},

    _updateButton : function() {},

    _onButtonClick : function() {
        this.toggleMod('opened');
    },

    _onButtonFocusChange : function(e, data) {
        this.setMod('focused', data.modVal);
    },

    _onPopupHide : function() {
        this.delMod('opened');
    },

    _onDocPointerPress : function(e) {
        if(this._isEventInPopup(e)) {
            e.pointerType === 'mouse' && e.preventDefault(); // prevents button blur in most desktop browsers
            this._isPointerPressInProgress = true;
            this._domEvents(bemDom.doc).on(
                'pointerrelease',
                { focusedHardMod : this._button.getMod('focused-hard') },
                this._onDocPointerRelease);
        }
    },

    _onDocPointerRelease : function(e) {
        this._isPointerPressInProgress = false;
        this._domEvents().un('pointerrelease', this._onDocPointerRelease);
        this._button
            .toggleMod('focused', true, '', this._isEventInPopup(e))
            .setMod('focused-hard', e.data.focusedHardMod);
    },

    _isEventInPopup : function(e) {
        return dom.contains(this._popup.domElem, $(e.target));
    }
}, /** @lends select */{
    lazyInit : true,
    onInit : function() {
        this._events(Button).on(
            { modName : 'focused', modVal : '*' },
            this.prototype._onButtonFocusChange);
    },

    _createControlHTML : function(name, val) {
        // Using string concatenation to not depend on template engines
        return '<input ' +
            'type="hidden" ' +
            'name="' + name + '" ' +
            'class="' + this._buildClassName('control') + '" ' +
            'value="' + escape.attr(val) + '"/>';
    }
}));

});
