(*
 *******************************************************************************
 * PasLibVlcUnit.pas - pascal interface for VideoLAN libvlc 2.2.0
 *
 * See copyright notice below.
 *
 * Last modified: 2015.03.08
 *
 * author: Robert Jdrzejczyk
 * e-mail: robert@prog.olsztyn.pl
 *    www: http://prog.olsztyn.pl/paslibvlc
 *
 *
 * See PasLibVlcUnit.txt for change log
 *
 *******************************************************************************
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Any non-GPL usage of this software or parts of this software is strictly
 * forbidden.
 *
 * The "appropriate copyright message" mentioned in section 2c of the GPLv2
 * must read: "Code from FAAD2 is copyright (c) Nero AG, www.nero.com"
 *
 * Commercial non-GPL licensing of this software is possible.
 * please contact robert@prog.olsztyn.pl
 *
 *******************************************************************************
 *
 * libvlc is part of project VideoLAN
 *
 * Copyright (c) 1996-2010 VideoLAN Team
 *
 * For more information about VideoLAN
 *
 * please visit http://www.videolan.org
 *
 *)

{$I compiler.inc}

unit PasLibVlcUnit;

interface

uses
  {$IFDEF UNIX}Unix, dynlibs,{$ENDIF}
  {$IFDEF MSWINDOWS}Windows,{$ENDIF}
  {$IFDEF LCLCARBON}CarbonPrivate, CarbonDef,{$ENDIF}
  {$IFDEF LCLGTK2}Gtk2, {$IFDEF UNIX}Gdk2x,{$ENDIF}{$ENDIF}
  {$IFDEF LCLQT}Qt4, QtWidgets,{$ENDIF}
  {$IFDEF UNIX}Dialogs,{$ENDIF}
  {$IFDEF MACOS}FMX.Dialogs,{$ENDIF}
  SysUtils;

(**
 * Real path to libvlc.dll
 *)

var
  libvlc_dynamic_dll_path: string;

(**
 * Real library file name
 *)
var
  libvlc_dynamic_dll_file: string;

(**
 * Current library version
 *)
var
  libvlc_dynamic_dll_version: string;

var
  libvlc_dynamic_dll_version_bin : LongWord;

(**
 * Last error message generated by LibVlc_Init() procedure
 *)
 
var
  libvlc_dynamic_dll_error: string;

(**
 * help read version nunber from string returned by
 * 
 *)
function read_dec_number(var ptr : PAnsiChar) : LongWord;

(**
 * help set function pointers to NIL
 * 
 *)
procedure libvlc_reset_function_pointers(force : boolean = FALSE);

(**
 * Dynamic loading libvlc.dll
 * Must be called before use any libvlc function
 *)

procedure libvlc_dynamic_dll_init();
procedure libvlc_dynamic_dll_init_with_path(vlc_install_path: string);

(**
 * Release libvlc.dll
 *)
 
procedure libvlc_dynamic_dll_done();

{$IFNDEF HAS_PATH_DELIM}
const
  PathDelim = '\';
{$ENDIF}

{$IFNDEF HAS_PPANSICHAR}
type
  PPAnsiChar = ^PAnsiChar;
{$ENDIF}

(*
 *******************************************************************************
 * type s and structures used in libvlc
 *******************************************************************************
 *)

type
  libvlc_instance_t_ptr          = type Pointer;
  libvlc_log_t_ptr               = type Pointer;
  libvlc_log_iterator_t_ptr      = type Pointer;
  libvlc_media_t_ptr             = type Pointer;
  libvlc_media_player_t_ptr      = type Pointer;
  libvlc_media_list_t_ptr        = type Pointer;
  libvlc_media_list_player_t_ptr = type Pointer;
  libvlc_media_library_t_ptr     = type Pointer;
  libvlc_media_discoverer_t_ptr  = type Pointer;
  libvlc_event_manager_t_ptr     = type Pointer;

type
  libvlc_media_open_cb = function(opaque: Pointer; var datap: Pointer;
    var sizep: Int64): Integer; cdecl;

  libvlc_media_read_cb = function(opaque: Pointer; buf: PByte; len: Integer): Integer; cdecl;

  libvlc_media_seek_cb = function(opaque: Pointer; offset: Int64): Integer; cdecl;

  libvlc_media_close_cb =function(opaque: Pointer): Integer; cdecl;


type
  libvlc_time_t_ptr = ^libvlc_time_t;
  libvlc_time_t = Int64;

type
  libvlc_log_message_t_ptr = ^libvlc_log_message_t;
  libvlc_log_message_t = packed record
    i_severity  : Integer;   // 0=INFO, 1=ERR, 2=WARN, 3=DBG
    psz_type    : PAnsiChar; // module type
    psz_name    : PAnsiChar; // module name
    psz_header  : PAnsiChar; // optional header
    psz_message : PAnsiChar; // message
  end;

type
  libvlc_meta_t = (
    libvlc_meta_Title,
    libvlc_meta_Artist,
    libvlc_meta_Genre,
    libvlc_meta_Copyright,
    libvlc_meta_Album,
    libvlc_meta_TrackNumber,
    libvlc_meta_Description,
    libvlc_meta_Rating,
    libvlc_meta_Date,
    libvlc_meta_Setting,
    libvlc_meta_URL,
    libvlc_meta_Language,
    libvlc_meta_NowPlaying,
    libvlc_meta_Publisher,
    libvlc_meta_EncodedBy,
    libvlc_meta_ArtworkURL,
    libvlc_meta_TrackID,
    // from VLC 2.2.0
    libvlc_meta_TrackTotal,
    libvlc_meta_Director,
    libvlc_meta_Season,
    libvlc_meta_Episode,
    libvlc_meta_ShowName,
    libvlc_meta_Actors
    // Add new meta types HERE
);

type
  libvlc_state_t = (
    libvlc_NothingSpecial, // 0,
    libvlc_Opening,        // 1,
    libvlc_Buffering,      // 2,
    libvlc_Playing,        // 3,
    libvlc_Paused,         // 4,
    libvlc_Stopped,        // 5,
    libvlc_Ended,          // 6,
    libvlc_Error           // 7
);

const
  libvlc_media_option_trusted = $0002;
  libvlc_media_option_unique  = $0100;

{$IFDEF HAS_ENUM_ORDINALITY}
type
  libvlc_track_type_t = (
    libvlc_track_unknown = -1,
    libvlc_track_audio   = 0,
    libvlc_track_video   = 1,
    libvlc_track_text    = 2
  );
{$ELSE}
type
  libvlc_track_type_t = -1..2;
const
  libvlc_track_unknown = -1;
  libvlc_track_audio   = 0;
  libvlc_track_video   = 1;
  libvlc_track_text    = 2;
{$ENDIF}

type
  libvlc_media_stats_t_ptr = ^libvlc_media_stats_t;
  libvlc_media_stats_t = packed record
    // Input
    i_read_bytes          : Integer;
    f_input_bitrate       : Single; // Float
    // Demux
    i_demux_read_bytes    : Integer;
    f_demux_bitrate       : Single;
    i_demux_corrupted     : Integer;
    i_demux_discontinuity : Integer;
    // Decoders
    i_decoded_video       : Integer;
    i_decoded_audio       : Integer;
    // Video Output
    i_displayed_pictures  : Integer;
    i_lost_pictures       : Integer;
    // Audio output
    i_played_abuffers     : Integer;
    i_lost_abuffers       : Integer;
    // Stream output
    i_sent_packets        : Integer;
    i_sent_bytes          : Integer;
    f_send_bitrate        : Single;
  end;

type
  libvlc_media_track_info_t_audio = packed record
    i_channels : LongWord;
    i_rate     : LongWord
  end;

type
  libvlc_media_track_info_t_video = packed record
    i_height : LongWord;
    i_width  : LongWord;
  end;

type
  libvlc_media_track_info_t_ptr = ^libvlc_media_track_info_t;
  libvlc_media_track_info_t = packed record
    // Codec fourcc
    i_codec : LongWord;
    i_id    : Integer;
    i_type  : libvlc_track_type_t;

    // Codec specific
    i_profile : Integer;
    i_level   : Integer;

    case Byte of
      0: (audio : libvlc_media_track_info_t_audio);
      1: (video : libvlc_media_track_info_t_video);
  end;

type
  libvlc_audio_track_t_ptr = ^libvlc_audio_track_t;
  libvlc_audio_track_t = packed record
    i_channels : LongWord;
    i_rate     : LongWord;
  end;

type
  libvlc_video_track_t_ptr = ^libvlc_video_track_t;
  libvlc_video_track_t = packed record
    i_height         : LongWord;
    i_width          : LongWord;
    i_sar_num        : LongWord;
    i_sar_den        : LongWord;
    i_frame_rate_num : LongWord;
    i_frame_rate_den : LongWord;
  end;

type
  libvlc_subtitle_track_t_ptr = ^libvlc_subtitle_track_t;
  libvlc_subtitle_track_t = packed record
    psz_encoding : PAnsiChar;
  end;

type
  libvlc_media_track_t_union = packed record
    case Byte of
      0: (audio : libvlc_audio_track_t);
      1: (video : libvlc_video_track_t);
      2: (subtitle : libvlc_subtitle_track_t);
  end;

type
  libvlc_media_track_t_ptr = ^libvlc_media_track_t;
  libvlc_media_track_t = packed record

    // Codec fourcc
    i_codec           : LongWord;
    i_original_fourcc : LongWord;
    i_id              : Integer;
    i_type            : libvlc_track_type_t;

    // Codec specific
    i_profile : Integer;
    i_level   : Integer;

    // union
    u : libvlc_media_track_t_union;

    i_bitrate : LongWord;
    psz_language : PAnsiChar;
    psz_description : PAnsiChar
  end;

(**
 * Description for video, audio tracks and subtitles. It contains
 * id, name (description string) and pointer to next record.
 *)
type
  libvlc_track_description_t_ptr = ^libvlc_track_description_t;
  libvlc_track_description_t = packed record
    i_id     : Integer;
    psz_name : PAnsiChar;
    p_next   : libvlc_track_description_t_ptr;
 end;

(**
 * Description for audio output. It contains
 * name, description and pointer to next record.
 *)
type
  libvlc_audio_output_t_ptr = ^libvlc_audio_output_t;
  libvlc_audio_output_t = packed record
    psz_name        : PAnsiChar;
    psz_description : PAnsiChar;
    p_next          : libvlc_audio_output_t_ptr;
  end;

(**
 * Description for audio output device.
 *)
type
  libvlc_audio_output_device_t_ptr = ^libvlc_audio_output_device_t;
  libvlc_audio_output_device_t = packed record
    p_next          : libvlc_audio_output_device_t_ptr; // Next entry in list
    psz_device      : PAnsiChar;                        // Device identifier string
    psz_description : PAnsiChar;                        // User-friendly device description
    // More fields may be added here in later versions
  end;

type
  libvlc_equalizer_t_ptr = Pointer;
  unsigned_t = Word;

(**
 * Rectangle type for video geometry
 *)
type
  libvlc_rectangle_t_ptr = ^libvlc_rectangle_t;
  libvlc_rectangle_t = packed record
    top    : Integer;
    left   : Integer;
    bottom : Integer;
    right  : Integer;
  end;

type
  libvlc_opacity_t = 0..255;

const
  libvlc_opacity_None   = 0;
  libvlc_opacity_Medium = 128;
  libvlc_opacity_Full   = 255;

(**
 * Marq options definition
 *)
type
  libvlc_video_marquee_option_t = (
    libvlc_marquee_Enable,   // 0,
    libvlc_marquee_Text,     // 1,
    libvlc_marquee_Color,    // 2,
    libvlc_marquee_Opacity,  // 3,
    libvlc_marquee_Position, // 4,
    libvlc_marquee_Refresh,  // 5,
    libvlc_marquee_Size,     // 6,
    libvlc_marquee_Timeout,  // 7,
    libvlc_marquee_X,        // 8,
    libvlc_marquee_Y         // 9
  );

const
  libvlc_video_marquee_color_Default = $f0000000;
  libvlc_video_marquee_color_Black   = $00000000;
  libvlc_video_marquee_color_Gray    = $00808080;
  libvlc_video_marquee_color_Silver  = $00C0C0C0;
  libvlc_video_marquee_color_White   = $00FFFFFF;
  libvlc_video_marquee_color_Maroon  = $00800000;
  libvlc_video_marquee_color_Red     = $00FF0000;
  libvlc_video_marquee_color_Fuchsia = $00FF00FF;
  libvlc_video_marquee_color_Yellow  = $00FFFF00;
  libvlc_video_marquee_color_Olive   = $00808000;
  libvlc_video_marquee_color_Green   = $00008000;
  libvlc_video_marquee_color_Teal    = $00008080;
  libvlc_video_marquee_color_Lime    = $0000FF00;
  libvlc_video_marquee_color_Purple  = $00800080;
  libvlc_video_marquee_color_Navy    = $00000080;
  libvlc_video_marquee_color_Blue    = $000000FF;
  libvlc_video_marquee_color_Aqua    = $0000FFFF;
type
  libvlc_video_marquee_color_t = libvlc_video_marquee_color_Black..libvlc_video_marquee_color_White;

const
  libvlc_video_marquee_default_font_size = 12;

type
  libvlc_video_logo_option_t = (
    libvlc_logo_Enable,   // 0,
    libvlc_logo_File,     // 1,
    libvlc_logo_X,        // 2,
    libvlc_logo_Y,        // 3,
    libvlc_logo_Delay,    // 4,
    libvlc_logo_Repeat,   // 5,
    libvlc_logo_Opacity,  // 6,
    libvlc_logo_Position  // 7
 );

(**
 * Audio device types
 *)

{$IFDEF HAS_ENUM_ORDINALITY}
type
  libvlc_audio_output_device_types_t = (
    libvlc_AudioOutputDevice_Error  = -1,
    libvlc_AudioOutputDevice_Mono   =  1,
    libvlc_AudioOutputDevice_Stereo =  2,
    libvlc_AudioOutputDevice_2F2R   =  4,
    libvlc_AudioOutputDevice_3F2R   =  5,
    libvlc_AudioOutputDevice_5_1    =  6,
    libvlc_AudioOutputDevice_6_1    =  7,
    libvlc_AudioOutputDevice_7_1    =  8,
    libvlc_AudioOutputDevice_SPDIF  = 10);
{$ELSE}
type
  libvlc_audio_output_device_types_t = -1..10;
const
  libvlc_AudioOutputDevice_Error  = -1;
  libvlc_AudioOutputDevice_Mono   =  1;
  libvlc_AudioOutputDevice_Stereo =  2;
  libvlc_AudioOutputDevice_2F2R   =  4;
  libvlc_AudioOutputDevice_3F2R   =  5;
  libvlc_AudioOutputDevice_5_1    =  6;
  libvlc_AudioOutputDevice_6_1    =  7;
  libvlc_AudioOutputDevice_7_1    =  8;
  libvlc_AudioOutputDevice_SPDIF  = 10;
{$ENDIF}

{$IFDEF HAS_ENUM_ORDINALITY}
type
  libvlc_audio_output_channel_t = (
    libvlc_AudioChannel_Error    = -1,
    libvlc_AudioChannel_NotAvail =  0,
    libvlc_AudioChannel_Stereo   =  1,
    libvlc_AudioChannel_RStereo  =  2,
    libvlc_AudioChannel_Left     =  3,
    libvlc_AudioChannel_Right    =  4,
    libvlc_AudioChannel_Dolbys   =  5
  );
{$ELSE}
type
  libvlc_audio_output_channel_t = -1..5;
const
  libvlc_AudioChannel_Error   = -1;
  libvlc_AudioChannel_Stereo  =  1;
  libvlc_AudioChannel_RStereo =  2;
  libvlc_AudioChannel_Left    =  3;
  libvlc_AudioChannel_Right   =  4;
  libvlc_AudioChannel_Dolbys  =  5;
{$ENDIF}

type
  libvlc_playback_mode_t = (
    libvlc_playback_mode_default,
    libvlc_playback_mode_loop,
    libvlc_playback_mode_repeat
  );

(**
 * Navigation mode
 *)

type
  libvlc_navigate_mode_t = (
    libvlc_navigate_activate,
    libvlc_navigate_up,
    libvlc_navigate_down,
    libvlc_navigate_left,
    libvlc_navigate_right
  );

(**
 * Enumeration of values used to set position (e.g. of video title).
 *)
{$IFDEF HAS_ENUM_ORDINALITY}
type
  libvlc_position_t = (
    libvlc_position_disable      = -1,
    libvlc_position_center       = 0,
    libvlc_position_left         = 1,
    libvlc_position_right        = 2,
    libvlc_position_top          = 4,
    libvlc_position_top_left     = 5,
    libvlc_position_top_right    = 6,
    libvlc_position_bottom       = 8,
    libvlc_position_bottom_left  = 9,
    libvlc_position_bottom_right = 10
  );
{$ELSE}
const
  libvlc_position_disable      = -1;
  libvlc_position_center       = 0;
  libvlc_position_left         = 1;
  libvlc_position_right        = 2;
  libvlc_position_top          = 4;
  libvlc_position_top_left     = 5;
  libvlc_position_top_right    = 6;
  libvlc_position_bottom       = 8;
  libvlc_position_bottom_left  = 9;
  libvlc_position_bottom_right = 10;
type
  libvlc_position_t = libvlc_position_disable..libvlc_position_bottom_right;
{$ENDIF}

(*
 ******************************************************************************
 * libvlc_event.h
 *******************************************************************************
 *)

{$IFDEF HAS_ENUM_ORDINALITY}
type
  libvlc_event_type_t = (
    (* Append new event types at the end of a category.
     * Do not remove, insert or re-order any entry.
     * Keep this in sync with src/control/event.c:libvlc_event_type_name(). *)
    libvlc_MediaMetaChanged = 0,
    libvlc_MediaSubItemAdded,
    libvlc_MediaDurationChanged,
    libvlc_MediaParsedChanged,
    libvlc_MediaFreed,
    libvlc_MediaStateChanged,
    libvlc_MediaSubItemTreeAdded,

    libvlc_MediaPlayerMediaChanged = $100,
    libvlc_MediaPlayerNothingSpecial,
    libvlc_MediaPlayerOpening,
    libvlc_MediaPlayerBuffering,
    libvlc_MediaPlayerPlaying,
    libvlc_MediaPlayerPaused,
    libvlc_MediaPlayerStopped,
    libvlc_MediaPlayerForward,
    libvlc_MediaPlayerBackward,
    libvlc_MediaPlayerEndReached,
    libvlc_MediaPlayerEncounteredError,
    libvlc_MediaPlayerTimeChanged,
    libvlc_MediaPlayerPositionChanged,
    libvlc_MediaPlayerSeekableChanged,
    libvlc_MediaPlayerPausableChanged,
    libvlc_MediaPlayerTitleChanged,
    libvlc_MediaPlayerSnapshotTaken,
    libvlc_MediaPlayerLengthChanged,
    libvlc_MediaPlayerVout,
    libvlc_MediaPlayerScrambledChanged,

    libvlc_MediaListItemAdded = $200,
    libvlc_MediaListWillAddItem,
    libvlc_MediaListItemDeleted,
    libvlc_MediaListWillDeleteItem,

    libvlc_MediaListViewItemAdded = $300,
    libvlc_MediaListViewWillAddItem,
    libvlc_MediaListViewItemDeleted,
    libvlc_MediaListViewWillDeleteItem,

    libvlc_MediaListPlayerPlayed = $400,
    libvlc_MediaListPlayerNextItemSet,
    libvlc_MediaListPlayerStopped,

    libvlc_MediaDiscovererStarted = $500,
    libvlc_MediaDiscovererEnded,

    libvlc_VlmMediaAdded = $600,
    libvlc_VlmMediaRemoved,
    libvlc_VlmMediaChanged,
    libvlc_VlmMediaInstanceStarted,
    libvlc_VlmMediaInstanceStopped,
    libvlc_VlmMediaInstanceStatusInit,
    libvlc_VlmMediaInstanceStatusOpening,
    libvlc_VlmMediaInstanceStatusPlaying,
    libvlc_VlmMediaInstanceStatusPause,
    libvlc_VlmMediaInstanceStatusEnd,
    libvlc_VlmMediaInstanceStatusError
  );
{$ELSE}
type
  libvlc_event_type_t = type integer;
const
  libvlc_MediaMetaChanged              = 0;
  libvlc_MediaSubItemAdded             = 1;
  libvlc_MediaDurationChanged          = 2;
  libvlc_MediaParsedChanged            = 3;
  libvlc_MediaFreed                    = 4;
  libvlc_MediaStateChanged             = 5;
  libvlc_MediaSubItemTreeAdded         = 6;

  libvlc_MediaPlayerMediaChanged       = $100;
  libvlc_MediaPlayerNothingSpecial     = $101;
  libvlc_MediaPlayerOpening            = $102;
  libvlc_MediaPlayerBuffering          = $103;
  libvlc_MediaPlayerPlaying            = $104;
  libvlc_MediaPlayerPaused             = $105;
  libvlc_MediaPlayerStopped            = $106;
  libvlc_MediaPlayerForward            = $107;
  libvlc_MediaPlayerBackward           = $108;
  libvlc_MediaPlayerEndReached         = $109;
  libvlc_MediaPlayerEncounteredError   = $10A;
  libvlc_MediaPlayerTimeChanged        = $10B;
  libvlc_MediaPlayerPositionChanged    = $10C;
  libvlc_MediaPlayerSeekableChanged    = $10D;
  libvlc_MediaPlayerPausableChanged    = $10E;
  libvlc_MediaPlayerTitleChanged       = $10F;
  libvlc_MediaPlayerSnapshotTaken      = $110;
  libvlc_MediaPlayerLengthChanged      = $111;
  libvlc_MediaPlayerVout               = $112;
  libvlc_MediaPlayerScrambledChanged   = $113;

  libvlc_MediaListItemAdded            = $200;
  libvlc_MediaListWillAddItem          = $201;
  libvlc_MediaListItemDeleted          = $202;
  libvlc_MediaListWillDeleteItem       = $203;

  libvlc_MediaListViewItemAdded        = $300;
  libvlc_MediaListViewWillAddItem      = $301;
  libvlc_MediaListViewItemDeleted      = $302;
  libvlc_MediaListViewWillDeleteItem   = $303;

  libvlc_MediaListPlayerPlayed         = $400;
  libvlc_MediaListPlayerNextItemSet    = $401;
  libvlc_MediaListPlayerStopped        = $402;

  libvlc_MediaDiscovererStarted        = $500;
  libvlc_MediaDiscovererEnded          = $501;

  libvlc_VlmMediaAdded                 = $600;
  libvlc_VlmMediaRemoved               = $601;
  libvlc_VlmMediaChanged               = $602;
  libvlc_VlmMediaInstanceStarted       = $603;
  libvlc_VlmMediaInstanceStopped       = $604;
  libvlc_VlmMediaInstanceStatusInit    = $605;
  libvlc_VlmMediaInstanceStatusOpening = $606;
  libvlc_VlmMediaInstanceStatusPlaying = $607;
  libvlc_VlmMediaInstanceStatusPause   = $608;
  libvlc_VlmMediaInstanceStatusEnd     = $609;
  libvlc_VlmMediaInstanceStatusError   = $60A;
{$ENDIF}

(*
 * Types below declared only for preserve structure of libvlc_event_t record
 *)

type
  event_media_meta_changed_t = packed record
    meta_type : libvlc_meta_t;
  end;

type
  event_media_subitem_added_t = packed record
    new_child : libvlc_media_t_ptr;
  end;

type
  event_media_duration_changed_t = packed record
    new_duration : Int64;
  end;

type
  media_parsed_changed_t = packed record
    new_status : Integer;
  end;

type
  media_freed_t = packed record
    md : libvlc_media_t_ptr;
  end;

type
  media_state_changed_t = packed record
    new_state : libvlc_state_t;
  end;

type
  media_subitemtree_added_t = packed record
    item : libvlc_media_t_ptr; 
  end;

type
  media_player_buffering_t = packed record
    new_cache : Single; // float
  end;

type
  media_player_media_changed_t = packed record
    new_media : libvlc_media_t_ptr;
  end;

type
  media_player_time_changed_t = packed record
    new_time : libvlc_time_t;
  end;
  
type
  media_player_position_changed_t = packed record
    new_position : Single; // float
  end;

type
  media_player_seekable_changed_t = packed record
    new_seekable : Integer;
  end;

type
  media_player_pausable_changed_t = packed record
    new_pausable : Integer;
  end;

type
  media_player_vout_t = packed record
    new_count : Integer;
  end;

type
  media_player_scrambled_changed_t = packed record
    new_scrambled : Integer;
  end;

type
  media_player_snapshot_taken_t = packed record
    psz_filename : PAnsiChar;
  end;

type
  media_player_length_changed_t = packed record
    new_length : libvlc_time_t;
  end;

type
  media_player_title_changed_t = packed record
    new_title : Integer;
  end;

type
  media_list_item_added_t = packed record
    item  : libvlc_media_t_ptr;
    index : Integer;
  end;

type
  media_list_will_add_item_t = packed record
    item  : libvlc_media_t_ptr;
    index : Integer;
  end;

type
  media_list_item_deleted_t = packed record
    item  : libvlc_media_t_ptr;
    index : Integer;
  end;

type
  media_list_will_delete_item_t = packed record
    item  : libvlc_media_t_ptr;
    index : Integer;
  end;

type
  media_list_player_next_item_set_t = packed record
    item : libvlc_media_t_ptr;
  end;

type
  vlm_media_event_t = packed record
    psz_media_name    : PAnsiChar;
    psz_instance_name : PAnsiChar;
  end;

{$IFDEF CPUX64}{$A8}{$Z4}{$ENDIF}

type
  libvlc_event_t_ptr = ^libvlc_event_t;
  libvlc_event_t = record
    event_type : libvlc_event_type_t;
    p_obj      : Pointer;              (* Object emitting the event *)

    case libvlc_event_type_t of

    // Extra MediaPlayer
    libvlc_MediaPlayerMediaChanged     : (media_player_media_changed      : media_player_media_changed_t);

    // media list player
    libvlc_MediaListPlayerNextItemSet  : (media_list_player_next_item_set : media_list_player_next_item_set_t);

    // snapshot taken
    libvlc_MediaPlayerSnapshotTaken    : (media_player_snapshot_taken     : media_player_snapshot_taken_t);

    // Length changed
    libvlc_MediaPlayerLengthChanged    : (media_player_length_changed     : media_player_length_changed_t);

    // media descriptor
    libvlc_MediaMetaChanged            : (media_meta_changed              : event_media_meta_changed_t);
    libvlc_MediaSubItemAdded           : (media_subitem_added             : event_media_subitem_added_t);
    libvlc_MediaDurationChanged        : (media_duration_changed          : event_media_duration_changed_t);
    libvlc_MediaParsedChanged          : (media_parsed_changed            : media_parsed_changed_t);
    libvlc_MediaFreed                  : (media_freed                     : media_freed_t);
    libvlc_MediaStateChanged           : (media_state_changed             : media_state_changed_t);
    libvlc_MediaSubItemTreeAdded       : (media_subitemtree_added         : media_subitemtree_added_t);

    // media instance
    libvlc_MediaPlayerBuffering        : (media_player_buffering          : media_player_buffering_t);
    libvlc_MediaPlayerPositionChanged  : (media_player_position_changed   : media_player_position_changed_t);
    libvlc_MediaPlayerTimeChanged      : (media_player_time_changed       : media_player_time_changed_t);
    libvlc_MediaPlayerTitleChanged     : (media_player_title_changed      : media_player_title_changed_t);
    libvlc_MediaPlayerSeekableChanged  : (media_player_seekable_changed   : media_player_seekable_changed_t);
    libvlc_MediaPlayerPausableChanged  : (media_player_pausable_changed   : media_player_pausable_changed_t);
    libvlc_MediaPlayerVout             : (media_player_vout               : media_player_vout_t);
    libvlc_MediaPlayerScrambledChanged : (media_player_scrambled_changed  : media_player_scrambled_changed_t);

    // media list
    libvlc_MediaListItemAdded          : (media_list_item_added           : media_list_item_added_t);
    libvlc_MediaListWillAddItem        : (media_list_will_add_item        : media_list_will_add_item_t);
    libvlc_MediaListItemDeleted        : (media_list_item_deleted         : media_list_item_deleted_t);
    libvlc_MediaListWillDeleteItem     : (media_list_will_delete_item     : media_list_will_delete_item_t);

    // VLM media
    libvlc_VlmMediaAdded,
    libvlc_VlmMediaRemoved,
    libvlc_VlmMediaChanged,
    libvlc_VlmMediaInstanceStarted,
    libvlc_VlmMediaInstanceStopped,
    libvlc_VlmMediaInstanceStatusInit,
    libvlc_VlmMediaInstanceStatusOpening,
    libvlc_VlmMediaInstanceStatusPlaying,
    libvlc_VlmMediaInstanceStatusPause,
    libvlc_VlmMediaInstanceStatusEnd,
    libvlc_VlmMediaInstanceStatusError : (vlm_media_event : vlm_media_event_t);
  end;

(**
 * Callback function notification
 * param p_event the event triggering the callback
 *)

type
   libvlc_event_callback_t = procedure(
     p_event : libvlc_event_t_ptr;
     data    :  Pointer
   ); cdecl;

(**
 * Option flags for libvlc_media_add_option_flag
 *)

{$IFDEF HAS_ENUM_ORDINALITY}
type
  input_item_option_e = (
    // Allow VLC to trust the given option.
    // By default options are untrusted
    VLC_INPUT_OPTION_TRUSTED = $0002,

    // Change the value associated to an option if already present,
    // otherwise add the option
    VLC_INPUT_OPTION_UNIQUE  = $0100
  );
{$ELSE}
type
  input_item_option_e = $0002..$0100;
const
  VLC_INPUT_OPTION_TRUSTED = $0002;
  VLC_INPUT_OPTION_UNIQUE  = $0100;
{$ENDIF}

(*
 *******************************************************************************
 * libvlc.h
 *******************************************************************************
 *)

(**
 * A human-readable error message for the last LibVLC error in the calling
 * thread. The resulting string is valid until another error occurs (at least
 * until the next LibVLC call).
 *
 * @warning
 * This will be NULL if there was no error.
 *)

 var
   libvlc_errmsg : function() : PAnsiChar; cdecl;

(**
 * Clears the LibVLC error status for the current thread. This is optional.
 * By default, the error status is automatically overriden when a new error
 * occurs, and destroyed when the thread exits.
 *)

var
  libvlc_clearerr : procedure(); cdecl;

(**
 * Sets the LibVLC error status and message for the current thread.
 * Any previous error is overridden.
 * param fmt the format string
 * param ap the arguments
 * return  a nul terminated string in any case
 *)

{$IFDEF FPC}
var
  libvlc_vprinterr : function(
    fmt : PAnsiChar;
    ap  : Pointer
  ) : PAnsiChar; cdecl;
{$ELSE}
var
  libvlc_vprinterr : function(
    fmt : PAnsiChar;
    ap  : array of const
  ) : PAnsiChar; cdecl;
{$ENDIF}

(**
 * Sets the LibVLC error status and message for the current thread.
 * Any previous error is overridden.
 * param fmt the format string
 * param args the arguments
 * return  a nul terminated string in any case
 *)

{$IFDEF FPC}
var
  libvlc_printerr : function(
    fmt : PAnsiChar;
    arg : Pointer
  ) : PAnsiChar; cdecl;
{$ELSE}
var
  libvlc_printerr : function(
    fmt : PAnsiChar;
    arg : array of const
  ) : PAnsiChar; cdecl;
{$ENDIF}

(**
 * Create and initialize a libvlc instance.
 * This functions accept a list of "command line" arguments similar to the
 * main(). These arguments affect the LibVLC instance default configuration.
 *
 * version
 * Arguments are meant to be passed from the command line to LibVLC, just like
 * VLC media player does. The list of valid arguments depends on the LibVLC
 * version, the operating system and platform, and set of available LibVLC
 * plugins. Invalid or unsupported arguments will cause the function to fail
 * (i.e. return NULL). Also, some arguments may alter the behaviour or
 * otherwise interfere with other LibVLC functions.
 *
 * warning
 * There is absolutely no warranty or promise of forward, backward and
 * cross-platform compatibility with regards to libvlc_new() arguments.
 * We recommend that you do not use them, other than when debugging.
 *
 * param argc the number of arguments (should be 0)
 * param argv list of arguments (should be NULL)
 * return the libvlc instance or NULL in case of error
 *)
var
  libvlc_new : function(
    argc : Integer;
    args : PPAnsiChar
  ) : libvlc_instance_t_ptr; cdecl;

(**
 * Decrement the reference count of a libvlc instance, and destroy it
 * if it reaches zero.
 *
 * param p_libvlc_instance the instance to destroy
 *)

var
  libvlc_release : procedure(
    p_instance : libvlc_instance_t_ptr
  ); cdecl;

(**
 * Increments the reference count of a libvlc instance.
 * The initial reference count is 1 after libvlc_new() returns.
 *
 * param p_instance the instance to reference
 *)

var
  libvlc_retain : procedure(
    p_instance : libvlc_instance_t_ptr
  ); cdecl;

(**
 * Try to start a user interface for the libvlc instance.
 *
 * param p_instance the instance
 * param name interface name, or NULL for default
 * return  0 on success, -1 on error.
 *)

var
  libvlc_add_intf : function(
    p_instance : libvlc_instance_t_ptr;
    name       : PAnsiChar
  ) : Integer; cdecl;

(**
 * Registers a callback for the LibVLC exit event. This is mostly useful if
 * the VLC playlist and/or at least one interface are started with
 * libvlc_playlist_play() or libvlc_add_intf() respectively.
 * Typically, this function will wake up your application main loop (from
 * another thread).
 *
 * note This function should be called before the playlist or interface are
 * started. Otherwise, there is a small race condition: the exit event could
 * be raised before the handler is registered.
 *
 * param p_instance LibVLC instance
 * param cb callback to invoke when LibVLC wants to exit,
 *           or NULL to disable the exit handler (as by default)
 * param opaque data pointer for the callback
 * warning This function and libvlc_wait() cannot be used at the same time.
 *)

type
  libvlc_exit_callback_t = procedure(data: Pointer);
var
  libvlc_set_exit_handler : procedure (
    p_instance : libvlc_instance_t_ptr;
    cb         : libvlc_exit_callback_t;
    opaque     : Pointer
  ); cdecl;


(**
 * Waits until an interface causes the instance to exit.
 * You should start at least one interface first, using libvlc_add_intf().
 *
 * param p_instance the instance
 * warning This function wastes one thread doing basically nothing.
 * libvlc_set_exit_handler() should be used instead.
 *)

var
  libvlc_wait : procedure(
    p_instance : libvlc_instance_t_ptr
  ); cdecl;

(**
 * Sets the application name. LibVLC passes this as the user agent string
 * when a protocol requires it.
 *
 * param p_instance LibVLC instance
 * param name human-readable application name, e.g. "FooBar player 1.2.3"
 * param http HTTP User Agent, e.g. "FooBar/1.2.3 Python/2.6.0"
 * version LibVLC 1.1.1 or later
 *)

var
  libvlc_set_user_agent : procedure(
    p_instance : libvlc_instance_t_ptr;
    name       : PAnsiChar;
    http       : PAnsiChar
  ); cdecl;

(**
 * Sets some meta-informations about the application.
 * See also libvlc_set_user_agent().
 *
 * param p_instance LibVLC instance
 * param id Java-style application identifier, e.g. "com.acme.foobar"
 * param version application version numbers, e.g. "1.2.3"
 * param icon application icon name, e.g. "foobar"
 * version LibVLC 2.1.0 or later.
 *)
var
  libvlc_set_app_id : procedure(
    p_instance : libvlc_instance_t_ptr;
    id         : PAnsiChar;
    version    : PAnsiChar;
    icon       : PAnsiChar); cdecl;

(**
 * Retrieve libvlc version.
 *
 * Example: "1.1.0-git The Luggage"
 *
 * return  a string containing the libvlc version
 *)

var
  libvlc_get_version : function() : PAnsiChar; cdecl;

(**
 * Retrieve libvlc compiler version.
 *
 * Example: "gcc version 4.2.3 (Ubuntu 4.2.3-2ubuntu6)"
 *
 * return  a string containing the libvlc compiler version
 *)

var
  libvlc_get_compiler : function() : PAnsiChar; cdecl;

(**
 * Retrieve libvlc changeset.
 *
 * Example: "aa9bce0bc4"
 *
 * return  a string containing the libvlc changeset
 *)

var
  libvlc_get_changeset : function() : PAnsiChar; cdecl;

(**
 * Frees an heap allocation returned by a LibVLC function.
 * If you know you're using the same underlying C run-time as the LibVLC
 * implementation, then you can call ANSI C free() directly instead.
 *
 * param ptr the pointer
 *)
var
  libvlc_free : procedure(
    ptr : Pointer
  ); cdecl;
 
(**
 * LibVLC emits asynchronous events.
 *
 * Several LibVLC objects (such @ref libvlc_instance_t as
 * @ref libvlc_media_player_t) generate events asynchronously. Each of them
 * provides @ref libvlc_event_manager_t event manager. You can subscribe to
 * events with libvlc_event_attach() and unsubscribe with
 * libvlc_event_detach().
 *
 * Event manager that belongs to a libvlc object, and from whom events can
 * be received.
 *)
    
(**
 * Register for an event notification.
 *
 * param p_event_manager the event manager to which you want to attach to.
 *       Generally it is obtained by vlc_my_object_event_manager() where
 *       my_object is the object you want to listen to.
 * param i_event_type the desired event to which we want to listen
 * param f_callback the function to call when i_event_type occurs
 * param user_data user provided data to carry with the event
 * return  0 on success, ENOMEM on error
 *)

var
  libvlc_event_attach : function(
    p_event_manager : libvlc_event_manager_t_ptr;
    i_event_type    : libvlc_event_type_t;
    f_callback      : libvlc_event_callback_t;
    user_data       : Pointer
  ) : Integer; cdecl;
  
(**
 * Unregister an event notification.
 *
 * param p_event_manager the event manager
 * param i_event_type the desired event to which we want to unregister
 * param f_callback the function to call when i_event_type occurs
 * param p_user_data user provided data to carry with the event
 *)

var
  libvlc_event_detach : procedure(
    p_event_manager : libvlc_event_manager_t_ptr;
    i_event_type    : libvlc_event_type_t;
    f_callback      : libvlc_event_callback_t;
    p_user_data     : Pointer
  ); cdecl;

(**
 * Get an event's type name.
 *
 * param event_type the desired event
 *)

var
  libvlc_event_type_name : function(
    event_type : libvlc_event_type_t
  ) : PAnsiChar; cdecl;

(**
 * Logging messages level.
 * note Future LibVLC versions may define new levels.
 *)

{$IFDEF HAS_ENUM_ORDINALITY}
type
  libvlc_log_level_t = (
    LIBVLC_LOG_DEBUG   = 0, // Debug message
    LIBVLC_LOG_NOTICE  = 2, // Important informational message
    LIBVLC_LOG_WARNING = 3, // Warning (potential error) message
    LIBVLC_LOG_ERROR   = 4  // Error message
  );
{$ELSE}
type
  libvlc_log_level_t = 0..4;
const
  LIBVLC_LOG_DEBUG   = 0;
  LIBVLC_LOG_NOTICE  = 2;
  LIBVLC_LOG_WARNING = 3;
  LIBVLC_LOG_ERROR   = 4;
{$ENDIF}

(**
 * libvlc_log_* functions provide access to the LibVLC messages log.
 * This is used for debugging or by advanced users.
 *)

(**
 * Gets debugging informations about a log message: the name of the VLC module
 * emitting the message and the message location within the source code.
 *
 * The returned module name and file name will be NULL if unknown.
 * The returned line number will similarly be zero if unknown.
 *
 * param ctx message context (as passed to the @ref libvlc_log_cb callback)
 * param module module name storage (or NULL) [OUT]
 * param file source code file name storage (or NULL) [OUT]
 * param line source code file line number storage (or NULL) [OUT]
 * warning The returned module name and source code file name, if non-NULL,
 * are only valid until the logging callback returns.
 *
 * version LibVLC 2.1.0 or later
 *)
var
  libvlc_log_get_context : procedure(
    ctx : libvlc_log_t_ptr;
    var module : PAnsiChar;
    var file_name : PANsiChar;
    var line : LongWord); cdecl;

(**
 * Gets VLC object informations about a log message: the type name of the VLC
 * object emitting the message, the object header if any and a temporaly-unique
 * object identifier. These informations are mainly meant for MANUAL
 * troubleshooting.
 *
 * The returned type name may be "generic" if unknown, but it cannot be NULL.
 * The returned header will be NULL if unset; in current versions, the header
 * is used to distinguish for VLM inputs.
 * The returned object ID will be zero if the message is not associated with
 * any VLC object.
 *
 * param ctx message context (as passed to the @ref libvlc_log_cb callback)
 * param name object name storage (or NULL) [OUT]
 * param header object header (or NULL) [OUT]
 * param line source code file line number storage (or NULL) [OUT]
 * warning The returned module name and source code file name, if non-NULL,
 * are only valid until the logging callback returns.
 *
 * version LibVLC 2.1.0 or later
 *)
var
  libvlc_log_get_object : procedure(
    ctx : libvlc_log_t_ptr;
    var name : PAnsiChar;
    var header : PAnsiChar;
    var id : Pointer); cdecl;

(**
 * Callback prototype for LibVLC log message handler.
 * param data data pointer as given to libvlc_log_set()
 * param level message level (@ref enum libvlc_log_level)
 * param ctx message context (meta-informations about the message)
 * param fmt printf() format string (as defined by ISO C11)
 * param args variable argument list for the format
 * note Log message handlers MUST be thread-safe.
 * warning The message context pointer, the format string parameters and the
 *          variable arguments are only valid until the callback returns.
 *)

{$IFDEF FPC}
type
  libvlc_log_cb = procedure(
    data  : Pointer;
    level : Integer;
    ctx   : libvlc_log_t_ptr;
    fmt   : PAnsiChar;
    args  : Pointer); cdecl;
{$ELSE}
type
  libvlc_log_cb = procedure(
    data  : Pointer;
    level : Integer;
    ctx   : libvlc_log_t_ptr;
    fmt   : PAnsiChar;
    args  : array of const); cdecl;
{$ENDIF}

(**
 * Unsets the logging callback for a LibVLC instance. This is rarely needed:
 * the callback is implicitly unset when the instance is destroyed.
 * This function will wait for any pending callbacks invocation to complete
 * (causing a deadlock if called from within the callback).
 *
 * param p_instance libvlc instance
 * version LibVLC 2.1.0 or later
 *)
var
  libvlc_log_unset : procedure(
    p_instance: libvlc_instance_t_ptr
  ); cdecl;

(**
 * Sets the logging callback for a LibVLC instance.
 * This function is thread-safe: it will wait for any pending callbacks
 * invocation to complete.
 *
 * param cb callback function pointer
 * param data opaque data pointer for the callback function
 *
 * note Some log messages (especially debug) are emitted by LibVLC while
 * is being initialized. These messages cannot be captured with this interface.
 *
 * warning A deadlock may occur if this function is called from the callback.
 *
 * param p_instance libvlc instance
 * version LibVLC 2.1.0 or later
 *)
var
  libvlc_log_set : procedure(
    p_instance: libvlc_instance_t_ptr;
    cb : libvlc_log_cb;
    data : Pointer); cdecl;

(**
 * Sets up logging to a file.
 * param p_instance libvlc instance
 * param stream FILE pointer opened for writing
 *         (the FILE pointer must remain valid until libvlc_log_unset())
 * version LibVLC 2.1.0 or later
 *)
var
  libvlc_log_set_file : procedure (
    p_instance: libvlc_instance_t_ptr;
    stream : THandle
  ); cdecl;

(**
 * Return the VLC messaging verbosity level.
 *
 * param p_instance libvlc instance
 * return  verbosity level for messages
 *)

// DEPRECATED

var
  libvlc_get_log_verbosity : function(
    p_instance : libvlc_instance_t_ptr
  ) : LongWord; cdecl;

(**
 * Set the VLC messaging verbosity level.
 *
 * param p_instance libvlc instance
 * param level log level
 *)

var
  libvlc_set_log_verbosity : procedure(
    p_instance : libvlc_instance_t_ptr;
    level      : LongWord
  ); cdecl;
  
(**
 * Open a VLC message log instance.
 *
 * param p_instance libvlc instance
 * return  log message instance or NULL on error
 *)

var
  libvlc_log_open : function(
    p_instance : libvlc_instance_t_ptr
  ) : libvlc_log_t_ptr; cdecl;

(**
 * Close a VLC message log instance.
 *
 * param p_log libvlc log instance or NULL
 *)

var
  libvlc_log_close : procedure(
    p_log : libvlc_log_t_ptr
  ); cdecl;

(**
 * Returns the number of messages in a log instance.
 *
 * param p_log libvlc log instance or NULL
 * return  number of log messages, 0 if p_log is NULL
 *)

var
  libvlc_log_count : function(
    p_log : libvlc_log_t_ptr
  ) : LongWord; cdecl;

(**
 * Clear a log instance.
 *
 * All messages in the log are removed. The log should be cleared on a
 * regular basis to avoid clogging.
 *
 * param p_log libvlc log instance or NULL
 *)

var
  libvlc_log_clear : procedure(
    p_log : libvlc_log_t_ptr
  ); cdecl;

(**
 * Allocate and returns a new iterator to messages in log.
 *
 * param p_log libvlc log instance
 * return  log iterator object or NULL on error
 *)

var
  libvlc_log_get_iterator : function(
    p_log: libvlc_log_t_ptr
  ) : libvlc_log_iterator_t_ptr; cdecl;

(**
 * Release a previoulsy allocated iterator.
 *
 * param p_iter libvlc log iterator or NULL
 *)

var
  libvlc_log_iterator_free : procedure(
    p_log : libvlc_log_t_ptr
  ); cdecl;

(**
 * Return whether log iterator has more messages.
 *
 * param p_iter libvlc log iterator or NULL
 * return  true if iterator has more message objects, else false
 *)

var
  libvlc_log_iterator_has_next : function(
    p_iter: libvlc_log_iterator_t_ptr
  ) : LongInt; cdecl;

(**
 * Return the next log message.
 *
 * The message contents must not be freed
 *
 * param p_iter libvlc log iterator or NULL
 * param p_buffer log buffer
 * return  log message object or NULL if none left
 *)

var
  libvlc_log_iterator_next : function(
    p_iter   : libvlc_log_iterator_t_ptr;
    p_buffer : libvlc_log_message_t_ptr
  ) : libvlc_log_message_t_ptr; cdecl;


(**
 * Description of a module.
 *)
type
  libvlc_module_description_t_ptr = ^libvlc_module_description_t;
  libvlc_module_description_t = packed record
    psz_name      : PAnsiChar;
    psz_shortname : PAnsiChar;
    psz_longname  : PAnsiChar;
    psz_help      : PAnsiChar;
    p_next        : libvlc_module_description_t_ptr;
  end;

(**
 * Release a list of module descriptions.
 *
 * param p_list the list to be released
 *)
var
  libvlc_module_description_list_release : procedure(
    p_list : libvlc_module_description_t_ptr
  ); cdecl;

(**
 * Returns a list of audio filters that are available.
 *
 * param p_instance libvlc instance
 *
 * return a list of module descriptions. It should be freed with libvlc_module_description_list_release().
 *         In case of an error, NULL is returned.
 *
 * see libvlc_module_description_t
 * see libvlc_module_description_list_release
 *)
var
  libvlc_audio_filter_list_get : function(
    p_instance  : libvlc_instance_t_ptr
  ) : libvlc_module_description_t_ptr; cdecl;

(**
 * Returns a list of video filters that are available.
 *
 * param p_instance libvlc instance
 *
 * return a list of module descriptions. It should be freed with libvlc_module_description_list_release().
 *         In case of an error, NULL is returned.
 *
 * see libvlc_module_description_t
 * see libvlc_module_description_list_release
 *)
var
  libvlc_video_filter_list_get : function(
    p_instance : libvlc_instance_t_ptr
  ) : libvlc_module_description_t_ptr; cdecl;

(**
 * Return the current time as defined by LibVLC. The unit is the microsecond.
 * Time increases monotonically (regardless of time zone changes and RTC
 * adjustements).
 * The origin is arbitrary but consistent across the whole system
 * (e.g. the system uptim, the time since the system was booted).
 * note On systems that support it, the POSIX monotonic clock is used.
 *)
var
  libvlc_clock : function() : Int64; cdecl;

(**
 * Return the delay (in microseconds) until a certain timestamp.
 * param pts timestamp
 * return negative if timestamp is in the past,
 * positive if it is in the future
 *)
function libvlc_delay(pts : Int64) : Int64;{$IFDEF DELPHI2005_UP}inline;{$ENDIF}

(*******************************************************************************
 * libvlc_media.h
 ******************************************************************************)

(**
 * Create a media with a certain given media resource location,
 * for instance a valid URL.
 *
 * To refer to a local file with this function,
 * the file://... URI syntax MUST be used (see IETF RFC3986).
 * We recommend using libvlc_media_new_path() instead when dealing with
 * local files.
 *
 * see libvlc_media_release
 *
 * param p_instance the instance
 * param psz_mrl the media location
 * return  the newly created media or NULL on error
 *)

var
  libvlc_media_new_location : function (
    p_instance : libvlc_instance_t_ptr;
    psz_mrl    : PAnsiChar
  ) : libvlc_media_t_ptr; cdecl;

(**
 * Create a media for a certain file path.
 *
 * see libvlc_media_release
 *
 * param p_instance the instance
 * param path local filesystem path
 * return  the newly created media or NULL on error
 *)

var
  libvlc_media_new_path : function(
    p_instance : libvlc_instance_t_ptr;
    path       : PAnsiChar
  ) : libvlc_media_t_ptr; cdecl;

(**
 * Create a media for an already open file descriptor.
 * The file descriptor shall be open for reading (or reading and writing).
 *
 * Regular file descriptors, pipe read descriptors and character device
 * descriptors (including TTYs) are supported on all platforms.
 * Block device descriptors are supported where available.
 * Directory descriptors are supported on systems that provide fdopendir().
 * Sockets are supported on all platforms where they are file descriptors,
 * i.e. all except Windows.
 *
 * This library will NOT automatically close the file descriptor
 * under any circumstance. Nevertheless, a file descriptor can usually only be
 * rendered once in a media player. To render it a second time, the file
 * descriptor should probably be rewound to the beginning with lseek().
 *
 * see libvlc_media_release
 *
 * version LibVLC 1.1.5 and later.
 *
 * param p_instance the instance
 * param fd open file descriptor
 * return  the newly created media or NULL on error
 *)

var
  libvlc_media_new_callbacks: function(
    p_instance: libvlc_instance_t_ptr;
    open_cb: libvlc_media_open_cb;
    read_cb: libvlc_media_read_cb;
    seek_cb: libvlc_media_seek_cb;
    close_cb: libvlc_media_close_cb;
    opaque: pointer
  ): libvlc_media_t_ptr; cdecl;
 
var
  libvlc_media_new_fd : function(
    p_instance : libvlc_instance_t_ptr;
    fd         : Integer
  ) : libvlc_media_t_ptr; cdecl;

(**
 * Create a media as an empty node with a given name.
 *
 * see libvlc_media_release
 *
 * param p_instance the instance
 * param psz_name the name of the node
 * return  the new empty media or NULL on error
 *)

var
  libvlc_media_new_as_node : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar
  ) : libvlc_media_t_ptr; cdecl;

(**
 * Add an option to the media.
 *
 * This option will be used to determine how the media_player will
 * read the media. This allows to use VLC's advanced
 * reading/streaming options on a per-media basis.
 *
 * note The options are listed in 'vlc --long-help' from the command line,
 * e.g. "-sout-all". Keep in mind that available options and their semantics
 * vary across LibVLC versions and builds.
 * warning Not all options affects libvlc_media_t objects:
 * Specifically, due to architectural issues most audio and video options,
 * such as text renderer options, have no effects on an individual media.
 * These options must be set through libvlc_new() instead.
 *
 * param p_md the media descriptor
 * param psz_options the options (as a string)
 *
 *)

var
  libvlc_media_add_option : procedure(
    p_md         : libvlc_media_t_ptr;
    ppsz_options : PAnsiChar
  ); cdecl;

(**
 * Add an option to the media with configurable flags.
 *
 * This option will be used to determine how the media_player will
 * read the media. This allows to use VLC's advanced
 * reading/streaming options on a per-media basis.
 *
 * The options are detailed in vlc --long-help, for instance
 * "--sout-all". Note that all options are not usable on medias:
 * specifically, due to architectural issues, video-related options
 * such as text renderer options cannot be set on a single media. They
 * must be set on the whole libvlc instance instead.
 *
 * param p_md the media descriptor
 * param psz_options the options (as a string)
 * param i_flags the flags for this option
 *)

var
  libvlc_media_add_option_flag : procedure(
    p_md         : libvlc_media_t_ptr;
    ppsz_options : PAnsiChar;
    i_flags      : input_item_option_e
  ); cdecl;

(**
 * Retain a reference to a media descriptor object (libvlc_media_t). Use
 * libvlc_media_release() to decrement the reference count of a
 * media descriptor object.
 *
 * param p_md the media descriptor
 *)

var
  libvlc_media_retain : procedure(
    p_md: libvlc_media_t_ptr
  ); cdecl;

(**
 * Decrement the reference count of a media descriptor object. If the
 * reference count is 0, then libvlc_media_release() will release the
 * media descriptor object. It will send out an libvlc_MediaFreed event
 * to all listeners. If the media descriptor object has been released it
 * should not be used again.
 *
 * param p_md the media descriptor
 *)

var
  libvlc_media_release : procedure(
    p_md : libvlc_media_t_ptr
  ); cdecl;


(**
 * Get the media resource locator (mrl) from a media descriptor object
 *
 * param p_md a media descriptor object
 * return  string with mrl of media descriptor object
 *)

var
  libvlc_media_get_mrl : function(
    p_md : libvlc_media_t_ptr
  ) : PAnsiChar; cdecl;

(**
 * Duplicate a media descriptor object.
 *
 * param p_md a media descriptor object.
 *)

var
  libvlc_media_duplicate : function (
    p_md : libvlc_media_t_ptr
  ) : libvlc_media_t_ptr; cdecl;

(**
 * Read the meta of the media.
 *
 * If the media has not yet been parsed this will return NULL.
 *
 * This methods automatically calls libvlc_media_parse_async(), so after calling
 * it you may receive a libvlc_MediaMetaChanged event. If you prefer a synchronous
 * version ensure that you call libvlc_media_parse() before get_meta().
 *
 * see libvlc_media_parse
 * see libvlc_media_parse_async
 * see libvlc_MediaMetaChanged
 *
 * param p_md the media descriptor
 * param e_meta the meta to read
 * return  the media's meta
 *)

var
  libvlc_media_get_meta : function(
    p_md   : libvlc_media_t_ptr;
    e_meta : libvlc_meta_t
  ) : PAnsiChar; cdecl;

(**
 * Set the meta of the media (this function will not save the meta, call
 * libvlc_media_save_meta in order to save the meta)
 *
 * param p_md the media descriptor
 * param e_meta the meta to write
 * param psz_value the media's meta
 *)

var
  libvlc_media_set_meta: procedure(
    p_md      : libvlc_media_t_ptr;
    e_meta    : libvlc_meta_t;
    psz_value : PAnsiChar
  ); cdecl;

(**
 * Save the meta previously set
 *
 * param p_md the media desriptor
 * return  true if the write operation was successfull
 *)

var
  libvlc_media_save_meta : function(
    p_md : libvlc_media_t_ptr
  ) : Integer; cdecl;

(**
 * Get current state of media descriptor object. Possible media states
 * are defined in libvlc_structures.c ( libvlc_NothingSpecial=0,
 * libvlc_Opening, libvlc_Buffering, libvlc_Playing, libvlc_Paused,
 * libvlc_Stopped, libvlc_Ended,
 * libvlc_Error).
 *
 * see libvlc_state_t
 * param p_meta_desc a media descriptor object
 * return  state of media descriptor object
 *)

var
  libvlc_media_get_state : function(
    p_md : libvlc_media_t_ptr
  ) : libvlc_state_t; cdecl;


(**
 * get the current statistics about the media
 * param p_md: media descriptor object
 * param p_stats: structure that contain the statistics about the media
 *                (this structure must be allocated by the caller)
 * return  true if the statistics are available, false otherwise
 *)

var
  libvlc_media_get_stats : function(
    p_md    : libvlc_media_t_ptr;
    p_stats : libvlc_media_stats_t_ptr
  ) : Integer; cdecl;

(**
 * Get subitems of media descriptor object. This will increment
 * the reference count of supplied media descriptor object. Use
 * libvlc_media_list_release() to decrement the reference counting.
 *
 * param p_md media descriptor object
 * return  list of media descriptor subitems or NULL
 *)

var
  libvlc_media_subitems : function(
    p_md : libvlc_media_t_ptr
  ) : libvlc_media_list_t_ptr; cdecl;

(**
 * Get event manager from media descriptor object.
 * NOTE: this function doesn't increment reference counting.
 *
 * param p_md a media descriptor object
 * return  event manager object
 *)

var
  libvlc_media_event_manager : function(
    p_md : libvlc_media_t_ptr
  ) : libvlc_event_manager_t_ptr; cdecl;

(**
 * Get duration (in ms) of media descriptor object item.
 *
 * param p_md media descriptor object
 * return  duration of media item or -1 on error
 *)

var
  libvlc_media_get_duration : function(
    p_md : libvlc_media_t_ptr
  ) : libvlc_time_t; cdecl;

(**
 * Parse a media.
 *
 * This fetches (local) meta data and tracks information.
 * The method is synchronous.
 *
 * see libvlc_media_parse_async
 * see libvlc_media_get_meta
 * see libvlc_media_get_tracks_info
 *
 * param media media descriptor object
 *)

var
  libvlc_media_parse : procedure(
    p_md : libvlc_media_t_ptr
  ); cdecl;

(**
 * Parse a media.
 *
 * This fetches (local) meta data and tracks information.
 * The method is the asynchronous of libvlc_media_parse().
 *
 * To track when this is over you can listen to libvlc_MediaParsedChanged
 * event. However if the media was already parsed you will not receive this
 * event.
 *
 * see libvlc_media_parse
 * see libvlc_MediaParsedChanged
 * see libvlc_media_get_meta
 * see libvlc_media_get_tracks_info
 *
 * param media media descriptor object
 *)

var
  libvlc_media_parse_async : procedure(
    p_md : libvlc_media_t_ptr
  ); cdecl;

(**
 * Get Parsed status for media descriptor object.
 *
 * see libvlc_MediaParsedChanged
 *
 * param p_md media descriptor object
 * return  true if media object has been parsed otherwise it returns false
 *)

var
  libvlc_media_is_parsed : function(
    p_md : libvlc_media_t_ptr
  ) : Integer; cdecl;

(**
 * Sets media descriptor's user_data. user_data is specialized data
 * accessed by the host application, VLC.framework uses it as a pointer to
 * an native object that references a libvlc_media_t pointer
 *
 * param p_md media descriptor object
 * param p_new_user_data pointer to user data
 *)

var
  libvlc_media_set_user_data : procedure(
    p_md            : libvlc_media_t_ptr;
    p_new_user_data : Pointer
  ); cdecl;

(**
 * Get media descriptor's user_data. user_data is specialized data
 * accessed by the host application, VLC.framework uses it as a pointer to
 * an native object that references a libvlc_media_t pointer
 *
 * param p_md media descriptor object
 *)

var
  libvlc_media_get_user_data : function(
    p_md : libvlc_media_t_ptr
  ) : Pointer; cdecl;

(**
 * Get media descriptor's elementary streams description
 *
 * Note, you need to play the media _one_ time with --sout="#description"
 * Not doing this will result in an empty array, and doing it more than once
 * will duplicate the entries in the array each time. Something like this:
 *
 * libvlc_media_player_t *player = libvlc_media_player_new_from_media(media);
 * libvlc_media_add_option_flag(media, "sout=\"#description\"");
 * libvlc_media_player_play(player);
 * // ... wait until playing
 * libvlc_media_player_release(player);
 *
 * This is very likely to change in next release, and be done at the parsing
 * phase.
 *
 * param media media descriptor object
 * param tracks address to store an allocated array of Elementary Streams
 * descriptions (must be freed by the caller)
 *
 * return  the number of Elementary Streams
 *)

var
  libvlc_media_get_tracks_info : function(
    p_md       : libvlc_media_t_ptr;
    var tracks : libvlc_media_track_info_t_ptr
  ) : Integer; cdecl;

(**
 * Get media descriptor's elementary streams description
 *
 * Note, you need to call libvlc_media_parse() or play the media at least once
 * before calling this function.
 * Not doing this will result in an empty array.
 *
 * version LibVLC 2.1.0 and later.
 *
 * param p_md media descriptor object
 * param tracks address to store an allocated array of Elementary Streams
 *        descriptions (must be freed with libvlc_media_tracks_release
          by the caller) [OUT]
 *
 * return  the number of Elementary Streams (zero on error)
 *)
var
  libvlc_media_tracks_get : function(
    p_md : libvlc_media_t_ptr;
    var tracks : Pointer) : LongWord; cdecl;


(**
 * Release media descriptor's elementary streams description array
 *
 * version LibVLC 2.1.0 and later.
 *
 * param p_tracks tracks info array to release
 * param i_count number of elements in the array
 *)
var
  libvlc_media_tracks_release : procedure(
    var tracks : Pointer;
    i_count : LongWord ); cdecl;

(*
 *******************************************************************************
 * libvlc_media_player.h
 *******************************************************************************
 *)

(**
 * Create an empty Media Player object
 *
 * param p_instance the libvlc instance in which the Media Player
 *        should be created.
 * return  a new media player object, or NULL on error.
 *)

var
  libvlc_media_player_new : function(
    p_instance : libvlc_instance_t_ptr
  ) : libvlc_media_player_t_ptr; cdecl;

(**
 * Create a Media Player object from a Media
 *
 * param p_md the media. Afterwards the p_md can be safely destroyed.
 * return  a new media player object, or NULL on error.
 *)

var
  libvlc_media_player_new_from_media : function(
    p_md : libvlc_media_t_ptr
  ) : libvlc_media_player_t_ptr; cdecl;

(**
 * Release a media_player after use
 * Decrement the reference count of a media player object. If the
 * reference count is 0, then libvlc_media_player_release() will
 * release the media player object. If the media player object
 * has been released, then it should not be used again.
 *
 * param p_mi the Media Player to free
 *)

var
  libvlc_media_player_release : procedure(
    p_mi : libvlc_media_player_t_ptr
  ); cdecl;

(**
 * Retain a reference to a media player object. Use
 * libvlc_media_player_release() to decrement reference count.
 *
 * param p_mi media player object
 *)

var
  libvlc_media_player_retain : procedure(
    p_mi : libvlc_media_player_t_ptr
  ); cdecl;

(**
 * Set the media that will be used by the media_player. If any,
 * previous md will be released.
 *
 * param p_mi the Media Player
 * param p_md the Media. Afterwards the p_md can be safely destroyed.
 *)

var
  libvlc_media_player_set_media : procedure(
    p_mi : libvlc_media_player_t_ptr;
    p_md : libvlc_media_t_ptr
  ); cdecl;

(**
 * Get the media used by the media_player.
 *
 * param p_mi the Media Player
 * return  the media associated with p_mi, or NULL if no media is associated
 *)

var
  libvlc_media_player_get_media : function(
    p_mi : libvlc_media_player_t_ptr
  ) : libvlc_media_t_ptr; cdecl;

(**
 * Get the Event Manager from which the media player send event.
 *
 * param p_mi the Media Player
 * return  the event manager associated with p_mi
 *)

var
  libvlc_media_player_event_manager : function(
    p_mi : libvlc_media_player_t_ptr
  ) : libvlc_event_manager_t_ptr; cdecl;

(**
 * is_playing
 *
 * param p_mi the Media Player
 * return  1 if the media player is playing, 0 otherwise
 *)

var
  libvlc_media_player_is_playing : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Play
 *
 * param p_mi the Media Player
 * return  0 if playback started (and was already started), or -1 on error.
 *)

var
  libvlc_media_player_play : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Pause or resume (no effect if there is no media)
 *
 * param mp the Media Player
 * param do_pause play/resume if zero, pause if non-zero
 * version LibVLC 1.1.1 or later
 *)
var
  libvlc_media_player_set_pause : procedure(
    p_mi     : libvlc_media_player_t_ptr;
    do_pause : Integer
  ); cdecl;

(**
 * Toggle pause (no effect if there is no media)
 *
 * param p_mi the Media Player
 *)

var
  libvlc_media_player_pause : procedure(
    p_mi : libvlc_media_player_t_ptr
  ); cdecl;

(**
 * Stop (no effect if there is no media)
 *
 * param p_mi the Media Player
 *)

var
  libvlc_media_player_stop : procedure(
    p_mi : libvlc_media_player_t_ptr
  ); cdecl;

(**
 * Callback prototype to allocate and lock a picture buffer.
 *
 * Whenever a new video frame needs to be decoded, the lock callback is
 * invoked. Depending on the video chroma, one or three pixel planes of
 * adequate dimensions must be returned via the second parameter. Those
 * planes must be aligned on 32-bytes boundaries.
 *
 * param opaque private pointer as passed to libvlc_video_set_callbacks() [IN]
 * param planes start address of the pixel planes (LibVLC allocates the array
 *             of void pointers, this callback must initialize the array) [OUT]
 * return  a private pointer for the display and unlock callbacks to identify the picture buffers
 *)

type
  libvlc_video_lock_cb = function(opaque : Pointer; var planes : Pointer) : Pointer; cdecl;

(**
 * Callback prototype to unlock a picture buffer.
 *
 * When the video frame decoding is complete, the unlock callback is invoked.
 * This callback might not be needed at all. It is only an indication that the
 * application can now read the pixel values if it needs to.
 *
 * warning A picture buffer is unlocked after the picture is decoded,
 *         but before the picture is displayed.
 *
 * param opaque private pointer as passed to libvlc_video_set_callbacks() [IN]
 * param picture private pointer returned from the @ref libvlc_video_lock_cb callback [IN]
 * param planes pixel planes as defined by the @ref libvlc_video_lock_cb callback
 *             (this parameter is only for convenience) [IN]
 *)

type
  libvlc_video_unlock_cb = function(opaque : Pointer; picture : Pointer; planes : Pointer) : Pointer; cdecl;

(**
 * Callback prototype to display a picture.
 *
 * When the video frame needs to be shown, as determined by the media playback
 * clock, the display callback is invoked.
 *
 * param opaque private pointer as passed to libvlc_video_set_callbacks() [IN]
 * param picture private pointer returned from the @ref libvlc_video_lock_cb callback [IN]
 *)

type
  libvlc_video_display_cb = function(opaque : Pointer; picture : Pointer) : Pointer; cdecl;

(**
 * Set callbacks and private data to render decoded video to a custom area
 * in memory.
 * Use libvlc_video_set_format() or libvlc_video_set_format_callbacks()
 * to configure the decoded format.
 *
 * param mp the media player
 * param lock callback to lock video memory (must not be NULL)
 * param unlock callback to unlock video memory (or NULL if not needed)
 * param display callback to display video (or NULL if not needed)
 * param opaque private pointer for the three callbacks (as first parameter)
 * version LibVLC 1.1.1 or later
 *)
 
var
  libvlc_video_set_callbacks : procedure(
    p_mi    : libvlc_media_player_t_ptr;
    lock    : libvlc_video_lock_cb;
    unlock  : libvlc_video_unlock_cb;
    display : libvlc_video_display_cb;
    opaque  : Pointer
  ); cdecl;

(**
 * Set decoded video chroma and dimensions.
 * This only works in combination with libvlc_video_set_callbacks(),
 * and is mutually exclusive with libvlc_video_set_format_callbacks().
 *
 * param mp the media player
 * param chroma a four-characters string identifying the chroma
 *              (e.g. "RV32" or "YUYV")
 * param width pixel width
 * param height pixel height
 * param pitch line pitch (in bytes)
 * version LibVLC 1.1.1 or later
 * bug All pixel planes are expected to have the same pitch.
 * To use the YCbCr color space with chrominance subsampling,
 * consider using libvlc_video_set_format_callbacks() instead.
 *)

var
  libvlc_video_set_format : procedure(
    p_mi   : libvlc_media_player_t_ptr;
    chroma : PAnsiChar;
    width  : Longword;
    height : Longword;
    pitch  : Longword
  ); cdecl;

(**
 * Callback prototype to configure picture buffers format.
 * This callback gets the format of the video as output by the video decoder
 * and the chain of video filters (if any). It can opt to change any parameter
 * as it needs. In that case, LibVLC will attempt to convert the video format
 * (rescaling and chroma conversion) but these operations can be CPU intensive.
 *
 * param opaque pointer to the private pointer passed to
 *               libvlc_video_set_callbacks() [IN/OUT]
 * param chroma pointer to the 4 bytes video format identifier [IN/OUT]
 * param width pointer to the pixel width [IN/OUT]
 * param height pointer to the pixel height [IN/OUT]
 * param pitches table of scanline pitches in bytes for each pixel plane
 *                (the table is allocated by LibVLC) [OUT]
 * param lines table of scanlines count for each plane [OUT]
 * return  the number of picture buffers allocated, 0 indicates failure
 *
 * note
 * For each pixels plane, the scanline pitch must be bigger than or equal to
 * the number of bytes per pixel multiplied by the pixel width.
 * Similarly, the number of scanlines must be bigger than of equal to
 * the pixel height.
 * Furthermore, we recommend that pitches and lines be multiple of 32
 * to not break assumption that might be made by various optimizations
 * in the video decoders, video filters and/or video converters.
 *)

type
  libvlc_video_format_cb = procedure (
    var opaque : Pointer;
    chroma : PAnsiChar;
    var width : LongWord;
    var height : LongWord;
    var pitches : LongWord;
    var lines : LongWord); cdecl;

(**
 * Callback prototype to configure picture buffers format.
 *
 * param opaque private pointer as passed to libvlc_video_set_callbacks()
 *               (and possibly modified by @ref libvlc_video_format_cb) [IN]
 *)

type
  libvlc_video_cleanup_cb = procedure(opaque : Pointer); cdecl;

(**
 * Set decoded video chroma and dimensions. This only works in combination with
 * libvlc_video_set_callbacks().
 *
 * param mp the media player
 * param setup callback to select the video format (cannot be NULL)
 * param cleanup callback to release any allocated resources (or NULL)
 * version LibVLC 2.0.0 or later
 *)
var
  libvlc_video_set_format_callbacks : procedure(
    p_mi    : libvlc_media_player_t_ptr;
    setup   : libvlc_video_format_cb;
    cleanup : libvlc_video_cleanup_cb); cdecl;

(**
 * Set the NSView handler where the media player should render its video output.
 *
 * Use the vout called "macosx".
 *
 * The drawable is an NSObject that follow the VLCOpenGLVideoViewEmbedding
 * protocol:
 *
 *)
 // @protocol VLCOpenGLVideoViewEmbedding <NSObject>
 // - (void)addVoutSubview:(NSView *)view;
 // - (void)removeVoutSubview:(NSView *)view;
 // @end

(*
 * Or it can be an NSView object.
 *
 * If you want to use it along with Qt4 see the QMacCocoaViewContainer. Then
 * the following code should work:
 * @begincode
 * {
 *     NSView *video = [[NSView alloc] init];
 *     QMacCocoaViewContainer *container = new QMacCocoaViewContainer(video, parent);
 *     libvlc_media_player_set_nsobject(mp, video);
 *     [video release];
 * }
 * @endcode
 *
 * You can find a live example in VLCVideoView in VLCKit.framework.
 *
 * param p_mi the Media Player
 * param drawable the drawable that is either an NSView or an object following
 *                the VLCOpenGLVideoViewEmbedding protocol.
 *)

var
  libvlc_media_player_set_nsobject : procedure(
    p_mi     : libvlc_media_player_t_ptr;
    drawable : Pointer
  ); cdecl;

(**
 * Get the NSView handler previously set with libvlc_media_player_set_nsobject().
 *
 * param p_mi the Media Player
 * return the NSView handler or 0 if none where set
 *)

var
  libvlc_media_player_get_nsobject : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Pointer; cdecl;

(**
 * Set the agl handler where the media player should render its video output.
 *
 * param p_mi the Media Player
 * param drawable the agl handler
 *)

var
  libvlc_media_player_set_agl : procedure(
    p_mi     : libvlc_media_player_t_ptr;
    drawable : LongWord
  ); cdecl;

(**
 * Get the agl handler previously set with libvlc_media_player_set_agl().
 *
 * param p_mi the Media Player
 * return  the agl handler or 0 if none where set
 *)

var
  libvlc_media_player_get_agl : function(
    p_mi : libvlc_media_player_t_ptr
  ) : LongWord; cdecl;

(**
 * Set an X Window System drawable where the media player should render its
 * video output. If LibVLC was built without X11 output support, then this has
 * no effects.
 *
 * The specified identifier must correspond to an existing Input/Output class
 * X11 window. Pixmaps are NOT supported. The caller shall ensure that
 * the X11 server is the same as the one the VLC instance has been configured
 * with. This function must be called before video playback is started;
 * otherwise it will only take effect after playback stop and restart.
 *
 * param p_mi the Media Player
 * param drawable the ID of the X window
 *)

var
  libvlc_media_player_set_xwindow : procedure(
    p_mi     : libvlc_media_player_t_ptr;
    drawable : LongWord
  ); cdecl;

(**
 * Get the X Window System window identifier previously set with
 * libvlc_media_player_set_xwindow(). Note that this will return the identifier
 * even if VLC is not currently using it (for instance if it is playing an
 * audio-only input).
 *
 * param p_mi the Media Player
 * return  an X window ID, or 0 if none where set.
 *)

var
  libvlc_media_player_get_xwindow : function(
    p_mi : libvlc_media_player_t_ptr
  ) : LongWord; cdecl;

(**
 * Set a Win32/Win64 API window handle (HWND) where the media player should
 * render its video output. If LibVLC was built without Win32/Win64 API output
 * support, then this has no effects.
 *
 * param p_mi the Media Player
 * param drawable windows handle of the drawable
 *)

var
  libvlc_media_player_set_hwnd : procedure(
    p_mi : libvlc_media_player_t_ptr;
    hwnd : THandle
  ); cdecl;

(**
 * Get the Windows API window handle (HWND) previously set with
 * libvlc_media_player_set_hwnd(). The handle will be returned even if LibVLC
 * is not currently outputting any video to it.
 *
 * param p_mi the Media Player
 * return a window handle or NULL if there are none.
 *)

var
  libvlc_media_player_get_hwnd : function(
    p_mi : libvlc_media_player_t_ptr
  ) : THandle; cdecl;


(**
 * Callback prototype for audio playback.
 * param data data pointer as passed to libvlc_audio_set_callbacks() [IN]
 * param samples pointer to the first audio sample to play back [IN]
 * param count number of audio samples to play back
 * param pts expected play time stamp (see libvlc_delay())
 *)
type
  libvlc_audio_play_cb = procedure(
    data : Pointer;
    const samples : Pointer;
    count : Cardinal;
    pts : Int64); cdecl;

(**
 * Callback prototype for audio pause.
 * note The pause callback is never called if the audio is already paused.
 * param data data pointer as passed to libvlc_audio_set_callbacks() [IN]
 * param pts time stamp of the pause request (should be elapsed already)
 *)
type
  libvlc_audio_pause_cb = procedure(
    data : Pointer;
    pts : Int64); cdecl;

(**
 * Callback prototype for audio resumption (i.e. restart from pause).
 * note The resume callback is never called if the audio is not paused.
 * param data data pointer as passed to libvlc_audio_set_callbacks() [IN]
 * param pts time stamp of the resumption request (should be elapsed already)
 *)
type
  libvlc_audio_resume_cb = procedure(
    data : Pointer;
    pts : Int64); cdecl;

(**
 * Callback prototype for audio buffer flush
 * (i.e. discard all pending buffers and stop playback as soon as possible).
 * param data data pointer as passed to libvlc_audio_set_callbacks() [IN]
 *)
type
  libvlc_audio_flush_cb = procedure(
    data : Pointer;
    pts : Int64); cdecl;

(**
 * Callback prototype for audio buffer drain
 * (i.e. wait for pending buffers to be played).
 * param data data pointer as passed to libvlc_audio_set_callbacks() [IN]
 *)
type
  libvlc_audio_drain_cb = procedure(data : Pointer);

(**
 * Callback prototype for audio volume change.
 * param data data pointer as passed to libvlc_audio_set_callbacks() [IN]
 * param volume software volume (1. = nominal, 0. = mute)
 * param mute muted flag
 *)
type
  libvlc_audio_set_volume_cb = procedure(
    data : Pointer;
    volume : Single;
    mute : Boolean);

(**
 * Set callbacks and private data for decoded audio.
 * Use libvlc_audio_set_format() or libvlc_audio_set_format_callbacks()
 * to configure the decoded audio format.
 *
 * param mp the media player
 * param play callback to play audio samples (must not be NULL)
 * param pause callback to pause playback (or NULL to ignore)
 * param resume callback to resume playback (or NULL to ignore)
 * param flush callback to flush audio buffers (or NULL to ignore)
 * param drain callback to drain audio buffers (or NULL to ignore)
 * param opaque private pointer for the audio callbacks (as first parameter)
 * version LibVLC 2.0.0 or later
 *)
var
  libvlc_audio_set_callbacks : procedure(
    p_mi   : libvlc_media_player_t_ptr;
    play   : libvlc_audio_play_cb;
    pause  : libvlc_audio_pause_cb;
    resume : libvlc_audio_resume_cb;
    flush  : libvlc_audio_flush_cb;
    drain  : libvlc_audio_drain_cb;
    opaque : Pointer ); cdecl;

(**
 * Set callbacks and private data for decoded audio. This only works in
 * combination with libvlc_audio_set_callbacks().
 * Use libvlc_audio_set_format() or libvlc_audio_set_format_callbacks()
 * to configure the decoded audio format.
 *
 * param mp the media player
 * param set_volume callback to apply audio volume,
 *                   or NULL to apply volume in software
 * version LibVLC 2.0.0 or later
 *)
var
  libvlc_audio_set_volume_callback : procedure(
    p_mi : libvlc_media_player_t_ptr;
    set_volume : libvlc_audio_set_volume_cb
  ); cdecl;

(**
 * Callback prototype to setup the audio playback.
 * This is called when the media player needs to create a new audio output.
 * param opaque pointer to the data pointer passed to
 *               libvlc_audio_set_callbacks() [IN/OUT]
 * param format 4 bytes sample format [IN/OUT]
 * param rate sample rate [IN/OUT]
 * param channels channels count [IN/OUT]
 * return 0 on success, anything else to skip audio playback
 *)
type
  libvlc_audio_setup_cb = function(
    var data : Pointer;
    var format : PAnsiChar;
    var rate  : Cardinal;
    var channels : Cardinal) : Integer; cdecl;

(**
 * Callback prototype for audio playback cleanup.
 * This is called when the media player no longer needs an audio output.
 * param opaque data pointer as passed to libvlc_audio_set_callbacks() [IN]
 *)
type
  libvlc_audio_cleanup_cb = procedure(
    data : Pointer
  ); cdecl;

(**
 * Set decoded audio format. This only works in combination with
 * libvlc_audio_set_callbacks().
 *
 * param mp the media player
 * param setup callback to select the audio format (cannot be NULL)
 * param cleanup callback to release any allocated resources (or NULL)
 * version LibVLC 2.0.0 or later
 *)
var
  libvlc_audio_set_format_callbacks : procedure(
    p_mi    : libvlc_media_player_t_ptr;
    setup   : libvlc_audio_setup_cb;
    cleanup : libvlc_audio_cleanup_cb ); cdecl;

(**
 * Set decoded audio format.
 * This only works in combination with libvlc_audio_set_callbacks(),
 * and is mutually exclusive with libvlc_audio_set_format_callbacks().
 *
 * param mp the media player
 * param format a four-characters string identifying the sample format
 *               (e.g. "S16N" or "FL32")
 * param rate sample rate (expressed in Hz)
 * param channels channels count
 * version LibVLC 2.0.0 or later
 *)
var
  libvlc_audio_set_format : procedure(
     p_mp         : libvlc_media_player_t_ptr;
     const format : PAnsiChar;
     rate         : Cardinal;
     channels     : Cardinal); cdecl;

(*
 *******************************************************************************
 * bug This might go away ... to be replaced by a broader system */
 *******************************************************************************
 *)

(**
 * Get the current movie length (in ms).
 *
 * param p_mi the Media Player
 * return  the movie length (in ms), or -1 if there is no media.
 *)

var
  libvlc_media_player_get_length : function(
    p_mi : libvlc_media_player_t_ptr
  ) : libvlc_time_t; cdecl;

(**
 * Get the current movie time (in ms).
 *
 * param p_mi the Media Player
 * return  the movie time (in ms), or -1 if there is no media.
 *)

var
  libvlc_media_player_get_time : function(
    p_mi : libvlc_media_player_t_ptr
  ) : libvlc_time_t; cdecl;

(**
 * Set the movie time (in ms). This has no effect if no media is being played.
 * Not all formats and protocols support this.
 *
 * param p_mi the Media Player
 * param i_time the movie time (in ms).
 *)

var
  libvlc_media_player_set_time : procedure(
    p_mi   : libvlc_media_player_t_ptr;
    i_time : libvlc_time_t
  ); cdecl;

(**
 * Get movie position as percentage between 0.0 and 1.0.
 *
 * param p_mi the Media Player
 * return  movie position in range 0..1, or -1. in case of error
 *)

var
  libvlc_media_player_get_position : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Single; cdecl;

(**
 * Set movie position as percentage between 0.0 and 1.0. 
 * This has no effect if playback is not enabled.
 * This might not work depending on the underlying input format and protocol.
 *
 * param p_mi the Media Player
 * param f_pos the position
 *)

var
  libvlc_media_player_set_position : procedure(
    p_mi  : libvlc_media_player_t_ptr;
    f_pos : Single // float
  ); cdecl;

(**
 * Set movie chapter (if applicable).
 *
 * param p_mi the Media Player
 * param i_chapter chapter number to play
 *)

var
  libvlc_media_player_set_chapter : procedure(
    p_mi      : libvlc_media_player_t_ptr;
    i_chapter : Integer
  ); cdecl;

(**
 * Get movie chapter.
 *
 * param p_mi the Media Player
 * return  chapter number currently playing, or -1 if there is no media.
 *)

var
  libvlc_media_player_get_chapter : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Get movie chapter count
 *
 * param p_mi the Media Player
 * return  number of chapters in movie, or -1.
 *)

var
  libvlc_media_player_get_chapter_count : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Is the player able to play
 *
 * param p_mi the Media Player
 * return  boolean
 *)

var
  libvlc_media_player_will_play : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Get title chapter count
 *
 * param p_mi the Media Player
 * param i_title title
 * return  number of chapters in title, or -1
 *)

var
  libvlc_media_player_get_chapter_count_for_title : function(
    p_mi    : libvlc_media_player_t_ptr;
    i_title : Integer
  ) : Integer; cdecl;

(**
 * Set movie title
 *
 * param p_mi the Media Player
 * param i_title title number to play
 *)

var
  libvlc_media_player_set_title : procedure(
    p_mi    : libvlc_media_player_t_ptr;
    i_title : Integer
  ); cdecl;

(**
 * Get movie title
 *
 * param p_mi the Media Player
 * return  title number currently playing, or -1
 *)

var
  libvlc_media_player_get_title : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Get movie title count
 *
 * param p_mi the Media Player
 * return  title number count, or -1
 *)

var
  libvlc_media_player_get_title_count : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Set previous chapter (if applicable)
 *
 * param p_mi the Media Player
 *)

var
  libvlc_media_player_previous_chapter : procedure(
    p_mi : libvlc_media_player_t_ptr
  ); cdecl;

(**
 * Set next chapter (if applicable)
 *
 * param p_mi the Media Player
 *)

var
  libvlc_media_player_next_chapter : procedure(
    p_mi : libvlc_media_player_t_ptr
  ); cdecl;

(**
 * Get the requested movie play rate.
 * @warning Depending on the underlying media, the requested rate may be
 * different from the real playback rate.
 *
 * param p_mi the Media Player
 * return  movie play rate
 *)

var
  libvlc_media_player_get_rate : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Single; cdecl;

(**
 * Set movie play rate
 *
 * param p_mi the Media Player
 * param rate movie play rate to set
 * return  -1 if an error was detected, 0 otherwise (but even then, it might
 * not actually work depending on the underlying media protocol)
 *)

var
  libvlc_media_player_set_rate : function(
    p_mi : libvlc_media_player_t_ptr;
    rate : Single // float
  ) : Integer; cdecl;

(**
 * Get current movie state
 *
 * param p_mi the Media Player
 * return  the current state of the media player (playing, paused, ...)
 * see libvlc_state_t
 *)

var
  libvlc_media_player_get_state : function(
    p_mi : libvlc_media_player_t_ptr
  ) : libvlc_state_t; cdecl;

(**
 * Get movie fps rate
 *
 * param p_mi the Media Player
 * return  frames per second (fps) for this playing movie, or 0 if unspecified
 *)

var
  libvlc_media_player_get_fps : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Single; cdecl;

(*
 *******************************************************************************
 * end bug
 *******************************************************************************
 *)

(**
 * How many video outputs does this media player have?
 *
 * param p_mi the media player
 * return  the number of video outputs
 *)

var
  libvlc_media_player_has_vout : function(
    p_mi : libvlc_media_player_t_ptr
  ) : LongWord; cdecl;

(**
 * Is this media player seekable?
 *
 * param p_mi the media player
 * return  true if the media player can seek
 *)

var
  libvlc_media_player_is_seekable : function(
    p_mi: libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Can this media player be paused?
 *
 * param p_mi the media player
 * return  true if the media player can pause
 *)

var
  libvlc_media_player_can_pause : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Check if the current program is scrambled
 *
 * param p_mi the media player
 * return true if the current program is scrambled
 *
 * libvlc_return_bool
 * version LibVLC 2.2.0 or later
 *)
var
  libvlc_media_player_program_scrambled: function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Display the next frame (if supported)
 *
 * param p_mi the media player
 *)

var
  libvlc_media_player_next_frame : procedure(
    p_mi : libvlc_media_player_t_ptr
  ); cdecl;

(**
 * Navigate through DVD Menu
 *
 * param p_mi the Media Player
 * param navigate the Navigation mode
 * version libVLC 2.0.0 or later
 *)
var
  libvlc_media_player_navigate : procedure (
    p_mi     : libvlc_media_player_t_ptr;
    navigate : libvlc_navigate_mode_t ); cdecl;

(**
 * Set if, and how, the video title will be shown when media is played.
 *
 * param p_mi       the media player
 * param position   position at which to display the title,
 *                  or libvlc_position_disable to prevent
 *                  the title from being displayed
 * param timeout    title display timeout in milliseconds
 *                  (ignored if libvlc_position_disable)
 * version libVLC 2.1.0 or later
 *)
var
  libvlc_media_player_set_video_title_display : procedure(
    p_mi     : libvlc_media_player_t_ptr;
    position : libvlc_position_t;
    timeout  : LongWord); cdecl;

(**
 * Release (free) libvlc_track_description_t
 *
 * param p_track_description the structure to release
 *)
var
  libvlc_track_description_list_release : procedure(
    p_track_description : libvlc_track_description_t_ptr
  ); cdecl;

(**
 * Release (free) libvlc_track_description_t
 *
 * param p_track_description the structure to release
 *)

var
  libvlc_track_description_release : procedure(
    p_track_description : libvlc_track_description_t_ptr
  ); cdecl;

(**
 * Toggle fullscreen status on non-embedded video outputs.
 *
 * @warning The same limitations applies to this function
 * as to libvlc_set_fullscreen().
 *
 * param p_mi the media player
 *)

var
  libvlc_toggle_fullscreen : procedure(
    p_mi : libvlc_media_player_t_ptr
  ); cdecl;

(**
 * Enable or disable fullscreen.
 *
 * @warning With most window managers, only a top-level windows can be in
 * full-screen mode. Hence, this function will not operate properly if
 * libvlc_media_player_set_xwindow() was used to embed the video in a
 * non-top-level window. In that case, the embedding window must be reparented
 * to the root window BEFORE fullscreen mode is enabled. You will want
 * to reparent it back to its normal parent when disabling fullscreen.
 *
 * param p_mi the media player
 * param b_fullscreen boolean for fullscreen status
 *)

var
  libvlc_set_fullscreen : procedure(
    p_mi         : libvlc_media_player_t_ptr;
    b_fullscreen : Integer
  ); cdecl;

(**
 * Get current fullscreen status.
 *
 * param p_mi the media player
 * return  the fullscreen status (boolean)
 *)

var
  libvlc_get_fullscreen : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Enable or disable key press events handling, according to the LibVLC hotkeys
 * configuration. By default and for historical reasons, keyboard events are
 * handled by the LibVLC video widget.
 *
 * On X11, there can be only one subscriber for key press and mouse
 * click events per window. If your application has subscribed to those events
 * for the X window ID of the video widget, then LibVLC will not be able to
 * handle key presses and mouse clicks in any case.
 *
 * This function is only implemented for X11 at the moment.
 *
 * param p_mi the media player
 * param on true to handle key press events, false to ignore them.
 *)

var
  libvlc_video_set_key_input : procedure(
    p_mi    : libvlc_media_player_t_ptr;
    keys_on : LongWord
  ); cdecl;

(**
 * Enable or disable mouse click events handling. By default, those events are
 * handled. This is needed for DVD menus to work, as well as a few video
 * filters such as "puzzle".
 *
 * See also libvlc_video_set_key_input().
 *
 * This function is only implemented for X11 at the moment.
 *
 * param p_mi the media player
 * param on true to handle mouse click events, false to ignore them.
 *)

var
  libvlc_video_set_mouse_input: procedure(
    p_mi     : libvlc_media_player_t_ptr;
    mouse_on : LongWord
  ); cdecl;


(**
 * Get the pixel dimensions of a video.
 *
 * param p_mi media player
 * param num number of the video (starting from, and most commonly 0)
 * param px pointer to get the pixel width [OUT]
 * param py pointer to get the pixel height [OUT]
 * return  0 on success, -1 if the specified video does not exist
 *)

var
  libvlc_video_get_size : function(
    p_mi : libvlc_media_player_t_ptr;
    num  : LongWord;
    var px, py: LongWord
  ) : Integer; cdecl;

{$IFDEF USE_VLC_DEPRECATED_API}
(**
 * Get current video height.
 * You should use libvlc_video_get_size() instead.
 *
 * param p_mi the media player
 * return  the video pixel height or 0 if not applicable
 *)

var
  libvlc_video_get_height : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;
{$ENDIF}

{$IFDEF USE_VLC_DEPRECATED_API}
(**
 * Get current video width.
 * You should use libvlc_video_get_size() instead.
 *
 * param p_mi the media player
 * return  the video pixel width or 0 if not applicable
 *)

var
  libvlc_video_get_width : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;
{$ENDIF}

(**
 * Get the mouse pointer coordinates over a video.
 * Coordinates are expressed in terms of the decoded video resolution,
 * NOT in terms of pixels on the screen/viewport (to get the latter,
 * you can query your windowing system directly).
 *
 * Either of the coordinates may be negative or larger than the corresponding
 * dimension of the video, if the cursor is outside the rendering area.
 *
 * @warning The coordinates may be out-of-date if the pointer is not located
 * on the video rendering area. LibVLC does not track the pointer if it is
 * outside of the video widget.
 *
 * @note LibVLC does not support multiple pointers (it does of course support
 * multiple input devices sharing the same pointer) at the moment.
 *
 * param p_mi media player
 * param num number of the video (starting from, and most commonly 0)
 * param px pointer to get the abscissa [OUT]
 * param py pointer to get the ordinate [OUT]
 * return  0 on success, -1 if the specified video does not exist
 *)

var
  libvlc_video_get_cursor : function(
    p_mi : libvlc_media_player_t_ptr;
    num  :  LongWord;
    var px, py : Integer
  ) : Integer; cdecl;

(**
 * Get the current video scaling factor.
 * See also libvlc_video_set_scale().
 *
 * param p_mi the media player
 * return  the currently configured zoom factor, or 0. if the video is set
 * to fit to the output window/drawable automatically.
 *)

var
  libvlc_video_get_scale : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Single; cdecl;

(**
 * Set the video scaling factor. That is the ratio of the number of pixels on
 * screen to the number of pixels in the original decoded video in each
 * dimension. Zero is a special value; it will adjust the video to the output
 * window/drawable (in windowed mode) or the entire screen.
 *
 * Note that not all video outputs support scaling.
 *
 * param p_mi the media player
 * param f_factor the scaling factor, or zero
 *)

var
  libvlc_video_set_scale : procedure(
    p_mi     : libvlc_media_player_t_ptr;
    f_factor : Single // float
  ); cdecl;

(**
 * Get current video aspect ratio.
 *
 * param p_mi the media player
 * return  the video aspect ratio or NULL if unspecified
 * (the result must be released with free() or libvlc_free()).
 *)

var
  libvlc_video_get_aspect_ratio : function(
    p_mi : libvlc_media_player_t_ptr
  ) : PAnsiChar; cdecl;

(**
 * Set new video aspect ratio.
 *
 * param p_mi the media player
 * param psz_aspect new video aspect-ratio or NULL to reset to default
 * Invalid aspect ratios are ignored.
 *)

var
  libvlc_video_set_aspect_ratio : procedure(
    p_mi       : libvlc_media_player_t_ptr;
    psz_aspect : PAnsiChar
  ); cdecl;

(**
 * Get current video subtitle.
 *
 * param p_mi the media player
 * return  the video subtitle selected, or -1 if none
 *)

var
  libvlc_video_get_spu : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Get the number of available video subtitles.
 *
 * param p_mi the media player
 * return  the number of available video subtitles
 *)

var
  libvlc_video_get_spu_count : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Get the description of available video subtitles.
 *
 * param p_mi the media player
 * return  list containing description of available video subtitles
 *)

var
  libvlc_video_get_spu_description : function(
    p_mi : libvlc_media_player_t_ptr
  ) : libvlc_track_description_t_ptr; cdecl;

(**
 * Set new video subtitle.
 *
 * param p_mi the media player
 * param i_spu video subtitle track to select (i_id from track description)
 * return  0 on success, -1 if out of range
 *)

var
  libvlc_video_set_spu : function(
    p_mi  : libvlc_media_player_t_ptr;
    i_spu : Integer
  ) : Integer; cdecl;

(**
 * Set new video subtitle file.
 *
 * param p_mi the media player
 * param psz_subtitle new video subtitle file
 * return  the success status (boolean)
 *)

var
  libvlc_video_set_subtitle_file : function(
    p_mi         : libvlc_media_player_t_ptr;
    psz_subtitle : PAnsiChar
  ) : Integer; cdecl;

(**
 * Get the current subtitle delay. Positive values means subtitles are being
 * displayed later, negative values earlier.
 *
 * param p_mi media player
 * return time (in microseconds) the display of subtitles is being delayed
 * version LibVLC 2.0.0 or later
 *)
var
 libvlc_video_get_spu_delay : function(
   p_mi : libvlc_media_player_t_ptr
 ): Int64; cdecl;

(**
 * Set the subtitle delay. This affects the timing of when the subtitle will
 * be displayed. Positive values result in subtitles being displayed later,
 * while negative values will result in subtitles being displayed earlier.
 *
 * The subtitle delay will be reset to zero each time the media changes.
 *
 * param p_mi media player
 * param i_delay time (in microseconds) the display of subtitles should be delayed
 * return 0 on success, -1 on error
 * version LibVLC 2.0.0 or later
 *)
var
  libvlc_video_set_spu_delay : function (
    p_mi : libvlc_media_player_t_ptr;
    i_delay : Int64
  ) : Integer; cdecl;


(**
 * Get the description of available titles.
 *
 * param p_mi the media player
 * return  list containing description of available titles
 *)

var
  libvlc_video_get_title_description : function(
    p_mi : libvlc_media_player_t_ptr
  ) : libvlc_track_description_t_ptr; cdecl;

(**
 * Get the description of available chapters for specific title.
 *
 * param p_mi the media player
 * param i_title selected title
 * return  list containing description of available chapter for title i_title
 *)

var
  libvlc_video_get_chapter_description : function(
    p_mi    : libvlc_media_player_t_ptr;
    i_title : Integer
  ) : libvlc_track_description_t_ptr; cdecl;

(**
 * Get current crop filter geometry.
 *
 * param p_mi the media player
 * return  the crop filter geometry or NULL if unset
 *)

var
  libvlc_video_get_crop_geometry : function(
    p_mi : libvlc_media_player_t_ptr
  ) : PAnsiChar; cdecl;

(**
 * Set new crop filter geometry.
 *
 * param p_mi the media player
 * param psz_geometry new crop filter geometry (NULL to unset)
 *)

var
  libvlc_video_set_crop_geometry : procedure(
    p_mi         : libvlc_media_player_t_ptr;
    psz_geometry : PAnsiChar
  ); cdecl;

(**
 * Get current teletext page requested.
 *
 * param p_mi the media player
 * return  the current teletext page requested.
 *)

var
  libvlc_video_get_teletext : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Set new teletext page to retrieve.
 *
 * param p_mi the media player
 * param i_page teletex page number requested
 *)

var
  libvlc_video_set_teletext : procedure(
    p_mi   : libvlc_media_player_t_ptr;
    i_page : Integer
  ); cdecl;

(**
 * Toggle teletext transparent status on video output.
 *
 * param p_mi the media player
 *)

var
  libvlc_toggle_teletext : procedure(
    p_mi : libvlc_media_player_t_ptr
  ); cdecl;

(**
 * Get number of available video tracks.
 *
 * param p_mi media player
 * return  the number of available video tracks (int)
 *)

var
  libvlc_video_get_track_count : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Get the description of available video tracks.
 *
 * param p_mi media player
 * return  list with description of available video tracks, or NULL on error
 *)

var
  libvlc_video_get_track_description : function(
    p_mi : libvlc_media_player_t_ptr
  ) : libvlc_track_description_t_ptr; cdecl;

(**
 * Get current video track.
 *
 * param p_mi media player
 * return  the video track ID (int) or -1 if no active input
 *)

var
  libvlc_video_get_track : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Set video track.
 *
 * param p_mi media player
 * param i_track the track ID (i_id field from track description)
 * return  0 on success, -1 if out of range
 *)

var
  libvlc_video_set_track : function(
    p_mi    : libvlc_media_player_t_ptr;
    i_track : Integer
  ) : Integer; cdecl;

(**
 * Take a snapshot of the current video window.
 *
 * If i_width AND i_height is 0, original size is used.
 * If i_width XOR i_height is 0, original aspect-ratio is preserved.
 *
 * param p_mi media player instance
 * param num number of video output (typically 0 for the first/only one)
 * param psz_filepath the path where to save the screenshot to
 * param i_width the snapshot's width
 * param i_height the snapshot's height
 * return  0 on success, -1 if the video was not found
 *)

var
  libvlc_video_take_snapshot : function(
    p_mi         : libvlc_media_player_t_ptr;
    num          : Integer;
    psz_filepath : PAnsiChar;
    i_width      : LongWord;
    i_height     : LongWord
  ) : Integer; cdecl;

(**
 * Enable or disable deinterlace filter
 *
 * param p_mi libvlc media player
 * param psz_mode type of deinterlace filter, NULL to disable
 *)

var
  libvlc_video_set_deinterlace : procedure(
    p_mi     : libvlc_media_player_t_ptr;
    psz_mode : PAnsiChar
  ); cdecl;

(**
 * Get an integer marquee option value
 *
 * param p_mi libvlc media player
 * param option marq option to get see libvlc_video_marquee_int_option_t
 *)

var
  libvlc_video_get_marquee_int : function(
    p_mi   : libvlc_media_player_t_ptr;
    option : libvlc_video_marquee_option_t
  ) : Integer; cdecl;

(**
 * Get a string marquee option value
 *
 * param p_mi libvlc media player
 * param option marq option to get
 *)

var
  libvlc_video_get_marquee_string : function(
    p_mi   : libvlc_media_player_t_ptr;
    option : libvlc_video_marquee_option_t
  ) : PAnsiChar; cdecl;

(**
 * Enable, disable or set an integer marquee option
 *
 * Setting libvlc_marquee_Enable has the side effect of enabling (arg !0)
 * or disabling (arg 0) the marq filter.
 *
 * param p_mi libvlc media player
 * param option marq option to set
 * param i_val marq option value
 *)

var
  libvlc_video_set_marquee_int : procedure(
    p_mi   : libvlc_media_player_t_ptr;
    option : libvlc_video_marquee_option_t;
    i_val  : Integer
  ); cdecl;

(**
 * Set a marquee string option
 *
 * param p_mi libvlc media player
 * param option marq option to set see libvlc_video_marquee_string_option_t
 * param psz_text marq option value
 *)

var
  libvlc_video_set_marquee_string : procedure(
    p_mi     : libvlc_media_player_t_ptr;
    option   : libvlc_video_marquee_option_t;
    psz_text : PAnsiChar
  ); cdecl;

(**
 * Get integer logo option.
 *
 * param p_mi libvlc media player instance
 * param option logo option to get, values of libvlc_video_logo_option_t
 *)

var
  libvlc_video_get_logo_int : function(
    p_mi   : libvlc_media_player_t_ptr;
    option : libvlc_video_logo_option_t
  ) : Integer; cdecl;

(**
 * Set logo option as integer. Options that take a different type value
 * are ignored.
 * Passing libvlc_logo_enable as option value has the side effect of
 * starting (arg !0) or stopping (arg 0) the logo filter.
 *
 * param p_mi libvlc media player instance
 * param option logo option to set, values of libvlc_video_logo_option_t
 * param value logo option value
 *)

var
  libvlc_video_set_logo_int : procedure(
    p_mi   : libvlc_media_player_t_ptr;
    option : libvlc_video_logo_option_t;
    value  : Integer
  ); cdecl;

(**
 * Set logo option as string. Options that take a different type value
 * are ignored.
 *
 * param p_mi libvlc media player instance
 * param option logo option to set, values of libvlc_video_logo_option_t
 * param psz_value logo option value
 *)

var
  libvlc_video_set_logo_string : procedure(
    p_mi      : libvlc_media_player_t_ptr;
    option    : libvlc_video_logo_option_t;
    psz_value : PAnsiChar
  ); cdecl;

(** option values for libvlc_video_{get,set}_adjust_{int,float,bool} *)
type
  libvlc_video_adjust_option_t = (
    libvlc_adjust_Enable,    // 0
    libvlc_adjust_Contrast,
    libvlc_adjust_Brightness,
    libvlc_adjust_Hue,
    libvlc_adjust_Saturation,
    libvlc_adjust_Gamma
  );

(**
 * Get integer adjust option.
 *
 * param p_mi libvlc media player instance
 * param option adjust option to get, values of libvlc_video_adjust_option_t
 * version LibVLC 1.1.1 and later.
 *)
var
  libvlc_video_get_adjust_int : function(
    p_mi   : libvlc_media_player_t_ptr;
    option : LongWord
  ) : Integer; cdecl;

(**
 * Set adjust option as integer. Options that take a different type value
 * are ignored.
 * Passing libvlc_adjust_enable as option value has the side effect of
 * starting (arg !0) or stopping (arg 0) the adjust filter.
 *
 * param p_mi libvlc media player instance
 * param option adust option to set, values of libvlc_video_adjust_option_t
 * param value adjust option value
 * version LibVLC 1.1.1 and later.
 *)
var
  libvlc_video_set_adjust_int : procedure(
    p_mi   : libvlc_media_player_t_ptr;
    option : LongWord;
    value  : Integer
  ); cdecl;

(**
 * Get float adjust option.
 *
 * param p_mi libvlc media player instance
 * param option adjust option to get, values of libvlc_video_adjust_option_t
 * version LibVLC 1.1.1 and later.
 *)
var
  libvlc_video_get_adjust_float : function(
    p_mi   : libvlc_media_player_t_ptr;
    option : LongWord
  ) : Single; cdecl;

(**
 * Set adjust option as float. Options that take a different type value
 * are ignored.
 *
 * param p_mi libvlc media player instance
 * param option adust option to set, values of libvlc_video_adjust_option_t
 * param value adjust option value
 * version LibVLC 1.1.1 and later.
 *)
var
  libvlc_video_set_adjust_float : procedure(
    p_mi   : libvlc_media_player_t_ptr;
    option : LongWord;
    value  : Single
  ); cdecl;

(**
 * Gets the list of available audio output modules
 *
 * param p_instance libvlc instance
 * return  list of available audio outputs. It must be freed it with
 * see libvlc_audio_output_list_release see libvlc_audio_output_t .
 * In case of error, NULL is returned.
 *)

var
  libvlc_audio_output_list_get : function(
    p_instance : libvlc_instance_t_ptr
  ) : libvlc_audio_output_t_ptr; cdecl;

(**
 * Free the list of available audio output modules
 *
 * param p_list list with audio outputs for release
 *)

var
  libvlc_audio_output_list_release : procedure(
    p_list : libvlc_audio_output_t_ptr
  ); cdecl;

(**
 * Selects an audio output module.
 * note Any change will take be effect only after playback is stopped and
 * restarted. Audio output cannot be changed while playing.
 *
 * param p_mi media player
 * param psz_name name of audio output,
 *               use psz_name of see libvlc_audio_output_t
 * return  0 if function succeded, -1 on error
 *)

var
  libvlc_audio_output_set : function(
    p_mi     : libvlc_media_player_t_ptr;
    psz_name : PAnsiChar
  ) : Integer; cdecl;

(**
 * Get count of devices for audio output, these devices are hardware oriented
 * like analog or digital output of sound card
 *
 * param p_instance libvlc instance
 * param psz_audio_output - name of audio output, see libvlc_audio_output_t
 * return  number of devices
 *)

{$IFDEF USE_VLC_DEPRECATED_API}
var
  libvlc_audio_output_device_count : function(
    p_instance       : libvlc_instance_t_ptr;
    psz_audio_output : PAnsiChar
  ) : Integer; cdecl;
{$ENDIF}

(**
 * Get long name of device, if not available short name given
 *
 * param p_instance libvlc instance
 * param psz_audio_output - name of audio output, see libvlc_audio_output_t
 * param i_device device index
 * return  long name of device
 *)

{$IFDEF USE_VLC_DEPRECATED_API}
var
  libvlc_audio_output_device_longname : function(
    p_instance       : libvlc_instance_t_ptr;
    psz_audio_output : PAnsiChar;
    i_device         : Integer
  ) : PAnsiChar; cdecl;
{$ENDIF}

(**
 * Get id name of device
 *
 * param p_instance libvlc instance
 * param psz_audio_output - name of audio output, see libvlc_audio_output_t
 * param i_device device index
 * return  id name of device, use for setting device, need to be free after use
 *)

{$IFDEF USE_VLC_DEPRECATED_API}
var
  libvlc_audio_output_device_id : function(
    p_instance       : libvlc_instance_t_ptr;
    psz_audio_output : PAnsiChar;
    i_device         : Integer
  ) : PAnsiChar; cdecl;
{$ENDIF}

(**
 * Gets a list of potential audio output devices,
 * see libvlc_audio_output_device_set().
 *
 * note Not all audio outputs support enumerating devices.
 * The audio output may be functional even if the list is empty (NULL).
 *
 * note The list may not be exhaustive.

 *
 * warning Some audio output devices in the list might not actually work in
 * some circumstances. By default, it is recommended to not specify any
 * explicit audio device.
 *
 * param mp media player


 * return A NULL-terminated linked list of potential audio output devices.
 * It must be freed it with libvlc_audio_output_device_list_release()
 * version LibVLC 2.2.0 or later.
 *)
var
  libvlc_audio_output_device_enum : function(
    p_mi : libvlc_media_player_t_ptr
  ) : libvlc_audio_output_device_t_ptr; cdecl;

(**
 * Gets a list of audio output devices for a given audio output.
 * see libvlc_audio_output_device_set().
 *
 * note Not all audio outputs support this. In particular, an empty (NULL)
 * list of devices does NOT imply that the specified audio output does
 * not work.
 *
 * note The list might not be exhaustive.
 *
 * warning Some audio output devices in the list might not actually work in
 * some circumstances. By default, it is recommended to not specify any
 * explicit audio device.
 *
 * param p_instance libvlc instance
 * param psz_aout audio output name
 *                 (as returned by libvlc_audio_output_list_get())
 * return  A NULL-terminated linked list of potential audio output devices.
 * It must be freed it with libvlc_audio_output_device_list_release()
 * version LibVLC 2.1.0 or later.
 *)
var
  libvlc_audio_output_device_list_get : function(
    p_instance : libvlc_instance_t_ptr;
    aout       : PAnsiChar
  ) : libvlc_audio_output_device_t_ptr; cdecl;


(**
 * Frees a list of available audio output devices.
 *
 * param p_list list with audio outputs for release
 * version LibVLC 2.1.0 or later.
 *)
var
  libvlc_audio_output_device_list_release : procedure(
    p_list : libvlc_audio_output_device_t_ptr
  ); cdecl;

(**
 * Configures an explicit audio output device.
 *
 * If the module paramater is NULL, audio output will be moved to the device
 * specified by the device identifier string immediately. This is the
 * recommended usage.
 *
 * A list of adequate potential device strings can be obtained with
 * libvlc_audio_output_device_enum().
 *
 * However passing NULL is supported in LibVLC version 2.2.0 and later only;
 * in earlier versions, this function would have no effects when the module
 * parameter was NULL.
 *
 * If the module parameter is not NULL, the device parameter of the
 * corresponding audio output, if it exists, will be set to the specified
 * string. Note that some audio output modules do not have such a parameter
 * (notably MMDevice and PulseAudio).
 *
 * A list of adequate potential device strings can be obtained with
 * libvlc_audio_output_device_list_get().
 *
 * note This function does not select the specified audio output plugin.
 * libvlc_audio_output_set() is used for that purpose.
 *
 * warning The syntax for the device parameter depends on the audio output.
 *
 * Some audio output modules require further parameters (e.g. a channels map
 * in the case of ALSA).
 *
 * param mp media player
 * param module If NULL, current audio output module.
 *               if non-NULL, name of audio output module
                 (see libvlc_audio_output_t)
 * param device_id device identifier string
 * return Nothing. Errors are ignored (this is a design bug).
 *)

var
  libvlc_audio_output_device_set : procedure(
    p_mi             : libvlc_media_player_t_ptr;
    psz_audio_output : PAnsiChar;
    psz_device_id    : PAnsiChar
  ); cdecl;

(**
 * Get current audio device type. Device type describes something like
 * character of output sound - stereo sound, 2.1, 5.1 etc
 *
 * param p_mi media player
 * return  the audio devices type see libvlc_audio_output_device_types_t
 *)

var
  libvlc_audio_output_get_device_type : function(
    p_mi : libvlc_media_player_t_ptr
  ) : libvlc_audio_output_device_types_t; cdecl;

(**
 * Set current audio device type.
 *
 * param p_mi vlc instance
 * param device_type the audio device type,
          according to see libvlc_audio_output_device_types_t
 *)

var
  libvlc_audio_output_set_device_type : procedure(
    p_mi        : libvlc_media_player_t_ptr;
    device_type : libvlc_audio_output_device_types_t
  ); cdecl;

(**
 * Toggle mute status.
 *
 * param p_mi media player
 * warning Toggling mute atomically is not always possible: On some platforms,
 * other processes can mute the VLC audio playback stream asynchronously. Thus,
 * there is a small race condition where toggling will not work.
 * See also the limitations of libvlc_audio_set_mute().
 *)

var
  libvlc_audio_toggle_mute : procedure(
    p_mi : libvlc_media_player_t_ptr
  ); cdecl;

(**
 * Get current mute status.
 *
 * param p_mi media player
 * return  the mute status (boolean) if defined, -1 if undefined/unapplicable
 *)

var
  libvlc_audio_get_mute : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Set mute status.
 *
 * param p_mi media player
 * param status If status is true then mute, otherwise unmute
 * warning This function does not always work. If there are no active audio
 * playback stream, the mute status might not be available. If digital
 * pass-through (S/PDIF, HDMI...) is in use, muting may be unapplicable. Also
 * some audio output plugins do not support muting at all.
 * note To force silent playback, disable all audio tracks. This is more
 * efficient and reliable than mute.
 *)

var
  libvlc_audio_set_mute : procedure(
    p_mi   : libvlc_media_player_t_ptr;
    status : Integer
  ); cdecl;

(**
 * Get current software audio volume.
 *
 * param p_mi media player
 * return the software volume in percents
 * (0 = mute, 100 = nominal / 0dB)
 *)

var
  libvlc_audio_get_volume : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Set current software audio volume.
 *
 * param p_mi media player
 * param i_volume the volume in percents (0 = mute, 100 = 0dB)
 * return 0 if the volume was set, -1 if it was out of range
 *)

var
  libvlc_audio_set_volume : function(
    p_mi     : libvlc_media_player_t_ptr;
    i_volume : Integer
  ) : Integer; cdecl;

(**
 * Get number of available audio tracks.
 *
 * param p_mi media player
 * return  the number of available audio tracks (int), or -1 if unavailable
 *)

var
  libvlc_audio_get_track_count : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Get the description of available audio tracks.
 *
 * param p_mi media player
 * return  list with description of available audio tracks, or NULL
 *)

var
  libvlc_audio_get_track_description : function(
    p_mi : libvlc_media_player_t_ptr
  ) : libvlc_track_description_t_ptr; cdecl;

(**
 * Get current audio track.
 *
 * param p_mi media player
 * return  the audio track ID or -1 if no active input.
 *)

var
  libvlc_audio_get_track : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;

(**
 * Set current audio track.
 *
 * param p_mi media player
 * param i_track the track ID (i_id field from track description)
 * return  0 on success, -1 on error
 *)

var
  libvlc_audio_set_track : function(
    p_mi    : libvlc_media_player_t_ptr;
    i_track : Integer
  ) : Integer; cdecl;

(**
 * Get current audio channel.
 *
 * param p_mi media player
 * return  the audio channel see libvlc_audio_output_channel_t
 *)

var
  libvlc_audio_get_channel : function(
    p_mi : libvlc_media_player_t_ptr
  ) : libvlc_audio_output_channel_t; cdecl;

(**
 * Set current audio channel.
 *
 * param p_mi media player
 * param channel the audio channel, see libvlc_audio_output_channel_t
 * return  0 on success, -1 on error
 *)

var
  libvlc_audio_set_channel : function(
    p_mi      : libvlc_media_player_t_ptr;
    i_channel : libvlc_audio_output_channel_t
  ) : Integer; cdecl;

(**
 * Get current audio delay.
 *
 * param p_mi media player
 * return the audio delay (microseconds)
 * version LibVLC 1.1.1 or later
 *)
var
  libvlc_audio_get_delay : function(
    p_mi : libvlc_media_player_t_ptr
  ) : Int64; cdecl;

(**
 * Set current audio delay. The audio delay will be reset to zero each time the media changes.
 *
 * param p_mi media player
 * param i_delay the audio delay (microseconds)
 * return 0 on success, -1 on error
 * version LibVLC 1.1.1 or later
 *)
var
  libvlc_audio_set_delay : function(
    p_mi    : libvlc_media_player_t_ptr;
    i_delay : Int64
  ) : Integer; cdecl;
  
(**
 * Get the number of equalizer presets.
 *
 * return number of presets
 * version LibVLC 2.2.0 or later
 *)
var
  libvlc_audio_equalizer_get_preset_count : function() : unsigned_t; cdecl; 

(**
 * Get the name of a particular equalizer preset.
 *
 * This name can be used, for example, to prepare a preset label or menu in a user
 * interface.
 *
 * param u_index index of the preset, counting from zero
 * return preset name, or NULL if there is no such preset
 * version LibVLC 2.2.0 or later
 *)
var
  libvlc_audio_equalizer_get_preset_name : function(
    u_index : unsigned_t
  ): PAnsiChar; cdecl;

(**
 * Get the number of distinct frequency bands for an equalizer.
 *
 * return number of frequency bands
 * version LibVLC 2.2.0 or later
 *)
var
  libvlc_audio_equalizer_get_band_count : function() : unsigned_t; cdecl;

(**
 * Get a particular equalizer band frequency.
 *
 * This value can be used, for example, to create a label for an equalizer band control
 * in a user interface.
 *
 * param u_index index of the band, counting from zero
 * return equalizer band frequency (Hz), or -1 if there is no such band
 * version LibVLC 2.2.0 or later
 *)
var
  libvlc_audio_equalizer_get_band_frequency : function(
    u_index : unsigned_t
  ) : Single; cdecl; // float 

(**
 * Create a new default equalizer, with all frequency values zeroed.
 *
 * The new equalizer can subsequently be applied to a media player by invoking
 * libvlc_media_player_set_equalizer().
 *
 * The returned handle should be freed via libvlc_audio_equalizer_release() when
 * it is no longer needed.
 *
 * return opaque equalizer handle, or NULL on error
 * version LibVLC 2.2.0 or later
 *)
var
  libvlc_audio_equalizer_new : function() : libvlc_equalizer_t_ptr; cdecl;

(**
 * Create a new equalizer, with initial frequency values copied from an existing
 * preset.
 *
 * The new equalizer can subsequently be applied to a media player by invoking
 * libvlc_media_player_set_equalizer().
 *
 * The returned handle should be freed via libvlc_audio_equalizer_release() when
 * it is no longer needed.
 *
 * param u_index index of the preset, counting from zero
 * return opaque equalizer handle, or NULL on error
 * version LibVLC 2.2.0 or later
 *)
var
  libvlc_audio_equalizer_new_from_preset : function(
    u_index : unsigned_t
  ) : libvlc_equalizer_t_ptr; cdecl;

(**
 * Release a previously created equalizer instance.
 *
 * The equalizer was previously created by using libvlc_audio_equalizer_new() or
 * libvlc_audio_equalizer_new_from_preset().
 *
 * It is safe to invoke this method with a NULL p_equalizer parameter for no effect.
 *
 * param p_equalizer opaque equalizer handle, or NULL
 * version LibVLC 2.2.0 or later
 *)
var
  libvlc_audio_equalizer_release : procedure(
    p_equalizer : libvlc_equalizer_t_ptr
  ); cdecl;

(**
 * Set a new pre-amplification value for an equalizer.
 *
 * The new equalizer settings are subsequently applied to a media player by invoking
 * libvlc_media_player_set_equalizer().
 *
 * The supplied amplification value will be clamped to the -20.0 to +20.0 range.
 *
 * param p_equalizer valid equalizer handle, must not be NULL
 * param f_preamp preamp value (-20.0 to 20.0 Hz)
 * return zero on success, -1 on error
 * version LibVLC 2.2.0 or later
 *)
var
  libvlc_audio_equalizer_set_preamp : function(
    p_equalizer : libvlc_equalizer_t_ptr;
    f_preamp    : Single // float
  ) : Integer; cdecl;

(**
 * Get the current pre-amplification value from an equalizer.
 *
 * param p_equalizer valid equalizer handle, must not be NULL
 * return preamp value (Hz)
 * version LibVLC 2.2.0 or later
 *)
var
  libvlc_audio_equalizer_get_preamp : function(
    p_equalizer : libvlc_equalizer_t_ptr
  ) : Single; cdecl; // float

(**
 * Set a new amplification value for a particular equalizer frequency band.
 *
 * The new equalizer settings are subsequently applied to a media player by invoking
 * libvlc_media_player_set_equalizer().
 *
 * The supplied amplification value will be clamped to the -20.0 to +20.0 range.
 *
 * param p_equalizer valid equalizer handle, must not be NULL
 * param f_amp amplification value (-20.0 to 20.0 Hz)
 * param u_band index, counting from zero, of the frequency band to set
 * return zero on success, -1 on error
 * version LibVLC 2.2.0 or later
 *)
var
  libvlc_audio_equalizer_set_amp_at_index : function(
    p_equalizer : libvlc_equalizer_t_ptr;
    f_amp : Single;// float
    u_band : unsigned_t
  ) : Integer; cdecl;

(**
 * Get the amplification value for a particular equalizer frequency band.
 *
 * param p_equalizer valid equalizer handle, must not be NULL
 * param u_band index, counting from zero, of the frequency band to get
 * return amplification value (Hz); NaN if there is no such frequency band
 * version LibVLC 2.2.0 or later
 *)
var
  libvlc_audio_equalizer_get_amp_at_index : function(
    p_equalizer : libvlc_equalizer_t_ptr;
    u_band : unsigned_t
  ) : Single; cdecl;

(**
 * Apply new equalizer settings to a media player.
 *
 * The equalizer is first created by invoking libvlc_audio_equalizer_new() or
 * libvlc_audio_equalizer_new_from_preset().
 *
 * It is possible to apply new equalizer settings to a media player whether the media
 * player is currently playing media or not.
 *
 * Invoking this method will immediately apply the new equalizer settings to the audio
 * output of the currently playing media if there is any.
 *
 * If there is no currently playing media, the new equalizer settings will be applied
 * later if and when new media is played.
 *
 * Equalizer settings will automatically be applied to subsequently played media.
 *
 * To disable the equalizer for a media player invoke this method passing NULL for the
 * p_equalizer parameter.
 *
 * The media player does not keep a reference to the supplied equalizer so it is safe
 * for an application to release the equalizer reference any time after this method
 * returns.
 *
 * param p_mi opaque media player handle
 * param p_equalizer opaque equalizer handle, or NULL to disable the equalizer for this media player
 * return zero on success, -1 on error
 * version LibVLC 2.2.0 or later
 *)
var
  libvlc_media_player_set_equalizer : function(
    p_mi        : libvlc_media_player_t_ptr;
    p_equalizer : libvlc_equalizer_t_ptr
  ) : Integer; cdecl;

(*
 *******************************************************************************
 * libvlc_media_list.h
 *******************************************************************************
 *)

(**
 * Create an empty media list.
 *
 * param p_instance libvlc instance
 * return  empty media list, or NULL on error
 *)

var
  libvlc_media_list_new : function(
    p_instance : libvlc_instance_t_ptr
  ) : libvlc_media_list_t_ptr; cdecl;

(**
 * Release media list created with libvlc_media_list_new().
 *
 * param p_ml a media list created with libvlc_media_list_new()
 *)

var
  libvlc_media_list_release : procedure(
    p_ml : libvlc_media_list_t_ptr
  ); cdecl;

(**
 * Retain reference to a media list
 *
 * param p_ml a media list created with libvlc_media_list_new()
 *)

var
  libvlc_media_list_retain : procedure(
    p_ml : libvlc_media_list_t_ptr
  ); cdecl;

{$IFDEF USE_VLC_DEPRECATED_API}
var
  libvlc_media_list_add_file_content:  function(
    p_ml    : libvlc_media_list_t_ptr;
    psz_uri : PAnsiChar
  ) : Integer; cdecl;
{$ENDIF}

(**
 * Associate media instance with this media list instance.
 * If another media instance was present it will be released.
 * The libvlc_media_list_lock should NOT be held upon entering this function.
 *
 * param p_ml a media list instance
 * param p_md media instance to add
 *)

var
  libvlc_media_list_set_media : procedure(
    p_ml : libvlc_media_list_t_ptr;
    p_md : libvlc_media_t_ptr
  ); cdecl;

(**
 * Get media instance from this media list instance. This action will increase
 * the refcount on the media instance.
 * The libvlc_media_list_lock should NOT be held upon entering this function.
 *
 * param p_ml a media list instance
 * return media instance
 *)

var
  libvlc_media_list_media : function(
    p_ml : libvlc_media_list_t_ptr
  ) : libvlc_media_t_ptr; cdecl;

(**
 * Add media instance to media list
 * The libvlc_media_list_lock should be held upon entering this function.
 *
 * param p_ml a media list instance
 * param p_md a media instance
 * return  0 on success, -1 if the media list is read-only
 *)

var
  libvlc_media_list_add_media : function(
    p_ml : libvlc_media_list_t_ptr;
    p_md : libvlc_media_t_ptr
  ) : Integer; cdecl;

(**
 * Insert media instance in media list on a position
 * The libvlc_media_list_lock should be held upon entering this function.
 *
 * param p_ml a media list instance
 * param p_md a media instance
 * param i_pos position in array where to insert
 * return  0 on success, -1 if the media list si read-only
 *)

var
  libvlc_media_list_insert_media : function(
    p_ml  : libvlc_media_list_t_ptr;
    p_md  : libvlc_media_t_ptr;
    i_pos : Integer
  ) : Integer; cdecl;

(**
 * Remove media instance from media list on a position
 * The libvlc_media_list_lock should be held upon entering this function.
 *
 * param p_ml a media list instance
 * param i_pos position in array where to insert
 * return  0 on success, -1 if the list is read-only or the item was not found
 *)

var
  libvlc_media_list_remove_index : function(
    p_ml  : libvlc_media_list_t_ptr;
    i_pos : Integer
  ) : Integer; cdecl;

(**
 * Get count on media list items
 * The libvlc_media_list_lock should be held upon entering this function.
 *
 * param p_ml a media list instance
 * return  number of items in media list
 *)

var
  libvlc_media_list_count : function(
    p_ml : libvlc_media_list_t_ptr
  ) : Integer; cdecl;

(**
 * List media instance in media list at a position
 * The libvlc_media_list_lock should be held upon entering this function.
 *
 * param p_ml a media list instance
 * param i_pos position in array where to insert
 * return  media instance at position i_pos, or -1 if not found.
 * In case of success, libvlc_media_retain() is called to increase the refcount
 * on the media.
 *)

var
  libvlc_media_list_item_at_index : function(
    p_ml  : libvlc_media_list_t_ptr;
    i_pos : Integer
  ) : libvlc_media_t_ptr; cdecl;

(**
 * Find index position of List media instance in media list.
 * Warning: the function will return the first matched position.
 * The libvlc_media_list_lock should be held upon entering this function.
 *
 * param p_ml a media list instance
 * param p_md media list instance
 * return  position of media instance
 *)

var
  libvlc_media_list_index_of_item : function(
    p_ml : libvlc_media_list_t_ptr;
    p_md : libvlc_media_t_ptr
  ) : Integer; cdecl;

(**
 * This indicates if this media list is read-only from a user point of view
 *
 * param p_ml media list instance
 * return  1 on readonly, 0 on readwrite
 *)

var
  libvlc_media_list_is_readonly : function(
    p_ml : libvlc_media_list_t_ptr
  ) : Integer; cdecl;

(**
 * Get lock on media list items
 *
 * param p_ml a media list instance
 *)

var
  libvlc_media_list_lock : procedure(
    p_ml : libvlc_media_list_t_ptr
  ); cdecl;

(**
 * Release lock on media list items
 * The libvlc_media_list_lock should be held upon entering this function.
 *
 * param p_ml a media list instance
 *)

var
  libvlc_media_list_unlock : procedure(
    p_ml : libvlc_media_list_t_ptr
  ); cdecl;

(**
 * Get libvlc_event_manager from this media list instance.
 * The p_event_manager is immutable, so you don't have to hold the lock
 *
 * param p_ml a media list instance
 * return  libvlc_event_manager
 *)

var
  libvlc_media_list_event_manager : function(
    p_ml : libvlc_media_list_t_ptr
  ) : libvlc_event_manager_t_ptr; cdecl;

(*******************************************************************************
 * libvlc_media_list_player.h
 *******************************************************************************
 *)

(**
 * The LibVLC media list player plays a @ref libvlc_media_list_t list of media,
 * in a certain order.
 * This is required to especially support playlist files.
 * The normal @ref libvlc_media_player_t LibVLC media player can only play a
 * single media, and does not handle playlist files properly.
 *)

(**
 * Create new media_list_player.
 *
 * param p_instance libvlc instance
 * return  media list player instance or NULL on error
 *)

var
  libvlc_media_list_player_new : function(
    p_instance : libvlc_instance_t_ptr
  ) : libvlc_media_list_player_t_ptr; cdecl;

(**
 * Release a media_list_player after use
 * Decrement the reference count of a media player object. If the
 * reference count is 0, then libvlc_media_list_player_release() will
 * release the media player object. If the media player object
 * has been released, then it should not be used again.
 *
 * param p_mlp media list player instance
 *)

var
  libvlc_media_list_player_release : procedure(
    p_mlp : libvlc_media_list_player_t_ptr
  ); cdecl;

(**
 * Retain a reference to a media player list object. Use
 * libvlc_media_list_player_release() to decrement reference count.
 *
 * param p_mlp media player list object
 *)
var
  libvlc_media_list_player_retain : procedure(
    p_mlp : libvlc_media_list_player_t_ptr
  ); cdecl;


(**
 * Return the event manager of this media_list_player.
 *
 * param p_mlp media list player instance
 * return  the event manager
 *)

var
  libvlc_media_list_player_event_manager : function(
    p_mlp : libvlc_media_list_player_t_ptr
  ) : libvlc_event_manager_t_ptr; cdecl;

(**
 * Replace media player in media_list_player with this instance.
 *
 * param p_mlp media list player instance
 * param p_mi media player instance
 *)

var
  libvlc_media_list_player_set_media_player : procedure(
    p_mlp : libvlc_media_list_player_t_ptr;
    p_mi  : libvlc_media_player_t_ptr
  ); cdecl;

(**
 * Set the media list associated with the player
 *
 * param p_mlp media list player instance
 * param p_mlist list of media
 *)

var
  libvlc_media_list_player_set_media_list : procedure(
    p_mlp   : libvlc_media_list_player_t_ptr;
    p_mlist : libvlc_media_list_t_ptr
  ); cdecl;

(**
 * Play media list
 *
 * param p_mlp media list player instance
 *)

var
  libvlc_media_list_player_play : procedure(
    p_mlp : libvlc_media_list_player_t_ptr
  ); cdecl;

(**
 * Toggle pause (or resume) media list
 *
 * param p_mlp media list player instance
 *)

var
  libvlc_media_list_player_pause : procedure(
    p_mlp : libvlc_media_list_player_t_ptr
  ); cdecl;

(**
 * Is media list playing?
 *
 * param p_mlp media list player instance
 * return  true for playing and false for not playing
 *)

var
  libvlc_media_list_player_is_playing : function(
    p_mlp : libvlc_media_list_player_t_ptr
  ) : Integer; cdecl;

(**
 * Get current libvlc_state of media list player
 *
 * param p_mlp media list player instance
 * return  libvlc_state_t for media list player
 *)

var
  libvlc_media_list_player_get_state : function(
    p_mlp : libvlc_media_list_player_t_ptr
  ) : libvlc_state_t; cdecl;

(**
 * Play media list item at position index
 *
 * param p_mlp media list player instance
 * param i_index index in media list to play
 * return  0 upon success -1 if the item wasn't found
 *)

var
  libvlc_media_list_player_play_item_at_index : function(
    p_mlp   : libvlc_media_list_player_t_ptr;
    i_index : Integer
  ) : Integer; cdecl;

(**
 * Play the given media item
 *
 * param p_mlp media list player instance
 * param p_md the media instance
 * return  0 upon success, -1 if the media is not part of the media list
 *)

var
  libvlc_media_list_player_play_item : function(
    p_mlp : libvlc_media_list_player_t_ptr;
    p_md  : libvlc_media_t_ptr
  ) : Integer; cdecl;

(**
 * Stop playing media list
 *
 * param p_mlp media list player instance
 *)

var
  libvlc_media_list_player_stop : procedure(
    p_mlp : libvlc_media_list_player_t_ptr
  ); cdecl;

(**
 * Play next item from media list
 *
 * param p_mlp media list player instance
 * return  0 upon success -1 if there is no next item
 *)

var
  libvlc_media_list_player_next : function(
    p_mlp : libvlc_media_list_player_t_ptr
  ) : Integer; cdecl;

(**
 * Play previous item from media list
 *
 * param p_mlp media list player instance
 * return  0 upon success -1 if there is no previous item
 *)

var
  libvlc_media_list_player_previous : function(
    p_mlp : libvlc_media_list_player_t_ptr
  ) : Integer; cdecl;

(**
 * Sets the playback mode for the playlist
 *
 * param p_mlp media list player instance
 * param e_mode playback mode specification
 *)

var
  libvlc_media_list_player_set_playback_mode : procedure(
    p_mlp  : libvlc_media_list_player_t_ptr;
    e_mode : libvlc_playback_mode_t
  ); cdecl;

(*
 *******************************************************************************
 * libvlc_media_library.h
 *******************************************************************************
 *)

(**
 * Create an new Media Library object
 *
 * param p_instance the libvlc instance
 * return  a new object or NULL on error
 *)

var
  libvlc_media_library_new : function(
    p_instance : libvlc_instance_t_ptr
  ) : libvlc_media_library_t_ptr; cdecl;

(**
 * Release media library object. This functions decrements the
 * reference count of the media library object. If it reaches 0,
 * then the object will be released.
 *
 * param p_mlib media library object
 *)

var
  libvlc_media_library_release : procedure(
    p_mlib : libvlc_media_library_t_ptr
  ); cdecl;

(**
 * Retain a reference to a media library object. This function will
 * increment the reference counting for this object. Use
 * libvlc_media_library_release() to decrement the reference count.
 *
 * param p_mlib media library object
 *)

var
  libvlc_media_library_retain : procedure(
    p_mlib : libvlc_media_library_t_ptr
  ); cdecl;

(**
 * Load media library.
 *
 * param p_mlib media library object
 * return  0 on success, -1 on error
 *)

var
  libvlc_media_library_load : function(
    p_mlib : libvlc_media_library_t_ptr
  ) : Integer; cdecl;

(**
 * Get media library subitems.
 *
 * param p_mlib media library object
 * return  media list subitems
 *)

var
  libvlc_media_library_media_list : function(
    p_mlib : libvlc_media_library_t_ptr
  ) : libvlc_media_list_t_ptr; cdecl;

(*******************************************************************************
 * libvlc_media_discoverer.h
 *******************************************************************************
 *)

(**
 * LibVLC media discovery finds available media via various means.
 * This corresponds to the service discovery functionality in VLC media player.
 * Different plugins find potential medias locally (e.g. user media directory),
 * from peripherals (e.g. video capture device), on the local network
 * (e.g. SAP) or on the Internet (e.g. Internet radios).
 *)

(**
 * Discover media service by name.
 *
 * param p_inst libvlc instance
 * param psz_name service name
 * return  media discover object or NULL in case of error
 *)

var
  libvlc_media_discoverer_new_from_name : function(
    p_inst   : libvlc_instance_t_ptr;
    psz_name : PAnsiChar
  ) : libvlc_media_discoverer_t_ptr; cdecl;

(**
 * Release media discover object. If the reference count reaches 0, then
 * the object will be released.
 *
 * param p_mdis media service discover object
 *)

var
  libvlc_media_discoverer_release : procedure(
    p_mdis : libvlc_media_discoverer_t_ptr
  ); cdecl;

(**
 * Get media service discover object its localized name.
 *
 * param p_mdis media discover object
 * return  localized name
 *)

var
  libvlc_media_discoverer_localized_name : function(
    p_mdis : libvlc_media_discoverer_t_ptr
  ) : PAnsiChar; cdecl;

(**
 * Get media service discover media list.
 *
 * param p_mdis media service discover object
 * return  list of media items
 *)

var
  libvlc_media_discoverer_media_list : function(
    p_mdis : libvlc_media_discoverer_t_ptr
  ) : libvlc_media_list_t_ptr; cdecl;

(**
 * Get event manager from media service discover object.
 *
 * param p_mdis media service discover object
 * return  event manager object.
 *)

var
  libvlc_media_discoverer_event_manager : function(
    p_mdis : libvlc_media_discoverer_t_ptr
  ) : libvlc_event_manager_t_ptr; cdecl;

(**
 * Query if media service discover object is running.
 *
 * param p_mdis media service discover object
 * return  true if running, false if not
 *)

var
  libvlc_media_discoverer_is_running : function(
    p_mdis : libvlc_media_discoverer_t_ptr
  ) : Integer; cdecl;

(*
 *******************************************************************************
 * libvlc_vlm.h
 *******************************************************************************
 *)

(**
 * Release the vlm instance related to the given libvlc_instance_t
 *
 * param p_instance the instance
 *)

var
  libvlc_vlm_release : procedure(
    p_instance : libvlc_instance_t_ptr
  ); cdecl;

(**
 * Add a broadcast, with one input.
 *
 * param p_instance the instance
 * param psz_name the name of the new broadcast
 * param psz_input the input MRL
 * param psz_output the output MRL (the parameter to the "sout" variable)
 * param i_options number of additional options
 * param ppsz_options additional options
 * param b_enabled boolean for enabling the new broadcast
 * param b_loop Should this broadcast be played in loop ?
 * return  0 on success, -1 on error
 *)

var
  libvlc_vlm_add_broadcast : function(
    p_instance   : libvlc_instance_t_ptr;
    psz_name     : PAnsiChar;
    psz_input    : PAnsiChar;
    psz_output   : PAnsiChar;
    i_options    : Integer;
    ppsz_options : PPAnsiChar;
    b_enabled    : Integer;
    b_loop       : Integer
  ) : Integer; cdecl;

(**
 * Add a vod, with one input.
 *
 * param p_instance the instance
 * param psz_name the name of the new vod media
 * param psz_input the input MRL
 * param i_options number of additional options
 * param ppsz_options additional options
 * param b_enabled boolean for enabling the new vod
 * param psz_mux the muxer of the vod media
 * return  0 on success, -1 on error
 *)

var
  libvlc_vlm_add_vod : function(
    p_instance   : libvlc_instance_t_ptr;
    psz_name     : PAnsiChar;
    psz_input    : PAnsiChar;
    i_options    : Integer;
    ppsz_options : PPAnsiChar;
    b_enabled    : Integer;
    psz_mux      : PAnsiChar
  ) : Integer; cdecl;

(**
 * Delete a media (VOD or broadcast).
 *
 * param p_instance the instance
 * param psz_name the media to delete
 * return  0 on success, -1 on error
 *)

var
  libvlc_vlm_del_media : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar
  ) : Integer; cdecl;

(**
 * Enable or disable a media (VOD or broadcast).
 *
 * param p_instance the instance
 * param psz_name the media to work on
 * param b_enabled the new status
 * return  0 on success, -1 on error
 *)

var
  libvlc_vlm_set_enabled : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar;
    b_enabled  : Integer
  ) : Integer; cdecl;

(**
 * Set the output for a media.
 *
 * param p_instance the instance
 * param psz_name the media to work on
 * param psz_output the output MRL (the parameter to the "sout" variable)
 * return  0 on success, -1 on error
 *)

var
 libvlc_vlm_set_output : function(
   p_instance : libvlc_instance_t_ptr;
   psz_name   : PAnsiChar;
   psz_output : PAnsiChar
 ) : Integer; cdecl;

(**
 * Set a media's input MRL. This will delete all existing inputs and
 * add the specified one.
 *
 * param p_instance the instance
 * param psz_name the media to work on
 * param psz_input the input MRL
 * return  0 on success, -1 on error
 *)

 var
  libvlc_vlm_set_input : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar;
    psz_input  : PAnsiChar
  ) : Integer; cdecl;

(**
 * Add a media's input MRL. This will add the specified one.
 *
 * param p_instance the instance
 * param psz_name the media to work on
 * param psz_input the input MRL
 * return  0 on success, -1 on error
 *)

var
  libvlc_vlm_add_input : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar;
    psz_input  : PAnsiChar
  ) : Integer; cdecl;

(**
 * Set a media's loop status.
 *
 * param p_instance the instance
 * param psz_name the media to work on
 * param b_loop the new status
 * return  0 on success, -1 on error
 *)

var
  libvlc_vlm_set_loop : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar;
    b_loop     : Integer
  ) : Integer; cdecl;

(**
 * Set a media's vod muxer.
 *
 * param p_instance the instance
 * param psz_name the media to work on
 * param psz_mux the new muxer
 * return 0 on success, -1 on error
 *)

var
  libvlc_vlm_set_mux : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar;
    psz_mux    : PAnsiChar
  ) : Integer; cdecl;

(**
 * Edit the parameters of a media. This will delete all existing inputs and
 * add the specified one.
 *
 * param p_instance the instance
 * param psz_name the name of the new broadcast
 * param psz_input the input MRL
 * param psz_output the output MRL (the parameter to the "sout" variable)
 * param i_options number of additional options
 * param ppsz_options additional options
 * param b_enabled boolean for enabling the new broadcast
 * param b_loop Should this broadcast be played in loop ?
 * return  0 on success, -1 on error
 *)

var
  libvlc_vlm_change_media : function(
    p_instance   : libvlc_instance_t_ptr;
    psz_name     : PAnsiChar;
    psz_input    : PAnsiChar;
    psz_output   : PAnsiChar;
    i_options    : Integer;
    ppsz_options : PPAnsiChar;
    b_enabled    : Integer;
    b_loop       : Integer
  ) : Integer; cdecl;  

(**
 * Play the named broadcast.
 *
 * param p_instance the instance
 * param psz_name the name of the broadcast
 * return  0 on success, -1 on error
 *)

var
  libvlc_vlm_play_media : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar
  ) : Integer; cdecl;

(**
 * Stop the named broadcast.
 *
 * param p_instance the instance
 * param psz_name the name of the broadcast
 * return  0 on success, -1 on error
 *)

var
  libvlc_vlm_stop_media : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar
  ) : Integer; cdecl;

(**
 * Pause the named broadcast.
 *
 * param p_instance the instance
 * param psz_name the name of the broadcast
 * return  0 on success, -1 on error
 *)

var
  libvlc_vlm_pause_media : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar
  ) : Integer; cdecl;

(**
 * Seek in the named broadcast.
 *
 * param p_instance the instance
 * param psz_name the name of the broadcast
 * param f_percentage the percentage to seek to
 * return  0 on success, -1 on error
 *)

var
  libvlc_vlm_seek_media : function(
    p_instance   : libvlc_instance_t_ptr;
    psz_name     : PAnsiChar;
    f_percentage : Single // float
  ) : Integer; cdecl;


(**
 * Return information about the named media as a JSON
 * string representation.
 *
 * This function is mainly intended for debugging use,
 * if you want programmatic access to the state of
 * a vlm_media_instance_t, please use the corresponding
 * libvlc_vlm_get_media_instance_xxx -functions.
 * Currently there are no such functions available for
 * vlm_media_t though.
 *
 * param p_instance the instance
 * param psz_name the name of the media,
 *      if the name is an empty string, all media is described
 * return  string with information about named media, or NULL on error
 *)

var
  libvlc_vlm_show_media : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar
  ) : PAnsiChar; cdecl;

(**
 * Get vlm_media instance position by name or instance id
 *
 * param p_instance a libvlc instance
 * param psz_name name of vlm media instance
 * param i_instance instance id
 * return  position as float or -1. on error
 *)

var
  libvlc_vlm_get_media_instance_position : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar;
    i_instance : Integer
  ) : Single; cdecl;

(**
 * Get vlm_media instance time by name or instance id
 *
 * param p_instance a libvlc instance
 * param psz_name name of vlm media instance
 * param i_instance instance id
 * return  time as integer or -1 on error
 *)

var
  libvlc_vlm_get_media_instance_time : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar;
    i_instance : Integer
  ) : Integer; cdecl;

(**
 * Get vlm_media instance length by name or instance id
 *
 * param p_instance a libvlc instance
 * param psz_name name of vlm media instance
 * param i_instance instance id
 * return  length of media item or -1 on error
 *)

var
  libvlc_vlm_get_media_instance_length : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar;
    i_instance : Integer
  ) : Integer; cdecl;

(**
 * Get vlm_media instance playback rate by name or instance id
 *
 * param p_instance a libvlc instance
 * param psz_name name of vlm media instance
 * param i_instance instance id
 * return  playback rate or -1 on error
 *)

var
  libvlc_vlm_get_media_instance_rate : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar;
    i_instance : Integer
  ) : Integer; cdecl;

{$IFDEF IS_0_GT_1}
(**
 * Get vlm_media instance title number by name or instance id
 * bug will always return 0
 *
 * param p_instance a libvlc instance
 * param psz_name name of vlm media instance
 * param i_instance instance id
 * return  title as number or -1 on error
 *)

var
  libvlc_vlm_get_media_instance_title : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar;
    i_instance : Integer
  ) : Integer; cdecl;

(**
 * Get vlm_media instance chapter number by name or instance id
 * bug will always return 0
 *
 * param p_instance a libvlc instance
 * param psz_name name of vlm media instance
 * param i_instance instance id
 * return  chapter as number or -1 on error
 *)

var
  libvlc_vlm_get_media_instance_chapter : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar;
    i_instance : Integer
  ) : Integer; cdecl;

(**
 * Is libvlc instance seekable ?
 * bug will always return 0
 *
 * param p_instance a libvlc instance
 * param psz_name name of vlm media instance
 * param i_instance instance id
 * return  1 if seekable, 0 if not, -1 if media does not exist
 *)

var
  libvlc_vlm_get_media_instance_seekable : function(
    p_instance : libvlc_instance_t_ptr;
    psz_name   : PAnsiChar;
    i_instance : Integer
  ) : Integer; cdecl;
{$ENDIF}

(**
 * Get libvlc_event_manager from a vlm media.
 * The p_event_manager is immutable, so you don't have to hold the lock
 *
 * param p_instance a libvlc instance
 * return  libvlc_event_manager
 *)

var
  libvlc_vlm_get_event_manager : function(
    p_instance : libvlc_instance_t_ptr
  ) : libvlc_event_manager_t_ptr; cdecl;

(*
 *******************************************************************************
 * decpecated.h
 *******************************************************************************
 *)

{$IFDEF USE_VLC_DEPRECATED_API}

(**
 * Start playing (if there is any item in the playlist).
 *
 * Additionnal playlist item options can be specified for addition to the
 * item before it is played.
 *
 * param p_instance the playlist instance
 * param i_id the item to play. If this is a negative number, the next
 *       item will be selected. Otherwise, the item with the given ID will be
 *       played
 * param i_options the number of options to add to the item
 * param ppsz_options the options to add to the item
 *)

var
  libvlc_playlist_play : procedure(
    p_instance   : libvlc_instance_t_ptr;
    i_id         : Integer;
    i_options    : Integer;
    ppsz_options : PPAnsiChar
  ); cdecl;
  
{$ENDIF}

{$IFDEF MSWINDOWS}
  {$IFDEF LCLGTK2}

    function GDK_WINDOW_HWND(window: Pointer) : THandle;

  {$ENDIF}
{$ENDIF}

{$IFDEF MACOS}
procedure libvlc_media_player_set_display_window(
  p_mi : libvlc_media_player_t_ptr;
  window_handle : Pointer);
{$ELSE}
procedure libvlc_media_player_set_display_window(
  p_mi : libvlc_media_player_t_ptr;
  window_handle : THandle);
{$ENDIF}


{$IFNDEF HAS_UTF8_ENCODE_DECODE}

function Utf8Encode(const sWS: WideString): AnsiString;
function Utf8Decode(const sUTF8: AnsiString): WideString;

{$ENDIF}

implementation

{$IFDEF DELPHI_XE6_UP}
uses
	System.AnsiStrings;
{$ENDIF}

const
  {$IFDEF MSWINDOWS}
  LIBVLC_DLL_NAME = 'libvlc.dll';
  {$ENDIF}

  {$IFDEF MACOS}
  LIBVLC_DLL_NAME = 'libvlc.dylib';
  {$ENDIF}

  {$IFDEF UNIX}
  LIBVLC_DLL_NAME = 'libvlc.so';
  {$ENDIF}
  
var
  libvlc_handle : THandle;

function libvlc_delay(pts : Int64) : Int64;{$IFDEF DELPHI2005_UP}inline;{$ENDIF}
begin
  Result := pts - libvlc_clock();
end;

{$IFNDEF HAS_UTF8_ENCODE_DECODE}

// for Delphi 4 and Delphi 5

function Utf8Encode(const sWS: WideString): AnsiString;
var
  dLeng : Integer;
  sIndx : Integer;
  sCode : Word;
begin
  Result := '';
  if (Length(sWS) = 0) then exit;
  SetLength(Result, Length(sWS) * 3); // assume, max 3 bytes per char
  dLeng := 0;
  for sIndx := 1 to Length(sWS) do
  begin
    sCode := Ord(sWS[sIndx]);
    // 0x00 .. 0x7F - 0xxxxxxx
    if (sCode <= $007F) then // (sCode >= $0000) and
    begin
      Inc(dLeng);
      Result[dLeng] := AnsiChar(sCode and $7F);
    end
    // 0x80 .. 0x7FF - 110xxxxx 10xxxxxx
    else if (sCode >= $0080) and (sCode <= $07FF) then
    begin
      Inc(dLeng);
      Result[dLeng] := AnsiChar($C0 or ((sCode shr 6) and $1F));
      Inc(dLeng);
      Result[dLeng] := AnsiChar($80 or (sCode and $3F));
      continue;
    end
    // 0x800 .. 0xFFFF - 1110xxxx 10xxxxxx 10xxxxxx
    else if (sCode >= $0800) then // and (sCode <= $FFFF)
    begin // $0800 .. $FFFF
      Inc(dLeng);
      Result[dLeng] := AnsiChar($E0 or ((sCode shr 12) and $0F));
      Inc(dLeng);
      Result[dLeng] := AnsiChar($80 or ((sCode shr 6) and $3F));
      Inc(dLeng);
      Result[dLeng] := AnsiChar($80 or (sCode and $3F));
    end;
    // 0x0010000 .. 0x001FFFFF - 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
    // 0x0200000 .. 0x03FFFFFF - 111110xx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
    // 0x4000000 .. 0x7FFFFFFF - 1111110x 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
  end;
  // correct length
  SetLength(Result, dLeng);
end;

function Utf8Decode(const sUTF8: AnsiString): WideString;
var
  sTmp0 : Word;
  sTmp1 : Word;
  sTmp2 : Word;
  sTmp3 : Word;
  sTmp4 : Word;
  sTmp5 : Word;
  sIndx : Integer;
  sLeft : Integer;
  dLeng : Integer;
begin
  Result := '';
  if (Length(sUTF8) = 0) then exit;
  SetLength(Result, Length(sUTF8));
  sIndx := 1;
  sLeft := Length(sUTF8);
  dLeng := 0;
  while (sLeft > 0) do
  begin
    sTmp0 := Ord(sUTF8[sIndx]);
    Inc(sIndx);
    Dec(sLeft);
    // 1-byte code => 0x00 do 0x7F - 0xxxxxxx
    if ((sTmp0 and $80) = 0) then
    begin
      Inc(dLeng);
      Result[dLeng] := WideChar(sTmp0);
      continue;
    end;
    // 2-byte code => 0x80 .. 0x7FF - 110x xxxx 10xx xxxx
    if (sLeft < 1) then break;
    if ((sTmp0 and $E0) = $C0) then
    begin
      sTmp1 := Ord(sUTF8[sIndx]);
      if (sTmp1 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);
      
      Inc(dLeng);
      Result[dLeng] := WideChar(((sTmp0 and $1F) shl 6) or (sTmp1 and $3F));
      continue;
    end;
    // 3-byte code => 0x800 .. 0xFFFF - 1110 xxxx 10xx xxxx 10xx xxxx
    if (sLeft < 2) then break;
    if ((sTmp0 and $E0) = $E0) then
    begin
      sTmp1 := Ord(sUTF8[sIndx]);
      if (sTmp1 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);

      sTmp2 := Ord(sUTF8[sIndx]);
      if (sTmp2 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);
      
      Inc(dLeng);
      Result[dLeng] := WideChar(((sTmp0 and $0F) shl 12) or ((sTmp1 and $3F) shl 6) or (sTmp2 and $3F));
      continue;
    end;
    // 4-byte code => 0x0010000 .. 0x001FFFFF - 1111 0xxx 10xx xxxx 10xx xxxx 10xx xxxx
    if (sLeft < 3) then break;
    if ((sTmp0 and $F8) = $F0) then
    begin
      sTmp1 := Ord(sUTF8[sIndx]);
      if (sTmp1 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);

      sTmp2 := Ord(sUTF8[sIndx]);
      if (sTmp2 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);

      sTmp3 := Ord(sUTF8[sIndx]);
      if (sTmp3 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);
      
      Result[dLeng] := '?';
      continue;
    end;
    // 5-byte code => 0x0200000 .. 0x03FFFFFF - 1111 10xx 10xx xxxx 10xx xxxx 10xx xxxx 10xx xxxx
    if (sLeft < 4) then break;
    if ((sTmp0 and $FC) = $F8) then
    begin
      sTmp1 := Ord(sUTF8[sIndx]);
      if (sTmp1 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);

      sTmp2 := Ord(sUTF8[sIndx]);
      if (sTmp2 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);

      sTmp3 := Ord(sUTF8[sIndx]);
      if (sTmp3 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);

      sTmp4 := Ord(sUTF8[sIndx]);
      if (sTmp4 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);
      
      Result[dLeng] := '?';
      continue;
    end;
    // 6-byte code => 0x4000000 .. 0x7FFFFFFF - 1111 110x 10xx xxxx 10xx xxxx 10xx xxxx 10xx xxxx 10xx xxxx
    if (sLeft < 5) then break;
    if ((sTmp0 and $FE) = $FC) then
    begin
      sTmp1 := Ord(sUTF8[sIndx]);
      if (sTmp1 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);
      
      sTmp2 := Ord(sUTF8[sIndx]);
      if (sTmp2 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);

      sTmp3 := Ord(sUTF8[sIndx]);
      if (sTmp3 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);

      sTmp4 := Ord(sUTF8[sIndx]);
      if (sTmp4 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);

      sTmp5 := Ord(sUTF8[sIndx]);
      if (sTmp5 and $C0) <> $80 then break;
      Inc(sIndx);
      Dec(sLeft);

      Result[dLeng] := '?';
      continue;
    end;
    break;
  end;
  // correct length
  SetLength(Result, dLeng);
end;

{$ENDIF}

{$IFNDEF HAS_FILE_EXISTS}

function FileExists(fileName : string) : Boolean;
var
  sr : TSearchRec;
  re : Integer;
begin
  re := SysUtils.FindFirst(fileName, faAnyFile, sr);
  SysUtils.FindClose(sr);
  Result := (re = 0) and ((sr.Attr and FILE_ATTRIBUTE_DIRECTORY) = 0);
end;

{$ENDIF}

{$IFNDEF HAS_DIRECTORY_EXISTS}

function DirectoryExists(fileName : string) : Boolean;
var
  sr : TSearchRec;
  re : Integer;
begin
  re := SysUtils.FindFirst(fileName, faAnyFile, sr);
  SysUtils.FindClose(sr);
  Result := (re = 0) and ((sr.Attr and FILE_ATTRIBUTE_DIRECTORY) = FILE_ATTRIBUTE_DIRECTORY);
end;

{$ENDIF}

function libvlc_get_install_path() : string;

{$IFDEF MSWINDOWS}
var
  reKey : HKEY;
  reRes : Longint;
  vType : DWORD;
  vSize : DWORD;
  vBuff : packed array[0..2047] of Char;
begin
  Result := '';
  FillChar(vBuff, sizeof(vBuff), 0);
  reKey := INVALID_HANDLE_VALUE;
  reRes := RegOpenKeyEx(HKEY_LOCAL_MACHINE, 'Software\VideoLAN\VLC', 0, KEY_READ, reKey);
  if (reRes = ERROR_SUCCESS) then
  begin
    vSize := sizeof(vBuff);
    vType := REG_SZ;
    reRes := RegQueryValueEx(reKey, 'InstallDir', NIL, PDWORD(@vType), Pointer(@vBuff), PDWORD(@vSize));
    if (reRes = ERROR_SUCCESS) and (vType = REG_SZ) then
    begin
      Result := string(PChar(@vBuff));
    end;
  end;
  RegCloseKey(reKey);
end;
{$ENDIF}

{$IFDEF MACOS}
const
  pathLst : array[0..2] of string = (
    '~/Desktop/VLC.app/Contents/MacOS/lib',
    '~/Applications/VLC.app/Contents/MacOS/lib',
    '/Applications/VLC.app/Contents/MacOS/lib'
  );
var
  pathIdx : Integer;
  pathStr : string;
  sr : TSearchRec;
  re : Integer;
begin
  for pathIdx := Low(pathLst) to High(pathLst) do
  begin
    pathStr := pathLst[pathIdx];

    if not DirectoryExists(pathStr) then continue;
    // look for libvlc.dylib
    if FileExists(pathStr + PathDelim + LIBVLC_DLL_NAME) then
    begin
      Result := pathStr;
      exit;
    end;
    // look for libvlc.x.dylib
    re := SysUtils.FindFirst(pathStr + PathDelim + 'libvlc.*.dylib', faAnyFile, sr);
    FindClose(sr);
    if (re = 0) then
    begin
      Result := pathStr;
      exit;
    end;
  end;
  Result := '';
end;
{$ENDIF}

{$IFDEF UNIX}
  // Linux searches a library in the paths of the environment variable
  // LD_LIBRARY_PATH, then in /lib, then /usr/lib and finally the paths of
  // /etc/ld.so.conf.
const
  pathLst : array[0..2] of string = (
    '/usr/lib',
    '/lib',
    '/usr/local/lib'
  );
var
  pathIdx : Integer;
  pathStr : string;
  sr : TSearchRec;
  re : Integer;
begin
  for pathIdx := Low(pathLst) to High(pathLst) do
  begin
    pathStr := pathLst[pathIdx];
    if not DirectoryExists(pathStr) then continue;
    // look for libvlc.so
    if FileExists(pathStr + PathDelim + LIBVLC_DLL_NAME) then
    begin
      Result := pathStr;
      exit;
    end;
    // look for libvlc.so.x.x
    re := SysUtils.FindFirst(pathStr + PathDelim + LIBVLC_DLL_NAME + '.*', faAnyFile, sr);
    FindClose(sr);
    if (re = 0) then
    begin
      Result := pathStr;
      exit;
    end;
  end;
  Result := '';
end;
{$ENDIF}

function libvlc_get_install_file(libvlc_install_path : string) : string;
{$IFDEF MSWINDOWS}
begin
  // look for libvlc.dll
  if (DirectoryExists(libvlc_install_path)) then
  begin
    Result := LIBVLC_DLL_NAME;
    if FileExists(libvlc_install_path + PathDelim + Result) then
    begin
      exit;
    end;
  end;
  Result := '';
end;
{$ENDIF}

{$IFDEF MACOS}
var
  sr : TSearchRec;
  re : Integer;
begin
  if (DirectoryExists(libvlc_install_path)) then
  begin
    Result := LIBVLC_DLL_NAME;
    // look for libvlc.dylib
    if FileExists(libvlc_install_path + PathDelim + Result) then
    begin
      exit;
    end;
    // look for libvlc.x.dylib
    re := SysUtils.FindFirst(libvlc_install_path + PathDelim + 'libvlc.*.dylib', faAnyFile, sr);
    SysUtils.FindClose(sr);
    if (re = 0) then
    begin
      Result := sr.Name;
      exit;
    end;
  end;
  Result := '';
end;
{$ENDIF}

{$IFDEF UNIX}
var
  sr : TSearchRec;
  re : Integer;
begin
  if (DirectoryExists(libvlc_install_path)) then
  begin
    Result := LIBVLC_DLL_NAME;
    // look for libvlc.so
    if FileExists(libvlc_install_path + PathDelim + Result) then exit;
    // look for libvlc.so.x.x
    re := SysUtils.FindFirst(libvlc_install_path + PathDelim + LIBVLC_DLL_NAME + '.*', faAnyFile, sr);
    SysUtils.FindClose(sr);
    if (re = 0) then
    begin
      Result := sr.Name;
      exit;
    end;
  end;
  Result := '';
end;
{$ENDIF}

{$IFDEF MSWINDOWS}
function libvlc_dll_get_proc_addr(
  var addr   : Pointer;
  const name : PAnsiChar
  ) : Boolean;
begin
  addr := GetProcAddress(libvlc_handle, name);
  Result := (addr <> NIL);
  if not Result then
  begin
    libvlc_dynamic_dll_error := 'Entry point "' + name + '" not found!';
  end;
end;
{$ENDIF}

{$IFDEF MACOS}
function libvlc_dll_get_proc_addr(
  var addr   : Pointer;
  const name : PWideChar
  ) : Boolean;
begin
  addr := GetProcAddress(libvlc_handle, name);
  Result := (addr <> NIL);
  if not Result then
  begin
    libvlc_dynamic_dll_error := 'Entry point "' + name + '" not found!';
  end;
end;
{$ENDIF}

{$IFDEF UNIX}
function libvlc_dll_get_proc_addr(
  var addr   : Pointer;
  const name : PAnsiChar
  ) : Boolean;
begin
  addr := GetProcedureAddress(libvlc_handle, name);
  Result := (addr <> NIL);
  if not Result then
  begin
    libvlc_dynamic_dll_error := 'Entry point "' + name + '" not found!';
  end;
end;
{$ENDIF}

procedure require_version(function_name, require_version: string);
begin
  {$IFDEF MSWINDOWS}
  MessageBox(0, PChar(function_name + ' require libvlc ' + require_version), 'libvlc', MB_ICONHAND or MB_OK);
  {$ELSE}
  ShowMessage(function_name + ' require libvlc ' + require_version);
  {$ENDIF}
end;

{$HINTS OFF}

{$IFDEF DELPHI2005_UP}{$REGION '1.1.0'}{$ENDIF}

function require_version_libvlc_errmsg() : PAnsiChar; cdecl;
begin
  Result := NIL;
  require_version('libvlc_errmsg', '1.1.0');
end;

procedure require_version_libvlc_clearerr(); cdecl;
begin
  require_version('version_libvlc_clearerr', '1.1.0');
end;

{$IFDEF FPC}
function require_version_libvlc_printerr(
    fmt : PAnsiChar;
    arg : Pointer
  ) : PAnsiChar; cdecl;
{$ELSE}
function require_version_libvlc_printerr(
    fmt : PAnsiChar;
    arg : array of const
  ) : PAnsiChar; cdecl;
{$ENDIF}
begin
  Result := NIL;
  require_version('libvlc_printerr', '1.1.0');
end;

{$IFNDEF FPC}
function require_version_libvlc_vprinterr(fmt : PAnsiChar; ap : array of const) : PAnsiChar; cdecl;
begin
  Result := NIL;
  require_version('libvlc_vprinterr', '1.1.0');
end;
{$ENDIF}

function require_version_libvlc_video_get_logo_int(
  p_mi   : libvlc_media_player_t_ptr;
  option : libvlc_video_logo_option_t) : Integer; cdecl;
begin
  Result := -1;
  require_version('libvlc_video_get_logo_int', '1.1.0');
end;

procedure require_version_libvlc_video_set_logo_int(
  p_mi   : libvlc_media_player_t_ptr;
  option : libvlc_video_logo_option_t;
  value  : Integer); cdecl;
begin
  require_version('libvlc_video_set_logo_int', '1.1.0');
end;

procedure require_version_libvlc_video_set_logo_string(
  p_mi      : libvlc_media_player_t_ptr;
  option    : libvlc_video_logo_option_t;
  psz_value : PAnsiChar); cdecl;
begin
  require_version('libvlc_video_set_logo_string', '1.1.0');
end;

function require_version_libvlc_video_get_marquee_int(
  p_mi   : libvlc_media_player_t_ptr;
  option : libvlc_video_marquee_option_t) : Integer; cdecl;
begin
  Result := -1;
  require_version('libvlc_video_get_marquee_int', '1.1.0');
end;

function require_version_libvlc_video_get_marquee_string(
  p_mi   : libvlc_media_player_t_ptr;
  option : libvlc_video_marquee_option_t) : PAnsiChar; cdecl;
begin
  Result := NIL;
  require_version('libvlc_video_get_marquee_string', '1.1.0');
end;

procedure require_version_libvlc_video_set_marquee_int(
  p_mi   : libvlc_media_player_t_ptr;
  option : libvlc_video_marquee_option_t;
  i_val  : Integer); cdecl;
begin
  require_version('libvlc_video_set_marquee_int', '1.1.0');
end;

procedure require_version_libvlc_video_set_marquee_string(
  p_mi     : libvlc_media_player_t_ptr;
  option   : libvlc_video_marquee_option_t;
  psz_text : PAnsiChar); cdecl;
begin
  require_version('libvlc_video_set_marquee_string', '1.1.0');
end;

function require_version_libvlc_video_get_cursor(
  p_mi : libvlc_media_player_t_ptr;
  num  : LongWord;
  var px, py : Integer) : Integer; cdecl;
begin
  Result := -1;
  require_version('libvlc_video_get_cursor', '1.1.0');
end;

function require_version_libvlc_video_get_size(
  p_mi : libvlc_media_player_t_ptr;
  num  : LongWord;
  var px, py : LongWord) : Integer; cdecl;
begin
  Result := -1;
  require_version('libvlc_video_get_size', '1.1.0');
end;

procedure require_version_libvlc_video_set_key_input(
  p_mi    : libvlc_media_player_t_ptr;
  keys_on : LongWord); cdecl;
begin
  require_version('libvlc_video_set_key_input', '1.1.0');
end;

procedure require_version_libvlc_video_set_mouse_input(
  p_mi     : libvlc_media_player_t_ptr;
  mouse_on : LongWord); cdecl;
begin
  require_version('libvlc_video_set_mouse_input', '1.1.0');
end;

procedure require_version_libvlc_video_set_deinterlace(
  p_mi     : libvlc_media_player_t_ptr;
  psz_mode : PAnsiChar
); cdecl;
begin
  require_version('libvlc_video_set_deinterlace', '1.1.0');
end;

procedure require_version_libvlc_media_player_next_frame(
  p_mi : libvlc_media_player_t_ptr); cdecl;
begin
  require_version('libvlc_media_player_next_frame', '1.1.0');
end;

procedure require_version_libvlc_media_list_player_set_playback_mode(
    p_mlp  : libvlc_media_list_player_t_ptr;
    e_mode : libvlc_playback_mode_t
  ); cdecl;
begin
  require_version('libvlc_media_list_player_set_playback_mode', '1.1.0');
end;

function require_version_libvlc_media_list_player_previous(
    p_mlp : libvlc_media_list_player_t_ptr
  ) : Integer; cdecl;
begin
  Result := -1;
  require_version('libvlc_media_list_player_previous', '1.1.0');
end;

function require_version_libvlc_media_list_player_event_manager(
  p_mlp : libvlc_media_list_player_t_ptr) : libvlc_event_manager_t_ptr; cdecl;
begin
  Result := NIL;
  require_version('libvlc_media_list_player_event_manager', '1.1.0');
end;

function require_version_libvlc_vlm_get_event_manager(
  p_instance : libvlc_instance_t_ptr) : libvlc_event_manager_t_ptr; cdecl;
begin
  Result := NIL;
  require_version('libvlc_vlm_get_event_manager', '1.1.0');
end;

procedure require_version_libvlc_media_parse(
  p_md : libvlc_media_t_ptr); cdecl;
begin
  require_version('libvlc_media_parse', '1.1.0');
end;

procedure require_version_libvlc_media_parse_async(
  p_md : libvlc_media_t_ptr); cdecl;
begin
  require_version('libvlc_media_parse_async', '1.1.0');
end;

function require_version_libvlc_media_is_parsed(
  p_md : libvlc_media_t_ptr) : Integer; cdecl;
begin
  Result := 0;
  require_version('libvlc_media_is_parsed', '1.1.0');
end;

function require_version_libvlc_media_new_location(
  p_instance : libvlc_instance_t_ptr;
  psz_mrl    : PAnsiChar) : libvlc_media_t_ptr; cdecl;
begin
  Result := NIL;
  require_version('libvlc_media_new_location', '1.1.0');
end;

function require_version_libvlc_media_new_path(
  p_instance : libvlc_instance_t_ptr;
  path       : PAnsiChar) : libvlc_media_t_ptr; cdecl;
begin
  Result := NIL;
  require_version('libvlc_media_new_path', '1.1.0');
end;

function require_version_libvlc_media_new_callbacks(
  p_instance: libvlc_instance_t_ptr;
  open_cb: libvlc_media_open_cb;
  read_cb: libvlc_media_read_cb;
  seek_cb: libvlc_media_seek_cb;
  close_cb: libvlc_media_close_cb;
  opaque: pointer): libvlc_media_t_ptr; cdecl;
begin
  Result := NIL;
  require_version('libvlc_media_new_callbacks', '3.0.0');
end;

procedure require_version_libvlc_media_add_option_flag(
  p_md         : libvlc_media_t_ptr;
  ppsz_options : PAnsiChar;
  i_flags      : input_item_option_e); cdecl;
begin
  require_version('libvlc_media_add_option_flag', '1.1.0');
end;

procedure require_version_libvlc_media_set_meta(
  p_md      : libvlc_media_t_ptr;
  e_meta    : libvlc_meta_t;
  psz_value : PAnsiChar); cdecl;
begin
  require_version('libvlc_media_set_meta', '1.1.0');
end;

function require_version_libvlc_media_save_meta(
  p_md : libvlc_media_t_ptr) : Integer; cdecl;
begin
  Result := 0;
  require_version('libvlc_media_save_meta', '1.1.0');
end;

function require_version_libvlc_media_get_stats(
  p_md    : libvlc_media_t_ptr;
  p_stats : libvlc_media_stats_t_ptr) : Integer; cdecl;
begin
  Result := 0;
  require_version('libvlc_media_get_stats', '1.1.0');
end;

function require_version_libvlc_media_get_tracks_info(
  p_md       : libvlc_media_t_ptr;
  var tracks : libvlc_media_track_info_t_ptr) : Integer; cdecl;
begin
  Result := 0;
  require_version('libvlc_media_get_tracks_info', '1.1.0');
end;

{$IFDEF DELPHI2005_UP}{$ENDREGION}{$ENDIF}

{$IFDEF DELPHI2005_UP}{$REGION '1.1.1'}{$ENDIF}

procedure require_version_libvlc_video_set_format(
  p_mi   : libvlc_media_player_t_ptr;
  chroma : PAnsiChar;
  width  : Longword;
  height : Longword; pitch : Longword); cdecl;
begin
  require_version('libvlc_video_set_format', '1.1.1');
end;

procedure require_version_libvlc_set_user_agent(
  p_instance : libvlc_instance_t_ptr;
  name       : PAnsiChar;
  http       : PAnsiChar); cdecl;
begin
  require_version('libvlc_set_user_agent', '1.1.1');
end;

procedure require_version_libvlc_video_set_callbacks(
  p_mi    : libvlc_media_player_t_ptr;
  lock    : libvlc_video_lock_cb;
  unlock  : libvlc_video_unlock_cb;
  display : libvlc_video_display_cb;
  opaque  : Pointer); cdecl;
begin
  require_version('libvlc_video_set_callbacks', '1.1.1');
end;

function require_version_libvlc_audio_get_delay(
  p_mi : libvlc_media_player_t_ptr) : Int64; cdecl;
begin
  Result := 0;
  require_version('libvlc_audio_get_delay', '1.1.1');
end;

function require_version_libvlc_audio_set_delay(
  p_mi    : libvlc_media_player_t_ptr;
  i_delay : Int64) : Integer; cdecl;
begin
  Result := -1;
  require_version('libvlc_audio_set_delay', '1.1.1');
end;

function require_version_libvlc_video_get_adjust_int(
  p_mi   : libvlc_media_player_t_ptr;
  option : LongWord) : Integer; cdecl;
begin
  Result := 0;
  require_version('libvlc_video_get_adjust_int', '1.1.1');
end;

procedure require_version_libvlc_video_set_adjust_int(
  p_mi   : libvlc_media_player_t_ptr;
  option : LongWord;
  value  : Integer); cdecl;
begin
  require_version('libvlc_video_set_adjust_int', '1.1.1');
end;

function require_version_libvlc_video_get_adjust_float(
  p_mi   : libvlc_media_player_t_ptr;
  option : LongWord) : Single; cdecl;
begin
  Result := 0;
  require_version('libvlc_video_get_adjust_float', '1.1.1');
end;

procedure require_version_libvlc_video_set_adjust_float(
  p_mi   : libvlc_media_player_t_ptr;
  option : LongWord;
  value  : Single); cdecl;
begin
  require_version('libvlc_video_set_adjust_float', '1.1.1');
end;

procedure require_version_libvlc_media_player_set_pause(
  p_mi     : libvlc_media_player_t_ptr;
  do_pause : Integer); cdecl;
begin
  require_version('libvlc_media_player_set_pause', '1.1.1');
end;

{$IFDEF DELPHI2005_UP}{$ENDREGION}{$ENDIF}

{$IFDEF DELPHI2005_UP}{$REGION '1.1.5'}{$ENDIF}

function require_version_libvlc_media_new_fd(
  p_instance : libvlc_instance_t_ptr;
  fd         : Integer) : libvlc_media_t_ptr; cdecl;
begin
  Result := NIL;
  require_version('libvlc_media_new_fd', '1.1.5');
end;

{$IFDEF DELPHI2005_UP}{$ENDREGION}{$ENDIF}

{$IFDEF DELPHI2005_UP}{$REGION '2.0.0'}{$ENDIF}

function require_version_libvlc_video_filter_list_get(
  p_instance : libvlc_instance_t_ptr) : libvlc_module_description_t_ptr; cdecl;
begin
  Result := NIL;
  require_version('libvlc_video_filter_list_get', '2.0.0');
end;

function require_version_libvlc_audio_filter_list_get(
  p_instance : libvlc_instance_t_ptr) : libvlc_module_description_t_ptr; cdecl;
begin
  Result := NIL;
  require_version('libvlc_audio_filter_list_get', '2.0.0');
end;

procedure require_version_libvlc_module_description_list_release(
  p_list : libvlc_module_description_t_ptr); cdecl;
begin
  require_version('libvlc_module_description_list_release', '2.0.0');
end;

procedure require_version_libvlc_set_exit_handler(
  p_instance : libvlc_instance_t_ptr;
  cb         : libvlc_exit_callback_t;
  opaque     : Pointer); cdecl;
begin
  require_version('libvlc_set_exit_handler', '2.0.0');
end;

procedure require_version_libvlc_free(
  ptr : Pointer); cdecl;
begin
  require_version('libvlc_free', '2.0.0');
end;

procedure require_version_libvlc_video_set_format_callbacks(
  p_mi    : libvlc_media_player_t_ptr;
  setup   : libvlc_video_format_cb;
  cleanup : libvlc_video_cleanup_cb); cdecl;
begin
  require_version('libvlc_video_set_format_callbacks', '2.0.0');
end;

procedure require_version_libvlc_audio_set_callbacks(
  p_mi   : libvlc_media_player_t_ptr;
  play   : libvlc_audio_play_cb;
  pause  : libvlc_audio_pause_cb;
  resume : libvlc_audio_resume_cb;
  flush  : libvlc_audio_flush_cb;
  drain  : libvlc_audio_drain_cb;
  opaque : Pointer ); cdecl;
begin
  require_version('libvlc_audio_set_callbacks', '2.0.0');
end;

procedure require_version_libvlc_audio_set_volume_callback(
  p_mi       : libvlc_media_player_t_ptr;
  set_volume : libvlc_audio_set_volume_cb); cdecl;
begin
  require_version('libvlc_audio_set_volume_callback', '2.0.0');
end;

procedure require_version_libvlc_audio_set_format_callbacks(
  p_mi    : libvlc_media_player_t_ptr;
  setup   : libvlc_audio_setup_cb;
  cleanup : libvlc_audio_cleanup_cb ); cdecl;
begin
  require_version('libvlc_audio_set_format_callbacks', '2.0.0');
end;

procedure require_version_libvlc_audio_set_format(
  p_mp         : libvlc_media_player_t_ptr;
  const format : PAnsiChar;
  rate         : Cardinal;
  channels     : Cardinal); cdecl;
begin
  require_version('libvlc_audio_set_format', '2.0.0');
end;

procedure require_version_libvlc_media_player_navigate(
  p_mi     : libvlc_media_player_t_ptr;
  navigate : libvlc_navigate_mode_t ); cdecl;
begin
  require_version('libvlc_media_player_navigate', '2.0.0');
end;

function require_version_libvlc_video_get_spu_delay(
  p_mi : libvlc_media_player_t_ptr): Int64; cdecl;
begin
  Result := 0;
  require_version('libvlc_video_get_spu_delay', '2.0.0');
end;

function require_version_libvlc_video_set_spu_delay(
  p_mi    : libvlc_media_player_t_ptr;
  i_delay : Int64) : Integer; cdecl;
begin
  Result := -1;
  require_version('libvlc_video_set_spu_delay', '2.0.0');
end;

procedure require_version_libvlc_media_list_player_retain(
  p_mlp : libvlc_media_list_player_t_ptr); cdecl;
begin
  require_version('libvlc_media_list_player_retain', '2.0.0');
end;

procedure require_version_libvlc_track_description_list_release(
  p_track_description : libvlc_track_description_t_ptr); cdecl;
begin
  require_version('libvlc_track_description_list_release', '2.0.0');
end;

{$IFDEF DELPHI2005_UP}{$ENDREGION}{$ENDIF}

{$IFDEF DELPHI2005_UP}{$REGION '2.1.0'}{$ENDIF}

procedure require_version_libvlc_media_player_set_video_title_display(
  p_mi     : libvlc_media_player_t_ptr;
  position : libvlc_position_t;
  timeout  : LongWord); cdecl;
begin
  require_version('libvlc_media_player_set_video_title_display', '2.1.0');
end;

function require_version_libvlc_audio_output_device_list_get(
    p_instance : libvlc_instance_t_ptr;
    aout       : PAnsiChar
  ) : libvlc_audio_output_device_t_ptr; cdecl;
begin
  Result := NIL;
  require_version('libvlc_audio_output_device_list_get', '2.1.0');
end;

procedure require_version_libvlc_audio_output_device_list_release(
    p_list : libvlc_audio_output_device_t_ptr
  ); cdecl;
begin
  require_version('libvlc_audio_output_device_list_release', '2.1.0');
end;

function require_version_libvlc_media_tracks_get(
    p_md       : libvlc_media_t_ptr;
    var tracks : Pointer
  ) : LongWord; cdecl;
begin
  Result := 0;
  require_version('libvlc_media_tracks_get', '2.1.0');
end;

procedure require_version_libvlc_media_tracks_release(
    var tracks : Pointer;
    i_count    : LongWord
  ); cdecl;
begin
  require_version('libvlc_media_tracks_release', '2.1.0');
end;

procedure require_version_libvlc_set_app_id (
    p_instance : libvlc_instance_t_ptr;
    id         : PAnsiChar;
    version    : PAnsiChar;
    icon       : PAnsiChar
  ); cdecl;
begin
  require_version('libvlc_set_app_id', '2.1.0');
end;

procedure require_version_libvlc_log_get_context(
    ctx           : libvlc_log_t_ptr;
    var module    : PAnsiChar;
    var file_name : PANsiChar;
    var line      : LongWord
  ); cdecl;
begin
  require_version('log_get_context', '2.1.0');
end;

procedure require_version_libvlc_log_get_object(
    ctx        : libvlc_log_t_ptr;
    var name   : PAnsiChar;
    var header : PAnsiChar;
    var id     : Pointer
  ); cdecl;
begin
  require_version('libvlc_log_get_object', '2.1.0');
end;

procedure require_version_libvlc_log_set(
    p_instance : libvlc_instance_t_ptr;
    cb         : libvlc_log_cb;
    data       : Pointer
  ); cdecl;
begin
  require_version('libvlc_log_unset', '2.1.0');
end;

procedure require_version_libvlc_log_unset(
    p_instance: libvlc_instance_t_ptr
  ); cdecl;
begin
  require_version('libvlc_log_unset', '2.1.0');
end;

procedure require_version_libvlc_log_set_file(
    p_instance: libvlc_instance_t_ptr;
    stream : THandle
  ); cdecl;
begin
  require_version('libvlc_log_set_file', '2.1.0');
end;

{$IFDEF DELPHI2005_UP}{$ENDREGION}{$ENDIF}

{$IFDEF DELPHI2005_UP}{$REGION '2.2.0'}{$ENDIF}

function require_version_libvlc_media_player_program_scrambled(
    p_mi : libvlc_media_player_t_ptr
  ) : Integer; cdecl;
begin
  Result := 0;
  require_version('libvlc_media_player_program_scrambled', '2.2.0');
end;
  
function require_version_libvlc_audio_output_device_enum(
    p_mi : libvlc_media_player_t_ptr
  ) : libvlc_audio_output_device_t_ptr; cdecl;
begin
  Result := NIL;
  require_version('libvlc_audio_output_device_enum ', '2.2.0');
end;

function require_version_libvlc_audio_equalizer_get_preset_count(
  ) : unsigned_t; cdecl;
begin
  Result := 0;
  require_version('libvlc_audio_equalizer_get_preset_count', '2.2.0');
end;

function require_version_libvlc_audio_equalizer_get_preset_name(
    u_index : unsigned_t
  ): PAnsiChar; cdecl;
begin
  Result := NIL;
  require_version('libvlc_audio_equalizer_get_preset_name', '2.2.0');
end;

function require_version_libvlc_audio_equalizer_get_band_count(
  ) : unsigned_t; cdecl;
begin
  Result := 0;
  require_version('libvlc_audio_equalizer_get_band_count', '2.2.0');
end;

function require_version_libvlc_audio_equalizer_get_band_frequency(
    u_index : unsigned_t
  ) : Single; cdecl;
begin
  Result := 0;
  require_version('libvlc_audio_equalizer_get_band_frequency', '2.2.0');
end;

function require_version_libvlc_audio_equalizer_new(
  ) : libvlc_equalizer_t_ptr; cdecl;
begin
  Result := NIL;
  require_version('libvlc_audio_equalizer_new', '2.2.0');
end;

function require_version_libvlc_audio_equalizer_new_from_preset(
    u_index : unsigned_t
  ) : libvlc_equalizer_t_ptr; cdecl;
begin
  Result := NIL;
  require_version('libvlc_audio_equalizer_new_from_preset', '2.2.0');
end;

procedure require_version_libvlc_audio_equalizer_release(
    p_equalizer : libvlc_equalizer_t_ptr
  ); cdecl;
begin
  require_version('libvlc_audio_equalizer_release', '2.2.0');
end;

function require_version_libvlc_audio_equalizer_set_preamp(
    p_equalizer : libvlc_equalizer_t_ptr;
    f_preamp    : Single
  ) : Integer; cdecl;
begin
  Result := 0;
  require_version('libvlc_audio_equalizer_set_preamp', '2.2.0');
end;

function require_version_libvlc_audio_equalizer_get_preamp(
    p_equalizer : libvlc_equalizer_t_ptr
  ) : Single; cdecl;
begin
  Result := 0;
  require_version('libvlc_audio_equalizer_get_preamp', '2.2.0');
end;

function require_version_libvlc_audio_equalizer_set_amp_at_index(
    p_equalizer : libvlc_equalizer_t_ptr;
    f_amp : Single;
    u_band : unsigned_t
  ) : Integer; cdecl;
begin
  Result := 0;
  require_version('libvlc_audio_equalizer_set_amp_at_index', '2.2.0');
end;

function require_version_libvlc_audio_equalizer_get_amp_at_index(
    p_equalizer : libvlc_equalizer_t_ptr;
    u_band      : unsigned_t
  ) : Single; cdecl;
begin
  Result := 0;
  require_version('libvlc_audio_equalizer_get_amp_at_index', '2.2.0');
end;

function require_version_libvlc_media_player_set_equalizer(
    p_mi        : libvlc_media_player_t_ptr;
    p_equalizer : libvlc_equalizer_t_ptr
  ) : Integer; cdecl;
begin
  Result := 0;
  require_version('libvlc_media_player_set_equalizer', '2.2.0');
end;

{$IFDEF DELPHI2005_UP}{$ENDREGION}{$ENDIF}

{$HINTS ON}

function read_dec_number(var ptr : PAnsiChar) : LongWord;
begin
  Result := 0;
  if (ptr <> NIL) then
  begin
    while ((ptr^ <> #00) and (ptr^ > '0') and (ptr^ <= '9')) do
    begin
      Result := Result * 10 + LongWord(Byte(ptr^) - 48);
      Inc(ptr);
    end;
    if (ptr^ = '.') then Inc(ptr);
  end;
end;

// inspired by "Marcio Wesley Borges" <marciowb@gmail.com>
procedure libvlc_reset_function_pointers(force : boolean = FALSE);
begin
  if (not force) then
  begin
    if (libvlc_handle <> 0) then exit;
  end;
  
  libvlc_get_version := NIL;
  
  libvlc_new     := NIL;
  libvlc_release := NIL;
  libvlc_retain  := NIL;

  libvlc_add_intf := NIL;
  libvlc_wait := NIL;
  libvlc_get_compiler := NIL;
  libvlc_get_changeset := NIL;
  libvlc_event_attach := NIL;
  libvlc_event_detach := NIL;
  libvlc_event_type_name := NIL;
  libvlc_get_log_verbosity := NIL;
  libvlc_set_log_verbosity := NIL;
  libvlc_log_open := NIL;
  libvlc_log_close := NIL;
  libvlc_log_count := NIL;
  libvlc_log_clear := NIL;
  libvlc_log_get_iterator := NIL;
  libvlc_log_iterator_free := NIL;
  libvlc_log_iterator_has_next := NIL;
  libvlc_log_iterator_next := NIL;
  libvlc_media_new_as_node := NIL;
  libvlc_media_add_option := NIL;
  libvlc_media_retain := NIL;
  libvlc_media_release := NIL;
  libvlc_media_get_mrl := NIL;
  libvlc_media_duplicate := NIL;
  libvlc_media_get_meta := NIL;
  libvlc_media_get_state := NIL;
  libvlc_media_subitems := NIL;
  libvlc_media_event_manager := NIL;
  libvlc_media_get_duration := NIL;
  libvlc_media_set_user_data := NIL;
  libvlc_media_get_user_data := NIL;
  libvlc_media_player_new := NIL;
  libvlc_media_player_new_from_media := NIL;
  libvlc_media_player_release := NIL;
  libvlc_media_player_retain := NIL;
  libvlc_media_player_set_media := NIL;
  libvlc_media_player_get_media := NIL;
  libvlc_media_player_event_manager := NIL;
  libvlc_media_player_is_playing := NIL;
  libvlc_media_player_play := NIL;
  libvlc_media_player_pause := NIL;
  libvlc_media_player_stop := NIL;
  libvlc_media_player_set_nsobject := NIL;
  libvlc_media_player_get_nsobject := NIL;
  libvlc_media_player_set_agl := NIL;
  libvlc_media_player_get_agl := NIL;
  libvlc_media_player_set_xwindow := NIL;
  libvlc_media_player_get_xwindow := NIL;
  libvlc_media_player_set_hwnd := NIL;
  libvlc_media_player_get_hwnd := NIL;
  libvlc_media_player_get_length := NIL;
  libvlc_media_player_get_time := NIL;
  libvlc_media_player_set_time := NIL;
  libvlc_media_player_get_position := NIL;
  libvlc_media_player_set_position := NIL;
  libvlc_media_player_set_chapter := NIL;
  libvlc_media_player_get_chapter := NIL;
  libvlc_media_player_get_chapter_count := NIL;
  libvlc_media_player_will_play := NIL;
  libvlc_media_player_get_chapter_count_for_title := NIL;
  libvlc_media_player_set_title := NIL;
  libvlc_media_player_get_title := NIL;
  libvlc_media_player_get_title_count := NIL;
  libvlc_media_player_previous_chapter := NIL;
  libvlc_media_player_next_chapter := NIL;
  libvlc_media_player_set_rate := NIL;
  libvlc_media_player_get_rate := NIL;
  libvlc_media_player_get_state := NIL;
  libvlc_media_player_get_fps := NIL;
  libvlc_media_player_has_vout := NIL;
  libvlc_media_player_is_seekable := NIL;
  libvlc_media_player_can_pause := NIL;
  libvlc_track_description_release := NIL;
  libvlc_toggle_fullscreen := NIL;
  libvlc_set_fullscreen := NIL;
  libvlc_get_fullscreen := NIL;
  
{$IFDEF USE_VLC_DEPRECATED_API}
  libvlc_video_get_height := NIL;
{$ENDIF}

{$IFDEF USE_VLC_DEPRECATED_API}
  libvlc_video_get_width := NIL;
{$ENDIF}

  libvlc_video_get_scale := NIL;
  libvlc_video_set_scale := NIL;
  libvlc_video_get_aspect_ratio := NIL;
  libvlc_video_set_aspect_ratio := NIL;
  libvlc_video_get_spu := NIL;
  libvlc_video_get_spu_count := NIL;
  libvlc_video_get_spu_description := NIL;
  libvlc_video_set_spu := NIL;
  libvlc_video_set_subtitle_file := NIL;
  libvlc_video_get_title_description := NIL;
  libvlc_video_get_chapter_description := NIL;
  libvlc_video_get_crop_geometry := NIL;
  libvlc_video_set_crop_geometry := NIL;
  libvlc_video_get_teletext := NIL;
  libvlc_video_set_teletext := NIL;
  libvlc_toggle_teletext := NIL;
  libvlc_video_get_track_count := NIL;
  libvlc_video_get_track_description := NIL;
  libvlc_video_get_track := NIL;
  libvlc_video_set_track := NIL;
  libvlc_video_take_snapshot := NIL;
  libvlc_audio_output_list_get := NIL;
  libvlc_audio_output_list_release := NIL;
  libvlc_audio_output_set := NIL;
{$IFDEF USE_VLC_DEPRECATED_API}
  libvlc_audio_output_device_count := NIL;
  libvlc_audio_output_device_longname := NIL;
  libvlc_audio_output_device_id := NIL;
{$ENDIF}
  libvlc_audio_output_device_set := NIL;
  libvlc_audio_output_get_device_type := NIL;
  libvlc_audio_output_set_device_type := NIL;
  libvlc_audio_toggle_mute := NIL;
  libvlc_audio_get_mute := NIL;
  libvlc_audio_set_mute := NIL;
  libvlc_audio_get_volume := NIL;
  libvlc_audio_set_volume := NIL;
  libvlc_audio_get_track_count := NIL;
  libvlc_audio_get_track_description := NIL;
  libvlc_audio_get_track := NIL;
  libvlc_audio_set_track := NIL;
  libvlc_audio_get_channel := NIL;
  libvlc_audio_set_channel := NIL;
  libvlc_media_list_new := NIL;
  libvlc_media_list_release := NIL;
  libvlc_media_list_retain := NIL;

{$IFDEF USE_VLC_DEPRECATED_API}
  libvlc_media_list_add_file_content := NIL;
{$ENDIF}

  libvlc_media_list_set_media := NIL;
  libvlc_media_list_media := NIL;
  libvlc_media_list_add_media := NIL;
  libvlc_media_list_insert_media := NIL;
  libvlc_media_list_remove_index := NIL;
  libvlc_media_list_count := NIL;
  libvlc_media_list_item_at_index := NIL;
  libvlc_media_list_index_of_item := NIL;
  libvlc_media_list_is_readonly := NIL;
  libvlc_media_list_lock := NIL;
  libvlc_media_list_unlock := NIL;
  libvlc_media_list_event_manager := NIL;
  libvlc_media_list_player_new := NIL;
  libvlc_media_list_player_release := NIL;
  libvlc_media_list_player_set_media_player := NIL;
  libvlc_media_list_player_set_media_list := NIL;
  libvlc_media_list_player_play := NIL;
  libvlc_media_list_player_pause := NIL;
  libvlc_media_list_player_is_playing := NIL;
  libvlc_media_list_player_get_state := NIL;
  libvlc_media_list_player_play_item_at_index := NIL;
  libvlc_media_list_player_play_item := NIL;
  libvlc_media_list_player_stop := NIL;
  libvlc_media_list_player_next := NIL;
  libvlc_media_library_new := NIL;
  libvlc_media_library_release := NIL;
  libvlc_media_library_retain := NIL;
  libvlc_media_library_load := NIL;
  libvlc_media_library_media_list := NIL;
  libvlc_media_discoverer_new_from_name := NIL;
  libvlc_media_discoverer_release := NIL;
  libvlc_media_discoverer_localized_name := NIL;
  libvlc_media_discoverer_media_list := NIL;
  libvlc_media_discoverer_event_manager := NIL;
  libvlc_media_discoverer_is_running := NIL;
  libvlc_vlm_release := NIL;
  libvlc_vlm_add_broadcast := NIL;
  libvlc_vlm_add_vod := NIL;
  libvlc_vlm_del_media := NIL;
  libvlc_vlm_set_enabled := NIL;
  libvlc_vlm_set_output := NIL;
  libvlc_vlm_set_input := NIL;
  libvlc_vlm_add_input := NIL;
  libvlc_vlm_set_loop := NIL;
  libvlc_vlm_set_mux := NIL;
  libvlc_vlm_change_media := NIL;
  libvlc_vlm_play_media := NIL;
  libvlc_vlm_stop_media := NIL;
  libvlc_vlm_pause_media := NIL;
  libvlc_vlm_seek_media := NIL;
  libvlc_vlm_show_media := NIL;
  libvlc_vlm_get_media_instance_position := NIL;
  libvlc_vlm_get_media_instance_time := NIL;
  libvlc_vlm_get_media_instance_length := NIL;
  libvlc_vlm_get_media_instance_rate := NIL;

{$IFDEF IS_0_GT_1}
   libvlc_vlm_get_media_instance_title := NIL;
   libvlc_vlm_get_media_instance_chapter := NIL;
   libvlc_vlm_get_media_instance_seekable := NIL;
{$ENDIF}

{$IFDEF USE_VLC_DEPRECATED_API}
  libvlc_playlist_play := NIL;
{$ENDIF}

  (* 1.1.0 *)

  libvlc_errmsg                          := @require_version_libvlc_errmsg;
  libvlc_clearerr                        := @require_version_libvlc_clearerr;
  libvlc_printerr                        := @require_version_libvlc_printerr;
{$IFNDEF FPC}
  libvlc_vprinterr                       := @require_version_libvlc_vprinterr;
{$ENDIF}
  libvlc_video_get_logo_int              := @require_version_libvlc_video_get_logo_int;
  libvlc_video_set_logo_int              := @require_version_libvlc_video_set_logo_int;
  libvlc_video_set_logo_string           := @require_version_libvlc_video_set_logo_string;
  libvlc_video_get_marquee_int           := @require_version_libvlc_video_get_marquee_int;
  libvlc_video_get_marquee_string        := @require_version_libvlc_video_get_marquee_string;
  libvlc_video_set_marquee_int           := @require_version_libvlc_video_set_marquee_int;
  libvlc_video_set_marquee_string        := @require_version_libvlc_video_set_marquee_string;
  libvlc_video_get_cursor                := @require_version_libvlc_video_get_cursor;
  libvlc_video_get_size                  := @require_version_libvlc_video_get_size;
  libvlc_video_set_key_input             := @require_version_libvlc_video_set_key_input;
  libvlc_video_set_mouse_input           := @require_version_libvlc_video_set_mouse_input;
  libvlc_video_set_deinterlace           := @require_version_libvlc_video_set_deinterlace;

  libvlc_media_player_next_frame         := @require_version_libvlc_media_player_next_frame;

  libvlc_media_list_player_event_manager     := @require_version_libvlc_media_list_player_event_manager;
  libvlc_media_list_player_set_playback_mode := @require_version_libvlc_media_list_player_set_playback_mode;
  libvlc_media_list_player_previous          := @require_version_libvlc_media_list_player_previous;

  libvlc_media_parse                     := @require_version_libvlc_media_parse;
  libvlc_media_parse_async               := @require_version_libvlc_media_parse_async;
  libvlc_media_is_parsed                 := @require_version_libvlc_media_is_parsed;
  libvlc_media_new_location              := @require_version_libvlc_media_new_location;
  libvlc_media_new_path                  := @require_version_libvlc_media_new_path;
  libvlc_media_new_callbacks             := @require_version_libvlc_media_new_callbacks;
  libvlc_media_add_option_flag           := @require_version_libvlc_media_add_option_flag;
  libvlc_media_set_meta                  := @require_version_libvlc_media_set_meta;
  libvlc_media_save_meta                 := @require_version_libvlc_media_save_meta;
  libvlc_media_get_tracks_info           := @require_version_libvlc_media_get_tracks_info;
  libvlc_media_get_stats                 := @require_version_libvlc_media_get_stats;

  libvlc_vlm_get_event_manager           := @require_version_libvlc_vlm_get_event_manager;
  
  (* 1.1.1 *)

  libvlc_set_user_agent                  := @require_version_libvlc_set_user_agent;
  libvlc_video_set_callbacks             := @require_version_libvlc_video_set_callbacks;
  libvlc_audio_get_delay                 := @require_version_libvlc_audio_get_delay;
  libvlc_audio_set_delay                 := @require_version_libvlc_audio_set_delay;
  libvlc_video_set_format                := @require_version_libvlc_video_set_format;
  libvlc_video_get_adjust_int            := @require_version_libvlc_video_get_adjust_int;
  libvlc_video_set_adjust_int            := @require_version_libvlc_video_set_adjust_int;
  libvlc_video_get_adjust_float          := @require_version_libvlc_video_get_adjust_float;
  libvlc_video_set_adjust_float          := @require_version_libvlc_video_set_adjust_float;
  libvlc_media_player_set_pause          := @require_version_libvlc_media_player_set_pause;

  (* 1.1.5 *)

  libvlc_media_new_fd                    := @require_version_libvlc_media_new_fd;

  (* 2.0.0 *)

  libvlc_audio_filter_list_get           := @require_version_libvlc_audio_filter_list_get;
  libvlc_audio_set_callbacks             := @require_version_libvlc_audio_set_callbacks;
  libvlc_audio_set_volume_callback       := @require_version_libvlc_audio_set_volume_callback;
  libvlc_audio_set_format_callbacks      := @require_version_libvlc_audio_set_format_callbacks;
  libvlc_audio_set_format                := @require_version_libvlc_audio_set_format;
  libvlc_video_filter_list_get           := @require_version_libvlc_video_filter_list_get;
  libvlc_video_set_format_callbacks      := @require_version_libvlc_video_set_format_callbacks;
  libvlc_video_get_spu_delay             := @require_version_libvlc_video_get_spu_delay;
  libvlc_video_set_spu_delay             := @require_version_libvlc_video_set_spu_delay;
  libvlc_media_player_navigate           := @require_version_libvlc_media_player_navigate;
  libvlc_media_list_player_retain        := @require_version_libvlc_media_list_player_retain;
  libvlc_module_description_list_release := @require_version_libvlc_module_description_list_release;
  libvlc_set_exit_handler                := @require_version_libvlc_set_exit_handler;
  libvlc_free                            := @require_version_libvlc_free;
  libvlc_track_description_list_release  := @require_version_libvlc_track_description_list_release;

  (* 2.1.0 *)
  libvlc_media_player_set_video_title_display := @require_version_libvlc_media_player_set_video_title_display;
  libvlc_audio_output_device_list_get         := @require_version_libvlc_audio_output_device_list_get;
  libvlc_audio_output_device_list_release     := @require_version_libvlc_audio_output_device_list_release;
  libvlc_media_tracks_get                     := @require_version_libvlc_media_tracks_get;
  libvlc_media_tracks_release                 := @require_version_libvlc_media_tracks_release;
  libvlc_set_app_id                           := @require_version_libvlc_set_app_id;
  libvlc_log_get_context                      := @require_version_libvlc_log_get_context;
  libvlc_log_get_object                       := @require_version_libvlc_log_get_object;
  libvlc_log_set                              := @require_version_libvlc_log_set;
  libvlc_log_unset                            := @require_version_libvlc_log_unset;
  libvlc_log_set_file                         := @require_version_libvlc_log_set_file;

  (* 2.2.0 *)
  libvlc_media_player_program_scrambled     := @require_version_libvlc_media_player_program_scrambled;
  libvlc_audio_output_device_enum           := @require_version_libvlc_audio_output_device_enum;
  libvlc_audio_equalizer_get_preset_count   := @require_version_libvlc_audio_equalizer_get_preset_count;
  libvlc_audio_equalizer_get_preset_name    := @require_version_libvlc_audio_equalizer_get_preset_name;
  libvlc_audio_equalizer_get_band_count     := @require_version_libvlc_audio_equalizer_get_band_count;
  libvlc_audio_equalizer_get_band_frequency := @require_version_libvlc_audio_equalizer_get_band_frequency;
  libvlc_audio_equalizer_new                := @require_version_libvlc_audio_equalizer_new;
  libvlc_audio_equalizer_new_from_preset    := @require_version_libvlc_audio_equalizer_new_from_preset;
  libvlc_audio_equalizer_release            := @require_version_libvlc_audio_equalizer_release;
  libvlc_audio_equalizer_set_preamp         := @require_version_libvlc_audio_equalizer_set_preamp;
  libvlc_audio_equalizer_get_preamp         := @require_version_libvlc_audio_equalizer_get_preamp;
  libvlc_audio_equalizer_set_amp_at_index   := @require_version_libvlc_audio_equalizer_set_amp_at_index;
  libvlc_audio_equalizer_get_amp_at_index   := @require_version_libvlc_audio_equalizer_get_amp_at_index;
  libvlc_media_player_set_equalizer         := @require_version_libvlc_media_player_set_equalizer;
  
  (* 3.0.0 *)
end;

function libvlc_dynamic_dll_architecture(file_path_name : string) : Integer;
{$IFDEF MSWINDOWS}
const
  IMAGE_FILE_MACHINE_I386    = $014c;
  IMAGE_FILE_MACHINE_IA64    = $0200;
  IMAGE_FILE_MACHINE_ALPHA64 = $0284;
  IMAGE_FILE_MACHINE_AMD64   = $8664;
var
  f_handle  : THandle;
  id_header : TImageDosHeader;
  pe_sign   : Cardinal;
  machine   : Word;
begin
  Result := -1;
  f_handle := FileOpen(file_path_name, fmOpenRead);
  if (f_handle <> INVALID_HANDLE_VALUE) then
  begin
    Result := 0;
    if (FileRead(f_handle, id_header, sizeof(id_header)) = sizeof(id_header)) then
    begin
      if (id_header.e_magic = IMAGE_DOS_SIGNATURE) then
      begin
        if (FileSeek(f_handle, id_header._lfanew, 0) = id_header._lfanew) then
        begin
          if (FileRead(f_handle, pe_sign, sizeof(pe_sign)) = sizeof(pe_sign)) then
          begin
            if (pe_sign = IMAGE_NT_SIGNATURE) then
            begin
              if (FileRead(f_handle, machine, sizeof(machine)) = sizeof(machine)) then
              begin
                case machine of
                  IMAGE_FILE_MACHINE_I386     : Result := 32;
                  IMAGE_FILE_MACHINE_IA64,
                  IMAGE_FILE_MACHINE_ALPHA64,
                  IMAGE_FILE_MACHINE_AMD64    : Result := 64;
                end;
              end;
            end;
          end;
        end;
      end;
    end;
    FileClose(f_handle);
  end;
end;
{$ENDIF}
{$IFDEF MACOS}
const
  MH_MAGIC_32 = $feedface;
  MH_MAGIC_64 = $feedfacf;
  FAT_MAGIC   = $cafebabe;
const
  CPU_ARCH_MASK  = $ff000000; // mask for architecture bits
  CPU_ARCH_ABI64 = $01000000; // 64 bit ABI
type
  mach_header_t = packed record
    magic      : Cardinal; // mach magic number identifier
	  cputype    : Cardinal; // cpu specifier
	  cpusubtype : Cardinal; // machine specifier
	  filetype   : Cardinal; // type of file
	  ncmds      : Cardinal; // number of load commands
	  sizeofcmds : Cardinal; // the size of all the load commands
	  flags      : Cardinal; // flags
  end;
type
  fat_arch_t = packed record
    cputype    : Cardinal;
    cpusubtype : Cardinal;
    offset     : Cardinal;
    size       : Cardinal;
    align      : Cardinal;
  end;
var
  f_handle  : THandle;
  magic     : Cardinal;
  cputype   : Cardinal;
  fat_count : Cardinal;
  fat_arch  : fat_arch_t;
begin
  Result := -1;
  f_handle := FileOpen(file_path_name, fmOpenRead);
  if (f_handle <> INVALID_HANDLE_VALUE) then
  begin
    Result := 0;
    if (FileRead(f_handle, magic, sizeof(magic)) = sizeof(magic)) then
    begin
      if (magic = MH_MAGIC_32) then
      begin
        FileRead(f_handle, cputype, sizeof(cputype));
        if ((cputype and CPU_ARCH_MASK) <> CPU_ARCH_ABI64) then
        begin
          Result := 32;
        end;
      end
      else if (magic = MH_MAGIC_64) then
      begin
        FileRead(f_handle, cputype, sizeof(cputype));
        if ((cputype and CPU_ARCH_MASK) = CPU_ARCH_ABI64) then
        begin
          Result := 64;
        end;
      end
      // Available in OS X v10.8 and later.
      // not tested
      else if (magic = FAT_MAGIC) then
      begin
        if (FileRead(f_handle, fat_count, sizeof(fat_count)) = sizeof(fat_count)) then
        begin
          while (fat_count > 0) do
          begin
            Dec(fat_count);
            if (FileRead(f_handle, fat_arch, sizeof(fat_arch)) <> sizeof(fat_arch)) then
            begin
              break;
            end;
            if ((cputype and CPU_ARCH_MASK) = CPU_ARCH_ABI64) then
            begin
              Result := Result or 64;
            end
            else
            begin
              Result := Result or 32;
            end;
            if (Result = 96) then break;            
          end;
        end;
      end;
    end;
    FileClose(f_handle);
  end;
end;
{$ENDIF}
{$IFDEF UNIX}
// http://en.wikipedia.org/wiki/Executable_and_Linkable_Format
type
  TElfHeader = packed record
    e_ident_MAGIC      : packed array[0..3] of Byte; // 0x7F followed by ELF in ASCII; these four bytes constitute the magic number.
    e_ident_CLASS      : Byte; // This byte is set to either 1 or 2 to signify 32- or 64-bit format, respectively.
    e_ident_DATA       : Byte; // This byte is set to either 1 or 2 to signify little or big endianness, respectively. This affects interpretation of multi-byte fields starting with offset 0x10.
    e_ident_VERSION    : Byte; // Set to 1 for the original version of ELF.
    e_ident_OSABI      : Byte; // 0x00	System V, 0x01	HP-UX, 0x02	NetBSD, 0x03	Linux, 0x06	Solaris, 0x07	AIX, 0x08	IRIX, 0x09	FreeBSD, 0x0C	OpenBSD, 0x0D	OpenVMS
    e_ident_ABIVERSION : Byte; // Further specifies the ABI version. Its interpretation depends on the target ABI. Linux kernel (after at least 2.6) has no definition of it.[4] In that case, offset and size of EI_PAD are 8.
    e_ident_PAD        : packed array[0..6] of Byte; //currently unused
    e_type             : Word; // 1, 2, 3, 4 specify whether the object is relocatable, executable, shared, or core, respectively.
    e_machine          : Word; // 0x02 SPARC, 0x03 x86, 0x08 MIPS, 0x14 PowerPC, 0x28 ARM, 0x2A SuperH, 0x32 IA-64, 0x3E x86-64, 0xB7 AArch64

    (*
    e_version          : Cardinal; // Set to 1 for the original version of ELF.

    // This fields is either 32 or 64 bits long depending on the format defined earlier (e_ident_CLASS).
    e_entry : Cardinal; // This is the memory address of the entry point from where the process starts executing.

    // This fields is either 32 or 64 bits long depending on the format defined earlier (e_ident_CLASS).
    e_phoff : Cardinal; // Points to the start of the program header table. It usually follows the file header immediately making the offset 0x40 for 64-bit ELF executables.

    // This fields is either 32 or 64 bits long depending on the format defined earlier.
    e_shoff : Cardinal; // Points to the start of the section header table.

    e_flags      : Cardinal;
    e_ehsize     : Word; // Contains the size of this header, normally 64 bytes for 64-bit and 52 for 32-bit format.
    e_phentsize  : Word; // Contains the size of a program header table entry.
    e_phnum      : Word; // Contains the number of entries in the program header table.
    e_shentsize  : Word; // Contains the size of a section header table entry.
    e_shnum      : Word; // Contains the number of entries in the section header table.
    e_shstrndx   : Word; // Contains index of the section header table entry that contains the section names.
    *)
  end;
var
  eh : TElfHeader;
  f_handle: THandle;
begin
  Result := 0;
  Result := -1;
  f_handle := FileOpen(file_path_name, fmOpenRead);
  if (f_handle <> -1) then
  begin
    Result := 0;
    if (FileRead(f_handle, eh, sizeof(eh)) = sizeof(eh)) then
    begin
      if (eh.e_ident_MAGIC[0] = $7F) and (eh.e_ident_MAGIC[1] = Byte('E')) and (eh.e_ident_MAGIC[2] = Byte('L')) and (eh.e_ident_MAGIC[3] = Byte('F'))then
      begin
        if (eh.e_machine = $03) or (eh.e_machine = $3E) then
        begin
          Result := Result or 32;
        end;
        if (eh.e_machine = $32) or  (eh.e_machine = $3E) or (eh.e_machine = $B7) then
        begin
          Result := Result or 64;
        end;
      end;
    end;
  end;
end;
{$ENDIF}

procedure libvlc_dynamic_dll_init_with_path(vlc_install_path: string);
var
  cdir : string;
  dll_path : string;
  dll_arch : Integer;
  va, vb, vc : LongWord;
  vp : PAnsiChar;
begin
  if (libvlc_handle <> 0) then exit;

  // no error
  libvlc_dynamic_dll_error := '';
  libvlc_dynamic_dll_path  := vlc_install_path;
  libvlc_dynamic_dll_file  := libvlc_get_install_file(vlc_install_path);

  // remove path separator from end of path
  while ((libvlc_dynamic_dll_path <> '') and (libvlc_dynamic_dll_path[Length(libvlc_dynamic_dll_path)] = PathDelim)) do
  begin
    SetLength(libvlc_dynamic_dll_path, Length(libvlc_dynamic_dll_path)-1);
  end;

  cdir := GetCurrentDir();

  // try load library
  // before loading libvlc.dll application must change dir to
  // libvlc_dynamic_dll_path
  if (libvlc_dynamic_dll_path <> '') then
  begin
    SetCurrentDir(libvlc_dynamic_dll_path);
    dll_path := libvlc_dynamic_dll_path + PathDelim + libvlc_dynamic_dll_file;
  end
  else
  begin
    dll_path := libvlc_dynamic_dll_file;
  end;

  // determine DLL architecture
  dll_arch := libvlc_dynamic_dll_architecture(dll_path);
{$IFDEF CPUX64}
  if (dll_arch = 32) then
  begin
    libvlc_dynamic_dll_error := 'Application is 64 bit, but ' + libvlc_dynamic_dll_file + ' library is 32 bit';
    if (libvlc_dynamic_dll_path <> '') then
    begin
      SetCurrentDir(cdir);
    end;
    exit;
  end;
{$ELSE}
  if (dll_arch = 64) then
  begin
    libvlc_dynamic_dll_error := 'Application is 32 bit, but ' + libvlc_dynamic_dll_file + ' library is 64 bit';
    if (libvlc_dynamic_dll_path <> '') then
    begin
      SetCurrentDir(cdir);
    end;
    exit;
  end;
{$ENDIF}

// if 64bit application, then we must preload libgcc_s_seh-1.dll
{$IFDEF MSWINDOWS}
  {$IFDEF CPUX64}
    LoadLibrary('libgcc_s_seh-1.dll');
  {$ENDIF}
{$ENDIF}


{$IFDEF SUPPORTS_UNICODE}
  libvlc_handle := LoadLibrary(PWideChar(dll_path));
{$ELSE}
  libvlc_handle := LoadLibrary(PAnsiChar(dll_path));
{$ENDIF}

  if (libvlc_dynamic_dll_path <> '') then
  begin
    SetCurrentDir(cdir);
  end;

  // exit, report error
  if (libvlc_handle = 0) then
  begin
    libvlc_dynamic_dll_error :=
      'Library not found ' + LIBVLC_DLL_NAME + ', '
{$IFDEF UNIX}
      + 'GetLastError() = ' + IntToStr(GetLastOSError())
{$ENDIF}
{$IFDEF MSWINDOWS}
      + 'GetLastError() = ' + IntToStr(GetLastError())
{$ENDIF}
{$IFDEF MACOS}
      + 'GetLastError() = ' + IntToStr(GetLastError())
{$ENDIF}
      + '';
    exit;
  end;
  
  libvlc_reset_function_pointers(TRUE);

  (****************************************************************************)

  if not libvlc_dll_get_proc_addr(@libvlc_get_version,
    'libvlc_get_version') then exit;

  vp := libvlc_get_version();
  libvlc_dynamic_dll_version := {$IFDEF DELPHI_XE2_UP}UTF8ToWideString{$ELSE}UTF8Decode{$ENDIF}(vp);

  va := read_dec_number(vp) and $ff;
  vb := read_dec_number(vp) and $ff;
  vc := read_dec_number(vp) and $ff;

  libvlc_dynamic_dll_version_bin := (va shl 16) or (vb shl 8) or vc;

  (****************************************************************************)

  if (libvlc_dynamic_dll_version_bin < $010000) then
  begin
    libvlc_dynamic_dll_error :=
      'Current libvlc version: ' + IntToStr(va) + '.' + IntToStr(vb) + '.' + IntToStr(vc) +
      ', minimum supported version is 1.0.0';
    exit;
  end;
  
  (****************************************************************************)

  if not libvlc_dll_get_proc_addr(@libvlc_new,
    'libvlc_new') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_release,
    'libvlc_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_retain,
    'libvlc_retain') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_add_intf,
    'libvlc_add_intf') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_wait,
    'libvlc_wait') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_get_compiler,
    'libvlc_get_compiler') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_get_changeset,
    'libvlc_get_changeset') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_event_attach,
    'libvlc_event_attach') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_event_detach,
    'libvlc_event_detach') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_event_type_name,
    'libvlc_event_type_name') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_get_log_verbosity,
    'libvlc_get_log_verbosity') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_set_log_verbosity,
    'libvlc_set_log_verbosity') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_log_open,
    'libvlc_log_open') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_log_close,
    'libvlc_log_close') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_log_count,
    'libvlc_log_count') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_log_clear,
    'libvlc_log_clear') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_log_get_iterator,
    'libvlc_log_get_iterator') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_log_iterator_free,
    'libvlc_log_iterator_free') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_log_iterator_has_next,
    'libvlc_log_iterator_has_next') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_log_iterator_next,
    'libvlc_log_iterator_next') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_new_as_node,
    'libvlc_media_new_as_node') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_add_option,
    'libvlc_media_add_option') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_retain,
    'libvlc_media_retain') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_release,
    'libvlc_media_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_get_mrl,
    'libvlc_media_get_mrl') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_duplicate,
    'libvlc_media_duplicate') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_get_meta,
    'libvlc_media_get_meta') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_get_state,
    'libvlc_media_get_state') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_subitems,
    'libvlc_media_subitems') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_event_manager,
    'libvlc_media_event_manager') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_get_duration,
    'libvlc_media_get_duration') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_set_user_data,
    'libvlc_media_set_user_data') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_get_user_data,
    'libvlc_media_get_user_data') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_new,
    'libvlc_media_player_new') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_new_from_media,
    'libvlc_media_player_new_from_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_release,
    'libvlc_media_player_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_retain,
    'libvlc_media_player_retain') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_set_media,
    'libvlc_media_player_set_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_media,
    'libvlc_media_player_get_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_event_manager,
    'libvlc_media_player_event_manager') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_is_playing,
    'libvlc_media_player_is_playing') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_play,
    'libvlc_media_player_play') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_pause,
    'libvlc_media_player_pause') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_stop,
    'libvlc_media_player_stop') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_set_nsobject,
    'libvlc_media_player_set_nsobject') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_nsobject,
    'libvlc_media_player_get_nsobject') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_set_agl,
    'libvlc_media_player_set_agl') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_agl,
    'libvlc_media_player_get_agl') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_set_xwindow,
    'libvlc_media_player_set_xwindow') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_xwindow,
    'libvlc_media_player_get_xwindow') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_set_hwnd,
    'libvlc_media_player_set_hwnd') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_hwnd,
    'libvlc_media_player_get_hwnd') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_length,
    'libvlc_media_player_get_length') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_time,
    'libvlc_media_player_get_time') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_set_time,
    'libvlc_media_player_set_time') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_position,
   'libvlc_media_player_get_position') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_set_position,
    'libvlc_media_player_set_position') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_set_chapter,
    'libvlc_media_player_set_chapter') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_chapter,
    'libvlc_media_player_get_chapter') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_chapter_count,
    'libvlc_media_player_get_chapter_count') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_will_play,
    'libvlc_media_player_will_play') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_chapter_count_for_title,
    'libvlc_media_player_get_chapter_count_for_title') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_set_title,
    'libvlc_media_player_set_title') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_title,
    'libvlc_media_player_get_title') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_title_count,
    'libvlc_media_player_get_title_count') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_previous_chapter,
    'libvlc_media_player_previous_chapter') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_next_chapter,
    'libvlc_media_player_next_chapter') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_set_rate,
    'libvlc_media_player_set_rate') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_rate,
    'libvlc_media_player_get_rate') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_state,
    'libvlc_media_player_get_state') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_get_fps,
    'libvlc_media_player_get_fps') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_has_vout,
    'libvlc_media_player_has_vout') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_is_seekable,
    'libvlc_media_player_is_seekable') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_can_pause,
    'libvlc_media_player_can_pause') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_track_description_release,
    'libvlc_track_description_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_toggle_fullscreen,
    'libvlc_toggle_fullscreen') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_set_fullscreen,
    'libvlc_set_fullscreen') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_get_fullscreen,
    'libvlc_get_fullscreen') then exit;

{$IFDEF USE_VLC_DEPRECATED_API}
  if not libvlc_dll_get_proc_addr(@libvlc_video_get_height,
    'libvlc_video_get_height') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_width,
    'libvlc_video_get_width') then exit;
{$ENDIF}

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_scale,
    'libvlc_video_get_scale') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_scale,
    'libvlc_video_set_scale') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_aspect_ratio,
    'libvlc_video_get_aspect_ratio') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_aspect_ratio,
    'libvlc_video_set_aspect_ratio') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_spu,
    'libvlc_video_get_spu') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_spu_count,
    'libvlc_video_get_spu_count') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_spu_description,
    'libvlc_video_get_spu_description') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_spu,
    'libvlc_video_set_spu') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_subtitle_file,
    'libvlc_video_set_subtitle_file') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_title_description,
    'libvlc_video_get_title_description') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_chapter_description,
    'libvlc_video_get_chapter_description') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_crop_geometry,
    'libvlc_video_get_crop_geometry') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_crop_geometry,
    'libvlc_video_set_crop_geometry') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_teletext,
    'libvlc_video_get_teletext') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_teletext,
    'libvlc_video_set_teletext') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_toggle_teletext,
    'libvlc_toggle_teletext') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_track_count,
    'libvlc_video_get_track_count') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_track_description,
    'libvlc_video_get_track_description') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_track,
    'libvlc_video_get_track') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_track,
    'libvlc_video_set_track') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_take_snapshot,
    'libvlc_video_take_snapshot') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_output_list_get,
    'libvlc_audio_output_list_get') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_output_list_release,
    'libvlc_audio_output_list_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_output_set,
    'libvlc_audio_output_set') then exit;

{$IFDEF USE_VLC_DEPRECATED_API}
  if not libvlc_dll_get_proc_addr(@libvlc_audio_output_device_count,
    'libvlc_audio_output_device_count') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_output_device_longname,
    'libvlc_audio_output_device_longname') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_output_device_id,
    'libvlc_audio_output_device_id') then exit;
{$ENDIF}

  if not libvlc_dll_get_proc_addr(@libvlc_audio_output_device_set,
    'libvlc_audio_output_device_set') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_output_get_device_type,
    'libvlc_audio_output_get_device_type') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_output_set_device_type,
    'libvlc_audio_output_set_device_type') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_toggle_mute,
    'libvlc_audio_toggle_mute') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_get_mute,
    'libvlc_audio_get_mute') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_set_mute,
    'libvlc_audio_set_mute') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_get_volume,
    'libvlc_audio_get_volume') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_set_volume,
    'libvlc_audio_set_volume') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_get_track_count,
    'libvlc_audio_get_track_count') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_get_track_description,
    'libvlc_audio_get_track_description') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_get_track,
    'libvlc_audio_get_track') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_set_track,
    'libvlc_audio_set_track') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_get_channel,
    'libvlc_audio_get_channel') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_set_channel,
    'libvlc_audio_set_channel') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_new,
    'libvlc_media_list_new') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_release,
    'libvlc_media_list_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_retain,
    'libvlc_media_list_retain') then exit;

{$IFDEF USE_VLC_DEPRECATED_API}
  if not libvlc_dll_get_proc_addr(@libvlc_media_list_add_file_content,
    'libvlc_media_list_add_file_content') then exit;
{$ENDIF}

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_set_media,
    'libvlc_media_list_set_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_media,
    'libvlc_media_list_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_add_media,
    'libvlc_media_list_add_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_insert_media,
    'libvlc_media_list_insert_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_remove_index,
    'libvlc_media_list_remove_index') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_count,
    'libvlc_media_list_count') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_item_at_index,
    'libvlc_media_list_item_at_index') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_index_of_item,
    'libvlc_media_list_index_of_item') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_is_readonly,
    'libvlc_media_list_is_readonly') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_lock,
    'libvlc_media_list_lock') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_unlock,
    'libvlc_media_list_unlock') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_event_manager,
    'libvlc_media_list_event_manager') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_new,
    'libvlc_media_list_player_new') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_release,
    'libvlc_media_list_player_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_set_media_player,
    'libvlc_media_list_player_set_media_player') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_set_media_list,
    'libvlc_media_list_player_set_media_list') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_play,
    'libvlc_media_list_player_play') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_pause,
    'libvlc_media_list_player_pause') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_is_playing,
    'libvlc_media_list_player_is_playing') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_get_state,
    'libvlc_media_list_player_get_state') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_play_item_at_index,
    'libvlc_media_list_player_play_item_at_index') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_play_item,
    'libvlc_media_list_player_play_item') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_stop,
    'libvlc_media_list_player_stop') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_next,
    'libvlc_media_list_player_next') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_library_new,
    'libvlc_media_library_new') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_library_release,
    'libvlc_media_library_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_library_retain,
    'libvlc_media_library_retain') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_library_load,
    'libvlc_media_library_load') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_library_media_list,
    'libvlc_media_library_media_list') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_discoverer_new_from_name,
    'libvlc_media_discoverer_new_from_name') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_discoverer_release,
    'libvlc_media_discoverer_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_discoverer_localized_name,
    'libvlc_media_discoverer_localized_name') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_discoverer_media_list,
    'libvlc_media_discoverer_media_list') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_discoverer_event_manager,
    'libvlc_media_discoverer_event_manager') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_discoverer_is_running,
    'libvlc_media_discoverer_is_running') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_release,
    'libvlc_vlm_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_add_broadcast,
    'libvlc_vlm_add_broadcast') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_add_vod,
    'libvlc_vlm_add_vod') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_del_media,
    'libvlc_vlm_del_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_set_enabled,
    'libvlc_vlm_set_enabled') then exit;
	
  if not libvlc_dll_get_proc_addr(@libvlc_vlm_set_output,
    'libvlc_vlm_set_output') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_set_input,
    'libvlc_vlm_set_input') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_add_input,
    'libvlc_vlm_add_input') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_set_loop,
    'libvlc_vlm_set_loop') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_set_mux,
    'libvlc_vlm_set_mux') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_change_media,
    'libvlc_vlm_change_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_play_media,
    'libvlc_vlm_play_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_stop_media,
    'libvlc_vlm_stop_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_pause_media,
    'libvlc_vlm_pause_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_seek_media,
    'libvlc_vlm_seek_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_show_media,
    'libvlc_vlm_show_media') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_get_media_instance_position,
    'libvlc_vlm_get_media_instance_position') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_get_media_instance_time,
    'libvlc_vlm_get_media_instance_time') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_get_media_instance_length,
    'libvlc_vlm_get_media_instance_length') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_get_media_instance_rate,
    'libvlc_vlm_get_media_instance_rate') then exit;

{$IFDEF IS_0_GT_1}
   if not libvlc_dll_get_proc_addr(@libvlc_vlm_get_media_instance_title,
     'libvlc_vlm_get_media_instance_title') then exit;
   if not libvlc_dll_get_proc_addr(@libvlc_vlm_get_media_instance_chapter,
     'libvlc_vlm_get_media_instance_chapter') then exit;
   if not libvlc_dll_get_proc_addr(@libvlc_vlm_get_media_instance_seekable,
     'libvlc_vlm_get_media_instance_seekable') then exit;
{$ENDIF}

{$IFDEF USE_VLC_DEPRECATED_API}
  if not libvlc_dll_get_proc_addr(@libvlc_playlist_play,
    'libvlc_playlist_play') then exit;
{$ENDIF}

  (****************************************************************************)
  if (libvlc_dynamic_dll_version_bin < $010100) then exit;
  (****************************************************************************)

  if not libvlc_dll_get_proc_addr(@libvlc_errmsg,
    'libvlc_errmsg') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_logo_int,
    'libvlc_video_get_logo_int') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_logo_int,
    'libvlc_video_set_logo_int') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_logo_string,
    'libvlc_video_set_logo_string') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_marquee_int,
    'libvlc_video_get_marquee_int') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_marquee_string,
    'libvlc_video_get_marquee_string') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_marquee_int,
    'libvlc_video_set_marquee_int') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_marquee_string,
    'libvlc_video_set_marquee_string') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_deinterlace,
    'libvlc_video_set_deinterlace') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_cursor,
    'libvlc_video_get_cursor') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_size,
    'libvlc_video_get_size') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_key_input,
    'libvlc_video_set_key_input') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_mouse_input,
    'libvlc_video_set_mouse_input') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_clearerr,
    'libvlc_clearerr') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vprinterr,
    'libvlc_vprinterr') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_printerr,
    'libvlc_printerr') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_event_manager,
    'libvlc_media_list_player_event_manager') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_set_playback_mode,
    'libvlc_media_list_player_set_playback_mode') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_previous,
    'libvlc_media_list_player_previous') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_next_frame,
    'libvlc_media_player_next_frame') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_vlm_get_event_manager,
    'libvlc_vlm_get_event_manager') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_parse,
    'libvlc_media_parse') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_parse_async,
    'libvlc_media_parse_async') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_is_parsed,
    'libvlc_media_is_parsed') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_new_location,
    'libvlc_media_new_location') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_new_path,
    'libvlc_media_new_path') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_new_callbacks,
    'libvlc_media_new_callbacks') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_add_option_flag,
    'libvlc_media_add_option_flag') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_set_meta,
    'libvlc_media_set_meta') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_save_meta,
    'libvlc_media_save_meta') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_get_stats,
    'libvlc_media_get_stats') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_get_tracks_info,
    'libvlc_media_get_tracks_info') then exit;

  (****************************************************************************)
  if (libvlc_dynamic_dll_version_bin < $010101) then exit;
  (****************************************************************************)
    
  if not libvlc_dll_get_proc_addr(@libvlc_video_set_format,
    'libvlc_video_set_format') then exit;
    
  if not libvlc_dll_get_proc_addr(@libvlc_set_user_agent,
    'libvlc_set_user_agent') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_set_pause,
    'libvlc_media_player_set_pause') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_callbacks,
    'libvlc_video_set_callbacks') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_get_delay,
    'libvlc_audio_get_delay') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_set_delay,
    'libvlc_audio_set_delay') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_adjust_int,
    'libvlc_video_get_adjust_int') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_adjust_int,
    'libvlc_video_set_adjust_int') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_adjust_float,
    'libvlc_video_get_adjust_float') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_adjust_float,
    'libvlc_video_set_adjust_float') then exit;

  (****************************************************************************)
  if (libvlc_dynamic_dll_version_bin < $010105) then exit;
  (****************************************************************************)

  if not libvlc_dll_get_proc_addr(@libvlc_media_new_fd,
    'libvlc_media_new_fd') then exit;

  (****************************************************************************)
  if (libvlc_dynamic_dll_version_bin < $020000) then exit;
  (****************************************************************************)

  if not libvlc_dll_get_proc_addr(@libvlc_video_filter_list_get,
    'libvlc_video_filter_list_get') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_filter_list_get,
    'libvlc_audio_filter_list_get') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_module_description_list_release,
    'libvlc_module_description_list_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_set_exit_handler,
    'libvlc_set_exit_handler') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_free,
    'libvlc_free') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_format_callbacks,
    'libvlc_video_set_format_callbacks') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_set_callbacks,
    'libvlc_audio_set_callbacks') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_set_volume_callback,
    'libvlc_audio_set_volume_callback') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_set_format_callbacks,
    'libvlc_audio_set_format_callbacks') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_set_format,
    'libvlc_audio_set_format') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_navigate,
    'libvlc_media_player_navigate') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_get_spu_delay,
    'libvlc_video_get_spu_delay') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_video_set_spu_delay,
    'libvlc_video_set_spu_delay') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_list_player_retain,
    'libvlc_media_list_player_retain') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_track_description_list_release,
    'libvlc_track_description_list_release') then exit;

  (****************************************************************************)
  if (libvlc_dynamic_dll_version_bin < $020100) then exit;
  (****************************************************************************)

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_set_video_title_display,
    'libvlc_media_player_set_video_title_display') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_output_device_list_get,
    'libvlc_audio_output_device_list_get') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_output_device_list_release,
    'libvlc_audio_output_device_list_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_tracks_get,
    'libvlc_media_tracks_get') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_tracks_release,
    'libvlc_media_tracks_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_set_app_id,
    'libvlc_set_app_id') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_log_get_context,
    'libvlc_log_get_context') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_log_get_object,
    'libvlc_log_get_object') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_log_set,
    'libvlc_log_set') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_log_unset,
    'libvlc_log_unset') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_log_set_file,
    'libvlc_log_set_file') then exit;

  (****************************************************************************)
  if (libvlc_dynamic_dll_version_bin < $020200) then exit;
  (****************************************************************************)

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_program_scrambled,
    'libvlc_media_player_program_scrambled') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_output_device_enum,
    'libvlc_audio_output_device_enum') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_equalizer_get_preset_count,
    'libvlc_audio_equalizer_get_preset_count') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_equalizer_get_preset_name,
    'libvlc_audio_equalizer_get_preset_name') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_equalizer_get_band_count,
    'libvlc_audio_equalizer_get_band_count') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_equalizer_get_band_frequency,
    'libvlc_audio_equalizer_get_band_frequency') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_equalizer_new,
    'libvlc_audio_equalizer_new') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_equalizer_new_from_preset,
    'libvlc_audio_equalizer_new_from_preset') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_equalizer_release,
    'libvlc_audio_equalizer_release') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_equalizer_set_preamp,
    'libvlc_audio_equalizer_set_preamp') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_equalizer_get_preamp,
    'libvlc_audio_equalizer_get_preamp') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_equalizer_set_amp_at_index,
    'libvlc_audio_equalizer_set_amp_at_index') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_audio_equalizer_get_amp_at_index,
    'libvlc_audio_equalizer_get_amp_at_index') then exit;

  if not libvlc_dll_get_proc_addr(@libvlc_media_player_set_equalizer,
    'libvlc_media_player_set_equalizer') then exit;

  (****************************************************************************)
  if (libvlc_dynamic_dll_version_bin < $030000) then exit;
  (****************************************************************************)
end;

procedure libvlc_dynamic_dll_init();
begin
  if (libvlc_handle <> 0) then exit;
  libvlc_dynamic_dll_init_with_path(libvlc_get_install_path());
end;

procedure libvlc_dynamic_dll_done();
begin
  if (libvlc_handle <> 0) then
  begin
    {$IFDEF MSWINDOWS}
    if FreeLibrary(libvlc_handle) then
    {$ENDIF}
    {$IFDEF MACOS}
    if FreeLibrary(libvlc_handle) then
    {$ENDIF}
    {$IFDEF UNIX}
    if UnloadLibrary(libvlc_handle) then
    {$ENDIF}
    begin
      libvlc_handle := 0;
    end;
  end;
  
  libvlc_reset_function_pointers();
end;

(******************************************************************************)
(* GDK_WINDOW_HWND is not defined in GDK2 for WIN32
(******************************************************************************)

{$IFDEF MSWINDOWS}
{$IFDEF LCLGTK2}

type
  PGTypeClass = ^TGTypeClass;
  TGTypeClass = packed record
    g_type : Cardinal;
  end;

  TGTypeInstance = packed record
    g_class : PGTypeClass;
  end;

  TGObject = packed record
    g_type_instance : TGTypeInstance;
    ref_count : Cardinal;
    qdata : Pointer;
  end;

  PGdkDrawable = ^TGdkDrawable;
  TGdkDrawable = packed record
   parent_instance : TGObject;
  end;

  PGdkWindowObject = ^TGdkWindowObject;
  TGdkWindowObject = packed record
    parent_instance : TGdkDrawable;
    impl : PGdkDrawable;
    parent : PGdkWindowObject;
    user_data : Pointer;
  end;


function gdk_win32_drawable_get_handle(d : PGdkDrawable) : THandle; cdecl; external 'libgdk-win32-2.0-0.dll';

function GDK_WINDOW_HWND(window : Pointer) : THandle;
begin
  Result := 0;
  if (window <> NIL) then
  begin
    Result := gdk_win32_drawable_get_handle(PGdkWindowObject(window)^.impl);
  end;
end;

{$ENDIF}
{$ENDIF}

// http://www.videolan.org/developers/vlc/doc/doxygen/html/group__libvlc.html

{$IFDEF MACOS}
procedure libvlc_media_player_set_display_window(
  p_mi : libvlc_media_player_t_ptr;
  window_handle : Pointer);
{$ELSE}
procedure libvlc_media_player_set_display_window(
  p_mi : libvlc_media_player_t_ptr;
  window_handle : THandle);
{$ENDIF}
begin
  {$IFDEF FPC}
    {$IFDEF MSWINDOWS}
      {$IFDEF LCLGTK2}
        libvlc_media_player_set_hwnd(p_mi, GDK_WINDOW_HWND(PGtkWidget(window_handle)^.window));
        exit;
      {$ENDIF}

      {$IFDEF LCLQT}
        libvlc_media_player_set_hwnd(p_mi, QWidget_winid(TQtWidget(window_handle).Widget));
        exit;
      {$ENDIF}

      libvlc_media_player_set_hwnd(p_mi, window_handle); // LCL WIN32
      exit;
    {$ENDIF}

    {$IFDEF DARWIN}
      {$IFDEF LCLCARBON}
        // libvlc_media_player_set_nsobject(p_mi, TCarbonWidget(window_handle).Widget); // Carbon ???
        // NSView *video = [[NSView alloc] init]; QMacCocoaViewContainer *container = new QMacCocoaViewContainer(video, parent); libvlc_media_player_set_nsobject(mp, video);
        // uses
        //   CocoaAll;
        //  {$mode objfpc}{$H+}
        //  {$modeswitch objectivec1}
        // uses
        //   FPCMacOSAll, AppKit, Foundation;
        // svn co http://svn.freepascal.org/svn/lazarus/trunk/lcl/interfaces/carbon/objc objc
        // svn co http://svn.freepascal.org/svn/lazarus/trunk/lcl/interfaces/carbon/pascocoa pascocoa
        // http://wiki.freepascal.org/PasCocoa
        exit;
      {$ENDIF}
      exit;
    {$ENDIF}

    {$IFDEF UNIX}
      {$IFDEF LCLGTK2}
        libvlc_media_player_set_xwindow(p_mi, GDK_WINDOW_XID(PGtkWidget(window_handle)^.window));
        exit;
      {$ENDIF}
      {$IFDEF LCLQT}
        libvlc_media_player_set_xwindow(p_mi, QWidget_winid(TQtWidget(window_handle).Widget));
        exit;
      {$ENDIF}
      libvlc_media_player_set_xwindow(p_mi, window_handle); // NOT GTK2 AND NOT QT
      exit;
    {$ENDIF}
    exit;
  {$ENDIF}

  // DELPHI

  {$IFDEF MSWINDOWS}
    libvlc_media_player_set_hwnd(p_mi, window_handle); // LCL WIN32
    exit;
  {$ENDIF}

  {$IFDEF MACOS}
    libvlc_media_player_set_nsobject(p_mi, window_handle);
    exit;
  {$ENDIF}

  {$IFDEF ANDROID}
    exit;
  {$ENDIF}
end;


initialization

  libvlc_handle := 0;

  libvlc_dynamic_dll_path        := '';
  libvlc_dynamic_dll_file        := '';
  libvlc_dynamic_dll_version     := '';
  libvlc_dynamic_dll_version_bin := 0;
  libvlc_dynamic_dll_error       := '';
  
finalization

  // without this code Lazarus heaptrc report memory leaks
  libvlc_dynamic_dll_path    := '';
  libvlc_dynamic_dll_file    := '';
  libvlc_dynamic_dll_version := '';
  libvlc_dynamic_dll_error   := '';

end.
