export {}

const html = "<h2 style=\"font-size: 3.5rem; line-height: 3.5rem;\">The Basics</h2><div class=\"preamble\" style=\"font-size: 16px; font-weight: 400;\"><p>Welcome to the first page of the handbook, if this is your first experience with TypeScript - you may want to start at one of the '<a href=\"https://www.typescriptlang.org/docs/handbook/intro.html#get-started\">Getting Started</a>' guides</p></div><article style=\"display: flex; width: 1105px; font-size: 16px; font-weight: 400;\"><div class=\"whitespace raised\" style=\"box-shadow: var(--raised-box-shadow); margin: 0px auto 1rem; overflow: hidden; padding: 2rem;\"><div class=\"markdown\"><p>Each and every value in JavaScript has a set of behaviors you can observe from running different operations. That sounds abstract, but as a quick example, consider some operations we might run on a variable named&nbsp;<code style=\"font-family: var(--code-font);\">message</code>.</p><pre class=\"shiki light-plus\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 128, 0);\">// Accessing the property 'toLowerCase'</span></div><div class=\"line\"><span style=\"color: rgb(0, 128, 0);\">// on 'message' and then calling it</span></div><div class=\"line\"><span style=\"color: rgb(0, 16, 128);\">message</span>.<span style=\"color: rgb(121, 94, 38);\">toLowerCase</span>();</div><div class=\"line\"></div><div class=\"line\"><span style=\"color: rgb(0, 128, 0);\">// Calling 'message'</span></div><div class=\"line\"><span style=\"color: rgb(121, 94, 38);\">message</span>();</div></code></div></pre><p>If we break this down, the first runnable line of code accesses a property called&nbsp;<code style=\"font-family: var(--code-font);\">toLowerCase</code>&nbsp;and then calls it. The second one tries to call&nbsp;<code style=\"font-family: var(--code-font);\">message</code>&nbsp;directly.</p><p>But assuming we don’t know the value of&nbsp;<code style=\"font-family: var(--code-font);\">message</code>&nbsp;- and that’s pretty common - we can’t reliably say what results we’ll get from trying to run any of this code. The behavior of each operation depends entirely on what value we had in the first place.</p><ul><li>Is&nbsp;<code style=\"font-family: var(--code-font);\">message</code>&nbsp;callable?</li><li>Does it have a property called&nbsp;<code style=\"font-family: var(--code-font);\">toLowerCase</code>&nbsp;on it?</li><li>If it does, is&nbsp;<code style=\"font-family: var(--code-font);\">toLowerCase</code>&nbsp;even callable?</li><li>If both of these values are callable, what do they return?</li></ul><p>The answers to these questions are usually things we keep in our heads when we write JavaScript, and we have to hope we got all the details right.</p><p>Let’s say&nbsp;<code style=\"font-family: var(--code-font);\">message</code>&nbsp;was defined in the following way.</p><pre class=\"shiki light-plus\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">const</span> <span style=\"color: rgb(0, 112, 193);\">message</span> = <span style=\"color: rgb(163, 21, 21);\">\"Hello World!\"</span>;</div></code></div></pre><p>As you can probably guess, if we try to run&nbsp;<code style=\"font-family: var(--code-font);\">message.toLowerCase()</code>, we’ll get the same string only in lower-case.</p><p>What about that second line of code? If you’re familiar with JavaScript, you’ll know this fails with an exception:</p><pre class=\"shiki light-plus\"><div class=\"code-container\"><code><div class=\"line\">TypeError: message is not a function</div></code></div></pre><p>It’d be great if we could avoid mistakes like this.</p><p>When we run our code, the way that our JavaScript runtime chooses what to do is by figuring out the&nbsp;<em>type</em>&nbsp;of the value - what sorts of behaviors and capabilities it has. That’s part of what that&nbsp;<code style=\"font-family: var(--code-font);\">TypeError</code>&nbsp;is alluding to - it’s saying that the string&nbsp;<code style=\"font-family: var(--code-font);\">\"Hello World!\"</code>&nbsp;cannot be called as a function.</p><p>For some values, such as the primitives&nbsp;<code style=\"font-family: var(--code-font);\">string</code>&nbsp;and&nbsp;<code style=\"font-family: var(--code-font);\">number</code>, we can identify their type at runtime using the&nbsp;<code style=\"font-family: var(--code-font);\">typeof</code>&nbsp;operator. But for other things like functions, there’s no corresponding runtime mechanism to identify their types. For example, consider this function:</p><pre class=\"shiki light-plus\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">function</span> <span style=\"color: rgb(121, 94, 38);\">fn</span>(<span style=\"color: rgb(0, 16, 128);\">x</span>) {</div><div class=\"line\">  <span style=\"color: rgb(175, 0, 219);\">return</span> <span style=\"color: rgb(0, 16, 128);\">x</span>.<span style=\"color: rgb(121, 94, 38);\">flip</span>();</div><div class=\"line\">}</div></code></div></pre><p>We can&nbsp;<em>observe</em>&nbsp;by reading the code that this function will only work if given an object with a callable&nbsp;<code style=\"font-family: var(--code-font);\">flip</code>&nbsp;property, but JavaScript doesn’t surface this information in a way that we can check while the code is running. The only way in pure JavaScript to tell what&nbsp;<code style=\"font-family: var(--code-font);\">fn</code>&nbsp;does with a particular value is to call it and see what happens. This kind of behavior makes it hard to predict what code will do before it runs, which means it’s harder to know what your code is going to do while you’re writing it.</p><p>Seen in this way, a&nbsp;<em>type</em>&nbsp;is the concept of describing which values can be passed to&nbsp;<code style=\"font-family: var(--code-font);\">fn</code>&nbsp;and which will crash. JavaScript only truly provides&nbsp;<em>dynamic</em>&nbsp;typing - running the code to see what happens.</p><p>The alternative is to use a&nbsp;<em>static</em>&nbsp;type system to make predictions about what code is expected&nbsp;<em>before</em>&nbsp;it runs.</p><h2 id=\"static-type-checking\" style=\"position: relative;\"><a href=\"https://www.typescriptlang.org/docs/handbook/2/basic-types.html#static-type-checking\" aria-label=\"static type checking permalink\" class=\"anchor before\"><svg aria-hidden=\"true\" focusable=\"false\" height=\"16\" version=\"1.1\" viewBox=\"0 0 16 16\" width=\"16\"><path fill-rule=\"evenodd\" d=\"M4 9h1v1H4c-1.5 0-3-1.69-3-3.5S2.55 3 4 3h4c1.45 0 3 1.69 3 3.5 0 1.41-.91 2.72-2 3.25V8.59c.58-.45 1-1.27 1-2.09C10 5.22 8.98 4 8 4H4c-.98 0-2 1.22-2 2.5S3 9 4 9zm9-3h-1v1h1c1 0 2 1.22 2 2.5S13.98 12 13 12H9c-.98 0-2-1.22-2-2.5 0-.83.42-1.64 1-2.09V6.25c-1.09.53-2 1.84-2 3.25C6 11.31 7.55 13 9 13h4c1.45 0 3-1.69 3-3.5S14.5 6 13 6z\"></path></svg></a>Static type-checking</h2><p>Think back to that&nbsp;<code style=\"font-family: var(--code-font);\">TypeError</code>&nbsp;we got earlier from trying to call a&nbsp;<code style=\"font-family: var(--code-font);\">string</code>&nbsp;as a function.&nbsp;<em>Most people</em>&nbsp;don’t like to get any sorts of errors when running their code - those are considered bugs! And when we write new code, we try our best to avoid introducing new bugs.</p><p>If we add just a bit of code, save our file, re-run the code, and immediately see the error, we might be able to isolate the problem quickly; but that’s not always the case. We might not have tested the feature thoroughly enough, so we might never actually run into a potential error that would be thrown! Or if we were lucky enough to witness the error, we might have ended up doing large refactorings and adding a lot of different code that we’re forced to dig through.</p><p>Ideally, we could have a tool that helps us find these bugs&nbsp;<em>before</em>&nbsp;our code runs. That’s what a static type-checker like TypeScript does.&nbsp;<em>Static types systems</em>&nbsp;describe the shapes and behaviors of what our values will be when we run our programs. A type-checker like TypeScript uses that information and tells us when things might be going off the rails.</p><pre class=\"shiki light-plus twoslash lsp\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">const</span> <span style=\"color: rgb(0, 112, 193);\"><data-lsp lsp=\"const message: &quot;hello!&quot;\">message</data-lsp></span> = <span style=\"color: rgb(163, 21, 21);\">\"hello!\"</span>;</div><div class=\"line\">&nbsp;</div><div class=\"line\"><span style=\"color: rgb(121, 94, 38);\"><data-err><data-lsp lsp=\"const message: &quot;hello!&quot;\">message</data-lsp></data-err></span>();</div><span class=\"error\">This expression is not callable.\n  Type 'String' has no call signatures.</span><span class=\"error-behind\">This expression is not callable.\n  Type 'String' has no call signatures.</span></code></div></pre><p>Running that last sample with TypeScript will give us an error message before we run the code in the first place.</p><h2 id=\"non-exception-failures\" style=\"position: relative;\"><a href=\"https://www.typescriptlang.org/docs/handbook/2/basic-types.html#non-exception-failures\" aria-label=\"non exception failures permalink\" class=\"anchor before\"><svg aria-hidden=\"true\" focusable=\"false\" height=\"16\" version=\"1.1\" viewBox=\"0 0 16 16\" width=\"16\"><path fill-rule=\"evenodd\" d=\"M4 9h1v1H4c-1.5 0-3-1.69-3-3.5S2.55 3 4 3h4c1.45 0 3 1.69 3 3.5 0 1.41-.91 2.72-2 3.25V8.59c.58-.45 1-1.27 1-2.09C10 5.22 8.98 4 8 4H4c-.98 0-2 1.22-2 2.5S3 9 4 9zm9-3h-1v1h1c1 0 2 1.22 2 2.5S13.98 12 13 12H9c-.98 0-2-1.22-2-2.5 0-.83.42-1.64 1-2.09V6.25c-1.09.53-2 1.84-2 3.25C6 11.31 7.55 13 9 13h4c1.45 0 3-1.69 3-3.5S14.5 6 13 6z\"></path></svg></a>Non-exception Failures</h2><p>So far we’ve been discussing certain things like runtime errors - cases where the JavaScript runtime tells us that it thinks something is nonsensical. Those cases come up because&nbsp;<a href=\"https://tc39.github.io/ecma262/\">the ECMAScript specification</a>&nbsp;has explicit instructions on how the language should behave when it runs into something unexpected.</p><p>For example, the specification says that trying to call something that isn’t callable should throw an error. Maybe that sounds like “obvious behavior”, but you could imagine that accessing a property that doesn’t exist on an object should throw an error too. Instead, JavaScript gives us different behavior and returns the value&nbsp;<code style=\"font-family: var(--code-font);\">undefined</code>:</p><pre class=\"shiki light-plus\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">const</span> <span style=\"color: rgb(0, 112, 193);\">user</span> = {</div><div class=\"line\">  <span style=\"color: rgb(0, 16, 128);\">name:</span> <span style=\"color: rgb(163, 21, 21);\">\"Daniel\"</span>,</div><div class=\"line\">  <span style=\"color: rgb(0, 16, 128);\">age:</span> <span style=\"color: rgb(9, 134, 88);\">26</span>,</div><div class=\"line\">};</div><div class=\"line\"></div><div class=\"line\"><span style=\"color: rgb(0, 16, 128);\">user</span>.<span style=\"color: rgb(0, 16, 128);\">location</span>; <span style=\"color: rgb(0, 128, 0);\">// returns undefined</span></div></code></div></pre><p>Ultimately, a static type system has to make the call over what code should be flagged as an error in its system, even if it’s “valid” JavaScript that won’t immediately throw an error. In TypeScript, the following code produces an error about&nbsp;<code style=\"font-family: var(--code-font);\">location</code>&nbsp;not being defined:</p><pre class=\"shiki light-plus twoslash lsp\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">const</span> <span style=\"color: rgb(0, 112, 193);\"><data-lsp lsp=\"const user: {\n    name: string;\n    age: number;\n}\">user</data-lsp></span> = {</div><div class=\"line\">  <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(property) name: string\">name</data-lsp>:</span> <span style=\"color: rgb(163, 21, 21);\">\"Daniel\"</span>,</div><div class=\"line\">  <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(property) age: number\">age</data-lsp>:</span> <span style=\"color: rgb(9, 134, 88);\">26</span>,</div><div class=\"line\">};</div><div class=\"line\">&nbsp;</div><div class=\"line\"><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"const user: {\n    name: string;\n    age: number;\n}\">user</data-lsp></span>.<span style=\"color: rgb(0, 16, 128);\"><data-err><data-lsp lsp=\"any\">location</data-lsp></data-err></span>;</div><span class=\"error\">Property 'location' does not exist on type '{ name: string; age: number; }'.</span><span class=\"error-behind\">Property 'location' does not exist on type '{ name: string; age: number; }'.</span></code></div></pre><p>While sometimes that implies a trade-off in what you can express, the intent is to catch legitimate bugs in our programs. And TypeScript catches&nbsp;<em>a lot</em>&nbsp;of legitimate bugs.</p><p>For example: typos,</p><pre class=\"shiki light-plus twoslash lsp\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">const</span> <span style=\"color: rgb(0, 112, 193);\"><data-lsp lsp=\"const announcement: &quot;Hello World!&quot;\">announcement</data-lsp></span> = <span style=\"color: rgb(163, 21, 21);\">\"Hello World!\"</span>;</div><div class=\"line\">&nbsp;</div><div class=\"line\"><span style=\"color: rgb(0, 128, 0);\">// How quickly can you spot the typos?</span></div><div class=\"line\"><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"const announcement: &quot;Hello World!&quot;\">announcement</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"any\">toLocaleLowercase</data-lsp></span>();</div><div class=\"line\"><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"const announcement: &quot;Hello World!&quot;\">announcement</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"any\">toLocalLowerCase</data-lsp></span>();</div><div class=\"line\">&nbsp;</div><div class=\"line\"><span style=\"color: rgb(0, 128, 0);\">// We probably meant to write this...</span></div><div class=\"line\"><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"const announcement: &quot;Hello World!&quot;\">announcement</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(method) String.toLocaleLowerCase(locales?: string | string[] | undefined): string\">toLocaleLowerCase</data-lsp></span>();</div></code></div></pre><p>uncalled functions,</p><pre class=\"shiki light-plus twoslash lsp\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">function</span> <span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"function flipCoin(): boolean\">flipCoin</data-lsp></span>() {</div><div class=\"line\">  <span style=\"color: rgb(0, 128, 0);\">// Meant to be Math.random()</span></div><div class=\"line\">  <span style=\"color: rgb(175, 0, 219);\">return</span> <span style=\"color: rgb(28, 98, 119);\"><data-lsp lsp=\"var Math: Math\">Math</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(method) Math.random(): number\">random</data-lsp></span> &lt; <span style=\"color: rgb(9, 134, 88);\">0.5</span>;</div><span class=\"error\">Operator '&lt;' cannot be applied to types '() =&gt; number' and 'number'.</span><span class=\"error-behind\">Operator '&lt;' cannot be applied to types '() =&gt; number' and 'number'.</span><div class=\"line\">}</div></code></div></pre><p>or basic logic errors.</p><pre class=\"shiki light-plus twoslash lsp\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">const</span> <span style=\"color: rgb(0, 112, 193);\"><data-lsp lsp=\"const value: &quot;a&quot; | &quot;b&quot;\">value</data-lsp></span> = <span style=\"color: rgb(28, 98, 119);\"><data-lsp lsp=\"var Math: Math\">Math</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(method) Math.random(): number\">random</data-lsp></span>() &lt; <span style=\"color: rgb(9, 134, 88);\">0.5</span> ? <span style=\"color: rgb(163, 21, 21);\">\"a\"</span> : <span style=\"color: rgb(163, 21, 21);\">\"b\"</span>;</div><div class=\"line\"><span style=\"color: rgb(175, 0, 219);\">if</span> (<span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"const value: &quot;a&quot; | &quot;b&quot;\">value</data-lsp></span> !== <span style=\"color: rgb(163, 21, 21);\">\"a\"</span>) {</div><div class=\"line\">  <span style=\"color: rgb(0, 128, 0);\">// ...</span></div><div class=\"line\">} <span style=\"color: rgb(175, 0, 219);\">else</span> <span style=\"color: rgb(175, 0, 219);\">if</span> (<span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"const value: &quot;a&quot;\">value</data-lsp></span> === <span style=\"color: rgb(163, 21, 21);\">\"b\"</span>) {</div><span class=\"error\">This condition will always return 'false' since the types '\"a\"' and '\"b\"' have no overlap.</span><span class=\"error-behind\">This condition will always return 'false' since the types '\"a\"' and '\"b\"' have no overlap.</span><div class=\"line\">  <span style=\"color: rgb(0, 128, 0);\">// Oops, unreachable</span></div><div class=\"line\">}</div></code></div></pre><h2 id=\"types-for-tooling\" style=\"position: relative;\"><a href=\"https://www.typescriptlang.org/docs/handbook/2/basic-types.html#types-for-tooling\" aria-label=\"types for tooling permalink\" class=\"anchor before\"><svg aria-hidden=\"true\" focusable=\"false\" height=\"16\" version=\"1.1\" viewBox=\"0 0 16 16\" width=\"16\"><path fill-rule=\"evenodd\" d=\"M4 9h1v1H4c-1.5 0-3-1.69-3-3.5S2.55 3 4 3h4c1.45 0 3 1.69 3 3.5 0 1.41-.91 2.72-2 3.25V8.59c.58-.45 1-1.27 1-2.09C10 5.22 8.98 4 8 4H4c-.98 0-2 1.22-2 2.5S3 9 4 9zm9-3h-1v1h1c1 0 2 1.22 2 2.5S13.98 12 13 12H9c-.98 0-2-1.22-2-2.5 0-.83.42-1.64 1-2.09V6.25c-1.09.53-2 1.84-2 3.25C6 11.31 7.55 13 9 13h4c1.45 0 3-1.69 3-3.5S14.5 6 13 6z\"></path></svg></a>Types for Tooling</h2><p>TypeScript can catch bugs when we make mistakes in our code. That’s great, but TypeScript can&nbsp;<em>also</em>&nbsp;prevent us from making those mistakes in the first place.</p><p>The type-checker has information to check things like whether we’re accessing the right properties on variables and other properties. Once it has that information, it can also start&nbsp;<em>suggesting</em>&nbsp;which properties you might want to use.</p><p>That means TypeScript can be leveraged for editing code too, and the core type-checker can provide error messages and code completion as you type in the editor. That’s part of what people often refer to when they talk about tooling in TypeScript.</p><pre class=\"shiki light-plus twoslash lsp\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(175, 0, 219);\">import</span> <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(alias) function express(): core.Express\n(alias) namespace express\nimport express\">express</data-lsp></span> <span style=\"color: rgb(175, 0, 219);\">from</span> <span style=\"color: rgb(163, 21, 21);\">\"express\"</span>;</div><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">const</span> <span style=\"color: rgb(0, 112, 193);\"><data-lsp lsp=\"const app: Express\">app</data-lsp></span> = <span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(alias) express(): Express\nimport express\">express</data-lsp></span>();</div><div class=\"line\">&nbsp;</div><div class=\"line\"><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"const app: Express\">app</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(property) Application.get: <ParamsDictionary, any, any, QueryString.ParsedQs>(path: PathParams, ...handlers: RequestHandler<ParamsDictionary, any, any, QueryString.ParsedQs>[]) => Express (+3 overloads)\">get</data-lsp></span>(<span style=\"color: rgb(163, 21, 21);\">\"/\"</span>, <span style=\"color: rgb(0, 0, 255);\">function</span> (<span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) req: Request<ParamsDictionary, any, any, QueryString.ParsedQs>\">req</data-lsp></span>, <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) res: Response<any>\">res</data-lsp></span>) {</div><div class=\"line\">  <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) res: Response<any>\">res</data-lsp></span>.<span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"any\">sen</data-lsp></span></div><div class=\"meta-line\">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class=\"inline-completions\"><ul class=\"dropdown\" style=\"font-family: var(--code-font);\"><li class=\"\"><span class=\"result-found\">sen</span>d</li><li class=\"\"><span class=\"result-found\">sen</span>dDate</li><li class=\"deprecated\"><span class=\"result-found\">sen</span>dfile</li><li class=\"\"><span class=\"result-found\">sen</span>dFile</li><li class=\"\"><span class=\"result-found\">sen</span>dStatus</li></ul></span></div><div class=\"line\">});</div><div class=\"line\">&nbsp;</div><div class=\"line\"><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"const app: Express\">app</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(method) Application.listen(port: number, callback?: ((...args: any[]) => void) | undefined): Server (+5 overloads)\">listen</data-lsp></span>(<span style=\"color: rgb(9, 134, 88);\">3000</span>);</div></code></div></pre><p>TypeScript takes tooling seriously, and that goes beyond completions and errors as you type. An editor that supports TypeScript can deliver “quick fixes” to automatically fix errors, refactorings to easily re-organize code, and useful navigation features for jumping to definitions of a variable, or finding all references to a given variable. All of this is built on top of the type-checker and is fully cross-platform, so it’s likely that&nbsp;<a href=\"https://github.com/Microsoft/TypeScript/wiki/TypeScript-Editor-Support\">your favorite editor has TypeScript support available</a>.</p><h2 id=\"tsc-the-typescript-compiler\" style=\"position: relative;\"><a href=\"https://www.typescriptlang.org/docs/handbook/2/basic-types.html#tsc-the-typescript-compiler\" aria-label=\"tsc the typescript compiler permalink\" class=\"anchor before\"><svg aria-hidden=\"true\" focusable=\"false\" height=\"16\" version=\"1.1\" viewBox=\"0 0 16 16\" width=\"16\"><path fill-rule=\"evenodd\" d=\"M4 9h1v1H4c-1.5 0-3-1.69-3-3.5S2.55 3 4 3h4c1.45 0 3 1.69 3 3.5 0 1.41-.91 2.72-2 3.25V8.59c.58-.45 1-1.27 1-2.09C10 5.22 8.98 4 8 4H4c-.98 0-2 1.22-2 2.5S3 9 4 9zm9-3h-1v1h1c1 0 2 1.22 2 2.5S13.98 12 13 12H9c-.98 0-2-1.22-2-2.5 0-.83.42-1.64 1-2.09V6.25c-1.09.53-2 1.84-2 3.25C6 11.31 7.55 13 9 13h4c1.45 0 3-1.69 3-3.5S14.5 6 13 6z\"></path></svg></a><code>tsc</code>, the TypeScript compiler</h2><p>We’ve been talking about type-checking, but we haven’t yet used our type-<em>checker</em>. Let’s get acquainted with our new friend&nbsp;<code style=\"font-family: var(--code-font);\">tsc</code>, the TypeScript compiler. First we’ll need to grab it via npm.</p><pre class=\"shiki light-plus\"><div class=\"code-container\"><code><div class=\"line\">npm install -g typescript</div></code></div></pre><blockquote style=\"border-left: 2px solid var(--border-color);\"><p>This installs the TypeScript Compiler&nbsp;<code style=\"font-family: var(--code-font);\">tsc</code>&nbsp;globally. You can use&nbsp;<code style=\"font-family: var(--code-font);\">npx</code>&nbsp;or similar tools if you’d prefer to run&nbsp;<code style=\"font-family: var(--code-font);\">tsc</code>&nbsp;from a local&nbsp;<code style=\"font-family: var(--code-font);\">node_modules</code>&nbsp;package instead.</p></blockquote><p>Now let’s move to an empty folder and try writing our first TypeScript program:&nbsp;<code style=\"font-family: var(--code-font);\">hello.ts</code>:</p><pre class=\"shiki light-plus twoslash lsp\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 128, 0);\">// Greets the world.</span></div><div class=\"line\"><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"var console: Console\">console</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(method) Console.log(...data: any[]): void\">log</data-lsp></span>(<span style=\"color: rgb(163, 21, 21);\">\"Hello world!\"</span>);</div></code></div></pre><p>Notice there are no frills here; this “hello world” program looks identical to what you’d write for a “hello world” program in JavaScript. And now let’s type-check it by running the command&nbsp;<code style=\"font-family: var(--code-font);\">tsc</code>&nbsp;which was installed for us by the&nbsp;<code style=\"font-family: var(--code-font);\">typescript</code>&nbsp;package.</p><pre class=\"shiki light-plus\"><div class=\"code-container\"><code><div class=\"line\">tsc hello.ts</div></code></div></pre><p>Tada!</p><p>Wait, “tada”&nbsp;<em>what</em>&nbsp;exactly? We ran&nbsp;<code style=\"font-family: var(--code-font);\">tsc</code>&nbsp;and nothing happened! Well, there were no type errors, so we didn’t get any output in our console since there was nothing to report.</p><p>But check again - we got some&nbsp;<em>file</em>&nbsp;output instead. If we look in our current directory, we’ll see a&nbsp;<code style=\"font-family: var(--code-font);\">hello.js</code>&nbsp;file next to&nbsp;<code style=\"font-family: var(--code-font);\">hello.ts</code>. That’s the output from our&nbsp;<code style=\"font-family: var(--code-font);\">hello.ts</code>&nbsp;file after&nbsp;<code style=\"font-family: var(--code-font);\">tsc</code>&nbsp;<em>compiles</em>&nbsp;or&nbsp;<em>transforms</em>&nbsp;it into a plain JavaScript file. And if we check the contents, we’ll see what TypeScript spits out after it processes a&nbsp;<code style=\"font-family: var(--code-font);\">.ts</code>&nbsp;file:</p><pre class=\"shiki light-plus\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 128, 0);\">// Greets the world.</span></div><div class=\"line\"><span style=\"color: rgb(0, 16, 128);\">console</span>.<span style=\"color: rgb(121, 94, 38);\">log</span>(<span style=\"color: rgb(163, 21, 21);\">\"Hello world!\"</span>);</div></code></div></pre><p>In this case, there was very little for TypeScript to transform, so it looks identical to what we wrote. The compiler tries to emit clean readable code that looks like something a person would write. While that’s not always so easy, TypeScript indents consistently, is mindful of when our code spans across different lines of code, and tries to keep comments around.</p><p>What about if we&nbsp;<em>did</em>&nbsp;introduce a type-checking error? Let’s rewrite&nbsp;<code style=\"font-family: var(--code-font);\">hello.ts</code>:</p><pre class=\"shiki light-plus twoslash lsp\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 128, 0);\">// This is an industrial-grade general-purpose greeter function:</span></div><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">function</span> <span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"function greet(person: any, date: any): void\">greet</data-lsp></span>(<span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) person: any\">person</data-lsp></span>, <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) date: any\">date</data-lsp></span>) {</div><div class=\"line\">  <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"var console: Console\">console</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(method) Console.log(...data: any[]): void\">log</data-lsp></span>(<span style=\"color: rgb(163, 21, 21);\">`Hello </span><span style=\"color: rgb(0, 0, 255);\">${</span><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) person: any\">person</data-lsp></span><span style=\"color: rgb(0, 0, 255);\">}</span><span style=\"color: rgb(163, 21, 21);\">, today is </span><span style=\"color: rgb(0, 0, 255);\">${</span><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) date: any\">date</data-lsp></span><span style=\"color: rgb(0, 0, 255);\">}</span><span style=\"color: rgb(163, 21, 21);\">!`</span>);</div><div class=\"line\">}</div><div class=\"line\">&nbsp;</div><div class=\"line\"><span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"function greet(person: any, date: any): void\">greet</data-lsp></span>(<span style=\"color: rgb(163, 21, 21);\">\"Brendan\"</span>);</div></code></div></pre><p>If we run&nbsp;<code style=\"font-family: var(--code-font);\">tsc hello.ts</code>&nbsp;again, notice that we get an error on the command line!</p><pre class=\"shiki light-plus\"><div class=\"code-container\"><code><div class=\"line\">Expected 2 arguments, but got 1.</div></code></div></pre><p>TypeScript is telling us we forgot to pass an argument to the&nbsp;<code style=\"font-family: var(--code-font);\">greet</code>&nbsp;function, and rightfully so. So far we’ve only written standard JavaScript, and yet type-checking was still able to find problems with our code. Thanks TypeScript!</p><h2 id=\"emitting-with-errors\" style=\"position: relative;\"><a href=\"https://www.typescriptlang.org/docs/handbook/2/basic-types.html#emitting-with-errors\" aria-label=\"emitting with errors permalink\" class=\"anchor before\"><svg aria-hidden=\"true\" focusable=\"false\" height=\"16\" version=\"1.1\" viewBox=\"0 0 16 16\" width=\"16\"><path fill-rule=\"evenodd\" d=\"M4 9h1v1H4c-1.5 0-3-1.69-3-3.5S2.55 3 4 3h4c1.45 0 3 1.69 3 3.5 0 1.41-.91 2.72-2 3.25V8.59c.58-.45 1-1.27 1-2.09C10 5.22 8.98 4 8 4H4c-.98 0-2 1.22-2 2.5S3 9 4 9zm9-3h-1v1h1c1 0 2 1.22 2 2.5S13.98 12 13 12H9c-.98 0-2-1.22-2-2.5 0-.83.42-1.64 1-2.09V6.25c-1.09.53-2 1.84-2 3.25C6 11.31 7.55 13 9 13h4c1.45 0 3-1.69 3-3.5S14.5 6 13 6z\"></path></svg></a>Emitting with Errors</h2><p>One thing you might not have noticed from the last example was that our&nbsp;<code style=\"font-family: var(--code-font);\">hello.js</code>&nbsp;file changed again. If we open that file up then we’ll see that the contents still basically look the same as our input file. That might be a bit surprising given the fact that&nbsp;<code style=\"font-family: var(--code-font);\">tsc</code>&nbsp;reported an error about our code, but this is based on one of TypeScript’s core values: much of the time,&nbsp;<em>you</em>&nbsp;will know better than TypeScript.</p><p>To reiterate from earlier, type-checking code limits the sorts of programs you can run, and so there’s a tradeoff on what sorts of things a type-checker finds acceptable. Most of the time that’s okay, but there are scenarios where those checks get in the way. For example, imagine yourself migrating JavaScript code over to TypeScript and introducing type-checking errors. Eventually you’ll get around to cleaning things up for the type-checker, but that original JavaScript code was already working! Why should converting it over to TypeScript stop you from running it?</p><p>So TypeScript doesn’t get in your way. Of course, over time, you may want to be a bit more defensive against mistakes, and make TypeScript act a bit more strictly. In that case, you can use the&nbsp;<code style=\"font-family: var(--code-font);\">--noEmitOnError</code>&nbsp;compiler option. Try changing your&nbsp;<code style=\"font-family: var(--code-font);\">hello.ts</code>&nbsp;file and running&nbsp;<code style=\"font-family: var(--code-font);\">tsc</code>&nbsp;with that flag:</p><pre class=\"shiki light-plus\"><div class=\"code-container\"><code><div class=\"line\">tsc --noEmitOnError hello.ts</div></code></div></pre><p>You’ll notice that&nbsp;<code style=\"font-family: var(--code-font);\">hello.js</code>&nbsp;never gets updated.</p><h2 id=\"explicit-types\" style=\"position: relative;\"><a href=\"https://www.typescriptlang.org/docs/handbook/2/basic-types.html#explicit-types\" aria-label=\"explicit types permalink\" class=\"anchor before\"><svg aria-hidden=\"true\" focusable=\"false\" height=\"16\" version=\"1.1\" viewBox=\"0 0 16 16\" width=\"16\"><path fill-rule=\"evenodd\" d=\"M4 9h1v1H4c-1.5 0-3-1.69-3-3.5S2.55 3 4 3h4c1.45 0 3 1.69 3 3.5 0 1.41-.91 2.72-2 3.25V8.59c.58-.45 1-1.27 1-2.09C10 5.22 8.98 4 8 4H4c-.98 0-2 1.22-2 2.5S3 9 4 9zm9-3h-1v1h1c1 0 2 1.22 2 2.5S13.98 12 13 12H9c-.98 0-2-1.22-2-2.5 0-.83.42-1.64 1-2.09V6.25c-1.09.53-2 1.84-2 3.25C6 11.31 7.55 13 9 13h4c1.45 0 3-1.69 3-3.5S14.5 6 13 6z\"></path></svg></a>Explicit Types</h2><p>Up until now, we haven’t told TypeScript what&nbsp;<code style=\"font-family: var(--code-font);\">person</code>&nbsp;or&nbsp;<code style=\"font-family: var(--code-font);\">date</code>&nbsp;are. Let’s edit the code to tell TypeScript that&nbsp;<code style=\"font-family: var(--code-font);\">person</code>&nbsp;is a&nbsp;<code style=\"font-family: var(--code-font);\">string</code>, and that&nbsp;<code style=\"font-family: var(--code-font);\">date</code>&nbsp;should be a&nbsp;<code style=\"font-family: var(--code-font);\">Date</code>&nbsp;object. We’ll also use the&nbsp;<code style=\"font-family: var(--code-font);\">toDateString()</code>&nbsp;method on&nbsp;<code style=\"font-family: var(--code-font);\">date</code>.</p><pre class=\"shiki light-plus twoslash lsp\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">function</span> <span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"function greet(person: string, date: Date): void\">greet</data-lsp></span>(<span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) person: string\">person</data-lsp></span>: <span style=\"color: rgb(28, 98, 119);\">string</span>, <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) date: Date\">date</data-lsp></span>: <span style=\"color: rgb(28, 98, 119);\"><data-lsp lsp=\"interface Date\">Date</data-lsp></span>) {</div><div class=\"line\">  <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"var console: Console\">console</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(method) Console.log(...data: any[]): void\">log</data-lsp></span>(<span style=\"color: rgb(163, 21, 21);\">`Hello </span><span style=\"color: rgb(0, 0, 255);\">${</span><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) person: string\">person</data-lsp></span><span style=\"color: rgb(0, 0, 255);\">}</span><span style=\"color: rgb(163, 21, 21);\">, today is </span><span style=\"color: rgb(0, 0, 255);\">${</span><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) date: Date\">date</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(method) Date.toDateString(): string\">toDateString</data-lsp></span>()<span style=\"color: rgb(0, 0, 255);\">}</span><span style=\"color: rgb(163, 21, 21);\">!`</span>);</div><div class=\"line\">}</div></code></div></pre><p>What we did was add&nbsp;<em>type annotations</em>&nbsp;on&nbsp;<code style=\"font-family: var(--code-font);\">person</code>&nbsp;and&nbsp;<code style=\"font-family: var(--code-font);\">date</code>&nbsp;to describe what types of values&nbsp;<code style=\"font-family: var(--code-font);\">greet</code>&nbsp;can be called with. You can read that signature as ”<code style=\"font-family: var(--code-font);\">greet</code>&nbsp;takes a&nbsp;<code style=\"font-family: var(--code-font);\">person</code>&nbsp;of type&nbsp;<code style=\"font-family: var(--code-font);\">string</code>, and a&nbsp;<code style=\"font-family: var(--code-font);\">date</code>&nbsp;of type&nbsp;<code style=\"font-family: var(--code-font);\">Date</code>“.</p><p>With this, TypeScript can tell us about other cases where we might have been called incorrectly. For example…</p><pre class=\"shiki light-plus twoslash lsp\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">function</span> <span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"function greet(person: string, date: Date): void\">greet</data-lsp></span>(<span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) person: string\">person</data-lsp></span>: <span style=\"color: rgb(28, 98, 119);\">string</span>, <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) date: Date\">date</data-lsp></span>: <span style=\"color: rgb(28, 98, 119);\"><data-lsp lsp=\"interface Date\">Date</data-lsp></span>) {</div><div class=\"line\">  <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"var console: Console\">console</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(method) Console.log(...data: any[]): void\">log</data-lsp></span>(<span style=\"color: rgb(163, 21, 21);\">`Hello </span><span style=\"color: rgb(0, 0, 255);\">${</span><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) person: string\">person</data-lsp></span><span style=\"color: rgb(0, 0, 255);\">}</span><span style=\"color: rgb(163, 21, 21);\">, today is </span><span style=\"color: rgb(0, 0, 255);\">${</span><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) date: Date\">date</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(method) Date.toDateString(): string\">toDateString</data-lsp></span>()<span style=\"color: rgb(0, 0, 255);\">}</span><span style=\"color: rgb(163, 21, 21);\">!`</span>);</div><div class=\"line\">}</div><div class=\"line\">&nbsp;</div><div class=\"line\"><span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"function greet(person: string, date: Date): void\">greet</data-lsp></span>(<span style=\"color: rgb(163, 21, 21);\">\"Maddison\"</span>, <span style=\"color: rgb(28, 98, 119);\"><data-lsp lsp=\"var Date: DateConstructor\n() => string\">Date</data-lsp></span>());</div><span class=\"error\">Argument of type 'string' is not assignable to parameter of type 'Date'.</span><span class=\"error-behind\">Argument of type 'string' is not assignable to parameter of type 'Date'.</span></code></div></pre><p>Huh? TypeScript reported an error on our second argument, but why?</p><p>Perhaps surprisingly, calling&nbsp;<code style=\"font-family: var(--code-font);\">Date()</code>&nbsp;in JavaScript returns a&nbsp;<code style=\"font-family: var(--code-font);\">string</code>. On the other hand, constructing a&nbsp;<code style=\"font-family: var(--code-font);\">Date</code>&nbsp;with&nbsp;<code style=\"font-family: var(--code-font);\">new Date()</code>&nbsp;actually gives us what we were expecting.</p><p>Anyway, we can quickly fix up the error:</p><pre class=\"shiki light-plus twoslash lsp\"><div class=\"code-container\"><code><div class=\"line dim\"><span style=\"color: rgb(0, 0, 255);\">function</span> <span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"function greet(person: string, date: Date): void\">greet</data-lsp></span>(<span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) person: string\">person</data-lsp></span>: <span style=\"color: rgb(28, 98, 119);\">string</span>, <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) date: Date\">date</data-lsp></span>: <span style=\"color: rgb(28, 98, 119);\"><data-lsp lsp=\"interface Date\">Date</data-lsp></span>) {</div><div class=\"line dim\">  <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"var console: Console\">console</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(method) Console.log(...data: any[]): void\">log</data-lsp></span>(<span style=\"color: rgb(163, 21, 21);\">`Hello </span><span style=\"color: rgb(0, 0, 255);\">${</span><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) person: string\">person</data-lsp></span><span style=\"color: rgb(0, 0, 255);\">}</span><span style=\"color: rgb(163, 21, 21);\">, today is </span><span style=\"color: rgb(0, 0, 255);\">${</span><span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"(parameter) date: Date\">date</data-lsp></span>.<span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"(method) Date.toDateString(): string\">toDateString</data-lsp></span>()<span style=\"color: rgb(0, 0, 255);\">}</span><span style=\"color: rgb(163, 21, 21);\">!`</span>);</div><div class=\"line dim\">}</div><div class=\"line dim\">&nbsp;</div><div class=\"line highlight\"><span style=\"color: rgb(121, 94, 38);\"><data-lsp lsp=\"function greet(person: string, date: Date): void\">greet</data-lsp></span>(<span style=\"color: rgb(163, 21, 21);\">\"Maddison\"</span>, <span style=\"color: rgb(0, 0, 255);\">new</span> <span style=\"color: rgb(28, 98, 119);\"><data-lsp lsp=\"var Date: DateConstructor\nnew () => Date (+4 overloads)\">Date</data-lsp></span>());</div></code></div></pre><p>Keep in mind, we don’t always have to write explicit type annotations. In many cases, TypeScript can even just&nbsp;<em>infer</em>&nbsp;(or “figure out”) the types for us even if we omit them.</p><pre class=\"shiki light-plus twoslash lsp\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">let</span> <span style=\"color: rgb(0, 16, 128);\"><data-lsp lsp=\"let msg: string\" style=\"border-bottom: 2px solid lightgrey;\">msg</data-lsp></span> = <span style=\"color: rgb(163, 21, 21);\">\"hello there!\"</span>;</div><div class=\"meta-line\"><span class=\"popover-prefix\">    </span><span class=\"popover\"><div class=\"arrow\"></div>let msg: string</span></div></code></div></pre><p>Even though we didn’t tell TypeScript that&nbsp;<code style=\"font-family: var(--code-font);\">msg</code>&nbsp;had the type&nbsp;<code style=\"font-family: var(--code-font);\">string</code>&nbsp;it was able to figure that out. That’s a feature, and it’s best not to add annotations when the type system would end up inferring the same type anyway.</p><blockquote style=\"border-left: 2px solid var(--border-color);\"><p>Note: the message bubble inside the code sample above. That is what your editor would show if you had hovered over the word.</p></blockquote><h2 id=\"erased-types\" style=\"position: relative;\"><a href=\"https://www.typescriptlang.org/docs/handbook/2/basic-types.html#erased-types\" aria-label=\"erased types permalink\" class=\"anchor before\"><svg aria-hidden=\"true\" focusable=\"false\" height=\"16\" version=\"1.1\" viewBox=\"0 0 16 16\" width=\"16\"><path fill-rule=\"evenodd\" d=\"M4 9h1v1H4c-1.5 0-3-1.69-3-3.5S2.55 3 4 3h4c1.45 0 3 1.69 3 3.5 0 1.41-.91 2.72-2 3.25V8.59c.58-.45 1-1.27 1-2.09C10 5.22 8.98 4 8 4H4c-.98 0-2 1.22-2 2.5S3 9 4 9zm9-3h-1v1h1c1 0 2 1.22 2 2.5S13.98 12 13 12H9c-.98 0-2-1.22-2-2.5 0-.83.42-1.64 1-2.09V6.25c-1.09.53-2 1.84-2 3.25C6 11.31 7.55 13 9 13h4c1.45 0 3-1.69 3-3.5S14.5 6 13 6z\"></path></svg></a>Erased Types</h2><p>Let’s take a look at what happens when we compile the above function&nbsp;<code style=\"font-family: var(--code-font);\">greet</code>&nbsp;with&nbsp;<code style=\"font-family: var(--code-font);\">tsc</code>&nbsp;to output JavaScript:</p><pre class=\"shiki light-plus twoslash lsp\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(163, 21, 21);\">\"use strict\"</span>;</div><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">function</span> <span style=\"color: rgb(121, 94, 38);\">greet</span>(<span style=\"color: rgb(0, 16, 128);\">person</span>, <span style=\"color: rgb(0, 16, 128);\">date</span>) {</div><div class=\"line\">    <span style=\"color: rgb(0, 16, 128);\">console</span>.<span style=\"color: rgb(121, 94, 38);\">log</span>(<span style=\"color: rgb(163, 21, 21);\">\"Hello \"</span> + <span style=\"color: rgb(0, 16, 128);\">person</span> + <span style=\"color: rgb(163, 21, 21);\">\", today is \"</span> + <span style=\"color: rgb(0, 16, 128);\">date</span>.<span style=\"color: rgb(121, 94, 38);\">toDateString</span>() + <span style=\"color: rgb(163, 21, 21);\">\"!\"</span>);</div><div class=\"line\">}</div><div class=\"line\"><span style=\"color: rgb(121, 94, 38);\">greet</span>(<span style=\"color: rgb(163, 21, 21);\">\"Maddison\"</span>, <span style=\"color: rgb(0, 0, 255);\">new</span> <span style=\"color: rgb(28, 98, 119);\">Date</span>());</div><div class=\"line\">&nbsp;</div></code></div></pre><p>Notice two things here:</p><ol><li>Our&nbsp;<code style=\"font-family: var(--code-font);\">person</code>&nbsp;and&nbsp;<code style=\"font-family: var(--code-font);\">date</code>&nbsp;parameters no longer have type annotations.</li><li>Our “template string” - that string that used backticks (the&nbsp;<code style=\"font-family: var(--code-font);\">`</code>&nbsp;character) - was converted to plain strings with concatenations (<code style=\"font-family: var(--code-font);\">+</code>).</li></ol><p>More on that second point later, but let’s now focus on that first point. Type annotations aren’t part of JavaScript (or ECMAScript to be pedantic), so there really aren’t any browsers or other runtimes that can just run TypeScript unmodified. That’s why TypeScript needs a compiler in the first place - it needs some way to strip out or transform any TypeScript-specific code so that you can run it. Most TypeScript-specific code gets erased away, and likewise, here our type annotations were completely erased.</p><blockquote style=\"border-left: 2px solid var(--border-color);\"><p><strong>Remember</strong>: Type annotations never change the runtime behavior of your program.</p></blockquote><h2 id=\"downleveling\" style=\"position: relative;\"><a href=\"https://www.typescriptlang.org/docs/handbook/2/basic-types.html#downleveling\" aria-label=\"downleveling permalink\" class=\"anchor before\"><svg aria-hidden=\"true\" focusable=\"false\" height=\"16\" version=\"1.1\" viewBox=\"0 0 16 16\" width=\"16\"><path fill-rule=\"evenodd\" d=\"M4 9h1v1H4c-1.5 0-3-1.69-3-3.5S2.55 3 4 3h4c1.45 0 3 1.69 3 3.5 0 1.41-.91 2.72-2 3.25V8.59c.58-.45 1-1.27 1-2.09C10 5.22 8.98 4 8 4H4c-.98 0-2 1.22-2 2.5S3 9 4 9zm9-3h-1v1h1c1 0 2 1.22 2 2.5S13.98 12 13 12H9c-.98 0-2-1.22-2-2.5 0-.83.42-1.64 1-2.09V6.25c-1.09.53-2 1.84-2 3.25C6 11.31 7.55 13 9 13h4c1.45 0 3-1.69 3-3.5S14.5 6 13 6z\"></path></svg></a>Downleveling</h2><p>One other difference from the above was that our template string was rewritten from</p><pre class=\"shiki light-plus\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(163, 21, 21);\">`Hello </span><span style=\"color: rgb(0, 0, 255);\">${</span><span style=\"color: rgb(0, 16, 128);\">person</span><span style=\"color: rgb(0, 0, 255);\">}</span><span style=\"color: rgb(163, 21, 21);\">, today is </span><span style=\"color: rgb(0, 0, 255);\">${</span><span style=\"color: rgb(0, 16, 128);\">date</span>.<span style=\"color: rgb(121, 94, 38);\">toDateString</span>()<span style=\"color: rgb(0, 0, 255);\">}</span><span style=\"color: rgb(163, 21, 21);\">!`</span>;</div></code></div></pre><p>to</p><pre class=\"shiki light-plus\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(163, 21, 21);\">\"Hello \"</span> + <span style=\"color: rgb(0, 16, 128);\">person</span> + <span style=\"color: rgb(163, 21, 21);\">\", today is \"</span> + <span style=\"color: rgb(0, 16, 128);\">date</span>.<span style=\"color: rgb(121, 94, 38);\">toDateString</span>() + <span style=\"color: rgb(163, 21, 21);\">\"!\"</span>;</div></code></div></pre><p>Why did this happen?</p><p>Template strings are a feature from a version of ECMAScript called ECMAScript 2015 (a.k.a. ECMAScript 6, ES2015, ES6, etc. -&nbsp;<em>don’t ask</em>). TypeScript has the ability to rewrite code from newer versions of ECMAScript to older ones such as ECMAScript 3 or ECMAScript 5 (a.k.a. ES3 and ES5). This process of moving from a newer or “higher” version of ECMAScript down to an older or “lower” one is sometimes called&nbsp;<em>downleveling</em>.</p><p>By default TypeScript targets ES3, an extremely old version of ECMAScript. We could have chosen something a little bit more recent by using the&nbsp;<code style=\"font-family: var(--code-font);\">--target</code>&nbsp;flag. Running with&nbsp;<code style=\"font-family: var(--code-font);\">--target es2015</code>&nbsp;changes TypeScript to target ECMAScript 2015, meaning code should be able to run wherever ECMAScript 2015 is supported. So running&nbsp;<code style=\"font-family: var(--code-font);\">tsc --target es2015 hello.ts</code>&nbsp;gives us the following output:</p><pre class=\"shiki light-plus\"><div class=\"code-container\"><code><div class=\"line\"><span style=\"color: rgb(0, 0, 255);\">function</span> <span style=\"color: rgb(121, 94, 38);\">greet</span>(<span style=\"color: rgb(0, 16, 128);\">person</span>, <span style=\"color: rgb(0, 16, 128);\">date</span>) {</div><div class=\"line\">  <span style=\"color: rgb(0, 16, 128);\">console</span>.<span style=\"color: rgb(121, 94, 38);\">log</span>(<span style=\"color: rgb(163, 21, 21);\">`Hello </span><span style=\"color: rgb(0, 0, 255);\">${</span><span style=\"color: rgb(0, 16, 128);\">person</span><span style=\"color: rgb(0, 0, 255);\">}</span><span style=\"color: rgb(163, 21, 21);\">, today is </span><span style=\"color: rgb(0, 0, 255);\">${</span><span style=\"color: rgb(0, 16, 128);\">date</span>.<span style=\"color: rgb(121, 94, 38);\">toDateString</span>()<span style=\"color: rgb(0, 0, 255);\">}</span><span style=\"color: rgb(163, 21, 21);\">!`</span>);</div><div class=\"line\">}</div><div class=\"line\"><span style=\"color: rgb(121, 94, 38);\">greet</span>(<span style=\"color: rgb(163, 21, 21);\">\"Maddison\"</span>, <span style=\"color: rgb(0, 0, 255);\">new</span> <span style=\"color: rgb(28, 98, 119);\">Date</span>());</div></code></div></pre><blockquote style=\"border-left: 2px solid var(--border-color);\"><p>While the default target is ES3, the great majority of current browsers support ES2015. Most developers can therefore safely specify ES2015 or above as a target, unless compatibility with certain ancient browsers is important.</p></blockquote><h2 id=\"strictness\" style=\"position: relative;\"><a href=\"https://www.typescriptlang.org/docs/handbook/2/basic-types.html#strictness\" aria-label=\"strictness permalink\" class=\"anchor before\"><svg aria-hidden=\"true\" focusable=\"false\" height=\"16\" version=\"1.1\" viewBox=\"0 0 16 16\" width=\"16\"><path fill-rule=\"evenodd\" d=\"M4 9h1v1H4c-1.5 0-3-1.69-3-3.5S2.55 3 4 3h4c1.45 0 3 1.69 3 3.5 0 1.41-.91 2.72-2 3.25V8.59c.58-.45 1-1.27 1-2.09C10 5.22 8.98 4 8 4H4c-.98 0-2 1.22-2 2.5S3 9 4 9zm9-3h-1v1h1c1 0 2 1.22 2 2.5S13.98 12 13 12H9c-.98 0-2-1.22-2-2.5 0-.83.42-1.64 1-2.09V6.25c-1.09.53-2 1.84-2 3.25C6 11.31 7.55 13 9 13h4c1.45 0 3-1.69 3-3.5S14.5 6 13 6z\"></path></svg></a>Strictness</h2><p>Different users come to TypeScript looking for different things in a type-checker. Some people are looking for a more loose opt-in experience which can help validate only some parts of their program, and still have decent tooling. This is the default experience with TypeScript, where types are optional, inference takes the most lenient types, and there’s no checking for potentially&nbsp;<code style=\"font-family: var(--code-font);\">null</code>/<code style=\"font-family: var(--code-font);\">undefined</code>&nbsp;values. Much like how&nbsp;<code style=\"font-family: var(--code-font);\">tsc</code>&nbsp;emits in the face of errors, these defaults are put in place to stay out of your way. If you’re migrating existing JavaScript, that might be a desirable first step.</p><p>In contrast, a lot of users prefer to have TypeScript validate as much as it can straight away, and that’s why the language provides strictness settings as well. These strictness settings turn static type-checking from a switch (either your code is checked or not) into something closer to a dial. The further you turn this dial up, the more TypeScript will check for you. This can require a little extra work, but generally speaking it pays for itself in the long run, and enables more thorough checks and more accurate tooling. When possible, a new codebase should always turn these strictness checks on.</p><p>TypeScript has several type-checking strictness flags that can be turned on or off, and all of our examples will be written with all of them enabled unless otherwise stated. The&nbsp;<code style=\"font-family: var(--code-font);\">--strict</code>&nbsp;flag in the CLI, or&nbsp;<code style=\"font-family: var(--code-font);\">\"strict\": true</code>&nbsp;in a&nbsp;<a href=\"https://www.typescriptlang.org/docs/handbook/tsconfig-json.html\"><code>tsconfig.json</code></a>&nbsp;toggles them all on simultaneously, but we can opt out of them individually. The two biggest ones you should know about are&nbsp;<code style=\"font-family: var(--code-font);\">noImplicitAny</code>&nbsp;and&nbsp;<code style=\"font-family: var(--code-font);\">strictNullChecks</code>.</p><h2 id=\"noimplicitany\" style=\"position: relative;\"><a href=\"https://www.typescriptlang.org/docs/handbook/2/basic-types.html#noimplicitany\" aria-label=\"noimplicitany permalink\" class=\"anchor before\"><svg aria-hidden=\"true\" focusable=\"false\" height=\"16\" version=\"1.1\" viewBox=\"0 0 16 16\" width=\"16\"><path fill-rule=\"evenodd\" d=\"M4 9h1v1H4c-1.5 0-3-1.69-3-3.5S2.55 3 4 3h4c1.45 0 3 1.69 3 3.5 0 1.41-.91 2.72-2 3.25V8.59c.58-.45 1-1.27 1-2.09C10 5.22 8.98 4 8 4H4c-.98 0-2 1.22-2 2.5S3 9 4 9zm9-3h-1v1h1c1 0 2 1.22 2 2.5S13.98 12 13 12H9c-.98 0-2-1.22-2-2.5 0-.83.42-1.64 1-2.09V6.25c-1.09.53-2 1.84-2 3.25C6 11.31 7.55 13 9 13h4c1.45 0 3-1.69 3-3.5S14.5 6 13 6z\"></path></svg></a><code>noImplicitAny</code></h2><p>Recall that in some places, TypeScript doesn’t try to infer any types for us and instead falls back to the most lenient type:&nbsp;<code style=\"font-family: var(--code-font);\">any</code>. This isn’t the worst thing that can happen - after all, falling back to&nbsp;<code style=\"font-family: var(--code-font);\">any</code>&nbsp;is just the plain JavaScript experience anyway.</p><p>However, using&nbsp;<code style=\"font-family: var(--code-font);\">any</code>&nbsp;often defeats the purpose of using TypeScript in the first place. The more typed your program is, the more validation and tooling you’ll get, meaning you’ll run into fewer bugs as you code. Turning on the&nbsp;<code style=\"font-family: var(--code-font);\">noImplicitAny</code>&nbsp;flag will issue an error on any variables whose type is implicitly inferred as&nbsp;<code style=\"font-family: var(--code-font);\">any</code>.</p><h2 id=\"strictnullchecks\" style=\"position: relative;\"><a href=\"https://www.typescriptlang.org/docs/handbook/2/basic-types.html#strictnullchecks\" aria-label=\"strictnullchecks permalink\" class=\"anchor before\"><svg aria-hidden=\"true\" focusable=\"false\" height=\"16\" version=\"1.1\" viewBox=\"0 0 16 16\" width=\"16\"><path fill-rule=\"evenodd\" d=\"M4 9h1v1H4c-1.5 0-3-1.69-3-3.5S2.55 3 4 3h4c1.45 0 3 1.69 3 3.5 0 1.41-.91 2.72-2 3.25V8.59c.58-.45 1-1.27 1-2.09C10 5.22 8.98 4 8 4H4c-.98 0-2 1.22-2 2.5S3 9 4 9zm9-3h-1v1h1c1 0 2 1.22 2 2.5S13.98 12 13 12H9c-.98 0-2-1.22-2-2.5 0-.83.42-1.64 1-2.09V6.25c-1.09.53-2 1.84-2 3.25C6 11.31 7.55 13 9 13h4c1.45 0 3-1.69 3-3.5S14.5 6 13 6z\"></path></svg></a><code>strictNullChecks</code></h2><p>By default, values like&nbsp;<code style=\"font-family: var(--code-font);\">null</code>&nbsp;and&nbsp;<code style=\"font-family: var(--code-font);\">undefined</code>&nbsp;are assignable to any other type. This can make writing some code easier, but forgetting to handle&nbsp;<code style=\"font-family: var(--code-font);\">null</code>&nbsp;and&nbsp;<code style=\"font-family: var(--code-font);\">undefined</code>&nbsp;is the cause of countless bugs in the world - some consider it a&nbsp;<a href=\"https://www.youtube.com/watch?v=ybrQvs4x0Ps\">billion dollar mistake</a>! The&nbsp;<code style=\"font-family: var(--code-font);\">strictNullChecks</code>&nbsp;flag makes handling&nbsp;<code style=\"font-family: var(--code-font);\">null</code>&nbsp;and&nbsp;<code style=\"font-family: var(--code-font);\">undefined</code>&nbsp;more explicit, and&nbsp;<em>spares</em>&nbsp;us from worrying about whether we&nbsp;<em>forgot</em>&nbsp;to handle&nbsp;<code style=\"font-family: var(--code-font);\">null</code>&nbsp;and&nbsp;<code style=\"font-family: var(--code-font);\">undefined</code>.</p></div></div></article>"

export default  html
