/*
 * Copyright 2018 LinkedIn Corp.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package azkaban.viewer.jobsummary;

import azkaban.executor.ExecutableFlow;
import azkaban.executor.ExecutableNode;
import azkaban.executor.ExecutorManagerAdapter;
import azkaban.executor.ExecutorManagerException;
import azkaban.project.Project;
import azkaban.project.ProjectManager;
import azkaban.server.session.Session;
import azkaban.user.Permission.Type;
import azkaban.user.User;
import azkaban.utils.ExternalLink;
import azkaban.utils.ExternalLinkUtils;
import azkaban.utils.ExternalLinkUtils.ExternalLinkScope;
import azkaban.utils.Props;
import azkaban.webapp.AzkabanWebServer;
import azkaban.webapp.plugin.PluginRegistry;
import azkaban.webapp.plugin.ViewerPlugin;
import azkaban.webapp.servlet.LoginAbstractAzkabanServlet;
import azkaban.webapp.servlet.Page;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.log4j.Logger;
import azkaban.Constants;


public class JobSummaryServlet extends LoginAbstractAzkabanServlet {

  private static final Logger logger = Logger.getLogger(JobSummaryServlet.class);
  private static final long serialVersionUID = 2057998465288828864L;

  private final Props props;
  private final File webResourcesPath;
  private final String viewerName;
  private final String viewerPath;

  private ExecutorManagerAdapter executorManagerAdapter;
  private ProjectManager projectManager;

  private List<ExternalLink> jobLevelExternalLinks;
  private int externalLinksTimeoutMs;

  public JobSummaryServlet(final Props props) {
    super(new ArrayList<>());
    this.props = props;
    this.viewerName = props.getString("viewer.name");
    this.viewerPath = props.getString("viewer.path");

    this.webResourcesPath =
        new File(new File(props.getSource()).getParentFile().getParentFile(),
            "web");
    this.webResourcesPath.mkdirs();
    setResourceDirectory(this.webResourcesPath);
  }

  @Override
  public void init(final ServletConfig config) throws ServletException {
    super.init(config);
    final AzkabanWebServer server = getApplication();
    this.executorManagerAdapter = server.getExecutorManager();
    this.projectManager = server.getProjectManager();

    final Props azkProps = server.getServerProps();
    this.jobLevelExternalLinks = ExternalLinkUtils.parseExternalLinks(azkProps, ExternalLinkScope.JOB);
    this.externalLinksTimeoutMs =
        this.props.getInt(Constants.ConfigurationKeys.AZKABAN_SERVER_EXTERNAL_ANALYZER_TIMEOUT_MS,
            Constants.DEFAULT_AZKABAN_SERVER_EXTERNAL_ANALYZER_TIMEOUT_MS);
  }

  private void handleViewer(final HttpServletRequest req, final HttpServletResponse resp,
      final Session session) throws ServletException {

    final Page page =
        newPage(req, resp, session,
            "azkaban/viewer/jobsummary/velocity/jobsummary.vm");
    page.add("viewerPath", this.viewerPath);
    page.add("viewerName", this.viewerName);

    final User user = session.getUser();
    final int execId = getIntParam(req, "execid");
    final String jobId = getParam(req, "jobid");
    final int attempt = getIntParam(req, "attempt", 0);

    page.add("execid", execId);
    page.add("jobid", jobId);
    page.add("attempt", attempt);

    ExecutableFlow flow = null;
    ExecutableNode node = null;
    try {
      flow = this.executorManagerAdapter.getExecutableFlow(execId);
      if (flow == null) {
        page.add("errorMsg", "Error loading executing flow " + execId
            + ": not found.");
        page.render();
        return;
      }

      node = flow.getExecutableNodePath(jobId);
      if (node == null) {
        page.add("errorMsg",
            "Job " + jobId + " doesn't exist in " + flow.getExecutionId());
        return;
      }

      final List<ViewerPlugin> jobViewerPlugins =
          PluginRegistry.getRegistry().getViewerPluginsForJobType(
              node.getType());
      page.add("jobViewerPlugins", jobViewerPlugins);
    } catch (final ExecutorManagerException e) {
      page.add("errorMsg", "Error loading executing flow: " + e.getMessage());
      page.render();
      return;
    }

    final int projectId = flow.getProjectId();
    final Project project = filterProjectByPermission(this.projectManager.getProject(projectId),
        user, Type.READ);
    if (project == null) {
      page.render();
      return;
    }

    page.add("projectName", project.getName());
    page.add("flowid", flow.getId());
    page.add("flowlist", flow.getId().split(Constants.PATH_DELIMITER, 0));
    page.add("pathDelimiter", Constants.PATH_DELIMITER);
    page.add("parentflowid", node.getParentFlow().getFlowId());
    page.add("jobname", node.getId());
    page.add("attemptStatus", attempt == node.getAttempt() ?
        node.getStatus() : node.getPastAttemptList().get(attempt).getStatus());

    final List<ExternalLink> externalAnalyzers =
        ExternalLinkUtils.buildExternalLinksForRequest(this.jobLevelExternalLinks,
            this.externalLinksTimeoutMs, req, execId, jobId);
    page.add("externalAnalyzers", externalAnalyzers);

    page.render();
  }

  private void handleDefault(final HttpServletRequest request,
      final HttpServletResponse response, final Session session) {
    final Page page =
        newPage(request, response, session,
            "azkaban/viewer/jobsummary/velocity/jobsummary.vm");
    page.add("viewerPath", this.viewerPath);
    page.add("viewerName", this.viewerName);
    page.add("errorMsg", "No job execution specified.");
    page.render();
  }

  @Override
  protected void handleGet(final HttpServletRequest request,
      final HttpServletResponse response, final Session session) throws ServletException {
    if (hasParam(request, "execid") && hasParam(request, "jobid")) {
      handleViewer(request, response, session);
    } else {
      handleDefault(request, response, session);
    }
  }

  @Override
  protected void handlePost(final HttpServletRequest request,
      final HttpServletResponse response, final Session session) {
  }
}
