export const variantSeperator = '_';

export const languages: Readonly<Record<string, string>> = {
  af: 'Afrikaans',
  ar: 'العربية',
  arg: 'aragonés',
  as: 'অসমীয়া',
  ast: 'asturianu',
  az: 'azərbaycan',
  ba: 'башҡортса',
  be: 'беларуская',
  bg: 'български',
  bn: 'বাংলা',
  bos: 'bosanski',
  br: 'brezhoneg',
  ca: 'català',
  cs: 'čeština',
  cv: 'чӑвашла',
  cy: 'Cymraeg',
  da: 'dansk',
  de: 'Deutsch',
  el: 'Ελληνικά',
  en: 'English',
  eo: 'esperanto',
  es: 'español',
  et: 'eesti',
  eu: 'euskara',
  fa: 'فارسی',
  fi: 'suomi',
  fo: 'føroyskt',
  fr: 'français',
  ga: 'Gaeilge',
  gd: 'Gàidhlig',
  gl: 'galego',
  gv: 'Gaelg',
  hbs: 'srpskohrvatski',
  he: 'עברית',
  hi: 'हिंदी',
  hrv: 'hrvatski',
  hu: 'magyar',
  hy: 'հայերեն',
  ia: 'interlingua',
  id: 'Bahasa Indonesia',
  is: 'íslenska',
  it: 'italiano',
  kk: 'қазақша',
  ko: '한국어',
  kum: 'къумукъча',
  ky: 'кыргызча',
  lg: 'Luganda',
  lo: 'ລາວ',
  lt: 'lietuvių',
  lv: 'latviešu',
  mfe: 'kreol morisien',
  mk: 'македонски',
  ml: 'മലയാളം',
  mr: 'मराठी',
  mrj: 'Мары йӹлмӹ',
  msa: 'bahasa malay',
  mt: 'Malti',
  myv: 'Эрзянь кель',
  nb: 'norsk bokmål',
  ne: 'नेपाली',
  nl: 'Nederlands',
  nn: 'nynorsk',
  nog: 'ногъайша',
  oc: 'occitan',
  os: 'ирон',
  pa: 'ਪੰਜਾਬੀ',
  pl: 'polski',
  pt: 'português',
  rm: 'rumantsch',
  rn: 'Ikirundi',
  ro: 'română',
  ru: 'русский',
  sah: 'сахалыы',
  se: 'davvisámegiella',
  si: 'සිංහල',
  sk: 'slovenčina',
  sl: 'slovenščina',
  sq: 'Shqip',
  srp: 'српски',
  sv: 'svenska',
  sw: 'Kiswahili',
  te: 'తెలుగు',
  tg: 'Тоҷикӣ',
  th: 'ไทย',
  tr: 'Türkçe',
  tt: 'татарча',
  tyv: 'тыва дылда',
  uig: 'ئۇيغۇرچە',
  uk: 'українська',
  ur: 'اردو',
  uz: 'oʻzbekcha',
  vi: 'Tiếng Việt',
  xh: 'isiXhosa',
  zh: '中文',
  zu: 'isiZulu',
};

const iso639Codes: Readonly<Record<string, string>> = {
  aar: 'aa',
  abk: 'ab',
  afr: 'af',
  aka: 'ak',
  amh: 'am',
  ara: 'ar',
  arg: 'an',
  asm: 'as',
  ava: 'av',
  ave: 'ae',
  aym: 'ay',
  azb: 'az',
  aze: 'az',
  bak: 'ba',
  bam: 'bm',
  bel: 'be',
  ben: 'bn',
  bih: 'bh',
  bis: 'bi',
  bod: 'bo',
  bos: 'bs',
  bre: 'br',
  bul: 'bg',
  cat: 'ca',
  ces: 'cs',
  cha: 'ch',
  che: 'ce',
  chu: 'cu',
  chv: 'cv',
  cor: 'kw',
  cos: 'co',
  cre: 'cr',
  cym: 'cy',
  dan: 'da',
  deu: 'de',
  div: 'dv',
  dzo: 'dz',
  ell: 'el',
  eng: 'en',
  epo: 'eo',
  est: 'et',
  eus: 'eu',
  ewe: 'ee',
  fao: 'fo',
  fas: 'fa',
  fij: 'fj',
  fin: 'fi',
  fra: 'fr',
  fry: 'fy',
  ful: 'ff',
  gla: 'gd',
  gle: 'ga',
  glg: 'gl',
  glv: 'gv',
  grn: 'gn',
  guj: 'gu',
  hat: 'ht',
  hau: 'ha',
  hbs: 'sh',
  heb: 'he',
  her: 'hz',
  hin: 'hi',
  hmo: 'ho',
  hrv: 'hr',
  hun: 'hu',
  hye: 'hy',
  ibo: 'ig',
  ido: 'io',
  iii: 'ii',
  iku: 'iu',
  ile: 'ie',
  ina: 'ia',
  ind: 'id',
  ipk: 'ik',
  isl: 'is',
  ita: 'it',
  jav: 'jv',
  jpn: 'ja',
  kal: 'kl',
  kan: 'kn',
  kas: 'ks',
  kat: 'ka',
  kau: 'kr',
  kaz: 'kk',
  khm: 'km',
  kik: 'ki',
  kin: 'rw',
  kir: 'ky',
  kom: 'kv',
  kon: 'kg',
  kor: 'ko',
  kua: 'kj',
  kur: 'ku',
  lao: 'lo',
  lat: 'la',
  lav: 'lv',
  lim: 'li',
  lin: 'ln',
  lit: 'lt',
  ltz: 'lb',
  lub: 'lu',
  lug: 'lg',
  mah: 'mh',
  mal: 'ml',
  mar: 'mr',
  mkd: 'mk',
  mlg: 'mg',
  mlt: 'mt',
  mon: 'mn',
  mri: 'mi',
  msa: 'ms',
  mya: 'my',
  nau: 'na',
  nav: 'nv',
  nbl: 'nr',
  nde: 'nd',
  ndo: 'ng',
  nep: 'ne',
  nld: 'nl',
  nno: 'nn',
  nob: 'nb',
  nor: 'no',
  nya: 'ny',
  oci: 'oc',
  oji: 'oj',
  ori: 'or',
  orm: 'om',
  oss: 'os',
  pan: 'pa',
  pes: 'fa',
  pli: 'pi',
  pol: 'pl',
  por: 'pt',
  pus: 'ps',
  que: 'qu',
  roh: 'rm',
  ron: 'ro',
  run: 'rn',
  rus: 'ru',
  sag: 'sg',
  san: 'sa',
  sin: 'si',
  slk: 'sk',
  slv: 'sl',
  sme: 'se',
  smo: 'sm',
  sna: 'sn',
  snd: 'sd',
  som: 'so',
  sot: 'st',
  spa: 'es',
  sqi: 'sq',
  srd: 'sc',
  srp: 'sr',
  ssw: 'ss',
  sun: 'su',
  swa: 'sw',
  swe: 'sv',
  tah: 'ty',
  tam: 'ta',
  tat: 'tt',
  tel: 'te',
  tgk: 'tg',
  tgl: 'tl',
  tha: 'th',
  tir: 'ti',
  ton: 'to',
  tsn: 'tn',
  tso: 'ts',
  tuk: 'tk',
  tur: 'tr',
  twi: 'tw',
  uig: 'ug',
  ukr: 'uk',
  urd: 'ur',
  uzb: 'uz',
  ven: 've',
  vie: 'vi',
  vol: 'vo',
  wln: 'wa',
  wol: 'wo',
  xho: 'xh',
  yid: 'yi',
  yor: 'yo',
  zha: 'za',
  zho: 'zh',
  zul: 'zu',
};

const rtlLanguages = new Set(['heb', 'ara', 'pes', 'urd', 'uig']);

const iso639CodesInverse: Record<string, string> = {};
Object.entries(iso639Codes).forEach(([alpha3, alpha2]) => (iso639CodesInverse[alpha2] = alpha3));

export function toAlpha2Code(code: string): string;
export function toAlpha2Code(code: string | null): string | null;
export function toAlpha2Code(code: string | null): string | null {
  if (!code) {
    return code;
  }

  const splitCode = code.split(variantSeperator, 2);
  const parentCode = splitCode[0];
  const variant = splitCode[1] ? `${variantSeperator}${splitCode[1]}` : '';

  if (parentCode in iso639CodesInverse) {
    return `${parentCode}${variant}`;
  }

  if (parentCode in iso639Codes) {
    return `${iso639Codes[parentCode]}${variant}`;
  }

  return null;
}

export function toAlpha3Code(code: string): string;
export function toAlpha3Code(code: string | null): string | null;
export function toAlpha3Code(code: string | null): string | null {
  if (!code) {
    return code;
  }

  const splitCode = code.split(variantSeperator, 2);
  const parentCode = splitCode[0];
  const variant = splitCode[1] ? `${variantSeperator}${splitCode[1]}` : '';

  if (parentCode in iso639Codes) {
    return `${parentCode}${variant}`;
  }

  if (parentCode in iso639CodesInverse) {
    return `${iso639CodesInverse[parentCode]}${variant}`;
  }

  return code;
}

export const langDirection = (code: string): string => {
  return rtlLanguages.has(code) ? 'rtl' : 'ltr';
};

export const parentLang = (code: string): string => code.split(variantSeperator, 2)[0];

export const isVariant = (code: string): boolean => code.includes(variantSeperator);
