{******************************************************************************}
{                                                                              }
{           ZLibFramework.Interfaces                                           }
{                                                                              }
{           Copyright (C) Antnio Jos Medeiros Schneider Jnior               }
{                                                                              }
{           https://github.com/antoniojmsjr/ZLibFramework                      }
{                                                                              }
{                                                                              }
{******************************************************************************}
{                                                                              }
{  Licensed under the Apache License, Version 2.0 (the "License");             }
{  you may not use this file except in compliance with the License.            }
{  You may obtain a copy of the License at                                     }
{                                                                              }
{      http://www.apache.org/licenses/LICENSE-2.0                              }
{                                                                              }
{  Unless required by applicable law or agreed to in writing, software         }
{  distributed under the License is distributed on an "AS IS" BASIS,           }
{  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.    }
{  See the License for the specific language governing permissions and         }
{  limitations under the License.                                              }
{                                                                              }
{******************************************************************************}
unit ZLibFramework.Interfaces;

interface

uses
  System.SysUtils, System.Classes, ZLibFramework.Types;

type

  IZLibAlgorithmCompression = interface;
  IZLibAlgorithmDecompression = interface;

  IZLibOperation = interface
    ['{050E6531-1417-4681-8007-E735CBE99EA3}']
    /// <summary>
    /// Processo de Compresso.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: TZLib.Base64.Compress.Deflate.Text('Texto12345', lResult);
    /// end;
    /// </code>
    /// </summary>
    /// <returns>Retorna uma interface do tipo IZLibAlgorithmCompression que implementa o conceito de interface fluente para guiar no uso da biblioteca.</returns>
    function Compress: IZLibAlgorithmCompression;
    /// <summary>
    /// Processo de Descompresso.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: TZLib.Base64.Decompress.Deflate.Text('eNozNDI2MQUAAvgBAA==', lResult);
    /// end;
    /// </code>
    /// </summary>
    /// <returns>Retorna uma interface do tipo IZLibAlgorithmDecompression que implementa o conceito de interface fluente para guiar no uso da biblioteca.</returns>
    function Decompress: IZLibAlgorithmDecompression;
  end;

  {$REGION 'COMPRESSION'}

  IZLibMethodsCompression = interface;

  IZLibAlgorithmCompression = interface
    ['{64C6AA18-01B6-44D8-8413-04DF96F9DDA8}']
    /// <summary>
    /// Processo de Compresso usando o algoritimo Deflate.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: TZLib.Data.Compress.Deflate.Text('Texto12345', lResult);
    /// end;
    /// </code>
    /// </summary>
    /// <returns>Retorna uma interface do tipo IZLibMethodsCompression que implementa o conceito de interface fluente para guiar no uso da biblioteca.</returns>
    function Deflate: IZLibMethodsCompression;
    /// <summary>
    /// Processo de Compresso usando o algoritimo GZip.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: TZLib.Data.Compress.GZip.Text('Texto12345', lResult);
    /// end;
    /// </code>
    /// </summary>
    /// <returns>Retorna uma interface do tipo IZLibMethodsCompression que implementa o conceito de interface fluente para guiar no uso da biblioteca.</returns>
    function GZip: IZLibMethodsCompression;
    function &End: IZLibOperation;
  end;

  IZLibMethodsCompression = interface
    ['{47AA25AE-CBD7-46CA-BF31-6CACF2A29F73}']
    /// <summary>
    /// Identifica o nvel de compresso utilizada para compactao.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: lResult := TZLib.Data.Compress.Deflate.Level(TZLibCompressionLevelType.Max).Text('Texto12345');
    /// end;
    /// </code>
    /// </summary>
    /// <remarks>
    /// <para>* Verificar na unit ZLibFramework.Types o tipo TZLibCompressionLevelType com as opes possveis para ser utilizado.</para>
    /// <para>* O CompressionLevel utilizado quando no informado,  TZLibCompressionLevelType.Max.</para>
    /// </remarks>
    /// <param name="CompressionLevel">
    /// Identificao do nvel de compresso.
    /// </param>
    /// <returns>Retorna uma interface do tipo IZLibMethodsCompression que implementa o conceito de fluent interface para guiar no uso da biblioteca.</returns>
    function Level(const CompressionLevel: TZLibCompressionLevelType): IZLibMethodsCompression;
    /// <summary>
    /// Compresso e codificao de uma string.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: lResult := TZLib.Data.Compress.Deflate.Text('Texto12345');
    /// end;
    /// </code>
    /// </summary>
    /// <remarks>
    /// <para>* Para codificar o parmetro de entrada em Base64, utilizar a property Base64 da classe TZLib.</para>
    /// <para>* O formato de compactao vai depender do algoritmo escolhido, Deflate ou GZip.</para>
    /// <para>* O Encoding utilizado quando no informado,  UTF8.</para>
    /// </remarks>
    /// <param name="Input">
    /// String na qual deseja-se compactar.
    /// </param>
    /// <returns>Retorna uma interface do tipo IZLibResult que contm o resultado do processo de compresso.</returns>
    function Text(const Input: string): IZLibResult; overload;
    /// <summary>
    /// Compresso e codificao de uma string, com a opo de identificar um Encoding.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: lResult := TZLib.Data.Compress.Deflate.Text('Texto12345', TEncoding.UTF8);
    /// end;
    /// </code>
    /// </summary>
    /// <remarks>
    /// <para>* Para codificar o parmetro de entrada em Base64, utilizar a property Base64 da classe TZLib.</para>
    /// <para>* O formato de compactao vai depender do algoritmo escolhido, Deflate ou GZip.</para>
    /// <para>* O Encoding utilizado quando no informado,  UTF8.</para>
    /// </remarks>
    /// <param name="Input">
    /// String na qual deseja-se compactar.
    /// </param>
    /// <param name="Encoding">
    /// Identificao do Encoding.
    /// </param>
    /// <returns>Retorna uma interface do tipo IZLibResult que contm o resultado do processo de compresso.</returns>
    function Text(const Input: string; const Encoding: TEncoding): IZLibResult; overload;
    /// <summary>
    /// Compresso e codificao de uma string, com a opo de salvar o resultado em um arquivo atravs do parmetro FileName.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: lResult := TZLib.Data.Compress.Deflate.SaveToFile('Texto12345', 'C:\Teste.txt');
    /// end;
    /// </code>
    /// </summary>
    /// <remarks>
    /// <para>* Para codificar o parmetro de entrada em Base64, utilizar a property Base64 da classe TZLib.</para>
    /// <para>* O formato de compactao vai depender do algoritmo escolhido, Deflate ou GZip.</para>
    /// <para>* O Encoding utilizado quando no informado,  UTF8.</para>
    /// </remarks>
    /// <param name="Input">
    /// String na qual deseja-se compactar.
    /// </param>
    /// <param name="FileName">
    /// Nome do arquivo que ser utilizado para salvar o resultado do processo de compactao.
    /// </param>
    /// <returns>Retorna uma interface do tipo IZLibResult que contm o resultado do processo de compresso.</returns>
    function SaveToFile(const Input: string; const FileName: TFileName): IZLibResult;
    /// <summary>
    /// Compresso e codificao de arquivo.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: lResult := TZLib.Data.Compress.Deflate.LoadFromFile('C:\Teste.txt');
    /// end;
    /// </code>
    /// </summary>
    /// <remarks>
    /// <para>* Utilizado em imagens, pdf, executveis entre outros.</para>
    /// <para>* Para codificar o parmetro de entrada em Base64, utilizar a property Base64 da classe TZLib.</para>
    /// <para>* O formato de compactao vai depender do algoritmo escolhido, Deflate ou GZip.</para>
    /// </remarks>
    /// <param name="Input">
    /// Arquivo na qual deseja-se compactar.
    /// </param>
    /// <returns>Retorna uma interface do tipo IZLibResult que contm o resultado do processo de compresso.</returns>
    function LoadFromFile(const Input: TFileName): IZLibResult;
    /// <summary>
    /// Compresso e codificao de uma variavl Stream.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: lResult := TZLib.Data.Compress.Deflate.LoadFromStream(TStream);
    /// end;
    /// </code>
    /// </summary>
    /// <remarks>
    /// <para>* Para codificar o parmetro de entrada em Base64, utilizar a property Base64 da classe TZLib.</para>
    /// <para>* O formato de compactao vai depender do algoritmo escolhido, Deflate ou GZip.</para>
    /// </remarks>
    /// <param name="Input">
    /// Stream na qual deseja-se compactar.
    /// </param>
    /// <returns>Retorna uma interface do tipo IZLibResult que contm o resultado do processo de compresso.</returns>
    function LoadFromStream(Input: TStream): IZLibResult;
  end;

  {$ENDREGION}

  {$REGION 'DECOMPRESSION'}

  IZLibMethodsDecompression = interface;

  IZLibAlgorithmDecompression = interface
    ['{D70E18FA-7E27-4A53-A228-1A907F4E97DB}']
    /// <summary>
    /// Processo de descompresso usando o algoritimo Deflate.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: lResult := TZLib.Base64.Decompress.Deflate.Text('eNozNDI2MQUAAvgBAA==');
    /// end;
    /// </code>
    /// </summary>
    /// <returns>Retorna uma interface do tipo IZLibMethodsDecompression que implementa o conceito de fluent interface para guiar no uso da biblioteca.</returns>
    function Deflate: IZLibMethodsDecompression;
    /// <summary>
    /// Processo de descompresso usando o algoritimo GZip.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: lResult := TZLib.Base64.Decompress.GZip.Text('eNozNDI2MQUAAvgBAA==');
    /// end;
    /// </code>
    /// </summary>
    /// <returns>Retorna uma interface do tipo IZLibMethodsDecompression que implementa o conceito de fluent interface para guiar no uso da biblioteca.</returns>
    function GZip: IZLibMethodsDecompression;
    function &End: IZLibOperation;
  end;

  IZLibMethodsDecompression = interface
    ['{45E581C3-39F6-41E4-BC23-C0CC2C87A2C1}']
    /// <summary>
    /// Decodificao e Descompresso de uma string.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: lResult := TZLib.Base64.Decompress.Deflate.Text('eNozNDI2MQUAAvgBAA==');
    /// end;
    /// </code>
    /// </summary>
    /// <remarks>
    /// <para>* Para decodificar o parmetro de entrada em Base64, utilizar a property Base64 da classe TZLib.</para>
    /// <para>* O formato de descompactao vai depender do algoritmo escolhido, Deflate ou GZip.</para>
    /// </remarks>
    /// <param name="Input">
    /// String na qual deseja-se descompactar.
    /// </param>
    /// <returns>Retorna uma interface do tipo IZLibResult que contm o resultado do processo de descompresso.</returns>
    function Text(const Input: string): IZLibResult;
    /// <summary>
    /// Decodificao e Descompresso de uma string, com a opo de salvar o resultado em um arquivo atravs do parmetro FileName.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: lResult := TZLib.Data.Decompress.Deflate.SaveToFile('eNozNDI2MQUAAvgBAA==', 'C:\Teste.txt');
    /// end;
    /// </code>
    /// </summary>
    /// <remarks>
    /// <para>* Para decodificar o parmetro de entrada em Base64, utilizar a property Base64 da classe TZLib.</para>
    /// <para>* O formato de descompactao vai depender do algoritmo escolhido, Deflate ou GZip.</para>
    /// <para>* O Encoding utilizado quando no informado,  UTF8.</para>
    /// </remarks>
    /// <param name="Input">
    /// String na qual deseja-se descompactar.
    /// </param>
    /// <param name="FileName">
    /// Nome do arquivo que ser utilizado para salvar o resultado do processo de descompactao.
    /// </param>
    /// <returns>Retorna uma interface do tipo IZLibResult que contm o resultado do processo de descompresso.</returns>
    function SaveToFile(const Input: string; const FileName: TFileName): IZLibResult;
    /// <summary>
    /// Decodificao e Descompresso de arquivo.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: IZLibResult := TZLib.Data.Decompress.Deflate.LoadFromFile('C:\Teste.txt');
    /// end;
    /// </code>
    /// </summary>
    /// <remarks>
    /// <para>* Utilizado em imagens, pdf, executveis entre outros.</para>
    /// <para>* Para decodificar o parmetro de entrada em Base64, utilizar a property Base64 da classe TZLib.</para>
    /// <para>* O formato de descompactao vai depender do algoritmo escolhido, Deflate ou GZip.</para>
    /// </remarks>
    /// <param name="Input">
    /// Arquivo na qual deseja-se descompactar.
    /// </param>
    /// <returns>Retorna uma interface do tipo IZLibResult que contm o resultado do processo de descompresso.</returns>
    function LoadFromFile(const Input: TFileName): IZLibResult;
    /// <summary>
    /// Decodificao e Descompresso de uma variavl Stream.
    /// <code>
    /// var
    ///   lResult: IZLibResult;
    /// begin
    ///   Exemplo: IZLibResult := TZLib.Data.Decompress.Deflate.LoadFromStream(TStream);
    /// end;
    /// </code>
    /// </summary>
    /// <remarks>
    /// <para>* Para decodificar o parmetro de entrada em Base64, utilizar a property Base64 da classe TZLib.</para>
    /// <para>* O formato de descompactao vai depender do algoritmo escolhido, Deflate ou GZip.</para>
    /// </remarks>
    /// <param name="Input">
    /// Stream na qual deseja-se descompactar.
    /// </param>
    /// <returns>Retorna uma interface do tipo IZLibResult que contm o resultado do processo de descompresso.</returns>
    function LoadFromStream(Input: TStream): IZLibResult; overload;
  end;

  {$ENDREGION}

implementation

end.

