PRAGMA foreign_keys=OFF;
BEGIN TRANSACTION;
CREATE TABLE instructions (platform TEXT, mnem TEXT, description TEXT);
INSERT INTO "instructions" VALUES('x86_64','AAA','AAA-ASCII Adjust After Addition
Opcode                     Instruction                Op/   64-bit    Compat/ Description
                                                      En    Mode      Leg Mode
37                         AAA                        NP    Invalid   Valid       ASCII adjust AL after addition.



                                              Instruction Operand Encoding
  Op/En              Operand 1                 Operand 2                      Operand 3                      Operand 4
     NP                  NA                       NA                             NA                                 NA

Description
Adjusts the sum of two unpacked BCD values to create an unpacked BCD result. The AL register is the implied
source and destination operand for this instruction. The AAA instruction is only useful when it follows an ADD
instruction that adds (binary addition) two unpacked BCD values and stores a byte result in the AL register. The
AAA instruction then adjusts the contents of the AL register to contain the correct 1-digit unpacked BCD result.
If the addition produces a decimal carry, the AH register increments by 1, and the CF and AF flags are set. If there
was no decimal carry, the CF and AF flags are cleared and the AH register is unchanged. In either case, bits 4
through 7 of the AL register are set to 0.
This instruction executes as described in compatibility mode and legacy mode. It is not valid in 64-bit mode.

Operation
IF 64-Bit Mode
    THEN
         #UD;
    ELSE
         IF ((AL AND 0FH) > 9) or (AF = 1)
               THEN
                   AX <- AX + 106H;
                   AF <- 1;
                   CF <- 1;
               ELSE
                   AF <- 0;
                   CF <- 0;
         FI;
         AL <- AL AND 0FH;
FI;

Flags Affected
The AF and CF flags are set to 1 if the adjustment results in a decimal carry; otherwise they are set to 0. The OF,
SF, ZF, and PF flags are undefined.

Protected Mode Exceptions
#UD                     If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as protected mode.





Compatibility Mode Exceptions
Same exceptions as protected mode.

64-Bit Mode Exceptions
#UD                      If in 64-bit mode.




');
INSERT INTO "instructions" VALUES('x86_64','AAD','AAD-ASCII Adjust AX Before Division
Opcode                    Instruction                    Op/   64-bit    Compat/ Description
                                                         En    Mode      Leg Mode
D5 0A                     AAD                            NP    Invalid   Valid      ASCII adjust AX before division.
D5 ib                     AAD imm8                       NP    Invalid   Valid      Adjust AX before division to number base
                                                                                    imm8.



                                                Instruction Operand Encoding
  Op/En             Operand 1                      Operand 2                     Operand 3                     Operand 4
    NP                  NA                               NA                         NA                             NA

Description
Adjusts two unpacked BCD digits (the least-significant digit in the AL register and the most-significant digit in the
AH register) so that a division operation performed on the result will yield a correct unpacked BCD value. The AAD
instruction is only useful when it precedes a DIV instruction that divides (binary division) the adjusted value in the
AX register by an unpacked BCD value.
The AAD instruction sets the value in the AL register to (AL + (10 * AH)), and then clears the AH register to 00H.
The value in the AX register is then equal to the binary equivalent of the original unpacked two-digit (base 10)
number in registers AH and AL.
The generalized version of this instruction allows adjustment of two unpacked digits of any number base (see the
"Operation" section below), by setting the imm8 byte to the selected number base (for example, 08H for octal, 0AH
for decimal, or 0CH for base 12 numbers). The AAD mnemonic is interpreted by all assemblers to mean adjust
ASCII (base 10) values. To adjust values in another number base, the instruction must be hand coded in machine
code (D5 imm8).
This instruction executes as described in compatibility mode and legacy mode. It is not valid in 64-bit mode.

Operation
IF 64-Bit Mode
    THEN
         #UD;
    ELSE
         tempAL <- AL;
         tempAH <- AH;
         AL <- (tempAL + (tempAH * imm8)) AND FFH;
         (* imm8 is set to 0AH for the AAD mnemonic.*)
         AH <- 0;
FI;
The immediate value (imm8) is taken from the second byte of the instruction.

Flags Affected
The SF, ZF, and PF flags are set according to the resulting binary value in the AL register; the OF, AF, and CF flags
are undefined.

Protected Mode Exceptions
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as protected mode.





Virtual-8086 Mode Exceptions
Same exceptions as protected mode.

Compatibility Mode Exceptions
Same exceptions as protected mode.

64-Bit Mode Exceptions
#UD                      If in 64-bit mode.




');
INSERT INTO "instructions" VALUES('x86_64','AAM','AAM-ASCII Adjust AX After Multiply
Opcode                   Instruction                 Op/    64-bit     Compat/ Description
                                                     En     Mode       Leg Mode
D4 0A                    AAM                         NP     Invalid    Valid       ASCII adjust AX after multiply.
D4 ib                    AAM imm8                    NP     Invalid    Valid       Adjust AX after multiply to number base
                                                                                   imm8.



                                               Instruction Operand Encoding
  Op/En             Operand 1                   Operand 2                      Operand 3                     Operand 4
    NP                 NA                           NA                            NA                             NA

Description
Adjusts the result of the multiplication of two unpacked BCD values to create a pair of unpacked (base 10) BCD
values. The AX register is the implied source and destination operand for this instruction. The AAM instruction is
only useful when it follows an MUL instruction that multiplies (binary multiplication) two unpacked BCD values and
stores a word result in the AX register. The AAM instruction then adjusts the contents of the AX register to contain
the correct 2-digit unpacked (base 10) BCD result.
The generalized version of this instruction allows adjustment of the contents of the AX to create two unpacked
digits of any number base (see the "Operation" section below). Here, the imm8 byte is set to the selected number
base (for example, 08H for octal, 0AH for decimal, or 0CH for base 12 numbers). The AAM mnemonic is interpreted
by all assemblers to mean adjust to ASCII (base 10) values. To adjust to values in another number base, the
instruction must be hand coded in machine code (D4 imm8).
This instruction executes as described in compatibility mode and legacy mode. It is not valid in 64-bit mode.

Operation
IF 64-Bit Mode
    THEN
         #UD;
    ELSE
         tempAL <- AL;
         AH <- tempAL / imm8; (* imm8 is set to 0AH for the AAM mnemonic *)
         AL <- tempAL MOD imm8;
FI;
The immediate value (imm8) is taken from the second byte of the instruction.

Flags Affected
The SF, ZF, and PF flags are set according to the resulting binary value in the AL register. The OF, AF, and CF flags
are undefined.

Protected Mode Exceptions
#DE                   If an immediate value of 0 is used.
#UD                   If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as protected mode.





Compatibility Mode Exceptions
Same exceptions as protected mode.

64-Bit Mode Exceptions
#UD                      If in 64-bit mode.




');
INSERT INTO "instructions" VALUES('x86_64','AAS','AAS-ASCII Adjust AL After Subtraction
Opcode                     Instruction                Op/   64-bit    Compat/ Description
                                                      En    Mode      Leg Mode
3F                         AAS                        NP    Invalid   Valid       ASCII adjust AL after subtraction.



                                              Instruction Operand Encoding
   Op/En              Operand 1                 Operand 2                     Operand 3                      Operand 4
     NP                   NA                          NA                         NA                              NA

Description
Adjusts the result of the subtraction of two unpacked BCD values to create a unpacked BCD result. The AL register
is the implied source and destination operand for this instruction. The AAS instruction is only useful when it follows
a SUB instruction that subtracts (binary subtraction) one unpacked BCD value from another and stores a byte
result in the AL register. The AAA instruction then adjusts the contents of the AL register to contain the correct 1-
digit unpacked BCD result.
If the subtraction produced a decimal carry, the AH register decrements by 1, and the CF and AF flags are set. If no
decimal carry occurred, the CF and AF flags are cleared, and the AH register is unchanged. In either case, the AL
register is left with its top four bits set to 0.
This instruction executes as described in compatibility mode and legacy mode. It is not valid in 64-bit mode.

Operation
IF 64-bit mode
    THEN
         #UD;
    ELSE
         IF ((AL AND 0FH) > 9) or (AF = 1)
               THEN
                   AX <- AX - 6;
                   AH <- AH - 1;
                   AF <- 1;
                   CF <- 1;
                   AL <- AL AND 0FH;
               ELSE
                   CF <- 0;
                   AF <- 0;
                   AL <- AL AND 0FH;
         FI;
FI;

Flags Affected
The AF and CF flags are set to 1 if there is a decimal borrow; otherwise, they are cleared to 0. The OF, SF, ZF, and
PF flags are undefined.

Protected Mode Exceptions
#UD                     If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as protected mode.





Virtual-8086 Mode Exceptions
Same exceptions as protected mode.

Compatibility Mode Exceptions
Same exceptions as protected mode.

64-Bit Mode Exceptions
#UD                      If in 64-bit mode.




');
INSERT INTO "instructions" VALUES('x86_64','ADC','ADC-Add with Carry
Opcode                     Instruction                  Op/    64-bit      Compat/ Description
                                                        En     Mode        Leg Mode
14 ib                      ADC AL, imm8                 I      Valid       Valid       Add with carry imm8 to AL.
15 iw                      ADC AX, imm16                I      Valid       Valid       Add with carry imm16 to AX.
15 id                      ADC EAX, imm32               I      Valid       Valid       Add with carry imm32 to EAX.
REX.W + 15 id              ADC RAX, imm32               I      Valid       N.E.        Add with carry imm32 sign extended to 64-
                                                                                       bits to RAX.
80 /2 ib                   ADC r/m8, imm8               MI     Valid       Valid       Add with carry imm8 to r/m8.
                                      *
REX + 80 /2 ib             ADC r/m8 , imm8              MI     Valid       N.E.        Add with carry imm8 to r/m8.
81 /2 iw                   ADC r/m16, imm16             MI     Valid       Valid       Add with carry imm16 to r/m16.
81 /2 id                   ADC r/m32, imm32             MI     Valid       Valid       Add with CF imm32 to r/m32.
REX.W + 81 /2 id           ADC r/m64, imm32             MI     Valid       N.E.        Add with CF imm32 sign extended to 64-bits
                                                                                       to r/m64.
83 /2 ib                   ADC r/m16, imm8              MI     Valid       Valid       Add with CF sign-extended imm8 to r/m16.
83 /2 ib                   ADC r/m32, imm8              MI     Valid       Valid       Add with CF sign-extended imm8 into r/m32.
REX.W + 83 /2 ib           ADC r/m64, imm8              MI     Valid       N.E.        Add with CF sign-extended imm8 into r/m64.
10 /r                      ADC r/m8, r8                 MR     Valid       Valid       Add with carry byte register to r/m8.
REX + 10 /r                ADC r/m8*, r8*               MR     Valid       N.E.        Add with carry byte register to r/m64.
11 /r                      ADC r/m16, r16               MR     Valid       Valid       Add with carry r16 to r/m16.
11 /r                      ADC r/m32, r32               MR     Valid       Valid       Add with CF r32 to r/m32.
REX.W + 11 /r              ADC r/m64, r64               MR     Valid       N.E.        Add with CF r64 to r/m64.
12 /r                      ADC r8, r/m8                 RM     Valid       Valid       Add with carry r/m8 to byte register.
REX + 12 /r                ADC r8*, r/m8*               RM     Valid       N.E.        Add with carry r/m64 to byte register.
13 /r                      ADC r16, r/m16               RM     Valid       Valid       Add with carry r/m16 to r16.
13 /r                      ADC r32, r/m32               RM     Valid       Valid       Add with CF r/m32 to r32.
REX.W + 13 /r              ADC r64, r/m64               RM     Valid       N.E.        Add with CF r/m64 to r64.
NOTES:
*In 64-bit mode, r/m8 can not be encoded to access the following byte registers if a REX prefix is used: AH, BH, CH, DH.



                                                 Instruction Operand Encoding
   Op/En              Operand 1                    Operand 2                       Operand 3                      Operand 4
    RM             ModRM:reg (r, w)              ModRM:r/m (r)                        NA                              NA
    MR             ModRM:r/m (r, w)              ModRM:reg (r)                        NA                              NA
    MI             ModRM:r/m (r, w)                  imm8                             NA                              NA
        I          AL/AX/EAX/RAX                     imm8                             NA                              NA

Description
Adds the destination operand (first operand), the source operand (second operand), and the carry (CF) flag and
stores the result in the destination operand. The destination operand can be a register or a memory location; the
source operand can be an immediate, a register, or a memory location. (However, two memory operands cannot be
used in one instruction.) The state of the CF flag represents a carry from a previous addition. When an immediate
value is used as an operand, it is sign-extended to the length of the destination operand format.





The ADC instruction does not distinguish between signed or unsigned operands. Instead, the processor evaluates
the result for both data types and sets the OF and CF flags to indicate a carry in the signed or unsigned result,
respectively. The SF flag indicates the sign of the signed result.
The ADC instruction is usually executed as part of a multibyte or multiword addition in which an ADD instruction is
followed by an ADC instruction.
This instruction can be used with a LOCK prefix to allow the instruction to be executed atomically.
In 64-bit mode, the instruction''s default operation size is 32 bits. Using a REX prefix in the form of REX.R permits
access to additional registers (R8-R15). Using a REX prefix in the form of REX.W promotes operation to 64 bits. See
the summary chart at the beginning of this section for encoding data and limits.

Operation
DEST <- DEST + SRC + CF;

Intel C/C++ Compiler Intrinsic Equivalent
ADC:       extern unsigned char _addcarry_u8(unsigned char c_in, unsigned char src1, unsigned char src2, unsigned char *sum_out);

ADC:       extern unsigned char _addcarry_u16(unsigned char c_in, unsigned short src1, unsigned short src2, unsigned short
*sum_out);

ADC:       extern unsigned char _addcarry_u32(unsigned char c_in, unsigned int src1, unsigned char int, unsigned int *sum_out);

ADC:       extern unsigned char _addcarry_u64(unsigned char c_in, unsigned __int64 src1, unsigned __int64 src2, unsigned __int64
*sum_out);

Flags Affected
The OF, SF, ZF, AF, CF, and PF flags are set according to the result.

Protected Mode Exceptions
#GP(0)                 If the destination is located in a non-writable segment.
                       If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                       If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                       selector.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                    If the LOCK prefix is used but the destination is not a memory operand.

Real-Address Mode Exceptions
#GP                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                    If a memory operand effective address is outside the SS segment limit.
#UD                    If the LOCK prefix is used but the destination is not a memory operand.

Virtual-8086 Mode Exceptions
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made.
#UD                    If the LOCK prefix is used but the destination is not a memory operand.





Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.




');
INSERT INTO "instructions" VALUES('x86_64','ADCX','ADCX - Unsigned Integer Addition of Two Operands with Carry Flag
 Opcode/                                 Op/    64/32bit      CPUID      Description
 Instruction                             En     Mode          Feature
                                                Support       Flag
 66 0F 38 F6 /r                          RM     V/V           ADX        Unsigned addition of r32 with CF, r/m32 to r32, writes CF.
 ADCX r32, r/m32
 66 REX.w 0F 38 F6 /r                    RM     V/NE          ADX        Unsigned addition of r64 with CF, r/m64 to r64, writes CF.
 ADCX r64, r/m64



                                                    Instruction Operand Encoding
      Op/En                 Operand 1                      Operand 2                    Operand 3                    Operand 4
       RM                ModRM:reg (r, w)                ModRM:r/m (r)                      NA                           NA

Description
Performs an unsigned addition of the destination operand (first operand), the source operand (second operand)
and the carry-flag (CF) and stores the result in the destination operand. The destination operand is a general-
purpose register, whereas the source operand can be a general-purpose register or memory location. The state of
CF can represent a carry from a previous addition. The instruction sets the CF flag with the carry generated by the
unsigned addition of the operands.
The ADCX instruction is executed in the context of multi-precision addition, where we add a series of operands with
a carry-chain. At the beginning of a chain of additions, we need to make sure the CF is in a desired initial state.
Often, this initial state needs to be 0, which can be achieved with an instruction to zero the CF (e.g. XOR).
This instruction is supported in real mode and virtual-8086 mode. The operand size is always 32 bits if not in 64-
bit mode.
In 64-bit mode, the default operation size is 32 bits. Using a REX Prefix in the form of REX.R permits access to addi-
tional registers (R8-15). Using REX Prefix in the form of REX.W promotes operation to 64 bits.
ADCX executes normally either inside or outside a transaction region.
Note: ADCX defines the OF flag differently than the ADD/ADC instructions as defined in Intel 64 and IA-32 Archi-
tectures Software Developer''s Manual, Volume 2A.

Operation
IF OperandSize is 64-bit
    THEN CF:DEST[63:0] <- DEST[63:0] + SRC[63:0] + CF;
    ELSE CF:DEST[31:0] <- DEST[31:0] + SRC[31:0] + CF;
FI;

Flags Affected
CF is updated based on result. OF, SF, ZF, AF and PF flags are unmodified.

Intel C/C++ Compiler Intrinsic Equivalent
unsigned char _addcarryx_u32 (unsigned char c_in, unsigned int src1, unsigned int src2, unsigned int *sum_out);
unsigned char _addcarryx_u64 (unsigned char c_in, unsigned __int64 src1, unsigned __int64 src2, unsigned __int64 *sum_out);

SIMD Floating-Point Exceptions
None

Protected Mode Exceptions
#UD                      If the LOCK prefix is used.
                         If CPUID.(EAX=07H, ECX=0H):EBX.ADX[bit 19] = 0.
#SS(0)                   For an illegal address in the SS segment.



#GP(0)              For an illegal memory operand effective address in the CS, DS, ES, FS or GS segments.
                    If the DS, ES, FS, or GS register is used to access memory and it contains a null segment
                    selector.
#PF(fault-code)     For a page fault.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.

Real-Address Mode Exceptions
#UD                 If the LOCK prefix is used.
                    If CPUID.(EAX=07H, ECX=0H):EBX.ADX[bit 19] = 0.
#SS(0)              For an illegal address in the SS segment.
#GP(0)              If any part of the operand lies outside the effective address space from 0 to FFFFH.

Virtual-8086 Mode Exceptions
#UD                 If the LOCK prefix is used.
                    If CPUID.(EAX=07H, ECX=0H):EBX.ADX[bit 19] = 0.
#SS(0)              For an illegal address in the SS segment.
#GP(0)              If any part of the operand lies outside the effective address space from 0 to FFFFH.
#PF(fault-code)     For a page fault.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#UD                 If the LOCK prefix is used.
                    If CPUID.(EAX=07H, ECX=0H):EBX.ADX[bit 19] = 0.
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#PF(fault-code)     For a page fault.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.




');
INSERT INTO "instructions" VALUES('x86_64','ADD','ADD-Add
Opcode                    Instruction                   Op/   64-bit       Compat/ Description
                                                        En    Mode         Leg Mode
04 ib                     ADD AL, imm8                  I     Valid        Valid        Add imm8 to AL.
05 iw                     ADD AX, imm16                 I     Valid        Valid        Add imm16 to AX.
05 id                     ADD EAX, imm32                I     Valid        Valid        Add imm32 to EAX.
REX.W + 05 id             ADD RAX, imm32                I     Valid        N.E.         Add imm32 sign-extended to 64-bits to RAX.
80 /0 ib                  ADD r/m8, imm8                MI    Valid        Valid        Add imm8 to r/m8.
REX + 80 /0 ib            ADD r/m8*, imm8               MI    Valid        N.E.         Add sign-extended imm8 to r/m64.
81 /0 iw                  ADD r/m16, imm16              MI    Valid        Valid        Add imm16 to r/m16.
81 /0 id                  ADD r/m32, imm32              MI    Valid        Valid        Add imm32 to r/m32.
REX.W + 81 /0 id          ADD r/m64, imm32              MI    Valid        N.E.         Add imm32 sign-extended to 64-bits to
                                                                                        r/m64.
83 /0 ib                  ADD r/m16, imm8               MI    Valid        Valid        Add sign-extended imm8 to r/m16.
83 /0 ib                  ADD r/m32, imm8               MI    Valid        Valid        Add sign-extended imm8 to r/m32.
REX.W + 83 /0 ib          ADD r/m64, imm8               MI    Valid        N.E.         Add sign-extended imm8 to r/m64.
00 /r                     ADD r/m8, r8                  MR    Valid        Valid        Add r8 to r/m8.
                                     *    *
REX + 00 /r               ADD r/m8 , r8                 MR    Valid        N.E.         Add r8 to r/m8.
01 /r                     ADD r/m16, r16                MR    Valid        Valid        Add r16 to r/m16.
01 /r                     ADD r/m32, r32                MR    Valid        Valid        Add r32 to r/m32.
REX.W + 01 /r             ADD r/m64, r64                MR    Valid        N.E.         Add r64 to r/m64.
02 /r                     ADD r8, r/m8                  RM    Valid        Valid        Add r/m8 to r8.
                                 *        *
REX + 02 /r               ADD r8 , r/m8                 RM    Valid        N.E.         Add r/m8 to r8.
03 /r                     ADD r16, r/m16                RM    Valid        Valid        Add r/m16 to r16.
03 /r                     ADD r32, r/m32                RM    Valid        Valid        Add r/m32 to r32.
REX.W + 03 /r             ADD r64, r/m64                RM    Valid        N.E.         Add r/m64 to r64.
NOTES:
*In 64-bit mode, r/m8 can not be encoded to access the following byte registers if a REX prefix is used: AH, BH, CH, DH.



                                                 Instruction Operand Encoding
  Op/En             Operand 1                     Operand 2                        Operand 3                     Operand 4
   RM            ModRM:reg (r, w)                ModRM:r/m (r)                        NA                             NA
   MR            ModRM:r/m (r, w)                ModRM:reg (r)                        NA                             NA
   MI            ModRM:r/m (r, w)                    imm8                             NA                             NA
    I            AL/AX/EAX/RAX                       imm8                             NA                             NA

Description
Adds the destination operand (first operand) and the source operand (second operand) and then stores the result
in the destination operand. The destination operand can be a register or a memory location; the source operand
can be an immediate, a register, or a memory location. (However, two memory operands cannot be used in one
instruction.) When an immediate value is used as an operand, it is sign-extended to the length of the destination
operand format.
The ADD instruction performs integer addition. It evaluates the result for both signed and unsigned integer oper-
ands and sets the CF and OF flags to indicate a carry (overflow) in the signed or unsigned result, respectively. The
SF flag indicates the sign of the signed result.



This instruction can be used with a LOCK prefix to allow the instruction to be executed atomically.
In 64-bit mode, the instruction''s default operation size is 32 bits. Using a REX prefix in the form of REX.R permits
access to additional registers (R8-R15). Using a REX prefix in the form of REX.W promotes operation to 64 bits. See
the summary chart at the beginning of this section for encoding data and limits.

Operation
DEST <- DEST + SRC;

Flags Affected
The OF, SF, ZF, AF, CF, and PF flags are set according to the result.

Protected Mode Exceptions
#GP(0)               If the destination is located in a non-writable segment.
                     If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                     If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                     selector.
#SS(0)               If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made while the
                     current privilege level is 3.
#UD                  If the LOCK prefix is used but the destination is not a memory operand.

Real-Address Mode Exceptions
#GP                  If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                  If a memory operand effective address is outside the SS segment limit.
#UD                  If the LOCK prefix is used but the destination is not a memory operand.

Virtual-8086 Mode Exceptions
#GP(0)               If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)               If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made.
#UD                  If the LOCK prefix is used but the destination is not a memory operand.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)               If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)               If the memory address is in a non-canonical form.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made while the
                     current privilege level is 3.
#UD                  If the LOCK prefix is used but the destination is not a memory operand.




');
INSERT INTO "instructions" VALUES('x86_64','ADDPD','ADDPD-Add Packed Double-Precision Floating-Point Values
 Opcode/                               Op /    64/32        CPUID          Description
 Instruction                           En      bit Mode     Feature
                                               Support      Flag
 66 0F 58 /r                           RM      V/V          SSE2           Add packed double-precision floating-point values from
 ADDPD xmm1, xmm2/m128                                                     xmm2/mem to xmm1 and store result in xmm1.
 VEX.NDS.128.66.0F.WIG 58 /r           RVM     V/V          AVX            Add packed double-precision floating-point values from
 VADDPD xmm1,xmm2,                                                         xmm3/mem to xmm2 and store result in xmm1.
 xmm3/m128
 VEX.NDS.256.66.0F.WIG 58 /r           RVM     V/V          AVX            Add packed double-precision floating-point values from
 VADDPD ymm1, ymm2,                                                        ymm3/mem to ymm2 and store result in ymm1.
 ymm3/m256
 EVEX.NDS.128.66.0F.W1 58 /r           FV      V/V          AVX512VL       Add packed double-precision floating-point values from
 VADDPD xmm1 {k1}{z}, xmm2,                                 AVX512F        xmm3/m128/m64bcst to xmm2 and store result in xmm1
 xmm3/m128/m64bcst                                                         with writemask k1.
 EVEX.NDS.256.66.0F.W1 58 /r           FV      V/V          AVX512VL       Add packed double-precision floating-point values from
 VADDPD ymm1 {k1}{z}, ymm2,                                 AVX512F        ymm3/m256/m64bcst to ymm2 and store result in ymm1
 ymm3/m256/m64bcst                                                         with writemask k1.
 EVEX.NDS.512.66.0F.W1 58 /r           FV      V/V          AVX512F        Add packed double-precision floating-point values from
 VADDPD zmm1 {k1}{z}, zmm2,                                                zmm3/m512/m64bcst to zmm2 and store result in zmm1
 zmm3/m512/m64bcst{er}                                                     with writemask k1.



                                                     Instruction Operand Encoding
        Op/En                  Operand 1                    Operand 2                     Operand 3                 Operand 4
         RM                 ModRM:reg (r, w)               ModRM:r/m (r)                      NA                        NA
         RVM                 ModRM:reg (w)                   VEX.vvvv                    ModRM:r/m (r)                  NA
       FV-RVM                ModRM:reg (w)                  EVEX.vvvv                    ModRM:r/m (r)                  NA

Description
Add two, four or eight packed double-precision floating-point values from the first source operand to the second
source operand, and stores the packed double-precision floating-point results in the destination operand.
EVEX encoded versions: The first source operand is a ZMM/YMM/XMM register. The second source operand can be
a ZMM/YMM/XMM register, a 512/256/128-bit memory location or a 512/256/128-bit vector broadcasted from a
64-bit memory location. The destination operand is a ZMM/YMM/XMM register conditionally updated with
writemask k1.
VEX.256 encoded version: The first source operand is a YMM register. The second source operand can be a YMM
register or a 256-bit memory location. The destination operand is a YMM register. The upper bits (MAX_VL-1:256)
of the corresponding ZMM register destination are zeroed.
VEX.128 encoded version: the first source operand is a XMM register. The second source operand is an XMM
register or 128-bit memory location. The destination operand is an XMM register. The upper bits (MAX_VL-1:128)
of the corresponding ZMM register destination are zeroed.
128-bit Legacy SSE version: The second source can be an XMM register or an 128-bit memory location. The desti-
nation is not distinct from the first source XMM register and the upper Bits (MAX_VL-1:128) of the corresponding
ZMM register destination are unmodified.

Operation
VADDPD (EVEX encoded versions) when src2 operand is a vector register
(KL, VL) = (2, 128), (4, 256), (8, 512)
IF (VL = 512) AND (EVEX.b = 1)
    THEN
         SET_RM(EVEX.RC);
    ELSE


         SET_RM(MXCSR.RM);
FI;
FOR j <- 0 TO KL-1
    i <- j * 64
    IF k1[j] OR *no writemask*
         THEN DEST[i+63:i] <- SRC1[i+63:i] + SRC2[i+63:i]
         ELSE
              IF *merging-masking*                ; merging-masking
                   THEN *DEST[i+63:i] remains unchanged*
                   ELSE                           ; zeroing-masking
                        DEST[i+63:i] <- 0
              FI
    FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VADDPD (EVEX encoded versions) when src2 operand is a memory source
(KL, VL) = (2, 128), (4, 256), (8, 512)

FOR j <- 0 TO KL-1
   i <- j * 64
   IF k1[j] OR *no writemask*
        THEN
             IF (EVEX.b = 1)
                  THEN
                       DEST[i+63:i] <- SRC1[i+63:i] + SRC2[63:0]
                  ELSE
                       DEST[i+63:i] <- SRC1[i+63:i] + SRC2[i+63:i]
             FI;
        ELSE
             IF *merging-masking*                 ; merging-masking
                  THEN *DEST[i+63:i] remains unchanged*
                  ELSE                            ; zeroing-masking
                       DEST[i+63:i] <- 0
             FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VADDPD (VEX.256 encoded version)
DEST[63:0] <- SRC1[63:0] + SRC2[63:0]
DEST[127:64] <- SRC1[127:64] + SRC2[127:64]
DEST[191:128] <- SRC1[191:128] + SRC2[191:128]
DEST[255:192] <- SRC1[255:192] + SRC2[255:192]
DEST[MAX_VL-1:256] <- 0
.





VADDPD (VEX.128 encoded version)
DEST[63:0] <- SRC1[63:0] + SRC2[63:0]
DEST[127:64] <- SRC1[127:64] + SRC2[127:64]
DEST[MAX_VL-1:128] <- 0

ADDPD (128-bit Legacy SSE version)
DEST[63:0] <- DEST[63:0] + SRC[63:0]
DEST[127:64] <- DEST[127:64] + SRC[127:64]
DEST[MAX_VL-1:128] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VADDPD __m512d _mm512_add_pd (__m512d a, __m512d b);
VADDPD __m512d _mm512_mask_add_pd (__m512d s, __mmask8 k, __m512d a, __m512d b);
VADDPD __m512d _mm512_maskz_add_pd (__mmask8 k, __m512d a, __m512d b);
VADDPD __m256d _mm256_mask_add_pd (__m256d s, __mmask8 k, __m256d a, __m256d b);
VADDPD __m256d _mm256_maskz_add_pd (__mmask8 k, __m256d a, __m256d b);
VADDPD __m128d _mm_mask_add_pd (__m128d s, __mmask8 k, __m128d a, __m128d b);
VADDPD __m128d _mm_maskz_add_pd (__mmask8 k, __m128d a, __m128d b);
VADDPD __m512d _mm512_add_round_pd (__m512d a, __m512d b, int);
VADDPD __m512d _mm512_mask_add_round_pd (__m512d s, __mmask8 k, __m512d a, __m512d b, int);
VADDPD __m512d _mm512_maskz_add_round_pd (__mmask8 k, __m512d a, __m512d b, int);
ADDPD __m256d _mm256_add_pd (__m256d a, __m256d b);
ADDPD __m128d _mm_add_pd (__m128d a, __m128d b);

SIMD Floating-Point Exceptions
Overflow, Underflow, Invalid, Precision, Denormal

Other Exceptions
VEX-encoded instruction, see Exceptions Type 2.
EVEX-encoded instruction, see Exceptions Type E2.




');
INSERT INTO "instructions" VALUES('x86_64','ADDPS','ADDPS-Add Packed Single-Precision Floating-Point Values
 Opcode/                                Op /   64/32       CPUID      Description
 Instruction                            En     bit Mode    Feature
                                               Support     Flag
 0F 58 /r                               RM     V/V         SSE        Add packed single-precision floating-point values from
 ADDPS xmm1, xmm2/m128                                                xmm2/m128 to xmm1 and store result in xmm1.
 VEX.NDS.128.0F.WIG 58 /r               RVM    V/V         AVX        Add packed single-precision floating-point values from
 VADDPS xmm1,xmm2, xmm3/m128                                          xmm3/m128 to xmm2 and store result in xmm1.
 VEX.NDS.256.0F.WIG 58 /r               RVM    V/V         AVX        Add packed single-precision floating-point values from
 VADDPS ymm1, ymm2, ymm3/m256                                         ymm3/m256 to ymm2 and store result in ymm1.
 EVEX.NDS.128.0F.W0 58 /r               FV     V/V         AVX512VL   Add packed single-precision floating-point values from
 VADDPS xmm1 {k1}{z}, xmm2,                                AVX512F    xmm3/m128/m32bcst to xmm2 and store result in
 xmm3/m128/m32bcst                                                    xmm1 with writemask k1.
 EVEX.NDS.256.0F.W0 58 /r               FV     V/V         AVX512VL   Add packed single-precision floating-point values from
 VADDPS ymm1 {k1}{z}, ymm2,                                AVX512F    ymm3/m256/m32bcst to ymm2 and store result in
 ymm3/m256/m32bcst                                                    ymm1 with writemask k1.
 EVEX.NDS.512.0F.W0 58 /r               FV     V/V         AVX512F    Add packed single-precision floating-point values from
 VADDPS zmm1 {k1}{z}, zmm2,                                           zmm3/m512/m32bcst to zmm2 and store result in
 zmm3/m512/m32bcst {er}                                               zmm1 with writemask k1.



                                               Instruction Operand Encoding
      Op/En             Operand 1                    Operand 2                Operand 3                     Operand 4
       RM            ModRM:reg (r, w)             ModRM:r/m (r)                  NA                            NA
       RVM            ModRM:reg (w)                  VEX.vvvv              ModRM:r/m (r)                       NA
     FV-RVM           ModRM:reg (w)                  EVEX.vvvv             ModRM:r/m (r)                       NA

Description
Add four, eight or sixteen packed single-precision floating-point values from the first source operand with the
second source operand, and stores the packed single-precision floating-point results in the destination operand.
EVEX encoded versions: The first source operand is a ZMM/YMM/XMM register. The second source operand can be
a ZMM/YMM/XMM register, a 512/256/128-bit memory location or a 512/256/128-bit vector broadcasted from a
32-bit memory location. The destination operand is a ZMM/YMM/XMM register conditionally updated with
writemask k1.
VEX.256 encoded version: The first source operand is a YMM register. The second source operand can be a YMM
register or a 256-bit memory location. The destination operand is a YMM register. The upper bits (MAX_VL-1:256)
of the corresponding ZMM register destination are zeroed.
VEX.128 encoded version: the first source operand is a XMM register. The second source operand is an XMM
register or 128-bit memory location. The destination operand is an XMM register. The upper bits (MAX_VL-1:128)
of the corresponding ZMM register destination are zeroed.
128-bit Legacy SSE version: The second source can be an XMM register or an 128-bit memory location. The desti-
nation is not distinct from the first source XMM register and the upper Bits (MAX_VL-1:128) of the corresponding
ZMM register destination are unmodified.

Operation
VADDPS (EVEX encoded versions) when src2 operand is a register
(KL, VL) = (4, 128), (8, 256), (16, 512)
IF (VL = 512) AND (EVEX.b = 1)
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);




FI;
FOR j <- 0 TO KL-1
    i <- j * 32
    IF k1[j] OR *no writemask*
         THEN DEST[i+31:i] <- SRC1[i+31:i] + SRC2[i+31:i]
         ELSE
              IF *merging-masking*                ; merging-masking
                   THEN *DEST[i+31:i] remains unchanged*
                   ELSE                           ; zeroing-masking
                        DEST[i+31:i] <- 0
              FI
    FI;
ENDFOR;
DEST[MAX_VL-1:VL] <- 0

VADDPS (EVEX encoded versions) when src2 operand is a memory source
(KL, VL) = (4, 128), (8, 256), (16, 512)

FOR j <- 0 TO KL-1
   i <-j * 32
   IF k1[j] OR *no writemask*
        THEN
             IF (EVEX.b = 1)
                  THEN
                       DEST[i+31:i] <- SRC1[i+31:i] + SRC2[31:0]
                  ELSE
                       DEST[i+31:i] <- SRC1[i+31:i] + SRC2[i+31:i]
             FI;
        ELSE
             IF *merging-masking*                 ; merging-masking
                  THEN *DEST[i+31:i] remains unchanged*
                  ELSE                            ; zeroing-masking
                       DEST[i+31:i] <- 0
             FI
   FI;
ENDFOR;
DEST[MAX_VL-1:VL] <- 0





VADDPS (VEX.256 encoded version)
DEST[31:0] <- SRC1[31:0] + SRC2[31:0]
DEST[63:32] <- SRC1[63:32] + SRC2[63:32]
DEST[95:64] <- SRC1[95:64] + SRC2[95:64]
DEST[127:96] <- SRC1[127:96] + SRC2[127:96]
DEST[159:128] <- SRC1[159:128] + SRC2[159:128]
DEST[191:160]<- SRC1[191:160] + SRC2[191:160]
DEST[223:192] <- SRC1[223:192] + SRC2[223:192]
DEST[255:224] <- SRC1[255:224] + SRC2[255:224].
DEST[MAX_VL-1:256] <- 0

VADDPS (VEX.128 encoded version)
DEST[31:0] <- SRC1[31:0] + SRC2[31:0]
DEST[63:32] <- SRC1[63:32] + SRC2[63:32]
DEST[95:64] <- SRC1[95:64] + SRC2[95:64]
DEST[127:96] <- SRC1[127:96] + SRC2[127:96]
DEST[MAX_VL-1:128] <- 0

ADDPS (128-bit Legacy SSE version)
DEST[31:0] <- SRC1[31:0] + SRC2[31:0]
DEST[63:32] <- SRC1[63:32] + SRC2[63:32]
DEST[95:64] <- SRC1[95:64] + SRC2[95:64]
DEST[127:96] <- SRC1[127:96] + SRC2[127:96]
DEST[MAX_VL-1:128] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VADDPS __m512 _mm512_add_ps (__m512 a, __m512 b);
VADDPS __m512 _mm512_mask_add_ps (__m512 s, __mmask16 k, __m512 a, __m512 b);
VADDPS __m512 _mm512_maskz_add_ps (__mmask16 k, __m512 a, __m512 b);
VADDPS __m256 _mm256_mask_add_ps (__m256 s, __mmask8 k, __m256 a, __m256 b);
VADDPS __m256 _mm256_maskz_add_ps (__mmask8 k, __m256 a, __m256 b);
VADDPS __m128 _mm_mask_add_ps (__m128d s, __mmask8 k, __m128 a, __m128 b);
VADDPS __m128 _mm_maskz_add_ps (__mmask8 k, __m128 a, __m128 b);
VADDPS __m512 _mm512_add_round_ps (__m512 a, __m512 b, int);
VADDPS __m512 _mm512_mask_add_round_ps (__m512 s, __mmask16 k, __m512 a, __m512 b, int);
VADDPS __m512 _mm512_maskz_add_round_ps (__mmask16 k, __m512 a, __m512 b, int);
ADDPS __m256 _mm256_add_ps (__m256 a, __m256 b);
ADDPS __m128 _mm_add_ps (__m128 a, __m128 b);

SIMD Floating-Point Exceptions
Overflow, Underflow, Invalid, Precision, Denormal

Other Exceptions
VEX-encoded instruction, see Exceptions Type 2.
EVEX-encoded instruction, see Exceptions Type E2.




');
INSERT INTO "instructions" VALUES('x86_64','ADDSD','ADDSD-Add Scalar Double-Precision Floating-Point Values
 Opcode/                            Op /     64/32        CPUID         Description
 Instruction                        En       bit Mode     Feature
                                             Support      Flag
 F2 0F 58 /r                        RM       V/V          SSE2          Add the low double-precision floating-point value from
 ADDSD xmm1, xmm2/m64                                                   xmm2/mem to xmm1 and store the result in xmm1.
 VEX.NDS.128.F2.0F.WIG 58 /r        RVM      V/V          AVX           Add the low double-precision floating-point value from
 VADDSD xmm1, xmm2,                                                     xmm3/mem to xmm2 and store the result in xmm1.
 xmm3/m64
 EVEX.NDS.LIG.F2.0F.W1 58 /r        T1S      V/V          AVX512F       Add the low double-precision floating-point value from
 VADDSD xmm1 {k1}{z},                                                   xmm3/m64 to xmm2 and store the result in xmm1 with
 xmm2, xmm3/m64{er}                                                     writemask k1.



                                                    Instruction Operand Encoding
        Op/En                    Operand 1                      Operand 2                 Operand 3                   Operand 4
          RM                 ModRM:reg (r, w)               ModRM:r/m (r)                     NA                         NA
         RVM                  ModRM:reg (w)                     VEX.vvvv                ModRM:r/m (r)                    NA
      T1S-RVM                 ModRM:reg (w)                     EVEX.vvvv               ModRM:r/m (r)                    NA

Description
Adds the low double-precision floating-point values from the second source operand and the first source operand
and stores the double-precision floating-point result in the destination operand.
The second source operand can be an XMM register or a 64-bit memory location. The first source and destination
operands are XMM registers.
128-bit Legacy SSE version: The first source and destination operands are the same. Bits (MAX_VL-1:64) of the
corresponding destination register remain unchanged.
EVEX and VEX.128 encoded version: The first source operand is encoded by EVEX.vvvv/VEX.vvvv. Bits (127:64) of
the XMM register destination are copied from corresponding bits in the first source operand. Bits (MAX_VL-1:128)
of the destination register are zeroed.
EVEX version: The low quadword element of the destination is updated according to the writemask.
Software should ensure VADDSD is encoded with VEX.L=0. Encoding VADDSD with VEX.L=1 may encounter
unpredictable behavior across different processor generations.

Operation
VADDSD (EVEX encoded version)
IF (EVEX.b = 1) AND SRC2 *is a register*
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);
FI;
IF k1[0] or *no writemask*
    THEN      DEST[63:0] <- SRC1[63:0] + SRC2[63:0]
    ELSE
         IF *merging-masking*              ; merging-masking
              THEN *DEST[63:0] remains unchanged*
              ELSE                         ; zeroing-masking
                  THEN DEST[63:0] <- 0
         FI;
FI;
DEST[127:64] <- SRC1[127:64]



DEST[MAX_VL-1:128] <- 0

VADDSD (VEX.128 encoded version)
DEST[63:0] <-SRC1[63:0] + SRC2[63:0]
DEST[127:64] <-SRC1[127:64]
DEST[MAX_VL-1:128] <-0

ADDSD (128-bit Legacy SSE version)
DEST[63:0] <-DEST[63:0] + SRC[63:0]
DEST[MAX_VL-1:64] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VADDSD __m128d _mm_mask_add_sd (__m128d s, __mmask8 k, __m128d a, __m128d b);
VADDSD __m128d _mm_maskz_add_sd (__mmask8 k, __m128d a, __m128d b);
VADDSD __m128d _mm_add_round_sd (__m128d a, __m128d b, int);
VADDSD __m128d _mm_mask_add_round_sd (__m128d s, __mmask8 k, __m128d a, __m128d b, int);
VADDSD __m128d _mm_maskz_add_round_sd (__mmask8 k, __m128d a, __m128d b, int);
ADDSD __m128d _mm_add_sd (__m128d a, __m128d b);

SIMD Floating-Point Exceptions
Overflow, Underflow, Invalid, Precision, Denormal

Other Exceptions
VEX-encoded instruction, see Exceptions Type 3.
EVEX-encoded instruction, see Exceptions Type E3.




');
INSERT INTO "instructions" VALUES('x86_64','ADDSS','ADDSS-Add Scalar Single-Precision Floating-Point Values
 Opcode/                             Op /    64/32        CPUID           Description
 Instruction                         En      bit Mode     Feature
                                             Support      Flag
 F3 0F 58 /r                         RM      V/V          SSE             Add the low single-precision floating-point value from
 ADDSS xmm1, xmm2/m32                                                     xmm2/mem to xmm1 and store the result in xmm1.
 VEX.NDS.128.F3.0F.WIG 58 /r         RVM     V/V          AVX             Add the low single-precision floating-point value from
 VADDSS xmm1,xmm2,                                                        xmm3/mem to xmm2 and store the result in xmm1.
 xmm3/m32
 EVEX.NDS.LIG.F3.0F.W0 58 /r         T1S     V/V          AVX512F         Add the low single-precision floating-point value from
 VADDSS xmm1{k1}{z}, xmm2,                                                xmm3/m32 to xmm2 and store the result in xmm1with
 xmm3/m32{er}                                                             writemask k1.



                                                     Instruction Operand Encoding
       Op/En                  Operand 1                    Operand 2                      Operand 3                     Operand 4
        RM                ModRM:reg (r, w)                ModRM:r/m (r)                       NA                            NA
        RVM                ModRM:reg (w)                    VEX.vvvv                     ModRM:r/m (r)                      NA
        T1S                ModRM:reg (w)                   EVEX.vvvv                     ModRM:r/m (r)                      NA

Description
Adds the low single-precision floating-point values from the second source operand and the first source operand,
and stores the double-precision floating-point result in the destination operand.
The second source operand can be an XMM register or a 64-bit memory location. The first source and destination
operands are XMM registers.
128-bit Legacy SSE version: The first source and destination operands are the same. Bits (MAX_VL-1:32) of the
corresponding the destination register remain unchanged.
EVEX and VEX.128 encoded version: The first source operand is encoded by EVEX.vvvv/VEX.vvvv. Bits (127:32) of
the XMM register destination are copied from corresponding bits in the first source operand. Bits (MAX_VL-1:128)
of the destination register are zeroed.
EVEX version: The low doubleword element of the destination is updated according to the writemask.
Software should ensure VADDSS is encoded with VEX.L=0. Encoding VADDSS with VEX.L=1 may encounter unpre-
dictable behavior across different processor generations.

Operation
VADDSS (EVEX encoded versions)
IF (EVEX.b = 1) AND SRC2 *is a register*
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);
FI;
IF k1[0] or *no writemask*
    THEN      DEST[31:0] <- SRC1[31:0] + SRC2[31:0]
    ELSE
         IF *merging-masking*              ; merging-masking
              THEN *DEST[31:0] remains unchanged*
              ELSE                         ; zeroing-masking
                  THEN DEST[31:0] <- 0
         FI;
FI;
DEST[127:32] <- SRC1[127:32]



DEST[MAX_VL-1:128] <- 0

VADDSS DEST, SRC1, SRC2 (VEX.128 encoded version)
DEST[31:0] <-SRC1[31:0] + SRC2[31:0]
DEST[127:32] <-SRC1[127:32]
DEST[MAX_VL-1:128] <-0

ADDSS DEST, SRC (128-bit Legacy SSE version)
DEST[31:0] <-DEST[31:0] + SRC[31:0]
DEST[MAX_VL-1:32] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VADDSS __m128 _mm_mask_add_ss (__m128 s, __mmask8 k, __m128 a, __m128 b);
VADDSS __m128 _mm_maskz_add_ss (__mmask8 k, __m128 a, __m128 b);
VADDSS __m128 _mm_add_round_ss (__m128 a, __m128 b, int);
VADDSS __m128 _mm_mask_add_round_ss (__m128 s, __mmask8 k, __m128 a, __m128 b, int);
VADDSS __m128 _mm_maskz_add_round_ss (__mmask8 k, __m128 a, __m128 b, int);
ADDSS __m128 _mm_add_ss (__m128 a, __m128 b);

SIMD Floating-Point Exceptions
Overflow, Underflow, Invalid, Precision, Denormal

Other Exceptions
VEX-encoded instruction, see Exceptions Type 3.
EVEX-encoded instruction, see Exceptions Type E3.




');
INSERT INTO "instructions" VALUES('x86_64','ADDSUBPD','ADDSUBPD-Packed Double-FP Add/Subtract
Opcode/                                         Op/       64/32-bit CPUID        Description
Instruction                                     En        Mode      Feature
                                                                    Flag
66 0F D0 /r                                     RM        V/V       SSE3         Add/subtract double-precision floating-point
ADDSUBPD xmm1, xmm2/m128                                                         values from xmm2/m128 to xmm1.

VEX.NDS.128.66.0F.WIG D0 /r                     RVM V/V             AVX          Add/subtract packed double-precision
VADDSUBPD xmm1, xmm2, xmm3/m128                                                  floating-point values from xmm3/mem to
                                                                                 xmm2 and stores result in xmm1.
VEX.NDS.256.66.0F.WIG D0 /r                     RVM V/V             AVX          Add / subtract packed double-precision
VADDSUBPD ymm1, ymm2, ymm3/m256                                                  floating-point values from ymm3/mem to
                                                                                 ymm2 and stores result in ymm1.



                                          Instruction Operand Encoding
  Op/En             Operand 1               Operand 2                      Operand 3                      Operand 4
    RM           ModRM:reg (r, w)         ModRM:r/m (r)                        NA                             NA
   RVM            ModRM:reg (w)            VEX.vvvv (r)                   ModRM:r/m (r)                       NA

Description
Adds odd-numbered double-precision floating-point values of the first source operand (second operand) with the
corresponding double-precision floating-point values from the second source operand (third operand); stores the
result in the odd-numbered values of the destination operand (first operand). Subtracts the even-numbered
double-precision floating-point values from the second source operand from the corresponding double-precision
floating values in the first source operand; stores the result into the even-numbered values of the destination
operand.
In 64-bit mode, using a REX prefix in the form of REX.R permits this instruction to access additional registers
(XMM8-XMM15).
128-bit Legacy SSE version: The second source can be an XMM register or an 128-bit memory location. The desti-
nation is not distinct from the first source XMM register and the upper bits (VLMAX-1:128) of the corresponding
YMM register destination are unmodified. See Figure 3-3.
VEX.128 encoded version: the first source operand is an XMM register or 128-bit memory location. The destination
operand is an XMM register. The upper bits (VLMAX-1:128) of the corresponding YMM register destination are
zeroed.
VEX.256 encoded version: The first source operand is a YMM register. The second source operand can be a YMM
register or a 256-bit memory location. The destination operand is a YMM register.





                                             ADDSUBPD xmm1, xmm2/m128


                                     [127:64]                           [63:0]              xmm2/m128




                                                                                            RESULT:
                         xmm1[127:64] + xmm2/m128[127:64]   xmm1[63:0] - xmm2/m128[63:0]
                                                                                            xmm1

                                     [127:64]                           [63:0]



                              Figure 3-3. ADDSUBPD-Packed Double-FP Add/Subtract


Operation
ADDSUBPD (128-bit Legacy SSE version)
DEST[63:0] <- DEST[63:0] - SRC[63:0]
DEST[127:64] <- DEST[127:64] + SRC[127:64]
DEST[VLMAX-1:128] (Unmodified)

VADDSUBPD (VEX.128 encoded version)
DEST[63:0] <- SRC1[63:0] - SRC2[63:0]
DEST[127:64] <- SRC1[127:64] + SRC2[127:64]
DEST[VLMAX-1:128] <- 0

VADDSUBPD (VEX.256 encoded version)
DEST[63:0] <- SRC1[63:0] - SRC2[63:0]
DEST[127:64] <- SRC1[127:64] + SRC2[127:64]
DEST[191:128] <- SRC1[191:128] - SRC2[191:128]
DEST[255:192] <- SRC1[255:192] + SRC2[255:192]

Intel C/C++ Compiler Intrinsic Equivalent
ADDSUBPD:      __m128d _mm_addsub_pd(__m128d a, __m128d b)

VADDSUBPD:     __m256d _mm256_addsub_pd (__m256d a, __m256d b)

Exceptions
When the source operand is a memory operand, it must be aligned on a 16-byte boundary or a general-protection
exception (#GP) will be generated.

SIMD Floating-Point Exceptions
Overflow, Underflow, Invalid, Precision, Denormal.

Other Exceptions
See Exceptions Type 2.




');
INSERT INTO "instructions" VALUES('x86_64','ADDSUBPS','ADDSUBPS-Packed Single-FP Add/Subtract
 Opcode/                                        Op/   64/32-bit CPUID        Description
 Instruction                                    En    Mode      Feature
                                                                Flag
 F2 0F D0 /r                                    RM    V/V       SSE3         Add/subtract single-precision floating-point
 ADDSUBPS xmm1, xmm2/m128                                                    values from xmm2/m128 to xmm1.

 VEX.NDS.128.F2.0F.WIG D0 /r                    RVM V/V         AVX          Add/subtract single-precision floating-point
 VADDSUBPS xmm1, xmm2, xmm3/m128                                             values from xmm3/mem to xmm2 and stores
                                                                             result in xmm1.
 VEX.NDS.256.F2.0F.WIG D0 /r                    RVM V/V         AVX          Add / subtract single-precision floating-point
 VADDSUBPS ymm1, ymm2, ymm3/m256                                             values from ymm3/mem to ymm2 and stores
                                                                             result in ymm1.



                                          Instruction Operand Encoding
  Op/En              Operand 1               Operand 2                  Operand 3                      Operand 4
    RM            ModRM:reg (r, w)         ModRM:r/m (r)                    NA                             NA
   RVM             ModRM:reg (w)            VEX.vvvv (r)               ModRM:r/m (r)                       NA

Description
Adds odd-numbered single-precision floating-point values of the first source operand (second operand) with the
corresponding single-precision floating-point values from the second source operand (third operand); stores the
result in the odd-numbered values of the destination operand (first operand). Subtracts the even-numbered
single-precision floating-point values from the second source operand from the corresponding single-precision
floating values in the first source operand; stores the result into the even-numbered values of the destination
operand.
In 64-bit mode, using a REX prefix in the form of REX.R permits this instruction to access additional registers
(XMM8-XMM15).
128-bit Legacy SSE version: The second source can be an XMM register or an 128-bit memory location. The desti-
nation is not distinct from the first source XMM register and the upper bits (VLMAX-1:128) of the corresponding
YMM register destination are unmodified. See Figure 3-4.
VEX.128 encoded version: the first source operand is an XMM register or 128-bit memory location. The destination
operand is an XMM register. The upper bits (VLMAX-1:128) of the corresponding YMM register destination are
zeroed.
VEX.256 encoded version: The first source operand is a YMM register. The second source operand can be a YMM
register or a 256-bit memory location. The destination operand is a YMM register.





                                           ADDSUBPS xmm1, xmm2/m128

                                                                                                    xmm2/
                        [127:96]             [95:64]              [63:32]            [31:0]
                                                                                                    m128




                      xmm1[127:96] +    xmm1[95:64] - xmm2/     xmm1[63:32] +      xmm1[31:0] -     RESULT:
                    xmm2/m128[127:96]      m128[95:64]        xmm2/m128[63:32]   xmm2/m128[31:0]    xmm1

                        [127:96]             [95:64]              [63:32]            [31:0]


                                                                                                        OM15992



                               Figure 3-4. ADDSUBPS-Packed Single-FP Add/Subtract


Operation
ADDSUBPS (128-bit Legacy SSE version)
DEST[31:0] <- DEST[31:0] - SRC[31:0]
DEST[63:32] <- DEST[63:32] + SRC[63:32]
DEST[95:64] <- DEST[95:64] - SRC[95:64]
DEST[127:96] <- DEST[127:96] + SRC[127:96]
DEST[VLMAX-1:128] (Unmodified)

VADDSUBPS (VEX.128 encoded version)
DEST[31:0] <- SRC1[31:0] - SRC2[31:0]
DEST[63:32] <- SRC1[63:32] + SRC2[63:32]
DEST[95:64] <- SRC1[95:64] - SRC2[95:64]
DEST[127:96] <- SRC1[127:96] + SRC2[127:96]
DEST[VLMAX-1:128] <- 0

VADDSUBPS (VEX.256 encoded version)
DEST[31:0] <- SRC1[31:0] - SRC2[31:0]
DEST[63:32] <- SRC1[63:32] + SRC2[63:32]
DEST[95:64] <- SRC1[95:64] - SRC2[95:64]
DEST[127:96] <- SRC1[127:96] + SRC2[127:96]
DEST[159:128] <- SRC1[159:128] - SRC2[159:128]
DEST[191:160]<- SRC1[191:160] + SRC2[191:160]
DEST[223:192] <- SRC1[223:192] - SRC2[223:192]
DEST[255:224] <- SRC1[255:224] + SRC2[255:224].

Intel C/C++ Compiler Intrinsic Equivalent
ADDSUBPS:      __m128 _mm_addsub_ps(__m128 a, __m128 b)

VADDSUBPS:     __m256 _mm256_addsub_ps (__m256 a, __m256 b)

Exceptions
When the source operand is a memory operand, the operand must be aligned on a 16-byte boundary or a general-
protection exception (#GP) will be generated.





SIMD Floating-Point Exceptions
Overflow, Underflow, Invalid, Precision, Denormal.

Other Exceptions
See Exceptions Type 2.




');
INSERT INTO "instructions" VALUES('x86_64','ADOX','ADOX - Unsigned Integer Addition of Two Operands with Overflow Flag
 Opcode/                           Op/     64/32bit    CPUID       Description
 Instruction                       En      Mode        Feature
                                           Support     Flag
 F3 0F 38 F6 /r                    RM      V/V         ADX         Unsigned addition of r32 with OF, r/m32 to r32, writes OF.
 ADOX r32, r/m32
 F3 REX.w 0F 38 F6 /r              RM      V/NE        ADX         Unsigned addition of r64 with OF, r/m64 to r64, writes OF.
 ADOX r64, r/m64



                                                  Instruction Operand Encoding
       Op/En               Operand 1                   Operand 2                     Operand 3                     Operand 4
        RM              ModRM:reg (r, w)              ModRM:r/m (r)                      NA                            NA

Description
Performs an unsigned addition of the destination operand (first operand), the source operand (second operand)
and the overflow-flag (OF) and stores the result in the destination operand. The destination operand is a general-
purpose register, whereas the source operand can be a general-purpose register or memory location. The state of
OF represents a carry from a previous addition. The instruction sets the OF flag with the carry generated by the
unsigned addition of the operands.
The ADOX instruction is executed in the context of multi-precision addition, where we add a series of operands with
a carry-chain. At the beginning of a chain of additions, we execute an instruction to zero the OF (e.g. XOR).
This instruction is supported in real mode and virtual-8086 mode. The operand size is always 32 bits if not in 64-bit
mode.
In 64-bit mode, the default operation size is 32 bits. Using a REX Prefix in the form of REX.R permits access to addi-
tional registers (R8-15). Using REX Prefix in the form of REX.W promotes operation to 64-bits.
ADOX executes normally either inside or outside a transaction region.
Note: ADOX defines the CF and OF flags differently than the ADD/ADC instructions as defined in Intel 64 and
IA-32 Architectures Software Developer''s Manual, Volume 2A.

Operation
IF OperandSize is 64-bit
    THEN OF:DEST[63:0] <- DEST[63:0] + SRC[63:0] + OF;
    ELSE OF:DEST[31:0] <- DEST[31:0] + SRC[31:0] + OF;
FI;

Flags Affected
OF is updated based on result. CF, SF, ZF, AF and PF flags are unmodified.

Intel C/C++ Compiler Intrinsic Equivalent
unsigned char _addcarryx_u32 (unsigned char c_in, unsigned int src1, unsigned int src2, unsigned int *sum_out);
unsigned char _addcarryx_u64 (unsigned char c_in, unsigned __int64 src1, unsigned __int64 src2, unsigned __int64 *sum_out);

SIMD Floating-Point Exceptions
None





Protected Mode Exceptions
#UD                      If the LOCK prefix is used.
                         If CPUID.(EAX=07H, ECX=0H):EBX.ADX[bit 19] = 0.
#SS(0)                   For an illegal address in the SS segment.
#GP(0)                   For an illegal memory operand effective address in the CS, DS, ES, FS or GS segments.
                         If the DS, ES, FS, or GS register is used to access memory and it contains a null segment
                         selector.
#PF(fault-code)          For a page fault.
#AC(0)                   If alignment checking is enabled and an unaligned memory reference is made while the
                         current privilege level is 3.

Real-Address Mode Exceptions
#UD                      If the LOCK prefix is used.
                         If CPUID.(EAX=07H, ECX=0H):EBX.ADX[bit 19] = 0.
#SS(0)                   For an illegal address in the SS segment.
#GP(0)                   If any part of the operand lies outside the effective address space from 0 to FFFFH.

Virtual-8086 Mode Exceptions
#UD                      If the LOCK prefix is used.
                         If CPUID.(EAX=07H, ECX=0H):EBX.ADX[bit 19] = 0.
#SS(0)                   For an illegal address in the SS segment.
#GP(0)                   If any part of the operand lies outside the effective address space from 0 to FFFFH.
#PF(fault-code)          For a page fault.
#AC(0)                   If alignment checking is enabled and an unaligned memory reference is made while the
                         current privilege level is 3.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#UD                      If the LOCK prefix is used.
                         If CPUID.(EAX=07H, ECX=0H):EBX.ADX[bit 19] = 0.
#SS(0)                   If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)                   If the memory address is in a non-canonical form.
#PF(fault-code)          For a page fault.
#AC(0)                   If alignment checking is enabled and an unaligned memory reference is made while the
                         current privilege level is 3.




');
INSERT INTO "instructions" VALUES('x86_64','AESDEC','AESDEC-Perform One Round of an AES Decryption Flow
Opcode/                                         Op/      64/32-bit CPUID        Description
Instruction                                     En       Mode      Feature
                                                                   Flag
66 0F 38 DE /r                                  RM       V/V       AES          Perform one round of an AES decryption flow,
AESDEC xmm1, xmm2/m128                                                          using the Equivalent Inverse Cipher, operating
                                                                                on a 128-bit data (state) from xmm1 with a
                                                                                128-bit round key from xmm2/m128.
VEX.NDS.128.66.0F38.WIG DE /r                   RVM V/V            Both AES     Perform one round of an AES decryption flow,
VAESDEC xmm1, xmm2, xmm3/m128                                      and          using the Equivalent Inverse Cipher, operating
                                                                   AVX flags    on a 128-bit data (state) from xmm2 with a
                                                                                128-bit round key from xmm3/m128; store
                                                                                the result in xmm1.



                                          Instruction Operand Encoding
  Op/En           Operand 1               Operand2                        Operand3                       Operand4
    RM         ModRM:reg (r, w)          ModRM:r/m (r)                        NA                             NA
   RVM         ModRM:reg (w)             VEX.vvvv (r)                    ModRM:r/m (r)                       NA

Description
This instruction performs a single round of the AES decryption flow using the Equivalent Inverse Cipher, with the
round key from the second source operand, operating on a 128-bit data (state) from the first source operand, and
store the result in the destination operand.
Use the AESDEC instruction for all but the last decryption round. For the last decryption round, use the AESDE-
CLAST instruction.
128-bit Legacy SSE version: The first source operand and the destination operand are the same and must be an
XMM register. The second source operand can be an XMM register or a 128-bit memory location. Bits (VLMAX-
1:128) of the corresponding YMM destination register remain unchanged.
VEX.128 encoded version: The first source operand and the destination operand are XMM registers. The second
source operand can be an XMM register or a 128-bit memory location. Bits (VLMAX-1:128) of the destination YMM
register are zeroed.

Operation
AESDEC
STATE <- SRC1;
RoundKey <- SRC2;
STATE <- InvShiftRows( STATE );
STATE <- InvSubBytes( STATE );
STATE <- InvMixColumns( STATE );
DEST[127:0] <- STATE XOR RoundKey;
DEST[VLMAX-1:128] (Unmodified)

VAESDEC
STATE <- SRC1;
RoundKey <- SRC2;
STATE <- InvShiftRows( STATE );
STATE <- InvSubBytes( STATE );
STATE <- InvMixColumns( STATE );
DEST[127:0] <- STATE XOR RoundKey;
DEST[VLMAX-1:128] <- 0





Intel C/C++ Compiler Intrinsic Equivalent
(V)AESDEC:      __m128i _mm_aesdec (__m128i, __m128i)

SIMD Floating-Point Exceptions
None

Other Exceptions
See Exceptions Type 4.




');
INSERT INTO "instructions" VALUES('x86_64','AESDECLAST','AESDECLAST-Perform Last Round of an AES Decryption Flow
Opcode/                                          Op/     64/32-bit CPUID         Description
Instruction                                      En      Mode      Feature
                                                                   Flag
66 0F 38 DF /r                                   RM      V/V       AES           Perform the last round of an AES decryption
AESDECLAST xmm1, xmm2/m128                                                       flow, using the Equivalent Inverse Cipher,
                                                                                 operating on a 128-bit data (state) from
                                                                                 xmm1 with a 128-bit round key from
                                                                                 xmm2/m128.
VEX.NDS.128.66.0F38.WIG DF /r                    RVM V/V           Both AES      Perform the last round of an AES decryption
VAESDECLAST xmm1, xmm2, xmm3/m128                                  and           flow, using the Equivalent Inverse Cipher,
                                                                   AVX flags     operating on a 128-bit data (state) from
                                                                                 xmm2 with a 128-bit round key from
                                                                                 xmm3/m128; store the result in xmm1.



                                            Instruction Operand Encoding
  Op/En             Operand 1                 Operand2                     Operand3                       Operand4
    RM           ModRM:reg (r, w)           ModRM:r/m (r)                      NA                             NA
   RVM           ModRM:reg (w)              VEX.vvvv (r)                 ModRM:r/m (r)                        NA

Description
This instruction performs the last round of the AES decryption flow using the Equivalent Inverse Cipher, with the
round key from the second source operand, operating on a 128-bit data (state) from the first source operand, and
store the result in the destination operand.
128-bit Legacy SSE version: The first source operand and the destination operand are the same and must be an
XMM register. The second source operand can be an XMM register or a 128-bit memory location. Bits (VLMAX-
1:128) of the corresponding YMM destination register remain unchanged.
VEX.128 encoded version: The first source operand and the destination operand are XMM registers. The second
source operand can be an XMM register or a 128-bit memory location. Bits (VLMAX-1:128) of the destination YMM
register are zeroed.

Operation
AESDECLAST
STATE <- SRC1;
RoundKey <- SRC2;
STATE <- InvShiftRows( STATE );
STATE <- InvSubBytes( STATE );
DEST[127:0] <- STATE XOR RoundKey;
DEST[VLMAX-1:128] (Unmodified)

VAESDECLAST
STATE <- SRC1;
RoundKey <- SRC2;
STATE <- InvShiftRows( STATE );
STATE <- InvSubBytes( STATE );
DEST[127:0] <- STATE XOR RoundKey;
DEST[VLMAX-1:128] <- 0

Intel C/C++ Compiler Intrinsic Equivalent
(V)AESDECLAST:     __m128i _mm_aesdeclast (__m128i, __m128i)




SIMD Floating-Point Exceptions
None

Other Exceptions
See Exceptions Type 4.




');
INSERT INTO "instructions" VALUES('x86_64','AESENC','AESENC-Perform One Round of an AES Encryption Flow
Opcode/                                         Op/    64/32-bit CPUID        Description
Instruction                                     En     Mode      Feature
                                                                 Flag
66 0F 38 DC /r                                  RM     V/V       AES          Perform one round of an AES encryption flow,
AESENC xmm1, xmm2/m128                                                        operating on a 128-bit data (state) from
                                                                              xmm1 with a 128-bit round key from
                                                                              xmm2/m128.
VEX.NDS.128.66.0F38.WIG DC /r                   RVM V/V          Both AES     Perform one round of an AES encryption flow,
VAESENC xmm1, xmm2, xmm3/m128                                    and          operating on a 128-bit data (state) from
                                                                 AVX flags    xmm2 with a 128-bit round key from the
                                                                              xmm3/m128; store the result in xmm1.



                                          Instruction Operand Encoding
  Op/En           Operand 1                 Operand2                    Operand3                        Operand4
    RM         ModRM:reg (r, w)            ModRM:r/m (r)                     NA                            NA
   RVM          ModRM:reg (w)              VEX.vvvv (r)                ModRM:r/m (r)                       NA

Description
This instruction performs a single round of an AES encryption flow using a round key from the second source
operand, operating on 128-bit data (state) from the first source operand, and store the result in the destination
operand.
Use the AESENC instruction for all but the last encryption rounds. For the last encryption round, use the AESENC-
CLAST instruction.
128-bit Legacy SSE version: The first source operand and the destination operand are the same and must be an
XMM register. The second source operand can be an XMM register or a 128-bit memory location. Bits (VLMAX-
1:128) of the corresponding YMM destination register remain unchanged.
VEX.128 encoded version: The first source operand and the destination operand are XMM registers. The second
source operand can be an XMM register or a 128-bit memory location. Bits (VLMAX-1:128) of the destination YMM
register are zeroed.

Operation
AESENC
STATE <- SRC1;
RoundKey <- SRC2;
STATE <- ShiftRows( STATE );
STATE <- SubBytes( STATE );
STATE <- MixColumns( STATE );
DEST[127:0] <- STATE XOR RoundKey;
DEST[VLMAX-1:128] (Unmodified)

VAESENC
STATE <- SRC1;
RoundKey <- SRC2;
STATE <- ShiftRows( STATE );
STATE <- SubBytes( STATE );
STATE <- MixColumns( STATE );
DEST[127:0] <- STATE XOR RoundKey;
DEST[VLMAX-1:128] <- 0





Intel C/C++ Compiler Intrinsic Equivalent
(V)AESENC:      __m128i _mm_aesenc (__m128i, __m128i)

SIMD Floating-Point Exceptions
None

Other Exceptions
See Exceptions Type 4.




');
INSERT INTO "instructions" VALUES('x86_64','AESENCLAST','AESENCLAST-Perform Last Round of an AES Encryption Flow
Opcode/                                           Op/     64/32-bit CPUID        Description
Instruction                                       En      Mode      Feature
                                                                    Flag
66 0F 38 DD /r                                    RM      V/V       AES          Perform the last round of an AES encryption
AESENCLAST xmm1, xmm2/m128                                                       flow, operating on a 128-bit data (state) from
                                                                                 xmm1 with a 128-bit round key from
                                                                                 xmm2/m128.
VEX.NDS.128.66.0F38.WIG DD /r                     RVM V/V           Both AES     Perform the last round of an AES encryption
VAESENCLAST xmm1, xmm2, xmm3/m128                                   and          flow, operating on a 128-bit data (state) from
                                                                    AVX flags    xmm2 with a 128 bit round key from
                                                                                 xmm3/m128; store the result in xmm1.



                                            Instruction Operand Encoding
  Op/En             Operand 1                  Operand2                       Operand3                     Operand4
    RM           ModRM:reg (r, w)            ModRM:r/m (r)                      NA                             NA
   RVM           ModRM:reg (w)               VEX.vvvv (r)                 ModRM:r/m (r)                        NA

Description
This instruction performs the last round of an AES encryption flow using a round key from the second source
operand, operating on 128-bit data (state) from the first source operand, and store the result in the destination
operand.
128-bit Legacy SSE version: The first source operand and the destination operand are the same and must be an
XMM register. The second source operand can be an XMM register or a 128-bit memory location. Bits (VLMAX-
1:128) of the corresponding YMM destination register remain unchanged.
VEX.128 encoded version: The first source operand and the destination operand are XMM registers. The second
source operand can be an XMM register or a 128-bit memory location. Bits (VLMAX-1:128) of the destination YMM
register are zeroed.

Operation
AESENCLAST
STATE <- SRC1;
RoundKey <- SRC2;
STATE <- ShiftRows( STATE );
STATE <- SubBytes( STATE );
DEST[127:0] <- STATE XOR RoundKey;
DEST[VLMAX-1:128] (Unmodified)

VAESENCLAST
STATE <- SRC1;
RoundKey <- SRC2;
STATE <- ShiftRows( STATE );
STATE <- SubBytes( STATE );
DEST[127:0] <- STATE XOR RoundKey;
DEST[VLMAX-1:128] <- 0

Intel C/C++ Compiler Intrinsic Equivalent
(V)AESENCLAST:     __m128i _mm_aesenclast (__m128i, __m128i)





SIMD Floating-Point Exceptions
None

Other Exceptions
See Exceptions Type 4.




');
INSERT INTO "instructions" VALUES('x86_64','AESIMC','AESIMC-Perform the AES InvMixColumn Transformation
Opcode/                                            Op/   64/32-bit CPUID        Description
Instruction                                        En    Mode      Feature
                                                                   Flag
66 0F 38 DB /r                                     RM    V/V       AES          Perform the InvMixColumn transformation on
AESIMC xmm1, xmm2/m128                                                          a 128-bit round key from xmm2/m128 and
                                                                                store the result in xmm1.
VEX.128.66.0F38.WIG DB /r                          RM    V/V       Both AES     Perform the InvMixColumn transformation on
VAESIMC xmm1, xmm2/m128                                            and          a 128-bit round key from xmm2/m128 and
                                                                   AVX flags    store the result in xmm1.



                                              Instruction Operand Encoding
  Op/En             Operand 1                   Operand2                     Operand3                     Operand4
    RM            ModRM:reg (w)               ModRM:r/m (r)                     NA                            NA

Description
Perform the InvMixColumns transformation on the source operand and store the result in the destination operand.
The destination operand is an XMM register. The source operand can be an XMM register or a 128-bit memory loca-
tion.
Note: the AESIMC instruction should be applied to the expanded AES round keys (except for the first and last round
key) in order to prepare them for decryption using the "Equivalent Inverse Cipher" (defined in FIPS 197).
128-bit Legacy SSE version: Bits (VLMAX-1:128) of the corresponding YMM destination register remain
unchanged.
VEX.128 encoded version: Bits (VLMAX-1:128) of the destination YMM register are zeroed.
Note: In VEX-encoded versions, VEX.vvvv is reserved and must be 1111b, otherwise instructions will #UD.

Operation
AESIMC
DEST[127:0] <- InvMixColumns( SRC );
DEST[VLMAX-1:128] (Unmodified)

VAESIMC
DEST[127:0] <- InvMixColumns( SRC );
DEST[VLMAX-1:128] <- 0;

Intel C/C++ Compiler Intrinsic Equivalent
(V)AESIMC:     __m128i _mm_aesimc (__m128i)

SIMD Floating-Point Exceptions
None

Other Exceptions
See Exceptions Type 4; additionally
#UD                   If VEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','AESKEYGENASSIST','AESKEYGENASSIST-AES Round Key Generation Assist
 Opcode/                                               Op/     64/32-bit CPUID        Description
 Instruction                                           En      Mode      Feature
                                                                         Flag
 66 0F 3A DF /r ib                                     RMI     V/V       AES          Assist in AES round key generation using an 8
 AESKEYGENASSIST xmm1, xmm2/m128, imm8                                                bits Round Constant (RCON) specified in the
                                                                                      immediate byte, operating on 128 bits of data
                                                                                      specified in xmm2/m128 and stores the
                                                                                      result in xmm1.
 VEX.128.66.0F3A.WIG DF /r ib                          RMI     V/V       Both AES     Assist in AES round key generation using 8
 VAESKEYGENASSIST xmm1, xmm2/m128, imm8                                  and          bits Round Constant (RCON) specified in the
                                                                         AVX flags    immediate byte, operating on 128 bits of data
                                                                                      specified in xmm2/m128 and stores the
                                                                                      result in xmm1.



                                                  Instruction Operand Encoding
   Op/En            Operand 1                       Operand2                    Operand3                        Operand4
    RMI           ModRM:reg (w)                   ModRM:r/m (r)                      imm8                          NA

Description
Assist in expanding the AES cipher key, by computing steps towards generating a round key for encryption, using
128-bit data specified in the source operand and an 8-bit round constant specified as an immediate, store the
result in the destination operand.
The destination operand is an XMM register. The source operand can be an XMM register or a 128-bit memory loca-
tion.
128-bit Legacy SSE version: Bits (VLMAX-1:128) of the corresponding YMM destination register remain
unchanged.
VEX.128 encoded version: Bits (VLMAX-1:128) of the destination YMM register are zeroed.
Note: In VEX-encoded versions, VEX.vvvv is reserved and must be 1111b, otherwise instructions will #UD.

Operation
AESKEYGENASSIST
X3[31:0] <- SRC [127: 96];
X2[31:0] <- SRC [95: 64];
X1[31:0] <- SRC [63: 32];
X0[31:0] <- SRC [31: 0];
RCON[31:0] <- ZeroExtend(Imm8[7:0]);
DEST[31:0] <- SubWord(X1);
DEST[63:32 ] <- RotWord( SubWord(X1) ) XOR RCON;
DEST[95:64] <- SubWord(X3);
DEST[127:96] <- RotWord( SubWord(X3) ) XOR RCON;
DEST[VLMAX-1:128] (Unmodified)





VAESKEYGENASSIST
X3[31:0] <- SRC [127: 96];
X2[31:0] <- SRC [95: 64];
X1[31:0] <- SRC [63: 32];
X0[31:0] <- SRC [31: 0];
RCON[31:0] <- ZeroExtend(Imm8[7:0]);
DEST[31:0] <- SubWord(X1);
DEST[63:32 ] <- RotWord( SubWord(X1) ) XOR RCON;
DEST[95:64] <- SubWord(X3);
DEST[127:96] <- RotWord( SubWord(X3) ) XOR RCON;
DEST[VLMAX-1:128] <- 0;

Intel C/C++ Compiler Intrinsic Equivalent
(V)AESKEYGENASSIST:    __m128i _mm_aeskeygenassist (__m128i, const int)

SIMD Floating-Point Exceptions
None

Other Exceptions
See Exceptions Type 4; additionally
#UD                   If VEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','AND','AND-Logical AND
 Opcode                    Instruction                  Op/    64-bit       Compat/ Description
                                                        En     Mode         Leg Mode
 24 ib                     AND AL, imm8                 I      Valid        Valid        AL AND imm8.
 25 iw                     AND AX, imm16                I      Valid        Valid        AX AND imm16.
 25 id                     AND EAX, imm32               I      Valid        Valid        EAX AND imm32.
 REX.W + 25 id             AND RAX, imm32               I      Valid        N.E.         RAX AND imm32 sign-extended to 64-bits.
 80 /4 ib                  AND r/m8, imm8               MI     Valid        Valid        r/m8 AND imm8.
 REX + 80 /4 ib            AND r/m8*, imm8              MI     Valid        N.E.         r/m8 AND imm8.
 81 /4 iw                  AND r/m16, imm16             MI     Valid        Valid        r/m16 AND imm16.
 81 /4 id                  AND r/m32, imm32             MI     Valid        Valid        r/m32 AND imm32.
 REX.W + 81 /4 id          AND r/m64, imm32             MI     Valid        N.E.         r/m64 AND imm32 sign extended to 64-bits.
 83 /4 ib                  AND r/m16, imm8              MI     Valid        Valid        r/m16 AND imm8 (sign-extended).
 83 /4 ib                  AND r/m32, imm8              MI     Valid        Valid        r/m32 AND imm8 (sign-extended).
 REX.W + 83 /4 ib          AND r/m64, imm8              MI     Valid        N.E.         r/m64 AND imm8 (sign-extended).
 20 /r                     AND r/m8, r8                 MR     Valid        Valid        r/m8 AND r8.
 REX + 20 /r               AND r/m8*, r8*               MR     Valid        N.E.         r/m64 AND r8 (sign-extended).
 21 /r                     AND r/m16, r16               MR     Valid        Valid        r/m16 AND r16.
 21 /r                     AND r/m32, r32               MR     Valid        Valid        r/m32 AND r32.
 REX.W + 21 /r             AND r/m64, r64               MR     Valid        N.E.         r/m64 AND r32.
 22 /r                     AND r8, r/m8                 RM     Valid        Valid        r8 AND r/m8.
 REX + 22 /r               AND r8*, r/m8*               RM     Valid        N.E.         r/m64 AND r8 (sign-extended).
 23 /r                     AND r16, r/m16               RM     Valid        Valid        r16 AND r/m16.
 23 /r                     AND r32, r/m32               RM     Valid        Valid        r32 AND r/m32.
 REX.W + 23 /r             AND r64, r/m64               RM     Valid        N.E.         r64 AND r/m64.
 NOTES:
 *In 64-bit mode, r/m8 can not be encoded to access the following byte registers if a REX prefix is used: AH, BH, CH, DH.



                                                  Instruction Operand Encoding
  Op/En              Operand 1                     Operand 2                        Operand 3                     Operand 4
    RM            ModRM:reg (r, w)               ModRM:r/m (r)                         NA                             NA
    MR            ModRM:r/m (r, w)               ModRM:reg (r)                         NA                             NA
    MI            ModRM:r/m (r, w)                    imm8                             NA                             NA
     I            AL/AX/EAX/RAX                       imm8                             NA                             NA

Description
Performs a bitwise AND operation on the destination (first) and source (second) operands and stores the result in
the destination operand location. The source operand can be an immediate, a register, or a memory location; the
destination operand can be a register or a memory location. (However, two memory operands cannot be used in
one instruction.) Each bit of the result is set to 1 if both corresponding bits of the first and second operands are 1;
otherwise, it is set to 0.
This instruction can be used with a LOCK prefix to allow the it to be executed atomically.
In 64-bit mode, the instruction''s default operation size is 32 bits. Using a REX prefix in the form of REX.R permits
access to additional registers (R8-R15). Using a REX prefix in the form of REX.W promotes operation to 64 bits. See
the summary chart at the beginning of this section for encoding data and limits.



Operation
DEST <- DEST AND SRC;

Flags Affected
The OF and CF flags are cleared; the SF, ZF, and PF flags are set according to the result. The state of the AF flag is
undefined.

Protected Mode Exceptions
#GP(0)                 If the destination operand points to a non-writable segment.
                       If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                       If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)        If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made while the
                     current privilege level is 3.
#UD                    If the LOCK prefix is used but the destination is not a memory operand.

Real-Address Mode Exceptions
#GP                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                    If a memory operand effective address is outside the SS segment limit.
#UD                    If the LOCK prefix is used but the destination is not a memory operand.

Virtual-8086 Mode Exceptions
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made.
#UD                    If the LOCK prefix is used but the destination is not a memory operand.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)                 If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)                 If the memory address is in a non-canonical form.
#PF(fault-code)        If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made while the
                     current privilege level is 3.
#UD                    If the LOCK prefix is used but the destination is not a memory operand.




');
INSERT INTO "instructions" VALUES('x86_64','ANDN','ANDN - Logical AND NOT
 Opcode/Instruction              Op/     64/32     CPUID        Description
                                 En      -bit      Feature
                                         Mode      Flag
 VEX.NDS.LZ.0F38.W0 F2 /r        RVM     V/V       BMI1         Bitwise AND of inverted r32b with r/m32, store result in r32a.
 ANDN r32a, r32b, r/m32
 VEX.NDS.LZ. 0F38.W1 F2 /r       RVM     V/NE      BMI1         Bitwise AND of inverted r64b with r/m64, store result in r64a.
 ANDN r64a, r64b, r/m64



                                                 Instruction Operand Encoding
  Op/En             Operand 1                    Operand 2                       Operand 3                      Operand 4
   RVM           ModRM:reg (w)                   VEX.vvvv (r)                  ModRM:r/m (r)                        NA

Description
Performs a bitwise logical AND of inverted second operand (the first source operand) with the third operand (the
second source operand). The result is stored in the first operand (destination operand).
This instruction is not supported in real mode and virtual-8086 mode. The operand size is always 32 bits if not in
64-bit mode. In 64-bit mode operand size 64 requires VEX.W1. VEX.W1 is ignored in non-64-bit modes. An
attempt to execute this instruction with VEX.L not equal to 0 will cause #UD.

Operation
DEST <- (NOT SRC1) bitwiseAND SRC2;
SF <- DEST[OperandSize -1];
ZF <- (DEST = 0);

Flags Affected
SF and ZF are updated based on result. OF and CF flags are cleared. AF and PF flags are undefined.

Intel C/C++ Compiler Intrinsic Equivalent
Auto-generated from high-level language.

SIMD Floating-Point Exceptions
None

Other Exceptions
See Section 2.5.1, "Exception Conditions for VEX-Encoded GPR Instructions", Table 2-29; additionally
#UD                      If VEX.W = 1.




');
INSERT INTO "instructions" VALUES('x86_64','ANDPD','ANDPD-Bitwise Logical AND of Packed Double Precision Floating-Point Values
 Opcode/                           Op /    64/32       CPUID          Description
 Instruction                       En      bit Mode    Feature
                                           Support     Flag
 66 0F 54 /r                       RM      V/V         SSE2           Return the bitwise logical AND of packed double-
 ANDPD xmm1, xmm2/m128                                                precision floating-point values in xmm1 and xmm2/mem.
 VEX.NDS.128.66.0F 54 /r           RVM     V/V         AVX            Return the bitwise logical AND of packed double-
 VANDPD xmm1, xmm2,                                                   precision floating-point values in xmm2 and xmm3/mem.
 xmm3/m128
 VEX.NDS.256.66.0F 54 /r           RVM     V/V         AVX            Return the bitwise logical AND of packed double-
 VANDPD ymm1, ymm2,                                                   precision floating-point values in ymm2 and ymm3/mem.
 ymm3/m256
 EVEX.NDS.128.66.0F.W1 54 /r       FV      V/V         AVX512VL       Return the bitwise logical AND of packed double-
 VANDPD xmm1 {k1}{z}, xmm2,                            AVX512DQ       precision floating-point values in xmm2 and
 xmm3/m128/m64bcst                                                    xmm3/m128/m64bcst subject to writemask k1.
 EVEX.NDS.256.66.0F.W1 54 /r       FV      V/V         AVX512VL       Return the bitwise logical AND of packed double-
 VANDPD ymm1 {k1}{z}, ymm2,                            AVX512DQ       precision floating-point values in ymm2 and
 ymm3/m256/m64bcst                                                    ymm3/m256/m64bcst subject to writemask k1.
 EVEX.NDS.512.66.0F.W1 54 /r       FV      V/V         AVX512DQ       Return the bitwise logical AND of packed double-
 VANDPD zmm1 {k1}{z}, zmm2,                                           precision floating-point values in zmm2 and
 zmm3/m512/m64bcst                                                    zmm3/m512/m64bcst subject to writemask k1.



                                           Instruction Operand Encoding
         Op/En             Operand 1                   Operand 2                       Operand 3                Operand 4
          RM            ModRM:reg (r, w)              ModRM:r/m (r)                        NA                       NA
         RVM             ModRM:reg (w)                  VEX.vvvv                     ModRM:r/m (r)                  NA
          FV             ModRM:reg (w)                 EVEX.vvvv                     ModRM:r/m (r)                  NA

Description
Performs a bitwise logical AND of the two, four or eight packed double-precision floating-point values from the first
source operand and the second source operand, and stores the result in the destination operand.
EVEX encoded versions: The first source operand is a ZMM/YMM/XMM register. The second source operand can be
a ZMM/YMM/XMM register, a 512/256/128-bit memory location, or a 512/256/128-bit vector broadcasted from a
64-bit memory location. The destination operand is a ZMM/YMM/XMM register conditionally updated with
writemask k1.
VEX.256 encoded version: The first source operand is a YMM register. The second source operand is a YMM register
or a 256-bit memory location. The destination operand is a YMM register. The upper bits (MAX_VL-1:256) of the
corresponding ZMM register destination are zeroed.
VEX.128 encoded version: The first source operand is an XMM register. The second source operand is an XMM
register or 128-bit memory location. The destination operand is an XMM register. The upper bits (MAX_VL-1:128)
of the corresponding ZMM register destination are zeroed.
128-bit Legacy SSE version: The second source can be an XMM register or an 128-bit memory location. The desti-
nation is not distinct from the first source XMM register and the upper bits (MAX_VL-1:128) of the corresponding
register destination are unmodified.





Operation
VANDPD (EVEX encoded versions)
(KL, VL) = (2, 128), (4, 256), (8, 512)
FOR j <- 0 TO KL-1
    i <- j * 64
    IF k1[j] OR *no writemask*
         THEN
               IF (EVEX.b == 1) AND (SRC2 *is memory*)
                    THEN
                         DEST[i+63:i] <- SRC1[i+63:i] BITWISE AND SRC2[63:0]
                    ELSE
                         DEST[i+63:i] <- SRC1[i+63:i] BITWISE AND SRC2[i+63:i]
               FI;
         ELSE
               IF *merging-masking*                 ; merging-masking
                    THEN *DEST[i+63:i] remains unchanged*
                    ELSE                            ; zeroing-masking
                         DEST[i+63:i] = 0
               FI;
    FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VANDPD (VEX.256 encoded version)
DEST[63:0] <- SRC1[63:0] BITWISE AND SRC2[63:0]
DEST[127:64] <- SRC1[127:64] BITWISE AND SRC2[127:64]
DEST[191:128] <- SRC1[191:128] BITWISE AND SRC2[191:128]
DEST[255:192] <- SRC1[255:192] BITWISE AND SRC2[255:192]
DEST[MAX_VL-1:256] <- 0

VANDPD (VEX.128 encoded version)
DEST[63:0] <- SRC1[63:0] BITWISE AND SRC2[63:0]
DEST[127:64] <- SRC1[127:64] BITWISE AND SRC2[127:64]
DEST[MAX_VL-1:128] <- 0

ANDPD (128-bit Legacy SSE version)
DEST[63:0] <- DEST[63:0] BITWISE AND SRC[63:0]
DEST[127:64] <- DEST[127:64] BITWISE AND SRC[127:64]
DEST[MAX_VL-1:128] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VANDPD __m512d _mm512_and_pd (__m512d a, __m512d b);
VANDPD __m512d _mm512_mask_and_pd (__m512d s, __mmask8 k, __m512d a, __m512d b);
VANDPD __m512d _mm512_maskz_and_pd (__mmask8 k, __m512d a, __m512d b);
VANDPD __m256d _mm256_mask_and_pd (__m256d s, __mmask8 k, __m256d a, __m256d b);
VANDPD __m256d _mm256_maskz_and_pd (__mmask8 k, __m256d a, __m256d b);
VANDPD __m128d _mm_mask_and_pd (__m128d s, __mmask8 k, __m128d a, __m128d b);
VANDPD __m128d _mm_maskz_and_pd (__mmask8 k, __m128d a, __m128d b);
VANDPD __m256d _mm256_and_pd (__m256d a, __m256d b);
ANDPD __m128d _mm_and_pd (__m128d a, __m128d b);

SIMD Floating-Point Exceptions
None




Other Exceptions
VEX-encoded instruction, see Exceptions Type 4.
EVEX-encoded instruction, see Exceptions Type E4.




');
INSERT INTO "instructions" VALUES('x86_64','ANDPS','ANDPS-Bitwise Logical AND of Packed Single Precision Floating-Point Values
 Opcode/                               Op /     64/32         CPUID           Description
 Instruction                           En       bit Mode      Feature
                                                Support       Flag
 0F 54 /r                              RM       V/V           SSE             Return the bitwise logical AND of packed single-precision
 ANDPS xmm1, xmm2/m128                                                        floating-point values in xmm1 and xmm2/mem.
 VEX.NDS.128.0F 54 /r                  RVM      V/V           AVX             Return the bitwise logical AND of packed single-precision
 VANDPS xmm1,xmm2,                                                            floating-point values in xmm2 and xmm3/mem.
 xmm3/m128
 VEX.NDS.256.0F 54 /r                  RVM      V/V           AVX             Return the bitwise logical AND of packed single-precision
 VANDPS ymm1, ymm2,                                                           floating-point values in ymm2 and ymm3/mem.
 ymm3/m256
 EVEX.NDS.128.0F.W0 54 /r              FV       V/V           AVX512VL        Return the bitwise logical AND of packed single-precision
 VANDPS xmm1 {k1}{z}, xmm2,                                   AVX512DQ        floating-point values in xmm2 and xmm3/m128/m32bcst
 xmm3/m128/m32bcst                                                            subject to writemask k1.
 EVEX.NDS.256.0F.W0 54 /r              FV       V/V           AVX512VL        Return the bitwise logical AND of packed single-precision
 VANDPS ymm1 {k1}{z}, ymm2,                                   AVX512DQ        floating-point values in ymm2 and ymm3/m256/m32bcst
 ymm3/m256/m32bcst                                                            subject to writemask k1.
 EVEX.NDS.512.0F.W0 54 /r              FV       V/V           AVX512DQ        Return the bitwise logical AND of packed single-precision
 VANDPS zmm1 {k1}{z}, zmm2,                                                   floating-point values in zmm2 and zmm3/m512/m32bcst
 zmm3/m512/m32bcst                                                            subject to writemask k1.



                                                      Instruction Operand Encoding
       Op/En                  Operand 1                         Operand 2                        Operand 3                Operand 4
         RM               ModRM:reg (r, w)                    ModRM:r/m (r)                          NA                       NA
        RVM                ModRM:reg (w)                         VEX.vvvv                      ModRM:r/m (r)                  NA
         FV                ModRM:reg (w)                        EVEX.vvvv                      ModRM:r/m (r)                  NA

Description
Performs a bitwise logical AND of the four, eight or sixteen packed single-precision floating-point values from the
first source operand and the second source operand, and stores the result in the destination operand.
EVEX encoded versions: The first source operand is a ZMM/YMM/XMM register. The second source operand can be
a ZMM/YMM/XMM register, a 512/256/128-bit memory location, or a 512/256/128-bit vector broadcasted from a
32-bit memory location. The destination operand is a ZMM/YMM/XMM register conditionally updated with
writemask k1.
VEX.256 encoded version: The first source operand is a YMM register. The second source operand is a YMM register
or a 256-bit memory location. The destination operand is a YMM register. The upper bits (MAX_VL-1:256) of the
corresponding ZMM register destination are zeroed.
VEX.128 encoded version: The first source operand is an XMM register. The second source operand is an XMM
register or 128-bit memory location. The destination operand is an XMM register. The upper bits (MAX_VL-1:128)
of the corresponding ZMM register destination are zeroed.
128-bit Legacy SSE version: The second source can be an XMM register or an 128-bit memory location. The desti-
nation is not distinct from the first source XMM register and the upper bits (MAX_VL-1:128) of the corresponding
ZMM register destination are unmodified.





Operation
VANDPS (EVEX encoded versions)
(KL, VL) = (4, 128), (8, 256), (16, 512)
FOR j <- 0 TO KL-1
    i <- j * 32
    IF k1[j] OR *no writemask*
               IF (EVEX.b == 1) AND (SRC2 *is memory*)
                    THEN
                         DEST[i+63:i] <- SRC1[i+31:i] BITWISE AND SRC2[31:0]
                    ELSE
                         DEST[i+31:i] <- SRC1[i+31:i] BITWISE AND SRC2[i+31:i]
               FI;
         ELSE
               IF *merging-masking*                 ; merging-masking
                    THEN *DEST[i+31:i] remains unchanged*
                    ELSE                            ; zeroing-masking
                         DEST[i+31:i] <- 0
               FI;
    FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0;

VANDPS (VEX.256 encoded version)
DEST[31:0] <- SRC1[31:0] BITWISE AND SRC2[31:0]
DEST[63:32] <- SRC1[63:32] BITWISE AND SRC2[63:32]
DEST[95:64] <- SRC1[95:64] BITWISE AND SRC2[95:64]
DEST[127:96] <- SRC1[127:96] BITWISE AND SRC2[127:96]
DEST[159:128] <- SRC1[159:128] BITWISE AND SRC2[159:128]
DEST[191:160] <- SRC1[191:160] BITWISE AND SRC2[191:160]
DEST[223:192] <- SRC1[223:192] BITWISE AND SRC2[223:192]
DEST[255:224] <- SRC1[255:224] BITWISE AND SRC2[255:224].
DEST[MAX_VL-1:256] <- 0;

VANDPS (VEX.128 encoded version)
DEST[31:0] <- SRC1[31:0] BITWISE AND SRC2[31:0]
DEST[63:32] <- SRC1[63:32] BITWISE AND SRC2[63:32]
DEST[95:64] <- SRC1[95:64] BITWISE AND SRC2[95:64]
DEST[127:96] <- SRC1[127:96] BITWISE AND SRC2[127:96]
DEST[MAX_VL-1:128] <- 0;

ANDPS (128-bit Legacy SSE version)
DEST[31:0] <- DEST[31:0] BITWISE AND SRC[31:0]
DEST[63:32] <- DEST[63:32] BITWISE AND SRC[63:32]
DEST[95:64] <- DEST[95:64] BITWISE AND SRC[95:64]
DEST[127:96] <- DEST[127:96] BITWISE AND SRC[127:96]
DEST[MAX_VL-1:128] (Unmodified)





Intel C/C++ Compiler Intrinsic Equivalent
VANDPS __m512 _mm512_and_ps (__m512 a, __m512 b);
VANDPS __m512 _mm512_mask_and_ps (__m512 s, __mmask16 k, __m512 a, __m512 b);
VANDPS __m512 _mm512_maskz_and_ps (__mmask16 k, __m512 a, __m512 b);
VANDPS __m256 _mm256_mask_and_ps (__m256 s, __mmask8 k, __m256 a, __m256 b);
VANDPS __m256 _mm256_maskz_and_ps (__mmask8 k, __m256 a, __m256 b);
VANDPS __m128 _mm_mask_and_ps (__m128 s, __mmask8 k, __m128 a, __m128 b);
VANDPS __m128 _mm_maskz_and_ps (__mmask8 k, __m128 a, __m128 b);
VANDPS __m256 _mm256_and_ps (__m256 a, __m256 b);
ANDPS __m128 _mm_and_ps (__m128 a, __m128 b);

SIMD Floating-Point Exceptions
None

Other Exceptions
VEX-encoded instruction, see Exceptions Type 4.
EVEX-encoded instruction, see Exceptions Type E4.




');
INSERT INTO "instructions" VALUES('x86_64','ANDNPD','ANDNPD-Bitwise Logical AND NOT of Packed Double Precision Floating-Point Values
 Opcode/                           Op /    64/32        CPUID           Description
 Instruction                       En      bit Mode     Feature
                                           Support      Flag
 66 0F 55 /r                       RM      V/V          SSE2            Return the bitwise logical AND NOT of packed double-
 ANDNPD xmm1, xmm2/m128                                                 precision floating-point values in xmm1 and xmm2/mem.
 VEX.NDS.128.66.0F 55 /r           RVM     V/V          AVX             Return the bitwise logical AND NOT of packed double-
 VANDNPD xmm1, xmm2,                                                    precision floating-point values in xmm2 and xmm3/mem.
 xmm3/m128
 VEX.NDS.256.66.0F 55/r            RVM     V/V          AVX             Return the bitwise logical AND NOT of packed double-
 VANDNPD ymm1, ymm2,                                                    precision floating-point values in ymm2 and ymm3/mem.
 ymm3/m256
 EVEX.NDS.128.66.0F.W1 55 /r       FV      V/V          AVX512VL        Return the bitwise logical AND NOT of packed double-
 VANDNPD xmm1 {k1}{z}, xmm2,                            AVX512DQ        precision floating-point values in xmm2 and
 xmm3/m128/m64bcst                                                      xmm3/m128/m64bcst subject to writemask k1.
 EVEX.NDS.256.66.0F.W1 55 /r       FV      V/V          AVX512VL        Return the bitwise logical AND NOT of packed double-
 VANDNPD ymm1 {k1}{z}, ymm2,                            AVX512DQ        precision floating-point values in ymm2 and
 ymm3/m256/m64bcst                                                      ymm3/m256/m64bcst subject to writemask k1.
 EVEX.NDS.512.66.0F.W1 55 /r       FV      V/V          AVX512DQ        Return the bitwise logical AND NOT of packed double-
 VANDNPD zmm1 {k1}{z}, zmm2,                                            precision floating-point values in zmm2 and
 zmm3/m512/m64bcst                                                      zmm3/m512/m64bcst subject to writemask k1.



                                           Instruction Operand Encoding
        Op/En              Operand 1                     Operand 2                        Operand 3                Operand 4
          RM            ModRM:reg (r, w)               ModRM:r/m (r)                          NA                       NA
         RVM             ModRM:reg (w)                    VEX.vvvv                      ModRM:r/m (r)                  NA
          FV             ModRM:reg (w)                   EVEX.vvvv                      ModRM:r/m (r)                  NA

Description
Performs a bitwise logical AND NOT of the two, four or eight packed double-precision floating-point values from the
first source operand and the second source operand, and stores the result in the destination operand.
EVEX encoded versions: The first source operand is a ZMM/YMM/XMM register. The second source operand can be
a ZMM/YMM/XMM register, a 512/256/128-bit memory location, or a 512/256/128-bit vector broadcasted from a
64-bit memory location. The destination operand is a ZMM/YMM/XMM register conditionally updated with
writemask k1.
VEX.256 encoded version: The first source operand is a YMM register. The second source operand is a YMM register
or a 256-bit memory location. The destination operand is a YMM register. The upper bits (MAX_VL-1:256) of the
corresponding ZMM register destination are zeroed.
VEX.128 encoded version: The first source operand is an XMM register. The second source operand is an XMM
register or 128-bit memory location. The destination operand is an XMM register. The upper bits (MAX_VL-1:128)
of the corresponding ZMM register destination are zeroed.
128-bit Legacy SSE version: The second source can be an XMM register or an 128-bit memory location. The desti-
nation is not distinct from the first source XMM register and the upper bits (MAX_VL-1:128) of the corresponding
register destination are unmodified.





Operation
VANDNPD (EVEX encoded versions)
(KL, VL) = (2, 128), (4, 256), (8, 512)
FOR j <- 0 TO KL-1
    i <- j * 64
    IF k1[j] OR *no writemask*
               IF (EVEX.b == 1) AND (SRC2 *is memory*)
                    THEN
                         DEST[i+63:i] <- (NOT(SRC1[i+63:i])) BITWISE AND SRC2[63:0]
                    ELSE
                         DEST[i+63:i] <- (NOT(SRC1[i+63:i])) BITWISE AND SRC2[i+63:i]
               FI;
         ELSE
               IF *merging-masking*                ; merging-masking
                    THEN *DEST[i+63:i] remains unchanged*
                    ELSE                           ; zeroing-masking
                         DEST[i+63:i] = 0
               FI;
    FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VANDNPD (VEX.256 encoded version)
DEST[63:0] <- (NOT(SRC1[63:0])) BITWISE AND SRC2[63:0]
DEST[127:64] <- (NOT(SRC1[127:64])) BITWISE AND SRC2[127:64]
DEST[191:128] <- (NOT(SRC1[191:128])) BITWISE AND SRC2[191:128]
DEST[255:192] <- (NOT(SRC1[255:192])) BITWISE AND SRC2[255:192]
DEST[MAX_VL-1:256] <- 0

VANDNPD (VEX.128 encoded version)
DEST[63:0] <- (NOT(SRC1[63:0])) BITWISE AND SRC2[63:0]
DEST[127:64] <- (NOT(SRC1[127:64])) BITWISE AND SRC2[127:64]
DEST[MAX_VL-1:128] <- 0

ANDNPD (128-bit Legacy SSE version)
DEST[63:0] <- (NOT(DEST[63:0])) BITWISE AND SRC[63:0]
DEST[127:64] <- (NOT(DEST[127:64])) BITWISE AND SRC[127:64]
DEST[MAX_VL-1:128] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VANDNPD __m512d _mm512_andnot_pd (__m512d a, __m512d b);
VANDNPD __m512d _mm512_mask_andnot_pd (__m512d s, __mmask8 k, __m512d a, __m512d b);
VANDNPD __m512d _mm512_maskz_andnot_pd (__mmask8 k, __m512d a, __m512d b);
VANDNPD __m256d _mm256_mask_andnot_pd (__m256d s, __mmask8 k, __m256d a, __m256d b);
VANDNPD __m256d _mm256_maskz_andnot_pd (__mmask8 k, __m256d a, __m256d b);
VANDNPD __m128d _mm_mask_andnot_pd (__m128d s, __mmask8 k, __m128d a, __m128d b);
VANDNPD __m128d _mm_maskz_andnot_pd (__mmask8 k, __m128d a, __m128d b);
VANDNPD __m256d _mm256_andnot_pd (__m256d a, __m256d b);
ANDNPD __m128d _mm_andnot_pd (__m128d a, __m128d b);

SIMD Floating-Point Exceptions
None





Other Exceptions
VEX-encoded instruction, see Exceptions Type 4.
EVEX-encoded instruction, see Exceptions Type E4.




');
INSERT INTO "instructions" VALUES('x86_64','ANDNPS','ANDNPS-Bitwise Logical AND NOT of Packed Single Precision Floating-Point Values
 Opcode/                              Op /    64/32         CPUID          Description
 Instruction                          En      bit Mode      Feature
                                              Support       Flag
 0F 55 /r                             RM      V/V           SSE            Return the bitwise logical AND NOT of packed single-precision
 ANDNPS xmm1, xmm2/m128                                                    floating-point values in xmm1 and xmm2/mem.
 VEX.NDS.128.0F 55 /r                 RVM     V/V           AVX            Return the bitwise logical AND NOT of packed single-precision
 VANDNPS xmm1, xmm2,                                                       floating-point values in xmm2 and xmm3/mem.
 xmm3/m128
 VEX.NDS.256.0F 55 /r                 RVM     V/V           AVX            Return the bitwise logical AND NOT of packed single-precision
 VANDNPS ymm1, ymm2,                                                       floating-point values in ymm2 and ymm3/mem.
 ymm3/m256
 EVEX.NDS.128.0F.W0 55 /r             FV      V/V           AVX512VL       Return the bitwise logical AND of packed single-precision
 VANDNPS xmm1 {k1}{z},                                      AVX512DQ       floating-point values in xmm2 and xmm3/m128/m32bcst
 xmm2, xmm3/m128/m32bcst                                                   subject to writemask k1.
 EVEX.NDS.256.0F.W0 55 /r             FV      V/V           AVX512VL       Return the bitwise logical AND of packed single-precision
 VANDNPS ymm1 {k1}{z},                                      AVX512DQ       floating-point values in ymm2 and ymm3/m256/m32bcst
 ymm2, ymm3/m256/m32bcst                                                   subject to writemask k1.
 EVEX.NDS.512.0F.W0 55 /r             FV      V/V           AVX512DQ       Return the bitwise logical AND of packed single-precision
 VANDNPS zmm1 {k1}{z},                                                     floating-point values in zmm2 and zmm3/m512/m32bcst
 zmm2, zmm3/m512/m32bcst                                                   subject to writemask k1.



                                                     Instruction Operand Encoding
          Op/En                   Operand 1                       Operand 2                     Operand 3               Operand 4
            RM                 ModRM:reg (r, w)                 ModRM:r/m (r)                       NA                      NA
           RVM                  ModRM:reg (w)                      VEX.vvvv                   ModRM:r/m (r)                 NA
            FV                  ModRM:reg (w)                     EVEX.vvvv                   ModRM:r/m (r)                 NA

Description
Performs a bitwise logical AND NOT of the four, eight or sixteen packed single-precision floating-point values from
the first source operand and the second source operand, and stores the result in the destination operand.
EVEX encoded versions: The first source operand is a ZMM/YMM/XMM register. The second source operand can be
a ZMM/YMM/XMM register, a 512/256/128-bit memory location, or a 512/256/128-bit vector broadcasted from a
32-bit memory location. The destination operand is a ZMM/YMM/XMM register conditionally updated with
writemask k1.
VEX.256 encoded version: The first source operand is a YMM register. The second source operand is a YMM register
or a 256-bit memory location. The destination operand is a YMM register. The upper bits (MAX_VL-1:256) of the
corresponding ZMM register destination are zeroed.
VEX.128 encoded version: The first source operand is an XMM register. The second source operand is an XMM
register or 128-bit memory location. The destination operand is an XMM register. The upper bits (MAX_VL-1:128)
of the corresponding ZMM register destination are zeroed.
128-bit Legacy SSE version: The second source can be an XMM register or an 128-bit memory location. The desti-
nation is not distinct from the first source XMM register and the upper bits (MAX_VL-1:128) of the corresponding
ZMM register destination are unmodified.





Operation
VANDNPS (EVEX encoded versions)
(KL, VL) = (4, 128), (8, 256), (16, 512)
FOR j <- 0 TO KL-1
    i <- j * 32
    IF k1[j] OR *no writemask*
               IF (EVEX.b == 1) AND (SRC2 *is memory*)
                    THEN
                         DEST[i+31:i] <- (NOT(SRC1[i+31:i])) BITWISE AND SRC2[31:0]
                    ELSE
                         DEST[i+31:i] <- (NOT(SRC1[i+31:i])) BITWISE AND SRC2[i+31:i]
               FI;
         ELSE
               IF *merging-masking*                ; merging-masking
                    THEN *DEST[i+31:i] remains unchanged*
                    ELSE                           ; zeroing-masking
                         DEST[i+31:i] = 0
               FI;
    FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VANDNPS (VEX.256 encoded version)
DEST[31:0] <- (NOT(SRC1[31:0])) BITWISE AND SRC2[31:0]
DEST[63:32] <- (NOT(SRC1[63:32])) BITWISE AND SRC2[63:32]
DEST[95:64] <- (NOT(SRC1[95:64])) BITWISE AND SRC2[95:64]
DEST[127:96] <- (NOT(SRC1[127:96])) BITWISE AND SRC2[127:96]
DEST[159:128] <- (NOT(SRC1[159:128])) BITWISE AND SRC2[159:128]
DEST[191:160] <- (NOT(SRC1[191:160])) BITWISE AND SRC2[191:160]
DEST[223:192] <- (NOT(SRC1[223:192])) BITWISE AND SRC2[223:192]
DEST[255:224] <- (NOT(SRC1[255:224])) BITWISE AND SRC2[255:224].
DEST[MAX_VL-1:256] <- 0

VANDNPS (VEX.128 encoded version)
DEST[31:0] <- (NOT(SRC1[31:0])) BITWISE AND SRC2[31:0]
DEST[63:32] <- (NOT(SRC1[63:32])) BITWISE AND SRC2[63:32]
DEST[95:64] <- (NOT(SRC1[95:64])) BITWISE AND SRC2[95:64]
DEST[127:96] <- (NOT(SRC1[127:96])) BITWISE AND SRC2[127:96]
DEST[MAX_VL-1:128] <- 0

ANDNPS (128-bit Legacy SSE version)
DEST[31:0] <- (NOT(DEST[31:0])) BITWISE AND SRC[31:0]
DEST[63:32] <- (NOT(DEST[63:32])) BITWISE AND SRC[63:32]
DEST[95:64] <- (NOT(DEST[95:64])) BITWISE AND SRC[95:64]
DEST[127:96] <- (NOT(DEST[127:96])) BITWISE AND SRC[127:96]
DEST[MAX_VL-1:128] (Unmodified)





Intel C/C++ Compiler Intrinsic Equivalent
VANDNPS __m512 _mm512_andnot_ps (__m512 a, __m512 b);
VANDNPS __m512 _mm512_mask_andnot_ps (__m512 s, __mmask16 k, __m512 a, __m512 b);
VANDNPS __m512 _mm512_maskz_andnot_ps (__mmask16 k, __m512 a, __m512 b);
VANDNPS __m256 _mm256_mask_andnot_ps (__m256 s, __mmask8 k, __m256 a, __m256 b);
VANDNPS __m256 _mm256_maskz_andnot_ps (__mmask8 k, __m256 a, __m256 b);
VANDNPS __m128 _mm_mask_andnot_ps (__m128 s, __mmask8 k, __m128 a, __m128 b);
VANDNPS __m128 _mm_maskz_andnot_ps (__mmask8 k, __m128 a, __m128 b);
VANDNPS __m256 _mm256_andnot_ps (__m256 a, __m256 b);
ANDNPS __m128 _mm_andnot_ps (__m128 a, __m128 b);

SIMD Floating-Point Exceptions
None

Other Exceptions
VEX-encoded instruction, see Exceptions Type 4.
EVEX-encoded instruction, see Exceptions Type E4.




');
INSERT INTO "instructions" VALUES('x86_64','ARPL','ARPL-Adjust RPL Field of Segment Selector
Opcode                  Instruction              Op/     64-bit   Compat/ Description
                                                 En      Mode     Leg Mode
63 /r                   ARPL r/m16, r16          NP      N. E.    Valid       Adjust RPL of r/m16 to not less than RPL of
                                                                              r16.



                                            Instruction Operand Encoding
  Op/En            Operand 1                 Operand 2                    Operand 3                      Operand 4
    NP           ModRM:r/m (w)             ModRM:reg (r)                     NA                             NA

Description
Compares the RPL fields of two segment selectors. The first operand (the destination operand) contains one
segment selector and the second operand (source operand) contains the other. (The RPL field is located in bits 0
and 1 of each operand.) If the RPL field of the destination operand is less than the RPL field of the source operand,
the ZF flag is set and the RPL field of the destination operand is increased to match that of the source operand.
Otherwise, the ZF flag is cleared and no change is made to the destination operand. (The destination operand can
be a word register or a memory location; the source operand must be a word register.)
The ARPL instruction is provided for use by operating-system procedures (however, it can also be used by applica-
tions). It is generally used to adjust the RPL of a segment selector that has been passed to the operating system
by an application program to match the privilege level of the application program. Here the segment selector
passed to the operating system is placed in the destination operand and segment selector for the application
program''s code segment is placed in the source operand. (The RPL field in the source operand represents the priv-
ilege level of the application program.) Execution of the ARPL instruction then ensures that the RPL of the segment
selector received by the operating system is no lower (does not have a higher privilege) than the privilege level of
the application program (the segment selector for the application program''s code segment can be read from the
stack following a procedure call).
This instruction executes as described in compatibility mode and legacy mode. It is not encodable in 64-bit mode.
See "Checking Caller Access Privileges" in Chapter 3, "Protected-Mode Memory Management," of the Intel 64 and
IA-32 Architectures Software Developer''s Manual, Volume 3A, for more information about the use of this instruc-
tion.

Operation
IF 64-BIT MODE
    THEN
        See MOVSXD;
    ELSE
        IF DEST[RPL] < SRC[RPL]
             THEN
                 ZF <- 1;
                 DEST[RPL] <- SRC[RPL];
             ELSE
                 ZF <- 0;
        FI;
FI;

Flags Affected
The ZF flag is set to 1 if the RPL field of the destination operand is less than that of the source operand; otherwise,
it is set to 0.





Protected Mode Exceptions
#GP(0)                   If the destination is located in a non-writable segment.
                         If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                         If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                         selector.
#SS(0)                   If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)          If a page fault occurs.
#AC(0)                   If alignment checking is enabled and an unaligned memory reference is made while the
                         current privilege level is 3.
#UD                      If the LOCK prefix is used.

Real-Address Mode Exceptions
#UD                      The ARPL instruction is not recognized in real-address mode.
                         If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#UD                      The ARPL instruction is not recognized in virtual-8086 mode.
                         If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Not applicable.




');
INSERT INTO "instructions" VALUES('x86_64','BLENDPD','BLENDPD - Blend Packed Double Precision Floating-Point Values
 Opcode/                                            Op/    64/32-bit CPUID       Description
 Instruction                                        En     Mode      Feature
                                                                     Flag
 66 0F 3A 0D /r ib                                  RMI    V/V       SSE4_1      Select packed DP-FP values from xmm1 and
 BLENDPD xmm1, xmm2/m128, imm8                                                   xmm2/m128 from mask specified in imm8
                                                                                 and store the values into xmm1.
 VEX.NDS.128.66.0F3A.WIG 0D /r ib                   RVMI V/V         AVX         Select packed double-precision floating-point
 VBLENDPD xmm1, xmm2, xmm3/m128, imm8                                            Values from xmm2 and xmm3/m128 from
                                                                                 mask in imm8 and store the values in xmm1.
 VEX.NDS.256.66.0F3A.WIG 0D /r ib                   RVMI V/V         AVX         Select packed double-precision floating-point
 VBLENDPD ymm1, ymm2, ymm3/m256, imm8                                            Values from ymm2 and ymm3/m256 from
                                                                                 mask in imm8 and store the values in ymm1.



                                            Instruction Operand Encoding
  Op/En               Operand 1              Operand 2                     Operand 3                       Operand 4
   RMI           ModRM:reg (r, w)           ModRM:r/m (r)                     imm8                             NA
   RVMI              ModRM:reg (w)          VEX.vvvv (r)                ModRM:r/m (r)                       imm8[3:0]

Description
Double-precision floating-point values from the second source operand (third operand) are conditionally merged
with values from the first source operand (second operand) and written to the destination operand (first operand).
The immediate bits [3:0] determine whether the corresponding double-precision floating-point value in the desti-
nation is copied from the second source or first source. If a bit in the mask, corresponding to a word, is "1", then
the double-precision floating-point value in the second source operand is copied, else the value in the first source
operand is copied.
128-bit Legacy SSE version: The second source can be an XMM register or an 128-bit memory location. The desti-
nation is not distinct from the first source XMM register and the upper bits (VLMAX-1:128) of the corresponding
YMM register destination are unmodified.
VEX.128 encoded version: the first source operand is an XMM register. The second source operand is an XMM
register or 128-bit memory location. The destination operand is an XMM register. The upper bits (VLMAX-1:128) of
the corresponding YMM register destination are zeroed.
VEX.256 encoded version: The first source operand is a YMM register. The second source operand can be a YMM
register or a 256-bit memory location. The destination operand is a YMM register.

Operation
BLENDPD (128-bit Legacy SSE version)
IF (IMM8[0] = 0)THEN DEST[63:0] <- DEST[63:0]
        ELSE DEST [63:0] <- SRC[63:0] FI
IF (IMM8[1] = 0) THEN DEST[127:64] <- DEST[127:64]
        ELSE DEST [127:64] <- SRC[127:64] FI
DEST[VLMAX-1:128] (Unmodified)

VBLENDPD (VEX.128 encoded version)
IF (IMM8[0] = 0)THEN DEST[63:0] <- SRC1[63:0]
        ELSE DEST [63:0] <- SRC2[63:0] FI
IF (IMM8[1] = 0) THEN DEST[127:64] <- SRC1[127:64]
        ELSE DEST [127:64] <- SRC2[127:64] FI
DEST[VLMAX-1:128] <- 0





VBLENDPD (VEX.256 encoded version)
IF (IMM8[0] = 0)THEN DEST[63:0] <- SRC1[63:0]
        ELSE DEST [63:0] <- SRC2[63:0] FI
IF (IMM8[1] = 0) THEN DEST[127:64] <- SRC1[127:64]
        ELSE DEST [127:64] <- SRC2[127:64] FI
IF (IMM8[2] = 0) THEN DEST[191:128] <- SRC1[191:128]
        ELSE DEST [191:128] <- SRC2[191:128] FI
IF (IMM8[3] = 0) THEN DEST[255:192] <- SRC1[255:192]
        ELSE DEST [255:192] <- SRC2[255:192] FI

Intel C/C++ Compiler Intrinsic Equivalent
BLENDPD:         __m128d _mm_blend_pd (__m128d v1, __m128d v2, const int mask);

VBLENDPD:        __m256d _mm256_blend_pd (__m256d a, __m256d b, const int mask);

SIMD Floating-Point Exceptions
None

Other Exceptions
See Exceptions Type 4.




');
INSERT INTO "instructions" VALUES('x86_64','BEXTR','BEXTR - Bit Field Extract
 Opcode/Instruction                Op/    64/32     CPUID      Description
                                   En     -bit      Feature
                                          Mode      Flag
 VEX.NDS.LZ.0F38.W0 F7 /r          RMV    V/V       BMI1       Contiguous bitwise extract from r/m32 using r32b as control; store
 BEXTR r32a, r/m32, r32b                                       result in r32a.
 VEX.NDS.LZ.0F38.W1 F7 /r          RMV    V/N.E.    BMI1       Contiguous bitwise extract from r/m64 using r64b as control; store
 BEXTR r64a, r/m64, r64b                                       result in r64a



                                                   Instruction Operand Encoding
   Op/En               Operand 1                     Operand 2                      Operand 3                        Operand 4
   RMV               ModRM:reg (w)                  ModRM:r/m (r)                  VEX.vvvv (r)                          NA

Description
Extracts contiguous bits from the first source operand (the second operand) using an index value and length value
specified in the second source operand (the third operand). Bit 7:0 of the second source operand specifies the
starting bit position of bit extraction. A START value exceeding the operand size will not extract any bits from the
second source operand. Bit 15:8 of the second source operand specifies the maximum number of bits (LENGTH)
beginning at the START position to extract. Only bit positions up to (OperandSize -1) of the first source operand are
extracted. The extracted bits are written to the destination register, starting from the least significant bit. All higher
order bits in the destination operand (starting at bit position LENGTH) are zeroed. The destination register is
cleared if no bits are extracted.
This instruction is not supported in real mode and virtual-8086 mode. The operand size is always 32 bits if not in
64-bit mode. In 64-bit mode operand size 64 requires VEX.W1. VEX.W1 is ignored in non-64-bit modes. An
attempt to execute this instruction with VEX.L not equal to 0 will cause #UD.

Operation
START <- SRC2[7:0];
LEN <- SRC2[15:8];
TEMP <- ZERO_EXTEND_TO_512 (SRC1 );
DEST <- ZERO_EXTEND(TEMP[START+LEN -1: START]);
ZF <- (DEST = 0);

Flags Affected
ZF is updated based on the result. AF, SF, and PF are undefined. All other flags are cleared.

Intel C/C++ Compiler Intrinsic Equivalent
BEXTR:         unsigned __int32 _bextr_u32(unsigned __int32 src, unsigned __int32 start. unsigned __int32 len);

BEXTR:         unsigned __int64 _bextr_u64(unsigned __int64 src, unsigned __int32 start. unsigned __int32 len);

SIMD Floating-Point Exceptions
None

Other Exceptions
See Section 2.5.1, "Exception Conditions for VEX-Encoded GPR Instructions", Table 2-29; additionally
#UD                      If VEX.W = 1.




');
INSERT INTO "instructions" VALUES('x86_64','BLENDPS','BLENDPS - Blend Packed Single Precision Floating-Point Values
 Opcode/                                                    Op/    64/32-bit CPUID       Description
 Instruction                                                En     Mode      Feature
                                                                             Flag
 66 0F 3A 0C /r ib                                          RMI    V/V       SSE4_1      Select packed single precision floating-point
 BLENDPS xmm1, xmm2/m128, imm8                                                           values from xmm1 and xmm2/m128 from
                                                                                         mask specified in imm8 and store the values
                                                                                         into xmm1.
 VEX.NDS.128.66.0F3A.WIG 0C /r ib                           RVMI V/V         AVX         Select packed single-precision floating-point
 VBLENDPS xmm1, xmm2, xmm3/m128, imm8                                                    values from xmm2 and xmm3/m128 from
                                                                                         mask in imm8 and store the values in xmm1.
 VEX.NDS.256.66.0F3A.WIG 0C /r ib                           RVMI V/V         AVX         Select packed single-precision floating-point
 VBLENDPS ymm1, ymm2, ymm3/m256, imm8                                                    values from ymm2 and ymm3/m256 from
                                                                                         mask in imm8 and store the values in ymm1.



                                                     Instruction Operand Encoding
   Op/En               Operand 1                       Operand 2                    Operand 3                      Operand 4
    RMI            ModRM:reg (r, w)                  ModRM:r/m (r)                     imm8                            NA
   RVMI              ModRM:reg (w)                    VEX.vvvv (r)                 ModRM:r/m (r)                     imm8

Description
Packed single-precision floating-point values from the second source operand (third operand) are conditionally
merged with values from the first source operand (second operand) and written to the destination operand (first
operand). The immediate bits [7:0] determine whether the corresponding single precision floating-point value in
the destination is copied from the second source or first source. If a bit in the mask, corresponding to a word, is
"1", then the single-precision floating-point value in the second source operand is copied, else the value in the first
source operand is copied.
128-bit Legacy SSE version: The second source can be an XMM register or an 128-bit memory location. The desti-
nation is not distinct from the first source XMM register and the upper bits (VLMAX-1:128) of the corresponding
YMM register destination are unmodified.
VEX.128 encoded version: The first source operand an XMM register. The second source operand is an XMM register
or 128-bit memory location. The destination operand is an XMM register. The upper bits (VLMAX-1:128) of the
corresponding YMM register destination are zeroed.
VEX.256 encoded version: The first source operand is a YMM register. The second source operand can be a YMM
register or a 256-bit memory location. The destination operand is a YMM register.

Operation
BLENDPS (128-bit Legacy SSE version)
IF (IMM8[0] = 0) THEN DEST[31:0] <-DEST[31:0]
        ELSE DEST [31:0] <- SRC[31:0] FI
IF (IMM8[1] = 0) THEN DEST[63:32] <- DEST[63:32]
        ELSE DEST [63:32] <- SRC[63:32] FI
IF (IMM8[2] = 0) THEN DEST[95:64] <- DEST[95:64]
        ELSE DEST [95:64] <- SRC[95:64] FI
IF (IMM8[3] = 0) THEN DEST[127:96] <- DEST[127:96]
        ELSE DEST [127:96] <- SRC[127:96] FI
DEST[VLMAX-1:128] (Unmodified)





VBLENDPS (VEX.128 encoded version)
IF (IMM8[0] = 0) THEN DEST[31:0] <-SRC1[31:0]
        ELSE DEST [31:0] <- SRC2[31:0] FI
IF (IMM8[1] = 0) THEN DEST[63:32] <- SRC1[63:32]
        ELSE DEST [63:32] <- SRC2[63:32] FI
IF (IMM8[2] = 0) THEN DEST[95:64] <- SRC1[95:64]
        ELSE DEST [95:64] <- SRC2[95:64] FI
IF (IMM8[3] = 0) THEN DEST[127:96] <- SRC1[127:96]
        ELSE DEST [127:96] <- SRC2[127:96] FI
DEST[VLMAX-1:128] <- 0

VBLENDPS (VEX.256 encoded version)
IF (IMM8[0] = 0) THEN DEST[31:0] <-SRC1[31:0]
        ELSE DEST [31:0] <- SRC2[31:0] FI
IF (IMM8[1] = 0) THEN DEST[63:32] <- SRC1[63:32]
        ELSE DEST [63:32] <- SRC2[63:32] FI
IF (IMM8[2] = 0) THEN DEST[95:64] <- SRC1[95:64]
        ELSE DEST [95:64] <- SRC2[95:64] FI
IF (IMM8[3] = 0) THEN DEST[127:96] <- SRC1[127:96]
        ELSE DEST [127:96] <- SRC2[127:96] FI
IF (IMM8[4] = 0) THEN DEST[159:128] <- SRC1[159:128]
        ELSE DEST [159:128] <- SRC2[159:128] FI
IF (IMM8[5] = 0) THEN DEST[191:160] <- SRC1[191:160]
        ELSE DEST [191:160] <- SRC2[191:160] FI
IF (IMM8[6] = 0) THEN DEST[223:192] <- SRC1[223:192]
        ELSE DEST [223:192] <- SRC2[223:192] FI
IF (IMM8[7] = 0) THEN DEST[255:224] <- SRC1[255:224]
        ELSE DEST [255:224] <- SRC2[255:224] FI.

Intel C/C++ Compiler Intrinsic Equivalent
BLENDPS:       __m128 _mm_blend_ps (__m128 v1, __m128 v2, const int mask);

VBLENDPS:      __m256 _mm256_blend_ps (__m256 a, __m256 b, const int mask);

SIMD Floating-Point Exceptions
None

Other Exceptions
See Exceptions Type 4.




');
INSERT INTO "instructions" VALUES('x86_64','BLENDVPD','BLENDVPD - Variable Blend Packed Double Precision Floating-Point Values
 Opcode/                                                   Op/       64/32-bit CPUID      Description
 Instruction                                               En        Mode      Feature
                                                                               Flag
 66 0F 38 15 /r                                            RM0       V/V       SSE4_1     Select packed DP FP values from xmm1 and
 BLENDVPD xmm1, xmm2/m128 , <XMM0>                                                        xmm2 from mask specified in XMM0 and
                                                                                          store the values in xmm1.
 VEX.NDS.128.66.0F3A.W0 4B /r /is4                         RVMR V/V            AVX        Conditionally copy double-precision floating-
 VBLENDVPD xmm1, xmm2, xmm3/m128, xmm4                                                    point values from xmm2 or xmm3/m128 to
                                                                                          xmm1, based on mask bits in the mask
                                                                                          operand, xmm4.
 VEX.NDS.256.66.0F3A.W0 4B /r /is4                         RVMR V/V            AVX        Conditionally copy double-precision floating-
 VBLENDVPD ymm1, ymm2, ymm3/m256, ymm4                                                    point values from ymm2 or ymm3/m256 to
                                                                                          ymm1, based on mask bits in the mask
                                                                                          operand, ymm4.



                                                     Instruction Operand Encoding
    Op/En             Operand 1                        Operand 2                     Operand 3                     Operand 4
    RM0            ModRM:reg (r, w)                  ModRM:r/m (r)                implicit XMM0                        NA
    RVMR            ModRM:reg (w)                     VEX.vvvv (r)                ModRM:r/m (r)                    imm8[7:4]

Description
Conditionally copy each quadword data element of double-precision floating-point value from the second source
operand and the first source operand depending on mask bits defined in the mask register operand. The mask bits
are the most significant bit in each quadword element of the mask register.
Each quadword element of the destination operand is copied from:
.    the corresponding quadword element in the second source operand, if a mask bit is "1"; or
.    the corresponding quadword element in the first source operand, if a mask bit is "0"
The register assignment of the implicit mask operand for BLENDVPD is defined to be the architectural register
XMM0.
128-bit Legacy SSE version: The first source operand and the destination operand is the same. Bits (VLMAX-1:128)
of the corresponding YMM destination register remain unchanged. The mask register operand is implicitly defined
to be the architectural register XMM0. An attempt to execute BLENDVPD with a VEX prefix will cause #UD.
VEX.128 encoded version: The first source operand and the destination operand are XMM registers. The second
source operand is an XMM register or 128-bit memory location. The mask operand is the third source register, and
encoded in bits[7:4] of the immediate byte(imm8). The bits[3:0] of imm8 are ignored. In 32-bit mode, imm8[7] is
ignored. The upper bits (VLMAX-1:128) of the corresponding YMM register (destination register) are zeroed.
VEX.W must be 0, otherwise, the instruction will #UD.
VEX.256 encoded version: The first source operand and destination operand are YMM registers. The second source
operand can be a YMM register or a 256-bit memory location. The mask operand is the third source register, and
encoded in bits[7:4] of the immediate byte(imm8). The bits[3:0] of imm8 are ignored. In 32-bit mode, imm8[7] is
ignored. VEX.W must be 0, otherwise, the instruction will #UD.
VBLENDVPD permits the mask to be any XMM or YMM register. In contrast, BLENDVPD treats XMM0 implicitly as the
mask and do not support non-destructive destination operation.





Operation
BLENDVPD (128-bit Legacy SSE version)
MASK <- XMM0
IF (MASK[63] = 0) THEN DEST[63:0] <- DEST[63:0]
        ELSE DEST [63:0] <- SRC[63:0] FI
IF (MASK[127] = 0) THEN DEST[127:64] <- DEST[127:64]
        ELSE DEST [127:64] <- SRC[127:64] FI
DEST[VLMAX-1:128] (Unmodified)

VBLENDVPD (VEX.128 encoded version)
MASK <- SRC3
IF (MASK[63] = 0) THEN DEST[63:0] <- SRC1[63:0]
        ELSE DEST [63:0] <- SRC2[63:0] FI
IF (MASK[127] = 0) THEN DEST[127:64] <- SRC1[127:64]
        ELSE DEST [127:64] <- SRC2[127:64] FI
DEST[VLMAX-1:128] <- 0

VBLENDVPD (VEX.256 encoded version)
MASK <- SRC3
IF (MASK[63] = 0) THEN DEST[63:0] <- SRC1[63:0]
        ELSE DEST [63:0] <- SRC2[63:0] FI
IF (MASK[127] = 0) THEN DEST[127:64] <- SRC1[127:64]
        ELSE DEST [127:64] <- SRC2[127:64] FI
IF (MASK[191] = 0) THEN DEST[191:128] <- SRC1[191:128]
        ELSE DEST [191:128] <- SRC2[191:128] FI
IF (MASK[255] = 0) THEN DEST[255:192] <- SRC1[255:192]
        ELSE DEST [255:192] <- SRC2[255:192] FI

Intel C/C++ Compiler Intrinsic Equivalent
BLENDVPD:      __m128d _mm_blendv_pd(__m128d v1, __m128d v2, __m128d v3);

VBLENDVPD:     __m128 _mm_blendv_pd (__m128d a, __m128d b, __m128d mask);

VBLENDVPD:     __m256 _mm256_blendv_pd (__m256d a, __m256d b, __m256d mask);

SIMD Floating-Point Exceptions
None

Other Exceptions
See Exceptions Type 4; additionally
#UD                  If VEX.W = 1.




');
INSERT INTO "instructions" VALUES('x86_64','BLENDVPS','BLENDVPS - Variable Blend Packed Single Precision Floating-Point Values
 Opcode/                                                    Op/      64/32-bit CPUID      Description
 Instruction                                                En       Mode      Feature
                                                                               Flag
 66 0F 38 14 /r                                             RM0      V/V       SSE4_1     Select packed single precision floating-point
 BLENDVPS xmm1, xmm2/m128, <XMM0>                                                         values from xmm1 and xmm2/m128 from
                                                                                          mask specified in XMM0 and store the values
                                                                                          into xmm1.
 VEX.NDS.128.66.0F3A.W0 4A /r /is4                          RVMR V/V           AVX        Conditionally copy single-precision floating-
 VBLENDVPS xmm1, xmm2, xmm3/m128, xmm4                                                    point values from xmm2 or xmm3/m128 to
                                                                                          xmm1, based on mask bits in the specified
                                                                                          mask operand, xmm4.
 VEX.NDS.256.66.0F3A.W0 4A /r /is4                          RVMR V/V           AVX        Conditionally copy single-precision floating-
 VBLENDVPS ymm1, ymm2, ymm3/m256, ymm4                                                    point values from ymm2 or ymm3/m256 to
                                                                                          ymm1, based on mask bits in the specified
                                                                                          mask register, ymm4.



                                                     Instruction Operand Encoding
    Op/En             Operand 1                        Operand 2                     Operand 3                      Operand 4
    RM0            ModRM:reg (r, w)                  ModRM:r/m (r)                implicit XMM0                         NA
    RVMR            ModRM:reg (w)                     VEX.vvvv (r)                ModRM:r/m (r)                     imm8[7:4]

Description
Conditionally copy each dword data element of single-precision floating-point value from the second source
operand and the first source operand depending on mask bits defined in the mask register operand. The mask bits
are the most significant bit in each dword element of the mask register.
Each quadword element of the destination operand is copied from:
.    the corresponding dword element in the second source operand, if a mask bit is "1"; or
.    the corresponding dword element in the first source operand, if a mask bit is "0"
The register assignment of the implicit mask operand for BLENDVPS is defined to be the architectural register
XMM0.
128-bit Legacy SSE version: The first source operand and the destination operand is the same. Bits (VLMAX-1:128)
of the corresponding YMM destination register remain unchanged. The mask register operand is implicitly defined
to be the architectural register XMM0. An attempt to execute BLENDVPS with a VEX prefix will cause #UD.
VEX.128 encoded version: The first source operand and the destination operand are XMM registers. The second
source operand is an XMM register or 128-bit memory location. The mask operand is the third source register, and
encoded in bits[7:4] of the immediate byte(imm8). The bits[3:0] of imm8 are ignored. In 32-bit mode, imm8[7] is
ignored. The upper bits (VLMAX-1:128) of the corresponding YMM register (destination register) are zeroed.
VEX.W must be 0, otherwise, the instruction will #UD.
VEX.256 encoded version: The first source operand and destination operand are YMM registers. The second source
operand can be a YMM register or a 256-bit memory location. The mask operand is the third source register, and
encoded in bits[7:4] of the immediate byte(imm8). The bits[3:0] of imm8 are ignored. In 32-bit mode, imm8[7] is
ignored. VEX.W must be 0, otherwise, the instruction will #UD.
VBLENDVPS permits the mask to be any XMM or YMM register. In contrast, BLENDVPS treats XMM0 implicitly as the
mask and do not support non-destructive destination operation.





Operation
BLENDVPS (128-bit Legacy SSE version)
MASK <- XMM0
IF (MASK[31] = 0) THEN DEST[31:0] <- DEST[31:0]
        ELSE DEST [31:0] <- SRC[31:0] FI
IF (MASK[63] = 0) THEN DEST[63:32] <- DEST[63:32]
        ELSE DEST [63:32] <- SRC[63:32] FI
IF (MASK[95] = 0) THEN DEST[95:64] <- DEST[95:64]
        ELSE DEST [95:64] <- SRC[95:64] FI
IF (MASK[127] = 0) THEN DEST[127:96] <- DEST[127:96]
        ELSE DEST [127:96] <- SRC[127:96] FI
DEST[VLMAX-1:128] (Unmodified)

VBLENDVPS (VEX.128 encoded version)
MASK <- SRC3
IF (MASK[31] = 0) THEN DEST[31:0] <- SRC1[31:0]
        ELSE DEST [31:0] <- SRC2[31:0] FI
IF (MASK[63] = 0) THEN DEST[63:32] <- SRC1[63:32]
        ELSE DEST [63:32] <- SRC2[63:32] FI
IF (MASK[95] = 0) THEN DEST[95:64] <- SRC1[95:64]
        ELSE DEST [95:64] <- SRC2[95:64] FI
IF (MASK[127] = 0) THEN DEST[127:96] <- SRC1[127:96]
        ELSE DEST [127:96] <- SRC2[127:96] FI
DEST[VLMAX-1:128] <- 0

VBLENDVPS (VEX.256 encoded version)
MASK <- SRC3
IF (MASK[31] = 0) THEN DEST[31:0] <- SRC1[31:0]
        ELSE DEST [31:0] <- SRC2[31:0] FI
IF (MASK[63] = 0) THEN DEST[63:32] <- SRC1[63:32]
        ELSE DEST [63:32] <- SRC2[63:32] FI
IF (MASK[95] = 0) THEN DEST[95:64] <- SRC1[95:64]
        ELSE DEST [95:64] <- SRC2[95:64] FI
IF (MASK[127] = 0) THEN DEST[127:96] <- SRC1[127:96]
        ELSE DEST [127:96] <- SRC2[127:96] FI
IF (MASK[159] = 0) THEN DEST[159:128] <- SRC1[159:128]
        ELSE DEST [159:128] <- SRC2[159:128] FI
IF (MASK[191] = 0) THEN DEST[191:160] <- SRC1[191:160]
        ELSE DEST [191:160] <- SRC2[191:160] FI
IF (MASK[223] = 0) THEN DEST[223:192] <- SRC1[223:192]
        ELSE DEST [223:192] <- SRC2[223:192] FI
IF (MASK[255] = 0) THEN DEST[255:224] <- SRC1[255:224]
        ELSE DEST [255:224] <- SRC2[255:224] FI

Intel C/C++ Compiler Intrinsic Equivalent
BLENDVPS:      __m128 _mm_blendv_ps(__m128 v1, __m128 v2, __m128 v3);

VBLENDVPS:     __m128 _mm_blendv_ps (__m128 a, __m128 b, __m128 mask);

VBLENDVPS:     __m256 _mm256_blendv_ps (__m256 a, __m256 b, __m256 mask);

SIMD Floating-Point Exceptions
None




Other Exceptions
See Exceptions Type 4; additionally
#UD                       If VEX.W = 1.




');
INSERT INTO "instructions" VALUES('x86_64','BLSI','BLSI - Extract Lowest Set Isolated Bit
 Opcode/Instruction                 Op/   64/32     CPUID     Description
                                    En    -bit      Feature
                                          Mode      Flag
 VEX.NDD.LZ.0F38.W0 F3 /3           VM    V/V       BMI1      Extract lowest set bit from r/m32 and set that bit in r32.
 BLSI r32, r/m32
 VEX.NDD.LZ.0F38.W1 F3 /3           VM    V/N.E.    BMI1      Extract lowest set bit from r/m64, and set that bit in r64.
 BLSI r64, r/m64



                                                   Instruction Operand Encoding
   Op/En               Operand 1                     Operand 2                    Operand 3                       Operand 4
    VM               VEX.vvvv (w)                  ModRM:r/m (r)                      NA                              NA

Description
Extracts the lowest set bit from the source operand and set the corresponding bit in the destination register. All
other bits in the destination operand are zeroed. If no bits are set in the source operand, BLSI sets all the bits in
the destination to 0 and sets ZF and CF.
This instruction is not supported in real mode and virtual-8086 mode. The operand size is always 32 bits if not in
64-bit mode. In 64-bit mode operand size 64 requires VEX.W1. VEX.W1 is ignored in non-64-bit modes. An
attempt to execute this instruction with VEX.L not equal to 0 will cause #UD.

Operation
temp <- (-SRC) bitwiseAND (SRC);
SF <- temp[OperandSize -1];
ZF <- (temp = 0);
IF SRC = 0
    CF <- 0;
ELSE
    CF <- 1;
FI
DEST <- temp;

Flags Affected
ZF and SF are updated based on the result. CF is set if the source is not zero. OF flags are cleared. AF and PF
flags are undefined.

Intel C/C++ Compiler Intrinsic Equivalent
BLSI:          unsigned __int32 _blsi_u32(unsigned __int32 src);

BLSI:          unsigned __int64 _blsi_u64(unsigned __int64 src);

SIMD Floating-Point Exceptions
None

Other Exceptions
See Section 2.5.1, "Exception Conditions for VEX-Encoded GPR Instructions", Table 2-29; additionally
#UD                      If VEX.W = 1.




');
INSERT INTO "instructions" VALUES('x86_64','BLSMSK','BLSMSK - Get Mask Up to Lowest Set Bit
 Opcode/Instruction                 Op/   64/32     CPUID       Description
                                    En    -bit      Feature
                                          Mode      Flag
 VEX.NDD.LZ.0F38.W0 F3 /2           VM    V/V       BMI1        Set all lower bits in r32 to "1" starting from bit 0 to lowest set bit in
 BLSMSK r32, r/m32                                              r/m32.
 VEX.NDD.LZ.0F38.W1 F3 /2           VM    V/N.E.    BMI1        Set all lower bits in r64 to "1" starting from bit 0 to lowest set bit in
 BLSMSK r64, r/m64                                              r/m64.



                                                   Instruction Operand Encoding
   Op/En              Operand 1                     Operand 2                        Operand 3                        Operand 4
    VM               VEX.vvvv (w)                  ModRM:r/m (r)                         NA                               NA

Description
Sets all the lower bits of the destination operand to "1" up to and including lowest set bit (=1) in the source
operand. If source operand is zero, BLSMSK sets all bits of the destination operand to 1 and also sets CF to 1.
This instruction is not supported in real mode and virtual-8086 mode. The operand size is always 32 bits if not in
64-bit mode. In 64-bit mode operand size 64 requires VEX.W1. VEX.W1 is ignored in non-64-bit modes. An
attempt to execute this instruction with VEX.L not equal to 0 will cause #UD.

Operation
temp <- (SRC-1) XOR (SRC) ;
SF <- temp[OperandSize -1];
ZF <- 0;
IF SRC = 0
    CF <- 1;
ELSE
    CF <- 0;
FI
DEST <- temp;

Flags Affected
SF is updated based on the result. CF is set if the source if zero. ZF and OF flags are cleared. AF and PF flag are
undefined.

Intel C/C++ Compiler Intrinsic Equivalent
BLSMSK:       unsigned __int32 _blsmsk_u32(unsigned __int32 src);

BLSMSK:       unsigned __int64 _blsmsk_u64(unsigned __int64 src);

SIMD Floating-Point Exceptions
None

Other Exceptions
See Section 2.5.1, "Exception Conditions for VEX-Encoded GPR Instructions", Table 2-29; additionally
#UD                     If VEX.W = 1.




');
INSERT INTO "instructions" VALUES('x86_64','BLSR','BLSR - Reset Lowest Set Bit
 Opcode/Instruction               Op/   64/32     CPUID        Description
                                  En    -bit      Feature
                                        Mode      Flag
 VEX.NDD.LZ.0F38.W0 F3 /1         VM    V/V       BMI1         Reset lowest set bit of r/m32, keep all other bits of r/m32 and write
 BLSR r32, r/m32                                               result to r32.
 VEX.NDD.LZ.0F38.W1 F3 /1         VM    V/N.E.    BMI1         Reset lowest set bit of r/m64, keep all other bits of r/m64 and write
 BLSR r64, r/m64                                               result to r64.



                                                 Instruction Operand Encoding
   Op/En              Operand 1                    Operand 2                       Operand 3                      Operand 4
    VM              VEX.vvvv (w)                 ModRM:r/m (r)                        NA                              NA

Description
Copies all bits from the source operand to the destination operand and resets (=0) the bit position in the destina-
tion operand that corresponds to the lowest set bit of the source operand. If the source operand is zero BLSR sets
CF.
This instruction is not supported in real mode and virtual-8086 mode. The operand size is always 32 bits if not in
64-bit mode. In 64-bit mode operand size 64 requires VEX.W1. VEX.W1 is ignored in non-64-bit modes. An
attempt to execute this instruction with VEX.L not equal to 0 will cause #UD.

Operation
temp <- (SRC-1) bitwiseAND ( SRC );
SF <- temp[OperandSize -1];
ZF <- (temp = 0);
IF SRC = 0
    CF <- 1;
ELSE
    CF <- 0;
FI
DEST <- temp;

Flags Affected
ZF and SF flags are updated based on the result. CF is set if the source is zero. OF flag is cleared. AF and PF flags
are undefined.

Intel C/C++ Compiler Intrinsic Equivalent
BLSR:          unsigned __int32 _blsr_u32(unsigned __int32 src);

BLSR:          unsigned __int64 _blsr_u64(unsigned __int64 src);

SIMD Floating-Point Exceptions
None

Other Exceptions
See Section 2.5.1, "Exception Conditions for VEX-Encoded GPR Instructions", Table 2-29; additionally
#UD                     If VEX.W = 1.




');
INSERT INTO "instructions" VALUES('x86_64','BNDCL','BNDCL-Check Lower Bound
 Opcode/                       Op/En      64/32       CPUID     Description
 Instruction                              bit Mode    Feature
                                          Support     Flag
 F3 0F 1A /r                   RM         NE/V        MPX       Generate a #BR if the address in r/m32 is lower than the lower
 BNDCL bnd, r/m32                                               bound in bnd.LB.
 F3 0F 1A /r                   RM         V/NE        MPX       Generate a #BR if the address in r/m64 is lower than the lower
 BNDCL bnd, r/m64                                               bound in bnd.LB.



                                                 Instruction Operand Encoding
      Op/En                Operand 1                             Operand 2                               Operand 3
       RM                 ModRM:reg (w)                         ModRM:r/m (r)                               NA

Description
Compare the address in the second operand with the lower bound in bnd. The second operand can be either a
register or memory operand. If the address is lower than the lower bound in bnd.LB, it will set BNDSTATUS to 01H
and signal a #BR exception.
This instruction does not cause any memory access, and does not read or write any flags.

Operation
BNDCL BND, reg
IF reg < BND.LB Then
    BNDSTATUS <- 01H;
    #BR;
FI;

BNDCL BND, mem
TEMP <- LEA(mem);
IF TEMP < BND.LB Then
    BNDSTATUS <- 01H;
    #BR;
FI;

Intel C/C++ Compiler Intrinsic Equivalent
BNDCL void _bnd_chk_ptr_lbounds(const void *q)

Flags Affected
None

Protected Mode Exceptions
#BR                   If lower bound check fails.
#UD                   If the LOCK prefix is used.
                      If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                      If 67H prefix is not used and CS.D=0.
                      If 67H prefix is used and CS.D=1.





Real-Address Mode Exceptions
#BR                 If lower bound check fails.
#UD                 If the LOCK prefix is used.
                    If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                    If 16-bit addressing is used.

Virtual-8086 Mode Exceptions
#BR                 If lower bound check fails.
#UD                 If the LOCK prefix is used.
                    If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                    If 16-bit addressing is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#UD                 If ModRM.r/m and REX encodes BND4-BND15 when Intel MPX is enabled.
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','BNDCU','BNDCU/BNDCN-Check Upper Bound
 Opcode/                        Op/En       64/32       CPUID     Description
 Instruction                                bit Mode    Feature
                                            Support     Flag
 F2 0F 1A /r                    RM          NE/V        MPX       Generate a #BR if the address in r/m32 is higher than the upper
 BNDCU bnd, r/m32                                                 bound in bnd.UB (bnb.UB in 1''s complement form).
 F2 0F 1A /r                    RM          V/NE        MPX       Generate a #BR if the address in r/m64 is higher than the upper
 BNDCU bnd, r/m64                                                 bound in bnd.UB (bnb.UB in 1''s complement form).
 F2 0F 1B /r                    RM          NE/V        MPX       Generate a #BR if the address in r/m32 is higher than the upper
 BNDCN bnd, r/m32                                                 bound in bnd.UB (bnb.UB not in 1''s complement form).
 F2 0F 1B /r                    RM          V/NE        MPX       Generate a #BR if the address in r/m64 is higher than the upper
 BNDCN bnd, r/m64                                                 bound in bnd.UB (bnb.UB not in 1''s complement form).



                                                   Instruction Operand Encoding
    Op/En                    Operand 1                             Operand 2                              Operand 3
      RM                    ModRM:reg (w)                         ModRM:r/m (r)                               NA

Description
Compare the address in the second operand with the upper bound in bnd. The second operand can be either a
register or a memory operand. If the address is higher than the upper bound in bnd.UB, it will set BNDSTATUS to
01H and signal a #BR exception.
BNDCU perform 1''s complement operation on the upper bound of bnd first before proceeding with address compar-
ison. BNDCN perform address comparison directly using the upper bound in bnd that is already reverted out of 1''s
complement form.
This instruction does not cause any memory access, and does not read or write any flags.
Effective address computation of m32/64 has identical behavior to LEA

Operation
BNDCU BND, reg
IF reg > NOT(BND.UB) Then
    BNDSTATUS <- 01H;
    #BR;
FI;

BNDCU BND, mem
TEMP <- LEA(mem);
IF TEMP > NOT(BND.UB) Then
    BNDSTATUS <- 01H;
    #BR;
FI;

BNDCN BND, reg
IF reg > BND.UB Then
    BNDSTATUS <- 01H;
    #BR;
FI;





BNDCN BND, mem
TEMP <- LEA(mem);
IF TEMP > BND.UB Then
    BNDSTATUS <- 01H;
    #BR;
FI;

Intel C/C++ Compiler Intrinsic Equivalent
BNDCU .void _bnd_chk_ptr_ubounds(const void *q)

Flags Affected
None

Protected Mode Exceptions
#BR                  If upper bound check fails.
#UD                  If the LOCK prefix is used.
                     If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                     If 67H prefix is not used and CS.D=0.
                     If 67H prefix is used and CS.D=1.

Real-Address Mode Exceptions
#BR                  If upper bound check fails.
#UD                  If the LOCK prefix is used.
                     If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                     If 16-bit addressing is used.

Virtual-8086 Mode Exceptions
#BR                  If upper bound check fails.
#UD                  If the LOCK prefix is used.
                     If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                     If 16-bit addressing is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#UD                  If ModRM.r/m and REX encodes BND4-BND15 when Intel MPX is enabled.
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','BNDCN','-R:BNDCU');
INSERT INTO "instructions" VALUES('x86_64','BNDLDX','BNDLDX-Load Extended Bounds Using Address Translation
 Opcode/                          Op/En     64/32       CPUID     Description
 Instruction                                bit Mode    Feature
                                            Support     Flag
 0F 1A /r                         RM        V/V         MPX       Load the bounds stored in a bound table entry (BTE) into bnd with
 BNDLDX bnd, mib                                                  address translation using the base of mib and conditional on the
                                                                  index of mib matching the pointer value in the BTE.



                                                 Instruction Operand Encoding
     Op/En                    Operand 1                            Operand 2                               Operand 3
                                                         SIB.base (r): Address of pointer
      RM                    ModRM:reg (w)                                                                     NA
                                                                   SIB.index(r)

Description
BNDLDX uses the linear address constructed from the base register and displacement of the SIB-addressing form
of the memory operand (mib) to perform address translation to access a bound table entry and conditionally load
the bounds in the BTE to the destination. The destination register is updated with the bounds in the BTE, if the
content of the index register of mib matches the pointer value stored in the BTE.
If the pointer value comparison fails, the destination is updated with INIT bounds (lb = 0x0, ub = 0x0) (note: as
articulated earlier, the upper bound is represented using 1''s complement, therefore, the 0x0 value of upper bound
allows for access to full memory).
This instruction does not cause memory access to the linear address of mib nor the effective address referenced by
the base, and does not read or write any flags.
Segment overrides apply to the linear address computation with the base of mib, and are used during address
translation to generate the address of the bound table entry. By default, the address of the BTE is assumed to be
linear address. There are no segmentation checks performed on the base of mib.
The base of mib will not be checked for canonical address violation as it does not access memory.
Any encoding of this instruction that does not specify base or index register will treat those registers as zero
(constant). The reg-reg form of this instruction will remain a NOP.
The scale field of the SIB byte has no effect on these instructions and is ignored.
The bound register may be partially updated on memory faults. The order in which memory operands are loaded is
implementation specific.

Operation
base <- mib.SIB.base ? mib.SIB.base + Disp: 0;
ptr_value <- mib.SIB.index ? mib.SIB.index : 0;

Outside 64-bit mode
A_BDE[31:0] <- (Zero_extend32(base[31:12] << 2) + (BNDCFG[31:12] <<12 );
A_BT[31:0] <- LoadFrom(A_BDE );
IF A_BT[0] equal 0 Then
    BNDSTATUS <- A_BDE | 02H;
    #BR;
FI;
A_BTE[31:0] <- (Zero_extend32(base[11:2] << 4) + (A_BT[31:2] << 2 );
Temp_lb[31:0] <- LoadFrom(A_BTE);
Temp_ub[31:0] <- LoadFrom(A_BTE + 4);
Temp_ptr[31:0] <- LoadFrom(A_BTE + 8);
IF Temp_ptr equal ptr_value Then
    BND.LB <- Temp_lb;
    BND.UB <- Temp_ub;



ELSE
    BND.LB <- 0;
    BND.UB <- 0;
FI;

In 64-bit mode
A_BDE[63:0] <- (Zero_extend64(base[47+MAWA:20] << 3) + (BNDCFG[63:20] <<12 );1
A_BT[63:0] <- LoadFrom(A_BDE);
IF A_BT[0] equal 0 Then
    BNDSTATUS <- A_BDE | 02H;
    #BR;
FI;
A_BTE[63:0] <- (Zero_extend64(base[19:3] << 5) + (A_BT[63:3] << 3 );
Temp_lb[63:0] <- LoadFrom(A_BTE);
Temp_ub[63:0] <- LoadFrom(A_BTE + 8);
Temp_ptr[63:0] <- LoadFrom(A_BTE + 16);
IF Temp_ptr equal ptr_value Then
    BND.LB <- Temp_lb;
    BND.UB <- Temp_ub;
ELSE
    BND.LB <- 0;
    BND.UB <- 0;
FI;

Intel C/C++ Compiler Intrinsic Equivalent
BNDLDX: Generated by compiler as needed.

Flags Affected
None

Protected Mode Exceptions
#BR                    If the bound directory entry is invalid.
#UD                    If the LOCK prefix is used.
                       If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                       If 67H prefix is not used and CS.D=0.
                       If 67H prefix is used and CS.D=1.
#GP(0)                 If a destination effective address of the Bound Table entry is outside the DS segment limit.
                       If DS register contains a NULL segment selector.
#PF(fault code)        If a page fault occurs.

Real-Address Mode Exceptions
#UD                    If the LOCK prefix is used.
                       If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                       If 16-bit addressing is used.
#GP(0)                 If a destination effective address of the Bound Table entry is outside the DS segment limit.




1. If CPL < 3, the supervisor MAWA (MAWAS) is used; this value is 0. If CPL = 3, the user MAWA (MAWAU) is used; this value is enumer-
   ated in CPUID.(EAX=07H,ECX=0H):ECX.MAWAU[bits 21:17]. See Section 17.3.1 of Intel 64 and IA-32 Architectures Software Devel-
   oper''s Manual, Volume 1.



Virtual-8086 Mode Exceptions
#UD                     If the LOCK prefix is used.
                        If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                        If 16-bit addressing is used.
#GP(0)                  If a destination effective address of the Bound Table entry is outside the DS segment limit.
#PF(fault code)         If a page fault occurs.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#BR                     If the bound directory entry is invalid.
#UD                     If ModRM is RIP relative.
                        If the LOCK prefix is used.
                        If ModRM.r/m and REX encodes BND4-BND15 when Intel MPX is enabled.
#GP(0)                  If the memory address (A_BDE or A_BTE) is in a non-canonical form.
#PF(fault code)         If a page fault occurs.




');
INSERT INTO "instructions" VALUES('x86_64','BNDMK','BNDMK-Make Bounds
 Opcode/                        Op/En     64/32       CPUID      Description
 Instruction                              bit Mode    Feature
                                          Support     Flag
 F3 0F 1B /r                    RM        NE/V        MPX        Make lower and upper bounds from m32 and store them in bnd.
 BNDMK bnd, m32
 F3 0F 1B /r                    RM        V/NE        MPX        Make lower and upper bounds from m64 and store them in bnd.
 BNDMK bnd, m64



                                                 Instruction Operand Encoding
    Op/En                   Operand 1                            Operand 2                             Operand 3
       RM                ModRM:reg (w)                          ModRM:r/m (r)                             NA

Description
Makes bounds from the second operand and stores the lower and upper bounds in the bound register bnd. The
second operand must be a memory operand. The content of the base register from the memory operand is stored
in the lower bound bnd.LB. The 1''s complement of the effective address of m32/m64 is stored in the upper bound
b.UB. Computation of m32/m64 has identical behavior to LEA.
This instruction does not cause any memory access, and does not read or write any flags.
If the instruction did not specify base register, the lower bound will be zero. The reg-reg form of this instruction
retains legacy behavior (NOP).
RIP relative instruction in 64-bit will #UD.

Operation
BND.LB <- SRCMEM.base;
IF 64-bit mode Then
    BND.UB <- NOT(LEA.64_bits(SRCMEM));
ELSE
    BND.UB <- Zero_Extend.64_bits(NOT(LEA.32_bits(SRCMEM)));
FI;

Intel C/C++ Compiler Intrinsic Equivalent
BNDMKvoid * _bnd_set_ptr_bounds(const void * q, size_t size);

Flags Affected
None

Protected Mode Exceptions
#UD                    If ModRM is RIP relative.
                       If the LOCK prefix is used.
                       If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                       If 67H prefix is not used and CS.D=0.
                       If 67H prefix is used and CS.D=1.

Real-Address Mode Exceptions
#UD                    If ModRM is RIP relative.
                       If the LOCK prefix is used.
                       If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                       If 16-bit addressing is used.




Virtual-8086 Mode Exceptions
#UD                 If ModRM is RIP relative.
                    If the LOCK prefix is used.
                    If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                    If 16-bit addressing is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#UD                 If ModRM.r/m and REX encodes BND4-BND15 when Intel MPX is enabled.
#SS(0)              If the memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','BNDMOV','BNDMOV-Move Bounds
 Opcode/                          Op/En     64/32        CPUID          Description
 Instruction                                bit Mode     Feature
                                            Support      Flag
 66 0F 1A /r                      RM        NE/V         MPX            Move lower and upper bound from bnd2/m64 to bound register
 BNDMOV bnd1, bnd2/m64                                                  bnd1.
 66 0F 1A /r                      RM        V/NE         MPX            Move lower and upper bound from bnd2/m128 to bound register
 BNDMOV bnd1, bnd2/m128                                                 bnd1.
 66 0F 1B /r                      MR        NE/V         MPX            Move lower and upper bound from bnd2 to bnd1/m64.
 BNDMOV bnd1/m64, bnd2
 66 0F 1B /r                      MR        V/NE         MPX            Move lower and upper bound from bnd2 to bound register
 BNDMOV bnd1/m128, bnd2                                                 bnd1/m128.



                                                   Instruction Operand Encoding
     Op/En                     Operand 1                                  Operand 2                                Operand 3
      RM                    ModRM:reg (w)                            ModRM:r/m (r)                                         NA
      MR                    ModRM:r/m (w)                               ModRM:reg (r)                                      NA

Description
BNDMOV moves a pair of lower and upper bound values from the source operand (the second operand) to the
destination (the first operand). Each operation is 128-bit move. The exceptions are same as the MOV instruction.
The memory format for loading/store bounds in 64-bit mode is shown in Figure 3-5.




                                                                                                BNDMOV to memory in 64-bit mode
                       Upper Bound (UB)                  Lower Bound (LB)


                    16                                                  8                                          0     Byte offset




                                                                                              BNDMOV to memory in 32-bit mode
                  Upper Bound (UB)                     Lower Bound (LB)


                  16                                                8                     4                    0       Byte offset




                                  Figure 3-5. Memory Layout of BNDMOV to/from Memory


This instruction does not change flags.

Operation
BNDMOV register to register
DEST.LB <- SRC.LB;
DEST.UB <- SRC.UB;





BNDMOV from memory
IF 64-bit mode THEN
         DEST.LB <- LOAD_QWORD(SRC);
         DEST.UB <- LOAD_QWORD(SRC+8);
    ELSE
         DEST.LB <- LOAD_DWORD_ZERO_EXT(SRC);
         DEST.UB <- LOAD_DWORD_ZERO_EXT(SRC+4);
FI;

BNDMOV to memory
IF 64-bit mode THEN
         DEST[63:0] <- SRC.LB;
         DEST[127:64] <- SRC.UB;
    ELSE
         DEST[31:0] <- SRC.LB;
         DEST[63:32] <- SRC.UB;
FI;

Intel C/C++ Compiler Intrinsic Equivalent
BNDMOV         void * _bnd_copy_ptr_bounds(const void *q, const void *r)

Flags Affected
None

Protected Mode Exceptions
#UD                   If the LOCK prefix is used but the destination is not a memory operand.
                      If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                      If 67H prefix is not used and CS.D=0.
                      If 67H prefix is used and CS.D=1.
#SS(0)                If the memory operand effective address is outside the SS segment limit.
#GP(0)                If the memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                      If the destination operand points to a non-writable segment
                      If the DS, ES, FS, or GS segment register contains a NULL segment selector.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made while CPL is 3.
#PF(fault code)       If a page fault occurs.

Real-Address Mode Exceptions
#UD                   If the LOCK prefix is used but the destination is not a memory operand.
                      If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                      If 16-bit addressing is used.
#GP(0)                If the memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                   If the memory operand effective address is outside the SS segment limit.





Virtual-8086 Mode Exceptions
#UD                 If the LOCK prefix is used but the destination is not a memory operand.
                    If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                    If 16-bit addressing is used.
#GP(0)              If the memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If the memory operand effective address is outside the SS segment limit.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while CPL is 3.
#PF(fault code)     If a page fault occurs.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#UD                 If the LOCK prefix is used but the destination is not a memory operand.
                    If ModRM.r/m and REX encodes BND4-BND15 when Intel MPX is enabled.
#SS(0)              If the memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while CPL is 3.
#PF(fault code)     If a page fault occurs.




');
INSERT INTO "instructions" VALUES('x86_64','BNDSTX','BNDSTX-Store Extended Bounds Using Address Translation
 Opcode/                          Op/En     64/32        CPUID     Description
 Instruction                                bit Mode     Feature
                                            Support      Flag
 0F 1B /r                         MR        V/V          MPX       Store the bounds in bnd and the pointer value in the index regis-
 BNDSTX mib, bnd                                                   ter of mib to a bound table entry (BTE) with address translation
                                                                   using the base of mib.



                                                   Instruction Operand Encoding
     Op/En                     Operand 1                             Operand 2                               Operand 3
                     SIB.base (r): Address of pointer
      MR                                                           ModRM:reg (r)                                 NA
                               SIB.index(r)

Description
BNDSTX uses the linear address constructed from the displacement and base register of the SIB-addressing form
of the memory operand (mib) to perform address translation to store to a bound table entry. The bounds in the
source operand bnd are written to the lower and upper bounds in the BTE. The content of the index register of mib
is written to the pointer value field in the BTE.
This instruction does not cause memory access to the linear address of mib nor the effective address referenced by
the base, and does not read or write any flags.
Segment overrides apply to the linear address computation with the base of mib, and are used during address
translation to generate the address of the bound table entry. By default, the address of the BTE is assumed to be
linear address. There are no segmentation checks performed on the base of mib.
The base of mib will not be checked for canonical address violation as it does not access memory.
Any encoding of this instruction that does not specify base or index register will treat those registers as zero
(constant). The reg-reg form of this instruction will remain a NOP.
The scale field of the SIB byte has no effect on these instructions and is ignored.
The bound register may be partially updated on memory faults. The order in which memory operands are loaded is
implementation specific.

Operation
base <- mib.SIB.base ? mib.SIB.base + Disp: 0;
ptr_value <- mib.SIB.index ? mib.SIB.index : 0;

Outside 64-bit mode
A_BDE[31:0] <- (Zero_extend32(base[31:12] << 2) + (BNDCFG[31:12] <<12 );
A_BT[31:0] <- LoadFrom(A_BDE);
IF A_BT[0] equal 0 Then
    BNDSTATUS <- A_BDE | 02H;
    #BR;
FI;
A_DEST[31:0] <- (Zero_extend32(base[11:2] << 4) + (A_BT[31:2] << 2 ); // address of Bound table entry
A_DEST[8][31:0] <- ptr_value;
A_DEST[0][31:0] <- BND.LB;
A_DEST[4][31:0] <- BND.UB;





In 64-bit mode
A_BDE[63:0] <- (Zero_extend64(base[47+MAWA:20] << 3) + (BNDCFG[63:20] <<12 );1
A_BT[63:0] <- LoadFrom(A_BDE);
IF A_BT[0] equal 0 Then
    BNDSTATUS <- A_BDE | 02H;
    #BR;
FI;
A_DEST[63:0] <- (Zero_extend64(base[19:3] << 5) + (A_BT[63:3] << 3 ); // address of Bound table entry
A_DEST[16][63:0] <- ptr_value;
A_DEST[0][63:0] <- BND.LB;
A_DEST[8][63:0] <- BND.UB;

Intel C/C++ Compiler Intrinsic Equivalent
BNDSTX: _bnd_store_ptr_bounds(const void **ptr_addr, const void *ptr_val);

Flags Affected
None

Protected Mode Exceptions
#BR                    If the bound directory entry is invalid.
#UD                    If the LOCK prefix is used.
                       If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                       If 67H prefix is not used and CS.D=0.
                       If 67H prefix is used and CS.D=1.
#GP(0)                 If a destination effective address of the Bound Table entry is outside the DS segment limit.
                       If DS register contains a NULL segment selector.
                       If the destination operand points to a non-writable segment
#PF(fault code)        If a page fault occurs.

Real-Address Mode Exceptions
#UD                    If the LOCK prefix is used.
                       If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                       If 16-bit addressing is used.
#GP(0)                 If a destination effective address of the Bound Table entry is outside the DS segment limit.

Virtual-8086 Mode Exceptions
#UD                    If the LOCK prefix is used.
                       If ModRM.r/m encodes BND4-BND7 when Intel MPX is enabled.
                       If 16-bit addressing is used.
#GP(0)                 If a destination effective address of the Bound Table entry is outside the DS segment limit.
#PF(fault code)        If a page fault occurs.

Compatibility Mode Exceptions
Same exceptions as in protected mode.




1. If CPL < 3, the supervisor MAWA (MAWAS) is used; this value is 0. If CPL = 3, the user MAWA (MAWAU) is used; this value is enumer-
   ated in CPUID.(EAX=07H,ECX=0H):ECX.MAWAU[bits 21:17]. See Section 17.3.1 of Intel 64 and IA-32 Architectures Software Devel-
   oper''s Manual, Volume 1.



64-Bit Mode Exceptions
#BR                     If the bound directory entry is invalid.
#UD                     If ModRM is RIP relative.
                        If the LOCK prefix is used.
                        If ModRM.r/m and REX encodes BND4-BND15 when Intel MPX is enabled.
#GP(0)                  If the memory address (A_BDE or A_BTE) is in a non-canonical form.
                        If the destination operand points to a non-writable segment
#PF(fault code)         If a page fault occurs.




');
INSERT INTO "instructions" VALUES('x86_64','BOUND','BOUND-Check Array Index Against Bounds
Opcode                  Instruction                  Op/   64-bit    Compat/ Description
                                                     En    Mode      Leg Mode
62 /r                   BOUND r16, m16&16            RM    Invalid   Valid     Check if r16 (array index) is within bounds
                                                                               specified by m16&16.
62 /r                   BOUND r32, m32&32            RM    Invalid   Valid     Check if r32 (array index) is within bounds
                                                                               specified by m32&32.



                                              Instruction Operand Encoding
  Op/En             Operand 1                      Operand 2                 Operand 3                    Operand 4
    RM             ModRM:reg (r)                  ModRM:r/m (r)                 NA                            NA

Description
BOUND determines if the first operand (array index) is within the bounds of an array specified the second operand
(bounds operand). The array index is a signed integer located in a register. The bounds operand is a memory loca-
tion that contains a pair of signed doubleword-integers (when the operand-size attribute is 32) or a pair of signed
word-integers (when the operand-size attribute is 16). The first doubleword (or word) is the lower bound of the
array and the second doubleword (or word) is the upper bound of the array. The array index must be greater than
or equal to the lower bound and less than or equal to the upper bound plus the operand size in bytes. If the index
is not within bounds, a BOUND range exceeded exception (#BR) is signaled. When this exception is generated, the
saved return instruction pointer points to the BOUND instruction.
The bounds limit data structure (two words or doublewords containing the lower and upper limits of the array) is
usually placed just before the array itself, making the limits addressable via a constant offset from the beginning of
the array. Because the address of the array already will be present in a register, this practice avoids extra bus cycles
to obtain the effective address of the array bounds.
This instruction executes as described in compatibility mode and legacy mode. It is not valid in 64-bit mode.

Operation
IF 64bit Mode
    THEN
         #UD;
    ELSE
         IF (ArrayIndex < LowerBound OR ArrayIndex > UpperBound)
         (* Below lower bound or above upper bound *)
              THEN #BR; FI;
FI;

Flags Affected
None.

Protected Mode Exceptions
#BR                   If the bounds test fails.
#UD                   If second operand is not a memory location.
                      If the LOCK prefix is used.
#GP(0)                If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                      If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)                If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)       If a page fault occurs.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made while the
                      current privilege level is 3.



Real-Address Mode Exceptions
#BR                    If the bounds test fails.
#UD                    If second operand is not a memory location.
                       If the LOCK prefix is used.
#GP                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                    If a memory operand effective address is outside the SS segment limit.

Virtual-8086 Mode Exceptions
#BR                    If the bounds test fails.
#UD                    If second operand is not a memory location.
                       If the LOCK prefix is used.
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#UD                    If in 64-bit mode.




');
INSERT INTO "instructions" VALUES('x86_64','BSF','BSF-Bit Scan Forward
 Opcode                   Instruction                 Op/   64-bit   Compat/ Description
                                                      En    Mode     Leg Mode
 0F BC /r                 BSF r16, r/m16              RM    Valid    Valid       Bit scan forward on r/m16.
 0F BC /r                 BSF r32, r/m32              RM    Valid    Valid       Bit scan forward on r/m32.
 REX.W + 0F BC /r         BSF r64, r/m64              RM    Valid    N.E.        Bit scan forward on r/m64.



                                              Instruction Operand Encoding
  Op/En              Operand 1                     Operand 2                 Operand 3                    Operand 4
    RM              ModRM:reg (w)                 ModRM:r/m (r)                 NA                              NA

Description
Searches the source operand (second operand) for the least significant set bit (1 bit). If a least significant 1 bit is
found, its bit index is stored in the destination operand (first operand). The source operand can be a register or a
memory location; the destination operand is a register. The bit index is an unsigned offset from bit 0 of the source
operand. If the content of the source operand is 0, the content of the destination operand is undefined.
In 64-bit mode, the instruction''s default operation size is 32 bits. Using a REX prefix in the form of REX.R permits
access to additional registers (R8-R15). Using a REX prefix in the form of REX.W promotes operation to 64 bits. See
the summary chart at the beginning of this section for encoding data and limits.

Operation
IF SRC = 0
    THEN
        ZF <- 1;
        DEST is undefined;
    ELSE
        ZF <- 0;
        temp <- 0;
        WHILE Bit(SRC, temp) = 0
        DO
            temp <- temp + 1;
        OD;
        DEST <- temp;
FI;

Flags Affected
The ZF flag is set to 1 if all the source operand is 0; otherwise, the ZF flag is cleared. The CF, OF, SF, AF, and PF, flags
are undefined.

Protected Mode Exceptions
#GP(0)                  If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                        If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)                  If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)         If a page fault occurs.
#AC(0)                  If alignment checking is enabled and an unaligned memory reference is made while the
                        current privilege level is 3.
#UD                     If the LOCK prefix is used.





Real-Address Mode Exceptions
#GP                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                    If a memory operand effective address is outside the SS segment limit.
#UD                    If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made.
#UD                    If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)                 If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)                 If the memory address is in a non-canonical form.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                    If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','BSR','BSR-Bit Scan Reverse
Opcode                   Instruction                  Op/     64-bit   Compat/ Description
                                                      En      Mode     Leg Mode
0F BD /r                 BSR r16, r/m16               RM      Valid    Valid      Bit scan reverse on r/m16.
0F BD /r                 BSR r32, r/m32               RM      Valid    Valid      Bit scan reverse on r/m32.
REX.W + 0F BD /r         BSR r64, r/m64               RM      Valid    N.E.       Bit scan reverse on r/m64.



                                              Instruction Operand Encoding
  Op/En             Operand 1                     Operand 2                    Operand 3                   Operand 4
   RM              ModRM:reg (w)                 ModRM:r/m (r)                    NA                              NA

Description
Searches the source operand (second operand) for the most significant set bit (1 bit). If a most significant 1 bit is
found, its bit index is stored in the destination operand (first operand). The source operand can be a register or a
memory location; the destination operand is a register. The bit index is an unsigned offset from bit 0 of the source
operand. If the content source operand is 0, the content of the destination operand is undefined.
In 64-bit mode, the instruction''s default operation size is 32 bits. Using a REX prefix in the form of REX.R permits
access to additional registers (R8-R15). Using a REX prefix in the form of REX.W promotes operation to 64 bits. See
the summary chart at the beginning of this section for encoding data and limits.

Operation
IF SRC = 0
    THEN
        ZF <- 1;
        DEST is undefined;
    ELSE
        ZF <- 0;
        temp <- OperandSize - 1;
        WHILE Bit(SRC, temp) = 0
        DO
            temp <- temp - 1;
        OD;
        DEST <- temp;
FI;

Flags Affected
The ZF flag is set to 1 if all the source operand is 0; otherwise, the ZF flag is cleared. The CF, OF, SF, AF, and PF, flags
are undefined.

Protected Mode Exceptions
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                       If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                    If the LOCK prefix is used.





Real-Address Mode Exceptions
#GP                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                    If a memory operand effective address is outside the SS segment limit.
#UD                    If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made.
#UD                    If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)                 If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)                 If the memory address is in a non-canonical form.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                    If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','BSWAP','BSWAP-Byte Swap
Opcode                    Instruction                   Op/   64-bit   Compat/ Description
                                                        En    Mode     Leg Mode
0F C8+rd                  BSWAP r32                     O     Valid*   Valid       Reverses the byte order of a 32-bit register.
REX.W + 0F C8+rd          BSWAP r64                     O     Valid    N.E.        Reverses the byte order of a 64-bit register.
NOTES:
* See IA-32 Architecture Compatibility section below.



                                                Instruction Operand Encoding
  Op/En             Operand 1                     Operand 2                    Operand 3                    Operand 4
    O            opcode + rd (r, w)                     NA                        NA                            NA

Description
Reverses the byte order of a 32-bit or 64-bit (destination) register. This instruction is provided for converting little-
endian values to big-endian format and vice versa. To swap bytes in a word value (16-bit register), use the XCHG
instruction. When the BSWAP instruction references a 16-bit register, the result is undefined.
In 64-bit mode, the instruction''s default operation size is 32 bits. Using a REX prefix in the form of REX.R permits
access to additional registers (R8-R15). Using a REX prefix in the form of REX.W promotes operation to 64 bits. See
the summary chart at the beginning of this section for encoding data and limits.

IA-32 Architecture Legacy Compatibility
The BSWAP instruction is not supported on IA-32 processors earlier than the Intel486 processor family. For
compatibility with this instruction, software should include functionally equivalent code for execution on Intel
processors earlier than the Intel486 processor family.

Operation
TEMP <- DEST
IF 64-bit mode AND OperandSize = 64
    THEN
         DEST[7:0] <- TEMP[63:56];
         DEST[15:8] <- TEMP[55:48];
         DEST[23:16] <- TEMP[47:40];
         DEST[31:24] <- TEMP[39:32];
         DEST[39:32] <- TEMP[31:24];
         DEST[47:40] <- TEMP[23:16];
         DEST[55:48] <- TEMP[15:8];
         DEST[63:56] <- TEMP[7:0];
    ELSE
         DEST[7:0] <- TEMP[31:24];
         DEST[15:8] <- TEMP[23:16];
         DEST[23:16] <- TEMP[15:8];
         DEST[31:24] <- TEMP[7:0];
FI;

Flags Affected
None.

Exceptions (All Operating Modes)
#UD                    If the LOCK prefix is used.



');
INSERT INTO "instructions" VALUES('x86_64','BT','BT-Bit Test
 Opcode                    Instruction                  Op/      64-bit   Compat/ Description
                                                        En       Mode     Leg Mode
 0F A3 /r                  BT r/m16, r16                MR       Valid    Valid       Store selected bit in CF flag.
 0F A3 /r                  BT r/m32, r32                MR       Valid    Valid       Store selected bit in CF flag.
 REX.W + 0F A3 /r          BT r/m64, r64                MR       Valid    N.E.        Store selected bit in CF flag.
 0F BA /4 ib               BT r/m16, imm8               MI       Valid    Valid       Store selected bit in CF flag.
 0F BA /4 ib               BT r/m32, imm8               MI       Valid    Valid       Store selected bit in CF flag.
 REX.W + 0F BA /4 ib       BT r/m64, imm8               MI       Valid    N.E.        Store selected bit in CF flag.



                                                   Instruction Operand Encoding
    Op/En              Operand 1                     Operand 2                    Operand 3                      Operand 4
     MR             ModRM:r/m (r)                  ModRM:reg (r)                     NA                                NA
     MI             ModRM:r/m (r)                      imm8                          NA                                NA

Description
Selects the bit in a bit string (specified with the first operand, called the bit base) at the bit-position designated by
the bit offset (specified by the second operand) and stores the value of the bit in the CF flag. The bit base operand
can be a register or a memory location; the bit offset operand can be a register or an immediate value:
.    If the bit base operand specifies a register, the instruction takes the modulo 16, 32, or 64 of the bit offset
     operand (modulo size depends on the mode and register size; 64-bit operands are available only in 64-bit
     mode).
.    If the bit base operand specifies a memory location, the operand represents the address of the byte in memory
     that contains the bit base (bit 0 of the specified byte) of the bit string. The range of the bit position that can be
     referenced by the offset operand depends on the operand size.
See also: Bit(BitBase, BitOffset) on page 3-11.
Some assemblers support immediate bit offsets larger than 31 by using the immediate bit offset field in combina-
tion with the displacement field of the memory operand. In this case, the low-order 3 or 5 bits (3 for 16-bit oper-
ands, 5 for 32-bit operands) of the immediate bit offset are stored in the immediate bit offset field, and the high-
order bits are shifted and combined with the byte displacement in the addressing mode by the assembler. The
processor will ignore the high order bits if they are not zero.
When accessing a bit in memory, the processor may access 4 bytes starting from the memory address for a 32-bit
operand size, using by the following relationship:

    Effective Address + (4 * (BitOffset DIV 32))
Or, it may access 2 bytes starting from the memory address for a 16-bit operand, using this relationship:

    Effective Address + (2 * (BitOffset DIV 16))
It may do so even when only a single byte needs to be accessed to reach the given bit. When using this bit
addressing mechanism, software should avoid referencing areas of memory close to address space holes. In partic-
ular, it should avoid references to memory-mapped I/O registers. Instead, software should use the MOV instruc-
tions to load from or store to these addresses, and use the register form of these instructions to manipulate the
data.
In 64-bit mode, the instruction''s default operation size is 32 bits. Using a REX prefix in the form of REX.R permits
access to additional registers (R8-R15). Using a REX prefix in the form of REX.W promotes operation to 64 bit oper-
ands. See the summary chart at the beginning of this section for encoding data and limits.

Operation
CF <- Bit(BitBase, BitOffset);



Flags Affected
The CF flag contains the value of the selected bit. The ZF flag is unaffected. The OF, SF, AF, and PF flags are
undefined.

Protected Mode Exceptions
#GP(0)               If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                     If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)               If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made while the
                     current privilege level is 3.
#UD                  If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                  If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                  If a memory operand effective address is outside the SS segment limit.
#UD                  If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)               If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)               If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made.
#UD                  If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)               If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)               If the memory address is in a non-canonical form.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made while the
                     current privilege level is 3.
#UD                  If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','BTC','BTC-Bit Test and Complement
 Opcode                    Instruction                    Op/   64-bit   Compat/ Description
                                                          En    Mode     Leg Mode
 0F BB /r                  BTC r/m16, r16                 MR    Valid    Valid        Store selected bit in CF flag and complement.
 0F BB /r                  BTC r/m32, r32                 MR    Valid    Valid        Store selected bit in CF flag and complement.
 REX.W + 0F BB /r          BTC r/m64, r64                 MR    Valid    N.E.         Store selected bit in CF flag and complement.
 0F BA /7 ib               BTC r/m16, imm8                MI    Valid    Valid        Store selected bit in CF flag and complement.
 0F BA /7 ib               BTC r/m32, imm8                MI    Valid    Valid        Store selected bit in CF flag and complement.
 REX.W + 0F BA /7 ib       BTC r/m64, imm8                MI    Valid    N.E.         Store selected bit in CF flag and complement.



                                                  Instruction Operand Encoding
    Op/En              Operand 1                    Operand 2                    Operand 3                     Operand 4
     MR           ModRM:r/m (r, w)                ModRM:reg (r)                     NA                             NA
     MI           ModRM:r/m (r, w)                       imm8                       NA                             NA

Description
Selects the bit in a bit string (specified with the first operand, called the bit base) at the bit-position designated by
the bit offset operand (second operand), stores the value of the bit in the CF flag, and complements the selected
bit in the bit string. The bit base operand can be a register or a memory location; the bit offset operand can be a
register or an immediate value:
.    If the bit base operand specifies a register, the instruction takes the modulo 16, 32, or 64 of the bit offset
     operand (modulo size depends on the mode and register size; 64-bit operands are available only in 64-bit
     mode). This allows any bit position to be selected.
.    If the bit base operand specifies a memory location, the operand represents the address of the byte in memory
     that contains the bit base (bit 0 of the specified byte) of the bit string. The range of the bit position that can be
     referenced by the offset operand depends on the operand size.
See also: Bit(BitBase, BitOffset) on page 3-11.
Some assemblers support immediate bit offsets larger than 31 by using the immediate bit offset field in combina-
tion with the displacement field of the memory operand. See "BT-Bit Test" in this chapter for more information on
this addressing mechanism.
This instruction can be used with a LOCK prefix to allow the instruction to be executed atomically.
In 64-bit mode, the instruction''s default operation size is 32 bits. Using a REX prefix in the form of REX.R permits
access to additional registers (R8-R15). Using a REX prefix in the form of REX.W promotes operation to 64 bits. See
the summary chart at the beginning of this section for encoding data and limits.

Operation
CF <- Bit(BitBase, BitOffset);
Bit(BitBase, BitOffset) <- NOT Bit(BitBase, BitOffset);

Flags Affected
The CF flag contains the value of the selected bit before it is complemented. The ZF flag is unaffected. The OF, SF,
AF, and PF flags are undefined.





Protected Mode Exceptions
#GP(0)              If the destination operand points to a non-writable segment.
                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.




');
INSERT INTO "instructions" VALUES('x86_64','BTR','BTR-Bit Test and Reset
 Opcode                      Instruction            Op/   64-bit     Compat/ Description
                                                    En    Mode       Leg Mode
 0F B3 /r                    BTR r/m16, r16         MR    Valid      Valid      Store selected bit in CF flag and clear.
 0F B3 /r                    BTR r/m32, r32         MR    Valid      Valid      Store selected bit in CF flag and clear.
 REX.W + 0F B3 /r            BTR r/m64, r64         MR    Valid      N.E.       Store selected bit in CF flag and clear.
 0F BA /6 ib                 BTR r/m16, imm8        MI    Valid      Valid      Store selected bit in CF flag and clear.
 0F BA /6 ib                 BTR r/m32, imm8        MI    Valid      Valid      Store selected bit in CF flag and clear.
 REX.W + 0F BA /6 ib         BTR r/m64, imm8        MI    Valid      N.E.       Store selected bit in CF flag and clear.



                                               Instruction Operand Encoding
    Op/En                Operand 1               Operand 2                   Operand 3                     Operand 4
     MR             ModRM:r/m (r, w)            ModRM:reg (r)                   NA                             NA
     MI             ModRM:r/m (r, w)               imm8                         NA                             NA

Description
Selects the bit in a bit string (specified with the first operand, called the bit base) at the bit-position designated by
the bit offset operand (second operand), stores the value of the bit in the CF flag, and clears the selected bit in the
bit string to 0. The bit base operand can be a register or a memory location; the bit offset operand can be a register
or an immediate value:
.    If the bit base operand specifies a register, the instruction takes the modulo 16, 32, or 64 of the bit offset
     operand (modulo size depends on the mode and register size; 64-bit operands are available only in 64-bit
     mode). This allows any bit position to be selected.
.    If the bit base operand specifies a memory location, the operand represents the address of the byte in memory
     that contains the bit base (bit 0 of the specified byte) of the bit string. The range of the bit position that can be
     referenced by the offset operand depends on the operand size.
See also: Bit(BitBase, BitOffset) on page 3-11.
Some assemblers support immediate bit offsets larger than 31 by using the immediate bit offset field in combina-
tion with the displacement field of the memory operand. See "BT-Bit Test" in this chapter for more information on
this addressing mechanism.
This instruction can be used with a LOCK prefix to allow the instruction to be executed atomically.
In 64-bit mode, the instruction''s default operation size is 32 bits. Using a REX prefix in the form of REX.R permits
access to additional registers (R8-R15). Using a REX prefix in the form of REX.W promotes operation to 64 bits. See
the summary chart at the beginning of this section for encoding data and limits.

Operation
CF <- Bit(BitBase, BitOffset);
Bit(BitBase, BitOffset) <- 0;

Flags Affected
The CF flag contains the value of the selected bit before it is cleared. The ZF flag is unaffected. The OF, SF, AF, and
PF flags are undefined.





Protected Mode Exceptions
#GP(0)              If the destination operand points to a non-writable segment.
                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.




');
INSERT INTO "instructions" VALUES('x86_64','BTS','BTS-Bit Test and Set
 Opcode                    Instruction             Op/     64-bit    Compat/ Description
                                                   En      Mode      Leg Mode
 0F AB /r                  BTS r/m16, r16          MR      Valid     Valid       Store selected bit in CF flag and set.
 0F AB /r                  BTS r/m32, r32          MR      Valid     Valid       Store selected bit in CF flag and set.
 REX.W + 0F AB /r          BTS r/m64, r64          MR      Valid     N.E.        Store selected bit in CF flag and set.
 0F BA /5 ib               BTS r/m16, imm8         MI      Valid     Valid       Store selected bit in CF flag and set.
 0F BA /5 ib               BTS r/m32, imm8         MI      Valid     Valid       Store selected bit in CF flag and set.
 REX.W + 0F BA /5 ib       BTS r/m64, imm8         MI      Valid     N.E.        Store selected bit in CF flag and set.



                                              Instruction Operand Encoding
    Op/En              Operand 1               Operand 2                     Operand 3                      Operand 4
     MR           ModRM:r/m (r, w)            ModRM:reg (r)                     NA                              NA
     MI           ModRM:r/m (r, w)               imm8                           NA                              NA

Description
Selects the bit in a bit string (specified with the first operand, called the bit base) at the bit-position designated by
the bit offset operand (second operand), stores the value of the bit in the CF flag, and sets the selected bit in the
bit string to 1. The bit base operand can be a register or a memory location; the bit offset operand can be a register
or an immediate value:
.    If the bit base operand specifies a register, the instruction takes the modulo 16, 32, or 64 of the bit offset
     operand (modulo size depends on the mode and register size; 64-bit operands are available only in 64-bit
     mode). This allows any bit position to be selected.
.    If the bit base operand specifies a memory location, the operand represents the address of the byte in memory
     that contains the bit base (bit 0 of the specified byte) of the bit string. The range of the bit position that can be
     referenced by the offset operand depends on the operand size.
See also: Bit(BitBase, BitOffset) on page 3-11.
Some assemblers support immediate bit offsets larger than 31 by using the immediate bit offset field in combina-
tion with the displacement field of the memory operand. See "BT-Bit Test" in this chapter for more information on
this addressing mechanism.
This instruction can be used with a LOCK prefix to allow the instruction to be executed atomically.
In 64-bit mode, the instruction''s default operation size is 32 bits. Using a REX prefix in the form of REX.R permits
access to additional registers (R8-R15). Using a REX prefix in the form of REX.W promotes operation to 64 bits. See
the summary chart at the beginning of this section for encoding data and limits.

Operation
CF <- Bit(BitBase, BitOffset);
Bit(BitBase, BitOffset) <- 1;

Flags Affected
The CF flag contains the value of the selected bit before it is set. The ZF flag is unaffected. The OF, SF, AF, and PF
flags are undefined.





Protected Mode Exceptions
#GP(0)              If the destination operand points to a non-writable segment.
                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.

Virtual-8086 Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.




');
INSERT INTO "instructions" VALUES('x86_64','BZHI','BZHI - Zero High Bits Starting with Specified Bit Position
  Opcode/Instruction                  Op/      64/32         CPUID       Description
                                      En       -bit          Feature
                                               Mode          Flag
  VEX.NDS.LZ.0F38.W0 F5 /r            RMV      V/V           BMI2        Zero bits in r/m32 starting with the position in r32b, write result to
  BZHI r32a, r/m32, r32b                                                 r32a.
  VEX.NDS.LZ.0F38.W1 F5 /r            RMV      V/N.E.        BMI2        Zero bits in r/m64 starting with the position in r64b, write result to
  BZHI r64a, r/m64, r64b                                                 r64a.



                                                        Instruction Operand Encoding
   Op/En                Operand 1                            Operand 2                      Operand 3                       Operand 4
    RMV              ModRM:reg (w)                      ModRM:r/m (r)                      VEX.vvvv (r)                         NA

Description
BZHI copies the bits of the first source operand (the second operand) into the destination operand (the first
operand) and clears the higher bits in the destination according to the INDEX value specified by the second source
operand (the third operand). The INDEX is specified by bits 7:0 of the second source operand. The INDEX value is
saturated at the value of OperandSize -1. CF is set, if the number contained in the 8 low bits of the third operand
is greater than OperandSize -1.
This instruction is not supported in real mode and virtual-8086 mode. The operand size is always 32 bits if not in
64-bit mode. In 64-bit mode operand size 64 requires VEX.W1. VEX.W1 is ignored in non-64-bit modes. An
attempt to execute this instruction with VEX.L not equal to 0 will cause #UD.

Operation
N <- SRC2[7:0]
DEST <- SRC1
IF (N < OperandSize)
    DEST[OperandSize-1:N] <- 0
FI
IF (N > OperandSize - 1)
    CF <- 1
ELSE
    CF <- 0
FI

Flags Affected
ZF, CF and SF flags are updated based on the result. OF flag is cleared. AF and PF flags are undefined.

Intel C/C++ Compiler Intrinsic Equivalent
BZHI:           unsigned __int32 _bzhi_u32(unsigned __int32 src, unsigned __int32 index);

BZHI:           unsigned __int64 _bzhi_u64(unsigned __int64 src, unsigned __int32 index);

SIMD Floating-Point Exceptions
None

Other Exceptions
See Section 2.5.1, "Exception Conditions for VEX-Encoded GPR Instructions", Table 2-29; additionally
#UD                        If VEX.W = 1.




');
INSERT INTO "instructions" VALUES('x86_64','CALL','CALL-Call Procedure
    Opcode             Instruction          Op/    64-bit    Compat/ Description
                                            En     Mode      Leg Mode
    E8 cw              CALL rel16           M      N.S.      Valid       Call near, relative, displacement relative to next
                                                                         instruction.
    E8 cd              CALL rel32           M      Valid     Valid       Call near, relative, displacement relative to next
                                                                         instruction. 32-bit displacement sign extended to
                                                                         64-bits in 64-bit mode.
    FF /2              CALL r/m16           M      N.E.      Valid       Call near, absolute indirect, address given in r/m16.
    FF /2              CALL r/m32           M      N.E.      Valid       Call near, absolute indirect, address given in r/m32.
    FF /2              CALL r/m64           M      Valid     N.E.        Call near, absolute indirect, address given in r/m64.
    9A cd              CALL ptr16:16        D      Invalid   Valid       Call far, absolute, address given in operand.
    9A cp              CALL ptr16:32        D      Invalid   Valid       Call far, absolute, address given in operand.
    FF /3              CALL m16:16          M      Valid     Valid       Call far, absolute indirect address given in m16:16.
                                                                         In 32-bit mode: if selector points to a gate, then RIP
                                                                         = 32-bit zero extended displacement taken from
                                                                         gate; else RIP = zero extended 16-bit offset from
                                                                         far pointer referenced in the instruction.
    FF /3              CALL m16:32          M      Valid     Valid       In 64-bit mode: If selector points to a gate, then RIP
                                                                         = 64-bit displacement taken from gate; else RIP =
                                                                         zero extended 32-bit offset from far pointer
                                                                         referenced in the instruction.
    REX.W + FF /3      CALL m16:64          M      Valid     N.E.        In 64-bit mode: If selector points to a gate, then RIP
                                                                         = 64-bit displacement taken from gate; else RIP =
                                                                         64-bit offset from far pointer referenced in the
                                                                         instruction.



                                             Instruction Operand Encoding
     Op/En           Operand 1                  Operand 2                  Operand 3                       Operand 4
       D               Offset                      NA                         NA                               NA
       M            ModRM:r/m (r)                  NA                         NA                               NA

Description
Saves procedure linking information on the stack and branches to the called procedure specified using the target
operand. The target operand specifies the address of the first instruction in the called procedure. The operand can
be an immediate value, a general-purpose register, or a memory location.
This instruction can be used to execute four types of calls:
.     Near Call - A call to a procedure in the current code segment (the segment currently pointed to by the CS
      register), sometimes referred to as an intra-segment call.
.     Far Call - A call to a procedure located in a different segment than the current code segment, sometimes
      referred to as an inter-segment call.
.     Inter-privilege-level far call - A far call to a procedure in a segment at a different privilege level than that
      of the currently executing program or procedure.
.     Task switch - A call to a procedure located in a different task.
The latter two call types (inter-privilege-level call and task switch) can only be executed in protected mode. See
"Calling Procedures Using Call and RET" in Chapter 6 of the Intel 64 and IA-32 Architectures Software Devel-
oper''s Manual, Volume 1, for additional information on near, far, and inter-privilege-level calls. See Chapter 7,
"Task Management," in the Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 3A, for infor-
mation on performing task switches with the CALL instruction.



Near Call. When executing a near call, the processor pushes the value of the EIP register (which contains the offset
of the instruction following the CALL instruction) on the stack (for use later as a return-instruction pointer). The
processor then branches to the address in the current code segment specified by the target operand. The target
operand specifies either an absolute offset in the code segment (an offset from the base of the code segment) or a
relative offset (a signed displacement relative to the current value of the instruction pointer in the EIP register; this
value points to the instruction following the CALL instruction). The CS register is not changed on near calls.
For a near call absolute, an absolute offset is specified indirectly in a general-purpose register or a memory location
(r/m16, r/m32, or r/m64). The operand-size attribute determines the size of the target operand (16, 32 or 64
bits). When in 64-bit mode, the operand size for near call (and all near branches) is forced to 64-bits. Absolute
offsets are loaded directly into the EIP(RIP) register. If the operand size attribute is 16, the upper two bytes of the
EIP register are cleared, resulting in a maximum instruction pointer size of 16 bits. When accessing an absolute
offset indirectly using the stack pointer [ESP] as the base register, the base value used is the value of the ESP
before the instruction executes.
A relative offset (rel16 or rel32) is generally specified as a label in assembly code. But at the machine code level, it
is encoded as a signed, 16- or 32-bit immediate value. This value is added to the value in the EIP(RIP) register. In
64-bit mode the relative offset is always a 32-bit immediate value which is sign extended to 64-bits before it is
added to the value in the RIP register for the target calculation. As with absolute offsets, the operand-size attribute
determines the size of the target operand (16, 32, or 64 bits). In 64-bit mode the target operand will always be 64-
bits because the operand size is forced to 64-bits for near branches.
Far Calls in Real-Address or Virtual-8086 Mode. When executing a far call in real- address or virtual-8086 mode, the
processor pushes the current value of both the CS and EIP registers on the stack for use as a return-instruction
pointer. The processor then performs a "far branch" to the code segment and offset specified with the target
operand for the called procedure. The target operand specifies an absolute far address either directly with a pointer
(ptr16:16 or ptr16:32) or indirectly with a memory location (m16:16 or m16:32). With the pointer method, the
segment and offset of the called procedure is encoded in the instruction using a 4-byte (16-bit operand size) or 6-
byte (32-bit operand size) far address immediate. With the indirect method, the target operand specifies a memory
location that contains a 4-byte (16-bit operand size) or 6-byte (32-bit operand size) far address. The operand-size
attribute determines the size of the offset (16 or 32 bits) in the far address. The far address is loaded directly into
the CS and EIP registers. If the operand-size attribute is 16, the upper two bytes of the EIP register are cleared.
Far Calls in Protected Mode. When the processor is operating in protected mode, the CALL instruction can be used to
perform the following types of far calls:
.   Far call to the same privilege level
.   Far call to a different privilege level (inter-privilege level call)
.   Task switch (far call to another task)
In protected mode, the processor always uses the segment selector part of the far address to access the corre-
sponding descriptor in the GDT or LDT. The descriptor type (code segment, call gate, task gate, or TSS) and access
rights determine the type of call operation to be performed.
If the selected descriptor is for a code segment, a far call to a code segment at the same privilege level is
performed. (If the selected code segment is at a different privilege level and the code segment is non-conforming,
a general-protection exception is generated.) A far call to the same privilege level in protected mode is very similar
to one carried out in real-address or virtual-8086 mode. The target operand specifies an absolute far address either
directly with a pointer (ptr16:16 or ptr16:32) or indirectly with a memory location (m16:16 or m16:32). The
operand- size attribute determines the size of the offset (16 or 32 bits) in the far address. The new code segment
selector and its descriptor are loaded into CS register; the offset from the instruction is loaded into the EIP register.
A call gate (described in the next paragraph) can also be used to perform a far call to a code segment at the same
privilege level. Using this mechanism provides an extra level of indirection and is the preferred method of making
calls between 16-bit and 32-bit code segments.
When executing an inter-privilege-level far call, the code segment for the procedure being called must be accessed
through a call gate. The segment selector specified by the target operand identifies the call gate. The target
operand can specify the call gate segment selector either directly with a pointer (ptr16:16 or ptr16:32) or indirectly
with a memory location (m16:16 or m16:32). The processor obtains the segment selector for the new code
segment and the new instruction pointer (offset) from the call gate descriptor. (The offset from the target operand
is ignored when a call gate is used.)




On inter-privilege-level calls, the processor switches to the stack for the privilege level of the called procedure. The
segment selector for the new stack segment is specified in the TSS for the currently running task. The branch to
the new code segment occurs after the stack switch. (Note that when using a call gate to perform a far call to a
segment at the same privilege level, no stack switch occurs.) On the new stack, the processor pushes the segment
selector and stack pointer for the calling procedure''s stack, an optional set of parameters from the calling proce-
dures stack, and the segment selector and instruction pointer for the calling procedure''s code segment. (A value in
the call gate descriptor determines how many parameters to copy to the new stack.) Finally, the processor
branches to the address of the procedure being called within the new code segment.
Executing a task switch with the CALL instruction is similar to executing a call through a call gate. The target
operand specifies the segment selector of the task gate for the new task activated by the switch (the offset in the
target operand is ignored). The task gate in turn points to the TSS for the new task, which contains the segment
selectors for the task''s code and stack segments. Note that the TSS also contains the EIP value for the next instruc-
tion that was to be executed before the calling task was suspended. This instruction pointer value is loaded into the
EIP register to re-start the calling task.
The CALL instruction can also specify the segment selector of the TSS directly, which eliminates the indirection of
the task gate. See Chapter 7, "Task Management," in the Intel 64 and IA-32 Architectures Software Developer''s
Manual, Volume 3A, for information on the mechanics of a task switch.
When you execute at task switch with a CALL instruction, the nested task flag (NT) is set in the EFLAGS register and
the new TSS''s previous task link field is loaded with the old task''s TSS selector. Code is expected to suspend this
nested task by executing an IRET instruction which, because the NT flag is set, automatically uses the previous
task link to return to the calling task. (See "Task Linking" in Chapter 7 of the Intel 64 and IA-32 Architectures
Software Developer''s Manual, Volume 3A, for information on nested tasks.) Switching tasks with the CALL instruc-
tion differs in this regard from JMP instruction. JMP does not set the NT flag and therefore does not expect an IRET
instruction to suspend the task.
Mixing 16-Bit and 32-Bit Calls. When making far calls between 16-bit and 32-bit code segments, use a call gate. If
the far call is from a 32-bit code segment to a 16-bit code segment, the call should be made from the first 64
KBytes of the 32-bit code segment. This is because the operand-size attribute of the instruction is set to 16, so only
a 16-bit return address offset can be saved. Also, the call should be made using a 16-bit call gate so that 16-bit
values can be pushed on the stack. See Chapter 21, "Mixing 16-Bit and 32-Bit Code," in the Intel 64 and IA-32
Architectures Software Developer''s Manual, Volume 3B, for more information.
Far Calls in Compatibility Mode. When the processor is operating in compatibility mode, the CALL instruction can be
used to perform the following types of far calls:
.   Far call to the same privilege level, remaining in compatibility mode
.   Far call to the same privilege level, transitioning to 64-bit mode
.   Far call to a different privilege level (inter-privilege level call), transitioning to 64-bit mode
Note that a CALL instruction can not be used to cause a task switch in compatibility mode since task switches are
not supported in IA-32e mode.
In compatibility mode, the processor always uses the segment selector part of the far address to access the corre-
sponding descriptor in the GDT or LDT. The descriptor type (code segment, call gate) and access rights determine
the type of call operation to be performed.
If the selected descriptor is for a code segment, a far call to a code segment at the same privilege level is
performed. (If the selected code segment is at a different privilege level and the code segment is non-conforming,
a general-protection exception is generated.) A far call to the same privilege level in compatibility mode is very
similar to one carried out in protected mode. The target operand specifies an absolute far address either directly
with a pointer (ptr16:16 or ptr16:32) or indirectly with a memory location (m16:16 or m16:32). The operand-size
attribute determines the size of the offset (16 or 32 bits) in the far address. The new code segment selector and its
descriptor are loaded into CS register and the offset from the instruction is loaded into the EIP register. The differ-
ence is that 64-bit mode may be entered. This specified by the L bit in the new code segment descriptor.
Note that a 64-bit call gate (described in the next paragraph) can also be used to perform a far call to a code
segment at the same privilege level. However, using this mechanism requires that the target code segment
descriptor have the L bit set, causing an entry to 64-bit mode.
When executing an inter-privilege-level far call, the code segment for the procedure being called must be accessed
through a 64-bit call gate. The segment selector specified by the target operand identifies the call gate. The target



operand can specify the call gate segment selector either directly with a pointer (ptr16:16 or ptr16:32) or indirectly
with a memory location (m16:16 or m16:32). The processor obtains the segment selector for the new code
segment and the new instruction pointer (offset) from the 16-byte call gate descriptor. (The offset from the target
operand is ignored when a call gate is used.)
On inter-privilege-level calls, the processor switches to the stack for the privilege level of the called procedure. The
segment selector for the new stack segment is set to NULL. The new stack pointer is specified in the TSS for the
currently running task. The branch to the new code segment occurs after the stack switch. (Note that when using
a call gate to perform a far call to a segment at the same privilege level, an implicit stack switch occurs as a result
of entering 64-bit mode. The SS selector is unchanged, but stack segment accesses use a segment base of 0x0,
the limit is ignored, and the default stack size is 64-bits. The full value of RSP is used for the offset, of which the
upper 32-bits are undefined.) On the new stack, the processor pushes the segment selector and stack pointer for
the calling procedure''s stack and the segment selector and instruction pointer for the calling procedure''s code
segment. (Parameter copy is not supported in IA-32e mode.) Finally, the processor branches to the address of the
procedure being called within the new code segment.
Near/(Far) Calls in 64-bit Mode. When the processor is operating in 64-bit mode, the CALL instruction can be used to
perform the following types of far calls:
.   Far call to the same privilege level, transitioning to compatibility mode
.   Far call to the same privilege level, remaining in 64-bit mode
.   Far call to a different privilege level (inter-privilege level call), remaining in 64-bit mode
Note that in this mode the CALL instruction can not be used to cause a task switch in 64-bit mode since task
switches are not supported in IA-32e mode.
In 64-bit mode, the processor always uses the segment selector part of the far address to access the corresponding
descriptor in the GDT or LDT. The descriptor type (code segment, call gate) and access rights determine the type
of call operation to be performed.
If the selected descriptor is for a code segment, a far call to a code segment at the same privilege level is
performed. (If the selected code segment is at a different privilege level and the code segment is non-conforming,
a general-protection exception is generated.) A far call to the same privilege level in 64-bit mode is very similar to
one carried out in compatibility mode. The target operand specifies an absolute far address indirectly with a
memory location (m16:16, m16:32 or m16:64). The form of CALL with a direct specification of absolute far
address is not defined in 64-bit mode. The operand-size attribute determines the size of the offset (16, 32, or 64
bits) in the far address. The new code segment selector and its descriptor are loaded into the CS register; the offset
from the instruction is loaded into the EIP register. The new code segment may specify entry either into compati-
bility or 64-bit mode, based on the L bit value.
A 64-bit call gate (described in the next paragraph) can also be used to perform a far call to a code segment at the
same privilege level. However, using this mechanism requires that the target code segment descriptor have the L
bit set.
When executing an inter-privilege-level far call, the code segment for the procedure being called must be accessed
through a 64-bit call gate. The segment selector specified by the target operand identifies the call gate. The target
operand can only specify the call gate segment selector indirectly with a memory location (m16:16, m16:32 or
m16:64). The processor obtains the segment selector for the new code segment and the new instruction pointer
(offset) from the 16-byte call gate descriptor. (The offset from the target operand is ignored when a call gate is
used.)
On inter-privilege-level calls, the processor switches to the stack for the privilege level of the called procedure. The
segment selector for the new stack segment is set to NULL. The new stack pointer is specified in the TSS for the
currently running task. The branch to the new code segment occurs after the stack switch.
Note that when using a call gate to perform a far call to a segment at the same privilege level, an implicit stack
switch occurs as a result of entering 64-bit mode. The SS selector is unchanged, but stack segment accesses use
a segment base of 0x0, the limit is ignored, and the default stack size is 64-bits. (The full value of RSP is used for
the offset.) On the new stack, the processor pushes the segment selector and stack pointer for the calling proce-
dure''s stack and the segment selector and instruction pointer for the calling procedure''s code segment. (Parameter
copy is not supported in IA-32e mode.) Finally, the processor branches to the address of the procedure being called
within the new code segment.





Operation
IF near call
    THEN IF near relative call
         THEN
             IF OperandSize = 64
                  THEN
                       tempDEST <- SignExtend(DEST); (* DEST is rel32 *)
                       tempRIP <- RIP + tempDEST;
                       IF stack not large enough for a 8-byte return address
                            THEN #SS(0); FI;
                       Push(RIP);
                       RIP <- tempRIP;
             FI;
             IF OperandSize = 32
                  THEN
                       tempEIP <- EIP + DEST; (* DEST is rel32 *)
                       IF tempEIP is not within code segment limit THEN #GP(0); FI;
                       IF stack not large enough for a 4-byte return address
                            THEN #SS(0); FI;
                       Push(EIP);
                       EIP <- tempEIP;
             FI;
             IF OperandSize = 16
                  THEN
                       tempEIP <- (EIP + DEST) AND 0000FFFFH; (* DEST is rel16 *)
                       IF tempEIP is not within code segment limit THEN #GP(0); FI;
                       IF stack not large enough for a 2-byte return address
                            THEN #SS(0); FI;
                       Push(IP);
                       EIP <- tempEIP;
             FI;
         ELSE (* Near absolute call *)
             IF OperandSize = 64
                  THEN
                       tempRIP <- DEST; (* DEST is r/m64 *)
                       IF stack not large enough for a 8-byte return address
                            THEN #SS(0); FI;
                       Push(RIP);
                       RIP <- tempRIP;
             FI;
             IF OperandSize = 32
                  THEN
                       tempEIP <- DEST; (* DEST is r/m32 *)
                       IF tempEIP is not within code segment limit THEN #GP(0); FI;
                       IF stack not large enough for a 4-byte return address
                            THEN #SS(0); FI;
                       Push(EIP);
                       EIP <- tempEIP;
             FI;
             IF OperandSize = 16
                  THEN
                       tempEIP <- DEST AND 0000FFFFH; (* DEST is r/m16 *)
                       IF tempEIP is not within code segment limit THEN #GP(0); FI;



                        IF stack not large enough for a 2-byte return address
                             THEN #SS(0); FI;
                        Push(IP);
                        EIP <- tempEIP;
                  FI;
     FI;rel/abs
FI; near

IF far call and (PE = 0 or (PE = 1 and VM = 1)) (* Real-address or virtual-8086 mode *)
    THEN
         IF OperandSize = 32
               THEN
                    IF stack not large enough for a 6-byte return address
                         THEN #SS(0); FI;
                    IF DEST[31:16] is not zero THEN #GP(0); FI;
                    Push(CS); (* Padded with 16 high-order bits *)
                    Push(EIP);
                    CS <- DEST[47:32]; (* DEST is ptr16:32 or [m16:32] *)
                    EIP <- DEST[31:0]; (* DEST is ptr16:32 or [m16:32] *)
               ELSE (* OperandSize = 16 *)
                    IF stack not large enough for a 4-byte return address
                         THEN #SS(0); FI;
                    Push(CS);
                    Push(IP);
                    CS <- DEST[31:16]; (* DEST is ptr16:16 or [m16:16] *)
                    EIP <- DEST[15:0]; (* DEST is ptr16:16 or [m16:16]; clear upper 16 bits *)
         FI;
FI;

IF far call and (PE = 1 and VM = 0) (* Protected mode or IA-32e Mode, not virtual-8086 mode*)
    THEN
         IF segment selector in target operand NULL
               THEN #GP(0); FI;
         IF segment selector index not within descriptor table limits
               THEN #GP(new code segment selector); FI;
         Read type and access rights of selected segment descriptor;
         IF IA32_EFER.LMA = 0
               THEN
                    IF segment type is not a conforming or nonconforming code segment, call
                    gate, task gate, or TSS
                         THEN #GP(segment selector); FI;
               ELSE
                    IF segment type is not a conforming or nonconforming code segment or
                    64-bit call gate,
                         THEN #GP(segment selector); FI;
         FI;
         Depending on type and access rights:
               GO TO CONFORMING-CODE-SEGMENT;
               GO TO NONCONFORMING-CODE-SEGMENT;
               GO TO CALL-GATE;
               GO TO TASK-GATE;
               GO TO TASK-STATE-SEGMENT;
FI;




CONFORMING-CODE-SEGMENT:
   IF L bit = 1 and D bit = 1 and IA32_EFER.LMA = 1
        THEN GP(new code segment selector); FI;
   IF DPL > CPL
        THEN #GP(new code segment selector); FI;
   IF segment not present
        THEN #NP(new code segment selector); FI;
   IF stack not large enough for return address
        THEN #SS(0); FI;
   tempEIP <- DEST(Offset);
   IF OperandSize = 16
        THEN
              tempEIP <- tempEIP AND 0000FFFFH; FI; (* Clear upper 16 bits *)
   IF (EFER.LMA = 0 or target mode = Compatibility mode) and (tempEIP outside new code
   segment limit)
        THEN #GP(0); FI;
   IF tempEIP is non-canonical
        THEN #GP(0); FI;
   IF OperandSize = 32
        THEN
              Push(CS); (* Padded with 16 high-order bits *)
              Push(EIP);
              CS <- DEST(CodeSegmentSelector);
              (* Segment descriptor information also loaded *)
              CS(RPL) <- CPL;
              EIP <- tempEIP;
        ELSE
              IF OperandSize = 16
                   THEN
                       Push(CS);
                       Push(IP);
                       CS <- DEST(CodeSegmentSelector);
                       (* Segment descriptor information also loaded *)
                       CS(RPL) <- CPL;
                       EIP <- tempEIP;
                   ELSE (* OperandSize = 64 *)
                       Push(CS); (* Padded with 48 high-order bits *)
                       Push(RIP);
                       CS <- DEST(CodeSegmentSelector);
                       (* Segment descriptor information also loaded *)
                       CS(RPL) <- CPL;
                       RIP <- tempEIP;
              FI;
   FI;
END;

NONCONFORMING-CODE-SEGMENT:
  IF L-Bit = 1 and D-BIT = 1 and IA32_EFER.LMA = 1
       THEN GP(new code segment selector); FI;
  IF (RPL > CPL) or (DPL != CPL)
       THEN #GP(new code segment selector); FI;
  IF segment not present
       THEN #NP(new code segment selector); FI;
  IF stack not large enough for return address



        THEN #SS(0); FI;
   tempEIP <- DEST(Offset);
   IF OperandSize = 16
        THEN tempEIP <- tempEIP AND 0000FFFFH; FI; (* Clear upper 16 bits *)
   IF (EFER.LMA = 0 or target mode = Compatibility mode) and (tempEIP outside new code
   segment limit)
        THEN #GP(0); FI;
   IF tempEIP is non-canonical
        THEN #GP(0); FI;
   IF OperandSize = 32
        THEN
             Push(CS); (* Padded with 16 high-order bits *)
             Push(EIP);
             CS <- DEST(CodeSegmentSelector);
             (* Segment descriptor information also loaded *)
             CS(RPL) <- CPL;
             EIP <- tempEIP;
        ELSE
             IF OperandSize = 16
                  THEN
                      Push(CS);
                      Push(IP);
                      CS <- DEST(CodeSegmentSelector);
                      (* Segment descriptor information also loaded *)
                      CS(RPL) <- CPL;
                      EIP <- tempEIP;
                  ELSE (* OperandSize = 64 *)
                      Push(CS); (* Padded with 48 high-order bits *)
                      Push(RIP);
                      CS <- DEST(CodeSegmentSelector);
                      (* Segment descriptor information also loaded *)
                      CS(RPL) <- CPL;
                      RIP <- tempEIP;
             FI;
   FI;
END;

CALL-GATE:
   IF call gate (DPL < CPL) or (RPL > DPL)
        THEN #GP(call-gate selector); FI;
   IF call gate not present
        THEN #NP(call-gate selector); FI;
   IF call-gate code-segment selector is NULL
        THEN #GP(0); FI;
   IF call-gate code-segment selector index is outside descriptor table limits
        THEN #GP(call-gate code-segment selector); FI;
   Read call-gate code-segment descriptor;
   IF call-gate code-segment descriptor does not indicate a code segment
   or call-gate code-segment descriptor DPL > CPL
        THEN #GP(call-gate code-segment selector); FI;
   IF IA32_EFER.LMA = 1 AND (call-gate code-segment descriptor is
   not a 64-bit code segment or call-gate code-segment descriptor has both L-bit and D-bit set)
        THEN #GP(call-gate code-segment selector); FI;
   IF call-gate code segment not present



        THEN #NP(call-gate code-segment selector); FI;
   IF call-gate code segment is non-conforming and DPL < CPL
        THEN go to MORE-PRIVILEGE;
        ELSE go to SAME-PRIVILEGE;
   FI;
END;

MORE-PRIVILEGE:
  IF current TSS is 32-bit
       THEN
            TSSstackAddress <- (new code-segment DPL * 8) + 4;
            IF (TSSstackAddress + 5) > current TSS limit
                 THEN #TS(current TSS selector); FI;
            NewSS <- 2 bytes loaded from (TSS base + TSSstackAddress + 4);
            NewESP <- 4 bytes loaded from (TSS base + TSSstackAddress);
       ELSE
            IF current TSS is 16-bit
                 THEN
                      TSSstackAddress <- (new code-segment DPL * 4) + 2
                      IF (TSSstackAddress + 3) > current TSS limit
                           THEN #TS(current TSS selector); FI;
                      NewSS <- 2 bytes loaded from (TSS base + TSSstackAddress + 2);
                      NewESP <- 2 bytes loaded from (TSS base + TSSstackAddress);
                 ELSE (* current TSS is 64-bit *)
                      TSSstackAddress <- (new code-segment DPL * 8) + 4;
                      IF (TSSstackAddress + 7) > current TSS limit
                           THEN #TS(current TSS selector); FI;
                      NewSS <- new code-segment DPL; (* NULL selector with RPL = new CPL *)
                      NewRSP <- 8 bytes loaded from (current TSS base + TSSstackAddress);
            FI;
  FI;
  IF IA32_EFER.LMA = 0 and NewSS is NULL
       THEN #TS(NewSS); FI;
  Read new code-segment descriptor and new stack-segment descriptor;
  IF IA32_EFER.LMA = 0 and (NewSS RPL != new code-segment DPL
  or new stack-segment DPL != new code-segment DPL or new stack segment is not a
  writable data segment)
       THEN #TS(NewSS); FI
  IF IA32_EFER.LMA = 0 and new stack segment not present
       THEN #SS(NewSS); FI;
  IF CallGateSize = 32
       THEN
            IF new stack does not have room for parameters plus 16 bytes
                 THEN #SS(NewSS); FI;
            IF CallGate(InstructionPointer) not within new code-segment limit
                 THEN #GP(0); FI;
            SS <- newSS; (* Segment descriptor information also loaded *)
            ESP <- newESP;
            CS:EIP <- CallGate(CS:InstructionPointer);
            (* Segment descriptor information also loaded *)
            Push(oldSS:oldESP); (* From calling procedure *)
            temp <- parameter count from call gate, masked to 5 bits;
            Push(parameters from calling procedure''s stack, temp)
            Push(oldCS:oldEIP); (* Return address to calling procedure *)



        ELSE
            IF CallGateSize = 16
                 THEN
                      IF new stack does not have room for parameters plus 8 bytes
                           THEN #SS(NewSS); FI;
                      IF (CallGate(InstructionPointer) AND FFFFH) not in new code-segment limit
                           THEN #GP(0); FI;
                      SS <- newSS; (* Segment descriptor information also loaded *)
                      ESP <- newESP;
                      CS:IP <- CallGate(CS:InstructionPointer);
                      (* Segment descriptor information also loaded *)
                      Push(oldSS:oldESP); (* From calling procedure *)
                      temp <- parameter count from call gate, masked to 5 bits;
                      Push(parameters from calling procedure''s stack, temp)
                      Push(oldCS:oldEIP); (* Return address to calling procedure *)
                 ELSE (* CallGateSize = 64 *)
                      IF pushing 32 bytes on the stack would use a non-canonical address
                           THEN #SS(NewSS); FI;
                      IF (CallGate(InstructionPointer) is non-canonical)
                           THEN #GP(0); FI;
                      SS <- NewSS; (* NewSS is NULL)
                      RSP <- NewESP;
                      CS:IP <- CallGate(CS:InstructionPointer);
                      (* Segment descriptor information also loaded *)
                      Push(oldSS:oldESP); (* From calling procedure *)
                      Push(oldCS:oldEIP); (* Return address to calling procedure *)
            FI;
   FI;
   CPL <- CodeSegment(DPL)
   CS(RPL) <- CPL
END;

SAME-PRIVILEGE:
  IF CallGateSize = 32
       THEN
            IF stack does not have room for 8 bytes
                  THEN #SS(0); FI;
            IF CallGate(InstructionPointer) not within code segment limit
                  THEN #GP(0); FI;
            CS:EIP <- CallGate(CS:EIP) (* Segment descriptor information also loaded *)
            Push(oldCS:oldEIP); (* Return address to calling procedure *)
       ELSE
            If CallGateSize = 16
                  THEN
                      IF stack does not have room for 4 bytes
                           THEN #SS(0); FI;
                      IF CallGate(InstructionPointer) not within code segment limit
                           THEN #GP(0); FI;
                      CS:IP <- CallGate(CS:instruction pointer);
                      (* Segment descriptor information also loaded *)
                      Push(oldCS:oldIP); (* Return address to calling procedure *)
                  ELSE (* CallGateSize = 64)
                      IF pushing 16 bytes on the stack touches non-canonical addresses
                           THEN #SS(0); FI;



                      IF RIP non-canonical
                           THEN #GP(0); FI;
                      CS:IP <- CallGate(CS:instruction pointer);
                      (* Segment descriptor information also loaded *)
                      Push(oldCS:oldIP); (* Return address to calling procedure *)
           FI;
   FI;
   CS(RPL) <- CPL
END;

TASK-GATE:
   IF task gate DPL < CPL or RPL
        THEN #GP(task gate selector); FI;
   IF task gate not present
        THEN #NP(task gate selector); FI;
   Read the TSS segment selector in the task-gate descriptor;
   IF TSS segment selector local/global bit is set to local
   or index not within GDT limits
        THEN #GP(TSS selector); FI;
   Access TSS descriptor in GDT;
   IF TSS descriptor specifies that the TSS is busy (low-order 5 bits set to 00001)
        THEN #GP(TSS selector); FI;
   IF TSS not present
        THEN #NP(TSS selector); FI;
   SWITCH-TASKS (with nesting) to TSS;
   IF EIP not within code segment limit
        THEN #GP(0); FI;
END;

TASK-STATE-SEGMENT:
   IF TSS DPL < CPL or RPL
   or TSS descriptor indicates TSS not available
        THEN #GP(TSS selector); FI;
   IF TSS is not present
        THEN #NP(TSS selector); FI;
   SWITCH-TASKS (with nesting) to TSS;
   IF EIP not within code segment limit
        THEN #GP(0); FI;
END;


Flags Affected
All flags are affected if a task switch occurs; no flags are affected if a task switch does not occur.





Protected Mode Exceptions
#GP(0)                If the target offset in destination operand is beyond the new code segment limit.
                      If the segment selector in the destination operand is NULL.
                      If the code segment selector in the gate is NULL.
                      If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                      If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                      selector.
#GP(selector)         If a code segment or gate or TSS selector index is outside descriptor table limits.
                      If the segment descriptor pointed to by the segment selector in the destination operand is not
                      for a conforming-code segment, nonconforming-code segment, call gate, task gate, or task
                      state segment.
                      If the DPL for a nonconforming-code segment is not equal to the CPL or the RPL for the
                      segment''s segment selector is greater than the CPL.
                      If the DPL for a conforming-code segment is greater than the CPL.
                      If the DPL from a call-gate, task-gate, or TSS segment descriptor is less than the CPL or than
                      the RPL of the call-gate, task-gate, or TSS''s segment selector.
                      If the segment descriptor for a segment selector from a call gate does not indicate it is a code
                      segment.
                      If the segment selector from a call gate is beyond the descriptor table limits.
                      If the DPL for a code-segment obtained from a call gate is greater than the CPL.
                      If the segment selector for a TSS has its local/global bit set for local.
                      If a TSS segment descriptor specifies that the TSS is busy or not available.
#SS(0)                If pushing the return address, parameters, or stack segment pointer onto the stack exceeds
                      the bounds of the stack segment, when no stack switch occurs.
                      If a memory operand effective address is outside the SS segment limit.
#SS(selector)         If pushing the return address, parameters, or stack segment pointer onto the stack exceeds
                      the bounds of the stack segment, when a stack switch occurs.
                      If the SS register is being loaded as part of a stack switch and the segment pointed to is
                      marked not present.
                      If stack segment does not have room for the return address, parameters, or stack segment
                      pointer, when stack switch occurs.
#NP(selector)         If a code segment, data segment, stack segment, call gate, task gate, or TSS is not present.
#TS(selector)         If the new stack segment selector and ESP are beyond the end of the TSS.
                      If the new stack segment selector is NULL.
                      If the RPL of the new stack segment selector in the TSS is not equal to the DPL of the code
                      segment being accessed.
                      If DPL of the stack segment descriptor for the new stack segment is not equal to the DPL of the
                      code segment descriptor.
                      If the new stack segment is not a writable data segment.
                      If segment-selector index for stack segment is outside descriptor table limits.
#PF(fault-code)       If a page fault occurs.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made while the
                      current privilege level is 3.
#UD                   If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                   If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                      If the target offset is beyond the code segment limit.
#UD                   If the LOCK prefix is used.




Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the target offset is beyond the code segment limit.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.
#GP(selector)       If a memory address accessed by the selector is in non-canonical space.
#GP(0)              If the target offset in the destination operand is non-canonical.

64-Bit Mode Exceptions
#GP(0)              If a memory address is non-canonical.
                    If target offset in destination operand is non-canonical.
                    If the segment selector in the destination operand is NULL.
                    If the code segment selector in the 64-bit gate is NULL.
#GP(selector)       If code segment or 64-bit call gate is outside descriptor table limits.
                    If code segment or 64-bit call gate overlaps non-canonical space.
                    If the segment descriptor pointed to by the segment selector in the destination operand is not
                    for a conforming-code segment, nonconforming-code segment, or 64-bit call gate.
                    If the segment descriptor pointed to by the segment selector in the destination operand is a
                    code segment and has both the D-bit and the L- bit set.
                    If the DPL for a nonconforming-code segment is not equal to the CPL, or the RPL for the
                    segment''s segment selector is greater than the CPL.
                    If the DPL for a conforming-code segment is greater than the CPL.
                    If the DPL from a 64-bit call-gate is less than the CPL or than the RPL of the 64-bit call-gate.
                    If the upper type field of a 64-bit call gate is not 0x0.
                    If the segment selector from a 64-bit call gate is beyond the descriptor table limits.
                    If the DPL for a code-segment obtained from a 64-bit call gate is greater than the CPL.
                    If the code segment descriptor pointed to by the selector in the 64-bit gate doesn''t have the L-
                    bit set and the D-bit clear.
                    If the segment descriptor for a segment selector from the 64-bit call gate does not indicate it
                    is a code segment.
#SS(0)              If pushing the return offset or CS selector onto the stack exceeds the bounds of the stack
                    segment when no stack switch occurs.
                    If a memory operand effective address is outside the SS segment limit.
                    If the stack address is in a non-canonical form.
#SS(selector)       If pushing the old values of SS selector, stack pointer, EFLAGS, CS selector, offset, or error
                    code onto the stack violates the canonical boundary when a stack switch occurs.
#NP(selector)       If a code segment or 64-bit call gate is not present.
#TS(selector)       If the load of the new RSP exceeds the limit of the TSS.
#UD                 (64-bit mode only) If a far call is direct to an absolute address in memory.
                    If the LOCK prefix is used.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.




');
INSERT INTO "instructions" VALUES('x86_64','CBW','CBW/CWDE/CDQE-Convert Byte to Word/Convert Word to Doubleword/Convert Doubleword to Quadword
Opcode                    Instruction                 Op/     64-bit     Compat/ Description
                                                      En      Mode       Leg Mode
98                        CBW                         NP      Valid      Valid       AX <- sign-extend of AL.
98                        CWDE                        NP      Valid      Valid       EAX <- sign-extend of AX.
REX.W + 98                CDQE                        NP      Valid      N.E.        RAX <- sign-extend of EAX.



                                                Instruction Operand Encoding
  Op/En             Operand 1                     Operand 2                      Operand 3                     Operand 4
     NP                 NA                           NA                             NA                            NA

Description
Double the size of the source operand by means of sign extension. The CBW (convert byte to word) instruction
copies the sign (bit 7) in the source operand into every bit in the AH register. The CWDE (convert word to double-
word) instruction copies the sign (bit 15) of the word in the AX register into the high 16 bits of the EAX register.
CBW and CWDE reference the same opcode. The CBW instruction is intended for use when the operand-size attri-
bute is 16; CWDE is intended for use when the operand-size attribute is 32. Some assemblers may force the
operand size. Others may treat these two mnemonics as synonyms (CBW/CWDE) and use the setting of the
operand-size attribute to determine the size of values to be converted.
In 64-bit mode, the default operation size is the size of the destination register. Use of the REX.W prefix promotes
this instruction (CDQE when promoted) to operate on 64-bit operands. In which case, CDQE copies the sign (bit
31) of the doubleword in the EAX register into the high 32 bits of RAX.

Operation
IF OperandSize = 16 (* Instruction = CBW *)
    THEN
        AX <- SignExtend(AL);
    ELSE IF (OperandSize = 32, Instruction = CWDE)
        EAX <- SignExtend(AX); FI;
    ELSE (* 64-Bit Mode, OperandSize = 64, Instruction = CDQE*)
        RAX <- SignExtend(EAX);
FI;

Flags Affected
None.

Exceptions (All Operating Modes)
#UD                    If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','CWDE','-R:CBW');
INSERT INTO "instructions" VALUES('x86_64','CDQE','-R:CBW');
INSERT INTO "instructions" VALUES('x86_64','CLAC','CLAC-Clear AC Flag in EFLAGS Register
Opcode/                                   Op /    64/32 bit   CPUID     Description
Instruction                               En      Mode        Feature
                                                  Support     Flag
0F 01 CA                                  NP      V/V         SMAP      Clear the AC flag in the EFLAGS register.
CLAC



                                          Instruction Operand Encoding
  Op/En           Operand 1                Operand 2                    Operand 3                      Operand 4
    NP               NA                          NA                        NA                              NA

Description
Clears the AC flag bit in EFLAGS register. This disables any alignment checking of user-mode data accesses. If the
SMAP bit is set in the CR4 register, this disallows explicit supervisor-mode data accesses to user-mode pages.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode. Attempts to execute CLAC when
CPL > 0 cause #UD.

Operation
EFLAGS.AC <- 0;

Flags Affected
AC cleared. Other flags are unaffected.

Protected Mode Exceptions
#UD                 If the LOCK prefix is used.
                    If the CPL > 0.
                    If CPUID.(EAX=07H, ECX=0H):EBX.SMAP[bit 20] = 0.

Real-Address Mode Exceptions
#UD                 If the LOCK prefix is used.
                    If CPUID.(EAX=07H, ECX=0H):EBX.SMAP[bit 20] = 0.

Virtual-8086 Mode Exceptions
#UD                 The CLAC instruction is not recognized in virtual-8086 mode.

Compatibility Mode Exceptions
#UD                 If the LOCK prefix is used.
                    If the CPL > 0.
                    If CPUID.(EAX=07H, ECX=0H):EBX.SMAP[bit 20] = 0.

64-Bit Mode Exceptions
#UD                 If the LOCK prefix is used.
                    If the CPL > 0.
                    If CPUID.(EAX=07H, ECX=0H):EBX.SMAP[bit 20] = 0.




');
INSERT INTO "instructions" VALUES('x86_64','CLC','CLC-Clear Carry Flag
 Opcode                    Instruction                 Op/   64-bit   Compat/ Description
                                                       En    Mode     Leg Mode
 F8                        CLC                         NP    Valid    Valid      Clear CF flag.



                                               Instruction Operand Encoding
   Op/En               Operand 1                 Operand 2                    Operand 3                   Operand 4
      NP                  NA                           NA                        NA                          NA

Description
Clears the CF flag in the EFLAGS register. Operation is the same in all modes.

Operation
CF <- 0;

Flags Affected
The CF flag is set to 0. The OF, ZF, SF, AF, and PF flags are unaffected.

Exceptions (All Operating Modes)
#UD                      If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','CLD','CLD-Clear Direction Flag
Opcode                 Instruction                 Op/   64-bit   Compat/ Description
                                                   En    Mode     Leg Mode
FC                     CLD                         NP    Valid    Valid         Clear DF flag.



                                            Instruction Operand Encoding
  Op/En            Operand 1                 Operand 2                    Operand 3               Operand 4
     NP               NA                           NA                        NA                       NA

Description
Clears the DF flag in the EFLAGS register. When the DF flag is set to 0, string operations increment the index regis-
ters (ESI and/or EDI). Operation is the same in all modes.

Operation
DF <- 0;

Flags Affected
The DF flag is set to 0. The CF, OF, ZF, SF, AF, and PF flags are unaffected.

Exceptions (All Operating Modes)
#UD                  If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','CLFLUSH','CLFLUSH-Flush Cache Line
 Opcode                    Instruction                 Op/    64-bit      Compat/ Description
                                                       En     Mode        Leg Mode
 0F AE /7                  CLFLUSH m8                  M      Valid       Valid       Flushes cache line containing m8.



                                                 Instruction Operand Encoding
  Op/En             Operand 1                     Operand 2                       Operand 3                    Operand 4
    M             ModRM:r/m (w)                       NA                             NA                            NA

Description
Invalidates from every level of the cache hierarchy in the cache coherence domain the cache line that contains the
linear address specified with the memory operand. If that cache line contains modified data at any level of the
cache hierarchy, that data is written back to memory. The source operand is a byte memory location.
The availability of CLFLUSH is indicated by the presence of the CPUID feature flag CLFSH
(CPUID.01H:EDX[bit 19]). The aligned cache line size affected is also indicated with the CPUID instruction (bits 8
through 15 of the EBX register when the initial value in the EAX register is 1).
The memory attribute of the page containing the affected line has no effect on the behavior of this instruction. It
should be noted that processors are free to speculatively fetch and cache data from system memory regions
assigned a memory-type allowing for speculative reads (such as, the WB, WC, and WT memory types). PREFETCHh
instructions can be used to provide the processor with hints for this speculative behavior. Because this speculative
fetching can occur at any time and is not tied to instruction execution, the CLFLUSH instruction is not ordered with
respect to PREFETCHh instructions or any of the speculative fetching mechanisms (that is, data can be specula-
tively loaded into a cache line just before, during, or after the execution of a CLFLUSH instruction that references
the cache line).
Executions of the CLFLUSH instruction are ordered with respect to each other and with respect to writes, locked
read-modify-write instructions, fence instructions, and executions of CLFLUSHOPT to the same cache line.1 They
are not ordered with respect to executions of CLFLUSHOPT to different cache lines.
The CLFLUSH instruction can be used at all privilege levels and is subject to all permission checking and faults asso-
ciated with a byte load (and in addition, a CLFLUSH instruction is allowed to flush a linear address in an execute-
only segment). Like a load, the CLFLUSH instruction sets the A bit but not the D bit in the page tables.
In some implementations, the CLFLUSH instruction may always cause transactional abort with Transactional
Synchronization Extensions (TSX). The CLFLUSH instruction is not expected to be commonly used inside typical
transactional regions. However, programmers must not rely on CLFLUSH instruction to force a transactional abort,
since whether they cause transactional abort is implementation dependent.
The CLFLUSH instruction was introduced with the SSE2 extensions; however, because it has its own CPUID feature
flag, it can be implemented in IA-32 processors that do not include the SSE2 extensions. Also, detecting the pres-
ence of the SSE2 extensions with the CPUID instruction does not guarantee that the CLFLUSH instruction is imple-
mented in the processor.
CLFLUSH operation is the same in non-64-bit modes and 64-bit mode.

Operation
Flush_Cache_Line(SRC);

Intel C/C++ Compiler Intrinsic Equivalents
CLFLUSH:    void _mm_clflush(void const *p)




1. Earlier versions of this manual specified that executions of the CLFLUSH instruction were ordered only by the MFENCE instruction.
   All processors implementing the CLFLUSH instruction also order it relative to the other operations enumerated above.



Protected Mode Exceptions
#GP(0)              For an illegal memory operand effective address in the CS, DS, ES, FS or GS segments.
#SS(0)              For an illegal address in the SS segment.
#PF(fault-code)     For a page fault.
#UD                 If CPUID.01H:EDX.CLFSH[bit 19] = 0.
                    If the LOCK prefix is used.
                    If an instruction prefix F2H or F3H is used.

Real-Address Mode Exceptions
#GP                 If any part of the operand lies outside the effective address space from 0 to FFFFH.
#UD                 If CPUID.01H:EDX.CLFSH[bit 19] = 0.
                    If the LOCK prefix is used.
                    If an instruction prefix F2H or F3H is used.

Virtual-8086 Mode Exceptions
Same exceptions as in real address mode.
#PF(fault-code)     For a page fault.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#PF(fault-code)     For a page fault.
#UD                 If CPUID.01H:EDX.CLFSH[bit 19] = 0.
                    If the LOCK prefix is used.
                    If an instruction prefix F2H or F3H is used.




');
INSERT INTO "instructions" VALUES('x86_64','CLFLUSHOPT','CLFLUSHOPT-Flush Cache Line Optimized
 Opcode                   Instruction                Op/     64-bit   Compat/ Description
                                                     En      Mode     Leg Mode
 66 0F AE /7              CLFLUSHOPT m8              M       Valid    Valid       Flushes cache line containing m8.



                                                Instruction Operand Encoding
  Op/En             Operand 1                    Operand 2                    Operand 3                    Operand 4
    M             ModRM:r/m (w)                     NA                           NA                           NA

Description
Invalidates from every level of the cache hierarchy in the cache coherence domain the cache line that contains the
linear address specified with the memory operand. If that cache line contains modified data at any level of the
cache hierarchy, that data is written back to memory. The source operand is a byte memory location.
The availability of CLFLUSHOPT is indicated by the presence of the CPUID feature flag CLFLUSHOPT
(CPUID.(EAX=7,ECX=0):EBX[bit 23]). The aligned cache line size affected is also indicated with the CPUID instruc-
tion (bits 8 through 15 of the EBX register when the initial value in the EAX register is 1).
The memory attribute of the page containing the affected line has no effect on the behavior of this instruction. It
should be noted that processors are free to speculatively fetch and cache data from system memory regions
assigned a memory-type allowing for speculative reads (such as, the WB, WC, and WT memory types). PREFETCHh
instructions can be used to provide the processor with hints for this speculative behavior. Because this speculative
fetching can occur at any time and is not tied to instruction execution, the CLFLUSH instruction is not ordered with
respect to PREFETCHh instructions or any of the speculative fetching mechanisms (that is, data can be specula-
tively loaded into a cache line just before, during, or after the execution of a CLFLUSH instruction that references
the cache line).
Executions of the CLFLUSHOPT instruction are ordered with respect to fence instructions and to locked read-
modify-write instructions; they are also ordered with respect to the following accesses to the cache line being
invalidated: writes, executions of CLFLUSH, and executions of CLFLUSHOPT. They are not ordered with respect to
writes, executions of CLFLUSH, or executions of CLFLUSHOPT that access other cache lines; to enforce ordering
with such an operation, software can insert an SFENCE instruction between CFLUSHOPT and that operation.
The CLFLUSHOPT instruction can be used at all privilege levels and is subject to all permission checking and faults
associated with a byte load (and in addition, a CLFLUSHOPT instruction is allowed to flush a linear address in an
execute-only segment). Like a load, the CLFLUSHOPT instruction sets the A bit but not the D bit in the page tables.
In some implementations, the CLFLUSHOPT instruction may always cause transactional abort with Transactional
Synchronization Extensions (TSX). The CLFLUSHOPT instruction is not expected to be commonly used inside
typical transactional regions. However, programmers must not rely on CLFLUSHOPT instruction to force a transac-
tional abort, since whether they cause transactional abort is implementation dependent.
CLFLUSHOPT operation is the same in non-64-bit modes and 64-bit mode.

Operation
Flush_Cache_Line_Optimized(SRC);

Intel C/C++ Compiler Intrinsic Equivalents
CLFLUSHOPT:void _mm_clflushopt(void const *p)





Protected Mode Exceptions
#GP(0)              For an illegal memory operand effective address in the CS, DS, ES, FS or GS segments.
#SS(0)              For an illegal address in the SS segment.
#PF(fault-code)     For a page fault.
#UD                 If CPUID.(EAX=7,ECX=0):EBX.CLFLUSHOPT[bit 23] = 0.
                    If the LOCK prefix is used.
                    If an instruction prefix F2H or F3H is used.

Real-Address Mode Exceptions
#GP                 If any part of the operand lies outside the effective address space from 0 to FFFFH.
#UD                 If CPUID.(EAX=7,ECX=0):EBX.CLFLUSHOPT[bit 23] = 0.
                    If the LOCK prefix is used.
                    If an instruction prefix F2H or F3H is used.

Virtual-8086 Mode Exceptions
Same exceptions as in real address mode.
#PF(fault-code)     For a page fault.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#PF(fault-code)     For a page fault.
#UD                 If CPUID.(EAX=7,ECX=0):EBX.CLFLUSHOPT[bit 23] = 0.
                    If the LOCK prefix is used.
                    If an instruction prefix F2H or F3H is used.




');
INSERT INTO "instructions" VALUES('x86_64','CLI','CLI - Clear Interrupt Flag
 Opcode                       Instruction                  Op/    64-bit        Compat/ Description
                                                           En     Mode          Leg Mode
 FA                           CLI                          NP     Valid         Valid         Clear interrupt flag; interrupts disabled when
                                                                                              interrupt flag cleared.



                                                    Instruction Operand Encoding
  Op/En                 Operand 1                     Operand 2                          Operand 3                        Operand 4
      NP                     NA                            NA                                 NA                              NA

Description
If protected-mode virtual interrupts are not enabled, CLI clears the IF flag in the EFLAGS register. No other flags
are affected. Clearing the IF flag causes the processor to ignore maskable external interrupts. The IF flag and the
CLI and STI instruction have no affect on the generation of exceptions and NMI interrupts.
When protected-mode virtual interrupts are enabled, CPL is 3, and IOPL is less than 3; CLI clears the VIF flag in the
EFLAGS register, leaving IF unaffected. Table 3-7 indicates the action of the CLI instruction depending on the
processor operating mode and the CPL/IOPL of the running program or procedure.
Operation is the same in all modes.


                                             Table 3-7. Decision Table for CLI Results
       PE               VM           IOPL            CPL            PVI                 VIP          VME                   CLI Result
           0             X             X              X              X                  X             X         IF = 0
           1             0           >= CPL            X              X                  X             X         IF = 0
           1             0           < CPL            3              1                  X             X         VIF = 0
           1             0           < CPL           <3              X                  X             X         GP Fault
           1             0           < CPL            X              0                  X             X         GP Fault
           1             1             3              X              X                  X             X         IF = 0
           1             1            <3              X              X                  X             1         VIF = 0
           1             1            <3              X              X                  X             0         GP Fault
 NOTES:
 * X = This setting has no impact.

Operation
IF PE = 0
    THEN
         IF <- 0; (* Reset Interrupt Flag *)
    ELSE
         IF VM = 0;
             THEN
                   IF IOPL >= CPL
                        THEN
                              IF <- 0; (* Reset Interrupt Flag *)
                   ELSE
                        IF ((IOPL < CPL) and (CPL = 3) and (PVI = 1))
                              THEN
                                  VIF <- 0; (* Reset Virtual Interrupt Flag *)
                              ELSE
                                  #GP(0);


                       FI;
                  FI;
              ELSE (* VM = 1 *)
                  IF IOPL = 3
                       THEN
                            IF <- 0; (* Reset Interrupt Flag *)
                       ELSE
                            IF (IOPL < 3) AND (VME = 1)
                                  THEN
                                      VIF <- 0; (* Reset Virtual Interrupt Flag *)
                                  ELSE
                                      #GP(0);
                            FI;
                  FI;
        FI;
FI;

Flags Affected
If protected-mode virtual interrupts are not enabled, IF is set to 0 if the CPL is equal to or less than the IOPL; other-
wise, it is not affected. Other flags are unaffected.
When protected-mode virtual interrupts are enabled, CPL is 3, and IOPL is less than 3; CLI clears the VIF flag in the
EFLAGS register, leaving IF unaffected. Other flags are unaffected.

Protected Mode Exceptions
#GP(0)                   If the CPL is greater (has less privilege) than the IOPL of the current program or procedure.
#UD                      If the LOCK prefix is used.

Real-Address Mode Exceptions
#UD                      If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)                   If the CPL is greater (has less privilege) than the IOPL of the current program or procedure.
#UD                      If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#GP(0)                   If the CPL is greater (has less privilege) than the IOPL of the current program or procedure.
#UD                      If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','CLTS','CLTS-Clear Task-Switched Flag in CR0
 Opcode                     Instruction                Op/    64-bit   Compat/ Description
                                                       En     Mode     Leg Mode
 0F 06                      CLTS                       NP     Valid    Valid       Clears TS flag in CR0.



                                                Instruction Operand Encoding
   Op/En              Operand 1                   Operand 2                    Operand 3                    Operand 4
    NP                    NA                           NA                         NA                           NA

Description
Clears the task-switched (TS) flag in the CR0 register. This instruction is intended for use in operating-system
procedures. It is a privileged instruction that can only be executed at a CPL of 0. It is allowed to be executed in real-
address mode to allow initialization for protected mode.
The processor sets the TS flag every time a task switch occurs. The flag is used to synchronize the saving of FPU
context in multitasking applications. See the description of the TS flag in the section titled "Control Registers" in
Chapter 2 of the Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 3A, for more information
about this flag.
CLTS operation is the same in non-64-bit modes and 64-bit mode.
See Chapter 25, "VMX Non-Root Operation," of the Intel 64 and IA-32 Architectures Software Developer''s
Manual, Volume 3C, for more information about the behavior of this instruction in VMX non-root operation.

Operation
CR0.TS[bit 3] <- 0;

Flags Affected
The TS flag in CR0 register is cleared.

Protected Mode Exceptions
#GP(0)                   If the current privilege level is not 0.
#UD                      If the LOCK prefix is used.

Real-Address Mode Exceptions
#UD                      If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)                   CLTS is not recognized in virtual-8086 mode.
#UD                      If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#GP(0)                   If the CPL is greater than 0.
#UD                      If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','CLWB','CLWB-Cache Line Write Back
 Opcode/                     Op/      64/32 bit   CPUID          Description
 Instruction                 En       Mode        Feature Flag
                                      Support
 66 0F AE /6                 M        V/V         CLWB           Writes back modified cache line containing m8, and may
 CLWB m8                                                         retain the line in cache hierarchy in non-modified state.


                                             Instruction Operand Encoding1
     Op/En              Operand 1                  Operand 2                   Operand 3                     Operand 4
        M             ModRM:r/m (w)                   NA                          NA                            NA

Description
Writes back to memory the cache line (if modified) that contains the linear address specified with the memory
operand from any level of the cache hierarchy in the cache coherence domain. The line may be retained in the
cache hierarchy in non-modified state. Retaining the line in the cache hierarchy is a performance optimization
(treated as a hint by hardware) to reduce the possibility of cache miss on a subsequent access. Hardware may
choose to retain the line at any of the levels in the cache hierarchy, and in some cases, may invalidate the line from
the cache hierarchy. The source operand is a byte memory location.
The availability of CLWB instruction is indicated by the presence of the CPUID feature flag CLWB (bit 24 of the EBX
register, see "CPUID - CPU Identification" in this chapter). The aligned cache line size affected is also indicated
with the CPUID instruction (bits 8 through 15 of the EBX register when the initial value in the EAX register is 1).
The memory attribute of the page containing the affected line has no effect on the behavior of this instruction. It
should be noted that processors are free to speculatively fetch and cache data from system memory regions that
are assigned a memory-type allowing for speculative reads (such as, the WB, WC, and WT memory types).
PREFETCHh instructions can be used to provide the processor with hints for this speculative behavior. Because this
speculative fetching can occur at any time and is not tied to instruction execution, the CLWB instruction is not
ordered with respect to PREFETCHh instructions or any of the speculative fetching mechanisms (that is, data can
be speculatively loaded into a cache line just before, during, or after the execution of a CLWB instruction that refer-
ences the cache line).
CLWB instruction is ordered only by store-fencing operations. For example, software can use an SFENCE, MFENCE,
XCHG, or LOCK-prefixed instructions to ensure that previous stores are included in the write-back. CLWB instruc-
tion need not be ordered by another CLWB or CLFLUSHOPT instruction. CLWB is implicitly ordered with older stores
executed by the logical processor to the same address.
For usages that require only writing back modified data from cache lines to memory (do not require the line to be
invalidated), and expect to subsequently access the data, software is recommended to use CLWB (with appropriate
fencing) instead of CLFLUSH or CLFLUSHOPT for improved performance.
The CLWB instruction can be used at all privilege levels and is subject to all permission checking and faults associ-
ated with a byte load. Like a load, the CLWB instruction sets the accessed flag but not the dirty flag in the page
tables.
In some implementations, the CLWB instruction may always cause transactional abort with Transactional Synchro-
nization Extensions (TSX). CLWB instruction is not expected to be commonly used inside typical transactional
regions. However, programmers must not rely on CLWB instruction to force a transactional abort, since whether
they cause transactional abort is implementation dependent.

Operation
Cache_Line_Write_Back(m8);

Flags Affected
None.


1. ModRM.MOD != 011B



C/C++ Compiler Intrinsic Equivalent
CLWB void _mm_clwb(void const *p);

Protected Mode Exceptions
#UD                     If the LOCK prefix is used.
                        If CPUID.(EAX=07H, ECX=0H):EBX.CLWB[bit 24] = 0.
#GP(0)                  For an illegal memory operand effective address in the CS, DS, ES, FS or GS segments.
#SS(0)                  For an illegal address in the SS segment.
#PF(fault-code)         For a page fault.

Real-Address Mode Exceptions
#UD                     If the LOCK prefix is used.
                        If CPUID.(EAX=07H, ECX=0H):EBX.CLWB[bit 24] = 0.
#GP                     If any part of the operand lies outside the effective address space from 0 to FFFFH.

Virtual-8086 Mode Exceptions
Same exceptions as in real address mode.
#PF(fault-code)         For a page fault.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#UD                     If the LOCK prefix is used.
                        If CPUID.(EAX=07H, ECX=0H):EBX.CLWB[bit 24] = 0.
#SS(0)                  If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)                  If the memory address is in a non-canonical form.
#PF(fault-code)         For a page fault.




');
INSERT INTO "instructions" VALUES('x86_64','CMC','CMC-Complement Carry Flag
Opcode                  Instruction                 Op/   64-bit   Compat/ Description
                                                    En    Mode     Leg Mode
F5                      CMC                         NP    Valid    Valid        Complement CF flag.



                                            Instruction Operand Encoding
  Op/En            Operand 1                 Operand 2                     Operand 3                     Operand 4
     NP               NA                        NA                            NA                             NA

Description
Complements the CF flag in the EFLAGS register. CMC operation is the same in non-64-bit modes and 64-bit mode.

Operation
EFLAGS.CF[bit 0]<- NOT EFLAGS.CF[bit 0];

Flags Affected
The CF flag contains the complement of its original value. The OF, ZF, SF, AF, and PF flags are unaffected.

Exceptions (All Operating Modes)
#UD                   If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','CMOVcc','CMOVcc-Conditional Move
 Opcode                   Instruction          Op/   64-Bit   Compat/ Description
                                               En    Mode     Leg Mode
 0F 47 /r                 CMOVA r16, r/m16     RM    Valid    Valid    Move if above (CF=0 and ZF=0).
 0F 47 /r                 CMOVA r32, r/m32     RM    Valid    Valid    Move if above (CF=0 and ZF=0).
 REX.W + 0F 47 /r         CMOVA r64, r/m64     RM    Valid    N.E.     Move if above (CF=0 and ZF=0).
 0F 43 /r                 CMOVAE r16, r/m16    RM    Valid    Valid    Move if above or equal (CF=0).
 0F 43 /r                 CMOVAE r32, r/m32    RM    Valid    Valid    Move if above or equal (CF=0).
 REX.W + 0F 43 /r         CMOVAE r64, r/m64    RM    Valid    N.E.     Move if above or equal (CF=0).
 0F 42 /r                 CMOVB r16, r/m16     RM    Valid    Valid    Move if below (CF=1).
 0F 42 /r                 CMOVB r32, r/m32     RM    Valid    Valid    Move if below (CF=1).
 REX.W + 0F 42 /r         CMOVB r64, r/m64     RM    Valid    N.E.     Move if below (CF=1).
 0F 46 /r                 CMOVBE r16, r/m16    RM    Valid    Valid    Move if below or equal (CF=1 or ZF=1).
 0F 46 /r                 CMOVBE r32, r/m32    RM    Valid    Valid    Move if below or equal (CF=1 or ZF=1).
 REX.W + 0F 46 /r         CMOVBE r64, r/m64    RM    Valid    N.E.     Move if below or equal (CF=1 or ZF=1).
 0F 42 /r                 CMOVC r16, r/m16     RM    Valid    Valid    Move if carry (CF=1).
 0F 42 /r                 CMOVC r32, r/m32     RM    Valid    Valid    Move if carry (CF=1).
 REX.W + 0F 42 /r         CMOVC r64, r/m64     RM    Valid    N.E.     Move if carry (CF=1).
 0F 44 /r                 CMOVE r16, r/m16     RM    Valid    Valid    Move if equal (ZF=1).
 0F 44 /r                 CMOVE r32, r/m32     RM    Valid    Valid    Move if equal (ZF=1).
 REX.W + 0F 44 /r         CMOVE r64, r/m64     RM    Valid    N.E.     Move if equal (ZF=1).
 0F 4F /r                 CMOVG r16, r/m16     RM    Valid    Valid    Move if greater (ZF=0 and SF=OF).
 0F 4F /r                 CMOVG r32, r/m32     RM    Valid    Valid    Move if greater (ZF=0 and SF=OF).
 REX.W + 0F 4F /r         CMOVG r64, r/m64     RM    V/N.E.   NA       Move if greater (ZF=0 and SF=OF).
 0F 4D /r                 CMOVGE r16, r/m16    RM    Valid    Valid    Move if greater or equal (SF=OF).
 0F 4D /r                 CMOVGE r32, r/m32    RM    Valid    Valid    Move if greater or equal (SF=OF).
 REX.W + 0F 4D /r         CMOVGE r64, r/m64    RM    Valid    N.E.     Move if greater or equal (SF=OF).
 0F 4C /r                 CMOVL r16, r/m16     RM    Valid    Valid    Move if less (SF!= OF).
 0F 4C /r                 CMOVL r32, r/m32     RM    Valid    Valid    Move if less (SF!= OF).
 REX.W + 0F 4C /r         CMOVL r64, r/m64     RM    Valid    N.E.     Move if less (SF!= OF).
 0F 4E /r                 CMOVLE r16, r/m16    RM    Valid    Valid    Move if less or equal (ZF=1 or SF!= OF).
 0F 4E /r                 CMOVLE r32, r/m32    RM    Valid    Valid    Move if less or equal (ZF=1 or SF!= OF).
 REX.W + 0F 4E /r         CMOVLE r64, r/m64    RM    Valid    N.E.     Move if less or equal (ZF=1 or SF!= OF).
 0F 46 /r                 CMOVNA r16, r/m16    RM    Valid    Valid    Move if not above (CF=1 or ZF=1).
 0F 46 /r                 CMOVNA r32, r/m32    RM    Valid    Valid    Move if not above (CF=1 or ZF=1).
 REX.W + 0F 46 /r         CMOVNA r64, r/m64    RM    Valid    N.E.     Move if not above (CF=1 or ZF=1).
 0F 42 /r                 CMOVNAE r16, r/m16   RM    Valid    Valid    Move if not above or equal (CF=1).
 0F 42 /r                 CMOVNAE r32, r/m32   RM    Valid    Valid    Move if not above or equal (CF=1).
 REX.W + 0F 42 /r         CMOVNAE r64, r/m64   RM    Valid    N.E.     Move if not above or equal (CF=1).
 0F 43 /r                 CMOVNB r16, r/m16    RM    Valid    Valid    Move if not below (CF=0).
 0F 43 /r                 CMOVNB r32, r/m32    RM    Valid    Valid    Move if not below (CF=0).
 REX.W + 0F 43 /r         CMOVNB r64, r/m64    RM    Valid    N.E.     Move if not below (CF=0).
 0F 47 /r                 CMOVNBE r16, r/m16   RM    Valid    Valid    Move if not below or equal (CF=0 and ZF=0).



 Opcode               Instruction          Op/   64-Bit   Compat/ Description
                                           En    Mode     Leg Mode
 0F 47 /r             CMOVNBE r32, r/m32   RM    Valid    Valid    Move if not below or equal (CF=0 and ZF=0).
 REX.W + 0F 47 /r     CMOVNBE r64, r/m64   RM    Valid    N.E.     Move if not below or equal (CF=0 and ZF=0).
 0F 43 /r             CMOVNC r16, r/m16    RM    Valid    Valid    Move if not carry (CF=0).
 0F 43 /r             CMOVNC r32, r/m32    RM    Valid    Valid    Move if not carry (CF=0).
 REX.W + 0F 43 /r     CMOVNC r64, r/m64    RM    Valid    N.E.     Move if not carry (CF=0).
 0F 45 /r             CMOVNE r16, r/m16    RM    Valid    Valid    Move if not equal (ZF=0).
 0F 45 /r             CMOVNE r32, r/m32    RM    Valid    Valid    Move if not equal (ZF=0).
 REX.W + 0F 45 /r     CMOVNE r64, r/m64    RM    Valid    N.E.     Move if not equal (ZF=0).
 0F 4E /r             CMOVNG r16, r/m16    RM    Valid    Valid    Move if not greater (ZF=1 or SF!= OF).
 0F 4E /r             CMOVNG r32, r/m32    RM    Valid    Valid    Move if not greater (ZF=1 or SF!= OF).
 REX.W + 0F 4E /r     CMOVNG r64, r/m64    RM    Valid    N.E.     Move if not greater (ZF=1 or SF!= OF).
 0F 4C /r             CMOVNGE r16, r/m16   RM    Valid    Valid    Move if not greater or equal (SF!= OF).
 0F 4C /r             CMOVNGE r32, r/m32   RM    Valid    Valid    Move if not greater or equal (SF!= OF).
 REX.W + 0F 4C /r     CMOVNGE r64, r/m64   RM    Valid    N.E.     Move if not greater or equal (SF!= OF).
 0F 4D /r             CMOVNL r16, r/m16    RM    Valid    Valid    Move if not less (SF=OF).
 0F 4D /r             CMOVNL r32, r/m32    RM    Valid    Valid    Move if not less (SF=OF).
 REX.W + 0F 4D /r     CMOVNL r64, r/m64    RM    Valid    N.E.     Move if not less (SF=OF).
 0F 4F /r             CMOVNLE r16, r/m16   RM    Valid    Valid    Move if not less or equal (ZF=0 and SF=OF).
 0F 4F /r             CMOVNLE r32, r/m32   RM    Valid    Valid    Move if not less or equal (ZF=0 and SF=OF).
 REX.W + 0F 4F /r     CMOVNLE r64, r/m64   RM    Valid    N.E.     Move if not less or equal (ZF=0 and SF=OF).
 0F 41 /r             CMOVNO r16, r/m16    RM    Valid    Valid    Move if not overflow (OF=0).
 0F 41 /r             CMOVNO r32, r/m32    RM    Valid    Valid    Move if not overflow (OF=0).
 REX.W + 0F 41 /r     CMOVNO r64, r/m64    RM    Valid    N.E.     Move if not overflow (OF=0).
 0F 4B /r             CMOVNP r16, r/m16    RM    Valid    Valid    Move if not parity (PF=0).
 0F 4B /r             CMOVNP r32, r/m32    RM    Valid    Valid    Move if not parity (PF=0).
 REX.W + 0F 4B /r     CMOVNP r64, r/m64    RM    Valid    N.E.     Move if not parity (PF=0).
 0F 49 /r             CMOVNS r16, r/m16    RM    Valid    Valid    Move if not sign (SF=0).
 0F 49 /r             CMOVNS r32, r/m32    RM    Valid    Valid    Move if not sign (SF=0).
 REX.W + 0F 49 /r     CMOVNS r64, r/m64    RM    Valid    N.E.     Move if not sign (SF=0).
 0F 45 /r             CMOVNZ r16, r/m16    RM    Valid    Valid    Move if not zero (ZF=0).
 0F 45 /r             CMOVNZ r32, r/m32    RM    Valid    Valid    Move if not zero (ZF=0).
 REX.W + 0F 45 /r     CMOVNZ r64, r/m64    RM    Valid    N.E.     Move if not zero (ZF=0).
 0F 40 /r             CMOVO r16, r/m16     RM    Valid    Valid    Move if overflow (OF=1).
 0F 40 /r             CMOVO r32, r/m32     RM    Valid    Valid    Move if overflow (OF=1).
 REX.W + 0F 40 /r     CMOVO r64, r/m64     RM    Valid    N.E.     Move if overflow (OF=1).
 0F 4A /r             CMOVP r16, r/m16     RM    Valid    Valid    Move if parity (PF=1).
 0F 4A /r             CMOVP r32, r/m32     RM    Valid    Valid    Move if parity (PF=1).
 REX.W + 0F 4A /r     CMOVP r64, r/m64     RM    Valid    N.E.     Move if parity (PF=1).
 0F 4A /r             CMOVPE r16, r/m16    RM    Valid    Valid    Move if parity even (PF=1).
 0F 4A /r             CMOVPE r32, r/m32    RM    Valid    Valid    Move if parity even (PF=1).
 REX.W + 0F 4A /r     CMOVPE r64, r/m64    RM    Valid    N.E.     Move if parity even (PF=1).



 Opcode                   Instruction                    Op/    64-Bit   Compat/ Description
                                                         En     Mode     Leg Mode
 0F 4B /r                 CMOVPO r16, r/m16              RM     Valid    Valid       Move if parity odd (PF=0).
 0F 4B /r                 CMOVPO r32, r/m32              RM     Valid    Valid       Move if parity odd (PF=0).
 REX.W + 0F 4B /r         CMOVPO r64, r/m64              RM     Valid    N.E.        Move if parity odd (PF=0).
 0F 48 /r                 CMOVS r16, r/m16               RM     Valid    Valid       Move if sign (SF=1).
 0F 48 /r                 CMOVS r32, r/m32               RM     Valid    Valid       Move if sign (SF=1).
 REX.W + 0F 48 /r         CMOVS r64, r/m64               RM     Valid    N.E.        Move if sign (SF=1).
 0F 44 /r                 CMOVZ r16, r/m16               RM     Valid    Valid       Move if zero (ZF=1).
 0F 44 /r                 CMOVZ r32, r/m32               RM     Valid    Valid       Move if zero (ZF=1).
 REX.W + 0F 44 /r         CMOVZ r64, r/m64               RM     Valid    N.E.        Move if zero (ZF=1).



                                                   Instruction Operand Encoding
  Op/En             Operand 1                       Operand 2                    Operand 3                    Operand 4
    RM           ModRM:reg (r, w)                  ModRM:r/m (r)                    NA                            NA

Description
The CMOVcc instructions check the state of one or more of the status flags in the EFLAGS register (CF, OF, PF, SF,
and ZF) and perform a move operation if the flags are in a specified state (or condition). A condition code (cc) is
associated with each instruction to indicate the condition being tested for. If the condition is not satisfied, a move
is not performed and execution continues with the instruction following the CMOVcc instruction.
These instructions can move 16-bit, 32-bit or 64-bit values from memory to a general-purpose register or from one
general-purpose register to another. Conditional moves of 8-bit register operands are not supported.
The condition for each CMOVcc mnemonic is given in the description column of the above table. The terms "less"
and "greater" are used for comparisons of signed integers and the terms "above" and "below" are used for
unsigned integers.
Because a particular state of the status flags can sometimes be interpreted in two ways, two mnemonics are
defined for some opcodes. For example, the CMOVA (conditional move if above) instruction and the CMOVNBE
(conditional move if not below or equal) instruction are alternate mnemonics for the opcode 0F 47H.
The CMOVcc instructions were introduced in P6 family processors; however, these instructions may not be
supported by all IA-32 processors. Software can determine if the CMOVcc instructions are supported by checking
the processor''s feature information with the CPUID instruction (see "CPUID-CPU Identification" in this chapter).
In 64-bit mode, the instruction''s default operation size is 32 bits. Use of the REX.R prefix permits access to addi-
tional registers (R8-R15). Use of the REX.W prefix promotes operation to 64 bits. See the summary chart at the
beginning of this section for encoding data and limits.

Operation
temp <- SRC

IF condition TRUE
    THEN
         DEST <- temp;
    FI;
ELSE
    IF (OperandSize = 32 and IA-32e mode active)
         THEN
              DEST[63:32] <- 0;
    FI;
FI;



Flags Affected
None.

Protected Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#UD                 If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','CMP','CMP-Compare Two Operands
Opcode                     Instruction                 Op/     64-Bit      Compat/ Description
                                                       En      Mode        Leg Mode
3C ib                      CMP AL, imm8                I       Valid       Valid       Compare imm8 with AL.
3D iw                      CMP AX, imm16               I       Valid       Valid       Compare imm16 with AX.
3D id                      CMP EAX, imm32              I       Valid       Valid       Compare imm32 with EAX.
REX.W + 3D id              CMP RAX, imm32              I       Valid       N.E.        Compare imm32 sign-extended to 64-bits
                                                                                       with RAX.
80 /7 ib                   CMP r/m8, imm8              MI      Valid       Valid       Compare imm8 with r/m8.
REX + 80 /7 ib             CMP r/m8*, imm8             MI      Valid       N.E.        Compare imm8 with r/m8.
81 /7 iw                   CMP r/m16, imm16            MI      Valid       Valid       Compare imm16 with r/m16.
81 /7 id                   CMP r/m32, imm32            MI      Valid       Valid       Compare imm32 with r/m32.
REX.W + 81 /7 id           CMP r/m64, imm32            MI      Valid       N.E.        Compare imm32 sign-extended to 64-bits
                                                                                       with r/m64.
83 /7 ib                   CMP r/m16, imm8             MI      Valid       Valid       Compare imm8 with r/m16.
83 /7 ib                   CMP r/m32, imm8             MI      Valid       Valid       Compare imm8 with r/m32.
REX.W + 83 /7 ib           CMP r/m64, imm8             MI      Valid       N.E.        Compare imm8 with r/m64.
38 /r                      CMP r/m8, r8                MR      Valid       Valid       Compare r8 with r/m8.
                                       *   *
REX + 38 /r                CMP r/m8 , r8               MR      Valid       N.E.        Compare r8 with r/m8.
39 /r                      CMP r/m16, r16              MR      Valid       Valid       Compare r16 with r/m16.
39 /r                      CMP r/m32, r32              MR      Valid       Valid       Compare r32 with r/m32.
REX.W + 39 /r              CMP r/m64,r64               MR      Valid       N.E.        Compare r64 with r/m64.
3A /r                      CMP r8, r/m8                RM      Valid       Valid       Compare r/m8 with r8.
                                   *       *
REX + 3A /r                CMP r8 , r/m8               RM      Valid       N.E.        Compare r/m8 with r8.
3B /r                      CMP r16, r/m16              RM      Valid       Valid       Compare r/m16 with r16.
3B /r                      CMP r32, r/m32              RM      Valid       Valid       Compare r/m32 with r32.
REX.W + 3B /r              CMP r64, r/m64              RM      Valid       N.E.        Compare r/m64 with r64.
NOTES:
* In 64-bit mode, r/m8 can not be encoded to access the following byte registers if a REX prefix is used: AH, BH, CH, DH.



                                                 Instruction Operand Encoding
  Op/En             Operand 1                      Operand 2                       Operand 3                     Operand 4
    RM             ModRM:reg (r)                 ModRM:r/m (r)                        NA                             NA
    MR             ModRM:r/m (r)                 ModRM:reg (r)                        NA                             NA
    MI             ModRM:r/m (r)                     imm8                             NA                             NA
     I           AL/AX/EAX/RAX (r)                   imm8                             NA                             NA

Description
Compares the first source operand with the second source operand and sets the status flags in the EFLAGS register
according to the results. The comparison is performed by subtracting the second operand from the first operand
and then setting the status flags in the same manner as the SUB instruction. When an immediate value is used as
an operand, it is sign-extended to the length of the first operand.
The condition codes used by the Jcc, CMOVcc, and SETcc instructions are based on the results of a CMP instruction.
Appendix B, "EFLAGS Condition Codes," in the Intel 64 and IA-32 Architectures Software Developer''s Manual,
Volume 1, shows the relationship of the status flags and the condition codes.



In 64-bit mode, the instruction''s default operation size is 32 bits. Use of the REX.R prefix permits access to addi-
tional registers (R8-R15). Use of the REX.W prefix promotes operation to 64 bits. See the summary chart at the
beginning of this section for encoding data and limits.

Operation
temp <- SRC1 - SignExtend(SRC2);
ModifyStatusFlags; (* Modify status flags in the same manner as the SUB instruction*)

Flags Affected
The CF, OF, SF, ZF, AF, and PF flags are set according to the result.

Protected Mode Exceptions
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                       If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                    If a memory operand effective address is outside the SS segment limit.

Virtual-8086 Mode Exceptions
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made.
#UD                    If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)                 If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)                 If the memory address is in a non-canonical form.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                    If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','CMPPD','CMPPD-Compare Packed Double-Precision Floating-Point Values
 Opcode/                                    Op /     64/32        CPUID      Description
 Instruction                                En       bit Mode     Feature
                                                     Support      Flag
 66 0F C2 /r ib                             RMI      V/V          SSE2       Compare packed double-precision floating-point values
 CMPPD xmm1, xmm2/m128, imm8                                                 in xmm2/m128 and xmm1 using bits 2:0 of imm8 as a
                                                                             comparison predicate.
 VEX.NDS.128.66.0F.WIG C2 /r ib             RVMI     V/V          AVX        Compare packed double-precision floating-point values
 VCMPPD xmm1, xmm2, xmm3/m128,                                               in xmm3/m128 and xmm2 using bits 4:0 of imm8 as a
 imm8                                                                        comparison predicate.
 VEX.NDS.256.66.0F.WIG C2 /r ib             RVMI     V/V          AVX        Compare packed double-precision floating-point values
 VCMPPD ymm1, ymm2, ymm3/m256,                                               in ymm3/m256 and ymm2 using bits 4:0 of imm8 as a
 imm8                                                                        comparison predicate.
 EVEX.NDS.128.66.0F.W1 C2 /r ib             FV       V/V          AVX512VL   Compare packed double-precision floating-point values
 VCMPPD k1 {k2}, xmm2,                                            AVX512F    in xmm3/m128/m64bcst and xmm2 using bits 4:0 of
 xmm3/m128/m64bcst, imm8                                                     imm8 as a comparison predicate with writemask k2
                                                                             and leave the result in mask register k1.
 EVEX.NDS.256.66.0F.W1 C2 /r ib             FV       V/V          AVX512VL   Compare packed double-precision floating-point values
 VCMPPD k1 {k2}, ymm2,                                            AVX512F    in ymm3/m256/m64bcst and ymm2 using bits 4:0 of
 ymm3/m256/m64bcst, imm8                                                     imm8 as a comparison predicate with writemask k2
                                                                             and leave the result in mask register k1.
 EVEX.NDS.512.66.0F.W1 C2 /r ib             FV       V/V          AVX512F    Compare packed double-precision floating-point values
 VCMPPD k1 {k2}, zmm2,                                                       in zmm3/m512/m64bcst and zmm2 using bits 4:0 of
 zmm3/m512/m64bcst{sae}, imm8                                                imm8 as a comparison predicate with writemask k2
                                                                             and leave the result in mask register k1.



                                                   Instruction Operand Encoding
      Op/En                Operand 1                       Operand 2              Operand 3                    Operand 4
       RMI              ModRM:reg (r, w)               ModRM:r/m (r)                Imm8                          NA
      RVMI               ModRM:reg (w)                     VEX.vvvv             ModRM:r/m (r)                    Imm8
       FV                ModRM:reg (w)                   EVEX.vvvv              ModRM:r/m (r)                    Imm8

Description
Performs a SIMD compare of the packed double-precision floating-point values in the second source operand and
the first source operand and returns the results of the comparison to the destination operand. The comparison
predicate operand (immediate byte) specifies the type of comparison performed on each pair of packed values in
the two source operands.
EVEX encoded versions: The first source operand (second operand) is a ZMM/YMM/XMM register. The second
source operand can be a ZMM/YMM/XMM register, a 512/256/128-bit memory location or a 512/256/128-bit vector
broadcasted from a 64-bit memory location. The destination operand (first operand) is an opmask register.
Comparison results are written to the destination operand under the writemask k2. Each comparison result is a
single mask bit of 1 (comparison true) or 0 (comparison false).
VEX.256 encoded version: The first source operand (second operand) is a YMM register. The second source
operand (third operand) can be a YMM register or a 256-bit memory location. The destination operand (first
operand) is a YMM register. Four comparisons are performed with results written to the destination operand. The
result of each comparison is a quadword mask of all 1s (comparison true) or all 0s (comparison false).
128-bit Legacy SSE version: The first source and destination operand (first operand) is an XMM register. The
second source operand (second operand) can be an XMM register or 128-bit memory location. Bits (MAX_VL-
1:128) of the corresponding ZMM destination register remain unchanged. Two comparisons are performed with
results written to bits 127:0 of the destination operand. The result of each comparison is a quadword mask of all
1s (comparison true) or all 0s (comparison false).





VEX.128 encoded version: The first source operand (second operand) is an XMM register. The second source
operand (third operand) can be an XMM register or a 128-bit memory location. Bits (MAX_VL-1:128) of the desti-
nation ZMM register are zeroed. Two comparisons are performed with results written to bits 127:0 of the destina-
tion operand.
The comparison predicate operand is an 8-bit immediate:
.   For instructions encoded using the VEX or EVEX prefix, bits 4:0 define the type of comparison to be performed
    (see Table 3-1). Bits 5 through 7 of the immediate are reserved.
.   For instruction encodings that do not use VEX prefix, bits 2:0 define the type of comparison to be made (see the
    first 8 rows of Table 3-1). Bits 3 through 7 of the immediate are reserved.


                         Table 3-1. Comparison Predicate for CMPPD and CMPPS Instructions
Predicate          imm8     Description                                     Result: A Is 1st Operand, B Is 2nd Operand Signals
                   Value                                                                                               #IA on
                                                                            A >B        A<B      A=B        Unordered1 QNAN

EQ_OQ (EQ)         0H       Equal (ordered, non-signaling)                  False       False     True       False          No
LT_OS (LT)         1H       Less-than (ordered, signaling)                  False       True      False      False          Yes
LE_OS (LE)         2H       Less-than-or-equal (ordered, signaling)         False       True      True       False          Yes
UNORD_Q (UNORD) 3H          Unordered (non-signaling)                       False       False     False      True           No
NEQ_UQ (NEQ)       4H       Not-equal (unordered, non-signaling)            True        True      False      True           No
NLT_US (NLT)       5H       Not-less-than (unordered, signaling)            True        False     True       True           Yes
NLE_US (NLE)       6H       Not-less-than-or-equal (unordered, signaling)   True        False     False      True           Yes
ORD_Q (ORD)        7H       Ordered (non-signaling)                         True        True      True       False          No
EQ_UQ              8H       Equal (unordered, non-signaling)                False       False     True       True           No
NGE_US (NGE)       9H       Not-greater-than-or-equal (unordered,           False       True      False      True           Yes
                            signaling)
NGT_US (NGT)       AH       Not-greater-than (unordered, signaling)         False       True      True       True           Yes
FALSE_OQ(FALSE)    BH       False (ordered, non-signaling)                  False       False     False      False          No
NEQ_OQ             CH       Not-equal (ordered, non-signaling)              True        True      False      False          No
GE_OS (GE)         DH       Greater-than-or-equal (ordered, signaling)      True        False     True        False         Yes
GT_OS (GT)         EH       Greater-than (ordered, signaling)               True        False     False      False          Yes
TRUE_UQ(TRUE)      FH       True (unordered, non-signaling)                 True        True      True       True           No
EQ_OS              10H      Equal (ordered, signaling)                      False       False     True       False          Yes
LT_OQ              11H      Less-than (ordered, nonsignaling)               False       True      False      False          No
LE_OQ              12H      Less-than-or-equal (ordered, nonsignaling)      False       True      True       False          No
UNORD_S            13H      Unordered (signaling)                           False       False     False      True           Yes
NEQ_US             14H      Not-equal (unordered, signaling)                True        True      False      True           Yes
NLT_UQ             15H      Not-less-than (unordered, nonsignaling)         True        False     True       True           No
NLE_UQ             16H      Not-less-than-or-equal (unordered, nonsig-      True        False     False      True           No
                            naling)
ORD_S              17H      Ordered (signaling)                             True        True      True       False          Yes


EQ_US              18H      Equal (unordered, signaling)                    False       False     True       True           Yes
NGE_UQ             19H      Not-greater-than-or-equal (unordered, non-      False       True      False      True           No
                            signaling)




                      Table 3-1. Comparison Predicate for CMPPD and CMPPS Instructions (Contd.)
    Predicate          imm8    Description                                  Result: A Is 1st Operand, B Is 2nd Operand Signals
                       Value                                                                                           #IA on
                                                                            A >B        A<B      A=B        Unordered1 QNAN

    NGT_UQ             1AH     Not-greater-than (unordered, nonsignaling)   False     True     True      True          No
    FALSE_OS           1BH     False (ordered, signaling)                   False     False    False     False         Yes
    NEQ_OS             1CH     Not-equal (ordered, signaling)               True      True     False     False         Yes
    GE_OQ              1DH     Greater-than-or-equal (ordered, nonsignal-   True      False    True      False         No
                               ing)
    GT_OQ              1EH     Greater-than (ordered, nonsignaling)         True      False    False     False         No
    TRUE_US            1FH     True (unordered, signaling)                  True      True     True      True          Yes


NOTES:
1. If either operand A or B is a NAN.

The unordered relationship is true when at least one of the two source operands being compared is a NaN; the
ordered relationship is true when neither source operand is a NaN.
A subsequent computational instruction that uses the mask result in the destination operand as an input operand
will not generate an exception, because a mask of all 0s corresponds to a floating-point value of +0.0 and a mask
of all 1s corresponds to a QNaN.
Note that processors with "CPUID.1H:ECX.AVX =0" do not implement the "greater-than", "greater-than-or-equal",
"not-greater than", and "not-greater-than-or-equal relations" predicates. These comparisons can be made either
by using the inverse relationship (that is, use the "not-less-than-or-equal" to make a "greater-than" comparison)
or by using software emulation. When using software emulation, the program must swap the operands (copying
registers when necessary to protect the data that will now be in the destination), and then perform the compare
using a different predicate. The predicate to be used for these emulations is listed in the first 8 rows of Table 3-7
(Intel 64 and IA-32 Architectures Software Developer''s Manual Volume 2A) under the heading Emulation.
Compilers and assemblers may implement the following two-operand pseudo-ops in addition to the three-operand
CMPPD instruction, for processors with "CPUID.1H:ECX.AVX =0". See Table 3-2. Compiler should treat reserved
Imm8 values as illegal syntax.
                                       Table 3-2. Pseudo-Op and CMPPD Implementation
:




    Pseudo-Op                                                         CMPPD Implementation
    CMPEQPD xmm1, xmm2                                                CMPPD xmm1, xmm2, 0
    CMPLTPD xmm1, xmm2                                                CMPPD xmm1, xmm2, 1
    CMPLEPD xmm1, xmm2                                                CMPPD xmm1, xmm2, 2
    CMPUNORDPD xmm1, xmm2                                             CMPPD xmm1, xmm2, 3
    CMPNEQPD xmm1, xmm2                                               CMPPD xmm1, xmm2, 4
    CMPNLTPD xmm1, xmm2                                               CMPPD xmm1, xmm2, 5
    CMPNLEPD xmm1, xmm2                                               CMPPD xmm1, xmm2, 6
    CMPORDPD xmm1, xmm2                                               CMPPD xmm1, xmm2, 7

The greater-than relations that the processor does not implement require more than one instruction to emulate in
software and therefore should not be implemented as pseudo-ops. (For these, the programmer should reverse the
operands of the corresponding less than relations and use move instructions to ensure that the mask is moved to
the correct destination register and that the source operand is left intact.)
Processors with "CPUID.1H:ECX.AVX =1" implement the full complement of 32 predicates shown in Table 3-3, soft-
ware emulation is no longer needed. Compilers and assemblers may implement the following three-operand
pseudo-ops in addition to the four-operand VCMPPD instruction. See Table 3-3, where the notations of reg1 reg2,
and reg3 represent either XMM registers or YMM registers. Compiler should treat reserved Imm8 values as illegal



syntax. Alternately, intrinsics can map the pseudo-ops to pre-defined constants to support a simpler intrinsic inter-
face. Compilers and assemblers may implement three-operand pseudo-ops for EVEX encoded VCMPPD instructions
in a similar fashion by extending the syntax listed in Table 3-3.
                                     Table 3-3. Pseudo-Op and VCMPPD Implementation
:




    Pseudo-Op                                                CMPPD Implementation
    VCMPEQPD reg1, reg2, reg3                                VCMPPD reg1, reg2, reg3, 0
    VCMPLTPD reg1, reg2, reg3                                VCMPPD reg1, reg2, reg3, 1
    VCMPLEPD reg1, reg2, reg3                                VCMPPD reg1, reg2, reg3, 2
    VCMPUNORDPD reg1, reg2, reg3                             VCMPPD reg1, reg2, reg3, 3
    VCMPNEQPD reg1, reg2, reg3                               VCMPPD reg1, reg2, reg3, 4
    VCMPNLTPD reg1, reg2, reg3                               VCMPPD reg1, reg2, reg3, 5
    VCMPNLEPD reg1, reg2, reg3                               VCMPPD reg1, reg2, reg3, 6
    VCMPORDPD reg1, reg2, reg3                               VCMPPD reg1, reg2, reg3, 7
    VCMPEQ_UQPD reg1, reg2, reg3                             VCMPPD reg1, reg2, reg3, 8
    VCMPNGEPD reg1, reg2, reg3                               VCMPPD reg1, reg2, reg3, 9
    VCMPNGTPD reg1, reg2, reg3                               VCMPPD reg1, reg2, reg3, 0AH
    VCMPFALSEPD reg1, reg2, reg3                             VCMPPD reg1, reg2, reg3, 0BH
    VCMPNEQ_OQPD reg1, reg2, reg3                            VCMPPD reg1, reg2, reg3, 0CH
    VCMPGEPD reg1, reg2, reg3                                VCMPPD reg1, reg2, reg3, 0DH
    VCMPGTPD reg1, reg2, reg3                                VCMPPD reg1, reg2, reg3, 0EH
    VCMPTRUEPD reg1, reg2, reg3                              VCMPPD reg1, reg2, reg3, 0FH
    VCMPEQ_OSPD reg1, reg2, reg3                             VCMPPD reg1, reg2, reg3, 10H
    VCMPLT_OQPD reg1, reg2, reg3                             VCMPPD reg1, reg2, reg3, 11H
    VCMPLE_OQPD reg1, reg2, reg3                             VCMPPD reg1, reg2, reg3, 12H
    VCMPUNORD_SPD reg1, reg2, reg3                           VCMPPD reg1, reg2, reg3, 13H
    VCMPNEQ_USPD reg1, reg2, reg3                            VCMPPD reg1, reg2, reg3, 14H
    VCMPNLT_UQPD reg1, reg2, reg3                            VCMPPD reg1, reg2, reg3, 15H
    VCMPNLE_UQPD reg1, reg2, reg3                            VCMPPD reg1, reg2, reg3, 16H
    VCMPORD_SPD reg1, reg2, reg3                             VCMPPD reg1, reg2, reg3, 17H
    VCMPEQ_USPD reg1, reg2, reg3                             VCMPPD reg1, reg2, reg3, 18H
    VCMPNGE_UQPD reg1, reg2, reg3                            VCMPPD reg1, reg2, reg3, 19H
    VCMPNGT_UQPD reg1, reg2, reg3                            VCMPPD reg1, reg2, reg3, 1AH
    VCMPFALSE_OSPD reg1, reg2, reg3                          VCMPPD reg1, reg2, reg3, 1BH
    VCMPNEQ_OSPD reg1, reg2, reg3                            VCMPPD reg1, reg2, reg3, 1CH
    VCMPGE_OQPD reg1, reg2, reg3                             VCMPPD reg1, reg2, reg3, 1DH
    VCMPGT_OQPD reg1, reg2, reg3                             VCMPPD reg1, reg2, reg3, 1EH
    VCMPTRUE_USPD reg1, reg2, reg3                           VCMPPD reg1, reg2, reg3, 1FH





Operation
CASE (COMPARISON PREDICATE) OF
0: OP3 <- EQ_OQ; OP5 <- EQ_OQ;
    1: OP3 <- LT_OS; OP5 <- LT_OS;
    2: OP3 <- LE_OS; OP5 <- LE_OS;
    3: OP3 <- UNORD_Q; OP5 <- UNORD_Q;
    4: OP3 <- NEQ_UQ; OP5 <- NEQ_UQ;
    5: OP3 <- NLT_US; OP5 <- NLT_US;
    6: OP3 <- NLE_US; OP5 <- NLE_US;
    7: OP3 <- ORD_Q; OP5 <- ORD_Q;
    8: OP5 <- EQ_UQ;
    9: OP5 <- NGE_US;
    10: OP5 <- NGT_US;
    11: OP5 <- FALSE_OQ;
    12: OP5 <- NEQ_OQ;
    13: OP5 <- GE_OS;
    14: OP5 <- GT_OS;
    15: OP5 <- TRUE_UQ;
    16: OP5 <- EQ_OS;
    17: OP5 <- LT_OQ;
    18: OP5 <- LE_OQ;
    19: OP5 <- UNORD_S;
    20: OP5 <- NEQ_US;
    21: OP5 <- NLT_UQ;
    22: OP5 <- NLE_UQ;
    23: OP5 <- ORD_S;
    24: OP5 <- EQ_US;
    25: OP5 <- NGE_UQ;
    26: OP5 <- NGT_UQ;
    27: OP5 <- FALSE_OS;
    28: OP5 <- NEQ_OS;
    29: OP5 <- GE_OQ;
    30: OP5 <- GT_OQ;
    31: OP5 <- TRUE_US;
    DEFAULT: Reserved;
ESAC;





VCMPPD (EVEX encoded versions)
(KL, VL) = (2, 128), (4, 256), (8, 512)
FOR j <- 0 TO KL-1
    i <- j * 64
    IF k2[j] OR *no writemask*
         THEN
               IF (EVEX.b = 1) AND (SRC2 *is memory*)
                    THEN
                         CMP <- SRC1[i+63:i] OP5 SRC2[63:0]
                    ELSE
                         CMP <- SRC1[i+63:i] OP5 SRC2[i+63:i]
               FI;
               IF CMP = TRUE
                    THEN DEST[j] <- 1;
                    ELSE DEST[j] <- 0; FI;
         ELSE       DEST[j] <- 0            ; zeroing-masking only
    FI;
ENDFOR
DEST[MAX_KL-1:KL] <- 0

VCMPPD (VEX.256 encoded version)
CMP0 <- SRC1[63:0] OP5 SRC2[63:0];
CMP1 <- SRC1[127:64] OP5 SRC2[127:64];
CMP2 <- SRC1[191:128] OP5 SRC2[191:128];
CMP3 <- SRC1[255:192] OP5 SRC2[255:192];
IF CMP0 = TRUE
    THEN DEST[63:0] <- FFFFFFFFFFFFFFFFH;
    ELSE DEST[63:0] <- 0000000000000000H; FI;
IF CMP1 = TRUE
    THEN DEST[127:64] <- FFFFFFFFFFFFFFFFH;
    ELSE DEST[127:64] <- 0000000000000000H; FI;
IF CMP2 = TRUE
    THEN DEST[191:128] <- FFFFFFFFFFFFFFFFH;
    ELSE DEST[191:128] <- 0000000000000000H; FI;
IF CMP3 = TRUE
    THEN DEST[255:192] <- FFFFFFFFFFFFFFFFH;
    ELSE DEST[255:192] <- 0000000000000000H; FI;
DEST[MAX_VL-1:256] <- 0

VCMPPD (VEX.128 encoded version)
CMP0 <- SRC1[63:0] OP5 SRC2[63:0];
CMP1 <- SRC1[127:64] OP5 SRC2[127:64];
IF CMP0 = TRUE
    THEN DEST[63:0] <- FFFFFFFFFFFFFFFFH;
    ELSE DEST[63:0] <- 0000000000000000H; FI;
IF CMP1 = TRUE
    THEN DEST[127:64] <- FFFFFFFFFFFFFFFFH;
    ELSE DEST[127:64] <- 0000000000000000H; FI;
DEST[MAX_VL-1:128] <- 0





CMPPD (128-bit Legacy SSE version)
CMP0 <- SRC1[63:0] OP3 SRC2[63:0];
CMP1 <- SRC1[127:64] OP3 SRC2[127:64];
IF CMP0 = TRUE
    THEN DEST[63:0] <- FFFFFFFFFFFFFFFFH;
    ELSE DEST[63:0] <- 0000000000000000H; FI;
IF CMP1 = TRUE
    THEN DEST[127:64] <- FFFFFFFFFFFFFFFFH;
    ELSE DEST[127:64] <- 0000000000000000H; FI;
DEST[MAX_VL-1:128] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VCMPPD __mmask8 _mm512_cmp_pd_mask( __m512d a, __m512d b, int imm);
VCMPPD __mmask8 _mm512_cmp_round_pd_mask( __m512d a, __m512d b, int imm, int sae);
VCMPPD __mmask8 _mm512_mask_cmp_pd_mask( __mmask8 k1, __m512d a, __m512d b, int imm);
VCMPPD __mmask8 _mm512_mask_cmp_round_pd_mask( __mmask8 k1, __m512d a, __m512d b, int imm, int sae);
VCMPPD __mmask8 _mm256_cmp_pd_mask( __m256d a, __m256d b, int imm);
VCMPPD __mmask8 _mm256_mask_cmp_pd_mask( __mmask8 k1, __m256d a, __m256d b, int imm);
VCMPPD __mmask8 _mm_cmp_pd_mask( __m128d a, __m128d b, int imm);
VCMPPD __mmask8 _mm_mask_cmp_pd_mask( __mmask8 k1, __m128d a, __m128d b, int imm);
VCMPPD __m256 _mm256_cmp_pd(__m256d a, __m256d b, int imm)
(V)CMPPD __m128 _mm_cmp_pd(__m128d a, __m128d b, int imm)

SIMD Floating-Point Exceptions
Invalid if SNaN operand and invalid if QNaN and predicate as listed in Table 3-1.
Denormal

Other Exceptions
VEX-encoded instructions, see Exceptions Type 2.
EVEX-encoded instructions, see Exceptions Type E2.




');
INSERT INTO "instructions" VALUES('x86_64','CMPPS','CMPPS-Compare Packed Single-Precision Floating-Point Values
 Opcode/                             Op /   64/32      CPUID      Description
 Instruction                         En     bit Mode   Feature
                                            Support    Flag
 0F C2 /r ib                         RMI    V/V        SSE        Compare packed single-precision floating-point values in
 CMPPS xmm1, xmm2/m128,                                           xmm2/m128 and xmm1 using bits 2:0 of imm8 as a
 imm8                                                             comparison predicate.
 VEX.NDS.128.0F.WIG C2 /r ib         RVMI   V/V        AVX        Compare packed single-precision floating-point values in
 VCMPPS xmm1, xmm2,                                               xmm3/m128 and xmm2 using bits 4:0 of imm8 as a
 xmm3/m128, imm8                                                  comparison predicate.
 VEX.NDS.256.0F.WIG C2 /r ib         RVMI   V/V        AVX        Compare packed single-precision floating-point values in
 VCMPPS ymm1, ymm2,                                               ymm3/m256 and ymm2 using bits 4:0 of imm8 as a
 ymm3/m256, imm8                                                  comparison predicate.
 EVEX.NDS.128.0F.W0 C2 /r ib         FV     V/V        AVX512VL   Compare packed single-precision floating-point values in
 VCMPPS k1 {k2}, xmm2,                                 AVX512F    xmm3/m128/m32bcst and xmm2 using bits 4:0 of imm8 as
 xmm3/m128/m32bcst, imm8                                          a comparison predicate with writemask k2 and leave the
                                                                  result in mask register k1.
 EVEX.NDS.256.0F.W0 C2 /r ib         FV     V/V        AVX512VL   Compare packed single-precision floating-point values in
 VCMPPS k1 {k2}, ymm2,                                 AVX512F    ymm3/m256/m32bcst and ymm2 using bits 4:0 of imm8 as
 ymm3/m256/m32bcst, imm8                                          a comparison predicate with writemask k2 and leave the
                                                                  result in mask register k1.
 EVEX.NDS.512.0F.W0 C2 /r ib         FV     V/V        AVX512F    Compare packed single-precision floating-point values in
 VCMPPS k1 {k2}, zmm2,                                            zmm3/m512/m32bcst and zmm2 using bits 4:0 of imm8 as
 zmm3/m512/m32bcst{sae}, imm8                                     a comparison predicate with writemask k2 and leave the
                                                                  result in mask register k1.



                                              Instruction Operand Encoding
     Op/En             Operand 1                   Operand 2                 Operand 3                    Operand 4
      RMI           ModRM:reg (r, w)              ModRM:r/m (r)                 Imm8                          NA
     RVMI            ModRM:reg (w)                  VEX.vvvv              ModRM:r/m (r)                     Imm8
       FV            ModRM:reg (w)                 EVEX.vvvv              ModRM:r/m (r)                     Imm8

Description
Performs a SIMD compare of the packed single-precision floating-point values in the second source operand and
the first source operand and returns the results of the comparison to the destination operand. The comparison
predicate operand (immediate byte) specifies the type of comparison performed on each of the pairs of packed
values.
EVEX encoded versions: The first source operand (second operand) is a ZMM/YMM/XMM register. The second
source operand can be a ZMM/YMM/XMM register, a 512/256/128-bit memory location or a 512/256/128-bit vector
broadcasted from a 32-bit memory location. The destination operand (first operand) is an opmask register.
Comparison results are written to the destination operand under the writemask k2. Each comparison result is a
single mask bit of 1 (comparison true) or 0 (comparison false).
VEX.256 encoded version: The first source operand (second operand) is a YMM register. The second source operand
(third operand) can be a YMM register or a 256-bit memory location. The destination operand (first operand) is a
YMM register. Eight comparisons are performed with results written to the destination operand. The result of each
comparison is a doubleword mask of all 1s (comparison true) or all 0s (comparison false).
128-bit Legacy SSE version: The first source and destination operand (first operand) is an XMM register. The
second source operand (second operand) can be an XMM register or 128-bit memory location. Bits (MAX_VL-
1:128) of the corresponding ZMM destination register remain unchanged. Four comparisons are performed with
results written to bits 127:0 of the destination operand. The result of each comparison is a doubleword mask of all
1s (comparison true) or all 0s (comparison false).





VEX.128 encoded version: The first source operand (second operand) is an XMM register. The second source
operand (third operand) can be an XMM register or a 128-bit memory location. Bits (MAX_VL-1:128) of the desti-
nation ZMM register are zeroed. Four comparisons are performed with results written to bits 127:0 of the destina-
tion operand.
The comparison predicate operand is an 8-bit immediate:
.     For instructions encoded using the VEX prefix and EVEX prefix, bits 4:0 define the type of comparison to be
      performed (see Table 3-1). Bits 5 through 7 of the immediate are reserved.
.     For instruction encodings that do not use VEX prefix, bits 2:0 define the type of comparison to be made (see
      the first 8 rows of Table 3-1). Bits 3 through 7 of the immediate are reserved.
The unordered relationship is true when at least one of the two source operands being compared is a NaN; the
ordered relationship is true when neither source operand is a NaN.
A subsequent computational instruction that uses the mask result in the destination operand as an input operand
will not generate an exception, because a mask of all 0s corresponds to a floating-point value of +0.0 and a mask
of all 1s corresponds to a QNaN.
Note that processors with "CPUID.1H:ECX.AVX =0" do not implement the "greater-than", "greater-than-or-equal",
"not-greater than", and "not-greater-than-or-equal relations" predicates. These comparisons can be made either
by using the inverse relationship (that is, use the "not-less-than-or-equal" to make a "greater-than" comparison)
or by using software emulation. When using software emulation, the program must swap the operands (copying
registers when necessary to protect the data that will now be in the destination), and then perform the compare
using a different predicate. The predicate to be used for these emulations is listed in the first 8 rows of Table 3-7
(Intel 64 and IA-32 Architectures Software Developer''s Manual Volume 2A) under the heading Emulation.
Compilers and assemblers may implement the following two-operand pseudo-ops in addition to the three-operand
CMPPS instruction, for processors with "CPUID.1H:ECX.AVX =0". See Table 3-4. Compiler should treat reserved
Imm8 values as illegal syntax.
                                       Table 3-4. Pseudo-Op and CMPPS Implementation
:




    Pseudo-Op                                                  CMPPS Implementation
    CMPEQPS xmm1, xmm2                                         CMPPS xmm1, xmm2, 0
    CMPLTPS xmm1, xmm2                                         CMPPS xmm1, xmm2, 1
    CMPLEPS xmm1, xmm2                                         CMPPS xmm1, xmm2, 2
    CMPUNORDPS xmm1, xmm2                                      CMPPS xmm1, xmm2, 3
    CMPNEQPS xmm1, xmm2                                        CMPPS xmm1, xmm2, 4
    CMPNLTPS xmm1, xmm2                                        CMPPS xmm1, xmm2, 5
    CMPNLEPS xmm1, xmm2                                        CMPPS xmm1, xmm2, 6
    CMPORDPS xmm1, xmm2                                        CMPPS xmm1, xmm2, 7

The greater-than relations that the processor does not implement require more than one instruction to emulate in
software and therefore should not be implemented as pseudo-ops. (For these, the programmer should reverse the
operands of the corresponding less than relations and use move instructions to ensure that the mask is moved to
the correct destination register and that the source operand is left intact.)
Processors with "CPUID.1H:ECX.AVX =1" implement the full complement of 32 predicates shown in Table 3-5, soft-
ware emulation is no longer needed. Compilers and assemblers may implement the following three-operand
pseudo-ops in addition to the four-operand VCMPPS instruction. See Table 3-5, where the notation of reg1 and
reg2 represent either XMM registers or YMM registers. Compiler should treat reserved Imm8 values as illegal
syntax. Alternately, intrinsics can map the pseudo-ops to pre-defined constants to support a simpler intrinsic inter-
face. Compilers and assemblers may implement three-operand pseudo-ops for EVEX encoded VCMPPS instructions
in a similar fashion by extending the syntax listed in Table 3-5.
:





                                   Table 3-5. Pseudo-Op and VCMPPS Implementation
 Pseudo-Op                                                  CMPPS Implementation
 VCMPEQPS reg1, reg2, reg3                                  VCMPPS reg1, reg2, reg3, 0
 VCMPLTPS reg1, reg2, reg3                                  VCMPPS reg1, reg2, reg3, 1
 VCMPLEPS reg1, reg2, reg3                                  VCMPPS reg1, reg2, reg3, 2
 VCMPUNORDPS reg1, reg2, reg3                               VCMPPS reg1, reg2, reg3, 3
 VCMPNEQPS reg1, reg2, reg3                                 VCMPPS reg1, reg2, reg3, 4
 VCMPNLTPS reg1, reg2, reg3                                 VCMPPS reg1, reg2, reg3, 5
 VCMPNLEPS reg1, reg2, reg3                                 VCMPPS reg1, reg2, reg3, 6
 VCMPORDPS reg1, reg2, reg3                                 VCMPPS reg1, reg2, reg3, 7
 VCMPEQ_UQPS reg1, reg2, reg3                               VCMPPS reg1, reg2, reg3, 8
 VCMPNGEPS reg1, reg2, reg3                                 VCMPPS reg1, reg2, reg3, 9
 VCMPNGTPS reg1, reg2, reg3                                 VCMPPS reg1, reg2, reg3, 0AH
 VCMPFALSEPS reg1, reg2, reg3                               VCMPPS reg1, reg2, reg3, 0BH
 VCMPNEQ_OQPS reg1, reg2, reg3                              VCMPPS reg1, reg2, reg3, 0CH
 VCMPGEPS reg1, reg2, reg3                                  VCMPPS reg1, reg2, reg3, 0DH
 VCMPGTPS reg1, reg2, reg3                                  VCMPPS reg1, reg2, reg3, 0EH
 VCMPTRUEPS reg1, reg2, reg3                                VCMPPS reg1, reg2, reg3, 0FH
 VCMPEQ_OSPS reg1, reg2, reg3                               VCMPPS reg1, reg2, reg3, 10H
 VCMPLT_OQPS reg1, reg2, reg3                               VCMPPS reg1, reg2, reg3, 11H
 VCMPLE_OQPS reg1, reg2, reg3                               VCMPPS reg1, reg2, reg3, 12H
 VCMPUNORD_SPS reg1, reg2, reg3                             VCMPPS reg1, reg2, reg3, 13H
 VCMPNEQ_USPS reg1, reg2, reg3                              VCMPPS reg1, reg2, reg3, 14H
 VCMPNLT_UQPS reg1, reg2, reg3                              VCMPPS reg1, reg2, reg3, 15H
 VCMPNLE_UQPS reg1, reg2, reg3                              VCMPPS reg1, reg2, reg3, 16H
 VCMPORD_SPS reg1, reg2, reg3                               VCMPPS reg1, reg2, reg3, 17H
 VCMPEQ_USPS reg1, reg2, reg3                               VCMPPS reg1, reg2, reg3, 18H
 VCMPNGE_UQPS reg1, reg2, reg3                              VCMPPS reg1, reg2, reg3, 19H
 VCMPNGT_UQPS reg1, reg2, reg3                              VCMPPS reg1, reg2, reg3, 1AH
 VCMPFALSE_OSPS reg1, reg2, reg3                            VCMPPS reg1, reg2, reg3, 1BH
 VCMPNEQ_OSPS reg1, reg2, reg3                              VCMPPS reg1, reg2, reg3, 1CH
 VCMPGE_OQPS reg1, reg2, reg3                               VCMPPS reg1, reg2, reg3, 1DH
 VCMPGT_OQPS reg1, reg2, reg3                               VCMPPS reg1, reg2, reg3, 1EH
 VCMPTRUE_USPS reg1, reg2, reg3                             VCMPPS reg1, reg2, reg3, 1FH





Operation
CASE (COMPARISON PREDICATE) OF
   0: OP3 <- EQ_OQ; OP5 <- EQ_OQ;
   1: OP3 <- LT_OS; OP5 <- LT_OS;
   2: OP3 <- LE_OS; OP5 <- LE_OS;
   3: OP3 <- UNORD_Q; OP5 <- UNORD_Q;
   4: OP3 <- NEQ_UQ; OP5 <- NEQ_UQ;
   5: OP3 <- NLT_US; OP5 <- NLT_US;
   6: OP3 <- NLE_US; OP5 <- NLE_US;
   7: OP3 <- ORD_Q; OP5 <- ORD_Q;
   8: OP5 <- EQ_UQ;
   9: OP5 <- NGE_US;
   10: OP5 <- NGT_US;
   11: OP5 <- FALSE_OQ;
   12: OP5 <- NEQ_OQ;
   13: OP5 <- GE_OS;
   14: OP5 <- GT_OS;
   15: OP5 <- TRUE_UQ;
   16: OP5 <- EQ_OS;
   17: OP5 <- LT_OQ;
   18: OP5 <- LE_OQ;
   19: OP5 <- UNORD_S;
   20: OP5 <- NEQ_US;
   21: OP5 <- NLT_UQ;
   22: OP5 <- NLE_UQ;
   23: OP5 <- ORD_S;
   24: OP5 <- EQ_US;
   25: OP5 <- NGE_UQ;
   26: OP5 <- NGT_UQ;
   27: OP5 <- FALSE_OS;
   28: OP5 <- NEQ_OS;
   29: OP5 <- GE_OQ;
   30: OP5 <- GT_OQ;
   31: OP5 <- TRUE_US;
   DEFAULT: Reserved
ESAC;





VCMPPS (EVEX encoded versions)
(KL, VL) = (4, 128), (8, 256), (16, 512)
FOR j <- 0 TO KL-1
    i <- j * 32
    IF k2[j] OR *no writemask*
         THEN
               IF (EVEX.b = 1) AND (SRC2 *is memory*)
                    THEN
                         CMP <- SRC1[i+31:i] OP5 SRC2[31:0]
                    ELSE
                         CMP <- SRC1[i+31:i] OP5 SRC2[i+31:i]
               FI;
               IF CMP = TRUE
                    THEN DEST[j] <- 1;
                    ELSE DEST[j] <- 0; FI;
         ELSE       DEST[j] <- 0                ; zeroing-masking onlyFI;
    FI;
ENDFOR
DEST[MAX_KL-1:KL] <- 0

VCMPPS (VEX.256 encoded version)
CMP0 <- SRC1[31:0] OP5 SRC2[31:0];
CMP1 <- SRC1[63:32] OP5 SRC2[63:32];
CMP2 <- SRC1[95:64] OP5 SRC2[95:64];
CMP3 <- SRC1[127:96] OP5 SRC2[127:96];
CMP4 <- SRC1[159:128] OP5 SRC2[159:128];
CMP5 <- SRC1[191:160] OP5 SRC2[191:160];
CMP6 <- SRC1[223:192] OP5 SRC2[223:192];
CMP7 <- SRC1[255:224] OP5 SRC2[255:224];
IF CMP0 = TRUE
    THEN DEST[31:0] <-FFFFFFFFH;
    ELSE DEST[31:0] <- 000000000H; FI;
IF CMP1 = TRUE
    THEN DEST[63:32] <- FFFFFFFFH;
    ELSE DEST[63:32] <-000000000H; FI;
IF CMP2 = TRUE
    THEN DEST[95:64] <- FFFFFFFFH;
    ELSE DEST[95:64] <- 000000000H; FI;
IF CMP3 = TRUE
    THEN DEST[127:96] <- FFFFFFFFH;
    ELSE DEST[127:96] <- 000000000H; FI;
IF CMP4 = TRUE
    THEN DEST[159:128] <- FFFFFFFFH;
    ELSE DEST[159:128] <- 000000000H; FI;
IF CMP5 = TRUE
    THEN DEST[191:160] <- FFFFFFFFH;
    ELSE DEST[191:160] <- 000000000H; FI;
IF CMP6 = TRUE
    THEN DEST[223:192] <- FFFFFFFFH;
    ELSE DEST[223:192] <-000000000H; FI;
IF CMP7 = TRUE
    THEN DEST[255:224] <- FFFFFFFFH;
    ELSE DEST[255:224] <- 000000000H; FI;
DEST[MAX_VL-1:256] <- 0



VCMPPS (VEX.128 encoded version)
CMP0 <- SRC1[31:0] OP5 SRC2[31:0];
CMP1 <- SRC1[63:32] OP5 SRC2[63:32];
CMP2 <- SRC1[95:64] OP5 SRC2[95:64];
CMP3 <- SRC1[127:96] OP5 SRC2[127:96];
IF CMP0 = TRUE
    THEN DEST[31:0] <-FFFFFFFFH;
    ELSE DEST[31:0] <- 000000000H; FI;
IF CMP1 = TRUE
    THEN DEST[63:32] <- FFFFFFFFH;
    ELSE DEST[63:32] <- 000000000H; FI;
IF CMP2 = TRUE
    THEN DEST[95:64] <- FFFFFFFFH;
    ELSE DEST[95:64] <- 000000000H; FI;
IF CMP3 = TRUE
    THEN DEST[127:96] <- FFFFFFFFH;
    ELSE DEST[127:96] <-000000000H; FI;
DEST[MAX_VL-1:128] <- 0

CMPPS (128-bit Legacy SSE version)
CMP0 <- SRC1[31:0] OP3 SRC2[31:0];
CMP1 <- SRC1[63:32] OP3 SRC2[63:32];
CMP2 <- SRC1[95:64] OP3 SRC2[95:64];
CMP3 <- SRC1[127:96] OP3 SRC2[127:96];
IF CMP0 = TRUE
    THEN DEST[31:0] <-FFFFFFFFH;
    ELSE DEST[31:0] <- 000000000H; FI;
IF CMP1 = TRUE
    THEN DEST[63:32] <- FFFFFFFFH;
    ELSE DEST[63:32] <- 000000000H; FI;
IF CMP2 = TRUE
    THEN DEST[95:64] <- FFFFFFFFH;
    ELSE DEST[95:64] <- 000000000H; FI;
IF CMP3 = TRUE
    THEN DEST[127:96] <- FFFFFFFFH;
    ELSE DEST[127:96] <-000000000H; FI;
DEST[MAX_VL-1:128] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VCMPPS __mmask16 _mm512_cmp_ps_mask( __m512 a, __m512 b, int imm);
VCMPPS __mmask16 _mm512_cmp_round_ps_mask( __m512 a, __m512 b, int imm, int sae);
VCMPPS __mmask16 _mm512_mask_cmp_ps_mask( __mmask16 k1, __m512 a, __m512 b, int imm);
VCMPPS __mmask16 _mm512_mask_cmp_round_ps_mask( __mmask16 k1, __m512 a, __m512 b, int imm, int sae);
VCMPPD __mmask8 _mm256_cmp_ps_mask( __m256 a, __m256 b, int imm);
VCMPPS __mmask8 _mm256_mask_cmp_ps_mask( __mmask8 k1, __m256 a, __m256 b, int imm);
VCMPPS __mmask8 _mm_cmp_ps_mask( __m128 a, __m128 b, int imm);
VCMPPS __mmask8 _mm_mask_cmp_ps_mask( __mmask8 k1, __m128 a, __m128 b, int imm);
VCMPPS __m256 _mm256_cmp_ps(__m256 a, __m256 b, int imm)
CMPPS __m128 _mm_cmp_ps(__m128 a, __m128 b, int imm)

SIMD Floating-Point Exceptions
Invalid if SNaN operand and invalid if QNaN and predicate as listed in Table 3-1.
Denormal




Other Exceptions
VEX-encoded instructions, see Exceptions Type 2.
EVEX-encoded instructions, see Exceptions Type E2.




');
INSERT INTO "instructions" VALUES('x86_64','CMPS','CMPS/CMPSB/CMPSW/CMPSD/CMPSQ-Compare String Operands
 Opcode             Instruction           Op/    64-Bit      Compat/ Description
                                          En     Mode        Leg Mode
 A6                 CMPS m8, m8           NP     Valid       Valid    For legacy mode, compare byte at address DS:(E)SI with
                                                                      byte at address ES:(E)DI; For 64-bit mode compare byte
                                                                      at address (R|E)SI to byte at address (R|E)DI. The status
                                                                      flags are set accordingly.
 A7                 CMPS m16, m16         NP     Valid       Valid    For legacy mode, compare word at address DS:(E)SI
                                                                      with word at address ES:(E)DI; For 64-bit mode
                                                                      compare word at address (R|E)SI with word at address
                                                                      (R|E)DI. The status flags are set accordingly.
 A7                 CMPS m32, m32         NP     Valid       Valid    For legacy mode, compare dword at address DS:(E)SI at
                                                                      dword at address ES:(E)DI; For 64-bit mode compare
                                                                      dword at address (R|E)SI at dword at address (R|E)DI.
                                                                      The status flags are set accordingly.
 REX.W + A7         CMPS m64, m64         NP     Valid       N.E.     Compares quadword at address (R|E)SI with quadword
                                                                      at address (R|E)DI and sets the status flags accordingly.
 A6                 CMPSB                 NP     Valid       Valid    For legacy mode, compare byte at address DS:(E)SI with
                                                                      byte at address ES:(E)DI; For 64-bit mode compare byte
                                                                      at address (R|E)SI with byte at address (R|E)DI. The
                                                                      status flags are set accordingly.
 A7                 CMPSW                 NP     Valid       Valid    For legacy mode, compare word at address DS:(E)SI
                                                                      with word at address ES:(E)DI; For 64-bit mode
                                                                      compare word at address (R|E)SI with word at address
                                                                      (R|E)DI. The status flags are set accordingly.
 A7                 CMPSD                 NP     Valid       Valid    For legacy mode, compare dword at address DS:(E)SI
                                                                      with dword at address ES:(E)DI; For 64-bit mode
                                                                      compare dword at address (R|E)SI with dword at
                                                                      address (R|E)DI. The status flags are set accordingly.
 REX.W + A7         CMPSQ                 NP     Valid       N.E.     Compares quadword at address (R|E)SI with quadword
                                                                      at address (R|E)DI and sets the status flags accordingly.



                                               Instruction Operand Encoding
   Op/En           Operand 1                     Operand 2                 Operand 3                      Operand 4
      NP               NA                           NA                         NA                             NA

Description
Compares the byte, word, doubleword, or quadword specified with the first source operand with the byte, word,
doubleword, or quadword specified with the second source operand and sets the status flags in the EFLAGS register
according to the results.
Both source operands are located in memory. The address of the first source operand is read from DS:SI, DS:ESI
or RSI (depending on the address-size attribute of the instruction is 16, 32, or 64, respectively). The address of the
second source operand is read from ES:DI, ES:EDI or RDI (again depending on the address-size attribute of the
instruction is 16, 32, or 64). The DS segment may be overridden with a segment override prefix, but the ES
segment cannot be overridden.
At the assembly-code level, two forms of this instruction are allowed: the "explicit-operands" form and the "no-
operands" form. The explicit-operands form (specified with the CMPS mnemonic) allows the two source operands
to be specified explicitly. Here, the source operands should be symbols that indicate the size and location of the
source values. This explicit-operand form is provided to allow documentation. However, note that the documenta-
tion provided by this form can be misleading. That is, the source operand symbols must specify the correct type
(size) of the operands (bytes, words, or doublewords, quadwords), but they do not have to specify the correct loca-




tion. Locations of the source operands are always specified by the DS:(E)SI (or RSI) and ES:(E)DI (or RDI) regis-
ters, which must be loaded correctly before the compare string instruction is executed.
The no-operands form provides "short forms" of the byte, word, and doubleword versions of the CMPS instructions.
Here also the DS:(E)SI (or RSI) and ES:(E)DI (or RDI) registers are assumed by the processor to specify the loca-
tion of the source operands. The size of the source operands is selected with the mnemonic: CMPSB (byte compar-
ison), CMPSW (word comparison), CMPSD (doubleword comparison), or CMPSQ (quadword comparison using
REX.W).
After the comparison, the (E/R)SI and (E/R)DI registers increment or decrement automatically according to the
setting of the DF flag in the EFLAGS register. (If the DF flag is 0, the (E/R)SI and (E/R)DI register increment; if the
DF flag is 1, the registers decrement.) The registers increment or decrement by 1 for byte operations, by 2 for word
operations, 4 for doubleword operations. If operand size is 64, RSI and RDI registers increment by 8 for quadword
operations.
The CMPS, CMPSB, CMPSW, CMPSD, and CMPSQ instructions can be preceded by the REP prefix for block compar-
isons. More often, however, these instructions will be used in a LOOP construct that takes some action based on the
setting of the status flags before the next comparison is made. See "REP/REPE/REPZ /REPNE/REPNZ-Repeat
String Operation Prefix" in Chapter 4 of the Intel 64 and IA-32 Architectures Software Developer''s Manual,
Volume 2B, for a description of the REP prefix.
In 64-bit mode, the instruction''s default address size is 64 bits, 32 bit address size is supported using the prefix
67H. Use of the REX.W prefix promotes doubleword operation to 64 bits (see CMPSQ). See the summary chart at
the beginning of this section for encoding data and limits.

Operation
temp <- SRC1 - SRC2;
SetStatusFlags(temp);

IF (64-Bit Mode)
    THEN
         IF (Byte comparison)
         THEN IF DF = 0
              THEN
                   (R|E)SI <- (R|E)SI + 1;
                   (R|E)DI <- (R|E)DI + 1;
              ELSE
                   (R|E)SI <- (R|E)SI - 1;
                   (R|E)DI <- (R|E)DI - 1;
              FI;
         ELSE IF (Word comparison)
              THEN IF DF = 0
                   THEN
                        (R|E)SI <- (R|E)SI + 2;
                        (R|E)DI <- (R|E)DI + 2;
                   ELSE
                        (R|E)SI <- (R|E)SI - 2;
                        (R|E)DI <- (R|E)DI - 2;
                   FI;
         ELSE IF (Doubleword comparison)
              THEN IF DF = 0
                   THEN
                        (R|E)SI <- (R|E)SI + 4;
                        (R|E)DI <- (R|E)DI + 4;
                   ELSE
                        (R|E)SI <- (R|E)SI - 4;
                        (R|E)DI <- (R|E)DI - 4;
                   FI;



          ELSE (* Quadword comparison *)
               THEN IF DF = 0
                    (R|E)SI <- (R|E)SI + 8;
                    (R|E)DI <- (R|E)DI + 8;
               ELSE
                    (R|E)SI <- (R|E)SI - 8;
                    (R|E)DI <- (R|E)DI - 8;
               FI;
          FI;
      ELSE (* Non-64-bit Mode *)
          IF (byte comparison)
          THEN IF DF = 0
               THEN
                    (E)SI <- (E)SI + 1;
                    (E)DI <- (E)DI + 1;
               ELSE
                    (E)SI <- (E)SI - 1;
                    (E)DI <- (E)DI - 1;
               FI;
          ELSE IF (Word comparison)
               THEN IF DF = 0
                    (E)SI <- (E)SI + 2;
                    (E)DI <- (E)DI + 2;
               ELSE
                    (E)SI <- (E)SI - 2;
                    (E)DI <- (E)DI - 2;
               FI;
          ELSE (* Doubleword comparison *)
               THEN IF DF = 0
                    (E)SI <- (E)SI + 4;
                    (E)DI <- (E)DI + 4;
               ELSE
                    (E)SI <- (E)SI - 4;
                    (E)DI <- (E)DI - 4;
               FI;
          FI;
FI;

Flags Affected
The CF, OF, SF, ZF, AF, and PF flags are set according to the temporary result of the comparison.

Protected Mode Exceptions
#GP(0)                   If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                         If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)                   If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)          If a page fault occurs.
#AC(0)                   If alignment checking is enabled and an unaligned memory reference is made while the
                         current privilege level is 3.
#UD                      If the LOCK prefix is used.





Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#UD                 If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','CMPSB','-R:CMPS');
INSERT INTO "instructions" VALUES('x86_64','CMPSW','-R:CMPS');
INSERT INTO "instructions" VALUES('x86_64','CMPSD','-R:CMPS');
INSERT INTO "instructions" VALUES('x86_64','CMPSQ','-R:CMPS');
INSERT INTO "instructions" VALUES('x86_64','CMPSD','CMPSD-Compare Scalar Double-Precision Floating-Point Value
    Opcode/                             Op /     64/32       CPUID     Description
    Instruction                         En       bit Mode    Feature
                                                 Support     Flag
    F2 0F C2 /r ib                      RMI      V/V         SSE2      Compare low double-precision floating-point value in
    CMPSD xmm1, xmm2/m64, imm8                                         xmm2/m64 and xmm1 using bits 2:0 of imm8 as comparison
                                                                       predicate.
    VEX.NDS.128.F2.0F.WIG C2 /r ib      RVMI     V/V         AVX       Compare low double-precision floating-point value in
    VCMPSD xmm1, xmm2,                                                 xmm3/m64 and xmm2 using bits 4:0 of imm8 as comparison
    xmm3/m64, imm8                                                     predicate.
    EVEX.NDS.LIG.F2.0F.W1 C2 /r ib      T1S      V/V         AVX512F   Compare low double-precision floating-point value in
    VCMPSD k1 {k2}, xmm2,                                              xmm3/m64 and xmm2 using bits 4:0 of imm8 as comparison
    xmm3/m64{sae}, imm8                                                predicate with writemask k2 and leave the result in mask
                                                                       register k1.



                                                    Instruction Operand Encoding
      Op/En             Operand 1                      Operand 2                Operand 3                    Operand 4
       RMI           ModRM:reg (r, w)                ModRM:r/m (r)                   Imm8                       NA
       RVMI           ModRM:reg (w)                    VEX.vvvv               ModRM:r/m (r)                    Imm8
       T1S            ModRM:reg (w)                    EVEX.vvvv              ModRM:r/m (r)                    Imm8

Description
Compares the low double-precision floating-point values in the second source operand and the first source operand
and returns the results in of the comparison to the destination operand. The comparison predicate operand (imme-
diate operand) specifies the type of comparison performed.
128-bit Legacy SSE version: The first source and destination operand (first operand) is an XMM register. The
second source operand (second operand) can be an XMM register or 64-bit memory location. Bits (MAX_VL-1:64)
of the corresponding YMM destination register remain unchanged. The comparison result is a quadword mask of all
1s (comparison true) or all 0s (comparison false).
VEX.128 encoded version: The first source operand (second operand) is an XMM register. The second source
operand (third operand) can be an XMM register or a 64-bit memory location. The result is stored in the low quad-
word of the destination operand; the high quadword is filled with the contents of the high quadword of the first
source operand. Bits (MAX_VL-1:128) of the destination ZMM register are zeroed. The comparison result is a quad-
word mask of all 1s (comparison true) or all 0s (comparison false).
EVEX encoded version: The first source operand (second operand) is an XMM register. The second source operand
can be a XMM register or a 64-bit memory location. The destination operand (first operand) is an opmask register.
The comparison result is a single mask bit of 1 (comparison true) or 0 (comparison false), written to the destination
starting from the LSB according to the writemask k2. Bits (MAX_KL-1:128) of the destination register are cleared.
The comparison predicate operand is an 8-bit immediate:
.     For instructions encoded using the VEX prefix, bits 4:0 define the type of comparison to be performed (see
      Table 3-1). Bits 5 through 7 of the immediate are reserved.
.     For instruction encodings that do not use VEX prefix, bits 2:0 define the type of comparison to be made (see
      the first 8 rows of Table 3-1). Bits 3 through 7 of the immediate are reserved.
The unordered relationship is true when at least one of the two source operands being compared is a NaN; the
ordered relationship is true when neither source operand is a NaN.
A subsequent computational instruction that uses the mask result in the destination operand as an input operand
will not generate an exception, because a mask of all 0s corresponds to a floating-point value of +0.0 and a mask
of all 1s corresponds to a QNaN.
Note that processors with "CPUID.1H:ECX.AVX =0" do not implement the "greater-than", "greater-than-or-equal",
"not-greater than", and "not-greater-than-or-equal relations" predicates. These comparisons can be made either
by using the inverse relationship (that is, use the "not-less-than-or-equal" to make a "greater-than" comparison)



or by using software emulation. When using software emulation, the program must swap the operands (copying
registers when necessary to protect the data that will now be in the destination), and then perform the compare
using a different predicate. The predicate to be used for these emulations is listed in the first 8 rows of Table 3-7
(Intel 64 and IA-32 Architectures Software Developer''s Manual Volume 2A) under the heading Emulation.
Compilers and assemblers may implement the following two-operand pseudo-ops in addition to the three-operand
CMPSD instruction, for processors with "CPUID.1H:ECX.AVX =0". See Table 3-6. Compiler should treat reserved
Imm8 values as illegal syntax.
                                    Table 3-6. Pseudo-Op and CMPSD Implementation
:




    Pseudo-Op                                                CMPSD Implementation
    CMPEQSD xmm1, xmm2                                       CMPSD xmm1, xmm2, 0
    CMPLTSD xmm1, xmm2                                       CMPSD xmm1, xmm2, 1
    CMPLESD xmm1, xmm2                                       CMPSD xmm1, xmm2, 2
    CMPUNORDSD xmm1, xmm2                                    CMPSD xmm1, xmm2, 3
    CMPNEQSD xmm1, xmm2                                      CMPSD xmm1, xmm2, 4
    CMPNLTSD xmm1, xmm2                                      CMPSD xmm1, xmm2, 5
    CMPNLESD xmm1, xmm2                                      CMPSD xmm1, xmm2, 6
    CMPORDSD xmm1, xmm2                                      CMPSD xmm1, xmm2, 7

The greater-than relations that the processor does not implement require more than one instruction to emulate in
software and therefore should not be implemented as pseudo-ops. (For these, the programmer should reverse the
operands of the corresponding less than relations and use move instructions to ensure that the mask is moved to
the correct destination register and that the source operand is left intact.)
Processors with "CPUID.1H:ECX.AVX =1" implement the full complement of 32 predicates shown in Table 3-7, soft-
ware emulation is no longer needed. Compilers and assemblers may implement the following three-operand
pseudo-ops in addition to the four-operand VCMPSD instruction. See Table 3-7, where the notations of reg1 reg2,
and reg3 represent either XMM registers or YMM registers. Compiler should treat reserved Imm8 values as illegal
syntax. Alternately, intrinsics can map the pseudo-ops to pre-defined constants to support a simpler intrinsic inter-
face. Compilers and assemblers may implement three-operand pseudo-ops for EVEX encoded VCMPSD instructions
in a similar fashion by extending the syntax listed in Table 3-7.
                                    Table 3-7. Pseudo-Op and VCMPSD Implementation
:




    Pseudo-Op                                                CMPSD Implementation
    VCMPEQSD reg1, reg2, reg3                                VCMPSD reg1, reg2, reg3, 0
    VCMPLTSD reg1, reg2, reg3                                VCMPSD reg1, reg2, reg3, 1
    VCMPLESD reg1, reg2, reg3                                VCMPSD reg1, reg2, reg3, 2
    VCMPUNORDSD reg1, reg2, reg3                             VCMPSD reg1, reg2, reg3, 3
    VCMPNEQSD reg1, reg2, reg3                               VCMPSD reg1, reg2, reg3, 4
    VCMPNLTSD reg1, reg2, reg3                               VCMPSD reg1, reg2, reg3, 5
    VCMPNLESD reg1, reg2, reg3                               VCMPSD reg1, reg2, reg3, 6
    VCMPORDSD reg1, reg2, reg3                               VCMPSD reg1, reg2, reg3, 7
    VCMPEQ_UQSD reg1, reg2, reg3                             VCMPSD reg1, reg2, reg3, 8
    VCMPNGESD reg1, reg2, reg3                               VCMPSD reg1, reg2, reg3, 9
    VCMPNGTSD reg1, reg2, reg3                               VCMPSD reg1, reg2, reg3, 0AH
    VCMPFALSESD reg1, reg2, reg3                             VCMPSD reg1, reg2, reg3, 0BH
    VCMPNEQ_OQSD reg1, reg2, reg3                            VCMPSD reg1, reg2, reg3, 0CH
    VCMPGESD reg1, reg2, reg3                                VCMPSD reg1, reg2, reg3, 0DH




                                       Table 3-7. Pseudo-Op and VCMPSD Implementation
 Pseudo-Op                                                      CMPSD Implementation
 VCMPGTSD reg1, reg2, reg3                                     VCMPSD reg1, reg2, reg3, 0EH
 VCMPTRUESD reg1, reg2, reg3                                   VCMPSD reg1, reg2, reg3, 0FH
 VCMPEQ_OSSD reg1, reg2, reg3                                  VCMPSD reg1, reg2, reg3, 10H
 VCMPLT_OQSD reg1, reg2, reg3                                  VCMPSD reg1, reg2, reg3, 11H
 VCMPLE_OQSD reg1, reg2, reg3                                  VCMPSD reg1, reg2, reg3, 12H
 VCMPUNORD_SSD reg1, reg2, reg3                                VCMPSD reg1, reg2, reg3, 13H
 VCMPNEQ_USSD reg1, reg2, reg3                                 VCMPSD reg1, reg2, reg3, 14H
 VCMPNLT_UQSD reg1, reg2, reg3                                 VCMPSD reg1, reg2, reg3, 15H
 VCMPNLE_UQSD reg1, reg2, reg3                                 VCMPSD reg1, reg2, reg3, 16H
 VCMPORD_SSD reg1, reg2, reg3                                  VCMPSD reg1, reg2, reg3, 17H
 VCMPEQ_USSD reg1, reg2, reg3                                  VCMPSD reg1, reg2, reg3, 18H
 VCMPNGE_UQSD reg1, reg2, reg3                                 VCMPSD reg1, reg2, reg3, 19H
 VCMPNGT_UQSD reg1, reg2, reg3                                 VCMPSD reg1, reg2, reg3, 1AH
 VCMPFALSE_OSSD reg1, reg2, reg3                               VCMPSD reg1, reg2, reg3, 1BH
 VCMPNEQ_OSSD reg1, reg2, reg3                                 VCMPSD reg1, reg2, reg3, 1CH
 VCMPGE_OQSD reg1, reg2, reg3                                  VCMPSD reg1, reg2, reg3, 1DH
 VCMPGT_OQSD reg1, reg2, reg3                                  VCMPSD reg1, reg2, reg3, 1EH
 VCMPTRUE_USSD reg1, reg2, reg3                                VCMPSD reg1, reg2, reg3, 1FH

Software should ensure VCMPSD is encoded with VEX.L=0. Encoding VCMPSD with VEX.L=1 may encounter unpre-
dictable behavior across different processor generations.

Operation
CASE (COMPARISON PREDICATE) OF
   0: OP3 <-EQ_OQ; OP5 <-EQ_OQ;
   1: OP3 <-LT_OS; OP5 <-LT_OS;
   2: OP3 <-LE_OS; OP5 <-LE_OS;
   3: OP3 <-UNORD_Q; OP5 <-UNORD_Q;
   4: OP3 <-NEQ_UQ; OP5 <-NEQ_UQ;
   5: OP3 <-NLT_US; OP5 <-NLT_US;
   6: OP3 <-NLE_US; OP5 <-NLE_US;
   7: OP3 <-ORD_Q; OP5 <-ORD_Q;
   8: OP5 <-EQ_UQ;
   9: OP5 <-NGE_US;
   10: OP5 <-NGT_US;
   11: OP5 <-FALSE_OQ;
   12: OP5 <-NEQ_OQ;
   13: OP5 <-GE_OS;
   14: OP5 <-GT_OS;
   15: OP5 <-TRUE_UQ;
   16: OP5 <-EQ_OS;
   17: OP5 <-LT_OQ;
   18: OP5 <-LE_OQ;
   19: OP5 <-UNORD_S;
   20: OP5 <-NEQ_US;
   21: OP5 <-NLT_UQ;



   22: OP5 <-NLE_UQ;
   23: OP5 <-ORD_S;
   24: OP5 <-EQ_US;
   25: OP5 <-NGE_UQ;
   26: OP5 <-NGT_UQ;
   27: OP5 <-FALSE_OS;
   28: OP5 <-NEQ_OS;
   29: OP5 <-GE_OQ;
   30: OP5 <-GT_OQ;
   31: OP5 <-TRUE_US;
   DEFAULT: Reserved
ESAC;

VCMPSD (EVEX encoded version)
CMP0 <- SRC1[63:0] OP5 SRC2[63:0];

IF k2[0] or *no writemask*
    THEN      IF CMP0 = TRUE
                   THEN DEST[0] <- 1;
                   ELSE DEST[0] <- 0; FI;
    ELSE      DEST[0] <- 0                ; zeroing-masking only
FI;
DEST[MAX_KL-1:1] <- 0

CMPSD (128-bit Legacy SSE version)
CMP0 <-DEST[63:0] OP3 SRC[63:0];
IF CMP0 = TRUE
THEN DEST[63:0] <-FFFFFFFFFFFFFFFFH;
ELSE DEST[63:0] <-0000000000000000H; FI;
DEST[MAX_VL-1:64] (Unmodified)

VCMPSD (VEX.128 encoded version)
CMP0 <-SRC1[63:0] OP5 SRC2[63:0];
IF CMP0 = TRUE
THEN DEST[63:0] <-FFFFFFFFFFFFFFFFH;
ELSE DEST[63:0] <-0000000000000000H; FI;
DEST[127:64] <-SRC1[127:64]
DEST[MAX_VL-1:128] <-0

Intel C/C++ Compiler Intrinsic Equivalent
VCMPSD __mmask8 _mm_cmp_sd_mask( __m128d a, __m128d b, int imm);
VCMPSD __mmask8 _mm_cmp_round_sd_mask( __m128d a, __m128d b, int imm, int sae);
VCMPSD __mmask8 _mm_mask_cmp_sd_mask( __mmask8 k1, __m128d a, __m128d b, int imm);
VCMPSD __mmask8 _mm_mask_cmp_round_sd_mask( __mmask8 k1, __m128d a, __m128d b, int imm, int sae);
(V)CMPSD __m128d _mm_cmp_sd(__m128d a, __m128d b, const int imm)

SIMD Floating-Point Exceptions
Invalid if SNaN operand, Invalid if QNaN and predicate as listed in Table 3-1 Denormal.

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3.
EVEX-encoded instructions, see Exceptions Type E3.




');
INSERT INTO "instructions" VALUES('x86_64','CMPSS','CMPSS-Compare Scalar Single-Precision Floating-Point Value
    Opcode/                                  Op /      64/32         CPUID     Description
    Instruction                              En        bit Mode      Feature
                                                       Support       Flag
    F3 0F C2 /r ib                           RMI       V/V           SSE       Compare low single-precision floating-point value in
    CMPSS xmm1, xmm2/m32, imm8                                                 xmm2/m32 and xmm1 using bits 2:0 of imm8 as
                                                                               comparison predicate.
    VEX.NDS.128.F3.0F.WIG C2 /r ib           RVMI      V/V           AVX       Compare low single-precision floating-point value in
    VCMPSS xmm1, xmm2, xmm3/m32,                                               xmm3/m32 and xmm2 using bits 4:0 of imm8 as
    imm8                                                                       comparison predicate.
    EVEX.NDS.LIG.F3.0F.W0 C2 /r ib           T1S       V/V           AVX512F   Compare low single-precision floating-point value in
    VCMPSS k1 {k2}, xmm2,                                                      xmm3/m32 and xmm2 using bits 4:0 of imm8 as
    xmm3/m32{sae}, imm8                                                        comparison predicate with writemask k2 and leave the
                                                                               result in mask register k1.



                                                     Instruction Operand Encoding
       Op/En              Operand 1                          Operand 2               Operand 3                   Operand 4
        RMI            ModRM:reg (r, w)                 ModRM:r/m (r)                  Imm8                          NA
       RVMI             ModRM:reg (w)                        VEX.vvvv              ModRM:r/m (r)                   Imm8
        T1S             ModRM:reg (w)                        EVEX.vvvv             ModRM:r/m (r)                   Imm8

Description
Compares the low single-precision floating-point values in the second source operand and the first source operand
and returns the results of the comparison to the destination operand. The comparison predicate operand (imme-
diate operand) specifies the type of comparison performed.
128-bit Legacy SSE version: The first source and destination operand (first operand) is an XMM register. The
second source operand (second operand) can be an XMM register or 32-bit memory location. Bits (MAX_VL-1:32)
of the corresponding YMM destination register remain unchanged. The comparison result is a doubleword mask of
all 1s (comparison true) or all 0s (comparison false).
VEX.128 encoded version: The first source operand (second operand) is an XMM register. The second source
operand (third operand) can be an XMM register or a 32-bit memory location. The result is stored in the low 32 bits
of the destination operand; bits 128:32 of the destination operand are copied from the first source operand. Bits
(MAX_VL-1:128) of the destination ZMM register are zeroed. The comparison result is a doubleword mask of all 1s
(comparison true) or all 0s (comparison false).
EVEX encoded version: The first source operand (second operand) is an XMM register. The second source operand
can be a XMM register or a 32-bit memory location. The destination operand (first operand) is an opmask register.
The comparison result is a single mask bit of 1 (comparison true) or 0 (comparison false), written to the destination
starting from the LSB according to the writemask k2. Bits (MAX_KL-1:128) of the destination register are cleared.
The comparison predicate operand is an 8-bit immediate:
.     For instructions encoded using the VEX prefix, bits 4:0 define the type of comparison to be performed (see
      Table 3-1). Bits 5 through 7 of the immediate are reserved.
.     For instruction encodings that do not use VEX prefix, bits 2:0 define the type of comparison to be made (see
      the first 8 rows of Table 3-1). Bits 3 through 7 of the immediate are reserved.


The unordered relationship is true when at least one of the two source operands being compared is a NaN; the
ordered relationship is true when neither source operand is a NaN.
A subsequent computational instruction that uses the mask result in the destination operand as an input operand
will not generate an exception, because a mask of all 0s corresponds to a floating-point value of +0.0 and a mask
of all 1s corresponds to a QNaN.
Note that processors with "CPUID.1H:ECX.AVX =0" do not implement the "greater-than", "greater-than-or-equal",
"not-greater than", and "not-greater-than-or-equal relations" predicates. These comparisons can be made either



by using the inverse relationship (that is, use the "not-less-than-or-equal" to make a "greater-than" comparison)
or by using software emulation. When using software emulation, the program must swap the operands (copying
registers when necessary to protect the data that will now be in the destination), and then perform the compare
using a different predicate. The predicate to be used for these emulations is listed in the first 8 rows of Table 3-7
(Intel 64 and IA-32 Architectures Software Developer''s Manual Volume 2A) under the heading Emulation.
Compilers and assemblers may implement the following two-operand pseudo-ops in addition to the three-operand
CMPSS instruction, for processors with "CPUID.1H:ECX.AVX =0". See Table 3-8. Compiler should treat reserved
Imm8 values as illegal syntax.
                                    Table 3-8. Pseudo-Op and CMPSS Implementation
:




    Pseudo-Op                                                CMPSS Implementation
    CMPEQSS xmm1, xmm2                                       CMPSS xmm1, xmm2, 0
    CMPLTSS xmm1, xmm2                                       CMPSS xmm1, xmm2, 1
    CMPLESS xmm1, xmm2                                       CMPSS xmm1, xmm2, 2
    CMPUNORDSS xmm1, xmm2                                    CMPSS xmm1, xmm2, 3
    CMPNEQSS xmm1, xmm2                                      CMPSS xmm1, xmm2, 4
    CMPNLTSS xmm1, xmm2                                      CMPSS xmm1, xmm2, 5
    CMPNLESS xmm1, xmm2                                      CMPSS xmm1, xmm2, 6
    CMPORDSS xmm1, xmm2                                      CMPSS xmm1, xmm2, 7

The greater-than relations that the processor does not implement require more than one instruction to emulate in
software and therefore should not be implemented as pseudo-ops. (For these, the programmer should reverse the
operands of the corresponding less than relations and use move instructions to ensure that the mask is moved to
the correct destination register and that the source operand is left intact.)
Processors with "CPUID.1H:ECX.AVX =1" implement the full complement of 32 predicates shown in Table 3-7, soft-
ware emulation is no longer needed. Compilers and assemblers may implement the following three-operand
pseudo-ops in addition to the four-operand VCMPSS instruction. See Table 3-9, where the notations of reg1 reg2,
and reg3 represent either XMM registers or YMM registers. Compiler should treat reserved Imm8 values as illegal
syntax. Alternately, intrinsics can map the pseudo-ops to pre-defined constants to support a simpler intrinsic inter-
face. Compilers and assemblers may implement three-operand pseudo-ops for EVEX encoded VCMPSS instructions
in a similar fashion by extending the syntax listed in Table 3-9.
                                    Table 3-9. Pseudo-Op and VCMPSS Implementation
:




    Pseudo-Op                                                CMPSS Implementation
    VCMPEQSS reg1, reg2, reg3                                VCMPSS reg1, reg2, reg3, 0
    VCMPLTSS reg1, reg2, reg3                                VCMPSS reg1, reg2, reg3, 1
    VCMPLESS reg1, reg2, reg3                                VCMPSS reg1, reg2, reg3, 2
    VCMPUNORDSS reg1, reg2, reg3                             VCMPSS reg1, reg2, reg3, 3
    VCMPNEQSS reg1, reg2, reg3                               VCMPSS reg1, reg2, reg3, 4
    VCMPNLTSS reg1, reg2, reg3                               VCMPSS reg1, reg2, reg3, 5
    VCMPNLESS reg1, reg2, reg3                               VCMPSS reg1, reg2, reg3, 6
    VCMPORDSS reg1, reg2, reg3                               VCMPSS reg1, reg2, reg3, 7
    VCMPEQ_UQSS reg1, reg2, reg3                             VCMPSS reg1, reg2, reg3, 8
    VCMPNGESS reg1, reg2, reg3                               VCMPSS reg1, reg2, reg3, 9
    VCMPNGTSS reg1, reg2, reg3                               VCMPSS reg1, reg2, reg3, 0AH
    VCMPFALSESS reg1, reg2, reg3                             VCMPSS reg1, reg2, reg3, 0BH
    VCMPNEQ_OQSS reg1, reg2, reg3                            VCMPSS reg1, reg2, reg3, 0CH
    VCMPGESS reg1, reg2, reg3                                VCMPSS reg1, reg2, reg3, 0DH



                                        Table 3-9. Pseudo-Op and VCMPSS Implementation
 Pseudo-Op                                                      CMPSS Implementation
 VCMPGTSS reg1, reg2, reg3                                      VCMPSS reg1, reg2, reg3, 0EH
 VCMPTRUESS reg1, reg2, reg3                                    VCMPSS reg1, reg2, reg3, 0FH
 VCMPEQ_OSSS reg1, reg2, reg3                                   VCMPSS reg1, reg2, reg3, 10H
 VCMPLT_OQSS reg1, reg2, reg3                                   VCMPSS reg1, reg2, reg3, 11H
 VCMPLE_OQSS reg1, reg2, reg3                                   VCMPSS reg1, reg2, reg3, 12H
 VCMPUNORD_SSS reg1, reg2, reg3                                 VCMPSS reg1, reg2, reg3, 13H
 VCMPNEQ_USSS reg1, reg2, reg3                                  VCMPSS reg1, reg2, reg3, 14H
 VCMPNLT_UQSS reg1, reg2, reg3                                  VCMPSS reg1, reg2, reg3, 15H
 VCMPNLE_UQSS reg1, reg2, reg3                                  VCMPSS reg1, reg2, reg3, 16H
 VCMPORD_SSS reg1, reg2, reg3                                   VCMPSS reg1, reg2, reg3, 17H
 VCMPEQ_USSS reg1, reg2, reg3                                   VCMPSS reg1, reg2, reg3, 18H
 VCMPNGE_UQSS reg1, reg2, reg3                                  VCMPSS reg1, reg2, reg3, 19H
 VCMPNGT_UQSS reg1, reg2, reg3                                  VCMPSS reg1, reg2, reg3, 1AH
 VCMPFALSE_OSSS reg1, reg2, reg3                                VCMPSS reg1, reg2, reg3, 1BH
 VCMPNEQ_OSSS reg1, reg2, reg3                                  VCMPSS reg1, reg2, reg3, 1CH
 VCMPGE_OQSS reg1, reg2, reg3                                   VCMPSS reg1, reg2, reg3, 1DH
 VCMPGT_OQSS reg1, reg2, reg3                                   VCMPSS reg1, reg2, reg3, 1EH
 VCMPTRUE_USSS reg1, reg2, reg3                                 VCMPSS reg1, reg2, reg3, 1FH

Software should ensure VCMPSS is encoded with VEX.L=0. Encoding VCMPSS with VEX.L=1 may encounter unpre-
dictable behavior across different processor generations.

Operation
CASE (COMPARISON PREDICATE) OF
   0: OP3 <-EQ_OQ; OP5 <-EQ_OQ;
   1: OP3 <-LT_OS; OP5 <-LT_OS;
   2: OP3 <-LE_OS; OP5 <-LE_OS;
   3: OP3 <-UNORD_Q; OP5 <-UNORD_Q;
   4: OP3 <-NEQ_UQ; OP5 <-NEQ_UQ;
   5: OP3 <-NLT_US; OP5 <-NLT_US;
   6: OP3 <-NLE_US; OP5 <-NLE_US;
   7: OP3 <-ORD_Q; OP5 <-ORD_Q;
   8: OP5 <-EQ_UQ;
   9: OP5 <-NGE_US;
   10: OP5 <-NGT_US;
   11: OP5 <-FALSE_OQ;
   12: OP5 <-NEQ_OQ;
   13: OP5 <-GE_OS;
   14: OP5 <-GT_OS;
   15: OP5 <-TRUE_UQ;
   16: OP5 <-EQ_OS;
   17: OP5 <-LT_OQ;
   18: OP5 <-LE_OQ;
   19: OP5 <-UNORD_S;
   20: OP5 <-NEQ_US;
   21: OP5 <-NLT_UQ;



   22: OP5 <-NLE_UQ;
   23: OP5 <-ORD_S;
   24: OP5 <-EQ_US;
   25: OP5 <-NGE_UQ;
   26: OP5 <-NGT_UQ;
   27: OP5 <-FALSE_OS;
   28: OP5 <-NEQ_OS;
   29: OP5 <-GE_OQ;
   30: OP5 <-GT_OQ;
   31: OP5 <-TRUE_US;
   DEFAULT: Reserved
ESAC;

VCMPSS (EVEX encoded version)
CMP0 <- SRC1[31:0] OP5 SRC2[31:0];

IF k2[0] or *no writemask*
    THEN      IF CMP0 = TRUE
                   THEN DEST[0] <- 1;
                   ELSE DEST[0] <- 0; FI;
    ELSE      DEST[0] <- 0                ; zeroing-masking only
FI;
DEST[MAX_KL-1:1] <- 0

CMPSS (128-bit Legacy SSE version)
CMP0 <-DEST[31:0] OP3 SRC[31:0];
IF CMP0 = TRUE
THEN DEST[31:0] <-FFFFFFFFH;
ELSE DEST[31:0] <-00000000H; FI;
DEST[MAX_VL-1:32] (Unmodified)

VCMPSS (VEX.128 encoded version)
CMP0 <-SRC1[31:0] OP5 SRC2[31:0];
IF CMP0 = TRUE
THEN DEST[31:0] <-FFFFFFFFH;
ELSE DEST[31:0] <-00000000H; FI;
DEST[127:32] <-SRC1[127:32]
DEST[MAX_VL-1:128] <-0

Intel C/C++ Compiler Intrinsic Equivalent
VCMPSS __mmask8 _mm_cmp_ss_mask( __m128 a, __m128 b, int imm);
VCMPSS __mmask8 _mm_cmp_round_ss_mask( __m128 a, __m128 b, int imm, int sae);
VCMPSS __mmask8 _mm_mask_cmp_ss_mask( __mmask8 k1, __m128 a, __m128 b, int imm);
VCMPSS __mmask8 _mm_mask_cmp_round_ss_mask( __mmask8 k1, __m128 a, __m128 b, int imm, int sae);
(V)CMPSS __m128 _mm_cmp_ss(__m128 a, __m128 b, const int imm)

SIMD Floating-Point Exceptions
Invalid if SNaN operand, Invalid if QNaN and predicate as listed in Table 3-1, Denormal.

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3.
EVEX-encoded instructions, see Exceptions Type E3.




');
INSERT INTO "instructions" VALUES('x86_64','CMPXCHG','CMPXCHG-Compare and Exchange
Opcode/                                Op/    64-Bit       Compat/ Description
Instruction                            En     Mode         Leg Mode
0F B0/r                                MR     Valid        Valid*      Compare AL with r/m8. If equal, ZF is set and r8 is loaded into
CMPXCHG r/m8, r8                                                       r/m8. Else, clear ZF and load r/m8 into AL.

REX + 0F B0/r                          MR     Valid        N.E.        Compare AL with r/m8. If equal, ZF is set and r8 is loaded into
CMPXCHG r/m8**,r8                                                      r/m8. Else, clear ZF and load r/m8 into AL.

0F B1/r                                MR     Valid        Valid*      Compare AX with r/m16. If equal, ZF is set and r16 is loaded
CMPXCHG r/m16, r16                                                     into r/m16. Else, clear ZF and load r/m16 into AX.
0F B1/r                                MR     Valid        Valid*      Compare EAX with r/m32. If equal, ZF is set and r32 is loaded
CMPXCHG r/m32, r32                                                     into r/m32. Else, clear ZF and load r/m32 into EAX.

REX.W + 0F B1/r                        MR     Valid        N.E.        Compare RAX with r/m64. If equal, ZF is set and r64 is loaded
CMPXCHG r/m64, r64                                                     into r/m64. Else, clear ZF and load r/m64 into RAX.

NOTES:
* See the IA-32 Architecture Compatibility section below.
** In 64-bit mode, r/m8 can not be encoded to access the following byte registers if a REX prefix is used: AH, BH, CH, DH.



                                                 Instruction Operand Encoding
  Op/En               Operand 1                         Operand 2                     Operand 3                    Operand 4
    MR             ModRM:r/m (r, w)                    ModRM:reg (r)                     NA                            NA

Description
Compares the value in the AL, AX, EAX, or RAX register with the first operand (destination operand). If the two
values are equal, the second operand (source operand) is loaded into the destination operand. Otherwise, the
destination operand is loaded into the AL, AX, EAX or RAX register. RAX register is available only in 64-bit mode.
This instruction can be used with a LOCK prefix to allow the instruction to be executed atomically. To simplify the
interface to the processor''s bus, the destination operand receives a write cycle without regard to the result of the
comparison. The destination operand is written back if the comparison fails; otherwise, the source operand is
written into the destination. (The processor never produces a locked read without also producing a locked write.)
In 64-bit mode, the instruction''s default operation size is 32 bits. Use of the REX.R prefix permits access to addi-
tional registers (R8-R15). Use of the REX.W prefix promotes operation to 64 bits. See the summary chart at the
beginning of this section for encoding data and limits.

IA-32 Architecture Compatibility
This instruction is not supported on Intel processors earlier than the Intel486 processors.

Operation
(* Accumulator = AL, AX, EAX, or RAX depending on whether a byte, word, doubleword, or quadword comparison is being performed *)
TEMP <- DEST
IF accumulator = TEMP
    THEN
        ZF <- 1;
        DEST <- SRC;
    ELSE
        ZF <- 0;
        accumulator <- TEMP;
        DEST <- TEMP;
FI;



Flags Affected
The ZF flag is set if the values in the destination operand and register AL, AX, or EAX are equal; otherwise it is
cleared. The CF, PF, AF, SF, and OF flags are set according to the results of the comparison operation.

Protected Mode Exceptions
#GP(0)               If the destination is located in a non-writable segment.
                     If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                     If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)               If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made while the
                     current privilege level is 3.
#UD                  If the LOCK prefix is used but the destination is not a memory operand.

Real-Address Mode Exceptions
#GP                  If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                  If a memory operand effective address is outside the SS segment limit.
#UD                  If the LOCK prefix is used but the destination is not a memory operand.

Virtual-8086 Mode Exceptions
#GP(0)               If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)               If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made.
#UD                  If the LOCK prefix is used but the destination is not a memory operand.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)               If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)               If the memory address is in a non-canonical form.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made while the
                     current privilege level is 3.
#UD                  If the LOCK prefix is used but the destination is not a memory operand.




');
INSERT INTO "instructions" VALUES('x86_64','CMPXCHG8B','CMPXCHG8B/CMPXCHG16B-Compare and Exchange Bytes
Opcode/                                Op/    64-Bit          Compat/ Description
Instruction                            En     Mode            Leg Mode
0F C7 /1 m64                           M      Valid           Valid*   Compare EDX:EAX with m64. If equal, set ZF and load
CMPXCHG8B m64                                                          ECX:EBX into m64. Else, clear ZF and load m64 into EDX:EAX.

REX.W + 0F C7 /1 m128                  M      Valid           N.E.     Compare RDX:RAX with m128. If equal, set ZF and load
CMPXCHG16B m128                                                        RCX:RBX into m128. Else, clear ZF and load m128 into
                                                                       RDX:RAX.
NOTES:
*See IA-32 Architecture Compatibility section below.



                                                Instruction Operand Encoding
  Op/En              Operand 1                        Operand 2                  Operand 3                    Operand 4
    M            ModRM:r/m (r, w)                        NA                         NA                            NA

Description
Compares the 64-bit value in EDX:EAX (or 128-bit value in RDX:RAX if operand size is 128 bits) with the operand
(destination operand). If the values are equal, the 64-bit value in ECX:EBX (or 128-bit value in RCX:RBX) is stored
in the destination operand. Otherwise, the value in the destination operand is loaded into EDX:EAX (or RDX:RAX).
The destination operand is an 8-byte memory location (or 16-byte memory location if operand size is 128 bits). For
the EDX:EAX and ECX:EBX register pairs, EDX and ECX contain the high-order 32 bits and EAX and EBX contain the
low-order 32 bits of a 64-bit value. For the RDX:RAX and RCX:RBX register pairs, RDX and RCX contain the high-
order 64 bits and RAX and RBX contain the low-order 64bits of a 128-bit value.
This instruction can be used with a LOCK prefix to allow the instruction to be executed atomically. To simplify the
interface to the processor''s bus, the destination operand receives a write cycle without regard to the result of the
comparison. The destination operand is written back if the comparison fails; otherwise, the source operand is
written into the destination. (The processor never produces a locked read without also producing a locked write.)
In 64-bit mode, default operation size is 64 bits. Use of the REX.W prefix promotes operation to 128 bits. Note that
CMPXCHG16B requires that the destination (memory) operand be 16-byte aligned. See the summary chart at the
beginning of this section for encoding data and limits. For information on the CPUID flag that indicates
CMPXCHG16B, see page 3-206.

IA-32 Architecture Compatibility
This instruction encoding is not supported on Intel processors earlier than the Pentium processors.





Operation
IF (64-Bit Mode and OperandSize = 64)
    THEN
         TEMP128 <- DEST
         IF (RDX:RAX = TEMP128)
              THEN
                  ZF <- 1;
                  DEST <- RCX:RBX;
              ELSE
                  ZF <- 0;
                  RDX:RAX <- TEMP128;
                  DEST <- TEMP128;
                  FI;
         FI
    ELSE
         TEMP64 <- DEST;
         IF (EDX:EAX = TEMP64)
              THEN
                  ZF <- 1;
                  DEST <- ECX:EBX;
              ELSE
                  ZF <- 0;
                  EDX:EAX <- TEMP64;
                  DEST <- TEMP64;
                  FI;
         FI;
FI;

Flags Affected
The ZF flag is set if the destination operand and EDX:EAX are equal; otherwise it is cleared. The CF, PF, AF, SF, and
OF flags are unaffected.

Protected Mode Exceptions
#UD                  If the destination is not a memory operand.
#GP(0)               If the destination is located in a non-writable segment.
                     If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                     If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)               If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made while the
                     current privilege level is 3.

Real-Address Mode Exceptions
#UD                  If the destination operand is not a memory location.
#GP                  If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                  If a memory operand effective address is outside the SS segment limit.





Virtual-8086 Mode Exceptions
#UD                  If the destination operand is not a memory location.
#GP(0)               If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)               If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)               If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)               If the memory address is in a non-canonical form.
                     If memory operand for CMPXCHG16B is not aligned on a 16-byte boundary.
                     If CPUID.01H:ECX.CMPXCHG16B[bit 13] = 0.
#UD                  If the destination operand is not a memory location.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made while the
                     current privilege level is 3.




');
INSERT INTO "instructions" VALUES('x86_64','CMPXCHG16B','-R:CMPXCHG8B');
INSERT INTO "instructions" VALUES('x86_64','COMISD','COMISD-Compare Scalar Ordered Double-Precision Floating-Point Values and Set EFLAGS
 Opcode/                              Op /    64/32        CPUID         Description
 Instruction                          En      bit Mode     Feature
                                              Support      Flag
 66 0F 2F /r                          RM      V/V          SSE2          Compare low double-precision floating-point values in
 COMISD xmm1, xmm2/m64                                                   xmm1 and xmm2/mem64 and set the EFLAGS flags
                                                                         accordingly.
 VEX.128.66.0F.WIG 2F /r              RM      V/V          AVX           Compare low double-precision floating-point values in
 VCOMISD xmm1, xmm2/m64                                                  xmm1 and xmm2/mem64 and set the EFLAGS flags
                                                                         accordingly.
 EVEX.LIG.66.0F.W1 2F /r              T1S     V/V          AVX512F       Compare low double-precision floating-point values in
 VCOMISD xmm1, xmm2/m64{sae}                                             xmm1 and xmm2/mem64 and set the EFLAGS flags
                                                                         accordingly.



                                             Instruction Operand Encoding
      Op/En            Operand 1                    Operand 2                   Operand 3                      Operand 4
       RM            ModRM:reg (w)              ModRM:r/m (r)                       NA                            NA
       T1S           ModRM:reg (w)              ModRM:r/m (r)                       NA                            NA

Description
Compares the double-precision floating-point values in the low quadwords of operand 1 (first operand) and
operand 2 (second operand), and sets the ZF, PF, and CF flags in the EFLAGS register according to the result (unor-
dered, greater than, less than, or equal). The OF, SF and AF flags in the EFLAGS register are set to 0. The unor-
dered result is returned if either source operand is a NaN (QNaN or SNaN).
Operand 1 is an XMM register; operand 2 can be an XMM register or a 64 bit memory
location. The COMISD instruction differs from the UCOMISD instruction in that it signals a SIMD floating-point
invalid operation exception (#I) when a source operand is either a QNaN or SNaN. The UCOMISD instruction signals
an invalid numeric exception only if a source operand is an SNaN.
The EFLAGS register is not updated if an unmasked SIMD floating-point exception is generated.
VEX.vvvv and EVEX.vvvv are reserved and must be 1111b, otherwise instructions will #UD.
Software should ensure VCOMISD is encoded with VEX.L=0. Encoding VCOMISD with VEX.L=1 may encounter
unpredictable behavior across different processor generations.

Operation
COMISD (all versions)
RESULT <- OrderedCompare(DEST[63:0] <> SRC[63:0]) {
(* Set EFLAGS *) CASE (RESULT) OF
    UNORDERED: ZF,PF,CF <- 111;
    GREATER_THAN: ZF,PF,CF <- 000;
    LESS_THAN: ZF,PF,CF <- 001;
    EQUAL: ZF,PF,CF <- 100;
ESAC;
OF, AF, SF <-0; }





Intel C/C++ Compiler Intrinsic Equivalent
VCOMISD int _mm_comi_round_sd(__m128d a, __m128d b, int imm, int sae);
VCOMISD int _mm_comieq_sd (__m128d a, __m128d b)
VCOMISD int _mm_comilt_sd (__m128d a, __m128d b)
VCOMISD int _mm_comile_sd (__m128d a, __m128d b)
VCOMISD int _mm_comigt_sd (__m128d a, __m128d b)
VCOMISD int _mm_comige_sd (__m128d a, __m128d b)
VCOMISD int _mm_comineq_sd (__m128d a, __m128d b)

SIMD Floating-Point Exceptions
Invalid (if SNaN or QNaN operands), Denormal.

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3;
EVEX-encoded instructions, see Exceptions Type E3NF.
#UD                      If VEX.vvvv != 1111B or EVEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','COMISS','COMISS-Compare Scalar Ordered Single-Precision Floating-Point Values and Set EFLAGS
 Opcode/                              Op /    64/32         CPUID          Description
 Instruction                          En      bit Mode      Feature
                                              Support       Flag
 0F 2F /r                             RM      V/V           SSE            Compare low single-precision floating-point values in
 COMISS xmm1, xmm2/m32                                                     xmm1 and xmm2/mem32 and set the EFLAGS flags
                                                                           accordingly.
 VEX.128.0F.WIG 2F /r                 RM      V/V           AVX            Compare low single-precision floating-point values in
 VCOMISS xmm1, xmm2/m32                                                    xmm1 and xmm2/mem32 and set the EFLAGS flags
                                                                           accordingly.
 EVEX.LIG.0F.W0 2F /r                 T1S     V/V           AVX512F        Compare low single-precision floating-point values in
 VCOMISS xmm1, xmm2/m32{sae}                                               xmm1 and xmm2/mem32 and set the EFLAGS flags
                                                                           accordingly.



                                             Instruction Operand Encoding
     Op/En             Operand 1                    Operand 2                      Operand 3                     Operand 4
       RM            ModRM:reg (w)              ModRM:r/m (r)                          NA                            NA
      T1S            ModRM:reg (w)              ModRM:r/m (r)                          NA                            NA

Description
Compares the single-precision floating-point values in the low quadwords of operand 1 (first operand) and operand
2 (second operand), and sets the ZF, PF, and CF flags in the EFLAGS register according to the result (unordered,
greater than, less than, or equal). The OF, SF and AF flags in the EFLAGS register are set to 0. The unordered result
is returned if either source operand is a NaN (QNaN or SNaN).
Operand 1 is an XMM register; operand 2 can be an XMM register or a 32 bit memory location.
The COMISS instruction differs from the UCOMISS instruction in that it signals a SIMD floating-point invalid opera-
tion exception (#I) when a source operand is either a QNaN or SNaN. The UCOMISS instruction signals an invalid
numeric exception only if a source operand is an SNaN.
The EFLAGS register is not updated if an unmasked SIMD floating-point exception is generated.
VEX.vvvv and EVEX.vvvv are reserved and must be 1111b, otherwise instructions will #UD.
Software should ensure VCOMISS is encoded with VEX.L=0. Encoding VCOMISS with VEX.L=1 may encounter
unpredictable behavior across different processor generations.

Operation
COMISS (all versions)
RESULT <- OrderedCompare(DEST[31:0] <> SRC[31:0]) {
(* Set EFLAGS *) CASE (RESULT) OF
    UNORDERED: ZF,PF,CF <- 111;
    GREATER_THAN: ZF,PF,CF <- 000;
    LESS_THAN: ZF,PF,CF <- 001;
    EQUAL: ZF,PF,CF <- 100;
ESAC;
OF, AF, SF <- 0; }





Intel C/C++ Compiler Intrinsic Equivalent
VCOMISS int _mm_comi_round_ss(__m128 a, __m128 b, int imm, int sae);
VCOMISS int _mm_comieq_ss (__m128 a, __m128 b)
VCOMISS int _mm_comilt_ss (__m128 a, __m128 b)
VCOMISS int _mm_comile_ss (__m128 a, __m128 b)
VCOMISS int _mm_comigt_ss (__m128 a, __m128 b)
VCOMISS int _mm_comige_ss (__m128 a, __m128 b)
VCOMISS int _mm_comineq_ss (__m128 a, __m128 b)

SIMD Floating-Point Exceptions
Invalid (if SNaN or QNaN operands), Denormal.

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3;
EVEX-encoded instructions, see Exceptions Type E3NF.
#UD                      If VEX.vvvv != 1111B or EVEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','CPUID','CPUID-CPU Identification
Opcode                    Instruction                  Op/   64-Bit      Compat/ Description
                                                       En    Mode        Leg Mode
0F A2                     CPUID                        NP    Valid       Valid       Returns processor identification and feature
                                                                                     information to the EAX, EBX, ECX, and EDX
                                                                                     registers, as determined by input entered in
                                                                                     EAX (in some cases, ECX as well).



                                                Instruction Operand Encoding
  Op/En               Operand 1                     Operand 2                     Operand 3                    Operand 4
    NP                   NA                             NA                            NA                           NA

Description
The ID flag (bit 21) in the EFLAGS register indicates support for the CPUID instruction. If a software procedure can
set and clear this flag, the processor executing the procedure supports the CPUID instruction. This instruction oper-
ates the same in non-64-bit modes and 64-bit mode.
CPUID returns processor identification and feature information in the EAX, EBX, ECX, and EDX registers.1 The
instruction''s output is dependent on the contents of the EAX register upon execution (in some cases, ECX as well).
For example, the following pseudocode loads EAX with 00H and causes CPUID to return a Maximum Return Value
and the Vendor Identification String in the appropriate registers:

   MOV EAX, 00H
   CPUID
Table 3-8 shows information returned, depending on the initial value loaded into the EAX register.
Two types of information are returned: basic and extended function information. If a value entered for CPUID.EAX
is higher than the maximum input value for basic or extended function for that processor then the data for the
highest basic information leaf is returned. For example, using the Intel Core i7 processor, the following is true:
   CPUID.EAX = 05H (* Returns MONITOR/MWAIT leaf. *)
   CPUID.EAX = 0AH (* Returns Architectural Performance Monitoring leaf. *)
   CPUID.EAX = 0BH (* Returns Extended Topology Enumeration leaf. *)
   CPUID.EAX = 0CH (* INVALID: Returns the same information as CPUID.EAX = 0BH. *)
   CPUID.EAX = 80000008H (* Returns linear/physical address size data. *)
   CPUID.EAX = 8000000AH (* INVALID: Returns same information as CPUID.EAX = 0BH. *)
If a value entered for CPUID.EAX is less than or equal to the maximum input value and the leaf is not supported on
that processor then 0 is returned in all the registers.
When CPUID returns the highest basic leaf information as a result of an invalid input EAX value, any dependence
on input ECX value in the basic leaf is honored.
CPUID can be executed at any privilege level to serialize instruction execution. Serializing instruction execution
guarantees that any modifications to flags, registers, and memory for previous instructions are completed before
the next instruction is fetched and executed.
See also:
"Serializing Instructions" in Chapter 8, "Multiple-Processor Management," in the Intel 64 and IA-32 Architectures
Software Developer''s Manual, Volume 3A.
"Caching Translation Information" in Chapter 4, "Paging," in the Intel 64 and IA-32 Architectures Software Devel-
oper''s Manual, Volume 3A.




1. On Intel 64 processors, CPUID clears the high 32 bits of the RAX/RBX/RCX/RDX registers in all modes.



                                     Table 3-8. Information Returned by CPUID Instruction
  Initial EAX
     Value                                          Information Provided about the Processor
                 Basic CPUID Information
      0H         EAX        Maximum Input Value for Basic CPUID Information.
                 EBX        "Genu"
                 ECX        "ntel"
                 EDX        "ineI"
      01H        EAX        Version Information: Type, Family, Model, and Stepping ID (see Figure 3-6).
                 EBX        Bits 07 - 00: Brand Index.
                            Bits 15 - 08: CLFLUSH line size (Value * 8 = cache line size in bytes; used also by CLFLUSHOPT).
                            Bits 23 - 16: Maximum number of addressable IDs for logical processors in this physical package*.
                            Bits 31 - 24: Initial APIC ID.
                 ECX        Feature Information (see Figure 3-7 and Table 3-10).
                 EDX        Feature Information (see Figure 3-8 and Table 3-11).
                            NOTES:
                            * The nearest power-of-2 integer that is not smaller than EBX[23:16] is the number of unique initial APIC
                              IDs reserved for addressing different logical processors in a physical package. This field is only valid if
                              CPUID.1.EDX.HTT[bit 28]= 1.
      02H        EAX        Cache and TLB Information (see Table 3-12).
                 EBX        Cache and TLB Information.
                 ECX        Cache and TLB Information.
                 EDX        Cache and TLB Information.
      03H        EAX        Reserved.
                 EBX        Reserved.
                 ECX        Bits 00 - 31 of 96 bit processor serial number. (Available in Pentium III processor only; otherwise, the
                            value in this register is reserved.)
                 EDX        Bits 32 - 63 of 96 bit processor serial number. (Available in Pentium III processor only; otherwise, the
                            value in this register is reserved.)
                            NOTES:
                              Processor serial number (PSN) is not supported in the Pentium 4 processor or later. On all models, use
                              the PSN flag (returned using CPUID) to check for PSN support before accessing the feature.
 CPUID leaves above 2 and below 80000000H are visible only when IA32_MISC_ENABLE[bit 22] has its default value of 0.
                 Deterministic Cache Parameters Leaf
      04H                   NOTES:
                              Leaf 04H output depends on the initial value in ECX.*
                              See also: "INPUT EAX = 04H: Returns Deterministic Cache Parameters for Each Level" on page 214.

                 EAX        Bits 04 - 00: Cache Type Field.
                              0 = Null - No more caches.
                              1 = Data Cache.
                              2 = Instruction Cache.
                              3 = Unified Cache.
                              4-31 = Reserved.





                           Table 3-8. Information Returned by CPUID Instruction (Contd.)
  Initial EAX
     Value                                         Information Provided about the Processor
                         Bits 07 - 05: Cache Level (starts at 1).
                         Bit 08: Self Initializing cache level (does not need SW initialization).
                         Bit 09: Fully Associative cache.
                         Bits 13 - 10: Reserved.
                         Bits 25 - 14: Maximum number of addressable IDs for logical processors sharing this cache**, ***.
                         Bits 31 - 26: Maximum number of addressable IDs for processor cores in the physical
                         package**, ****, *****.
                EBX      Bits 11 - 00: L = System Coherency Line Size**.
                         Bits 21 - 12: P = Physical Line partitions**.
                         Bits 31 - 22: W = Ways of associativity**.
                ECX      Bits 31-00: S = Number of Sets**.
                EDX      Bit 00: Write-Back Invalidate/Invalidate.
                           0 = WBINVD/INVD from threads sharing this cache acts upon lower level caches for threads sharing this
                           cache.
                           1 = WBINVD/INVD is not guaranteed to act upon lower level caches of non-originating threads sharing
                           this cache.
                         Bit 01: Cache Inclusiveness.
                           0 = Cache is not inclusive of lower cache levels.
                           1 = Cache is inclusive of lower cache levels.
                         Bit 02: Complex Cache Indexing.
                           0 = Direct mapped cache.
                           1 = A complex function is used to index the cache, potentially using all address bits.
                         Bits 31 - 03: Reserved = 0.
                         NOTES:
                         * If ECX contains an invalid sub leaf index, EAX/EBX/ECX/EDX return 0. Sub-leaf index n+1 is invalid if sub-
                            leaf n returns EAX[4:0] as 0.
                         ** Add one to the return value to get the result.
                         ***The nearest power-of-2 integer that is not smaller than (1 + EAX[25:14]) is the number of unique ini-
                            tial APIC IDs reserved for addressing different logical processors sharing this cache.
                         **** The nearest power-of-2 integer that is not smaller than (1 + EAX[31:26]) is the number of unique
                            Core_IDs reserved for addressing different processor cores in a physical package. Core ID is a subset of
                            bits of the initial APIC ID.
                         ***** The returned value is constant for valid initial values in ECX. Valid ECX values start from 0.
                MONITOR/MWAIT Leaf
     05H        EAX      Bits 15 - 00: Smallest monitor-line size in bytes (default is processor''s monitor granularity).
                         Bits 31 - 16: Reserved = 0.
                EBX      Bits 15 - 00: Largest monitor-line size in bytes (default is processor''s monitor granularity).
                         Bits 31 - 16: Reserved = 0.
                ECX      Bit 00: Enumeration of Monitor-Mwait extensions (beyond EAX and EBX registers) supported.
                         Bit 01: Supports treating interrupts as break-event for MWAIT, even when interrupts disabled.
                         Bits 31 - 02: Reserved.





                             Table 3-8. Information Returned by CPUID Instruction (Contd.)
  Initial EAX
     Value                                         Information Provided about the Processor
                 EDX       Bits 03 - 00: Number of C0* sub C-states supported using MWAIT.
                           Bits 07 - 04: Number of C1* sub C-states supported using MWAIT.
                           Bits 11 - 08: Number of C2* sub C-states supported using MWAIT.
                           Bits 15 - 12: Number of C3* sub C-states supported using MWAIT.
                           Bits 19 - 16: Number of C4* sub C-states supported using MWAIT.
                           Bits 23 - 20: Number of C5* sub C-states supported using MWAIT.
                           Bits 27 - 24: Number of C6* sub C-states supported using MWAIT.
                           Bits 31 - 28: Number of C7* sub C-states supported using MWAIT.
                           NOTE:
                           * The definition of C0 through C7 states for MWAIT extension are processor-specific C-states, not ACPI C-
                              states.
                 Thermal and Power Management Leaf
      06H        EAX       Bit 00: Digital temperature sensor is supported if set.
                           Bit 01: Intel Turbo Boost Technology Available (see description of IA32_MISC_ENABLE[38]).
                           Bit 02: ARAT. APIC-Timer-always-running feature is supported if set.
                           Bit 03: Reserved.
                           Bit 04: PLN. Power limit notification controls are supported if set.
                           Bit 05: ECMD. Clock modulation duty cycle extension is supported if set.
                           Bit 06: PTM. Package thermal management is supported if set.
                           Bit 07: HWP. HWP base registers (IA32_PM_ENABLE[bit 0], IA32_HWP_CAPABILITIES,
                           IA32_HWP_REQUEST, IA32_HWP_STATUS) are supported if set.
                           Bit 08: HWP_Notification. IA32_HWP_INTERRUPT MSR is supported if set.
                           Bit 09: HWP_Activity_Window. IA32_HWP_REQUEST[bits 41:32] is supported if set.
                           Bit 10: HWP_Energy_Performance_Preference. IA32_HWP_REQUEST[bits 31:24] is supported if set.
                           Bit 11: HWP_Package_Level_Request. IA32_HWP_REQUEST_PKG MSR is supported if set.
                           Bit 12: Reserved.
                           Bit 13: HDC. HDC base registers IA32_PKG_HDC_CTL, IA32_PM_CTL1, IA32_THREAD_STALL MSRs are
                           supported if set.
                           Bits 31 - 15: Reserved.
                 EBX       Bits 03 - 00: Number of Interrupt Thresholds in Digital Thermal Sensor.
                           Bits 31 - 04: Reserved.
                 ECX       Bit 00: Hardware Coordination Feedback Capability (Presence of IA32_MPERF and IA32_APERF). The
                           capability to provide a measure of delivered processor performance (since last reset of the counters), as
                           a percentage of the expected processor performance when running at the TSC frequency.
                           Bits 02 - 01: Reserved = 0.
                           Bit 03: The processor supports performance-energy bias preference if CPUID.06H:ECX.SETBH[bit 3] is set
                           and it also implies the presence of a new architectural MSR called IA32_ENERGY_PERF_BIAS (1B0H).
                           Bits 31 - 04: Reserved = 0.
                 EDX       Reserved = 0.





                             Table 3-8. Information Returned by CPUID Instruction (Contd.)
  Initial EAX
     Value                                         Information Provided about the Processor
                Structured Extended Feature Flags Enumeration Leaf (Output depends on ECX input value)
     07H                     Sub-leaf 0 (Input ECX = 0). *


                EAX        Bits 31 - 00: Reports the maximum input value for supported leaf 7 sub-leaves.
                EBX        Bit 00: FSGSBASE. Supports RDFSBASE/RDGSBASE/WRFSBASE/WRGSBASE if 1.
                           Bit 01: IA32_TSC_ADJUST MSR is supported if 1.
                           Bit 02: SGX. Supports Intel Software Guard Extensions (Intel SGX Extensions) if 1.
                           Bit 03: BMI1.
                           Bit 04: HLE.
                           Bit 05: AVX2.
                           Bit 06: FDP_EXCPTN_ONLY. x87 FPU Data Pointer updated only on x87 exceptions if 1.
                           Bit 07: SMEP. Supports Supervisor-Mode Execution Prevention if 1.
                           Bit 08: BMI2.
                           Bit 09: Supports Enhanced REP MOVSB/STOSB if 1.
                           Bit 10: INVPCID. If 1, supports INVPCID instruction for system software that manages process-context
                           identifiers.
                           Bit 11: RTM.
                           Bit 12: RDT-M. Supports Intel Resource Director Technology (Intel RDT) Monitoring capability if 1.
                           Bit 13: Deprecates FPU CS and FPU DS values if 1.
                           Bit 14: MPX. Supports Intel Memory Protection Extensions if 1.
                           Bit 15: RDT-A. Supports Intel Resource Director Technology (Intel RDT) Allocation capability if 1.
                           Bits 17:16: Reserved.
                           Bit 18: RDSEED.
                           Bit 19: ADX.
                           Bit 20: SMAP. Supports Supervisor-Mode Access Prevention (and the CLAC/STAC instructions) if 1.
                           Bits 22 - 21: Reserved.
                           Bit 23: CLFLUSHOPT.
                           Bit 24: CLWB.
                           Bit 25: Intel Processor Trace.
                           Bits 28 - 26: Reserved.
                           Bit 29: SHA. supports Intel Secure Hash Algorithm Extensions (Intel SHA Extensions) if 1.
                           Bits 31 - 30: Reserved.
                ECX        Bit 00: PREFETCHWT1.
                           Bit 01: Reserved.
                           Bit 02: UMIP. Supports user-mode instruction prevention if 1.
                           Bit 03: PKU. Supports protection keys for user-mode pages if 1.
                           Bit 04: OSPKE. If 1, OS has set CR4.PKE to enable protection keys (and the RDPKRU/WRPKRU instruc-
                           tions).
                           Bits 16 - 5: Reserved.
                           Bits 21 - 17: The value of MAWAU used by the BNDLDX and BNDSTX instructions in 64-bit mode.
                           Bit 22: RDPID. Supports Read Processor ID if 1.
                           Bits 29 - 23: Reserved.
                           Bit 30: SGX_LC. Supports SGX Launch Configuration if 1.
                           Bit 31: Reserved.
                EDX        Reserved.

                           NOTE:
                           * If ECX contains an invalid sub-leaf index, EAX/EBX/ECX/EDX return 0. Sub-leaf index n is invalid if n
                              exceeds the value that sub-leaf 0 returns in EAX.





                              Table 3-8. Information Returned by CPUID Instruction (Contd.)
  Initial EAX
     Value                                           Information Provided about the Processor
                 Direct Cache Access Information Leaf
      09H        EAX        Value of bits [31:0] of IA32_PLATFORM_DCA_CAP MSR (address 1F8H).
                 EBX        Reserved.
                 ECX        Reserved.
                 EDX        Reserved.
                 Architectural Performance Monitoring Leaf
      0AH        EAX        Bits 07 - 00: Version ID of architectural performance monitoring.
                            Bits 15 - 08: Number of general-purpose performance monitoring counter per logical processor.
                            Bits 23 - 16: Bit width of general-purpose, performance monitoring counter.
                            Bits 31 - 24: Length of EBX bit vector to enumerate architectural performance monitoring events.
                 EBX        Bit 00: Core cycle event not available if 1.
                            Bit 01: Instruction retired event not available if 1.
                            Bit 02: Reference cycles event not available if 1.
                            Bit 03: Last-level cache reference event not available if 1.
                            Bit 04: Last-level cache misses event not available if 1.
                            Bit 05: Branch instruction retired event not available if 1.
                            Bit 06: Branch mispredict retired event not available if 1.
                            Bits 31 - 07: Reserved = 0.
                 ECX        Reserved = 0.
                 EDX        Bits 04 - 00: Number of fixed-function performance counters (if Version ID > 1).
                            Bits 12 - 05: Bit width of fixed-function performance counters (if Version ID > 1).
                            Reserved = 0.
                 Extended Topology Enumeration Leaf
      0BH                   NOTES:
                              Most of Leaf 0BH output depends on the initial value in ECX.
                              The EDX output of leaf 0BH is always valid and does not vary with input value in ECX.
                              Output value in ECX[7:0] always equals input value in ECX[7:0].
                              For sub-leaves that return an invalid level-type of 0 in ECX[15:8]; EAX and EBX will return 0.
                              If an input value n in ECX returns the invalid level-type of 0 in ECX[15:8], other input values with ECX >
                              n also return 0 in ECX[15:8].


                 EAX        Bits 04 - 00: Number of bits to shift right on x2APIC ID to get a unique topology ID of the next level type*.
                            All logical processors with the same next level ID share current level.
                            Bits 31 - 05: Reserved.
                 EBX        Bits 15 - 00: Number of logical processors at this level type. The number reflects configuration as shipped
                            by Intel**.
                            Bits 31- 16: Reserved.
                 ECX        Bits 07 - 00: Level number. Same value in ECX input.
                            Bits 15 - 08: Level type***.
                            Bits 31 - 16: Reserved.
                 EDX        Bits 31- 00: x2APIC ID the current logical processor.
                            NOTES:
                            * Software should use this field (EAX[4:0]) to enumerate processor topology of the system.





                             Table 3-8. Information Returned by CPUID Instruction (Contd.)
  Initial EAX
     Value                                          Information Provided about the Processor
                           ** Software must not use EBX[15:0] to enumerate processor topology of the system. This value in this
                           field (EBX[15:0]) is only intended for display/diagnostic purposes. The actual number of logical processors
                           available to BIOS/OS/Applications may be different from the value of EBX[15:0], depending on software
                           and platform hardware configurations.

                           *** The value of the "level type" field is not related to level numbers in any way, higher "level type" val-
                           ues do not mean higher levels. Level type field has the following encoding:
                           0: Invalid.
                           1: SMT.
                           2: Core.
                           3-255: Reserved.
                Processor Extended State Enumeration Main Leaf (EAX = 0DH, ECX = 0)
     0DH                   NOTES:
                             Leaf 0DH main leaf (ECX = 0).
                EAX        Bits 31 - 00: Reports the supported bits of the lower 32 bits of XCR0. XCR0[n] can be set to 1 only if
                           EAX[n] is 1.
                           Bit 00: x87 state.
                           Bit 01: SSE state.
                           Bit 02: AVX state.
                           Bits 04 - 03: MPX state.
                           Bits 07 - 05: AVX-512 state.
                           Bit 08: Used for IA32_XSS.
                           Bit 09: PKRU state.
                           Bits 31 - 10: Reserved.
                EBX        Bits 31 - 00: Maximum size (bytes, from the beginning of the XSAVE/XRSTOR save area) required by
                           enabled features in XCR0. May be different than ECX if some features at the end of the XSAVE save area
                           are not enabled.
                ECX        Bit 31 - 00: Maximum size (bytes, from the beginning of the XSAVE/XRSTOR save area) of the
                           XSAVE/XRSTOR save area required by all supported features in the processor, i.e., all the valid bit fields in
                           XCR0.
                EDX        Bit 31 - 00: Reports the supported bits of the upper 32 bits of XCR0. XCR0[n+32] can be set to 1 only if
                           EDX[n] is 1.
                           Bits 31 - 00: Reserved.
                Processor Extended State Enumeration Sub-leaf (EAX = 0DH, ECX = 1)
     0DH        EAX        Bit 00: XSAVEOPT is available.
                           Bit 01: Supports XSAVEC and the compacted form of XRSTOR if set.
                           Bit 02: Supports XGETBV with ECX = 1 if set.
                           Bit 03: Supports XSAVES/XRSTORS and IA32_XSS if set.
                           Bits 31 - 04: Reserved.
                EBX        Bits 31 - 00: The size in bytes of the XSAVE area containing all states enabled by XCRO | IA32_XSS.
                ECX        Bits 31 - 00: Reports the supported bits of the lower 32 bits of the IA32_XSS MSR. IA32_XSS[n] can be
                           set to 1 only if ECX[n] is 1.
                           Bits 07 - 00: Used for XCR0.
                           Bit 08: PT state.
                           Bit 09: Used for XCR0.
                           Bits 31 - 10: Reserved.
                EDX        Bits 31 - 00: Reports the supported bits of the upper 32 bits of the IA32_XSS MSR. IA32_XSS[n+32] can
                           be set to 1 only if EDX[n] is 1.
                           Bits 31 - 00: Reserved.




                              Table 3-8. Information Returned by CPUID Instruction (Contd.)
  Initial EAX
     Value                                            Information Provided about the Processor
                 Processor Extended State Enumeration Sub-leaves (EAX = 0DH, ECX = n, n > 1)
      0DH                   NOTES:
                              Leaf 0DH output depends on the initial value in ECX.
                              Each sub-leaf index (starting at position 2) is supported if it corresponds to a supported bit in either the
                              XCR0 register or the IA32_XSS MSR.
                             * If ECX contains an invalid sub-leaf index, EAX/EBX/ECX/EDX return 0. Sub-leaf n (0 <= n <= 31) is invalid
                               if sub-leaf 0 returns 0 in EAX[n] and sub-leaf 1 returns 0 in ECX[n]. Sub-leaf n (32 <= n <= 63) is invalid if
                               sub-leaf 0 returns 0 in EDX[n-32] and sub-leaf 1 returns 0 in EDX[n-32].
                 EAX        Bits 31 - 0: The size in bytes (from the offset specified in EBX) of the save area for an extended state
                            feature associated with a valid sub-leaf index, n.
                 EBX        Bits 31 - 0: The offset in bytes of this extended state component''s save area from the beginning of the
                            XSAVE/XRSTOR area.
                            This field reports 0 if the sub-leaf index, n, does not map to a valid bit in the XCR0 register*.
                 ECX        Bit 00 is set if the bit n (corresponding to the sub-leaf index) is supported in the IA32_XSS MSR; it is clear
                            if bit n is instead supported in XCR0.
                            Bit 01 is set if, when the compacted format of an XSAVE area is used, this extended state component
                            located on the next 64-byte boundary following the preceding state component (otherwise, it is located
                            immediately following the preceding state component).
                            Bits 31 - 02 are reserved.
                            This field reports 0 if the sub-leaf index, n, is invalid*.
                 EDX        This field reports 0 if the sub-leaf index, n, is invalid*; otherwise it is reserved.
                 Intel Resource Director Technology (Intel RDT) Monitoring Enumeration Sub-leaf (EAX = 0FH, ECX = 0)
      0FH                   NOTES:
                              Leaf 0FH output depends on the initial value in ECX.
                              Sub-leaf index 0 reports valid resource type starting at bit position 1 of EDX.
                 EAX        Reserved.
                 EBX        Bits 31 - 00: Maximum range (zero-based) of RMID within this physical processor of all types.
                 ECX        Reserved.
                 EDX        Bit 00: Reserved.
                            Bit 01: Supports L3 Cache Intel RDT Monitoring if 1.
                            Bits 31 - 02: Reserved.
                 L3 Cache Intel RDT Monitoring Capability Enumeration Sub-leaf (EAX = 0FH, ECX = 1)
      0FH                   NOTES:
                              Leaf 0FH output depends on the initial value in ECX.
                 EAX        Reserved.
                 EBX        Bits 31 - 00: Conversion factor from reported IA32_QM_CTR value to occupancy metric (bytes).
                 ECX        Maximum range (zero-based) of RMID of this resource type.
                 EDX        Bit 00: Supports L3 occupancy monitoring if 1.
                            Bit 01: Supports L3 Total Bandwidth monitoring if 1.
                            Bit 02: Supports L3 Local Bandwidth monitoring if 1.
                            Bits 31 - 03: Reserved.





                             Table 3-8. Information Returned by CPUID Instruction (Contd.)
  Initial EAX
     Value                                           Information Provided about the Processor
                Intel Resource Director Technology (Intel RDT) Allocation Enumeration Sub-leaf (EAX = 10H, ECX = 0)
     10H                   NOTES:
                             Leaf 10H output depends on the initial value in ECX.
                             Sub-leaf index 0 reports valid resource identification (ResID) starting at bit position 1 of EBX.
                EAX        Reserved.
                EBX        Bit 00: Reserved.
                           Bit 01: Supports L3 Cache Allocation Technology if 1.
                           Bit 02: Supports L2 Cache Allocation Technology if 1.
                           Bits 31 - 03: Reserved.
                ECX        Reserved.
                EDX        Reserved.
                L3 Cache Allocation Technology Enumeration Sub-leaf (EAX = 10H, ECX = ResID =1)
     10H                   NOTES:
                             Leaf 10H output depends on the initial value in ECX.
                EAX        Bits 4 - 00: Length of the capacity bit mask for the corresponding ResID using minus-one notation.
                           Bits 31 - 05: Reserved.
                EBX        Bits 31 - 00: Bit-granular map of isolation/contention of allocation units.
                ECX        Bit 00: Reserved.
                           Bit 01: Updates of COS should be infrequent if 1.
                           Bit 02: Code and Data Prioritization Technology supported if 1.
                           Bits 31 - 03: Reserved.
                EDX        Bits 15 - 00: Highest COS number supported for this ResID.
                           Bits 31 - 16: Reserved.
                L2 Cache Allocation Technology Enumeration Sub-leaf (EAX = 10H, ECX = ResID =2)
     10H                   NOTES:
                             Leaf 10H output depends on the initial value in ECX.
                EAX        Bits 4 - 00: Length of the capacity bit mask for the corresponding ResID using minus-one notation.
                           Bits 31 - 05: Reserved.
                EBX        Bits 31 - 00: Bit-granular map of isolation/contention of allocation units.
                ECX        Bits 31 - 00: Reserved.
                EDX        Bits 15 - 00: Highest COS number supported for this ResID.
                           Bits 31 - 16: Reserved.
                Intel SGX Capability Enumeration Leaf, sub-leaf 0 (EAX = 12H, ECX = 0)
     12H                   NOTES:
                             Leaf 12H sub-leaf 0 (ECX = 0) is supported if CPUID.(EAX=07H, ECX=0H):EBX[SGX] = 1.
                EAX        Bit 00: SGX1. If 1, Indicates Intel SGX supports the collection of SGX1 leaf functions.
                           Bit 01: SGX2. If 1, Indicates Intel SGX supports the collection of SGX2 leaf functions.
                           Bit 31 - 02: Reserved.
                EBX        Bit 31 - 00: MISCSELECT. Bit vector of supported extended SGX features.
                ECX        Bit 31 - 00: Reserved.





                              Table 3-8. Information Returned by CPUID Instruction (Contd.)
  Initial EAX
     Value                                           Information Provided about the Processor
                 EDX        Bit 07 - 00: MaxEnclaveSize_Not64. The maximum supported enclave size in non-64-bit mode is
                            2^(EDX[7:0]).
                            Bit 15 - 08: MaxEnclaveSize_64. The maximum supported enclave size in 64-bit mode is 2^(EDX[15:8]).
                            Bits 31 - 16: Reserved.
                 Intel SGX Attributes Enumeration Leaf, sub-leaf 1 (EAX = 12H, ECX = 1)
      12H                   NOTES:
                              Leaf 12H sub-leaf 1 (ECX = 1) is supported if CPUID.(EAX=07H, ECX=0H):EBX[SGX] = 1.
                 EAX        Bit 31 - 00: Reports the valid bits of SECS.ATTRIBUTES[31:0] that software can set with ECREATE.
                 EBX        Bit 31 - 00: Reports the valid bits of SECS.ATTRIBUTES[63:32] that software can set with ECREATE.
                 ECX        Bit 31 - 00: Reports the valid bits of SECS.ATTRIBUTES[95:64] that software can set with ECREATE.
                 EDX        Bit 31 - 00: Reports the valid bits of SECS.ATTRIBUTES[127:96] that software can set with ECREATE.
                 Intel SGX EPC Enumeration Leaf, sub-leaves (EAX = 12H, ECX = 2 or higher)
      12H                   NOTES:
                              Leaf 12H sub-leaf 2 or higher (ECX >= 2) is supported if CPUID.(EAX=07H, ECX=0H):EBX[SGX] = 1.
                              For sub-leaves (ECX = 2 or higher), definition of EDX,ECX,EBX,EAX[31:4] depends on the sub-leaf type
                              listed below.
                 EAX        Bit 03 - 00: Sub-leaf Type
                              0000b: Indicates this sub-leaf is invalid.
                              0001b: This sub-leaf enumerates an EPC section. EBX:EAX and EDX:ECX provide information on the
                              Enclave Page Cache (EPC) section.
                              All other type encodings are reserved.
                 Type       0000b. This sub-leaf is invalid.
                              EDX:ECX:EBX:EAX return 0.
                 Type       0001b. This sub-leaf enumerates an EPC sections with EDX:ECX, EBX:EAX defined as follows.
                              EAX[11:04]: Reserved (enumerate 0).
                              EAX[31:12]: Bits 31:12 of the physical address of the base of the EPC section.

                              EBX[19:00]: Bits 51:32 of the physical address of the base of the EPC section.
                              EBX[31:20]: Reserved.

                              ECX[03:00]: EPC section property encoding defined as follows:
                               If EAX[3:0] 0000b, then all bits of the EDX:ECX pair are enumerated as 0.
                               If EAX[3:0] 0001b, then this section has confidentiality and integrity protection.
                               All other encodings are reserved.
                              ECX[11:04]: Reserved (enumerate 0).
                              ECX[31:12]: Bits 31:12 of the size of the corresponding EPC section within the Processor Reserved
                              Memory.

                              EDX[19:00]: Bits 51:32 of the size of the corresponding EPC section within the Processor Reserved
                              Memory.
                              EDX[31:20]: Reserved.





                             Table 3-8. Information Returned by CPUID Instruction (Contd.)
  Initial EAX
     Value                                           Information Provided about the Processor
                Intel Processor Trace Enumeration Main Leaf (EAX = 14H, ECX = 0)
     14H                   NOTES:
                             Leaf 14H main leaf (ECX = 0).
                EAX        Bits 31 - 00: Reports the maximum sub-leaf supported in leaf 14H.
                EBX        Bit 00: If 1, indicates that IA32_RTIT_CTL.CR3Filter can be set to 1, and that IA32_RTIT_CR3_MATCH
                           MSR can be accessed.
                           Bit 01: If 1, indicates support of Configurable PSB and Cycle-Accurate Mode.
                           Bit 02: If 1, indicates support of IP Filtering, TraceStop filtering, and preservation of Intel PT MSRs across
                           warm reset.
                           Bit 03: If 1, indicates support of MTC timing packet and suppression of COFI-based packets.
                           Bit 04: If 1, indicates support of PTWRITE. Writes can set IA32_RTIT_CTL[12] (PTWEn) and
                           IA32_RTIT_CTL[5] (FUPonPTW), and PTWRITE can generate packets.
                           Bit 05: If 1, indicates support of Power Event Trace. Writes can set IA32_RTIT_CTL[4] (PwrEvtEn),
                           enabling Power Event Trace packet generation.
                           Bit 31 - 06: Reserved.
                ECX        Bit 00: If 1, Tracing can be enabled with IA32_RTIT_CTL.ToPA = 1, hence utilizing the ToPA output
                           scheme; IA32_RTIT_OUTPUT_BASE and IA32_RTIT_OUTPUT_MASK_PTRS MSRs can be accessed.
                           Bit 01: If 1, ToPA tables can hold any number of output entries, up to the maximum allowed by the Mas-
                           kOrTableOffset field of IA32_RTIT_OUTPUT_MASK_PTRS.
                           Bit 02: If 1, indicates support of Single-Range Output scheme.
                           Bit 03: If 1, indicates support of output to Trace Transport subsystem.
                           Bit 30 - 04: Reserved.
                           Bit 31: If 1, generated packets which contain IP payloads have LIP values, which include the CS base com-
                           ponent.
                EDX        Bits 31 - 00: Reserved.
                Intel Processor Trace Enumeration Sub-leaf (EAX = 14H, ECX = 1)
     14H        EAX        Bits 02 - 00: Number of configurable Address Ranges for filtering.
                           Bits 15 - 03: Reserved.
                           Bits 31 - 16: Bitmap of supported MTC period encodings.
                EBX        Bits 15 - 00: Bitmap of supported Cycle Threshold value encodings.
                           Bit 31 - 16: Bitmap of supported Configurable PSB frequency encodings.
                ECX        Bits 31 - 00: Reserved.
                EDX        Bits 31 - 00: Reserved.
                Time Stamp Counter and Nominal Core Crystal Clock Information Leaf
     15H                   NOTES:
                             If EBX[31:0] is 0, the TSC/"core crystal clock" ratio is not enumerated.
                             EBX[31:0]/EAX[31:0] indicates the ratio of the TSC frequency and the core crystal clock frequency.
                             If ECX is 0, the nominal core crystal clock frequency is not enumerated.
                             "TSC frequency" = "core crystal clock frequency" * EBX/EAX.
                            The core crystal clock may differ from the reference clock, bus clock, or core clock frequencies.
                EAX        Bits 31 - 00: An unsigned integer which is the denominator of the TSC/"core crystal clock" ratio.
                EBX        Bits 31 - 00: An unsigned integer which is the numerator of the TSC/"core crystal clock" ratio.
                ECX        Bits 31 - 00: An unsigned integer which is the nominal frequency of the core crystal clock in Hz.
                EDX        Bits 31 - 00: Reserved = 0.





                              Table 3-8. Information Returned by CPUID Instruction (Contd.)
  Initial EAX
     Value                                          Information Provided about the Processor
                 Processor Frequency Information Leaf
      16H        EAX        Bits 15 - 00: Processor Base Frequency (in MHz).
                            Bits 31 - 16: Reserved =0.
                 EBX        Bits 15 - 00: Maximum Frequency (in MHz).
                            Bits 31 - 16: Reserved = 0.
                 ECX        Bits 15 - 00: Bus (Reference) Frequency (in MHz).
                            Bits 31 - 16: Reserved = 0.
                 EDX        Reserved.
                            NOTES:
                            * Data is returned from this interface in accordance with the processor''s specification and does not reflect
                            actual values. Suitable use of this data includes the display of processor information in like manner to the
                            processor brand string and for determining the appropriate range to use when displaying processor
                            information e.g. frequency history graphs. The returned information should not be used for any other
                            purpose as the returned information does not accurately correlate to information / counters returned by
                            other processor interfaces.

                            While a processor may support the Processor Frequency Information leaf, fields that return a value of
                            zero are not supported.
                 System-On-Chip Vendor Attribute Enumeration Main Leaf (EAX = 17H, ECX = 0)
      17H                   NOTES:
                              Leaf 17H main leaf (ECX = 0).
                              Leaf 17H output depends on the initial value in ECX.
                              Leaf 17H sub-leaves 1 through 3 reports SOC Vendor Brand String.
                             Leaf 17H is valid if MaxSOCID_Index >= 3.
                             Leaf 17H sub-leaves 4 and above are reserved.


                 EAX        Bits 31 - 00: MaxSOCID_Index. Reports the maximum input value of supported sub-leaf in leaf 17H.
                 EBX        Bits 15 - 00: SOC Vendor ID.
                            Bit 16: IsVendorScheme. If 1, the SOC Vendor ID field is assigned via an industry standard enumeration
                            scheme. Otherwise, the SOC Vendor ID field is assigned by Intel.
                            Bits 31 - 17: Reserved = 0.
                 ECX        Bits 31 - 00: Project ID. A unique number an SOC vendor assigns to its SOC projects.
                 EDX        Bits 31 - 00: Stepping ID. A unique number within an SOC project that an SOC vendor assigns.
                 System-On-Chip Vendor Attribute Enumeration Sub-leaf (EAX = 17H, ECX = 1..3)
      17H        EAX        Bit 31 - 00: SOC Vendor Brand String. UTF-8 encoded string.
                 EBX        Bit 31 - 00: SOC Vendor Brand String. UTF-8 encoded string.
                 ECX        Bit 31 - 00: SOC Vendor Brand String. UTF-8 encoded string.
                 EDX        Bit 31 - 00: SOC Vendor Brand String. UTF-8 encoded string.
                            NOTES:
                             Leaf 17H output depends on the initial value in ECX.
                             SOC Vendor Brand String is a UTF-8 encoded string padded with trailing bytes of 00H.
                             The complete SOC Vendor Brand String is constructed by concatenating in ascending order of
                             EAX:EBX:ECX:EDX and from the sub-leaf 1 fragment towards sub-leaf 3.





                             Table 3-8. Information Returned by CPUID Instruction (Contd.)
  Initial EAX
     Value                                          Information Provided about the Processor
                System-On-Chip Vendor Attribute Enumeration Sub-leaves (EAX = 17H, ECX > MaxSOCID_Index)
     17H                   NOTES:
                            Leaf 17H output depends on the initial value in ECX.


                EAX        Bits 31 - 00: Reserved = 0.
                EBX        Bits 31 - 00: Reserved = 0.
                ECX        Bits 31 - 00: Reserved = 0.
                EDX        Bits 31 - 00: Reserved = 0.
                Unimplemented CPUID Leaf Functions
 40000000H                 Invalid. No existing or future CPU will return processor identification or feature information if the initial
     -                     EAX value is in the range 40000000H to 4FFFFFFFH.
 4FFFFFFFH
                Extended Function CPUID Information
 80000000H EAX             Maximum Input Value for Extended Function CPUID Information.
                EBX        Reserved.
                ECX        Reserved.
                EDX        Reserved.
 80000001H EAX             Extended Processor Signature and Feature Bits.
                EBX        Reserved.
                ECX        Bit 00: LAHF/SAHF available in 64-bit mode.
                           Bits 04 - 01: Reserved.
                           Bit 05: LZCNT.
                           Bits 07 - 06: Reserved.
                           Bit 08: PREFETCHW.
                           Bits 31 - 09: Reserved.
                EDX        Bits 10 - 00: Reserved.
                           Bit 11: SYSCALL/SYSRET available in 64-bit mode.
                           Bits 19 - 12: Reserved = 0.
                           Bit 20: Execute Disable Bit available.
                           Bits 25 - 21: Reserved = 0.
                           Bit 26: 1-GByte pages are available if 1.
                           Bit 27: RDTSCP and IA32_TSC_AUX are available if 1.
                           Bit 28: Reserved = 0.
                           Bit 29: Intel 64 Architecture available if 1.
                           Bits 31 - 30: Reserved = 0.
 80000002H EAX             Processor Brand String.
           EBX             Processor Brand String Continued.
           ECX             Processor Brand String Continued.
           EDX             Processor Brand String Continued.
 80000003H EAX             Processor Brand String Continued.
           EBX             Processor Brand String Continued.
           ECX             Processor Brand String Continued.
           EDX             Processor Brand String Continued.





                             Table 3-8. Information Returned by CPUID Instruction (Contd.)
  Initial EAX
     Value                                          Information Provided about the Processor
 80000004H EAX             Processor Brand String Continued.
           EBX             Processor Brand String Continued.
           ECX             Processor Brand String Continued.
           EDX             Processor Brand String Continued.
 80000005H EAX             Reserved = 0.
           EBX             Reserved = 0.
           ECX             Reserved = 0.
           EDX             Reserved = 0.
 80000006H EAX             Reserved = 0.
           EBX             Reserved = 0.
                 ECX       Bits 07 - 00: Cache Line size in bytes.
                           Bits 11 - 08: Reserved.
                           Bits 15 - 12: L2 Associativity field *.
                           Bits 31 - 16: Cache size in 1K units.
                 EDX       Reserved = 0.
                           NOTES:
                           * L2 associativity field encodings:
                              00H - Disabled.
                              01H - Direct mapped.
                              02H - 2-way.
                              04H - 4-way.
                              06H - 8-way.
                              08H - 16-way.
                              0FH - Fully associative.
 80000007H EAX             Reserved = 0.
           EBX             Reserved = 0.
           ECX             Reserved = 0.
           EDX             Bits 07 - 00: Reserved = 0.
                           Bit 08: Invariant TSC available if 1.
                           Bits 31 - 09: Reserved = 0.
 80000008H EAX             Linear/Physical Address size.
                           Bits 07 - 00: #Physical Address Bits*.
                           Bits 15 - 08: #Linear Address Bits.
                           Bits 31 - 16: Reserved = 0.
                 EBX       Reserved = 0.
                 ECX       Reserved = 0.
                 EDX       Reserved = 0.

                           NOTES:
                           * If CPUID.80000008H:EAX[7:0] is supported, the maximum physical address number supported should
                             come from this field.


INPUT EAX = 0: Returns CPUID''s Highest Value for Basic Processor Information and the Vendor Identification String
When CPUID executes with EAX set to 0, the processor returns the highest value the CPUID recognizes for
returning basic processor information. The value is returned in the EAX register and is processor specific.





A vendor identification string is also returned in EBX, EDX, and ECX. For Intel processors, the string is "Genuin-
eIntel" and is expressed:
    EBX <- 756e6547h (* "Genu", with G in the low eight bits of BL *)
    EDX <- 49656e69h (* "ineI", with i in the low eight bits of DL *)
    ECX <- 6c65746eh (* "ntel", with n in the low eight bits of CL *)

INPUT EAX = 80000000H: Returns CPUID''s Highest Value for Extended Processor Information
When CPUID executes with EAX set to 80000000H, the processor returns the highest value the processor recog-
nizes for returning extended processor information. The value is returned in the EAX register and is processor
specific.

IA32_BIOS_SIGN_ID Returns Microcode Update Signature
For processors that support the microcode update facility, the IA32_BIOS_SIGN_ID MSR is loaded with the update
signature whenever CPUID executes. The signature is returned in the upper DWORD. For details, see Chapter 9 in
the Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 3A.

INPUT EAX = 01H: Returns Model, Family, Stepping Information
When CPUID executes with EAX set to 01H, version information is returned in EAX (see Figure 3-6). For example:
model, family, and processor type for the Intel Xeon processor 5100 series is as follows:
.   Model - 1111B
.   Family - 0101B
.   Processor Type - 00B
See Table 3-9 for available processor type values. Stepping IDs are provided as needed.




                            31     28 27                20 19      16 15 14 13 12 11    8 7           4   3        0


                                           Extended        Extended                Family                 Stepping
                     EAX                                                                      Model
                                           Family ID       Model ID                  ID                      ID


                    Extended Family ID (0)
                    Extended Model ID (0)
                    Processor Type
                    Family (0FH for the Pentium 4 Processor Family)
                    Model

                           Reserved
                                                                                                              OM16525


                                 Figure 3-6. Version Information Returned by CPUID in EAX





                                                   Table 3-9. Processor Type Field
                                Type                                                      Encoding
 Original OEM Processor                                                                     00B
                  
 Intel OverDrive Processor                                                                  01B
 Dual processor (not applicable to Intel486 processors)                                     10B
 Intel reserved                                                                             11B

                                                                   NOTE
         See Chapter 19 in the Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 1,
         for information on identifying earlier IA-32 processors.


The Extended Family ID needs to be examined only when the Family ID is 0FH. Integrate the fields into a display
using the following rule:

    IF Family_ID != 0FH
         THEN DisplayFamily = Family_ID;
         ELSE DisplayFamily = Extended_Family_ID + Family_ID;
         (* Right justify and zero-extend 4-bit field. *)
    FI;
    (* Show DisplayFamily as HEX field. *)
The Extended Model ID needs to be examined only when the Family ID is 06H or 0FH. Integrate the field into a
display using the following rule:

    IF (Family_ID = 06H or Family_ID = 0FH)
         THEN DisplayModel = (Extended_Model_ID << 4) + Model_ID;
         (* Right justify and zero-extend 4-bit field; display Model_ID as HEX field.*)
         ELSE DisplayModel = Model_ID;
    FI;
    (* Show DisplayModel as HEX field. *)

INPUT EAX = 01H: Returns Additional Information in EBX
When CPUID executes with EAX set to 01H, additional information is returned to the EBX register:
.   Brand index (low byte of EBX) - this number provides an entry into a brand string table that contains brand
    strings for IA-32 processors. More information about this field is provided later in this section.
.   CLFLUSH instruction cache line size (second byte of EBX) - this number indicates the size of the cache line
    flushed by the CLFLUSH and CLFLUSHOPT instructions in 8-byte increments. This field was introduced in the
    Pentium 4 processor.
.   Local APIC ID (high byte of EBX) - this number is the 8-bit ID that is assigned to the local APIC on the
    processor during power up. This field was introduced in the Pentium 4 processor.

INPUT EAX = 01H: Returns Feature Information in ECX and EDX
When CPUID executes with EAX set to 01H, feature information is returned in ECX and EDX.
.   Figure 3-7 and Table 3-10 show encodings for ECX.
.   Figure 3-8 and Table 3-11 show encodings for EDX.
For all feature flags, a 1 indicates that the feature is supported. Use Intel to properly interpret feature flags.

                                                                   NOTE
         Software must confirm that a processor feature is present using feature flags returned by CPUID
         prior to using the feature. Software should not depend on future offerings retaining all features.





                                        31 30 29 28 27 26 25 24 23 22 21 20 19 18 17 16 15 14 13 12 11 10 9 8 7 6 5 4   3 2 1   0



                                 ECX
                                        0


                        RDRAND
                        F16C
                        AVX
                        OSXSAVE
                        XSAVE
                        AES
                        TSC-Deadline
                        POPCNT
                        MOVBE
                        x2APIC
                        SSE4_2 - SSE4.2
                        SSE4_1 - SSE4.1
                        DCA - Direct Cache Access
                        PCID - Process-context Identifiers
                        PDCM - Perf/Debug Capability MSR
                        xTPR Update Control
                        CMPXCHG16B
                        FMA - Fused Multiply Add
                        SDBG
                        CNXT-ID - L1 Context ID
                        SSSE3 - SSSE3 Extensions
                        TM2 - Thermal Monitor 2
                        EIST - Enhanced Intel SpeedStep Technology
                        SMX - Safer Mode Extensions
                        VMX - Virtual Machine Extensions
                        DS-CPL - CPL Qualified Debug Store
                        MONITOR - MONITOR/MWAIT
                        DTES64 - 64-bit DS Area
                        PCLMULQDQ - Carryless Multiplication
                        SSE3 - SSE3 Extensions
                                                                                                                          OM16524b
                          Reserved

                                 Figure 3-7. Feature Information Returned in the ECX Register


                                Table 3-10. Feature Information Returned in the ECX Register
 Bit #          Mnemonic             Description
 0              SSE3                 Streaming SIMD Extensions 3 (SSE3). A value of 1 indicates the processor supports this
                                     technology.
 1              PCLMULQDQ            PCLMULQDQ. A value of 1 indicates the processor supports the PCLMULQDQ instruction.
 2              DTES64               64-bit DS Area. A value of 1 indicates the processor supports DS area using 64-bit layout.
 3              MONITOR              MONITOR/MWAIT. A value of 1 indicates the processor supports this feature.
 4              DS-CPL               CPL Qualified Debug Store. A value of 1 indicates the processor supports the extensions to the
                                     Debug Store feature to allow for branch message storage qualified by CPL.
 5              VMX                  Virtual Machine Extensions. A value of 1 indicates that the processor supports this technology.
 6              SMX                  Safer Mode Extensions. A value of 1 indicates that the processor supports this technology. See
                                     Chapter 6, "Safer Mode Extensions Reference".
 7              EIST                 Enhanced Intel SpeedStep technology. A value of 1 indicates that the processor supports this
                                     technology.
 8              TM2                  Thermal Monitor 2. A value of 1 indicates whether the processor supports this technology.
 9              SSSE3                A value of 1 indicates the presence of the Supplemental Streaming SIMD Extensions 3 (SSSE3). A
                                     value of 0 indicates the instruction extensions are not present in the processor.



                           Table 3-10. Feature Information Returned in the ECX Register (Contd.)
  Bit #        Mnemonic           Description
  10           CNXT-ID            L1 Context ID. A value of 1 indicates the L1 data cache mode can be set to either adaptive mode
                                  or shared mode. A value of 0 indicates this feature is not supported. See definition of the
                                  IA32_MISC_ENABLE MSR Bit 24 (L1 Data Cache Context Mode) for details.
  11           SDBG               A value of 1 indicates the processor supports IA32_DEBUG_INTERFACE MSR for silicon debug.
  12           FMA                A value of 1 indicates the processor supports FMA extensions using YMM state.
  13           CMPXCHG16B         CMPXCHG16B Available. A value of 1 indicates that the feature is available. See the
                                  "CMPXCHG8B/CMPXCHG16B-Compare and Exchange Bytes" section in this chapter for a
                                  description.
  14           xTPR Update        xTPR Update Control. A value of 1 indicates that the processor supports changing
               Control            IA32_MISC_ENABLE[bit 23].
  15           PDCM               Perfmon and Debug Capability: A value of 1 indicates the processor supports the performance
                                  and debug feature indication MSR IA32_PERF_CAPABILITIES.
  16           Reserved           Reserved
  17           PCID               Process-context identifiers. A value of 1 indicates that the processor supports PCIDs and that
                                  software may set CR4.PCIDE to 1.
  18           DCA                A value of 1 indicates the processor supports the ability to prefetch data from a memory mapped
                                  device.
  19           SSE4.1             A value of 1 indicates that the processor supports SSE4.1.
  20           SSE4.2             A value of 1 indicates that the processor supports SSE4.2.
  21           x2APIC             A value of 1 indicates that the processor supports x2APIC feature.
  22           MOVBE              A value of 1 indicates that the processor supports MOVBE instruction.
  23           POPCNT             A value of 1 indicates that the processor supports the POPCNT instruction.
  24           TSC-Deadline       A value of 1 indicates that the processor''s local APIC timer supports one-shot operation using a
                                  TSC deadline value.
  25           AESNI              A value of 1 indicates that the processor supports the AESNI instruction extensions.
  26           XSAVE              A value of 1 indicates that the processor supports the XSAVE/XRSTOR processor extended states
                                  feature, the XSETBV/XGETBV instructions, and XCR0.
  27           OSXSAVE            A value of 1 indicates that the OS has set CR4.OSXSAVE[bit 18] to enable XSETBV/XGETBV
                                  instructions to access XCR0 and to support processor extended state management using
                                  XSAVE/XRSTOR.
  28           AVX                A value of 1 indicates the processor supports the AVX instruction extensions.
  29           F16C               A value of 1 indicates that processor supports 16-bit floating-point conversion instructions.
  30           RDRAND             A value of 1 indicates that processor supports RDRAND instruction.
  31           Not Used           Always returns 0.





                                  31 30 29 28 27 26 25 24 23 22 21 20 19 18 17 16 15 14 13 12 11 10 9 8 7 6 5 4 3 2 1    0


                           EDX


                PBE-Pend. Brk. EN.
                TM-Therm. Monitor
                HTT-Multi-threading
                SS-Self Snoop
                SSE2-SSE2 Extensions
                SSE-SSE Extensions
                FXSR-FXSAVE/FXRSTOR
                MMX-MMX Technology
                ACPI-Thermal Monitor and Clock Ctrl
                DS-Debug Store
                CLFSH-CLFLUSH instruction
                PSN-Processor Serial Number
                PSE-36 - Page Size Extension
                PAT-Page Attribute Table
                CMOV-Conditional Move/Compare Instruction
                MCA-Machine Check Architecture
                PGE-PTE Global Bit
                MTRR-Memory Type Range Registers
                SEP-SYSENTER and SYSEXIT
                APIC-APIC on Chip
                CX8-CMPXCHG8B Inst.
                MCE-Machine Check Exception
                PAE-Physical Address Extensions
                MSR-RDMSR and WRMSR Support
                TSC-Time Stamp Counter
                PSE-Page Size Extensions
                DE-Debugging Extensions
                VME-Virtual-8086 Mode Enhancement
                FPU-x87 FPU on Chip

                     Reserved
                                                                                                                 OM16523



                          Figure 3-8. Feature Information Returned in the EDX Register





                             Table 3-11. More on Feature Information Returned in the EDX Register
 Bit #    Mnemonic         Description
    0     FPU              Floating Point Unit On-Chip. The processor contains an x87 FPU.
    1     VME              Virtual 8086 Mode Enhancements. Virtual 8086 mode enhancements, including CR4.VME for controlling the
                           feature, CR4.PVI for protected mode virtual interrupts, software interrupt indirection, expansion of the TSS
                           with the software indirection bitmap, and EFLAGS.VIF and EFLAGS.VIP flags.
    2     DE               Debugging Extensions. Support for I/O breakpoints, including CR4.DE for controlling the feature, and optional
                           trapping of accesses to DR4 and DR5.
    3     PSE              Page Size Extension. Large pages of size 4 MByte are supported, including CR4.PSE for controlling the
                           feature, the defined dirty bit in PDE (Page Directory Entries), optional reserved bit trapping in CR3, PDEs, and
                           PTEs.
    4     TSC              Time Stamp Counter. The RDTSC instruction is supported, including CR4.TSD for controlling privilege.
    5     MSR              Model Specific Registers RDMSR and WRMSR Instructions. The RDMSR and WRMSR instructions are
                           supported. Some of the MSRs are implementation dependent.
    6     PAE              Physical Address Extension. Physical addresses greater than 32 bits are supported: extended page table
                           entry formats, an extra level in the page translation tables is defined, 2-MByte pages are supported instead of
                           4 Mbyte pages if PAE bit is 1.
    7     MCE              Machine Check Exception. Exception 18 is defined for Machine Checks, including CR4.MCE for controlling the
                           feature. This feature does not define the model-specific implementations of machine-check error logging,
                           reporting, and processor shutdowns. Machine Check exception handlers may have to depend on processor
                           version to do model specific processing of the exception, or test for the presence of the Machine Check feature.
    8     CX8              CMPXCHG8B Instruction. The compare-and-exchange 8 bytes (64 bits) instruction is supported (implicitly
                           locked and atomic).
    9     APIC             APIC On-Chip. The processor contains an Advanced Programmable Interrupt Controller (APIC), responding to
                           memory mapped commands in the physical address range FFFE0000H to FFFE0FFFH (by default - some
                           processors permit the APIC to be relocated).
   10     Reserved         Reserved
   11     SEP              SYSENTER and SYSEXIT Instructions. The SYSENTER and SYSEXIT and associated MSRs are supported.
   12     MTRR             Memory Type Range Registers. MTRRs are supported. The MTRRcap MSR contains feature bits that describe
                           what memory types are supported, how many variable MTRRs are supported, and whether fixed MTRRs are
                           supported.
   13     PGE              Page Global Bit. The global bit is supported in paging-structure entries that map a page, indicating TLB entries
                           that are common to different processes and need not be flushed. The CR4.PGE bit controls this feature.
   14     MCA              Machine Check Architecture. A value of 1 indicates the Machine Check Architecture of reporting machine
                           errors is supported. The MCG_CAP MSR contains feature bits describing how many banks of error reporting
                           MSRs are supported.
   15     CMOV             Conditional Move Instructions. The conditional move instruction CMOV is supported. In addition, if x87 FPU is
                           present as indicated by the CPUID.FPU feature bit, then the FCOMI and FCMOV instructions are supported
   16     PAT              Page Attribute Table. Page Attribute Table is supported. This feature augments the Memory Type Range
                           Registers (MTRRs), allowing an operating system to specify attributes of memory accessed through a linear
                           address on a 4KB granularity.
   17     PSE-36           36-Bit Page Size Extension. 4-MByte pages addressing physical memory beyond 4 GBytes are supported with
                           32-bit paging. This feature indicates that upper bits of the physical address of a 4-MByte page are encoded in
                           bits 20:13 of the page-directory entry. Such physical addresses are limited by MAXPHYADDR and may be up to
                           40 bits in size.
   18     PSN              Processor Serial Number. The processor supports the 96-bit processor identification number feature and the
                           feature is enabled.
   19     CLFSH            CLFLUSH Instruction. CLFLUSH Instruction is supported.
   20     Reserved         Reserved



                    Table 3-11. More on Feature Information Returned in the EDX Register (Contd.)
 Bit #   Mnemonic    Description
    21   DS          Debug Store. The processor supports the ability to write debug information into a memory resident buffer.
                     This feature is used by the branch trace store (BTS) and precise event-based sampling (PEBS) facilities (see
                     Chapter 23, "Introduction to Virtual-Machine Extensions," in the Intel 64 and IA-32 Architectures Software
                     Developer''s Manual, Volume 3C).
    22   ACPI        Thermal Monitor and Software Controlled Clock Facilities. The processor implements internal MSRs that
                     allow processor temperature to be monitored and processor performance to be modulated in predefined duty
                     cycles under software control.
    23   MMX         Intel MMX Technology. The processor supports the Intel MMX technology.
    24   FXSR        FXSAVE and FXRSTOR Instructions. The FXSAVE and FXRSTOR instructions are supported for fast save and
                     restore of the floating point context. Presence of this bit also indicates that CR4.OSFXSR is available for an
                     operating system to indicate that it supports the FXSAVE and FXRSTOR instructions.
    25   SSE         SSE. The processor supports the SSE extensions.
    26   SSE2        SSE2. The processor supports the SSE2 extensions.
    27   SS          Self Snoop. The processor supports the management of conflicting memory types by performing a snoop of its
                     own cache structure for transactions issued to the bus.
    28   HTT         Max APIC IDs reserved field is Valid. A value of 0 for HTT indicates there is only a single logical processor in
                     the package and software should assume only a single APIC ID is reserved. A value of 1 for HTT indicates the
                     value in CPUID.1.EBX[23:16] (the Maximum number of addressable IDs for logical processors in this package) is
                     valid for the package.
    29   TM          Thermal Monitor. The processor implements the thermal monitor automatic thermal control circuitry (TCC).
    30   Reserved    Reserved
    31   PBE         Pending Break Enable. The processor supports the use of the FERR#/PBE# pin when the processor is in the
                     stop-clock state (STPCLK# is asserted) to signal the processor that an interrupt is pending and that the
                     processor should return to normal operation to handle the interrupt. Bit 10 (PBE enable) in the
                     IA32_MISC_ENABLE MSR enables this capability.



INPUT EAX = 02H: TLB/Cache/Prefetch Information Returned in EAX, EBX, ECX, EDX
When CPUID executes with EAX set to 02H, the processor returns information about the processor''s internal TLBs,
cache and prefetch hardware in the EAX, EBX, ECX, and EDX registers. The information is reported in encoded form
and fall into the following categories:
.    The least-significant byte in register EAX (register AL) will always return 01H. Software should ignore this value
     and not interpret it as an informational descriptor.
.    The most significant bit (bit 31) of each register indicates whether the register contains valid information (set
     to 0) or is reserved (set to 1).
.    If a register contains valid information, the information is contained in 1 byte descriptors. There are four types
     of encoding values for the byte descriptor, the encoding type is noted in the second column of Table 3-12. Table
     3-12 lists the encoding of these descriptors. Note that the order of descriptors in the EAX, EBX, ECX, and EDX
     registers is not defined; that is, specific bytes are not designated to contain descriptors for specific cache,
     prefetch, or TLB types. The descriptors may appear in any order. Note also a processor may report a general
     descriptor type (FFH) and not report any byte descriptor of "cache type" via CPUID leaf 2.





                                      Table 3-12. Encoding of CPUID Leaf 2 Descriptors
  Value      Type                                                           Description
  00H       General    Null descriptor, this byte contains no information
  01H         TLB      Instruction TLB: 4 KByte pages, 4-way set associative, 32 entries
  02H         TLB      Instruction TLB: 4 MByte pages, fully associative, 2 entries
  03H         TLB      Data TLB: 4 KByte pages, 4-way set associative, 64 entries
  04H         TLB      Data TLB: 4 MByte pages, 4-way set associative, 8 entries
  05H         TLB      Data TLB1: 4 MByte pages, 4-way set associative, 32 entries
  06H        Cache     1st-level instruction cache: 8 KBytes, 4-way set associative, 32 byte line size
  08H        Cache     1st-level instruction cache: 16 KBytes, 4-way set associative, 32 byte line size
  09H        Cache     1st-level instruction cache: 32KBytes, 4-way set associative, 64 byte line size
  0AH        Cache     1st-level data cache: 8 KBytes, 2-way set associative, 32 byte line size
  0BH         TLB      Instruction TLB: 4 MByte pages, 4-way set associative, 4 entries
  0CH        Cache     1st-level data cache: 16 KBytes, 4-way set associative, 32 byte line size
  0DH        Cache     1st-level data cache: 16 KBytes, 4-way set associative, 64 byte line size
  0EH        Cache     1st-level data cache: 24 KBytes, 6-way set associative, 64 byte line size
  1DH        Cache     2nd-level cache: 128 KBytes, 2-way set associative, 64 byte line size
  21H        Cache     2nd-level cache: 256 KBytes, 8-way set associative, 64 byte line size
  22H        Cache     3rd-level cache: 512 KBytes, 4-way set associative, 64 byte line size, 2 lines per sector
  23H        Cache     3rd-level cache: 1 MBytes, 8-way set associative, 64 byte line size, 2 lines per sector
  24H        Cache     2nd-level cache: 1 MBytes, 16-way set associative, 64 byte line size
  25H        Cache     3rd-level cache: 2 MBytes, 8-way set associative, 64 byte line size, 2 lines per sector
  29H        Cache     3rd-level cache: 4 MBytes, 8-way set associative, 64 byte line size, 2 lines per sector
  2CH        Cache     1st-level data cache: 32 KBytes, 8-way set associative, 64 byte line size
  30H        Cache     1st-level instruction cache: 32 KBytes, 8-way set associative, 64 byte line size
  40H        Cache     No 2nd-level cache or, if processor contains a valid 2nd-level cache, no 3rd-level cache
  41H        Cache     2nd-level cache: 128 KBytes, 4-way set associative, 32 byte line size
  42H        Cache     2nd-level cache: 256 KBytes, 4-way set associative, 32 byte line size
  43H        Cache     2nd-level cache: 512 KBytes, 4-way set associative, 32 byte line size
  44H        Cache     2nd-level cache: 1 MByte, 4-way set associative, 32 byte line size
  45H        Cache     2nd-level cache: 2 MByte, 4-way set associative, 32 byte line size
  46H        Cache     3rd-level cache: 4 MByte, 4-way set associative, 64 byte line size
  47H        Cache     3rd-level cache: 8 MByte, 8-way set associative, 64 byte line size
  48H        Cache     2nd-level cache: 3MByte, 12-way set associative, 64 byte line size
  49H        Cache     3rd-level cache: 4MB, 16-way set associative, 64-byte line size (Intel Xeon processor MP, Family 0FH, Model
                       06H);
                       2nd-level cache: 4 MByte, 16-way set associative, 64 byte line size
  4AH        Cache     3rd-level cache: 6MByte, 12-way set associative, 64 byte line size
  4BH        Cache     3rd-level cache: 8MByte, 16-way set associative, 64 byte line size
  4CH        Cache     3rd-level cache: 12MByte, 12-way set associative, 64 byte line size
  4DH        Cache     3rd-level cache: 16MByte, 16-way set associative, 64 byte line size
  4EH        Cache     2nd-level cache: 6MByte, 24-way set associative, 64 byte line size
  4FH         TLB      Instruction TLB: 4 KByte pages, 32 entries




                              Table 3-12. Encoding of CPUID Leaf 2 Descriptors (Contd.)
 Value      Type                                                         Description
  50H       TLB     Instruction TLB: 4 KByte and 2-MByte or 4-MByte pages, 64 entries
  51H       TLB     Instruction TLB: 4 KByte and 2-MByte or 4-MByte pages, 128 entries
  52H       TLB     Instruction TLB: 4 KByte and 2-MByte or 4-MByte pages, 256 entries
  55H       TLB     Instruction TLB: 2-MByte or 4-MByte pages, fully associative, 7 entries
  56H       TLB     Data TLB0: 4 MByte pages, 4-way set associative, 16 entries
  57H       TLB     Data TLB0: 4 KByte pages, 4-way associative, 16 entries
  59H       TLB     Data TLB0: 4 KByte pages, fully associative, 16 entries
  5AH       TLB     Data TLB0: 2 MByte or 4 MByte pages, 4-way set associative, 32 entries
  5BH       TLB     Data TLB: 4 KByte and 4 MByte pages, 64 entries
  5CH       TLB     Data TLB: 4 KByte and 4 MByte pages,128 entries
  5DH       TLB     Data TLB: 4 KByte and 4 MByte pages,256 entries
  60H       Cache   1st-level data cache: 16 KByte, 8-way set associative, 64 byte line size
  61H       TLB     Instruction TLB: 4 KByte pages, fully associative, 48 entries
  63H       TLB     Data TLB: 2 MByte or 4 MByte pages, 4-way set associative, 32 entries and a separate array with 1 GByte
                    pages, 4-way set associative, 4 entries
  64H       TLB     Data TLB: 4 KByte pages, 4-way set associative, 512 entries
  66H       Cache   1st-level data cache: 8 KByte, 4-way set associative, 64 byte line size
  67H       Cache   1st-level data cache: 16 KByte, 4-way set associative, 64 byte line size
  68H       Cache   1st-level data cache: 32 KByte, 4-way set associative, 64 byte line size
  6AH       Cache   uTLB: 4 KByte pages, 8-way set associative, 64 entries
  6BH       Cache   DTLB: 4 KByte pages, 8-way set associative, 256 entries
  6CH       Cache   DTLB: 2M/4M pages, 8-way set associative, 128 entries
  6DH       Cache   DTLB: 1 GByte pages, fully associative, 16 entries
  70H       Cache   Trace cache: 12 K-muop, 8-way set associative
  71H       Cache   Trace cache: 16 K-muop, 8-way set associative
  72H       Cache   Trace cache: 32 K-muop, 8-way set associative
  76H       TLB     Instruction TLB: 2M/4M pages, fully associative, 8 entries
  78H       Cache   2nd-level cache: 1 MByte, 4-way set associative, 64byte line size
  79H       Cache   2nd-level cache: 128 KByte, 8-way set associative, 64 byte line size, 2 lines per sector
  7AH       Cache   2nd-level cache: 256 KByte, 8-way set associative, 64 byte line size, 2 lines per sector
  7BH       Cache   2nd-level cache: 512 KByte, 8-way set associative, 64 byte line size, 2 lines per sector
  7CH       Cache   2nd-level cache: 1 MByte, 8-way set associative, 64 byte line size, 2 lines per sector
  7DH       Cache   2nd-level cache: 2 MByte, 8-way set associative, 64byte line size
  7FH       Cache   2nd-level cache: 512 KByte, 2-way set associative, 64-byte line size
  80H       Cache   2nd-level cache: 512 KByte, 8-way set associative, 64-byte line size
  82H       Cache   2nd-level cache: 256 KByte, 8-way set associative, 32 byte line size
  83H       Cache   2nd-level cache: 512 KByte, 8-way set associative, 32 byte line size
  84H       Cache   2nd-level cache: 1 MByte, 8-way set associative, 32 byte line size
  85H       Cache   2nd-level cache: 2 MByte, 8-way set associative, 32 byte line size
  86H       Cache   2nd-level cache: 512 KByte, 4-way set associative, 64 byte line size
  87H       Cache   2nd-level cache: 1 MByte, 8-way set associative, 64 byte line size




                                 Table 3-12. Encoding of CPUID Leaf 2 Descriptors (Contd.)
  Value      Type                                                        Description
  A0H        DTLB      DTLB: 4k pages, fully associative, 32 entries
  B0H         TLB      Instruction TLB: 4 KByte pages, 4-way set associative, 128 entries
  B1H         TLB      Instruction TLB: 2M pages, 4-way, 8 entries or 4M pages, 4-way, 4 entries
  B2H         TLB      Instruction TLB: 4KByte pages, 4-way set associative, 64 entries
  B3H         TLB      Data TLB: 4 KByte pages, 4-way set associative, 128 entries
  B4H         TLB      Data TLB1: 4 KByte pages, 4-way associative, 256 entries
  B5H         TLB      Instruction TLB: 4KByte pages, 8-way set associative, 64 entries
  B6H         TLB      Instruction TLB: 4KByte pages, 8-way set associative, 128 entries
  BAH         TLB      Data TLB1: 4 KByte pages, 4-way associative, 64 entries
  C0H         TLB      Data TLB: 4 KByte and 4 MByte pages, 4-way associative, 8 entries
  C1H        STLB      Shared 2nd-Level TLB: 4 KByte/2MByte pages, 8-way associative, 1024 entries
  C2H        DTLB      DTLB: 4 KByte/2 MByte pages, 4-way associative, 16 entries
  C3H        STLB      Shared 2nd-Level TLB: 4 KByte /2 MByte pages, 6-way associative, 1536 entries. Also 1GBbyte pages, 4-way,
                       16 entries.
  C4H        DTLB      DTLB: 2M/4M Byte pages, 4-way associative, 32 entries
  CAH        STLB      Shared 2nd-Level TLB: 4 KByte pages, 4-way associative, 512 entries
  D0H        Cache     3rd-level cache: 512 KByte, 4-way set associative, 64 byte line size
  D1H        Cache     3rd-level cache: 1 MByte, 4-way set associative, 64 byte line size
  D2H        Cache     3rd-level cache: 2 MByte, 4-way set associative, 64 byte line size
  D6H        Cache     3rd-level cache: 1 MByte, 8-way set associative, 64 byte line size
  D7H        Cache     3rd-level cache: 2 MByte, 8-way set associative, 64 byte line size
  D8H        Cache     3rd-level cache: 4 MByte, 8-way set associative, 64 byte line size
  DCH        Cache     3rd-level cache: 1.5 MByte, 12-way set associative, 64 byte line size
  DDH        Cache     3rd-level cache: 3 MByte, 12-way set associative, 64 byte line size
  DEH        Cache     3rd-level cache: 6 MByte, 12-way set associative, 64 byte line size
  E2H        Cache     3rd-level cache: 2 MByte, 16-way set associative, 64 byte line size
  E3H        Cache     3rd-level cache: 4 MByte, 16-way set associative, 64 byte line size
  E4H        Cache     3rd-level cache: 8 MByte, 16-way set associative, 64 byte line size
  EAH        Cache     3rd-level cache: 12MByte, 24-way set associative, 64 byte line size
  EBH        Cache     3rd-level cache: 18MByte, 24-way set associative, 64 byte line size
  ECH        Cache     3rd-level cache: 24MByte, 24-way set associative, 64 byte line size
  F0H      Prefetch    64-Byte prefetching
  F1H      Prefetch    128-Byte prefetching
  FFH       General    CPUID leaf 2 does not report cache descriptor information, use CPUID leaf 4 to query cache parameters


Example 3-1. Example of Cache and TLB Interpretation
The first member of the family of Pentium 4 processors returns the following information about caches and TLBs
when the CPUID executes with an input value of 2:

   EAX       66 5B 50 01H
   EBX       0H
   ECX       0H
   EDX       00 7A 70 00H




Which means:
.   The least-significant byte (byte 0) of register EAX is set to 01H. This value should be ignored.
.   The most-significant bit of all four registers (EAX, EBX, ECX, and EDX) is set to 0, indicating that each register
    contains valid 1-byte descriptors.
.   Bytes 1, 2, and 3 of register EAX indicate that the processor has:
    - 50H - a 64-entry instruction TLB, for mapping 4-KByte and 2-MByte or 4-MByte pages.
    - 5BH - a 64-entry data TLB, for mapping 4-KByte and 4-MByte pages.
    - 66H - an 8-KByte 1st level data cache, 4-way set associative, with a 64-Byte cache line size.
.   The descriptors in registers EBX and ECX are valid, but contain NULL descriptors.
.   Bytes 0, 1, 2, and 3 of register EDX indicate that the processor has:
    - 00H - NULL descriptor.
    - 70H - Trace cache: 12 K-muop, 8-way set associative.
    - 7AH - a 256-KByte 2nd level cache, 8-way set associative, with a sectored, 64-byte cache line size.
    - 00H - NULL descriptor.

INPUT EAX = 04H: Returns Deterministic Cache Parameters for Each Level
When CPUID executes with EAX set to 04H and ECX contains an index value, the processor returns encoded data
that describe a set of deterministic cache parameters (for the cache level associated with the input in ECX). Valid
index values start from 0.
Software can enumerate the deterministic cache parameters for each level of the cache hierarchy starting with an
index value of 0, until the parameters report the value associated with the cache type field is 0. The architecturally
defined fields reported by deterministic cache parameters are documented in Table 3-8.
This Cache Size in Bytes
= (Ways + 1) * (Partitions + 1) * (Line_Size + 1) * (Sets + 1)
= (EBX[31:22] + 1) * (EBX[21:12] + 1) * (EBX[11:0] + 1) * (ECX + 1)


The CPUID leaf 04H also reports data that can be used to derive the topology of processor cores in a physical
package. This information is constant for all valid index values. Software can query the raw data reported by
executing CPUID with EAX=04H and ECX=0 and use it as part of the topology enumeration algorithm described in
Chapter 8, "Multiple-Processor Management," in the Intel 64 and IA-32 Architectures Software Developer''s
Manual, Volume 3A.

INPUT EAX = 05H: Returns MONITOR and MWAIT Features
When CPUID executes with EAX set to 05H, the processor returns information about features available to
MONITOR/MWAIT instructions. The MONITOR instruction is used for address-range monitoring in conjunction with
MWAIT instruction. The MWAIT instruction optionally provides additional extensions for advanced power manage-
ment. See Table 3-8.

INPUT EAX = 06H: Returns Thermal and Power Management Features
When CPUID executes with EAX set to 06H, the processor returns information about thermal and power manage-
ment features. See Table 3-8.

INPUT EAX = 07H: Returns Structured Extended Feature Enumeration Information
When CPUID executes with EAX set to 07H and ECX = 0, the processor returns information about the maximum
input value for sub-leaves that contain extended feature flags. See Table 3-8.
When CPUID executes with EAX set to 07H and the input value of ECX is invalid (see leaf 07H entry in Table 3-8),
the processor returns 0 in EAX/EBX/ECX/EDX. In subleaf 0, EAX returns the maximum input value of the highest
leaf 7 sub-leaf, and EBX, ECX & EDX contain information of extended feature flags.




INPUT EAX = 09H: Returns Direct Cache Access Information
When CPUID executes with EAX set to 09H, the processor returns information about Direct Cache Access capabili-
ties. See Table 3-8.

INPUT EAX = 0AH: Returns Architectural Performance Monitoring Features
When CPUID executes with EAX set to 0AH, the processor returns information about support for architectural
performance monitoring capabilities. Architectural performance monitoring is supported if the version ID (see
Table 3-8) is greater than Pn 0. See Table 3-8.
For each version of architectural performance monitoring capability, software must enumerate this leaf to discover
the programming facilities and the architectural performance events available in the processor. The details are
described in Chapter 23, "Introduction to Virtual-Machine Extensions," in the Intel 64 and IA-32 Architectures
Software Developer''s Manual, Volume 3C.

INPUT EAX = 0BH: Returns Extended Topology Information
When CPUID executes with EAX set to 0BH, the processor returns information about extended topology enumera-
tion data. Software must detect the presence of CPUID leaf 0BH by verifying (a) the highest leaf index supported
by CPUID is >= 0BH, and (b) CPUID.0BH:EBX[15:0] reports a non-zero value. See Table 3-8.

INPUT EAX = 0DH: Returns Processor Extended States Enumeration Information
When CPUID executes with EAX set to 0DH and ECX = 0, the processor returns information about the bit-vector
representation of all processor state extensions that are supported in the processor and storage size requirements
of the XSAVE/XRSTOR area. See Table 3-8.
When CPUID executes with EAX set to 0DH and ECX = n (n > 1, and is a valid sub-leaf index), the processor returns
information about the size and offset of each processor extended state save area within the XSAVE/XRSTOR area.
See Table 3-8. Software can use the forward-extendable technique depicted below to query the valid sub-leaves
and obtain size and offset information for each processor extended state save area:

For i = 2 to 62 // sub-leaf 1 is reserved
   IF (CPUID.(EAX=0DH, ECX=0):VECTOR[i] = 1 ) // VECTOR is the 64-bit value of EDX:EAX
         Execute CPUID.(EAX=0DH, ECX = i) to examine size and offset for sub-leaf i;
   FI;

INPUT EAX = 0FH: Returns Intel Resource Director Technology (Intel RDT) Monitoring Enumeration Information
When CPUID executes with EAX set to 0FH and ECX = 0, the processor returns information about the bit-vector
representation of QoS monitoring resource types that are supported in the processor and maximum range of RMID
values the processor can use to monitor of any supported resource types. Each bit, starting from bit 1, corresponds
to a specific resource type if the bit is set. The bit position corresponds to the sub-leaf index (or ResID) that soft-
ware must use to query QoS monitoring capability available for that type. See Table 3-8.
When CPUID executes with EAX set to 0FH and ECX = n (n >= 1, and is a valid ResID), the processor returns infor-
mation software can use to program IA32_PQR_ASSOC, IA32_QM_EVTSEL MSRs before reading QoS data from the
IA32_QM_CTR MSR.

INPUT EAX = 10H: Returns Intel Resource Director Technology (Intel RDT) Allocation Enumeration Information
When CPUID executes with EAX set to 10H and ECX = 0, the processor returns information about the bit-vector
representation of QoS Enforcement resource types that are supported in the processor. Each bit, starting from bit
1, corresponds to a specific resource type if the bit is set. The bit position corresponds to the sub-leaf index (or
ResID) that software must use to query QoS enforcement capability available for that type. See Table 3-8.
When CPUID executes with EAX set to 10H and ECX = n (n >= 1, and is a valid ResID), the processor returns infor-
mation about available classes of service and range of QoS mask MSRs that software can use to configure each
class of services using capability bit masks in the QoS Mask registers, IA32_resourceType_Mask_n.





INPUT EAX = 12H: Returns Intel SGX Enumeration Information
When CPUID executes with EAX set to 12H and ECX = 0H, the processor returns information about Intel SGX capa-
bilities. See Table 3-8.
When CPUID executes with EAX set to 12H and ECX = 1H, the processor returns information about Intel SGX attri-
butes. See Table 3-8.
When CPUID executes with EAX set to 12H and ECX = n (n > 1), the processor returns information about Intel SGX
Enclave Page Cache. See Table 3-8.

INPUT EAX = 14H: Returns Intel Processor Trace Enumeration Information
When CPUID executes with EAX set to 14H and ECX = 0H, the processor returns information about Intel Processor
Trace extensions. See Table 3-8.
When CPUID executes with EAX set to 14H and ECX = n (n > 0 and less than the number of non-zero bits in
CPUID.(EAX=14H, ECX= 0H).EAX), the processor returns information about packet generation in Intel Processor
Trace. See Table 3-8.

INPUT EAX = 15H: Returns Time Stamp Counter and Nominal Core Crystal Clock Information
When CPUID executes with EAX set to 15H and ECX = 0H, the processor returns information about Time Stamp
Counter and Core Crystal Clock. See Table 3-8.

INPUT EAX = 16H: Returns Processor Frequency Information
When CPUID executes with EAX set to 16H, the processor returns information about Processor Frequency Informa-
tion. See Table 3-8.

INPUT EAX = 17H: Returns System-On-Chip Information
When CPUID executes with EAX set to 17H, the processor returns information about the System-On-Chip Vendor
Attribute Enumeration. See Table 3-8.

METHODS FOR RETURNING BRANDING INFORMATION
Use the following techniques to access branding information:
1. Processor brand string method.
2. Processor brand index; this method uses a software supplied brand string table.
These two methods are discussed in the following sections. For methods that are available in early processors, see
Section: "Identification of Earlier IA-32 Processors" in Chapter 19 of the Intel 64 and IA-32 Architectures Soft-
ware Developer''s Manual, Volume 1.

The Processor Brand String Method
Figure 3-9 describes the algorithm used for detection of the brand string. Processor brand identification software
should execute this algorithm on all Intel 64 and IA-32 processors.
This method (introduced with Pentium 4 processors) returns an ASCII brand identification string and the Processor
Base frequency of the processor to the EAX, EBX, ECX, and EDX registers.





                                Input: EAX=
                                0x80000000

                             CPUID


                                                          False          Processor Brand
                           IF (EAX & 0x80000000)                            String Not
                                                                            Supported


                              CPUID
                                       True >=
                            Function
                                       Extended
                           Supported

                            EAX Return Value =
                           Max. Extended CPUID
                              Function Index




                                                          True           Processor Brand
                            IF (EAX Return Value
                               >= 0x80000004)                             String Supported


                                                                                        OM15194



                           Figure 3-9. Determination of Support for the Processor Brand String


How Brand Strings Work
To use the brand string method, execute CPUID with EAX input of 8000002H through 80000004H. For each input
value, CPUID returns 16 ASCII characters using EAX, EBX, ECX, and EDX. The returned string will be NULL-termi-
nated.





Table 3-13 shows the brand string that is returned by the first processor in the Pentium 4 processor family.

                      Table 3-13. Processor Brand String Returned with Pentium 4 Processor
         EAX Input Value                               Return Values                                 ASCII Equivalent
 80000002H                         EAX = 20202020H                                     "   "
                                   EBX = 20202020H                                     " "
                                   ECX = 20202020H                                     " "
                                   EDX = 6E492020H                                     "nI "
 80000003H                         EAX = 286C6574H                                     "(let"
                                   EBX = 50202952H                                     "P )R"
                                   ECX = 69746E65H                                     "itne"
                                   EDX = 52286D75H                                     "R(mu"
 80000004H                         EAX = 20342029H                                     " 4 )"
                                   EBX = 20555043H                                     " UPC"
                                   ECX = 30303531H                                     "0051"
                                   EDX = 007A484DH                                     "\0zHM"



Extracting the Processor Frequency from Brand Strings
Figure 3-10 provides an algorithm which software can use to extract the Processor Base frequency from the
processor brand string.


                              Scan "Brand String" in
                               Reverse Byte Order

                              "zHM", or
                                           Match
                              "zHG", or
                                           Substring
                                 "zHT"


                                                              False
                              IF Substring Matched                         Report Error




                       Determine "Freq"    True                If "zHM"
                        and "Multiplier"                                   Multiplier = 1 x 106

                                                               If "zHG"
                                                                           Multiplier = 1 x 109
                              Determine "Multiplier"           If "zHT"
                                                                           Multiplier = 1 x 1012


                                                          Scan Digits
                                                          Until Blank        Reverse Digits
                                 Determine "Freq"
                                                        In Reverse Order    To Decimal Value




                                 Processor Base
                                  Frequency =
                                                                             "Freq" = X.YZ if
                               "Freq" x "Multiplier"
                                                                             Digits = "ZY.X"

                                                                                                   OM15195


                              Figure 3-10. Algorithm for Extracting Processor Frequency




The Processor Brand Index Method
The brand index method (introduced with Pentium III Xeon processors) provides an entry point into a brand
identification table that is maintained in memory by system software and is accessible from system- and user-level
code. In this table, each brand index is associate with an ASCII brand identification string that identifies the official
Intel family and model number of a processor.
When CPUID executes with EAX set to 1, the processor returns a brand index to the low byte in EBX. Software can
then use this index to locate the brand identification string for the processor in the brand identification table. The
first entry (brand index 0) in this table is reserved, allowing for backward compatibility with processors that do not
support the brand identification feature. Starting with processor signature family ID = 0FH, model = 03H, brand
index method is no longer supported. Use brand string method instead.
Table 3-14 shows brand indices that have identification strings associated with them.
                  Table 3-14. Mapping of Brand Indices; and Intel 64 and IA-32 Processor Brand Strings
    Brand Index                                                        Brand String
        00H           This processor does not support the brand identification feature
        01H           Intel(R) Celeron(R) processor1
        02H           Intel(R) Pentium(R) III processor1
        03H           Intel(R) Pentium(R) III Xeon(R) processor; If processor signature = 000006B1h, then Intel(R) Celeron(R)
                      processor
        04H           Intel(R) Pentium(R) III processor
        06H           Mobile Intel(R) Pentium(R) III processor-M
        07H           Mobile Intel(R) Celeron(R) processor1
        08H           Intel(R) Pentium(R) 4 processor
        09H           Intel(R) Pentium(R) 4 processor
        0AH           Intel(R) Celeron(R) processor1
        0BH           Intel(R) Xeon(R) processor; If processor signature = 00000F13h, then Intel(R) Xeon(R) processor MP
        0CH           Intel(R) Xeon(R) processor MP
        0EH           Mobile Intel(R) Pentium(R) 4 processor-M; If processor signature = 00000F13h, then Intel(R) Xeon(R) processor
        0FH           Mobile Intel(R) Celeron(R) processor1
        11H           Mobile Genuine Intel(R) processor
        12H           Intel(R) Celeron(R) M processor
        13H           Mobile Intel(R) Celeron(R) processor1
        14H           Intel(R) Celeron(R) processor
        15H           Mobile Genuine Intel(R) processor
        16H           Intel(R) Pentium(R) M processor
        17H           Mobile Intel(R) Celeron(R) processor1
    18H - 0FFH        RESERVED
 NOTES:
 1. Indicates versions of these processors that were introduced after the Pentium III

IA-32 Architecture Compatibility
CPUID is not supported in early models of the Intel486 processor or in any IA-32 processor earlier than the
Intel486 processor.





Operation
IA32_BIOS_SIGN_ID MSR <- Update with installed microcode revision number;

CASE (EAX) OF
   EAX = 0:
       EAX <- Highest basic function input value understood by CPUID;
       EBX <- Vendor identification string;
       EDX <- Vendor identification string;
       ECX <- Vendor identification string;
   BREAK;
   EAX = 1H:
       EAX[3:0] <- Stepping ID;
       EAX[7:4] <- Model;
       EAX[11:8] <- Family;
       EAX[13:12] <- Processor type;
       EAX[15:14] <- Reserved;
       EAX[19:16] <- Extended Model;
       EAX[27:20] <- Extended Family;
       EAX[31:28] <- Reserved;
       EBX[7:0] <- Brand Index; (* Reserved if the value is zero. *)
       EBX[15:8] <- CLFLUSH Line Size;
       EBX[16:23] <- Reserved; (* Number of threads enabled = 2 if MT enable fuse set. *)
       EBX[24:31] <- Initial APIC ID;
       ECX <- Feature flags; (* See Figure 3-7. *)
       EDX <- Feature flags; (* See Figure 3-8. *)
   BREAK;
   EAX = 2H:
       EAX <- Cache and TLB information;
       EBX <- Cache and TLB information;
       ECX <- Cache and TLB information;
       EDX <- Cache and TLB information;
   BREAK;
   EAX = 3H:
       EAX <- Reserved;
       EBX <- Reserved;
       ECX <- ProcessorSerialNumber[31:0];
       (* Pentium III processors only, otherwise reserved. *)
       EDX <- ProcessorSerialNumber[63:32];
       (* Pentium III processors only, otherwise reserved. *
   BREAK
   EAX = 4H:
       EAX <- Deterministic Cache Parameters Leaf; (* See Table 3-8. *)
       EBX <- Deterministic Cache Parameters Leaf;
       ECX <- Deterministic Cache Parameters Leaf;
       EDX <- Deterministic Cache Parameters Leaf;
   BREAK;
   EAX = 5H:
       EAX <- MONITOR/MWAIT Leaf; (* See Table 3-8. *)
       EBX <- MONITOR/MWAIT Leaf;
       ECX <- MONITOR/MWAIT Leaf;
       EDX <- MONITOR/MWAIT Leaf;
   BREAK;





   EAX = 6H:
       EAX <- Thermal and Power Management Leaf; (* See Table 3-8. *)
       EBX <- Thermal and Power Management Leaf;
       ECX <- Thermal and Power Management Leaf;
       EDX <- Thermal and Power Management Leaf;
   BREAK;
   EAX = 7H:
       EAX <- Structured Extended Feature Flags Enumeration Leaf; (* See Table 3-8. *)
       EBX <- Structured Extended Feature Flags Enumeration Leaf;
       ECX <- Structured Extended Feature Flags Enumeration Leaf;
       EDX <- Structured Extended Feature Flags Enumeration Leaf;
   BREAK;
   EAX = 8H:
       EAX <- Reserved = 0;
       EBX <- Reserved = 0;
       ECX <- Reserved = 0;
       EDX <- Reserved = 0;
   BREAK;
   EAX = 9H:
       EAX <- Direct Cache Access Information Leaf; (* See Table 3-8. *)
       EBX <- Direct Cache Access Information Leaf;
       ECX <- Direct Cache Access Information Leaf;
       EDX <- Direct Cache Access Information Leaf;
   BREAK;
   EAX = AH:
       EAX <- Architectural Performance Monitoring Leaf; (* See Table 3-8. *)
       EBX <- Architectural Performance Monitoring Leaf;
       ECX <- Architectural Performance Monitoring Leaf;
       EDX <- Architectural Performance Monitoring Leaf;
       BREAK
   EAX = BH:
       EAX <- Extended Topology Enumeration Leaf; (* See Table 3-8. *)
       EBX <- Extended Topology Enumeration Leaf;
       ECX <- Extended Topology Enumeration Leaf;
       EDX <- Extended Topology Enumeration Leaf;
   BREAK;
   EAX = CH:
       EAX <- Reserved = 0;
       EBX <- Reserved = 0;
       ECX <- Reserved = 0;
       EDX <- Reserved = 0;
   BREAK;
   EAX = DH:
       EAX <- Processor Extended State Enumeration Leaf; (* See Table 3-8. *)
       EBX <- Processor Extended State Enumeration Leaf;
       ECX <- Processor Extended State Enumeration Leaf;
       EDX <- Processor Extended State Enumeration Leaf;
   BREAK;
   EAX = EH:
       EAX <- Reserved = 0;
       EBX <- Reserved = 0;
       ECX <- Reserved = 0;
       EDX <- Reserved = 0;
   BREAK;



   EAX = FH:
       EAX <- Intel Resource Director Technology Monitoring Enumeration Leaf; (* See Table 3-8. *)
       EBX <- Intel Resource Director Technology Monitoring Enumeration Leaf;
       ECX <- Intel Resource Director Technology Monitoring Enumeration Leaf;
       EDX <- Intel Resource Director Technology Monitoring Enumeration Leaf;
   BREAK;
   EAX = 10H:
       EAX <- Intel Resource Director Technology Allocation Enumeration Leaf; (* See Table 3-8. *)
       EBX <- Intel Resource Director Technology Allocation Enumeration Leaf;
       ECX <- Intel Resource Director Technology Allocation Enumeration Leaf;
       EDX <- Intel Resource Director Technology Allocation Enumeration Leaf;
   BREAK;
       EAX = 12H:
       EAX <- Intel SGX Enumeration Leaf; (* See Table 3-8. *)
       EBX <- Intel SGX Enumeration Leaf;
       ECX <- Intel SGX Enumeration Leaf;
       EDX <- Intel SGX Enumeration Leaf;
   BREAK;
   EAX = 14H:
       EAX <- Intel Processor Trace Enumeration Leaf; (* See Table 3-8. *)
       EBX <- Intel Processor Trace Enumeration Leaf;
       ECX <- Intel Processor Trace Enumeration Leaf;
       EDX <- Intel Processor Trace Enumeration Leaf;
   BREAK;
   EAX = 15H:
       EAX <- Time Stamp Counter and Nominal Core Crystal Clock Information Leaf; (* See Table 3-8. *)
       EBX <- Time Stamp Counter and Nominal Core Crystal Clock Information Leaf;
       ECX <- Time Stamp Counter and Nominal Core Crystal Clock Information Leaf;
       EDX <- Time Stamp Counter and Nominal Core Crystal Clock Information Leaf;
   BREAK;
   EAX = 16H:
       EAX <- Processor Frequency Information Enumeration Leaf; (* See Table 3-8. *)
       EBX <- Processor Frequency Information Enumeration Leaf;
       ECX <- Processor Frequency Information Enumeration Leaf;
       EDX <- Processor Frequency Information Enumeration Leaf;
   BREAK;
   EAX = 17H:
       EAX <- System-On-Chip Vendor Attribute Enumeration Leaf; (* See Table 3-8. *)
       EBX <- System-On-Chip Vendor Attribute Enumeration Leaf;
       ECX <- System-On-Chip Vendor Attribute Enumeration Leaf;
       EDX <- System-On-Chip Vendor Attribute Enumeration Leaf;
   BREAK;
   EAX = 80000000H:
       EAX <- Highest extended function input value understood by CPUID;
       EBX <- Reserved;
       ECX <- Reserved;
       EDX <- Reserved;
   BREAK;
   EAX = 80000001H:
       EAX <- Reserved;
       EBX <- Reserved;
       ECX <- Extended Feature Bits (* See Table 3-8.*);
       EDX <- Extended Feature Bits (* See Table 3-8. *);
   BREAK;



   EAX = 80000002H:
       EAX <- Processor Brand String;
       EBX <- Processor Brand String, continued;
       ECX <- Processor Brand String, continued;
       EDX <- Processor Brand String, continued;
   BREAK;
   EAX = 80000003H:
       EAX <- Processor Brand String, continued;
       EBX <- Processor Brand String, continued;
       ECX <- Processor Brand String, continued;
       EDX <- Processor Brand String, continued;
   BREAK;
   EAX = 80000004H:
       EAX <- Processor Brand String, continued;
       EBX <- Processor Brand String, continued;
       ECX <- Processor Brand String, continued;
       EDX <- Processor Brand String, continued;
   BREAK;
   EAX = 80000005H:
       EAX <- Reserved = 0;
       EBX <- Reserved = 0;
       ECX <- Reserved = 0;
       EDX <- Reserved = 0;
   BREAK;
   EAX = 80000006H:
       EAX <- Reserved = 0;
       EBX <- Reserved = 0;
       ECX <- Cache information;
       EDX <- Reserved = 0;
   BREAK;
   EAX = 80000007H:
       EAX <- Reserved = 0;
       EBX <- Reserved = 0;
       ECX <- Reserved = 0;
       EDX <- Reserved = Misc Feature Flags;
   BREAK;
   EAX = 80000008H:
       EAX <- Reserved = Physical Address Size Information;
       EBX <- Reserved = Virtual Address Size Information;
       ECX <- Reserved = 0;
       EDX <- Reserved = 0;
   BREAK;
   EAX >= 40000000H and EAX <= 4FFFFFFFH:
   DEFAULT: (* EAX = Value outside of recognized range for CPUID. *)
       (* If the highest basic information leaf data depend on ECX input value, ECX is honored.*)
       EAX <- Reserved; (* Information returned for highest basic information leaf. *)
       EBX <- Reserved; (* Information returned for highest basic information leaf. *)
       ECX <- Reserved; (* Information returned for highest basic information leaf. *)
       EDX <- Reserved; (* Information returned for highest basic information leaf. *)
   BREAK;
ESAC;

Flags Affected
None.



Exceptions (All Operating Modes)
#UD                 If the LOCK prefix is used.
                    In earlier IA-32 processors that do not support the CPUID instruction, execution of the instruc-
                    tion results in an invalid opcode (#UD) exception being generated.




');
INSERT INTO "instructions" VALUES('x86_64','CRC32','CRC32 - Accumulate CRC32 Value
Opcode/                                                 Op/     64-Bit     Compat/ Description
Instruction                                             En      Mode       Leg Mode
F2 0F 38 F0 /r                                          RM      Valid      Valid        Accumulate CRC32 on r/m8.
CRC32 r32, r/m8
F2 REX 0F 38 F0 /r                                      RM      Valid      N.E.         Accumulate CRC32 on r/m8.
CRC32 r32, r/m8*
F2 0F 38 F1 /r                                          RM      Valid      Valid        Accumulate CRC32 on r/m16.
CRC32 r32, r/m16
F2 0F 38 F1 /r                                          RM      Valid      Valid        Accumulate CRC32 on r/m32.
CRC32 r32, r/m32
F2 REX.W 0F 38 F0 /r                                    RM      Valid      N.E.         Accumulate CRC32 on r/m8.
CRC32 r64, r/m8
F2 REX.W 0F 38 F1 /r                                    RM      Valid      N.E.         Accumulate CRC32 on r/m64.
CRC32 r64, r/m64
NOTES:
*In 64-bit mode, r/m8 can not be encoded to access the following byte registers if a REX prefix is used: AH, BH, CH, DH.



                                                 Instruction Operand Encoding
  Op/En              Operand 1                      Operand 2                      Operand 3                      Operand 4
    RM            ModRM:reg (r, w)                ModRM:r/m (r)                       NA                             NA

Description
Starting with an initial value in the first operand (destination operand), accumulates a CRC32 (polynomial
11EDC6F41H) value for the second operand (source operand) and stores the result in the destination operand. The
source operand can be a register or a memory location. The destination operand must be an r32 or r64 register. If
the destination is an r64 register, then the 32-bit result is stored in the least significant double word and
00000000H is stored in the most significant double word of the r64 register.
The initial value supplied in the destination operand is a double word integer stored in the r32 register or the least
significant double word of the r64 register. To incrementally accumulate a CRC32 value, software retains the result
of the previous CRC32 operation in the destination operand, then executes the CRC32 instruction again with new
input data in the source operand. Data contained in the source operand is processed in reflected bit order. This
means that the most significant bit of the source operand is treated as the least significant bit of the quotient, and
so on, for all the bits of the source operand. Likewise, the result of the CRC operation is stored in the destination
operand in reflected bit order. This means that the most significant bit of the resulting CRC (bit 31) is stored in the
least significant bit of the destination operand (bit 0), and so on, for all the bits of the CRC.

Operation

Notes:
   BIT_REFLECT64: DST[63-0] = SRC[0-63]
   BIT_REFLECT32: DST[31-0] = SRC[0-31]
   BIT_REFLECT16: DST[15-0] = SRC[0-15]
   BIT_REFLECT8: DST[7-0] = SRC[0-7]
   MOD2: Remainder from Polynomial division modulus 2





CRC32 instruction for 64-bit source operand and 64-bit destination operand:

   TEMP1[63-0] <- BIT_REFLECT64 (SRC[63-0])
   TEMP2[31-0] <- BIT_REFLECT32 (DEST[31-0])
   TEMP3[95-0] <- TEMP1[63-0] << 32
   TEMP4[95-0] <- TEMP2[31-0] << 64
   TEMP5[95-0] <- TEMP3[95-0] XOR TEMP4[95-0]
   TEMP6[31-0] <- TEMP5[95-0] MOD2 11EDC6F41H
   DEST[31-0] <- BIT_REFLECT (TEMP6[31-0])
   DEST[63-32] <- 00000000H
CRC32 instruction for 32-bit source operand and 32-bit destination operand:

   TEMP1[31-0] <- BIT_REFLECT32 (SRC[31-0])
   TEMP2[31-0] <- BIT_REFLECT32 (DEST[31-0])
   TEMP3[63-0] <- TEMP1[31-0] << 32
   TEMP4[63-0] <- TEMP2[31-0] << 32
   TEMP5[63-0] <- TEMP3[63-0] XOR TEMP4[63-0]
   TEMP6[31-0] <- TEMP5[63-0] MOD2 11EDC6F41H
   DEST[31-0] <- BIT_REFLECT (TEMP6[31-0])
CRC32 instruction for 16-bit source operand and 32-bit destination operand:

   TEMP1[15-0] <- BIT_REFLECT16 (SRC[15-0])
   TEMP2[31-0] <- BIT_REFLECT32 (DEST[31-0])
   TEMP3[47-0] <- TEMP1[15-0] << 32
   TEMP4[47-0] <- TEMP2[31-0] << 16
   TEMP5[47-0] <- TEMP3[47-0] XOR TEMP4[47-0]
   TEMP6[31-0] <- TEMP5[47-0] MOD2 11EDC6F41H
   DEST[31-0] <- BIT_REFLECT (TEMP6[31-0])
CRC32 instruction for 8-bit source operand and 64-bit destination operand:

   TEMP1[7-0] <- BIT_REFLECT8(SRC[7-0])
   TEMP2[31-0] <- BIT_REFLECT32 (DEST[31-0])
   TEMP3[39-0] <- TEMP1[7-0] << 32
   TEMP4[39-0] <- TEMP2[31-0] << 8
   TEMP5[39-0] <- TEMP3[39-0] XOR TEMP4[39-0]
   TEMP6[31-0] <- TEMP5[39-0] MOD2 11EDC6F41H
   DEST[31-0] <- BIT_REFLECT (TEMP6[31-0])
   DEST[63-32] <- 00000000H
CRC32 instruction for 8-bit source operand and 32-bit destination operand:

   TEMP1[7-0] <- BIT_REFLECT8(SRC[7-0])
   TEMP2[31-0] <- BIT_REFLECT32 (DEST[31-0])
   TEMP3[39-0] <- TEMP1[7-0] << 32
   TEMP4[39-0] <- TEMP2[31-0] << 8
   TEMP5[39-0] <- TEMP3[39-0] XOR TEMP4[39-0]
   TEMP6[31-0] <- TEMP5[39-0] MOD2 11EDC6F41H
   DEST[31-0] <- BIT_REFLECT (TEMP6[31-0])

Flags Affected
None





Intel C/C++ Compiler Intrinsic Equivalent
unsigned int _mm_crc32_u8( unsigned int crc, unsigned char data )
unsigned int _mm_crc32_u16( unsigned int crc, unsigned short data )
unsigned int _mm_crc32_u32( unsigned int crc, unsigned int data )
unsinged __int64 _mm_crc32_u64( unsinged __int64 crc, unsigned __int64 data )

SIMD Floating Point Exceptions
None

Protected Mode Exceptions
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS or GS segments.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#PF (fault-code)       For a page fault.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                    If CPUID.01H:ECX.SSE4_2 [Bit 20] = 0.
                       If LOCK prefix is used.

Real-Address Mode Exceptions
#GP(0)                 If any part of the operand lies outside of the effective address space from 0 to 0FFFFH.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#UD                    If CPUID.01H:ECX.SSE4_2 [Bit 20] = 0.
                       If LOCK prefix is used.

Virtual 8086 Mode Exceptions
#GP(0)                 If any part of the operand lies outside of the effective address space from 0 to 0FFFFH.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#PF (fault-code)       For a page fault.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made.
#UD                    If CPUID.01H:ECX.SSE4_2 [Bit 20] = 0.
                       If LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in Protected Mode.

64-Bit Mode Exceptions
#GP(0)                 If the memory address is in a non-canonical form.
#SS(0)                 If a memory address referencing the SS segment is in a non-canonical form.
#PF (fault-code)       For a page fault.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                    If CPUID.01H:ECX.SSE4_2 [Bit 20] = 0.
                       If LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','CVTDQ2PD','CVTDQ2PD-Convert Packed Doubleword Integers to Packed Double-Precision Floating-Point
Values
 Opcode/                            Op /    64/32         CPUID          Description
 Instruction                        En      bit Mode      Feature
                                            Support       Flag
 F3 0F E6 /r                        RM      V/V           SSE2           Convert two packed signed doubleword integers from
 CVTDQ2PD xmm1, xmm2/m64                                                 xmm2/mem to two packed double-precision floating-
                                                                         point values in xmm1.
 VEX.128.F3.0F.WIG E6 /r            RM      V/V           AVX            Convert two packed signed doubleword integers from
 VCVTDQ2PD xmm1, xmm2/m64                                                xmm2/mem to two packed double-precision floating-
                                                                         point values in xmm1.
 VEX.256.F3.0F.WIG E6 /r            RM      V/V           AVX            Convert four packed signed doubleword integers from
 VCVTDQ2PD ymm1, xmm2/m128                                               xmm2/mem to four packed double-precision floating-
                                                                         point values in ymm1.
 EVEX.128.F3.0F.W0 E6 /r            HV      V/V           AVX512VL       Convert 2 packed signed doubleword integers from
 VCVTDQ2PD xmm1 {k1}{z},                                  AVX512F        xmm2/m128/m32bcst to eight packed double-precision
 xmm2/m128/m32bcst                                                       floating-point values in xmm1 with writemask k1.
 EVEX.256.F3.0F.W0 E6 /r            HV      V/V           AVX512VL       Convert 4 packed signed doubleword integers from
 VCVTDQ2PD ymm1 {k1}{z},                                  AVX512F        xmm2/m128/m32bcst to 4 packed double-precision
 xmm2/m128/m32bcst                                                       floating-point values in ymm1 with writemask k1.
 EVEX.512.F3.0F.W0 E6 /r            HV      V/V           AVX512F        Convert eight packed signed doubleword integers from
 VCVTDQ2PD zmm1 {k1}{z},                                                 ymm2/m256/m32bcst to eight packed double-precision
 ymm2/m256/m32bcst                                                       floating-point values in zmm1 with writemask k1.



                                             Instruction Operand Encoding
     Op/En            Operand 1                   Operand 2                     Operand 3                      Operand 4
       RM           ModRM:reg (w)               ModRM:r/m (r)                       NA                             NA
       HV           ModRM:reg (w)               ModRM:r/m (r)                       NA                             NA

Description
Converts two, four or eight packed signed doubleword integers in the source operand (the second operand) to two,
four or eight packed double-precision floating-point values in the destination operand (the first operand).
EVEX encoded versions: The source operand can be a YMM/XMM/XMM (low 64 bits) register, a 256/128/64-bit
memory location or a 256/128/64-bit vector broadcasted from a 32-bit memory location. The destination operand
is a ZMM/YMM/XMM register conditionally updated with writemask k1. Attempt to encode this instruction with EVEX
embedded rounding is ignored.
VEX.256 encoded version: The source operand is an XMM register or 128- bit memory location. The destination
operand is a YMM register.
VEX.128 encoded version: The source operand is an XMM register or 64- bit memory location. The destination
operand is a XMM register. The upper Bits (MAX_VL-1:128) of the corresponding ZMM register destination are
zeroed.
128-bit Legacy SSE version: The source operand is an XMM register or 64- bit memory location. The destination
operand is an XMM register. The upper Bits (MAX_VL-1:128) of the corresponding ZMM register destination are
unmodified.
VEX.vvvv and EVEX.vvvv are reserved and must be 1111b, otherwise instructions will #UD.





                            SRC                                              X3         X2     X1        X0




                            DEST        X3                     X2                  X1               X0



                                      Figure 3-11. CVTDQ2PD (VEX.256 encoded version)


Operation
VCVTDQ2PD (EVEX encoded versions) when src operand is a register
(KL, VL) = (2, 128), (4, 256), (8, 512)
FOR j <- 0 TO KL-1
    i <- j * 64
    k <- j * 32
    IF k1[j] OR *no writemask*
         THEN DEST[i+63:i] <-
               Convert_Integer_To_Double_Precision_Floating_Point(SRC[k+31:k])
         ELSE
               IF *merging-masking*                ; merging-masking
                    THEN *DEST[i+63:i] remains unchanged*
                    ELSE                           ; zeroing-masking
                         DEST[i+63:i] <- 0
               FI
    FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0





VCVTDQ2PD (EVEX encoded versions) when src operand is a memory source
(KL, VL) = (2, 128), (4, 256), (8, 512)

FOR j <- 0 TO KL-1
   i <- j * 64
   k <- j * 32
   IF k1[j] OR *no writemask*
        THEN
             IF (EVEX.b = 1)
                  THEN
                       DEST[i+63:i] <-
             Convert_Integer_To_Double_Precision_Floating_Point(SRC[31:0])
                  ELSE
                       DEST[i+63:i] <-
             Convert_Integer_To_Double_Precision_Floating_Point(SRC[k+31:k])
             FI;
        ELSE
             IF *merging-masking*                ; merging-masking
                  THEN *DEST[i+63:i] remains unchanged*
                  ELSE                           ; zeroing-masking
                       DEST[i+63:i] <- 0
             FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VCVTDQ2PD (VEX.256 encoded version)
DEST[63:0] <- Convert_Integer_To_Double_Precision_Floating_Point(SRC[31:0])
DEST[127:64] <- Convert_Integer_To_Double_Precision_Floating_Point(SRC[63:32])
DEST[191:128] <- Convert_Integer_To_Double_Precision_Floating_Point(SRC[95:64])
DEST[255:192] <- Convert_Integer_To_Double_Precision_Floating_Point(SRC[127:96)
DEST[MAX_VL-1:256] <- 0

VCVTDQ2PD (VEX.128 encoded version)
DEST[63:0] <- Convert_Integer_To_Double_Precision_Floating_Point(SRC[31:0])
DEST[127:64] <- Convert_Integer_To_Double_Precision_Floating_Point(SRC[63:32])
DEST[MAX_VL-1:128] <- 0

CVTDQ2PD (128-bit Legacy SSE version)
DEST[63:0] <- Convert_Integer_To_Double_Precision_Floating_Point(SRC[31:0])
DEST[127:64] <- Convert_Integer_To_Double_Precision_Floating_Point(SRC[63:32])
DEST[MAX_VL-1:128] (unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VCVTDQ2PD __m512d _mm512_cvtepi32_pd( __m256i a);
VCVTDQ2PD __m512d _mm512_mask_cvtepi32_pd( __m512d s, __mmask8 k, __m256i a);
VCVTDQ2PD __m512d _mm512_maskz_cvtepi32_pd( __mmask8 k, __m256i a);
VCVTDQ2PD __m256d _mm256_mask_cvtepi32_pd( __m256d s, __mmask8 k, __m256i a);
VCVTDQ2PD __m256d _mm256_maskz_cvtepi32_pd( __mmask8 k, __m256i a);
VCVTDQ2PD __m128d _mm_mask_cvtepi32_pd( __m128d s, __mmask8 k, __m128i a);
VCVTDQ2PD __m128d _mm_maskz_cvtepi32_pd( __mmask8 k, __m128i a);
CVTDQ2PD __m256d _mm256_cvtepi32_pd (__m128i src)
CVTDQ2PD __m128d _mm_cvtepi32_pd (__m128i src)




Other Exceptions
VEX-encoded instructions, see Exceptions Type 5;
EVEX-encoded instructions, see Exceptions Type E5.
#UD                      If VEX.vvvv != 1111B or EVEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTDQ2PS','CVTDQ2PS-Convert Packed Doubleword Integers to Packed Single-Precision Floating-Point
Values
 Opcode                               Op /    64/32         CPUID          Description
 Instruction                          En      bit Mode      Feature
                                              Support       Flag
 0F 5B /r                             RM      V/V           SSE2           Convert four packed signed doubleword integers from
 CVTDQ2PS xmm1, xmm2/m128                                                  xmm2/mem to four packed single-precision floating-
                                                                           point values in xmm1.
 VEX.128.0F.WIG 5B /r                 RM       V/V          AVX            Convert four packed signed doubleword integers from
 VCVTDQ2PS xmm1, xmm2/m128                                                 xmm2/mem to four packed single-precision floating-
                                                                           point values in xmm1.
 VEX.256.0F.WIG 5B /r                 RM       V/V          AVX            Convert eight packed signed doubleword integers from
 VCVTDQ2PS ymm1, ymm2/m256                                                 ymm2/mem to eight packed single-precision floating-
                                                                           point values in ymm1.
 EVEX.128.0F.W0 5B /r                 FV      V/V           AVX512VL       Convert four packed signed doubleword integers from
 VCVTDQ2PS xmm1 {k1}{z},                                    AVX512F        xmm2/m128/m32bcst to four packed single-precision
 xmm2/m128/m32bcst                                                         floating-point values in xmm1with writemask k1.
 EVEX.256.0F.W0 5B /r                 FV      V/V           AVX512VL       Convert eight packed signed doubleword integers from
 VCVTDQ2PS ymm1 {k1}{z},                                    AVX512F        ymm2/m256/m32bcst to eight packed single-precision
 ymm2/m256/m32bcst                                                         floating-point values in ymm1with writemask k1.
 EVEX.512.0F.W0 5B /r                 FV      V/V           AVX512F        Convert sixteen packed signed doubleword integers
 VCVTDQ2PS zmm1 {k1}{z},                                                   from zmm2/m512/m32bcst to sixteen packed single-
 zmm2/m512/m32bcst{er}                                                     precision floating-point values in zmm1with writemask
                                                                           k1.



                                              Instruction Operand Encoding
       Op/En            Operand 1                    Operand 2                    Operand 3                      Operand 4
        RM            ModRM:reg (w)               ModRM:r/m (r)                       NA                             NA
        FV            ModRM:reg (w)               ModRM:r/m (r)                       NA                             NA

Description
Converts four, eight or sixteen packed signed doubleword integers in the source operand to four, eight or sixteen
packed single-precision floating-point values in the destination operand.
EVEX encoded versions: The source operand can be a ZMM/YMM/XMM register, a 512/256/128-bit memory loca-
tion or a 512/256/128-bit vector broadcasted from a 32-bit memory location. The destination operand is a
ZMM/YMM/XMM register conditionally updated with writemask k1.
VEX.256 encoded version: The source operand is a YMM register or 256- bit memory location. The destination
operand is a YMM register. Bits (MAX_VL-1:256) of the corresponding register destination are zeroed.
VEX.128 encoded version: The source operand is an XMM register or 128- bit memory location. The destination
operand is a XMM register. The upper bits (MAX_VL-1:128) of the corresponding register destination are zeroed.
128-bit Legacy SSE version: The source operand is an XMM register or 128- bit memory location. The destination
operand is an XMM register. The upper Bits (MAX_VL-1:128) of the corresponding register destination are unmod-
ified.
VEX.vvvv and EVEX.vvvv are reserved and must be 1111b, otherwise instructions will #UD.





Operation
VCVTDQ2PS (EVEX encoded versions) when SRC operand is a register
(KL, VL) = (4, 128), (8, 256), (16, 512)
IF (VL = 512) AND (EVEX.b = 1)
    THEN
         SET_RM(EVEX.RC);              ; refer to Table 2-4 in the Intel Architecture Instruction Set Extensions Programming Reference
    ELSE
         SET_RM(MXCSR.RM);             ; refer to Table 2-4 in the Intel Architecture Instruction Set Extensions Programming Reference
FI;

FOR j <- 0 TO KL-1
   i <- j * 32
   IF k1[j] OR *no writemask*
        THEN DEST[i+31:i] <-
             Convert_Integer_To_Single_Precision_Floating_Point(SRC[i+31:i])
        ELSE
             IF *merging-masking*                ; merging-masking
                  THEN *DEST[i+31:i] remains unchanged*
                  ELSE                           ; zeroing-masking
                       DEST[i+31:i] <- 0
             FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VCVTDQ2PS (EVEX encoded versions) when SRC operand is a memory source
(KL, VL) = (4, 128), (8, 256), (16, 512)

FOR j <- 0 TO KL-1
   i <-j * 32
   IF k1[j] OR *no writemask*
        THEN
             IF (EVEX.b = 1)
                  THEN
                       DEST[i+31:i] <-
             Convert_Integer_To_Single_Precision_Floating_Point(SRC[31:0])
                  ELSE
                       DEST[i+31:i] <-
             Convert_Integer_To_Single_Precision_Floating_Point(SRC[i+31:i])
             FI;
        ELSE
             IF *merging-masking*                ; merging-masking
                  THEN *DEST[i+31:i] remains unchanged*
                  ELSE                           ; zeroing-masking
                       DEST[i+31:i] <- 0
             FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0





VCVTDQ2PS (VEX.256 encoded version)
DEST[31:0] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[31:0])
DEST[63:32] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[63:32])
DEST[95:64] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[95:64])
DEST[127:96] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[127:96)
DEST[159:128] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[159:128])
DEST[191:160] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[191:160])
DEST[223:192] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[223:192])
DEST[255:224] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[255:224)
DEST[MAX_VL-1:256] <- 0

VCVTDQ2PS (VEX.128 encoded version)
DEST[31:0] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[31:0])
DEST[63:32] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[63:32])
DEST[95:64] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[95:64])
DEST[127:96] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[127z:96)
DEST[MAX_VL-1:128] <- 0

CVTDQ2PS (128-bit Legacy SSE version)
DEST[31:0] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[31:0])
DEST[63:32] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[63:32])
DEST[95:64] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[95:64])
DEST[127:96] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[127z:96)
DEST[MAX_VL-1:128] (unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VCVTDQ2PS __m512 _mm512_cvtepi32_ps( __m512i a);
VCVTDQ2PS __m512 _mm512_mask_cvtepi32_ps( __m512 s, __mmask16 k, __m512i a);
VCVTDQ2PS __m512 _mm512_maskz_cvtepi32_ps( __mmask16 k, __m512i a);
VCVTDQ2PS __m512 _mm512_cvt_roundepi32_ps( __m512i a, int r);
VCVTDQ2PS __m512 _mm512_mask_cvt_roundepi_ps( __m512 s, __mmask16 k, __m512i a, int r);
VCVTDQ2PS __m512 _mm512_maskz_cvt_roundepi32_ps( __mmask16 k, __m512i a, int r);
VCVTDQ2PS __m256 _mm256_mask_cvtepi32_ps( __m256 s, __mmask8 k, __m256i a);
VCVTDQ2PS __m256 _mm256_maskz_cvtepi32_ps( __mmask8 k, __m256i a);
VCVTDQ2PS __m128 _mm_mask_cvtepi32_ps( __m128 s, __mmask8 k, __m128i a);
VCVTDQ2PS __m128 _mm_maskz_cvtepi32_ps( __mmask8 k, __m128i a);
CVTDQ2PS __m256 _mm256_cvtepi32_ps (__m256i src)
CVTDQ2PS __m128 _mm_cvtepi32_ps (__m128i src)

SIMD Floating-Point Exceptions
Precision

Other Exceptions
VEX-encoded instructions, see Exceptions Type 2;
EVEX-encoded instructions, see Exceptions Type E2.
#UD                   If VEX.vvvv != 1111B or EVEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTPD2DQ','CVTPD2DQ-Convert Packed Double-Precision Floating-Point Values to Packed Doubleword
Integers
 Opcode                                      Op /    64/32         CPUID         Description
 Instruction                                 En      bit Mode      Feature
                                                     Support       Flag
 F2 0F E6 /r                                 RM      V/V           SSE2          Convert two packed double-precision floating-point
 CVTPD2DQ xmm1, xmm2/m128                                                        values in xmm2/mem to two signed doubleword
                                                                                 integers in xmm1.
 VEX.128.F2.0F.WIG E6 /r                     RM      V/V           AVX           Convert two packed double-precision floating-point
 VCVTPD2DQ xmm1, xmm2/m128                                                       values in xmm2/mem to two signed doubleword
                                                                                 integers in xmm1.
 VEX.256.F2.0F.WIG E6 /r                     RM      V/V           AVX           Convert four packed double-precision floating-point
 VCVTPD2DQ xmm1, ymm2/m256                                                       values in ymm2/mem to four signed doubleword
                                                                                 integers in xmm1.
 EVEX.128.F2.0F.W1 E6 /r                     FV      V/V           AVX512VL      Convert two packed double-precision floating-point
 VCVTPD2DQ xmm1 {k1}{z},                                           AVX512F       values in xmm2/m128/m64bcst to two signed
 xmm2/m128/m64bcst                                                               doubleword integers in xmm1 subject to writemask k1.
 EVEX.256.F2.0F.W1 E6 /r                     FV      V/V           AVX512VL      Convert four packed double-precision floating-point
 VCVTPD2DQ xmm1 {k1}{z},                                           AVX512F       values in ymm2/m256/m64bcst to four signed
 ymm2/m256/m64bcst                                                               doubleword integers in xmm1 subject to writemask k1.
 EVEX.512.F2.0F.W1 E6 /r                     FV      V/V           AVX512F       Convert eight packed double-precision floating-point
 VCVTPD2DQ ymm1 {k1}{z},                                                         values in zmm2/m512/m64bcst to eight signed
 zmm2/m512/m64bcst{er}                                                           doubleword integers in ymm1 subject to writemask k1.



                                                    Instruction Operand Encoding
      Op/En                Operand 1                    Operand 2                      Operand 3                  Operand 4
       RM                ModRM:reg (w)                ModRM:r/m (r)                        NA                        NA
        FV               ModRM:reg (w)                ModRM:r/m (r)                        NA                        NA

Description
Converts packed double-precision floating-point values in the source operand (second operand) to packed signed
doubleword integers in the destination operand (first operand).
When a conversion is inexact, the value returned is rounded according to the rounding control bits in the MXCSR
register or the embedded rounding control bits. If a converted result cannot be represented in the destination
format, the floating-point invalid exception is raised, and if this exception is masked, the indefinite integer value
(2w-1, where w represents the number of bits in the destination format) is returned.
EVEX encoded versions: The source operand is a ZMM/YMM/XMM register, a 512-bit memory location, or a 512-bit
vector broadcasted from a 64-bit memory location. The destination operand is a ZMM/YMM/XMM register condi-
tionally updated with writemask k1. The upper bits (MAX_VL-1:256/128/64) of the corresponding destination are
zeroed.
VEX.256 encoded version: The source operand is a YMM register or 256- bit memory location. The destination
operand is an XMM register. The upper bits (MAX_VL-1:128) of the corresponding ZMM register destination are
zeroed.
VEX.128 encoded version: The source operand is an XMM register or 128- bit memory location. The destination
operand is a XMM register. The upper bits (MAX_VL-1:64) of the corresponding ZMM register destination are
zeroed.
128-bit Legacy SSE version: The source operand is an XMM register or 128- bit memory location. The destination
operand is an XMM register. Bits[127:64] of the destination XMM register are zeroed. However, the upper bits
(MAX_VL-1:128) of the corresponding ZMM register destination are unmodified.
VEX.vvvv and EVEX.vvvv are reserved and must be 1111b, otherwise instructions will #UD.





                SRC        X3                   X2                   X1                  X0




                DEST                   0                      X3          X2       X1         X0



                                 Figure 3-12. VCVTPD2DQ (VEX.256 encoded version)


Operation
VCVTPD2DQ (EVEX encoded versions) when src operand is a register
(KL, VL) = (2, 128), (4, 256), (8, 512)
IF (VL = 512) AND (EVEX.b = 1)
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);
FI;

FOR j <- 0 TO KL-1
   i <- j * 32
   k <- j * 64
   IF k1[j] OR *no writemask*
        THEN DEST[i+31:i] <-
             Convert_Double_Precision_Floating_Point_To_Integer(SRC[k+63:k])
        ELSE
             IF *merging-masking*                ; merging-masking
                  THEN *DEST[i+31:i] remains unchanged*
                  ELSE                           ; zeroing-masking
                       DEST[i+31:i] <- 0
             FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL/2] <- 0





VCVTPD2DQ (EVEX encoded versions) when src operand is a memory source
(KL, VL) = (2, 128), (4, 256), (8, 512)
FOR j <- 0 TO KL-1
    i <- j * 32
    k <- j * 64
    IF k1[j] OR *no writemask*
         THEN
               IF (EVEX.b = 1)
                    THEN
                         DEST[i+31:i] <-
               Convert_Double_Precision_Floating_Point_To_Integer(SRC[63:0])
                    ELSE
                         DEST[i+31:i] <-
               Convert_Double_Precision_Floating_Point_To_Integer(SRC[k+63:k])
               FI;
         ELSE
               IF *merging-masking*                ; merging-masking
                    THEN *DEST[i+31:i] remains unchanged*
                    ELSE                           ; zeroing-masking
                         DEST[i+31:i] <- 0
               FI
    FI;
ENDFOR
DEST[MAX_VL-1:VL/2] <- 0

VCVTPD2DQ (VEX.256 encoded version)
DEST[31:0] <-Convert_Double_Precision_Floating_Point_To_Integer(SRC[63:0])
DEST[63:32] <-Convert_Double_Precision_Floating_Point_To_Integer(SRC[127:64])
DEST[95:64] <-Convert_Double_Precision_Floating_Point_To_Integer(SRC[191:128])
DEST[127:96] <-Convert_Double_Precision_Floating_Point_To_Integer(SRC[255:192)
DEST[MAX_VL-1:128]<-0

VCVTPD2DQ (VEX.128 encoded version)
DEST[31:0] <-Convert_Double_Precision_Floating_Point_To_Integer(SRC[63:0])
DEST[63:32] <-Convert_Double_Precision_Floating_Point_To_Integer(SRC[127:64])
DEST[MAX_VL-1:64]<-0

CVTPD2DQ (128-bit Legacy SSE version)
DEST[31:0] <-Convert_Double_Precision_Floating_Point_To_Integer(SRC[63:0])
DEST[63:32] <-Convert_Double_Precision_Floating_Point_To_Integer(SRC[127:64])
DEST[127:64] <-0
DEST[MAX_VL-1:128] (unmodified)





Intel C/C++ Compiler Intrinsic Equivalent
VCVTPD2DQ __m256i _mm512_cvtpd_epi32( __m512d a);
VCVTPD2DQ __m256i _mm512_mask_cvtpd_epi32( __m256i s, __mmask8 k, __m512d a);
VCVTPD2DQ __m256i _mm512_maskz_cvtpd_epi32( __mmask8 k, __m512d a);
VCVTPD2DQ __m256i _mm512_cvt_roundpd_epi32( __m512d a, int r);
VCVTPD2DQ __m256i _mm512_mask_cvt_roundpd_epi32( __m256i s, __mmask8 k, __m512d a, int r);
VCVTPD2DQ __m256i _mm512_maskz_cvt_roundpd_epi32( __mmask8 k, __m512d a, int r);
VCVTPD2DQ __m128i _mm256_mask_cvtpd_epi32( __m128i s, __mmask8 k, __m256d a);
VCVTPD2DQ __m128i _mm256_maskz_cvtpd_epi32( __mmask8 k, __m256d a);
VCVTPD2DQ __m128i _mm_mask_cvtpd_epi32( __m128i s, __mmask8 k, __m128d a);
VCVTPD2DQ __m128i _mm_maskz_cvtpd_epi32( __mmask8 k, __m128d a);
VCVTPD2DQ __m128i _mm256_cvtpd_epi32 (__m256d src)
CVTPD2DQ __m128i _mm_cvtpd_epi32 (__m128d src)

SIMD Floating-Point Exceptions
Invalid, Precision

Other Exceptions
See Exceptions Type 2; additionally
EVEX-encoded instructions, see Exceptions Type E2.
#UD                  If VEX.vvvv != 1111B or EVEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTPD2PI','CVTPD2PI-Convert Packed Double-Precision FP Values to Packed Dword Integers
Opcode/                                                  Op/    64-Bit        Compat/ Description
Instruction                                              En     Mode          Leg Mode
66 0F 2D /r                                              RM     Valid         Valid     Convert two packed double-precision floating-
CVTPD2PI mm, xmm/m128                                                                   point values from xmm/m128 to two packed
                                                                                        signed doubleword integers in mm.



                                                   Instruction Operand Encoding
   Op/En               Operand 1                        Operand 2                     Operand 3                   Operand 4
    RM               ModRM:reg (w)                    ModRM:r/m (r)                      NA                          NA

Description
Converts two packed double-precision floating-point values in the source operand (second operand) to two packed
signed doubleword integers in the destination operand (first operand).
The source operand can be an XMM register or a 128-bit memory location. The destination operand is an MMX tech-
nology register.
When a conversion is inexact, the value returned is rounded according to the rounding control bits in the MXCSR
register. If a converted result is larger than the maximum signed doubleword integer, the floating-point invalid
exception is raised, and if this exception is masked, the indefinite integer value (80000000H) is returned.
This instruction causes a transition from x87 FPU to MMX technology operation (that is, the x87 FPU top-of-stack
pointer is set to 0 and the x87 FPU tag word is set to all 0s [valid]). If this instruction is executed while an x87 FPU
floating-point exception is pending, the exception is handled before the CVTPD2PI instruction is executed.
In 64-bit mode, use of the REX.R prefix permits this instruction to access additional registers (XMM8-XMM15).

Operation
DEST[31:0] <- Convert_Double_Precision_Floating_Point_To_Integer32(SRC[63:0]);
DEST[63:32] <- Convert_Double_Precision_Floating_Point_To_Integer32(SRC[127:64]);

Intel C/C++ Compiler Intrinsic Equivalent
CVTPD1PI:      __m64 _mm_cvtpd_pi32(__m128d a)

SIMD Floating-Point Exceptions
Invalid, Precision.

Other Exceptions
See Table 22-4, "Exception Conditions for Legacy SIMD/MMX Instructions with FP Exception and 16-Byte Align-
ment," in the Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 3B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTPD2PS','CVTPD2PS-Convert Packed Double-Precision Floating-Point Values to Packed Single-Precision
Floating-Point Values
 Opcode/                                  Op /     64/32        CPUID          Description
 Instruction                              En       bit Mode     Feature
                                                   Support      Flag
 66 0F 5A /r                              RM       V/V          SSE2           Convert two packed double-precision floating-point
 CVTPD2PS xmm1, xmm2/m128                                                      values in xmm2/mem to two single-precision
                                                                               floating-point values in xmm1.
 VEX.128.66.0F.WIG 5A /r                  RM       V/V          AVX            Convert two packed double-precision floating-point
 VCVTPD2PS xmm1, xmm2/m128                                                     values in xmm2/mem to two single-precision
                                                                               floating-point values in xmm1.
 VEX.256.66.0F.WIG 5A /r                  RM       V/V          AVX            Convert four packed double-precision floating-point
 VCVTPD2PS xmm1, ymm2/m256                                                     values in ymm2/mem to four single-precision
                                                                               floating-point values in xmm1.
 EVEX.128.66.0F.W1 5A /r                  FV       V/V          AVX512VL       Convert two packed double-precision floating-point
 VCVTPD2PS xmm1 {k1}{z},                                        AVX512F        values in xmm2/m128/m64bcst to two single-
 xmm2/m128/m64bcst                                                             precision floating-point values in xmm1with
                                                                               writemask k1.
 EVEX.256.66.0F.W1 5A /r                  FV       V/V          AVX512VL       Convert four packed double-precision floating-point
 VCVTPD2PS xmm1 {k1}{z},                                        AVX512F        values in ymm2/m256/m64bcst to four single-
 ymm2/m256/m64bcst                                                             precision floating-point values in xmm1with
                                                                               writemask k1.
 EVEX.512.66.0F.W1 5A /r                  FV       V/V          AVX512F        Convert eight packed double-precision floating-point
 VCVTPD2PS ymm1 {k1}{z},                                                       values in zmm2/m512/m64bcst to eight single-
 zmm2/m512/m64bcst{er}                                                         precision floating-point values in ymm1with
                                                                               writemask k1.



                                                 Instruction Operand Encoding
      Op/En             Operand 1                        Operand 2                    Operand 3                      Operand 4
       RM             ModRM:reg (w)                  ModRM:r/m (r)                        NA                             NA
       FV             ModRM:reg (w)                  ModRM:r/m (r)                        NA                             NA

Description
Converts two, four or eight packed double-precision floating-point values in the source operand (second operand)
to two, four or eight packed single-precision floating-point values in the destination operand (first operand).
When a conversion is inexact, the value returned is rounded according to the rounding control bits in the MXCSR
register or the embedded rounding control bits.
EVEX encoded versions: The source operand is a ZMM/YMM/XMM register, a 512/256/128-bit memory location, or
a 512/256/128-bit vector broadcasted from a 64-bit memory location. The destination operand is a
YMM/XMM/XMM (low 64-bits) register conditionally updated with writemask k1. The upper bits (MAX_VL-
1:256/128/64) of the corresponding destination are zeroed.
VEX.256 encoded version: The source operand is a YMM register or 256- bit memory location. The destination
operand is an XMM register. The upper bits (MAX_VL-1:128) of the corresponding ZMM register destination are
zeroed.
VEX.128 encoded version: The source operand is an XMM register or 128- bit memory location. The destination
operand is a XMM register. The upper bits (MAX_VL-1:64) of the corresponding ZMM register destination are
zeroed.
128-bit Legacy SSE version: The source operand is an XMM register or 128- bit memory location. The destination
operand is an XMM register. Bits[127:64] of the destination XMM register are zeroed. However, the upper Bits
(MAX_VL-1:128) of the corresponding ZMM register destination are unmodified.
VEX.vvvv and EVEX.vvvv are reserved and must be 1111b otherwise instructions will #UD.




                   SRC         X3                      X2                   X1                   X0




                   DEST                      0                        X3         X2       X1          X0



                                      Figure 3-13. VCVTPD2PS (VEX.256 encoded version)


Operation
VCVTPD2PS (EVEX encoded version) when src operand is a register
(KL, VL) = (2, 128), (4, 256), (8, 512)
IF (VL = 512) AND (EVEX.b = 1)
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);
FI;

FOR j <- 0 TO KL-1
   i <- j * 32
   k <- j * 64
   IF k1[j] OR *no writemask*
        THEN
             DEST[i+31:i] <- Convert_Double_Precision_Floating_Point_To_Single_Precision_Floating_Point(SRC[k+63:k])
        ELSE
             IF *merging-masking*                ; merging-masking
                  THEN *DEST[i+31:i] remains unchanged*
                  ELSE                           ; zeroing-masking
                       DEST[i+31:i] <- 0
             FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL/2] <- 0





VCVTPD2PS (EVEX encoded version) when src operand is a memory source
(KL, VL) = (2, 128), (4, 256), (8, 512)

FOR j <- 0 TO KL-1
   i <- j * 32
   k <- j * 64
   IF k1[j] OR *no writemask*
        THEN
             IF (EVEX.b = 1)
                  THEN
                       DEST[i+31:i] <-Convert_Double_Precision_Floating_Point_To_Single_Precision_Floating_Point(SRC[63:0])
                  ELSE
                       DEST[i+31:i] <- Convert_Double_Precision_Floating_Point_To_Single_Precision_Floating_Point(SRC[k+63:k])
             FI;
        ELSE
             IF *merging-masking*                ; merging-masking
                  THEN *DEST[i+31:i] remains unchanged*
                  ELSE                           ; zeroing-masking
                       DEST[i+31:i] <- 0
             FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL/2] <- 0

VCVTPD2PS (VEX.256 encoded version)
DEST[31:0] <- Convert_Double_Precision_To_Single_Precision_Floating_Point(SRC[63:0])
DEST[63:32] <- Convert_Double_Precision_To_Single_Precision_Floating_Point(SRC[127:64])
DEST[95:64] <- Convert_Double_Precision_To_Single_Precision_Floating_Point(SRC[191:128])
DEST[127:96] <- Convert_Double_Precision_To_Single_Precision_Floating_Point(SRC[255:192)
DEST[MAX_VL-1:128] <- 0

VCVTPD2PS (VEX.128 encoded version)
DEST[31:0] <- Convert_Double_Precision_To_Single_Precision_Floating_Point(SRC[63:0])
DEST[63:32] <- Convert_Double_Precision_To_Single_Precision_Floating_Point(SRC[127:64])
DEST[MAX_VL-1:64] <- 0

CVTPD2PS (128-bit Legacy SSE version)
DEST[31:0] <- Convert_Double_Precision_To_Single_Precision_Floating_Point(SRC[63:0])
DEST[63:32] <- Convert_Double_Precision_To_Single_Precision_Floating_Point(SRC[127:64])
DEST[127:64] <- 0
DEST[MAX_VL-1:128] (unmodified)





Intel C/C++ Compiler Intrinsic Equivalent
VCVTPD2PS __m256 _mm512_cvtpd_ps( __m512d a);
VCVTPD2PS __m256 _mm512_mask_cvtpd_ps( __m256 s, __mmask8 k, __m512d a);
VCVTPD2PS __m256 _mm512_maskz_cvtpd_ps( __mmask8 k, __m512d a);
VCVTPD2PS __m256 _mm512_cvt_roundpd_ps( __m512d a, int r);
VCVTPD2PS __m256 _mm512_mask_cvt_roundpd_ps( __m256 s, __mmask8 k, __m512d a, int r);
VCVTPD2PS __m256 _mm512_maskz_cvt_roundpd_ps( __mmask8 k, __m512d a, int r);
VCVTPD2PS __m128 _mm256_mask_cvtpd_ps( __m128 s, __mmask8 k, __m256d a);
VCVTPD2PS __m128 _mm256_maskz_cvtpd_ps( __mmask8 k, __m256d a);
VCVTPD2PS __m128 _mm_mask_cvtpd_ps( __m128 s, __mmask8 k, __m128d a);
VCVTPD2PS __m128 _mm_maskz_cvtpd_ps( __mmask8 k, __m128d a);
VCVTPD2PS __m128 _mm256_cvtpd_ps (__m256d a)
CVTPD2PS __m128 _mm_cvtpd_ps (__m128d a)

SIMD Floating-Point Exceptions
Invalid, Precision, Underflow, Overflow, Denormal

Other Exceptions
VEX-encoded instructions, see Exceptions Type 2;
EVEX-encoded instructions, see Exceptions Type E2.
#UD                       If VEX.vvvv != 1111B or EVEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTPI2PD','CVTPI2PD-Convert Packed Dword Integers to Packed Double-Precision FP Values
Opcode/                                                 Op/    64-Bit       Compat/ Description
Instruction                                             En     Mode         Leg Mode
66 0F 2A /r                                             RM     Valid        Valid       Convert two packed signed doubleword
CVTPI2PD xmm, mm/m64*                                                                   integers from mm/mem64 to two packed
                                                                                        double-precision floating-point values in xmm.
NOTES:
*Operation is different for different operand sets; see the Description section.



                                                  Instruction Operand Encoding
    Op/En             Operand 1                      Operand 2                      Operand 3                      Operand 4
     RM             ModRM:reg (w)                  ModRM:r/m (r)                        NA                             NA

Description
Converts two packed signed doubleword integers in the source operand (second operand) to two packed double-
precision floating-point values in the destination operand (first operand).
The source operand can be an MMX technology register or a 64-bit memory location. The destination operand is an
XMM register. In addition, depending on the operand configuration:
.    For operands xmm, mm: the instruction causes a transition from x87 FPU to MMX technology operation (that
     is, the x87 FPU top-of-stack pointer is set to 0 and the x87 FPU tag word is set to all 0s [valid]). If this
     instruction is executed while an x87 FPU floating-point exception is pending, the exception is handled before
     the CVTPI2PD instruction is executed.
.    For operands xmm, m64: the instruction does not cause a transition to MMX technology and does not take
     x87 FPU exceptions.
In 64-bit mode, use of the REX.R prefix permits this instruction to access additional registers (XMM8-XMM15).

Operation
DEST[63:0] <- Convert_Integer_To_Double_Precision_Floating_Point(SRC[31:0]);
DEST[127:64] <- Convert_Integer_To_Double_Precision_Floating_Point(SRC[63:32]);

Intel C/C++ Compiler Intrinsic Equivalent
CVTPI2PD:       __m128d _mm_cvtpi32_pd(__m64 a)

SIMD Floating-Point Exceptions
None

Other Exceptions
See Table 22-6, "Exception Conditions for Legacy SIMD/MMX Instructions with XMM and without FP Exception," in
the Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 3B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTPI2PS','CVTPI2PS-Convert Packed Dword Integers to Packed Single-Precision FP Values
Opcode/                                                    Op/    64-Bit      Compat/ Description
Instruction                                                En     Mode        Leg Mode
0F 2A /r                                                   RM     Valid       Valid       Convert two signed doubleword integers
CVTPI2PS xmm, mm/m64                                                                      from mm/m64 to two single-precision
                                                                                          floating-point values in xmm.



                                                    Instruction Operand Encoding
   Op/En              Operand 1                       Operand 2                       Operand 3                  Operand 4
    RM              ModRM:reg (w)                   ModRM:r/m (r)                        NA                          NA

Description
Converts two packed signed doubleword integers in the source operand (second operand) to two packed single-
precision floating-point values in the destination operand (first operand).
The source operand can be an MMX technology register or a 64-bit memory location. The destination operand is an
XMM register. The results are stored in the low quadword of the destination operand, and the high quadword
remains unchanged. When a conversion is inexact, the value returned is rounded according to the rounding control
bits in the MXCSR register.
This instruction causes a transition from x87 FPU to MMX technology operation (that is, the x87 FPU top-of-stack
pointer is set to 0 and the x87 FPU tag word is set to all 0s [valid]). If this instruction is executed while an x87 FPU
floating-point exception is pending, the exception is handled before the CVTPI2PS instruction is executed.
In 64-bit mode, use of the REX.R prefix permits this instruction to access additional registers (XMM8-XMM15).

Operation
DEST[31:0] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[31:0]);
DEST[63:32] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[63:32]);
(* High quadword of destination unchanged *)

Intel C/C++ Compiler Intrinsic Equivalent
CVTPI2PS:      __m128 _mm_cvtpi32_ps(__m128 a, __m64 b)

SIMD Floating-Point Exceptions
Precision

Other Exceptions
See Table 22-5, "Exception Conditions for Legacy SIMD/MMX Instructions with XMM and FP Exception," in the
Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 3B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTPS2DQ','CVTPS2DQ-Convert Packed Single-Precision Floating-Point Values to Packed Signed
Doubleword Integer Values
 Opcode/                              Op /   64/32       CPUID          Description
 Instruction                          En     bit Mode    Feature
                                             Support     Flag
 66 0F 5B /r                          RM     V/V         SSE2           Convert four packed single-precision floating-point values
 CVTPS2DQ xmm1, xmm2/m128                                               from xmm2/mem to four packed signed doubleword
                                                                        values in xmm1.
 VEX.128.66.0F.WIG 5B /r              RM     V/V         AVX            Convert four packed single-precision floating-point values
 VCVTPS2DQ xmm1, xmm2/m128                                              from xmm2/mem to four packed signed doubleword
                                                                        values in xmm1.
 VEX.256.66.0F.WIG 5B /r              RM     V/V         AVX            Convert eight packed single-precision floating-point values
 VCVTPS2DQ ymm1, ymm2/m256                                              from ymm2/mem to eight packed signed doubleword
                                                                        values in ymm1.
 EVEX.128.66.0F.W0 5B /r              FV     V/V         AVX512VL       Convert four packed single precision floating-point values
 VCVTPS2DQ xmm1 {k1}{z},                                 AVX512F        from xmm2/m128/m32bcst to four packed signed
 xmm2/m128/m32bcst                                                      doubleword values in xmm1 subject to writemask k1.
 EVEX.256.66.0F.W0 5B /r              FV     V/V         AVX512VL       Convert eight packed single precision floating-point values
 VCVTPS2DQ ymm1 {k1}{z},                                 AVX512F        from ymm2/m256/m32bcst to eight packed signed
 ymm2/m256/m32bcst                                                      doubleword values in ymm1 subject to writemask k1.
 EVEX.512.66.0F.W0 5B /r              FV     V/V         AVX512F        Convert sixteen packed single-precision floating-point
 VCVTPS2DQ zmm1 {k1}{z},                                                values from zmm2/m512/m32bcst to sixteen packed
 zmm2/m512/m32bcst{er}                                                  signed doubleword values in zmm1 subject to writemask
                                                                        k1.



                                              Instruction Operand Encoding
     Op/En              Operand 1                   Operand 2                      Operand 3                     Operand 4
       RM             ModRM:reg (w)                ModRM:r/m (r)                       NA                            NA
       FV             ModRM:reg (w)                ModRM:r/m (r)                       NA                            NA

Description
Converts four, eight or sixteen packed single-precision floating-point values in the source operand to four, eight or
sixteen signed doubleword integers in the destination operand.
When a conversion is inexact, the value returned is rounded according to the rounding control bits in the MXCSR
register or the embedded rounding control bits. If a converted result cannot be represented in the destination
format, the floating-point invalid exception is raised, and if this exception is masked, the indefinite integer value
(2w-1, where w represents the number of bits in the destination format) is returned.
EVEX encoded versions: The source operand is a ZMM register, a 512-bit memory location or a 512-bit vector
broadcasted from a 32-bit memory location. The destination operand is a ZMM register conditionally updated with
writemask k1.
VEX.256 encoded version: The source operand is a YMM register or 256- bit memory location. The destination
operand is a YMM register. The upper bits (MAX_VL-1:256) of the corresponding ZMM register destination are
zeroed.
VEX.128 encoded version: The source operand is an XMM register or 128- bit memory location. The destination
operand is a XMM register. The upper bits (MAX_VL-1:128) of the corresponding ZMM register destination are
zeroed.
128-bit Legacy SSE version: The source operand is an XMM register or 128- bit memory location. The destination
operand is an XMM register. The upper bits (MAX_VL-1:128) of the corresponding ZMM register destination are
unmodified.
VEX.vvvv and EVEX.vvvv are reserved and must be 1111b otherwise instructions will #UD.





Operation
VCVTPS2DQ (encoded versions) when src operand is a register
(KL, VL) = (4, 128), (8, 256), (16, 512)
IF (VL = 512) AND (EVEX.b = 1)
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);
FI;

FOR j <- 0 TO KL-1
   i <- j * 32
   IF k1[j] OR *no writemask*
        THEN DEST[i+31:i] <-
             Convert_Single_Precision_Floating_Point_To_Integer(SRC[i+31:i])
        ELSE
             IF *merging-masking*                ; merging-masking
                  THEN *DEST[i+31:i] remains unchanged*
                  ELSE                           ; zeroing-masking
                       DEST[i+31:i] <- 0
             FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VCVTPS2DQ (EVEX encoded versions) when src operand is a memory source
(KL, VL) = (4, 128), (8, 256), (16, 512)

FOR j <- 0 TO 15
   i <- j * 32
   IF k1[j] OR *no writemask*
        THEN
             IF (EVEX.b = 1)
                  THEN
                       DEST[i+31:i] <-
             Convert_Single_Precision_Floating_Point_To_Integer(SRC[31:0])
                  ELSE
                       DEST[i+31:i] <-
             Convert_Single_Precision_Floating_Point_To_Integer(SRC[i+31:i])
             FI;
        ELSE
             IF *merging-masking*                ; merging-masking
                  THEN *DEST[i+31:i] remains unchanged*
                  ELSE                           ; zeroing-masking
                       DEST[i+31:i] <- 0
             FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0





VCVTPS2DQ (VEX.256 encoded version)
DEST[31:0] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[31:0])
DEST[63:32] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[63:32])
DEST[95:64] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[95:64])
DEST[127:96] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[127:96)
DEST[159:128] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[159:128])
DEST[191:160] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[191:160])
DEST[223:192] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[223:192])
DEST[255:224] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[255:224])

VCVTPS2DQ (VEX.128 encoded version)
DEST[31:0] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[31:0])
DEST[63:32] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[63:32])
DEST[95:64] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[95:64])
DEST[127:96] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[127:96])
DEST[MAX_VL-1:128] <-0

CVTPS2DQ (128-bit Legacy SSE version)
DEST[31:0] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[31:0])
DEST[63:32] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[63:32])
DEST[95:64] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[95:64])
DEST[127:96] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[127:96])
DEST[MAX_VL-1:128] (unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VCVTPS2DQ __m512i _mm512_cvtps_epi32( __m512 a);
VCVTPS2DQ __m512i _mm512_mask_cvtps_epi32( __m512i s, __mmask16 k, __m512 a);
VCVTPS2DQ __m512i _mm512_maskz_cvtps_epi32( __mmask16 k, __m512 a);
VCVTPS2DQ __m512i _mm512_cvt_roundps_epi32( __m512 a, int r);
VCVTPS2DQ __m512i _mm512_mask_cvt_roundps_epi32( __m512i s, __mmask16 k, __m512 a, int r);
VCVTPS2DQ __m512i _mm512_maskz_cvt_roundps_epi32( __mmask16 k, __m512 a, int r);
VCVTPS2DQ __m256i _mm256_mask_cvtps_epi32( __m256i s, __mmask8 k, __m256 a);
VCVTPS2DQ __m256i _mm256_maskz_cvtps_epi32( __mmask8 k, __m256 a);
VCVTPS2DQ __m128i _mm_mask_cvtps_epi32( __m128i s, __mmask8 k, __m128 a);
VCVTPS2DQ __m128i _mm_maskz_cvtps_epi32( __mmask8 k, __m128 a);
VCVTPS2DQ __ m256i _mm256_cvtps_epi32 (__m256 a)
CVTPS2DQ __m128i _mm_cvtps_epi32 (__m128 a)

SIMD Floating-Point Exceptions
Invalid, Precision

Other Exceptions
VEX-encoded instructions, see Exceptions Type 2;
EVEX-encoded instructions, see Exceptions Type E2.
#UD                   If VEX.vvvv != 1111B or EVEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTPS2PD','CVTPS2PD-Convert Packed Single-Precision Floating-Point Values to Packed Double-Precision
Floating-Point Values
 Opcode/                                  Op /    64/32         CPUID           Description
 Instruction                              En      bit Mode      Feature
                                                  Support       Flag
 0F 5A /r                                 RM      V/V           SSE2            Convert two packed single-precision floating-point values in
 CVTPS2PD xmm1, xmm2/m64                                                        xmm2/m64 to two packed double-precision floating-point
                                                                                values in xmm1.
 VEX.128.0F.WIG 5A /r                     RM      V/V           AVX             Convert two packed single-precision floating-point values in
 VCVTPS2PD xmm1, xmm2/m64                                                       xmm2/m64 to two packed double-precision floating-point
                                                                                values in xmm1.
 VEX.256.0F.WIG 5A /r                     RM      V/V           AVX             Convert four packed single-precision floating-point values
 VCVTPS2PD ymm1, xmm2/m128                                                      in xmm2/m128 to four packed double-precision floating-
                                                                                point values in ymm1.
 EVEX.128.0F.W0 5A /r                     HV      V/V           AVX512VL        Convert two packed single-precision floating-point values in
 VCVTPS2PD xmm1 {k1}{z},                                        AVX512F         xmm2/m64/m32bcst to packed double-precision floating-
 xmm2/m64/m32bcst                                                               point values in xmm1 with writemask k1.
 EVEX.256.0F.W0 5A /r                     HV      V/V           AVX512VL        Convert four packed single-precision floating-point values
 VCVTPS2PD ymm1 {k1}{z},                                                        in xmm2/m128/m32bcst to packed double-precision
 xmm2/m128/m32bcst                                                              floating-point values in ymm1 with writemask k1.
 EVEX.512.0F.W0 5A /r                     HV      V/V           AVX512F         Convert eight packed single-precision floating-point values
 VCVTPS2PD zmm1 {k1}{z},                                                        in ymm2/m256/b32bcst to eight packed double-precision
 ymm2/m256/m32bcst{sae}                                                         floating-point values in zmm1 with writemask k1.



                                                     Instruction Operand Encoding
      Op/En                 Operand 1                      Operand 2                      Operand 3                      Operand 4
       RM                ModRM:reg (w)                   ModRM:r/m (r)                        NA                            NA
       HV                ModRM:reg (w)                   ModRM:r/m (r)                        NA                            NA

Description
Converts two, four or eight packed single-precision floating-point values in the source operand (second operand)
to two, four or eight packed double-precision floating-point values in the destination operand (first operand).
EVEX encoded versions: The source operand is a YMM/XMM/XMM (low 64-bits) register, a 256/128/64-bit memory
location or a 256/128/64-bit vector broadcasted from a 32-bit memory location. The destination operand is a
ZMM/YMM/XMM register conditionally updated with writemask k1.
VEX.256 encoded version: The source operand is an XMM register or 128- bit memory location. The destination
operand is a YMM register. Bits (MAX_VL-1:256) of the corresponding destination ZMM register are zeroed.
VEX.128 encoded version: The source operand is an XMM register or 64- bit memory location. The destination
operand is a XMM register. The upper Bits (MAX_VL-1:128) of the corresponding ZMM register destination are
zeroed.
128-bit Legacy SSE version: The source operand is an XMM register or 64- bit memory location. The destination
operand is an XMM register. The upper Bits (MAX_VL-1:128) of the corresponding ZMM register destination are
unmodified.
Note: VEX.vvvv and EVEX.vvvv are reserved and must be 1111b otherwise instructions will #UD.





                 SRC                                              X3         X2        X1         X0




                 DEST       X3                     X2                   X1                   X0




                                   Figure 3-14. CVTPS2PD (VEX.256 encoded version)


Operation
VCVTPS2PD (EVEX encoded versions) when src operand is a register
(KL, VL) = (2, 128), (4, 256), (8, 512)
FOR j <- 0 TO KL-1
    i <- j * 64
    k <- j * 32
    IF k1[j] OR *no writemask*
         THEN DEST[i+63:i] <-
               Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC[k+31:k])
         ELSE
               IF *merging-masking*                ; merging-masking
                    THEN *DEST[i+63:i] remains unchanged*
                    ELSE                           ; zeroing-masking
                         DEST[i+63:i] <- 0
               FI
    FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VCVTPS2PD (EVEX encoded versions) when src operand is a memory source
(KL, VL) = (2, 128), (4, 256), (8, 512)

FOR j <- 0 TO KL-1
   i <- j * 64
   k <- j * 32
   IF k1[j] OR *no writemask*
        THEN
             IF (EVEX.b = 1)
                  THEN
                       DEST[i+63:i] <-
             Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC[31:0])
                  ELSE
                       DEST[i+63:i] <-
             Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC[k+31:k])
             FI;
        ELSE



              IF *merging-masking*                ; merging-masking
                   THEN *DEST[i+63:i] remains unchanged*
                   ELSE                           ; zeroing-masking
                       DEST[i+63:i] <- 0
              FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VCVTPS2PD (VEX.256 encoded version)
DEST[63:0] <- Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC[31:0])
DEST[127:64] <- Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC[63:32])
DEST[191:128] <- Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC[95:64])
DEST[255:192] <- Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC[127:96)
DEST[MAX_VL-1:256] <- 0

VCVTPS2PD (VEX.128 encoded version)
DEST[63:0] <- Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC[31:0])
DEST[127:64] <- Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC[63:32])
DEST[MAX_VL-1:128] <- 0

CVTPS2PD (128-bit Legacy SSE version)
DEST[63:0] <- Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC[31:0])
DEST[127:64] <- Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC[63:32])
DEST[MAX_VL-1:128] (unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VCVTPS2PD __m512d _mm512_cvtps_pd( __m256 a);
VCVTPS2PD __m512d _mm512_mask_cvtps_pd( __m512d s, __mmask8 k, __m256 a);
VCVTPS2PD __m512d _mm512_maskz_cvtps_pd( __mmask8 k, __m256 a);
VCVTPS2PD __m512d _mm512_cvt_roundps_pd( __m256 a, int sae);
VCVTPS2PD __m512d _mm512_mask_cvt_roundps_pd( __m512d s, __mmask8 k, __m256 a, int sae);
VCVTPS2PD __m512d _mm512_maskz_cvt_roundps_pd( __mmask8 k, __m256 a, int sae);
VCVTPS2PD __m256d _mm256_mask_cvtps_pd( __m256d s, __mmask8 k, __m128 a);
VCVTPS2PD __m256d _mm256_maskz_cvtps_pd( __mmask8 k, __m128a);
VCVTPS2PD __m128d _mm_mask_cvtps_pd( __m128d s, __mmask8 k, __m128 a);
VCVTPS2PD __m128d _mm_maskz_cvtps_pd( __mmask8 k, __m128 a);
VCVTPS2PD __m256d _mm256_cvtps_pd (__m128 a)
CVTPS2PD __m128d _mm_cvtps_pd (__m128 a)

SIMD Floating-Point Exceptions
Invalid, Denormal

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3;
EVEX-encoded instructions, see Exceptions Type E3.
#UD                       If VEX.vvvv != 1111B or EVEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTPS2PI','CVTPS2PI-Convert Packed Single-Precision FP Values to Packed Dword Integers
 Opcode/                                             Op/   64-Bit      Compat/ Description
 Instruction                                         En    Mode        Leg Mode
 0F 2D /r                                            RM    Valid       Valid        Convert two packed single-precision floating-
 CVTPS2PI mm, xmm/m64                                                               point values from xmm/m64 to two packed
                                                                                    signed doubleword integers in mm.



                                              Instruction Operand Encoding
  Op/En              Operand 1                    Operand 2                      Operand 3                      Operand 4
    RM             ModRM:reg (w)                 ModRM:r/m (r)                       NA                             NA

Description
Converts two packed single-precision floating-point values in the source operand (second operand) to two packed
signed doubleword integers in the destination operand (first operand).
The source operand can be an XMM register or a 128-bit memory location. The destination operand is an MMX tech-
nology register. When the source operand is an XMM register, the two single-precision floating-point values are
contained in the low quadword of the register. When a conversion is inexact, the value returned is rounded
according to the rounding control bits in the MXCSR register. If a converted result is larger than the maximum
signed doubleword integer, the floating-point invalid exception is raised, and if this exception is masked, the indef-
inite integer value (80000000H) is returned.
CVTPS2PI causes a transition from x87 FPU to MMX technology operation (that is, the x87 FPU top-of-stack pointer
is set to 0 and the x87 FPU tag word is set to all 0s [valid]). If this instruction is executed while an x87 FPU floating-
point exception is pending, the exception is handled before the CVTPS2PI instruction is executed.
In 64-bit mode, use of the REX.R prefix permits this instruction to access additional registers (XMM8-XMM15).

Operation
DEST[31:0] <- Convert_Single_Precision_Floating_Point_To_Integer(SRC[31:0]);
DEST[63:32] <- Convert_Single_Precision_Floating_Point_To_Integer(SRC[63:32]);

Intel C/C++ Compiler Intrinsic Equivalent
CVTPS2PI:       __m64 _mm_cvtps_pi32(__m128 a)

SIMD Floating-Point Exceptions
Invalid, Precision

Other Exceptions
See Table 22-5, "Exception Conditions for Legacy SIMD/MMX Instructions with XMM and FP Exception," in the
Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 3B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTSD2SI','CVTSD2SI-Convert Scalar Double-Precision Floating-Point Value to Doubleword Integer
 Opcode/                                     Op /    64/32         CPUID          Description
 Instruction                                 En      bit Mode      Feature
                                                     Support       Flag
 F2 0F 2D /r                                 RM      V/V           SSE2           Convert one double-precision floating-point value from
 CVTSD2SI r32, xmm1/m64                                                           xmm1/m64 to one signed doubleword integer r32.
 F2 REX.W 0F 2D /r                           RM      V/N.E.        SSE2           Convert one double-precision floating-point value from
 CVTSD2SI r64, xmm1/m64                                                           xmm1/m64 to one signed quadword integer sign-
                                                                                  extended into r64.
 VEX.128.F2.0F.W0 2D /r                      RM      V/V           AVX            Convert one double-precision floating-point value from
 VCVTSD2SI r32, xmm1/m64                                                          xmm1/m64 to one signed doubleword integer r32.
 VEX.128.F2.0F.W1 2D /r                      RM      V/N.E.1       AVX            Convert one double-precision floating-point value from
 VCVTSD2SI r64, xmm1/m64                                                          xmm1/m64 to one signed quadword integer sign-
                                                                                  extended into r64.
 EVEX.LIG.F2.0F.W0 2D /r                     T1F     V/V           AVX512F        Convert one double-precision floating-point value from
 VCVTSD2SI r32, xmm1/m64{er}                                                      xmm1/m64 to one signed doubleword integer r32.
 EVEX.LIG.F2.0F.W1 2D /r                     T1F     V/N.E.1       AVX512F        Convert one double-precision floating-point value from
 VCVTSD2SI r64, xmm1/m64{er}                                                      xmm1/m64 to one signed quadword integer sign-
                                                                                  extended into r64.
NOTES:
1. VEX.W1/EVEX.W1 in non-64 bit is ignored; the instructions behaves as if the W0 version is used.



                                                     Instruction Operand Encoding
       Op/En                  Operand 1                       Operand 2                   Operand 3                    Operand 4
         RM                ModRM:reg (w)                   ModRM:r/m (r)                        NA                        NA
        T1F                ModRM:reg (w)                   ModRM:r/m (r)                        NA                        NA

Description
Converts a double-precision floating-point value in the source operand (the second operand) to a signed double-
word integer in the destination operand (first operand). The source operand can be an XMM register or a 64-bit
memory location. The destination operand is a general-purpose register. When the source operand is an XMM
register, the double-precision floating-point value is contained in the low quadword of the register.
When a conversion is inexact, the value returned is rounded according to the rounding control bits in the MXCSR
register.
If a converted result exceeds the range limits of signed doubleword integer (in non-64-bit modes or 64-bit mode
with REX.W/VEX.W/EVEX.W=0), the floating-point invalid exception is raised, and if this exception is masked, the
indefinite integer value (80000000H) is returned.
If a converted result exceeds the range limits of signed quadword integer (in 64-bit mode and
REX.W/VEX.W/EVEX.W = 1), the floating-point invalid exception is raised, and if this exception is masked, the
indefinite integer value (80000000_00000000H) is returned.
Legacy SSE instruction: Use of the REX.W prefix promotes the instruction to produce 64-bit data in 64-bit mode.
See the summary chart at the beginning of this section for encoding data and limits.
Note: VEX.vvvv and EVEX.vvvv are reserved and must be 1111b, otherwise instructions will #UD.
Software should ensure VCVTSD2SI is encoded with VEX.L=0. Encoding VCVTSD2SI with VEX.L=1 may encounter
unpredictable behavior across different processor generations.





Operation
VCVTSD2SI (EVEX encoded version)
IF SRC *is register* AND (EVEX.b = 1)
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);
FI;
IF 64-Bit Mode and OperandSize = 64
    THEN      DEST[63:0] <- Convert_Double_Precision_Floating_Point_To_Integer(SRC[63:0]);
    ELSE      DEST[31:0] <- Convert_Double_Precision_Floating_Point_To_Integer(SRC[63:0]);
FI

(V)CVTSD2SI
IF 64-Bit Mode and OperandSize = 64
THEN
    DEST[63:0] <-Convert_Double_Precision_Floating_Point_To_Integer(SRC[63:0]);
ELSE
    DEST[31:0] <-Convert_Double_Precision_Floating_Point_To_Integer(SRC[63:0]);
FI;

Intel C/C++ Compiler Intrinsic Equivalent
VCVTSD2SI int _mm_cvtsd_i32(__m128d);
VCVTSD2SI int _mm_cvt_roundsd_i32(__m128d, int r);
VCVTSD2SI __int64 _mm_cvtsd_i64(__m128d);
VCVTSD2SI __int64 _mm_cvt_roundsd_i64(__m128d, int r);
CVTSD2SI __int64 _mm_cvtsd_si64(__m128d);
CVTSD2SI int _mm_cvtsd_si32(__m128d a)

SIMD Floating-Point Exceptions
Invalid, Precision

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3;
EVEX-encoded instructions, see Exceptions Type E3NF.
#UD                   If VEX.vvvv != 1111B or EVEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTSD2SS','CVTSD2SS-Convert Scalar Double-Precision Floating-Point Value to Scalar Single-Precision
Floating-Point Value
 Opcode/                                      Op /     64/32         CPUID           Description
 Instruction                                  En       bit Mode      Feature
                                                       Support       Flag
 F2 0F 5A /r                                  RM       V/V           SSE2            Convert one double-precision floating-point value in
 CVTSD2SS xmm1, xmm2/m64                                                             xmm2/m64 to one single-precision floating-point value
                                                                                     in xmm1.
 VEX.NDS.128.F2.0F.WIG 5A /r                  RVM      V/V           AVX             Convert one double-precision floating-point value in
 VCVTSD2SS xmm1,xmm2,                                                                xmm3/m64 to one single-precision floating-point value
 xmm3/m64                                                                            and merge with high bits in xmm2.
 EVEX.NDS.LIG.F2.0F.W1 5A /r                  T1S      V/V           AVX512F         Convert one double-precision floating-point value in
 VCVTSD2SS xmm1 {k1}{z}, xmm2,                                                       xmm3/m64 to one single-precision floating-point value
 xmm3/m64{er}                                                                        and merge with high bits in xmm2 under writemask k1.



                                                      Instruction Operand Encoding
      Op/En                  Operand 1                        Operand 2                       Operand 3                      Operand 4
        RM                 ModRM:reg (w)                     ModRM:r/m (r)                         NA                           NA
       RVM                 ModRM:reg (w)                       VEX.vvvv                     ModRM:r/m (r)                       NA
       T1S                 ModRM:reg (w)                      EVEX.vvvv                     ModRM:r/m (r)                       NA

Description
Converts a double-precision floating-point value in the "convert-from" source operand (the second operand in
SSE2 version, otherwise the third operand) to a single-precision floating-point value in the destination operand.
When the "convert-from" operand is an XMM register, the double-precision floating-point value is contained in the
low quadword of the register. The result is stored in the low doubleword of the destination operand. When the
conversion is inexact, the value returned is rounded according to the rounding control bits in the MXCSR register.
128-bit Legacy SSE version: The "convert-from" source operand (the second operand) is an XMM register or
memory location. Bits (MAX_VL-1:32) of the corresponding destination register remain unchanged. The destina-
tion operand is an XMM register.
VEX.128 and EVEX encoded versions: The "convert-from" source operand (the third operand) can be an XMM
register or a 64-bit memory location. The first source and destination operands are XMM registers. Bits (127:32) of
the XMM register destination are copied from the corresponding bits in the first source operand. Bits (MAX_VL-
1:128) of the destination register are zeroed.
EVEX encoded version: the converted result in written to the low doubleword element of the destination under the
writemask.
Software should ensure VCVTSD2SS is encoded with VEX.L=0. Encoding VCVTSD2SS with VEX.L=1 may encounter
unpredictable behavior across different processor generations.





Operation
VCVTSD2SS (EVEX encoded version)
IF (SRC2 *is register*) AND (EVEX.b = 1)
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);
FI;
IF k1[0] or *no writemask*
    THEN      DEST[31:0] <- Convert_Double_Precision_To_Single_Precision_Floating_Point(SRC2[63:0]);
    ELSE
         IF *merging-masking*              ; merging-masking
              THEN *DEST[31:0] remains unchanged*
              ELSE                         ; zeroing-masking
                   THEN DEST[31:0] <- 0
         FI;
FI;
DEST[127:32] <- SRC1[127:32]
DEST[MAX_VL-1:128] <- 0

VCVTSD2SS (VEX.128 encoded version)
DEST[31:0] <-Convert_Double_Precision_To_Single_Precision_Floating_Point(SRC2[63:0]);
DEST[127:32] <-SRC1[127:32]
DEST[MAX_VL-1:128] <-0

CVTSD2SS (128-bit Legacy SSE version)
DEST[31:0] <-Convert_Double_Precision_To_Single_Precision_Floating_Point(SRC[63:0]);
(* DEST[MAX_VL-1:32] Unmodified *)

Intel C/C++ Compiler Intrinsic Equivalent
VCVTSD2SS __m128 _mm_mask_cvtsd_ss(__m128 s, __mmask8 k, __m128 a, __m128d b);
VCVTSD2SS __m128 _mm_maskz_cvtsd_ss( __mmask8 k, __m128 a,__m128d b);
VCVTSD2SS __m128 _mm_cvt_roundsd_ss(__m128 a, __m128d b, int r);
VCVTSD2SS __m128 _mm_mask_cvt_roundsd_ss(__m128 s, __mmask8 k, __m128 a, __m128d b, int r);
VCVTSD2SS __m128 _mm_maskz_cvt_roundsd_ss( __mmask8 k, __m128 a,__m128d b, int r);
CVTSD2SS __m128_mm_cvtsd_ss(__m128 a, __m128d b)

SIMD Floating-Point Exceptions
Overflow, Underflow, Invalid, Precision, Denormal

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3.
EVEX-encoded instructions, see Exceptions Type E3.




');
INSERT INTO "instructions" VALUES('x86_64','CVTSI2SD','CVTSI2SD-Convert Doubleword Integer to Scalar Double-Precision Floating-Point Value
 Opcode/                                           Op /      64/32        CPUID        Description
 Instruction                                       En        bit Mode     Feature
                                                             Support      Flag
 F2 0F 2A /r                                       RM        V/V          SSE2         Convert one signed doubleword integer from
 CVTSI2SD xmm1, r32/m32                                                                r32/m32 to one double-precision floating-point
                                                                                       value in xmm1.
 F2 REX.W 0F 2A /r                                 RM        V/N.E.       SSE2         Convert one signed quadword integer from r/m64
 CVTSI2SD xmm1, r/m64                                                                  to one double-precision floating-point value in
                                                                                       xmm1.
 VEX.NDS.128.F2.0F.W0 2A /r                        RVM       V/V          AVX          Convert one signed doubleword integer from
 VCVTSI2SD xmm1, xmm2, r/m32                                                           r/m32 to one double-precision floating-point
                                                                                       value in xmm1.
 VEX.NDS.128.F2.0F.W1 2A /r                        RVM       V/N.E.1      AVX          Convert one signed quadword integer from r/m64
 VCVTSI2SD xmm1, xmm2, r/m64                                                           to one double-precision floating-point value in
                                                                                       xmm1.
 EVEX.NDS.LIG.F2.0F.W0 2A /r                       T1S       V/V          AVX512F      Convert one signed doubleword integer from
 VCVTSI2SD xmm1, xmm2, r/m32                                                           r/m32 to one double-precision floating-point
                                                                                       value in xmm1.
 EVEX.NDS.LIG.F2.0F.W1 2A /r                       T1S       V/N.E.1      AVX512F      Convert one signed quadword integer from r/m64
 VCVTSI2SD xmm1, xmm2, r/m64{er}                                                       to one double-precision floating-point value in
                                                                                       xmm1.
NOTES:
1. VEX.W1/EVEX.W1 in non-64 bit is ignored; the instructions behaves as if the W0 version is used.



                                                     Instruction Operand Encoding
      Op/En                 Operand 1                      Operand 2                   Operand 3                    Operand 4
        RM                ModRM:reg (w)                   ModRM:r/m (r)                    NA                          NA
       RVM                ModRM:reg (w)                     VEX.vvvv                  ModRM:r/m (r)                    NA
       T1S                ModRM:reg (w)                    EVEX.vvvv                  ModRM:r/m (r)                    NA

Description
Converts a signed doubleword integer (or signed quadword integer if operand size is 64 bits) in the "convert-from"
source operand to a double-precision floating-point value in the destination operand. The result is stored in the low
quadword of the destination operand, and the high quadword left unchanged. When conversion is inexact, the
value returned is rounded according to the rounding control bits in the MXCSR register.
The second source operand can be a general-purpose register or a 32/64-bit memory location. The first source and
destination operands are XMM registers.
128-bit Legacy SSE version: Use of the REX.W prefix promotes the instruction to 64-bit operands. The "convert-
from" source operand (the second operand) is a general-purpose register or memory location. The destination is
an XMM register Bits (MAX_VL-1:64) of the corresponding destination register remain unchanged.
VEX.128 and EVEX encoded versions: The "convert-from" source operand (the third operand) can be a general-
purpose register or a memory location. The first source and destination operands are XMM registers. Bits (127:64)
of the XMM register destination are copied from the corresponding bits in the first source operand. Bits (MAX_VL-
1:128) of the destination register are zeroed.
EVEX.W0 version: attempt to encode this instruction with EVEX embedded rounding is ignored.
VEX.W1 and EVEX.W1 versions: promotes the instruction to use 64-bit input value in 64-bit mode.
Software should ensure VCVTSI2SD is encoded with VEX.L=0. Encoding VCVTSI2SD with VEX.L=1 may encounter
unpredictable behavior across different processor generations.





Operation
VCVTSI2SD (EVEX encoded version)
IF (SRC2 *is register*) AND (EVEX.b = 1)
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);
FI;
IF 64-Bit Mode And OperandSize = 64
THEN
    DEST[63:0] <- Convert_Integer_To_Double_Precision_Floating_Point(SRC2[63:0]);
ELSE
    DEST[63:0] <- Convert_Integer_To_Double_Precision_Floating_Point(SRC2[31:0]);
FI;
DEST[127:64] <- SRC1[127:64]
DEST[MAX_VL-1:128] <- 0

VCVTSI2SD (VEX.128 encoded version)
IF 64-Bit Mode And OperandSize = 64
THEN
    DEST[63:0] <-Convert_Integer_To_Double_Precision_Floating_Point(SRC2[63:0]);
ELSE
    DEST[63:0] <-Convert_Integer_To_Double_Precision_Floating_Point(SRC2[31:0]);
FI;
DEST[127:64] <-SRC1[127:64]
DEST[MAX_VL-1:128] <-0

CVTSI2SD
IF 64-Bit Mode And OperandSize = 64
THEN
    DEST[63:0] <-Convert_Integer_To_Double_Precision_Floating_Point(SRC[63:0]);
ELSE
    DEST[63:0] <-Convert_Integer_To_Double_Precision_Floating_Point(SRC[31:0]);
FI;
DEST[MAX_VL-1:64] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VCVTSI2SD __m128d _mm_cvti32_sd(__m128d s, int a);
VCVTSI2SD __m128d _mm_cvt_roundi32_sd(__m128d s, int a, int r);
VCVTSI2SD __m128d _mm_cvti64_sd(__m128d s, __int64 a);
VCVTSI2SD __m128d _mm_cvt_roundi64_sd(__m128d s, __int64 a, int r);
CVTSI2SD __m128d _mm_cvtsi64_sd(__m128d s, __int64 a);
CVTSI2SD __m128d_mm_cvtsi32_sd(__m128d a, int b)

SIMD Floating-Point Exceptions
Precision

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3 if W1, else Type 5.
EVEX-encoded instructions, see Exceptions Type E3NF if W1, else Type E10NF.




');
INSERT INTO "instructions" VALUES('x86_64','CVTSI2SS','CVTSI2SS-Convert Doubleword Integer to Scalar Single-Precision Floating-Point Value
 Opcode/                                         Op /     64/32         CPUID         Description
 Instruction                                     En       bit Mode      Feature
                                                          Support       Flag
 F3 0F 2A /r                                     RM       V/V           SSE           Convert one signed doubleword integer from r/m32
 CVTSI2SS xmm1, r/m32                                                                 to one single-precision floating-point value in xmm1.
 F3 REX.W 0F 2A /r                               RM       V/N.E.        SSE           Convert one signed quadword integer from r/m64
 CVTSI2SS xmm1, r/m64                                                                 to one single-precision floating-point value in xmm1.
 VEX.NDS.128.F3.0F.W0 2A /r                      RVM      V/V           AVX           Convert one signed doubleword integer from r/m32
 VCVTSI2SS xmm1, xmm2, r/m32                                                          to one single-precision floating-point value in xmm1.
 VEX.NDS.128.F3.0F.W1 2A /r                      RVM      V/N.E.1       AVX           Convert one signed quadword integer from r/m64
 VCVTSI2SS xmm1, xmm2, r/m64                                                          to one single-precision floating-point value in xmm1.
 EVEX.NDS.LIG.F3.0F.W0 2A /r                     T1S      V/V           AVX512F       Convert one signed doubleword integer from r/m32
 VCVTSI2SS xmm1, xmm2, r/m32{er}                                                      to one single-precision floating-point value in xmm1.
 EVEX.NDS.LIG.F3.0F.W1 2A /r                     T1S      V/N.E.1       AVX512F       Convert one signed quadword integer from r/m64
 VCVTSI2SS xmm1, xmm2, r/m64{er}                                                      to one single-precision floating-point value in xmm1.
NOTES:
1. VEX.W1/EVEX.W1 in non-64 bit is ignored; the instructions behaves as if the W0 version is used.



                                                      Instruction Operand Encoding
       Op/En                 Operand 1                      Operand 2                    Operand 3                     Operand 4
        RM                 ModRM:reg (w)                  ModRM:r/m (r)                      NA                            NA
       RVM                 ModRM:reg (w)                     VEX.vvvv                  ModRM:r/m (r)                       NA
        T1S                ModRM:reg (w)                    EVEX.vvvv                  ModRM:r/m (r)                       NA

Description
Converts a signed doubleword integer (or signed quadword integer if operand size is 64 bits) in the "convert-from"
source operand to a single-precision floating-point value in the destination operand (first operand). The "convert-
from" source operand can be a general-purpose register or a memory location. The destination operand is an XMM
register. The result is stored in the low doubleword of the destination operand, and the upper three doublewords
are left unchanged. When a conversion is inexact, the value returned is rounded according to the rounding control
bits in the MXCSR register or the embedded rounding control bits.
128-bit Legacy SSE version: In 64-bit mode, Use of the REX.W prefix promotes the instruction to use 64-bit input
value. The "convert-from" source operand (the second operand) is a general-purpose register or memory location.
Bits (MAX_VL-1:32) of the corresponding destination register remain unchanged.
VEX.128 and EVEX encoded versions: The "convert-from" source operand (the third operand) can be a general-
purpose register or a memory location. The first source and destination operands are XMM registers. Bits (127:32)
of the XMM register destination are copied from corresponding bits in the first source operand. Bits (MAX_VL-
1:128) of the destination register are zeroed.
EVEX encoded version: the converted result in written to the low doubleword element of the destination under the
writemask.
Software should ensure VCVTSI2SS is encoded with VEX.L=0. Encoding VCVTSI2SS with VEX.L=1 may encounter
unpredictable behavior across different processor generations.





Operation
VCVTSI2SS (EVEX encoded version)
IF (SRC2 *is register*) AND (EVEX.b = 1)
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);
FI;
IF 64-Bit Mode And OperandSize = 64
THEN
    DEST[31:0] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[63:0]);
ELSE
    DEST[31:0] <- Convert_Integer_To_Single_Precision_Floating_Point(SRC[31:0]);
FI;
DEST[127:32] <- SRC1[127:32]
DEST[MAX_VL-1:128] <- 0

VCVTSI2SS (VEX.128 encoded version)
IF 64-Bit Mode And OperandSize = 64
THEN
    DEST[31:0] <-Convert_Integer_To_Single_Precision_Floating_Point(SRC[63:0]);
ELSE
    DEST[31:0] <-Convert_Integer_To_Single_Precision_Floating_Point(SRC[31:0]);
FI;
DEST[127:32] <-SRC1[127:32]
DEST[MAX_VL-1:128] <-0

CVTSI2SS (128-bit Legacy SSE version)
IF 64-Bit Mode And OperandSize = 64
THEN
    DEST[31:0] <-Convert_Integer_To_Single_Precision_Floating_Point(SRC[63:0]);
ELSE
    DEST[31:0] <-Convert_Integer_To_Single_Precision_Floating_Point(SRC[31:0]);
FI;
DEST[MAX_VL-1:32] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VCVTSI2SS __m128 _mm_cvti32_ss(__m128 s, int a);
VCVTSI2SS __m128 _mm_cvt_roundi32_ss(__m128 s, int a, int r);
VCVTSI2SS __m128 _mm_cvti64_ss(__m128 s, __int64 a);
VCVTSI2SS __m128 _mm_cvt_roundi64_ss(__m128 s, __int64 a, int r);
CVTSI2SS __m128 _mm_cvtsi64_ss(__m128 s, __int64 a);
CVTSI2SS __m128 _mm_cvtsi32_ss(__m128 a, int b);

SIMD Floating-Point Exceptions
Precision

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3.
EVEX-encoded instructions, see Exceptions Type E3NF.




');
INSERT INTO "instructions" VALUES('x86_64','CVTSS2SD','CVTSS2SD-Convert Scalar Single-Precision Floating-Point Value to Scalar Double-Precision
Floating-Point Value
 Opcode/                                      Op /    64/32          CPUID          Description
 Instruction                                  En      bit Mode       Feature
                                                      Support        Flag
 F3 0F 5A /r                                  RM      V/V            SSE2           Convert one single-precision floating-point value in
 CVTSS2SD xmm1, xmm2/m32                                                            xmm2/m32 to one double-precision floating-point value
                                                                                    in xmm1.
 VEX.NDS.128.F3.0F.WIG 5A /r                  RVM      V/V           AVX            Convert one single-precision floating-point value in
 VCVTSS2SD xmm1, xmm2,                                                              xmm3/m32 to one double-precision floating-point value
 xmm3/m32                                                                           and merge with high bits of xmm2.
 EVEX.NDS.LIG.F3.0F.W0 5A /r                  T1S     V/V            AVX512F        Convert one single-precision floating-point value in
 VCVTSS2SD xmm1 {k1}{z}, xmm2,                                                      xmm3/m32 to one double-precision floating-point value
 xmm3/m32{sae}                                                                      and merge with high bits of xmm2 under writemask k1.



                                                      Instruction Operand Encoding
       Op/En                  Operand 1                       Operand 2                      Operand 3                      Operand 4
        RM                  ModRM:reg (w)                    ModRM:r/m (r)                        NA                           NA
        RVM                 ModRM:reg (w)                      VEX.vvvv                    ModRM:r/m (r)                       NA
        T1S                 ModRM:reg (w)                     EVEX.vvvv                    ModRM:r/m (r)                       NA

Description
Converts a single-precision floating-point value in the "convert-from" source operand to a double-precision
floating-point value in the destination operand. When the "convert-from" source operand is an XMM register, the
single-precision floating-point value is contained in the low doubleword of the register. The result is stored in the
low quadword of the destination operand.
128-bit Legacy SSE version: The "convert-from" source operand (the second operand) is an XMM register or
memory location. Bits (MAX_VL-1:64) of the corresponding destination register remain unchanged. The destina-
tion operand is an XMM register.
VEX.128 and EVEX encoded versions: The "convert-from" source operand (the third operand) can be an XMM
register or a 32-bit memory location. The first source and destination operands are XMM registers. Bits (127:64) of
the XMM register destination are copied from the corresponding bits in the first source operand. Bits (MAX_VL-
1:128) of the destination register are zeroed.
Software should ensure VCVTSS2SD is encoded with VEX.L=0. Encoding VCVTSS2SD with VEX.L=1 may encounter
unpredictable behavior across different processor generations.

Operation
VCVTSS2SD (EVEX encoded version)
IF k1[0] or *no writemask*
    THEN      DEST[63:0] <- Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC2[31:0]);
    ELSE
         IF *merging-masking*               ; merging-masking
              THEN *DEST[63:0] remains unchanged*
              ELSE                          ; zeroing-masking
                  THEN DEST[63:0] = 0
         FI;
FI;
DEST[127:64] <- SRC1[127:64]
DEST[MAX_VL-1:128] <- 0





VCVTSS2SD (VEX.128 encoded version)
DEST[63:0] <-Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC2[31:0])
DEST[127:64] <-SRC1[127:64]
DEST[MAX_VL-1:128] <-0

CVTSS2SD (128-bit Legacy SSE version)
DEST[63:0] <-Convert_Single_Precision_To_Double_Precision_Floating_Point(SRC[31:0]);
DEST[MAX_VL-1:64] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VCVTSS2SD __m128d _mm_cvt_roundss_sd(__m128d a, __m128 b, int r);
VCVTSS2SD __m128d _mm_mask_cvt_roundss_sd(__m128d s, __mmask8 m, __m128d a,__m128 b, int r);
VCVTSS2SD __m128d _mm_maskz_cvt_roundss_sd(__mmask8 k, __m128d a, __m128 a, int r);
VCVTSS2SD __m128d _mm_mask_cvtss_sd(__m128d s, __mmask8 m, __m128d a,__m128 b);
VCVTSS2SD __m128d _mm_maskz_cvtss_sd(__mmask8 m, __m128d a,__m128 b);
CVTSS2SD __m128d_mm_cvtss_sd(__m128d a, __m128 a);

SIMD Floating-Point Exceptions
Invalid, Denormal

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3.
EVEX-encoded instructions, see Exceptions Type E3.




');
INSERT INTO "instructions" VALUES('x86_64','CVTSS2SI','CVTSS2SI-Convert Scalar Single-Precision Floating-Point Value to Doubleword Integer
 Opcode/                                    Op /     64/32          CPUID         Description
 Instruction                                En       bit Mode       Feature
                                                     Support        Flag
 F3 0F 2D /r                                RM       V/V            SSE           Convert one single-precision floating-point value from
 CVTSS2SI r32, xmm1/m32                                                           xmm1/m32 to one signed doubleword integer in r32.
 F3 REX.W 0F 2D /r                          RM       V/N.E.         SSE           Convert one single-precision floating-point value from
 CVTSS2SI r64, xmm1/m32                                                           xmm1/m32 to one signed quadword integer in r64.
 VEX.128.F3.0F.W0 2D /r                     RM       V/V            AVX           Convert one single-precision floating-point value from
 VCVTSS2SI r32, xmm1/m32                                                          xmm1/m32 to one signed doubleword integer in r32.
 VEX.128.F3.0F.W1 2D /r                     RM       V/N.E.1        AVX           Convert one single-precision floating-point value from
 VCVTSS2SI r64, xmm1/m32                                                          xmm1/m32 to one signed quadword integer in r64.
 EVEX.LIG.F3.0F.W0 2D /r                    T1F      V/V            AVX512F       Convert one single-precision floating-point value from
 VCVTSS2SI r32, xmm1/m32{er}                                                      xmm1/m32 to one signed doubleword integer in r32.
 EVEX.LIG.F3.0F.W1 2D /r                    T1F      V/N.E.1        AVX512F       Convert one single-precision floating-point value from
 VCVTSS2SI r64, xmm1/m32{er}                                                      xmm1/m32 to one signed quadword integer in r64.
NOTES:
1. VEX.W1/EVEX.W1 in non-64 bit is ignored; the instructions behaves as if the W0 version is used.



                                                     Instruction Operand Encoding
       Op/En                  Operand 1                        Operand 2                   Operand 3                    Operand 4
        RM                  ModRM:reg (w)                     ModRM:r/m (r)                     NA                          NA
        T1F                 ModRM:reg (w)                     ModRM:r/m (r)                     NA                          NA

Description
Converts a single-precision floating-point value in the source operand (the second operand) to a signed double-
word integer (or signed quadword integer if operand size is 64 bits) in the destination operand (the first operand).
The source operand can be an XMM register or a memory location. The destination operand is a general-purpose
register. When the source operand is an XMM register, the single-precision floating-point value is contained in the
low doubleword of the register.
When a conversion is inexact, the value returned is rounded according to the rounding control bits in the MXCSR
register or the embedded rounding control bits. If a converted result cannot be represented in the destination
format, the floating-point invalid exception is raised, and if this exception is masked, the indefinite integer value
(2w-1, where w represents the number of bits in the destination format) is returned.
Legacy SSE instructions: In 64-bit mode, Use of the REX.W prefix promotes the instruction to produce 64-bit data.
See the summary chart at the beginning of this section for encoding data and limits.
VEX.W1 and EVEX.W1 versions: promotes the instruction to produce 64-bit data in 64-bit mode.
Note: VEX.vvvv and EVEX.vvvv are reserved and must be 1111b, otherwise instructions will #UD.
Software should ensure VCVTSS2SI is encoded with VEX.L=0. Encoding VCVTSS2SI with VEX.L=1 may encounter
unpredictable behavior across different processor generations.





Operation
VCVTSS2SI (EVEX encoded version)
IF (SRC *is register*) AND (EVEX.b = 1)
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);
FI;
IF 64-bit Mode and OperandSize = 64
THEN
    DEST[63:0] <- Convert_Single_Precision_Floating_Point_To_Integer(SRC[31:0]);
ELSE
    DEST[31:0] <- Convert_Single_Precision_Floating_Point_To_Integer(SRC[31:0]);
FI;

(V)CVTSS2SI (Legacy and VEX.128 encoded version)
IF 64-bit Mode and OperandSize = 64
THEN
    DEST[63:0] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[31:0]);
ELSE
    DEST[31:0] <-Convert_Single_Precision_Floating_Point_To_Integer(SRC[31:0]);
FI;

Intel C/C++ Compiler Intrinsic Equivalent
VCVTSS2SI int _mm_cvtss_i32( __m128 a);
VCVTSS2SI int _mm_cvt_roundss_i32( __m128 a, int r);
VCVTSS2SI __int64 _mm_cvtss_i64( __m128 a);
VCVTSS2SI __int64 _mm_cvt_roundss_i64( __m128 a, int r);

SIMD Floating-Point Exceptions
Invalid, Precision

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3; additionally
#UD                   If VEX.vvvv != 1111B.
EVEX-encoded instructions, see Exceptions Type E3NF.




');
INSERT INTO "instructions" VALUES('x86_64','CVTTPD2DQ','CVTTPD2DQ-Convert with Truncation Packed Double-Precision Floating-Point Values to
Packed Doubleword Integers
 Opcode/                                    Op /    64/32         CPUID          Description
 Instruction                                En      bit Mode      Feature
                                                    Support       Flag
 66 0F E6 /r                                RM      V/V           SSE2           Convert two packed double-precision floating-point
 CVTTPD2DQ xmm1, xmm2/m128                                                       values in xmm2/mem to two signed doubleword
                                                                                 integers in xmm1 using truncation.
 VEX.128.66.0F.WIG E6 /r                    RM      V/V           AVX            Convert two packed double-precision floating-point
 VCVTTPD2DQ xmm1, xmm2/m128                                                      values in xmm2/mem to two signed doubleword
                                                                                 integers in xmm1 using truncation.
 VEX.256.66.0F.WIG E6 /r                    RM      V/V           AVX            Convert four packed double-precision floating-point
 VCVTTPD2DQ xmm1, ymm2/m256                                                      values in ymm2/mem to four signed doubleword
                                                                                 integers in xmm1 using truncation.
 EVEX.128.66.0F.W1 E6 /r                    FV      V/V           AVX512VL       Convert two packed double-precision floating-point
 VCVTTPD2DQ xmm1 {k1}{z},                                         AVX512F        values in xmm2/m128/m64bcst to two signed
 xmm2/m128/m64bcst                                                               doubleword integers in xmm1 using truncation subject
                                                                                 to writemask k1.
 EVEX.256.66.0F.W1 E6 /r                    FV      V/V           AVX512VL       Convert four packed double-precision floating-point
 VCVTTPD2DQ xmm1 {k1}{z},                                         AVX512F        values in ymm2/m256/m64bcst to four signed
 ymm2/m256/m64bcst                                                               doubleword integers in xmm1 using truncation subject
                                                                                 to writemask k1.
 EVEX.512.66.0F.W1 E6 /r                    FV      V/V           AVX512F        Convert eight packed double-precision floating-point
 VCVTTPD2DQ ymm1 {k1}{z},                                                        values in zmm2/m512/m64bcst to eight signed
 zmm2/m512/m64bcst{sae}                                                          doubleword integers in ymm1 using truncation subject
                                                                                 to writemask k1.



                                                    Instruction Operand Encoding
      Op/En                Operand 1                      Operand 2                    Operand 3                   Operand 4
        RM               ModRM:reg (w)                 ModRM:r/m (r)                       NA                          NA
        FV               ModRM:reg (w)                 ModRM:r/m (r)                       NA                          NA

Description
Converts two, four or eight packed double-precision floating-point values in the source operand (second operand)
to two, four or eight packed signed doubleword integers in the destination operand (first operand).
When a conversion is inexact, a truncated (round toward zero) value is returned. If a converted result is larger than
the maximum signed doubleword integer, the floating-point invalid exception is raised, and if this exception is
masked, the indefinite integer value (80000000H) is returned.
EVEX encoded versions: The source operand is a ZMM/YMM/XMM register, a 512/256/128-bit memory location, or
a 512/256/128-bit vector broadcasted from a 64-bit memory location. The destination operand is a
YMM/XMM/XMM (low 64 bits) register conditionally updated with writemask k1. The upper bits (MAX_VL-1:256) of
the corresponding destination are zeroed.
VEX.256 encoded version: The source operand is a YMM register or 256- bit memory location. The destination
operand is an XMM register. The upper bits (MAX_VL-1:128) of the corresponding ZMM register destination are
zeroed.
VEX.128 encoded version: The source operand is an XMM register or 128- bit memory location. The destination
operand is a XMM register. The upper bits (MAX_VL-1:64) of the corresponding ZMM register destination are
zeroed.
128-bit Legacy SSE version: The source operand is an XMM register or 128- bit memory location. The destination
operand is an XMM register. The upper bits (MAX_VL-1:128) of the corresponding ZMM register destination are
unmodified.
Note: VEX.vvvv and EVEX.vvvv are reserved and must be 1111b, otherwise instructions will #UD.



                SRC        X3                     X2                  X1                  X0




                DEST                    0                       X3         X2       X1         X0




                                 Figure 3-15. VCVTTPD2DQ (VEX.256 encoded version)


Operation
VCVTTPD2DQ (EVEX encoded versions) when src operand is a register
(KL, VL) = (2, 128), (4, 256), (8, 512)

FOR j <- 0 TO KL-1
   i <- j * 32
   k <- j * 64
   IF k1[j] OR *no writemask*
        THEN DEST[i+31:i] <-
             Convert_Double_Precision_Floating_Point_To_Integer_Truncate(SRC[k+63:k])
        ELSE
             IF *merging-masking*                ; merging-masking
                  THEN *DEST[i+31:i] remains unchanged*
                  ELSE                           ; zeroing-masking
                       DEST[i+31:i] <- 0
             FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL/2] <- 0





VCVTTPD2DQ (EVEX encoded versions) when src operand is a memory source
(KL, VL) = (2, 128), (4, 256), (8, 512)

FOR j <- 0 TO KL-1
   i <- j * 32
   k <- j * 64
   IF k1[j] OR *no writemask*
        THEN
             IF (EVEX.b = 1)
                  THEN
                       DEST[i+31:i] <-
             Convert_Double_Precision_Floating_Point_To_Integer_Truncate(SRC[63:0])
                  ELSE
                       DEST[i+31:i] <-
             Convert_Double_Precision_Floating_Point_To_Integer_Truncate(SRC[k+63:k])
             FI;
        ELSE
             IF *merging-masking*                ; merging-masking
                  THEN *DEST[i+31:i] remains unchanged*
                  ELSE                           ; zeroing-masking
                       DEST[i+31:i] <- 0
             FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL/2] <- 0

VCVTTPD2DQ (VEX.256 encoded version)
DEST[31:0] <-Convert_Double_Precision_Floating_Point_To_Integer_Truncate(SRC[63:0])
DEST[63:32] <-Convert_Double_Precision_Floating_Point_To_Integer_Truncate(SRC[127:64])
DEST[95:64] <-Convert_Double_Precision_Floating_Point_To_Integer_Truncate(SRC[191:128])
DEST[127:96] <-Convert_Double_Precision_Floating_Point_To_Integer_Truncate(SRC[255:192)
DEST[MAX_VL-1:128]<-0

VCVTTPD2DQ (VEX.128 encoded version)
DEST[31:0] <-Convert_Double_Precision_Floating_Point_To_Integer_Truncate(SRC[63:0])
DEST[63:32] <-Convert_Double_Precision_Floating_Point_To_Integer_Truncate(SRC[127:64])
DEST[MAX_VL-1:64]<-0

CVTTPD2DQ (128-bit Legacy SSE version)
DEST[31:0] <-Convert_Double_Precision_Floating_Point_To_Integer_Truncate(SRC[63:0])
DEST[63:32] <-Convert_Double_Precision_Floating_Point_To_Integer_Truncate(SRC[127:64])
DEST[127:64] <-0
DEST[MAX_VL-1:128] (unmodified)





Intel C/C++ Compiler Intrinsic Equivalent
VCVTTPD2DQ __m256i _mm512_cvttpd_epi32( __m512d a);
VCVTTPD2DQ __m256i _mm512_mask_cvttpd_epi32( __m256i s, __mmask8 k, __m512d a);
VCVTTPD2DQ __m256i _mm512_maskz_cvttpd_epi32( __mmask8 k, __m512d a);
VCVTTPD2DQ __m256i _mm512_cvtt_roundpd_epi32( __m512d a, int sae);
VCVTTPD2DQ __m256i _mm512_mask_cvtt_roundpd_epi32( __m256i s, __mmask8 k, __m512d a, int sae);
VCVTTPD2DQ __m256i _mm512_maskz_cvtt_roundpd_epi32( __mmask8 k, __m512d a, int sae);
VCVTTPD2DQ __m128i _mm256_mask_cvttpd_epi32( __m128i s, __mmask8 k, __m256d a);
VCVTTPD2DQ __m128i _mm256_maskz_cvttpd_epi32( __mmask8 k, __m256d a);
VCVTTPD2DQ __m128i _mm_mask_cvttpd_epi32( __m128i s, __mmask8 k, __m128d a);
VCVTTPD2DQ __m128i _mm_maskz_cvttpd_epi32( __mmask8 k, __m128d a);
VCVTTPD2DQ __m128i _mm256_cvttpd_epi32 (__m256d src);
CVTTPD2DQ __m128i _mm_cvttpd_epi32 (__m128d src);

SIMD Floating-Point Exceptions
Invalid, Precision

Other Exceptions
VEX-encoded instructions, see Exceptions Type 2;
EVEX-encoded instructions, see Exceptions Type E2.
#UD                  If VEX.vvvv != 1111B or EVEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTTPD2PI','CVTTPD2PI-Convert with Truncation Packed Double-Precision FP Values to Packed Dword
Integers
 Opcode/                                                  Op/   64-Bit       Compat/ Description
 Instruction                                              En    Mode         Leg Mode
 66 0F 2C /r                                              RM    Valid        Valid        Convert two packer double-precision floating-
 CVTTPD2PI mm, xmm/m128                                                                   point values from xmm/m128 to two packed
                                                                                          signed doubleword integers in mm using
                                                                                          truncation.



                                                   Instruction Operand Encoding
   Op/En              Operand 1                        Operand 2                      Operand 3                     Operand 4
    RM              ModRM:reg (w)                    ModRM:r/m (r)                        NA                           NA

Description
Converts two packed double-precision floating-point values in the source operand (second operand) to two packed
signed doubleword integers in the destination operand (first operand). The source operand can be an XMM register
or a 128-bit memory location. The destination operand is an MMX technology register.
When a conversion is inexact, a truncated (round toward zero) result is returned. If a converted result is larger
than the maximum signed doubleword integer, the floating-point invalid exception is raised, and if this exception is
masked, the indefinite integer value (80000000H) is returned.
This instruction causes a transition from x87 FPU to MMX technology operation (that is, the x87 FPU top-of-stack
pointer is set to 0 and the x87 FPU tag word is set to all 0s [valid]). If this instruction is executed while an x87 FPU
floating-point exception is pending, the exception is handled before the CVTTPD2PI instruction is executed.
In 64-bit mode, use of the REX.R prefix permits this instruction to access additional registers (XMM8-XMM15).

Operation
DEST[31:0] <- Convert_Double_Precision_Floating_Point_To_Integer32_Truncate(SRC[63:0]);
DEST[63:32] <- Convert_Double_Precision_Floating_Point_To_Integer32_
               Truncate(SRC[127:64]);

Intel C/C++ Compiler Intrinsic Equivalent
CVTTPD1PI:       __m64 _mm_cvttpd_pi32(__m128d a)

SIMD Floating-Point Exceptions
Invalid, Precision

Other Mode Exceptions
See Table 22-4, "Exception Conditions for Legacy SIMD/MMX Instructions with FP Exception and 16-Byte Align-
ment," in the Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 3B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTTPS2DQ','CVTTPS2DQ-Convert with Truncation Packed Single-Precision Floating-Point Values to Packed
Signed Doubleword Integer Values
 Opcode/                                 Op /    64/32         CPUID          Description
 Instruction                             En      bit Mode      Feature
                                                 Support       Flag
 F3 0F 5B /r                             RM      V/V           SSE2           Convert four packed single-precision floating-point
 CVTTPS2DQ xmm1, xmm2/m128                                                    values from xmm2/mem to four packed signed
                                                                              doubleword values in xmm1 using truncation.
 VEX.128.F3.0F.WIG 5B /r                 RM      V/V           AVX            Convert four packed single-precision floating-point
 VCVTTPS2DQ xmm1, xmm2/m128                                                   values from xmm2/mem to four packed signed
                                                                              doubleword values in xmm1 using truncation.
 VEX.256.F3.0F.WIG 5B /r                 RM      V/V           AVX            Convert eight packed single-precision floating-point
 VCVTTPS2DQ ymm1, ymm2/m256                                                   values from ymm2/mem to eight packed signed
                                                                              doubleword values in ymm1 using truncation.
 EVEX.128.F3.0F.W0 5B /r                 FV      V/V           AVX512VL       Convert four packed single precision floating-point
 VCVTTPS2DQ xmm1 {k1}{z},                                      AVX512F        values from xmm2/m128/m32bcst to four packed
 xmm2/m128/m32bcst                                                            signed doubleword values in xmm1 using truncation
                                                                              subject to writemask k1.
 EVEX.256.F3.0F.W0 5B /r                 FV      V/V           AVX512VL       Convert eight packed single precision floating-point
 VCVTTPS2DQ ymm1 {k1}{z},                                      AVX512F        values from ymm2/m256/m32bcst to eight packed
 ymm2/m256/m32bcst                                                            signed doubleword values in ymm1 using truncation
                                                                              subject to writemask k1.
 EVEX.512.F3.0F.W0 5B /r                 FV      V/V           AVX512F        Convert sixteen packed single-precision floating-point
 VCVTTPS2DQ zmm1 {k1}{z},                                                     values from zmm2/m512/m32bcst to sixteen packed
 zmm2/m512/m32bcst {sae}                                                      signed doubleword values in zmm1 using truncation
                                                                              subject to writemask k1.



                                                Instruction Operand Encoding
     Op/En            Operand 1                     Operand 2                       Operand 3                       Operand 4
      RM            ModRM:reg (w)                  ModRM:r/m (r)                        NA                              NA
      FV            ModRM:reg (w)                  ModRM:r/m (r)                        NA                              NA

Description
Converts four, eight or sixteen packed single-precision floating-point values in the source operand to four, eight or
sixteen signed doubleword integers in the destination operand.
When a conversion is inexact, a truncated (round toward zero) value is returned. If a converted result is larger than
the maximum signed doubleword integer, the floating-point invalid exception is raised, and if this exception is
masked, the indefinite integer value (80000000H) is returned.
EVEX encoded versions: The source operand is a ZMM/YMM/XMM register, a 512/256/128-bit memory location or
a 512/256/128-bit vector broadcasted from a 32-bit memory location. The destination operand is a
ZMM/YMM/XMM register conditionally updated with writemask k1.
VEX.256 encoded version: The source operand is a YMM register or 256- bit memory location. The destination
operand is a YMM register. The upper bits (MAX_VL-1:256) of the corresponding ZMM register destination are
zeroed.
VEX.128 encoded version: The source operand is an XMM register or 128- bit memory location. The destination
operand is a XMM register. The upper bits (MAX_VL-1:128) of the corresponding ZMM register destination are
zeroed.
128-bit Legacy SSE version: The source operand is an XMM register or 128- bit memory location. The destination
operand is an XMM register. The upper bits (MAX_VL-1:128) of the corresponding ZMM register destination are
unmodified.
Note: VEX.vvvv and EVEX.vvvv are reserved and must be 1111b otherwise instructions will #UD.




Operation
VCVTTPS2DQ (EVEX encoded versions) when src operand is a register
(KL, VL) = (4, 128), (8, 256), (16, 512)

FOR j <- 0 TO KL-1
   i <- j * 32
   IF k1[j] OR *no writemask*
        THEN DEST[i+31:i] <-
             Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[i+31:i])
        ELSE
             IF *merging-masking*                ; merging-masking
                  THEN *DEST[i+31:i] remains unchanged*
                  ELSE                           ; zeroing-masking
                       DEST[i+31:i] <- 0
             FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VCVTTPS2DQ (EVEX encoded versions) when src operand is a memory source
(KL, VL) = (4, 128), (8, 256), (16, 512)

FOR j <- 0 TO 15
   i <- j * 32
   IF k1[j] OR *no writemask*
        THEN
             IF (EVEX.b = 1)
                  THEN
                       DEST[i+31:i] <-
             Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[31:0])
                  ELSE
                       DEST[i+31:i] <-
             Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[i+31:i])
             FI;
        ELSE
             IF *merging-masking*                ; merging-masking
                  THEN *DEST[i+31:i] remains unchanged*
                  ELSE                           ; zeroing-masking
                       DEST[i+31:i] <- 0
             FI
   FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VCVTTPS2DQ (VEX.256 encoded version)
DEST[31:0] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[31:0])
DEST[63:32] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[63:32])
DEST[95:64] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[95:64])
DEST[127:96] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[127:96)
DEST[159:128] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[159:128])
DEST[191:160] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[191:160])
DEST[223:192] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[223:192])
DEST[255:224] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[255:224])



VCVTTPS2DQ (VEX.128 encoded version)
DEST[31:0] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[31:0])
DEST[63:32] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[63:32])
DEST[95:64] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[95:64])
DEST[127:96] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[127:96])
DEST[MAX_VL-1:128] <-0

CVTTPS2DQ (128-bit Legacy SSE version)
DEST[31:0] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[31:0])
DEST[63:32] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[63:32])
DEST[95:64] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[95:64])
DEST[127:96] <-Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[127:96])
DEST[MAX_VL-1:128] (unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VCVTTPS2DQ __m512i _mm512_cvttps_epi32( __m512 a);
VCVTTPS2DQ __m512i _mm512_mask_cvttps_epi32( __m512i s, __mmask16 k, __m512 a);
VCVTTPS2DQ __m512i _mm512_maskz_cvttps_epi32( __mmask16 k, __m512 a);
VCVTTPS2DQ __m512i _mm512_cvtt_roundps_epi32( __m512 a, int sae);
VCVTTPS2DQ __m512i _mm512_mask_cvtt_roundps_epi32( __m512i s, __mmask16 k, __m512 a, int sae);
VCVTTPS2DQ __m512i _mm512_maskz_cvtt_roundps_epi32( __mmask16 k, __m512 a, int sae);
VCVTTPS2DQ __m256i _mm256_mask_cvttps_epi32( __m256i s, __mmask8 k, __m256 a);
VCVTTPS2DQ __m256i _mm256_maskz_cvttps_epi32( __mmask8 k, __m256 a);
VCVTTPS2DQ __m128i _mm_mask_cvttps_epi32( __m128i s, __mmask8 k, __m128 a);
VCVTTPS2DQ __m128i _mm_maskz_cvttps_epi32( __mmask8 k, __m128 a);
VCVTTPS2DQ __m256i _mm256_cvttps_epi32 (__m256 a)
CVTTPS2DQ __m128i _mm_cvttps_epi32 (__m128 a)

SIMD Floating-Point Exceptions
Invalid, Precision

Other Exceptions
VEX-encoded instructions, see Exceptions Type 2; additionally
EVEX-encoded instructions, see Exceptions Type E2.
#UD                   If VEX.vvvv != 1111B or EVEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTTPS2PI','CVTTPS2PI-Convert with Truncation Packed Single-Precision FP Values to Packed Dword
Integers
 Opcode/                                                   Op/    64-Bit       Compat/ Description
 Instruction                                               En     Mode         Leg Mode
 0F 2C /r                                                  RM     Valid        Valid       Convert two single-precision floating-point
 CVTTPS2PI mm, xmm/m64                                                                     values from xmm/m64 to two signed
                                                                                           doubleword signed integers in mm using
                                                                                           truncation.



                                                    Instruction Operand Encoding
   Op/En              Operand 1                       Operand 2                        Operand 3                     Operand 4
    RM              ModRM:reg (w)                   ModRM:r/m (r)                          NA                            NA

Description
Converts two packed single-precision floating-point values in the source operand (second operand) to two packed
signed doubleword integers in the destination operand (first operand). The source operand can be an XMM register
or a 64-bit memory location. The destination operand is an MMX technology register. When the source operand is
an XMM register, the two single-precision floating-point values are contained in the low quadword of the register.
When a conversion is inexact, a truncated (round toward zero) result is returned. If a converted result is larger
than the maximum signed doubleword integer, the floating-point invalid exception is raised, and if this exception is
masked, the indefinite integer value (80000000H) is returned.
This instruction causes a transition from x87 FPU to MMX technology operation (that is, the x87 FPU top-of-stack
pointer is set to 0 and the x87 FPU tag word is set to all 0s [valid]). If this instruction is executed while an x87 FPU
floating-point exception is pending, the exception is handled before the CVTTPS2PI instruction is executed.
In 64-bit mode, use of the REX.R prefix permits this instruction to access additional registers (XMM8-XMM15).

Operation
DEST[31:0] <- Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[31:0]);
DEST[63:32] <- Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[63:32]);

Intel C/C++ Compiler Intrinsic Equivalent
CVTTPS2PI:        __m64 _mm_cvttps_pi32(__m128 a)

SIMD Floating-Point Exceptions
Invalid, Precision

Other Exceptions
See Table 22-5, "Exception Conditions for Legacy SIMD/MMX Instructions with XMM and FP Exception," in the
Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 3B.




');
INSERT INTO "instructions" VALUES('x86_64','CVTTSD2SI','CVTTSD2SI-Convert with Truncation Scalar Double-Precision Floating-Point Value to Signed
Integer
 Opcode/                              Op /    64/32         CPUID         Description
 Instruction                          En      bit Mode      Feature
                                              Support       Flag
 F2 0F 2C /r                          RM      V/V           SSE2          Convert one double-precision floating-point value from
 CVTTSD2SI r32, xmm1/m64                                                  xmm1/m64 to one signed doubleword integer in r32
                                                                          using truncation.
 F2 REX.W 0F 2C /r                    RM      V/N.E.        SSE2          Convert one double-precision floating-point value from
 CVTTSD2SI r64, xmm1/m64                                                  xmm1/m64 to one signed quadword integer in r64
                                                                          using truncation.
 VEX.128.F2.0F.W0 2C /r               RM      V/V           AVX           Convert one double-precision floating-point value from
 VCVTTSD2SI r32, xmm1/m64                                                 xmm1/m64 to one signed doubleword integer in r32
                                                                          using truncation.
 VEX.128.F2.0F.W1 2C /r               T1F     V/N.E.1       AVX           Convert one double-precision floating-point value from
 VCVTTSD2SI r64, xmm1/m64                                                 xmm1/m64 to one signed quadword integer in r64
                                                                          using truncation.
 EVEX.LIG.F2.0F.W0 2C /r              T1F     V/V           AVX512F       Convert one double-precision floating-point value from
 VCVTTSD2SI r32, xmm1/m64{sae}                                            xmm1/m64 to one signed doubleword integer in r32
                                                                          using truncation.
 EVEX.LIG.F2.0F.W1 2C /r              T1F     V/N.E.1       AVX512F       Convert one double-precision floating-point value from
 VCVTTSD2SI r64, xmm1/m64{sae}                                            xmm1/m64 to one signed quadword integer in r64
                                                                          using truncation.
NOTES:
1. For this specific instruction, VEX.W/EVEX.W in non-64 bit is ignored; the instructions behaves as if the W0 ver-
  sion is used.



                                             Instruction Operand Encoding
     Op/En            Operand 1                 Operand 2                       Operand 3                       Operand 4
      RM            ModRM:reg (w)              ModRM:r/m (r)                        NA                              NA
      T1F           ModRM:reg (w)              ModRM:r/m (r)                        NA                              NA

Description
Converts a double-precision floating-point value in the source operand (the second operand) to a signed double-
word integer (or signed quadword integer if operand size is 64 bits) in the destination operand (the first operand).
The source operand can be an XMM register or a 64-bit memory location. The destination operand is a general
purpose register. When the source operand is an XMM register, the double-precision floating-point value is
contained in the low quadword of the register.
When a conversion is inexact, the value returned is rounded according to the rounding control bits in the MXCSR
register.
If a converted result exceeds the range limits of signed doubleword integer (in non-64-bit modes or 64-bit mode
with REX.W/VEX.W/EVEX.W=0), the floating-point invalid exception is raised, and if this exception is masked, the
indefinite integer value (80000000H) is returned.
If a converted result exceeds the range limits of signed quadword integer (in 64-bit mode and
REX.W/VEX.W/EVEX.W = 1), the floating-point invalid exception is raised, and if this exception is masked, the
indefinite integer value (80000000_00000000H) is returned.
Legacy SSE instructions: In 64-bit mode, Use of the REX.W prefix promotes the instruction to 64-bit operation. See
the summary chart at the beginning of this section for encoding data and limits.
VEX.W1 and EVEX.W1 versions: promotes the instruction to produce 64-bit data in 64-bit mode.
Note: VEX.vvvv and EVEX.vvvv are reserved and must be 1111b, otherwise instructions will #UD.




Software should ensure VCVTTSD2SI is encoded with VEX.L=0. Encoding VCVTTSD2SI with VEX.L=1 may
encounter unpredictable behavior across different processor generations.

Operation
(V)CVTTSD2SI (All versions)
IF 64-Bit Mode and OperandSize = 64
THEN
    DEST[63:0] <- Convert_Double_Precision_Floating_Point_To_Integer_Truncate(SRC[63:0]);
ELSE
    DEST[31:0] <- Convert_Double_Precision_Floating_Point_To_Integer_Truncate(SRC[63:0]);
FI;

Intel C/C++ Compiler Intrinsic Equivalent
VCVTTSD2SI int _mm_cvttsd_i32( __m128d a);
VCVTTSD2SI int _mm_cvtt_roundsd_i32( __m128d a, int sae);
VCVTTSD2SI __int64 _mm_cvttsd_i64( __m128d a);
VCVTTSD2SI __int64 _mm_cvtt_roundsd_i64( __m128d a, int sae);
CVTTSD2SI int _mm_cvttsd_si32( __m128d a);
CVTTSD2SI __int64 _mm_cvttsd_si64( __m128d a);

SIMD Floating-Point Exceptions
Invalid, Precision

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3; additionally
#UD                       If VEX.vvvv != 1111B.
EVEX-encoded instructions, see Exceptions Type E3NF.




');
INSERT INTO "instructions" VALUES('x86_64','CVTTSS2SI','CVTTSS2SI-Convert with Truncation Scalar Single-Precision Floating-Point Value to Integer
 Opcode/                              Op /    64/32        CPUID          Description
 Instruction                          En      bit Mode     Feature
                                              Support      Flag
 F3 0F 2C /r                          RM      V/V          SSE            Convert one single-precision floating-point value from
 CVTTSS2SI r32, xmm1/m32                                                  xmm1/m32 to one signed doubleword integer in r32
                                                                          using truncation.
 F3 REX.W 0F 2C /r                    RM      V/N.E.       SSE            Convert one single-precision floating-point value from
 CVTTSS2SI r64, xmm1/m32                                                  xmm1/m32 to one signed quadword integer in r64
                                                                          using truncation.
 VEX.128.F3.0F.W0 2C /r               RM      V/V          AVX            Convert one single-precision floating-point value from
 VCVTTSS2SI r32, xmm1/m32                                                 xmm1/m32 to one signed doubleword integer in r32
                                                                          using truncation.
 VEX.128.F3.0F.W1 2C /r               RM      V/N.E.1      AVX            Convert one single-precision floating-point value from
 VCVTTSS2SI r64, xmm1/m32                                                 xmm1/m32 to one signed quadword integer in r64
                                                                          using truncation.
 EVEX.LIG.F3.0F.W0 2C /r              T1F     V/V          AVX512F        Convert one single-precision floating-point value from
 VCVTTSS2SI r32, xmm1/m32{sae}                                            xmm1/m32 to one signed doubleword integer in r32
                                                                          using truncation.
 EVEX.LIG.F3.0F.W1 2C /r              T1F     V/N.E.1      AVX512F        Convert one single-precision floating-point value from
 VCVTTSS2SI r64, xmm1/m32{sae}                                            xmm1/m32 to one signed quadword integer in r64
                                                                          using truncation.
NOTES:
1. For this specific instruction, VEX.W/EVEX.W in non-64 bit is ignored; the instructions behaves as if the W0 ver-
  sion is used.



                                             Instruction Operand Encoding
      Op/En             Operand 1                   Operand 2                     Operand 3                      Operand 4
       RM             ModRM:reg (w)             ModRM:r/m (r)                         NA                             NA
       T1F            ModRM:reg (w)             ModRM:r/m (r)                         NA                             NA

Description
Converts a single-precision floating-point value in the source operand (the second operand) to a signed doubleword
integer (or signed quadword integer if operand size is 64 bits) in the destination operand (the first operand). The
source operand can be an XMM register or a 32-bit memory location. The destination operand is a general purpose
register. When the source operand is an XMM register, the single-precision floating-point value is contained in the
low doubleword of the register.
When a conversion is inexact, a truncated (round toward zero) result is returned. If a converted result is larger than
the maximum signed doubleword integer, the floating-point invalid exception is raised. If this exception is masked,
the indefinite integer value (80000000H or 80000000_00000000H if operand size is 64 bits) is returned.
Legacy SSE instructions: In 64-bit mode, Use of the REX.W prefix promotes the instruction to 64-bit operation. See
the summary chart at the beginning of this section for encoding data and limits.
VEX.W1 and EVEX.W1 versions: promotes the instruction to produce 64-bit data in 64-bit mode.
Note: VEX.vvvv and EVEX.vvvv are reserved and must be 1111b, otherwise instructions will #UD.
Software should ensure VCVTTSS2SI is encoded with VEX.L=0. Encoding VCVTTSS2SI with VEX.L=1 may
encounter unpredictable behavior across different processor generations.





Operation
(V)CVTTSS2SI (All versions)
IF 64-Bit Mode and OperandSize = 64
THEN
    DEST[63:0] <- Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[31:0]);
ELSE
    DEST[31:0] <- Convert_Single_Precision_Floating_Point_To_Integer_Truncate(SRC[31:0]);
FI;

Intel C/C++ Compiler Intrinsic Equivalent
VCVTTSS2SI int _mm_cvttss_i32( __m128 a);
VCVTTSS2SI int _mm_cvtt_roundss_i32( __m128 a, int sae);
VCVTTSS2SI __int64 _mm_cvttss_i64( __m128 a);
VCVTTSS2SI __int64 _mm_cvtt_roundss_i64( __m128 a, int sae);
CVTTSS2SI int _mm_cvttss_si32( __m128 a);
CVTTSS2SI __int64 _mm_cvttss_si64( __m128 a);

SIMD Floating-Point Exceptions
Invalid, Precision

Other Exceptions
See Exceptions Type 3; additionally
#UD                       If VEX.vvvv != 1111B.
EVEX-encoded instructions, see Exceptions Type E3NF.




');
INSERT INTO "instructions" VALUES('x86_64','CWD','CWD/CDQ/CQO-Convert Word to Doubleword/Convert Doubleword to Quadword
Opcode                    Instruction                 Op/    64-Bit     Compat/ Description
                                                      En     Mode       Leg Mode
99                        CWD                         NP     Valid      Valid       DX:AX <- sign-extend of AX.
99                        CDQ                         NP     Valid      Valid       EDX:EAX <- sign-extend of EAX.
REX.W + 99                CQO                         NP     Valid      N.E.        RDX:RAX<- sign-extend of RAX.



                                                Instruction Operand Encoding
     Op/En             Operand 1                     Operand 2                   Operand 3                    Operand 4
      NP                  NA                            NA                           NA                          NA

Description
Doubles the size of the operand in register AX, EAX, or RAX (depending on the operand size) by means of sign
extension and stores the result in registers DX:AX, EDX:EAX, or RDX:RAX, respectively. The CWD instruction
copies the sign (bit 15) of the value in the AX register into every bit position in the DX register. The CDQ instruction
copies the sign (bit 31) of the value in the EAX register into every bit position in the EDX register. The CQO instruc-
tion (available in 64-bit mode only) copies the sign (bit 63) of the value in the RAX register into every bit position
in the RDX register.
The CWD instruction can be used to produce a doubleword dividend from a word before word division. The CDQ
instruction can be used to produce a quadword dividend from a doubleword before doubleword division. The CQO
instruction can be used to produce a double quadword dividend from a quadword before a quadword division.
The CWD and CDQ mnemonics reference the same opcode. The CWD instruction is intended for use when the
operand-size attribute is 16 and the CDQ instruction for when the operand-size attribute is 32. Some assemblers
may force the operand size to 16 when CWD is used and to 32 when CDQ is used. Others may treat these
mnemonics as synonyms (CWD/CDQ) and use the current setting of the operand-size attribute to determine the
size of values to be converted, regardless of the mnemonic used.
In 64-bit mode, use of the REX.W prefix promotes operation to 64 bits. The CQO mnemonics reference the same
opcode as CWD/CDQ. See the summary chart at the beginning of this section for encoding data and limits.

Operation

IF OperandSize = 16 (* CWD instruction *)
    THEN
        DX <- SignExtend(AX);
    ELSE IF OperandSize = 32 (* CDQ instruction *)
        EDX <- SignExtend(EAX); FI;
    ELSE IF 64-Bit Mode and OperandSize = 64 (* CQO instruction*)
        RDX <- SignExtend(RAX); FI;
FI;

Flags Affected
None

Exceptions (All Operating Modes)
#UD                    If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','CDQ','-R:CWD');
INSERT INTO "instructions" VALUES('x86_64','CQO','-R:CWD');
INSERT INTO "instructions" VALUES('x86_64','DAA','DAA-Decimal Adjust AL after Addition
 Opcode                     Instruction                Op/     64-Bit    Compat/ Description
                                                       En      Mode      Leg Mode
 27                         DAA                        NP      Invalid   Valid     Decimal adjust AL after addition.



                                                 Instruction Operand Encoding
      Op/En              Operand 1                    Operand 2                  Operand 3                    Operand 4
       NP                    NA                           NA                        NA                            NA

Description
Adjusts the sum of two packed BCD values to create a packed BCD result. The AL register is the implied source and
destination operand. The DAA instruction is only useful when it follows an ADD instruction that adds (binary addi-
tion) two 2-digit, packed BCD values and stores a byte result in the AL register. The DAA instruction then adjusts
the contents of the AL register to contain the correct 2-digit, packed BCD result. If a decimal carry is detected, the
CF and AF flags are set accordingly.
This instruction executes as described above in compatibility mode and legacy mode. It is not valid in 64-bit mode.

Operation
IF 64-Bit Mode
    THEN
         #UD;
    ELSE
         old_AL <- AL;
         old_CF <- CF;
         CF <- 0;
         IF (((AL AND 0FH) > 9) or AF = 1)
                THEN
                    AL <- AL + 6;
                    CF <- old_CF or (Carry from AL <- AL + 6);
                    AF <- 1;
                ELSE
                    AF <- 0;
         FI;
         IF ((old_AL > 99H) or (old_CF = 1))
               THEN
                    AL <- AL + 60H;
                    CF <- 1;
               ELSE
                    CF <- 0;
         FI;
FI;

Example
ADD      AL, BL   Before: AL=79H BL=35H EFLAGS(OSZAPC)=XXXXXX
                  After: AL=AEH BL=35H EFLAGS(0SZAPC)=110000
DAA               Before: AL=AEH BL=35H EFLAGS(OSZAPC)=110000
                  After: AL=14H BL=35H EFLAGS(0SZAPC)=X00111
DAA               Before: AL=2EH BL=35H EFLAGS(OSZAPC)=110000
                  After: AL=34H BL=35H EFLAGS(0SZAPC)=X00101





Flags Affected
The CF and AF flags are set if the adjustment of the value results in a decimal carry in either digit of the result (see
the "Operation" section above). The SF, ZF, and PF flags are set according to the result. The OF flag is undefined.

Protected Mode Exceptions
#UD                  If the LOCK prefix is used.

Real-Address Mode Exceptions
#UD                  If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#UD                  If the LOCK prefix is used.

Compatibility Mode Exceptions
#UD                  If the LOCK prefix is used.

64-Bit Mode Exceptions
#UD                  If in 64-bit mode.




');
INSERT INTO "instructions" VALUES('x86_64','DAS','DAS-Decimal Adjust AL after Subtraction
 Opcode                     Instruction               Op/      64-Bit    Compat/ Description
                                                      En       Mode      Leg Mode
 2F                         DAS                       NP       Invalid   Valid      Decimal adjust AL after subtraction.



                                                Instruction Operand Encoding
      Op/En             Operand 1                  Operand 2                     Operand 3                    Operand 4
       NP                   NA                         NA                           NA                            NA

Description
Adjusts the result of the subtraction of two packed BCD values to create a packed BCD result. The AL register is the
implied source and destination operand. The DAS instruction is only useful when it follows a SUB instruction that
subtracts (binary subtraction) one 2-digit, packed BCD value from another and stores a byte result in the AL
register. The DAS instruction then adjusts the contents of the AL register to contain the correct 2-digit, packed BCD
result. If a decimal borrow is detected, the CF and AF flags are set accordingly.
This instruction executes as described above in compatibility mode and legacy mode. It is not valid in 64-bit mode.

Operation
IF 64-Bit Mode
    THEN
         #UD;
    ELSE
         old_AL <- AL;
         old_CF <- CF;
         CF <- 0;
         IF (((AL AND 0FH) > 9) or AF = 1)
               THEN
                   AL <- AL - 6;
                   CF <- old_CF or (Borrow from AL <- AL - 6);
                   AF <- 1;
               ELSE
                   AF <- 0;
         FI;
         IF ((old_AL > 99H) or (old_CF = 1))
                THEN
                   AL <- AL - 60H;
                   CF <- 1;
         FI;
FI;

Example
SUB      AL, BL   Before: AL = 35H, BL = 47H, EFLAGS(OSZAPC) = XXXXXX
                  After: AL = EEH, BL = 47H, EFLAGS(0SZAPC) = 010111
DAA               Before: AL = EEH, BL = 47H, EFLAGS(OSZAPC) = 010111
                  After: AL = 88H, BL = 47H, EFLAGS(0SZAPC) = X10111

Flags Affected
The CF and AF flags are set if the adjustment of the value results in a decimal borrow in either digit of the result
(see the "Operation" section above). The SF, ZF, and PF flags are set according to the result. The OF flag is unde-
fined.



Protected Mode Exceptions
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
#UD                 If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
#UD                 If the LOCK prefix is used.

64-Bit Mode Exceptions
#UD                 If in 64-bit mode.




');
INSERT INTO "instructions" VALUES('x86_64','DEC','DEC-Decrement by 1
 Opcode                    Instruction                  Op/      64-Bit    Compat/ Description
                                                        En       Mode      Leg Mode
 FE /1                     DEC r/m8                     M        Valid     Valid        Decrement r/m8 by 1.
                                      *
 REX + FE /1               DEC r/m8                     M        Valid     N.E.         Decrement r/m8 by 1.
 FF /1                     DEC r/m16                    M        Valid     Valid        Decrement r/m16 by 1.
 FF /1                     DEC r/m32                    M        Valid     Valid        Decrement r/m32 by 1.
 REX.W + FF /1             DEC r/m64                    M        Valid     N.E.         Decrement r/m64 by 1.
 48+rw                     DEC r16                      O        N.E.      Valid        Decrement r16 by 1.
 48+rd                     DEC r32                      O        N.E.      Valid        Decrement r32 by 1.
 NOTES:
 * In 64-bit mode, r/m8 can not be encoded to access the following byte registers if a REX prefix is used: AH, BH, CH, DH.



                                                  Instruction Operand Encoding
    Op/En               Operand 1                       Operand 2                     Operand 3                    Operand 4
         M           ModRM:r/m (r, w)                       NA                            NA                           NA
         O           opcode + rd (r, w)                     NA                            NA                           NA

Description
Subtracts 1 from the destination operand, while preserving the state of the CF flag. The destination operand can be
a register or a memory location. This instruction allows a loop counter to be updated without disturbing the CF flag.
(To perform a decrement operation that updates the CF flag, use a SUB instruction with an immediate operand of
1.)
This instruction can be used with a LOCK prefix to allow the instruction to be executed atomically.
In 64-bit mode, DEC r16 and DEC r32 are not encodable (because opcodes 48H through 4FH are REX prefixes).
Otherwise, the instruction''s 64-bit mode default operation size is 32 bits. Use of the REX.R prefix permits access to
additional registers (R8-R15). Use of the REX.W prefix promotes operation to 64 bits.
See the summary chart at the beginning of this section for encoding data and limits.

Operation
DEST <- DEST - 1;

Flags Affected
The CF flag is not affected. The OF, SF, ZF, AF, and PF flags are set according to the result.

Protected Mode Exceptions
#GP(0)                  If the destination operand is located in a non-writable segment.
                        If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                        If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)                  If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)         If a page fault occurs.
#AC(0)                  If alignment checking is enabled and an unaligned memory reference is made while the
                        current privilege level is 3.
#UD                     If the LOCK prefix is used but the destination is not a memory operand.





Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used but the destination is not a memory operand.




');
INSERT INTO "instructions" VALUES('x86_64','DIV','DIV-Unsigned Divide
Opcode                     Instruction                  Op/    64-Bit      Compat/ Description
                                                        En     Mode        Leg Mode
F6 /6                      DIV r/m8                     M      Valid       Valid       Unsigned divide AX by r/m8, with result
                                                                                       stored in AL <- Quotient, AH <- Remainder.
REX + F6 /6                DIV r/m8*                    M      Valid       N.E.        Unsigned divide AX by r/m8, with result
                                                                                       stored in AL <- Quotient, AH <- Remainder.
F7 /6                      DIV r/m16                    M      Valid       Valid       Unsigned divide DX:AX by r/m16, with result
                                                                                       stored in AX <- Quotient, DX <- Remainder.
F7 /6                      DIV r/m32                    M      Valid       Valid       Unsigned divide EDX:EAX by r/m32, with
                                                                                       result stored in EAX <- Quotient, EDX <-
                                                                                       Remainder.
REX.W + F7 /6              DIV r/m64                    M      Valid       N.E.        Unsigned divide RDX:RAX by r/m64, with
                                                                                       result stored in RAX <- Quotient, RDX <-
                                                                                       Remainder.
NOTES:
* In 64-bit mode, r/m8 can not be encoded to access the following byte registers if a REX prefix is used: AH, BH, CH, DH.



                                                  Instruction Operand Encoding
   Op/En                Operand 1                   Operand 2                      Operand 3                      Operand 4
     M                ModRM:r/m (w)                     NA                             NA                             NA

Description
Divides unsigned the value in the AX, DX:AX, EDX:EAX, or RDX:RAX registers (dividend) by the source operand
(divisor) and stores the result in the AX (AH:AL), DX:AX, EDX:EAX, or RDX:RAX registers. The source operand can
be a general-purpose register or a memory location. The action of this instruction depends on the operand size
(dividend/divisor). Division using 64-bit operand is available only in 64-bit mode.
Non-integral results are truncated (chopped) towards 0. The remainder is always less than the divisor in magni-
tude. Overflow is indicated with the #DE (divide error) exception rather than with the CF flag.
In 64-bit mode, the instruction''s default operation size is 32 bits. Use of the REX.R prefix permits access to addi-
tional registers (R8-R15). Use of the REX.W prefix promotes operation to 64 bits. In 64-bit mode when REX.W is
applied, the instruction divides the unsigned value in RDX:RAX by the source operand and stores the quotient in
RAX, the remainder in RDX.
See the summary chart at the beginning of this section for encoding data and limits. See Table 3-15.
                                                    Table 3-15. DIV Action
                                                                                                                  Maximum
 Operand Size                          Dividend         Divisor         Quotient            Remainder             Quotient
 Word/byte                             AX               r/m8            AL                  AH                    255
 Doubleword/word                       DX:AX            r/m16           AX                  DX                    65,535
 Quadword/doubleword                   EDX:EAX          r/m32           EAX                 EDX                   232 - 1
 Doublequadword/                       RDX:RAX          r/m64           RAX                 RDX                   264 - 1
 quadword





Operation
IF SRC = 0
    THEN #DE; FI; (* Divide Error *)
IF OperandSize = 8 (* Word/Byte Operation *)
    THEN
        temp <- AX / SRC;
        IF temp > FFH
             THEN #DE; (* Divide error *)
             ELSE
                  AL <- temp;
                  AH <- AX MOD SRC;
        FI;
    ELSE IF OperandSize = 16 (* Doubleword/word operation *)
        THEN
             temp <- DX:AX / SRC;
             IF temp > FFFFH
                  THEN #DE; (* Divide error *)
             ELSE
                  AX <- temp;
                  DX <- DX:AX MOD SRC;
             FI;
        FI;
    ELSE IF Operandsize = 32 (* Quadword/doubleword operation *)
        THEN
             temp <- EDX:EAX / SRC;
             IF temp > FFFFFFFFH
                  THEN #DE; (* Divide error *)
             ELSE
                  EAX <- temp;
                  EDX <- EDX:EAX MOD SRC;
             FI;
        FI;
    ELSE IF 64-Bit Mode and Operandsize = 64 (* Doublequadword/quadword operation *)
        THEN
             temp <- RDX:RAX / SRC;
             IF temp > FFFFFFFFFFFFFFFFH
                  THEN #DE; (* Divide error *)
             ELSE
                  RAX <- temp;
                  RDX <- RDX:RAX MOD SRC;
             FI;
        FI;
FI;

Flags Affected
The CF, OF, SF, ZF, AF, and PF flags are undefined.





Protected Mode Exceptions
#DE                   If the source operand (divisor) is 0
                      If the quotient is too large for the designated register.
#GP(0)                If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                      If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)                If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)       If a page fault occurs.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made while the
                      current privilege level is 3.
#UD                   If the LOCK prefix is used.

Real-Address Mode Exceptions
#DE                   If the source operand (divisor) is 0.
                      If the quotient is too large for the designated register.
#GP                   If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                      If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)                If a memory operand effective address is outside the SS segment limit.
#UD                   If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#DE                   If the source operand (divisor) is 0.
                      If the quotient is too large for the designated register.
#GP(0)                If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                   If a memory operand effective address is outside the SS segment limit.
#PF(fault-code)       If a page fault occurs.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made.
#UD                   If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)                If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)                If the memory address is in a non-canonical form.
#DE                   If the source operand (divisor) is 0
                      If the quotient is too large for the designated register.
#PF(fault-code)       If a page fault occurs.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made while the
                      current privilege level is 3.
#UD                   If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','DIVPD','DIVPD-Divide Packed Double-Precision Floating-Point Values
 Opcode/                              Op /   64/32        CPUID      Description
 Instruction                          En     bit Mode     Feature
                                             Support      Flag
 66 0F 5E /r                          RM     V/V          SSE2       Divide packed double-precision floating-point values
 DIVPD xmm1, xmm2/m128                                               in xmm1 by packed double-precision floating-point
                                                                     values in xmm2/mem.
 VEX.NDS.128.66.0F.WIG 5E /r          RVM    V/V          AVX        Divide packed double-precision floating-point values
 VDIVPD xmm1, xmm2, xmm3/m128                                        in xmm2 by packed double-precision floating-point
                                                                     values in xmm3/mem.
 VEX.NDS.256.66.0F.WIG 5E /r          RVM    V/V          AVX        Divide packed double-precision floating-point values
 VDIVPD ymm1, ymm2, ymm3/m256                                        in ymm2 by packed double-precision floating-point
                                                                     values in ymm3/mem.
 EVEX.NDS.128.66.0F.W1 5E /r          FV     V/V          AVX512VL   Divide packed double-precision floating-point values
 VDIVPD xmm1 {k1}{z}, xmm2,                               AVX512F    in xmm2 by packed double-precision floating-point
 xmm3/m128/m64bcst                                                   values in xmm3/m128/m64bcst and write results to
                                                                     xmm1 subject to writemask k1.
 EVEX.NDS.256.66.0F.W1 5E /r          FV     V/V          AVX512VL   Divide packed double-precision floating-point values
 VDIVPD ymm1 {k1}{z}, ymm2,                               AVX512F    in ymm2 by packed double-precision floating-point
 ymm3/m256/m64bcst                                                   values in ymm3/m256/m64bcst and write results to
                                                                     ymm1 subject to writemask k1.
 EVEX.NDS.512.66.0F.W1 5E /r          FV     V/V          AVX512F    Divide packed double-precision floating-point values
 VDIVPD zmm1 {k1}{z}, zmm2,                                          in zmm2 by packed double-precision FP values in
 zmm3/m512/m64bcst{er}                                               zmm3/m512/m64bcst and write results to zmm1
                                                                     subject to writemask k1.



                                           Instruction Operand Encoding
     Op/En            Operand 1               Operand 2                  Operand 3                       Operand 4
      RM           ModRM:reg (r, w)          ModRM:r/m (r)                   NA                              NA
     RVM            ModRM:reg (w)              VEX.vvvv                ModRM:r/m (r)                         NA
      FV            ModRM:reg (w)             EVEX.vvvv                ModRM:r/m (r)                         NA

Description
Performs a SIMD divide of the double-precision floating-point values in the first source operand by the floating-
point values in the second source operand (the third operand). Results are written to the destination operand (the
first operand).
EVEX encoded versions: The first source operand (the second operand) is a ZMM/YMM/XMM register. The second
source operand can be a ZMM/YMM/XMM register, a 512/256/128-bit memory location or a 512/256/128-bit vector
broadcasted from a 64-bit memory location. The destination operand is a ZMM/YMM/XMM register conditionally
updated with writemask k1.
VEX.256 encoded version: The first source operand (the second operand) is a YMM register. The second source
operand can be a YMM register or a 256-bit memory location. The destination operand is a YMM register. The upper
bits (MAX_VL-1:256) of the corresponding destination are zeroed.
VEX.128 encoded version: The first source operand (the second operand) is a XMM register. The second source
operand can be a XMM register or a 128-bit memory location. The destination operand is a XMM register. The upper
bits (MAX_VL-1:128) of the corresponding destination are zeroed.
128-bit Legacy SSE version: The second source operand (the second operand) can be an XMM register or an 128-
bit memory location. The destination is the same as the first source operand. The upper bits (MAX_VL-1:128) of the
corresponding destination are unmodified.





Operation
VDIVPD (EVEX encoded versions)
(KL, VL) = (2, 128), (4, 256), (8, 512)
IF (VL = 512) AND (EVEX.b = 1) AND SRC2 *is a register*
    THEN
         SET_RM(EVEX.RC);          ; refer to Table 2-4 in the Intel Architecture Instruction Set Extensions Programming Reference
    ELSE
         SET_RM(MXCSR.RM);
FI;
FOR j <- 0 TO KL-1
    i <- j * 64
    IF k1[j] OR *no writemask*
         THEN
               IF (EVEX.b = 1) AND (SRC2 *is memory*)
                    THEN
                         DEST[i+63:i] <- SRC1[i+63:i] / SRC2[63:0]
                    ELSE
                         DEST[i+63:i] <- SRC1[i+63:i] / SRC2[i+63:i]
               FI;
         ELSE
               IF *merging-masking*                    ; merging-masking
                    THEN *DEST[i+63:i] remains unchanged*
                    ELSE                               ; zeroing-masking
                         DEST[i+63:i] <- 0
               FI
    FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VDIVPD (VEX.256 encoded version)
DEST[63:0] <-SRC1[63:0] / SRC2[63:0]
DEST[127:64] <-SRC1[127:64] / SRC2[127:64]
DEST[191:128] <-SRC1[191:128] / SRC2[191:128]
DEST[255:192] <-SRC1[255:192] / SRC2[255:192]
DEST[MAX_VL-1:256] <-0;

VDIVPD (VEX.128 encoded version)
DEST[63:0] <-SRC1[63:0] / SRC2[63:0]
DEST[127:64] <-SRC1[127:64] / SRC2[127:64]
DEST[MAX_VL-1:128] <-0;

DIVPD (128-bit Legacy SSE version)
DEST[63:0] <-SRC1[63:0] / SRC2[63:0]
DEST[127:64] <-SRC1[127:64] / SRC2[127:64]
DEST[MAX_VL-1:128] (Unmodified)





Intel C/C++ Compiler Intrinsic Equivalent
VDIVPD __m512d _mm512_div_pd( __m512d a, __m512d b);
VDIVPD __m512d _mm512_mask_div_pd(__m512d s, __mmask8 k, __m512d a, __m512d b);
VDIVPD __m512d _mm512_maskz_div_pd( __mmask8 k, __m512d a, __m512d b);
VDIVPD __m256d _mm256_mask_div_pd(__m256d s, __mmask8 k, __m256d a, __m256d b);
VDIVPD __m256d _mm256_maskz_div_pd( __mmask8 k, __m256d a, __m256d b);
VDIVPD __m128d _mm_mask_div_pd(__m128d s, __mmask8 k, __m128d a, __m128d b);
VDIVPD __m128d _mm_maskz_div_pd( __mmask8 k, __m128d a, __m128d b);
VDIVPD __m512d _mm512_div_round_pd( __m512d a, __m512d b, int);
VDIVPD __m512d _mm512_mask_div_round_pd(__m512d s, __mmask8 k, __m512d a, __m512d b, int);
VDIVPD __m512d _mm512_maskz_div_round_pd( __mmask8 k, __m512d a, __m512d b, int);
VDIVPD __m256d _mm256_div_pd (__m256d a, __m256d b);
DIVPD __m128d _mm_div_pd (__m128d a, __m128d b);

SIMD Floating-Point Exceptions
Overflow, Underflow, Invalid, Divide-by-Zero, Precision, Denormal

Other Exceptions
VEX-encoded instructions, see Exceptions Type 2.
EVEX-encoded instructions, see Exceptions Type E2.




');
INSERT INTO "instructions" VALUES('x86_64','DIVPS','DIVPS-Divide Packed Single-Precision Floating-Point Values
 Opcode/                                          Op /       64/32        CPUID      Description
 Instruction                                      En         bit Mode     Feature
                                                             Support      Flag
 0F 5E /r                                         RM         V/V          SSE        Divide packed single-precision floating-point values
 DIVPS xmm1, xmm2/m128                                                               in xmm1 by packed single-precision floating-point
                                                                                     values in xmm2/mem.
 VEX.NDS.128.0F.WIG 5E /r                         RVM        V/V          AVX        Divide packed single-precision floating-point values
 VDIVPS xmm1, xmm2, xmm3/m128                                                        in xmm2 by packed single-precision floating-point
                                                                                     values in xmm3/mem.
 VEX.NDS.256.0F.WIG 5E /r                         RVM        V/V          AVX        Divide packed single-precision floating-point values
 VDIVPS ymm1, ymm2, ymm3/m256                                                        in ymm2 by packed single-precision floating-point
                                                                                     values in ymm3/mem.
 EVEX.NDS.128.0F.W0 5E /r                         FV         V/V          AVX512VL   Divide packed single-precision floating-point values
 VDIVPS xmm1 {k1}{z}, xmm2,                                               AVX512F    in xmm2 by packed single-precision floating-point
 xmm3/m128/m32bcst                                                                   values in xmm3/m128/m32bcst and write results to
                                                                                     xmm1 subject to writemask k1.
 EVEX.NDS.256.0F.W0 5E /r                         FV         V/V          AVX512VL   Divide packed single-precision floating-point values
 VDIVPS ymm1 {k1}{z}, ymm2,                                               AVX512F    in ymm2 by packed single-precision floating-point
 ymm3/m256/m32bcst                                                                   values in ymm3/m256/m32bcst and write results to
                                                                                     ymm1 subject to writemask k1.
 EVEX.NDS.512.0F.W0 5E /r                         FV         V/V          AVX512F    Divide packed single-precision floating-point values
 VDIVPS zmm1 {k1}{z}, zmm2,                                                          in zmm2 by packed single-precision floating-point
 zmm3/m512/m32bcst{er}                                                               values in zmm3/m512/m32bcst and write results to
                                                                                     zmm1 subject to writemask k1.



                                                       Instruction Operand Encoding
      Op/En                 Operand 1                         Operand 2                  Operand 3                    Operand 4
       RM               ModRM:reg (r, w)                     ModRM:r/m (r)                  NA                            NA
      RVM                ModRM:reg (w)                         VEX.vvvv                ModRM:r/m (r)                      NA
       FV                ModRM:reg (w)                        EVEX.vvvv                ModRM:r/m (r)                      NA

Description
Performs a SIMD divide of the four, eight or sixteen packed single-precision floating-point values in the first source
operand (the second operand) by the four, eight or sixteen packed single-precision floating-point values in the
second source operand (the third operand). Results are written to the destination operand (the first operand).
EVEX encoded versions: The first source operand (the second operand) is a ZMM/YMM/XMM register. The second
source operand can be a ZMM/YMM/XMM register, a 512/256/128-bit memory location or a 512/256/128-bit vector
broadcasted from a 32-bit memory location. The destination operand is a ZMM/YMM/XMM register conditionally
updated with writemask k1.
VEX.256 encoded version: The first source operand is a YMM register. The second source operand can be a YMM
register or a 256-bit memory location. The destination operand is a YMM register.
VEX.128 encoded version: The first source operand is a XMM register. The second source operand can be a XMM
register or a 128-bit memory location. The destination operand is a XMM register. The upper bits (MAX_VL-1:128)
of the corresponding ZMM register destination are zeroed.
128-bit Legacy SSE version: The second source can be an XMM register or an 128-bit memory location. The desti-
nation is not distinct from the first source XMM register and the upper bits (MAX_VL-1:128) of the corresponding
ZMM register destination are unmodified.





Operation
VDIVPS (EVEX encoded versions)
(KL, VL) = (4, 128), (8, 256), (16, 512)
IF (VL = 512) AND (EVEX.b = 1) AND SRC2 *is a register*
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);
FI;
FOR j <- 0 TO KL-1
    i <- j * 32
    IF k1[j] OR *no writemask*
         THEN
               IF (EVEX.b = 1) AND (SRC2 *is memory*)
                    THEN
                         DEST[i+31:i] <- SRC1[i+31:i] / SRC2[31:0]
                    ELSE
                         DEST[i+31:i] <- SRC1[i+31:i] / SRC2[i+31:i]
               FI;
         ELSE
               IF *merging-masking*                 ; merging-masking
                    THEN *DEST[i+31:i] remains unchanged*
                    ELSE                            ; zeroing-masking
                         DEST[i+31:i] <- 0
               FI
    FI;
ENDFOR
DEST[MAX_VL-1:VL] <- 0

VDIVPS (VEX.256 encoded version)
DEST[31:0] <-SRC1[31:0] / SRC2[31:0]
DEST[63:32] <-SRC1[63:32] / SRC2[63:32]
DEST[95:64] <-SRC1[95:64] / SRC2[95:64]
DEST[127:96] <-SRC1[127:96] / SRC2[127:96]
DEST[159:128] <-SRC1[159:128] / SRC2[159:128]
DEST[191:160]<-SRC1[191:160] / SRC2[191:160]
DEST[223:192] <-SRC1[223:192] / SRC2[223:192]
DEST[255:224] <-SRC1[255:224] / SRC2[255:224].
DEST[MAX_VL-1:256] <-0;

VDIVPS (VEX.128 encoded version)
DEST[31:0] <-SRC1[31:0] / SRC2[31:0]
DEST[63:32] <-SRC1[63:32] / SRC2[63:32]
DEST[95:64] <-SRC1[95:64] / SRC2[95:64]
DEST[127:96] <-SRC1[127:96] / SRC2[127:96]
DEST[MAX_VL-1:128] <-0





DIVPS (128-bit Legacy SSE version)
DEST[31:0] <-SRC1[31:0] / SRC2[31:0]
DEST[63:32] <-SRC1[63:32] / SRC2[63:32]
DEST[95:64] <-SRC1[95:64] / SRC2[95:64]
DEST[127:96] <-SRC1[127:96] / SRC2[127:96]
DEST[MAX_VL-1:128] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VDIVPS __m512 _mm512_div_ps( __m512 a, __m512 b);
VDIVPS __m512 _mm512_mask_div_ps(__m512 s, __mmask16 k, __m512 a, __m512 b);
VDIVPS __m512 _mm512_maskz_div_ps(__mmask16 k, __m512 a, __m512 b);
VDIVPD __m256d _mm256_mask_div_pd(__m256d s, __mmask8 k, __m256d a, __m256d b);
VDIVPD __m256d _mm256_maskz_div_pd( __mmask8 k, __m256d a, __m256d b);
VDIVPD __m128d _mm_mask_div_pd(__m128d s, __mmask8 k, __m128d a, __m128d b);
VDIVPD __m128d _mm_maskz_div_pd( __mmask8 k, __m128d a, __m128d b);
VDIVPS __m512 _mm512_div_round_ps( __m512 a, __m512 b, int);
VDIVPS __m512 _mm512_mask_div_round_ps(__m512 s, __mmask16 k, __m512 a, __m512 b, int);
VDIVPS __m512 _mm512_maskz_div_round_ps(__mmask16 k, __m512 a, __m512 b, int);
VDIVPS __m256 _mm256_div_ps (__m256 a, __m256 b);
DIVPS __m128 _mm_div_ps (__m128 a, __m128 b);

SIMD Floating-Point Exceptions
Overflow, Underflow, Invalid, Divide-by-Zero, Precision, Denormal

Other Exceptions
VEX-encoded instructions, see Exceptions Type 2.
EVEX-encoded instructions, see Exceptions Type E2.




');
INSERT INTO "instructions" VALUES('x86_64','DIVSD','DIVSD-Divide Scalar Double-Precision Floating-Point Value
 Opcode/                                Op /     64/32        CPUID     Description
 Instruction                            En       bit Mode     Feature
                                                 Support      Flag
 F2 0F 5E /r                            RM       V/V          SSE2      Divide low double-precision floating-point value in
 DIVSD xmm1, xmm2/m64                                                   xmm1 by low double-precision floating-point value
                                                                        in xmm2/m64.
 VEX.NDS.128.F2.0F.WIG 5E /r            RVM      V/V          AVX       Divide low double-precision floating-point value in
 VDIVSD xmm1, xmm2, xmm3/m64                                            xmm2 by low double-precision floating-point value
                                                                        in xmm3/m64.
 EVEX.NDS.LIG.F2.0F.W1 5E /r            T1S      V/V          AVX512F   Divide low double-precision floating-point value in
 VDIVSD xmm1 {k1}{z}, xmm2,                                             xmm2 by low double-precision floating-point value
 xmm3/m64{er}                                                           in xmm3/m64.



                                              Instruction Operand Encoding
     Op/En             Operand 1                  Operand 2                Operand 3                       Operand 4
       RM           ModRM:reg (r, w)            ModRM:r/m (r)                  NA                              NA
      RVM            ModRM:reg (w)                VEX.vvvv               ModRM:r/m (r)                         NA
      T1S            ModRM:reg (w)               EVEX.vvvv               ModRM:r/m (r)                         NA

Description
Divides the low double-precision floating-point value in the first source operand by the low double-precision
floating-point value in the second source operand, and stores the double-precision floating-point result in the desti-
nation operand. The second source operand can be an XMM register or a 64-bit memory location. The first source
and destination are XMM registers.
128-bit Legacy SSE version: The first source operand and the destination operand are the same. Bits (MAX_VL-
1:64) of the corresponding ZMM destination register remain unchanged.
VEX.128 encoded version: The first source operand is an xmm register encoded by VEX.vvvv. The quadword at bits
127:64 of the destination operand is copied from the corresponding quadword of the first source operand. Bits
(MAX_VL-1:128) of the destination register are zeroed.
EVEX.128 encoded version: The first source operand is an xmm register encoded by EVEX.vvvv. The quadword
element of the destination operand at bits 127:64 are copied from the first source operand. Bits (MAX_VL-1:128)
of the destination register are zeroed.
EVEX version: The low quadword element of the destination is updated according to the writemask.
Software should ensure VDIVSD is encoded with VEX.L=0. Encoding VDIVSD with VEX.L=1 may encounter unpre-
dictable behavior across different processor generations.





Operation
VDIVSD (EVEX encoded version)
IF (EVEX.b = 1) AND SRC2 *is a register*
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);
FI;
IF k1[0] or *no writemask*
    THEN      DEST[63:0] <- SRC1[63:0] / SRC2[63:0]
    ELSE
         IF *merging-masking*               ; merging-masking
              THEN *DEST[63:0] remains unchanged*
              ELSE                          ; zeroing-masking
                  THEN DEST[63:0] <- 0
         FI;
FI;
DEST[127:64] <- SRC1[127:64]
DEST[MAX_VL-1:128] <- 0

VDIVSD (VEX.128 encoded version)
DEST[63:0] <-SRC1[63:0] / SRC2[63:0]
DEST[127:64] <-SRC1[127:64]
DEST[MAX_VL-1:128] <-0

DIVSD (128-bit Legacy SSE version)
DEST[63:0] <-DEST[63:0] / SRC[63:0]
DEST[MAX_VL-1:64] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VDIVSD __m128d _mm_mask_div_sd(__m128d s, __mmask8 k, __m128d a, __m128d b);
VDIVSD __m128d _mm_maskz_div_sd( __mmask8 k, __m128d a, __m128d b);
VDIVSD __m128d _mm_div_round_sd( __m128d a, __m128d b, int);
VDIVSD __m128d _mm_mask_div_round_sd(__m128d s, __mmask8 k, __m128d a, __m128d b, int);
VDIVSD __m128d _mm_maskz_div_round_sd( __mmask8 k, __m128d a, __m128d b, int);
DIVSD __m128d _mm_div_sd (__m128d a, __m128d b);

SIMD Floating-Point Exceptions
Overflow, Underflow, Invalid, Divide-by-Zero, Precision, Denormal

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3.
EVEX-encoded instructions, see Exceptions Type E3.




');
INSERT INTO "instructions" VALUES('x86_64','DIVSS','DIVSS-Divide Scalar Single-Precision Floating-Point Values
 Opcode/                                Op /     64/32         CPUID     Description
 Instruction                            En       bit Mode      Feature
                                                 Support       Flag
 F3 0F 5E /r                            RM       V/V           SSE       Divide low single-precision floating-point value in
 DIVSS xmm1, xmm2/m32                                                    xmm1 by low single-precision floating-point value in
                                                                         xmm2/m32.
 VEX.NDS.128.F3.0F.WIG 5E /r            RVM      V/V           AVX       Divide low single-precision floating-point value in
 VDIVSS xmm1, xmm2, xmm3/m32                                             xmm2 by low single-precision floating-point value in
                                                                         xmm3/m32.
 EVEX.NDS.LIG.F3.0F.W0 5E /r            T1S      V/V           AVX512F   Divide low single-precision floating-point value in
 VDIVSS xmm1 {k1}{z}, xmm2,                                              xmm2 by low single-precision floating-point value in
 xmm3/m32{er}                                                            xmm3/m32.



                                               Instruction Operand Encoding
     Op/En             Operand 1                   Operand 2                Operand 3                        Operand 4
      RM            ModRM:reg (r, w)             ModRM:r/m (r)                   NA                              NA
      RVM            ModRM:reg (w)                 VEX.vvvv                ModRM:r/m (r)                         NA
      T1S            ModRM:reg (w)                 EVEX.vvvv               ModRM:r/m (r)                         NA

Description
Divides the low single-precision floating-point value in the first source operand by the low single-precision floating-
point value in the second source operand, and stores the single-precision floating-point result in the destination
operand. The second source operand can be an XMM register or a 32-bit memory location.
128-bit Legacy SSE version: The first source operand and the destination operand are the same. Bits (MAX_VL-
1:32) of the corresponding YMM destination register remain unchanged.
VEX.128 encoded version: The first source operand is an xmm register encoded by VEX.vvvv. The three high-order
doublewords of the destination operand are copied from the first source operand. Bits (MAX_VL-1:128) of the
destination register are zeroed.
EVEX.128 encoded version: The first source operand is an xmm register encoded by EVEX.vvvv. The doubleword
elements of the destination operand at bits 127:32 are copied from the first source operand. Bits (MAX_VL-1:128)
of the destination register are zeroed.
EVEX version: The low doubleword element of the destination is updated according to the writemask.
Software should ensure VDIVSS is encoded with VEX.L=0. Encoding VDIVSS with VEX.L=1 may encounter unpre-
dictable behavior across different processor generations.





Operation
VDIVSS (EVEX encoded version)
IF (EVEX.b = 1) AND SRC2 *is a register*
    THEN
         SET_RM(EVEX.RC);
    ELSE
         SET_RM(MXCSR.RM);
FI;
IF k1[0] or *no writemask*
    THEN      DEST[31:0] <- SRC1[31:0] / SRC2[31:0]
    ELSE
         IF *merging-masking*               ; merging-masking
              THEN *DEST[31:0] remains unchanged*
              ELSE                          ; zeroing-masking
                  THEN DEST[31:0] <- 0
         FI;
FI;
DEST[127:32] <- SRC1[127:32]
DEST[MAX_VL-1:128] <- 0

VDIVSS (VEX.128 encoded version)
DEST[31:0] <-SRC1[31:0] / SRC2[31:0]
DEST[127:32] <-SRC1[127:32]
DEST[MAX_VL-1:128] <-0

DIVSS (128-bit Legacy SSE version)
DEST[31:0] <-DEST[31:0] / SRC[31:0]
DEST[MAX_VL-1:32] (Unmodified)

Intel C/C++ Compiler Intrinsic Equivalent
VDIVSS __m128 _mm_mask_div_ss(__m128 s, __mmask8 k, __m128 a, __m128 b);
VDIVSS __m128 _mm_maskz_div_ss( __mmask8 k, __m128 a, __m128 b);
VDIVSS __m128 _mm_div_round_ss( __m128 a, __m128 b, int);
VDIVSS __m128 _mm_mask_div_round_ss(__m128 s, __mmask8 k, __m128 a, __m128 b, int);
VDIVSS __m128 _mm_maskz_div_round_ss( __mmask8 k, __m128 a, __m128 b, int);
DIVSS __m128 _mm_div_ss(__m128 a, __m128 b);

SIMD Floating-Point Exceptions
Overflow, Underflow, Invalid, Divide-by-Zero, Precision, Denormal

Other Exceptions
VEX-encoded instructions, see Exceptions Type 3.
EVEX-encoded instructions, see Exceptions Type E3.




');
INSERT INTO "instructions" VALUES('x86_64','DPPD','DPPD - Dot Product of Packed Double Precision Floating-Point Values
 Opcode/                                         Op/      64/32-bit CPUID        Description
 Instruction                                     En       Mode      Feature
                                                                    Flag
 66 0F 3A 41 /r ib                               RMI      V/V       SSE4_1       Selectively multiply packed DP floating-point
 DPPD xmm1, xmm2/m128, imm8                                                      values from xmm1 with packed DP floating-
                                                                                 point values from xmm2, add and selectively
                                                                                 store the packed DP floating-point values to
                                                                                 xmm1.
 VEX.NDS.128.66.0F3A.WIG 41 /r ib                RVMI V/V           AVX          Selectively multiply packed DP floating-point
 VDPPD xmm1,xmm2, xmm3/m128, imm8                                                values from xmm2 with packed DP floating-
                                                                                 point values from xmm3, add and selectively
                                                                                 store the packed DP floating-point values to
                                                                                 xmm1.



                                           Instruction Operand Encoding
  Op/En               Operand 1            Operand 2                      Operand 3                         Operand 4
   RMI           ModRM:reg (r, w)         ModRM:r/m (r)                     imm8                                NA
   RVMI              ModRM:reg (w)         VEX.vvvv (r)                 ModRM:r/m (r)                          imm8

Description
Conditionally multiplies the packed double-precision floating-point values in the destination operand (first operand)
with the packed double-precision floating-point values in the source (second operand) depending on a mask
extracted from bits [5:4] of the immediate operand (third operand). If a condition mask bit is zero, the corre-
sponding multiplication is replaced by a value of 0.0 in the manner described by Section 12.8.4 of Intel 64 and
IA-32 Architectures Software Developer''s Manual, Volume 1.
The two resulting double-precision values are summed into an intermediate result. The intermediate result is
conditionally broadcasted to the destination using a broadcast mask specified by bits [1:0] of the immediate byte.
If a broadcast mask bit is "1", the intermediate result is copied to the corresponding qword element in the destina-
tion operand. If a broadcast mask bit is zero, the corresponding element in the destination is set to zero.
DPPD follows the NaN forwarding rules stated in the Software Developer''s Manual, vol. 1, table 4.7. These rules do
not cover horizontal prioritization of NaNs. Horizontal propagation of NaNs to the destination and the positioning of
those NaNs in the destination is implementation dependent. NaNs on the input sources or computationally gener-
ated NaNs will have at least one NaN propagated to the destination.
128-bit Legacy SSE version: The second source can be an XMM register or an 128-bit memory location. The desti-
nation is not distinct from the first source XMM register and the upper bits (VLMAX-1:128) of the corresponding
YMM register destination are unmodified.
VEX.128 encoded version: the first source operand is an XMM register or 128-bit memory location. The destination
operand is an XMM register. The upper bits (VLMAX-1:128) of the corresponding YMM register destination are
zeroed.
If VDPPD is encoded with VEX.L= 1, an attempt to execute the instruction encoded with VEX.L= 1 will cause an
#UD exception.





Operation
DP_primitive (SRC1, SRC2)
IF (imm8[4] = 1)
     THEN Temp1[63:0] <- DEST[63:0] * SRC[63:0]; // update SIMD exception flags
     ELSE Temp1[63:0] <- +0.0; FI;
IF (imm8[5] = 1)
     THEN Temp1[127:64] <- DEST[127:64] * SRC[127:64]; // update SIMD exception flags
     ELSE Temp1[127:64] <- +0.0; FI;
/* if unmasked exception reported, execute exception handler*/

Temp2[63:0] <- Temp1[63:0] + Temp1[127:64]; // update SIMD exception flags
/* if unmasked exception reported, execute exception handler*/

IF (imm8[0] = 1)
     THEN DEST[63:0] <- Temp2[63:0];
     ELSE DEST[63:0] <- +0.0; FI;
IF (imm8[1] = 1)
     THEN DEST[127:64] <- Temp2[63:0];
     ELSE DEST[127:64] <- +0.0; FI;

DPPD (128-bit Legacy SSE version)
DEST[127:0]<-DP_Primitive(SRC1[127:0], SRC2[127:0]);
DEST[VLMAX-1:128] (Unmodified)

VDPPD (VEX.128 encoded version)
DEST[127:0]<-DP_Primitive(SRC1[127:0], SRC2[127:0]);
DEST[VLMAX-1:128] <- 0

Flags Affected
None

Intel C/C++ Compiler Intrinsic Equivalent
DPPD:     __m128d _mm_dp_pd ( __m128d a, __m128d b, const int mask);

SIMD Floating-Point Exceptions
Overflow, Underflow, Invalid, Precision, Denormal
Exceptions are determined separately for each add and multiply operation. Unmasked exceptions will leave the
destination untouched.

Other Exceptions
See Exceptions Type 2; additionally
#UD                       If VEX.L= 1.




');
INSERT INTO "instructions" VALUES('x86_64','DPPS','DPPS - Dot Product of Packed Single Precision Floating-Point Values
 Opcode/                                        Op/    64/32-bit CPUID         Description
 Instruction                                    En     Mode      Feature
                                                                 Flag
 66 0F 3A 40 /r ib                              RMI    V/V       SSE4_1        Selectively multiply packed SP floating-point
 DPPS xmm1, xmm2/m128, imm8                                                    values from xmm1 with packed SP floating-
                                                                               point values from xmm2, add and selectively
                                                                               store the packed SP floating-point values or
                                                                               zero values to xmm1.
 VEX.NDS.128.66.0F3A.WIG 40 /r ib               RVMI V/V         AVX           Multiply packed SP floating point values from
 VDPPS xmm1,xmm2, xmm3/m128, imm8                                              xmm1 with packed SP floating point values
                                                                               from xmm2/mem selectively add and store to
                                                                               xmm1.
 VEX.NDS.256.66.0F3A.WIG 40 /r ib               RVMI V/V         AVX           Multiply packed single-precision floating-point
 VDPPS ymm1, ymm2, ymm3/m256, imm8                                             values from ymm2 with packed SP floating
                                                                               point values from ymm3/mem, selectively add
                                                                               pairs of elements and store to ymm1.



                                           Instruction Operand Encoding
   Op/En                Operand 1            Operand 2                   Operand 3                         Operand 4
    RMI              ModRM:reg (r, w)       ModRM:r/m (r)                   imm8                               NA
   RVMI              ModRM:reg (w)           VEX.vvvv (r)              ModRM:r/m (r)                          imm8

Description
Conditionally multiplies the packed single precision floating-point values in the destination operand (first operand)
with the packed single-precision floats in the source (second operand) depending on a mask extracted from the
high 4 bits of the immediate byte (third operand). If a condition mask bit in Imm8[7:4] is zero, the corresponding
multiplication is replaced by a value of 0.0 in the manner described by Section 12.8.4 of Intel 64 and IA-32 Archi-
tectures Software Developer''s Manual, Volume 1.
The four resulting single-precision values are summed into an intermediate result. The intermediate result is condi-
tionally broadcasted to the destination using a broadcast mask specified by bits [3:0] of the immediate byte.
If a broadcast mask bit is "1", the intermediate result is copied to the corresponding dword element in the destina-
tion operand. If a broadcast mask bit is zero, the corresponding element in the destination is set to zero.
DPPS follows the NaN forwarding rules stated in the Software Developer''s Manual, vol. 1, table 4.7. These rules do
not cover horizontal prioritization of NaNs. Horizontal propagation of NaNs to the destination and the positioning of
those NaNs in the destination is implementation dependent. NaNs on the input sources or computationally gener-
ated NaNs will have at least one NaN propagated to the destination.
128-bit Legacy SSE version: The second source can be an XMM register or an 128-bit memory location. The desti-
nation is not distinct from the first source XMM register and the upper bits (VLMAX-1:128) of the corresponding
YMM register destination are unmodified.
VEX.128 encoded version: the first source operand is an XMM register or 128-bit memory location. The destination
operand is an XMM register. The upper bits (VLMAX-1:128) of the corresponding YMM register destination are
zeroed.
VEX.256 encoded version: The first source operand is a YMM register. The second source operand can be a YMM
register or a 256-bit memory location. The destination operand is a YMM register.





Operation
DP_primitive (SRC1, SRC2)
IF (imm8[4] = 1)
     THEN Temp1[31:0] <- DEST[31:0] * SRC[31:0]; // update SIMD exception flags
     ELSE Temp1[31:0] <- +0.0; FI;
IF (imm8[5] = 1)
     THEN Temp1[63:32] <- DEST[63:32] * SRC[63:32]; // update SIMD exception flags
     ELSE Temp1[63:32] <- +0.0; FI;
IF (imm8[6] = 1)
     THEN Temp1[95:64] <- DEST[95:64] * SRC[95:64]; // update SIMD exception flags
     ELSE Temp1[95:64] <- +0.0; FI;
IF (imm8[7] = 1)
     THEN Temp1[127:96] <- DEST[127:96] * SRC[127:96]; // update SIMD exception flags
     ELSE Temp1[127:96] <- +0.0; FI;

Temp2[31:0] <- Temp1[31:0] + Temp1[63:32]; // update SIMD exception flags
/* if unmasked exception reported, execute exception handler*/
Temp3[31:0] <- Temp1[95:64] + Temp1[127:96]; // update SIMD exception flags
/* if unmasked exception reported, execute exception handler*/
Temp4[31:0] <- Temp2[31:0] + Temp3[31:0]; // update SIMD exception flags
/* if unmasked exception reported, execute exception handler*/

IF (imm8[0] = 1)
     THEN DEST[31:0] <- Temp4[31:0];
     ELSE DEST[31:0] <- +0.0; FI;
IF (imm8[1] = 1)
     THEN DEST[63:32] <- Temp4[31:0];
     ELSE DEST[63:32] <- +0.0; FI;
IF (imm8[2] = 1)
     THEN DEST[95:64] <- Temp4[31:0];
     ELSE DEST[95:64] <- +0.0; FI;
IF (imm8[3] = 1)
     THEN DEST[127:96] <- Temp4[31:0];
     ELSE DEST[127:96] <- +0.0; FI;

DPPS (128-bit Legacy SSE version)
DEST[127:0]<-DP_Primitive(SRC1[127:0], SRC2[127:0]);
DEST[VLMAX-1:128] (Unmodified)

VDPPS (VEX.128 encoded version)
DEST[127:0]<-DP_Primitive(SRC1[127:0], SRC2[127:0]);
DEST[VLMAX-1:128] <- 0

VDPPS (VEX.256 encoded version)
DEST[127:0]<-DP_Primitive(SRC1[127:0], SRC2[127:0]);
DEST[255:128]<-DP_Primitive(SRC1[255:128], SRC2[255:128]);

Flags Affected
None





Intel C/C++ Compiler Intrinsic Equivalent
(V)DPPS:    __m128 _mm_dp_ps ( __m128 a, __m128 b, const int mask);

VDPPS:      __m256 _mm256_dp_ps ( __m256 a, __m256 b, const int mask);

SIMD Floating-Point Exceptions
Overflow, Underflow, Invalid, Precision, Denormal
Exceptions are determined separately for each add and multiply operation, in the order of their execution.
Unmasked exceptions will leave the destination operands unchanged.

Other Exceptions
See Exceptions Type 2.




');
INSERT INTO "instructions" VALUES('x86_64','EMMS','EMMS-Empty MMX Technology State
Opcode                  Instruction                 Op/   64-Bit   Compat/ Description
                                                    En    Mode     Leg Mode
0F 77                   EMMS                        NP    Valid    Valid      Set the x87 FPU tag word to empty.



                                              Instruction Operand Encoding
    Op/En             Operand 1                  Operand 2                 Operand 3                  Operand 4
      NP                 NA                          NA                       NA                         NA

Description
Sets the values of all the tags in the x87 FPU tag word to empty (all 1s). This operation marks the x87 FPU data
registers (which are aliased to the MMX technology registers) as available for use by x87 FPU floating-point instruc-
tions. (See Figure 8-7 in the Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 1, for the
format of the x87 FPU tag word.) All other MMX instructions (other than the EMMS instruction) set all the tags in
x87 FPU tag word to valid (all 0s).
The EMMS instruction must be used to clear the MMX technology state at the end of all MMX technology procedures
or subroutines and before calling other procedures or subroutines that may execute x87 floating-point instructions.
If a floating-point instruction loads one of the registers in the x87 FPU data register stack before the x87 FPU tag
word has been reset by the EMMS instruction, an x87 floating-point register stack overflow can occur that will
result in an x87 floating-point exception or incorrect result.
EMMS operation is the same in non-64-bit modes and 64-bit mode.

Operation
x87FPUTagWord <- FFFFH;

Intel C/C++ Compiler Intrinsic Equivalent
void _mm_empty()

Flags Affected
None

Protected Mode Exceptions
#UD                   If CR0.EM[bit 2] = 1.
#NM                   If CR0.TS[bit 3] = 1.
#MF                   If there is a pending FPU exception.
#UD                   If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.



');
INSERT INTO "instructions" VALUES('x86_64','ENTER','ENTER-Make Stack Frame for Procedure Parameters
Opcode                 Instruction              Op/      64-Bit   Compat/ Description
                                                En       Mode     Leg Mode
C8 iw 00               ENTER imm16, 0           II       Valid    Valid       Create a stack frame for a procedure.
C8 iw 01               ENTER imm16,1            II       Valid    Valid       Create a stack frame with a nested pointer for
                                                                              a procedure.
C8 iw ib               ENTER imm16, imm8        II       Valid    Valid       Create a stack frame with nested pointers for
                                                                              a procedure.



                                           Instruction Operand Encoding
    Op/En          Operand 1                 Operand 2                    Operand 3                     Operand 4
      II              iw                       imm8                          NA                            NA

Description
Creates a stack frame (comprising of space for dynamic storage and 1-32 frame pointer storage) for a procedure.
The first operand (imm16) specifies the size of the dynamic storage in the stack frame (that is, the number of bytes
of dynamically allocated on the stack for the procedure). The second operand (imm8) gives the lexical nesting level
(0 to 31) of the procedure. The nesting level (imm8 mod 32) and the OperandSize attribute determine the size in
bytes of the storage space for frame pointers.
The nesting level determines the number of frame pointers that are copied into the "display area" of the new stack
frame from the preceding frame. The default size of the frame pointer is the StackAddrSize attribute, but can be
overridden using the 66H prefix. Thus, the OperandSize attribute determines the size of each frame pointer that
will be copied into the stack frame and the data being transferred from SP/ESP/RSP register into the BP/EBP/RBP
register.
The ENTER and companion LEAVE instructions are provided to support block structured languages. The ENTER
instruction (when used) is typically the first instruction in a procedure and is used to set up a new stack frame for
a procedure. The LEAVE instruction is then used at the end of the procedure (just before the RET instruction) to
release the stack frame.
If the nesting level is 0, the processor pushes the frame pointer from the BP/EBP/RBP register onto the stack,
copies the current stack pointer from the SP/ESP/RSP register into the BP/EBP/RBP register, and loads the
SP/ESP/RSP register with the current stack-pointer value minus the value in the size operand. For nesting levels of
1 or greater, the processor pushes additional frame pointers on the stack before adjusting the stack pointer. These
additional frame pointers provide the called procedure with access points to other nested frames on the stack. See
"Procedure Calls for Block-Structured Languages" in Chapter 6 of the Intel 64 and IA-32 Architectures Software
Developer''s Manual, Volume 1, for more information about the actions of the ENTER instruction.
The ENTER instruction causes a page fault whenever a write using the final value of the stack pointer (within the
current stack segment) would do so.
In 64-bit mode, default operation size is 64 bits; 32-bit operation size cannot be encoded. Use of 66H prefix
changes frame pointer operand size to 16 bits.
When the 66H prefix is used and causing the OperandSize attribute to be less than the StackAddrSize, software is
responsible for the following:
.   The companion LEAVE instruction must also use the 66H prefix,
.   The value in the RBP/EBP register prior to executing "66H ENTER" must be within the same 16KByte region of
    the current stack pointer (RSP/ESP), such that the value of RBP/EBP after "66H ENTER" remains a valid address
    in the stack. This ensures "66H LEAVE" can restore 16-bits of data from the stack.





Operation
AllocSize <- imm16;
NestingLevel <- imm8 MOD 32;
IF (OperandSize = 64)
    THEN
         Push(RBP); (* RSP decrements by 8 *)
         FrameTemp <- RSP;
    ELSE IF OperandSize = 32
         THEN
             Push(EBP); (* (E)SP decrements by 4 *)
             FrameTemp <- ESP; FI;
    ELSE (* OperandSize = 16 *)
             Push(BP); (* RSP or (E)SP decrements by 2 *)
             FrameTemp <- SP;
FI;

IF NestingLevel = 0
    THEN GOTO CONTINUE;
FI;

IF (NestingLevel > 1)
    THEN FOR i <- 1 to (NestingLevel - 1)
         DO
             IF (OperandSize = 64)
                  THEN
                      RBP <- RBP - 8;
                      Push([RBP]); (* Quadword push *)
                  ELSE IF OperandSize = 32
                      THEN
                           IF StackSize = 32
                                EBP <- EBP - 4;
                                Push([EBP]); (* Doubleword push *)
                           ELSE (* StackSize = 16 *)
                                BP <- BP - 4;
                                Push([BP]); (* Doubleword push *)
                           FI;
                      FI;
                  ELSE (* OperandSize = 16 *)
                      IF StackSize = 32
                           THEN
                                EBP <- EBP - 2;
                                Push([EBP]); (* Word push *)
                           ELSE (* StackSize = 16 *)
                                BP <- BP - 2;
                                Push([BP]); (* Word push *)
                      FI;
                  FI;
    OD;
FI;

IF (OperandSize = 64) (* nestinglevel 1 *)
    THEN
        Push(FrameTemp); (* Quadword push and RSP decrements by 8 *)
    ELSE IF OperandSize = 32




          THEN
               Push(FrameTemp); FI; (* Doubleword push and (E)SP decrements by 4 *)
      ELSE (* OperandSize = 16 *)
               Push(FrameTemp); (* Word push and RSP|ESP|SP decrements by 2 *)
FI;

CONTINUE:
IF 64-Bit Mode (StackSize = 64)
    THEN
             RBP <- FrameTemp;
             RSP <- RSP - AllocSize;
    ELSE IF OperandSize = 32
         THEN
             EBP <- FrameTemp;
             ESP <- ESP - AllocSize; FI;
    ELSE (* OperandSize = 16 *)
             BP <- FrameTemp[15:1]; (* Bits 16 and above of applicable RBP/EBP are unmodified *)
             SP <- SP - AllocSize;
FI;

END;

Flags Affected
None.

Protected Mode Exceptions
#SS(0)                   If the new value of the SP or ESP register is outside the stack segment limit.
#PF(fault-code)          If a page fault occurs or if a write using the final value of the stack pointer (within the current
                         stack segment) would cause a page fault.
#UD                      If the LOCK prefix is used.

Real-Address Mode Exceptions
#SS                      If the new value of the SP or ESP register is outside the stack segment limit.
#UD                      If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#SS(0)                   If the new value of the SP or ESP register is outside the stack segment limit.
#PF(fault-code)          If a page fault occurs or if a write using the final value of the stack pointer (within the current
                         stack segment) would cause a page fault.
#UD                      If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)                   If the stack address is in a non-canonical form.
#PF(fault-code)          If a page fault occurs or if a write using the final value of the stack pointer (within the current
                         stack segment) would cause a page fault.
#UD                      If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','EXTRACTPS','EXTRACTPS-Extract Packed Floating-Point Values
 Opcode/                                         Op /    64/32         CPUID     Description
 Instruction                                     En      bit Mode      Feature
                                                         Support       Flag
 66 0F 3A 17 /r ib                               RMI     VV            SSE4_1    Extract one single-precision floating-point value
 EXTRACTPS reg/m32, xmm1, imm8                                                   from xmm1 at the offset specified by imm8 and
                                                                                 store the result in reg or m32. Zero extend the
                                                                                 results in 64-bit register if applicable.
 VEX.128.66.0F3A.WIG 17 /r ib                    RMI     V/V           AVX       Extract one single-precision floating-point value
 VEXTRACTPS reg/m32, xmm1, imm8                                                  from xmm1 at the offset specified by imm8 and
                                                                                 store the result in reg or m32. Zero extend the
                                                                                 results in 64-bit register if applicable.
 EVEX.128.66.0F3A.WIG 17 /r ib                   T1S     V/V           AVX512F   Extract one single-precision floating-point value
 VEXTRACTPS reg/m32, xmm1, imm8                                                  from xmm1 at the offset specified by imm8 and
                                                                                 store the result in reg or m32. Zero extend the
                                                                                 results in 64-bit register if applicable.



                                                       Instruction Operand Encoding
        Op/En                  Operand 1                        Operand 2               Operand 3                   Operand 4
         RMI                ModRM:r/m (w)                      ModRM:reg (r)              Imm8                          NA
         T1S                ModRM:r/m (w)                      ModRM:reg (r)              Imm8                          NA

Description
Extracts a single-precision floating-point value from the source operand (second operand) at the 32-bit offset spec-
ified from imm8. Immediate bits higher than the most significant offset for the vector length are ignored.
The extracted single-precision floating-point value is stored in the low 32-bits of the destination operand
In 64-bit mode, destination register operand has default operand size of 64 bits. The upper 32-bits of the register
are filled with zero. REX.W is ignored.
VEX.128 and EVEX encoded version: When VEX.W1 or EVEX.W1 form is used in 64-bit mode with a general
purpose register (GPR) as a destination operand, the packed single quantity is zero extended to 64 bits.
VEX.vvvv/EVEX.vvvv is reserved and must be 1111b otherwise instructions will #UD.
128-bit Legacy SSE version: When a REX.W prefix is used in 64-bit mode with a general purpose register (GPR) as
a destination operand, the packed single quantity is zero extended to 64 bits.
The source register is an XMM register. Imm8[1:0] determine the starting DWORD offset from which to extract the
32-bit floating-point value.
If VEXTRACTPS is encoded with VEX.L= 1, an attempt to execute the instruction encoded with VEX.L= 1 will cause
an #UD exception.

Operation
VEXTRACTPS (EVEX and VEX.128 encoded version)
SRC_OFFSET <- IMM8[1:0]
IF (64-Bit Mode and DEST is register)
    DEST[31:0] <- (SRC[127:0] >> (SRC_OFFSET*32)) AND 0FFFFFFFFh
    DEST[63:32] <- 0
ELSE
    DEST[31:0] <- (SRC[127:0] >> (SRC_OFFSET*32)) AND 0FFFFFFFFh
FI





EXTRACTPS (128-bit Legacy SSE version)
SRC_OFFSET <-IMM8[1:0]
IF (64-Bit Mode and DEST is register)
    DEST[31:0] <-(SRC[127:0] >> (SRC_OFFSET*32)) AND 0FFFFFFFFh
    DEST[63:32] <-0
ELSE
    DEST[31:0] <-(SRC[127:0] >> (SRC_OFFSET*32)) AND 0FFFFFFFFh
FI

Intel C/C++ Compiler Intrinsic Equivalent
EXTRACTPS int _mm_extract_ps (__m128 a, const int nidx);

SIMD Floating-Point Exceptions
None

Other Exceptions
VEX-encoded instructions, see Exceptions Type 5; Additionally
EVEX-encoded instructions, see Exceptions Type E9NF.
#UD                   IF VEX.L = 0.
#UD                   If VEX.vvvv != 1111B or EVEX.vvvv != 1111B.




');
INSERT INTO "instructions" VALUES('x86_64','F2XM1','F2XM1-Compute 2x-1
Opcode                   Instruction              64-Bit      Compat/       Description
                                                  Mode        Leg Mode
D9 F0                    F2XM1                    Valid       Valid         Replace ST(0) with (2ST(0) - 1).



Description
Computes the exponential value of 2 to the power of the source operand minus 1. The source operand is located in
register ST(0) and the result is also stored in ST(0). The value of the source operand must lie in the range -1.0 to
+1.0. If the source value is outside this range, the result is undefined.
The following table shows the results obtained when computing the exponential value of various classes of
numbers, assuming that neither overflow nor underflow occurs.
                                         Table 3-16. Results Obtained from F2XM1
                           ST(0) SRC                                                   ST(0) DEST
                           - 1.0 to -0                                                 - 0.5 to - 0
                               -0                                                           -0
                               +0                                                           +0
                           + 0 to +1.0                                                  + 0 to 1.0

Values other than 2 can be exponentiated using the following formula:

     xy <- 2(y * log2x)


This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
ST(0) <- (2ST(0) - 1);

FPU Flags Affected
C1                       Set to 0 if stack underflow occurred.
                         Set if result was rounded up; cleared otherwise.
C0, C2, C3               Undefined.

Floating-Point Exceptions
#IS                      Stack underflow occurred.
#IA                      Source operand is an SNaN value or unsupported format.
#D                       Source is a denormal value.
#U                       Result is too small for destination format.
#P                       Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#NM                      CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                      If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.



Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FABS','FABS-Absolute Value
 Opcode                 Instruction               64-Bit   Compat/    Description
                                                  Mode     Leg Mode
 D9 E1                  FABS                      Valid    Valid      Replace ST with its absolute value.



Description
Clears the sign bit of ST(0) to create the absolute value of the operand. The following table shows the results
obtained when creating the absolute value of various classes of numbers.

                                        Table 3-17. Results Obtained from FABS
                            ST(0) SRC                                                ST(0) DEST
                               -inf                                                       +inf
                               -F                                                         +F
                               -0                                                         +0
                               +0                                                         +0
                               +F                                                         +F
                               +inf                                                       +inf
                               NaN                                                       NaN
 NOTES:
 F Means finite floating-point value.

This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
ST(0) <- |ST(0)|;

FPU Flags Affected
C1                      Set to 0.
C0, C2, C3              Undefined.

Floating-Point Exceptions
#IS                     Stack underflow occurred.

Protected Mode Exceptions
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                     If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.


');
INSERT INTO "instructions" VALUES('x86_64','FADD','FADD/FADDP/FIADD-Add
Opcode              Instruction               64-Bit     Compat/     Description
                                              Mode       Leg Mode
D8 /0               FADD m32fp                Valid      Valid       Add m32fp to ST(0) and store result in ST(0).
DC /0               FADD m64fp                Valid      Valid       Add m64fp to ST(0) and store result in ST(0).
D8 C0+i             FADD ST(0), ST(i)         Valid      Valid       Add ST(0) to ST(i) and store result in ST(0).
DC C0+i             FADD ST(i), ST(0)         Valid      Valid       Add ST(i) to ST(0) and store result in ST(i).
DE C0+i             FADDP ST(i), ST(0)        Valid      Valid       Add ST(0) to ST(i), store result in ST(i), and pop the
                                                                     register stack.
DE C1               FADDP                     Valid      Valid       Add ST(0) to ST(1), store result in ST(1), and pop the
                                                                     register stack.
DA /0               FIADD m32int              Valid      Valid       Add m32int to ST(0) and store result in ST(0).
DE /0               FIADD m16int              Valid      Valid       Add m16int to ST(0) and store result in ST(0).



Description
Adds the destination and source operands and stores the sum in the destination location. The destination operand
is always an FPU register; the source operand can be a register or a memory location. Source operands in memory
can be in single-precision or double-precision floating-point format or in word or doubleword integer format.
The no-operand version of the instruction adds the contents of the ST(0) register to the ST(1) register. The one-
operand version adds the contents of a memory location (either a floating-point or an integer value) to the contents
of the ST(0) register. The two-operand version, adds the contents of the ST(0) register to the ST(i) register or vice
versa. The value in ST(0) can be doubled by coding:

   FADD ST(0), ST(0);
The FADDP instructions perform the additional operation of popping the FPU register stack after storing the result.
To pop the register stack, the processor marks the ST(0) register as empty and increments the stack pointer (TOP)
by 1. (The no-operand version of the floating-point add instructions always results in the register stack being
popped. In some assemblers, the mnemonic for this instruction is FADD rather than FADDP.)
The FIADD instructions convert an integer source operand to double extended-precision floating-point format
before performing the addition.
The table on the following page shows the results obtained when adding various classes of numbers, assuming that
neither overflow nor underflow occurs.
When the sum of two operands with opposite signs is 0, the result is +0, except for the round toward -inf mode, in
which case the result is -0. When the source operand is an integer 0, it is treated as a +0.
When both operand are infinities of the same sign, the result is inf of the expected sign. If both operands are infini-
ties of opposite signs, an invalid-operation exception is generated. See Table 3-18.





                                            Table 3-18. FADD/FADDP/FIADD Results
                                                                          DEST
                               -inf           -F               -0             +0    +F               +inf           NaN
              -inf               -inf           -inf               -inf             -inf    -inf               *            NaN
              - F or - I       -inf           -F               SRC            SRC   +- F or +- 0       +inf           NaN
 SRC          -0               -inf           DEST             -0             +-0    DEST             +inf           NaN
              +0               -inf           DEST             +-0             +0    DEST             +inf           NaN
              + F or + I       -inf           +- F or +- 0       SRC            SRC   +F               +inf           NaN
              +inf               *            +inf               +inf             +inf    +inf               +inf           NaN
              NaN              NaN          NaN              NaN            NaN   NaN              NaN          NaN
 NOTES:
 F Means finite floating-point value.
 I Means integer.
 * Indicates floating-point invalid-arithmetic-operand (#IA) exception.

This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
IF Instruction = FIADD
     THEN
         DEST <- DEST + ConvertToDoubleExtendedPrecisionFP(SRC);
     ELSE (* Source operand is floating-point value *)
         DEST <- DEST + SRC;
FI;

IF Instruction = FADDP
     THEN
         PopRegisterStack;
FI;

FPU Flags Affected
C1                         Set to 0 if stack underflow occurred.
                           Set if result was rounded up; cleared otherwise.
C0, C2, C3                 Undefined.

Floating-Point Exceptions
#IS                        Stack underflow occurred.
#IA                        Operand is an SNaN value or unsupported format.
                           Operands are infinities of unlike sign.
#D                         Source operand is a denormal value.
#U                         Result is too small for destination format.
#O                         Result is too large for destination format.
#P                         Value cannot be represented exactly in destination format.





Protected Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                 If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FADDP','-R:FADD');
INSERT INTO "instructions" VALUES('x86_64','FIADD','-R:FADD');
INSERT INTO "instructions" VALUES('x86_64','FBLD','FBLD-Load Binary Coded Decimal
 Opcode                Instruction                64-Bit   Compat/    Description
                                                  Mode     Leg Mode
 DF /4                 FBLD m80dec                Valid    Valid      Convert BCD value to floating-point and push onto the
                                                                      FPU stack.



Description
Converts the BCD source operand into double extended-precision floating-point format and pushes the value onto
the FPU stack. The source operand is loaded without rounding errors. The sign of the source operand is preserved,
including that of -0.
The packed BCD digits are assumed to be in the range 0 through 9; the instruction does not check for invalid digits
(AH through FH). Attempting to load an invalid encoding produces an undefined result.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
TOP <- TOP - 1;
ST(0) <- ConvertToDoubleExtendedPrecisionFP(SRC);

FPU Flags Affected
C1                      Set to 1 if stack overflow occurred; otherwise, set to 0.
C0, C2, C3              Undefined.

Floating-Point Exceptions
#IS                     Stack overflow occurred.

Protected Mode Exceptions
#GP(0)                  If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                        If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)                  If a memory operand effective address is outside the SS segment limit.
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)         If a page fault occurs.
#AC(0)                  If alignment checking is enabled and an unaligned memory reference is made while the
                        current privilege level is 3.
#UD                     If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                     If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                     If a memory operand effective address is outside the SS segment limit.
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                     If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)                  If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)                  If a memory operand effective address is outside the SS segment limit.
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)         If a page fault occurs.
#AC(0)                  If alignment checking is enabled and an unaligned memory reference is made.
#UD                     If the LOCK prefix is used.



Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FBSTP','FBSTP-Store BCD Integer and Pop
Opcode              Instruction                   64-Bit     Compat/    Description
                                                  Mode       Leg Mode
DF /6               FBSTP m80bcd                  Valid      Valid      Store ST(0) in m80bcd and pop ST(0).



Description
Converts the value in the ST(0) register to an 18-digit packed BCD integer, stores the result in the destination
operand, and pops the register stack. If the source value is a non-integral value, it is rounded to an integer value,
according to rounding mode specified by the RC field of the FPU control word. To pop the register stack, the
processor marks the ST(0) register as empty and increments the stack pointer (TOP) by 1.
The destination operand specifies the address where the first byte destination value is to be stored. The BCD value
(including its sign bit) requires 10 bytes of space in memory.
The following table shows the results obtained when storing various classes of numbers in packed BCD format.
                                                     Table 3-19. FBSTP Results
                              ST(0)                                                           DEST
            - inf or Value Too Large for DEST Format                                              *
                              F<=-1                                                             -D
                            -1 < F < -0                                                         **
                                  -0                                                           -0
                                  +0                                                           +0
                           + 0 < F < +1                                                         **
                              F >= +1                                                           +D
            + inf or Value Too Large for DEST Format                                              *
                               NaN                                                              *
 NOTES:
 F Means finite floating-point value.
 D Means packed-BCD number.
 * Indicates floating-point invalid-operation (#IA) exception.
 ** +-0 or +-1, depending on the rounding mode.

If the converted value is too large for the destination format, or if the source operand is an inf, SNaN, QNAN, or is in
an unsupported format, an invalid-arithmetic-operand condition is signaled. If the invalid-operation exception is
not masked, an invalid-arithmetic-operand exception (#IA) is generated and no value is stored in the destination
operand. If the invalid-operation exception is masked, the packed BCD indefinite value is stored in memory.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
DEST <- BCD(ST(0));
PopRegisterStack;

FPU Flags Affected
C1                      Set to 0 if stack underflow occurred.
                        Set if result was rounded up; cleared otherwise.
C0, C2, C3              Undefined.





Floating-Point Exceptions
#IS                 Stack underflow occurred.
#IA                 Converted value that exceeds 18 BCD digits in length.
                    Source operand is an SNaN, QNaN, +-inf, or in an unsupported format.
#P                  Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#GP(0)              If a segment register is being loaded with a segment selector that points to a non-writable
                    segment.
                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                 If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FCHS','FCHS-Change Sign
Opcode                 Instruction             64-Bit   Compat/    Description
                                               Mode     Leg Mode
D9 E0                  FCHS                    Valid    Valid      Complements sign of ST(0).



Description
Complements the sign bit of ST(0). This operation changes a positive value into a negative value of equal magni-
tude or vice versa. The following table shows the results obtained when changing the sign of various classes of
numbers.
                                                Table 3-20. FCHS Results
                            ST(0) SRC                                              ST(0) DEST
 -inf                                                        +inf
 -F                                                        +F
 -0                                                        +0
 +0                                                        -0
 +F                                                        -F
 +inf                                                        -inf
 NaN                                                       NaN
 NOTES:
 * F means finite floating-point value.

This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
SignBit(ST(0)) <- NOT (SignBit(ST(0)));

FPU Flags Affected
C1                      Set to 0.
C0, C2, C3              Undefined.

Floating-Point Exceptions
#IS                     Stack underflow occurred.

Protected Mode Exceptions
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                     If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.





64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FCLEX','FCLEX/FNCLEX - Clear Exceptions
 Opcode*                Instruction             64-Bit   Compat/    Description
                                                Mode     Leg Mode
 9B DB E2               FCLEX                   Valid    Valid      Clear floating-point exception flags after checking for
                                                                    pending unmasked floating-point exceptions.
 DB E2                  FNCLEX*                 Valid    Valid      Clear floating-point exception flags without checking for
                                                                    pending unmasked floating-point exceptions.
 NOTES:
 * See IA-32 Architecture Compatibility section below.



Description
Clears the floating-point exception flags (PE, UE, OE, ZE, DE, and IE), the exception summary status flag (ES), the
stack fault flag (SF), and the busy flag (B) in the FPU status word. The FCLEX instruction checks for and handles
any pending unmasked floating-point exceptions before clearing the exception flags; the FNCLEX instruction does
not.
The assembler issues two instructions for the FCLEX instruction (an FWAIT instruction followed by an FNCLEX
instruction), and the processor executes each of these instructions separately. If an exception is generated for
either of these instructions, the save EIP points to the instruction that caused the exception.

IA-32 Architecture Compatibility
When operating a Pentium or Intel486 processor in MS-DOS* compatibility mode, it is possible (under unusual
circumstances) for an FNCLEX instruction to be interrupted prior to being executed to handle a pending FPU excep-
tion. See the section titled "No-Wait FPU Instructions Can Get FPU Interrupt in Window" in Appendix D of the Intel
64 and IA-32 Architectures Software Developer''s Manual, Volume 1, for a description of these circumstances. An
FNCLEX instruction cannot be interrupted in this way on later Intel processors, except for the Intel QuarkTM X1000
processor.
This instruction affects only the x87 FPU floating-point exception flags. It does not affect the SIMD floating-point
exception flags in the MXCRS register.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
FPUStatusWord[0:7] <- 0;
FPUStatusWord[15] <- 0;

FPU Flags Affected
The PE, UE, OE, ZE, DE, IE, ES, SF, and B flags in the FPU status word are cleared. The C0, C1, C2, and C3 flags are
undefined.

Floating-Point Exceptions
None

Protected Mode Exceptions
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                     If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.





Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FNCLEX','-R:FCLEX');
INSERT INTO "instructions" VALUES('x86_64','FCMOVcc','FCMOVcc-Floating-Point Conditional Move
Opcode*            Instruction                          64-Bit    Compat/   Description
                                                        Mode      Leg Mode*
DA C0+i            FCMOVB ST(0), ST(i)                  Valid     Valid     Move if below (CF=1).
DA C8+i            FCMOVE ST(0), ST(i)                  Valid     Valid     Move if equal (ZF=1).
DA D0+i            FCMOVBE ST(0), ST(i)                 Valid     Valid     Move if below or equal (CF=1 or ZF=1).
DA D8+i            FCMOVU ST(0), ST(i)                  Valid     Valid     Move if unordered (PF=1).
DB C0+i            FCMOVNB ST(0), ST(i)                 Valid     Valid     Move if not below (CF=0).
DB C8+i            FCMOVNE ST(0), ST(i)                 Valid     Valid     Move if not equal (ZF=0).
DB D0+i            FCMOVNBE ST(0), ST(i)                Valid     Valid     Move if not below or equal (CF=0 and ZF=0).
DB D8+i            FCMOVNU ST(0), ST(i)                 Valid     Valid     Move if not unordered (PF=0).
NOTES:
* See IA-32 Architecture Compatibility section below.



Description
Tests the status flags in the EFLAGS register and moves the source operand (second operand) to the destination
operand (first operand) if the given test condition is true. The condition for each mnemonic os given in the Descrip-
tion column above and in Chapter 8 in the Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume
1. The source operand is always in the ST(i) register and the destination operand is always ST(0).
The FCMOVcc instructions are useful for optimizing small IF constructions. They also help eliminate branching
overhead for IF operations and the possibility of branch mispredictions by the processor.
A processor may not support the FCMOVcc instructions. Software can check if the FCMOVcc instructions are
supported by checking the processor''s feature information with the CPUID instruction (see "COMISS-Compare
Scalar Ordered Single-Precision Floating-Point Values and Set EFLAGS" in this chapter). If both the CMOV and FPU
feature bits are set, the FCMOVcc instructions are supported.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

IA-32 Architecture Compatibility
The FCMOVcc instructions were introduced to the IA-32 Architecture in the P6 family processors and are not avail-
able in earlier IA-32 processors.

Operation
IF condition TRUE
    THEN ST(0) <- ST(i);
FI;

FPU Flags Affected
C1                        Set to 0 if stack underflow occurred.
C0, C2, C3                Undefined.

Floating-Point Exceptions
#IS                       Stack underflow occurred.

Integer Flags Affected
None.





Protected Mode Exceptions
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FCOM','FCOM/FCOMP/FCOMPP-Compare Floating Point Values
 Opcode               Instruction                     64-Bit       Compat/       Description
                                                      Mode         Leg Mode
 D8 /2                FCOM m32fp                      Valid        Valid         Compare ST(0) with m32fp.
 DC /2                FCOM m64fp                      Valid        Valid         Compare ST(0) with m64fp.
 D8 D0+i              FCOM ST(i)                      Valid        Valid         Compare ST(0) with ST(i).
 D8 D1                FCOM                            Valid        Valid         Compare ST(0) with ST(1).
 D8 /3                FCOMP m32fp                     Valid        Valid         Compare ST(0) with m32fp and pop register stack.
 DC /3                FCOMP m64fp                     Valid        Valid         Compare ST(0) with m64fp and pop register stack.
 D8 D8+i              FCOMP ST(i)                     Valid        Valid         Compare ST(0) with ST(i) and pop register stack.
 D8 D9                FCOMP                           Valid        Valid         Compare ST(0) with ST(1) and pop register stack.
 DE D9                FCOMPP                          Valid        Valid         Compare ST(0) with ST(1) and pop register stack
                                                                                 twice.



Description
Compares the contents of register ST(0) and source value and sets condition code flags C0, C2, and C3 in the FPU
status word according to the results (see the table below). The source operand can be a data register or a memory
location. If no source operand is given, the value in ST(0) is compared with the value in ST(1). The sign of zero is
ignored, so that -0.0 is equal to +0.0.
                                          Table 3-21. FCOM/FCOMP/FCOMPP Results
                  Condition                               C3                            C2                          C0
                 ST(0) > SRC                               0                            0                            0
                 ST(0) < SRC                               0                            0                            1
                 ST(0) = SRC                               1                            0                            0
                 Unordered*                                1                            1                            1
 NOTES:
 * Flags not set if unmasked invalid-arithmetic-operand (#IA) exception is generated.

This instruction checks the class of the numbers being compared (see "FXAM-Examine Floating-Point" in this
chapter). If either operand is a NaN or is in an unsupported format, an invalid-arithmetic-operand exception (#IA)
is raised and, if the exception is masked, the condition flags are set to "unordered." If the invalid-arithmetic-
operand exception is unmasked, the condition code flags are not set.
The FCOMP instruction pops the register stack following the comparison operation and the FCOMPP instruction
pops the register stack twice following the comparison operation. To pop the register stack, the processor marks
the ST(0) register as empty and increments the stack pointer (TOP) by 1.
The FCOM instructions perform the same operation as the FUCOM instructions. The only difference is how they
handle QNaN operands. The FCOM instructions raise an invalid-arithmetic-operand exception (#IA) when either or
both of the operands is a NaN value or is in an unsupported format. The FUCOM instructions perform the same
operation as the FCOM instructions, except that they do not generate an invalid-arithmetic-operand exception for
QNaNs.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.





Operation
CASE (relation of operands) OF
   ST > SRC:          C3, C2, C0 <- 000;
   ST < SRC:          C3, C2, C0 <- 001;
   ST = SRC:          C3, C2, C0 <- 100;
ESAC;

IF ST(0) or SRC = NaN or unsupported format
    THEN
         #IA
         IF FPUControlWord.IM = 1
              THEN
                  C3, C2, C0 <- 111;
         FI;
FI;

IF Instruction = FCOMP
     THEN
         PopRegisterStack;
FI;

IF Instruction = FCOMPP
     THEN
         PopRegisterStack;
         PopRegisterStack;
FI;

FPU Flags Affected
C1                     Set to 0.
C0, C2, C3             See table on previous page.

Floating-Point Exceptions
#IS                    Stack underflow occurred.
#IA                    One or both operands are NaN values or have unsupported formats.
                       Register is marked empty.
#D                     One or both operands are denormal values.

Protected Mode Exceptions
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                       If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                    If a memory operand effective address is outside the SS segment limit.
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                    If the LOCK prefix is used.


Virtual-8086 Mode Exceptions
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made.
#UD                    If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)                 If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)                 If the memory address is in a non-canonical form.
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                    If there is a pending x87 FPU exception.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                    If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FCOMP','-R:FCOM');
INSERT INTO "instructions" VALUES('x86_64','FCOMPP','-R:FCOM');
INSERT INTO "instructions" VALUES('x86_64','FCOMI','FCOMI/FCOMIP/ FUCOMI/FUCOMIP-Compare Floating Point Values and Set EFLAGS
 Opcode              Instruction                  64-Bit     Compat/       Description
                                                  Mode       Leg Mode
 DB F0+i             FCOMI ST, ST(i)              Valid      Valid         Compare ST(0) with ST(i) and set status flags accordingly.
 DF F0+i             FCOMIP ST, ST(i)             Valid      Valid         Compare ST(0) with ST(i), set status flags accordingly, and
                                                                           pop register stack.
 DB E8+i             FUCOMI ST, ST(i)             Valid      Valid        Compare ST(0) with ST(i), check for ordered values, and set
                                                                          status flags accordingly.
 DF E8+i             FUCOMIP ST, ST(i)            Valid      Valid         Compare ST(0) with ST(i), check for ordered values, set
                                                                           status flags accordingly, and pop register stack.



Description
Performs an unordered comparison of the contents of registers ST(0) and ST(i) and sets the status flags ZF, PF, and
CF in the EFLAGS register according to the results (see the table below). The sign of zero is ignored for compari-
sons, so that -0.0 is equal to +0.0.
                                   Table 3-22. FCOMI/FCOMIP/ FUCOMI/FUCOMIP Results
                     Comparison Results*                                    ZF                      PF                     CF
                           ST0 > ST(i)                                       0                       0                      0
                           ST0 < ST(i)                                       0                       0                      1
                           ST0 = ST(i)                                       1                       0                      0
                          Unordered**                                        1                       1                      1
 NOTES:
 * See the IA-32 Architecture Compatibility section below.
 ** Flags not set if unmasked invalid-arithmetic-operand (#IA) exception is generated.

An unordered comparison checks the class of the numbers being compared (see "FXAM-Examine Floating-Point"
in this chapter). The FUCOMI/FUCOMIP instructions perform the same operations as the FCOMI/FCOMIP instruc-
tions. The only difference is that the FUCOMI/FUCOMIP instructions raise the invalid-arithmetic-operand exception
(#IA) only when either or both operands are an SNaN or are in an unsupported format; QNaNs cause the condition
code flags to be set to unordered, but do not cause an exception to be generated. The FCOMI/FCOMIP instructions
raise an invalid-operation exception when either or both of the operands are a NaN value of any kind or are in an
unsupported format.
If the operation results in an invalid-arithmetic-operand exception being raised, the status flags in the EFLAGS
register are set only if the exception is masked.
The FCOMI/FCOMIP and FUCOMI/FUCOMIP instructions set the OF, SF and AF flags to zero in the EFLAGS register
(regardless of whether an invalid-operation exception is detected).
The FCOMIP and FUCOMIP instructions also pop the register stack following the comparison operation. To pop the
register stack, the processor marks the ST(0) register as empty and increments the stack pointer (TOP) by 1.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

IA-32 Architecture Compatibility
The FCOMI/FCOMIP/FUCOMI/FUCOMIP instructions were introduced to the IA-32 Architecture in the P6 family
processors and are not available in earlier IA-32 processors.





Operation
CASE (relation of operands) OF
   ST(0) > ST(i):     ZF, PF, CF <- 000;
   ST(0) < ST(i):     ZF, PF, CF <- 001;
   ST(0) = ST(i):     ZF, PF, CF <- 100;
ESAC;

IF Instruction is FCOMI or FCOMIP
     THEN
         IF ST(0) or ST(i) = NaN or unsupported format
              THEN
                   #IA
                   IF FPUControlWord.IM = 1
                        THEN
                             ZF, PF, CF <- 111;
                   FI;
         FI;
FI;

IF Instruction is FUCOMI or FUCOMIP
    THEN
         IF ST(0) or ST(i) = QNaN, but not SNaN or unsupported format
              THEN
                   ZF, PF, CF <- 111;
              ELSE (* ST(0) or ST(i) is SNaN or unsupported format *)
                    #IA;
                   IF FPUControlWord.IM = 1
                         THEN
                             ZF, PF, CF <- 111;
                   FI;
         FI;
FI;

IF Instruction is FCOMIP or FUCOMIP
     THEN
         PopRegisterStack;
FI;

FPU Flags Affected
C1                      Set to 0.
C0, C2, C3              Not affected.

Floating-Point Exceptions
#IS                     Stack underflow occurred.
#IA                     (FCOMI or FCOMIP instruction) One or both operands are NaN values or have unsupported
                        formats.
                        (FUCOMI or FUCOMIP instruction) One or both operands are SNaN values (but not QNaNs) or
                        have undefined formats. Detection of a QNaN value does not raise an invalid-operand excep-
                        tion.





Protected Mode Exceptions
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FCOMIP','-R:FCOMI');
INSERT INTO "instructions" VALUES('x86_64','FUCOMI','-R:FCOMI');
INSERT INTO "instructions" VALUES('x86_64','FUCOMIP','-R:FCOMI');
INSERT INTO "instructions" VALUES('x86_64','FCOS','FCOS- Cosine
Opcode                 Instruction                 64-Bit     Compat/     Description
                                                   Mode       Leg Mode
D9 FF                  FCOS                        Valid      Valid       Replace ST(0) with its approximate cosine.



Description
Computes the approximate cosine of the source operand in register ST(0) and stores the result in ST(0). The
source operand must be given in radians and must be within the range -263 to +263. The following table shows the
results obtained when taking the cosine of various classes of numbers.
                                                       Table 3-23. FCOS Results
                            ST(0) SRC                                                     ST(0) DEST
                               -inf                                                              *
                               -F                                                          -1 to +1
                               -0                                                             +1
                               +0                                                             +1
                               +F                                                          - 1 to + 1
                               +inf                                                              *
                               NaN                                                            NaN
 NOTES:
 F Means finite floating-point value.
 * Indicates floating-point invalid-arithmetic-operand (#IA) exception.

If the source operand is outside the acceptable range, the C2 flag in the FPU status word is set, and the value in
register ST(0) remains unchanged. The instruction does not raise an exception when the source operand is out of
range. It is up to the program to check the C2 flag for out-of-range conditions. Source values outside the range -
263 to +263 can be reduced to the range of the instruction by subtracting an appropriate integer multiple of 2pi.
However, even within the range -263 to +263, inaccurate results can occur because the finite approximation of pi
used internally for argument reduction is not sufficient in all cases. Therefore, for accurate results it is safe to apply
FCOS only to arguments reduced accurately in software, to a value smaller in absolute value than 3pi/8. See the
sections titled "Approximation of Pi" and "Transcendental Instruction Accuracy" in Chapter 8 of the Intel 64 and
IA-32 Architectures Software Developer''s Manual, Volume 1, for a discussion of the proper value to use for pi in
performing such reductions.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
IF |ST(0)| < 263
THEN
     C2 <- 0;
     ST(0) <- FCOS(ST(0)); // approximation of cosine
ELSE (* Source operand is out-of-range *)
     C2 <- 1;
FI;





FPU Flags Affected
C1                   Set to 0 if stack underflow occurred.
                     Set if result was rounded up; cleared otherwise.
                     Undefined if C2 is 1.
C2                   Set to 1 if outside range (-263 < source operand < +263); otherwise, set to 0.
C0, C3               Undefined.

Floating-Point Exceptions
#IS                  Stack underflow occurred.
#IA                  Source operand is an SNaN value, inf, or unsupported format.
#D                   Source is a denormal value.
#P                   Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#NM                  CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                  If there is a pending x87 FPU exception.
#UD                  If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FDECSTP','FDECSTP-Decrement Stack-Top Pointer
 Opcode                Instruction            64-Bit      Compat/       Description
                                              Mode        Leg Mode
 D9 F6                 FDECSTP                Valid       Valid         Decrement TOP field in FPU status word.



Description
Subtracts one from the TOP field of the FPU status word (decrements the top-of-stack pointer). If the TOP field
contains a 0, it is set to 7. The effect of this instruction is to rotate the stack by one position. The contents of the
FPU data registers and tag register are not affected.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
IF TOP = 0
    THEN TOP <- 7;
    ELSE TOP <- TOP - 1;
FI;

FPU Flags Affected
The C1 flag is set to 0. The C0, C2, and C3 flags are undefined.

Floating-Point Exceptions
None.

Protected Mode Exceptions
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                    If there is a pending x87 FPU exception.
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FDIV','FDIV/FDIVP/FIDIV-Divide
Opcode              Instruction                64-Bit   Compat/      Description
                                               Mode     Leg Mode
D8 /6               FDIV m32fp                 Valid    Valid        Divide ST(0) by m32fp and store result in ST(0).
DC /6               FDIV m64fp                 Valid    Valid        Divide ST(0) by m64fp and store result in ST(0).
D8 F0+i             FDIV ST(0), ST(i)          Valid    Valid        Divide ST(0) by ST(i) and store result in ST(0).
DC F8+i             FDIV ST(i), ST(0)          Valid    Valid        Divide ST(i) by ST(0) and store result in ST(i).
DE F8+i             FDIVP ST(i), ST(0)         Valid    Valid        Divide ST(i) by ST(0), store result in ST(i), and pop the
                                                                     register stack.
DE F9               FDIVP                      Valid    Valid        Divide ST(1) by ST(0), store result in ST(1), and pop
                                                                     the register stack.
DA /6               FIDIV m32int               Valid    Valid        Divide ST(0) by m32int and store result in ST(0).
DE /6               FIDIV m16int               Valid    Valid        Divide ST(0) by m16int and store result in ST(0).



Description
Divides the destination operand by the source operand and stores the result in the destination location. The desti-
nation operand (dividend) is always in an FPU register; the source operand (divisor) can be a register or a memory
location. Source operands in memory can be in single-precision or double-precision floating-point format, word or
doubleword integer format.
The no-operand version of the instruction divides the contents of the ST(1) register by the contents of the ST(0)
register. The one-operand version divides the contents of the ST(0) register by the contents of a memory location
(either a floating-point or an integer value). The two-operand version, divides the contents of the ST(0) register by
the contents of the ST(i) register or vice versa.
The FDIVP instructions perform the additional operation of popping the FPU register stack after storing the result.
To pop the register stack, the processor marks the ST(0) register as empty and increments the stack pointer (TOP)
by 1. The no-operand version of the floating-point divide instructions always results in the register stack being
popped. In some assemblers, the mnemonic for this instruction is FDIV rather than FDIVP.
The FIDIV instructions convert an integer source operand to double extended-precision floating-point format
before performing the division. When the source operand is an integer 0, it is treated as a +0.
If an unmasked divide-by-zero exception (#Z) is generated, no result is stored; if the exception is masked, an inf of
the appropriate sign is stored in the destination operand.
The following table shows the results obtained when dividing various classes of numbers, assuming that neither
overflow nor underflow occurs.





                                             Table 3-24. FDIV/FDIVP/FIDIV Results
                                                                           DEST
                              -inf             -F              -0              +0    +F        +inf           NaN
               -inf             *              +0              +0              -0    -0        *            NaN
               -F             +inf             +F              +0              -0    -F        -inf           NaN
               -I             +inf             +F              +0              -0    -F        -inf           NaN
     SRC       -0             +inf             **              *               *     **        -inf           NaN
               +0             -inf             **              *               *     **        +inf           NaN
               +I             -inf             -F              -0              +0    +F        +inf           NaN
               +F             -inf             -F              -0              +0    +F        +inf           NaN
               +inf             *              -0              -0              +0    +0        *            NaN
               NaN            NaN            NaN             NaN             NaN   NaN       NaN          NaN
  NOTES:
  F Means finite floating-point value.
  I Means integer.
  * Indicates floating-point invalid-arithmetic-operand (#IA) exception.
  ** Indicates floating-point zero-divide (#Z) exception.

This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
IF SRC = 0
    THEN
        #Z;
    ELSE
        IF Instruction is FIDIV
              THEN
                  DEST <- DEST / ConvertToDoubleExtendedPrecisionFP(SRC);
              ELSE (* Source operand is floating-point value *)
                  DEST <- DEST / SRC;
        FI;
FI;

IF Instruction = FDIVP
     THEN
         PopRegisterStack;
FI;

FPU Flags Affected
C1                        Set to 0 if stack underflow occurred.
                          Set if result was rounded up; cleared otherwise.
C0, C2, C3                Undefined.





Floating-Point Exceptions
#IS                 Stack underflow occurred.
#IA                 Operand is an SNaN value or unsupported format.
                    +-inf / +-inf; +-0 / +-0
#D                  Source is a denormal value.
#Z                  DEST / +-0, where DEST is not equal to +-0.
#U                  Result is too small for destination format.
#O                  Result is too large for destination format.
#P                  Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                 If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FDIVP','-R:FDIV');
INSERT INTO "instructions" VALUES('x86_64','FIDIV','-R:FDIV');
INSERT INTO "instructions" VALUES('x86_64','FDIVR','FDIVR/FDIVRP/FIDIVR-Reverse Divide
Opcode               Instruction             64-Bit   Compat/     Description
                                             Mode     Leg Mode
D8 /7                FDIVR m32fp             Valid    Valid       Divide m32fp by ST(0) and store result in ST(0).
DC /7                FDIVR m64fp             Valid    Valid       Divide m64fp by ST(0) and store result in ST(0).
D8 F8+i              FDIVR ST(0), ST(i)      Valid    Valid       Divide ST(i) by ST(0) and store result in ST(0).
DC F0+i              FDIVR ST(i), ST(0)      Valid    Valid       Divide ST(0) by ST(i) and store result in ST(i).
DE F0+i              FDIVRP ST(i), ST(0)     Valid    Valid       Divide ST(0) by ST(i), store result in ST(i), and pop the
                                                                  register stack.
DE F1                FDIVRP                  Valid    Valid       Divide ST(0) by ST(1), store result in ST(1), and pop the
                                                                  register stack.
DA /7                FIDIVR m32int           Valid    Valid       Divide m32int by ST(0) and store result in ST(0).
DE /7                FIDIVR m16int           Valid    Valid       Divide m16int by ST(0) and store result in ST(0).



Description
Divides the source operand by the destination operand and stores the result in the destination location. The desti-
nation operand (divisor) is always in an FPU register; the source operand (dividend) can be a register or a memory
location. Source operands in memory can be in single-precision or double-precision floating-point format, word or
doubleword integer format.
These instructions perform the reverse operations of the FDIV, FDIVP, and FIDIV instructions. They are provided to
support more efficient coding.
The no-operand version of the instruction divides the contents of the ST(0) register by the contents of the ST(1)
register. The one-operand version divides the contents of a memory location (either a floating-point or an integer
value) by the contents of the ST(0) register. The two-operand version, divides the contents of the ST(i) register by
the contents of the ST(0) register or vice versa.
The FDIVRP instructions perform the additional operation of popping the FPU register stack after storing the result.
To pop the register stack, the processor marks the ST(0) register as empty and increments the stack pointer (TOP)
by 1. The no-operand version of the floating-point divide instructions always results in the register stack being
popped. In some assemblers, the mnemonic for this instruction is FDIVR rather than FDIVRP.
The FIDIVR instructions convert an integer source operand to double extended-precision floating-point format
before performing the division.
If an unmasked divide-by-zero exception (#Z) is generated, no result is stored; if the exception is masked, an inf of
the appropriate sign is stored in the destination operand.
The following table shows the results obtained when dividing various classes of numbers, assuming that neither
overflow nor underflow occurs.





                                          Table 3-25. FDIVR/FDIVRP/FIDIVR Results
                                                                          DEST
                                  -inf              -F              -0             +0    +F             +inf            NaN
                   -inf              *             +inf              +inf              -inf    -inf               *           NaN
     SRC           -F             +0              +F              **             **    -F              -0           NaN
                   -I             +0              +F              **             **    -F              -0           NaN
                   -0             +0              +0               *              *    -0              -0           NaN
                   +0             -0              -0               *              *    +0              +0           NaN
                   +I             -0              -F              **             **    +F              +0           NaN
                   +F             -0              -F              **             **    +F              +0           NaN
                   +inf              *              -inf             -inf              +inf    +inf               *           NaN
                  NaN            NaN             NaN             NaN             NaN   NaN            NaN           NaN
 NOTES:
 F Means finite floating-point value.
 I Means integer.
 * Indicates floating-point invalid-arithmetic-operand (#IA) exception.
 ** Indicates floating-point zero-divide (#Z) exception.

When the source operand is an integer 0, it is treated as a +0. This instruction''s operation is the same in non-64-bit
modes and 64-bit mode.

Operation
IF DEST = 0
    THEN
        #Z;
    ELSE
        IF Instruction = FIDIVR
              THEN
                  DEST <- ConvertToDoubleExtendedPrecisionFP(SRC) / DEST;
              ELSE (* Source operand is floating-point value *)
                  DEST <- SRC / DEST;
        FI;
FI;

IF Instruction = FDIVRP
     THEN
         PopRegisterStack;
FI;

FPU Flags Affected
C1                      Set to 0 if stack underflow occurred.
                        Set if result was rounded up; cleared otherwise.
C0, C2, C3              Undefined.





Floating-Point Exceptions
#IS                     Stack underflow occurred.
#IA                     Operand is an SNaN value or unsupported format.
                       +-inf / +-inf; +-0 / +-0
#D                      Source is a denormal value.
#Z                     SRC / +-0, where SRC is not equal to +-0.
#U                      Result is too small for destination format.
#O                      Result is too large for destination format.
#P                      Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#GP(0)                  If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                       If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)                  If a memory operand effective address is outside the SS segment limit.
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                     If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                     If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                     If a memory operand effective address is outside the SS segment limit.
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                     If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)                  If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)                  If a memory operand effective address is outside the SS segment limit.
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)        If a page fault occurs.
#AC(0)                  If alignment checking is enabled and an unaligned memory reference is made.
#UD                     If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)                 If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)                 If the memory address is in a non-canonical form.
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                     If there is a pending x87 FPU exception.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                     If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FDIVRP','-R:FDIVR');
INSERT INTO "instructions" VALUES('x86_64','FIDIVR','-R:FDIVR');
INSERT INTO "instructions" VALUES('x86_64','FFREE','FFREE-Free Floating-Point Register
Opcode             Instruction              64-Bit    Compat/       Description
                                            Mode      Leg Mode
DD C0+i            FFREE ST(i)              Valid     Valid         Sets tag for ST(i) to empty.



Description
Sets the tag in the FPU tag register associated with register ST(i) to empty (11B). The contents of ST(i) and the FPU
stack-top pointer (TOP) are not affected.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
TAG(i) <- 11B;

FPU Flags Affected
C0, C1, C2, C3 undefined.

Floating-Point Exceptions
None

Protected Mode Exceptions
#NM                  CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                  If there is a pending x87 FPU exception.
#UD                  If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FICOM','FICOM/FICOMP-Compare Integer
 Opcode                Instruction              64-Bit   Compat/    Description
                                                Mode     Leg Mode
 DE /2                 FICOM m16int             Valid    Valid      Compare ST(0) with m16int.
 DA /2                 FICOM m32int             Valid    Valid      Compare ST(0) with m32int.
 DE /3                 FICOMP m16int            Valid    Valid      Compare ST(0) with m16int and pop stack register.
 DA /3                 FICOMP m32int            Valid    Valid      Compare ST(0) with m32int and pop stack register.



Description
Compares the value in ST(0) with an integer source operand and sets the condition code flags C0, C2, and C3 in
the FPU status word according to the results (see table below). The integer value is converted to double extended-
precision floating-point format before the comparison is made.
                                          Table 3-26. FICOM/FICOMP Results
                      Condition                          C3                       C2                      C0
                      ST(0) > SRC                         0                       0                        0
                     ST(0) < SRC                          0                       0                        1
                     ST(0) = SRC                          1                       0                        0
                      Unordered                           1                       1                        1

These instructions perform an "unordered comparison." An unordered comparison also checks the class of the
numbers being compared (see "FXAM-Examine Floating-Point" in this chapter). If either operand is a NaN or is in
an undefined format, the condition flags are set to "unordered."
The sign of zero is ignored, so that -0.0 <- +0.0.
The FICOMP instructions pop the register stack following the comparison. To pop the register stack, the processor
marks the ST(0) register empty and increments the stack pointer (TOP) by 1.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
CASE (relation of operands) OF
   ST(0) > SRC:       C3, C2, C0 <- 000;
   ST(0) < SRC:       C3, C2, C0 <- 001;
   ST(0) = SRC:       C3, C2, C0 <- 100;
   Unordered:         C3, C2, C0 <- 111;
ESAC;

IF Instruction = FICOMP
     THEN
         PopRegisterStack;
FI;

FPU Flags Affected
C1                     Set to 0.
C0, C2, C3             See table on previous page.

Floating-Point Exceptions
#IS                    Stack underflow occurred.
#IA                    One or both operands are NaN values or have unsupported formats.
#D                     One or both operands are denormal values.



Protected Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                 If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FICOMP','-R:FICOM');
INSERT INTO "instructions" VALUES('x86_64','FILD','FILD-Load Integer
 Opcode               Instruction          64-Bit    Compat/      Description
                                           Mode      Leg Mode
 DF /0                FILD m16int          Valid     Valid        Push m16int onto the FPU register stack.
 DB /0                FILD m32int          Valid     Valid        Push m32int onto the FPU register stack.
 DF /5                FILD m64int          Valid     Valid        Push m64int onto the FPU register stack.



Description
Converts the signed-integer source operand into double extended-precision floating-point format and pushes the
value onto the FPU register stack. The source operand can be a word, doubleword, or quadword integer. It is loaded
without rounding errors. The sign of the source operand is preserved.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
TOP <- TOP - 1;
ST(0) <- ConvertToDoubleExtendedPrecisionFP(SRC);

FPU Flags Affected
C1                   Set to 1 if stack overflow occurred; set to 0 otherwise.
C0, C2, C3           Undefined.

Floating-Point Exceptions
#IS                  Stack overflow occurred.

Protected Mode Exceptions
#GP(0)               If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                     If the DS, ES, FS, or GS register contains a NULL segment selector.
#SS(0)               If a memory operand effective address is outside the SS segment limit.
#NM                  CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made while the
                     current privilege level is 3.
#UD                  If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                  If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                  If a memory operand effective address is outside the SS segment limit.
#NM                  CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                  If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)               If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)               If a memory operand effective address is outside the SS segment limit.
#NM                  CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)      If a page fault occurs.
#AC(0)               If alignment checking is enabled and an unaligned memory reference is made.
#UD                  If the LOCK prefix is used.




Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FINCSTP','FINCSTP-Increment Stack-Top Pointer
 Opcode                 Instruction        64-Bit     Compat/      Description
                                           Mode       Leg Mode
 D9 F7                  FINCSTP            Valid      Valid        Increment the TOP field in the FPU status register.



Description
Adds one to the TOP field of the FPU status word (increments the top-of-stack pointer). If the TOP field contains a
7, it is set to 0. The effect of this instruction is to rotate the stack by one position. The contents of the FPU data
registers and tag register are not affected. This operation is not equivalent to popping the stack, because the tag
for the previous top-of-stack register is not marked empty.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
IF TOP = 7
    THEN TOP <- 0;
    ELSE TOP <- TOP + 1;
FI;

FPU Flags Affected
The C1 flag is set to 0. The C0, C2, and C3 flags are undefined.

Floating-Point Exceptions
None

Protected Mode Exceptions
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                     If there is a pending x87 FPU exception.
#UD                     If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FINIT','FINIT/FNINIT-Initialize Floating-Point Unit
Opcode               Instruction              64-Bit    Compat/    Description
                                              Mode      Leg Mode
9B DB E3             FINIT                    Valid     Valid      Initialize FPU after checking for pending unmasked
                                                                   floating-point exceptions.
DB E3                FNINIT*                  Valid     Valid      Initialize FPU without checking for pending unmasked
                                                                   floating-point exceptions.
NOTES:
* See IA-32 Architecture Compatibility section below.



Description
Sets the FPU control, status, tag, instruction pointer, and data pointer registers to their default states. The FPU
control word is set to 037FH (round to nearest, all exceptions masked, 64-bit precision). The status word is cleared
(no exception flags set, TOP is set to 0). The data registers in the register stack are left unchanged, but they are all
tagged as empty (11B). Both the instruction and data pointers are cleared.
The FINIT instruction checks for and handles any pending unmasked floating-point exceptions before performing
the initialization; the FNINIT instruction does not.
The assembler issues two instructions for the FINIT instruction (an FWAIT instruction followed by an FNINIT
instruction), and the processor executes each of these instructions in separately. If an exception is generated for
either of these instructions, the save EIP points to the instruction that caused the exception.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

IA-32 Architecture Compatibility
When operating a Pentium or Intel486 processor in MS-DOS compatibility mode, it is possible (under unusual
circumstances) for an FNINIT instruction to be interrupted prior to being executed to handle a pending FPU excep-
tion. See the section titled "No-Wait FPU Instructions Can Get FPU Interrupt in Window" in Appendix D of the Intel
64 and IA-32 Architectures Software Developer''s Manual, Volume 1, for a description of these circumstances. An
FNINIT instruction cannot be interrupted in this way on later Intel processors, except for the Intel QuarkTM X1000
processor.
In the Intel387 math coprocessor, the FINIT/FNINIT instruction does not clear the instruction and data pointers.
This instruction affects only the x87 FPU. It does not affect the XMM and MXCSR registers.

Operation
FPUControlWord <- 037FH;
FPUStatusWord <- 0;
FPUTagWord <- FFFFH;
FPUDataPointer <- 0;
FPUInstructionPointer <- 0;
FPULastInstructionOpcode <- 0;

FPU Flags Affected
C0, C1, C2, C3 set to 0.

Floating-Point Exceptions
None





Protected Mode Exceptions
#NM                        CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                        If there is a pending x87 FPU exception.
#UD                        If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FNINIT','-R:FINIT');
INSERT INTO "instructions" VALUES('x86_64','FIST','FIST/FISTP-Store Integer
Opcode                     Instruction                64-Bit    Compat/       Description
                                                      Mode      Leg Mode
DF /2                      FIST m16int                Valid     Valid         Store ST(0) in m16int.
DB /2                      FIST m32int                Valid     Valid         Store ST(0) in m32int.
DF /3                      FISTP m16int               Valid     Valid         Store ST(0) in m16int and pop register stack.
DB /3                      FISTP m32int               Valid     Valid         Store ST(0) in m32int and pop register stack.
DF /7                      FISTP m64int               Valid     Valid         Store ST(0) in m64int and pop register stack.



Description
The FIST instruction converts the value in the ST(0) register to a signed integer and stores the result in the desti-
nation operand. Values can be stored in word or doubleword integer format. The destination operand specifies the
address where the first byte of the destination value is to be stored.
The FISTP instruction performs the same operation as the FIST instruction and then pops the register stack. To pop
the register stack, the processor marks the ST(0) register as empty and increments the stack pointer (TOP) by 1.
The FISTP instruction also stores values in quadword integer format.
The following table shows the results obtained when storing various classes of numbers in integer format.
                                                 Table 3-27. FIST/FISTP Results
                             ST(0)                                                          DEST
           - inf or Value Too Large for DEST Format                                            *
                             F <= -1                                                          -I
                          -1 < F < -0                                                        **
                              -0                                                             0
                              +0                                                             0
                         +0<F<+1                                                             **
                            F>=+1                                                             +I
           + inf or Value Too Large for DEST Format                                            *
                              NaN                                                            *
NOTES:
F Means finite floating-point value.
I Means integer.
* Indicates floating-point invalid-operation (#IA) exception.
** 0 or +-1, depending on the rounding mode.


If the source value is a non-integral value, it is rounded to an integer value, according to the rounding mode spec-
ified by the RC field of the FPU control word.
If the converted value is too large for the destination format, or if the source operand is an inf, SNaN, QNAN, or is in
an unsupported format, an invalid-arithmetic-operand condition is signaled. If the invalid-operation exception is
not masked, an invalid-arithmetic-operand exception (#IA) is generated and no value is stored in the destination
operand. If the invalid-operation exception is masked, the integer indefinite value is stored in memory.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.





Operation
DEST <- Integer(ST(0));

IF Instruction = FISTP
     THEN
         PopRegisterStack;
FI;

FPU Flags Affected
C1                         Set to 0 if stack underflow occurred.
                           Indicates rounding direction of if the inexact exception (#P) is generated: 0 <- not roundup; 1
                           <- roundup.
                           Set to 0 otherwise.
C0, C2, C3                 Undefined.

Floating-Point Exceptions
#IS                        Stack underflow occurred.
#IA                        Converted value is too large for the destination format.
                           Source operand is an SNaN, QNaN, +-inf, or unsupported format.
#P                         Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#GP(0)                     If the destination is located in a non-writable segment.
                           If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                           If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                           selector.
#SS(0)                     If a memory operand effective address is outside the SS segment limit.
#NM                        CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)            If a page fault occurs.
#AC(0)                     If alignment checking is enabled and an unaligned memory reference is made while the
                           current privilege level is 3.
#UD                        If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                        If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                        If a memory operand effective address is outside the SS segment limit.
#NM                        CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                        If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)                     If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)                     If a memory operand effective address is outside the SS segment limit.
#NM                        CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)            If a page fault occurs.
#AC(0)                     If alignment checking is enabled and an unaligned memory reference is made.
#UD                        If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.




64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FISTP','-R:FIST');
INSERT INTO "instructions" VALUES('x86_64','FISTTP','FISTTP-Store Integer with Truncation
 Opcode                 Instruction                  64-Bit Mode Compat/         Description
                                                                 Leg Mode
 DF /1                  FISTTP m16int                Valid      Valid            Store ST(0) in m16int with truncation.
 DB /1                  FISTTP m32int                Valid      Valid            Store ST(0) in m32int with truncation.
 DD /1                  FISTTP m64int                Valid      Valid            Store ST(0) in m64int with truncation.



Description
FISTTP converts the value in ST into a signed integer using truncation (chop) as rounding mode, transfers the
result to the destination, and pop ST. FISTTP accepts word, short integer, and long integer destinations.
The following table shows the results obtained when storing various classes of numbers in integer format.
                                                    Table 3-28. FISTTP Results
                               ST(0)                                                           DEST
- inf or Value Too Large for DEST Format                                                          *
F<= -1                                                                                           -I
-1<F<+1                                                                                         0
FS+1                                                                                            +I
+ inf or Value Too Large for DEST Format                                                          *
NaN                                                                                             *
NOTES:
F Means finite floating-point value.
I Means integer.
* Indicates floating-point invalid-operation (#IA) exception.

This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
DEST <- ST;
pop ST;

Flags Affected
C1 is cleared; C0, C2, C3 undefined.

Numeric Exceptions
Invalid, Stack Invalid (stack underflow), Precision.

Protected Mode Exceptions
#GP(0)                   If the destination is in a nonwritable segment.
                         For an illegal memory operand effective address in the CS, DS, ES, FS or GS segments.
#SS(0)                   For an illegal address in the SS segment.
#PF(fault-code)          For a page fault.
#AC(0)                   If alignment checking is enabled and an unaligned memory reference is made while the
                         current privilege level is 3.
#NM                      If CR0.EM[bit 2] = 1.
                         If CR0.TS[bit 3] = 1.
#UD                      If CPUID.01H:ECX.SSE3[bit 0] = 0.
                         If the LOCK prefix is used.



Real Address Mode Exceptions
GP(0)               If any part of the operand would lie outside of the effective address space from 0 to 0FFFFH.
#NM                 If CR0.EM[bit 2] = 1.
                    If CR0.TS[bit 3] = 1.
#UD                 If CPUID.01H:ECX.SSE3[bit 0] = 0.
                    If the LOCK prefix is used.

Virtual 8086 Mode Exceptions
GP(0)               If any part of the operand would lie outside of the effective address space from 0 to 0FFFFH.
#NM                 If CR0.EM[bit 2] = 1.
                    If CR0.TS[bit 3] = 1.
#UD                 If CPUID.01H:ECX.SSE3[bit 0] = 0.
                    If the LOCK prefix is used.
#PF(fault-code)     For a page fault.
#AC(0)              For unaligned memory reference if the current privilege is 3.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
                    If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FLD','FLD-Load Floating Point Value
 Opcode                   Instruction        64-Bit   Compat/     Description
                                             Mode     Leg Mode
 D9 /0                    FLD m32fp          Valid    Valid       Push m32fp onto the FPU register stack.
 DD /0                    FLD m64fp          Valid    Valid       Push m64fp onto the FPU register stack.
 DB /5                    FLD m80fp          Valid    Valid       Push m80fp onto the FPU register stack.
 D9 C0+i                  FLD ST(i)          Valid    Valid       Push ST(i) onto the FPU register stack.



Description
Pushes the source operand onto the FPU register stack. The source operand can be in single-precision, double-
precision, or double extended-precision floating-point format. If the source operand is in single-precision or
double-precision floating-point format, it is automatically converted to the double extended-precision floating-
point format before being pushed on the stack.
The FLD instruction can also push the value in a selected FPU register [ST(i)] onto the stack. Here, pushing register
ST(0) duplicates the stack top.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
IF SRC is ST(i)
    THEN
         temp <- ST(i);
FI;

TOP <- TOP    - 1;
IF SRC is memory-operand
    THEN
         ST(0) <- ConvertToDoubleExtendedPrecisionFP(SRC);
    ELSE (* SRC is ST(i) *)
         ST(0) <- temp;
FI;

FPU Flags Affected
C1                       Set to 1 if stack overflow occurred; otherwise, set to 0.
C0, C2, C3               Undefined.

Floating-Point Exceptions
#IS                      Stack underflow or overflow occurred.
#IA                      Source operand is an SNaN. Does not occur if the source operand is in double extended-preci-
                         sion floating-point format (FLD m80fp or FLD ST(i)).
#D                       Source operand is a denormal value. Does not occur if the source operand is in double
                         extended-precision floating-point format.





Protected Mode Exceptions
#GP(0)              If destination is located in a non-writable segment.
                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                    selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                 If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FLD1','FLD1/FLDL2T/FLDL2E/FLDPI/FLDLG2/FLDLN2/FLDZ-Load Constant
Opcode*               Instruction             64-Bit    Compat/    Description
                                              Mode      Leg Mode
D9 E8                 FLD1                    Valid     Valid      Push +1.0 onto the FPU register stack.
D9 E9                 FLDL2T                  Valid     Valid      Push log210 onto the FPU register stack.
D9 EA                 FLDL2E                  Valid     Valid      Push log2e onto the FPU register stack.
D9 EB                 FLDPI                   Valid     Valid      Push pi onto the FPU register stack.
D9 EC                 FLDLG2                  Valid     Valid      Push log102 onto the FPU register stack.
D9 ED                 FLDLN2                  Valid     Valid      Push loge2 onto the FPU register stack.
D9 EE                 FLDZ                    Valid     Valid      Push +0.0 onto the FPU register stack.
NOTES:
* See IA-32 Architecture Compatibility section below.



Description
Push one of seven commonly used constants (in double extended-precision floating-point format) onto the FPU
register stack. The constants that can be loaded with these instructions include +1.0, +0.0, log210, log2e, pi, log102,
and loge2. For each constant, an internal 66-bit constant is rounded (as specified by the RC field in the FPU control
word) to double extended-precision floating-point format. The inexact-result exception (#P) is not generated as a
result of the rounding, nor is the C1 flag set in the x87 FPU status word if the value is rounded up.
See the section titled "Approximation of Pi" in Chapter 8 of the Intel 64 and IA-32 Architectures Software Devel-
oper''s Manual, Volume 1, for a description of the pi constant.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

IA-32 Architecture Compatibility
When the RC field is set to round-to-nearest, the FPU produces the same constants that is produced by the Intel
8087 and Intel 287 math coprocessors.

Operation
TOP <- TOP - 1;
ST(0) <- CONSTANT;

FPU Flags Affected
C1                     Set to 1 if stack overflow occurred; otherwise, set to 0.
C0, C2, C3             Undefined.

Floating-Point Exceptions
#IS                    Stack overflow occurred.

Protected Mode Exceptions
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                    If there is a pending x87 FPU exception.
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.



Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FLDL2T','-R:FLD1');
INSERT INTO "instructions" VALUES('x86_64','FLDL2E','-R:FLD1');
INSERT INTO "instructions" VALUES('x86_64','FLDPI','-R:FLD1');
INSERT INTO "instructions" VALUES('x86_64','FLDLG2','-R:FLD1');
INSERT INTO "instructions" VALUES('x86_64','FLDLN2','-R:FLD1');
INSERT INTO "instructions" VALUES('x86_64','FLDZ','-R:FLD1');
INSERT INTO "instructions" VALUES('x86_64','FLDCW','FLDCW-Load x87 FPU Control Word
 Opcode                  Instruction                 64-Bit   Compat/    Description
                                                     Mode     Leg Mode
 D9 /5                   FLDCW m2byte                Valid    Valid      Load FPU control word from m2byte.



Description
Loads the 16-bit source operand into the FPU control word. The source operand is a memory location. This instruc-
tion is typically used to establish or change the FPU''s mode of operation.
If one or more exception flags are set in the FPU status word prior to loading a new FPU control word and the new
control word unmasks one or more of those exceptions, a floating-point exception will be generated upon execution
of the next floating-point instruction (except for the no-wait floating-point instructions, see the section titled "Soft-
ware Exception Handling" in Chapter 8 of the Intel 64 and IA-32 Architectures Software Developer''s Manual,
Volume 1). To avoid raising exceptions when changing FPU operating modes, clear any pending exceptions (using
the FCLEX or FNCLEX instruction) before loading the new control word.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
FPUControlWord <- SRC;

FPU Flags Affected
C0, C1, C2, C3 undefined.

Floating-Point Exceptions
None; however, this operation might unmask a pending exception in the FPU status word. That exception is then
generated upon execution of the next "waiting" floating-point instruction.

Protected Mode Exceptions
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                       If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                       selector.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                    If a memory operand effective address is outside the SS segment limit.
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                    If the LOCK prefix is used.





Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FLDENV','FLDENV-Load x87 FPU Environment
Opcode            Instruction                      64-Bit   Compat/     Description
                                                   Mode     Leg Mode
D9 /4             FLDENV m14/28byte                Valid    Valid       Load FPU environment from m14byte or m28byte.



Description
Loads the complete x87 FPU operating environment from memory into the FPU registers. The source operand spec-
ifies the first byte of the operating-environment data in memory. This data is typically written to the specified
memory location by a FSTENV or FNSTENV instruction.
The FPU operating environment consists of the FPU control word, status word, tag word, instruction pointer, data
pointer, and last opcode. Figures 8-9 through 8-12 in the Intel 64 and IA-32 Architectures Software Developer''s
Manual, Volume 1, show the layout in memory of the loaded environment, depending on the operating mode of the
processor (protected or real) and the current operand-size attribute (16-bit or 32-bit). In virtual-8086 mode, the
real mode layouts are used.
The FLDENV instruction should be executed in the same operating mode as the corresponding FSTENV/FNSTENV
instruction.
If one or more unmasked exception flags are set in the new FPU status word, a floating-point exception will be
generated upon execution of the next floating-point instruction (except for the no-wait floating-point instructions,
see the section titled "Software Exception Handling" in Chapter 8 of the Intel 64 and IA-32 Architectures Soft-
ware Developer''s Manual, Volume 1). To avoid generating exceptions when loading a new environment, clear all
the exception flags in the FPU status word that is being loaded.
If a page or limit fault occurs during the execution of this instruction, the state of the x87 FPU registers as seen by
the fault handler may be different than the state being loaded from memory. In such situations, the fault handler
should ignore the status of the x87 FPU registers, handle the fault, and return. The FLDENV instruction will then
complete the loading of the x87 FPU registers with no resulting context inconsistency.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
FPUControlWord <- SRC[FPUControlWord];
FPUStatusWord <- SRC[FPUStatusWord];
FPUTagWord <- SRC[FPUTagWord];
FPUDataPointer <- SRC[FPUDataPointer];
FPUInstructionPointer <- SRC[FPUInstructionPointer];
FPULastInstructionOpcode <- SRC[FPULastInstructionOpcode];

FPU Flags Affected
The C0, C1, C2, C3 flags are loaded.

Floating-Point Exceptions
None; however, if an unmasked exception is loaded in the status word, it is generated upon execution of the next
"waiting" floating-point instruction.





Protected Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                    selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                 If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FMUL','FMUL/FMULP/FIMUL-Multiply
 Opcode            Instruction                  64-Bit   Compat/     Description
                                                Mode     Leg Mode
 D8 /1             FMUL m32fp                   Valid    Valid       Multiply ST(0) by m32fp and store result in ST(0).
 DC /1             FMUL m64fp                   Valid    Valid       Multiply ST(0) by m64fp and store result in ST(0).
 D8 C8+i           FMUL ST(0), ST(i)            Valid    Valid       Multiply ST(0) by ST(i) and store result in ST(0).
 DC C8+i           FMUL ST(i), ST(0)            Valid    Valid       Multiply ST(i) by ST(0) and store result in ST(i).
 DE C8+i           FMULP ST(i), ST(0)           Valid    Valid       Multiply ST(i) by ST(0), store result in ST(i), and pop the
                                                                     register stack.
 DE C9             FMULP                        Valid    Valid       Multiply ST(1) by ST(0), store result in ST(1), and pop
                                                                     the register stack.
 DA /1             FIMUL m32int                 Valid    Valid       Multiply ST(0) by m32int and store result in ST(0).
 DE /1             FIMUL m16int                 Valid    Valid       Multiply ST(0) by m16int and store result in ST(0).



Description
Multiplies the destination and source operands and stores the product in the destination location. The destination
operand is always an FPU data register; the source operand can be an FPU data register or a memory location.
Source operands in memory can be in single-precision or double-precision floating-point format or in word or
doubleword integer format.
The no-operand version of the instruction multiplies the contents of the ST(1) register by the contents of the ST(0)
register and stores the product in the ST(1) register. The one-operand version multiplies the contents of the ST(0)
register by the contents of a memory location (either a floating point or an integer value) and stores the product in
the ST(0) register. The two-operand version, multiplies the contents of the ST(0) register by the contents of the
ST(i) register, or vice versa, with the result being stored in the register specified with the first operand (the desti-
nation operand).
The FMULP instructions perform the additional operation of popping the FPU register stack after storing the
product. To pop the register stack, the processor marks the ST(0) register as empty and increments the stack
pointer (TOP) by 1. The no-operand version of the floating-point multiply instructions always results in the register
stack being popped. In some assemblers, the mnemonic for this instruction is FMUL rather than FMULP.
The FIMUL instructions convert an integer source operand to double extended-
precision floating-point format before performing the multiplication.
The sign of the result is always the exclusive-OR of the source signs, even if one or more of the values being multi-
plied is 0 or inf. When the source operand is an integer 0, it is treated as a +0.
The following table shows the results obtained when multiplying various classes of numbers, assuming that neither
overflow nor underflow occurs.





                                           Table 3-29. FMUL/FMULP/FIMUL Results
                                                                  DEST
                                 -inf              -F         -0           +0          +F      +inf           NaN
                  -inf             +inf              +inf          *             *         -inf      -inf           NaN
                   -F            +inf              +F         +0           -0          -F      -inf           NaN
                   -I            +inf              +F         +0           -0          -F      -inf           NaN
      SRC          -0              *             +0         +0           -0          -0       *           NaN
                   +0              *             -0         -0           +0          +0       *           NaN
                   +I            -inf              -F         -0           +0          +F      +inf           NaN
                   +F            -inf              -F         -0           +0          +F      +inf           NaN
                  +inf             -inf              -inf          *             *         +inf      +inf           NaN
                  NaN            NaN             NaN       NaN           NaN         NaN    NaN           NaN
 NOTES:
 F Means finite floating-point value.
 I Means Integer.
 * Indicates invalid-arithmetic-operand (#IA) exception.

This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
IF Instruction = FIMUL
     THEN
         DEST <- DEST * ConvertToDoubleExtendedPrecisionFP(SRC);
     ELSE (* Source operand is floating-point value *)
         DEST <- DEST * SRC;
FI;

IF Instruction = FMULP
     THEN
         PopRegisterStack;
FI;

FPU Flags Affected
C1                      Set to 0 if stack underflow occurred.
                        Set if result was rounded up; cleared otherwise.
C0, C2, C3              Undefined.

Floating-Point Exceptions
#IS                     Stack underflow occurred.
#IA                     Operand is an SNaN value or unsupported format.
                        One operand is +-0 and the other is +-inf.
#D                      Source operand is a denormal value.
#U                      Result is too small for destination format.
#O                      Result is too large for destination format.
#P                      Value cannot be represented exactly in destination format.





Protected Mode Exceptions
#GP(0)                If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                      If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                      selector.
#SS(0)                If a memory operand effective address is outside the SS segment limit.
#NM                   CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)       If a page fault occurs.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made while the
                      current privilege level is 3.
#UD                   If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                   If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                   If a memory operand effective address is outside the SS segment limit.
#NM                   CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                   If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)                If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)                If a memory operand effective address is outside the SS segment limit.
#NM                   CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)       If a page fault occurs.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made.
#UD                   If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)                If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)                If the memory address is in a non-canonical form.
#NM                   CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                   If there is a pending x87 FPU exception.
#PF(fault-code)       If a page fault occurs.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made while the
                      current privilege level is 3.
#UD                   If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FMULP','-R:FMUL');
INSERT INTO "instructions" VALUES('x86_64','FIMUL','-R:FMUL');
INSERT INTO "instructions" VALUES('x86_64','FNOP','FNOP-No Operation
 Opcode                  Instruction        64-Bit   Compat/     Description
                                            Mode     Leg Mode
 D9 D0                   FNOP               Valid    Valid       No operation is performed.



Description
Performs no FPU operation. This instruction takes up space in the instruction stream but does not affect the FPU or
machine context, except the EIP register and the FPU Instruction Pointer.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

FPU Flags Affected
C0, C1, C2, C3 undefined.

Floating-Point Exceptions
None

Protected Mode Exceptions
#NM                  CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                  If there is a pending x87 FPU exception.
#UD                  If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FPATAN','FPATAN-Partial Arctangent
Opcode*              Instruction            64-Bit        Compat/         Description
                                            Mode          Leg Mode
D9 F3                FPATAN                 Valid         Valid           Replace ST(1) with arctan(ST(1)/ST(0)) and pop the register stack.
NOTES:
* See IA-32 Architecture Compatibility section below.



Description
Computes the arctangent of the source operand in register ST(1) divided by the source operand in register ST(0),
stores the result in ST(1), and pops the FPU register stack. The result in register ST(0) has the same sign as the
source operand ST(1) and a magnitude less than +pi.
The FPATAN instruction returns the angle between the X axis and the line from the origin to the point (X,Y), where
Y (the ordinate) is ST(1) and X (the abscissa) is ST(0). The angle depends on the sign of X and Y independently,
not just on the sign of the ratio Y/X. This is because a point (-X,Y) is in the second quadrant, resulting in an angle
between pi/2 and pi, while a point (X,-Y) is in the fourth quadrant, resulting in an angle between 0 and -pi/2. A point
(-X,-Y) is in the third quadrant, giving an angle between -pi/2 and -pi.
The following table shows the results obtained when computing the arctangent of various classes of numbers,
assuming that underflow does not occur.
                                                    Table 3-30. FPATAN Results
                                                                              ST(0)
                             -inf            -F                     -0              +0            +F              +inf                 NaN
              -inf             - 3pi/4*       - pi/2                  - pi/2           - pi/2         - pi/2           - pi/4*             NaN
   ST(1)      -F             -p            -pi to -pi/2             -pi/2            -pi/2          -pi/2 to -0      -0                 NaN
              -0             -p            -p                     -p*             - 0*          -0              -0                 NaN
              +0             +p            +p                     + pi*            + 0*          +0              +0                 NaN
              +F             +p            +pi to +pi/2             + pi/2           +pi/2          +pi/2 to +0      +0                 NaN
              +inf             +3pi/4*        +pi/2                   +pi/2            +pi/2          + pi/2           + pi/4*             NaN
                   NaN             NaN              NaN                 NaN            NaN           NaN             NaN           NaN
 NOTES:
 F Means finite floating-point value.
 * Table 8-10 in the Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 1, specifies that the ratios 0/0 and inf/inf
   generate the floating-point invalid arithmetic-operation exception and, if this exception is masked, the floating-point QNaN indefi-
   nite value is returned. With the FPATAN instruction, the 0/0 or inf/inf value is actually not calculated using division. Instead, the arc-
   tangent of the two variables is derived from a standard mathematical formulation that is generalized to allow complex numbers as
   arguments. In this complex variable formulation, arctangent(0,0) etc. has well defined values. These values are needed to develop
   a library to compute transcendental functions with complex arguments, based on the FPU functions that only allow floating-point
   values as arguments.

There is no restriction on the range of source operands that FPATAN can accept.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

IA-32 Architecture Compatibility
The source operands for this instruction are restricted for the 80287 math coprocessor to the following range:

0 <= |ST(1)| < |ST(0)| < +inf





Operation
ST(1) <- arctan(ST(1) / ST(0));
PopRegisterStack;

FPU Flags Affected
C1                      Set to 0 if stack underflow occurred.
                        Set if result was rounded up; cleared otherwise.
C0, C2, C3              Undefined.

Floating-Point Exceptions
#IS                     Stack underflow occurred.
#IA                     Source operand is an SNaN value or unsupported format.
#D                      Source operand is a denormal value.
#U                      Result is too small for destination format.
#P                      Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                     If there is a pending x87 FPU exception.
#UD                     If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FPREM','FPREM-Partial Remainder
Opcode                   Instruction          64-Bit       Compat/         Description
                                              Mode         Leg Mode
D9 F8                    FPREM                Valid        Valid           Replace ST(0) with the remainder obtained from dividing
                                                                           ST(0) by ST(1).



Description
Computes the remainder obtained from dividing the value in the ST(0) register (the dividend) by the value in the
ST(1) register (the divisor or modulus), and stores the result in ST(0). The remainder represents the following
value:
Remainder <- ST(0) - (Q * ST(1))
Here, Q is an integer value that is obtained by truncating the floating-point number quotient of [ST(0) / ST(1)]
toward zero. The sign of the remainder is the same as the sign of the dividend. The magnitude of the remainder is
less than that of the modulus, unless a partial remainder was computed (as described below).
This instruction produces an exact result; the inexact-result exception does not occur and the rounding control has
no effect. The following table shows the results obtained when computing the remainder of various classes of
numbers, assuming that underflow does not occur.
                                                      Table 3-31. FPREM Results
                                                                          ST(1)
                                       -inf          -F              -0             +0            +F             +inf            NaN
                   -inf                   *             *             *              *             *              *            NaN
 ST(0)              -F             ST(0)        -F or -0           **             **          -F or -0       ST(0)           NaN
                    -0                 -0          -0               *              *            -0             -0            NaN
                   +0                  +0         +0                *              *            +0             +0            NaN
                   +F              ST(0)       +F or +0            **             **         +F or +0        ST(0)           NaN
                   +inf                   *             *             *              *             *              *            NaN
                  NaN                  NaN        NaN              NaN            NaN          NaN            NaN            NaN
 NOTES:
 F Means finite floating-point value.
 * Indicates floating-point invalid-arithmetic-operand (#IA) exception.
 ** Indicates floating-point zero-divide (#Z) exception.

When the result is 0, its sign is the same as that of the dividend. When the modulus is inf, the result is equal to the
value in ST(0).
The FPREM instruction does not compute the remainder specified in IEEE Std 754. The IEEE specified remainder
can be computed with the FPREM1 instruction. The FPREM instruction is provided for compatibility with the Intel
8087 and Intel287 math coprocessors.
The FPREM instruction gets its name "partial remainder" because of the way it computes the remainder. This
instruction arrives at a remainder through iterative subtraction. It can, however, reduce the exponent of ST(0) by
no more than 63 in one execution of the instruction. If the instruction succeeds in producing a remainder that is
less than the modulus, the operation is complete and the C2 flag in the FPU status word is cleared. Otherwise, C2
is set, and the result in ST(0) is called the partial remainder. The exponent of the partial remainder will be less
than the exponent of the original dividend by at least 32. Software can re-execute the instruction (using the partial
remainder in ST(0) as the dividend) until C2 is cleared. (Note that while executing such a remainder-computation
loop, a higher-priority interrupting routine that needs the FPU can force a context switch in-between the instruc-
tions in the loop.)
An important use of the FPREM instruction is to reduce the arguments of periodic functions. When reduction is
complete, the instruction stores the three least-significant bits of the quotient in the C3, C1, and C0 flags of the FPU



status word. This information is important in argument reduction for the tangent function (using a modulus of pi/4),
because it locates the original angle in the correct one of eight sectors of the unit circle.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
D <- exponent(ST(0)) - exponent(ST(1));

IF D < 64
    THEN
        Q <- Integer(TruncateTowardZero(ST(0) / ST(1)));
        ST(0) <- ST(0) - (ST(1) * Q);
        C2 <- 0;
        C0, C3, C1 <- LeastSignificantBits(Q); (* Q2, Q1, Q0 *)
    ELSE
        C2 <- 1;
        N <- An implementation-dependent number between 32 and 63;
        QQ <- Integer(TruncateTowardZero((ST(0) / ST(1)) / 2(D - N)));
        ST(0) <- ST(0) - (ST(1) * QQ * 2(D - N));
FI;

FPU Flags Affected
C0                    Set to bit 2 (Q2) of the quotient.
C1                    Set to 0 if stack underflow occurred; otherwise, set to least significant bit of quotient (Q0).
C2                    Set to 0 if reduction complete; set to 1 if incomplete.
C3                    Set to bit 1 (Q1) of the quotient.

Floating-Point Exceptions
#IS                   Stack underflow occurred.
#IA                   Source operand is an SNaN value, modulus is 0, dividend is inf, or unsupported format.
#D                    Source operand is a denormal value.
#U                    Result is too small for destination format.

Protected Mode Exceptions
#NM                   CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                   If there is a pending x87 FPU exception.
#UD                   If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FPREM1','FPREM1-Partial Remainder
 Opcode              Instruction               64-Bit       Compat/       Description
                                               Mode         Leg Mode
 D9 F5               FPREM1                    Valid        Valid         Replace ST(0) with the IEEE remainder obtained from dividing
                                                                          ST(0) by ST(1).



Description
Computes the IEEE remainder obtained from dividing the value in the ST(0) register (the dividend) by the value in
the ST(1) register (the divisor or modulus), and stores the result in ST(0). The remainder represents the following
value:
Remainder <- ST(0) - (Q * ST(1))
Here, Q is an integer value that is obtained by rounding the floating-point number quotient of [ST(0) / ST(1)]
toward the nearest integer value. The magnitude of the remainder is less than or equal to half the magnitude of the
modulus, unless a partial remainder was computed (as described below).
This instruction produces an exact result; the precision (inexact) exception does not occur and the rounding control
has no effect. The following table shows the results obtained when computing the remainder of various classes of
numbers, assuming that underflow does not occur.
                                                   Table 3-32. FPREM1 Results
                                                                          ST(1)
                                   -inf             -F                -0            +0             +F            +inf            NaN
                   -inf               *              *                 *             *              *              *           NaN
 ST(0)             -F            ST(0)         +-F or -0             **            **         +- F or - 0       ST(0)          NaN
                   -0              -0             -0                 *             *             -0             -0           NaN
                   +0              +0             +0                 *             *             +0             +0           NaN
                   +F            ST(0)         +- F or + 0           **            **         +- F or + 0       ST(0)          NaN
                   +inf               *              *                 *             *              *              *           NaN
                  NaN              NaN           NaN                NaN           NaN           NaN            NaN           NaN
 NOTES:
 F Means finite floating-point value.
 * Indicates floating-point invalid-arithmetic-operand (#IA) exception.
 ** Indicates floating-point zero-divide (#Z) exception.

When the result is 0, its sign is the same as that of the dividend. When the modulus is inf, the result is equal to the
value in ST(0).
The FPREM1 instruction computes the remainder specified in IEEE Standard 754. This instruction operates differ-
ently from the FPREM instruction in the way that it rounds the quotient of ST(0) divided by ST(1) to an integer (see
the "Operation" section below).
Like the FPREM instruction, FPREM1 computes the remainder through iterative subtraction, but can reduce the
exponent of ST(0) by no more than 63 in one execution of the instruction. If the instruction succeeds in producing
a remainder that is less than one half the modulus, the operation is complete and the C2 flag in the FPU status word
is cleared. Otherwise, C2 is set, and the result in ST(0) is called the partial remainder. The exponent of the partial
remainder will be less than the exponent of the original dividend by at least 32. Software can re-execute the
instruction (using the partial remainder in ST(0) as the dividend) until C2 is cleared. (Note that while executing
such a remainder-computation loop, a higher-priority interrupting routine that needs the FPU can force a context
switch in-between the instructions in the loop.)
An important use of the FPREM1 instruction is to reduce the arguments of periodic functions. When reduction is
complete, the instruction stores the three least-significant bits of the quotient in the C3, C1, and C0 flags of the FPU




status word. This information is important in argument reduction for the tangent function (using a modulus of pi/4),
because it locates the original angle in the correct one of eight sectors of the unit circle.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
D <- exponent(ST(0)) - exponent(ST(1));

IF D < 64
    THEN
        Q <- Integer(RoundTowardNearestInteger(ST(0) / ST(1)));
        ST(0) <- ST(0) - (ST(1) * Q);
        C2 <- 0;
        C0, C3, C1 <- LeastSignificantBits(Q); (* Q2, Q1, Q0 *)
    ELSE
        C2 <- 1;
        N <- An implementation-dependent number between 32 and 63;
        QQ <- Integer(TruncateTowardZero((ST(0) / ST(1)) / 2(D - N)));
        ST(0) <- ST(0) - (ST(1) * QQ * 2(D - N));
FI;

FPU Flags Affected
C0                    Set to bit 2 (Q2) of the quotient.
C1                    Set to 0 if stack underflow occurred; otherwise, set to least significant bit of quotient (Q0).
C2                    Set to 0 if reduction complete; set to 1 if incomplete.
C3                    Set to bit 1 (Q1) of the quotient.

Floating-Point Exceptions
#IS                   Stack underflow occurred.
#IA                   Source operand is an SNaN value, modulus (divisor) is 0, dividend is inf, or unsupported
                      format.
#D                    Source operand is a denormal value.
#U                    Result is too small for destination format.

Protected Mode Exceptions
#NM                   CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                   If there is a pending x87 FPU exception.
#UD                   If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FPTAN','FPTAN-Partial Tangent
Opcode                  Instruction              64-Bit        Compat/    Description
                                                 Mode          Leg Mode
D9 F2                   FPTAN                    Valid         Valid      Replace ST(0) with its approximate tangent and push 1
                                                                          onto the FPU stack.



Description
Computes the approximate tangent of the source operand in register ST(0), stores the result in ST(0), and pushes
a 1.0 onto the FPU register stack. The source operand must be given in radians and must be less than +-263. The
following table shows the unmasked results obtained when computing the partial tangent of various classes of
numbers, assuming that underflow does not occur.
                                                   Table 3-33. FPTAN Results
                           ST(0) SRC                                                      ST(0) DEST
                              -inf                                                               *
                              -F                                                           - F to + F
                              -0                                                              -0
                              +0                                                              +0
                              +F                                                           - F to + F
                              +inf                                                               *
                              NaN                                                            NaN
NOTES:
F Means finite floating-point value.
* Indicates floating-point invalid-arithmetic-operand (#IA) exception.

If the source operand is outside the acceptable range, the C2 flag in the FPU status word is set, and the value in
register ST(0) remains unchanged. The instruction does not raise an exception when the source operand is out of
range. It is up to the program to check the C2 flag for out-of-range conditions. Source values outside the range -
263 to +263 can be reduced to the range of the instruction by subtracting an appropriate integer multiple of 2pi.
However, even within the range -263 to +263, inaccurate results can occur because the finite approximation of pi
used internally for argument reduction is not sufficient in all cases. Therefore, for accurate results it is safe to apply
FPTAN only to arguments reduced accurately in software, to a value smaller in absolute value than 3pi/8. See the
sections titled "Approximation of Pi" and "Transcendental Instruction Accuracy" in Chapter 8 of the Intel 64 and
IA-32 Architectures Software Developer''s Manual, Volume 1, for a discussion of the proper value to use for pi in
performing such reductions.
The value 1.0 is pushed onto the register stack after the tangent has been computed to maintain compatibility with
the Intel 8087 and Intel287 math coprocessors. This operation also simplifies the calculation of other trigonometric
functions. For instance, the cotangent (which is the reciprocal of the tangent) can be computed by executing a
FDIVR instruction after the FPTAN instruction.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.





Operation
IF ST(0) < 263
    THEN
         C2 <- 0;
         ST(0) <- fptan(ST(0)); // approximation of tan
         TOP <- TOP - 1;
         ST(0) <- 1.0;
    ELSE (* Source operand is out-of-range *)
         C2 <- 1;
FI;

FPU Flags Affected
C1                      Set to 0 if stack underflow occurred; set to 1 if stack overflow occurred.
                        Set if result was rounded up; cleared otherwise.
C2                      Set to 1 if outside range (-263 < source operand < +263); otherwise, set to 0.
C0, C3                  Undefined.

Floating-Point Exceptions
#IS                     Stack underflow or overflow occurred.
#IA                     Source operand is an SNaN value, inf, or unsupported format.
#D                      Source operand is a denormal value.
#U                      Result is too small for destination format.
#P                      Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                     If there is a pending x87 FPU exception.
#UD                     If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FRNDINT','FRNDINT-Round to Integer
Opcode                     Instruction      64-Bit     Compat/      Description
                                            Mode       Leg Mode
D9 FC                      FRNDINT          Valid      Valid        Round ST(0) to an integer.



Description
Rounds the source value in the ST(0) register to the nearest integral value, depending on the current rounding
mode (setting of the RC field of the FPU control word), and stores the result in ST(0).
If the source value is inf, the value is not changed. If the source value is not an integral value, the floating-point
inexact-result exception (#P) is generated.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
ST(0) <- RoundToIntegralValue(ST(0));

FPU Flags Affected
C1                     Set to 0 if stack underflow occurred.
                       Set if result was rounded up; cleared otherwise.
C0, C2, C3             Undefined.

Floating-Point Exceptions
#IS                    Stack underflow occurred.
#IA                    Source operand is an SNaN value or unsupported format.
#D                     Source operand is a denormal value.
#P                     Source operand is not an integral value.

Protected Mode Exceptions
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                    If there is a pending x87 FPU exception.
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FRSTOR','FRSTOR-Restore x87 FPU State
Opcode                Instruction                    64-Bit   Compat/      Description
                                                     Mode     Leg Mode
DD /4                 FRSTOR m94/108byte             Valid    Valid        Load FPU state from m94byte or m108byte.



Description
Loads the FPU state (operating environment and register stack) from the memory area specified with the source
operand. This state data is typically written to the specified memory location by a previous FSAVE/FNSAVE instruc-
tion.
The FPU operating environment consists of the FPU control word, status word, tag word, instruction pointer, data
pointer, and last opcode. Figures 8-9 through 8-12 in the Intel 64 and IA-32 Architectures Software Developer''s
Manual, Volume 1, show the layout in memory of the stored environment, depending on the operating mode of the
processor (protected or real) and the current operand-size attribute (16-bit or 32-bit). In virtual-8086 mode, the
real mode layouts are used. The contents of the FPU register stack are stored in the 80 bytes immediately following
the operating environment image.
The FRSTOR instruction should be executed in the same operating mode as the corresponding FSAVE/FNSAVE
instruction.
If one or more unmasked exception bits are set in the new FPU status word, a floating-point exception will be
generated. To avoid raising exceptions when loading a new operating environment, clear all the exception flags in
the FPU status word that is being loaded.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
FPUControlWord <- SRC[FPUControlWord];
FPUStatusWord <- SRC[FPUStatusWord];
FPUTagWord <- SRC[FPUTagWord];
FPUDataPointer <- SRC[FPUDataPointer];
FPUInstructionPointer <- SRC[FPUInstructionPointer];
FPULastInstructionOpcode <- SRC[FPULastInstructionOpcode];

ST(0) <- SRC[ST(0)];
ST(1) <- SRC[ST(1)];
ST(2) <- SRC[ST(2)];
ST(3) <- SRC[ST(3)];
ST(4) <- SRC[ST(4)];
ST(5) <- SRC[ST(5)];
ST(6) <- SRC[ST(6)];
ST(7) <- SRC[ST(7)];

FPU Flags Affected
The C0, C1, C2, C3 flags are loaded.

Floating-Point Exceptions
None; however, this operation might unmask an existing exception that has been detected but not generated,
because it was masked. Here, the exception is generated at the completion of the instruction.





Protected Mode Exceptions
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                      If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                      selector.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)        If a page fault occurs.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made while the
                      current privilege level is 3.
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                    If a memory operand effective address is outside the SS segment limit.
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                    If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made.
#UD                    If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)                 If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)                 If the memory address is in a non-canonical form.
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)        If a page fault occurs.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made while the
                      current privilege level is 3.
#UD                    If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FSAVE','FSAVE/FNSAVE-Store x87 FPU State
 Opcode           Instruction                            64-Bit   Compat/    Description
                                                         Mode     Leg Mode
 9B DD /6         FSAVE m94/108byte                      Valid    Valid      Store FPU state to m94byte or m108byte after
                                                                             checking for pending unmasked floating-point
                                                                             exceptions. Then re-initialize the FPU.
 DD /6            FNSAVE* m94/108byte                    Valid    Valid      Store FPU environment to m94byte or m108byte
                                                                             without checking for pending unmasked floating-
                                                                             point exceptions. Then re-initialize the FPU.
 NOTES:
 * See IA-32 Architecture Compatibility section below.



Description
Stores the current FPU state (operating environment and register stack) at the specified destination in memory,
and then re-initializes the FPU. The FSAVE instruction checks for and handles pending unmasked floating-point
exceptions before storing the FPU state; the FNSAVE instruction does not.
The FPU operating environment consists of the FPU control word, status word, tag word, instruction pointer, data
pointer, and last opcode. Figures 8-9 through 8-12 in the Intel 64 and IA-32 Architectures Software Developer''s
Manual, Volume 1, show the layout in memory of the stored environment, depending on the operating mode of the
processor (protected or real) and the current operand-size attribute (16-bit or 32-bit). In virtual-8086 mode, the
real mode layouts are used. The contents of the FPU register stack are stored in the 80 bytes immediately follow
the operating environment image.
The saved image reflects the state of the FPU after all floating-point instructions preceding the FSAVE/FNSAVE
instruction in the instruction stream have been executed.
After the FPU state has been saved, the FPU is reset to the same default values it is set to with the FINIT/FNINIT
instructions (see "FINIT/FNINIT-Initialize Floating-Point Unit" in this chapter).
The FSAVE/FNSAVE instructions are typically used when the operating system needs to perform a context switch,
an exception handler needs to use the FPU, or an application program needs to pass a "clean" FPU to a procedure.
The assembler issues two instructions for the FSAVE instruction (an FWAIT instruction followed by an FNSAVE
instruction), and the processor executes each of these instructions separately. If an exception is generated for
either of these instructions, the save EIP points to the instruction that caused the exception.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

IA-32 Architecture Compatibility
For Intel math coprocessors and FPUs prior to the Intel Pentium processor, an FWAIT instruction should be
executed before attempting to read from the memory image stored with a prior FSAVE/FNSAVE instruction. This
FWAIT instruction helps ensure that the storage operation has been completed.
When operating a Pentium or Intel486 processor in MS-DOS compatibility mode, it is possible (under unusual
circumstances) for an FNSAVE instruction to be interrupted prior to being executed to handle a pending FPU excep-
tion. See the section titled "No-Wait FPU Instructions Can Get FPU Interrupt in Window" in Appendix D of the Intel
64 and IA-32 Architectures Software Developer''s Manual, Volume 1, for a description of these circumstances. An
FNSAVE instruction cannot be interrupted in this way on later Intel processors, except for the Intel QuarkTM X1000
processor.





Operation
(* Save FPU State and Registers *)

DEST[FPUControlWord] <- FPUControlWord;
DEST[FPUStatusWord] <- FPUStatusWord;
DEST[FPUTagWord] <- FPUTagWord;
DEST[FPUDataPointer] <- FPUDataPointer;
DEST[FPUInstructionPointer] <- FPUInstructionPointer;
DEST[FPULastInstructionOpcode] <- FPULastInstructionOpcode;

DEST[ST(0)] <- ST(0);
DEST[ST(1)] <- ST(1);
DEST[ST(2)] <- ST(2);
DEST[ST(3)] <- ST(3);
DEST[ST(4)]<- ST(4);
DEST[ST(5)] <- ST(5);
DEST[ST(6)] <- ST(6);
DEST[ST(7)] <- ST(7);

(* Initialize FPU *)

FPUControlWord <- 037FH;
FPUStatusWord <- 0;
FPUTagWord <- FFFFH;
FPUDataPointer <- 0;
FPUInstructionPointer <- 0;
FPULastInstructionOpcode <- 0;

FPU Flags Affected
The C0, C1, C2, and C3 flags are saved and then cleared.

Floating-Point Exceptions
None.

Protected Mode Exceptions
#GP(0)                 If destination is located in a non-writable segment.
                       If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                       If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                       selector.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                    If a memory operand effective address is outside the SS segment limit.
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                    If the LOCK prefix is used.





Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.




');
INSERT INTO "instructions" VALUES('x86_64','FNSAVE','-R:FSAVE');
INSERT INTO "instructions" VALUES('x86_64','FSCALE','FSCALE-Scale
Opcode                     Instruction           64-Bit      Compat/       Description
                                                 Mode        Leg Mode
D9 FD                      FSCALE                Valid       Valid         Scale ST(0) by ST(1).



Description
Truncates the value in the source operand (toward 0) to an integral value and adds that value to the exponent of
the destination operand. The destination and source operands are floating-point values located in registers ST(0)
and ST(1), respectively. This instruction provides rapid multiplication or division by integral powers of 2. The
following table shows the results obtained when scaling various classes of numbers, assuming that neither over-
flow nor underflow occurs.


                                                   Table 3-34. FSCALE Results
                   ST(1)
                                         -inf       -F           -0          +0             +F           +inf           NaN
                       -inf            NaN          -inf           -inf          -inf            -inf            -inf           NaN
 ST(0)                 -F                -0       -F           -F          -F             -F           -inf           NaN
                       -0                -0       -0           -0          -0             -0           NaN          NaN
                       +0                +0       +0           +0          +0             +0           NaN          NaN
                       +F                +0       +F           +F          +F             +F           +inf           NaN
                       +inf            NaN          +inf           +inf          +inf            +inf            +inf           NaN
                      NaN            NaN          NaN         NaN         NaN            NaN           NaN          NaN
 NOTES:
 F Means finite floating-point value.

In most cases, only the exponent is changed and the mantissa (significand) remains unchanged. However, when
the value being scaled in ST(0) is a denormal value, the mantissa is also changed and the result may turn out to be
a normalized number. Similarly, if overflow or underflow results from a scale operation, the resulting mantissa will
differ from the source''s mantissa.
The FSCALE instruction can also be used to reverse the action of the FXTRACT instruction, as shown in the following
example:
     FXTRACT;
     FSCALE;
     FSTP ST(1);
In this example, the FXTRACT instruction extracts the significand and exponent from the value in ST(0) and stores
them in ST(0) and ST(1) respectively. The FSCALE then scales the significand in ST(0) by the exponent in ST(1),
recreating the original value before the FXTRACT operation was performed. The FSTP ST(1) instruction overwrites
the exponent (extracted by the FXTRACT instruction) with the recreated value, which returns the stack to its orig-
inal state with only one register [ST(0)] occupied.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
ST(0) <- ST(0) * 2RoundTowardZero(ST(1));

FPU Flags Affected
C1                          Set to 0 if stack underflow occurred.
                            Set if result was rounded up; cleared otherwise.
C0, C2, C3                  Undefined.



Floating-Point Exceptions
#IS                 Stack underflow occurred.
#IA                 Source operand is an SNaN value or unsupported format.
#D                  Source operand is a denormal value.
#U                  Result is too small for destination format.
#O                  Result is too large for destination format.
#P                  Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FSIN','FSIN-Sine
Opcode                   Instruction             64-Bit        Compat/            Description
                                                 Mode          Leg Mode
D9 FE                    FSIN                    Valid         Valid          Replace ST(0) with the approximate of its sine.



Description
Computes an approximation of the sine of the source operand in register ST(0) and stores the result in ST(0). The
source operand must be given in radians and must be within the range -263 to +263. The following table shows the
results obtained when taking the sine of various classes of numbers, assuming that underflow does not occur.
                                                     Table 3-35. FSIN Results
                           SRC (ST(0))                                                          DEST (ST(0))
                                -inf                                                                    *
                                -F                                                               - 1 to + 1
                                -0                                                                   -0
                                +0                                                                  +0
                                +F                                                                - 1 to +1
                                +inf                                                                    *
                                NaN                                                                 NaN
 NOTES:
 F Means finite floating-point value.
 * Indicates floating-point invalid-arithmetic-operand (#IA) exception.

If the source operand is outside the acceptable range, the C2 flag in the FPU status word is set, and the value in
register ST(0) remains unchanged. The instruction does not raise an exception when the source operand is out of
range. It is up to the program to check the C2 flag for out-of-range conditions. Source values outside the range -
263 to +263 can be reduced to the range of the instruction by subtracting an appropriate integer multiple of 2pi.
However, even within the range -263 to +263, inaccurate results can occur because the finite approximation of pi
used internally for argument reduction is not sufficient in all cases. Therefore, for accurate results it is safe to apply
FSIN only to arguments reduced accurately in software, to a value smaller in absolute value than 3pi/4. See the
sections titled "Approximation of Pi" and "Transcendental Instruction Accuracy" in Chapter 8 of the Intel 64 and
IA-32 Architectures Software Developer''s Manual, Volume 1, for a discussion of the proper value to use for pi in
performing such reductions.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
IF -263 < ST(0) < 263
    THEN
         C2 <- 0;
         ST(0) <- fsin(ST(0)); // approximation of the mathematical sin function
    ELSE (* Source operand out of range *)
         C2 <- 1;
FI;

FPU Flags Affected
C1                      Set to 0 if stack underflow occurred.
                        Set if result was rounded up; cleared otherwise.
C2                      Set to 1 if outside range (-263 < source operand < +263); otherwise, set to 0.
C0, C3                  Undefined.




Floating-Point Exceptions
#IS                 Stack underflow occurred.
#IA                 Source operand is an SNaN value, inf, or unsupported format.
#D                  Source operand is a denormal value.
#P                  Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FSINCOS','FSINCOS-Sine and Cosine
 Opcode                   Instruction            64-Bit    Compat/         Description
                                                 Mode      Leg Mode
 D9 FB                    FSINCOS                Valid     Valid           Compute the sine and cosine of ST(0); replace ST(0) with the
                                                                           approximate sine, and push the approximate cosine onto the
                                                                           register stack.



Description
Computes both the approximate sine and the cosine of the source operand in register ST(0), stores the sine in
ST(0), and pushes the cosine onto the top of the FPU register stack. (This instruction is faster than executing the
FSIN and FCOS instructions in succession.)
The source operand must be given in radians and must be within the range -263 to +263. The following table shows
the results obtained when taking the sine and cosine of various classes of numbers, assuming that underflow does
not occur.
                                                   Table 3-36. FSINCOS Results
                    SRC                                                                  DEST
                   ST(0)                                    ST(1) Cosine                                    ST(0) Sine
                    -inf                                              *                                            *
                    -F                                        - 1 to + 1                                     - 1 to + 1
                    -0                                             +1                                           -0
                    +0                                             +1                                           +0
                    +F                                        - 1 to + 1                                     - 1 to + 1
                    +inf                                              *                                            *
                   NaN                                             NaN                                         NaN
 NOTES:
 F Means finite floating-point value.
 * Indicates floating-point invalid-arithmetic-operand (#IA) exception.

If the source operand is outside the acceptable range, the C2 flag in the FPU status word is set, and the value in
register ST(0) remains unchanged. The instruction does not raise an exception when the source operand is out of
range. It is up to the program to check the C2 flag for out-of-range conditions. Source values outside the range -
263 to +263 can be reduced to the range of the instruction by subtracting an appropriate integer multiple of 2pi.
However, even within the range -263 to +263, inaccurate results can occur because the finite approximation of pi
used internally for argument reduction is not sufficient in all cases. Therefore, for accurate results it is safe to apply
FSINCOS only to arguments reduced accurately in software, to a value smaller in absolute value than 3pi/8. See the
sections titled "Approximation of Pi" and "Transcendental Instruction Accuracy" in Chapter 8 of the Intel 64 and
IA-32 Architectures Software Developer''s Manual, Volume 1, for a discussion of the proper value to use for pi in
performing such reductions.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.





Operation
IF ST(0) < 263
    THEN
         C2 <- 0;
         TEMP <- fcos(ST(0)); // approximation of cosine
         ST(0) <- fsin(ST(0)); // approximation of sine
         TOP <- TOP - 1;
         ST(0) <- TEMP;
    ELSE (* Source operand out of range *)
         C2 <- 1;
FI;

FPU Flags Affected
C1                     Set to 0 if stack underflow occurred; set to 1 of stack overflow occurs.
                       Set if result was rounded up; cleared otherwise.
C2                     Set to 1 if outside range (-263 < source operand < +263); otherwise, set to 0.
C0, C3                 Undefined.

Floating-Point Exceptions
#IS                    Stack underflow or overflow occurred.
#IA                    Source operand is an SNaN value, inf, or unsupported format.
#D                     Source operand is a denormal value.
#U                     Result is too small for destination format.
#P                     Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                    If there is a pending x87 FPU exception.
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FSQRT','FSQRT-Square Root
Opcode                  Instruction            64-Bit     Compat/         Description
                                               Mode       Leg Mode
D9 FA                   FSQRT                  Valid      Valid           Computes square root of ST(0) and stores the result in ST(0).



Description
Computes the square root of the source value in the ST(0) register and stores the result in ST(0).
The following table shows the results obtained when taking the square root of various classes of numbers,
assuming that neither overflow nor underflow occurs.
                                                    Table 3-37. FSQRT Results
                           SRC (ST(0))                                                         DEST (ST(0))
                                -inf                                                                   *
                                -F                                                                   *
                                -0                                                                  -0
                                +0                                                                  +0
                                +F                                                                  +F
                                +inf                                                                  +inf
                              NaN                                                                   NaN
 NOTES:
 F Means finite floating-point value.
 * Indicates floating-point invalid-arithmetic-operand (#IA) exception.

This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
ST(0) <- SquareRoot(ST(0));

FPU Flags Affected
C1                      Set to 0 if stack underflow occurred.
                        Set if result was rounded up; cleared otherwise.
C0, C2, C3              Undefined.

Floating-Point Exceptions
#IS                     Stack underflow occurred.
#IA                     Source operand is an SNaN value or unsupported format.
                        Source operand is a negative value (except for -0).
#D                      Source operand is a denormal value.
#P                      Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                     If there is a pending x87 FPU exception.
#UD                     If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.




Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FST','FST/FSTP-Store Floating Point Value
 Opcode                   Instruction         64-Bit   Compat/       Description
                                              Mode     Leg Mode
 D9 /2                    FST m32fp           Valid    Valid         Copy ST(0) to m32fp.
 DD /2                    FST m64fp           Valid    Valid         Copy ST(0) to m64fp.
 DD D0+i                  FST ST(i)           Valid    Valid         Copy ST(0) to ST(i).
 D9 /3                    FSTP m32fp          Valid    Valid         Copy ST(0) to m32fp and pop register stack.
 DD /3                    FSTP m64fp          Valid    Valid         Copy ST(0) to m64fp and pop register stack.
 DB /7                    FSTP m80fp          Valid    Valid         Copy ST(0) to m80fp and pop register stack.
 DD D8+i                  FSTP ST(i)          Valid    Valid         Copy ST(0) to ST(i) and pop register stack.



Description
The FST instruction copies the value in the ST(0) register to the destination operand, which can be a memory loca-
tion or another register in the FPU register stack. When storing the value in memory, the value is converted to
single-precision or double-precision floating-point format.
The FSTP instruction performs the same operation as the FST instruction and then pops the register stack. To pop
the register stack, the processor marks the ST(0) register as empty and increments the stack pointer (TOP) by 1.
The FSTP instruction can also store values in memory in double extended-precision floating-point format.
If the destination operand is a memory location, the operand specifies the address where the first byte of the desti-
nation value is to be stored. If the destination operand is a register, the operand specifies a register in the register
stack relative to the top of the stack.
If the destination size is single-precision or double-precision, the significand of the value being stored is rounded
to the width of the destination (according to the rounding mode specified by the RC field of the FPU control word),
and the exponent is converted to the width and bias of the destination format. If the value being stored is too large
for the destination format, a numeric overflow exception (#O) is generated and, if the exception is unmasked, no
value is stored in the destination operand. If the value being stored is a denormal value, the denormal exception
(#D) is not generated. This condition is simply signaled as a numeric underflow exception (#U) condition.
If the value being stored is +-0, +-inf, or a NaN, the least-significant bits of the significand and the exponent are trun-
cated to fit the destination format. This operation preserves the value''s identity as a 0, inf, or NaN.
If the destination operand is a non-empty register, the invalid-operation exception is not generated.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
DEST <- ST(0);

IF Instruction = FSTP
     THEN
         PopRegisterStack;
FI;

FPU Flags Affected
C1                       Set to 0 if stack underflow occurred.
                         Indicates rounding direction of if the floating-point inexact exception (#P) is generated: 0 <-
                         not roundup; 1 <- roundup.
C0, C2, C3               Undefined.





Floating-Point Exceptions
#IS                 Stack underflow occurred.
#IA                 If destination result is an SNaN value or unsupported format, except when the destination
                    format is in double extended-precision floating-point format.
#U                  Result is too small for the destination format.
#O                  Result is too large for the destination format.
#P                  Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#GP(0)              If the destination is located in a non-writable segment.
                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                    selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                 If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FSTP','-R:FST');
INSERT INTO "instructions" VALUES('x86_64','FSTCW','FSTCW/FNSTCW-Store x87 FPU Control Word
Opcode               Instruction                    64-Bit   Compat/    Description
                                                    Mode     Leg Mode
9B D9 /7             FSTCW m2byte                   Valid    Valid      Store FPU control word to m2byte after checking for
                                                                        pending unmasked floating-point exceptions.
D9 /7                FNSTCW* m2byte                 Valid    Valid      Store FPU control word to m2byte without checking for
                                                                        pending unmasked floating-point exceptions.
NOTES:
* See IA-32 Architecture Compatibility section below.



Description
Stores the current value of the FPU control word at the specified destination in memory. The FSTCW instruction
checks for and handles pending unmasked floating-point exceptions before storing the control word; the FNSTCW
instruction does not.
The assembler issues two instructions for the FSTCW instruction (an FWAIT instruction followed by an FNSTCW
instruction), and the processor executes each of these instructions in separately. If an exception is generated for
either of these instructions, the save EIP points to the instruction that caused the exception.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

IA-32 Architecture Compatibility
When operating a Pentium or Intel486 processor in MS-DOS compatibility mode, it is possible (under unusual
circumstances) for an FNSTCW instruction to be interrupted prior to being executed to handle a pending FPU
exception. See the section titled "No-Wait FPU Instructions Can Get FPU Interrupt in Window" in Appendix D of the
Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 1, for a description of these circum-
stances. An FNSTCW instruction cannot be interrupted in this way on later Intel processors, except for the Intel
QuarkTM X1000 processor.

Operation
DEST <- FPUControlWord;

FPU Flags Affected
The C0, C1, C2, and C3 flags are undefined.

Floating-Point Exceptions
None.

Protected Mode Exceptions
#GP(0)                 If the destination is located in a non-writable segment.
                       If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                       If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                       selector.
#SS(0)                 If a memory operand effective address is outside the SS segment limit.
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)        If a page fault occurs.
#AC(0)                 If alignment checking is enabled and an unaligned memory reference is made while the
                       current privilege level is 3.
#UD                    If the LOCK prefix is used.





Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                 If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FNSTCW','-R:FSTCW');
INSERT INTO "instructions" VALUES('x86_64','FSTENV','FSTENV/FNSTENV-Store x87 FPU Environment
 Opcode               Instruction                        64-Bit   Compat/    Description
                                                         Mode     Leg Mode
 9B D9 /6             FSTENV m14/28byte                  Valid    Valid      Store FPU environment to m14byte or m28byte
                                                                             after checking for pending unmasked floating-point
                                                                             exceptions. Then mask all floating-point exceptions.
 D9 /6                FNSTENV* m14/28byte                Valid    Valid      Store FPU environment to m14byte or m28byte
                                                                             without checking for pending unmasked floating-
                                                                             point exceptions. Then mask all floating-
                                                                             point exceptions.
 NOTES:
 * See IA-32 Architecture Compatibility section below.



Description
Saves the current FPU operating environment at the memory location specified with the destination operand, and
then masks all floating-point exceptions. The FPU operating environment consists of the FPU control word, status
word, tag word, instruction pointer, data pointer, and last opcode. Figures 8-9 through 8-12 in the Intel 64 and
IA-32 Architectures Software Developer''s Manual, Volume 1, show the layout in memory of the stored environ-
ment, depending on the operating mode of the processor (protected or real) and the current operand-size attribute
(16-bit or 32-bit). In virtual-8086 mode, the real mode layouts are used.
The FSTENV instruction checks for and handles any pending unmasked floating-point exceptions before storing
the FPU environment; the FNSTENV instruction does not. The saved image reflects the state of the FPU after all
floating-point instructions preceding the FSTENV/FNSTENV instruction in the instruction stream have been
executed.
These instructions are often used by exception handlers because they provide access to the FPU instruction and
data pointers. The environment is typically saved in the stack. Masking all exceptions after saving the environment
prevents floating-point exceptions from interrupting the exception handler.
The assembler issues two instructions for the FSTENV instruction (an FWAIT instruction followed by an FNSTENV
instruction), and the processor executes each of these instructions separately. If an exception is generated for
either of these instructions, the save EIP points to the instruction that caused the exception.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

IA-32 Architecture Compatibility
When operating a Pentium or Intel486 processor in MS-DOS compatibility mode, it is possible (under unusual
circumstances) for an FNSTENV instruction to be interrupted prior to being executed to handle a pending FPU
exception. See the section titled "No-Wait FPU Instructions Can Get FPU Interrupt in Window" in Appendix D of the
Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 1, for a description of these circum-
stances. An FNSTENV instruction cannot be interrupted in this way on later Intel processors, except for the Intel
QuarkTM X1000 processor.

Operation
DEST[FPUControlWord] <- FPUControlWord;
DEST[FPUStatusWord] <- FPUStatusWord;
DEST[FPUTagWord] <- FPUTagWord;
DEST[FPUDataPointer] <- FPUDataPointer;
DEST[FPUInstructionPointer] <- FPUInstructionPointer;
DEST[FPULastInstructionOpcode] <- FPULastInstructionOpcode;

FPU Flags Affected
The C0, C1, C2, and C3 are undefined.




Floating-Point Exceptions
None

Protected Mode Exceptions
#GP(0)              If the destination is located in a non-writable segment.
                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                    selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                 If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FNSTENV','-R:FSTENV');
INSERT INTO "instructions" VALUES('x86_64','FSTSW','FSTSW/FNSTSW-Store x87 FPU Status Word
Opcode               Instruction                    64-Bit      Compat/    Description
                                                    Mode        Leg Mode
9B DD /7             FSTSW m2byte                       Valid   Valid      Store FPU status word at m2byte after checking
                                                                           for pending unmasked floating-point exceptions.
9B DF E0             FSTSW AX                           Valid   Valid      Store FPU status word in AX register after
                                                                           checking for pending unmasked floating-point
                                                                           exceptions.
DD /7                FNSTSW* m2byte                     Valid   Valid      Store FPU status word at m2byte without
                                                                           checking for pending unmasked floating-point
                                                                           exceptions.
DF E0                FNSTSW* AX                         Valid   Valid      Store FPU status word in AX register without
                                                                           checking for pending unmasked floating-point
                                                                           exceptions.
NOTES:
* See IA-32 Architecture Compatibility section below.



Description
Stores the current value of the x87 FPU status word in the destination location. The destination operand can be
either a two-byte memory location or the AX register. The FSTSW instruction checks for and handles pending
unmasked floating-point exceptions before storing the status word; the FNSTSW instruction does not.
The FNSTSW AX form of the instruction is used primarily in conditional branching (for instance, after an FPU
comparison instruction or an FPREM, FPREM1, or FXAM instruction), where the direction of the branch depends on
the state of the FPU condition code flags. (See the section titled "Branching and Conditional Moves on FPU Condi-
tion Codes" in Chapter 8 of the Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 1.) This
instruction can also be used to invoke exception handlers (by examining the exception flags) in environments that
do not use interrupts. When the FNSTSW AX instruction is executed, the AX register is updated before the
processor executes any further instructions. The status stored in the AX register is thus guaranteed to be from the
completion of the prior FPU instruction.
The assembler issues two instructions for the FSTSW instruction (an FWAIT instruction followed by an FNSTSW
instruction), and the processor executes each of these instructions separately. If an exception is generated for
either of these instructions, the save EIP points to the instruction that caused the exception.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

IA-32 Architecture Compatibility
When operating a Pentium or Intel486 processor in MS-DOS compatibility mode, it is possible (under unusual
circumstances) for an FNSTSW instruction to be interrupted prior to being executed to handle a pending FPU
exception. See the section titled "No-Wait FPU Instructions Can Get FPU Interrupt in Window" in Appendix D of the
Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 1, for a description of these circum-
stances. An FNSTSW instruction cannot be interrupted in this way on later Intel processors, except for the Intel
QuarkTM X1000 processor.

Operation
DEST <- FPUStatusWord;

FPU Flags Affected
The C0, C1, C2, and C3 are undefined.

Floating-Point Exceptions
None




Protected Mode Exceptions
#GP(0)              If the destination is located in a non-writable segment.
                    If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                    selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FNSTSW','-R:FSTSW');
INSERT INTO "instructions" VALUES('x86_64','FSUB','FSUB/FSUBP/FISUB-Subtract
Opcode             Instruction               64-Bit      Compat/       Description
                                             Mode        Leg Mode
D8 /4              FSUB m32fp                Valid       Valid         Subtract m32fp from ST(0) and store result in ST(0).
DC /4              FSUB m64fp                Valid       Valid         Subtract m64fp from ST(0) and store result in ST(0).
D8 E0+i            FSUB ST(0), ST(i)         Valid       Valid         Subtract ST(i) from ST(0) and store result in ST(0).
DC E8+i            FSUB ST(i), ST(0)         Valid       Valid         Subtract ST(0) from ST(i) and store result in ST(i).
DE E8+i            FSUBP ST(i), ST(0)        Valid       Valid         Subtract ST(0) from ST(i), store result in ST(i), and
                                                                       pop register stack.
DE E9              FSUBP                     Valid       Valid         Subtract ST(0) from ST(1), store result in ST(1), and
                                                                       pop register stack.
DA /4              FISUB m32int              Valid       Valid         Subtract m32int from ST(0) and store result in ST(0).
DE /4              FISUB m16int              Valid       Valid         Subtract m16int from ST(0) and store result in ST(0).



Description
Subtracts the source operand from the destination operand and stores the difference in the destination location.
The destination operand is always an FPU data register; the source operand can be a register or a memory location.
Source operands in memory can be in single-precision or double-precision floating-point format or in word or
doubleword integer format.
The no-operand version of the instruction subtracts the contents of the ST(0) register from the ST(1) register and
stores the result in ST(1). The one-operand version subtracts the contents of a memory location (either a floating-
point or an integer value) from the contents of the ST(0) register and stores the result in ST(0). The two-operand
version, subtracts the contents of the ST(0) register from the ST(i) register or vice versa.
The FSUBP instructions perform the additional operation of popping the FPU register stack following the subtrac-
tion. To pop the register stack, the processor marks the ST(0) register as empty and increments the stack pointer
(TOP) by 1. The no-operand version of the floating-point subtract instructions always results in the register stack
being popped. In some assemblers, the mnemonic for this instruction is FSUB rather than FSUBP.
The FISUB instructions convert an integer source operand to double extended-precision floating-point format
before performing the subtraction.
Table 3-38 shows the results obtained when subtracting various classes of numbers from one another, assuming
that neither overflow nor underflow occurs. Here, the SRC value is subtracted from the DEST value (DEST - SRC =
result).
When the difference between two operands of like sign is 0, the result is +0, except for the round toward -inf mode,
in which case the result is -0. This instruction also guarantees that +0 - (-0) = +0, and that -0 - (+0) = -0. When the
source operand is an integer 0, it is treated as a +0.
When one operand is inf, the result is inf of the expected sign. If both operands are inf of the same sign, an invalid-
operation exception is generated.





                                            Table 3-38. FSUB/FSUBP/FISUB Results
                                                                          SRC
                                  -inf           - F or - I        -0             +0     + F or + I    +inf            NaN
                   -inf              *             -inf              -inf             -inf       -inf          -inf            NaN
                   -F             +inf           +-F or +-0         DEST            DEST      -F         -inf            NaN
 DEST              -0             +inf            -SRC              +-0            -0      - SRC        -inf            NaN
                   +0             +inf            -SRC             +0              +-0     - SRC        -inf            NaN
                   +F             +inf              +F            DEST            DEST   +-F or +-0      -inf            NaN
                   +inf             +inf             +inf              +inf             +inf       +inf            *           NaN
                  NaN            NaN             NaN             NaN            NaN      NaN         NaN           NaN
 NOTES:
 F Means finite floating-point value.
 I Means integer.
 * Indicates floating-point invalid-arithmetic-operand (#IA) exception.

This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
IF Instruction = FISUB
     THEN
         DEST <- DEST - ConvertToDoubleExtendedPrecisionFP(SRC);
     ELSE (* Source operand is floating-point value *)
         DEST <- DEST - SRC;
FI;

IF Instruction = FSUBP
     THEN
         PopRegisterStack;
FI;

FPU Flags Affected
C1                      Set to 0 if stack underflow occurred.
                        Set if result was rounded up; cleared otherwise.
C0, C2, C3              Undefined.

Floating-Point Exceptions
#IS                     Stack underflow occurred.
#IA                     Operand is an SNaN value or unsupported format.
                        Operands are infinities of like sign.
#D                      Source operand is a denormal value.
#U                      Result is too small for destination format.
#O                      Result is too large for destination format.
#P                      Value cannot be represented exactly in destination format.





Protected Mode Exceptions
#GP(0)                If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                      If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                      selector.
#SS(0)                If a memory operand effective address is outside the SS segment limit.
#NM                   CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)       If a page fault occurs.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made while the
                      current privilege level is 3.
#UD                   If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                   If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                   If a memory operand effective address is outside the SS segment limit.
#NM                   CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                   If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)                If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)                If a memory operand effective address is outside the SS segment limit.
#NM                   CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)       If a page fault occurs.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made.
#UD                   If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)                If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)                If the memory address is in a non-canonical form.
#NM                   CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                   If there is a pending x87 FPU exception.
#PF(fault-code)       If a page fault occurs.
#AC(0)                If alignment checking is enabled and an unaligned memory reference is made while the
                      current privilege level is 3.
#UD                   If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FSUBP','-R:FSUB');
INSERT INTO "instructions" VALUES('x86_64','FISUB','-R:FSUB');
INSERT INTO "instructions" VALUES('x86_64','FSUBR','FSUBR/FSUBRP/FISUBR-Reverse Subtract
 Opcode           Instruction                   64-Bit    Compat/      Description
                                                Mode      Leg Mode
 D8 /5            FSUBR m32fp                   Valid     Valid        Subtract ST(0) from m32fp and store result in ST(0).
 DC /5            FSUBR m64fp                   Valid     Valid        Subtract ST(0) from m64fp and store result in ST(0).
 D8 E8+i          FSUBR ST(0), ST(i)            Valid     Valid        Subtract ST(0) from ST(i) and store result in ST(0).
 DC E0+i          FSUBR ST(i), ST(0)            Valid     Valid        Subtract ST(i) from ST(0) and store result in ST(i).
 DE E0+i          FSUBRP ST(i), ST(0)           Valid     Valid        Subtract ST(i) from ST(0), store result in ST(i), and
                                                                       pop register stack.
 DE E1            FSUBRP                        Valid     Valid        Subtract ST(1) from ST(0), store result in ST(1), and
                                                                       pop register stack.
 DA /5            FISUBR m32int                 Valid     Valid        Subtract ST(0) from m32int and store result in ST(0).
 DE /5            FISUBR m16int                 Valid     Valid        Subtract ST(0) from m16int and store result in ST(0).



Description
Subtracts the destination operand from the source operand and stores the difference in the destination location.
The destination operand is always an FPU register; the source operand can be a register or a memory location.
Source operands in memory can be in single-precision or double-precision floating-point format or in word or
doubleword integer format.
These instructions perform the reverse operations of the FSUB, FSUBP, and FISUB instructions. They are provided
to support more efficient coding.
The no-operand version of the instruction subtracts the contents of the ST(1) register from the ST(0) register and
stores the result in ST(1). The one-operand version subtracts the contents of the ST(0) register from the contents
of a memory location (either a floating-point or an integer value) and stores the result in ST(0). The two-operand
version, subtracts the contents of the ST(i) register from the ST(0) register or vice versa.
The FSUBRP instructions perform the additional operation of popping the FPU register stack following the subtrac-
tion. To pop the register stack, the processor marks the ST(0) register as empty and increments the stack pointer
(TOP) by 1. The no-operand version of the floating-point reverse subtract instructions always results in the register
stack being popped. In some assemblers, the mnemonic for this instruction is FSUBR rather than FSUBRP.
The FISUBR instructions convert an integer source operand to double extended-precision floating-point format
before performing the subtraction.
The following table shows the results obtained when subtracting various classes of numbers from one another,
assuming that neither overflow nor underflow occurs. Here, the DEST value is subtracted from the SRC value (SRC
- DEST = result).
When the difference between two operands of like sign is 0, the result is +0, except for the round toward -inf mode,
in which case the result is -0. This instruction also guarantees that +0 - (-0) = +0, and that -0 - (+0) = -0. When the
source operand is an integer 0, it is treated as a +0.
When one operand is inf, the result is inf of the expected sign. If both operands are inf of the same sign, an invalid-
operation exception is generated.





                                         Table 3-39. FSUBR/FSUBRP/FISUBR Results
                                                                          SRC
                                  -inf           -F or -I           -0             +0     +F or +I       +inf           NaN
                   -inf              *              +inf             +inf              +inf       +inf           +inf           NaN
                   -F             -inf           +-F or +-0        -DEST            -DEST     +F           +inf           NaN
 DEST              -0             -inf             SRC              +-0             +0      SRC           +inf           NaN
                   +0             -inf             SRC              -0             +-0      SRC           +inf           NaN
                   +F             -inf              -F           -DEST            -DEST   +-F or +-0       +inf           NaN
                   +inf             -inf              -inf             -inf              -inf       -inf             *          NaN
                  NaN            NaN             NaN             NaN            NaN      NaN           NaN          NaN
 NOTES:
 F Means finite floating-point value.
 I Means integer.
 * Indicates floating-point invalid-arithmetic-operand (#IA) exception.

This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
IF Instruction = FISUBR
     THEN
         DEST <- ConvertToDoubleExtendedPrecisionFP(SRC) - DEST;
     ELSE (* Source operand is floating-point value *)
         DEST <- SRC - DEST; FI;
IF Instruction = FSUBRP
     THEN
         PopRegisterStack; FI;

FPU Flags Affected
C1                      Set to 0 if stack underflow occurred.
                        Set if result was rounded up; cleared otherwise.
C0, C2, C3              Undefined.

Floating-Point Exceptions
#IS                     Stack underflow occurred.
#IA                     Operand is an SNaN value or unsupported format.
                        Operands are infinities of like sign.
#D                      Source operand is a denormal value.
#U                      Result is too small for destination format.
#O                      Result is too large for destination format.
#P                      Value cannot be represented exactly in destination format.





Protected Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
                    If the DS, ES, FS, or GS register is used to access memory and it contains a NULL segment
                    selector.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                 If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS                 If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#UD                 If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
#GP(0)              If a memory operand effective address is outside the CS, DS, ES, FS, or GS segment limit.
#SS(0)              If a memory operand effective address is outside the SS segment limit.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made.
#UD                 If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#PF(fault-code)     If a page fault occurs.
#AC(0)              If alignment checking is enabled and an unaligned memory reference is made while the
                    current privilege level is 3.
#UD                 If the LOCK prefix is used.




');
INSERT INTO "instructions" VALUES('x86_64','FSUBRP','-R:FSUBR');
INSERT INTO "instructions" VALUES('x86_64','FISUBR','-R:FSUBR');
INSERT INTO "instructions" VALUES('x86_64','FTST','FTST-TEST
Opcode                    Instruction           64-Bit    Compat/         Description
                                                Mode      Leg Mode
D9 E4                     FTST                  Valid     Valid           Compare ST(0) with 0.0.



Description
Compares the value in the ST(0) register with 0.0 and sets the condition code flags C0, C2, and C3 in the FPU
status word according to the results (see table below).
                                               Table 3-40. FTST Results
                      Condition                              C3                    C2                      C0
                     ST(0) > 0.0                             0                     0                       0
                     ST(0) < 0.0                             0                     0                       1
                     ST(0) = 0.0                             1                     0                       0
                     Unordered                               1                     1                       1

This instruction performs an "unordered comparison." An unordered comparison also checks the class of the
numbers being compared (see "FXAM-Examine Floating-Point" in this chapter). If the value in register ST(0) is a
NaN or is in an undefined format, the condition flags are set to "unordered" and the invalid operation exception is
generated.
The sign of zero is ignored, so that (- 0.0 <- +0.0).
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
CASE (relation of operands) OF
   Not comparable: C3, C2, C0 <- 111;
   ST(0) > 0.0:       C3, C2, C0 <- 000;
   ST(0) < 0.0:       C3, C2, C0 <- 001;
   ST(0) = 0.0:       C3, C2, C0 <- 100;
ESAC;

FPU Flags Affected
C1                     Set to 0.
C0, C2, C3             See Table 3-40.

Floating-Point Exceptions
#IS                    Stack underflow occurred.
#IA                    The source operand is a NaN value or is in an unsupported format.
#D                     The source operand is a denormal value.

Protected Mode Exceptions
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                    If there is a pending x87 FPU exception.
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.





Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FUCOM','FUCOM/FUCOMP/FUCOMPP-Unordered Compare Floating Point Values
Opcode                Instruction               64-Bit       Compat/      Description
                                                Mode         Leg Mode
DD E0+i               FUCOM ST(i)               Valid        Valid        Compare ST(0) with ST(i).
DD E1                 FUCOM                     Valid        Valid        Compare ST(0) with ST(1).
DD E8+i               FUCOMP ST(i)              Valid        Valid        Compare ST(0) with ST(i) and pop register stack.
DD E9                 FUCOMP                    Valid        Valid        Compare ST(0) with ST(1) and pop register stack.
DA E9                 FUCOMPP                   Valid        Valid        Compare ST(0) with ST(1) and pop register stack twice.



Description
Performs an unordered comparison of the contents of register ST(0) and ST(i) and sets condition code flags C0, C2,
and C3 in the FPU status word according to the results (see the table below). If no operand is specified, the
contents of registers ST(0) and ST(1) are compared. The sign of zero is ignored, so that -0.0 is equal to +0.0.
                                        Table 3-41. FUCOM/FUCOMP/FUCOMPP Results
                     Comparison Results*                                   C3                     C2                    C0
                          ST0 > ST(i)                                       0                      0                     0
                          ST0 < ST(i)                                       0                      0                     1
                          ST0 = ST(i)                                       1                      0                     0
                           Unordered                                        1                      1                     1
 NOTES:
 * Flags not set if unmasked invalid-arithmetic-operand (#IA) exception is generated.

An unordered comparison checks the class of the numbers being compared (see "FXAM-Examine Floating-Point"
in this chapter). The FUCOM/FUCOMP/FUCOMPP instructions perform the same operations as the
FCOM/FCOMP/FCOMPP instructions. The only difference is that the FUCOM/FUCOMP/FUCOMPP instructions raise
the invalid-arithmetic-operand exception (#IA) only when either or both operands are an SNaN or are in an unsup-
ported format; QNaNs cause the condition code flags to be set to unordered, but do not cause an exception to be
generated. The FCOM/FCOMP/FCOMPP instructions raise an invalid-operation exception when either or both of the
operands are a NaN value of any kind or are in an unsupported format.
As with the FCOM/FCOMP/FCOMPP instructions, if the operation results in an invalid-arithmetic-operand exception
being raised, the condition code flags are set only if the exception is masked.
The FUCOMP instruction pops the register stack following the comparison operation and the FUCOMPP instruction
pops the register stack twice following the comparison operation. To pop the register stack, the processor marks
the ST(0) register as empty and increments the stack pointer (TOP) by 1.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.





Operation
CASE (relation of operands) OF
   ST > SRC:          C3, C2, C0 <- 000;
   ST < SRC:          C3, C2, C0 <- 001;
   ST = SRC:          C3, C2, C0 <- 100;
ESAC;

IF ST(0) or SRC = QNaN, but not SNaN or unsupported format
    THEN
         C3, C2, C0 <- 111;
    ELSE (* ST(0) or SRC is SNaN or unsupported format *)
          #IA;
         IF FPUControlWord.IM = 1
               THEN
                   C3, C2, C0 <- 111;
         FI;
FI;

IF Instruction = FUCOMP
     THEN
         PopRegisterStack;
FI;

IF Instruction = FUCOMPP
     THEN
         PopRegisterStack;
FI;

FPU Flags Affected
C1                     Set to 0 if stack underflow occurred.
C0, C2, C3             See Table 3-41.

Floating-Point Exceptions
#IS                    Stack underflow occurred.
#IA                    One or both operands are SNaN values or have unsupported formats. Detection of a QNaN
                       value in and of itself does not raise an invalid-operand exception.
#D                     One or both operands are denormal values.

Protected Mode Exceptions
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                    If there is a pending x87 FPU exception.
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.





64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FUCOMP','-R:FUCOM');
INSERT INTO "instructions" VALUES('x86_64','FUCOMPP','-R:FUCOM');
INSERT INTO "instructions" VALUES('x86_64','FXAM','FXAM-Examine Floating-Point
    Opcode                 Instruction              64-Bit     Compat/       Description
                                                    Mode       Leg Mode
    D9 E5                  FXAM                     Valid       Valid        Classify value or number in ST(0).



Description
Examines the contents of the ST(0) register and sets the condition code flags C0, C2, and C3 in the FPU status word
to indicate the class of value or number in the register (see the table below).
                                                       Table 3-42. FXAM Results
.




                             Class                                      C3                   C2                      C0
    Unsupported                                                         0                     0                      0
    NaN                                                                 0                     0                      1
    Normal finite number                                                0                     1                      0
    Infinity                                                            0                     1                      1
    Zero                                                                1                     0                      0
    Empty                                                               1                     0                      1
    Denormal number                                                     1                     1                      0

The C1 flag is set to the sign of the value in ST(0), regardless of whether the register is empty or full.
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
C1 <- sign bit of ST; (* 0 for positive, 1 for negative *)

CASE (class of value or number in ST(0)) OF
   Unsupported:C3, C2, C0 <- 000;
   NaN:           C3, C2, C0 <- 001;
   Normal:        C3, C2, C0 <- 010;
   Infinity:      C3, C2, C0 <- 011;
   Zero:          C3, C2, C0 <- 100;
   Empty:         C3, C2, C0 <- 101;
   Denormal:      C3, C2, C0 <- 110;
ESAC;

FPU Flags Affected
C1                         Sign of value in ST(0).
C0, C2, C3                 See Table 3-42.

Floating-Point Exceptions
None

Protected Mode Exceptions
#NM                        CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                        If there is a pending x87 FPU exception.
#UD                        If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.



Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FXCH','FXCH-Exchange Register Contents
Opcode                  Instruction         64-Bit    Compat/       Description
                                            Mode      Leg Mode
D9 C8+i                 FXCH ST(i)          Valid     Valid         Exchange the contents of ST(0) and ST(i).
D9 C9                   FXCH                Valid     Valid         Exchange the contents of ST(0) and ST(1).



Description
Exchanges the contents of registers ST(0) and ST(i). If no source operand is specified, the contents of ST(0) and
ST(1) are exchanged.
This instruction provides a simple means of moving values in the FPU register stack to the top of the stack [ST(0)],
so that they can be operated on by those floating-point instructions that can only operate on values in ST(0). For
example, the following instruction sequence takes the square root of the third register from the top of the register
stack:
     FXCH ST(3);
     FSQRT;
     FXCH ST(3);
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
IF (Number-of-operands) is 1
    THEN
        temp <- ST(0);
        ST(0) <- SRC;
        SRC <- temp;
    ELSE
        temp <- ST(0);
        ST(0) <- ST(1);
        ST(1) <- temp;
FI;

FPU Flags Affected
C1                     Set to 0.
C0, C2, C3             Undefined.

Floating-Point Exceptions
#IS                    Stack underflow occurred.

Protected Mode Exceptions
#NM                    CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                    If there is a pending x87 FPU exception.
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.





Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FXRSTOR','FXRSTOR-Restore x87 FPU, MMX, XMM, and MXCSR State
 Opcode/                                         Op/     64-Bit   Compat/ Description
 Instruction                                     En      Mode     Leg Mode
 0F AE /1                                        M       Valid    Valid        Restore the x87 FPU, MMX, XMM, and MXCSR
 FXRSTOR m512byte                                                              register state from m512byte.

 REX.W+ 0F AE /1                                 M       Valid    N.E.         Restore the x87 FPU, MMX, XMM, and MXCSR
 FXRSTOR64 m512byte                                                            register state from m512byte.



                                            Instruction Operand Encoding
   Op/En            Operand 1                Operand 2                    Operand 3                     Operand 4
     M             ModRM:r/m (r)                 NA                           NA                           NA

Description
Reloads the x87 FPU, MMX technology, XMM, and MXCSR registers from the 512-byte memory image specified in
the source operand. This data should have been written to memory previously using the FXSAVE instruction, and
in the same format as required by the operating modes. The first byte of the data should be located on a 16-byte
boundary. There are three distinct layouts of the FXSAVE state map: one for legacy and compatibility mode, a
second format for 64-bit mode FXSAVE/FXRSTOR with REX.W=0, and the third format is for 64-bit mode with
FXSAVE64/FXRSTOR64. Table 3-43 shows the layout of the legacy/compatibility mode state information in memory
and describes the fields in the memory image for the FXRSTOR and FXSAVE instructions. Table 3-46 shows the
layout of the 64-bit mode state information when REX.W is set (FXSAVE64/FXRSTOR64). Table 3-47 shows the
layout of the 64-bit mode state information when REX.W is clear (FXSAVE/FXRSTOR).
The state image referenced with an FXRSTOR instruction must have been saved using an FXSAVE instruction or be
in the same format as required by Table 3-43, Table 3-46, or Table 3-47. Referencing a state image saved with an
FSAVE, FNSAVE instruction or incompatible field layout will result in an incorrect state restoration.
The FXRSTOR instruction does not flush pending x87 FPU exceptions. To check and raise exceptions when loading
x87 FPU state information with the FXRSTOR instruction, use an FWAIT instruction after the FXRSTOR instruction.
If the OSFXSR bit in control register CR4 is not set, the FXRSTOR instruction may not restore the states of the XMM
and MXCSR registers. This behavior is implementation dependent.
If the MXCSR state contains an unmasked exception with a corresponding status flag also set, loading the register
with the FXRSTOR instruction will not result in a SIMD floating-point error condition being generated. Only the next
occurrence of this unmasked exception will result in the exception being generated.
Bits 16 through 32 of the MXCSR register are defined as reserved and should be set to 0. Attempting to write a 1 in
any of these bits from the saved state image will result in a general protection exception (#GP) being generated.
Bytes 464:511 of an FXSAVE image are available for software use. FXRSTOR ignores the content of bytes 464:511
in an FXSAVE state image.

Operation
IF 64-Bit Mode
    THEN
        (x87 FPU, MMX, XMM15-XMM0, MXCSR) Load(SRC);
    ELSE
         (x87 FPU, MMX, XMM7-XMM0, MXCSR) <- Load(SRC);
FI;

x87 FPU and SIMD Floating-Point Exceptions
None.





Protected Mode Exceptions
#GP(0)                 For an illegal memory operand effective address in the CS, DS, ES, FS or GS segments.
                       If a memory operand is not aligned on a 16-byte boundary, regardless of segment. (See align-
                       ment check exception [#AC] below.)
                       For an attempt to set reserved bits in MXCSR.
#SS(0)                 For an illegal address in the SS segment.
#PF(fault-code)        For a page fault.
#NM                    If CR0.TS[bit 3] = 1.
                       If CR0.EM[bit 2] = 1.
#UD                    If CPUID.01H:EDX.FXSR[bit 24] = 0.
                       If instruction is preceded by a LOCK prefix.
#AC                    If this exception is disabled a general protection exception (#GP) is signaled if the memory
                       operand is not aligned on a 16-byte boundary, as described above. If the alignment check
                       exception (#AC) is enabled (and the CPL is 3), signaling of #AC is not guaranteed and may
                       vary with implementation, as follows. In all implementations where #AC is not signaled, a
                       general protection exception is signaled in its place. In addition, the width of the alignment
                       check may also vary with implementation. For instance, for a given implementation, an align-
                       ment check exception might be signaled for a 2-byte misalignment, whereas a general protec-
                       tion exception might be signaled for all other misalignments (4-, 8-, or 16-byte
                       misalignments).
#UD                    If the LOCK prefix is used.

Real-Address Mode Exceptions
#GP                    If a memory operand is not aligned on a 16-byte boundary, regardless of segment.
                       If any part of the operand lies outside the effective address space from 0 to FFFFH.
                       For an attempt to set reserved bits in MXCSR.
#NM                    If CR0.TS[bit 3] = 1.
                       If CR0.EM[bit 2] = 1.
#UD                    If CPUID.01H:EDX.FXSR[bit 24] = 0.
                       If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
Same exceptions as in real address mode.
#PF(fault-code)        For a page fault.
#AC                    For unaligned memory reference.
#UD                    If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.





64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
                    If memory operand is not aligned on a 16-byte boundary, regardless of segment.
                    For an attempt to set reserved bits in MXCSR.
#PF(fault-code)     For a page fault.
#NM                 If CR0.TS[bit 3] = 1.
                    If CR0.EM[bit 2] = 1.
#UD                 If CPUID.01H:EDX.FXSR[bit 24] = 0.
                    If instruction is preceded by a LOCK prefix.
#AC                 If this exception is disabled a general protection exception (#GP) is signaled if the memory
                    operand is not aligned on a 16-byte boundary, as described above. If the alignment check
                    exception (#AC) is enabled (and the CPL is 3), signaling of #AC is not guaranteed and may
                    vary with implementation, as follows. In all implementations where #AC is not signaled, a
                    general protection exception is signaled in its place. In addition, the width of the alignment
                    check may also vary with implementation. For instance, for a given implementation, an align-
                    ment check exception might be signaled for a 2-byte misalignment, whereas a general protec-
                    tion exception might be signaled for all other misalignments (4-, 8-, or 16-byte
                    misalignments).




');
INSERT INTO "instructions" VALUES('x86_64','FXSAVE','FXSAVE-Save x87 FPU, MMX Technology, and SSE State
 Opcode/                                                Op/      64-Bit       Compat/ Description
 Instruction                                            En       Mode         Leg Mode
 0F AE /0                                               M        Valid        Valid          Save the x87 FPU, MMX, XMM, and MXCSR
 FXSAVE m512byte                                                                             register state to m512byte.

 REX.W+ 0F AE /0                                        M        Valid        N.E.           Save the x87 FPU, MMX, XMM, and MXCSR
 FXSAVE64 m512byte                                                                           register state to m512byte.



                                                Instruction Operand Encoding
   Op/En                Operand 1                    Operand 2                           Operand 3                  Operand 4
     M             ModRM:r/m (w)                        NA                                  NA                          NA

Description
Saves the current state of the x87 FPU, MMX technology, XMM, and MXCSR registers to a 512-byte memory loca-
tion specified in the destination operand. The content layout of the 512 byte region depends on whether the
processor is operating in non-64-bit operating modes or 64-bit sub-mode of IA-32e mode.
Bytes 464:511 are available to software use. The processor does not write to bytes 464:511 of an FXSAVE area.
The operation of FXSAVE in non-64-bit modes is described first.

Non-64-Bit Mode Operation
Table 3-43 shows the layout of the state information in memory when the processor is operating in legacy modes.

                               Table 3-43. Non-64-bit-Mode Layout of FXSAVE and FXRSTOR
                                                     Memory Region
   15      14      13     12        11 10        9      8          7      6          5         4     3         2    1    0
     Rsvd           FCS                 FIP[31:0]                   FOP         Rsvd        FTW          FSW         FCW           0
         MXCSR_MASK                     MXCSR                       Rsrvd                FDS                FDP[31:0]              16
                 Reserved                                                            ST0/MM0                                       32
                 Reserved                                                            ST1/MM1                                       48
                 Reserved                                                            ST2/MM2                                       64
                 Reserved                                                            ST3/MM3                                       80
                 Reserved                                                            ST4/MM4                                       96
                 Reserved                                                            ST5/MM5                                      112
                 Reserved                                                            ST6/MM6                                      128
                 Reserved                                                            ST7/MM7                                      144
                                                             XMM0                                                                 160
                                                             XMM1                                                                 176
                                                             XMM2                                                                 192
                                                             XMM3                                                                 208
                                                             XMM4                                                                 224
                                                             XMM5                                                                 240
                                                             XMM6                                                                 256
                                                             XMM7                                                                 272
                                                            Reserved                                                              288



                          Table 3-43. Non-64-bit-Mode Layout of FXSAVE and FXRSTOR
                                           Memory Region (Contd.)
   15    14     13   12       11 10           9     8          7    6       5        4           3   2         1    0
                                                        Reserved                                                             304
                                                        Reserved                                                             320
                                                        Reserved                                                             336
                                                        Reserved                                                             352
                                                        Reserved                                                             368
                                                        Reserved                                                             384
                                                        Reserved                                                             400
                                                        Reserved                                                             416
                                                        Reserved                                                             432
                                                        Reserved                                                             448
                                                        Available                                                            464
                                                        Available                                                            480
                                                        Available                                                            496

The destination operand contains the first byte of the memory image, and it must be aligned on a 16-byte
boundary. A misaligned destination operand will result in a general-protection (#GP) exception being generated (or
in some cases, an alignment check exception [#AC]).
The FXSAVE instruction is used when an operating system needs to perform a context switch or when an exception
handler needs to save and examine the current state of the x87 FPU, MMX technology, and/or XMM and MXCSR
registers.
The fields in Table 3-43 are defined in Table 3-44.


                                              Table 3-44. Field Definitions
 Field                     Definition
 FCW                       x87 FPU Control Word (16 bits). See Figure 8-6 in the Intel 64 and IA-32 Architectures Software
                           Developer''s Manual, Volume 1, for the layout of the x87 FPU control word.
 FSW                       x87 FPU Status Word (16 bits). See Figure 8-4 in the Intel 64 and IA-32 Architectures Software
                           Developer''s Manual, Volume 1, for the layout of the x87 FPU status word.
 Abridged FTW              x87 FPU Tag Word (8 bits). The tag information saved here is abridged, as described in the following
                           paragraphs.
 FOP                       x87 FPU Opcode (16 bits). The lower 11 bits of this field contain the opcode, upper 5 bits are reserved.
                           See Figure 8-8 in the Intel 64 and IA-32 Architectures Software Developer''s Manual, Volume 1, for
                           the layout of the x87 FPU opcode field.
 FIP                       x87 FPU Instruction Pointer Offset (64 bits). The contents of this field differ depending on the current
                           addressing mode (32-bit, 16-bit, or 64-bit) of the processor when the FXSAVE instruction was
                           executed:
                           32-bit mode - 32-bit IP offset.
                           16-bit mode - low 16 bits are IP offset; high 16 bits are reserved.
                           64-bit mode with REX.W - 64-bit IP offset.
                           64-bit mode without REX.W - 32-bit IP offset.
                           See "x87 FPU Instruction and Operand (Data) Pointers" in Chapter 8 of the Intel 64 and IA-32
                           Architectures Software Developer''s Manual, Volume 1, for a description of the x87 FPU instruction
                           pointer.





                                            Table 3-44. Field Definitions (Contd.)
    Field                     Definition
    FCS                       x87 FPU Instruction Pointer Selector (16 bits). If CPUID.(EAX=07H,ECX=0H):EBX[bit 13] = 1, the
                              processor deprecates FCS and FDS, and this field is saved as 0000H.
    FDP                       x87 FPU Instruction Operand (Data) Pointer Offset (64 bits). The contents of this field differ
                              depending on the current addressing mode (32-bit, 16-bit, or 64-bit) of the processor when the
                              FXSAVE instruction was executed:
                              32-bit mode - 32-bit DP offset.
                              16-bit mode - low 16 bits are DP offset; high 16 bits are reserved.
                              64-bit mode with REX.W - 64-bit DP offset.
                              64-bit mode without REX.W - 32-bit DP offset.
                              See "x87 FPU Instruction and Operand (Data) Pointers" in Chapter 8 of the Intel 64 and IA-32
                              Architectures Software Developer''s Manual, Volume 1, for a description of the x87 FPU operand
                              pointer.
    FDS                       x87 FPU Instruction Operand (Data) Pointer Selector (16 bits). If CPUID.(EAX=07H,ECX=0H):EBX[bit
                              13] = 1, the processor deprecates FCS and FDS, and this field is saved as 0000H.
    MXCSR                     MXCSR Register State (32 bits). See Figure 10-3 in the Intel 64 and IA-32 Architectures Software
                              Developer''s Manual, Volume 1, for the layout of the MXCSR register. If the OSFXSR bit in control
                              register CR4 is not set, the FXSAVE instruction may not save this register. This behavior is
                              implementation dependent.
    MXCSR_                    MXCSR_MASK (32 bits). This mask can be used to adjust values written to the MXCSR register,
    MASK                      ensuring that reserved bits are set to 0. Set the mask bits and flags in MXCSR to the mode of
                              operation desired for SSE and SSE2 SIMD floating-point instructions. See "Guidelines for Writing to the
                              MXCSR Register" in Chapter 11 of the Intel 64 and IA-32 Architectures Software Developer''s Manual,
                              Volume 1, for instructions for how to determine and use the MXCSR_MASK value.
    ST0/MM0 through           x87 FPU or MMX technology registers. These 80-bit fields contain the x87 FPU data registers or the
    ST7/MM7                   MMX technology registers, depending on the state of the processor prior to the execution of the
                              FXSAVE instruction. If the processor had been executing x87 FPU instruction prior to the FXSAVE
                              instruction, the x87 FPU data registers are saved; if it had been executing MMX instructions (or SSE or
                              SSE2 instructions that operated on the MMX technology registers), the MMX technology registers are
                              saved. When the MMX technology registers are saved, the high 16 bits of the field are reserved.
    XMM0 through XMM7         XMM registers (128 bits per field). If the OSFXSR bit in control register CR4 is not set, the FXSAVE
                              instruction may not save these registers. This behavior is implementation dependent.

The FXSAVE instruction saves an abridged version of the x87 FPU tag word in the FTW field (unlike the FSAVE
instruction, which saves the complete tag word). The tag information is saved in physical register order (R0
through R7), rather than in top-of-stack (TOS) order. With the FXSAVE instruction, however, only a single bit (1 for
valid or 0 for empty) is saved for each tag. For example, assume that the tag word is currently set as follows:

     R7 R6 R5 R4 R3 R2 R1 R0
     11 xx xx xx 11 11 11 11
Here, 11B indicates empty stack elements and "xx" indicates valid (00B), zero (01B), or special (10B).
For this example, the FXSAVE instruction saves only the following 8 bits of information:

     R7 R6 R5 R4 R3 R2 R1 R0
     0 1 1 1 0 0 0 0
Here, a 1 is saved for any valid, zero, or special tag, and a 0 is saved for any empty tag.
The operation of the FXSAVE instruction differs from that of the FSAVE instruction, the as follows:
.     FXSAVE instruction does not check for pending unmasked floating-point exceptions. (The FXSAVE operation in
      this regard is similar to the operation of the FNSAVE instruction).
.     After the FXSAVE instruction has saved the state of the x87 FPU, MMX technology, XMM, and MXCSR registers,
      the processor retains the contents of the registers. Because of this behavior, the FXSAVE instruction cannot be



      used by an application program to pass a "clean" x87 FPU state to a procedure, since it retains the current
      state. To clean the x87 FPU state, an application must explicitly execute an FINIT instruction after an FXSAVE
      instruction to reinitialize the x87 FPU state.
.     The format of the memory image saved with the FXSAVE instruction is the same regardless of the current
      addressing mode (32-bit or 16-bit) and operating mode (protected, real address, or system management).
      This behavior differs from the FSAVE instructions, where the memory image format is different depending on
      the addressing mode and operating mode. Because of the different image formats, the memory image saved
      with the FXSAVE instruction cannot be restored correctly with the FRSTOR instruction, and likewise the state
      saved with the FSAVE instruction cannot be restored correctly with the FXRSTOR instruction.
The FSAVE format for FTW can be recreated from the FTW valid bits and the stored 80-bit FP data (assuming the
stored data was not the contents of MMX technology registers) using Table 3-45.

                                         Table 3-45. Recreating FSAVE Format
       Exponent             Exponent       Fraction           J and M         FTW valid bit
         all 1''s              all 0''s       all 0''s             bits                                      x87 FTW
            0                   0             0                 0x                  1           Special              10
            0                   0             0                 1x                  1           Valid                00
            0                   0             1                 00                  1           Special              10
            0                   0             1                 10                  1           Valid                00
            0                   1             0                 0x                  1           Special              10
            0                   1             0                 1x                  1           Special              10
            0                   1             1                 00                  1           Zero                 01
            0                   1             1                 10                  1           Special              10
            1                   0             0                 1x                  1           Special              10
            1                   0             0                 1x                  1           Special              10
            1                   0             1                 00                  1           Special              10
            1                   0             1                 10                  1           Special              10
    For all legal combinations above.                                               0           Empty                11

The J-bit is defined to be the 1-bit binary integer to the left of the decimal place in the significand. The M-bit is
defined to be the most significant bit of the fractional portion of the significand (i.e., the bit immediately to the right
of the decimal place).
When the M-bit is the most significant bit of the fractional portion of the significand, it must be 0 if the fraction is all
0''s.

IA-32e Mode Operation
In compatibility sub-mode of IA-32e mode, legacy SSE registers, XMM0 through XMM7, are saved according to the
legacy FXSAVE map. In 64-bit mode, all of the SSE registers, XMM0 through XMM15, are saved. Additionally, there
are two different layouts of the FXSAVE map in 64-bit mode, corresponding to FXSAVE64 (which requires
REX.W=1) and FXSAVE (REX.W=0). In the FXSAVE64 map (Table 3-46), the FPU IP and FPU DP pointers are 64-bit
wide. In the FXSAVE map for 64-bit mode (Table 3-47), the FPU IP and FPU DP pointers are 32-bits.





                                      Table 3-46. Layout of the 64-bit-mode FXSAVE64 Map
                                                      (requires REX.W = 1)
   15    14      13   12         11    10     9      8      7    6      5        4    3      2       1   0
                           FIP                               FOP     Reserved   FTW   FSW            FCW          0
        MXCSR_MASK                      MXCSR                                   FDP                               16
                Reserved                                                 ST0/MM0                                  32
                Reserved                                                 ST1/MM1                                  48
                Reserved                                                 ST2/MM2                                  64
                Reserved                                                 ST3/MM3                                  80
                Reserved                                                 ST4/MM4                                  96
                Reserved                                                 ST5/MM5                                 112
                Reserved                                                 ST6/MM6                                 128
                Reserved                                                 ST7/MM7                                 144
                                                          XMM0                                                   160
                                                          XMM1                                                   176
                                                          XMM2                                                   192
                                                          XMM3                                                   208
                                                          XMM4                                                   224
                                                          XMM5                                                   240
                                                          XMM6                                                   256
                                                          XMM7                                                   272
                                                          XMM8                                                   288
                                                          XMM9                                                   304
                                                         XMM10                                                   320
                                                         XMM11                                                   336
                                                         XMM12                                                   352
                                                         XMM13                                                   368
                                                         XMM14                                                   384
                                                         XMM15                                                   400
                                                         Reserved                                                416
                                                         Reserved                                                432
                                                         Reserved                                                448
                                                         Available                                               464
                                                         Available                                               480
                                                         Available                                               496





                           Table 3-47. Layout of the 64-bit-mode FXSAVE Map (REX.W = 0)
   15    14      13   12     11   10       9   8       7       6      5           4         3   2        1    0
   Reserved       FCS              FIP[31:0]            FOP        Reserved      FTW        FSW          FCW            0
        MXCSR_MASK                 MXCSR              Reserved             FDS                  FDP[31:0]               16
                Reserved                                                  ST0/MM0                                       32
                Reserved                                                  ST1/MM1                                       48
                Reserved                                                  ST2/MM2                                       64
                Reserved                                                  ST3/MM3                                       80
                Reserved                                                  ST4/MM4                                       96
                Reserved                                                  ST5/MM5                                      112
                Reserved                                                  ST6/MM6                                      128
                Reserved                                                  ST7/MM7                                      144
                                                    XMM0                                                               160
                                                    XMM1                                                               176
                                                    XMM2                                                               192
                                                    XMM3                                                               208
                                                    XMM4                                                               224
                                                    XMM5                                                               240
                                                    XMM6                                                               256
                                                    XMM7                                                               272
                                                    XMM8                                                               288
                                                    XMM9                                                               304
                                                   XMM10                                                               320
                                                   XMM11                                                               336
                                                   XMM12                                                               352
                                                   XMM13                                                               368
                                                   XMM14                                                               384
                                                   XMM15                                                               400
                                                   Reserved                                                            416
                                                   Reserved                                                            432
                                                   Reserved                                                            448
                                                   Available                                                           464
                                                   Available                                                           480
                                                   Available                                                           496





Operation
IF 64-Bit Mode
    THEN
         IF REX.W = 1
              THEN
                  DEST <- Save64BitPromotedFxsave(x87 FPU, MMX, XMM15-XMM0,
                  MXCSR);
              ELSE
                  DEST <- Save64BitDefaultFxsave(x87 FPU, MMX, XMM15-XMM0, MXCSR);
         FI;
    ELSE
         DEST <- SaveLegacyFxsave(x87 FPU, MMX, XMM7-XMM0, MXCSR);
FI;

Protected Mode Exceptions
#GP(0)                 For an illegal memory operand effective address in the CS, DS, ES, FS or GS segments.
                       If a memory operand is not aligned on a 16-byte boundary, regardless of segment. (See the
                       description of the alignment check exception [#AC] below.)
#SS(0)                 For an illegal address in the SS segment.
#PF(fault-code)        For a page fault.
#NM                    If CR0.TS[bit 3] = 1.
                       If CR0.EM[bit 2] = 1.
#UD                    If CPUID.01H:EDX.FXSR[bit 24] = 0.
#UD                    If the LOCK prefix is used.
#AC                    If this exception is disabled a general protection exception (#GP) is signaled if the memory
                       operand is not aligned on a 16-byte boundary, as described above. If the alignment check
                       exception (#AC) is enabled (and the CPL is 3), signaling of #AC is not guaranteed and may
                       vary with implementation, as follows. In all implementations where #AC is not signaled, a
                       general protection exception is signaled in its place. In addition, the width of the alignment
                       check may also vary with implementation. For instance, for a given implementation, an align-
                       ment check exception might be signaled for a 2-byte misalignment, whereas a general protec-
                       tion exception might be signaled for all other misalignments (4-, 8-, or 16-byte
                       misalignments).

Real-Address Mode Exceptions
#GP                    If a memory operand is not aligned on a 16-byte boundary, regardless of segment.
                       If any part of the operand lies outside the effective address space from 0 to FFFFH.
#NM                    If CR0.TS[bit 3] = 1.
                       If CR0.EM[bit 2] = 1.
#UD                    If CPUID.01H:EDX.FXSR[bit 24] = 0.
                       If the LOCK prefix is used.

Virtual-8086 Mode Exceptions
Same exceptions as in real address mode.
#PF(fault-code)        For a page fault.
#AC                    For unaligned memory reference.
#UD                    If the LOCK prefix is used.

Compatibility Mode Exceptions
Same exceptions as in protected mode.




64-Bit Mode Exceptions
#SS(0)              If a memory address referencing the SS segment is in a non-canonical form.
#GP(0)              If the memory address is in a non-canonical form.
                    If memory operand is not aligned on a 16-byte boundary, regardless of segment.
#PF(fault-code)     For a page fault.
#NM                 If CR0.TS[bit 3] = 1.
                    If CR0.EM[bit 2] = 1.
#UD                 If CPUID.01H:EDX.FXSR[bit 24] = 0.
                    If the LOCK prefix is used.
#AC                 If this exception is disabled a general protection exception (#GP) is signaled if the memory
                    operand is not aligned on a 16-byte boundary, as described above. If the alignment check
                    exception (#AC) is enabled (and the CPL is 3), signaling of #AC is not guaranteed and may
                    vary with implementation, as follows. In all implementations where #AC is not signaled, a
                    general protection exception is signaled in its place. In addition, the width of the alignment
                    check may also vary with implementation. For instance, for a given implementation, an align-
                    ment check exception might be signaled for a 2-byte misalignment, whereas a general protec-
                    tion exception might be signaled for all other misalignments (4-, 8-, or 16-byte
                    misalignments).

Implementation Note
The order in which the processor signals general-protection (#GP) and page-fault (#PF) exceptions when they both
occur on an instruction boundary is given in Table 5-2 in the Intel 64 and IA-32 Architectures Software Devel-
oper''s Manual, Volume 3B. This order vary for FXSAVE for different processor implementations.




');
INSERT INTO "instructions" VALUES('x86_64','FXTRACT','FXTRACT-Extract Exponent and Significand
 Opcode/                                       64-Bit   Compat/    Description
 Instruction                                   Mode     Leg Mode
 D9 F4                                         Valid    Valid      Separate value in ST(0) into exponent and significand, store
 FXTRACT                                                           exponent in ST(0), and push the significand onto the register
                                                                   stack.



Description
Separates the source value in the ST(0) register into its exponent and significand, stores the exponent in ST(0),
and pushes the significand onto the register stack. Following this operation, the new top-of-stack register ST(0)
contains the value of the original significand expressed as a floating-point value. The sign and significand of this
value are the same as those found in the source operand, and the exponent is 3FFFH (biased value for a true expo-
nent of zero). The ST(1) register contains the value of the original operand''s true (unbiased) exponent expressed
as a floating-point value. (The operation performed by this instruction is a superset of the IEEE-recommended
logb(x) function.)
This instruction and the F2XM1 instruction are useful for performing power and range scaling operations. The
FXTRACT instruction is also useful for converting numbers in double extended-precision floating-point format to
decimal representations (e.g., for printing or displaying).
If the floating-point zero-divide exception (#Z) is masked and the source operand is zero, an exponent value of -
inf is stored in register ST(1) and 0 with the sign of the source operand is stored in register ST(0).
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
TEMP <- Significand(ST(0));
ST(0) <- Exponent(ST(0));
TOP<- TOP - 1;
ST(0) <- TEMP;

FPU Flags Affected
C1                      Set to 0 if stack underflow occurred; set to 1 if stack overflow occurred.
C0, C2, C3              Undefined.

Floating-Point Exceptions
#IS                     Stack underflow or overflow occurred.
#IA                     Source operand is an SNaN value or unsupported format.
#Z                      ST(0) operand is +-0.
#D                      Source operand is a denormal value.

Protected Mode Exceptions
#NM                     CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                     If there is a pending x87 FPU exception.
#UD                     If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.





Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FYL2X','FYL2X-Compute y * log2x
Opcode                   Instruction             64-Bit           Compat/           Description
                                                 Mode             Leg Mode
D9 F1                    FYL2X                   Valid            Valid             Replace ST(1) with (ST(1) * log2ST(0)) and pop the
                                                                                    register stack.



Description
Computes (ST(1) * log2 (ST(0))), stores the result in resister ST(1), and pops the FPU register stack. The source
operand in ST(0) must be a non-zero positive number.
The following table shows the results obtained when taking the log of various classes of numbers, assuming that
neither overflow nor underflow occurs.
                                                    Table 3-48. FYL2X Results
                                                                            ST(0)
                                 -inf         -F            +-0              +0<+F<+1            +1        +F>+1           +inf           NaN

                   -inf             *          *            +inf                 +inf                   *        -inf           -inf           NaN

     ST(1)         -F             *          *            **                 +F               -0           -F           -inf           NaN

                   -0             *          *                *              +0               -0           -0             *          NaN

                   +0             *          *                *              -0               +0           +0             *          NaN

                   +F             *          *            **                 -F               +0           +F           +inf           NaN

                   +inf             *          *            -   inf              -inf                   *        +inf           +inf           NaN

                   NaN           NaN       NaN            NaN                NaN              NaN          NaN          NaN          NaN

 NOTES:
 F Means finite floating-point value.
 * Indicates floating-point invalid-operation (#IA) exception.
 ** Indicates floating-point zero-divide (#Z) exception.


If the divide-by-zero exception is masked and register ST(0) contains +-0, the instruction returns inf with a sign that
is the opposite of the sign of the source operand in register ST(1).
The FYL2X instruction is designed with a built-in multiplication to optimize the calculation of logarithms with an
arbitrary positive base (b):

     logbx <- (log2b)-1 * log2x
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
ST(1) <- ST(1) * log2ST(0);
PopRegisterStack;

FPU Flags Affected
C1                        Set to 0 if stack underflow occurred.
                          Set if result was rounded up; cleared otherwise.
C0, C2, C3                Undefined.





Floating-Point Exceptions
#IS                 Stack underflow occurred.
#IA                 Either operand is an SNaN or unsupported format.
                    Source operand in register ST(0) is a negative finite value
                    (not -0).
#Z                  Source operand in register ST(0) is +-0.
#D                  Source operand is a denormal value.
#U                  Result is too small for destination format.
#O                  Result is too large for destination format.
#P                  Value cannot be represented exactly in destination format.

Protected Mode Exceptions
#NM                 CR0.EM[bit 2] or CR0.TS[bit 3] = 1.
#MF                 If there is a pending x87 FPU exception.
#UD                 If the LOCK prefix is used.

Real-Address Mode Exceptions
Same exceptions as in protected mode.

Virtual-8086 Mode Exceptions
Same exceptions as in protected mode.

Compatibility Mode Exceptions
Same exceptions as in protected mode.

64-Bit Mode Exceptions
Same exceptions as in protected mode.




');
INSERT INTO "instructions" VALUES('x86_64','FYL2XP1','FYL2XP1-Compute y * log2(x +1)
 Opcode                   Instruction              64-Bit        Compat/             Description
                                                   Mode          Leg Mode
 D9 F9                    FYL2XP1                  Valid         Valid               Replace ST(1) with ST(1) * log2(ST(0) + 1.0) and pop the
                                                                                     register stack.



Description
Computes (ST(1) * log2(ST(0) + 1.0)), stores the result in register ST(1), and pops the FPU register stack. The
source operand in ST(0) must be in the range:
     - ( 1 - 2 / 2 ) )to ( 1 - 2 / 2 )
The source operand in ST(1) can range from -inf to +inf. If the ST(0) operand is outside of its acceptable range, the
result is undefined and software should not rely on an exception being generated. Under some circumstances
exceptions may be generated when ST(0) is out of range, but this behavior is implementation specific and not
guaranteed.
The following table shows the results obtained when taking the log epsilon of various classes of numbers, assuming
that underflow does not occur.
                                                     Table 3-49. FYL2XP1 Results
                                                                             ST(0)
                                     -(1 - ( 2 / 2 )) to -0          -0              +0             +0 to +(1 - ( 2 / 2 ))          NaN
                     -inf                       +inf                         *            *                     -inf                      NaN
     ST(1)           -F                       +F                     +0               -0                     -F                     NaN
                     -0                       +0                     +0               -0                     -0                     NaN
                     +0                       -0                    -0               +0                      +0                     NaN
                     +F                       -F                    -0               +0                      +F                     NaN
                     +inf                       -inf                         *            *                     +inf                      NaN
                    NaN                      NaN                    NaN              NaN                    NaN                     NaN
 NOTES:
 F Means finite floating-point value.
 * Indicates floating-point invalid-operation (#IA) exception.

This instruction provides optimal accuracy for values of epsilon [the value in register ST(0)] that are close to 0. For
small epsilon (e) values, more significant digits can be retained by using the FYL2XP1 instruction than by using
(e+1) as an argument to the FYL2X instruction. The (e+1) expression is commonly found in compound interest and
annuity calculations. The result can be simply converted into a value in another logarithm base by including a scale
factor in the ST(1) source operand. The following equation is used to calculate the scale factor for a particular loga-
rithm base, where n is the logarithm base desired for the result of the FYL2XP1 instruction:

   scale factor <- logn 2
This instruction''s operation is the same in non-64-bit modes and 64-bit mode.

Operation
ST(1) <- ST(1) * log2(ST(0) + 1.0);
PopRegisterStack;





FPU Flags Affected
C1                   Set to 0 if stack underflow occurred.
                     Set if result was rounded up; cleared otherwise.
C0, C2, C3       