---
order: 2
title: Internationalization
---



Internationalization refers to the default configurations built into the component library to adapt to the major regional languages in the world, including text translation and date formatting, as well as the default configurations of some components, such as the first day of the week, whether the month is before the year, and so on.

The internationalization of `Ant Design Blazor` is provided by the built-in `LocaleProvider` static class, which sets the regional language package of the thread for different UI threads, which is embedded in the assembly.

In addition, we provide app-level localization components, see [Localization](docs/localization).


## Default language

By default, components automatically get the locale language from the environment as the default language when the program calls `Services.AddAntDesign()`. Therefore, different operating environments have different Settings:

- Blazor WebAssembly: The default language is retrieved from the visitor's browser language Settings. Note that it is not obtained from the system operating system or server.
- Blazor Static/ Interactive server rendering: The default language is set according to the language of the operating system environment in which the service is running, rather than the user's request.
- Blazor WebApp Auto mode: It is obtained from the server system in the static phase and from the browser in the WebAssembly phase, so care needs to be taken to keep the language consistent and avoid the problem of switching.
- Blazor Hybrid: Gets the default language from the client.

If you need to manually set the default language, you can configure it during initialization or modify it during running. For details, see the following solution.


## Global configuration

Set at the entry point where the application runs

```csharp

service.AddAntDesign();

// Set current language
LocaleProvider.DefaultLanguage = "en-US";
```

## Change locale at run time

```csharp

@code {

    private void ChangeLocale()
    {
        LocaleProvider.SetLocale("en-US");
        StateHasChanged();
    }
}
```

When the language you set does not have a built-in language pack, we will fallback to the parent culture/locale, then `LocaleProvider.DefaultLanguage`, then "en-US". (To learn more about the term of parent culture, see [Globalization and localization terms](https://docs.microsoft.com/en-us/aspnet/core/fundamentals/localization?view=aspnetcore-5.0#globalization-and-localization-terms))

You can set default language with `LocaleProvider.DefaultLanguage = "languageName"`.

You can also add a language pack by using the `LocaleProvider.SetLocale` method.

```csharp
// Set default language
LocaleProvider.DefaultLanguage = "en-US";

// Create a `Locale` object
var zhHK = new Locale { ... };
// Set an additional language
LocaleProvider.SetLocale("zh-HK", zhHK);
```

Note: `en-US` is the package name, follow below.

Built-in languages:

| Language                 | Filename |
| ------------------------ | -------- |
| Arabic                   | ar-EG    |
| Azerbaijani              | az-AZ    |
| Bulgarian                | bg-BG    |
| Bangla (Bangladesh)      | bn-BD    |
| Belarusian               | by-BY    |
| Catalan                  | ca-ES    |
| Czech                    | cs-CZ    |
| Danish                   | da-DK    |
| German                   | de-DE    |
| Greek                    | el-GR    |
| English (United Kingdom) | en-GB    |
| English                  | en-US    |
| Spanish                  | es-ES    |
| Estonian                 | et-EE    |
| Persian                  | fa-IR    |
| Finnish                  | fi-FI    |
| French (Belgium)         | fr-BE    |
| French (Canada)          | fr-CA    |
| French (France)          | fr-FR    |
| Irish (Ireland)          | ga-IE    |
| Galician (Spain)         | gl-ES    |
| Hebrew                   | he-IL    |
| Hindi                    | hi-IN    |
| Croatian                 | hr-HR    |
| Hungarian                | hu-HU    |
| Armenian                 | hy-AM    |
| Indonesian               | id-ID    |
| Italian                  | it-IT    |
| Icelandic                | is-IS    |
| Japanese                 | ja-JP    |
| Georgian                 | ka-GE    |
| Kurdish (Kurmanji)       | kmr-IQ   |
| Kannada                  | kn-IN    |
| Kazakh                   | kk-KZ    |
| Khmer                    | km-KH    |
| Korean                   | ko-KR    |
| Lithuanian               | lt-LT    |
| Latvian                  | lv-LV    |
| Macedonian               | mk-MK    |
| Malayalam (India)        | ml-IN    |
| Mongolian                | mn-MN    |
| Malay (Malaysia)         | ms-MY    |
| Norwegian                | nb-NO    |
| Nepal                    | ne-NP    |
| Dutch (Belgium)          | nl-BE    |
| Dutch                    | nl-NL    |
| Polish                   | pl-PL    |
| Portuguese (Brazil)      | pt-BR    |
| Portuguese               | pt-PT    |
| Romanian                 | ro-RO    |
| Russian                  | ru-RU    |
| Sinhalese / Sinhala      | si-LK    |
| Slovak                   | sk-SK    |
| Serbian                  | sr-RS    |
| Slovenian                | sl-SI    |
| Swedish                  | sv-SE    |
| Tamil                    | ta-IN    |
| Thai                     | th-TH    |
| Turkish                  | tr-TR    |
| Turkmen                  | tk-TK    |
| Urdu (Pakistan)          | ur-PK    |
| Ukrainian                | uk-UA    |
| Uzbek                    | uz-UZ    |
| Vietnamese               | vi-VN    |
| Chinese (Simplified)     | zh-CN    |
| Chinese (Traditional)    | zh-HK    |
| Chinese (Traditional)    | zh-TW    |

Of course, we are not proficient in all languages, if you find a translation error, please submit a PR to help us improve.