# Statische Website erstellen

Analog unterstützt die statische Webseiten-Generierung bei der Erstellung für die Bereitstellung. Dazu gehört das Vorberechnen bereitgestellter Routen in statische HTML-Dateien zusammen mit der clientseitigen Anwendung.

## Statische Website-Generierung

### Aus der Routenliste

Um Seiten vorzuberechnen, verwende die Eigenschaft `prerender`, um Routen zu konfigurieren, die zur Erstellungszeit berechnet werden. Die Routen, die vorgerendert werden sollen, können auch asynchron bereitgestellt werden.

```ts
import { defineConfig } from 'vite';
import analog from '@analogjs/platform';

// https://vitejs.dev/config/
export default defineConfig(({ mode }) => ({
  plugins: [
    analog({
      prerender: {
        routes: async () => [
          '/',
          '/about',
          '/blog',
          '/blog/posts/2023-02-01-my-first-post',
        ],
      },
    }),
  ],
}));
```

### Aus dem Content-Verzeichnis

Möglicherweise sollen alle Routen, die das Ergebnis eines gerenderten Inhaltsverzeichnisses sind, vorberechnet werden.
Wenn zum Beispiel ein Blog verwendet wird und alle Artikel als Markdown-Dateien im Verzeichnis `contents` ablegt sind.
Für solche Szenarien kann ein Objekt zur `routes`-Konfiguration hinzugefügt werden, um alles innerhalb eines Verzeichnisses darzustellen.
Beachte, dass die Verzeichnisstruktur möglicherweise nicht 1:1 in Anwendungspfaden wiedergegeben wird.
Daher muss eine `transform`-Funktion übergeben werden, die die Dateipfade auf die URLs abbildet.
Der zurückgegebene String sollte der URL-Pfad in der Anwendung sein.
Mit `transform` können auch einige Routen herausgefiltert werden, indem `false` zurückgeben wird.
Das schließt diese nicht in den Vorberechnungsprozess ein, wie z.B. Dateien, die im Frontmatter als `draft` markiert sind.
Der `contentDir`-Wert dieses Objekts kann ein glob-Muster oder ein bestimmter Pfad sein.

```ts
import { defineConfig } from 'vite';
import analog, { type PrerenderContentFile } from '@analogjs/platform';

// https://vitejs.dev/config/
export default defineConfig(({ mode }) => ({
  plugins: [
    analog({
      prerender: {
        routes: async () => [
          '/',
          '/blog',
          {
            contentDir: 'src/content/blog',
            transform: (file: PrerenderContentFile) => {
              // do not include files marked as draft in frontmatter
              if (file.attributes.draft) {
                return false;
              }
              // use the slug from frontmatter if defined, otherwise use the files basename
              const slug = file.attributes.slug || file.name;
              return `/blog/${slug}`;
            },
          },
        ],
      },
    }),
  ],
}));
```

### Nur statische Seiten

Um nur die statischen Seiten vorzuberechnen, verwenden den Flag `static: true`.

```ts
import { defineConfig } from 'vite';
import analog from '@analogjs/platform';

// https://vitejs.dev/config/
export default defineConfig(({ mode }) => ({
  plugins: [
    analog({
      static: true,
      prerender: {
        routes: async () => [
          '/',
          '/about',
          '/blog',
          '/blog/posts/2023-02-01-my-first-post',
        ],
      },
    }),
  ],
}));
```

Die statischen Seiten können aus dem Verzeichnis `dist/analog/public` bereitgestellt werden.

### Erstellung der Sitemap

Analog unterstützt auch die automatische Erstellung von Sitemaps. Analog erzeugt eine Sitemap im Verzeichnis `dist/analog/public`
Verzeichnis, wenn ein Build ausgeführt wird und eine Sitemap-Konfiguration angegeben ist.

```ts
import { defineConfig } from 'vite';
import analog from '@analogjs/platform';

// https://vitejs.dev/config/
export default defineConfig(({ mode }) => ({
  plugins: [
    analog({
      prerender: {
        routes: async () => ['/', '/blog'],
        sitemap: {
          host: 'https://analogjs.org/',
        },
      },
    }),
  ],
}));
```

Solange Routen angegeben werden, erzeugt Analog eine Datei `sitemap.xml`, die eine
Abbildung der Eigenschaften `<loc>` und `<lastmod>` der Seiten enthält.

```xml
<?xml version="1.0" encoding="UTF-8"?>
<urlset...>
    <!--This file was automatically generated by Analog.-->
    <url>
        <loc>https://analogjs.org/</loc>
        <lastmod>2023-07-01</lastmod>
    </url>
    <url>
        <loc>https://analogjs.org/blog</loc>
        <lastmod>2023-07-01</lastmod>
    </url>
</urlset...>
```

### Post-Rendering-Hooks

Analog unterstützt die Post-Rendering-Hooks während des Vorberechnungs-Prozesses. Der Anwendungsfall für Post-Rendering-Hooks kann das Inlining kritischer CSS, das Hinzufügen/Entfernen von Skripten in HTML-Dateien usw. sein.

Das folgende Codebeispiel zeigt, wie `postRenderingHooks` im Code verwendet werden können:

```ts
import analog from '@analogjs/platform';
import { defineConfig } from 'vite';
import { PrerenderRoute } from 'nitropack';

// https://vitejs.dev/config/
export default defineConfig(() => {
  return {
    publicDir: 'src/public',
    build: {
      target: ['es2020'],
    },
    plugins: [
      analog({
        static: true,
        prerender: {
          routes: async () => [],
          postRenderingHooks: [
            async (route: PrerenderRoute) => console.log(route),
          ],
        },
      }),
    ],
  };
});
```

Die `PrerenderRoute` gibt Informationen über `Route`, `Inhalt`, `Daten` und `Dateiname`, die nützlich sein können, um während der Vorberechnungs-Phase Änderungen an den Inhalt vorzunehmen.

Nachfolgend ein kleines Beispiel, in dem ein Skript angehängt wird, um Google Analytics während des Vorberechnungs-Prozesses mit `postRenderingHooks` einzubinden:

```ts
/// <reference types="vitest" />

import analog from '@analogjs/platform';
import { defineConfig, splitVendorChunkPlugin } from 'vite';
import { nxViteTsPaths } from '@nx/vite/plugins/nx-tsconfig-paths.plugin';
import { PrerenderRoute } from 'nitropack';

// https://vitejs.dev/config/
export default defineConfig(() => {
  return {
    publicDir: 'src/public',
    build: {
      target: ['es2020'],
    },
    plugins: [
      analog({
        static: true,
        prerender: {
          routes: async () => ['/', '/aboutus'],
          postRenderingHooks: [
            async (route: PrerenderRoute) => {
              const gTag = `<script>
              (function(i,s,o,g,r,a,m){i['GoogleAnalyticsObject']=r;i[r]=i[r]||function(){
                (i[r].q=i[r].q||[]).push(arguments)},i[r].l=1*new Date();a=s.createElement(o),
                m=s.getElementsByTagName(o)[0];a.async=1;a.src=g;m.parentNode.insertBefore(a,m)
                })(window,document,'script','//www.google-analytics.com/analytics.js','ga');

                ga('create', 'UA-xxxxxx-1', 'auto');
                ga('send', 'pageview');
              </script>`;
              if (route.route === '/aboutus') {
                route.contents = route.contents?.concat(gTag);
              }
            },
          ],
        },
      }),
    ],
  };
});
```
