/*
 * The Alluxio Open Foundation licenses this work under the Apache License, version 2.0
 * (the "License"). You may not use this work except in compliance with the License, which is
 * available at www.apache.org/licenses/LICENSE-2.0
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied, as more fully set forth in the License.
 *
 * See the NOTICE file distributed with this work for information regarding copyright ownership.
 */

package alluxio.worker.block;

import alluxio.ClientContext;
import alluxio.conf.Configuration;
import alluxio.conf.PropertyKey;
import alluxio.master.MasterClientContext;
import alluxio.resource.ResourcePool;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.io.Closer;
import com.google.inject.Inject;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

/**
 * Class for managing block master clients. After obtaining a client with
 * {@link ResourcePool#acquire()}, {@link ResourcePool#release(Object)} must be called when the
 * thread is done using the client.
 */
@ThreadSafe
// TODO(jiacheng): This is not removed because it is used by the PagedDoraWorker.
//  It shall be removed together with a refactor on the Dora worker refactor on register.
public class BlockMasterClientPool extends ResourcePool<BlockMasterClient> {
  private final Queue<BlockMasterClient> mClientList;
  private final MasterClientContext mMasterContext;

  /** If not specified, the client pool will create clients connecting to the primary master. **/
  @Nullable
  private final InetSocketAddress mMasterAddress;

  /**
   * A factory class for testing purpose.
   */
  @VisibleForTesting
  static class Factory {
    BlockMasterClientPool create() {
      return new BlockMasterClientPool();
    }

    BlockMasterClientPool create(@Nullable InetSocketAddress address) {
      return new BlockMasterClientPool(address);
    }
  }

  /**
   * Creates a new block master client pool.
   */
  @Inject
  public BlockMasterClientPool() {
    this(null);
  }

  /**
   * Creates a new block master client pool.
   * @param address the block master address
   */
  public BlockMasterClientPool(@Nullable InetSocketAddress address) {
    super(Configuration.getInt(PropertyKey.WORKER_BLOCK_MASTER_CLIENT_POOL_SIZE));
    mClientList = new ConcurrentLinkedQueue<>();
    mMasterContext = MasterClientContext
        .newBuilder(ClientContext.create(Configuration.global())).build();
    mMasterAddress = address;
  }

  @Override
  public void close() throws IOException {
    BlockMasterClient client;
    Closer closer = Closer.create();
    while ((client = mClientList.poll()) != null) {
      closer.register(client);
    }
    closer.close();
  }

  @Override
  public BlockMasterClient createNewResource() {
    final BlockMasterClient client;
    if (mMasterAddress != null) {
      // If an address is specified, that means all clients in this pool connect
      // to the specific master no matter it is a primary or standby
      client = new BlockMasterClient(mMasterContext, mMasterAddress);
    } else {
      client = new BlockMasterClient(mMasterContext);
    }
    mClientList.add(client);
    return client;
  }

  protected InetSocketAddress getMasterAddress() {
    return mMasterAddress;
  }

  protected MasterClientContext getMasterContext() {
    return mMasterContext;
  }

  protected Queue<BlockMasterClient> getClientList() {
    return mClientList;
  }
}
