---
title: Advanced Custom Fields integration with Algolia Search
description: Integrate Algolia Search for WordPress with Advanced Custom Fields.
layout: page.html
---

## Introduction

The [Advanced Custom Fields plugin](https://www.advancedcustomfields.com/) is a very popular plugin that allows you to create your custom fields directly from the admin section of your WordPress website.

On this page we will give some examples on how to push these custom fields to Algolia, make them searchable and have them impact the ranking of your results.

If you haven't read the [basics of extending this plugin](extend-basics.html), you should probably read that first.

To get an in depth overview of methods introduced by the Advanced Custom Fields, check out the [Official Advanced Custom Fields documentation](https://www.advancedcustomfields.com/resources/).

## Push custom fields to Algolia

In the following example, we add the `bio` field to the posts that have a post type of `speaker`.

Here we assume you have a custom post type `speaker` which is not natively available in WordPress.

We add the `bio` attribute to both `searchable_posts` and `posts` indices.

```php
<?php

add_filter( 'algolia_post_shared_attributes', 'my_post_attributes', 10, 2 );
add_filter( 'algolia_searchable_post_shared_attributes', 'my_post_attributes', 10, 2 );

/**
 * @param array   $attributes
 * @param WP_Post $post
 *
 * @return array
 */
function my_post_attributes( array $attributes, WP_Post $post ) {

	if ( 'speaker' !== $post->post_type ) {
		// We only want to add an attribute for the 'speaker' post type.
		// Here the post isn't a 'speaker', so we return the attributes unaltered.
		return $attributes;
	}

	// Get the field value with the 'get_field' method and assign it to the attributes array.
	// @see https://www.advancedcustomfields.com/resources/get_field/
	$attributes['bio'] = get_field( 'bio', $post->ID );

	// Always return the value we are filtering.
	return $attributes;
}
```

## Make custom fields searchable

In this example we make sure that the `bio` attribute we pushed earlier is taken into account in the Algolia relevancy algorithm.

Here we only alter the settings of the `posts_speaker` index because in the `searchable_posts` index not all posts have
the `bio` attribute, so basing the relevancy on that field might lead to results that are difficult to understand for the end users.

In this example, we also ask for Algolia to return a snippet of 50 chars of the `bio` attribute along with the record.
This will allow you to display a relevant snippet of the `bio` attribute to your end users.
The snippet is dynamically generated by the Algolia engine to show the most relevant part of the attribute regarding the current user's request.

```php
<?php

add_filter( 'algolia_posts_speaker_index_settings', 'my_posts_index_settings' );
// We could also have used a more general hook 'algolia_posts_index_settings',
// but in that case we would have needed to add a check on the post type
// to ensure we only adjust settings for the 'speaker' post_type.

/**
 * @param array $settings
 *
 * @return array
 */
function my_posts_index_settings( array $settings ) {

	// Make Algolia search into the 'bio' field when searching for results.
	// Using ordered instead of unordered would make words matching in the beginning of the attribute
	// make the record score higher.
	// @see https://www.algolia.com/doc/api-client/ruby/parameters#attributestoindex
	$settings['attributesToIndex'][] = 'unordered(bio)';

	// Make Algolia return a pre-computed snippet of 50 chars as part of the result set.
	// @see https://www.algolia.com/doc/api-client/ruby/parameters#attributestohighlight
	$settings['attributesToSnippet'][] = 'bio:50';

	// Always return the value we are filtering.
	return $settings;
}
```

## Integrate custom fields in your custom ranking

You can integrate your custom metrics in the ranking algorithm of Algolia.
That way, you can provide your end users with results ranked according to your business requirements.

In this example, let's say you are a company selling videos on your WordPress website.
Your website is tracking the number of views of each video in a custom attribute called `views_count`.

In this example we will show you how to add the `views_count` attribute to the custom ranking.
The idea being that if several videos are considered a good match regarding the current's user search query,
we will order the results regarding the `views_count`. Read more about the custom ranking here: https://www.algolia.com/doc/guides/relevance/ranking#custom-ranking

Note that you can add as many attributes to the custom ranking as you want. The order in which the attributes appear in the custom ranking matters as under the hood
Algolia uses a Tie Breaking algorithm. Read more about it here: https://www.algolia.com/doc/guides/relevance/ranking#ranking-formula-a-tie-breaking-algorithm

Here we assume you already made it so that the `views_count` are pushed to the `posts_video` index as explained in the '[push custom fields to Algolia](#push-custom-fields-to-algolia)' section.

```php
<?php

add_filter( 'algolia_posts_video_index_settings', 'my_video_index_settings' );
// We could also have used a more general hook 'algolia_posts_index_settings',
// but in that case we would have needed to add a check on the post type
// to ensure we only adjust settings for the 'video' post_type.

/**
 * @param array $settings
 *
 * @return array
 */
function my_video_index_settings( array $settings ) {

	// By default, the plugin uses 'is_sticky' and the 'post_date' in the custom ranking.
	// Here we first retrieve the custom ranking to be able to alter it with more control.
	$customRanking = $settings['customRanking'];

	// We add our custom ranking rule at the beginning of the rules so that
	// it is the first one considered in the algorithm.
	array_unshift( $customRanking, 'desc(views_count)' );

	// We override the initial custom ranking rules.
	$settings['customRanking'] = $customRanking;


	// Always return the value we are filtering.
	return $settings;
}
```
