// Copyright (c) arkade author(s) 2022. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

package apps

import (
	"fmt"

	"github.com/alexellis/arkade/pkg/k8s"

	"github.com/alexellis/arkade/pkg"
	"github.com/alexellis/arkade/pkg/apps"
	"github.com/alexellis/arkade/pkg/config"
	"github.com/alexellis/arkade/pkg/types"
	"github.com/spf13/cobra"
)

func MakeInstallOpenFaaSLoki() *cobra.Command {
	var OpenFaaSlokiApp = &cobra.Command{
		Use:          "openfaas-loki",
		Short:        "Install Loki-OpenFaaS and Configure Loki logs provider for OpenFaaS",
		Long:         "Install Loki-OpenFaaS and Configure Loki logs provider for OpenFaaS",
		Example:      "arkade install openfaas-loki",
		SilenceUsage: true,
	}

	OpenFaaSlokiApp.Flags().StringP("namespace", "n", "default", "The namespace to install loki (default: default")
	OpenFaaSlokiApp.Flags().Bool("update-repo", true, "Update the helm repo")
	OpenFaaSlokiApp.Flags().MarkDeprecated("update-repo", "this flag is deprecated, it is no longer needed for the openfaas-loki")
	OpenFaaSlokiApp.Flags().String("openfaas-namespace", "openfaas", "set the namespace that OpenFaaS is installed into")
	OpenFaaSlokiApp.Flags().String("loki-url", "http://loki-stack.default:3100", "set the loki url (default http://loki-stack.default:3100)")
	OpenFaaSlokiApp.Flags().String("version", "", "Override chart version, must be >= 1.7.3")
	OpenFaaSlokiApp.Flags().StringArray("set", []string{}, "Use custom flags or override existing flags \n(example --set grafana.enabled=true)")
	OpenFaaSlokiApp.RunE = func(command *cobra.Command, args []string) error {
		kubeConfigPath, _ := command.Flags().GetString("kubeconfig")

		namespace, _ := OpenFaaSlokiApp.Flags().GetString("namespace")

		openfaasNamespace, _ := OpenFaaSlokiApp.Flags().GetString("openfaas-namespace")
		lokiURL, _ := OpenFaaSlokiApp.Flags().GetString("loki-url")

		overrides := map[string]string{}
		overrides["lokiURL"] = lokiURL

		customFlags, _ := command.Flags().GetStringArray("set")

		if err := config.MergeFlags(overrides, customFlags); err != nil {
			return err
		}

		version, _ := command.Flags().GetString("version")

		lokiOptions := types.DefaultInstallOptions().
			WithNamespace(namespace).
			WithHelmURL("oci://ghcr.io/lucasroesler/charts/openfaas-loki").
			WithOverrides(overrides).
			WithKubeconfigPath(kubeConfigPath).
			WithHelmRepoVersion(version)

		// The default options includes the `values.yaml` file but this is already
		// implied when using the OCI chart.
		lokiOptions.Helm.ValuesFile = ""

		_, err := apps.MakeInstallChart(lokiOptions)
		if err != nil {
			return err
		}

		// Post install config of openfaas-loki
		k8s.Kubectl("-n", openfaasNamespace,
			"set", "env", "deployment/gateway",
			"-c", "gateway",
			"-e", fmt.Sprintf("logs_provider_url=http://openfaas-loki.%s:9191/", namespace))

		println(lokiOFInstallMsg)
		return nil
	}

	return OpenFaaSlokiApp
}

const LokiOFInfoMsg = `# Get started with openfaas-loki here:

# If you are authenticated with your openfaas gateway
faas-cli logs

# If you installed loki with grafana, with 'arkade install loki --grafana'
# You can use the grafana dashboard to see the OpenFaaS Logs, you can see
# how to get your grafana password with 'arkade info loki'

# We have automatically configured OpenFaaS to use the Loki logs URL, you can set 'gateway.logsProviderURL'
# When installing openfaas with Helm or use the '--log-provider-url' flag in arkade.
# The url is in the format 'http://loki-stack.namespace:3100/' (where namespace is the installed namespace for loki-stack)
`

const lokiOFInstallMsg = `=======================================================================
= OpenFaaS loki has been installed.                                   =
=======================================================================` +
	"\n\n" + LokiOFInfoMsg + "\n\n" + pkg.SupportMessageShort
