# Error Handling

In programming, errors are inevitable. They can occur due to various reasons such as incorrect user input, file not found, or division by zero. Python provides a robust way to handle these errors gracefully using exceptions. This ensures that your program can handle unexpected situations without crashing.

## Exceptions

An exception is an event that disrupts the normal flow of a program. When an error occurs, Python raises an exception. If the exception is not handled, the program will terminate.

### Common Exceptions

- `ZeroDivisionError`: Raised when you divide by zero.
- `FileNotFoundError`: Raised when a file operation (like open) fails because the file does not exist.
- `ValueError`: Raised when a function receives an argument of the correct type but an inappropriate value.

## Try, Except, Finally Blocks

To handle exceptions, you can use `try`, `except`, and `finally` blocks.

### Try and Except

The `try` block contains the code that might raise an exception. The `except` block contains the code that runs if an exception occurs.

```python
try:
    # Code that might raise an exception
    result = 10 / 0
except ZeroDivisionError:
    # Code that runs if a ZeroDivisionError occurs
    print("You can't divide by zero!")
```

### Finally

The `finally` block contains code that will run no matter what, whether an exception occurs or not. It is often used for cleanup actions, like closing a file.

```python
try:
    file = open('example.txt', 'r')
    # Perform file operations
except FileNotFoundError:
    print("The file was not found.")
finally:
    file.close()
```

## Custom Exceptions

You can also define your own exceptions by creating a new class that inherits from the built-in `Exception` class.

```python
class CustomError(Exception):
    pass

try:
    raise CustomError("This is a custom error")
except CustomError as e:
    print(e)
```

## Summary

- **Exceptions** are errors that disrupt the normal flow of a program.
- Use **try** and **except** blocks to handle exceptions.
- The **finally** block is used for code that should run no matter what.
- You can create **custom exceptions** by defining a new class that inherits from `Exception`.

By handling errors properly, you can make your programs more robust and user-friendly.

<div style="text-align: right;">
    <a href="10.md">Next</a>
</div>
<div style="text-align: left;">
    <a href="8.md">Prev</a>
</div>