# File Handling

File handling is an essential skill in Python that allows you to read from and write to files. This is useful for tasks like saving data, reading configuration files, or processing text files. Let's break down the basics of file handling in Python.

## Reading from Files

To read from a file, you need to open it first. Python provides the `open()` function for this purpose. Here's how you can read the contents of a file:

```python
# Open the file in read mode
file = open('example.txt', 'r')

# Read the contents of the file
content = file.read()

# Print the contents
print(content)

# Close the file
file.close()
```

### Using `with` Statement

A better way to handle files is by using the `with` statement. This ensures that the file is properly closed after its suite finishes, even if an exception is raised. Here's an example:

```python
# Open the file using 'with' statement
with open('example.txt', 'r') as file:
    content = file.read()
    print(content)
```

## Writing to Files

To write to a file, you need to open it in write mode (`'w'`). If the file doesn't exist, it will be created. If it does exist, its contents will be overwritten. Here's how you can write to a file:

```python
# Open the file in write mode
with open('example.txt', 'w') as file:
    file.write('Hello, world!')
```

### Appending to Files

If you want to add content to an existing file without overwriting it, you can open the file in append mode (`'a'`):

```python
# Open the file in append mode
with open('example.txt', 'a') as file:
    file.write('\nThis is a new line.')
```

## Working with File Paths

Sometimes, you need to work with file paths to access files in different directories. Python's `os` module provides functions to handle file paths:

```python
import os

# Get the current working directory
current_directory = os.getcwd()
print(current_directory)

# Join paths
file_path = os.path.join(current_directory, 'example.txt')
print(file_path)
```

## Summary

- **Reading from files**: Use `open('filename', 'r')` and `file.read()`.
- **Writing to files**: Use `open('filename', 'w')` and `file.write()`.
- **Appending to files**: Use `open('filename', 'a')` and `file.write()`.
- **Using `with` statement**: Ensures files are properly closed.
- **Working with file paths**: Use the `os` module.

File handling is a fundamental skill that will help you manage data efficiently in your Python programs. Practice reading from and writing to files to become comfortable with these operations.

<div style="text-align: right;">
    <a href="8.md">Next</a>
</div>
<div style="text-align: left;">
    <a href="6.md">Prev</a>
</div>