﻿//-----------------------------------------------------------------------
// <copyright file="ClusterSettings.cs" company="Akka.NET Project">
//     Copyright (C) 2009-2024 Lightbend Inc. <http://www.lightbend.com>
//     Copyright (C) 2013-2024 .NET Foundation <https://github.com/akkadotnet/akka.net>
// </copyright>
//-----------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Linq;
using System.Reflection;
using Akka.Actor;
using Akka.Configuration;
using Akka.Dispatch;
using Akka.Util;

namespace Akka.Cluster
{
    /// <summary>
    /// This class represents configuration information used when setting up a cluster.
    /// </summary>
    public sealed class ClusterSettings
    {
        private readonly Config _failureDetectorConfig;
        private readonly string _useDispatcher;

        /// <summary>
        /// Initializes a new instance of the <see cref="ClusterSettings"/> class.
        /// </summary>
        /// <param name="config">The configuration to use when setting up the cluster.</param>
        /// <param name="systemName">The name of the actor system hosting the cluster.</param>
        public ClusterSettings(Config config, string systemName)
        {
            //TODO: Requiring!
            var clusterConfig = config.GetConfig("akka.cluster");
            if (clusterConfig?.GetConfig("failure-detector") == null)
                throw new ConfigurationException($"Failed to instantiate {nameof(ClusterSettings)}: Configuration does not contain `akka.cluster` node. Did you forgot to set the 'akka.cluster.provider' HOCON property to 'cluster'?");

            LogInfoVerbose = clusterConfig.GetBoolean("log-info-verbose", false);
            LogInfo = LogInfoVerbose || clusterConfig.GetBoolean("log-info", false);
            _failureDetectorConfig = clusterConfig.GetConfig("failure-detector");
            FailureDetectorImplementationClass = _failureDetectorConfig.GetString("implementation-class", null);
            HeartbeatInterval = _failureDetectorConfig.GetTimeSpan("heartbeat-interval", null);
            HeartbeatExpectedResponseAfter = _failureDetectorConfig.GetTimeSpan("expected-response-after", null);
            MonitoredByNrOfMembers = _failureDetectorConfig.GetInt("monitored-by-nr-of-members", 0);

            SeedNodes = clusterConfig.GetStringList("seed-nodes", new string[] { }).Select(Address.Parse).ToImmutableList();
            SeedNodeTimeout = clusterConfig.GetTimeSpan("seed-node-timeout", null);
            RetryUnsuccessfulJoinAfter = clusterConfig.GetTimeSpanWithOffSwitch("retry-unsuccessful-join-after");
            ShutdownAfterUnsuccessfulJoinSeedNodes = clusterConfig.GetTimeSpanWithOffSwitch("shutdown-after-unsuccessful-join-seed-nodes");
            PeriodicTasksInitialDelay = clusterConfig.GetTimeSpan("periodic-tasks-initial-delay", null);
            GossipInterval = clusterConfig.GetTimeSpan("gossip-interval", null);
            GossipTimeToLive = clusterConfig.GetTimeSpan("gossip-time-to-live", null);
            LeaderActionsInterval = clusterConfig.GetTimeSpan("leader-actions-interval", null);
            UnreachableNodesReaperInterval = clusterConfig.GetTimeSpan("unreachable-nodes-reaper-interval", null);
            PublishStatsInterval = clusterConfig.GetTimeSpanWithOffSwitch("publish-stats-interval");

            var key = "down-removal-margin";
            var useDownRemoval = clusterConfig.GetString(key, "");
            DownRemovalMargin =
                (
                    useDownRemoval.ToLowerInvariant().Equals("off") ||
                    useDownRemoval.ToLowerInvariant().Equals("false") ||
                    useDownRemoval.ToLowerInvariant().Equals("no")
                ) ? TimeSpan.Zero :
                clusterConfig.GetTimeSpan("down-removal-margin", null);

#pragma warning disable CS0618
            AutoDownUnreachableAfter = clusterConfig.GetTimeSpanWithOffSwitch("auto-down-unreachable-after");
#pragma warning restore CS0618

            Roles = clusterConfig.GetStringList("roles", new string[] { }).ToImmutableHashSet();
            AppVersion = Util.AppVersion.Create(clusterConfig.GetString("app-version"));

            MinNrOfMembers = clusterConfig.GetInt("min-nr-of-members", 0);

            _useDispatcher = clusterConfig.GetString("use-dispatcher", null);
            if (string.IsNullOrEmpty(_useDispatcher)) _useDispatcher = Dispatchers.InternalDispatcherId;
            GossipDifferentViewProbability = clusterConfig.GetDouble("gossip-different-view-probability", 0);
            ReduceGossipDifferentViewProbability = clusterConfig.GetInt("reduce-gossip-different-view-probability", 0);
            SchedulerTickDuration = clusterConfig.GetTimeSpan("scheduler.tick-duration", null);
            SchedulerTicksPerWheel = clusterConfig.GetInt("scheduler.ticks-per-wheel", 0);

            MinNrOfMembersOfRole = clusterConfig.GetConfig("role").Root.GetObject().Items
                .ToImmutableDictionary(kv => kv.Key, kv => kv.Value.GetObject().GetKey("min-nr-of-members").GetInt());

            VerboseHeartbeatLogging = clusterConfig.GetBoolean("debug.verbose-heartbeat-logging", false);
            VerboseGossipReceivedLogging = clusterConfig.GetBoolean("debug.verbose-receive-gossip-logging", false);

            var downingProviderClassName = clusterConfig.GetString("downing-provider-class", null);
            if (!string.IsNullOrEmpty(downingProviderClassName))
                DowningProviderType = Type.GetType(downingProviderClassName, true);
#pragma warning disable CS0618
            else if (AutoDownUnreachableAfter.HasValue)
#pragma warning restore CS0618
                DowningProviderType = typeof(AutoDowning);
            else
                DowningProviderType = typeof(NoDowning);

            RunCoordinatedShutdownWhenDown = clusterConfig.GetBoolean("run-coordinated-shutdown-when-down", false);
            
            TimeSpan GetWeaklyUpDuration()
            {
                var cKey = "allow-weakly-up-members";
                switch (clusterConfig.GetString(cKey, string.Empty)
                    .ToLowerInvariant())
                {
                    case "off":
                        return TimeSpan.Zero;
                    case "on":

                        return TimeSpan.FromSeconds(7); // for backwards compatibility when it wasn't a duration
                    default:
                        var val = clusterConfig.GetTimeSpan(cKey, TimeSpan.FromSeconds(7));
                        if(!(val > TimeSpan.Zero))
                            throw new ConfigurationException($"Valid settings for [akka.cluster.{cKey}] are 'off', 'on', or a timespan greater than 0s. Received [{val}]");
                        return val;
                }
            }

            WeaklyUpAfter = GetWeaklyUpDuration();

            UseLegacyHeartbeatMessage = clusterConfig.GetBoolean("use-legacy-heartbeat-message", false);
        }

        /// <summary>
        /// Determine whether to log verbose <see cref="Akka.Event.LogLevel.InfoLevel"/> messages for temporary troubleshooting.
        /// </summary>
        public bool LogInfoVerbose { get; }

        /// <summary>
        /// Determine whether to log <see cref="Akka.Event.LogLevel.InfoLevel"/> messages.
        /// </summary>
        public bool LogInfo { get; }

        /// <summary>
        /// The configuration for the underlying failure detector used by Akka.Cluster.
        /// </summary>
        public Config FailureDetectorConfig => _failureDetectorConfig;

        /// <summary>
        /// The fully qualified type name of the failure detector class that will be used.
        /// </summary>
        public string FailureDetectorImplementationClass { get; }

        /// <summary>
        /// The amount of time between when heartbeat messages are sent.
        /// </summary>
        public TimeSpan HeartbeatInterval { get; }

        /// <summary>
        /// The amount of time we expect a heartbeat response after first contact with a new node.
        /// </summary>
        public TimeSpan HeartbeatExpectedResponseAfter { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public int MonitoredByNrOfMembers { get; }

        /// <summary>
        /// A list of designated seed nodes for the cluster.
        /// </summary>
        public ImmutableList<Address> SeedNodes { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public TimeSpan SeedNodeTimeout { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public TimeSpan? RetryUnsuccessfulJoinAfter { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public TimeSpan? ShutdownAfterUnsuccessfulJoinSeedNodes { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public TimeSpan PeriodicTasksInitialDelay { get; }

        /// <summary>
        /// The amount of time between when gossip messages are sent.
        /// </summary>
        public TimeSpan GossipInterval { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public TimeSpan GossipTimeToLive { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public TimeSpan LeaderActionsInterval { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public TimeSpan UnreachableNodesReaperInterval { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public TimeSpan? PublishStatsInterval { get; }

        /// <summary>
        /// Obsolete. No longer used as of Akka.NET v1.5.
        /// </summary>
        [Obsolete(message:"Deprecated as of Akka.NET v1.5.2 - clustering defaults to using KeepMajority SBR instead")]
        public TimeSpan? AutoDownUnreachableAfter { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public ImmutableHashSet<string> Roles { get; }

        /// <summary>
        /// Application version
        /// </summary>
        public AppVersion AppVersion { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public double GossipDifferentViewProbability { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public int ReduceGossipDifferentViewProbability { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public string UseDispatcher => _useDispatcher;

        /// <summary>
        /// TBD
        /// </summary>
        public TimeSpan SchedulerTickDuration { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public int SchedulerTicksPerWheel { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public int MinNrOfMembers { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public ImmutableDictionary<string, int> MinNrOfMembersOfRole { get; }

        internal TimeSpan DownRemovalMargin { get; }

        /// <summary>
        /// Determine whether or not to log heartbeat message in verbose mode.
        /// </summary>
        public bool VerboseHeartbeatLogging { get; }

        /// <summary>
        /// Determines whether or not to log gossip consumption logging in verbose mode
        /// </summary>
        public bool VerboseGossipReceivedLogging { get; }

        /// <summary>
        /// TBD
        /// </summary>
        public Type DowningProviderType { get; }

        /// <summary>
        /// Trigger the <see cref="CoordinatedShutdown"/> even if this node was removed by non-graceful
        /// means, such as being downed.
        /// </summary>
        public bool RunCoordinatedShutdownWhenDown { get; }

        /// <summary>
        /// If this is set to "off", the leader will not move <see cref="MemberStatus.Joining"/> members to <see cref="MemberStatus.Up"/> during a network
        /// split. This feature allows the leader to accept <see cref="MemberStatus.Joining"/> members to be <see cref="MemberStatus.WeaklyUp"/>
        /// so they become part of the cluster even during a network split.
        ///
        /// The leader will move <see cref="MemberStatus.WeaklyUp"/> members to <see cref="MemberStatus.Up"/> status once convergence has been reached.
        /// </summary>
        public bool AllowWeaklyUpMembers => WeaklyUpAfter != TimeSpan.Zero;

        /// <summary>
        /// The duration after which a member who is currently <see cref="MemberStatus.Joining"/> will be marked as
        /// <see cref="MemberStatus.WeaklyUp"/> in the event that members of the cluster are currently unreachable.
        ///
        /// This is designed to allow new cluster members to perform work even in the event of a cluster split.
        /// 
        /// The leader will move <see cref="MemberStatus.WeaklyUp"/> members to <see cref="MemberStatus.Up"/> status once convergence has been reached.
        /// </summary>
        public TimeSpan WeaklyUpAfter { get; }
        
        /// <summary>
        /// Enable/disable legacy pre-1.4.19 <see cref="ClusterHeartbeatSender.Heartbeat"/> and
        /// <see cref="ClusterHeartbeatSender.HeartbeatRsp"/> wire format serialization support.
        /// Set this to true if you're doing a rolling update from Akka.NET version older than 1.4.19.
        /// </summary>
        public bool UseLegacyHeartbeatMessage { get; }
    }
}

