---
uid: AK1000
title: Akka.Analyzers Rule AK1000 - "Do not use `new` to create actors"
---

# AK1000 - Error

Do not use the `new` keyword to create Akka.NET actors.

## Cause

A class inheriting from [`Akka.Actor.ActorBase`](xref:Akka.Actor.ActorBase) or one of its descendants was instantiated directly via the `new` keyword, which is illegal. Actors can only be instantiated inside a [`Props.Create` method](xref:Akka.Actor.Props) or via an [`IIndirectActorProducer`](xref:Akka.Actor.IIndirectActorProducer) implementation (such as [`Akka.DependencyInjection`](xref:dependency-injection).)

An example:

```csharp
using Akka.Actor;

class MyActor : ActorBase {
    protected override bool Receive(object message) {
        return true;
    }
}

class Test
{
    void Method()
    {
        MyActor actorInstance = new MyActor(); // not supported by Akka.NET
    }
}
```

## Resolution

The correct way to get a reference back to an actor is to wrap your actor's constructor call inside a `Props.Create` method and then pass that into either [`ActorSystem.ActorOf`](xref:Akka.Actor.ActorSystem) (when creating a top-level actor) or [`Context.ActorOf`](xref:Akka.Actor.IActorRefFactory#Akka_Actor_IActorRefFactory_ActorOf_Akka_Actor_Props_System_String_) (when creating a child actor.)

Here's an example below:

```csharp
using Akka.Actor;

class MyActor : ReceiveActor {
    private readonly string _name;
    private readonly int _myVar;

    public MyActor(string name, int myVar){
        _name = name;
        _myVar = myVar;
        ReceiveAny(_ => {
            Sender.Tell(_name + _myVar);
        });
    }
}

class Test
{
    void Method()
    {
        // obviously, don't create a new ActorSystem every time - this is just an example.
        ActorSystem sys = ActorSystem.Create("MySys");
        Akka.Actor.Props props = Akka.Actor.Props.Create(() => new MyActor("foo", 1));
        IActorRef realActorInstance = sys.ActorOf(props);
    }
}
```
