import {
  Box,
  Button,
  Code,
  Flex,
  createStandaloneToast,
  useColorModeValue,
} from '@chakra-ui/react';
import { Prism as SyntaxHighlighter } from 'react-syntax-highlighter';
import {
  dracula,
  oneLight,
} from 'react-syntax-highlighter/dist/cjs/styles/prism';

const { toast } = createStandaloneToast();

const handleClickCopy = (childrenValue: string) => async () => {
  const toastId = childrenValue.substring(0, 10);

  await navigator.clipboard.writeText(childrenValue);
  if (!toast.isActive(toastId)) {
    toast({
      id: toastId,
      status: 'success',
      position: 'top-right',
      title: 'Copied',
      isClosable: true,
    });
  }
};

const languageClassNameRegex = /language-(\w+)/;
const enterRegex = /\n$/;

export const CodeBlock = ({
  className,
  children,
  ...props
}: React.HTMLAttributes<HTMLElement>) => {
  /** https://github.com/remarkjs/react-markdown#use-custom-components-syntax-highlight */
  const match = languageClassNameRegex.exec(className || '');
  const language = match?.[1];
  const childrenValue = String(children).replace(enterRegex, '');
  const themes = useColorModeValue(oneLight, dracula);

  return match ? (
    <Box width="100%">
      <Flex alignItems="center">
        {language && <Code>{language}</Code>}
        <Button
          marginLeft="auto"
          colorScheme="green"
          height={6}
          padding={0}
          fontSize={12}
          textTransform="lowercase"
          onClick={handleClickCopy(childrenValue)}
        >
          Copy
        </Button>
      </Flex>
      <SyntaxHighlighter
        language={language}
        style={themes}
        showLineNumbers={false}
      >
        {childrenValue}
      </SyntaxHighlighter>
    </Box>
  ) : (
    <code className={className} {...props}>
      {children}
    </code>
  );
};
