/**
 * This class provides a service to communicate with the OpenAI API
 * and generate text based on user input. It uses a HashMap to store
 * the context of each user conversation and a whitelist to restrict
 * access to the bot. It also provides a method to reset the context
 * of a user.
 */

package es.achousa.services;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.pengrad.telegrambot.model.Chat;
import com.pengrad.telegrambot.model.Update;
import es.achousa.clients.OpenAIApiClient;
import es.achousa.model.request.ChatRequest;
import es.achousa.model.request.Message;
import es.achousa.model.response.ChatResponse;
import es.achousa.utils.MessageLog;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;

@Service
public class GptService {

    protected final Log log = LogFactory.getLog(this.getClass());
    @Autowired
    private OpenAIApiClient client;
    @Value("${openai.maxtokens}")
    private Integer maxtokens;
    @Value("${openai.model}")
    private String model;
    @Value("${openai.temperature}")
    private Double temperature;
    @Value("${openai.systemprompt}")
    private String systemprompt;
    @Value("${openai.max.message.pool.size}")
    Integer maxMessagePoolSize;
    @Value("${bot.presentation}")
    private String presentation;
    @Value("#{'${bot.whitelist:}'.empty ? null : '${bot.whitelist}'.split(',')}")
    private ArrayList<String> whiteList;
    private HashSet<String> whiteSet;

    // UserContext is a Map, with the userid and a list of previous messages
    private HashMap<Long, MessageLog<Message>> userContext = new HashMap<>();

    private ArrayList<String> examples;

    @Autowired
    private Environment env;

    private Integer ntokens = 0;

    public Integer getNumTokens() {
        return this.ntokens;
    }

    /**
     * This method is called after the bean has been instantiated and
     * is used to initialize the examples list and the whitelist.
     */
    @PostConstruct
    private void init(){
        // Read examples from configuration
        this.examples = getPropertyList("openai.example");
        // turn whitelist into a hashset for quicker access
        if(this.whiteList!=null && !this.whiteList.isEmpty()) {
            this.whiteSet = new HashSet<>();
            for(String name : this.whiteList) {
                this.whiteSet.add(name.toLowerCase());
            }
        } else {
            this.whiteSet = null;
        }

    }

    /**
     * This method gets a list of properties from the Spring Environment
     * and returns an ArrayList of Strings. It is used to load examples
     * from configuration.
     * @param name the name of the property to retrieve
     * @return an ArrayList with the values of the property
     */
    private ArrayList<String> getPropertyList(String name) {
        ArrayList<String> list = new ArrayList<>();
        int i = 1;
        while (env.containsProperty(name+ "." + i)) {
            list.add(env.getProperty(name +"." + i));
            i++;
        }
        return list;
    }

    /**
     * This method sends a message to the OpenAI API to generate a text
     * response based on user input. It checks if the user is authorized
     * to talk to the bot and if the conversation is private or not.
     * @param update the update object containing user input
     * @return the text response generated by the OpenAI API
     */
    public synchronized String SendMessage(Update update) {

        if(!checkPermission(update)){
            return "I'm sorry but I can't talk to you, you're not on the access list.";
        }

        ChatResponse response = null;
        try {

            // Compose new request
            ObjectMapper mapper = new ObjectMapper();
            ChatRequest chatRequest = new ChatRequest();
            chatRequest.setModel(model);
            chatRequest.setTemperature(temperature);
            chatRequest.setMaxTokens(maxtokens);

            // Set up array of messages
            ArrayList<Message> messages = new ArrayList<>();
            Message systemMessage = new Message();

            // System prompt
            systemMessage.setRole("system");
            systemMessage.setContent(systemprompt);
            messages.add(systemMessage);

            // List of user messages
            if(isPrivate(update)) {
                // If this is a private chat, then we send along a number of previous messages for context
                if(!userContext.containsKey(update.message().from().id())) {
                    // If the user is not in the map push a new entry for him
                    userContext.put(update.message().from().id(),new MessageLog<>(maxMessagePoolSize));
                    // If there are examples, add them to the context (this is only made the first time)
                    if(!this.examples.isEmpty()) {
                        userContext.get(update.message().from().id()).addAll(getExamples(this.examples));
                    }
                }
                // Add the new message to the message log, previous messages will be erased
                // when context reaches it's max length
                Message newUserMessage = new Message();
                newUserMessage.setRole("user");
                newUserMessage.setContent(update.message().text());
                userContext.get(update.message().from().id()).add(newUserMessage);
                // add to userMessages
                messages.addAll(userContext.get(update.message().from().id()));
            } else {
                // If this is a group, the message is processed without any context
                // If there are examples, we add them to the list of messages prior the user message
                if(!this.examples.isEmpty()) {
                    messages.addAll(getExamples(this.examples));
                }
                Message userMessage = new Message();
                userMessage.setRole("user");
                userMessage.setContent(update.message().text());
                messages.add(userMessage);
            }

            // set messages to the request
            chatRequest.setMessages(messages);

            // Send synchronous request to the OpenAI Api
            response = client.getCompletion(chatRequest);
            // Incremet token counter
            ntokens = ntokens + response.getUsage().getTotalTokens();

            // If this is a private conversation, add the assistant response to the user context
            if(isPrivate(update)) {
                Message assistantMessage = new Message();
                assistantMessage.setRole("assistant");
                assistantMessage.setContent(response.getChoices().get(0).getMessage().getContent());
                userContext.get(update.message().from().id()).add(assistantMessage);
            }

            // Return text to be sent to the user
            return response.getChoices().get(0).getMessage().getContent();

        } catch (Exception e) {
            e.printStackTrace();
            return "I'm sorry, something went wrong";
        }
    }

    /**
     * This method returns an ArrayList of Messages based on an ArrayList
     * of Strings. Each message is created with a role and a content.
     * @param examples the ArrayList of Strings to convert into Messages
     * @return an ArrayList of Messages
     */
    private ArrayList<Message> getExamples(ArrayList<String> examples) {
        ArrayList<Message> results = new ArrayList<>();
        for(String example: examples) {
            try {
                String role = example.split(":",2)[0];
                String content = example.split(":",2)[1];
                if(StringUtils.isNotEmpty(role) && StringUtils.isNotEmpty(content)) {
                    Message exampleMessage = new Message();
                    exampleMessage.setRole(role.toLowerCase());
                    exampleMessage.setContent(content);
                    results.add(exampleMessage);
                } else {
                    log.error("Something went wrong with this example: " + example);
                }
            } catch(Exception e) {
                log.error("Something went wrong with this example: " + example + " " + e.getMessage());
            }
        }
        return results;
    }

    /**
     * This method checks if the user is authorized to talk to the bot
     * based on the whitelist. If the whitelist is empty, everyone can
     * talk to the bot. Otherwise, the method checks if the user or the
     * group is in the whitelist.
     * @param update the update object containing user information
     * @return true if the user is authorized, false otherwise
     */
    private boolean checkPermission(Update update) {
        // if the whitelist is empty, everyone can talk to the bot
        // otherwise it must be checked against
        String userName = "none";
        String groupName = "none";
        if (this.whiteSet!=null && !this.whiteSet.isEmpty()) {
            if(update.message().from().firstName()!=null) {
                userName = update.message().from().firstName().toLowerCase();
            }
            if(update.message().chat().title()!=null) {
                groupName = update.message().chat().title().toLowerCase();
            }
            // either name on the list, grants access
            if(this.whiteSet.contains(userName)) {
                // access is granted for the user
                return true;
            }
            if(this.whiteSet.contains(groupName)) {
                // access is granted for the group
                return true;
            }

            log.error("An unauthorised user tried to talk to me: " + userName);
            return false;
        }
        // no whitelist
        return true;

    }

    /**
     * This method sends a message to the OpenAI API to generate a text
     * response. The message is not based on user input, but on a custom
     * presentation text.
     * @param update the update object containing user information
     * @param text the custom presentation text
     * @return the text response generated by the OpenAI API
     */
    public String sendCustomMessage(Update update, String text) {
        ChatResponse response = null;
        try {

            // Compose new request
            ObjectMapper mapper = new ObjectMapper();
            ChatRequest chatRequest = new ChatRequest();
            chatRequest.setModel(model);
            chatRequest.setTemperature(temperature);
            chatRequest.setMaxTokens(maxtokens);
            // Set up array of messages
            ArrayList<Message> messages = new ArrayList<>();
            Message systemMessage = new Message();
            // System prompt
            systemMessage.setRole("system");
            systemMessage.setContent(systemprompt);
            messages.add(systemMessage);
            // Custom message
            Message userMessage = new Message();
            userMessage.setRole("user");
            userMessage.setContent(this.presentation);
            messages.add(userMessage);
            // set messages to the request
            chatRequest.setMessages(messages);

            // Send synchronous request to the OpenAI Api
            response = client.getCompletion(chatRequest);
            // Incremet token counter
            ntokens = ntokens + response.getUsage().getTotalTokens();
            // Return text to be sent to the user
            return response.getChoices().get(0).getMessage().getContent();

        } catch (Exception e) {
            e.printStackTrace();
            return "Lo siento, algo ha ido mal.";
        }
    }

    /**
     * Determines whether a message was sent privately to the bot or in a group chat.
     * @param update The message received from the user.
     * @return true if the message is private, false otherwise.
     */
    private boolean isPrivate(Update update) {
        if (update.message().chat().type().equals(Chat.Type.Private)) {
            return true;
        } else {
            return false;
        }
    }

    public String resetUserContext(Update update) {
        // If this is a private chat, reset context for current user
        if(isPrivate(update)){
           if (this.userContext.containsKey(update.message().from().id())) {
               this.userContext.remove(update.message().from().id());
               return "Se ha reseteado el contexto del usuario " + update.message().from().firstName();
           } else {
               return "No he encontrado contexto para el usuario " + update.message().from().firstName();
           }
        } else {
            return "No hay nada que resetear, ya es que esto es un grupo.";
        }
    }
}
