<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\Console\Descriptor;

use Symfony\Component\Console\Application;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Exception\InvalidArgumentException;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputDefinition;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * @author Jean-François Simon <jeanfrancois.simon@sensiolabs.com>
 *
 * @internal
 */
abstract class Descriptor implements DescriptorInterface
{
    /**
     * @var \Symfony\Component\Console\Output\OutputInterface
     */
    protected $output;

    /**
     * @param object $object
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     * @param mixed[] $options
     * @return void
     */
    public function describe($output, $object, $options = [])
    {
        $this->output = $output;

        switch (true) {
            case $object instanceof InputArgument:
                $this->describeInputArgument($object, $options);
                break;
            case $object instanceof InputOption:
                $this->describeInputOption($object, $options);
                break;
            case $object instanceof InputDefinition:
                $this->describeInputDefinition($object, $options);
                break;
            case $object instanceof Command:
                $this->describeCommand($object, $options);
                break;
            case $object instanceof Application:
                $this->describeApplication($object, $options);
                break;
            default:
                throw new InvalidArgumentException(sprintf('Object of type "%s" is not describable.', get_debug_type($object)));
        }
    }

    /**
     * @param string $content
     * @param bool $decorated
     * @return void
     */
    protected function write($content, $decorated = false)
    {
        $this->output->write($content, false, $decorated ? OutputInterface::OUTPUT_NORMAL : OutputInterface::OUTPUT_RAW);
    }

    /**
     * Describes an InputArgument instance.
     * @param \Symfony\Component\Console\Input\InputArgument $argument
     * @param mixed[] $options
     * @return void
     */
    abstract protected function describeInputArgument($argument, $options = []);

    /**
     * Describes an InputOption instance.
     * @param \Symfony\Component\Console\Input\InputOption $option
     * @param mixed[] $options
     * @return void
     */
    abstract protected function describeInputOption($option, $options = []);

    /**
     * Describes an InputDefinition instance.
     * @param \Symfony\Component\Console\Input\InputDefinition $definition
     * @param mixed[] $options
     * @return void
     */
    abstract protected function describeInputDefinition($definition, $options = []);

    /**
     * Describes a Command instance.
     * @param \Symfony\Component\Console\Command\Command $command
     * @param mixed[] $options
     * @return void
     */
    abstract protected function describeCommand($command, $options = []);

    /**
     * Describes an Application instance.
     * @param \Symfony\Component\Console\Application $application
     * @param mixed[] $options
     * @return void
     */
    abstract protected function describeApplication($application, $options = []);
}
