<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\Console\Output;

use Symfony\Component\Console\Formatter\OutputFormatterInterface;

/**
 * OutputInterface is the interface implemented by all Output classes.
 *
 * @author Fabien Potencier <fabien@symfony.com>
 */
interface OutputInterface
{
    const VERBOSITY_QUIET = 16;
    const VERBOSITY_NORMAL = 32;
    const VERBOSITY_VERBOSE = 64;
    const VERBOSITY_VERY_VERBOSE = 128;
    const VERBOSITY_DEBUG = 256;

    const OUTPUT_NORMAL = 1;
    const OUTPUT_RAW = 2;
    const OUTPUT_PLAIN = 4;

    /**
     * Writes a message to the output.
     *
     * @param bool $newline Whether to add a newline
     * @param int  $options A bitmask of options (one of the OUTPUT or VERBOSITY constants),
     *                      0 is considered the same as self::OUTPUT_NORMAL | self::VERBOSITY_NORMAL
     *
     * @return void
     * @param string|mixed[] $messages
     */
    public function write($messages, $newline = false, $options = 0);

    /**
     * Writes a message to the output and adds a newline at the end.
     *
     * @param int $options A bitmask of options (one of the OUTPUT or VERBOSITY constants),
     *                     0 is considered the same as self::OUTPUT_NORMAL | self::VERBOSITY_NORMAL
     *
     * @return void
     * @param string|mixed[] $messages
     */
    public function writeln($messages, $options = 0);

    /**
     * Sets the verbosity of the output.
     *
     * @return void
     * @param int $level
     */
    public function setVerbosity($level);

    /**
     * Gets the current verbosity of the output.
     */
    public function getVerbosity();

    /**
     * Returns whether verbosity is quiet (-q).
     */
    public function isQuiet();

    /**
     * Returns whether verbosity is verbose (-v).
     */
    public function isVerbose();

    /**
     * Returns whether verbosity is very verbose (-vv).
     */
    public function isVeryVerbose();

    /**
     * Returns whether verbosity is debug (-vvv).
     */
    public function isDebug();

    /**
     * Sets the decorated flag.
     *
     * @return void
     * @param bool $decorated
     */
    public function setDecorated($decorated);

    /**
     * Gets the decorated flag.
     */
    public function isDecorated();

    /**
     * @return void
     * @param \Symfony\Component\Console\Formatter\OutputFormatterInterface $formatter
     */
    public function setFormatter($formatter);

    /**
     * Returns current output formatter instance.
     */
    public function getFormatter();
}
