from cgi import FieldStorage
import cgitb
import os

__version__ = "0.7.3"
_cookie_var = "HTTP_COOKIE"


class BWCGI:
    """ handy cgi stuff """
    _header_state = False
    cgi_cookies = dict()
    cgi_headers = dict()

    def __init__(self, **kwargs):
        self.set_header("Content-type", kwargs.get("content_type", "text/html"))
        if _cookie_var in os.environ:
            self.parse_cookies()

    def set_header(self, k, v):
        if k in self.cgi_headers:
            if isinstance(self.cgi_headers[k], list):
                self.cgi_headers[k].append(v)
            else:
                self.cgi_headers[k] = [self.cgi_headers[k], v]
        else:
            self.cgi_headers[k] = str(v)
        return v

    def get_header(self, k):
        return self.cgi_headers.get(k, None)

    def send_header(self):
        """send the header(s), only once"""
        if self._header_state:
            return
        for k in self.cgi_headers:
            value = self.cgi_headers[k]
            if isinstance(value, list):
                for v in value:
                    print("{}: {}".format(k, v))
                else:
                    print("{}: {}".format(k, value))
        print()
        self._header_state = True
        cgitb.enable()  # only after the header has been sent

    def set_cookie(self, key, value, **kwargs):
        """ kwargs can include expires path, or domain """
        cookie = "{}={}".format(str(key), str(value))
        if kwargs.keys():
            for k in kwargs.keys():
                cookie = "{}; {}={}".format(cookie, k, kwargs[k])
        self.set_header("Set-Cookie", cookie)

    def parse_cookies(self):
        if os.environ[_cookie_var]:
            for ck in os.environ[_cookie_var].split(";"):
                lhs, rhs = ck.strip().split("=")
                self.cgi_cookies[lhs.strip()] = rhs.strip()

    def get_cookies(self):
        return self.cgi_cookies

    def get_cookie(self, key):
        return self.cgi_cookies.get(key, None)

    @staticmethod
    def linkback():
        """ return a relative URL for use as a linkback to this scrip"""
        for e in ("REQUEST_URL", "SCRIPT_NAME"):
            if e in os.environ:
                lb = os.environ[e]
                break
        else:
            return "*** cannot make linkback ***"
        if "?" in lb:
            lb = lb[0:lb.find("?")]
        return os.path.basename(lb)

    @staticmethod
    def vars():
        return FieldStorage()

    # utility methods

    @staticmethod
    def entity_encode(s):
        """
        convert unicode to XML entities
        :param s:
        :return: encoded string
        """
        outbytes = bytearray()
        for c in s:
            if ord(c) > 127:
                outbytes += bytes("&#{:d};".format(ord(c)), encoding="utf_8")
            else:
                outbytes.append(ord(c))
        return str(outbytes, encoding="utf_8")


def test():
    if _cookie_var not in os.environ:
        os.environ[_cookie_var] = "one=1; two=2; three=3"
    cgi = BWCGI(content_type="text/plain")
    cgi.set_header("X-BWCGI", __version__)
    cgi.set_header("X-number", 42)
    cgi.set_cookie("one", 1)
    cgi.set_cookie("two", 2)
    cgi.set_cookie("three", 3, path="/", expires="31-Dec-2023 23:59:59 GMT", domain="./.org")
    cgi.set_cookie("five", 5)
    cgi.send_header()
    cgi.send_header()
    cgi.send_header()
    print("Hello, CGI")
    print("header X-BWCGI:", cgi.get_header("X-BWCGI"))
    print("header Eggs:", cgi.get_header("Eggs"))
    print("Cookies:")
    print(sorted(cgi.get_header("X-BWCGI")))
    print("cookie one:", cgi.get_cookie("one"))
    print("cookie seven:", cgi.get_cookie("seven"))


if __name__ == '__main__':
    test()
