import dotenv from "dotenv";
import { OpenAI } from "openai";
import {
  ChatCompletionCreateParamsNonStreaming,
  ChatCompletionCreateParamsStreaming,
} from "openai/resources/chat/completions";

import { ChatOpenaiInputParamsSchema } from "../../schema/providerSchemas/chatOpenai.schema";

dotenv.config();

/**
 * Generate a response using the OpenAI API based on the input parameters.
 * @param {Object} params - The input parameters for creating the chat completion request.
 * @param {boolean} [stream=false] - A boolean flag indicating whether to use stream mode for the response.
 * @param {boolean} [isAzure=false] - A boolean flag indicating whether to use Azure OpenAI API configuration.
 * @returns {Promise<any>} The response generated by the OpenAI API.
 */
export default async function generateResponse({
  params,
  stream = false,
  isAzure = false,
  timeout,
  maxRetries,
}: {
  params: ChatOpenaiInputParamsSchema;
  stream?: boolean;
  isAzure?: boolean;
  timeout: number;
  maxRetries: number;
}): Promise<any> {
  let openai: OpenAI;
  if (isAzure) {
    openai = new OpenAI({
      apiKey: process.env.AZURE_AOAI_OPENAI_SUBSCRIPTION_KEY,
      baseURL: process.env.AZURE_AOAI_OPENAI_API_BASE_URL + `/openai/deployments/${params?.model}`,
      defaultQuery: { "api-version": process.env.AZURE_OPENAI_API_VERSION },
      defaultHeaders: {
        "ocp-apim-subscription-key": process.env.AZURE_AOAI_OPENAI_SUBSCRIPTION_KEY,
      },
      timeout: timeout,
      maxRetries: maxRetries,
    });
  } else {
    openai = new OpenAI({
      apiKey: process.env.OPENAI_API_KEY,
      organization: process.env.OPENAI_ORG,
      timeout: timeout,
      maxRetries: maxRetries,
    });
  }
  if (stream) {
    const stream = await openai.chat.completions.create({
      ...params,
    } as ChatCompletionCreateParamsStreaming);
    return stream;
  } else {
    const completion = await openai.chat.completions.create({
      ...params,
    } as ChatCompletionCreateParamsNonStreaming);
    return completion;
  }
}
