"""
Django settings for catalog project.

Generated by 'django-admin startproject' using Django 2.0.5.

For more information on this file, see
https://docs.djangoproject.com/en/2.0/topics/settings/

For the full list of settings and their values, see
https://docs.djangoproject.com/en/2.0/ref/settings/
"""

from datetime import timedelta
from pathlib import Path
from socket import gethostbyname, gethostname

import sentry_sdk
from decouple import config
from sentry_sdk.integrations.django import DjangoIntegration
from sentry_sdk.integrations.logging import ignore_logger

from catalog.configuration.aws import AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY
from catalog.configuration.elasticsearch import ES, MEDIA_INDEX_MAPPING
from catalog.configuration.link_validation_cache import (
    LinkValidationCacheExpiryConfiguration,
)
from catalog.configuration.logging import LOGGING


# Build paths inside the project like this: BASE_DIR.join('dir', 'subdir'...)
BASE_DIR = Path(__file__).resolve().parent.parent

# Where to collect static files in production/development deployments
STATIC_ROOT = config("STATIC_ROOT", default="/var/api_static_content/static")

FILTER_DEAD_LINKS_BY_DEFAULT = config(
    "FILTER_DEAD_LINKS_BY_DEFAULT", cast=bool, default=True
)

# Quick-start development settings - unsuitable for production
# See https://docs.djangoproject.com/en/2.0/howto/deployment/checklist/

# SECURITY WARNING: keep the secret key used in production secret!
SECRET_KEY = config("DJANGO_SECRET_KEY")  # required

# SECURITY WARNING: don't run with debug turned on in production!
DEBUG = config("DJANGO_DEBUG_ENABLED", default=False, cast=bool)

ENVIRONMENT = config("ENVIRONMENT", default="local")

ALLOWED_HOSTS = config("ALLOWED_HOSTS").split(",") + [
    gethostname(),
    gethostbyname(gethostname()),
]

if lb_url := config("LOAD_BALANCER_URL", default=""):
    ALLOWED_HOSTS.append(lb_url)

if DEBUG:
    ALLOWED_HOSTS += [
        "dev.openverse.test",  # used in local development
        "localhost",
        "127.0.0.1",
        "0.0.0.0",
    ]

USE_S3 = config("USE_S3", default=False, cast=bool)

# Application definition

INSTALLED_APPS = [
    "catalog",
    "catalog.api",
    "drf_yasg",
    "django.contrib.admin",
    "django.contrib.auth",
    "django.contrib.contenttypes",
    "django.contrib.sessions",
    "django.contrib.messages",
    "django.contrib.staticfiles",
    "oauth2_provider",
    "rest_framework",
    "corsheaders",
    "sslserver",
]

if USE_S3:
    DEFAULT_FILE_STORAGE = "storages.backends.s3boto3.S3Boto3Storage"
    AWS_STORAGE_BUCKET_NAME = config("LOGOS_BUCKET", default="openverse_api-logos-prod")
    AWS_S3_SIGNATURE_VERSION = "s3v4"
    INSTALLED_APPS.append("storages")

# https://github.com/dabapps/django-log-request-id#logging-all-requests
LOG_REQUESTS = True
# https://github.com/dabapps/django-log-request-id#installation-and-usage
REQUEST_ID_RESPONSE_HEADER = "X-Request-Id"

MIDDLEWARE = [
    # https://github.com/dabapps/django-log-request-id
    "log_request_id.middleware.RequestIDMiddleware",
    "django.middleware.security.SecurityMiddleware",
    "django.contrib.sessions.middleware.SessionMiddleware",
    "corsheaders.middleware.CorsMiddleware",
    "django.middleware.common.CommonMiddleware",
    "django.middleware.csrf.CsrfViewMiddleware",
    "django.contrib.auth.middleware.AuthenticationMiddleware",
    "django.contrib.messages.middleware.MessageMiddleware",
    "django.middleware.clickjacking.XFrameOptionsMiddleware",
    "oauth2_provider.middleware.OAuth2TokenMiddleware",
]

SWAGGER_SETTINGS = {
    "DEFAULT_INFO": "catalog.urls.swagger.open_api_info",
    "SECURITY_DEFINITIONS": {},
}

OAUTH2_PROVIDER = {
    "SCOPES": {
        "read": "Read scope",
        "write": "Write scope",
    },
    "ACCESS_TOKEN_EXPIRE_SECONDS": config(
        "ACCESS_TOKEN_EXPIRE_SECONDS", default=3600 * 12, cast=int
    ),
}

OAUTH2_PROVIDER_APPLICATION_MODEL = "api.ThrottledApplication"

THROTTLE_ANON_BURST = config("THROTTLE_ANON_BURST", default="5/hour")
THROTTLE_ANON_SUSTAINED = config("THROTTLE_ANON_SUSTAINED", default="100/day")
THROTTLE_ANON_THUMBS = config("THROTTLE_ANON_THUMBS", default="150/minute")
THROTTLE_OAUTH2_THUMBS = config("THROTTLE_OAUTH2_THUMBS", default="500/minute")

REST_FRAMEWORK = {
    "DEFAULT_AUTHENTICATION_CLASSES": (
        "oauth2_provider.contrib.rest_framework.OAuth2Authentication",
    ),
    "DEFAULT_VERSIONING_CLASS": "rest_framework.versioning.URLPathVersioning",
    "DEFAULT_RENDERER_CLASSES": (
        "rest_framework.renderers.JSONRenderer",
        "rest_framework.renderers.BrowsableAPIRenderer",
        "rest_framework_xml.renderers.XMLRenderer",
    ),
    "DEFAULT_THROTTLE_CLASSES": (
        "catalog.api.utils.throttle.BurstRateThrottle",
        "catalog.api.utils.throttle.SustainedRateThrottle",
        "catalog.api.utils.throttle.AnonThumbnailRateThrottle",
        "catalog.api.utils.throttle.OAuth2IdThumbnailRateThrottle",
        "catalog.api.utils.throttle.OAuth2IdSustainedRateThrottle",
        "catalog.api.utils.throttle.OAuth2IdBurstRateThrottle",
        "catalog.api.utils.throttle.EnhancedOAuth2IdSustainedRateThrottle",
        "catalog.api.utils.throttle.EnhancedOAuth2IdBurstRateThrottle",
        "catalog.api.utils.throttle.ExemptOAuth2IdRateThrottle",
    ),
    "DEFAULT_THROTTLE_RATES": {
        "anon_burst": THROTTLE_ANON_BURST,
        "anon_sustained": THROTTLE_ANON_SUSTAINED,
        "anon_thumbnail": THROTTLE_ANON_THUMBS,
        "oauth2_client_credentials_thumbnail": THROTTLE_OAUTH2_THUMBS,
        "oauth2_client_credentials_sustained": "10000/day",
        "oauth2_client_credentials_burst": "100/min",
        "enhanced_oauth2_client_credentials_sustained": "20000/day",
        "enhanced_oauth2_client_credentials_burst": "200/min",
        # ``None`` completely by-passes the rate limiting
        "exempt_oauth2_client_credentials": None,
    },
    "EXCEPTION_HANDLER": "catalog.api.utils.exceptions.exception_handler",
}

if config("DISABLE_GLOBAL_THROTTLING", default=True, cast=bool):
    # Set all to ``None`` rather than deleting so that explicitly configured
    # throttled views in tests still have the default rates to fall back onto
    REST_FRAMEWORK["DEFAULT_THROTTLE_RATES"].update(
        **{k: None for k, _ in REST_FRAMEWORK["DEFAULT_THROTTLE_RATES"].items()}
    )
    del REST_FRAMEWORK["DEFAULT_THROTTLE_CLASSES"]

REDIS_HOST = config("REDIS_HOST", default="localhost")
REDIS_PORT = config("REDIS_PORT", default=6379, cast=int)
REDIS_PASSWORD = config("REDIS_PASSWORD", default="")


def _make_cache_config(dbnum: int, **overrides) -> dict:
    return {
        "BACKEND": "django_redis.cache.RedisCache",
        "LOCATION": f"redis://{REDIS_HOST}:{REDIS_PORT}/{dbnum}",
        "OPTIONS": {
            "CLIENT_CLASS": "django_redis.client.DefaultClient",
        }
        | overrides.pop("OPTIONS", {}),
    } | overrides


CACHES = {
    # Site cache writes to 'default'
    "default": _make_cache_config(0),
    # For rapidly changing stats that we don't want to hammer the database with
    "traffic_stats": _make_cache_config(1),
    # For ensuring consistency among multiple Django workers and servers.
    # Used by Redlock.
    "locks": _make_cache_config(2),
    # Used for tracking tallied figures that shouldn't expire and are indexed
    # with a timestamp range (for example, the key could a timestamp valid
    # for a given week), allowing historical data analysis.
    "tallies": _make_cache_config(3, TIMEOUT=None),
}

# If key is not present then the authentication header won't be sent
# and query forwarding will not work as expected. Lack of query forwarding
# is not an issue for local development so this compromise is okay.
PHOTON_AUTH_KEY = config("PHOTON_AUTH_KEY", default=None)

# Produce CC-hosted thumbnails dynamically through a proxy.
PHOTON_ENDPOINT = config("PHOTON_ENDPOINT", default="https://i0.wp.com/")

# These do not need to be cast to int because we don't use them directly,
# they're just passed through to Photon's API
# Keeping them as strings makes the tests slightly less verbose (for not needing
# to cast them in assertions to match the parsed param types)
THUMBNAIL_WIDTH_PX = config("THUMBNAIL_WIDTH_PX", default="600")
THUMBNAIL_QUALITY = config("THUMBNAIL_JPG_QUALITY", default="80")

THUMBNAIL_TIMEOUT_PREFIX = config(
    "THUMBNAIL_TIMEOUT_PREFIX", default="thumbnail_timeout:"
)

AUTHENTICATION_BACKENDS = (
    "oauth2_provider.backends.OAuth2Backend",
    "django.contrib.auth.backends.ModelBackend",
)

ROOT_URLCONF = "catalog.urls"

TEMPLATES = [
    {
        "BACKEND": "django.template.backends.django.DjangoTemplates",
        "DIRS": [BASE_DIR.joinpath("catalog", "templates")],
        "APP_DIRS": True,
        "OPTIONS": {
            "context_processors": [
                "django.template.context_processors.debug",
                "django.template.context_processors.request",
                "django.contrib.auth.context_processors.auth",
                "django.contrib.messages.context_processors.messages",
            ],
        },
    },
]

WSGI_APPLICATION = "catalog.wsgi.application"

# Database
# https://docs.djangoproject.com/en/2.0/ref/settings/#databases

DATABASES = {
    "default": {
        "ENGINE": "django.db.backends.postgresql",
        "HOST": config("DJANGO_DATABASE_HOST", default="localhost"),
        "PORT": config("DJANGO_DATABASE_PORT", default=5432, cast=int),
        "USER": config("DJANGO_DATABASE_USER", default="deploy"),
        "PASSWORD": config("DJANGO_DATABASE_PASSWORD", default="deploy"),
        "NAME": config("DJANGO_DATABASE_NAME", default="openledger"),
    },
}

# Password validation
# https://docs.djangoproject.com/en/2.0/ref/settings/#auth-password-validators

AUTH_PASSWORD_VALIDATORS = [
    {
        "NAME": "django.contrib.auth.password_validation"
        ".UserAttributeSimilarityValidator",
    },
    {
        "NAME": "django.contrib.auth.password_validation" ".MinimumLengthValidator",
    },
    {
        "NAME": "django.contrib.auth.password_validation" ".CommonPasswordValidator",
    },
    {
        "NAME": "django.contrib.auth.password_validation" ".NumericPasswordValidator",
    },
]

# Internationalization
# https://docs.djangoproject.com/en/2.0/topics/i18n/

LANGUAGE_CODE = "en-us"

TIME_ZONE = "UTC"

USE_I18N = True

USE_L10N = True

USE_TZ = True

# Static files (CSS, JavaScript, Images)
# https://docs.djangoproject.com/en/2.0/howto/static-files/

STATIC_URL = "/static/"

# Allow anybody to access the API from any domain
CORS_ORIGIN_ALLOW_ALL = True

# The version of the API. We follow the semantic version specification.
API_VERSION = config("SEMANTIC_VERSION", default="Version not specified")

OUTBOUND_USER_AGENT_TEMPLATE = config(
    "OUTBOUND_USER_AGENT_TEMPLATE",
    default=f"Openverse{{purpose}}/{API_VERSION} (https://wordpress.org/openverse)",
)

# The contact email of the Openverse team
CONTACT_EMAIL = config("CONTACT_EMAIL", default="openverse@wordpress.org")

WATERMARK_ENABLED = config("WATERMARK_ENABLED", default=False, cast=bool)

EMAIL_SENDER = config("EMAIL_SENDER", default="")
EMAIL_HOST = config("EMAIL_HOST", default="")
EMAIL_PORT = config("EMAIL_PORT", default=587, cast=int)
EMAIL_HOST_USER = config("EMAIL_HOST_USER", default="")
EMAIL_HOST_PASSWORD = config("EMAIL_HOST_PASSWORD", default="")
EMAIL_SUBJECT_PREFIX = "[noreply]"
EMAIL_USE_TLS = True
DEFAULT_FROM_EMAIL = config("DEFAULT_FROM_EMAIL", default="")

if EMAIL_HOST_USER or EMAIL_HOST_PASSWORD:
    EMAIL_BACKEND = "django.core.mail.backends.smtp.EmailBackend"
else:
    EMAIL_BACKEND = "django.core.mail.backends.console.EmailBackend"

# Log full Elasticsearch response
VERBOSE_ES_RESPONSE = config("DEBUG_SCORES", default=False, cast=bool)

# Whether to boost results by authority and popularity
USE_RANK_FEATURES = config("USE_RANK_FEATURES", default=True, cast=bool)

# The scheme to use for the hyperlinks in the API responses
API_LINK_SCHEME = config("API_LINK_SCHEME", default=None)

# Proxy handling, for production
if config("IS_PROXIED", default=True, cast=bool):
    # https://docs.djangoproject.com/en/4.0/ref/settings/#use-x-forwarded-host
    USE_X_FORWARDED_HOST = True
    # https://docs.djangoproject.com/en/4.0/ref/settings/#secure-proxy-ssl-header
    SECURE_PROXY_SSL_HEADER = ("HTTP_X_FORWARDED_PROTO", "https")

# Trusted origins for CSRF
# https://docs.djangoproject.com/en/4.0/releases/4.0/#csrf-trusted-origins-changes-4-0
CSRF_TRUSTED_ORIGINS = ["https://*.openverse.engineering"]

SENTRY_DSN = config("SENTRY_DSN", default="")

SENTRY_SAMPLE_RATE = config("SENTRY_SAMPLE_RATE", default=1.0, cast=float)

if not DEBUG and SENTRY_DSN:
    sentry_sdk.init(
        dsn=SENTRY_DSN,
        integrations=[DjangoIntegration()],
        traces_sample_rate=SENTRY_SAMPLE_RATE,
        send_default_pii=False,
        environment=ENVIRONMENT,
    )

    # ALLOW_HOSTS is correctly configured so ignore this to prevent
    # spammy bots like https://github.com/robertdavidgraham/masscan
    # from pushing un-actionable alerts to Sentry like
    # https://sentry.io/share/issue/9af3cdf8ef74420aa7bbb6697760a82c/
    ignore_logger("django.security.DisallowedHost")

# Custom link validation expiration times
# Overrides can be set via LINK_VALIDATION_CACHE_EXPIRY__<http integer status code>
# and should be set as kwarg dicts for datetime.timedelta
# E.g. LINK_VALIDATION_CACHE_EXPIRY__200='{"days": 1}' will set the expiration time
# for links with HTTP status 200 to 1 day
LINK_VALIDATION_CACHE_EXPIRY_CONFIGURATION = LinkValidationCacheExpiryConfiguration()

MAX_ANONYMOUS_PAGE_SIZE = 20
MAX_AUTHED_PAGE_SIZE = 500
MAX_PAGINATION_DEPTH = 20

BASE_URL = config("BASE_URL", default="https://wordpress.org/openverse/")
